use v6;
use lib 'lib';
use Test;
use TXN::Parser;
use TXN::Parser::Types;

plan 5;

subtest
{
    my Str $file = 't/data/sample/sample.txn';
    my TXN::Parser::Actions $actions .= new;
    my $match-ledger = TXN::Parser::Grammar.parsefile($file, :$actions);

    is(
        $match-ledger.WHAT,
        Match,
        q:to/EOF/
        ♪ [Grammar.parse($document)] - 1 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ Parses accounting ledger successfully
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );

    is(
        $match-ledger.made.WHAT,
        Array[TXN::Parser::AST::Entry:D],
        q:to/EOF/
        ♪ [Is array?] - 2 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made.WHAT ~~ Array[TXN::Parser::AST::Entry:D]
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );

    is(
        $match-ledger.made[0].header.date,
        '2014-01-01',
        q:to/EOF/
        ♪ [Is expected value?] - 3 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].header.date ~~ '2014-01-01'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].header.description,
        'I started the year with $1000 in Bankwest cheque account',
        q:to/EOF/
        ♪ [Is expected value?] - 4 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].header.description
        ┃   Success   ┃        ~~ '...'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].header.important,
        0,
        q:to/EOF/
        ♪ [Is expected value?] - 5 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].header.important
        ┃   Success   ┃        == 0
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].header.tag[0],
        'TAG1',
        q:to/EOF/
        ♪ [Is expected value?] - 6 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].header.tag[0]
        ┃   Success   ┃        ~~ 'TAG1'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].header.tag[1],
        'TAG2',
        q:to/EOF/
        ♪ [Is expected value?] - 7 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].header.tag[1]
        ┃   Success   ┃        ~~ 'TAG2'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].id.number,
        0,
        q:to/EOF/
        ♪ [Is expected value?] - 8 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].id.number == 0
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].id.text,
        "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\" #TAG1 #TAG2 -- EODESC COMMENT\n  -- this is a comment line\n  Assets:Personal:Bankwest:Cheque    \$1000.00 USD\n  -- this is a second comment line\n  Equity:Personal                    \$1000.00 USD -- EOL COMMENT",
        q:to/EOF/
        ♪ [Is expected value?] - 9 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].id.text ~~ "..."
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].id.xxhash,
        1219414156,
        q:to/EOF/
        ♪ [Is expected value?] - 10 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].id.xxhash == 1219414156
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].account.entity,
        'Personal',
        q:to/EOF/
        ♪ [Is expected value?] - 11 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].account.entity
        ┃   Success   ┃        ~~ 'Personal'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].account.silo,
        TXN::Parser::Types::Silo::ASSETS,
        q:to/EOF/
        ♪ [Is expected value?] - 12 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].account.silo
        ┃   Success   ┃        ~~ TXN::Parser::Types::Silo::ASSETS
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].account.path[0],
        'Bankwest',
        q:to/EOF/
        ♪ [Is expected value?] - 13 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].account.path[0]
        ┃   Success   ┃        ~~ 'Bankwest'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].account.path[1],
        'Cheque',
        q:to/EOF/
        ♪ [Is expected value?] - 14 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].account.path[1]
        ┃   Success   ┃        ~~ 'Cheque'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].amount.asset-code,
        'USD',
        q:to/EOF/
        ♪ [Is expected value?] - 15 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].amount.asset-code
        ┃   Success   ┃        ~~ 'USD'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].amount.asset-quantity,
        1000.0,
        q:to/EOF/
        ♪ [Is expected value?] - 16 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].amount.asset-quantity
        ┃   Success   ┃        == 1000.0
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].amount.asset-symbol,
        '$',
        q:to/EOF/
        ♪ [Is expected value?] - 17 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].amount.asset-symbol
        ┃   Success   ┃        ~~ '$'
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].annot,
        TXN::Parser::AST::Entry::Posting::Annot,
        q:to/EOF/
        ♪ [Is expected value?] - 18 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].annot
        ┃   Success   ┃        ~~ TXN::Parser::AST::Entry::Posting::Annot
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].amount.plus-or-minus,
        TXN::Parser::Types::PlusMinus,
        q:to/EOF/
        ♪ [Is expected value?] - 19 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].amount.plus-or-minus
        ┃   Success   ┃        ~~ TXN::Parser::Types::PlusMinus
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].decinc,
        TXN::Parser::Types::DecInc::INC,
        q:to/EOF/
        ♪ [Is expected value?] - 20 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].decinc
        ┃   Success   ┃        ~~ TXN::Parser::Types::DecInc::INC
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].drcr,
        TXN::Parser::Types::DrCr::DEBIT,
        q:to/EOF/
        ♪ [Is expected value?] - 21 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].drcr
        ┃   Success   ┃        ~~ TXN::Parser::Types::DrCR::DEBIT
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.number,
        0,
        q:to/EOF/
        ♪ [Is expected value?] - 22 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.number == 0
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.text,
        'Assets:Personal:Bankwest:Cheque    $1000.00 USD',
        q:to/EOF/
        ♪ [Is expected value?] - 23 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.text ~~ '...'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.xxhash,
        352942826,
        q:to/EOF/
        ♪ [Is expected value?] - 24 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.xxhash
        ┃   Success   ┃        == 352942826
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.entry-id.number,
        0,
        q:to/EOF/
        ♪ [Is expected value?] - 25 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.entry-id.number
        ┃   Success   ┃        == 0
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.entry-id.text,
        "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\" #TAG1 #TAG2 -- EODESC COMMENT\n  -- this is a comment line\n  Assets:Personal:Bankwest:Cheque    \$1000.00 USD\n  -- this is a second comment line\n  Equity:Personal                    \$1000.00 USD -- EOL COMMENT",
        q:to/EOF/
        ♪ [Is expected value?] - 26 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.entry-id.text
        ┃   Success   ┃        ~~ "..."
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        $match-ledger.made[0].posting[0].id.entry-id.xxhash,
        1219414156,
        q:to/EOF/
        ♪ [Is expected value?] - 27 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ $match-ledger.made[0].posting[0].id.entry-id.xxhash
        ┃   Success   ┃        == 1219414156
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
}

# verify existence of primary and inherited exchange rate
subtest
{
    my Str $txn = q:to/EOF/;
    2016-04-26 '''
    I receive a gift of 5 BTC

    - market price is $466/BTC
    - donor's basis was $0.04/BTC
    '''
    Assets:Personal:ColdStorage    ฿5 BTC @ $466 USD « $0.04 USD
    Income:Personal:Gifts          ฿5 BTC @ $466 USD
    EOF

    my @entry = TXN::Parser.parse($txn).made;

    is(
        @entry[0].posting[0].amount.asset-code,
        'BTC',
        q:to/EOF/
        ♪ [Is expected value?] - 28 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-code eq 'BTC'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].amount.asset-quantity,
        5,
        q:to/EOF/
        ♪ [Is expected value?] - 29 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-quantity == 5
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].amount.asset-symbol,
        '฿',
        q:to/EOF/
        ♪ [Is expected value?] - 30 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-symbol eq '฿'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-code,
        'USD',
        q:to/EOF/
        ♪ [Is expected value?] - 31 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-code eq 'USD'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-price,
        466,
        q:to/EOF/
        ♪ [Is expected value?] - 32 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-price == 466
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-symbol,
        '$',
        q:to/EOF/
        ♪ [Is expected value?] - 33 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-symbol eq '$'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.inherit.asset-code,
        'USD',
        q:to/EOF/
        ♪ [Is expected value?] - 34 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.inherit.asset-code eq 'USD'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.inherit.asset-price,
        0.04,
        q:to/EOF/
        ♪ [Is expected value?] - 35 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.inherit.asset-price == 0.04
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.inherit.asset-symbol,
        '$',
        q:to/EOF/
        ♪ [Is expected value?] - 36 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.inherit.asset-symbol eq '$'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
}

# verify existence of primary exchange rate with lot sales
subtest
{
    my Str $txn = q:to/EOF/;
    2016-06-25 "I bought bitcoin for a Netflix subscription instapurchase"
      Assets:Personal:Coinbase           ฿0.1 BTC @ $700 USD → [instapurchase]
      Assets:Personal:BankWest:Check    -$70 USD

    2016-06-25 "I instapurchased Netflix subscription with bitcoin"
      Assets:Personal:NetflixPrepaid     $70 USD
      Assets:Personal:Coinbase          -฿0.1 BTC @ $700 USD ← [instapurchase]
    EOF

    my TXN::Parser::AST::Entry @entry = TXN::Parser.parse($txn).made;

    is(
        @entry[0].posting[0].amount.asset-code,
        'BTC',
        q:to/EOF/
        ♪ [Is expected value?] - 37 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-code eq 'BTC'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].amount.asset-quantity,
        0.1,
        q:to/EOF/
        ♪ [Is expected value?] - 38 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-quantity == 0.1
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].amount.asset-symbol,
        '฿',
        q:to/EOF/
        ♪ [Is expected value?] - 39 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].amount.asset-symbol eq '฿'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-code,
        'USD',
        q:to/EOF/
        ♪ [Is expected value?] - 40 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-code eq 'USD'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-price,
        700,
        q:to/EOF/
        ♪ [Is expected value?] - 41 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-price == 700
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.xe.asset-symbol,
        '$',
        q:to/EOF/
        ♪ [Is expected value?] - 42 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.xe.asset-symbol eq '$'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.lot.name,
        'instapurchase',
        q:to/EOF/
        ♪ [Is expected value?] - 43 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.lot.name eq 'instapurchase'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[0].posting[0].annot.lot.decinc,
        TXN::Parser::Types::DecInc::INC,
        q:to/EOF/
        ♪ [Is expected value?] - 44 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[0].posting[0].annot.lot.decinc
        ┃   Success   ┃         eq TXN::Parser::Types::DecInc::INC
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[1].posting[1].annot.lot.name,
        'instapurchase',
        q:to/EOF/
        ♪ [Is expected value?] - 45 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[1].posting[1].annot.lot.name eq 'instapurchase'
        ┃   Success   ┃
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
    is(
        @entry[1].posting[1].annot.lot.decinc,
        TXN::Parser::Types::DecInc::DEC,
        q:to/EOF/
        ♪ [Is expected value?] - 46 of 8334
        ┏━━━━━━━━━━━━━┓
        ┃             ┃  ∙ @entry[1].posting[1].annot.lot.decinc
        ┃   Success   ┃         eq TXN::Parser::Types::DecInc::DEC
        ┃             ┃
        ┗━━━━━━━━━━━━━┛
        EOF
    );
}

subtest
{
    my Str $file = 't/data/with-includes/with-includes.txn';
    my TXN::Parser::AST::Entry @entry = TXN::Parser.parsefile($file).made;

    # tests {{{
    ok @entry[0].header.date eqv Date.new(2011,1,1);
    ok @entry[0].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[0].header.important == 0;
    ok @entry[0].id.number eq Array[UInt:D].new(0, 0);
    ok @entry[0].id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].id.xxhash == 839297870;
    ok @entry[0].posting[0].account.entity eqv "FooCorp";
    ok @entry[0].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[0].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[0].posting[0].amount.asset-code eqv "USD";
    ok @entry[0].posting[0].amount.asset-quantity == 1000;
    ok @entry[0].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[0].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[0].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[0].posting[0].decinc eqv DecInc::INC;
    ok @entry[0].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[0].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0);
    ok @entry[0].posting[0].id.entry-id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[0].id.entry-id.xxhash == 839297870;
    ok @entry[0].posting[0].id.number == 0;
    ok @entry[0].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[0].posting[0].id.xxhash == 3244003616;
    ok @entry[0].posting[1].account.entity eqv "FooCorp";
    ok @entry[0].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[0].posting[1].amount.asset-code eqv "USD";
    ok @entry[0].posting[1].amount.asset-quantity == 1000;
    ok @entry[0].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[0].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[0].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[0].posting[1].decinc eqv DecInc::INC;
    ok @entry[0].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[0].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0);
    ok @entry[0].posting[1].id.entry-id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[1].id.entry-id.xxhash == 839297870;
    ok @entry[0].posting[1].id.number == 1;
    ok @entry[0].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[1].id.xxhash == 1025058054;
    ok @entry[1].header.date eqv Date.new(2011,1,1);
    ok @entry[1].header.description eqv "Wigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[1].header.important == 0;
    ok @entry[1].id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[1].id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].id.xxhash == 3127303444;
    ok @entry[1].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[1].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[1].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[1].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[1].posting[0].amount.asset-quantity == 1;
    ok @entry[1].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[1].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[1].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[1].posting[0].decinc eqv DecInc::INC;
    ok @entry[1].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[1].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[1].posting[0].id.entry-id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[0].id.entry-id.xxhash == 3127303444;
    ok @entry[1].posting[0].id.number == 0;
    ok @entry[1].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD";
    ok @entry[1].posting[0].id.xxhash == 3785012149;
    ok @entry[1].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[1].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[1].posting[1].amount.asset-code eqv "USD";
    ok @entry[1].posting[1].amount.asset-quantity == 1.01;
    ok @entry[1].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[1].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[1].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[1].posting[1].decinc eqv DecInc::DEC;
    ok @entry[1].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[1].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[1].posting[1].id.entry-id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[1].id.entry-id.xxhash == 3127303444;
    ok @entry[1].posting[1].id.number == 1;
    ok @entry[1].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[1].id.xxhash == 1442732716;
    ok @entry[2].header.date eqv Date.new(2011,1,16);
    ok @entry[2].header.description eqv "Wigwam LLC received a dividend of \$2011.0116 USD";
    ok @entry[2].header.important == 0;
    ok @entry[2].id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[2].id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].id.xxhash == 4018141641;
    ok @entry[2].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[2].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[2].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[2].posting[0].amount.asset-code eqv "USD";
    ok @entry[2].posting[0].amount.asset-quantity == 2011.0116;
    ok @entry[2].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[2].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[2].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[2].posting[0].decinc eqv DecInc::INC;
    ok @entry[2].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[2].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[2].posting[0].id.entry-id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[0].id.entry-id.xxhash == 4018141641;
    ok @entry[2].posting[0].id.number == 0;
    ok @entry[2].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD";
    ok @entry[2].posting[0].id.xxhash == 228891826;
    ok @entry[2].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[2].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[2].posting[1].amount.asset-code eqv "USD";
    ok @entry[2].posting[1].amount.asset-quantity == 2011.0116;
    ok @entry[2].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[2].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[2].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[2].posting[1].decinc eqv DecInc::INC;
    ok @entry[2].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[2].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[2].posting[1].id.entry-id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[1].id.entry-id.xxhash == 4018141641;
    ok @entry[2].posting[1].id.number == 1;
    ok @entry[2].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[1].id.xxhash == 3822606469;
    ok @entry[3].header.date eqv Date.new(2011,1,31);
    ok @entry[3].header.description eqv "Wigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[3].header.important == 0;
    ok @entry[3].id.number eq Array[UInt:D].new(0, 1, 2);
    ok @entry[3].id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].id.xxhash == 2478358144;
    ok @entry[3].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[3].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[3].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[3].posting[0].amount.asset-code eqv "USD";
    ok @entry[3].posting[0].amount.asset-quantity == 1.01;
    ok @entry[3].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[3].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[3].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[3].posting[0].decinc eqv DecInc::INC;
    ok @entry[3].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[3].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2);
    ok @entry[3].posting[0].id.entry-id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[0].id.entry-id.xxhash == 2478358144;
    ok @entry[3].posting[0].id.number == 0;
    ok @entry[3].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.01 USD";
    ok @entry[3].posting[0].id.xxhash == 1462893106;
    ok @entry[3].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[3].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[3].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[3].posting[1].amount.asset-quantity == 1;
    ok @entry[3].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[3].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[3].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[3].posting[1].decinc eqv DecInc::DEC;
    ok @entry[3].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[3].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2);
    ok @entry[3].posting[1].id.entry-id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[1].id.entry-id.xxhash == 2478358144;
    ok @entry[3].posting[1].id.number == 1;
    ok @entry[3].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[1].id.xxhash == 3589948839;
    ok @entry[4].header.date eqv Date.new(2011,2,1);
    ok @entry[4].header.description eqv "Wigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[4].header.important == 0;
    ok @entry[4].id.number eq Array[UInt:D].new(0, 1, 3);
    ok @entry[4].id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].id.xxhash == 2320468031;
    ok @entry[4].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[4].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[4].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[4].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[4].posting[0].amount.asset-quantity == 1;
    ok @entry[4].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[4].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[4].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[4].posting[0].decinc eqv DecInc::INC;
    ok @entry[4].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[4].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3);
    ok @entry[4].posting[0].id.entry-id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[0].id.entry-id.xxhash == 2320468031;
    ok @entry[4].posting[0].id.number == 0;
    ok @entry[4].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD";
    ok @entry[4].posting[0].id.xxhash == 826351545;
    ok @entry[4].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[4].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[4].posting[1].amount.asset-code eqv "USD";
    ok @entry[4].posting[1].amount.asset-quantity == 1.02;
    ok @entry[4].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[4].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[4].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[4].posting[1].decinc eqv DecInc::DEC;
    ok @entry[4].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[4].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3);
    ok @entry[4].posting[1].id.entry-id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[1].id.entry-id.xxhash == 2320468031;
    ok @entry[4].posting[1].id.number == 1;
    ok @entry[4].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[1].id.xxhash == 2965421556;
    ok @entry[5].header.date eqv Date.new(2011,2,16);
    ok @entry[5].header.description eqv "Wigwam LLC received a dividend of \$2011.0216 USD";
    ok @entry[5].header.important == 0;
    ok @entry[5].id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[5].id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].id.xxhash == 1177608739;
    ok @entry[5].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[5].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[5].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[5].posting[0].amount.asset-code eqv "USD";
    ok @entry[5].posting[0].amount.asset-quantity == 2011.0216;
    ok @entry[5].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[5].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[5].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[5].posting[0].decinc eqv DecInc::INC;
    ok @entry[5].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[5].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[5].posting[0].id.entry-id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[0].id.entry-id.xxhash == 1177608739;
    ok @entry[5].posting[0].id.number == 0;
    ok @entry[5].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD";
    ok @entry[5].posting[0].id.xxhash == 2162966727;
    ok @entry[5].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[5].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[5].posting[1].amount.asset-code eqv "USD";
    ok @entry[5].posting[1].amount.asset-quantity == 2011.0216;
    ok @entry[5].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[5].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[5].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[5].posting[1].decinc eqv DecInc::INC;
    ok @entry[5].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[5].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[5].posting[1].id.entry-id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[1].id.entry-id.xxhash == 1177608739;
    ok @entry[5].posting[1].id.number == 1;
    ok @entry[5].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[1].id.xxhash == 1447769892;
    ok @entry[6].header.date eqv Date.new(2011,2,28);
    ok @entry[6].header.description eqv "Wigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[6].header.important == 0;
    ok @entry[6].id.number eq Array[UInt:D].new(0, 1, 5);
    ok @entry[6].id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].id.xxhash == 3178583936;
    ok @entry[6].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[6].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[6].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[6].posting[0].amount.asset-code eqv "USD";
    ok @entry[6].posting[0].amount.asset-quantity == 1.02;
    ok @entry[6].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[6].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[6].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[6].posting[0].decinc eqv DecInc::INC;
    ok @entry[6].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[6].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 5);
    ok @entry[6].posting[0].id.entry-id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[0].id.entry-id.xxhash == 3178583936;
    ok @entry[6].posting[0].id.number == 0;
    ok @entry[6].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.02 USD";
    ok @entry[6].posting[0].id.xxhash == 1509993572;
    ok @entry[6].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[6].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[6].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[6].posting[1].amount.asset-quantity == 1;
    ok @entry[6].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[6].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[6].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[6].posting[1].decinc eqv DecInc::DEC;
    ok @entry[6].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[6].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 5);
    ok @entry[6].posting[1].id.entry-id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[1].id.entry-id.xxhash == 3178583936;
    ok @entry[6].posting[1].id.number == 1;
    ok @entry[6].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[1].id.xxhash == 991307650;
    ok @entry[7].header.date eqv Date.new(2011,3,1);
    ok @entry[7].header.description eqv "Wigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[7].header.important == 0;
    ok @entry[7].id.number eq Array[UInt:D].new(0, 1, 6);
    ok @entry[7].id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].id.xxhash == 1567632724;
    ok @entry[7].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[7].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[7].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[7].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[7].posting[0].amount.asset-quantity == 1;
    ok @entry[7].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[7].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[7].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[7].posting[0].decinc eqv DecInc::INC;
    ok @entry[7].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[7].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 6);
    ok @entry[7].posting[0].id.entry-id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[0].id.entry-id.xxhash == 1567632724;
    ok @entry[7].posting[0].id.number == 0;
    ok @entry[7].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD";
    ok @entry[7].posting[0].id.xxhash == 2399027560;
    ok @entry[7].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[7].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[7].posting[1].amount.asset-code eqv "USD";
    ok @entry[7].posting[1].amount.asset-quantity == 1.03;
    ok @entry[7].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[7].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[7].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[7].posting[1].decinc eqv DecInc::DEC;
    ok @entry[7].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[7].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 6);
    ok @entry[7].posting[1].id.entry-id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[1].id.entry-id.xxhash == 1567632724;
    ok @entry[7].posting[1].id.number == 1;
    ok @entry[7].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[1].id.xxhash == 1721285682;
    ok @entry[8].header.date eqv Date.new(2011,3,16);
    ok @entry[8].header.description eqv "Wigwam LLC received a dividend of \$2011.0316 USD";
    ok @entry[8].header.important == 0;
    ok @entry[8].id.number eq Array[UInt:D].new(0, 1, 7, 0);
    ok @entry[8].id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].id.xxhash == 3569777874;
    ok @entry[8].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[8].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[8].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[8].posting[0].amount.asset-code eqv "USD";
    ok @entry[8].posting[0].amount.asset-quantity == 2011.0316;
    ok @entry[8].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[8].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[8].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[8].posting[0].decinc eqv DecInc::INC;
    ok @entry[8].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[8].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 7, 0);
    ok @entry[8].posting[0].id.entry-id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[0].id.entry-id.xxhash == 3569777874;
    ok @entry[8].posting[0].id.number == 0;
    ok @entry[8].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD";
    ok @entry[8].posting[0].id.xxhash == 3064719331;
    ok @entry[8].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[8].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[8].posting[1].amount.asset-code eqv "USD";
    ok @entry[8].posting[1].amount.asset-quantity == 2011.0316;
    ok @entry[8].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[8].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[8].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[8].posting[1].decinc eqv DecInc::INC;
    ok @entry[8].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[8].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 7, 0);
    ok @entry[8].posting[1].id.entry-id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[1].id.entry-id.xxhash == 3569777874;
    ok @entry[8].posting[1].id.number == 1;
    ok @entry[8].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[1].id.xxhash == 1632975976;
    ok @entry[9].header.date eqv Date.new(2011,3,31);
    ok @entry[9].header.description eqv "Wigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[9].header.important == 0;
    ok @entry[9].id.number eq Array[UInt:D].new(0, 1, 8);
    ok @entry[9].id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].id.xxhash == 521313438;
    ok @entry[9].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[9].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[9].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[9].posting[0].amount.asset-code eqv "USD";
    ok @entry[9].posting[0].amount.asset-quantity == 1.03;
    ok @entry[9].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[9].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[9].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[9].posting[0].decinc eqv DecInc::INC;
    ok @entry[9].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[9].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 8);
    ok @entry[9].posting[0].id.entry-id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[0].id.entry-id.xxhash == 521313438;
    ok @entry[9].posting[0].id.number == 0;
    ok @entry[9].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.03 USD";
    ok @entry[9].posting[0].id.xxhash == 3019123955;
    ok @entry[9].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[9].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[9].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[9].posting[1].amount.asset-quantity == 1;
    ok @entry[9].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[9].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[9].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[9].posting[1].decinc eqv DecInc::DEC;
    ok @entry[9].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[9].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 8);
    ok @entry[9].posting[1].id.entry-id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[1].id.entry-id.xxhash == 521313438;
    ok @entry[9].posting[1].id.number == 1;
    ok @entry[9].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[1].id.xxhash == 2676696051;
    ok @entry[10].header.date eqv Date.new(2011,4,1);
    ok @entry[10].header.description eqv "Wigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[10].header.important == 0;
    ok @entry[10].id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[10].id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].id.xxhash == 234894800;
    ok @entry[10].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[10].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[10].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[10].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[10].posting[0].amount.asset-quantity == 1;
    ok @entry[10].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[10].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[10].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(26, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[10].posting[0].decinc eqv DecInc::INC;
    ok @entry[10].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[10].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[10].posting[0].id.entry-id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[0].id.entry-id.xxhash == 234894800;
    ok @entry[10].posting[0].id.number == 0;
    ok @entry[10].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD";
    ok @entry[10].posting[0].id.xxhash == 2392165153;
    ok @entry[10].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[10].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[10].posting[1].amount.asset-code eqv "USD";
    ok @entry[10].posting[1].amount.asset-quantity == 1.04;
    ok @entry[10].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[10].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[10].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[10].posting[1].decinc eqv DecInc::DEC;
    ok @entry[10].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[10].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[10].posting[1].id.entry-id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[1].id.entry-id.xxhash == 234894800;
    ok @entry[10].posting[1].id.number == 1;
    ok @entry[10].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[1].id.xxhash == 3179024211;
    ok @entry[11].header.date eqv Date.new(2011,4,16);
    ok @entry[11].header.description eqv "Wigwam LLC received a dividend of \$2011.0416 USD";
    ok @entry[11].header.important == 0;
    ok @entry[11].id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[11].id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].id.xxhash == 2888576925;
    ok @entry[11].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[11].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[11].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[11].posting[0].amount.asset-code eqv "USD";
    ok @entry[11].posting[0].amount.asset-quantity == 2011.0416;
    ok @entry[11].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[11].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[11].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[11].posting[0].decinc eqv DecInc::INC;
    ok @entry[11].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[11].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[11].posting[0].id.entry-id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[0].id.entry-id.xxhash == 2888576925;
    ok @entry[11].posting[0].id.number == 0;
    ok @entry[11].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD";
    ok @entry[11].posting[0].id.xxhash == 1643697461;
    ok @entry[11].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[11].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[11].posting[1].amount.asset-code eqv "USD";
    ok @entry[11].posting[1].amount.asset-quantity == 2011.0416;
    ok @entry[11].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[11].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[11].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[11].posting[1].decinc eqv DecInc::INC;
    ok @entry[11].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[11].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[11].posting[1].id.entry-id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[1].id.entry-id.xxhash == 2888576925;
    ok @entry[11].posting[1].id.number == 1;
    ok @entry[11].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[1].id.xxhash == 1601475876;
    ok @entry[12].header.date eqv Date.new(2011,4,30);
    ok @entry[12].header.description eqv "Wigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[12].header.important == 0;
    ok @entry[12].id.number eq Array[UInt:D].new(0, 2, 2);
    ok @entry[12].id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].id.xxhash == 801753479;
    ok @entry[12].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[12].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[12].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[12].posting[0].amount.asset-code eqv "USD";
    ok @entry[12].posting[0].amount.asset-quantity == 1.04;
    ok @entry[12].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[12].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[12].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[12].posting[0].decinc eqv DecInc::INC;
    ok @entry[12].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[12].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2);
    ok @entry[12].posting[0].id.entry-id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[0].id.entry-id.xxhash == 801753479;
    ok @entry[12].posting[0].id.number == 0;
    ok @entry[12].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.04 USD";
    ok @entry[12].posting[0].id.xxhash == 1481608029;
    ok @entry[12].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[12].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[12].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[12].posting[1].amount.asset-quantity == 1;
    ok @entry[12].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[12].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[12].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(26, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[12].posting[1].decinc eqv DecInc::DEC;
    ok @entry[12].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[12].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2);
    ok @entry[12].posting[1].id.entry-id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[1].id.entry-id.xxhash == 801753479;
    ok @entry[12].posting[1].id.number == 1;
    ok @entry[12].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[1].id.xxhash == 701076721;
    ok @entry[13].header.date eqv Date.new(2011,5,1);
    ok @entry[13].header.description eqv "Wigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[13].header.important == 0;
    ok @entry[13].id.number eq Array[UInt:D].new(0, 2, 3);
    ok @entry[13].id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].id.xxhash == 2708781669;
    ok @entry[13].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[13].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[13].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[13].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[13].posting[0].amount.asset-quantity == 1;
    ok @entry[13].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[13].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[13].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[13].posting[0].decinc eqv DecInc::INC;
    ok @entry[13].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[13].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3);
    ok @entry[13].posting[0].id.entry-id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[0].id.entry-id.xxhash == 2708781669;
    ok @entry[13].posting[0].id.number == 0;
    ok @entry[13].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD";
    ok @entry[13].posting[0].id.xxhash == 2741913114;
    ok @entry[13].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[13].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[13].posting[1].amount.asset-code eqv "USD";
    ok @entry[13].posting[1].amount.asset-quantity == 1.05;
    ok @entry[13].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[13].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[13].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[13].posting[1].decinc eqv DecInc::DEC;
    ok @entry[13].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[13].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3);
    ok @entry[13].posting[1].id.entry-id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[1].id.entry-id.xxhash == 2708781669;
    ok @entry[13].posting[1].id.number == 1;
    ok @entry[13].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[1].id.xxhash == 3177866826;
    ok @entry[14].header.date eqv Date.new(2011,5,16);
    ok @entry[14].header.description eqv "Wigwam LLC received a dividend of \$2011.0516 USD";
    ok @entry[14].header.important == 0;
    ok @entry[14].id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[14].id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].id.xxhash == 2585914;
    ok @entry[14].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[14].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[14].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[14].posting[0].amount.asset-code eqv "USD";
    ok @entry[14].posting[0].amount.asset-quantity == 2011.0516;
    ok @entry[14].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[14].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[14].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[14].posting[0].decinc eqv DecInc::INC;
    ok @entry[14].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[14].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[14].posting[0].id.entry-id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[0].id.entry-id.xxhash == 2585914;
    ok @entry[14].posting[0].id.number == 0;
    ok @entry[14].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD";
    ok @entry[14].posting[0].id.xxhash == 3043997489;
    ok @entry[14].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[14].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[14].posting[1].amount.asset-code eqv "USD";
    ok @entry[14].posting[1].amount.asset-quantity == 2011.0516;
    ok @entry[14].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[14].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[14].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[14].posting[1].decinc eqv DecInc::INC;
    ok @entry[14].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[14].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[14].posting[1].id.entry-id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[1].id.entry-id.xxhash == 2585914;
    ok @entry[14].posting[1].id.number == 1;
    ok @entry[14].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[1].id.xxhash == 2731775541;
    ok @entry[15].header.date eqv Date.new(2011,5,31);
    ok @entry[15].header.description eqv "Wigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[15].header.important == 0;
    ok @entry[15].id.number eq Array[UInt:D].new(0, 2, 5);
    ok @entry[15].id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].id.xxhash == 793767881;
    ok @entry[15].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[15].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[15].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[15].posting[0].amount.asset-code eqv "USD";
    ok @entry[15].posting[0].amount.asset-quantity == 1.05;
    ok @entry[15].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[15].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[15].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[15].posting[0].decinc eqv DecInc::INC;
    ok @entry[15].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[15].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 5);
    ok @entry[15].posting[0].id.entry-id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[0].id.entry-id.xxhash == 793767881;
    ok @entry[15].posting[0].id.number == 0;
    ok @entry[15].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.05 USD";
    ok @entry[15].posting[0].id.xxhash == 2669757498;
    ok @entry[15].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[15].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[15].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[15].posting[1].amount.asset-quantity == 1;
    ok @entry[15].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[15].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[15].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[15].posting[1].decinc eqv DecInc::DEC;
    ok @entry[15].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[15].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 5);
    ok @entry[15].posting[1].id.entry-id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[1].id.entry-id.xxhash == 793767881;
    ok @entry[15].posting[1].id.number == 1;
    ok @entry[15].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[1].id.xxhash == 1957446417;
    ok @entry[16].header.date eqv Date.new(2011,6,1);
    ok @entry[16].header.description eqv "Wigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[16].header.important == 0;
    ok @entry[16].id.number eq Array[UInt:D].new(0, 2, 6);
    ok @entry[16].id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].id.xxhash == 1054837725;
    ok @entry[16].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[16].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[16].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[16].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[16].posting[0].amount.asset-quantity == 1;
    ok @entry[16].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[16].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[16].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[16].posting[0].decinc eqv DecInc::INC;
    ok @entry[16].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[16].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 6);
    ok @entry[16].posting[0].id.entry-id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[0].id.entry-id.xxhash == 1054837725;
    ok @entry[16].posting[0].id.number == 0;
    ok @entry[16].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD";
    ok @entry[16].posting[0].id.xxhash == 826810409;
    ok @entry[16].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[16].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[16].posting[1].amount.asset-code eqv "USD";
    ok @entry[16].posting[1].amount.asset-quantity == 1.06;
    ok @entry[16].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[16].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[16].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[16].posting[1].decinc eqv DecInc::DEC;
    ok @entry[16].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[16].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 6);
    ok @entry[16].posting[1].id.entry-id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[1].id.entry-id.xxhash == 1054837725;
    ok @entry[16].posting[1].id.number == 1;
    ok @entry[16].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[1].id.xxhash == 1066001694;
    ok @entry[17].header.date eqv Date.new(2011,6,16);
    ok @entry[17].header.description eqv "Wigwam LLC received a dividend of \$2011.0616 USD";
    ok @entry[17].header.important == 0;
    ok @entry[17].id.number eq Array[UInt:D].new(0, 2, 7, 0);
    ok @entry[17].id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].id.xxhash == 2233477133;
    ok @entry[17].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[17].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[17].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[17].posting[0].amount.asset-code eqv "USD";
    ok @entry[17].posting[0].amount.asset-quantity == 2011.0616;
    ok @entry[17].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[17].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[17].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[17].posting[0].decinc eqv DecInc::INC;
    ok @entry[17].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[17].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 7, 0);
    ok @entry[17].posting[0].id.entry-id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[0].id.entry-id.xxhash == 2233477133;
    ok @entry[17].posting[0].id.number == 0;
    ok @entry[17].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD";
    ok @entry[17].posting[0].id.xxhash == 3145971049;
    ok @entry[17].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[17].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[17].posting[1].amount.asset-code eqv "USD";
    ok @entry[17].posting[1].amount.asset-quantity == 2011.0616;
    ok @entry[17].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[17].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[17].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[17].posting[1].decinc eqv DecInc::INC;
    ok @entry[17].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[17].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 7, 0);
    ok @entry[17].posting[1].id.entry-id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[1].id.entry-id.xxhash == 2233477133;
    ok @entry[17].posting[1].id.number == 1;
    ok @entry[17].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[1].id.xxhash == 2894491481;
    ok @entry[18].header.date eqv Date.new(2011,6,30);
    ok @entry[18].header.description eqv "Wigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[18].header.important == 0;
    ok @entry[18].id.number eq Array[UInt:D].new(0, 2, 8);
    ok @entry[18].id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].id.xxhash == 2726916770;
    ok @entry[18].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[18].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[18].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[18].posting[0].amount.asset-code eqv "USD";
    ok @entry[18].posting[0].amount.asset-quantity == 1.06;
    ok @entry[18].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[18].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[18].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[18].posting[0].decinc eqv DecInc::INC;
    ok @entry[18].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[18].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 8);
    ok @entry[18].posting[0].id.entry-id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[0].id.entry-id.xxhash == 2726916770;
    ok @entry[18].posting[0].id.number == 0;
    ok @entry[18].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.06 USD";
    ok @entry[18].posting[0].id.xxhash == 4047159393;
    ok @entry[18].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[18].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[18].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[18].posting[1].amount.asset-quantity == 1;
    ok @entry[18].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[18].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[18].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[18].posting[1].decinc eqv DecInc::DEC;
    ok @entry[18].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[18].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 8);
    ok @entry[18].posting[1].id.entry-id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[1].id.entry-id.xxhash == 2726916770;
    ok @entry[18].posting[1].id.number == 1;
    ok @entry[18].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[1].id.xxhash == 1322105725;
    ok @entry[19].header.date eqv Date.new(2011,7,1);
    ok @entry[19].header.description eqv "Wigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[19].header.important == 0;
    ok @entry[19].id.number eq Array[UInt:D].new(0, 3, 0);
    ok @entry[19].id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].id.xxhash == 2522748302;
    ok @entry[19].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[19].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[19].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[19].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[19].posting[0].amount.asset-quantity == 1;
    ok @entry[19].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[19].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[19].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(107, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[19].posting[0].decinc eqv DecInc::INC;
    ok @entry[19].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[19].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 0);
    ok @entry[19].posting[0].id.entry-id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[0].id.entry-id.xxhash == 2522748302;
    ok @entry[19].posting[0].id.number == 0;
    ok @entry[19].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD";
    ok @entry[19].posting[0].id.xxhash == 4103355834;
    ok @entry[19].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[19].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[19].posting[1].amount.asset-code eqv "USD";
    ok @entry[19].posting[1].amount.asset-quantity == 1.07;
    ok @entry[19].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[19].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[19].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[19].posting[1].decinc eqv DecInc::DEC;
    ok @entry[19].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[19].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 0);
    ok @entry[19].posting[1].id.entry-id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[1].id.entry-id.xxhash == 2522748302;
    ok @entry[19].posting[1].id.number == 1;
    ok @entry[19].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[1].id.xxhash == 879230496;
    ok @entry[20].header.date eqv Date.new(2011,7,16);
    ok @entry[20].header.description eqv "Wigwam LLC received a dividend of \$2011.0716 USD";
    ok @entry[20].header.important == 0;
    ok @entry[20].id.number eq Array[UInt:D].new(0, 3, 1, 0);
    ok @entry[20].id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].id.xxhash == 66189993;
    ok @entry[20].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[20].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[20].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[20].posting[0].amount.asset-code eqv "USD";
    ok @entry[20].posting[0].amount.asset-quantity == 2011.0716;
    ok @entry[20].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[20].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[20].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[20].posting[0].decinc eqv DecInc::INC;
    ok @entry[20].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[20].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 1, 0);
    ok @entry[20].posting[0].id.entry-id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[0].id.entry-id.xxhash == 66189993;
    ok @entry[20].posting[0].id.number == 0;
    ok @entry[20].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD";
    ok @entry[20].posting[0].id.xxhash == 1214634887;
    ok @entry[20].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[20].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[20].posting[1].amount.asset-code eqv "USD";
    ok @entry[20].posting[1].amount.asset-quantity == 2011.0716;
    ok @entry[20].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[20].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[20].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[20].posting[1].decinc eqv DecInc::INC;
    ok @entry[20].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[20].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 1, 0);
    ok @entry[20].posting[1].id.entry-id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[1].id.entry-id.xxhash == 66189993;
    ok @entry[20].posting[1].id.number == 1;
    ok @entry[20].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[1].id.xxhash == 1354653370;
    ok @entry[21].header.date eqv Date.new(2011,7,31);
    ok @entry[21].header.description eqv "Wigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[21].header.important == 0;
    ok @entry[21].id.number eq Array[UInt:D].new(0, 3, 2);
    ok @entry[21].id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].id.xxhash == 992681704;
    ok @entry[21].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[21].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[21].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[21].posting[0].amount.asset-code eqv "USD";
    ok @entry[21].posting[0].amount.asset-quantity == 1.07;
    ok @entry[21].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[21].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[21].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[21].posting[0].decinc eqv DecInc::INC;
    ok @entry[21].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[21].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 2);
    ok @entry[21].posting[0].id.entry-id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[0].id.entry-id.xxhash == 992681704;
    ok @entry[21].posting[0].id.number == 0;
    ok @entry[21].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.07 USD";
    ok @entry[21].posting[0].id.xxhash == 1539269153;
    ok @entry[21].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[21].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[21].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[21].posting[1].amount.asset-quantity == 1;
    ok @entry[21].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[21].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[21].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(107, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[21].posting[1].decinc eqv DecInc::DEC;
    ok @entry[21].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[21].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 2);
    ok @entry[21].posting[1].id.entry-id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[1].id.entry-id.xxhash == 992681704;
    ok @entry[21].posting[1].id.number == 1;
    ok @entry[21].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[1].id.xxhash == 1406482422;
    ok @entry[22].header.date eqv Date.new(2011,8,1);
    ok @entry[22].header.description eqv "Wigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[22].header.important == 0;
    ok @entry[22].id.number eq Array[UInt:D].new(0, 3, 3);
    ok @entry[22].id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].id.xxhash == 2932388883;
    ok @entry[22].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[22].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[22].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[22].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[22].posting[0].amount.asset-quantity == 1;
    ok @entry[22].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[22].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[22].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(27, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[22].posting[0].decinc eqv DecInc::INC;
    ok @entry[22].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[22].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 3);
    ok @entry[22].posting[0].id.entry-id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[0].id.entry-id.xxhash == 2932388883;
    ok @entry[22].posting[0].id.number == 0;
    ok @entry[22].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD";
    ok @entry[22].posting[0].id.xxhash == 4060607404;
    ok @entry[22].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[22].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[22].posting[1].amount.asset-code eqv "USD";
    ok @entry[22].posting[1].amount.asset-quantity == 1.08;
    ok @entry[22].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[22].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[22].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[22].posting[1].decinc eqv DecInc::DEC;
    ok @entry[22].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[22].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 3);
    ok @entry[22].posting[1].id.entry-id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[1].id.entry-id.xxhash == 2932388883;
    ok @entry[22].posting[1].id.number == 1;
    ok @entry[22].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[1].id.xxhash == 1923859345;
    ok @entry[23].header.date eqv Date.new(2011,8,16);
    ok @entry[23].header.description eqv "Wigwam LLC received a dividend of \$2011.0816 USD";
    ok @entry[23].header.important == 0;
    ok @entry[23].id.number eq Array[UInt:D].new(0, 3, 4, 0);
    ok @entry[23].id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].id.xxhash == 165262646;
    ok @entry[23].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[23].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[23].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[23].posting[0].amount.asset-code eqv "USD";
    ok @entry[23].posting[0].amount.asset-quantity == 2011.0816;
    ok @entry[23].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[23].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[23].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[23].posting[0].decinc eqv DecInc::INC;
    ok @entry[23].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[23].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 4, 0);
    ok @entry[23].posting[0].id.entry-id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[0].id.entry-id.xxhash == 165262646;
    ok @entry[23].posting[0].id.number == 0;
    ok @entry[23].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD";
    ok @entry[23].posting[0].id.xxhash == 541135489;
    ok @entry[23].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[23].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[23].posting[1].amount.asset-code eqv "USD";
    ok @entry[23].posting[1].amount.asset-quantity == 2011.0816;
    ok @entry[23].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[23].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[23].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[23].posting[1].decinc eqv DecInc::INC;
    ok @entry[23].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[23].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 4, 0);
    ok @entry[23].posting[1].id.entry-id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[1].id.entry-id.xxhash == 165262646;
    ok @entry[23].posting[1].id.number == 1;
    ok @entry[23].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[1].id.xxhash == 4267062116;
    ok @entry[24].header.date eqv Date.new(2011,8,31);
    ok @entry[24].header.description eqv "Wigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[24].header.important == 0;
    ok @entry[24].id.number eq Array[UInt:D].new(0, 3, 5);
    ok @entry[24].id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].id.xxhash == 2421394429;
    ok @entry[24].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[24].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[24].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[24].posting[0].amount.asset-code eqv "USD";
    ok @entry[24].posting[0].amount.asset-quantity == 1.08;
    ok @entry[24].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[24].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[24].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[24].posting[0].decinc eqv DecInc::INC;
    ok @entry[24].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[24].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 5);
    ok @entry[24].posting[0].id.entry-id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[0].id.entry-id.xxhash == 2421394429;
    ok @entry[24].posting[0].id.number == 0;
    ok @entry[24].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.08 USD";
    ok @entry[24].posting[0].id.xxhash == 4017504886;
    ok @entry[24].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[24].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[24].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[24].posting[1].amount.asset-quantity == 1;
    ok @entry[24].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[24].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[24].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(27, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[24].posting[1].decinc eqv DecInc::DEC;
    ok @entry[24].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[24].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 5);
    ok @entry[24].posting[1].id.entry-id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[1].id.entry-id.xxhash == 2421394429;
    ok @entry[24].posting[1].id.number == 1;
    ok @entry[24].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[1].id.xxhash == 1142673963;
    ok @entry[25].header.date eqv Date.new(2011,9,1);
    ok @entry[25].header.description eqv "Wigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[25].header.important == 0;
    ok @entry[25].id.number eq Array[UInt:D].new(0, 3, 6);
    ok @entry[25].id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].id.xxhash == 3100476228;
    ok @entry[25].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[25].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[25].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[25].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[25].posting[0].amount.asset-quantity == 1;
    ok @entry[25].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[25].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[25].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(109, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[25].posting[0].decinc eqv DecInc::INC;
    ok @entry[25].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[25].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 6);
    ok @entry[25].posting[0].id.entry-id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[0].id.entry-id.xxhash == 3100476228;
    ok @entry[25].posting[0].id.number == 0;
    ok @entry[25].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD";
    ok @entry[25].posting[0].id.xxhash == 2201473302;
    ok @entry[25].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[25].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[25].posting[1].amount.asset-code eqv "USD";
    ok @entry[25].posting[1].amount.asset-quantity == 1.09;
    ok @entry[25].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[25].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[25].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[25].posting[1].decinc eqv DecInc::DEC;
    ok @entry[25].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[25].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 6);
    ok @entry[25].posting[1].id.entry-id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[1].id.entry-id.xxhash == 3100476228;
    ok @entry[25].posting[1].id.number == 1;
    ok @entry[25].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[1].id.xxhash == 1895357630;
    ok @entry[26].header.date eqv Date.new(2011,9,16);
    ok @entry[26].header.description eqv "Wigwam LLC received a dividend of \$2011.0916 USD";
    ok @entry[26].header.important == 0;
    ok @entry[26].id.number eq Array[UInt:D].new(0, 3, 7, 0);
    ok @entry[26].id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].id.xxhash == 1591575233;
    ok @entry[26].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[26].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[26].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[26].posting[0].amount.asset-code eqv "USD";
    ok @entry[26].posting[0].amount.asset-quantity == 2011.0916;
    ok @entry[26].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[26].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[26].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[26].posting[0].decinc eqv DecInc::INC;
    ok @entry[26].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[26].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 7, 0);
    ok @entry[26].posting[0].id.entry-id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[0].id.entry-id.xxhash == 1591575233;
    ok @entry[26].posting[0].id.number == 0;
    ok @entry[26].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD";
    ok @entry[26].posting[0].id.xxhash == 3079066074;
    ok @entry[26].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[26].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[26].posting[1].amount.asset-code eqv "USD";
    ok @entry[26].posting[1].amount.asset-quantity == 2011.0916;
    ok @entry[26].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[26].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[26].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[26].posting[1].decinc eqv DecInc::INC;
    ok @entry[26].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[26].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 7, 0);
    ok @entry[26].posting[1].id.entry-id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[1].id.entry-id.xxhash == 1591575233;
    ok @entry[26].posting[1].id.number == 1;
    ok @entry[26].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[1].id.xxhash == 3486713981;
    ok @entry[27].header.date eqv Date.new(2011,9,30);
    ok @entry[27].header.description eqv "Wigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[27].header.important == 0;
    ok @entry[27].id.number eq Array[UInt:D].new(0, 3, 8);
    ok @entry[27].id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].id.xxhash == 1027238853;
    ok @entry[27].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[27].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[27].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[27].posting[0].amount.asset-code eqv "USD";
    ok @entry[27].posting[0].amount.asset-quantity == 1.09;
    ok @entry[27].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[27].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[27].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[27].posting[0].decinc eqv DecInc::INC;
    ok @entry[27].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[27].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3, 8);
    ok @entry[27].posting[0].id.entry-id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[0].id.entry-id.xxhash == 1027238853;
    ok @entry[27].posting[0].id.number == 0;
    ok @entry[27].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.09 USD";
    ok @entry[27].posting[0].id.xxhash == 2763187408;
    ok @entry[27].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[27].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[27].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[27].posting[1].amount.asset-quantity == 1;
    ok @entry[27].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[27].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[27].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(109, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[27].posting[1].decinc eqv DecInc::DEC;
    ok @entry[27].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[27].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3, 8);
    ok @entry[27].posting[1].id.entry-id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[1].id.entry-id.xxhash == 1027238853;
    ok @entry[27].posting[1].id.number == 1;
    ok @entry[27].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[1].id.xxhash == 223171015;
    ok @entry[28].header.date eqv Date.new(2011,10,1);
    ok @entry[28].header.description eqv "Wigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[28].header.important == 0;
    ok @entry[28].id.number eq Array[UInt:D].new(0, 4, 0);
    ok @entry[28].id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].id.xxhash == 2088813850;
    ok @entry[28].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[28].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[28].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[28].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[28].posting[0].amount.asset-quantity == 1;
    ok @entry[28].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[28].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[28].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(11, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[28].posting[0].decinc eqv DecInc::INC;
    ok @entry[28].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[28].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 0);
    ok @entry[28].posting[0].id.entry-id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[0].id.entry-id.xxhash == 2088813850;
    ok @entry[28].posting[0].id.number == 0;
    ok @entry[28].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD";
    ok @entry[28].posting[0].id.xxhash == 4077097927;
    ok @entry[28].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[28].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[28].posting[1].amount.asset-code eqv "USD";
    ok @entry[28].posting[1].amount.asset-quantity == 1.1;
    ok @entry[28].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[28].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[28].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[28].posting[1].decinc eqv DecInc::DEC;
    ok @entry[28].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[28].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 0);
    ok @entry[28].posting[1].id.entry-id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[1].id.entry-id.xxhash == 2088813850;
    ok @entry[28].posting[1].id.number == 1;
    ok @entry[28].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[1].id.xxhash == 39322455;
    ok @entry[29].header.date eqv Date.new(2011,10,16);
    ok @entry[29].header.description eqv "Wigwam LLC received a dividend of \$2011.1016 USD";
    ok @entry[29].header.important == 0;
    ok @entry[29].id.number eq Array[UInt:D].new(0, 4, 1, 0);
    ok @entry[29].id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].id.xxhash == 1630761716;
    ok @entry[29].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[29].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[29].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[29].posting[0].amount.asset-code eqv "USD";
    ok @entry[29].posting[0].amount.asset-quantity == 2011.1016;
    ok @entry[29].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[29].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[29].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[29].posting[0].decinc eqv DecInc::INC;
    ok @entry[29].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[29].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 1, 0);
    ok @entry[29].posting[0].id.entry-id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[0].id.entry-id.xxhash == 1630761716;
    ok @entry[29].posting[0].id.number == 0;
    ok @entry[29].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD";
    ok @entry[29].posting[0].id.xxhash == 1180154985;
    ok @entry[29].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[29].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[29].posting[1].amount.asset-code eqv "USD";
    ok @entry[29].posting[1].amount.asset-quantity == 2011.1016;
    ok @entry[29].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[29].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[29].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[29].posting[1].decinc eqv DecInc::INC;
    ok @entry[29].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[29].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 1, 0);
    ok @entry[29].posting[1].id.entry-id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[1].id.entry-id.xxhash == 1630761716;
    ok @entry[29].posting[1].id.number == 1;
    ok @entry[29].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[1].id.xxhash == 1286546360;
    ok @entry[30].header.date eqv Date.new(2011,10,31);
    ok @entry[30].header.description eqv "Wigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[30].header.important == 0;
    ok @entry[30].id.number eq Array[UInt:D].new(0, 4, 2);
    ok @entry[30].id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].id.xxhash == 3547803812;
    ok @entry[30].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[30].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[30].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[30].posting[0].amount.asset-code eqv "USD";
    ok @entry[30].posting[0].amount.asset-quantity == 1.1;
    ok @entry[30].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[30].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[30].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[30].posting[0].decinc eqv DecInc::INC;
    ok @entry[30].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[30].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 2);
    ok @entry[30].posting[0].id.entry-id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[0].id.entry-id.xxhash == 3547803812;
    ok @entry[30].posting[0].id.number == 0;
    ok @entry[30].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.10 USD";
    ok @entry[30].posting[0].id.xxhash == 7326839;
    ok @entry[30].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[30].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[30].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[30].posting[1].amount.asset-quantity == 1;
    ok @entry[30].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[30].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[30].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(11, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[30].posting[1].decinc eqv DecInc::DEC;
    ok @entry[30].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[30].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 2);
    ok @entry[30].posting[1].id.entry-id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[1].id.entry-id.xxhash == 3547803812;
    ok @entry[30].posting[1].id.number == 1;
    ok @entry[30].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[1].id.xxhash == 532507333;
    ok @entry[31].header.date eqv Date.new(2011,11,1);
    ok @entry[31].header.description eqv "Wigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[31].header.important == 0;
    ok @entry[31].id.number eq Array[UInt:D].new(0, 4, 3);
    ok @entry[31].id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].id.xxhash == 3153707624;
    ok @entry[31].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[31].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[31].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[31].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[31].posting[0].amount.asset-quantity == 1;
    ok @entry[31].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[31].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[31].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(111, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[31].posting[0].decinc eqv DecInc::INC;
    ok @entry[31].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[31].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 3);
    ok @entry[31].posting[0].id.entry-id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[0].id.entry-id.xxhash == 3153707624;
    ok @entry[31].posting[0].id.number == 0;
    ok @entry[31].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD";
    ok @entry[31].posting[0].id.xxhash == 3758225172;
    ok @entry[31].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[31].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[31].posting[1].amount.asset-code eqv "USD";
    ok @entry[31].posting[1].amount.asset-quantity == 1.11;
    ok @entry[31].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[31].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[31].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[31].posting[1].decinc eqv DecInc::DEC;
    ok @entry[31].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[31].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 3);
    ok @entry[31].posting[1].id.entry-id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[1].id.entry-id.xxhash == 3153707624;
    ok @entry[31].posting[1].id.number == 1;
    ok @entry[31].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[1].id.xxhash == 4143645507;
    ok @entry[32].header.date eqv Date.new(2011,11,16);
    ok @entry[32].header.description eqv "Wigwam LLC received a dividend of \$2011.1116 USD";
    ok @entry[32].header.important == 0;
    ok @entry[32].id.number eq Array[UInt:D].new(0, 4, 4, 0);
    ok @entry[32].id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].id.xxhash == 784831183;
    ok @entry[32].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[32].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[32].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[32].posting[0].amount.asset-code eqv "USD";
    ok @entry[32].posting[0].amount.asset-quantity == 2011.1116;
    ok @entry[32].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[32].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[32].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[32].posting[0].decinc eqv DecInc::INC;
    ok @entry[32].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[32].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 4, 0);
    ok @entry[32].posting[0].id.entry-id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[0].id.entry-id.xxhash == 784831183;
    ok @entry[32].posting[0].id.number == 0;
    ok @entry[32].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD";
    ok @entry[32].posting[0].id.xxhash == 3948144430;
    ok @entry[32].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[32].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[32].posting[1].amount.asset-code eqv "USD";
    ok @entry[32].posting[1].amount.asset-quantity == 2011.1116;
    ok @entry[32].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[32].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[32].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[32].posting[1].decinc eqv DecInc::INC;
    ok @entry[32].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[32].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 4, 0);
    ok @entry[32].posting[1].id.entry-id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[1].id.entry-id.xxhash == 784831183;
    ok @entry[32].posting[1].id.number == 1;
    ok @entry[32].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[1].id.xxhash == 3507307506;
    ok @entry[33].header.date eqv Date.new(2011,11,30);
    ok @entry[33].header.description eqv "Wigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[33].header.important == 0;
    ok @entry[33].id.number eq Array[UInt:D].new(0, 4, 5);
    ok @entry[33].id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].id.xxhash == 3036041749;
    ok @entry[33].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[33].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[33].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[33].posting[0].amount.asset-code eqv "USD";
    ok @entry[33].posting[0].amount.asset-quantity == 1.11;
    ok @entry[33].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[33].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[33].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[33].posting[0].decinc eqv DecInc::INC;
    ok @entry[33].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[33].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 5);
    ok @entry[33].posting[0].id.entry-id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[0].id.entry-id.xxhash == 3036041749;
    ok @entry[33].posting[0].id.number == 0;
    ok @entry[33].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.11 USD";
    ok @entry[33].posting[0].id.xxhash == 2788789124;
    ok @entry[33].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[33].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[33].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[33].posting[1].amount.asset-quantity == 1;
    ok @entry[33].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[33].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[33].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(111, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[33].posting[1].decinc eqv DecInc::DEC;
    ok @entry[33].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[33].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 5);
    ok @entry[33].posting[1].id.entry-id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[1].id.entry-id.xxhash == 3036041749;
    ok @entry[33].posting[1].id.number == 1;
    ok @entry[33].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[1].id.xxhash == 942077384;
    ok @entry[34].header.date eqv Date.new(2011,12,1);
    ok @entry[34].header.description eqv "Wigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[34].header.important == 0;
    ok @entry[34].id.number eq Array[UInt:D].new(0, 4, 6);
    ok @entry[34].id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].id.xxhash == 3901453053;
    ok @entry[34].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[34].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[34].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[34].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[34].posting[0].amount.asset-quantity == 1;
    ok @entry[34].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[34].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[34].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(28, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[34].posting[0].decinc eqv DecInc::INC;
    ok @entry[34].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[34].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 6);
    ok @entry[34].posting[0].id.entry-id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[0].id.entry-id.xxhash == 3901453053;
    ok @entry[34].posting[0].id.number == 0;
    ok @entry[34].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD";
    ok @entry[34].posting[0].id.xxhash == 3674014973;
    ok @entry[34].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[34].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[34].posting[1].amount.asset-code eqv "USD";
    ok @entry[34].posting[1].amount.asset-quantity == 1.12;
    ok @entry[34].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[34].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[34].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[34].posting[1].decinc eqv DecInc::DEC;
    ok @entry[34].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[34].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 6);
    ok @entry[34].posting[1].id.entry-id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[1].id.entry-id.xxhash == 3901453053;
    ok @entry[34].posting[1].id.number == 1;
    ok @entry[34].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[1].id.xxhash == 1505204997;
    ok @entry[35].header.date eqv Date.new(2011,12,16);
    ok @entry[35].header.description eqv "Wigwam LLC received a dividend of \$2011.1216 USD";
    ok @entry[35].header.important == 0;
    ok @entry[35].id.number eq Array[UInt:D].new(0, 4, 7, 0);
    ok @entry[35].id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].id.xxhash == 2743061682;
    ok @entry[35].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[35].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[35].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[35].posting[0].amount.asset-code eqv "USD";
    ok @entry[35].posting[0].amount.asset-quantity == 2011.1216;
    ok @entry[35].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[35].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[35].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[35].posting[0].decinc eqv DecInc::INC;
    ok @entry[35].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[35].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 7, 0);
    ok @entry[35].posting[0].id.entry-id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[0].id.entry-id.xxhash == 2743061682;
    ok @entry[35].posting[0].id.number == 0;
    ok @entry[35].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD";
    ok @entry[35].posting[0].id.xxhash == 1332472062;
    ok @entry[35].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[35].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[35].posting[1].amount.asset-code eqv "USD";
    ok @entry[35].posting[1].amount.asset-quantity == 2011.1216;
    ok @entry[35].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[35].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[35].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[35].posting[1].decinc eqv DecInc::INC;
    ok @entry[35].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[35].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 7, 0);
    ok @entry[35].posting[1].id.entry-id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[1].id.entry-id.xxhash == 2743061682;
    ok @entry[35].posting[1].id.number == 1;
    ok @entry[35].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[1].id.xxhash == 1858587862;
    ok @entry[36].header.date eqv Date.new(2011,12,31);
    ok @entry[36].header.description eqv "Wigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[36].header.important == 0;
    ok @entry[36].id.number eq Array[UInt:D].new(0, 4, 8);
    ok @entry[36].id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].id.xxhash == 3355735851;
    ok @entry[36].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[36].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[36].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[36].posting[0].amount.asset-code eqv "USD";
    ok @entry[36].posting[0].amount.asset-quantity == 1.12;
    ok @entry[36].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[36].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[36].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[36].posting[0].decinc eqv DecInc::INC;
    ok @entry[36].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[36].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 4, 8);
    ok @entry[36].posting[0].id.entry-id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[0].id.entry-id.xxhash == 3355735851;
    ok @entry[36].posting[0].id.number == 0;
    ok @entry[36].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.12 USD";
    ok @entry[36].posting[0].id.xxhash == 3597539835;
    ok @entry[36].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[36].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[36].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[36].posting[1].amount.asset-quantity == 1;
    ok @entry[36].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[36].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[36].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(28, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[36].posting[1].decinc eqv DecInc::DEC;
    ok @entry[36].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[36].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 4, 8);
    ok @entry[36].posting[1].id.entry-id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[1].id.entry-id.xxhash == 3355735851;
    ok @entry[36].posting[1].id.number == 1;
    ok @entry[36].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[1].id.xxhash == 1323199045;
    ok @entry[37].header.date eqv Date.new(2012,1,1);
    ok @entry[37].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[37].header.important == 0;
    ok @entry[37].id.number eq Array[UInt:D].new(1, 0);
    ok @entry[37].id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].id.xxhash == 581231782;
    ok @entry[37].posting[0].account.entity eqv "FooCorp";
    ok @entry[37].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[37].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[37].posting[0].amount.asset-code eqv "USD";
    ok @entry[37].posting[0].amount.asset-quantity == 1000;
    ok @entry[37].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[37].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[37].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[37].posting[0].decinc eqv DecInc::INC;
    ok @entry[37].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[37].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 0);
    ok @entry[37].posting[0].id.entry-id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[0].id.entry-id.xxhash == 581231782;
    ok @entry[37].posting[0].id.number == 0;
    ok @entry[37].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[37].posting[0].id.xxhash == 3244003616;
    ok @entry[37].posting[1].account.entity eqv "FooCorp";
    ok @entry[37].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[37].posting[1].amount.asset-code eqv "USD";
    ok @entry[37].posting[1].amount.asset-quantity == 1000;
    ok @entry[37].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[37].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[37].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[37].posting[1].decinc eqv DecInc::INC;
    ok @entry[37].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[37].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 0);
    ok @entry[37].posting[1].id.entry-id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[1].id.entry-id.xxhash == 581231782;
    ok @entry[37].posting[1].id.number == 1;
    ok @entry[37].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[1].id.xxhash == 1025058054;
    ok @entry[38].header.date eqv Date.new(2012,1,1);
    ok @entry[38].header.description eqv "Wigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[38].header.important == 0;
    ok @entry[38].id.number eq Array[UInt:D].new(1, 1, 0);
    ok @entry[38].id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].id.xxhash == 2118065854;
    ok @entry[38].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[38].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[38].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[38].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[38].posting[0].amount.asset-quantity == 1;
    ok @entry[38].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[38].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[38].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(201, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[38].posting[0].decinc eqv DecInc::INC;
    ok @entry[38].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[38].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 0);
    ok @entry[38].posting[0].id.entry-id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[0].id.entry-id.xxhash == 2118065854;
    ok @entry[38].posting[0].id.number == 0;
    ok @entry[38].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD";
    ok @entry[38].posting[0].id.xxhash == 3765185300;
    ok @entry[38].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[38].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[38].posting[1].amount.asset-code eqv "USD";
    ok @entry[38].posting[1].amount.asset-quantity == 2.01;
    ok @entry[38].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[38].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[38].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[38].posting[1].decinc eqv DecInc::DEC;
    ok @entry[38].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[38].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 0);
    ok @entry[38].posting[1].id.entry-id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[1].id.entry-id.xxhash == 2118065854;
    ok @entry[38].posting[1].id.number == 1;
    ok @entry[38].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[1].id.xxhash == 2966987256;
    ok @entry[39].header.date eqv Date.new(2012,1,16);
    ok @entry[39].header.description eqv "Wigwam LLC received a dividend of \$2012.0116 USD";
    ok @entry[39].header.important == 0;
    ok @entry[39].id.number eq Array[UInt:D].new(1, 1, 1, 0);
    ok @entry[39].id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].id.xxhash == 822843219;
    ok @entry[39].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[39].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[39].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[39].posting[0].amount.asset-code eqv "USD";
    ok @entry[39].posting[0].amount.asset-quantity == 2012.0116;
    ok @entry[39].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[39].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[39].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[39].posting[0].decinc eqv DecInc::INC;
    ok @entry[39].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[39].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 1, 0);
    ok @entry[39].posting[0].id.entry-id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[0].id.entry-id.xxhash == 822843219;
    ok @entry[39].posting[0].id.number == 0;
    ok @entry[39].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD";
    ok @entry[39].posting[0].id.xxhash == 4018452130;
    ok @entry[39].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[39].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[39].posting[1].amount.asset-code eqv "USD";
    ok @entry[39].posting[1].amount.asset-quantity == 2012.0116;
    ok @entry[39].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[39].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[39].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[39].posting[1].decinc eqv DecInc::INC;
    ok @entry[39].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[39].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 1, 0);
    ok @entry[39].posting[1].id.entry-id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[1].id.entry-id.xxhash == 822843219;
    ok @entry[39].posting[1].id.number == 1;
    ok @entry[39].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[1].id.xxhash == 1403571913;
    ok @entry[40].header.date eqv Date.new(2012,1,31);
    ok @entry[40].header.description eqv "Wigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[40].header.important == 0;
    ok @entry[40].id.number eq Array[UInt:D].new(1, 1, 2);
    ok @entry[40].id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].id.xxhash == 1801463403;
    ok @entry[40].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[40].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[40].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[40].posting[0].amount.asset-code eqv "USD";
    ok @entry[40].posting[0].amount.asset-quantity == 2.01;
    ok @entry[40].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[40].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[40].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[40].posting[0].decinc eqv DecInc::INC;
    ok @entry[40].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[40].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 2);
    ok @entry[40].posting[0].id.entry-id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[0].id.entry-id.xxhash == 1801463403;
    ok @entry[40].posting[0].id.number == 0;
    ok @entry[40].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.01 USD";
    ok @entry[40].posting[0].id.xxhash == 1866242286;
    ok @entry[40].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[40].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[40].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[40].posting[1].amount.asset-quantity == 1;
    ok @entry[40].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[40].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[40].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(201, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[40].posting[1].decinc eqv DecInc::DEC;
    ok @entry[40].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[40].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 2);
    ok @entry[40].posting[1].id.entry-id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[1].id.entry-id.xxhash == 1801463403;
    ok @entry[40].posting[1].id.number == 1;
    ok @entry[40].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[1].id.xxhash == 2649719130;
    ok @entry[41].header.date eqv Date.new(2012,2,1);
    ok @entry[41].header.description eqv "Wigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[41].header.important == 0;
    ok @entry[41].id.number eq Array[UInt:D].new(1, 1, 3);
    ok @entry[41].id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].id.xxhash == 1339543688;
    ok @entry[41].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[41].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[41].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[41].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[41].posting[0].amount.asset-quantity == 1;
    ok @entry[41].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[41].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[41].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[41].posting[0].decinc eqv DecInc::INC;
    ok @entry[41].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[41].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 3);
    ok @entry[41].posting[0].id.entry-id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[0].id.entry-id.xxhash == 1339543688;
    ok @entry[41].posting[0].id.number == 0;
    ok @entry[41].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD";
    ok @entry[41].posting[0].id.xxhash == 1544971153;
    ok @entry[41].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[41].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[41].posting[1].amount.asset-code eqv "USD";
    ok @entry[41].posting[1].amount.asset-quantity == 2.02;
    ok @entry[41].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[41].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[41].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[41].posting[1].decinc eqv DecInc::DEC;
    ok @entry[41].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[41].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 3);
    ok @entry[41].posting[1].id.entry-id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[1].id.entry-id.xxhash == 1339543688;
    ok @entry[41].posting[1].id.number == 1;
    ok @entry[41].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[1].id.xxhash == 4217279969;
    ok @entry[42].header.date eqv Date.new(2012,2,16);
    ok @entry[42].header.description eqv "Wigwam LLC received a dividend of \$2012.0216 USD";
    ok @entry[42].header.important == 0;
    ok @entry[42].id.number eq Array[UInt:D].new(1, 1, 4, 0);
    ok @entry[42].id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].id.xxhash == 230490045;
    ok @entry[42].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[42].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[42].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[42].posting[0].amount.asset-code eqv "USD";
    ok @entry[42].posting[0].amount.asset-quantity == 2012.0216;
    ok @entry[42].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[42].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[42].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[42].posting[0].decinc eqv DecInc::INC;
    ok @entry[42].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[42].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 4, 0);
    ok @entry[42].posting[0].id.entry-id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[0].id.entry-id.xxhash == 230490045;
    ok @entry[42].posting[0].id.number == 0;
    ok @entry[42].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD";
    ok @entry[42].posting[0].id.xxhash == 3895133532;
    ok @entry[42].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[42].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[42].posting[1].amount.asset-code eqv "USD";
    ok @entry[42].posting[1].amount.asset-quantity == 2012.0216;
    ok @entry[42].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[42].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[42].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[42].posting[1].decinc eqv DecInc::INC;
    ok @entry[42].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[42].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 4, 0);
    ok @entry[42].posting[1].id.entry-id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[1].id.entry-id.xxhash == 230490045;
    ok @entry[42].posting[1].id.number == 1;
    ok @entry[42].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[1].id.xxhash == 2386315820;
    ok @entry[43].header.date eqv Date.new(2012,2,29);
    ok @entry[43].header.description eqv "Wigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[43].header.important == 0;
    ok @entry[43].id.number eq Array[UInt:D].new(1, 1, 5);
    ok @entry[43].id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].id.xxhash == 2961867692;
    ok @entry[43].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[43].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[43].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[43].posting[0].amount.asset-code eqv "USD";
    ok @entry[43].posting[0].amount.asset-quantity == 2.02;
    ok @entry[43].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[43].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[43].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[43].posting[0].decinc eqv DecInc::INC;
    ok @entry[43].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[43].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 5);
    ok @entry[43].posting[0].id.entry-id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[0].id.entry-id.xxhash == 2961867692;
    ok @entry[43].posting[0].id.number == 0;
    ok @entry[43].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.02 USD";
    ok @entry[43].posting[0].id.xxhash == 257731380;
    ok @entry[43].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[43].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[43].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[43].posting[1].amount.asset-quantity == 1;
    ok @entry[43].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[43].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[43].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[43].posting[1].decinc eqv DecInc::DEC;
    ok @entry[43].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[43].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 5);
    ok @entry[43].posting[1].id.entry-id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[1].id.entry-id.xxhash == 2961867692;
    ok @entry[43].posting[1].id.number == 1;
    ok @entry[43].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[1].id.xxhash == 2786971912;
    ok @entry[44].header.date eqv Date.new(2012,3,1);
    ok @entry[44].header.description eqv "Wigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[44].header.important == 0;
    ok @entry[44].id.number eq Array[UInt:D].new(1, 1, 6);
    ok @entry[44].id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].id.xxhash == 462880268;
    ok @entry[44].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[44].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[44].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[44].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[44].posting[0].amount.asset-quantity == 1;
    ok @entry[44].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[44].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[44].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(203, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[44].posting[0].decinc eqv DecInc::INC;
    ok @entry[44].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[44].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 6);
    ok @entry[44].posting[0].id.entry-id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[0].id.entry-id.xxhash == 462880268;
    ok @entry[44].posting[0].id.number == 0;
    ok @entry[44].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD";
    ok @entry[44].posting[0].id.xxhash == 2548327836;
    ok @entry[44].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[44].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[44].posting[1].amount.asset-code eqv "USD";
    ok @entry[44].posting[1].amount.asset-quantity == 2.03;
    ok @entry[44].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[44].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[44].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[44].posting[1].decinc eqv DecInc::DEC;
    ok @entry[44].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[44].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 6);
    ok @entry[44].posting[1].id.entry-id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[1].id.entry-id.xxhash == 462880268;
    ok @entry[44].posting[1].id.number == 1;
    ok @entry[44].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[1].id.xxhash == 48485376;
    ok @entry[45].header.date eqv Date.new(2012,3,16);
    ok @entry[45].header.description eqv "Wigwam LLC received a dividend of \$2012.0316 USD";
    ok @entry[45].header.important == 0;
    ok @entry[45].id.number eq Array[UInt:D].new(1, 1, 7, 0);
    ok @entry[45].id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].id.xxhash == 1496080492;
    ok @entry[45].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[45].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[45].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[45].posting[0].amount.asset-code eqv "USD";
    ok @entry[45].posting[0].amount.asset-quantity == 2012.0316;
    ok @entry[45].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[45].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[45].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[45].posting[0].decinc eqv DecInc::INC;
    ok @entry[45].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[45].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 7, 0);
    ok @entry[45].posting[0].id.entry-id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[0].id.entry-id.xxhash == 1496080492;
    ok @entry[45].posting[0].id.number == 0;
    ok @entry[45].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD";
    ok @entry[45].posting[0].id.xxhash == 2099532214;
    ok @entry[45].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[45].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[45].posting[1].amount.asset-code eqv "USD";
    ok @entry[45].posting[1].amount.asset-quantity == 2012.0316;
    ok @entry[45].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[45].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[45].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[45].posting[1].decinc eqv DecInc::INC;
    ok @entry[45].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[45].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 7, 0);
    ok @entry[45].posting[1].id.entry-id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[1].id.entry-id.xxhash == 1496080492;
    ok @entry[45].posting[1].id.number == 1;
    ok @entry[45].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[1].id.xxhash == 1016359530;
    ok @entry[46].header.date eqv Date.new(2012,3,31);
    ok @entry[46].header.description eqv "Wigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[46].header.important == 0;
    ok @entry[46].id.number eq Array[UInt:D].new(1, 1, 8);
    ok @entry[46].id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].id.xxhash == 1725880199;
    ok @entry[46].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[46].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[46].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[46].posting[0].amount.asset-code eqv "USD";
    ok @entry[46].posting[0].amount.asset-quantity == 2.03;
    ok @entry[46].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[46].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[46].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[46].posting[0].decinc eqv DecInc::INC;
    ok @entry[46].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[46].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 1, 8);
    ok @entry[46].posting[0].id.entry-id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[0].id.entry-id.xxhash == 1725880199;
    ok @entry[46].posting[0].id.number == 0;
    ok @entry[46].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.03 USD";
    ok @entry[46].posting[0].id.xxhash == 3066988641;
    ok @entry[46].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[46].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[46].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[46].posting[1].amount.asset-quantity == 1;
    ok @entry[46].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[46].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[46].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(203, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[46].posting[1].decinc eqv DecInc::DEC;
    ok @entry[46].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[46].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 1, 8);
    ok @entry[46].posting[1].id.entry-id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[1].id.entry-id.xxhash == 1725880199;
    ok @entry[46].posting[1].id.number == 1;
    ok @entry[46].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[1].id.xxhash == 3495293318;
    ok @entry[47].header.date eqv Date.new(2012,4,1);
    ok @entry[47].header.description eqv "Wigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[47].header.important == 0;
    ok @entry[47].id.number eq Array[UInt:D].new(1, 2, 0);
    ok @entry[47].id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].id.xxhash == 4013949028;
    ok @entry[47].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[47].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[47].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[47].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[47].posting[0].amount.asset-quantity == 1;
    ok @entry[47].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[47].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[47].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[47].posting[0].decinc eqv DecInc::INC;
    ok @entry[47].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[47].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 0);
    ok @entry[47].posting[0].id.entry-id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[0].id.entry-id.xxhash == 4013949028;
    ok @entry[47].posting[0].id.number == 0;
    ok @entry[47].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD";
    ok @entry[47].posting[0].id.xxhash == 3088262806;
    ok @entry[47].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[47].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[47].posting[1].amount.asset-code eqv "USD";
    ok @entry[47].posting[1].amount.asset-quantity == 2.04;
    ok @entry[47].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[47].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[47].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[47].posting[1].decinc eqv DecInc::DEC;
    ok @entry[47].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[47].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 0);
    ok @entry[47].posting[1].id.entry-id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[1].id.entry-id.xxhash == 4013949028;
    ok @entry[47].posting[1].id.number == 1;
    ok @entry[47].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[1].id.xxhash == 1215240546;
    ok @entry[48].header.date eqv Date.new(2012,4,16);
    ok @entry[48].header.description eqv "Wigwam LLC received a dividend of \$2012.0416 USD";
    ok @entry[48].header.important == 0;
    ok @entry[48].id.number eq Array[UInt:D].new(1, 2, 1, 0);
    ok @entry[48].id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].id.xxhash == 2366118516;
    ok @entry[48].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[48].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[48].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[48].posting[0].amount.asset-code eqv "USD";
    ok @entry[48].posting[0].amount.asset-quantity == 2012.0416;
    ok @entry[48].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[48].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[48].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[48].posting[0].decinc eqv DecInc::INC;
    ok @entry[48].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[48].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 1, 0);
    ok @entry[48].posting[0].id.entry-id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[0].id.entry-id.xxhash == 2366118516;
    ok @entry[48].posting[0].id.number == 0;
    ok @entry[48].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD";
    ok @entry[48].posting[0].id.xxhash == 1583826195;
    ok @entry[48].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[48].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[48].posting[1].amount.asset-code eqv "USD";
    ok @entry[48].posting[1].amount.asset-quantity == 2012.0416;
    ok @entry[48].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[48].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[48].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[48].posting[1].decinc eqv DecInc::INC;
    ok @entry[48].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[48].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 1, 0);
    ok @entry[48].posting[1].id.entry-id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[1].id.entry-id.xxhash == 2366118516;
    ok @entry[48].posting[1].id.number == 1;
    ok @entry[48].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[1].id.xxhash == 3098158574;
    ok @entry[49].header.date eqv Date.new(2012,4,30);
    ok @entry[49].header.description eqv "Wigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[49].header.important == 0;
    ok @entry[49].id.number eq Array[UInt:D].new(1, 2, 2);
    ok @entry[49].id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].id.xxhash == 534726936;
    ok @entry[49].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[49].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[49].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[49].posting[0].amount.asset-code eqv "USD";
    ok @entry[49].posting[0].amount.asset-quantity == 2.04;
    ok @entry[49].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[49].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[49].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[49].posting[0].decinc eqv DecInc::INC;
    ok @entry[49].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[49].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 2);
    ok @entry[49].posting[0].id.entry-id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[0].id.entry-id.xxhash == 534726936;
    ok @entry[49].posting[0].id.number == 0;
    ok @entry[49].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.04 USD";
    ok @entry[49].posting[0].id.xxhash == 3201881197;
    ok @entry[49].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[49].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[49].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[49].posting[1].amount.asset-quantity == 1;
    ok @entry[49].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[49].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[49].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[49].posting[1].decinc eqv DecInc::DEC;
    ok @entry[49].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[49].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 2);
    ok @entry[49].posting[1].id.entry-id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[1].id.entry-id.xxhash == 534726936;
    ok @entry[49].posting[1].id.number == 1;
    ok @entry[49].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[1].id.xxhash == 2492372061;
    ok @entry[50].header.date eqv Date.new(2012,5,1);
    ok @entry[50].header.description eqv "Wigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[50].header.important == 0;
    ok @entry[50].id.number eq Array[UInt:D].new(1, 2, 3);
    ok @entry[50].id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].id.xxhash == 3245023882;
    ok @entry[50].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[50].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[50].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[50].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[50].posting[0].amount.asset-quantity == 1;
    ok @entry[50].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[50].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[50].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(41, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[50].posting[0].decinc eqv DecInc::INC;
    ok @entry[50].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[50].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 3);
    ok @entry[50].posting[0].id.entry-id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[0].id.entry-id.xxhash == 3245023882;
    ok @entry[50].posting[0].id.number == 0;
    ok @entry[50].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD";
    ok @entry[50].posting[0].id.xxhash == 1524838205;
    ok @entry[50].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[50].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[50].posting[1].amount.asset-code eqv "USD";
    ok @entry[50].posting[1].amount.asset-quantity == 2.05;
    ok @entry[50].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[50].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[50].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[50].posting[1].decinc eqv DecInc::DEC;
    ok @entry[50].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[50].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 3);
    ok @entry[50].posting[1].id.entry-id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[1].id.entry-id.xxhash == 3245023882;
    ok @entry[50].posting[1].id.number == 1;
    ok @entry[50].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[1].id.xxhash == 1182404444;
    ok @entry[51].header.date eqv Date.new(2012,5,16);
    ok @entry[51].header.description eqv "Wigwam LLC received a dividend of \$2012.0516 USD";
    ok @entry[51].header.important == 0;
    ok @entry[51].id.number eq Array[UInt:D].new(1, 2, 4, 0);
    ok @entry[51].id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].id.xxhash == 2131847179;
    ok @entry[51].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[51].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[51].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[51].posting[0].amount.asset-code eqv "USD";
    ok @entry[51].posting[0].amount.asset-quantity == 2012.0516;
    ok @entry[51].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[51].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[51].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[51].posting[0].decinc eqv DecInc::INC;
    ok @entry[51].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[51].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 4, 0);
    ok @entry[51].posting[0].id.entry-id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[0].id.entry-id.xxhash == 2131847179;
    ok @entry[51].posting[0].id.number == 0;
    ok @entry[51].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD";
    ok @entry[51].posting[0].id.xxhash == 1792523213;
    ok @entry[51].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[51].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[51].posting[1].amount.asset-code eqv "USD";
    ok @entry[51].posting[1].amount.asset-quantity == 2012.0516;
    ok @entry[51].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[51].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[51].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[51].posting[1].decinc eqv DecInc::INC;
    ok @entry[51].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[51].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 4, 0);
    ok @entry[51].posting[1].id.entry-id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[1].id.entry-id.xxhash == 2131847179;
    ok @entry[51].posting[1].id.number == 1;
    ok @entry[51].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[1].id.xxhash == 263105630;
    ok @entry[52].header.date eqv Date.new(2012,5,31);
    ok @entry[52].header.description eqv "Wigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[52].header.important == 0;
    ok @entry[52].id.number eq Array[UInt:D].new(1, 2, 5);
    ok @entry[52].id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].id.xxhash == 3774317231;
    ok @entry[52].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[52].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[52].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[52].posting[0].amount.asset-code eqv "USD";
    ok @entry[52].posting[0].amount.asset-quantity == 2.05;
    ok @entry[52].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[52].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[52].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[52].posting[0].decinc eqv DecInc::INC;
    ok @entry[52].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[52].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 5);
    ok @entry[52].posting[0].id.entry-id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[0].id.entry-id.xxhash == 3774317231;
    ok @entry[52].posting[0].id.number == 0;
    ok @entry[52].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.05 USD";
    ok @entry[52].posting[0].id.xxhash == 530861043;
    ok @entry[52].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[52].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[52].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[52].posting[1].amount.asset-quantity == 1;
    ok @entry[52].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[52].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[52].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(41, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[52].posting[1].decinc eqv DecInc::DEC;
    ok @entry[52].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[52].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 5);
    ok @entry[52].posting[1].id.entry-id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[1].id.entry-id.xxhash == 3774317231;
    ok @entry[52].posting[1].id.number == 1;
    ok @entry[52].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[1].id.xxhash == 3748210809;
    ok @entry[53].header.date eqv Date.new(2012,6,1);
    ok @entry[53].header.description eqv "Wigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[53].header.important == 0;
    ok @entry[53].id.number eq Array[UInt:D].new(1, 2, 6);
    ok @entry[53].id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].id.xxhash == 1722267067;
    ok @entry[53].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[53].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[53].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[53].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[53].posting[0].amount.asset-quantity == 1;
    ok @entry[53].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[53].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[53].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[53].posting[0].decinc eqv DecInc::INC;
    ok @entry[53].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[53].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 6);
    ok @entry[53].posting[0].id.entry-id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[0].id.entry-id.xxhash == 1722267067;
    ok @entry[53].posting[0].id.number == 0;
    ok @entry[53].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD";
    ok @entry[53].posting[0].id.xxhash == 1530406877;
    ok @entry[53].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[53].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[53].posting[1].amount.asset-code eqv "USD";
    ok @entry[53].posting[1].amount.asset-quantity == 2.06;
    ok @entry[53].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[53].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[53].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[53].posting[1].decinc eqv DecInc::DEC;
    ok @entry[53].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[53].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 6);
    ok @entry[53].posting[1].id.entry-id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[1].id.entry-id.xxhash == 1722267067;
    ok @entry[53].posting[1].id.number == 1;
    ok @entry[53].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[1].id.xxhash == 2192104612;
    ok @entry[54].header.date eqv Date.new(2012,6,16);
    ok @entry[54].header.description eqv "Wigwam LLC received a dividend of \$2012.0616 USD";
    ok @entry[54].header.important == 0;
    ok @entry[54].id.number eq Array[UInt:D].new(1, 2, 7, 0);
    ok @entry[54].id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].id.xxhash == 3942872006;
    ok @entry[54].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[54].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[54].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[54].posting[0].amount.asset-code eqv "USD";
    ok @entry[54].posting[0].amount.asset-quantity == 2012.0616;
    ok @entry[54].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[54].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[54].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[54].posting[0].decinc eqv DecInc::INC;
    ok @entry[54].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[54].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 7, 0);
    ok @entry[54].posting[0].id.entry-id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[0].id.entry-id.xxhash == 3942872006;
    ok @entry[54].posting[0].id.number == 0;
    ok @entry[54].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD";
    ok @entry[54].posting[0].id.xxhash == 1318665303;
    ok @entry[54].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[54].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[54].posting[1].amount.asset-code eqv "USD";
    ok @entry[54].posting[1].amount.asset-quantity == 2012.0616;
    ok @entry[54].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[54].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[54].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[54].posting[1].decinc eqv DecInc::INC;
    ok @entry[54].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[54].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 7, 0);
    ok @entry[54].posting[1].id.entry-id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[1].id.entry-id.xxhash == 3942872006;
    ok @entry[54].posting[1].id.number == 1;
    ok @entry[54].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[1].id.xxhash == 2809751684;
    ok @entry[55].header.date eqv Date.new(2012,6,30);
    ok @entry[55].header.description eqv "Wigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[55].header.important == 0;
    ok @entry[55].id.number eq Array[UInt:D].new(1, 2, 8);
    ok @entry[55].id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].id.xxhash == 4168296451;
    ok @entry[55].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[55].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[55].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[55].posting[0].amount.asset-code eqv "USD";
    ok @entry[55].posting[0].amount.asset-quantity == 2.06;
    ok @entry[55].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[55].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[55].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[55].posting[0].decinc eqv DecInc::INC;
    ok @entry[55].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[55].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 2, 8);
    ok @entry[55].posting[0].id.entry-id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[0].id.entry-id.xxhash == 4168296451;
    ok @entry[55].posting[0].id.number == 0;
    ok @entry[55].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.06 USD";
    ok @entry[55].posting[0].id.xxhash == 2175989140;
    ok @entry[55].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[55].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[55].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[55].posting[1].amount.asset-quantity == 1;
    ok @entry[55].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[55].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[55].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[55].posting[1].decinc eqv DecInc::DEC;
    ok @entry[55].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[55].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 2, 8);
    ok @entry[55].posting[1].id.entry-id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[1].id.entry-id.xxhash == 4168296451;
    ok @entry[55].posting[1].id.number == 1;
    ok @entry[55].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[1].id.xxhash == 3753283423;
    ok @entry[56].header.date eqv Date.new(2012,7,1);
    ok @entry[56].header.description eqv "Wigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[56].header.important == 0;
    ok @entry[56].id.number eq Array[UInt:D].new(1, 3, 0);
    ok @entry[56].id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].id.xxhash == 220833898;
    ok @entry[56].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[56].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[56].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[56].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[56].posting[0].amount.asset-quantity == 1;
    ok @entry[56].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[56].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[56].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(207, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[56].posting[0].decinc eqv DecInc::INC;
    ok @entry[56].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[56].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 0);
    ok @entry[56].posting[0].id.entry-id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[0].id.entry-id.xxhash == 220833898;
    ok @entry[56].posting[0].id.number == 0;
    ok @entry[56].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD";
    ok @entry[56].posting[0].id.xxhash == 1672261378;
    ok @entry[56].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[56].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[56].posting[1].amount.asset-code eqv "USD";
    ok @entry[56].posting[1].amount.asset-quantity == 2.07;
    ok @entry[56].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[56].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[56].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[56].posting[1].decinc eqv DecInc::DEC;
    ok @entry[56].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[56].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 0);
    ok @entry[56].posting[1].id.entry-id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[1].id.entry-id.xxhash == 220833898;
    ok @entry[56].posting[1].id.number == 1;
    ok @entry[56].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[1].id.xxhash == 2474022905;
    ok @entry[57].header.date eqv Date.new(2012,7,16);
    ok @entry[57].header.description eqv "Wigwam LLC received a dividend of \$2012.0716 USD";
    ok @entry[57].header.important == 0;
    ok @entry[57].id.number eq Array[UInt:D].new(1, 3, 1, 0);
    ok @entry[57].id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].id.xxhash == 2173105363;
    ok @entry[57].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[57].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[57].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[57].posting[0].amount.asset-code eqv "USD";
    ok @entry[57].posting[0].amount.asset-quantity == 2012.0716;
    ok @entry[57].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[57].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[57].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[57].posting[0].decinc eqv DecInc::INC;
    ok @entry[57].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[57].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 1, 0);
    ok @entry[57].posting[0].id.entry-id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[0].id.entry-id.xxhash == 2173105363;
    ok @entry[57].posting[0].id.number == 0;
    ok @entry[57].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD";
    ok @entry[57].posting[0].id.xxhash == 982703979;
    ok @entry[57].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[57].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[57].posting[1].amount.asset-code eqv "USD";
    ok @entry[57].posting[1].amount.asset-quantity == 2012.0716;
    ok @entry[57].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[57].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[57].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[57].posting[1].decinc eqv DecInc::INC;
    ok @entry[57].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[57].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 1, 0);
    ok @entry[57].posting[1].id.entry-id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[1].id.entry-id.xxhash == 2173105363;
    ok @entry[57].posting[1].id.number == 1;
    ok @entry[57].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[1].id.xxhash == 1671141747;
    ok @entry[58].header.date eqv Date.new(2012,7,31);
    ok @entry[58].header.description eqv "Wigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[58].header.important == 0;
    ok @entry[58].id.number eq Array[UInt:D].new(1, 3, 2);
    ok @entry[58].id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].id.xxhash == 2998128583;
    ok @entry[58].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[58].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[58].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[58].posting[0].amount.asset-code eqv "USD";
    ok @entry[58].posting[0].amount.asset-quantity == 2.07;
    ok @entry[58].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[58].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[58].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[58].posting[0].decinc eqv DecInc::INC;
    ok @entry[58].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[58].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 2);
    ok @entry[58].posting[0].id.entry-id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[0].id.entry-id.xxhash == 2998128583;
    ok @entry[58].posting[0].id.number == 0;
    ok @entry[58].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.07 USD";
    ok @entry[58].posting[0].id.xxhash == 2444392457;
    ok @entry[58].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[58].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[58].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[58].posting[1].amount.asset-quantity == 1;
    ok @entry[58].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[58].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[58].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(207, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[58].posting[1].decinc eqv DecInc::DEC;
    ok @entry[58].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[58].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 2);
    ok @entry[58].posting[1].id.entry-id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[1].id.entry-id.xxhash == 2998128583;
    ok @entry[58].posting[1].id.number == 1;
    ok @entry[58].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[1].id.xxhash == 1106340009;
    ok @entry[59].header.date eqv Date.new(2012,8,1);
    ok @entry[59].header.description eqv "Wigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[59].header.important == 0;
    ok @entry[59].id.number eq Array[UInt:D].new(1, 3, 3);
    ok @entry[59].id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].id.xxhash == 752691327;
    ok @entry[59].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[59].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[59].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[59].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[59].posting[0].amount.asset-quantity == 1;
    ok @entry[59].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[59].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[59].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(52, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[59].posting[0].decinc eqv DecInc::INC;
    ok @entry[59].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[59].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 3);
    ok @entry[59].posting[0].id.entry-id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[0].id.entry-id.xxhash == 752691327;
    ok @entry[59].posting[0].id.number == 0;
    ok @entry[59].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD";
    ok @entry[59].posting[0].id.xxhash == 2144803209;
    ok @entry[59].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[59].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[59].posting[1].amount.asset-code eqv "USD";
    ok @entry[59].posting[1].amount.asset-quantity == 2.08;
    ok @entry[59].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[59].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[59].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[59].posting[1].decinc eqv DecInc::DEC;
    ok @entry[59].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[59].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 3);
    ok @entry[59].posting[1].id.entry-id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[1].id.entry-id.xxhash == 752691327;
    ok @entry[59].posting[1].id.number == 1;
    ok @entry[59].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[1].id.xxhash == 2749172222;
    ok @entry[60].header.date eqv Date.new(2012,8,16);
    ok @entry[60].header.description eqv "Wigwam LLC received a dividend of \$2012.0816 USD";
    ok @entry[60].header.important == 0;
    ok @entry[60].id.number eq Array[UInt:D].new(1, 3, 4, 0);
    ok @entry[60].id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].id.xxhash == 1870601986;
    ok @entry[60].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[60].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[60].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[60].posting[0].amount.asset-code eqv "USD";
    ok @entry[60].posting[0].amount.asset-quantity == 2012.0816;
    ok @entry[60].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[60].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[60].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[60].posting[0].decinc eqv DecInc::INC;
    ok @entry[60].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[60].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 4, 0);
    ok @entry[60].posting[0].id.entry-id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[0].id.entry-id.xxhash == 1870601986;
    ok @entry[60].posting[0].id.number == 0;
    ok @entry[60].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD";
    ok @entry[60].posting[0].id.xxhash == 3457680319;
    ok @entry[60].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[60].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[60].posting[1].amount.asset-code eqv "USD";
    ok @entry[60].posting[1].amount.asset-quantity == 2012.0816;
    ok @entry[60].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[60].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[60].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[60].posting[1].decinc eqv DecInc::INC;
    ok @entry[60].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[60].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 4, 0);
    ok @entry[60].posting[1].id.entry-id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[1].id.entry-id.xxhash == 1870601986;
    ok @entry[60].posting[1].id.number == 1;
    ok @entry[60].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[1].id.xxhash == 2033582180;
    ok @entry[61].header.date eqv Date.new(2012,8,31);
    ok @entry[61].header.description eqv "Wigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[61].header.important == 0;
    ok @entry[61].id.number eq Array[UInt:D].new(1, 3, 5);
    ok @entry[61].id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].id.xxhash == 3036777254;
    ok @entry[61].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[61].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[61].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[61].posting[0].amount.asset-code eqv "USD";
    ok @entry[61].posting[0].amount.asset-quantity == 2.08;
    ok @entry[61].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[61].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[61].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[61].posting[0].decinc eqv DecInc::INC;
    ok @entry[61].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[61].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 5);
    ok @entry[61].posting[0].id.entry-id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[0].id.entry-id.xxhash == 3036777254;
    ok @entry[61].posting[0].id.number == 0;
    ok @entry[61].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.08 USD";
    ok @entry[61].posting[0].id.xxhash == 1814447104;
    ok @entry[61].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[61].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[61].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[61].posting[1].amount.asset-quantity == 1;
    ok @entry[61].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[61].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[61].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(52, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[61].posting[1].decinc eqv DecInc::DEC;
    ok @entry[61].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[61].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 5);
    ok @entry[61].posting[1].id.entry-id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[1].id.entry-id.xxhash == 3036777254;
    ok @entry[61].posting[1].id.number == 1;
    ok @entry[61].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[1].id.xxhash == 3797888419;
    ok @entry[62].header.date eqv Date.new(2012,9,1);
    ok @entry[62].header.description eqv "Wigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[62].header.important == 0;
    ok @entry[62].id.number eq Array[UInt:D].new(1, 3, 6);
    ok @entry[62].id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].id.xxhash == 3451152685;
    ok @entry[62].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[62].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[62].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[62].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[62].posting[0].amount.asset-quantity == 1;
    ok @entry[62].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[62].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[62].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(209, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[62].posting[0].decinc eqv DecInc::INC;
    ok @entry[62].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[62].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 6);
    ok @entry[62].posting[0].id.entry-id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[0].id.entry-id.xxhash == 3451152685;
    ok @entry[62].posting[0].id.number == 0;
    ok @entry[62].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD";
    ok @entry[62].posting[0].id.xxhash == 1324828912;
    ok @entry[62].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[62].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[62].posting[1].amount.asset-code eqv "USD";
    ok @entry[62].posting[1].amount.asset-quantity == 2.09;
    ok @entry[62].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[62].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[62].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[62].posting[1].decinc eqv DecInc::DEC;
    ok @entry[62].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[62].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 6);
    ok @entry[62].posting[1].id.entry-id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[1].id.entry-id.xxhash == 3451152685;
    ok @entry[62].posting[1].id.number == 1;
    ok @entry[62].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[1].id.xxhash == 243411132;
    ok @entry[63].header.date eqv Date.new(2012,9,16);
    ok @entry[63].header.description eqv "Wigwam LLC received a dividend of \$2012.0916 USD";
    ok @entry[63].header.important == 0;
    ok @entry[63].id.number eq Array[UInt:D].new(1, 3, 7, 0);
    ok @entry[63].id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].id.xxhash == 3511177135;
    ok @entry[63].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[63].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[63].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[63].posting[0].amount.asset-code eqv "USD";
    ok @entry[63].posting[0].amount.asset-quantity == 2012.0916;
    ok @entry[63].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[63].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[63].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[63].posting[0].decinc eqv DecInc::INC;
    ok @entry[63].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[63].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 7, 0);
    ok @entry[63].posting[0].id.entry-id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[0].id.entry-id.xxhash == 3511177135;
    ok @entry[63].posting[0].id.number == 0;
    ok @entry[63].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD";
    ok @entry[63].posting[0].id.xxhash == 2772381157;
    ok @entry[63].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[63].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[63].posting[1].amount.asset-code eqv "USD";
    ok @entry[63].posting[1].amount.asset-quantity == 2012.0916;
    ok @entry[63].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[63].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[63].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[63].posting[1].decinc eqv DecInc::INC;
    ok @entry[63].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[63].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 7, 0);
    ok @entry[63].posting[1].id.entry-id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[1].id.entry-id.xxhash == 3511177135;
    ok @entry[63].posting[1].id.number == 1;
    ok @entry[63].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[1].id.xxhash == 3854836884;
    ok @entry[64].header.date eqv Date.new(2012,9,30);
    ok @entry[64].header.description eqv "Wigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[64].header.important == 0;
    ok @entry[64].id.number eq Array[UInt:D].new(1, 3, 8);
    ok @entry[64].id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].id.xxhash == 1903824288;
    ok @entry[64].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[64].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[64].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[64].posting[0].amount.asset-code eqv "USD";
    ok @entry[64].posting[0].amount.asset-quantity == 2.09;
    ok @entry[64].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[64].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[64].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[64].posting[0].decinc eqv DecInc::INC;
    ok @entry[64].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[64].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 3, 8);
    ok @entry[64].posting[0].id.entry-id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[0].id.entry-id.xxhash == 1903824288;
    ok @entry[64].posting[0].id.number == 0;
    ok @entry[64].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.09 USD";
    ok @entry[64].posting[0].id.xxhash == 1045883229;
    ok @entry[64].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[64].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[64].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[64].posting[1].amount.asset-quantity == 1;
    ok @entry[64].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[64].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[64].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(209, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[64].posting[1].decinc eqv DecInc::DEC;
    ok @entry[64].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[64].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 3, 8);
    ok @entry[64].posting[1].id.entry-id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[1].id.entry-id.xxhash == 1903824288;
    ok @entry[64].posting[1].id.number == 1;
    ok @entry[64].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[1].id.xxhash == 951147748;
    ok @entry[65].header.date eqv Date.new(2012,10,1);
    ok @entry[65].header.description eqv "Wigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[65].header.important == 0;
    ok @entry[65].id.number eq Array[UInt:D].new(1, 4, 0);
    ok @entry[65].id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].id.xxhash == 3197209626;
    ok @entry[65].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[65].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[65].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[65].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[65].posting[0].amount.asset-quantity == 1;
    ok @entry[65].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[65].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[65].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[65].posting[0].decinc eqv DecInc::INC;
    ok @entry[65].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[65].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 0);
    ok @entry[65].posting[0].id.entry-id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[0].id.entry-id.xxhash == 3197209626;
    ok @entry[65].posting[0].id.number == 0;
    ok @entry[65].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD";
    ok @entry[65].posting[0].id.xxhash == 3960426057;
    ok @entry[65].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[65].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[65].posting[1].amount.asset-code eqv "USD";
    ok @entry[65].posting[1].amount.asset-quantity == 2.1;
    ok @entry[65].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[65].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[65].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[65].posting[1].decinc eqv DecInc::DEC;
    ok @entry[65].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[65].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 0);
    ok @entry[65].posting[1].id.entry-id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[1].id.entry-id.xxhash == 3197209626;
    ok @entry[65].posting[1].id.number == 1;
    ok @entry[65].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[1].id.xxhash == 2882172050;
    ok @entry[66].header.date eqv Date.new(2012,10,16);
    ok @entry[66].header.description eqv "Wigwam LLC received a dividend of \$2012.1016 USD";
    ok @entry[66].header.important == 0;
    ok @entry[66].id.number eq Array[UInt:D].new(1, 4, 1, 0);
    ok @entry[66].id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].id.xxhash == 518309921;
    ok @entry[66].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[66].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[66].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[66].posting[0].amount.asset-code eqv "USD";
    ok @entry[66].posting[0].amount.asset-quantity == 2012.1016;
    ok @entry[66].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[66].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[66].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[66].posting[0].decinc eqv DecInc::INC;
    ok @entry[66].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[66].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 1, 0);
    ok @entry[66].posting[0].id.entry-id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[0].id.entry-id.xxhash == 518309921;
    ok @entry[66].posting[0].id.number == 0;
    ok @entry[66].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD";
    ok @entry[66].posting[0].id.xxhash == 847137267;
    ok @entry[66].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[66].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[66].posting[1].amount.asset-code eqv "USD";
    ok @entry[66].posting[1].amount.asset-quantity == 2012.1016;
    ok @entry[66].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[66].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[66].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[66].posting[1].decinc eqv DecInc::INC;
    ok @entry[66].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[66].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 1, 0);
    ok @entry[66].posting[1].id.entry-id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[1].id.entry-id.xxhash == 518309921;
    ok @entry[66].posting[1].id.number == 1;
    ok @entry[66].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[1].id.xxhash == 3525777378;
    ok @entry[67].header.date eqv Date.new(2012,10,31);
    ok @entry[67].header.description eqv "Wigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[67].header.important == 0;
    ok @entry[67].id.number eq Array[UInt:D].new(1, 4, 2);
    ok @entry[67].id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].id.xxhash == 2477064034;
    ok @entry[67].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[67].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[67].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[67].posting[0].amount.asset-code eqv "USD";
    ok @entry[67].posting[0].amount.asset-quantity == 2.1;
    ok @entry[67].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[67].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[67].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[67].posting[0].decinc eqv DecInc::INC;
    ok @entry[67].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[67].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 2);
    ok @entry[67].posting[0].id.entry-id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[0].id.entry-id.xxhash == 2477064034;
    ok @entry[67].posting[0].id.number == 0;
    ok @entry[67].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.10 USD";
    ok @entry[67].posting[0].id.xxhash == 2084107816;
    ok @entry[67].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[67].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[67].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[67].posting[1].amount.asset-quantity == 1;
    ok @entry[67].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[67].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[67].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[67].posting[1].decinc eqv DecInc::DEC;
    ok @entry[67].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[67].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 2);
    ok @entry[67].posting[1].id.entry-id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[1].id.entry-id.xxhash == 2477064034;
    ok @entry[67].posting[1].id.number == 1;
    ok @entry[67].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[1].id.xxhash == 53602413;
    ok @entry[68].header.date eqv Date.new(2012,11,1);
    ok @entry[68].header.description eqv "Wigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[68].header.important == 0;
    ok @entry[68].id.number eq Array[UInt:D].new(1, 4, 3);
    ok @entry[68].id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].id.xxhash == 729373328;
    ok @entry[68].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[68].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[68].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[68].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[68].posting[0].amount.asset-quantity == 1;
    ok @entry[68].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[68].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[68].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(211, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[68].posting[0].decinc eqv DecInc::INC;
    ok @entry[68].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[68].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 3);
    ok @entry[68].posting[0].id.entry-id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[0].id.entry-id.xxhash == 729373328;
    ok @entry[68].posting[0].id.number == 0;
    ok @entry[68].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD";
    ok @entry[68].posting[0].id.xxhash == 3794157360;
    ok @entry[68].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[68].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[68].posting[1].amount.asset-code eqv "USD";
    ok @entry[68].posting[1].amount.asset-quantity == 2.11;
    ok @entry[68].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[68].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[68].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[68].posting[1].decinc eqv DecInc::DEC;
    ok @entry[68].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[68].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 3);
    ok @entry[68].posting[1].id.entry-id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[1].id.entry-id.xxhash == 729373328;
    ok @entry[68].posting[1].id.number == 1;
    ok @entry[68].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[1].id.xxhash == 974733802;
    ok @entry[69].header.date eqv Date.new(2012,11,16);
    ok @entry[69].header.description eqv "Wigwam LLC received a dividend of \$2012.1116 USD";
    ok @entry[69].header.important == 0;
    ok @entry[69].id.number eq Array[UInt:D].new(1, 4, 4, 0);
    ok @entry[69].id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].id.xxhash == 2908538066;
    ok @entry[69].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[69].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[69].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[69].posting[0].amount.asset-code eqv "USD";
    ok @entry[69].posting[0].amount.asset-quantity == 2012.1116;
    ok @entry[69].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[69].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[69].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[69].posting[0].decinc eqv DecInc::INC;
    ok @entry[69].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[69].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 4, 0);
    ok @entry[69].posting[0].id.entry-id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[0].id.entry-id.xxhash == 2908538066;
    ok @entry[69].posting[0].id.number == 0;
    ok @entry[69].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD";
    ok @entry[69].posting[0].id.xxhash == 3423083241;
    ok @entry[69].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[69].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[69].posting[1].amount.asset-code eqv "USD";
    ok @entry[69].posting[1].amount.asset-quantity == 2012.1116;
    ok @entry[69].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[69].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[69].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[69].posting[1].decinc eqv DecInc::INC;
    ok @entry[69].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[69].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 4, 0);
    ok @entry[69].posting[1].id.entry-id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[1].id.entry-id.xxhash == 2908538066;
    ok @entry[69].posting[1].id.number == 1;
    ok @entry[69].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[1].id.xxhash == 400356518;
    ok @entry[70].header.date eqv Date.new(2012,11,30);
    ok @entry[70].header.description eqv "Wigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[70].header.important == 0;
    ok @entry[70].id.number eq Array[UInt:D].new(1, 4, 5);
    ok @entry[70].id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].id.xxhash == 228106705;
    ok @entry[70].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[70].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[70].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[70].posting[0].amount.asset-code eqv "USD";
    ok @entry[70].posting[0].amount.asset-quantity == 2.11;
    ok @entry[70].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[70].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[70].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[70].posting[0].decinc eqv DecInc::INC;
    ok @entry[70].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[70].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 5);
    ok @entry[70].posting[0].id.entry-id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[0].id.entry-id.xxhash == 228106705;
    ok @entry[70].posting[0].id.number == 0;
    ok @entry[70].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.11 USD";
    ok @entry[70].posting[0].id.xxhash == 2795591875;
    ok @entry[70].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[70].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[70].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[70].posting[1].amount.asset-quantity == 1;
    ok @entry[70].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[70].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[70].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(211, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[70].posting[1].decinc eqv DecInc::DEC;
    ok @entry[70].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[70].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 5);
    ok @entry[70].posting[1].id.entry-id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[1].id.entry-id.xxhash == 228106705;
    ok @entry[70].posting[1].id.number == 1;
    ok @entry[70].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[1].id.xxhash == 1515324218;
    ok @entry[71].header.date eqv Date.new(2012,12,1);
    ok @entry[71].header.description eqv "Wigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[71].header.important == 0;
    ok @entry[71].id.number eq Array[UInt:D].new(1, 4, 6);
    ok @entry[71].id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].id.xxhash == 118051768;
    ok @entry[71].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[71].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[71].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[71].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[71].posting[0].amount.asset-quantity == 1;
    ok @entry[71].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[71].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[71].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[71].posting[0].decinc eqv DecInc::INC;
    ok @entry[71].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[71].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 6);
    ok @entry[71].posting[0].id.entry-id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[0].id.entry-id.xxhash == 118051768;
    ok @entry[71].posting[0].id.number == 0;
    ok @entry[71].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD";
    ok @entry[71].posting[0].id.xxhash == 228959427;
    ok @entry[71].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[71].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[71].posting[1].amount.asset-code eqv "USD";
    ok @entry[71].posting[1].amount.asset-quantity == 2.12;
    ok @entry[71].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[71].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[71].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[71].posting[1].decinc eqv DecInc::DEC;
    ok @entry[71].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[71].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 6);
    ok @entry[71].posting[1].id.entry-id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[1].id.entry-id.xxhash == 118051768;
    ok @entry[71].posting[1].id.number == 1;
    ok @entry[71].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[1].id.xxhash == 1223007802;
    ok @entry[72].header.date eqv Date.new(2012,12,16);
    ok @entry[72].header.description eqv "Wigwam LLC received a dividend of \$2012.1216 USD";
    ok @entry[72].header.important == 0;
    ok @entry[72].id.number eq Array[UInt:D].new(1, 4, 7, 0);
    ok @entry[72].id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].id.xxhash == 3874590774;
    ok @entry[72].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[72].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[72].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[72].posting[0].amount.asset-code eqv "USD";
    ok @entry[72].posting[0].amount.asset-quantity == 2012.1216;
    ok @entry[72].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[72].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[72].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[72].posting[0].decinc eqv DecInc::INC;
    ok @entry[72].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[72].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 7, 0);
    ok @entry[72].posting[0].id.entry-id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[0].id.entry-id.xxhash == 3874590774;
    ok @entry[72].posting[0].id.number == 0;
    ok @entry[72].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD";
    ok @entry[72].posting[0].id.xxhash == 3994190483;
    ok @entry[72].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[72].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[72].posting[1].amount.asset-code eqv "USD";
    ok @entry[72].posting[1].amount.asset-quantity == 2012.1216;
    ok @entry[72].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[72].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[72].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[72].posting[1].decinc eqv DecInc::INC;
    ok @entry[72].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[72].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 7, 0);
    ok @entry[72].posting[1].id.entry-id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[1].id.entry-id.xxhash == 3874590774;
    ok @entry[72].posting[1].id.number == 1;
    ok @entry[72].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[1].id.xxhash == 3325655088;
    ok @entry[73].header.date eqv Date.new(2012,12,31);
    ok @entry[73].header.description eqv "Wigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[73].header.important == 0;
    ok @entry[73].id.number eq Array[UInt:D].new(1, 4, 8);
    ok @entry[73].id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].id.xxhash == 2276583731;
    ok @entry[73].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[73].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[73].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[73].posting[0].amount.asset-code eqv "USD";
    ok @entry[73].posting[0].amount.asset-quantity == 2.12;
    ok @entry[73].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[73].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[73].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[73].posting[0].decinc eqv DecInc::INC;
    ok @entry[73].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[73].posting[0].id.entry-id.number eq Array[UInt:D].new(1, 4, 8);
    ok @entry[73].posting[0].id.entry-id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[0].id.entry-id.xxhash == 2276583731;
    ok @entry[73].posting[0].id.number == 0;
    ok @entry[73].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.12 USD";
    ok @entry[73].posting[0].id.xxhash == 2132552848;
    ok @entry[73].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[73].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[73].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[73].posting[1].amount.asset-quantity == 1;
    ok @entry[73].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[73].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[73].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[73].posting[1].decinc eqv DecInc::DEC;
    ok @entry[73].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[73].posting[1].id.entry-id.number eq Array[UInt:D].new(1, 4, 8);
    ok @entry[73].posting[1].id.entry-id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[1].id.entry-id.xxhash == 2276583731;
    ok @entry[73].posting[1].id.number == 1;
    ok @entry[73].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[1].id.xxhash == 3096402053;
    ok @entry[74].header.date eqv Date.new(2013,1,1);
    ok @entry[74].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[74].header.important == 0;
    ok @entry[74].id.number eq Array[UInt:D].new(2, 0);
    ok @entry[74].id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].id.xxhash == 2316435396;
    ok @entry[74].posting[0].account.entity eqv "FooCorp";
    ok @entry[74].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[74].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[74].posting[0].amount.asset-code eqv "USD";
    ok @entry[74].posting[0].amount.asset-quantity == 1000;
    ok @entry[74].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[74].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[74].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[74].posting[0].decinc eqv DecInc::INC;
    ok @entry[74].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[74].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 0);
    ok @entry[74].posting[0].id.entry-id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[0].id.entry-id.xxhash == 2316435396;
    ok @entry[74].posting[0].id.number == 0;
    ok @entry[74].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[74].posting[0].id.xxhash == 3244003616;
    ok @entry[74].posting[1].account.entity eqv "FooCorp";
    ok @entry[74].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[74].posting[1].amount.asset-code eqv "USD";
    ok @entry[74].posting[1].amount.asset-quantity == 1000;
    ok @entry[74].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[74].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[74].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[74].posting[1].decinc eqv DecInc::INC;
    ok @entry[74].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[74].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 0);
    ok @entry[74].posting[1].id.entry-id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[1].id.entry-id.xxhash == 2316435396;
    ok @entry[74].posting[1].id.number == 1;
    ok @entry[74].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[1].id.xxhash == 1025058054;
    ok @entry[75].header.date eqv Date.new(2013,1,1);
    ok @entry[75].header.description eqv "Wigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[75].header.important == 0;
    ok @entry[75].id.number eq Array[UInt:D].new(2, 1, 0);
    ok @entry[75].id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].id.xxhash == 265230743;
    ok @entry[75].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[75].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[75].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[75].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[75].posting[0].amount.asset-quantity == 1;
    ok @entry[75].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[75].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[75].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(301, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[75].posting[0].decinc eqv DecInc::INC;
    ok @entry[75].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[75].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 0);
    ok @entry[75].posting[0].id.entry-id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[0].id.entry-id.xxhash == 265230743;
    ok @entry[75].posting[0].id.number == 0;
    ok @entry[75].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD";
    ok @entry[75].posting[0].id.xxhash == 1130731432;
    ok @entry[75].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[75].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[75].posting[1].amount.asset-code eqv "USD";
    ok @entry[75].posting[1].amount.asset-quantity == 3.01;
    ok @entry[75].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[75].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[75].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[75].posting[1].decinc eqv DecInc::DEC;
    ok @entry[75].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[75].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 0);
    ok @entry[75].posting[1].id.entry-id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[1].id.entry-id.xxhash == 265230743;
    ok @entry[75].posting[1].id.number == 1;
    ok @entry[75].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[1].id.xxhash == 35511673;
    ok @entry[76].header.date eqv Date.new(2013,1,16);
    ok @entry[76].header.description eqv "Wigwam LLC received a dividend of \$2013.0116 USD";
    ok @entry[76].header.important == 0;
    ok @entry[76].id.number eq Array[UInt:D].new(2, 1, 1, 0);
    ok @entry[76].id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].id.xxhash == 1201569906;
    ok @entry[76].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[76].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[76].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[76].posting[0].amount.asset-code eqv "USD";
    ok @entry[76].posting[0].amount.asset-quantity == 2013.0116;
    ok @entry[76].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[76].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[76].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[76].posting[0].decinc eqv DecInc::INC;
    ok @entry[76].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[76].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 1, 0);
    ok @entry[76].posting[0].id.entry-id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[0].id.entry-id.xxhash == 1201569906;
    ok @entry[76].posting[0].id.number == 0;
    ok @entry[76].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD";
    ok @entry[76].posting[0].id.xxhash == 4225789096;
    ok @entry[76].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[76].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[76].posting[1].amount.asset-code eqv "USD";
    ok @entry[76].posting[1].amount.asset-quantity == 2013.0116;
    ok @entry[76].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[76].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[76].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[76].posting[1].decinc eqv DecInc::INC;
    ok @entry[76].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[76].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 1, 0);
    ok @entry[76].posting[1].id.entry-id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[1].id.entry-id.xxhash == 1201569906;
    ok @entry[76].posting[1].id.number == 1;
    ok @entry[76].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[1].id.xxhash == 1105245657;
    ok @entry[77].header.date eqv Date.new(2013,1,31);
    ok @entry[77].header.description eqv "Wigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[77].header.important == 0;
    ok @entry[77].id.number eq Array[UInt:D].new(2, 1, 2);
    ok @entry[77].id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].id.xxhash == 3717945709;
    ok @entry[77].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[77].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[77].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[77].posting[0].amount.asset-code eqv "USD";
    ok @entry[77].posting[0].amount.asset-quantity == 3.01;
    ok @entry[77].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[77].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[77].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[77].posting[0].decinc eqv DecInc::INC;
    ok @entry[77].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[77].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 2);
    ok @entry[77].posting[0].id.entry-id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[0].id.entry-id.xxhash == 3717945709;
    ok @entry[77].posting[0].id.number == 0;
    ok @entry[77].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.01 USD";
    ok @entry[77].posting[0].id.xxhash == 2990923830;
    ok @entry[77].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[77].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[77].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[77].posting[1].amount.asset-quantity == 1;
    ok @entry[77].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[77].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[77].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(301, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[77].posting[1].decinc eqv DecInc::DEC;
    ok @entry[77].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[77].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 2);
    ok @entry[77].posting[1].id.entry-id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[1].id.entry-id.xxhash == 3717945709;
    ok @entry[77].posting[1].id.number == 1;
    ok @entry[77].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[1].id.xxhash == 1596641458;
    ok @entry[78].header.date eqv Date.new(2013,2,1);
    ok @entry[78].header.description eqv "Wigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[78].header.important == 0;
    ok @entry[78].id.number eq Array[UInt:D].new(2, 1, 3);
    ok @entry[78].id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].id.xxhash == 4024926956;
    ok @entry[78].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[78].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[78].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[78].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[78].posting[0].amount.asset-quantity == 1;
    ok @entry[78].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[78].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[78].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(151, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[78].posting[0].decinc eqv DecInc::INC;
    ok @entry[78].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[78].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 3);
    ok @entry[78].posting[0].id.entry-id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[0].id.entry-id.xxhash == 4024926956;
    ok @entry[78].posting[0].id.number == 0;
    ok @entry[78].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD";
    ok @entry[78].posting[0].id.xxhash == 1475938137;
    ok @entry[78].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[78].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[78].posting[1].amount.asset-code eqv "USD";
    ok @entry[78].posting[1].amount.asset-quantity == 3.02;
    ok @entry[78].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[78].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[78].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[78].posting[1].decinc eqv DecInc::DEC;
    ok @entry[78].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[78].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 3);
    ok @entry[78].posting[1].id.entry-id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[1].id.entry-id.xxhash == 4024926956;
    ok @entry[78].posting[1].id.number == 1;
    ok @entry[78].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[1].id.xxhash == 3350274487;
    ok @entry[79].header.date eqv Date.new(2013,2,16);
    ok @entry[79].header.description eqv "Wigwam LLC received a dividend of \$2013.0216 USD";
    ok @entry[79].header.important == 0;
    ok @entry[79].id.number eq Array[UInt:D].new(2, 1, 4, 0);
    ok @entry[79].id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].id.xxhash == 2203509452;
    ok @entry[79].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[79].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[79].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[79].posting[0].amount.asset-code eqv "USD";
    ok @entry[79].posting[0].amount.asset-quantity == 2013.0216;
    ok @entry[79].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[79].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[79].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[79].posting[0].decinc eqv DecInc::INC;
    ok @entry[79].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[79].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 4, 0);
    ok @entry[79].posting[0].id.entry-id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[0].id.entry-id.xxhash == 2203509452;
    ok @entry[79].posting[0].id.number == 0;
    ok @entry[79].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD";
    ok @entry[79].posting[0].id.xxhash == 2242748384;
    ok @entry[79].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[79].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[79].posting[1].amount.asset-code eqv "USD";
    ok @entry[79].posting[1].amount.asset-quantity == 2013.0216;
    ok @entry[79].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[79].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[79].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[79].posting[1].decinc eqv DecInc::INC;
    ok @entry[79].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[79].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 4, 0);
    ok @entry[79].posting[1].id.entry-id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[1].id.entry-id.xxhash == 2203509452;
    ok @entry[79].posting[1].id.number == 1;
    ok @entry[79].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[1].id.xxhash == 2807440849;
    ok @entry[80].header.date eqv Date.new(2013,2,28);
    ok @entry[80].header.description eqv "Wigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[80].header.important == 0;
    ok @entry[80].id.number eq Array[UInt:D].new(2, 1, 5);
    ok @entry[80].id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].id.xxhash == 1337772394;
    ok @entry[80].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[80].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[80].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[80].posting[0].amount.asset-code eqv "USD";
    ok @entry[80].posting[0].amount.asset-quantity == 3.02;
    ok @entry[80].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[80].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[80].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[80].posting[0].decinc eqv DecInc::INC;
    ok @entry[80].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[80].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 5);
    ok @entry[80].posting[0].id.entry-id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[0].id.entry-id.xxhash == 1337772394;
    ok @entry[80].posting[0].id.number == 0;
    ok @entry[80].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.02 USD";
    ok @entry[80].posting[0].id.xxhash == 1829432264;
    ok @entry[80].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[80].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[80].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[80].posting[1].amount.asset-quantity == 1;
    ok @entry[80].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[80].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[80].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(151, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[80].posting[1].decinc eqv DecInc::DEC;
    ok @entry[80].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[80].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 5);
    ok @entry[80].posting[1].id.entry-id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[1].id.entry-id.xxhash == 1337772394;
    ok @entry[80].posting[1].id.number == 1;
    ok @entry[80].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[1].id.xxhash == 1035364396;
    ok @entry[81].header.date eqv Date.new(2013,3,1);
    ok @entry[81].header.description eqv "Wigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[81].header.important == 0;
    ok @entry[81].id.number eq Array[UInt:D].new(2, 1, 6);
    ok @entry[81].id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].id.xxhash == 3993675371;
    ok @entry[81].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[81].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[81].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[81].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[81].posting[0].amount.asset-quantity == 1;
    ok @entry[81].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[81].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[81].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(303, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[81].posting[0].decinc eqv DecInc::INC;
    ok @entry[81].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[81].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 6);
    ok @entry[81].posting[0].id.entry-id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[0].id.entry-id.xxhash == 3993675371;
    ok @entry[81].posting[0].id.number == 0;
    ok @entry[81].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD";
    ok @entry[81].posting[0].id.xxhash == 3084675232;
    ok @entry[81].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[81].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[81].posting[1].amount.asset-code eqv "USD";
    ok @entry[81].posting[1].amount.asset-quantity == 3.03;
    ok @entry[81].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[81].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[81].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[81].posting[1].decinc eqv DecInc::DEC;
    ok @entry[81].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[81].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 6);
    ok @entry[81].posting[1].id.entry-id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[1].id.entry-id.xxhash == 3993675371;
    ok @entry[81].posting[1].id.number == 1;
    ok @entry[81].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[1].id.xxhash == 1840852121;
    ok @entry[82].header.date eqv Date.new(2013,3,16);
    ok @entry[82].header.description eqv "Wigwam LLC received a dividend of \$2013.0316 USD";
    ok @entry[82].header.important == 0;
    ok @entry[82].id.number eq Array[UInt:D].new(2, 1, 7, 0);
    ok @entry[82].id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].id.xxhash == 1562886410;
    ok @entry[82].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[82].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[82].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[82].posting[0].amount.asset-code eqv "USD";
    ok @entry[82].posting[0].amount.asset-quantity == 2013.0316;
    ok @entry[82].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[82].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[82].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[82].posting[0].decinc eqv DecInc::INC;
    ok @entry[82].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[82].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 7, 0);
    ok @entry[82].posting[0].id.entry-id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[0].id.entry-id.xxhash == 1562886410;
    ok @entry[82].posting[0].id.number == 0;
    ok @entry[82].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD";
    ok @entry[82].posting[0].id.xxhash == 4042680049;
    ok @entry[82].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[82].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[82].posting[1].amount.asset-code eqv "USD";
    ok @entry[82].posting[1].amount.asset-quantity == 2013.0316;
    ok @entry[82].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[82].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[82].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[82].posting[1].decinc eqv DecInc::INC;
    ok @entry[82].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[82].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 7, 0);
    ok @entry[82].posting[1].id.entry-id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[1].id.entry-id.xxhash == 1562886410;
    ok @entry[82].posting[1].id.number == 1;
    ok @entry[82].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[1].id.xxhash == 1244356843;
    ok @entry[83].header.date eqv Date.new(2013,3,31);
    ok @entry[83].header.description eqv "Wigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[83].header.important == 0;
    ok @entry[83].id.number eq Array[UInt:D].new(2, 1, 8);
    ok @entry[83].id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].id.xxhash == 1634210627;
    ok @entry[83].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[83].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[83].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[83].posting[0].amount.asset-code eqv "USD";
    ok @entry[83].posting[0].amount.asset-quantity == 3.03;
    ok @entry[83].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[83].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[83].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[83].posting[0].decinc eqv DecInc::INC;
    ok @entry[83].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[83].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 1, 8);
    ok @entry[83].posting[0].id.entry-id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[0].id.entry-id.xxhash == 1634210627;
    ok @entry[83].posting[0].id.number == 0;
    ok @entry[83].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.03 USD";
    ok @entry[83].posting[0].id.xxhash == 2812452536;
    ok @entry[83].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[83].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[83].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[83].posting[1].amount.asset-quantity == 1;
    ok @entry[83].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[83].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[83].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(303, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[83].posting[1].decinc eqv DecInc::DEC;
    ok @entry[83].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[83].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 1, 8);
    ok @entry[83].posting[1].id.entry-id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[1].id.entry-id.xxhash == 1634210627;
    ok @entry[83].posting[1].id.number == 1;
    ok @entry[83].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[1].id.xxhash == 3941709198;
    ok @entry[84].header.date eqv Date.new(2013,4,1);
    ok @entry[84].header.description eqv "Wigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[84].header.important == 0;
    ok @entry[84].id.number eq Array[UInt:D].new(2, 2, 0);
    ok @entry[84].id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].id.xxhash == 713752082;
    ok @entry[84].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[84].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[84].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[84].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[84].posting[0].amount.asset-quantity == 1;
    ok @entry[84].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[84].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[84].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(76, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[84].posting[0].decinc eqv DecInc::INC;
    ok @entry[84].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[84].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 0);
    ok @entry[84].posting[0].id.entry-id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[0].id.entry-id.xxhash == 713752082;
    ok @entry[84].posting[0].id.number == 0;
    ok @entry[84].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD";
    ok @entry[84].posting[0].id.xxhash == 2233409856;
    ok @entry[84].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[84].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[84].posting[1].amount.asset-code eqv "USD";
    ok @entry[84].posting[1].amount.asset-quantity == 3.04;
    ok @entry[84].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[84].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[84].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[84].posting[1].decinc eqv DecInc::DEC;
    ok @entry[84].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[84].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 0);
    ok @entry[84].posting[1].id.entry-id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[1].id.entry-id.xxhash == 713752082;
    ok @entry[84].posting[1].id.number == 1;
    ok @entry[84].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[1].id.xxhash == 2267252423;
    ok @entry[85].header.date eqv Date.new(2013,4,16);
    ok @entry[85].header.description eqv "Wigwam LLC received a dividend of \$2013.0416 USD";
    ok @entry[85].header.important == 0;
    ok @entry[85].id.number eq Array[UInt:D].new(2, 2, 1, 0);
    ok @entry[85].id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].id.xxhash == 2573101041;
    ok @entry[85].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[85].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[85].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[85].posting[0].amount.asset-code eqv "USD";
    ok @entry[85].posting[0].amount.asset-quantity == 2013.0416;
    ok @entry[85].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[85].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[85].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[85].posting[0].decinc eqv DecInc::INC;
    ok @entry[85].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[85].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 1, 0);
    ok @entry[85].posting[0].id.entry-id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[0].id.entry-id.xxhash == 2573101041;
    ok @entry[85].posting[0].id.number == 0;
    ok @entry[85].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD";
    ok @entry[85].posting[0].id.xxhash == 3978994459;
    ok @entry[85].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[85].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[85].posting[1].amount.asset-code eqv "USD";
    ok @entry[85].posting[1].amount.asset-quantity == 2013.0416;
    ok @entry[85].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[85].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[85].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[85].posting[1].decinc eqv DecInc::INC;
    ok @entry[85].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[85].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 1, 0);
    ok @entry[85].posting[1].id.entry-id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[1].id.entry-id.xxhash == 2573101041;
    ok @entry[85].posting[1].id.number == 1;
    ok @entry[85].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[1].id.xxhash == 419651900;
    ok @entry[86].header.date eqv Date.new(2013,4,30);
    ok @entry[86].header.description eqv "Wigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[86].header.important == 0;
    ok @entry[86].id.number eq Array[UInt:D].new(2, 2, 2);
    ok @entry[86].id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].id.xxhash == 2698432083;
    ok @entry[86].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[86].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[86].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[86].posting[0].amount.asset-code eqv "USD";
    ok @entry[86].posting[0].amount.asset-quantity == 3.04;
    ok @entry[86].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[86].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[86].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[86].posting[0].decinc eqv DecInc::INC;
    ok @entry[86].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[86].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 2);
    ok @entry[86].posting[0].id.entry-id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[0].id.entry-id.xxhash == 2698432083;
    ok @entry[86].posting[0].id.number == 0;
    ok @entry[86].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.04 USD";
    ok @entry[86].posting[0].id.xxhash == 170876880;
    ok @entry[86].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[86].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[86].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[86].posting[1].amount.asset-quantity == 1;
    ok @entry[86].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[86].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[86].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(76, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[86].posting[1].decinc eqv DecInc::DEC;
    ok @entry[86].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[86].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 2);
    ok @entry[86].posting[1].id.entry-id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[1].id.entry-id.xxhash == 2698432083;
    ok @entry[86].posting[1].id.number == 1;
    ok @entry[86].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[1].id.xxhash == 3893093683;
    ok @entry[87].header.date eqv Date.new(2013,5,1);
    ok @entry[87].header.description eqv "Wigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[87].header.important == 0;
    ok @entry[87].id.number eq Array[UInt:D].new(2, 2, 3);
    ok @entry[87].id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].id.xxhash == 607841021;
    ok @entry[87].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[87].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[87].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[87].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[87].posting[0].amount.asset-quantity == 1;
    ok @entry[87].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[87].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[87].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(61, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[87].posting[0].decinc eqv DecInc::INC;
    ok @entry[87].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[87].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 3);
    ok @entry[87].posting[0].id.entry-id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[0].id.entry-id.xxhash == 607841021;
    ok @entry[87].posting[0].id.number == 0;
    ok @entry[87].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD";
    ok @entry[87].posting[0].id.xxhash == 3108085666;
    ok @entry[87].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[87].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[87].posting[1].amount.asset-code eqv "USD";
    ok @entry[87].posting[1].amount.asset-quantity == 3.05;
    ok @entry[87].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[87].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[87].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[87].posting[1].decinc eqv DecInc::DEC;
    ok @entry[87].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[87].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 3);
    ok @entry[87].posting[1].id.entry-id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[1].id.entry-id.xxhash == 607841021;
    ok @entry[87].posting[1].id.number == 1;
    ok @entry[87].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[1].id.xxhash == 2590724923;
    ok @entry[88].header.date eqv Date.new(2013,5,16);
    ok @entry[88].header.description eqv "Wigwam LLC received a dividend of \$2013.0516 USD";
    ok @entry[88].header.important == 0;
    ok @entry[88].id.number eq Array[UInt:D].new(2, 2, 4, 0);
    ok @entry[88].id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].id.xxhash == 3814087496;
    ok @entry[88].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[88].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[88].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[88].posting[0].amount.asset-code eqv "USD";
    ok @entry[88].posting[0].amount.asset-quantity == 2013.0516;
    ok @entry[88].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[88].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[88].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[88].posting[0].decinc eqv DecInc::INC;
    ok @entry[88].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[88].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 4, 0);
    ok @entry[88].posting[0].id.entry-id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[0].id.entry-id.xxhash == 3814087496;
    ok @entry[88].posting[0].id.number == 0;
    ok @entry[88].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD";
    ok @entry[88].posting[0].id.xxhash == 2233760271;
    ok @entry[88].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[88].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[88].posting[1].amount.asset-code eqv "USD";
    ok @entry[88].posting[1].amount.asset-quantity == 2013.0516;
    ok @entry[88].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[88].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[88].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[88].posting[1].decinc eqv DecInc::INC;
    ok @entry[88].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[88].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 4, 0);
    ok @entry[88].posting[1].id.entry-id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[1].id.entry-id.xxhash == 3814087496;
    ok @entry[88].posting[1].id.number == 1;
    ok @entry[88].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[1].id.xxhash == 1692039403;
    ok @entry[89].header.date eqv Date.new(2013,5,31);
    ok @entry[89].header.description eqv "Wigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[89].header.important == 0;
    ok @entry[89].id.number eq Array[UInt:D].new(2, 2, 5);
    ok @entry[89].id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].id.xxhash == 3166921593;
    ok @entry[89].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[89].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[89].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[89].posting[0].amount.asset-code eqv "USD";
    ok @entry[89].posting[0].amount.asset-quantity == 3.05;
    ok @entry[89].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[89].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[89].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[89].posting[0].decinc eqv DecInc::INC;
    ok @entry[89].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[89].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 5);
    ok @entry[89].posting[0].id.entry-id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[0].id.entry-id.xxhash == 3166921593;
    ok @entry[89].posting[0].id.number == 0;
    ok @entry[89].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.05 USD";
    ok @entry[89].posting[0].id.xxhash == 2097158499;
    ok @entry[89].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[89].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[89].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[89].posting[1].amount.asset-quantity == 1;
    ok @entry[89].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[89].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[89].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(61, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[89].posting[1].decinc eqv DecInc::DEC;
    ok @entry[89].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[89].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 5);
    ok @entry[89].posting[1].id.entry-id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[1].id.entry-id.xxhash == 3166921593;
    ok @entry[89].posting[1].id.number == 1;
    ok @entry[89].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[1].id.xxhash == 895907392;
    ok @entry[90].header.date eqv Date.new(2013,6,1);
    ok @entry[90].header.description eqv "Wigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[90].header.important == 0;
    ok @entry[90].id.number eq Array[UInt:D].new(2, 2, 6);
    ok @entry[90].id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].id.xxhash == 927323051;
    ok @entry[90].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[90].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[90].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[90].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[90].posting[0].amount.asset-quantity == 1;
    ok @entry[90].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[90].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[90].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(153, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[90].posting[0].decinc eqv DecInc::INC;
    ok @entry[90].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[90].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 6);
    ok @entry[90].posting[0].id.entry-id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[0].id.entry-id.xxhash == 927323051;
    ok @entry[90].posting[0].id.number == 0;
    ok @entry[90].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD";
    ok @entry[90].posting[0].id.xxhash == 3153491063;
    ok @entry[90].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[90].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[90].posting[1].amount.asset-code eqv "USD";
    ok @entry[90].posting[1].amount.asset-quantity == 3.06;
    ok @entry[90].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[90].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[90].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[90].posting[1].decinc eqv DecInc::DEC;
    ok @entry[90].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[90].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 6);
    ok @entry[90].posting[1].id.entry-id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[1].id.entry-id.xxhash == 927323051;
    ok @entry[90].posting[1].id.number == 1;
    ok @entry[90].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[1].id.xxhash == 1187163231;
    ok @entry[91].header.date eqv Date.new(2013,6,16);
    ok @entry[91].header.description eqv "Wigwam LLC received a dividend of \$2013.0616 USD";
    ok @entry[91].header.important == 0;
    ok @entry[91].id.number eq Array[UInt:D].new(2, 2, 7, 0);
    ok @entry[91].id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].id.xxhash == 2504499649;
    ok @entry[91].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[91].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[91].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[91].posting[0].amount.asset-code eqv "USD";
    ok @entry[91].posting[0].amount.asset-quantity == 2013.0616;
    ok @entry[91].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[91].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[91].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[91].posting[0].decinc eqv DecInc::INC;
    ok @entry[91].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[91].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 7, 0);
    ok @entry[91].posting[0].id.entry-id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[0].id.entry-id.xxhash == 2504499649;
    ok @entry[91].posting[0].id.number == 0;
    ok @entry[91].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD";
    ok @entry[91].posting[0].id.xxhash == 2945489261;
    ok @entry[91].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[91].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[91].posting[1].amount.asset-code eqv "USD";
    ok @entry[91].posting[1].amount.asset-quantity == 2013.0616;
    ok @entry[91].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[91].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[91].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[91].posting[1].decinc eqv DecInc::INC;
    ok @entry[91].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[91].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 7, 0);
    ok @entry[91].posting[1].id.entry-id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[1].id.entry-id.xxhash == 2504499649;
    ok @entry[91].posting[1].id.number == 1;
    ok @entry[91].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[1].id.xxhash == 398263546;
    ok @entry[92].header.date eqv Date.new(2013,6,30);
    ok @entry[92].header.description eqv "Wigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[92].header.important == 0;
    ok @entry[92].id.number eq Array[UInt:D].new(2, 2, 8);
    ok @entry[92].id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].id.xxhash == 1870422416;
    ok @entry[92].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[92].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[92].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[92].posting[0].amount.asset-code eqv "USD";
    ok @entry[92].posting[0].amount.asset-quantity == 3.06;
    ok @entry[92].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[92].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[92].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[92].posting[0].decinc eqv DecInc::INC;
    ok @entry[92].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[92].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 2, 8);
    ok @entry[92].posting[0].id.entry-id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[0].id.entry-id.xxhash == 1870422416;
    ok @entry[92].posting[0].id.number == 0;
    ok @entry[92].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.06 USD";
    ok @entry[92].posting[0].id.xxhash == 90479374;
    ok @entry[92].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[92].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[92].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[92].posting[1].amount.asset-quantity == 1;
    ok @entry[92].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[92].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[92].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(153, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[92].posting[1].decinc eqv DecInc::DEC;
    ok @entry[92].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[92].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 2, 8);
    ok @entry[92].posting[1].id.entry-id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[1].id.entry-id.xxhash == 1870422416;
    ok @entry[92].posting[1].id.number == 1;
    ok @entry[92].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[1].id.xxhash == 1167336265;
    ok @entry[93].header.date eqv Date.new(2013,7,1);
    ok @entry[93].header.description eqv "Wigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[93].header.important == 0;
    ok @entry[93].id.number eq Array[UInt:D].new(2, 3, 0);
    ok @entry[93].id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].id.xxhash == 4095708493;
    ok @entry[93].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[93].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[93].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[93].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[93].posting[0].amount.asset-quantity == 1;
    ok @entry[93].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[93].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[93].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(307, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[93].posting[0].decinc eqv DecInc::INC;
    ok @entry[93].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[93].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 0);
    ok @entry[93].posting[0].id.entry-id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[0].id.entry-id.xxhash == 4095708493;
    ok @entry[93].posting[0].id.number == 0;
    ok @entry[93].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD";
    ok @entry[93].posting[0].id.xxhash == 688453593;
    ok @entry[93].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[93].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[93].posting[1].amount.asset-code eqv "USD";
    ok @entry[93].posting[1].amount.asset-quantity == 3.07;
    ok @entry[93].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[93].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[93].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[93].posting[1].decinc eqv DecInc::DEC;
    ok @entry[93].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[93].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 0);
    ok @entry[93].posting[1].id.entry-id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[1].id.entry-id.xxhash == 4095708493;
    ok @entry[93].posting[1].id.number == 1;
    ok @entry[93].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[1].id.xxhash == 2675724246;
    ok @entry[94].header.date eqv Date.new(2013,7,16);
    ok @entry[94].header.description eqv "Wigwam LLC received a dividend of \$2013.0716 USD";
    ok @entry[94].header.important == 0;
    ok @entry[94].id.number eq Array[UInt:D].new(2, 3, 1, 0);
    ok @entry[94].id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].id.xxhash == 1271983877;
    ok @entry[94].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[94].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[94].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[94].posting[0].amount.asset-code eqv "USD";
    ok @entry[94].posting[0].amount.asset-quantity == 2013.0716;
    ok @entry[94].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[94].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[94].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[94].posting[0].decinc eqv DecInc::INC;
    ok @entry[94].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[94].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 1, 0);
    ok @entry[94].posting[0].id.entry-id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[0].id.entry-id.xxhash == 1271983877;
    ok @entry[94].posting[0].id.number == 0;
    ok @entry[94].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD";
    ok @entry[94].posting[0].id.xxhash == 463766995;
    ok @entry[94].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[94].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[94].posting[1].amount.asset-code eqv "USD";
    ok @entry[94].posting[1].amount.asset-quantity == 2013.0716;
    ok @entry[94].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[94].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[94].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[94].posting[1].decinc eqv DecInc::INC;
    ok @entry[94].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[94].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 1, 0);
    ok @entry[94].posting[1].id.entry-id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[1].id.entry-id.xxhash == 1271983877;
    ok @entry[94].posting[1].id.number == 1;
    ok @entry[94].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[1].id.xxhash == 1429453554;
    ok @entry[95].header.date eqv Date.new(2013,7,31);
    ok @entry[95].header.description eqv "Wigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[95].header.important == 0;
    ok @entry[95].id.number eq Array[UInt:D].new(2, 3, 2);
    ok @entry[95].id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].id.xxhash == 585905153;
    ok @entry[95].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[95].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[95].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[95].posting[0].amount.asset-code eqv "USD";
    ok @entry[95].posting[0].amount.asset-quantity == 3.07;
    ok @entry[95].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[95].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[95].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[95].posting[0].decinc eqv DecInc::INC;
    ok @entry[95].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[95].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 2);
    ok @entry[95].posting[0].id.entry-id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[0].id.entry-id.xxhash == 585905153;
    ok @entry[95].posting[0].id.number == 0;
    ok @entry[95].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.07 USD";
    ok @entry[95].posting[0].id.xxhash == 1932435839;
    ok @entry[95].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[95].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[95].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[95].posting[1].amount.asset-quantity == 1;
    ok @entry[95].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[95].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[95].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(307, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[95].posting[1].decinc eqv DecInc::DEC;
    ok @entry[95].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[95].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 2);
    ok @entry[95].posting[1].id.entry-id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[1].id.entry-id.xxhash == 585905153;
    ok @entry[95].posting[1].id.number == 1;
    ok @entry[95].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[1].id.xxhash == 3307372531;
    ok @entry[96].header.date eqv Date.new(2013,8,1);
    ok @entry[96].header.description eqv "Wigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[96].header.important == 0;
    ok @entry[96].id.number eq Array[UInt:D].new(2, 3, 3);
    ok @entry[96].id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].id.xxhash == 444744934;
    ok @entry[96].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[96].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[96].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[96].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[96].posting[0].amount.asset-quantity == 1;
    ok @entry[96].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[96].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[96].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(77, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[96].posting[0].decinc eqv DecInc::INC;
    ok @entry[96].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[96].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 3);
    ok @entry[96].posting[0].id.entry-id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[0].id.entry-id.xxhash == 444744934;
    ok @entry[96].posting[0].id.number == 0;
    ok @entry[96].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD";
    ok @entry[96].posting[0].id.xxhash == 1041307584;
    ok @entry[96].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[96].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[96].posting[1].amount.asset-code eqv "USD";
    ok @entry[96].posting[1].amount.asset-quantity == 3.08;
    ok @entry[96].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[96].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[96].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[96].posting[1].decinc eqv DecInc::DEC;
    ok @entry[96].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[96].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 3);
    ok @entry[96].posting[1].id.entry-id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[1].id.entry-id.xxhash == 444744934;
    ok @entry[96].posting[1].id.number == 1;
    ok @entry[96].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[1].id.xxhash == 3039180281;
    ok @entry[97].header.date eqv Date.new(2013,8,16);
    ok @entry[97].header.description eqv "Wigwam LLC received a dividend of \$2013.0816 USD";
    ok @entry[97].header.important == 0;
    ok @entry[97].id.number eq Array[UInt:D].new(2, 3, 4, 0);
    ok @entry[97].id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].id.xxhash == 796956488;
    ok @entry[97].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[97].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[97].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[97].posting[0].amount.asset-code eqv "USD";
    ok @entry[97].posting[0].amount.asset-quantity == 2013.0816;
    ok @entry[97].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[97].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[97].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[97].posting[0].decinc eqv DecInc::INC;
    ok @entry[97].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[97].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 4, 0);
    ok @entry[97].posting[0].id.entry-id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[0].id.entry-id.xxhash == 796956488;
    ok @entry[97].posting[0].id.number == 0;
    ok @entry[97].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD";
    ok @entry[97].posting[0].id.xxhash == 4281436752;
    ok @entry[97].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[97].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[97].posting[1].amount.asset-code eqv "USD";
    ok @entry[97].posting[1].amount.asset-quantity == 2013.0816;
    ok @entry[97].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[97].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[97].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[97].posting[1].decinc eqv DecInc::INC;
    ok @entry[97].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[97].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 4, 0);
    ok @entry[97].posting[1].id.entry-id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[1].id.entry-id.xxhash == 796956488;
    ok @entry[97].posting[1].id.number == 1;
    ok @entry[97].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[1].id.xxhash == 1969770397;
    ok @entry[98].header.date eqv Date.new(2013,8,31);
    ok @entry[98].header.description eqv "Wigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[98].header.important == 0;
    ok @entry[98].id.number eq Array[UInt:D].new(2, 3, 5);
    ok @entry[98].id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].id.xxhash == 1033193703;
    ok @entry[98].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[98].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[98].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[98].posting[0].amount.asset-code eqv "USD";
    ok @entry[98].posting[0].amount.asset-quantity == 3.08;
    ok @entry[98].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[98].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[98].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[98].posting[0].decinc eqv DecInc::INC;
    ok @entry[98].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[98].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 5);
    ok @entry[98].posting[0].id.entry-id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[0].id.entry-id.xxhash == 1033193703;
    ok @entry[98].posting[0].id.number == 0;
    ok @entry[98].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.08 USD";
    ok @entry[98].posting[0].id.xxhash == 259022758;
    ok @entry[98].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[98].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[98].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[98].posting[1].amount.asset-quantity == 1;
    ok @entry[98].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[98].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[98].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(77, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[98].posting[1].decinc eqv DecInc::DEC;
    ok @entry[98].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[98].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 5);
    ok @entry[98].posting[1].id.entry-id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[1].id.entry-id.xxhash == 1033193703;
    ok @entry[98].posting[1].id.number == 1;
    ok @entry[98].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[1].id.xxhash == 115568189;
    ok @entry[99].header.date eqv Date.new(2013,9,1);
    ok @entry[99].header.description eqv "Wigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[99].header.important == 0;
    ok @entry[99].id.number eq Array[UInt:D].new(2, 3, 6);
    ok @entry[99].id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].id.xxhash == 2081861311;
    ok @entry[99].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[99].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[99].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[99].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[99].posting[0].amount.asset-quantity == 1;
    ok @entry[99].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[99].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[99].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(309, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[99].posting[0].decinc eqv DecInc::INC;
    ok @entry[99].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[99].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 6);
    ok @entry[99].posting[0].id.entry-id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[0].id.entry-id.xxhash == 2081861311;
    ok @entry[99].posting[0].id.number == 0;
    ok @entry[99].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD";
    ok @entry[99].posting[0].id.xxhash == 2102045153;
    ok @entry[99].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[99].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[99].posting[1].amount.asset-code eqv "USD";
    ok @entry[99].posting[1].amount.asset-quantity == 3.09;
    ok @entry[99].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[99].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[99].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[99].posting[1].decinc eqv DecInc::DEC;
    ok @entry[99].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[99].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 6);
    ok @entry[99].posting[1].id.entry-id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[1].id.entry-id.xxhash == 2081861311;
    ok @entry[99].posting[1].id.number == 1;
    ok @entry[99].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[1].id.xxhash == 3596800179;
    ok @entry[100].header.date eqv Date.new(2013,9,16);
    ok @entry[100].header.description eqv "Wigwam LLC received a dividend of \$2013.0916 USD";
    ok @entry[100].header.important == 0;
    ok @entry[100].id.number eq Array[UInt:D].new(2, 3, 7, 0);
    ok @entry[100].id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].id.xxhash == 705777824;
    ok @entry[100].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[100].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[100].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[100].posting[0].amount.asset-code eqv "USD";
    ok @entry[100].posting[0].amount.asset-quantity == 2013.0916;
    ok @entry[100].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[100].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[100].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[100].posting[0].decinc eqv DecInc::INC;
    ok @entry[100].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[100].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 7, 0);
    ok @entry[100].posting[0].id.entry-id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[0].id.entry-id.xxhash == 705777824;
    ok @entry[100].posting[0].id.number == 0;
    ok @entry[100].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD";
    ok @entry[100].posting[0].id.xxhash == 2426431672;
    ok @entry[100].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[100].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[100].posting[1].amount.asset-code eqv "USD";
    ok @entry[100].posting[1].amount.asset-quantity == 2013.0916;
    ok @entry[100].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[100].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[100].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[100].posting[1].decinc eqv DecInc::INC;
    ok @entry[100].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[100].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 7, 0);
    ok @entry[100].posting[1].id.entry-id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[1].id.entry-id.xxhash == 705777824;
    ok @entry[100].posting[1].id.number == 1;
    ok @entry[100].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[1].id.xxhash == 742012739;
    ok @entry[101].header.date eqv Date.new(2013,9,30);
    ok @entry[101].header.description eqv "Wigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[101].header.important == 0;
    ok @entry[101].id.number eq Array[UInt:D].new(2, 3, 8);
    ok @entry[101].id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].id.xxhash == 2792675818;
    ok @entry[101].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[101].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[101].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[101].posting[0].amount.asset-code eqv "USD";
    ok @entry[101].posting[0].amount.asset-quantity == 3.09;
    ok @entry[101].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[101].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[101].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[101].posting[0].decinc eqv DecInc::INC;
    ok @entry[101].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[101].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 3, 8);
    ok @entry[101].posting[0].id.entry-id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[0].id.entry-id.xxhash == 2792675818;
    ok @entry[101].posting[0].id.number == 0;
    ok @entry[101].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.09 USD";
    ok @entry[101].posting[0].id.xxhash == 3876439597;
    ok @entry[101].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[101].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[101].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[101].posting[1].amount.asset-quantity == 1;
    ok @entry[101].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[101].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[101].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(309, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[101].posting[1].decinc eqv DecInc::DEC;
    ok @entry[101].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[101].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 3, 8);
    ok @entry[101].posting[1].id.entry-id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[1].id.entry-id.xxhash == 2792675818;
    ok @entry[101].posting[1].id.number == 1;
    ok @entry[101].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[1].id.xxhash == 4150514617;
    ok @entry[102].header.date eqv Date.new(2013,10,1);
    ok @entry[102].header.description eqv "Wigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[102].header.important == 0;
    ok @entry[102].id.number eq Array[UInt:D].new(2, 4, 0);
    ok @entry[102].id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].id.xxhash == 574545503;
    ok @entry[102].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[102].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[102].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[102].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[102].posting[0].amount.asset-quantity == 1;
    ok @entry[102].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[102].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[102].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(31, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[102].posting[0].decinc eqv DecInc::INC;
    ok @entry[102].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[102].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 0);
    ok @entry[102].posting[0].id.entry-id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[0].id.entry-id.xxhash == 574545503;
    ok @entry[102].posting[0].id.number == 0;
    ok @entry[102].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD";
    ok @entry[102].posting[0].id.xxhash == 264778505;
    ok @entry[102].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[102].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[102].posting[1].amount.asset-code eqv "USD";
    ok @entry[102].posting[1].amount.asset-quantity == 3.1;
    ok @entry[102].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[102].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[102].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[102].posting[1].decinc eqv DecInc::DEC;
    ok @entry[102].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[102].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 0);
    ok @entry[102].posting[1].id.entry-id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[1].id.entry-id.xxhash == 574545503;
    ok @entry[102].posting[1].id.number == 1;
    ok @entry[102].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[1].id.xxhash == 806384794;
    ok @entry[103].header.date eqv Date.new(2013,10,16);
    ok @entry[103].header.description eqv "Wigwam LLC received a dividend of \$2013.1016 USD";
    ok @entry[103].header.important == 0;
    ok @entry[103].id.number eq Array[UInt:D].new(2, 4, 1, 0);
    ok @entry[103].id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].id.xxhash == 1123984249;
    ok @entry[103].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[103].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[103].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[103].posting[0].amount.asset-code eqv "USD";
    ok @entry[103].posting[0].amount.asset-quantity == 2013.1016;
    ok @entry[103].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[103].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[103].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[103].posting[0].decinc eqv DecInc::INC;
    ok @entry[103].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[103].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 1, 0);
    ok @entry[103].posting[0].id.entry-id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[0].id.entry-id.xxhash == 1123984249;
    ok @entry[103].posting[0].id.number == 0;
    ok @entry[103].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD";
    ok @entry[103].posting[0].id.xxhash == 1130097971;
    ok @entry[103].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[103].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[103].posting[1].amount.asset-code eqv "USD";
    ok @entry[103].posting[1].amount.asset-quantity == 2013.1016;
    ok @entry[103].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[103].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[103].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[103].posting[1].decinc eqv DecInc::INC;
    ok @entry[103].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[103].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 1, 0);
    ok @entry[103].posting[1].id.entry-id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[1].id.entry-id.xxhash == 1123984249;
    ok @entry[103].posting[1].id.number == 1;
    ok @entry[103].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[1].id.xxhash == 719072295;
    ok @entry[104].header.date eqv Date.new(2013,10,31);
    ok @entry[104].header.description eqv "Wigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[104].header.important == 0;
    ok @entry[104].id.number eq Array[UInt:D].new(2, 4, 2);
    ok @entry[104].id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].id.xxhash == 1605620712;
    ok @entry[104].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[104].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[104].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[104].posting[0].amount.asset-code eqv "USD";
    ok @entry[104].posting[0].amount.asset-quantity == 3.1;
    ok @entry[104].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[104].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[104].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[104].posting[0].decinc eqv DecInc::INC;
    ok @entry[104].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[104].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 2);
    ok @entry[104].posting[0].id.entry-id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[0].id.entry-id.xxhash == 1605620712;
    ok @entry[104].posting[0].id.number == 0;
    ok @entry[104].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.10 USD";
    ok @entry[104].posting[0].id.xxhash == 4175703663;
    ok @entry[104].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[104].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[104].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[104].posting[1].amount.asset-quantity == 1;
    ok @entry[104].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[104].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[104].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(31, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[104].posting[1].decinc eqv DecInc::DEC;
    ok @entry[104].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[104].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 2);
    ok @entry[104].posting[1].id.entry-id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[1].id.entry-id.xxhash == 1605620712;
    ok @entry[104].posting[1].id.number == 1;
    ok @entry[104].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[1].id.xxhash == 1943956588;
    ok @entry[105].header.date eqv Date.new(2013,11,1);
    ok @entry[105].header.description eqv "Wigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[105].header.important == 0;
    ok @entry[105].id.number eq Array[UInt:D].new(2, 4, 3);
    ok @entry[105].id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].id.xxhash == 3963180771;
    ok @entry[105].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[105].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[105].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[105].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[105].posting[0].amount.asset-quantity == 1;
    ok @entry[105].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[105].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[105].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(311, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[105].posting[0].decinc eqv DecInc::INC;
    ok @entry[105].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[105].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 3);
    ok @entry[105].posting[0].id.entry-id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[0].id.entry-id.xxhash == 3963180771;
    ok @entry[105].posting[0].id.number == 0;
    ok @entry[105].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD";
    ok @entry[105].posting[0].id.xxhash == 3475412129;
    ok @entry[105].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[105].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[105].posting[1].amount.asset-code eqv "USD";
    ok @entry[105].posting[1].amount.asset-quantity == 3.11;
    ok @entry[105].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[105].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[105].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[105].posting[1].decinc eqv DecInc::DEC;
    ok @entry[105].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[105].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 3);
    ok @entry[105].posting[1].id.entry-id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[1].id.entry-id.xxhash == 3963180771;
    ok @entry[105].posting[1].id.number == 1;
    ok @entry[105].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[1].id.xxhash == 1726070033;
    ok @entry[106].header.date eqv Date.new(2013,11,16);
    ok @entry[106].header.description eqv "Wigwam LLC received a dividend of \$2013.1116 USD";
    ok @entry[106].header.important == 0;
    ok @entry[106].id.number eq Array[UInt:D].new(2, 4, 4, 0);
    ok @entry[106].id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].id.xxhash == 3185930480;
    ok @entry[106].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[106].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[106].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[106].posting[0].amount.asset-code eqv "USD";
    ok @entry[106].posting[0].amount.asset-quantity == 2013.1116;
    ok @entry[106].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[106].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[106].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[106].posting[0].decinc eqv DecInc::INC;
    ok @entry[106].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[106].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 4, 0);
    ok @entry[106].posting[0].id.entry-id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[0].id.entry-id.xxhash == 3185930480;
    ok @entry[106].posting[0].id.number == 0;
    ok @entry[106].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD";
    ok @entry[106].posting[0].id.xxhash == 2304408211;
    ok @entry[106].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[106].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[106].posting[1].amount.asset-code eqv "USD";
    ok @entry[106].posting[1].amount.asset-quantity == 2013.1116;
    ok @entry[106].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[106].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[106].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[106].posting[1].decinc eqv DecInc::INC;
    ok @entry[106].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[106].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 4, 0);
    ok @entry[106].posting[1].id.entry-id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[1].id.entry-id.xxhash == 3185930480;
    ok @entry[106].posting[1].id.number == 1;
    ok @entry[106].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[1].id.xxhash == 690089900;
    ok @entry[107].header.date eqv Date.new(2013,11,30);
    ok @entry[107].header.description eqv "Wigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[107].header.important == 0;
    ok @entry[107].id.number eq Array[UInt:D].new(2, 4, 5);
    ok @entry[107].id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].id.xxhash == 2315332478;
    ok @entry[107].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[107].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[107].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[107].posting[0].amount.asset-code eqv "USD";
    ok @entry[107].posting[0].amount.asset-quantity == 3.11;
    ok @entry[107].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[107].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[107].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[107].posting[0].decinc eqv DecInc::INC;
    ok @entry[107].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[107].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 5);
    ok @entry[107].posting[0].id.entry-id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[0].id.entry-id.xxhash == 2315332478;
    ok @entry[107].posting[0].id.number == 0;
    ok @entry[107].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.11 USD";
    ok @entry[107].posting[0].id.xxhash == 4002911623;
    ok @entry[107].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[107].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[107].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[107].posting[1].amount.asset-quantity == 1;
    ok @entry[107].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[107].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[107].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(311, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[107].posting[1].decinc eqv DecInc::DEC;
    ok @entry[107].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[107].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 5);
    ok @entry[107].posting[1].id.entry-id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[1].id.entry-id.xxhash == 2315332478;
    ok @entry[107].posting[1].id.number == 1;
    ok @entry[107].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[1].id.xxhash == 2318076742;
    ok @entry[108].header.date eqv Date.new(2013,12,1);
    ok @entry[108].header.description eqv "Wigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[108].header.important == 0;
    ok @entry[108].id.number eq Array[UInt:D].new(2, 4, 6);
    ok @entry[108].id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].id.xxhash == 646922698;
    ok @entry[108].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[108].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[108].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[108].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[108].posting[0].amount.asset-quantity == 1;
    ok @entry[108].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[108].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[108].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(78, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[108].posting[0].decinc eqv DecInc::INC;
    ok @entry[108].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[108].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 6);
    ok @entry[108].posting[0].id.entry-id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[0].id.entry-id.xxhash == 646922698;
    ok @entry[108].posting[0].id.number == 0;
    ok @entry[108].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD";
    ok @entry[108].posting[0].id.xxhash == 2535125365;
    ok @entry[108].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[108].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[108].posting[1].amount.asset-code eqv "USD";
    ok @entry[108].posting[1].amount.asset-quantity == 3.12;
    ok @entry[108].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[108].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[108].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[108].posting[1].decinc eqv DecInc::DEC;
    ok @entry[108].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[108].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 6);
    ok @entry[108].posting[1].id.entry-id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[1].id.entry-id.xxhash == 646922698;
    ok @entry[108].posting[1].id.number == 1;
    ok @entry[108].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[1].id.xxhash == 3978925433;
    ok @entry[109].header.date eqv Date.new(2013,12,16);
    ok @entry[109].header.description eqv "Wigwam LLC received a dividend of \$2013.1216 USD";
    ok @entry[109].header.important == 0;
    ok @entry[109].id.number eq Array[UInt:D].new(2, 4, 7, 0);
    ok @entry[109].id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].id.xxhash == 2042360109;
    ok @entry[109].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[109].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[109].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[109].posting[0].amount.asset-code eqv "USD";
    ok @entry[109].posting[0].amount.asset-quantity == 2013.1216;
    ok @entry[109].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[109].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[109].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[109].posting[0].decinc eqv DecInc::INC;
    ok @entry[109].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[109].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 7, 0);
    ok @entry[109].posting[0].id.entry-id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[0].id.entry-id.xxhash == 2042360109;
    ok @entry[109].posting[0].id.number == 0;
    ok @entry[109].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD";
    ok @entry[109].posting[0].id.xxhash == 578573857;
    ok @entry[109].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[109].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[109].posting[1].amount.asset-code eqv "USD";
    ok @entry[109].posting[1].amount.asset-quantity == 2013.1216;
    ok @entry[109].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[109].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[109].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[109].posting[1].decinc eqv DecInc::INC;
    ok @entry[109].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[109].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 7, 0);
    ok @entry[109].posting[1].id.entry-id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[1].id.entry-id.xxhash == 2042360109;
    ok @entry[109].posting[1].id.number == 1;
    ok @entry[109].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[1].id.xxhash == 2455119990;
    ok @entry[110].header.date eqv Date.new(2013,12,31);
    ok @entry[110].header.description eqv "Wigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[110].header.important == 0;
    ok @entry[110].id.number eq Array[UInt:D].new(2, 4, 8);
    ok @entry[110].id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].id.xxhash == 1933516675;
    ok @entry[110].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[110].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[110].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[110].posting[0].amount.asset-code eqv "USD";
    ok @entry[110].posting[0].amount.asset-quantity == 3.12;
    ok @entry[110].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[110].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[110].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[110].posting[0].decinc eqv DecInc::INC;
    ok @entry[110].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[110].posting[0].id.entry-id.number eq Array[UInt:D].new(2, 4, 8);
    ok @entry[110].posting[0].id.entry-id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[0].id.entry-id.xxhash == 1933516675;
    ok @entry[110].posting[0].id.number == 0;
    ok @entry[110].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.12 USD";
    ok @entry[110].posting[0].id.xxhash == 3769395453;
    ok @entry[110].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[110].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[110].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[110].posting[1].amount.asset-quantity == 1;
    ok @entry[110].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[110].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[110].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(78, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[110].posting[1].decinc eqv DecInc::DEC;
    ok @entry[110].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[110].posting[1].id.entry-id.number eq Array[UInt:D].new(2, 4, 8);
    ok @entry[110].posting[1].id.entry-id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[1].id.entry-id.xxhash == 1933516675;
    ok @entry[110].posting[1].id.number == 1;
    ok @entry[110].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[1].id.xxhash == 3832507616;
    ok @entry[111].header.date eqv Date.new(2014,1,1);
    ok @entry[111].header.description eqv "I started the year with \$1000 in Bankwest cheque account";
    ok @entry[111].header.important == 0;
    ok @entry[111].id.number eq Array[UInt:D].new(3);
    ok @entry[111].id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].id.xxhash == 1468523538;
    ok @entry[111].posting[0].account.entity eqv "Personal";
    ok @entry[111].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[111].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[111].posting[0].amount.asset-code eqv "USD";
    ok @entry[111].posting[0].amount.asset-quantity == 1000;
    ok @entry[111].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[111].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[111].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[111].posting[0].decinc eqv DecInc::INC;
    ok @entry[111].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[111].posting[0].id.entry-id.number eq Array[UInt:D].new(3);
    ok @entry[111].posting[0].id.entry-id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[0].id.entry-id.xxhash == 1468523538;
    ok @entry[111].posting[0].id.number == 0;
    ok @entry[111].posting[0].id.text eqv "Assets:Personal:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[111].posting[0].id.xxhash == 4134277096;
    ok @entry[111].posting[1].account.entity eqv "Personal";
    ok @entry[111].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[111].posting[1].amount.asset-code eqv "USD";
    ok @entry[111].posting[1].amount.asset-quantity == 1000;
    ok @entry[111].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[111].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[111].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[111].posting[1].decinc eqv DecInc::INC;
    ok @entry[111].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[111].posting[1].id.entry-id.number eq Array[UInt:D].new(3);
    ok @entry[111].posting[1].id.entry-id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[1].id.entry-id.xxhash == 1468523538;
    ok @entry[111].posting[1].id.number == 1;
    ok @entry[111].posting[1].id.text eqv "Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[1].id.xxhash == 344831063;
    # end tests }}}
}

# check that we repeat results of above test when including from txn-dir
subtest
{
    my Str $txn = "include <with-includes>\n";
    my TXN::Parser::AST::Entry @entry =
        TXN::Parser.parse($txn, :txn-dir<t/data/with-includes>).made;

    # tests {{{
    ok @entry[0].header.date eqv Date.new(2011,1,1);
    ok @entry[0].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[0].header.important == 0;
    ok @entry[0].id.number eq Array[UInt:D].new(0, 0, 0);
    ok @entry[0].id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].id.xxhash == 839297870;
    ok @entry[0].posting[0].account.entity eqv "FooCorp";
    ok @entry[0].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[0].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[0].posting[0].amount.asset-code eqv "USD";
    ok @entry[0].posting[0].amount.asset-quantity == 1000;
    ok @entry[0].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[0].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[0].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[0].posting[0].decinc eqv DecInc::INC;
    ok @entry[0].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[0].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 0);
    ok @entry[0].posting[0].id.entry-id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[0].id.entry-id.xxhash == 839297870;
    ok @entry[0].posting[0].id.number == 0;
    ok @entry[0].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[0].posting[0].id.xxhash == 3244003616;
    ok @entry[0].posting[1].account.entity eqv "FooCorp";
    ok @entry[0].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[0].posting[1].amount.asset-code eqv "USD";
    ok @entry[0].posting[1].amount.asset-quantity == 1000;
    ok @entry[0].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[0].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[0].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[0].posting[1].decinc eqv DecInc::INC;
    ok @entry[0].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[0].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 0);
    ok @entry[0].posting[1].id.entry-id.text eqv "2011-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[1].id.entry-id.xxhash == 839297870;
    ok @entry[0].posting[1].id.number == 1;
    ok @entry[0].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[0].posting[1].id.xxhash == 1025058054;
    ok @entry[1].header.date eqv Date.new(2011,1,1);
    ok @entry[1].header.description eqv "Wigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[1].header.important == 0;
    ok @entry[1].id.number eq Array[UInt:D].new(0, 0, 1, 0);
    ok @entry[1].id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].id.xxhash == 3127303444;
    ok @entry[1].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[1].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[1].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[1].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[1].posting[0].amount.asset-quantity == 1;
    ok @entry[1].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[1].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[1].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[1].posting[0].decinc eqv DecInc::INC;
    ok @entry[1].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[1].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 0);
    ok @entry[1].posting[0].id.entry-id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[0].id.entry-id.xxhash == 3127303444;
    ok @entry[1].posting[0].id.number == 0;
    ok @entry[1].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD";
    ok @entry[1].posting[0].id.xxhash == 3785012149;
    ok @entry[1].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[1].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[1].posting[1].amount.asset-code eqv "USD";
    ok @entry[1].posting[1].amount.asset-quantity == 1.01;
    ok @entry[1].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[1].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[1].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[1].posting[1].decinc eqv DecInc::DEC;
    ok @entry[1].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[1].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 0);
    ok @entry[1].posting[1].id.entry-id.text eqv "2011-01-01 '''\nWigwam LLC bought one wigwam for \$1.01 USD in early January 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[1].id.entry-id.xxhash == 3127303444;
    ok @entry[1].posting[1].id.number == 1;
    ok @entry[1].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.01 USD";
    ok @entry[1].posting[1].id.xxhash == 1442732716;
    ok @entry[2].header.date eqv Date.new(2011,1,16);
    ok @entry[2].header.description eqv "Wigwam LLC received a dividend of \$2011.0116 USD";
    ok @entry[2].header.important == 0;
    ok @entry[2].id.number eq Array[UInt:D].new(0, 0, 1, 1, 0);
    ok @entry[2].id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].id.xxhash == 4018141641;
    ok @entry[2].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[2].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[2].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[2].posting[0].amount.asset-code eqv "USD";
    ok @entry[2].posting[0].amount.asset-quantity == 2011.0116;
    ok @entry[2].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[2].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[2].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[2].posting[0].decinc eqv DecInc::INC;
    ok @entry[2].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[2].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 1, 0);
    ok @entry[2].posting[0].id.entry-id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[0].id.entry-id.xxhash == 4018141641;
    ok @entry[2].posting[0].id.number == 0;
    ok @entry[2].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD";
    ok @entry[2].posting[0].id.xxhash == 228891826;
    ok @entry[2].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[2].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[2].posting[1].amount.asset-code eqv "USD";
    ok @entry[2].posting[1].amount.asset-quantity == 2011.0116;
    ok @entry[2].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[2].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[2].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[2].posting[1].decinc eqv DecInc::INC;
    ok @entry[2].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[2].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 1, 0);
    ok @entry[2].posting[1].id.entry-id.text eqv "2011-01-16 'Wigwam LLC received a dividend of \$2011.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0116 USD\n  Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[1].id.entry-id.xxhash == 4018141641;
    ok @entry[2].posting[1].id.number == 1;
    ok @entry[2].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0116 USD";
    ok @entry[2].posting[1].id.xxhash == 3822606469;
    ok @entry[3].header.date eqv Date.new(2011,1,31);
    ok @entry[3].header.description eqv "Wigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[3].header.important == 0;
    ok @entry[3].id.number eq Array[UInt:D].new(0, 0, 1, 2);
    ok @entry[3].id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].id.xxhash == 2478358144;
    ok @entry[3].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[3].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[3].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[3].posting[0].amount.asset-code eqv "USD";
    ok @entry[3].posting[0].amount.asset-quantity == 1.01;
    ok @entry[3].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[3].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[3].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[3].posting[0].decinc eqv DecInc::INC;
    ok @entry[3].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[3].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 2);
    ok @entry[3].posting[0].id.entry-id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[0].id.entry-id.xxhash == 2478358144;
    ok @entry[3].posting[0].id.number == 0;
    ok @entry[3].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.01 USD";
    ok @entry[3].posting[0].id.xxhash == 1462893106;
    ok @entry[3].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[3].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[3].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[3].posting[1].amount.asset-quantity == 1;
    ok @entry[3].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[3].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[3].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[3].posting[1].decinc eqv DecInc::DEC;
    ok @entry[3].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[3].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 2);
    ok @entry[3].posting[1].id.entry-id.text eqv "2011-01-31 '''\nWigwam LLC sold one wigwam for \$1.01 USD in late January 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[1].id.entry-id.xxhash == 2478358144;
    ok @entry[3].posting[1].id.number == 1;
    ok @entry[3].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.01 USD";
    ok @entry[3].posting[1].id.xxhash == 3589948839;
    ok @entry[4].header.date eqv Date.new(2011,2,1);
    ok @entry[4].header.description eqv "Wigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[4].header.important == 0;
    ok @entry[4].id.number eq Array[UInt:D].new(0, 0, 1, 3);
    ok @entry[4].id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].id.xxhash == 2320468031;
    ok @entry[4].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[4].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[4].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[4].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[4].posting[0].amount.asset-quantity == 1;
    ok @entry[4].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[4].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[4].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[4].posting[0].decinc eqv DecInc::INC;
    ok @entry[4].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[4].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 3);
    ok @entry[4].posting[0].id.entry-id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[0].id.entry-id.xxhash == 2320468031;
    ok @entry[4].posting[0].id.number == 0;
    ok @entry[4].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD";
    ok @entry[4].posting[0].id.xxhash == 826351545;
    ok @entry[4].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[4].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[4].posting[1].amount.asset-code eqv "USD";
    ok @entry[4].posting[1].amount.asset-quantity == 1.02;
    ok @entry[4].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[4].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[4].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[4].posting[1].decinc eqv DecInc::DEC;
    ok @entry[4].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[4].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 3);
    ok @entry[4].posting[1].id.entry-id.text eqv "2011-02-01 '''\nWigwam LLC bought one wigwam for \$1.02 USD in early February 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[1].id.entry-id.xxhash == 2320468031;
    ok @entry[4].posting[1].id.number == 1;
    ok @entry[4].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.02 USD";
    ok @entry[4].posting[1].id.xxhash == 2965421556;
    ok @entry[5].header.date eqv Date.new(2011,2,16);
    ok @entry[5].header.description eqv "Wigwam LLC received a dividend of \$2011.0216 USD";
    ok @entry[5].header.important == 0;
    ok @entry[5].id.number eq Array[UInt:D].new(0, 0, 1, 4, 0);
    ok @entry[5].id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].id.xxhash == 1177608739;
    ok @entry[5].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[5].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[5].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[5].posting[0].amount.asset-code eqv "USD";
    ok @entry[5].posting[0].amount.asset-quantity == 2011.0216;
    ok @entry[5].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[5].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[5].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[5].posting[0].decinc eqv DecInc::INC;
    ok @entry[5].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[5].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 4, 0);
    ok @entry[5].posting[0].id.entry-id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[0].id.entry-id.xxhash == 1177608739;
    ok @entry[5].posting[0].id.number == 0;
    ok @entry[5].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD";
    ok @entry[5].posting[0].id.xxhash == 2162966727;
    ok @entry[5].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[5].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[5].posting[1].amount.asset-code eqv "USD";
    ok @entry[5].posting[1].amount.asset-quantity == 2011.0216;
    ok @entry[5].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[5].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[5].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[5].posting[1].decinc eqv DecInc::INC;
    ok @entry[5].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[5].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 4, 0);
    ok @entry[5].posting[1].id.entry-id.text eqv "2011-02-16 'Wigwam LLC received a dividend of \$2011.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0216 USD\n  Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[1].id.entry-id.xxhash == 1177608739;
    ok @entry[5].posting[1].id.number == 1;
    ok @entry[5].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0216 USD";
    ok @entry[5].posting[1].id.xxhash == 1447769892;
    ok @entry[6].header.date eqv Date.new(2011,2,28);
    ok @entry[6].header.description eqv "Wigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[6].header.important == 0;
    ok @entry[6].id.number eq Array[UInt:D].new(0, 0, 1, 5);
    ok @entry[6].id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].id.xxhash == 3178583936;
    ok @entry[6].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[6].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[6].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[6].posting[0].amount.asset-code eqv "USD";
    ok @entry[6].posting[0].amount.asset-quantity == 1.02;
    ok @entry[6].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[6].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[6].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[6].posting[0].decinc eqv DecInc::INC;
    ok @entry[6].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[6].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 5);
    ok @entry[6].posting[0].id.entry-id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[0].id.entry-id.xxhash == 3178583936;
    ok @entry[6].posting[0].id.number == 0;
    ok @entry[6].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.02 USD";
    ok @entry[6].posting[0].id.xxhash == 1509993572;
    ok @entry[6].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[6].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[6].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[6].posting[1].amount.asset-quantity == 1;
    ok @entry[6].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[6].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[6].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[6].posting[1].decinc eqv DecInc::DEC;
    ok @entry[6].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[6].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 5);
    ok @entry[6].posting[1].id.entry-id.text eqv "2011-02-28 '''\nWigwam LLC sold one wigwam for \$1.02 USD in late February 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[1].id.entry-id.xxhash == 3178583936;
    ok @entry[6].posting[1].id.number == 1;
    ok @entry[6].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.02 USD";
    ok @entry[6].posting[1].id.xxhash == 991307650;
    ok @entry[7].header.date eqv Date.new(2011,3,1);
    ok @entry[7].header.description eqv "Wigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[7].header.important == 0;
    ok @entry[7].id.number eq Array[UInt:D].new(0, 0, 1, 6);
    ok @entry[7].id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].id.xxhash == 1567632724;
    ok @entry[7].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[7].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[7].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[7].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[7].posting[0].amount.asset-quantity == 1;
    ok @entry[7].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[7].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[7].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[7].posting[0].decinc eqv DecInc::INC;
    ok @entry[7].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[7].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 6);
    ok @entry[7].posting[0].id.entry-id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[0].id.entry-id.xxhash == 1567632724;
    ok @entry[7].posting[0].id.number == 0;
    ok @entry[7].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD";
    ok @entry[7].posting[0].id.xxhash == 2399027560;
    ok @entry[7].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[7].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[7].posting[1].amount.asset-code eqv "USD";
    ok @entry[7].posting[1].amount.asset-quantity == 1.03;
    ok @entry[7].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[7].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[7].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[7].posting[1].decinc eqv DecInc::DEC;
    ok @entry[7].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[7].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 6);
    ok @entry[7].posting[1].id.entry-id.text eqv "2011-03-01 '''\nWigwam LLC bought one wigwam for \$1.03 USD in early March 2011\nFILE:  「includes/2011/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[1].id.entry-id.xxhash == 1567632724;
    ok @entry[7].posting[1].id.number == 1;
    ok @entry[7].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.03 USD";
    ok @entry[7].posting[1].id.xxhash == 1721285682;
    ok @entry[8].header.date eqv Date.new(2011,3,16);
    ok @entry[8].header.description eqv "Wigwam LLC received a dividend of \$2011.0316 USD";
    ok @entry[8].header.important == 0;
    ok @entry[8].id.number eq Array[UInt:D].new(0, 0, 1, 7, 0);
    ok @entry[8].id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].id.xxhash == 3569777874;
    ok @entry[8].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[8].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[8].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[8].posting[0].amount.asset-code eqv "USD";
    ok @entry[8].posting[0].amount.asset-quantity == 2011.0316;
    ok @entry[8].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[8].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[8].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[8].posting[0].decinc eqv DecInc::INC;
    ok @entry[8].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[8].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 7, 0);
    ok @entry[8].posting[0].id.entry-id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[0].id.entry-id.xxhash == 3569777874;
    ok @entry[8].posting[0].id.number == 0;
    ok @entry[8].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD";
    ok @entry[8].posting[0].id.xxhash == 3064719331;
    ok @entry[8].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[8].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[8].posting[1].amount.asset-code eqv "USD";
    ok @entry[8].posting[1].amount.asset-quantity == 2011.0316;
    ok @entry[8].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[8].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[8].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[8].posting[1].decinc eqv DecInc::INC;
    ok @entry[8].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[8].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 7, 0);
    ok @entry[8].posting[1].id.entry-id.text eqv "2011-03-16 'Wigwam LLC received a dividend of \$2011.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0316 USD\n  Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[1].id.entry-id.xxhash == 3569777874;
    ok @entry[8].posting[1].id.number == 1;
    ok @entry[8].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0316 USD";
    ok @entry[8].posting[1].id.xxhash == 1632975976;
    ok @entry[9].header.date eqv Date.new(2011,3,31);
    ok @entry[9].header.description eqv "Wigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[9].header.important == 0;
    ok @entry[9].id.number eq Array[UInt:D].new(0, 0, 1, 8);
    ok @entry[9].id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].id.xxhash == 521313438;
    ok @entry[9].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[9].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[9].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[9].posting[0].amount.asset-code eqv "USD";
    ok @entry[9].posting[0].amount.asset-quantity == 1.03;
    ok @entry[9].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[9].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[9].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[9].posting[0].decinc eqv DecInc::INC;
    ok @entry[9].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[9].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 8);
    ok @entry[9].posting[0].id.entry-id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[0].id.entry-id.xxhash == 521313438;
    ok @entry[9].posting[0].id.number == 0;
    ok @entry[9].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.03 USD";
    ok @entry[9].posting[0].id.xxhash == 3019123955;
    ok @entry[9].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[9].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[9].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[9].posting[1].amount.asset-quantity == 1;
    ok @entry[9].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[9].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[9].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[9].posting[1].decinc eqv DecInc::DEC;
    ok @entry[9].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[9].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 1, 8);
    ok @entry[9].posting[1].id.entry-id.text eqv "2011-03-31 '''\nWigwam LLC sold one wigwam for \$1.03 USD in late March 2011\nFILE:  「includes/2011/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[1].id.entry-id.xxhash == 521313438;
    ok @entry[9].posting[1].id.number == 1;
    ok @entry[9].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.03 USD";
    ok @entry[9].posting[1].id.xxhash == 2676696051;
    ok @entry[10].header.date eqv Date.new(2011,4,1);
    ok @entry[10].header.description eqv "Wigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[10].header.important == 0;
    ok @entry[10].id.number eq Array[UInt:D].new(0, 0, 2, 0);
    ok @entry[10].id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].id.xxhash == 234894800;
    ok @entry[10].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[10].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[10].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[10].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[10].posting[0].amount.asset-quantity == 1;
    ok @entry[10].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[10].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[10].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(26, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[10].posting[0].decinc eqv DecInc::INC;
    ok @entry[10].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[10].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 0);
    ok @entry[10].posting[0].id.entry-id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[0].id.entry-id.xxhash == 234894800;
    ok @entry[10].posting[0].id.number == 0;
    ok @entry[10].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD";
    ok @entry[10].posting[0].id.xxhash == 2392165153;
    ok @entry[10].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[10].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[10].posting[1].amount.asset-code eqv "USD";
    ok @entry[10].posting[1].amount.asset-quantity == 1.04;
    ok @entry[10].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[10].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[10].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[10].posting[1].decinc eqv DecInc::DEC;
    ok @entry[10].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[10].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 0);
    ok @entry[10].posting[1].id.entry-id.text eqv "2011-04-01 '''\nWigwam LLC bought one wigwam for \$1.04 USD in early April 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[1].id.entry-id.xxhash == 234894800;
    ok @entry[10].posting[1].id.number == 1;
    ok @entry[10].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.04 USD";
    ok @entry[10].posting[1].id.xxhash == 3179024211;
    ok @entry[11].header.date eqv Date.new(2011,4,16);
    ok @entry[11].header.description eqv "Wigwam LLC received a dividend of \$2011.0416 USD";
    ok @entry[11].header.important == 0;
    ok @entry[11].id.number eq Array[UInt:D].new(0, 0, 2, 1, 0);
    ok @entry[11].id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].id.xxhash == 2888576925;
    ok @entry[11].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[11].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[11].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[11].posting[0].amount.asset-code eqv "USD";
    ok @entry[11].posting[0].amount.asset-quantity == 2011.0416;
    ok @entry[11].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[11].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[11].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[11].posting[0].decinc eqv DecInc::INC;
    ok @entry[11].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[11].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 1, 0);
    ok @entry[11].posting[0].id.entry-id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[0].id.entry-id.xxhash == 2888576925;
    ok @entry[11].posting[0].id.number == 0;
    ok @entry[11].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD";
    ok @entry[11].posting[0].id.xxhash == 1643697461;
    ok @entry[11].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[11].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[11].posting[1].amount.asset-code eqv "USD";
    ok @entry[11].posting[1].amount.asset-quantity == 2011.0416;
    ok @entry[11].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[11].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[11].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[11].posting[1].decinc eqv DecInc::INC;
    ok @entry[11].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[11].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 1, 0);
    ok @entry[11].posting[1].id.entry-id.text eqv "2011-04-16 'Wigwam LLC received a dividend of \$2011.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0416 USD\n  Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[1].id.entry-id.xxhash == 2888576925;
    ok @entry[11].posting[1].id.number == 1;
    ok @entry[11].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0416 USD";
    ok @entry[11].posting[1].id.xxhash == 1601475876;
    ok @entry[12].header.date eqv Date.new(2011,4,30);
    ok @entry[12].header.description eqv "Wigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[12].header.important == 0;
    ok @entry[12].id.number eq Array[UInt:D].new(0, 0, 2, 2);
    ok @entry[12].id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].id.xxhash == 801753479;
    ok @entry[12].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[12].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[12].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[12].posting[0].amount.asset-code eqv "USD";
    ok @entry[12].posting[0].amount.asset-quantity == 1.04;
    ok @entry[12].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[12].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[12].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[12].posting[0].decinc eqv DecInc::INC;
    ok @entry[12].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[12].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 2);
    ok @entry[12].posting[0].id.entry-id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[0].id.entry-id.xxhash == 801753479;
    ok @entry[12].posting[0].id.number == 0;
    ok @entry[12].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.04 USD";
    ok @entry[12].posting[0].id.xxhash == 1481608029;
    ok @entry[12].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[12].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[12].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[12].posting[1].amount.asset-quantity == 1;
    ok @entry[12].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[12].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[12].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(26, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[12].posting[1].decinc eqv DecInc::DEC;
    ok @entry[12].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[12].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 2);
    ok @entry[12].posting[1].id.entry-id.text eqv "2011-04-30 '''\nWigwam LLC sold one wigwam for \$1.04 USD in late April 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[1].id.entry-id.xxhash == 801753479;
    ok @entry[12].posting[1].id.number == 1;
    ok @entry[12].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.04 USD";
    ok @entry[12].posting[1].id.xxhash == 701076721;
    ok @entry[13].header.date eqv Date.new(2011,5,1);
    ok @entry[13].header.description eqv "Wigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[13].header.important == 0;
    ok @entry[13].id.number eq Array[UInt:D].new(0, 0, 2, 3);
    ok @entry[13].id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].id.xxhash == 2708781669;
    ok @entry[13].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[13].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[13].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[13].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[13].posting[0].amount.asset-quantity == 1;
    ok @entry[13].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[13].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[13].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[13].posting[0].decinc eqv DecInc::INC;
    ok @entry[13].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[13].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 3);
    ok @entry[13].posting[0].id.entry-id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[0].id.entry-id.xxhash == 2708781669;
    ok @entry[13].posting[0].id.number == 0;
    ok @entry[13].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD";
    ok @entry[13].posting[0].id.xxhash == 2741913114;
    ok @entry[13].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[13].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[13].posting[1].amount.asset-code eqv "USD";
    ok @entry[13].posting[1].amount.asset-quantity == 1.05;
    ok @entry[13].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[13].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[13].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[13].posting[1].decinc eqv DecInc::DEC;
    ok @entry[13].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[13].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 3);
    ok @entry[13].posting[1].id.entry-id.text eqv "2011-05-01 '''\nWigwam LLC bought one wigwam for \$1.05 USD in early May 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[1].id.entry-id.xxhash == 2708781669;
    ok @entry[13].posting[1].id.number == 1;
    ok @entry[13].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.05 USD";
    ok @entry[13].posting[1].id.xxhash == 3177866826;
    ok @entry[14].header.date eqv Date.new(2011,5,16);
    ok @entry[14].header.description eqv "Wigwam LLC received a dividend of \$2011.0516 USD";
    ok @entry[14].header.important == 0;
    ok @entry[14].id.number eq Array[UInt:D].new(0, 0, 2, 4, 0);
    ok @entry[14].id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].id.xxhash == 2585914;
    ok @entry[14].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[14].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[14].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[14].posting[0].amount.asset-code eqv "USD";
    ok @entry[14].posting[0].amount.asset-quantity == 2011.0516;
    ok @entry[14].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[14].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[14].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[14].posting[0].decinc eqv DecInc::INC;
    ok @entry[14].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[14].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 4, 0);
    ok @entry[14].posting[0].id.entry-id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[0].id.entry-id.xxhash == 2585914;
    ok @entry[14].posting[0].id.number == 0;
    ok @entry[14].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD";
    ok @entry[14].posting[0].id.xxhash == 3043997489;
    ok @entry[14].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[14].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[14].posting[1].amount.asset-code eqv "USD";
    ok @entry[14].posting[1].amount.asset-quantity == 2011.0516;
    ok @entry[14].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[14].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[14].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[14].posting[1].decinc eqv DecInc::INC;
    ok @entry[14].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[14].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 4, 0);
    ok @entry[14].posting[1].id.entry-id.text eqv "2011-05-16 'Wigwam LLC received a dividend of \$2011.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0516 USD\n  Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[1].id.entry-id.xxhash == 2585914;
    ok @entry[14].posting[1].id.number == 1;
    ok @entry[14].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0516 USD";
    ok @entry[14].posting[1].id.xxhash == 2731775541;
    ok @entry[15].header.date eqv Date.new(2011,5,31);
    ok @entry[15].header.description eqv "Wigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[15].header.important == 0;
    ok @entry[15].id.number eq Array[UInt:D].new(0, 0, 2, 5);
    ok @entry[15].id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].id.xxhash == 793767881;
    ok @entry[15].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[15].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[15].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[15].posting[0].amount.asset-code eqv "USD";
    ok @entry[15].posting[0].amount.asset-quantity == 1.05;
    ok @entry[15].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[15].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[15].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[15].posting[0].decinc eqv DecInc::INC;
    ok @entry[15].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[15].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 5);
    ok @entry[15].posting[0].id.entry-id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[0].id.entry-id.xxhash == 793767881;
    ok @entry[15].posting[0].id.number == 0;
    ok @entry[15].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.05 USD";
    ok @entry[15].posting[0].id.xxhash == 2669757498;
    ok @entry[15].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[15].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[15].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[15].posting[1].amount.asset-quantity == 1;
    ok @entry[15].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[15].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[15].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[15].posting[1].decinc eqv DecInc::DEC;
    ok @entry[15].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[15].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 5);
    ok @entry[15].posting[1].id.entry-id.text eqv "2011-05-31 '''\nWigwam LLC sold one wigwam for \$1.05 USD in late May 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[1].id.entry-id.xxhash == 793767881;
    ok @entry[15].posting[1].id.number == 1;
    ok @entry[15].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.05 USD";
    ok @entry[15].posting[1].id.xxhash == 1957446417;
    ok @entry[16].header.date eqv Date.new(2011,6,1);
    ok @entry[16].header.description eqv "Wigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[16].header.important == 0;
    ok @entry[16].id.number eq Array[UInt:D].new(0, 0, 2, 6);
    ok @entry[16].id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].id.xxhash == 1054837725;
    ok @entry[16].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[16].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[16].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[16].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[16].posting[0].amount.asset-quantity == 1;
    ok @entry[16].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[16].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[16].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[16].posting[0].decinc eqv DecInc::INC;
    ok @entry[16].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[16].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 6);
    ok @entry[16].posting[0].id.entry-id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[0].id.entry-id.xxhash == 1054837725;
    ok @entry[16].posting[0].id.number == 0;
    ok @entry[16].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD";
    ok @entry[16].posting[0].id.xxhash == 826810409;
    ok @entry[16].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[16].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[16].posting[1].amount.asset-code eqv "USD";
    ok @entry[16].posting[1].amount.asset-quantity == 1.06;
    ok @entry[16].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[16].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[16].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[16].posting[1].decinc eqv DecInc::DEC;
    ok @entry[16].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[16].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 6);
    ok @entry[16].posting[1].id.entry-id.text eqv "2011-06-01 '''\nWigwam LLC bought one wigwam for \$1.06 USD in early June 2011\nFILE:  「includes/2011/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[1].id.entry-id.xxhash == 1054837725;
    ok @entry[16].posting[1].id.number == 1;
    ok @entry[16].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.06 USD";
    ok @entry[16].posting[1].id.xxhash == 1066001694;
    ok @entry[17].header.date eqv Date.new(2011,6,16);
    ok @entry[17].header.description eqv "Wigwam LLC received a dividend of \$2011.0616 USD";
    ok @entry[17].header.important == 0;
    ok @entry[17].id.number eq Array[UInt:D].new(0, 0, 2, 7, 0);
    ok @entry[17].id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].id.xxhash == 2233477133;
    ok @entry[17].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[17].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[17].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[17].posting[0].amount.asset-code eqv "USD";
    ok @entry[17].posting[0].amount.asset-quantity == 2011.0616;
    ok @entry[17].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[17].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[17].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[17].posting[0].decinc eqv DecInc::INC;
    ok @entry[17].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[17].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 7, 0);
    ok @entry[17].posting[0].id.entry-id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[0].id.entry-id.xxhash == 2233477133;
    ok @entry[17].posting[0].id.number == 0;
    ok @entry[17].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD";
    ok @entry[17].posting[0].id.xxhash == 3145971049;
    ok @entry[17].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[17].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[17].posting[1].amount.asset-code eqv "USD";
    ok @entry[17].posting[1].amount.asset-quantity == 2011.0616;
    ok @entry[17].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[17].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[17].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[17].posting[1].decinc eqv DecInc::INC;
    ok @entry[17].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[17].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 7, 0);
    ok @entry[17].posting[1].id.entry-id.text eqv "2011-06-16 'Wigwam LLC received a dividend of \$2011.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0616 USD\n  Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[1].id.entry-id.xxhash == 2233477133;
    ok @entry[17].posting[1].id.number == 1;
    ok @entry[17].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0616 USD";
    ok @entry[17].posting[1].id.xxhash == 2894491481;
    ok @entry[18].header.date eqv Date.new(2011,6,30);
    ok @entry[18].header.description eqv "Wigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[18].header.important == 0;
    ok @entry[18].id.number eq Array[UInt:D].new(0, 0, 2, 8);
    ok @entry[18].id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].id.xxhash == 2726916770;
    ok @entry[18].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[18].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[18].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[18].posting[0].amount.asset-code eqv "USD";
    ok @entry[18].posting[0].amount.asset-quantity == 1.06;
    ok @entry[18].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[18].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[18].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[18].posting[0].decinc eqv DecInc::INC;
    ok @entry[18].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[18].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 8);
    ok @entry[18].posting[0].id.entry-id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[0].id.entry-id.xxhash == 2726916770;
    ok @entry[18].posting[0].id.number == 0;
    ok @entry[18].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.06 USD";
    ok @entry[18].posting[0].id.xxhash == 4047159393;
    ok @entry[18].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[18].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[18].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[18].posting[1].amount.asset-quantity == 1;
    ok @entry[18].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[18].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[18].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[18].posting[1].decinc eqv DecInc::DEC;
    ok @entry[18].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[18].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 2, 8);
    ok @entry[18].posting[1].id.entry-id.text eqv "2011-06-30 '''\nWigwam LLC sold one wigwam for \$1.06 USD in late June 2011\nFILE:  「includes/2011/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[1].id.entry-id.xxhash == 2726916770;
    ok @entry[18].posting[1].id.number == 1;
    ok @entry[18].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.06 USD";
    ok @entry[18].posting[1].id.xxhash == 1322105725;
    ok @entry[19].header.date eqv Date.new(2011,7,1);
    ok @entry[19].header.description eqv "Wigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[19].header.important == 0;
    ok @entry[19].id.number eq Array[UInt:D].new(0, 0, 3, 0);
    ok @entry[19].id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].id.xxhash == 2522748302;
    ok @entry[19].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[19].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[19].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[19].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[19].posting[0].amount.asset-quantity == 1;
    ok @entry[19].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[19].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[19].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(107, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[19].posting[0].decinc eqv DecInc::INC;
    ok @entry[19].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[19].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 0);
    ok @entry[19].posting[0].id.entry-id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[0].id.entry-id.xxhash == 2522748302;
    ok @entry[19].posting[0].id.number == 0;
    ok @entry[19].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD";
    ok @entry[19].posting[0].id.xxhash == 4103355834;
    ok @entry[19].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[19].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[19].posting[1].amount.asset-code eqv "USD";
    ok @entry[19].posting[1].amount.asset-quantity == 1.07;
    ok @entry[19].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[19].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[19].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[19].posting[1].decinc eqv DecInc::DEC;
    ok @entry[19].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[19].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 0);
    ok @entry[19].posting[1].id.entry-id.text eqv "2011-07-01 '''\nWigwam LLC bought one wigwam for \$1.07 USD in early July 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[1].id.entry-id.xxhash == 2522748302;
    ok @entry[19].posting[1].id.number == 1;
    ok @entry[19].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.07 USD";
    ok @entry[19].posting[1].id.xxhash == 879230496;
    ok @entry[20].header.date eqv Date.new(2011,7,16);
    ok @entry[20].header.description eqv "Wigwam LLC received a dividend of \$2011.0716 USD";
    ok @entry[20].header.important == 0;
    ok @entry[20].id.number eq Array[UInt:D].new(0, 0, 3, 1, 0);
    ok @entry[20].id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].id.xxhash == 66189993;
    ok @entry[20].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[20].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[20].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[20].posting[0].amount.asset-code eqv "USD";
    ok @entry[20].posting[0].amount.asset-quantity == 2011.0716;
    ok @entry[20].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[20].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[20].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[20].posting[0].decinc eqv DecInc::INC;
    ok @entry[20].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[20].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 1, 0);
    ok @entry[20].posting[0].id.entry-id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[0].id.entry-id.xxhash == 66189993;
    ok @entry[20].posting[0].id.number == 0;
    ok @entry[20].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD";
    ok @entry[20].posting[0].id.xxhash == 1214634887;
    ok @entry[20].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[20].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[20].posting[1].amount.asset-code eqv "USD";
    ok @entry[20].posting[1].amount.asset-quantity == 2011.0716;
    ok @entry[20].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[20].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[20].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[20].posting[1].decinc eqv DecInc::INC;
    ok @entry[20].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[20].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 1, 0);
    ok @entry[20].posting[1].id.entry-id.text eqv "2011-07-16 'Wigwam LLC received a dividend of \$2011.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0716 USD\n  Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[1].id.entry-id.xxhash == 66189993;
    ok @entry[20].posting[1].id.number == 1;
    ok @entry[20].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0716 USD";
    ok @entry[20].posting[1].id.xxhash == 1354653370;
    ok @entry[21].header.date eqv Date.new(2011,7,31);
    ok @entry[21].header.description eqv "Wigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[21].header.important == 0;
    ok @entry[21].id.number eq Array[UInt:D].new(0, 0, 3, 2);
    ok @entry[21].id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].id.xxhash == 992681704;
    ok @entry[21].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[21].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[21].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[21].posting[0].amount.asset-code eqv "USD";
    ok @entry[21].posting[0].amount.asset-quantity == 1.07;
    ok @entry[21].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[21].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[21].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[21].posting[0].decinc eqv DecInc::INC;
    ok @entry[21].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[21].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 2);
    ok @entry[21].posting[0].id.entry-id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[0].id.entry-id.xxhash == 992681704;
    ok @entry[21].posting[0].id.number == 0;
    ok @entry[21].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.07 USD";
    ok @entry[21].posting[0].id.xxhash == 1539269153;
    ok @entry[21].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[21].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[21].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[21].posting[1].amount.asset-quantity == 1;
    ok @entry[21].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[21].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[21].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(107, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[21].posting[1].decinc eqv DecInc::DEC;
    ok @entry[21].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[21].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 2);
    ok @entry[21].posting[1].id.entry-id.text eqv "2011-07-31 '''\nWigwam LLC sold one wigwam for \$1.07 USD in late July 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[1].id.entry-id.xxhash == 992681704;
    ok @entry[21].posting[1].id.number == 1;
    ok @entry[21].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.07 USD";
    ok @entry[21].posting[1].id.xxhash == 1406482422;
    ok @entry[22].header.date eqv Date.new(2011,8,1);
    ok @entry[22].header.description eqv "Wigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[22].header.important == 0;
    ok @entry[22].id.number eq Array[UInt:D].new(0, 0, 3, 3);
    ok @entry[22].id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].id.xxhash == 2932388883;
    ok @entry[22].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[22].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[22].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[22].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[22].posting[0].amount.asset-quantity == 1;
    ok @entry[22].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[22].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[22].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(27, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[22].posting[0].decinc eqv DecInc::INC;
    ok @entry[22].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[22].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 3);
    ok @entry[22].posting[0].id.entry-id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[0].id.entry-id.xxhash == 2932388883;
    ok @entry[22].posting[0].id.number == 0;
    ok @entry[22].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD";
    ok @entry[22].posting[0].id.xxhash == 4060607404;
    ok @entry[22].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[22].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[22].posting[1].amount.asset-code eqv "USD";
    ok @entry[22].posting[1].amount.asset-quantity == 1.08;
    ok @entry[22].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[22].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[22].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[22].posting[1].decinc eqv DecInc::DEC;
    ok @entry[22].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[22].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 3);
    ok @entry[22].posting[1].id.entry-id.text eqv "2011-08-01 '''\nWigwam LLC bought one wigwam for \$1.08 USD in early August 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[1].id.entry-id.xxhash == 2932388883;
    ok @entry[22].posting[1].id.number == 1;
    ok @entry[22].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.08 USD";
    ok @entry[22].posting[1].id.xxhash == 1923859345;
    ok @entry[23].header.date eqv Date.new(2011,8,16);
    ok @entry[23].header.description eqv "Wigwam LLC received a dividend of \$2011.0816 USD";
    ok @entry[23].header.important == 0;
    ok @entry[23].id.number eq Array[UInt:D].new(0, 0, 3, 4, 0);
    ok @entry[23].id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].id.xxhash == 165262646;
    ok @entry[23].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[23].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[23].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[23].posting[0].amount.asset-code eqv "USD";
    ok @entry[23].posting[0].amount.asset-quantity == 2011.0816;
    ok @entry[23].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[23].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[23].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[23].posting[0].decinc eqv DecInc::INC;
    ok @entry[23].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[23].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 4, 0);
    ok @entry[23].posting[0].id.entry-id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[0].id.entry-id.xxhash == 165262646;
    ok @entry[23].posting[0].id.number == 0;
    ok @entry[23].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD";
    ok @entry[23].posting[0].id.xxhash == 541135489;
    ok @entry[23].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[23].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[23].posting[1].amount.asset-code eqv "USD";
    ok @entry[23].posting[1].amount.asset-quantity == 2011.0816;
    ok @entry[23].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[23].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[23].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[23].posting[1].decinc eqv DecInc::INC;
    ok @entry[23].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[23].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 4, 0);
    ok @entry[23].posting[1].id.entry-id.text eqv "2011-08-16 'Wigwam LLC received a dividend of \$2011.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0816 USD\n  Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[1].id.entry-id.xxhash == 165262646;
    ok @entry[23].posting[1].id.number == 1;
    ok @entry[23].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0816 USD";
    ok @entry[23].posting[1].id.xxhash == 4267062116;
    ok @entry[24].header.date eqv Date.new(2011,8,31);
    ok @entry[24].header.description eqv "Wigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[24].header.important == 0;
    ok @entry[24].id.number eq Array[UInt:D].new(0, 0, 3, 5);
    ok @entry[24].id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].id.xxhash == 2421394429;
    ok @entry[24].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[24].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[24].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[24].posting[0].amount.asset-code eqv "USD";
    ok @entry[24].posting[0].amount.asset-quantity == 1.08;
    ok @entry[24].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[24].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[24].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[24].posting[0].decinc eqv DecInc::INC;
    ok @entry[24].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[24].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 5);
    ok @entry[24].posting[0].id.entry-id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[0].id.entry-id.xxhash == 2421394429;
    ok @entry[24].posting[0].id.number == 0;
    ok @entry[24].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.08 USD";
    ok @entry[24].posting[0].id.xxhash == 4017504886;
    ok @entry[24].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[24].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[24].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[24].posting[1].amount.asset-quantity == 1;
    ok @entry[24].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[24].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[24].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(27, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[24].posting[1].decinc eqv DecInc::DEC;
    ok @entry[24].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[24].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 5);
    ok @entry[24].posting[1].id.entry-id.text eqv "2011-08-31 '''\nWigwam LLC sold one wigwam for \$1.08 USD in late August 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[1].id.entry-id.xxhash == 2421394429;
    ok @entry[24].posting[1].id.number == 1;
    ok @entry[24].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.08 USD";
    ok @entry[24].posting[1].id.xxhash == 1142673963;
    ok @entry[25].header.date eqv Date.new(2011,9,1);
    ok @entry[25].header.description eqv "Wigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[25].header.important == 0;
    ok @entry[25].id.number eq Array[UInt:D].new(0, 0, 3, 6);
    ok @entry[25].id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].id.xxhash == 3100476228;
    ok @entry[25].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[25].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[25].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[25].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[25].posting[0].amount.asset-quantity == 1;
    ok @entry[25].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[25].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[25].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(109, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[25].posting[0].decinc eqv DecInc::INC;
    ok @entry[25].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[25].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 6);
    ok @entry[25].posting[0].id.entry-id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[0].id.entry-id.xxhash == 3100476228;
    ok @entry[25].posting[0].id.number == 0;
    ok @entry[25].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD";
    ok @entry[25].posting[0].id.xxhash == 2201473302;
    ok @entry[25].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[25].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[25].posting[1].amount.asset-code eqv "USD";
    ok @entry[25].posting[1].amount.asset-quantity == 1.09;
    ok @entry[25].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[25].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[25].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[25].posting[1].decinc eqv DecInc::DEC;
    ok @entry[25].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[25].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 6);
    ok @entry[25].posting[1].id.entry-id.text eqv "2011-09-01 '''\nWigwam LLC bought one wigwam for \$1.09 USD in early September 2011\nFILE:  「includes/2011/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[1].id.entry-id.xxhash == 3100476228;
    ok @entry[25].posting[1].id.number == 1;
    ok @entry[25].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.09 USD";
    ok @entry[25].posting[1].id.xxhash == 1895357630;
    ok @entry[26].header.date eqv Date.new(2011,9,16);
    ok @entry[26].header.description eqv "Wigwam LLC received a dividend of \$2011.0916 USD";
    ok @entry[26].header.important == 0;
    ok @entry[26].id.number eq Array[UInt:D].new(0, 0, 3, 7, 0);
    ok @entry[26].id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].id.xxhash == 1591575233;
    ok @entry[26].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[26].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[26].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[26].posting[0].amount.asset-code eqv "USD";
    ok @entry[26].posting[0].amount.asset-quantity == 2011.0916;
    ok @entry[26].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[26].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[26].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[26].posting[0].decinc eqv DecInc::INC;
    ok @entry[26].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[26].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 7, 0);
    ok @entry[26].posting[0].id.entry-id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[0].id.entry-id.xxhash == 1591575233;
    ok @entry[26].posting[0].id.number == 0;
    ok @entry[26].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD";
    ok @entry[26].posting[0].id.xxhash == 3079066074;
    ok @entry[26].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[26].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[26].posting[1].amount.asset-code eqv "USD";
    ok @entry[26].posting[1].amount.asset-quantity == 2011.0916;
    ok @entry[26].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[26].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[26].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[26].posting[1].decinc eqv DecInc::INC;
    ok @entry[26].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[26].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 7, 0);
    ok @entry[26].posting[1].id.entry-id.text eqv "2011-09-16 'Wigwam LLC received a dividend of \$2011.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.0916 USD\n  Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[1].id.entry-id.xxhash == 1591575233;
    ok @entry[26].posting[1].id.number == 1;
    ok @entry[26].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.0916 USD";
    ok @entry[26].posting[1].id.xxhash == 3486713981;
    ok @entry[27].header.date eqv Date.new(2011,9,30);
    ok @entry[27].header.description eqv "Wigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[27].header.important == 0;
    ok @entry[27].id.number eq Array[UInt:D].new(0, 0, 3, 8);
    ok @entry[27].id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].id.xxhash == 1027238853;
    ok @entry[27].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[27].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[27].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[27].posting[0].amount.asset-code eqv "USD";
    ok @entry[27].posting[0].amount.asset-quantity == 1.09;
    ok @entry[27].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[27].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[27].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[27].posting[0].decinc eqv DecInc::INC;
    ok @entry[27].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[27].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 8);
    ok @entry[27].posting[0].id.entry-id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[0].id.entry-id.xxhash == 1027238853;
    ok @entry[27].posting[0].id.number == 0;
    ok @entry[27].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.09 USD";
    ok @entry[27].posting[0].id.xxhash == 2763187408;
    ok @entry[27].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[27].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[27].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[27].posting[1].amount.asset-quantity == 1;
    ok @entry[27].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[27].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[27].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(109, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[27].posting[1].decinc eqv DecInc::DEC;
    ok @entry[27].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[27].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 3, 8);
    ok @entry[27].posting[1].id.entry-id.text eqv "2011-09-30 '''\nWigwam LLC sold one wigwam for \$1.09 USD in late September 2011\nFILE:  「includes/2011/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[1].id.entry-id.xxhash == 1027238853;
    ok @entry[27].posting[1].id.number == 1;
    ok @entry[27].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.09 USD";
    ok @entry[27].posting[1].id.xxhash == 223171015;
    ok @entry[28].header.date eqv Date.new(2011,10,1);
    ok @entry[28].header.description eqv "Wigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[28].header.important == 0;
    ok @entry[28].id.number eq Array[UInt:D].new(0, 0, 4, 0);
    ok @entry[28].id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].id.xxhash == 2088813850;
    ok @entry[28].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[28].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[28].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[28].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[28].posting[0].amount.asset-quantity == 1;
    ok @entry[28].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[28].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[28].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(11, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[28].posting[0].decinc eqv DecInc::INC;
    ok @entry[28].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[28].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 0);
    ok @entry[28].posting[0].id.entry-id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[0].id.entry-id.xxhash == 2088813850;
    ok @entry[28].posting[0].id.number == 0;
    ok @entry[28].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD";
    ok @entry[28].posting[0].id.xxhash == 4077097927;
    ok @entry[28].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[28].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[28].posting[1].amount.asset-code eqv "USD";
    ok @entry[28].posting[1].amount.asset-quantity == 1.1;
    ok @entry[28].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[28].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[28].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[28].posting[1].decinc eqv DecInc::DEC;
    ok @entry[28].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[28].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 0);
    ok @entry[28].posting[1].id.entry-id.text eqv "2011-10-01 '''\nWigwam LLC bought one wigwam for \$1.10 USD in early October 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[1].id.entry-id.xxhash == 2088813850;
    ok @entry[28].posting[1].id.number == 1;
    ok @entry[28].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.10 USD";
    ok @entry[28].posting[1].id.xxhash == 39322455;
    ok @entry[29].header.date eqv Date.new(2011,10,16);
    ok @entry[29].header.description eqv "Wigwam LLC received a dividend of \$2011.1016 USD";
    ok @entry[29].header.important == 0;
    ok @entry[29].id.number eq Array[UInt:D].new(0, 0, 4, 1, 0);
    ok @entry[29].id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].id.xxhash == 1630761716;
    ok @entry[29].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[29].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[29].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[29].posting[0].amount.asset-code eqv "USD";
    ok @entry[29].posting[0].amount.asset-quantity == 2011.1016;
    ok @entry[29].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[29].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[29].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[29].posting[0].decinc eqv DecInc::INC;
    ok @entry[29].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[29].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 1, 0);
    ok @entry[29].posting[0].id.entry-id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[0].id.entry-id.xxhash == 1630761716;
    ok @entry[29].posting[0].id.number == 0;
    ok @entry[29].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD";
    ok @entry[29].posting[0].id.xxhash == 1180154985;
    ok @entry[29].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[29].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[29].posting[1].amount.asset-code eqv "USD";
    ok @entry[29].posting[1].amount.asset-quantity == 2011.1016;
    ok @entry[29].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[29].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[29].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[29].posting[1].decinc eqv DecInc::INC;
    ok @entry[29].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[29].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 1, 0);
    ok @entry[29].posting[1].id.entry-id.text eqv "2011-10-16 'Wigwam LLC received a dividend of \$2011.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1016 USD\n  Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[1].id.entry-id.xxhash == 1630761716;
    ok @entry[29].posting[1].id.number == 1;
    ok @entry[29].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1016 USD";
    ok @entry[29].posting[1].id.xxhash == 1286546360;
    ok @entry[30].header.date eqv Date.new(2011,10,31);
    ok @entry[30].header.description eqv "Wigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[30].header.important == 0;
    ok @entry[30].id.number eq Array[UInt:D].new(0, 0, 4, 2);
    ok @entry[30].id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].id.xxhash == 3547803812;
    ok @entry[30].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[30].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[30].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[30].posting[0].amount.asset-code eqv "USD";
    ok @entry[30].posting[0].amount.asset-quantity == 1.1;
    ok @entry[30].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[30].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[30].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[30].posting[0].decinc eqv DecInc::INC;
    ok @entry[30].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[30].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 2);
    ok @entry[30].posting[0].id.entry-id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[0].id.entry-id.xxhash == 3547803812;
    ok @entry[30].posting[0].id.number == 0;
    ok @entry[30].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.10 USD";
    ok @entry[30].posting[0].id.xxhash == 7326839;
    ok @entry[30].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[30].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[30].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[30].posting[1].amount.asset-quantity == 1;
    ok @entry[30].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[30].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[30].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(11, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[30].posting[1].decinc eqv DecInc::DEC;
    ok @entry[30].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[30].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 2);
    ok @entry[30].posting[1].id.entry-id.text eqv "2011-10-31 '''\nWigwam LLC sold one wigwam for \$1.10 USD in late October 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[1].id.entry-id.xxhash == 3547803812;
    ok @entry[30].posting[1].id.number == 1;
    ok @entry[30].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.10 USD";
    ok @entry[30].posting[1].id.xxhash == 532507333;
    ok @entry[31].header.date eqv Date.new(2011,11,1);
    ok @entry[31].header.description eqv "Wigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[31].header.important == 0;
    ok @entry[31].id.number eq Array[UInt:D].new(0, 0, 4, 3);
    ok @entry[31].id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].id.xxhash == 3153707624;
    ok @entry[31].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[31].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[31].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[31].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[31].posting[0].amount.asset-quantity == 1;
    ok @entry[31].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[31].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[31].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(111, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[31].posting[0].decinc eqv DecInc::INC;
    ok @entry[31].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[31].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 3);
    ok @entry[31].posting[0].id.entry-id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[0].id.entry-id.xxhash == 3153707624;
    ok @entry[31].posting[0].id.number == 0;
    ok @entry[31].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD";
    ok @entry[31].posting[0].id.xxhash == 3758225172;
    ok @entry[31].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[31].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[31].posting[1].amount.asset-code eqv "USD";
    ok @entry[31].posting[1].amount.asset-quantity == 1.11;
    ok @entry[31].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[31].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[31].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[31].posting[1].decinc eqv DecInc::DEC;
    ok @entry[31].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[31].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 3);
    ok @entry[31].posting[1].id.entry-id.text eqv "2011-11-01 '''\nWigwam LLC bought one wigwam for \$1.11 USD in early November 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[1].id.entry-id.xxhash == 3153707624;
    ok @entry[31].posting[1].id.number == 1;
    ok @entry[31].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.11 USD";
    ok @entry[31].posting[1].id.xxhash == 4143645507;
    ok @entry[32].header.date eqv Date.new(2011,11,16);
    ok @entry[32].header.description eqv "Wigwam LLC received a dividend of \$2011.1116 USD";
    ok @entry[32].header.important == 0;
    ok @entry[32].id.number eq Array[UInt:D].new(0, 0, 4, 4, 0);
    ok @entry[32].id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].id.xxhash == 784831183;
    ok @entry[32].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[32].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[32].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[32].posting[0].amount.asset-code eqv "USD";
    ok @entry[32].posting[0].amount.asset-quantity == 2011.1116;
    ok @entry[32].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[32].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[32].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[32].posting[0].decinc eqv DecInc::INC;
    ok @entry[32].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[32].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 4, 0);
    ok @entry[32].posting[0].id.entry-id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[0].id.entry-id.xxhash == 784831183;
    ok @entry[32].posting[0].id.number == 0;
    ok @entry[32].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD";
    ok @entry[32].posting[0].id.xxhash == 3948144430;
    ok @entry[32].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[32].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[32].posting[1].amount.asset-code eqv "USD";
    ok @entry[32].posting[1].amount.asset-quantity == 2011.1116;
    ok @entry[32].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[32].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[32].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[32].posting[1].decinc eqv DecInc::INC;
    ok @entry[32].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[32].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 4, 0);
    ok @entry[32].posting[1].id.entry-id.text eqv "2011-11-16 'Wigwam LLC received a dividend of \$2011.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1116 USD\n  Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[1].id.entry-id.xxhash == 784831183;
    ok @entry[32].posting[1].id.number == 1;
    ok @entry[32].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1116 USD";
    ok @entry[32].posting[1].id.xxhash == 3507307506;
    ok @entry[33].header.date eqv Date.new(2011,11,30);
    ok @entry[33].header.description eqv "Wigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[33].header.important == 0;
    ok @entry[33].id.number eq Array[UInt:D].new(0, 0, 4, 5);
    ok @entry[33].id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].id.xxhash == 3036041749;
    ok @entry[33].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[33].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[33].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[33].posting[0].amount.asset-code eqv "USD";
    ok @entry[33].posting[0].amount.asset-quantity == 1.11;
    ok @entry[33].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[33].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[33].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[33].posting[0].decinc eqv DecInc::INC;
    ok @entry[33].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[33].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 5);
    ok @entry[33].posting[0].id.entry-id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[0].id.entry-id.xxhash == 3036041749;
    ok @entry[33].posting[0].id.number == 0;
    ok @entry[33].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.11 USD";
    ok @entry[33].posting[0].id.xxhash == 2788789124;
    ok @entry[33].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[33].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[33].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[33].posting[1].amount.asset-quantity == 1;
    ok @entry[33].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[33].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[33].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(111, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[33].posting[1].decinc eqv DecInc::DEC;
    ok @entry[33].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[33].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 5);
    ok @entry[33].posting[1].id.entry-id.text eqv "2011-11-30 '''\nWigwam LLC sold one wigwam for \$1.11 USD in late November 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[1].id.entry-id.xxhash == 3036041749;
    ok @entry[33].posting[1].id.number == 1;
    ok @entry[33].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.11 USD";
    ok @entry[33].posting[1].id.xxhash == 942077384;
    ok @entry[34].header.date eqv Date.new(2011,12,1);
    ok @entry[34].header.description eqv "Wigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[34].header.important == 0;
    ok @entry[34].id.number eq Array[UInt:D].new(0, 0, 4, 6);
    ok @entry[34].id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].id.xxhash == 3901453053;
    ok @entry[34].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[34].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[34].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[34].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[34].posting[0].amount.asset-quantity == 1;
    ok @entry[34].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[34].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[34].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(28, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[34].posting[0].decinc eqv DecInc::INC;
    ok @entry[34].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[34].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 6);
    ok @entry[34].posting[0].id.entry-id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[0].id.entry-id.xxhash == 3901453053;
    ok @entry[34].posting[0].id.number == 0;
    ok @entry[34].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD";
    ok @entry[34].posting[0].id.xxhash == 3674014973;
    ok @entry[34].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[34].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[34].posting[1].amount.asset-code eqv "USD";
    ok @entry[34].posting[1].amount.asset-quantity == 1.12;
    ok @entry[34].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[34].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[34].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[34].posting[1].decinc eqv DecInc::DEC;
    ok @entry[34].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[34].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 6);
    ok @entry[34].posting[1].id.entry-id.text eqv "2011-12-01 '''\nWigwam LLC bought one wigwam for \$1.12 USD in early December 2011\nFILE:  「includes/2011/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 1.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[1].id.entry-id.xxhash == 3901453053;
    ok @entry[34].posting[1].id.number == 1;
    ok @entry[34].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -1.12 USD";
    ok @entry[34].posting[1].id.xxhash == 1505204997;
    ok @entry[35].header.date eqv Date.new(2011,12,16);
    ok @entry[35].header.description eqv "Wigwam LLC received a dividend of \$2011.1216 USD";
    ok @entry[35].header.important == 0;
    ok @entry[35].id.number eq Array[UInt:D].new(0, 0, 4, 7, 0);
    ok @entry[35].id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].id.xxhash == 2743061682;
    ok @entry[35].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[35].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[35].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[35].posting[0].amount.asset-code eqv "USD";
    ok @entry[35].posting[0].amount.asset-quantity == 2011.1216;
    ok @entry[35].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[35].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[35].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[35].posting[0].decinc eqv DecInc::INC;
    ok @entry[35].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[35].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 7, 0);
    ok @entry[35].posting[0].id.entry-id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[0].id.entry-id.xxhash == 2743061682;
    ok @entry[35].posting[0].id.number == 0;
    ok @entry[35].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD";
    ok @entry[35].posting[0].id.xxhash == 1332472062;
    ok @entry[35].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[35].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[35].posting[1].amount.asset-code eqv "USD";
    ok @entry[35].posting[1].amount.asset-quantity == 2011.1216;
    ok @entry[35].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[35].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[35].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[35].posting[1].decinc eqv DecInc::INC;
    ok @entry[35].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[35].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 7, 0);
    ok @entry[35].posting[1].id.entry-id.text eqv "2011-12-16 'Wigwam LLC received a dividend of \$2011.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2011.1216 USD\n  Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[1].id.entry-id.xxhash == 2743061682;
    ok @entry[35].posting[1].id.number == 1;
    ok @entry[35].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2011.1216 USD";
    ok @entry[35].posting[1].id.xxhash == 1858587862;
    ok @entry[36].header.date eqv Date.new(2011,12,31);
    ok @entry[36].header.description eqv "Wigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[36].header.important == 0;
    ok @entry[36].id.number eq Array[UInt:D].new(0, 0, 4, 8);
    ok @entry[36].id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].id.xxhash == 3355735851;
    ok @entry[36].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[36].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[36].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[36].posting[0].amount.asset-code eqv "USD";
    ok @entry[36].posting[0].amount.asset-quantity == 1.12;
    ok @entry[36].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[36].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[36].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[36].posting[0].decinc eqv DecInc::INC;
    ok @entry[36].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[36].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 8);
    ok @entry[36].posting[0].id.entry-id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[0].id.entry-id.xxhash == 3355735851;
    ok @entry[36].posting[0].id.number == 0;
    ok @entry[36].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       1.12 USD";
    ok @entry[36].posting[0].id.xxhash == 3597539835;
    ok @entry[36].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[36].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[36].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[36].posting[1].amount.asset-quantity == 1;
    ok @entry[36].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[36].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[36].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(28, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[36].posting[1].decinc eqv DecInc::DEC;
    ok @entry[36].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[36].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 0, 4, 8);
    ok @entry[36].posting[1].id.entry-id.text eqv "2011-12-31 '''\nWigwam LLC sold one wigwam for \$1.12 USD in late December 2011\nFILE:  「includes/2011/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       1.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[1].id.entry-id.xxhash == 3355735851;
    ok @entry[36].posting[1].id.number == 1;
    ok @entry[36].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 1.12 USD";
    ok @entry[36].posting[1].id.xxhash == 1323199045;
    ok @entry[37].header.date eqv Date.new(2012,1,1);
    ok @entry[37].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[37].header.important == 0;
    ok @entry[37].id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[37].id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].id.xxhash == 581231782;
    ok @entry[37].posting[0].account.entity eqv "FooCorp";
    ok @entry[37].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[37].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[37].posting[0].amount.asset-code eqv "USD";
    ok @entry[37].posting[0].amount.asset-quantity == 1000;
    ok @entry[37].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[37].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[37].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[37].posting[0].decinc eqv DecInc::INC;
    ok @entry[37].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[37].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[37].posting[0].id.entry-id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[0].id.entry-id.xxhash == 581231782;
    ok @entry[37].posting[0].id.number == 0;
    ok @entry[37].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[37].posting[0].id.xxhash == 3244003616;
    ok @entry[37].posting[1].account.entity eqv "FooCorp";
    ok @entry[37].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[37].posting[1].amount.asset-code eqv "USD";
    ok @entry[37].posting[1].amount.asset-quantity == 1000;
    ok @entry[37].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[37].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[37].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[37].posting[1].decinc eqv DecInc::INC;
    ok @entry[37].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[37].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 0);
    ok @entry[37].posting[1].id.entry-id.text eqv "2012-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[1].id.entry-id.xxhash == 581231782;
    ok @entry[37].posting[1].id.number == 1;
    ok @entry[37].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[37].posting[1].id.xxhash == 1025058054;
    ok @entry[38].header.date eqv Date.new(2012,1,1);
    ok @entry[38].header.description eqv "Wigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[38].header.important == 0;
    ok @entry[38].id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[38].id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].id.xxhash == 2118065854;
    ok @entry[38].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[38].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[38].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[38].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[38].posting[0].amount.asset-quantity == 1;
    ok @entry[38].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[38].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[38].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(201, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[38].posting[0].decinc eqv DecInc::INC;
    ok @entry[38].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[38].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[38].posting[0].id.entry-id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[0].id.entry-id.xxhash == 2118065854;
    ok @entry[38].posting[0].id.number == 0;
    ok @entry[38].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD";
    ok @entry[38].posting[0].id.xxhash == 3765185300;
    ok @entry[38].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[38].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[38].posting[1].amount.asset-code eqv "USD";
    ok @entry[38].posting[1].amount.asset-quantity == 2.01;
    ok @entry[38].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[38].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[38].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[38].posting[1].decinc eqv DecInc::DEC;
    ok @entry[38].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[38].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 0);
    ok @entry[38].posting[1].id.entry-id.text eqv "2012-01-01 '''\nWigwam LLC bought one wigwam for \$2.01 USD in early January 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[1].id.entry-id.xxhash == 2118065854;
    ok @entry[38].posting[1].id.number == 1;
    ok @entry[38].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.01 USD";
    ok @entry[38].posting[1].id.xxhash == 2966987256;
    ok @entry[39].header.date eqv Date.new(2012,1,16);
    ok @entry[39].header.description eqv "Wigwam LLC received a dividend of \$2012.0116 USD";
    ok @entry[39].header.important == 0;
    ok @entry[39].id.number eq Array[UInt:D].new(0, 1, 1, 1, 0);
    ok @entry[39].id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].id.xxhash == 822843219;
    ok @entry[39].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[39].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[39].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[39].posting[0].amount.asset-code eqv "USD";
    ok @entry[39].posting[0].amount.asset-quantity == 2012.0116;
    ok @entry[39].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[39].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[39].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[39].posting[0].decinc eqv DecInc::INC;
    ok @entry[39].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[39].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 1, 0);
    ok @entry[39].posting[0].id.entry-id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[0].id.entry-id.xxhash == 822843219;
    ok @entry[39].posting[0].id.number == 0;
    ok @entry[39].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD";
    ok @entry[39].posting[0].id.xxhash == 4018452130;
    ok @entry[39].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[39].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[39].posting[1].amount.asset-code eqv "USD";
    ok @entry[39].posting[1].amount.asset-quantity == 2012.0116;
    ok @entry[39].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[39].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[39].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[39].posting[1].decinc eqv DecInc::INC;
    ok @entry[39].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[39].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 1, 0);
    ok @entry[39].posting[1].id.entry-id.text eqv "2012-01-16 'Wigwam LLC received a dividend of \$2012.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0116 USD\n  Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[1].id.entry-id.xxhash == 822843219;
    ok @entry[39].posting[1].id.number == 1;
    ok @entry[39].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0116 USD";
    ok @entry[39].posting[1].id.xxhash == 1403571913;
    ok @entry[40].header.date eqv Date.new(2012,1,31);
    ok @entry[40].header.description eqv "Wigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[40].header.important == 0;
    ok @entry[40].id.number eq Array[UInt:D].new(0, 1, 1, 2);
    ok @entry[40].id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].id.xxhash == 1801463403;
    ok @entry[40].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[40].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[40].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[40].posting[0].amount.asset-code eqv "USD";
    ok @entry[40].posting[0].amount.asset-quantity == 2.01;
    ok @entry[40].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[40].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[40].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[40].posting[0].decinc eqv DecInc::INC;
    ok @entry[40].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[40].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 2);
    ok @entry[40].posting[0].id.entry-id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[0].id.entry-id.xxhash == 1801463403;
    ok @entry[40].posting[0].id.number == 0;
    ok @entry[40].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.01 USD";
    ok @entry[40].posting[0].id.xxhash == 1866242286;
    ok @entry[40].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[40].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[40].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[40].posting[1].amount.asset-quantity == 1;
    ok @entry[40].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[40].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[40].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(201, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[40].posting[1].decinc eqv DecInc::DEC;
    ok @entry[40].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[40].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 2);
    ok @entry[40].posting[1].id.entry-id.text eqv "2012-01-31 '''\nWigwam LLC sold one wigwam for \$2.01 USD in late January 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[1].id.entry-id.xxhash == 1801463403;
    ok @entry[40].posting[1].id.number == 1;
    ok @entry[40].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.01 USD";
    ok @entry[40].posting[1].id.xxhash == 2649719130;
    ok @entry[41].header.date eqv Date.new(2012,2,1);
    ok @entry[41].header.description eqv "Wigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[41].header.important == 0;
    ok @entry[41].id.number eq Array[UInt:D].new(0, 1, 1, 3);
    ok @entry[41].id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].id.xxhash == 1339543688;
    ok @entry[41].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[41].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[41].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[41].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[41].posting[0].amount.asset-quantity == 1;
    ok @entry[41].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[41].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[41].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[41].posting[0].decinc eqv DecInc::INC;
    ok @entry[41].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[41].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 3);
    ok @entry[41].posting[0].id.entry-id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[0].id.entry-id.xxhash == 1339543688;
    ok @entry[41].posting[0].id.number == 0;
    ok @entry[41].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD";
    ok @entry[41].posting[0].id.xxhash == 1544971153;
    ok @entry[41].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[41].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[41].posting[1].amount.asset-code eqv "USD";
    ok @entry[41].posting[1].amount.asset-quantity == 2.02;
    ok @entry[41].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[41].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[41].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[41].posting[1].decinc eqv DecInc::DEC;
    ok @entry[41].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[41].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 3);
    ok @entry[41].posting[1].id.entry-id.text eqv "2012-02-01 '''\nWigwam LLC bought one wigwam for \$2.02 USD in early February 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[1].id.entry-id.xxhash == 1339543688;
    ok @entry[41].posting[1].id.number == 1;
    ok @entry[41].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.02 USD";
    ok @entry[41].posting[1].id.xxhash == 4217279969;
    ok @entry[42].header.date eqv Date.new(2012,2,16);
    ok @entry[42].header.description eqv "Wigwam LLC received a dividend of \$2012.0216 USD";
    ok @entry[42].header.important == 0;
    ok @entry[42].id.number eq Array[UInt:D].new(0, 1, 1, 4, 0);
    ok @entry[42].id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].id.xxhash == 230490045;
    ok @entry[42].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[42].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[42].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[42].posting[0].amount.asset-code eqv "USD";
    ok @entry[42].posting[0].amount.asset-quantity == 2012.0216;
    ok @entry[42].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[42].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[42].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[42].posting[0].decinc eqv DecInc::INC;
    ok @entry[42].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[42].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 4, 0);
    ok @entry[42].posting[0].id.entry-id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[0].id.entry-id.xxhash == 230490045;
    ok @entry[42].posting[0].id.number == 0;
    ok @entry[42].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD";
    ok @entry[42].posting[0].id.xxhash == 3895133532;
    ok @entry[42].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[42].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[42].posting[1].amount.asset-code eqv "USD";
    ok @entry[42].posting[1].amount.asset-quantity == 2012.0216;
    ok @entry[42].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[42].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[42].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[42].posting[1].decinc eqv DecInc::INC;
    ok @entry[42].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[42].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 4, 0);
    ok @entry[42].posting[1].id.entry-id.text eqv "2012-02-16 'Wigwam LLC received a dividend of \$2012.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0216 USD\n  Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[1].id.entry-id.xxhash == 230490045;
    ok @entry[42].posting[1].id.number == 1;
    ok @entry[42].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0216 USD";
    ok @entry[42].posting[1].id.xxhash == 2386315820;
    ok @entry[43].header.date eqv Date.new(2012,2,29);
    ok @entry[43].header.description eqv "Wigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[43].header.important == 0;
    ok @entry[43].id.number eq Array[UInt:D].new(0, 1, 1, 5);
    ok @entry[43].id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].id.xxhash == 2961867692;
    ok @entry[43].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[43].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[43].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[43].posting[0].amount.asset-code eqv "USD";
    ok @entry[43].posting[0].amount.asset-quantity == 2.02;
    ok @entry[43].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[43].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[43].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[43].posting[0].decinc eqv DecInc::INC;
    ok @entry[43].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[43].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 5);
    ok @entry[43].posting[0].id.entry-id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[0].id.entry-id.xxhash == 2961867692;
    ok @entry[43].posting[0].id.number == 0;
    ok @entry[43].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.02 USD";
    ok @entry[43].posting[0].id.xxhash == 257731380;
    ok @entry[43].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[43].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[43].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[43].posting[1].amount.asset-quantity == 1;
    ok @entry[43].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[43].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[43].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(101, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[43].posting[1].decinc eqv DecInc::DEC;
    ok @entry[43].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[43].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 5);
    ok @entry[43].posting[1].id.entry-id.text eqv "2012-02-29 '''\nWigwam LLC sold one wigwam for \$2.02 USD in late February 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[1].id.entry-id.xxhash == 2961867692;
    ok @entry[43].posting[1].id.number == 1;
    ok @entry[43].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.02 USD";
    ok @entry[43].posting[1].id.xxhash == 2786971912;
    ok @entry[44].header.date eqv Date.new(2012,3,1);
    ok @entry[44].header.description eqv "Wigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[44].header.important == 0;
    ok @entry[44].id.number eq Array[UInt:D].new(0, 1, 1, 6);
    ok @entry[44].id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].id.xxhash == 462880268;
    ok @entry[44].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[44].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[44].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[44].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[44].posting[0].amount.asset-quantity == 1;
    ok @entry[44].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[44].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[44].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(203, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[44].posting[0].decinc eqv DecInc::INC;
    ok @entry[44].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[44].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 6);
    ok @entry[44].posting[0].id.entry-id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[0].id.entry-id.xxhash == 462880268;
    ok @entry[44].posting[0].id.number == 0;
    ok @entry[44].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD";
    ok @entry[44].posting[0].id.xxhash == 2548327836;
    ok @entry[44].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[44].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[44].posting[1].amount.asset-code eqv "USD";
    ok @entry[44].posting[1].amount.asset-quantity == 2.03;
    ok @entry[44].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[44].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[44].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[44].posting[1].decinc eqv DecInc::DEC;
    ok @entry[44].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[44].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 6);
    ok @entry[44].posting[1].id.entry-id.text eqv "2012-03-01 '''\nWigwam LLC bought one wigwam for \$2.03 USD in early March 2012\nFILE:  「includes/2012/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[1].id.entry-id.xxhash == 462880268;
    ok @entry[44].posting[1].id.number == 1;
    ok @entry[44].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.03 USD";
    ok @entry[44].posting[1].id.xxhash == 48485376;
    ok @entry[45].header.date eqv Date.new(2012,3,16);
    ok @entry[45].header.description eqv "Wigwam LLC received a dividend of \$2012.0316 USD";
    ok @entry[45].header.important == 0;
    ok @entry[45].id.number eq Array[UInt:D].new(0, 1, 1, 7, 0);
    ok @entry[45].id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].id.xxhash == 1496080492;
    ok @entry[45].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[45].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[45].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[45].posting[0].amount.asset-code eqv "USD";
    ok @entry[45].posting[0].amount.asset-quantity == 2012.0316;
    ok @entry[45].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[45].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[45].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[45].posting[0].decinc eqv DecInc::INC;
    ok @entry[45].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[45].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 7, 0);
    ok @entry[45].posting[0].id.entry-id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[0].id.entry-id.xxhash == 1496080492;
    ok @entry[45].posting[0].id.number == 0;
    ok @entry[45].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD";
    ok @entry[45].posting[0].id.xxhash == 2099532214;
    ok @entry[45].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[45].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[45].posting[1].amount.asset-code eqv "USD";
    ok @entry[45].posting[1].amount.asset-quantity == 2012.0316;
    ok @entry[45].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[45].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[45].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[45].posting[1].decinc eqv DecInc::INC;
    ok @entry[45].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[45].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 7, 0);
    ok @entry[45].posting[1].id.entry-id.text eqv "2012-03-16 'Wigwam LLC received a dividend of \$2012.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0316 USD\n  Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[1].id.entry-id.xxhash == 1496080492;
    ok @entry[45].posting[1].id.number == 1;
    ok @entry[45].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0316 USD";
    ok @entry[45].posting[1].id.xxhash == 1016359530;
    ok @entry[46].header.date eqv Date.new(2012,3,31);
    ok @entry[46].header.description eqv "Wigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[46].header.important == 0;
    ok @entry[46].id.number eq Array[UInt:D].new(0, 1, 1, 8);
    ok @entry[46].id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].id.xxhash == 1725880199;
    ok @entry[46].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[46].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[46].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[46].posting[0].amount.asset-code eqv "USD";
    ok @entry[46].posting[0].amount.asset-quantity == 2.03;
    ok @entry[46].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[46].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[46].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[46].posting[0].decinc eqv DecInc::INC;
    ok @entry[46].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[46].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 8);
    ok @entry[46].posting[0].id.entry-id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[0].id.entry-id.xxhash == 1725880199;
    ok @entry[46].posting[0].id.number == 0;
    ok @entry[46].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.03 USD";
    ok @entry[46].posting[0].id.xxhash == 3066988641;
    ok @entry[46].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[46].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[46].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[46].posting[1].amount.asset-quantity == 1;
    ok @entry[46].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[46].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[46].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(203, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[46].posting[1].decinc eqv DecInc::DEC;
    ok @entry[46].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[46].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 1, 8);
    ok @entry[46].posting[1].id.entry-id.text eqv "2012-03-31 '''\nWigwam LLC sold one wigwam for \$2.03 USD in late March 2012\nFILE:  「includes/2012/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[1].id.entry-id.xxhash == 1725880199;
    ok @entry[46].posting[1].id.number == 1;
    ok @entry[46].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.03 USD";
    ok @entry[46].posting[1].id.xxhash == 3495293318;
    ok @entry[47].header.date eqv Date.new(2012,4,1);
    ok @entry[47].header.description eqv "Wigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[47].header.important == 0;
    ok @entry[47].id.number eq Array[UInt:D].new(0, 1, 2, 0);
    ok @entry[47].id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].id.xxhash == 4013949028;
    ok @entry[47].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[47].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[47].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[47].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[47].posting[0].amount.asset-quantity == 1;
    ok @entry[47].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[47].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[47].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[47].posting[0].decinc eqv DecInc::INC;
    ok @entry[47].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[47].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 0);
    ok @entry[47].posting[0].id.entry-id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[0].id.entry-id.xxhash == 4013949028;
    ok @entry[47].posting[0].id.number == 0;
    ok @entry[47].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD";
    ok @entry[47].posting[0].id.xxhash == 3088262806;
    ok @entry[47].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[47].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[47].posting[1].amount.asset-code eqv "USD";
    ok @entry[47].posting[1].amount.asset-quantity == 2.04;
    ok @entry[47].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[47].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[47].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[47].posting[1].decinc eqv DecInc::DEC;
    ok @entry[47].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[47].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 0);
    ok @entry[47].posting[1].id.entry-id.text eqv "2012-04-01 '''\nWigwam LLC bought one wigwam for \$2.04 USD in early April 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[1].id.entry-id.xxhash == 4013949028;
    ok @entry[47].posting[1].id.number == 1;
    ok @entry[47].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.04 USD";
    ok @entry[47].posting[1].id.xxhash == 1215240546;
    ok @entry[48].header.date eqv Date.new(2012,4,16);
    ok @entry[48].header.description eqv "Wigwam LLC received a dividend of \$2012.0416 USD";
    ok @entry[48].header.important == 0;
    ok @entry[48].id.number eq Array[UInt:D].new(0, 1, 2, 1, 0);
    ok @entry[48].id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].id.xxhash == 2366118516;
    ok @entry[48].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[48].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[48].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[48].posting[0].amount.asset-code eqv "USD";
    ok @entry[48].posting[0].amount.asset-quantity == 2012.0416;
    ok @entry[48].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[48].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[48].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[48].posting[0].decinc eqv DecInc::INC;
    ok @entry[48].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[48].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 1, 0);
    ok @entry[48].posting[0].id.entry-id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[0].id.entry-id.xxhash == 2366118516;
    ok @entry[48].posting[0].id.number == 0;
    ok @entry[48].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD";
    ok @entry[48].posting[0].id.xxhash == 1583826195;
    ok @entry[48].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[48].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[48].posting[1].amount.asset-code eqv "USD";
    ok @entry[48].posting[1].amount.asset-quantity == 2012.0416;
    ok @entry[48].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[48].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[48].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[48].posting[1].decinc eqv DecInc::INC;
    ok @entry[48].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[48].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 1, 0);
    ok @entry[48].posting[1].id.entry-id.text eqv "2012-04-16 'Wigwam LLC received a dividend of \$2012.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0416 USD\n  Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[1].id.entry-id.xxhash == 2366118516;
    ok @entry[48].posting[1].id.number == 1;
    ok @entry[48].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0416 USD";
    ok @entry[48].posting[1].id.xxhash == 3098158574;
    ok @entry[49].header.date eqv Date.new(2012,4,30);
    ok @entry[49].header.description eqv "Wigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[49].header.important == 0;
    ok @entry[49].id.number eq Array[UInt:D].new(0, 1, 2, 2);
    ok @entry[49].id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].id.xxhash == 534726936;
    ok @entry[49].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[49].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[49].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[49].posting[0].amount.asset-code eqv "USD";
    ok @entry[49].posting[0].amount.asset-quantity == 2.04;
    ok @entry[49].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[49].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[49].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[49].posting[0].decinc eqv DecInc::INC;
    ok @entry[49].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[49].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 2);
    ok @entry[49].posting[0].id.entry-id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[0].id.entry-id.xxhash == 534726936;
    ok @entry[49].posting[0].id.number == 0;
    ok @entry[49].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.04 USD";
    ok @entry[49].posting[0].id.xxhash == 3201881197;
    ok @entry[49].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[49].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[49].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[49].posting[1].amount.asset-quantity == 1;
    ok @entry[49].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[49].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[49].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(51, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[49].posting[1].decinc eqv DecInc::DEC;
    ok @entry[49].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[49].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 2);
    ok @entry[49].posting[1].id.entry-id.text eqv "2012-04-30 '''\nWigwam LLC sold one wigwam for \$2.04 USD in late April 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[1].id.entry-id.xxhash == 534726936;
    ok @entry[49].posting[1].id.number == 1;
    ok @entry[49].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.04 USD";
    ok @entry[49].posting[1].id.xxhash == 2492372061;
    ok @entry[50].header.date eqv Date.new(2012,5,1);
    ok @entry[50].header.description eqv "Wigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[50].header.important == 0;
    ok @entry[50].id.number eq Array[UInt:D].new(0, 1, 2, 3);
    ok @entry[50].id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].id.xxhash == 3245023882;
    ok @entry[50].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[50].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[50].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[50].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[50].posting[0].amount.asset-quantity == 1;
    ok @entry[50].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[50].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[50].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(41, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[50].posting[0].decinc eqv DecInc::INC;
    ok @entry[50].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[50].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 3);
    ok @entry[50].posting[0].id.entry-id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[0].id.entry-id.xxhash == 3245023882;
    ok @entry[50].posting[0].id.number == 0;
    ok @entry[50].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD";
    ok @entry[50].posting[0].id.xxhash == 1524838205;
    ok @entry[50].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[50].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[50].posting[1].amount.asset-code eqv "USD";
    ok @entry[50].posting[1].amount.asset-quantity == 2.05;
    ok @entry[50].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[50].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[50].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[50].posting[1].decinc eqv DecInc::DEC;
    ok @entry[50].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[50].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 3);
    ok @entry[50].posting[1].id.entry-id.text eqv "2012-05-01 '''\nWigwam LLC bought one wigwam for \$2.05 USD in early May 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[1].id.entry-id.xxhash == 3245023882;
    ok @entry[50].posting[1].id.number == 1;
    ok @entry[50].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.05 USD";
    ok @entry[50].posting[1].id.xxhash == 1182404444;
    ok @entry[51].header.date eqv Date.new(2012,5,16);
    ok @entry[51].header.description eqv "Wigwam LLC received a dividend of \$2012.0516 USD";
    ok @entry[51].header.important == 0;
    ok @entry[51].id.number eq Array[UInt:D].new(0, 1, 2, 4, 0);
    ok @entry[51].id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].id.xxhash == 2131847179;
    ok @entry[51].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[51].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[51].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[51].posting[0].amount.asset-code eqv "USD";
    ok @entry[51].posting[0].amount.asset-quantity == 2012.0516;
    ok @entry[51].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[51].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[51].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[51].posting[0].decinc eqv DecInc::INC;
    ok @entry[51].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[51].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 4, 0);
    ok @entry[51].posting[0].id.entry-id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[0].id.entry-id.xxhash == 2131847179;
    ok @entry[51].posting[0].id.number == 0;
    ok @entry[51].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD";
    ok @entry[51].posting[0].id.xxhash == 1792523213;
    ok @entry[51].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[51].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[51].posting[1].amount.asset-code eqv "USD";
    ok @entry[51].posting[1].amount.asset-quantity == 2012.0516;
    ok @entry[51].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[51].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[51].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[51].posting[1].decinc eqv DecInc::INC;
    ok @entry[51].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[51].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 4, 0);
    ok @entry[51].posting[1].id.entry-id.text eqv "2012-05-16 'Wigwam LLC received a dividend of \$2012.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0516 USD\n  Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[1].id.entry-id.xxhash == 2131847179;
    ok @entry[51].posting[1].id.number == 1;
    ok @entry[51].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0516 USD";
    ok @entry[51].posting[1].id.xxhash == 263105630;
    ok @entry[52].header.date eqv Date.new(2012,5,31);
    ok @entry[52].header.description eqv "Wigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[52].header.important == 0;
    ok @entry[52].id.number eq Array[UInt:D].new(0, 1, 2, 5);
    ok @entry[52].id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].id.xxhash == 3774317231;
    ok @entry[52].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[52].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[52].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[52].posting[0].amount.asset-code eqv "USD";
    ok @entry[52].posting[0].amount.asset-quantity == 2.05;
    ok @entry[52].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[52].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[52].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[52].posting[0].decinc eqv DecInc::INC;
    ok @entry[52].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[52].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 5);
    ok @entry[52].posting[0].id.entry-id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[0].id.entry-id.xxhash == 3774317231;
    ok @entry[52].posting[0].id.number == 0;
    ok @entry[52].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.05 USD";
    ok @entry[52].posting[0].id.xxhash == 530861043;
    ok @entry[52].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[52].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[52].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[52].posting[1].amount.asset-quantity == 1;
    ok @entry[52].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[52].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[52].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(41, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[52].posting[1].decinc eqv DecInc::DEC;
    ok @entry[52].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[52].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 5);
    ok @entry[52].posting[1].id.entry-id.text eqv "2012-05-31 '''\nWigwam LLC sold one wigwam for \$2.05 USD in late May 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[1].id.entry-id.xxhash == 3774317231;
    ok @entry[52].posting[1].id.number == 1;
    ok @entry[52].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.05 USD";
    ok @entry[52].posting[1].id.xxhash == 3748210809;
    ok @entry[53].header.date eqv Date.new(2012,6,1);
    ok @entry[53].header.description eqv "Wigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[53].header.important == 0;
    ok @entry[53].id.number eq Array[UInt:D].new(0, 1, 2, 6);
    ok @entry[53].id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].id.xxhash == 1722267067;
    ok @entry[53].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[53].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[53].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[53].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[53].posting[0].amount.asset-quantity == 1;
    ok @entry[53].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[53].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[53].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[53].posting[0].decinc eqv DecInc::INC;
    ok @entry[53].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[53].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 6);
    ok @entry[53].posting[0].id.entry-id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[0].id.entry-id.xxhash == 1722267067;
    ok @entry[53].posting[0].id.number == 0;
    ok @entry[53].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD";
    ok @entry[53].posting[0].id.xxhash == 1530406877;
    ok @entry[53].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[53].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[53].posting[1].amount.asset-code eqv "USD";
    ok @entry[53].posting[1].amount.asset-quantity == 2.06;
    ok @entry[53].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[53].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[53].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[53].posting[1].decinc eqv DecInc::DEC;
    ok @entry[53].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[53].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 6);
    ok @entry[53].posting[1].id.entry-id.text eqv "2012-06-01 '''\nWigwam LLC bought one wigwam for \$2.06 USD in early June 2012\nFILE:  「includes/2012/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[1].id.entry-id.xxhash == 1722267067;
    ok @entry[53].posting[1].id.number == 1;
    ok @entry[53].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.06 USD";
    ok @entry[53].posting[1].id.xxhash == 2192104612;
    ok @entry[54].header.date eqv Date.new(2012,6,16);
    ok @entry[54].header.description eqv "Wigwam LLC received a dividend of \$2012.0616 USD";
    ok @entry[54].header.important == 0;
    ok @entry[54].id.number eq Array[UInt:D].new(0, 1, 2, 7, 0);
    ok @entry[54].id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].id.xxhash == 3942872006;
    ok @entry[54].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[54].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[54].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[54].posting[0].amount.asset-code eqv "USD";
    ok @entry[54].posting[0].amount.asset-quantity == 2012.0616;
    ok @entry[54].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[54].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[54].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[54].posting[0].decinc eqv DecInc::INC;
    ok @entry[54].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[54].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 7, 0);
    ok @entry[54].posting[0].id.entry-id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[0].id.entry-id.xxhash == 3942872006;
    ok @entry[54].posting[0].id.number == 0;
    ok @entry[54].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD";
    ok @entry[54].posting[0].id.xxhash == 1318665303;
    ok @entry[54].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[54].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[54].posting[1].amount.asset-code eqv "USD";
    ok @entry[54].posting[1].amount.asset-quantity == 2012.0616;
    ok @entry[54].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[54].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[54].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[54].posting[1].decinc eqv DecInc::INC;
    ok @entry[54].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[54].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 7, 0);
    ok @entry[54].posting[1].id.entry-id.text eqv "2012-06-16 'Wigwam LLC received a dividend of \$2012.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0616 USD\n  Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[1].id.entry-id.xxhash == 3942872006;
    ok @entry[54].posting[1].id.number == 1;
    ok @entry[54].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0616 USD";
    ok @entry[54].posting[1].id.xxhash == 2809751684;
    ok @entry[55].header.date eqv Date.new(2012,6,30);
    ok @entry[55].header.description eqv "Wigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[55].header.important == 0;
    ok @entry[55].id.number eq Array[UInt:D].new(0, 1, 2, 8);
    ok @entry[55].id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].id.xxhash == 4168296451;
    ok @entry[55].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[55].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[55].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[55].posting[0].amount.asset-code eqv "USD";
    ok @entry[55].posting[0].amount.asset-quantity == 2.06;
    ok @entry[55].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[55].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[55].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[55].posting[0].decinc eqv DecInc::INC;
    ok @entry[55].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[55].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 8);
    ok @entry[55].posting[0].id.entry-id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[0].id.entry-id.xxhash == 4168296451;
    ok @entry[55].posting[0].id.number == 0;
    ok @entry[55].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.06 USD";
    ok @entry[55].posting[0].id.xxhash == 2175989140;
    ok @entry[55].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[55].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[55].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[55].posting[1].amount.asset-quantity == 1;
    ok @entry[55].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[55].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[55].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(103, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[55].posting[1].decinc eqv DecInc::DEC;
    ok @entry[55].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[55].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 2, 8);
    ok @entry[55].posting[1].id.entry-id.text eqv "2012-06-30 '''\nWigwam LLC sold one wigwam for \$2.06 USD in late June 2012\nFILE:  「includes/2012/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[1].id.entry-id.xxhash == 4168296451;
    ok @entry[55].posting[1].id.number == 1;
    ok @entry[55].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.06 USD";
    ok @entry[55].posting[1].id.xxhash == 3753283423;
    ok @entry[56].header.date eqv Date.new(2012,7,1);
    ok @entry[56].header.description eqv "Wigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[56].header.important == 0;
    ok @entry[56].id.number eq Array[UInt:D].new(0, 1, 3, 0);
    ok @entry[56].id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].id.xxhash == 220833898;
    ok @entry[56].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[56].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[56].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[56].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[56].posting[0].amount.asset-quantity == 1;
    ok @entry[56].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[56].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[56].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(207, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[56].posting[0].decinc eqv DecInc::INC;
    ok @entry[56].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[56].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 0);
    ok @entry[56].posting[0].id.entry-id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[0].id.entry-id.xxhash == 220833898;
    ok @entry[56].posting[0].id.number == 0;
    ok @entry[56].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD";
    ok @entry[56].posting[0].id.xxhash == 1672261378;
    ok @entry[56].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[56].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[56].posting[1].amount.asset-code eqv "USD";
    ok @entry[56].posting[1].amount.asset-quantity == 2.07;
    ok @entry[56].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[56].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[56].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[56].posting[1].decinc eqv DecInc::DEC;
    ok @entry[56].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[56].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 0);
    ok @entry[56].posting[1].id.entry-id.text eqv "2012-07-01 '''\nWigwam LLC bought one wigwam for \$2.07 USD in early July 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[1].id.entry-id.xxhash == 220833898;
    ok @entry[56].posting[1].id.number == 1;
    ok @entry[56].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.07 USD";
    ok @entry[56].posting[1].id.xxhash == 2474022905;
    ok @entry[57].header.date eqv Date.new(2012,7,16);
    ok @entry[57].header.description eqv "Wigwam LLC received a dividend of \$2012.0716 USD";
    ok @entry[57].header.important == 0;
    ok @entry[57].id.number eq Array[UInt:D].new(0, 1, 3, 1, 0);
    ok @entry[57].id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].id.xxhash == 2173105363;
    ok @entry[57].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[57].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[57].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[57].posting[0].amount.asset-code eqv "USD";
    ok @entry[57].posting[0].amount.asset-quantity == 2012.0716;
    ok @entry[57].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[57].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[57].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[57].posting[0].decinc eqv DecInc::INC;
    ok @entry[57].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[57].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 1, 0);
    ok @entry[57].posting[0].id.entry-id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[0].id.entry-id.xxhash == 2173105363;
    ok @entry[57].posting[0].id.number == 0;
    ok @entry[57].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD";
    ok @entry[57].posting[0].id.xxhash == 982703979;
    ok @entry[57].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[57].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[57].posting[1].amount.asset-code eqv "USD";
    ok @entry[57].posting[1].amount.asset-quantity == 2012.0716;
    ok @entry[57].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[57].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[57].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[57].posting[1].decinc eqv DecInc::INC;
    ok @entry[57].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[57].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 1, 0);
    ok @entry[57].posting[1].id.entry-id.text eqv "2012-07-16 'Wigwam LLC received a dividend of \$2012.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0716 USD\n  Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[1].id.entry-id.xxhash == 2173105363;
    ok @entry[57].posting[1].id.number == 1;
    ok @entry[57].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0716 USD";
    ok @entry[57].posting[1].id.xxhash == 1671141747;
    ok @entry[58].header.date eqv Date.new(2012,7,31);
    ok @entry[58].header.description eqv "Wigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[58].header.important == 0;
    ok @entry[58].id.number eq Array[UInt:D].new(0, 1, 3, 2);
    ok @entry[58].id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].id.xxhash == 2998128583;
    ok @entry[58].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[58].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[58].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[58].posting[0].amount.asset-code eqv "USD";
    ok @entry[58].posting[0].amount.asset-quantity == 2.07;
    ok @entry[58].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[58].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[58].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[58].posting[0].decinc eqv DecInc::INC;
    ok @entry[58].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[58].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 2);
    ok @entry[58].posting[0].id.entry-id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[0].id.entry-id.xxhash == 2998128583;
    ok @entry[58].posting[0].id.number == 0;
    ok @entry[58].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.07 USD";
    ok @entry[58].posting[0].id.xxhash == 2444392457;
    ok @entry[58].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[58].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[58].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[58].posting[1].amount.asset-quantity == 1;
    ok @entry[58].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[58].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[58].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(207, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[58].posting[1].decinc eqv DecInc::DEC;
    ok @entry[58].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[58].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 2);
    ok @entry[58].posting[1].id.entry-id.text eqv "2012-07-31 '''\nWigwam LLC sold one wigwam for \$2.07 USD in late July 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[1].id.entry-id.xxhash == 2998128583;
    ok @entry[58].posting[1].id.number == 1;
    ok @entry[58].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.07 USD";
    ok @entry[58].posting[1].id.xxhash == 1106340009;
    ok @entry[59].header.date eqv Date.new(2012,8,1);
    ok @entry[59].header.description eqv "Wigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[59].header.important == 0;
    ok @entry[59].id.number eq Array[UInt:D].new(0, 1, 3, 3);
    ok @entry[59].id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].id.xxhash == 752691327;
    ok @entry[59].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[59].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[59].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[59].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[59].posting[0].amount.asset-quantity == 1;
    ok @entry[59].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[59].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[59].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(52, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[59].posting[0].decinc eqv DecInc::INC;
    ok @entry[59].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[59].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 3);
    ok @entry[59].posting[0].id.entry-id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[0].id.entry-id.xxhash == 752691327;
    ok @entry[59].posting[0].id.number == 0;
    ok @entry[59].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD";
    ok @entry[59].posting[0].id.xxhash == 2144803209;
    ok @entry[59].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[59].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[59].posting[1].amount.asset-code eqv "USD";
    ok @entry[59].posting[1].amount.asset-quantity == 2.08;
    ok @entry[59].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[59].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[59].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[59].posting[1].decinc eqv DecInc::DEC;
    ok @entry[59].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[59].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 3);
    ok @entry[59].posting[1].id.entry-id.text eqv "2012-08-01 '''\nWigwam LLC bought one wigwam for \$2.08 USD in early August 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[1].id.entry-id.xxhash == 752691327;
    ok @entry[59].posting[1].id.number == 1;
    ok @entry[59].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.08 USD";
    ok @entry[59].posting[1].id.xxhash == 2749172222;
    ok @entry[60].header.date eqv Date.new(2012,8,16);
    ok @entry[60].header.description eqv "Wigwam LLC received a dividend of \$2012.0816 USD";
    ok @entry[60].header.important == 0;
    ok @entry[60].id.number eq Array[UInt:D].new(0, 1, 3, 4, 0);
    ok @entry[60].id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].id.xxhash == 1870601986;
    ok @entry[60].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[60].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[60].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[60].posting[0].amount.asset-code eqv "USD";
    ok @entry[60].posting[0].amount.asset-quantity == 2012.0816;
    ok @entry[60].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[60].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[60].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[60].posting[0].decinc eqv DecInc::INC;
    ok @entry[60].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[60].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 4, 0);
    ok @entry[60].posting[0].id.entry-id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[0].id.entry-id.xxhash == 1870601986;
    ok @entry[60].posting[0].id.number == 0;
    ok @entry[60].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD";
    ok @entry[60].posting[0].id.xxhash == 3457680319;
    ok @entry[60].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[60].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[60].posting[1].amount.asset-code eqv "USD";
    ok @entry[60].posting[1].amount.asset-quantity == 2012.0816;
    ok @entry[60].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[60].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[60].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[60].posting[1].decinc eqv DecInc::INC;
    ok @entry[60].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[60].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 4, 0);
    ok @entry[60].posting[1].id.entry-id.text eqv "2012-08-16 'Wigwam LLC received a dividend of \$2012.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0816 USD\n  Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[1].id.entry-id.xxhash == 1870601986;
    ok @entry[60].posting[1].id.number == 1;
    ok @entry[60].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0816 USD";
    ok @entry[60].posting[1].id.xxhash == 2033582180;
    ok @entry[61].header.date eqv Date.new(2012,8,31);
    ok @entry[61].header.description eqv "Wigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[61].header.important == 0;
    ok @entry[61].id.number eq Array[UInt:D].new(0, 1, 3, 5);
    ok @entry[61].id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].id.xxhash == 3036777254;
    ok @entry[61].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[61].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[61].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[61].posting[0].amount.asset-code eqv "USD";
    ok @entry[61].posting[0].amount.asset-quantity == 2.08;
    ok @entry[61].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[61].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[61].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[61].posting[0].decinc eqv DecInc::INC;
    ok @entry[61].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[61].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 5);
    ok @entry[61].posting[0].id.entry-id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[0].id.entry-id.xxhash == 3036777254;
    ok @entry[61].posting[0].id.number == 0;
    ok @entry[61].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.08 USD";
    ok @entry[61].posting[0].id.xxhash == 1814447104;
    ok @entry[61].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[61].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[61].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[61].posting[1].amount.asset-quantity == 1;
    ok @entry[61].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[61].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[61].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(52, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[61].posting[1].decinc eqv DecInc::DEC;
    ok @entry[61].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[61].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 5);
    ok @entry[61].posting[1].id.entry-id.text eqv "2012-08-31 '''\nWigwam LLC sold one wigwam for \$2.08 USD in late August 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[1].id.entry-id.xxhash == 3036777254;
    ok @entry[61].posting[1].id.number == 1;
    ok @entry[61].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.08 USD";
    ok @entry[61].posting[1].id.xxhash == 3797888419;
    ok @entry[62].header.date eqv Date.new(2012,9,1);
    ok @entry[62].header.description eqv "Wigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[62].header.important == 0;
    ok @entry[62].id.number eq Array[UInt:D].new(0, 1, 3, 6);
    ok @entry[62].id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].id.xxhash == 3451152685;
    ok @entry[62].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[62].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[62].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[62].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[62].posting[0].amount.asset-quantity == 1;
    ok @entry[62].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[62].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[62].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(209, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[62].posting[0].decinc eqv DecInc::INC;
    ok @entry[62].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[62].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 6);
    ok @entry[62].posting[0].id.entry-id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[0].id.entry-id.xxhash == 3451152685;
    ok @entry[62].posting[0].id.number == 0;
    ok @entry[62].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD";
    ok @entry[62].posting[0].id.xxhash == 1324828912;
    ok @entry[62].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[62].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[62].posting[1].amount.asset-code eqv "USD";
    ok @entry[62].posting[1].amount.asset-quantity == 2.09;
    ok @entry[62].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[62].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[62].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[62].posting[1].decinc eqv DecInc::DEC;
    ok @entry[62].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[62].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 6);
    ok @entry[62].posting[1].id.entry-id.text eqv "2012-09-01 '''\nWigwam LLC bought one wigwam for \$2.09 USD in early September 2012\nFILE:  「includes/2012/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[1].id.entry-id.xxhash == 3451152685;
    ok @entry[62].posting[1].id.number == 1;
    ok @entry[62].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.09 USD";
    ok @entry[62].posting[1].id.xxhash == 243411132;
    ok @entry[63].header.date eqv Date.new(2012,9,16);
    ok @entry[63].header.description eqv "Wigwam LLC received a dividend of \$2012.0916 USD";
    ok @entry[63].header.important == 0;
    ok @entry[63].id.number eq Array[UInt:D].new(0, 1, 3, 7, 0);
    ok @entry[63].id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].id.xxhash == 3511177135;
    ok @entry[63].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[63].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[63].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[63].posting[0].amount.asset-code eqv "USD";
    ok @entry[63].posting[0].amount.asset-quantity == 2012.0916;
    ok @entry[63].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[63].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[63].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[63].posting[0].decinc eqv DecInc::INC;
    ok @entry[63].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[63].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 7, 0);
    ok @entry[63].posting[0].id.entry-id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[0].id.entry-id.xxhash == 3511177135;
    ok @entry[63].posting[0].id.number == 0;
    ok @entry[63].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD";
    ok @entry[63].posting[0].id.xxhash == 2772381157;
    ok @entry[63].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[63].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[63].posting[1].amount.asset-code eqv "USD";
    ok @entry[63].posting[1].amount.asset-quantity == 2012.0916;
    ok @entry[63].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[63].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[63].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[63].posting[1].decinc eqv DecInc::INC;
    ok @entry[63].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[63].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 7, 0);
    ok @entry[63].posting[1].id.entry-id.text eqv "2012-09-16 'Wigwam LLC received a dividend of \$2012.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.0916 USD\n  Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[1].id.entry-id.xxhash == 3511177135;
    ok @entry[63].posting[1].id.number == 1;
    ok @entry[63].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.0916 USD";
    ok @entry[63].posting[1].id.xxhash == 3854836884;
    ok @entry[64].header.date eqv Date.new(2012,9,30);
    ok @entry[64].header.description eqv "Wigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[64].header.important == 0;
    ok @entry[64].id.number eq Array[UInt:D].new(0, 1, 3, 8);
    ok @entry[64].id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].id.xxhash == 1903824288;
    ok @entry[64].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[64].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[64].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[64].posting[0].amount.asset-code eqv "USD";
    ok @entry[64].posting[0].amount.asset-quantity == 2.09;
    ok @entry[64].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[64].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[64].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[64].posting[0].decinc eqv DecInc::INC;
    ok @entry[64].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[64].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 8);
    ok @entry[64].posting[0].id.entry-id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[0].id.entry-id.xxhash == 1903824288;
    ok @entry[64].posting[0].id.number == 0;
    ok @entry[64].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.09 USD";
    ok @entry[64].posting[0].id.xxhash == 1045883229;
    ok @entry[64].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[64].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[64].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[64].posting[1].amount.asset-quantity == 1;
    ok @entry[64].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[64].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[64].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(209, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[64].posting[1].decinc eqv DecInc::DEC;
    ok @entry[64].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[64].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 3, 8);
    ok @entry[64].posting[1].id.entry-id.text eqv "2012-09-30 '''\nWigwam LLC sold one wigwam for \$2.09 USD in late September 2012\nFILE:  「includes/2012/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[1].id.entry-id.xxhash == 1903824288;
    ok @entry[64].posting[1].id.number == 1;
    ok @entry[64].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.09 USD";
    ok @entry[64].posting[1].id.xxhash == 951147748;
    ok @entry[65].header.date eqv Date.new(2012,10,1);
    ok @entry[65].header.description eqv "Wigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[65].header.important == 0;
    ok @entry[65].id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[65].id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].id.xxhash == 3197209626;
    ok @entry[65].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[65].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[65].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[65].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[65].posting[0].amount.asset-quantity == 1;
    ok @entry[65].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[65].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[65].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[65].posting[0].decinc eqv DecInc::INC;
    ok @entry[65].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[65].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[65].posting[0].id.entry-id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[0].id.entry-id.xxhash == 3197209626;
    ok @entry[65].posting[0].id.number == 0;
    ok @entry[65].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD";
    ok @entry[65].posting[0].id.xxhash == 3960426057;
    ok @entry[65].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[65].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[65].posting[1].amount.asset-code eqv "USD";
    ok @entry[65].posting[1].amount.asset-quantity == 2.1;
    ok @entry[65].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[65].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[65].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[65].posting[1].decinc eqv DecInc::DEC;
    ok @entry[65].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[65].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 0);
    ok @entry[65].posting[1].id.entry-id.text eqv "2012-10-01 '''\nWigwam LLC bought one wigwam for \$2.10 USD in early October 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[1].id.entry-id.xxhash == 3197209626;
    ok @entry[65].posting[1].id.number == 1;
    ok @entry[65].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.10 USD";
    ok @entry[65].posting[1].id.xxhash == 2882172050;
    ok @entry[66].header.date eqv Date.new(2012,10,16);
    ok @entry[66].header.description eqv "Wigwam LLC received a dividend of \$2012.1016 USD";
    ok @entry[66].header.important == 0;
    ok @entry[66].id.number eq Array[UInt:D].new(0, 1, 4, 1, 0);
    ok @entry[66].id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].id.xxhash == 518309921;
    ok @entry[66].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[66].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[66].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[66].posting[0].amount.asset-code eqv "USD";
    ok @entry[66].posting[0].amount.asset-quantity == 2012.1016;
    ok @entry[66].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[66].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[66].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[66].posting[0].decinc eqv DecInc::INC;
    ok @entry[66].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[66].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 1, 0);
    ok @entry[66].posting[0].id.entry-id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[0].id.entry-id.xxhash == 518309921;
    ok @entry[66].posting[0].id.number == 0;
    ok @entry[66].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD";
    ok @entry[66].posting[0].id.xxhash == 847137267;
    ok @entry[66].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[66].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[66].posting[1].amount.asset-code eqv "USD";
    ok @entry[66].posting[1].amount.asset-quantity == 2012.1016;
    ok @entry[66].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[66].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[66].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[66].posting[1].decinc eqv DecInc::INC;
    ok @entry[66].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[66].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 1, 0);
    ok @entry[66].posting[1].id.entry-id.text eqv "2012-10-16 'Wigwam LLC received a dividend of \$2012.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1016 USD\n  Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[1].id.entry-id.xxhash == 518309921;
    ok @entry[66].posting[1].id.number == 1;
    ok @entry[66].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1016 USD";
    ok @entry[66].posting[1].id.xxhash == 3525777378;
    ok @entry[67].header.date eqv Date.new(2012,10,31);
    ok @entry[67].header.description eqv "Wigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[67].header.important == 0;
    ok @entry[67].id.number eq Array[UInt:D].new(0, 1, 4, 2);
    ok @entry[67].id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].id.xxhash == 2477064034;
    ok @entry[67].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[67].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[67].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[67].posting[0].amount.asset-code eqv "USD";
    ok @entry[67].posting[0].amount.asset-quantity == 2.1;
    ok @entry[67].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[67].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[67].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[67].posting[0].decinc eqv DecInc::INC;
    ok @entry[67].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[67].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 2);
    ok @entry[67].posting[0].id.entry-id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[0].id.entry-id.xxhash == 2477064034;
    ok @entry[67].posting[0].id.number == 0;
    ok @entry[67].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.10 USD";
    ok @entry[67].posting[0].id.xxhash == 2084107816;
    ok @entry[67].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[67].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[67].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[67].posting[1].amount.asset-quantity == 1;
    ok @entry[67].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[67].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[67].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(21, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[67].posting[1].decinc eqv DecInc::DEC;
    ok @entry[67].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[67].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 2);
    ok @entry[67].posting[1].id.entry-id.text eqv "2012-10-31 '''\nWigwam LLC sold one wigwam for \$2.10 USD in late October 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[1].id.entry-id.xxhash == 2477064034;
    ok @entry[67].posting[1].id.number == 1;
    ok @entry[67].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.10 USD";
    ok @entry[67].posting[1].id.xxhash == 53602413;
    ok @entry[68].header.date eqv Date.new(2012,11,1);
    ok @entry[68].header.description eqv "Wigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[68].header.important == 0;
    ok @entry[68].id.number eq Array[UInt:D].new(0, 1, 4, 3);
    ok @entry[68].id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].id.xxhash == 729373328;
    ok @entry[68].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[68].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[68].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[68].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[68].posting[0].amount.asset-quantity == 1;
    ok @entry[68].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[68].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[68].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(211, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[68].posting[0].decinc eqv DecInc::INC;
    ok @entry[68].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[68].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 3);
    ok @entry[68].posting[0].id.entry-id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[0].id.entry-id.xxhash == 729373328;
    ok @entry[68].posting[0].id.number == 0;
    ok @entry[68].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD";
    ok @entry[68].posting[0].id.xxhash == 3794157360;
    ok @entry[68].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[68].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[68].posting[1].amount.asset-code eqv "USD";
    ok @entry[68].posting[1].amount.asset-quantity == 2.11;
    ok @entry[68].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[68].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[68].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[68].posting[1].decinc eqv DecInc::DEC;
    ok @entry[68].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[68].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 3);
    ok @entry[68].posting[1].id.entry-id.text eqv "2012-11-01 '''\nWigwam LLC bought one wigwam for \$2.11 USD in early November 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[1].id.entry-id.xxhash == 729373328;
    ok @entry[68].posting[1].id.number == 1;
    ok @entry[68].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.11 USD";
    ok @entry[68].posting[1].id.xxhash == 974733802;
    ok @entry[69].header.date eqv Date.new(2012,11,16);
    ok @entry[69].header.description eqv "Wigwam LLC received a dividend of \$2012.1116 USD";
    ok @entry[69].header.important == 0;
    ok @entry[69].id.number eq Array[UInt:D].new(0, 1, 4, 4, 0);
    ok @entry[69].id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].id.xxhash == 2908538066;
    ok @entry[69].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[69].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[69].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[69].posting[0].amount.asset-code eqv "USD";
    ok @entry[69].posting[0].amount.asset-quantity == 2012.1116;
    ok @entry[69].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[69].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[69].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[69].posting[0].decinc eqv DecInc::INC;
    ok @entry[69].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[69].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 4, 0);
    ok @entry[69].posting[0].id.entry-id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[0].id.entry-id.xxhash == 2908538066;
    ok @entry[69].posting[0].id.number == 0;
    ok @entry[69].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD";
    ok @entry[69].posting[0].id.xxhash == 3423083241;
    ok @entry[69].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[69].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[69].posting[1].amount.asset-code eqv "USD";
    ok @entry[69].posting[1].amount.asset-quantity == 2012.1116;
    ok @entry[69].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[69].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[69].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[69].posting[1].decinc eqv DecInc::INC;
    ok @entry[69].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[69].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 4, 0);
    ok @entry[69].posting[1].id.entry-id.text eqv "2012-11-16 'Wigwam LLC received a dividend of \$2012.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1116 USD\n  Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[1].id.entry-id.xxhash == 2908538066;
    ok @entry[69].posting[1].id.number == 1;
    ok @entry[69].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1116 USD";
    ok @entry[69].posting[1].id.xxhash == 400356518;
    ok @entry[70].header.date eqv Date.new(2012,11,30);
    ok @entry[70].header.description eqv "Wigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[70].header.important == 0;
    ok @entry[70].id.number eq Array[UInt:D].new(0, 1, 4, 5);
    ok @entry[70].id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].id.xxhash == 228106705;
    ok @entry[70].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[70].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[70].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[70].posting[0].amount.asset-code eqv "USD";
    ok @entry[70].posting[0].amount.asset-quantity == 2.11;
    ok @entry[70].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[70].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[70].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[70].posting[0].decinc eqv DecInc::INC;
    ok @entry[70].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[70].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 5);
    ok @entry[70].posting[0].id.entry-id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[0].id.entry-id.xxhash == 228106705;
    ok @entry[70].posting[0].id.number == 0;
    ok @entry[70].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.11 USD";
    ok @entry[70].posting[0].id.xxhash == 2795591875;
    ok @entry[70].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[70].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[70].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[70].posting[1].amount.asset-quantity == 1;
    ok @entry[70].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[70].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[70].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(211, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[70].posting[1].decinc eqv DecInc::DEC;
    ok @entry[70].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[70].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 5);
    ok @entry[70].posting[1].id.entry-id.text eqv "2012-11-30 '''\nWigwam LLC sold one wigwam for \$2.11 USD in late November 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[1].id.entry-id.xxhash == 228106705;
    ok @entry[70].posting[1].id.number == 1;
    ok @entry[70].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.11 USD";
    ok @entry[70].posting[1].id.xxhash == 1515324218;
    ok @entry[71].header.date eqv Date.new(2012,12,1);
    ok @entry[71].header.description eqv "Wigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[71].header.important == 0;
    ok @entry[71].id.number eq Array[UInt:D].new(0, 1, 4, 6);
    ok @entry[71].id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].id.xxhash == 118051768;
    ok @entry[71].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[71].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[71].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[71].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[71].posting[0].amount.asset-quantity == 1;
    ok @entry[71].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[71].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[71].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[71].posting[0].decinc eqv DecInc::INC;
    ok @entry[71].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[71].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 6);
    ok @entry[71].posting[0].id.entry-id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[0].id.entry-id.xxhash == 118051768;
    ok @entry[71].posting[0].id.number == 0;
    ok @entry[71].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD";
    ok @entry[71].posting[0].id.xxhash == 228959427;
    ok @entry[71].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[71].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[71].posting[1].amount.asset-code eqv "USD";
    ok @entry[71].posting[1].amount.asset-quantity == 2.12;
    ok @entry[71].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[71].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[71].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[71].posting[1].decinc eqv DecInc::DEC;
    ok @entry[71].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[71].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 6);
    ok @entry[71].posting[1].id.entry-id.text eqv "2012-12-01 '''\nWigwam LLC bought one wigwam for \$2.12 USD in early December 2012\nFILE:  「includes/2012/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 2.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[1].id.entry-id.xxhash == 118051768;
    ok @entry[71].posting[1].id.number == 1;
    ok @entry[71].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -2.12 USD";
    ok @entry[71].posting[1].id.xxhash == 1223007802;
    ok @entry[72].header.date eqv Date.new(2012,12,16);
    ok @entry[72].header.description eqv "Wigwam LLC received a dividend of \$2012.1216 USD";
    ok @entry[72].header.important == 0;
    ok @entry[72].id.number eq Array[UInt:D].new(0, 1, 4, 7, 0);
    ok @entry[72].id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].id.xxhash == 3874590774;
    ok @entry[72].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[72].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[72].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[72].posting[0].amount.asset-code eqv "USD";
    ok @entry[72].posting[0].amount.asset-quantity == 2012.1216;
    ok @entry[72].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[72].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[72].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[72].posting[0].decinc eqv DecInc::INC;
    ok @entry[72].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[72].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 7, 0);
    ok @entry[72].posting[0].id.entry-id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[0].id.entry-id.xxhash == 3874590774;
    ok @entry[72].posting[0].id.number == 0;
    ok @entry[72].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD";
    ok @entry[72].posting[0].id.xxhash == 3994190483;
    ok @entry[72].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[72].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[72].posting[1].amount.asset-code eqv "USD";
    ok @entry[72].posting[1].amount.asset-quantity == 2012.1216;
    ok @entry[72].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[72].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[72].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[72].posting[1].decinc eqv DecInc::INC;
    ok @entry[72].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[72].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 7, 0);
    ok @entry[72].posting[1].id.entry-id.text eqv "2012-12-16 'Wigwam LLC received a dividend of \$2012.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2012.1216 USD\n  Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[1].id.entry-id.xxhash == 3874590774;
    ok @entry[72].posting[1].id.number == 1;
    ok @entry[72].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2012.1216 USD";
    ok @entry[72].posting[1].id.xxhash == 3325655088;
    ok @entry[73].header.date eqv Date.new(2012,12,31);
    ok @entry[73].header.description eqv "Wigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[73].header.important == 0;
    ok @entry[73].id.number eq Array[UInt:D].new(0, 1, 4, 8);
    ok @entry[73].id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].id.xxhash == 2276583731;
    ok @entry[73].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[73].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[73].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[73].posting[0].amount.asset-code eqv "USD";
    ok @entry[73].posting[0].amount.asset-quantity == 2.12;
    ok @entry[73].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[73].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[73].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[73].posting[0].decinc eqv DecInc::INC;
    ok @entry[73].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[73].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 8);
    ok @entry[73].posting[0].id.entry-id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[0].id.entry-id.xxhash == 2276583731;
    ok @entry[73].posting[0].id.number == 0;
    ok @entry[73].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       2.12 USD";
    ok @entry[73].posting[0].id.xxhash == 2132552848;
    ok @entry[73].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[73].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[73].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[73].posting[1].amount.asset-quantity == 1;
    ok @entry[73].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[73].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[73].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(53, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[73].posting[1].decinc eqv DecInc::DEC;
    ok @entry[73].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[73].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 1, 4, 8);
    ok @entry[73].posting[1].id.entry-id.text eqv "2012-12-31 '''\nWigwam LLC sold one wigwam for \$2.12 USD in late December 2012\nFILE:  「includes/2012/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       2.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[1].id.entry-id.xxhash == 2276583731;
    ok @entry[73].posting[1].id.number == 1;
    ok @entry[73].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 2.12 USD";
    ok @entry[73].posting[1].id.xxhash == 3096402053;
    ok @entry[74].header.date eqv Date.new(2013,1,1);
    ok @entry[74].header.description eqv "FooCorp started the year with \$1000 in Bankwest cheque account";
    ok @entry[74].header.important == 0;
    ok @entry[74].id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[74].id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].id.xxhash == 2316435396;
    ok @entry[74].posting[0].account.entity eqv "FooCorp";
    ok @entry[74].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[74].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[74].posting[0].amount.asset-code eqv "USD";
    ok @entry[74].posting[0].amount.asset-quantity == 1000;
    ok @entry[74].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[74].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[74].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[74].posting[0].decinc eqv DecInc::INC;
    ok @entry[74].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[74].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[74].posting[0].id.entry-id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[0].id.entry-id.xxhash == 2316435396;
    ok @entry[74].posting[0].id.number == 0;
    ok @entry[74].posting[0].id.text eqv "Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[74].posting[0].id.xxhash == 3244003616;
    ok @entry[74].posting[1].account.entity eqv "FooCorp";
    ok @entry[74].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[74].posting[1].amount.asset-code eqv "USD";
    ok @entry[74].posting[1].amount.asset-quantity == 1000;
    ok @entry[74].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[74].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[74].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[74].posting[1].decinc eqv DecInc::INC;
    ok @entry[74].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[74].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 0);
    ok @entry[74].posting[1].id.entry-id.text eqv "2013-01-01 \"FooCorp started the year with \$1000 in Bankwest cheque account\"\n  Assets:FooCorp:Bankwest:Cheque      \$1000.00 USD\n  Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[1].id.entry-id.xxhash == 2316435396;
    ok @entry[74].posting[1].id.number == 1;
    ok @entry[74].posting[1].id.text eqv "Equity:FooCorp                      \$1000.00 USD";
    ok @entry[74].posting[1].id.xxhash == 1025058054;
    ok @entry[75].header.date eqv Date.new(2013,1,1);
    ok @entry[75].header.description eqv "Wigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n";
    ok @entry[75].header.important == 0;
    ok @entry[75].id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[75].id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].id.xxhash == 265230743;
    ok @entry[75].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[75].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[75].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[75].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[75].posting[0].amount.asset-quantity == 1;
    ok @entry[75].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[75].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[75].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(301, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[75].posting[0].decinc eqv DecInc::INC;
    ok @entry[75].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[75].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[75].posting[0].id.entry-id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[0].id.entry-id.xxhash == 265230743;
    ok @entry[75].posting[0].id.number == 0;
    ok @entry[75].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD";
    ok @entry[75].posting[0].id.xxhash == 1130731432;
    ok @entry[75].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[75].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[75].posting[1].amount.asset-code eqv "USD";
    ok @entry[75].posting[1].amount.asset-quantity == 3.01;
    ok @entry[75].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[75].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[75].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[75].posting[1].decinc eqv DecInc::DEC;
    ok @entry[75].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[75].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 0);
    ok @entry[75].posting[1].id.entry-id.text eqv "2013-01-01 '''\nWigwam LLC bought one wigwam for \$3.01 USD in early January 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/January'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.01 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[1].id.entry-id.xxhash == 265230743;
    ok @entry[75].posting[1].id.number == 1;
    ok @entry[75].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.01 USD";
    ok @entry[75].posting[1].id.xxhash == 35511673;
    ok @entry[76].header.date eqv Date.new(2013,1,16);
    ok @entry[76].header.description eqv "Wigwam LLC received a dividend of \$2013.0116 USD";
    ok @entry[76].header.important == 0;
    ok @entry[76].id.number eq Array[UInt:D].new(0, 2, 1, 1, 0);
    ok @entry[76].id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].id.xxhash == 1201569906;
    ok @entry[76].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[76].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[76].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[76].posting[0].amount.asset-code eqv "USD";
    ok @entry[76].posting[0].amount.asset-quantity == 2013.0116;
    ok @entry[76].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[76].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[76].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[76].posting[0].decinc eqv DecInc::INC;
    ok @entry[76].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[76].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 1, 0);
    ok @entry[76].posting[0].id.entry-id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[0].id.entry-id.xxhash == 1201569906;
    ok @entry[76].posting[0].id.number == 0;
    ok @entry[76].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD";
    ok @entry[76].posting[0].id.xxhash == 4225789096;
    ok @entry[76].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[76].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[76].posting[1].amount.asset-code eqv "USD";
    ok @entry[76].posting[1].amount.asset-quantity == 2013.0116;
    ok @entry[76].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[76].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[76].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[76].posting[1].decinc eqv DecInc::INC;
    ok @entry[76].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[76].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 1, 0);
    ok @entry[76].posting[1].id.entry-id.text eqv "2013-01-16 'Wigwam LLC received a dividend of \$2013.0116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0116 USD\n  Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[1].id.entry-id.xxhash == 1201569906;
    ok @entry[76].posting[1].id.number == 1;
    ok @entry[76].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0116 USD";
    ok @entry[76].posting[1].id.xxhash == 1105245657;
    ok @entry[77].header.date eqv Date.new(2013,1,31);
    ok @entry[77].header.description eqv "Wigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n";
    ok @entry[77].header.important == 0;
    ok @entry[77].id.number eq Array[UInt:D].new(0, 2, 1, 2);
    ok @entry[77].id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].id.xxhash == 3717945709;
    ok @entry[77].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[77].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[77].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[77].posting[0].amount.asset-code eqv "USD";
    ok @entry[77].posting[0].amount.asset-quantity == 3.01;
    ok @entry[77].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[77].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[77].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[77].posting[0].decinc eqv DecInc::INC;
    ok @entry[77].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[77].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 2);
    ok @entry[77].posting[0].id.entry-id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[0].id.entry-id.xxhash == 3717945709;
    ok @entry[77].posting[0].id.number == 0;
    ok @entry[77].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.01 USD";
    ok @entry[77].posting[0].id.xxhash == 2990923830;
    ok @entry[77].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[77].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[77].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[77].posting[1].amount.asset-quantity == 1;
    ok @entry[77].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[77].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[77].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(301, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[77].posting[1].decinc eqv DecInc::DEC;
    ok @entry[77].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[77].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 2);
    ok @entry[77].posting[1].id.entry-id.text eqv "2013-01-31 '''\nWigwam LLC sold one wigwam for \$3.01 USD in late January 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/January'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.01 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[1].id.entry-id.xxhash == 3717945709;
    ok @entry[77].posting[1].id.number == 1;
    ok @entry[77].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.01 USD";
    ok @entry[77].posting[1].id.xxhash == 1596641458;
    ok @entry[78].header.date eqv Date.new(2013,2,1);
    ok @entry[78].header.description eqv "Wigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n";
    ok @entry[78].header.important == 0;
    ok @entry[78].id.number eq Array[UInt:D].new(0, 2, 1, 3);
    ok @entry[78].id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].id.xxhash == 4024926956;
    ok @entry[78].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[78].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[78].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[78].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[78].posting[0].amount.asset-quantity == 1;
    ok @entry[78].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[78].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[78].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(151, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[78].posting[0].decinc eqv DecInc::INC;
    ok @entry[78].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[78].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 3);
    ok @entry[78].posting[0].id.entry-id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[0].id.entry-id.xxhash == 4024926956;
    ok @entry[78].posting[0].id.number == 0;
    ok @entry[78].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD";
    ok @entry[78].posting[0].id.xxhash == 1475938137;
    ok @entry[78].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[78].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[78].posting[1].amount.asset-code eqv "USD";
    ok @entry[78].posting[1].amount.asset-quantity == 3.02;
    ok @entry[78].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[78].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[78].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[78].posting[1].decinc eqv DecInc::DEC;
    ok @entry[78].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[78].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 3);
    ok @entry[78].posting[1].id.entry-id.text eqv "2013-02-01 '''\nWigwam LLC bought one wigwam for \$3.02 USD in early February 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/February'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.02 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[1].id.entry-id.xxhash == 4024926956;
    ok @entry[78].posting[1].id.number == 1;
    ok @entry[78].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.02 USD";
    ok @entry[78].posting[1].id.xxhash == 3350274487;
    ok @entry[79].header.date eqv Date.new(2013,2,16);
    ok @entry[79].header.description eqv "Wigwam LLC received a dividend of \$2013.0216 USD";
    ok @entry[79].header.important == 0;
    ok @entry[79].id.number eq Array[UInt:D].new(0, 2, 1, 4, 0);
    ok @entry[79].id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].id.xxhash == 2203509452;
    ok @entry[79].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[79].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[79].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[79].posting[0].amount.asset-code eqv "USD";
    ok @entry[79].posting[0].amount.asset-quantity == 2013.0216;
    ok @entry[79].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[79].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[79].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[79].posting[0].decinc eqv DecInc::INC;
    ok @entry[79].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[79].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 4, 0);
    ok @entry[79].posting[0].id.entry-id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[0].id.entry-id.xxhash == 2203509452;
    ok @entry[79].posting[0].id.number == 0;
    ok @entry[79].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD";
    ok @entry[79].posting[0].id.xxhash == 2242748384;
    ok @entry[79].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[79].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[79].posting[1].amount.asset-code eqv "USD";
    ok @entry[79].posting[1].amount.asset-quantity == 2013.0216;
    ok @entry[79].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[79].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[79].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[79].posting[1].decinc eqv DecInc::INC;
    ok @entry[79].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[79].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 4, 0);
    ok @entry[79].posting[1].id.entry-id.text eqv "2013-02-16 'Wigwam LLC received a dividend of \$2013.0216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0216 USD\n  Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[1].id.entry-id.xxhash == 2203509452;
    ok @entry[79].posting[1].id.number == 1;
    ok @entry[79].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0216 USD";
    ok @entry[79].posting[1].id.xxhash == 2807440849;
    ok @entry[80].header.date eqv Date.new(2013,2,28);
    ok @entry[80].header.description eqv "Wigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n";
    ok @entry[80].header.important == 0;
    ok @entry[80].id.number eq Array[UInt:D].new(0, 2, 1, 5);
    ok @entry[80].id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].id.xxhash == 1337772394;
    ok @entry[80].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[80].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[80].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[80].posting[0].amount.asset-code eqv "USD";
    ok @entry[80].posting[0].amount.asset-quantity == 3.02;
    ok @entry[80].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[80].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[80].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[80].posting[0].decinc eqv DecInc::INC;
    ok @entry[80].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[80].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 5);
    ok @entry[80].posting[0].id.entry-id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[0].id.entry-id.xxhash == 1337772394;
    ok @entry[80].posting[0].id.number == 0;
    ok @entry[80].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.02 USD";
    ok @entry[80].posting[0].id.xxhash == 1829432264;
    ok @entry[80].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[80].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[80].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[80].posting[1].amount.asset-quantity == 1;
    ok @entry[80].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[80].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[80].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(151, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[80].posting[1].decinc eqv DecInc::DEC;
    ok @entry[80].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[80].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 5);
    ok @entry[80].posting[1].id.entry-id.text eqv "2013-02-28 '''\nWigwam LLC sold one wigwam for \$3.02 USD in late February 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/February'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.02 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[1].id.entry-id.xxhash == 1337772394;
    ok @entry[80].posting[1].id.number == 1;
    ok @entry[80].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.02 USD";
    ok @entry[80].posting[1].id.xxhash == 1035364396;
    ok @entry[81].header.date eqv Date.new(2013,3,1);
    ok @entry[81].header.description eqv "Wigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n";
    ok @entry[81].header.important == 0;
    ok @entry[81].id.number eq Array[UInt:D].new(0, 2, 1, 6);
    ok @entry[81].id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].id.xxhash == 3993675371;
    ok @entry[81].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[81].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[81].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[81].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[81].posting[0].amount.asset-quantity == 1;
    ok @entry[81].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[81].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[81].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(303, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[81].posting[0].decinc eqv DecInc::INC;
    ok @entry[81].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[81].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 6);
    ok @entry[81].posting[0].id.entry-id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[0].id.entry-id.xxhash == 3993675371;
    ok @entry[81].posting[0].id.number == 0;
    ok @entry[81].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD";
    ok @entry[81].posting[0].id.xxhash == 3084675232;
    ok @entry[81].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[81].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[81].posting[1].amount.asset-code eqv "USD";
    ok @entry[81].posting[1].amount.asset-quantity == 3.03;
    ok @entry[81].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[81].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[81].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[81].posting[1].decinc eqv DecInc::DEC;
    ok @entry[81].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[81].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 6);
    ok @entry[81].posting[1].id.entry-id.text eqv "2013-03-01 '''\nWigwam LLC bought one wigwam for \$3.03 USD in early March 2013\nFILE:  「includes/2013/Q1.txn」\nBEFORE:「include 'Q1/March'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.03 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[1].id.entry-id.xxhash == 3993675371;
    ok @entry[81].posting[1].id.number == 1;
    ok @entry[81].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.03 USD";
    ok @entry[81].posting[1].id.xxhash == 1840852121;
    ok @entry[82].header.date eqv Date.new(2013,3,16);
    ok @entry[82].header.description eqv "Wigwam LLC received a dividend of \$2013.0316 USD";
    ok @entry[82].header.important == 0;
    ok @entry[82].id.number eq Array[UInt:D].new(0, 2, 1, 7, 0);
    ok @entry[82].id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].id.xxhash == 1562886410;
    ok @entry[82].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[82].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[82].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[82].posting[0].amount.asset-code eqv "USD";
    ok @entry[82].posting[0].amount.asset-quantity == 2013.0316;
    ok @entry[82].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[82].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[82].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[82].posting[0].decinc eqv DecInc::INC;
    ok @entry[82].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[82].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 7, 0);
    ok @entry[82].posting[0].id.entry-id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[0].id.entry-id.xxhash == 1562886410;
    ok @entry[82].posting[0].id.number == 0;
    ok @entry[82].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD";
    ok @entry[82].posting[0].id.xxhash == 4042680049;
    ok @entry[82].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[82].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[82].posting[1].amount.asset-code eqv "USD";
    ok @entry[82].posting[1].amount.asset-quantity == 2013.0316;
    ok @entry[82].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[82].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[82].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[82].posting[1].decinc eqv DecInc::INC;
    ok @entry[82].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[82].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 7, 0);
    ok @entry[82].posting[1].id.entry-id.text eqv "2013-03-16 'Wigwam LLC received a dividend of \$2013.0316 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0316 USD\n  Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[1].id.entry-id.xxhash == 1562886410;
    ok @entry[82].posting[1].id.number == 1;
    ok @entry[82].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0316 USD";
    ok @entry[82].posting[1].id.xxhash == 1244356843;
    ok @entry[83].header.date eqv Date.new(2013,3,31);
    ok @entry[83].header.description eqv "Wigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n";
    ok @entry[83].header.important == 0;
    ok @entry[83].id.number eq Array[UInt:D].new(0, 2, 1, 8);
    ok @entry[83].id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].id.xxhash == 1634210627;
    ok @entry[83].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[83].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[83].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[83].posting[0].amount.asset-code eqv "USD";
    ok @entry[83].posting[0].amount.asset-quantity == 3.03;
    ok @entry[83].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[83].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[83].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[83].posting[0].decinc eqv DecInc::INC;
    ok @entry[83].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[83].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 8);
    ok @entry[83].posting[0].id.entry-id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[0].id.entry-id.xxhash == 1634210627;
    ok @entry[83].posting[0].id.number == 0;
    ok @entry[83].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.03 USD";
    ok @entry[83].posting[0].id.xxhash == 2812452536;
    ok @entry[83].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[83].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[83].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[83].posting[1].amount.asset-quantity == 1;
    ok @entry[83].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[83].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[83].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(303, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[83].posting[1].decinc eqv DecInc::DEC;
    ok @entry[83].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[83].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 1, 8);
    ok @entry[83].posting[1].id.entry-id.text eqv "2013-03-31 '''\nWigwam LLC sold one wigwam for \$3.03 USD in late March 2013\nFILE:  「includes/2013/Q1.txn」\nAFTER: 「include 'Q1/March'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.03 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[1].id.entry-id.xxhash == 1634210627;
    ok @entry[83].posting[1].id.number == 1;
    ok @entry[83].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.03 USD";
    ok @entry[83].posting[1].id.xxhash == 3941709198;
    ok @entry[84].header.date eqv Date.new(2013,4,1);
    ok @entry[84].header.description eqv "Wigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n";
    ok @entry[84].header.important == 0;
    ok @entry[84].id.number eq Array[UInt:D].new(0, 2, 2, 0);
    ok @entry[84].id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].id.xxhash == 713752082;
    ok @entry[84].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[84].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[84].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[84].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[84].posting[0].amount.asset-quantity == 1;
    ok @entry[84].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[84].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[84].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(76, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[84].posting[0].decinc eqv DecInc::INC;
    ok @entry[84].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[84].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 0);
    ok @entry[84].posting[0].id.entry-id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[0].id.entry-id.xxhash == 713752082;
    ok @entry[84].posting[0].id.number == 0;
    ok @entry[84].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD";
    ok @entry[84].posting[0].id.xxhash == 2233409856;
    ok @entry[84].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[84].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[84].posting[1].amount.asset-code eqv "USD";
    ok @entry[84].posting[1].amount.asset-quantity == 3.04;
    ok @entry[84].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[84].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[84].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[84].posting[1].decinc eqv DecInc::DEC;
    ok @entry[84].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[84].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 0);
    ok @entry[84].posting[1].id.entry-id.text eqv "2013-04-01 '''\nWigwam LLC bought one wigwam for \$3.04 USD in early April 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/April'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.04 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[1].id.entry-id.xxhash == 713752082;
    ok @entry[84].posting[1].id.number == 1;
    ok @entry[84].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.04 USD";
    ok @entry[84].posting[1].id.xxhash == 2267252423;
    ok @entry[85].header.date eqv Date.new(2013,4,16);
    ok @entry[85].header.description eqv "Wigwam LLC received a dividend of \$2013.0416 USD";
    ok @entry[85].header.important == 0;
    ok @entry[85].id.number eq Array[UInt:D].new(0, 2, 2, 1, 0);
    ok @entry[85].id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].id.xxhash == 2573101041;
    ok @entry[85].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[85].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[85].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[85].posting[0].amount.asset-code eqv "USD";
    ok @entry[85].posting[0].amount.asset-quantity == 2013.0416;
    ok @entry[85].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[85].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[85].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[85].posting[0].decinc eqv DecInc::INC;
    ok @entry[85].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[85].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 1, 0);
    ok @entry[85].posting[0].id.entry-id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[0].id.entry-id.xxhash == 2573101041;
    ok @entry[85].posting[0].id.number == 0;
    ok @entry[85].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD";
    ok @entry[85].posting[0].id.xxhash == 3978994459;
    ok @entry[85].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[85].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[85].posting[1].amount.asset-code eqv "USD";
    ok @entry[85].posting[1].amount.asset-quantity == 2013.0416;
    ok @entry[85].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[85].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[85].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[85].posting[1].decinc eqv DecInc::INC;
    ok @entry[85].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[85].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 1, 0);
    ok @entry[85].posting[1].id.entry-id.text eqv "2013-04-16 'Wigwam LLC received a dividend of \$2013.0416 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0416 USD\n  Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[1].id.entry-id.xxhash == 2573101041;
    ok @entry[85].posting[1].id.number == 1;
    ok @entry[85].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0416 USD";
    ok @entry[85].posting[1].id.xxhash == 419651900;
    ok @entry[86].header.date eqv Date.new(2013,4,30);
    ok @entry[86].header.description eqv "Wigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n";
    ok @entry[86].header.important == 0;
    ok @entry[86].id.number eq Array[UInt:D].new(0, 2, 2, 2);
    ok @entry[86].id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].id.xxhash == 2698432083;
    ok @entry[86].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[86].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[86].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[86].posting[0].amount.asset-code eqv "USD";
    ok @entry[86].posting[0].amount.asset-quantity == 3.04;
    ok @entry[86].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[86].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[86].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[86].posting[0].decinc eqv DecInc::INC;
    ok @entry[86].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[86].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 2);
    ok @entry[86].posting[0].id.entry-id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[0].id.entry-id.xxhash == 2698432083;
    ok @entry[86].posting[0].id.number == 0;
    ok @entry[86].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.04 USD";
    ok @entry[86].posting[0].id.xxhash == 170876880;
    ok @entry[86].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[86].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[86].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[86].posting[1].amount.asset-quantity == 1;
    ok @entry[86].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[86].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[86].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(76, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[86].posting[1].decinc eqv DecInc::DEC;
    ok @entry[86].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[86].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 2);
    ok @entry[86].posting[1].id.entry-id.text eqv "2013-04-30 '''\nWigwam LLC sold one wigwam for \$3.04 USD in late April 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/April'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.04 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[1].id.entry-id.xxhash == 2698432083;
    ok @entry[86].posting[1].id.number == 1;
    ok @entry[86].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.04 USD";
    ok @entry[86].posting[1].id.xxhash == 3893093683;
    ok @entry[87].header.date eqv Date.new(2013,5,1);
    ok @entry[87].header.description eqv "Wigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n";
    ok @entry[87].header.important == 0;
    ok @entry[87].id.number eq Array[UInt:D].new(0, 2, 2, 3);
    ok @entry[87].id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].id.xxhash == 607841021;
    ok @entry[87].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[87].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[87].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[87].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[87].posting[0].amount.asset-quantity == 1;
    ok @entry[87].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[87].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[87].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(61, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[87].posting[0].decinc eqv DecInc::INC;
    ok @entry[87].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[87].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 3);
    ok @entry[87].posting[0].id.entry-id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[0].id.entry-id.xxhash == 607841021;
    ok @entry[87].posting[0].id.number == 0;
    ok @entry[87].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD";
    ok @entry[87].posting[0].id.xxhash == 3108085666;
    ok @entry[87].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[87].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[87].posting[1].amount.asset-code eqv "USD";
    ok @entry[87].posting[1].amount.asset-quantity == 3.05;
    ok @entry[87].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[87].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[87].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[87].posting[1].decinc eqv DecInc::DEC;
    ok @entry[87].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[87].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 3);
    ok @entry[87].posting[1].id.entry-id.text eqv "2013-05-01 '''\nWigwam LLC bought one wigwam for \$3.05 USD in early May 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/May'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.05 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[1].id.entry-id.xxhash == 607841021;
    ok @entry[87].posting[1].id.number == 1;
    ok @entry[87].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.05 USD";
    ok @entry[87].posting[1].id.xxhash == 2590724923;
    ok @entry[88].header.date eqv Date.new(2013,5,16);
    ok @entry[88].header.description eqv "Wigwam LLC received a dividend of \$2013.0516 USD";
    ok @entry[88].header.important == 0;
    ok @entry[88].id.number eq Array[UInt:D].new(0, 2, 2, 4, 0);
    ok @entry[88].id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].id.xxhash == 3814087496;
    ok @entry[88].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[88].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[88].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[88].posting[0].amount.asset-code eqv "USD";
    ok @entry[88].posting[0].amount.asset-quantity == 2013.0516;
    ok @entry[88].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[88].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[88].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[88].posting[0].decinc eqv DecInc::INC;
    ok @entry[88].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[88].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 4, 0);
    ok @entry[88].posting[0].id.entry-id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[0].id.entry-id.xxhash == 3814087496;
    ok @entry[88].posting[0].id.number == 0;
    ok @entry[88].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD";
    ok @entry[88].posting[0].id.xxhash == 2233760271;
    ok @entry[88].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[88].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[88].posting[1].amount.asset-code eqv "USD";
    ok @entry[88].posting[1].amount.asset-quantity == 2013.0516;
    ok @entry[88].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[88].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[88].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[88].posting[1].decinc eqv DecInc::INC;
    ok @entry[88].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[88].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 4, 0);
    ok @entry[88].posting[1].id.entry-id.text eqv "2013-05-16 'Wigwam LLC received a dividend of \$2013.0516 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0516 USD\n  Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[1].id.entry-id.xxhash == 3814087496;
    ok @entry[88].posting[1].id.number == 1;
    ok @entry[88].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0516 USD";
    ok @entry[88].posting[1].id.xxhash == 1692039403;
    ok @entry[89].header.date eqv Date.new(2013,5,31);
    ok @entry[89].header.description eqv "Wigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n";
    ok @entry[89].header.important == 0;
    ok @entry[89].id.number eq Array[UInt:D].new(0, 2, 2, 5);
    ok @entry[89].id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].id.xxhash == 3166921593;
    ok @entry[89].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[89].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[89].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[89].posting[0].amount.asset-code eqv "USD";
    ok @entry[89].posting[0].amount.asset-quantity == 3.05;
    ok @entry[89].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[89].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[89].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[89].posting[0].decinc eqv DecInc::INC;
    ok @entry[89].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[89].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 5);
    ok @entry[89].posting[0].id.entry-id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[0].id.entry-id.xxhash == 3166921593;
    ok @entry[89].posting[0].id.number == 0;
    ok @entry[89].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.05 USD";
    ok @entry[89].posting[0].id.xxhash == 2097158499;
    ok @entry[89].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[89].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[89].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[89].posting[1].amount.asset-quantity == 1;
    ok @entry[89].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[89].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[89].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(61, 20), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[89].posting[1].decinc eqv DecInc::DEC;
    ok @entry[89].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[89].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 5);
    ok @entry[89].posting[1].id.entry-id.text eqv "2013-05-31 '''\nWigwam LLC sold one wigwam for \$3.05 USD in late May 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/May'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.05 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[1].id.entry-id.xxhash == 3166921593;
    ok @entry[89].posting[1].id.number == 1;
    ok @entry[89].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.05 USD";
    ok @entry[89].posting[1].id.xxhash == 895907392;
    ok @entry[90].header.date eqv Date.new(2013,6,1);
    ok @entry[90].header.description eqv "Wigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n";
    ok @entry[90].header.important == 0;
    ok @entry[90].id.number eq Array[UInt:D].new(0, 2, 2, 6);
    ok @entry[90].id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].id.xxhash == 927323051;
    ok @entry[90].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[90].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[90].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[90].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[90].posting[0].amount.asset-quantity == 1;
    ok @entry[90].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[90].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[90].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(153, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[90].posting[0].decinc eqv DecInc::INC;
    ok @entry[90].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[90].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 6);
    ok @entry[90].posting[0].id.entry-id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[0].id.entry-id.xxhash == 927323051;
    ok @entry[90].posting[0].id.number == 0;
    ok @entry[90].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD";
    ok @entry[90].posting[0].id.xxhash == 3153491063;
    ok @entry[90].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[90].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[90].posting[1].amount.asset-code eqv "USD";
    ok @entry[90].posting[1].amount.asset-quantity == 3.06;
    ok @entry[90].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[90].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[90].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[90].posting[1].decinc eqv DecInc::DEC;
    ok @entry[90].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[90].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 6);
    ok @entry[90].posting[1].id.entry-id.text eqv "2013-06-01 '''\nWigwam LLC bought one wigwam for \$3.06 USD in early June 2013\nFILE:  「includes/2013/Q2.txn」\nBEFORE:「include 'Q2/June'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.06 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[1].id.entry-id.xxhash == 927323051;
    ok @entry[90].posting[1].id.number == 1;
    ok @entry[90].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.06 USD";
    ok @entry[90].posting[1].id.xxhash == 1187163231;
    ok @entry[91].header.date eqv Date.new(2013,6,16);
    ok @entry[91].header.description eqv "Wigwam LLC received a dividend of \$2013.0616 USD";
    ok @entry[91].header.important == 0;
    ok @entry[91].id.number eq Array[UInt:D].new(0, 2, 2, 7, 0);
    ok @entry[91].id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].id.xxhash == 2504499649;
    ok @entry[91].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[91].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[91].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[91].posting[0].amount.asset-code eqv "USD";
    ok @entry[91].posting[0].amount.asset-quantity == 2013.0616;
    ok @entry[91].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[91].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[91].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[91].posting[0].decinc eqv DecInc::INC;
    ok @entry[91].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[91].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 7, 0);
    ok @entry[91].posting[0].id.entry-id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[0].id.entry-id.xxhash == 2504499649;
    ok @entry[91].posting[0].id.number == 0;
    ok @entry[91].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD";
    ok @entry[91].posting[0].id.xxhash == 2945489261;
    ok @entry[91].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[91].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[91].posting[1].amount.asset-code eqv "USD";
    ok @entry[91].posting[1].amount.asset-quantity == 2013.0616;
    ok @entry[91].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[91].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[91].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[91].posting[1].decinc eqv DecInc::INC;
    ok @entry[91].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[91].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 7, 0);
    ok @entry[91].posting[1].id.entry-id.text eqv "2013-06-16 'Wigwam LLC received a dividend of \$2013.0616 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0616 USD\n  Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[1].id.entry-id.xxhash == 2504499649;
    ok @entry[91].posting[1].id.number == 1;
    ok @entry[91].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0616 USD";
    ok @entry[91].posting[1].id.xxhash == 398263546;
    ok @entry[92].header.date eqv Date.new(2013,6,30);
    ok @entry[92].header.description eqv "Wigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n";
    ok @entry[92].header.important == 0;
    ok @entry[92].id.number eq Array[UInt:D].new(0, 2, 2, 8);
    ok @entry[92].id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].id.xxhash == 1870422416;
    ok @entry[92].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[92].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[92].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[92].posting[0].amount.asset-code eqv "USD";
    ok @entry[92].posting[0].amount.asset-quantity == 3.06;
    ok @entry[92].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[92].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[92].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[92].posting[0].decinc eqv DecInc::INC;
    ok @entry[92].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[92].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 8);
    ok @entry[92].posting[0].id.entry-id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[0].id.entry-id.xxhash == 1870422416;
    ok @entry[92].posting[0].id.number == 0;
    ok @entry[92].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.06 USD";
    ok @entry[92].posting[0].id.xxhash == 90479374;
    ok @entry[92].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[92].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[92].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[92].posting[1].amount.asset-quantity == 1;
    ok @entry[92].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[92].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[92].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(153, 50), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[92].posting[1].decinc eqv DecInc::DEC;
    ok @entry[92].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[92].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 2, 8);
    ok @entry[92].posting[1].id.entry-id.text eqv "2013-06-30 '''\nWigwam LLC sold one wigwam for \$3.06 USD in late June 2013\nFILE:  「includes/2013/Q2.txn」\nAFTER: 「include 'Q2/June'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.06 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[1].id.entry-id.xxhash == 1870422416;
    ok @entry[92].posting[1].id.number == 1;
    ok @entry[92].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.06 USD";
    ok @entry[92].posting[1].id.xxhash == 1167336265;
    ok @entry[93].header.date eqv Date.new(2013,7,1);
    ok @entry[93].header.description eqv "Wigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n";
    ok @entry[93].header.important == 0;
    ok @entry[93].id.number eq Array[UInt:D].new(0, 2, 3, 0);
    ok @entry[93].id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].id.xxhash == 4095708493;
    ok @entry[93].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[93].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[93].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[93].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[93].posting[0].amount.asset-quantity == 1;
    ok @entry[93].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[93].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[93].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(307, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[93].posting[0].decinc eqv DecInc::INC;
    ok @entry[93].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[93].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 0);
    ok @entry[93].posting[0].id.entry-id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[0].id.entry-id.xxhash == 4095708493;
    ok @entry[93].posting[0].id.number == 0;
    ok @entry[93].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD";
    ok @entry[93].posting[0].id.xxhash == 688453593;
    ok @entry[93].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[93].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[93].posting[1].amount.asset-code eqv "USD";
    ok @entry[93].posting[1].amount.asset-quantity == 3.07;
    ok @entry[93].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[93].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[93].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[93].posting[1].decinc eqv DecInc::DEC;
    ok @entry[93].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[93].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 0);
    ok @entry[93].posting[1].id.entry-id.text eqv "2013-07-01 '''\nWigwam LLC bought one wigwam for \$3.07 USD in early July 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/July'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.07 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[1].id.entry-id.xxhash == 4095708493;
    ok @entry[93].posting[1].id.number == 1;
    ok @entry[93].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.07 USD";
    ok @entry[93].posting[1].id.xxhash == 2675724246;
    ok @entry[94].header.date eqv Date.new(2013,7,16);
    ok @entry[94].header.description eqv "Wigwam LLC received a dividend of \$2013.0716 USD";
    ok @entry[94].header.important == 0;
    ok @entry[94].id.number eq Array[UInt:D].new(0, 2, 3, 1, 0);
    ok @entry[94].id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].id.xxhash == 1271983877;
    ok @entry[94].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[94].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[94].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[94].posting[0].amount.asset-code eqv "USD";
    ok @entry[94].posting[0].amount.asset-quantity == 2013.0716;
    ok @entry[94].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[94].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[94].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[94].posting[0].decinc eqv DecInc::INC;
    ok @entry[94].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[94].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 1, 0);
    ok @entry[94].posting[0].id.entry-id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[0].id.entry-id.xxhash == 1271983877;
    ok @entry[94].posting[0].id.number == 0;
    ok @entry[94].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD";
    ok @entry[94].posting[0].id.xxhash == 463766995;
    ok @entry[94].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[94].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[94].posting[1].amount.asset-code eqv "USD";
    ok @entry[94].posting[1].amount.asset-quantity == 2013.0716;
    ok @entry[94].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[94].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[94].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[94].posting[1].decinc eqv DecInc::INC;
    ok @entry[94].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[94].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 1, 0);
    ok @entry[94].posting[1].id.entry-id.text eqv "2013-07-16 'Wigwam LLC received a dividend of \$2013.0716 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0716 USD\n  Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[1].id.entry-id.xxhash == 1271983877;
    ok @entry[94].posting[1].id.number == 1;
    ok @entry[94].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0716 USD";
    ok @entry[94].posting[1].id.xxhash == 1429453554;
    ok @entry[95].header.date eqv Date.new(2013,7,31);
    ok @entry[95].header.description eqv "Wigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n";
    ok @entry[95].header.important == 0;
    ok @entry[95].id.number eq Array[UInt:D].new(0, 2, 3, 2);
    ok @entry[95].id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].id.xxhash == 585905153;
    ok @entry[95].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[95].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[95].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[95].posting[0].amount.asset-code eqv "USD";
    ok @entry[95].posting[0].amount.asset-quantity == 3.07;
    ok @entry[95].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[95].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[95].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[95].posting[0].decinc eqv DecInc::INC;
    ok @entry[95].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[95].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 2);
    ok @entry[95].posting[0].id.entry-id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[0].id.entry-id.xxhash == 585905153;
    ok @entry[95].posting[0].id.number == 0;
    ok @entry[95].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.07 USD";
    ok @entry[95].posting[0].id.xxhash == 1932435839;
    ok @entry[95].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[95].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[95].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[95].posting[1].amount.asset-quantity == 1;
    ok @entry[95].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[95].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[95].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(307, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[95].posting[1].decinc eqv DecInc::DEC;
    ok @entry[95].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[95].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 2);
    ok @entry[95].posting[1].id.entry-id.text eqv "2013-07-31 '''\nWigwam LLC sold one wigwam for \$3.07 USD in late July 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/July'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.07 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[1].id.entry-id.xxhash == 585905153;
    ok @entry[95].posting[1].id.number == 1;
    ok @entry[95].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.07 USD";
    ok @entry[95].posting[1].id.xxhash == 3307372531;
    ok @entry[96].header.date eqv Date.new(2013,8,1);
    ok @entry[96].header.description eqv "Wigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n";
    ok @entry[96].header.important == 0;
    ok @entry[96].id.number eq Array[UInt:D].new(0, 2, 3, 3);
    ok @entry[96].id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].id.xxhash == 444744934;
    ok @entry[96].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[96].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[96].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[96].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[96].posting[0].amount.asset-quantity == 1;
    ok @entry[96].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[96].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[96].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(77, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[96].posting[0].decinc eqv DecInc::INC;
    ok @entry[96].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[96].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 3);
    ok @entry[96].posting[0].id.entry-id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[0].id.entry-id.xxhash == 444744934;
    ok @entry[96].posting[0].id.number == 0;
    ok @entry[96].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD";
    ok @entry[96].posting[0].id.xxhash == 1041307584;
    ok @entry[96].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[96].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[96].posting[1].amount.asset-code eqv "USD";
    ok @entry[96].posting[1].amount.asset-quantity == 3.08;
    ok @entry[96].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[96].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[96].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[96].posting[1].decinc eqv DecInc::DEC;
    ok @entry[96].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[96].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 3);
    ok @entry[96].posting[1].id.entry-id.text eqv "2013-08-01 '''\nWigwam LLC bought one wigwam for \$3.08 USD in early August 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/August'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.08 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[1].id.entry-id.xxhash == 444744934;
    ok @entry[96].posting[1].id.number == 1;
    ok @entry[96].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.08 USD";
    ok @entry[96].posting[1].id.xxhash == 3039180281;
    ok @entry[97].header.date eqv Date.new(2013,8,16);
    ok @entry[97].header.description eqv "Wigwam LLC received a dividend of \$2013.0816 USD";
    ok @entry[97].header.important == 0;
    ok @entry[97].id.number eq Array[UInt:D].new(0, 2, 3, 4, 0);
    ok @entry[97].id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].id.xxhash == 796956488;
    ok @entry[97].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[97].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[97].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[97].posting[0].amount.asset-code eqv "USD";
    ok @entry[97].posting[0].amount.asset-quantity == 2013.0816;
    ok @entry[97].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[97].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[97].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[97].posting[0].decinc eqv DecInc::INC;
    ok @entry[97].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[97].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 4, 0);
    ok @entry[97].posting[0].id.entry-id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[0].id.entry-id.xxhash == 796956488;
    ok @entry[97].posting[0].id.number == 0;
    ok @entry[97].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD";
    ok @entry[97].posting[0].id.xxhash == 4281436752;
    ok @entry[97].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[97].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[97].posting[1].amount.asset-code eqv "USD";
    ok @entry[97].posting[1].amount.asset-quantity == 2013.0816;
    ok @entry[97].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[97].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[97].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[97].posting[1].decinc eqv DecInc::INC;
    ok @entry[97].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[97].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 4, 0);
    ok @entry[97].posting[1].id.entry-id.text eqv "2013-08-16 'Wigwam LLC received a dividend of \$2013.0816 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0816 USD\n  Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[1].id.entry-id.xxhash == 796956488;
    ok @entry[97].posting[1].id.number == 1;
    ok @entry[97].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0816 USD";
    ok @entry[97].posting[1].id.xxhash == 1969770397;
    ok @entry[98].header.date eqv Date.new(2013,8,31);
    ok @entry[98].header.description eqv "Wigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n";
    ok @entry[98].header.important == 0;
    ok @entry[98].id.number eq Array[UInt:D].new(0, 2, 3, 5);
    ok @entry[98].id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].id.xxhash == 1033193703;
    ok @entry[98].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[98].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[98].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[98].posting[0].amount.asset-code eqv "USD";
    ok @entry[98].posting[0].amount.asset-quantity == 3.08;
    ok @entry[98].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[98].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[98].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[98].posting[0].decinc eqv DecInc::INC;
    ok @entry[98].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[98].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 5);
    ok @entry[98].posting[0].id.entry-id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[0].id.entry-id.xxhash == 1033193703;
    ok @entry[98].posting[0].id.number == 0;
    ok @entry[98].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.08 USD";
    ok @entry[98].posting[0].id.xxhash == 259022758;
    ok @entry[98].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[98].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[98].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[98].posting[1].amount.asset-quantity == 1;
    ok @entry[98].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[98].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[98].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(77, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[98].posting[1].decinc eqv DecInc::DEC;
    ok @entry[98].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[98].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 5);
    ok @entry[98].posting[1].id.entry-id.text eqv "2013-08-31 '''\nWigwam LLC sold one wigwam for \$3.08 USD in late August 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/August'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.08 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[1].id.entry-id.xxhash == 1033193703;
    ok @entry[98].posting[1].id.number == 1;
    ok @entry[98].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.08 USD";
    ok @entry[98].posting[1].id.xxhash == 115568189;
    ok @entry[99].header.date eqv Date.new(2013,9,1);
    ok @entry[99].header.description eqv "Wigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n";
    ok @entry[99].header.important == 0;
    ok @entry[99].id.number eq Array[UInt:D].new(0, 2, 3, 6);
    ok @entry[99].id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].id.xxhash == 2081861311;
    ok @entry[99].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[99].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[99].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[99].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[99].posting[0].amount.asset-quantity == 1;
    ok @entry[99].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[99].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[99].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(309, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[99].posting[0].decinc eqv DecInc::INC;
    ok @entry[99].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[99].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 6);
    ok @entry[99].posting[0].id.entry-id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[0].id.entry-id.xxhash == 2081861311;
    ok @entry[99].posting[0].id.number == 0;
    ok @entry[99].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD";
    ok @entry[99].posting[0].id.xxhash == 2102045153;
    ok @entry[99].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[99].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[99].posting[1].amount.asset-code eqv "USD";
    ok @entry[99].posting[1].amount.asset-quantity == 3.09;
    ok @entry[99].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[99].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[99].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[99].posting[1].decinc eqv DecInc::DEC;
    ok @entry[99].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[99].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 6);
    ok @entry[99].posting[1].id.entry-id.text eqv "2013-09-01 '''\nWigwam LLC bought one wigwam for \$3.09 USD in early September 2013\nFILE:  「includes/2013/Q3.txn」\nBEFORE:「include 'Q3/September'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.09 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[1].id.entry-id.xxhash == 2081861311;
    ok @entry[99].posting[1].id.number == 1;
    ok @entry[99].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.09 USD";
    ok @entry[99].posting[1].id.xxhash == 3596800179;
    ok @entry[100].header.date eqv Date.new(2013,9,16);
    ok @entry[100].header.description eqv "Wigwam LLC received a dividend of \$2013.0916 USD";
    ok @entry[100].header.important == 0;
    ok @entry[100].id.number eq Array[UInt:D].new(0, 2, 3, 7, 0);
    ok @entry[100].id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].id.xxhash == 705777824;
    ok @entry[100].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[100].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[100].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[100].posting[0].amount.asset-code eqv "USD";
    ok @entry[100].posting[0].amount.asset-quantity == 2013.0916;
    ok @entry[100].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[100].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[100].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[100].posting[0].decinc eqv DecInc::INC;
    ok @entry[100].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[100].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 7, 0);
    ok @entry[100].posting[0].id.entry-id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[0].id.entry-id.xxhash == 705777824;
    ok @entry[100].posting[0].id.number == 0;
    ok @entry[100].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD";
    ok @entry[100].posting[0].id.xxhash == 2426431672;
    ok @entry[100].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[100].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[100].posting[1].amount.asset-code eqv "USD";
    ok @entry[100].posting[1].amount.asset-quantity == 2013.0916;
    ok @entry[100].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[100].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[100].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[100].posting[1].decinc eqv DecInc::INC;
    ok @entry[100].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[100].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 7, 0);
    ok @entry[100].posting[1].id.entry-id.text eqv "2013-09-16 'Wigwam LLC received a dividend of \$2013.0916 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.0916 USD\n  Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[1].id.entry-id.xxhash == 705777824;
    ok @entry[100].posting[1].id.number == 1;
    ok @entry[100].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.0916 USD";
    ok @entry[100].posting[1].id.xxhash == 742012739;
    ok @entry[101].header.date eqv Date.new(2013,9,30);
    ok @entry[101].header.description eqv "Wigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n";
    ok @entry[101].header.important == 0;
    ok @entry[101].id.number eq Array[UInt:D].new(0, 2, 3, 8);
    ok @entry[101].id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].id.xxhash == 2792675818;
    ok @entry[101].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[101].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[101].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[101].posting[0].amount.asset-code eqv "USD";
    ok @entry[101].posting[0].amount.asset-quantity == 3.09;
    ok @entry[101].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[101].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[101].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[101].posting[0].decinc eqv DecInc::INC;
    ok @entry[101].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[101].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 8);
    ok @entry[101].posting[0].id.entry-id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[0].id.entry-id.xxhash == 2792675818;
    ok @entry[101].posting[0].id.number == 0;
    ok @entry[101].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.09 USD";
    ok @entry[101].posting[0].id.xxhash == 3876439597;
    ok @entry[101].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[101].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[101].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[101].posting[1].amount.asset-quantity == 1;
    ok @entry[101].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[101].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[101].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(309, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[101].posting[1].decinc eqv DecInc::DEC;
    ok @entry[101].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[101].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 3, 8);
    ok @entry[101].posting[1].id.entry-id.text eqv "2013-09-30 '''\nWigwam LLC sold one wigwam for \$3.09 USD in late September 2013\nFILE:  「includes/2013/Q3.txn」\nAFTER: 「include 'Q3/September'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.09 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[1].id.entry-id.xxhash == 2792675818;
    ok @entry[101].posting[1].id.number == 1;
    ok @entry[101].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.09 USD";
    ok @entry[101].posting[1].id.xxhash == 4150514617;
    ok @entry[102].header.date eqv Date.new(2013,10,1);
    ok @entry[102].header.description eqv "Wigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n";
    ok @entry[102].header.important == 0;
    ok @entry[102].id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[102].id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].id.xxhash == 574545503;
    ok @entry[102].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[102].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[102].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[102].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[102].posting[0].amount.asset-quantity == 1;
    ok @entry[102].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[102].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[102].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(31, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[102].posting[0].decinc eqv DecInc::INC;
    ok @entry[102].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[102].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[102].posting[0].id.entry-id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[0].id.entry-id.xxhash == 574545503;
    ok @entry[102].posting[0].id.number == 0;
    ok @entry[102].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD";
    ok @entry[102].posting[0].id.xxhash == 264778505;
    ok @entry[102].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[102].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[102].posting[1].amount.asset-code eqv "USD";
    ok @entry[102].posting[1].amount.asset-quantity == 3.1;
    ok @entry[102].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[102].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[102].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[102].posting[1].decinc eqv DecInc::DEC;
    ok @entry[102].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[102].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 0);
    ok @entry[102].posting[1].id.entry-id.text eqv "2013-10-01 '''\nWigwam LLC bought one wigwam for \$3.10 USD in early October 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/October'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.10 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[1].id.entry-id.xxhash == 574545503;
    ok @entry[102].posting[1].id.number == 1;
    ok @entry[102].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.10 USD";
    ok @entry[102].posting[1].id.xxhash == 806384794;
    ok @entry[103].header.date eqv Date.new(2013,10,16);
    ok @entry[103].header.description eqv "Wigwam LLC received a dividend of \$2013.1016 USD";
    ok @entry[103].header.important == 0;
    ok @entry[103].id.number eq Array[UInt:D].new(0, 2, 4, 1, 0);
    ok @entry[103].id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].id.xxhash == 1123984249;
    ok @entry[103].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[103].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[103].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[103].posting[0].amount.asset-code eqv "USD";
    ok @entry[103].posting[0].amount.asset-quantity == 2013.1016;
    ok @entry[103].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[103].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[103].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[103].posting[0].decinc eqv DecInc::INC;
    ok @entry[103].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[103].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 1, 0);
    ok @entry[103].posting[0].id.entry-id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[0].id.entry-id.xxhash == 1123984249;
    ok @entry[103].posting[0].id.number == 0;
    ok @entry[103].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD";
    ok @entry[103].posting[0].id.xxhash == 1130097971;
    ok @entry[103].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[103].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[103].posting[1].amount.asset-code eqv "USD";
    ok @entry[103].posting[1].amount.asset-quantity == 2013.1016;
    ok @entry[103].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[103].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[103].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[103].posting[1].decinc eqv DecInc::INC;
    ok @entry[103].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[103].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 1, 0);
    ok @entry[103].posting[1].id.entry-id.text eqv "2013-10-16 'Wigwam LLC received a dividend of \$2013.1016 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1016 USD\n  Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[1].id.entry-id.xxhash == 1123984249;
    ok @entry[103].posting[1].id.number == 1;
    ok @entry[103].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1016 USD";
    ok @entry[103].posting[1].id.xxhash == 719072295;
    ok @entry[104].header.date eqv Date.new(2013,10,31);
    ok @entry[104].header.description eqv "Wigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n";
    ok @entry[104].header.important == 0;
    ok @entry[104].id.number eq Array[UInt:D].new(0, 2, 4, 2);
    ok @entry[104].id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].id.xxhash == 1605620712;
    ok @entry[104].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[104].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[104].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[104].posting[0].amount.asset-code eqv "USD";
    ok @entry[104].posting[0].amount.asset-quantity == 3.1;
    ok @entry[104].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[104].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[104].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[104].posting[0].decinc eqv DecInc::INC;
    ok @entry[104].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[104].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 2);
    ok @entry[104].posting[0].id.entry-id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[0].id.entry-id.xxhash == 1605620712;
    ok @entry[104].posting[0].id.number == 0;
    ok @entry[104].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.10 USD";
    ok @entry[104].posting[0].id.xxhash == 4175703663;
    ok @entry[104].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[104].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[104].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[104].posting[1].amount.asset-quantity == 1;
    ok @entry[104].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[104].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[104].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(31, 10), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[104].posting[1].decinc eqv DecInc::DEC;
    ok @entry[104].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[104].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 2);
    ok @entry[104].posting[1].id.entry-id.text eqv "2013-10-31 '''\nWigwam LLC sold one wigwam for \$3.10 USD in late October 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/October'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.10 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[1].id.entry-id.xxhash == 1605620712;
    ok @entry[104].posting[1].id.number == 1;
    ok @entry[104].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.10 USD";
    ok @entry[104].posting[1].id.xxhash == 1943956588;
    ok @entry[105].header.date eqv Date.new(2013,11,1);
    ok @entry[105].header.description eqv "Wigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n";
    ok @entry[105].header.important == 0;
    ok @entry[105].id.number eq Array[UInt:D].new(0, 2, 4, 3);
    ok @entry[105].id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].id.xxhash == 3963180771;
    ok @entry[105].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[105].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[105].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[105].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[105].posting[0].amount.asset-quantity == 1;
    ok @entry[105].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[105].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[105].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(311, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[105].posting[0].decinc eqv DecInc::INC;
    ok @entry[105].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[105].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 3);
    ok @entry[105].posting[0].id.entry-id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[0].id.entry-id.xxhash == 3963180771;
    ok @entry[105].posting[0].id.number == 0;
    ok @entry[105].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD";
    ok @entry[105].posting[0].id.xxhash == 3475412129;
    ok @entry[105].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[105].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[105].posting[1].amount.asset-code eqv "USD";
    ok @entry[105].posting[1].amount.asset-quantity == 3.11;
    ok @entry[105].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[105].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[105].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[105].posting[1].decinc eqv DecInc::DEC;
    ok @entry[105].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[105].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 3);
    ok @entry[105].posting[1].id.entry-id.text eqv "2013-11-01 '''\nWigwam LLC bought one wigwam for \$3.11 USD in early November 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/November'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.11 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[1].id.entry-id.xxhash == 3963180771;
    ok @entry[105].posting[1].id.number == 1;
    ok @entry[105].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.11 USD";
    ok @entry[105].posting[1].id.xxhash == 1726070033;
    ok @entry[106].header.date eqv Date.new(2013,11,16);
    ok @entry[106].header.description eqv "Wigwam LLC received a dividend of \$2013.1116 USD";
    ok @entry[106].header.important == 0;
    ok @entry[106].id.number eq Array[UInt:D].new(0, 2, 4, 4, 0);
    ok @entry[106].id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].id.xxhash == 3185930480;
    ok @entry[106].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[106].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[106].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[106].posting[0].amount.asset-code eqv "USD";
    ok @entry[106].posting[0].amount.asset-quantity == 2013.1116;
    ok @entry[106].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[106].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[106].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[106].posting[0].decinc eqv DecInc::INC;
    ok @entry[106].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[106].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 4, 0);
    ok @entry[106].posting[0].id.entry-id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[0].id.entry-id.xxhash == 3185930480;
    ok @entry[106].posting[0].id.number == 0;
    ok @entry[106].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD";
    ok @entry[106].posting[0].id.xxhash == 2304408211;
    ok @entry[106].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[106].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[106].posting[1].amount.asset-code eqv "USD";
    ok @entry[106].posting[1].amount.asset-quantity == 2013.1116;
    ok @entry[106].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[106].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[106].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[106].posting[1].decinc eqv DecInc::INC;
    ok @entry[106].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[106].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 4, 0);
    ok @entry[106].posting[1].id.entry-id.text eqv "2013-11-16 'Wigwam LLC received a dividend of \$2013.1116 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1116 USD\n  Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[1].id.entry-id.xxhash == 3185930480;
    ok @entry[106].posting[1].id.number == 1;
    ok @entry[106].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1116 USD";
    ok @entry[106].posting[1].id.xxhash == 690089900;
    ok @entry[107].header.date eqv Date.new(2013,11,30);
    ok @entry[107].header.description eqv "Wigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n";
    ok @entry[107].header.important == 0;
    ok @entry[107].id.number eq Array[UInt:D].new(0, 2, 4, 5);
    ok @entry[107].id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].id.xxhash == 2315332478;
    ok @entry[107].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[107].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[107].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[107].posting[0].amount.asset-code eqv "USD";
    ok @entry[107].posting[0].amount.asset-quantity == 3.11;
    ok @entry[107].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[107].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[107].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[107].posting[0].decinc eqv DecInc::INC;
    ok @entry[107].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[107].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 5);
    ok @entry[107].posting[0].id.entry-id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[0].id.entry-id.xxhash == 2315332478;
    ok @entry[107].posting[0].id.number == 0;
    ok @entry[107].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.11 USD";
    ok @entry[107].posting[0].id.xxhash == 4002911623;
    ok @entry[107].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[107].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[107].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[107].posting[1].amount.asset-quantity == 1;
    ok @entry[107].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[107].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[107].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(311, 100), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[107].posting[1].decinc eqv DecInc::DEC;
    ok @entry[107].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[107].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 5);
    ok @entry[107].posting[1].id.entry-id.text eqv "2013-11-30 '''\nWigwam LLC sold one wigwam for \$3.11 USD in late November 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/November'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.11 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[1].id.entry-id.xxhash == 2315332478;
    ok @entry[107].posting[1].id.number == 1;
    ok @entry[107].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.11 USD";
    ok @entry[107].posting[1].id.xxhash == 2318076742;
    ok @entry[108].header.date eqv Date.new(2013,12,1);
    ok @entry[108].header.description eqv "Wigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n";
    ok @entry[108].header.important == 0;
    ok @entry[108].id.number eq Array[UInt:D].new(0, 2, 4, 6);
    ok @entry[108].id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].id.xxhash == 646922698;
    ok @entry[108].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[108].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[108].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Inventory", "Wigwams");
    ok @entry[108].posting[0].amount.asset-code eqv "Wigwam";
    ok @entry[108].posting[0].amount.asset-quantity == 1;
    ok @entry[108].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[108].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[108].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(78, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[108].posting[0].decinc eqv DecInc::INC;
    ok @entry[108].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[108].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 6);
    ok @entry[108].posting[0].id.entry-id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[0].id.entry-id.xxhash == 646922698;
    ok @entry[108].posting[0].id.number == 0;
    ok @entry[108].posting[0].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD";
    ok @entry[108].posting[0].id.xxhash == 2535125365;
    ok @entry[108].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[108].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[108].posting[1].amount.asset-code eqv "USD";
    ok @entry[108].posting[1].amount.asset-quantity == 3.12;
    ok @entry[108].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[108].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[108].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[108].posting[1].decinc eqv DecInc::DEC;
    ok @entry[108].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[108].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 6);
    ok @entry[108].posting[1].id.entry-id.text eqv "2013-12-01 '''\nWigwam LLC bought one wigwam for \$3.12 USD in early December 2013\nFILE:  「includes/2013/Q4.txn」\nBEFORE:「include 'Q4/December'」\n'''\nAssets:WigwamLLC:Inventory:Wigwams     1 Wigwam \@ 3.12 USD\nAssets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[1].id.entry-id.xxhash == 646922698;
    ok @entry[108].posting[1].id.number == 1;
    ok @entry[108].posting[1].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank      -3.12 USD";
    ok @entry[108].posting[1].id.xxhash == 3978925433;
    ok @entry[109].header.date eqv Date.new(2013,12,16);
    ok @entry[109].header.description eqv "Wigwam LLC received a dividend of \$2013.1216 USD";
    ok @entry[109].header.important == 0;
    ok @entry[109].id.number eq Array[UInt:D].new(0, 2, 4, 7, 0);
    ok @entry[109].id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].id.xxhash == 2042360109;
    ok @entry[109].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[109].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[109].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[109].posting[0].amount.asset-code eqv "USD";
    ok @entry[109].posting[0].amount.asset-quantity == 2013.1216;
    ok @entry[109].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[109].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[109].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[109].posting[0].decinc eqv DecInc::INC;
    ok @entry[109].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[109].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 7, 0);
    ok @entry[109].posting[0].id.entry-id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[0].id.entry-id.xxhash == 2042360109;
    ok @entry[109].posting[0].id.number == 0;
    ok @entry[109].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD";
    ok @entry[109].posting[0].id.xxhash == 578573857;
    ok @entry[109].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[109].posting[1].account.silo eqv Silo::INCOME;
    ok @entry[109].posting[1].amount.asset-code eqv "USD";
    ok @entry[109].posting[1].amount.asset-quantity == 2013.1216;
    ok @entry[109].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[109].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[109].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[109].posting[1].decinc eqv DecInc::INC;
    ok @entry[109].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[109].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 7, 0);
    ok @entry[109].posting[1].id.entry-id.text eqv "2013-12-16 'Wigwam LLC received a dividend of \$2013.1216 USD'\n  Assets:WigwamLLC:KoopaTroopaBank    \$2013.1216 USD\n  Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[1].id.entry-id.xxhash == 2042360109;
    ok @entry[109].posting[1].id.number == 1;
    ok @entry[109].posting[1].id.text eqv "Income:WigwamLLC:Dividends          \$2013.1216 USD";
    ok @entry[109].posting[1].id.xxhash == 2455119990;
    ok @entry[110].header.date eqv Date.new(2013,12,31);
    ok @entry[110].header.description eqv "Wigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n";
    ok @entry[110].header.important == 0;
    ok @entry[110].id.number eq Array[UInt:D].new(0, 2, 4, 8);
    ok @entry[110].id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].id.xxhash == 1933516675;
    ok @entry[110].posting[0].account.entity eqv "WigwamLLC";
    ok @entry[110].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[110].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("KoopaTroopaBank");
    ok @entry[110].posting[0].amount.asset-code eqv "USD";
    ok @entry[110].posting[0].amount.asset-quantity == 3.12;
    ok @entry[110].posting[0].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[110].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[110].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[110].posting[0].decinc eqv DecInc::INC;
    ok @entry[110].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[110].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 8);
    ok @entry[110].posting[0].id.entry-id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[0].id.entry-id.xxhash == 1933516675;
    ok @entry[110].posting[0].id.number == 0;
    ok @entry[110].posting[0].id.text eqv "Assets:WigwamLLC:KoopaTroopaBank       3.12 USD";
    ok @entry[110].posting[0].id.xxhash == 3769395453;
    ok @entry[110].posting[1].account.entity eqv "WigwamLLC";
    ok @entry[110].posting[1].account.silo eqv Silo::ASSETS;
    ok @entry[110].posting[1].amount.asset-code eqv "Wigwam";
    ok @entry[110].posting[1].amount.asset-quantity == 1;
    ok @entry[110].posting[1].amount.asset-symbol eqv TXN::Parser::Types::AssetSymbol;
    ok @entry[110].posting[1].amount.plus-or-minus eqv "-";
    ok @entry[110].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot.new(inherit => TXN::Parser::AST::Entry::Posting::Annot::Inherit, lot => TXN::Parser::AST::Entry::Posting::Annot::Lot, xe => TXN::Parser::AST::Entry::Posting::Annot::XE.new(asset-code => "USD", asset-price => FatRat.new(78, 25), asset-symbol => TXN::Parser::Types::AssetSymbol));
    ok @entry[110].posting[1].decinc eqv DecInc::DEC;
    ok @entry[110].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[110].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 2, 4, 8);
    ok @entry[110].posting[1].id.entry-id.text eqv "2013-12-31 '''\nWigwam LLC sold one wigwam for \$3.12 USD in late December 2013\nFILE:  「includes/2013/Q4.txn」\nAFTER: 「include 'Q4/December'」\n'''\nAssets:WigwamLLC:KoopaTroopaBank       3.12 USD\nAssets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[1].id.entry-id.xxhash == 1933516675;
    ok @entry[110].posting[1].id.number == 1;
    ok @entry[110].posting[1].id.text eqv "Assets:WigwamLLC:Inventory:Wigwams    -1 Wigwam \@ 3.12 USD";
    ok @entry[110].posting[1].id.xxhash == 3832507616;
    ok @entry[111].header.date eqv Date.new(2014,1,1);
    ok @entry[111].header.description eqv "I started the year with \$1000 in Bankwest cheque account";
    ok @entry[111].header.important == 0;
    ok @entry[111].id.number eq Array[UInt:D].new(0, 3);
    ok @entry[111].id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].id.xxhash == 1468523538;
    ok @entry[111].posting[0].account.entity eqv "Personal";
    ok @entry[111].posting[0].account.silo eqv Silo::ASSETS;
    ok @entry[111].posting[0].account.path eqv Array[TXN::Parser::Types::VarName:D].new("Bankwest", "Cheque");
    ok @entry[111].posting[0].amount.asset-code eqv "USD";
    ok @entry[111].posting[0].amount.asset-quantity == 1000;
    ok @entry[111].posting[0].amount.asset-symbol eqv "\$";
    ok @entry[111].posting[0].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[111].posting[0].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[111].posting[0].decinc eqv DecInc::INC;
    ok @entry[111].posting[0].drcr eqv DrCr::DEBIT;
    ok @entry[111].posting[0].id.entry-id.number eq Array[UInt:D].new(0, 3);
    ok @entry[111].posting[0].id.entry-id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[0].id.entry-id.xxhash == 1468523538;
    ok @entry[111].posting[0].id.number == 0;
    ok @entry[111].posting[0].id.text eqv "Assets:Personal:Bankwest:Cheque      \$1000.00 USD";
    ok @entry[111].posting[0].id.xxhash == 4134277096;
    ok @entry[111].posting[1].account.entity eqv "Personal";
    ok @entry[111].posting[1].account.silo eqv Silo::EQUITY;
    ok @entry[111].posting[1].amount.asset-code eqv "USD";
    ok @entry[111].posting[1].amount.asset-quantity == 1000;
    ok @entry[111].posting[1].amount.asset-symbol eqv "\$";
    ok @entry[111].posting[1].amount.plus-or-minus eqv TXN::Parser::Types::PlusMinus;
    ok @entry[111].posting[1].annot eqv TXN::Parser::AST::Entry::Posting::Annot;
    ok @entry[111].posting[1].decinc eqv DecInc::INC;
    ok @entry[111].posting[1].drcr eqv DrCr::CREDIT;
    ok @entry[111].posting[1].id.entry-id.number eq Array[UInt:D].new(0, 3);
    ok @entry[111].posting[1].id.entry-id.text eqv "2014-01-01 \"I started the year with \$1000 in Bankwest cheque account\"\n  Assets:Personal:Bankwest:Cheque      \$1000.00 USD\n  Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[1].id.entry-id.xxhash == 1468523538;
    ok @entry[111].posting[1].id.number == 1;
    ok @entry[111].posting[1].id.text eqv "Equity:Personal                      \$1000.00 USD";
    ok @entry[111].posting[1].id.xxhash == 344831063;
    # end tests }}}
}

# vim: set filetype=perl6 foldmethod=marker foldlevel=0:
