#!/usr/bin/perl

# ABSTRACT: IP Address Normalization tests for Net::Google::SafeBrowsing4::URI class

use strict;
use warnings;

use Test::More 0.92 qw(no_plan);

use Net::Google::SafeBrowsing4::URI;

my @invalid_uris = (
	# Single number IPv4 (decimal/octal/hexadecimal) out of range
	'http://4294967296/',
	'http://00000040000000000/',
	'http://0x100000000/',
	# Dotted-decimal IPv4 with too much elements:
	'http://195.56.65.250.1',
	'1.1.1.1.1',
	# One segment out of bounds
	'195.56.65.256',
	'195.0x100.65.250',
	'0310.0x100.65.250',
	'195.56.65536',
	# Multibyte first segment is not accepted by major Browsers
	'49976.65.250',
	# Multibyte middle segment is not accepted by major Browsers
	'195.14401.250',
);

my %uris = (
	# 1.1.1.1 IPv4 Addresses
	'195.56.65.250' => 'http://195.56.65.250/',
	'http://195.56.65.250' => 'http://195.56.65.250/',
	'http://195.56.65.0372' => 'http://195.56.65.250/',
	'http://195.56.65.0xFa' => 'http://195.56.65.250/',
	'http://195.56.0101.250' => 'http://195.56.65.250/',
	'http://195.56.0101.0372' => 'http://195.56.65.250/',
	'http://195.56.0101.0xFa' => 'http://195.56.65.250/',
	'http://195.56.0x41.250' => 'http://195.56.65.250/',
	'http://195.56.0x41.0372' => 'http://195.56.65.250/',
	'http://195.56.0x41.0xFa' => 'http://195.56.65.250/',
	'http://195.070.65.250' => 'http://195.56.65.250/',
	'http://195.070.65.0372' => 'http://195.56.65.250/',
	'http://195.070.65.0xFa' => 'http://195.56.65.250/',
	'http://195.070.0101.250' => 'http://195.56.65.250/',
	'http://195.070.0101.0372' => 'http://195.56.65.250/',
	'http://195.070.0101.0xFa' => 'http://195.56.65.250/',
	'http://195.070.0x41.250' => 'http://195.56.65.250/',
	'http://195.070.0x41.0372' => 'http://195.56.65.250/',
	'http://195.070.0x41.0xFa' => 'http://195.56.65.250/',
	'http://195.0x38.65.250' => 'http://195.56.65.250/',
	'http://195.0x38.65.0372' => 'http://195.56.65.250/',
	'http://195.0x38.65.0xFa' => 'http://195.56.65.250/',
	'http://195.0x38.0101.250' => 'http://195.56.65.250/',
	'http://195.0x38.0101.0372' => 'http://195.56.65.250/',
	'http://195.0x38.0101.0xFa' => 'http://195.56.65.250/',
	'http://195.0x38.0x41.250' => 'http://195.56.65.250/',
	'http://195.0x38.0x41.0372' => 'http://195.56.65.250/',
	'http://195.0x38.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0303.56.65.250' => 'http://195.56.65.250/',
	'http://0303.56.65.0372' => 'http://195.56.65.250/',
	'http://0303.56.65.0xFa' => 'http://195.56.65.250/',
	'http://0303.56.0101.250' => 'http://195.56.65.250/',
	'http://0303.56.0101.0372' => 'http://195.56.65.250/',
	'http://0303.56.0101.0xFa' => 'http://195.56.65.250/',
	'http://0303.56.0x41.250' => 'http://195.56.65.250/',
	'http://0303.56.0x41.0372' => 'http://195.56.65.250/',
	'http://0303.56.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0303.070.65.250' => 'http://195.56.65.250/',
	'http://0303.070.65.0372' => 'http://195.56.65.250/',
	'http://0303.070.65.0xFa' => 'http://195.56.65.250/',
	'http://0303.070.0101.250' => 'http://195.56.65.250/',
	'http://0303.070.0101.0372' => 'http://195.56.65.250/',
	'http://0303.070.0101.0xFa' => 'http://195.56.65.250/',
	'http://0303.070.0x41.250' => 'http://195.56.65.250/',
	'http://0303.070.0x41.0372' => 'http://195.56.65.250/',
	'http://0303.070.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0303.0x38.65.250' => 'http://195.56.65.250/',
	'http://0303.0x38.65.0372' => 'http://195.56.65.250/',
	'http://0303.0x38.65.0xFa' => 'http://195.56.65.250/',
	'http://0303.0x38.0101.250' => 'http://195.56.65.250/',
	'http://0303.0x38.0101.0372' => 'http://195.56.65.250/',
	'http://0303.0x38.0101.0xFa' => 'http://195.56.65.250/',
	'http://0303.0x38.0x41.250' => 'http://195.56.65.250/',
	'http://0303.0x38.0x41.0372' => 'http://195.56.65.250/',
	'http://0303.0x38.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.56.65.250' => 'http://195.56.65.250/',
	'http://0xc3.56.65.0372' => 'http://195.56.65.250/',
	'http://0xc3.56.65.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.56.0101.250' => 'http://195.56.65.250/',
	'http://0xc3.56.0101.0372' => 'http://195.56.65.250/',
	'http://0xc3.56.0101.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.56.0x41.250' => 'http://195.56.65.250/',
	'http://0xc3.56.0x41.0372' => 'http://195.56.65.250/',
	'http://0xc3.56.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.070.65.250' => 'http://195.56.65.250/',
	'http://0xc3.070.65.0372' => 'http://195.56.65.250/',
	'http://0xc3.070.65.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.070.0101.250' => 'http://195.56.65.250/',
	'http://0xc3.070.0101.0372' => 'http://195.56.65.250/',
	'http://0xc3.070.0101.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.070.0x41.250' => 'http://195.56.65.250/',
	'http://0xc3.070.0x41.0372' => 'http://195.56.65.250/',
	'http://0xc3.070.0x41.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.0x38.65.250' => 'http://195.56.65.250/',
	'http://0xc3.0x38.65.0372' => 'http://195.56.65.250/',
	'http://0xc3.0x38.65.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0101.250' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0101.0372' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0101.0xFa' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0x41.250' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0x41.0372' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0x41.0xFa' => 'http://195.56.65.250/',
	# 1 + 1 + 2 bytes split IPv4
	'http://195.56.16890' => 'http://195.56.65.250/',
	'http://195.56.040772' => 'http://195.56.65.250/',
	'http://195.56.0x41fa' => 'http://195.56.65.250/',
	'http://195.070.16890' => 'http://195.56.65.250/',
	'http://195.070.040772' => 'http://195.56.65.250/',
	'http://195.070.0x41fa' => 'http://195.56.65.250/',
	'http://195.0x38.16890' => 'http://195.56.65.250/',
	'http://195.0x38.040772' => 'http://195.56.65.250/',
	'http://195.0x38.0x41fa' => 'http://195.56.65.250/',
	'http://0303.56.16890' => 'http://195.56.65.250/',
	'http://0303.56.040772' => 'http://195.56.65.250/',
	'http://0303.56.0x41fa' => 'http://195.56.65.250/',
	'http://0303.070.16890' => 'http://195.56.65.250/',
	'http://0303.070.040772' => 'http://195.56.65.250/',
	'http://0303.070.0x41fa' => 'http://195.56.65.250/',
	'http://0303.0x38.16890' => 'http://195.56.65.250/',
	'http://0303.0x38.040772' => 'http://195.56.65.250/',
	'http://0303.0x38.0x41fa' => 'http://195.56.65.250/',
	'http://0xc3.56.16890' => 'http://195.56.65.250/',
	'http://0xc3.56.040772' => 'http://195.56.65.250/',
	'http://0xc3.56.0x41fa' => 'http://195.56.65.250/',
	'http://0xc3.070.16890' => 'http://195.56.65.250/',
	'http://0xc3.070.040772' => 'http://195.56.65.250/',
	'http://0xc3.070.0x41fa' => 'http://195.56.65.250/',
	'http://0xc3.0x38.16890' => 'http://195.56.65.250/',
	'http://0xc3.0x38.040772' => 'http://195.56.65.250/',
	'http://0xc3.0x38.0x41fa' => 'http://195.56.65.250/',
	# 1 + 3 bytes split IPv4
	'http://195.3686906' => 'http://195.56.65.250/',
	'http://195.016040772' => 'http://195.56.65.250/',
	'http://195.0x3841fa' => 'http://195.56.65.250/',
	'http://0303.3686906' => 'http://195.56.65.250/',
	'http://0303.016040772' => 'http://195.56.65.250/',
	'http://0303.0x3841fa' => 'http://195.56.65.250/',
	'http://0xc3.3686906' => 'http://195.56.65.250/',
	'http://0xc3.016040772' => 'http://195.56.65.250/',
	'http://0xc3.0x3841fa' => 'http://195.56.65.250/',
	# Single decimal/octal/hexadecimal as IPv4
	'http://3275244026' => 'http://195.56.65.250/',
	'http://030316040772' => 'http://195.56.65.250/',
	'http://0xc33841fa' => 'http://195.56.65.250/',
);


foreach my $uri (sort(@invalid_uris)) {
	my $gsb_uri = Net::Google::SafeBrowsing4::URI->new($uri);
	is($gsb_uri, undef, "Invalid URI '". $uri ."'  detected");
}

foreach my $uri (sort(keys(%uris))) {
	my $gsb_uri = Net::Google::SafeBrowsing4::URI->new($uri);
	ok($gsb_uri, "URI parsed: ". $uri);
	is($gsb_uri->as_string(), $uris{$uri}, "Normalize URI '". $uri ."'  to '". $uris{$uri} ."' (got: '". $gsb_uri->as_string() ."')");
}
