# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl test.pl'

#########################

use Test;
BEGIN { plan tests => 5 };
use AI::DecisionTree;
ok(1); # If we made it this far, we're ok.

#########################

my $dtree = AI::DecisionTree->new(noise_mode => 'pick_best');
ok $dtree;

my @names = split /, /, <DATA>;
chomp $names[-1];

# Train on first 600 instances
printf "Loading 600 training instances with %d attribute types each\n", scalar @names;
while (<DATA>) {
  last unless 2..601;
  
  chomp;
  my @values = split /, /, $_;
  my $result = pop @values;
  my %pairs = map {$names[$_], $values[$_]} 0..$#names;

  $dtree->add_instance(attributes => \%pairs,
		       result => $result,
		      );
}
print "Building decision tree\n";
$dtree->train;
ok(1);

# Test on rest of data, get at least 80%
print "Testing on remainder of data\n";
my ($good, $bad) = (0,0);
while (<DATA>) {
  chomp;
  my @values = split /, /, $_;
  my $result = pop @values;
  my %pairs = map {$names[$_], $values[$_]} 0..$#names;

  my ($guess, $confidence) = $dtree->get_result(attributes => \%pairs);
  $guess ||= '';  $confidence ||= '';
  ($guess eq $result ? $good : $bad)++;
  
  #print "$guess : $result : $confidence\n";
}
my $accuracy = $good/($good + $bad);
ok $accuracy > .8;
print "Accuracy=$accuracy\n";

#use YAML; print Dump($dtree->rule_tree);
#print map "$_\n", $dtree->rule_statements;

if (eval "use GraphViz; 1") {
  my $graphviz = $dtree->as_graphviz;
  ok $graphviz;

  if (0) {
    # Only works on Mac OS X
    my $file = '/tmp/tree2.png';
    open my($fh), "> $file" or die "$file: $!";
    print $fh $graphviz->as_png;
    close $fh;
    system('open', $file);
  }
} else {
  skip("Skipping: GraphViz is not installed", 0);
}

# The following data comes from the "C4.5" software package, in the
# "soybean.data" data file.  It is somewhat noisy.  I chose it because
# it was a pretty big data set, and because there are published
# results on it that I can compare to.  Since the data seemed to be in
# order from most-information to least-information or something in the
# C4.5 distribution, I randomized the order of the instances.  Note
# also that I'm treating the '?' value as a normal string value.

# It looks like the original data source is
#     (a) Michalski,R.S. Learning by being told and learning from
#         examples: an experimental comparison of the two methodes of knowledge
#         acquisition in the context of developing an expert system for soybean
#         desease diagnoiss", International Journal of Policy Analysis and
#         Information Systems, 1980, 4(2), 125-161.

# The "C4.5" package is written by J.R. Quinlan and may be downloaded
# and used for free, but it is not supported and may not be
# redistributed.

__DATA__
date, plant-stand, precip, temp, hail, crop-hist, area-damaged, severity, seed-tmt, germination, plant-growth, leaves, leafspots-halo, leafspots-marg, leafspot-size, leaf-shread, leaf-malf, leaf-mild, stem, lodging, stem-cankers, canker-lesion, fruiting-bodies, external decay, mycelium, int-discolor, sclerotia, fruit-pods, fruit spots, seed, mold-growth, seed-discolor, seed-size, shriveling, roots
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, norm, yes, same-lst-yr, low-areas, severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
june, lt-normal, norm, lt-norm, yes, same-lst-sev-yrs, low-areas, severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
october, normal, lt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
may, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, gt-norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
april, lt-normal, ?, lt-norm, ?, same-lst-yr, whole-field, ?, ?, ?, abnorm, abnorm, no-yellow-halos, no-w-s-marg, gt-1/8, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
september, lt-normal, gt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
september, normal, lt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, lt-normal, gt-norm, gt-norm, no, diff-lst-year, scattered, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
june, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, ?, ?, ?, ?, ?, upper-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
may, lt-normal, ?, lt-norm, ?, diff-lst-year, scattered, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
august, normal, gt-norm, norm, no, same-lst-yr, low-areas, minor, fungicide, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, normal, gt-norm, norm, no, same-lst-two-yrs, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, gt-norm, norm, no, same-lst-two-yrs, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
october, ?, ?, ?, ?, ?, low-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
september, normal, lt-norm, norm, yes, same-lst-yr, scattered, pot-severe, fungicide, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
august, lt-normal, lt-norm, lt-norm, yes, same-lst-two-yrs, upper-areas, severe, none, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, normal, norm, lt-norm, no, same-lst-sev-yrs, whole-field, minor, none, 80-89%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
may, lt-normal, gt-norm, lt-norm, yes, diff-lst-year, low-areas, pot-severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
september, lt-normal, gt-norm, norm, no, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, gt-norm, norm, no, same-lst-sev-yrs, low-areas, severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
july, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
july, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, lt-normal, gt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, minor, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, norm, present, norm, anthracnose
september, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
may, normal, gt-norm, norm, no, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, lt-normal, ?, lt-norm, ?, same-lst-yr, whole-field, ?, ?, ?, abnorm, abnorm, no-yellow-halos, no-w-s-marg, gt-1/8, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
june, normal, norm, norm, yes, same-lst-sev-yrs, low-areas, minor, none, 90-100%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
june, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
august, normal, gt-norm, gt-norm, no, same-lst-sev-yrs, upper-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, present, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, lt-normal, lt-norm, lt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
october, normal, gt-norm, lt-norm, no, same-lst-two-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
august, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, gt-norm, no, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, normal, norm, gt-norm, no, diff-lst-year, low-areas, minor, none, 80-89%, abnorm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
august, normal, gt-norm, norm, yes, diff-lst-year, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
october, normal, lt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
august, normal, gt-norm, norm, no, diff-lst-year, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
october, lt-normal, lt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
may, lt-normal, gt-norm, gt-norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, normal, gt-norm, norm, no, same-lst-yr, whole-field, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
june, ?, ?, ?, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
june, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, low-areas, pot-severe, fungicide, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, ?, lt-norm, ?, diff-lst-year, scattered, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
september, normal, gt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, absent, present, lt-norm, present, norm, frog-eye-leaf-spot
august, lt-normal, norm, gt-norm, yes, same-lst-yr, low-areas, minor, other, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-yr, low-areas, minor, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, lt-normal, gt-norm, gt-norm, yes, diff-lst-year, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
july, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, scattered, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
april, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, fungicide, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
june, normal, gt-norm, norm, no, same-lst-sev-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
august, lt-normal, norm, gt-norm, yes, same-lst-yr, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
august, lt-normal, norm, lt-norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, 90-100%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-yr, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, lt-normal, gt-norm, lt-norm, yes, diff-lst-year, low-areas, pot-severe, fungicide, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
june, lt-normal, norm, lt-norm, yes, diff-lst-year, scattered, minor, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, normal, lt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, lt-normal, gt-norm, gt-norm, no, same-lst-yr, low-areas, minor, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
october, normal, lt-norm, gt-norm, no, diff-lst-year, upper-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, lt-norm, gt-norm, no, same-lst-yr, upper-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
october, ?, ?, ?, ?, ?, upper-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
october, normal, norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, ?, ?, ?, ?, ?, upper-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
september, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, ?, ?, ?, ?, ?, low-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
september, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, present, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, norm, norm, no, diff-lst-year, low-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dna, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
september, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, other, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, norm, absent, norm, anthracnose
august, normal, gt-norm, norm, no, same-lst-sev-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, normal, gt-norm, gt-norm, ?, same-lst-yr, whole-field, ?, ?, 90-100%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
september, normal, gt-norm, norm, yes, same-lst-yr, low-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
october, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, minor, other, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
august, normal, gt-norm, norm, yes, diff-lst-year, scattered, minor, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, normal, lt-norm, norm, no, diff-lst-year, whole-field, minor, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
april, ?, ?, ?, ?, ?, scattered, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, no, same-lst-yr, scattered, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, normal, lt-norm, norm, no, same-lst-sev-yrs, scattered, pot-severe, none, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, normal, gt-norm, lt-norm, no, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
june, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
june, normal, norm, norm, no, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
may, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, ?, ?, ?, ?, same-lst-sev-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
august, lt-normal, norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
july, lt-normal, gt-norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, ?, ?, ?, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
july, normal, norm, norm, yes, same-lst-two-yrs, scattered, minor, none, 90-100%, abnorm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
april, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
june, lt-normal, norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, galls-cysts, bacterial-pustule
october, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
october, normal, norm, gt-norm, no, same-lst-sev-yrs, whole-field, minor, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, present, norm, absent, norm, alternarialeaf-spot
june, lt-normal, gt-norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
august, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
september, normal, norm, norm, yes, same-lst-two-yrs, low-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, lt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
june, normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, lt-normal, lt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, norm, norm, no, same-lst-yr, whole-field, minor, none, 80-89%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, lt-norm, absent, norm, bacterial-pustule
october, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, gt-norm, lt-norm, no, diff-lst-year, scattered, minor, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, norm, no, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
may, lt-normal, norm, norm, yes, same-lst-yr, scattered, minor, fungicide, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, lt-norm, absent, rotted, bacterial-pustule
june, lt-normal, norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, norm, gt-norm, ?, same-lst-sev-yrs, scattered, ?, ?, lt-80%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
september, lt-normal, norm, gt-norm, no, same-lst-two-yrs, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
april, lt-normal, gt-norm, norm, no, same-lst-yr, low-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
may, lt-normal, lt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, other, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
october, normal, norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
august, lt-normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
august, normal, norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
october, normal, norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, gt-norm, lt-norm, no, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
june, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, norm, no, same-lst-sev-yrs, scattered, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
may, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
may, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, gt-norm, yes, same-lst-yr, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, lt-norm, no, same-lst-two-yrs, upper-areas, minor, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, normal, lt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, lt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
july, normal, gt-norm, norm, yes, diff-lst-year, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
april, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, other, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
october, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
may, normal, norm, norm, no, same-lst-sev-yrs, low-areas, minor, fungicide, 80-89%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, lt-norm, absent, norm, bacterial-pustule
june, normal, gt-norm, lt-norm, no, diff-lst-year, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
july, normal, lt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, lt-normal, gt-norm, gt-norm, no, same-lst-yr, low-areas, minor, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
april, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, lt-normal, norm, norm, no, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, colored, norm, absent, absent, norm, absent, norm, brown-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, lt-normal, gt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
july, ?, ?, ?, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
june, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
july, ?, ?, ?, ?, same-lst-sev-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
september, normal, lt-norm, gt-norm, yes, diff-lst-year, upper-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
july, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
july, lt-normal, gt-norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, gt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
october, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, upper-areas, minor, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, lt-normal, norm, lt-norm, yes, diff-lst-year, scattered, minor, other, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, lt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, lt-normal, norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, gt-norm, norm, no, diff-lst-year, scattered, pot-severe, fungicide, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, no, above-soil, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, lt-normal, norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
july, normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, whole-field, minor, none, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
july, lt-normal, gt-norm, gt-norm, no, same-lst-yr, low-areas, minor, other, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, gt-norm, norm, yes, diff-lst-year, upper-areas, pot-severe, none, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, lt-norm, absent, rotted, bacterial-pustule
july, normal, lt-norm, norm, no, diff-lst-year, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
august, normal, lt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, normal, gt-norm, norm, no, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, lt-normal, norm, norm, ?, diff-lst-year, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, normal, gt-norm, norm, yes, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, brown, absent, firm-and-dry, absent, none, absent, norm, absent, abnorm, absent, present, lt-norm, present, norm, anthracnose
june, normal, lt-norm, gt-norm, no, same-lst-sev-yrs, scattered, pot-severe, fungicide, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
june, lt-normal, norm, gt-norm, yes, same-lst-sev-yrs, scattered, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
may, lt-normal, gt-norm, norm, no, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
august, normal, norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, gt-norm, no, same-lst-yr, scattered, pot-severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, normal, lt-norm, gt-norm, no, same-lst-yr, whole-field, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
may, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, lt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
april, lt-normal, norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, severe, other, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, present, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, no, diff-lst-year, scattered, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, scattered, pot-severe, other, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, no, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, norm, norm, ?, diff-lst-year, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
april, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, other, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, gt-norm, norm, yes, diff-lst-year, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, gt-norm, norm, no, diff-lst-year, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
august, normal, gt-norm, lt-norm, no, diff-lst-year, low-areas, severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, lt-normal, norm, norm, yes, same-lst-yr, scattered, minor, none, lt-80%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
june, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, normal, norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
october, ?, gt-norm, gt-norm, ?, same-lst-two-yrs, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
september, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-yr, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, lt-normal, ?, lt-norm, ?, diff-lst-year, whole-field, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
october, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, norm, norm, no, same-lst-two-yrs, whole-field, minor, none, 80-89%, abnorm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, lt-norm, absent, norm, bacterial-pustule
april, lt-normal, norm, norm, no, same-lst-two-yrs, low-areas, severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
july, ?, ?, ?, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
august, normal, lt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
may, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, lt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
june, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
october, normal, lt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, whole-field, ?, ?, 90-100%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
june, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, other, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, norm, gt-norm, ?, diff-lst-year, scattered, ?, ?, lt-80%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
april, lt-normal, gt-norm, lt-norm, yes, diff-lst-year, low-areas, pot-severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, rotted, rhizoctonia-root-rot
august, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, ?, gt-norm, gt-norm, ?, same-lst-yr, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
july, lt-normal, norm, norm, yes, diff-lst-year, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, normal, lt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
june, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, lt-norm, yes, diff-lst-year, low-areas, severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, lt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, lt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, normal, lt-norm, norm, no, same-lst-yr, low-areas, pot-severe, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, lt-normal, norm, norm, yes, same-lst-two-yrs, whole-field, minor, none, lt-80%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
july, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, low-areas, severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
september, normal, lt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, lt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
july, normal, lt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
july, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
may, lt-normal, gt-norm, gt-norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, normal, gt-norm, lt-norm, no, diff-lst-year, scattered, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
august, lt-normal, gt-norm, gt-norm, no, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, absent, norm, anthracnose
august, lt-normal, norm, gt-norm, yes, same-lst-yr, upper-areas, minor, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
july, lt-normal, norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, normal, gt-norm, norm, yes, diff-lst-year, low-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, ?, ?, ?, ?, ?, scattered, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
july, normal, gt-norm, norm, no, diff-lst-year, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, lt-normal, norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, ?, ?, ?, ?, ?, low-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
april, lt-normal, norm, norm, no, same-lst-two-yrs, upper-areas, minor, other, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, lt-normal, ?, lt-norm, ?, same-lst-yr, scattered, ?, ?, ?, abnorm, abnorm, no-yellow-halos, no-w-s-marg, gt-1/8, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
july, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, no, same-lst-sev-yrs, low-areas, severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, below-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-yr, scattered, severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, gt-norm, lt-norm, no, same-lst-sev-yrs, low-areas, severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, below-soil, brown, absent, firm-and-dry, present, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
may, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, lt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
august, normal, lt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, lt-normal, norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, ?, diff-lst-year, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, lt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
august, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, norm, lt-norm, no, same-lst-sev-yrs, upper-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
may, lt-normal, ?, lt-norm, ?, same-lst-yr, whole-field, ?, ?, ?, abnorm, abnorm, no-yellow-halos, no-w-s-marg, gt-1/8, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
august, normal, norm, lt-norm, no, same-lst-two-yrs, whole-field, pot-severe, fungicide, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, whole-field, ?, ?, 90-100%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, lt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, gt-norm, norm, yes, same-lst-yr, scattered, severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
september, normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, whole-field, ?, ?, 90-100%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
october, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, present, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, norm, absent, norm, anthracnose
june, normal, norm, gt-norm, no, same-lst-two-yrs, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, norm, absent, norm, bacterial-pustule
july, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
october, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, abnorm, absent, present, lt-norm, present, norm, anthracnose
july, ?, ?, ?, ?, same-lst-sev-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
august, normal, lt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, lt-norm, gt-norm, no, diff-lst-year, upper-areas, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dk-brown-blk, present, firm-and-dry, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
august, normal, norm, norm, yes, same-lst-yr, scattered, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, brown-spot
october, normal, norm, norm, no, same-lst-two-yrs, upper-areas, minor, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, upper-areas, pot-severe, none, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, rotted, bacterial-pustule
september, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, normal, lt-norm, gt-norm, no, same-lst-yr, whole-field, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, lt-normal, norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, norm, yes, diff-lst-year, low-areas, severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
august, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
may, normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
july, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
july, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, lt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, lt-normal, lt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
may, lt-normal, lt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, other, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
september, normal, lt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, lt-normal, gt-norm, norm, yes, diff-lst-year, whole-field, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, lt-normal, norm, norm, yes, same-lst-two-yrs, scattered, minor, fungicide, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, rotted, bacterial-pustule
august, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
june, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, severe, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, low-areas, severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
june, lt-normal, gt-norm, norm, no, same-lst-yr, low-areas, severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
june, lt-normal, gt-norm, lt-norm, yes, diff-lst-year, low-areas, pot-severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, ?, ?, ?, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
july, lt-normal, norm, gt-norm, yes, same-lst-yr, scattered, minor, none, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, lt-normal, lt-norm, lt-norm, yes, same-lst-two-yrs, scattered, severe, none, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, lt-normal, norm, norm, yes, same-lst-two-yrs, whole-field, severe, none, 90-100%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
june, lt-normal, norm, norm, no, same-lst-two-yrs, upper-areas, pot-severe, fungicide, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, lt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
august, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, lt-norm, norm, no, same-lst-two-yrs, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
september, normal, gt-norm, gt-norm, no, same-lst-yr, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
may, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, gt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, lt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, normal, lt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, fungicide, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
april, ?, ?, ?, ?, ?, whole-field, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
june, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, ?, gt-norm, gt-norm, ?, same-lst-two-yrs, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
august, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
june, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
october, normal, gt-norm, norm, yes, diff-lst-year, whole-field, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, lt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, norm, no, same-lst-two-yrs, upper-areas, minor, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, above-sec-nde, brown, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, lt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, minor, other, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, norm, absent, abnorm, absent, present, lt-norm, present, norm, anthracnose
june, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, low-areas, severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, lt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
october, lt-normal, norm, gt-norm, no, same-lst-yr, low-areas, minor, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, lt-norm, norm, no, same-lst-yr, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
october, normal, gt-norm, norm, yes, same-lst-yr, low-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
april, normal, norm, norm, yes, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
july, ?, ?, ?, ?, same-lst-two-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
june, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
may, normal, lt-norm, norm, no, diff-lst-year, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, normal, lt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
july, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, norm, lt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, lt-normal, gt-norm, norm, yes, same-lst-yr, upper-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, lt-norm, yes, same-lst-yr, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, gt-norm, ?, same-lst-two-yrs, whole-field, ?, ?, 90-100%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
may, lt-normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, normal, lt-norm, gt-norm, no, same-lst-sev-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
july, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, no, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
june, lt-normal, gt-norm, gt-norm, no, diff-lst-year, scattered, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, brown, absent, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, anthracnose
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, lt-normal, gt-norm, gt-norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, norm, norm, no, diff-lst-year, scattered, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-yr, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, ?, ?, ?, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
august, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, rotted, bacterial-pustule
october, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, ?, ?, ?, ?, ?, scattered, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
may, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, norm, lt-norm, no, same-lst-two-yrs, upper-areas, minor, fungicide, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, lt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, lt-normal, lt-norm, norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
may, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, lt-norm, absent, rotted, bacterial-pustule
june, normal, gt-norm, norm, no, same-lst-yr, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
june, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
october, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, normal, gt-norm, norm, yes, diff-lst-year, scattered, minor, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
may, lt-normal, norm, norm, no, same-lst-two-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, lt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
july, normal, lt-norm, lt-norm, no, same-lst-yr, scattered, severe, none, 90-100%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
june, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, 90-100%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, gt-norm, lt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, norm, yes, diff-lst-year, whole-field, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, lt-norm, yes, same-lst-two-yrs, upper-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
october, normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, whole-field, ?, ?, 80-89%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
october, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, diff-lst-year, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, lt-normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
august, ?, ?, ?, ?, ?, whole-field, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
october, normal, gt-norm, gt-norm, ?, same-lst-two-yrs, whole-field, ?, ?, 80-89%, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
june, ?, ?, ?, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
september, normal, lt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
june, lt-normal, gt-norm, gt-norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, lt-normal, gt-norm, norm, no, same-lst-yr, whole-field, pot-severe, fungicide, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, norm, norm, yes, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, none, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, absent, absent, none, absent, norm, colored, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
august, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, upper-areas, minor, other, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
june, lt-normal, norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
june, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
may, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, lt-norm, lt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, low-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, lt-norm, yes, diff-lst-year, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
july, normal, gt-norm, norm, no, same-lst-sev-yrs, whole-field, minor, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, norm, yes, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
may, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
may, normal, gt-norm, norm, yes, diff-lst-year, scattered, minor, other, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, brown, absent, firm-and-dry, absent, none, absent, norm, absent, abnorm, absent, present, norm, present, norm, anthracnose
june, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, ?, ?, ?, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
june, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, lt-normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, lt-norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 80-89%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, lt-normal, gt-norm, gt-norm, no, same-lst-two-yrs, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, lt-norm, norm, no, same-lst-sev-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
july, normal, gt-norm, norm, no, same-lst-yr, upper-areas, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
october, normal, norm, lt-norm, no, diff-lst-year, scattered, minor, fungicide, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
october, ?, gt-norm, gt-norm, ?, same-lst-yr, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
october, normal, gt-norm, lt-norm, no, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
june, lt-normal, norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
may, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, anthracnose
july, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, absent, tan, present, firm-and-dry, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, norm, norm, yes, same-lst-yr, whole-field, pot-severe, other, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, absent, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
october, normal, lt-norm, norm, no, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, lt-norm, gt-norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
august, normal, gt-norm, norm, no, diff-lst-year, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, no, absent, tan, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, lt-80%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
october, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, scattered, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
april, normal, norm, norm, yes, same-lst-two-yrs, scattered, minor, other, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, normal, norm, lt-norm, yes, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-pustule
july, lt-normal, lt-norm, lt-norm, yes, same-lst-sev-yrs, scattered, pot-severe, fungicide, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, no, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
september, ?, ?, ?, ?, ?, scattered, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
september, normal, lt-norm, gt-norm, yes, same-lst-two-yrs, scattered, pot-severe, fungicide, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
july, lt-normal, gt-norm, gt-norm, no, same-lst-two-yrs, scattered, pot-severe, none, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
september, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
may, lt-normal, gt-norm, gt-norm, no, same-lst-sev-yrs, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, gt-norm, norm, no, same-lst-yr, scattered, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
september, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
july, lt-normal, norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
october, normal, norm, norm, no, diff-lst-year, scattered, minor, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
august, normal, gt-norm, norm, yes, same-lst-yr, scattered, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, norm, lt-norm, yes, same-lst-two-yrs, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, no-w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, rotted, bacterial-pustule
september, normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
april, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, low-areas, severe, none, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, brown, absent, firm-and-dry, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, rhizoctonia-root-rot
august, ?, ?, ?, ?, same-lst-sev-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
september, lt-normal, norm, lt-norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, abnorm, norm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, severe, fungicide, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
july, ?, ?, ?, ?, ?, whole-field, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
august, normal, lt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, absent, dna, absent, absent, absent, brown, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, ?, gt-norm, gt-norm, ?, same-lst-sev-yrs, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
august, normal, norm, norm, no, same-lst-sev-yrs, low-areas, minor, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, lt-normal, norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, below-soil, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
june, ?, ?, ?, ?, ?, low-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
april, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
june, lt-normal, gt-norm, norm, ?, same-lst-yr, low-areas, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, ?, above-soil, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, lt-norm, lt-norm, no, same-lst-sev-yrs, upper-areas, severe, none, 80-89%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
october, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
april, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, 80-89%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, below-soil, dk-brown-blk, absent, absent, absent, none, absent, dna, dna, norm, absent, absent, norm, absent, norm, phytophthora-rot
june, lt-normal, gt-norm, gt-norm, yes, same-lst-yr, upper-areas, pot-severe, fungicide, lt-80%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, rotted, bacterial-pustule
july, normal, gt-norm, norm, no, same-lst-yr, low-areas, pot-severe, none, 80-89%, norm, abnorm, yellow-halos, no-w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, present, lt-norm, absent, rotted, bacterial-pustule
august, normal, gt-norm, gt-norm, yes, same-lst-yr, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, lt-normal, gt-norm, gt-norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
april, lt-normal, ?, lt-norm, ?, diff-lst-year, scattered, ?, ?, ?, abnorm, abnorm, absent, dna, dna, absent, present, ?, abnorm, ?, ?, ?, ?, ?, ?, ?, ?, dna, ?, ?, ?, ?, ?, ?, rotted, herbicide-injury
june, normal, gt-norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, lt-norm, norm, no, same-lst-two-yrs, whole-field, minor, none, lt-80%, abnorm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, present, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
september, lt-normal, gt-norm, gt-norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
june, normal, norm, norm, yes, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
august, normal, gt-norm, gt-norm, no, diff-lst-year, scattered, minor, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, lt-1/8, absent, absent, absent, norm, no, absent, tan, absent, absent, absent, none, absent, diseased, colored, abnorm, absent, present, norm, absent, norm, purple-seed-stain
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, scattered, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, normal, gt-norm, norm, yes, same-lst-yr, scattered, minor, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, pot-severe, none, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
july, normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, none, 90-100%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, norm, absent, absent, norm, absent, norm, anthracnose
july, normal, norm, norm, yes, same-lst-yr, scattered, minor, none, 80-89%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
may, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
october, normal, gt-norm, norm, yes, same-lst-yr, low-areas, pot-severe, fungicide, 80-89%, norm, norm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, absent, lt-norm, present, norm, anthracnose
october, normal, norm, norm, no, same-lst-two-yrs, scattered, pot-severe, fungicide, 90-100%, abnorm, norm, absent, dna, gt-1/8, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, brown, absent, norm, dna, norm, absent, absent, norm, absent, norm, brown-stem-rot
september, normal, gt-norm, gt-norm, yes, same-lst-two-yrs, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, alternarialeaf-spot
july, normal, norm, norm, yes, same-lst-yr, upper-areas, minor, none, 90-100%, norm, abnorm, yellow-halos, w-s-marg, lt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, bacterial-blight
april, normal, gt-norm, norm, yes, same-lst-sev-yrs, whole-field, pot-severe, fungicide, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-soil, dk-brown-blk, absent, firm-and-dry, absent, none, absent, norm, absent, abnorm, present, absent, lt-norm, absent, norm, anthracnose
october, ?, gt-norm, gt-norm, ?, diff-lst-year, whole-field, ?, ?, ?, norm, norm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dna, present, absent, absent, none, absent, diseased, brown-w/blk-specks, abnorm, present, present, lt-norm, present, ?, diaporthe-pod-&-stem-blight
may, lt-normal, gt-norm, norm, yes, same-lst-two-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, norm, gt-norm, yes, same-lst-yr, low-areas, minor, fungicide, 80-89%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
april, ?, ?, ?, ?, ?, upper-areas, ?, ?, ?, ?, abnorm, absent, dna, dna, ?, present, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 2-4-d-injury
september, normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
july, lt-normal, gt-norm, norm, ?, same-lst-two-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, absent, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
september, normal, norm, lt-norm, no, diff-lst-year, scattered, minor, none, 80-89%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
september, normal, gt-norm, norm, yes, same-lst-two-yrs, scattered, pot-severe, fungicide, 90-100%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, dna, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
september, lt-normal, norm, gt-norm, no, diff-lst-year, scattered, pot-severe, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, absent, present, norm, absent, norm, alternarialeaf-spot
august, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, upper-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
may, normal, lt-norm, lt-norm, no, diff-lst-year, scattered, minor, none, 90-100%, norm, abnorm, absent, dna, dna, absent, absent, upper-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, powdery-mildew
august, normal, gt-norm, norm, yes, same-lst-two-yrs, whole-field, minor, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, normal, norm, norm, yes, same-lst-two-yrs, upper-areas, pot-severe, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
august, normal, gt-norm, gt-norm, yes, same-lst-yr, whole-field, pot-severe, fungicide, 90-100%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, dk-brown-blk, absent, firm-and-dry, absent, none, absent, diseased, colored, norm, absent, absent, norm, absent, norm, frog-eye-leaf-spot
june, ?, ?, ?, ?, same-lst-two-yrs, upper-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, norm, ?, ?, ?, ?, ?, ?, ?, ?, few-present, ?, abnorm, absent, ?, lt-norm, ?, galls-cysts, cyst-nematode
may, lt-normal, gt-norm, norm, ?, diff-lst-year, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, absent, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
may, lt-normal, gt-norm, norm, ?, same-lst-sev-yrs, low-areas, ?, ?, ?, abnorm, abnorm, ?, ?, ?, ?, ?, ?, abnorm, ?, above-sec-nde, dk-brown-blk, ?, watery, absent, none, absent, ?, ?, ?, ?, ?, ?, ?, rotted, phytophthora-rot
august, normal, gt-norm, norm, no, same-lst-yr, scattered, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, firm-and-dry, absent, none, absent, norm, dna, norm, absent, absent, norm, absent, norm, diaporthe-stem-canker
june, lt-normal, norm, gt-norm, yes, same-lst-sev-yrs, whole-field, minor, other, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, present, absent, absent, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, phyllosticta-leaf-spot
september, normal, lt-norm, gt-norm, no, same-lst-two-yrs, upper-areas, pot-severe, none, lt-80%, abnorm, abnorm, absent, dna, dna, absent, absent, absent, abnorm, no, absent, tan, absent, absent, absent, black, present, norm, dna, norm, absent, absent, norm, absent, norm, charcoal-rot
september, lt-normal, gt-norm, lt-norm, yes, same-lst-sev-yrs, whole-field, minor, fungicide, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, lower-surf, norm, yes, absent, dna, absent, absent, absent, none, absent, norm, absent, abnorm, present, absent, norm, absent, norm, downy-mildew
july, lt-normal, gt-norm, norm, yes, same-lst-sev-yrs, low-areas, pot-severe, none, lt-80%, norm, abnorm, no-yellow-halos, w-s-marg, gt-1/8, absent, absent, absent, abnorm, yes, above-sec-nde, brown, present, absent, absent, none, absent, norm, absent, norm, absent, absent, norm, absent, norm, brown-spot
