package Mo::utils::Language;

use base qw(Exporter);
use strict;
use warnings;

use Error::Pure qw(err);
use List::Util qw(none);
use Locale::Language;
use Readonly;

Readonly::Array our @EXPORT_OK => qw(check_language);

our $VERSION = 0.01;

sub check_language {
	my ($self, $key) = @_;

	_check_key($self, $key) && return;

	if (none { $_ eq $self->{$key} } all_language_codes()) {
		err "Language code '".$self->{$key}."' isn't ISO 639-1 code.";
	}

	return;
}

sub _check_key {
	my ($self, $key) = @_;

	if (! exists $self->{$key} || ! defined $self->{$key}) {
		return 1;
	}

	return 0;
}

1;

__END__

=pod

=encoding utf8

=head1 NAME

Mo::utils::Language - Mo language utilities.

=head1 SYNOPSIS

 use Mo::utils qw(check_language);

 check_language($self, $key);

=head1 DESCRIPTION

Mo language utilities for checking of data objects.

=head1 SUBROUTINES

=head2 C<check_language>

 check_language($self, $key);

Check parameter defined by C<$key> if it's ISO 639-1 language code and if language exists.
Value could be undefined.

Returns undef.

=head1 ERRORS

 check_language():
         Language code '%s' isn't ISO 639-1 code.
         Language with ISO 639-1 code '%s' doesn't exist.

=head1 DEPENDENCIES

L<Error::Pure>,
L<Exporter>,
L<List::Util>,
L<Locale::Language>,
L<Readonly>.

=head1 SEE ALSO

=over

=item L<Mo>

Micro Objects. Mo is less.

=item L<Mo::utils>

Mo utilities.

=back

=head1 REPOSITORY

L<https://github.com/michal-josef-spacek/Mo-utils-Language>

=head1 AUTHOR

Michal Josef Špaček L<mailto:skim@cpan.org>

L<http://skim.cz>

=head1 LICENSE AND COPYRIGHT

© 2022-2023 Michal Josef Špaček

BSD 2-Clause License

=head1 VERSION

0.01

=cut
