#! /bin/false

# Copyright (C) 2019 Guido Flohr <guido.flohr@cantanea.com>,
# all rights reserved.

# This program is free software. It comes without any warranty, to
# the extent permitted by applicable law. You can redistribute it
# and/or modify it under the terms of the Do What the Fuck You Want
# to Public License, Version 2, as published by Sam Hocevar. See
# http://www.wtfpl.net/ for more details.

# Make Dist::Zilla happy.
# ABSTRACT: Read chess opening books in polyglot format

package Chess::Opening::ECO;
$Chess::Opening::ECO::VERSION = '0.6';
use common::sense;

use Locale::TextDomain 'com.cantanea.Chess-Opening';

sub positions {
	return {
		'rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00a',
			# TRANSLATORS: A00a:
			variation => N__('Start'),
			moves => {
				'a2a3' => 1,
				'a2a4' => 1,
				'b1a3' => 1,
				'b1c3' => 1,
				'b2b3' => 1,
				'b2b4' => 1,
				'c2c3' => 1,
				'c2c4' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'f2f3' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
				'g1h3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
				'h2h3' => 1,
				'h2h4' => 1,
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/5P2/PPPPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A00b',
			# TRANSLATORS: A00b: 1. f3
			variation => N__('Barnes Opening'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/5P2/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A00b',
			# TRANSLATORS: A00b: 1. f3 e5
			variation => N__('Barnes Opening'),
			moves => {
				'e1f2' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/5P2/PPPPPKPP/RNBQ1BNR b kq' => {
			eco => 'A00b',
			# TRANSLATORS: A00b: 1. f3 e5 2. Kf2
			variation => N__('Fried fox'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/7P/8/PPPPPPP1/RNBQKBNR b KQkq' => {
			eco => 'A00c',
			# TRANSLATORS: A00c: 1. h4
			variation => N__('Kadas Opening'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/7P/PPPPPPP1/RNBQKBNR b KQkq' => {
			eco => 'A00d',
			# TRANSLATORS: A00d: 1. h3
			variation => N__('Clemenz Opening'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/P7/8/1PPPPPPP/RNBQKBNR b KQkq' => {
			eco => 'A00e',
			# TRANSLATORS: A00e: 1. a4
			variation => N__('Ware Opening'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/P7/1PPPPPPP/RNBQKBNR b KQkq' => {
			eco => 'A00f',
			# TRANSLATORS: A00f: 1. a3
			variation => N__('Anderssen Opening'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/P7/1PPPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00f',
			# TRANSLATORS: A00f: 1. a3 e5
			variation => N__('Anderssen Opening'),
			moves => {
				'h2h3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/P6P/1PPPPPP1/RNBQKBNR b KQkq' => {
			eco => 'A00f',
			# TRANSLATORS: A00f: 1. a3 e5 2. h3
			variation => N__('Anderssen Opening'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/8/P6P/1PPPPPP1/RNBQKBNR w KQkq' => {
			eco => 'A00f',
			# TRANSLATORS: A00f: 1. a3 e5 2. h3 d5
			variation => N__('Creepy Crawly Opening (Basman)'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/7N/PPPPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A00g',
			# TRANSLATORS: A00g: 1. Nh3
			variation => N__('Amar/Paris Opening'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/7N/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A00g',
			# TRANSLATORS: A00g: 1. Nh3 d5
			variation => N__('Amar/Paris Opening'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/6PN/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A00g',
			# TRANSLATORS: A00g: 1. Nh3 d5 2. g3
			variation => N__('Amar/Paris Opening'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/8/6PN/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A00g',
			# TRANSLATORS: A00g: 1. Nh3 d5 2. g3 e5
			variation => N__('Amar/Paris Opening'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/5P2/6PN/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A00g',
			# TRANSLATORS: A00g: 1. Nh3 d5 2. g3 e5 3. f4
			variation => N__('Amar: Paris Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/N7/PPPPPPPP/R1BQKBNR b KQkq' => {
			eco => 'A00h',
			# TRANSLATORS: A00h: 1. Na3
			variation => N__('Durkin'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/2P5/PP1PPPPP/RNBQKBNR b KQkq' => {
			eco => 'A00i',
			# TRANSLATORS: A00i: 1. c3
			variation => N__('Saragossa'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/3P4/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A00j',
			# TRANSLATORS: A00j: 1. d3
			variation => N__('Mieses'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/3P4/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A00j',
			# TRANSLATORS: A00j: 1. d3 d5
			variation => N__('Mieses: 1... d5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/3P4/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A00j',
			# TRANSLATORS: A00j: 1. d3 e5
			variation => N__('Mieses: 1... e5'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/8/3P4/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A00j',
			# TRANSLATORS: A00j: 1. d3 g6
			variation => N__('Mieses'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/6P1/3P4/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A00j',
			# TRANSLATORS: A00j: 1. d3 g6 2. g4
			variation => N__('Spike Deferred'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/4P3/PPPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'A00k',
			# TRANSLATORS: A00k: 1. e3
			variation => N__('Van Kruijs'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/2N5/PPPPPPPP/R1BQKBNR b KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3
			variation => N__('Van Geet (Dunst) Opening'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/2N5/PPPPPPPP/R1BQKBNR w KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 c5
			variation => N__('Van Geet: Sicilian Variation'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/2N2N2/PPPPPPPP/R1BQKB1R b KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 c5 2. Nf3
			variation => N__('Van Geet: Sicilian Variation, 2. Nf3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/8/2N2N2/PPPPPPPP/R1BQKB1R w KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 c5 2. Nf3 Nc6
			variation => N__('Van Geet: Sicilian Variation, 2. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/2N5/PPPPPPPP/R1BQKBNR w KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 e5
			variation => N__('Van Geet: 1... e5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/2N2N2/PPPPPPPP/R1BQKB1R b KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 e5 2. Nf3
			variation => N__('Van Geet: 1... e5 2. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/2N5/PPPPPPPP/R1BQKBNR w KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 Nf6
			variation => N__('Van Geet: 1... Nf6'),
			moves => {
				'g1f3' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/2N2N2/PPPPPPPP/R1BQKB1R b KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 Nf6 2. Nf3
			variation => N__('Van Geet: 1... Nf6 2. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/6P1/2N5/PPPPPP1P/R1BQKBNR b KQkq' => {
			eco => 'A00l',
			# TRANSLATORS: A00l: 1. Nc3 Nf6 2. g4
			variation => N__('Van Geet: Tübingen Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/2N5/PPPPPPPP/R1BQKBNR w KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5
			variation => N__('Van Geet: 1... d5'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. e4
			variation => N__('Van Geet: 1... d5 2. e4'),
			moves => {
				'd5d4' => 1,
				'd5e4' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/3pP3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. e4 d4
			variation => N__('Van Geet: 1... d5 2. e4 d4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/4p3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. e4 dxe4
			variation => N__('Van Geet: 1... d5 2. e4 dxe4'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2B1p3/2N5/PPPP1PPP/R1BQK1NR b KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. e4 dxe4 3. Bc4
			variation => N__('Van Geet: Hector Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/2N2N2/PPPPPPPP/R1BQKB1R b KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. Nf3
			variation => N__('Van Geet: 1... d5 2. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/2N2N2/PPPPPPPP/R1BQKB1R w KQkq' => {
			eco => 'A00m',
			# TRANSLATORS: A00m: 1. Nc3 d5 2. Nf3 Nf6
			variation => N__('Van Geet: 1... d5 2. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/6P1/8/PPPPPP1P/RNBQKBNR b KQkq' => {
			eco => 'A00n',
			# TRANSLATORS: A00n: 1. g4
			variation => N__('Grob'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
				'f7f5' => 1,
				'g7g5' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/6P1/8/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00n',
			# TRANSLATORS: A00n: 1. g4 e5
			variation => N__('Grob: 1... e5'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/6P1/8/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00n',
			# TRANSLATORS: A00n: 1. g4 f5
			variation => N__('Grob: Alessi Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/6P1/8/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00n',
			# TRANSLATORS: A00n: 1. g4 g5
			variation => N__('Grob: Double Grob'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/6P1/8/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5
			variation => N__('Grob: 1... d5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/6P1/8/PPPPPPBP/RNBQK1NR b KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2
			variation => N__('Grob Gambit'),
			moves => {
				'c7c6' => 1,
				'c8g4' => 1,
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/6P1/8/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 c6
			variation => N__('Grob Gambit: 2... c6'),
			moves => {
				'g4g5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p2P1/8/8/PPPPPPBP/RNBQK1NR b KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 c6 3. g5
			variation => N__('Grob Gambit: Spike Attack'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/6b1/8/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 Bxg4
			variation => N__('Grob Gambit Accepted'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/2P3b1/8/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 Bxg4 3. c4
			variation => N__('Grob Gambit Accepted: Fritz Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/6P1/8/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 e5
			variation => N__('Grob Gambit: e5'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/2P3P1/8/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A00o',
			# TRANSLATORS: A00o: 1. g4 d5 2. Bg2 e5 3. c4
			variation => N__('Grob Gambit: Hurst Attack'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/1P6/8/P1PPPPPP/RNBQKBNR b KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4
			variation => N__('Polish (Sokolsky; Orang-Utan)'),
			moves => {
				'c7c5' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/1P6/8/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 c5
			variation => N__('Polish: Birmingham Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/1P6/8/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 c6
			variation => N__('Polish: 1... c6'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/1P6/8/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 c6 2. Bb2
			variation => N__('Polish: 1... c6'),
			moves => {
				'a7a5' => 1,
			},
		},
		'rnbqkbnr/1p1ppppp/2p5/p7/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 c6 2. Bb2 a5
			variation => N__('Polish: 1... c6'),
			moves => {
				'b4b5' => 1,
			},
		},
		'rnbqkbnr/1p1ppppp/2p5/pP6/8/8/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 c6 2. Bb2 a5 3. b5
			variation => N__('Polish: Schühler Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/1P6/8/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 Nf6
			variation => N__('Polish: 1... Nf6'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/1P6/8/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 Nf6 2. Bb2
			variation => N__('Polish: 1... Nf6 2. Bb2'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00p',
			# TRANSLATORS: A00p: 1. b4 Nf6 2. Bb2 e6
			variation => N__('Polish: 1... Nf6 2. Bb2 e6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/1P6/8/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00q',
			# TRANSLATORS: A00q: 1. b4 d5
			variation => N__('Polish: 1... d5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/1P6/8/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00q',
			# TRANSLATORS: A00q: 1. b4 d5 2. Bb2
			variation => N__('Polish: 1... d5 2. Bb2'),
			moves => {
				'c8f5' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p1b2/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00q',
			# TRANSLATORS: A00q: 1. b4 d5 2. Bb2 Bf5
			variation => N__('Polish: 1... d5 2. Bb2 Bf5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00q',
			# TRANSLATORS: A00q: 1. b4 d5 2. Bb2 Nf6
			variation => N__('Polish: 1... d5 2. Bb2 Nf6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/1P6/4P3/PBPP1PPP/RN1QKBNR b KQkq' => {
			eco => 'A00q',
			# TRANSLATORS: A00q: 1. b4 d5 2. Bb2 Nf6 3. e3
			variation => N__('Polish: 1... d5 2. Bb2 Nf6 3. e3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/1P6/8/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5
			variation => N__('Polish: 1... e5'),
			moves => {
				'a2a3' => 1,
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/1P6/P7/2PPPPPP/RNBQKBNR b KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. a3
			variation => N__('Polish: Bugayev Attack'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/1P6/8/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2
			variation => N__('Polish: 1... e5 2. Bb2'),
			moves => {
				'c7c5' => 1,
				'd7d6' => 1,
				'f7f6' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2p1p3/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2 c5
			variation => N__('Polish: Wolfertz Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2 d6
			variation => N__('Polish: 1... e5 2. Bb2 d6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/5p2/4p3/1P6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2 f6
			variation => N__('Polish: 1... e5 2. Bb2 f6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/5p2/4p3/1P2P3/8/PBPP1PPP/RN1QKBNR b KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2 f6 3. e4
			variation => N__('Polish: 1... e5 2. Bb2 f6'),
			moves => {
				'f8b4' => 1,
			},
		},
		'rnbqk1nr/pppp2pp/5p2/4p3/1b2P3/8/PBPP1PPP/RN1QKBNR w KQkq' => {
			eco => 'A00r',
			# TRANSLATORS: A00r: 1. b4 e5 2. Bb2 f6 3. e4 Bxb4
			variation => N__('Polish: Tartakower Gambit'),
			moves => {
			},
		},
		'rnbqk1nr/pppp1ppp/8/4p3/1b6/8/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00s',
			# TRANSLATORS: A00s: 1. b4 e5 2. Bb2 Bxb4
			variation => N__('Polish: 2...Bxb4'),
			moves => {
				'b2e5' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/8/4B3/1b6/8/P1PPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00s',
			# TRANSLATORS: A00s: 1. b4 e5 2. Bb2 Bxb4 3. Bxe5
			variation => N__('Polish: 2...Bxb4 3.Bxe5'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4B3/1b6/8/P1PPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A00s',
			# TRANSLATORS: A00s: 1. b4 e5 2. Bb2 Bxb4 3. Bxe5 Nf6
			variation => N__('Polish: 2...Bxb4 3.Bxe5 Nf6'),
			moves => {
				'c2c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4B3/1bP5/8/P2PPPPP/RN1QKBNR b KQkq' => {
			eco => 'A00s',
			# TRANSLATORS: A00s: 1. b4 e5 2. Bb2 Bxb4 3. Bxe5 Nf6 4. c4
			variation => N__('Polish: 2...Bxb4 3.Bxe5 Nf6 4. c4'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4B3/1b6/5N2/P1PPPPPP/RN1QKB1R b KQkq' => {
			eco => 'A00s',
			# TRANSLATORS: A00s: 1. b4 e5 2. Bb2 Bxb4 3. Bxe5 Nf6 4. Nf3
			variation => N__('Polish: 2...Bxb4 3.Bxe5 Nf6 4. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/6P1/PPPPPP1P/RNBQKBNR b KQkq' => {
			eco => 'A00t',
			# TRANSLATORS: A00t: 1. g3
			variation => N__('Benko Opening'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/8/6P1/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00t',
			# TRANSLATORS: A00t: 1. g3 g6
			variation => N__('Benko Opening: Symmetrical'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/6P1/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00t',
			# TRANSLATORS: A00t: 1. g3 Nf6
			variation => N__('Benko Opening'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/6P1/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00u',
			# TRANSLATORS: A00u: 1. g3 e5
			variation => N__('Benko Opening'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/6P1/PPPPPP1P/RNBQKBNR w KQkq' => {
			eco => 'A00v',
			# TRANSLATORS: A00v: 1. g3 d5
			variation => N__('Benko Opening'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/6P1/PPPPPPBP/RNBQK1NR b KQkq' => {
			eco => 'A00v',
			# TRANSLATORS: A00v: 1. g3 d5 2. Bg2
			variation => N__('Benko Opening'),
			moves => {
				'c7c6' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/8/6P1/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00v',
			# TRANSLATORS: A00v: 1. g3 d5 2. Bg2 c6
			variation => N__('Benko Opening'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/8/3pp3/8/6P1/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00v',
			# TRANSLATORS: A00v: 1. g3 d5 2. Bg2 e5
			variation => N__('Benko Opening'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/6P1/PPPPPPBP/RNBQK1NR w KQkq' => {
			eco => 'A00v',
			# TRANSLATORS: A00v: 1. g3 d5 2. Bg2 Nf6
			variation => N__('Benko Opening'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/1P6/P1PPPPPP/RNBQKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3
			variation => N__('Nimzowitsch-Larsen Attack'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'c7c5' => 1,
				'd7d5' => 1,
				'e7e5' => 1,
				'f7f5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/8/1p6/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 b5
			variation => N__('Nimzowitsch-Larsen: Polish Variation'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 b6
			variation => N__('Nimzowitsch-Larsen: Symmetrical'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 c5
			variation => N__('Nimzowitsch-Larsen: English Variation'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 d5
			variation => N__('Nimzowitsch-Larsen: 1... d5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/1P6/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 d5 2. Bb2
			variation => N__('Nimzowitsch-Larsen: 1... d5 2. Bb2'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5
			variation => N__('Nimzowitsch-Larsen: 1... e5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/8/1P6/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5 2. Bb2
			variation => N__('Nimzowitsch-Larsen: 1... e5 2. Bb2'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/8/1P6/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5 2. Bb2 Nc6
			variation => N__('Nimzowitsch-Larsen: 1... e5 2. Bb2 Nc6'),
			moves => {
				'e2e3' => 1,
				'f2f4' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/8/1P2P3/PBPP1PPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5 2. Bb2 Nc6 3. e3
			variation => N__('Nimzowitsch-Larsen: 1... e5 2. Bb2 Nc6 3. e3'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/5P2/1P6/PBPPP1PP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5 2. Bb2 Nc6 3. f4
			variation => N__('Nimzowitsch-Larsen: Paschmann Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/8/1P6/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 e5 2. Bb2 d6
			variation => N__('Nimzowitsch-Larsen: 1... e5 2. Bb2 d6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 f5
			variation => N__('Nimzowitsch-Larsen: Dutch Variation'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/8/1P6/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 f5 2. Bb2
			variation => N__('Nimzowitsch-Larsen: Dutch Variation'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/8/1P6/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 f5 2. Bb2 e6
			variation => N__('Nimzowitsch-Larsen: Dutch Variation'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/4P3/1P6/PBPP1PPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 f5 2. Bb2 e6 3. e4
			variation => N__('Nimzowitsch-Larsen: Ringelbach Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/1P6/P1PPPPPP/RNBQKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 Nf6
			variation => N__('Nimzowitsch-Larsen: Indian Variation'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/1P6/PBPPPPPP/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 Nf6 2. Bb2
			variation => N__('Nimzowitsch-Larsen: Indian Variation'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/8/1P6/PBPPPPPP/RN1QKBNR w KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 Nf6 2. Bb2 g6
			variation => N__('Nimzowitsch-Larsen: Indian Variation'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/6P1/1P6/PBPPPP1P/RN1QKBNR b KQkq' => {
			eco => 'A01',
			# TRANSLATORS: A01: 1. b3 Nf6 2. Bb2 g6 3. g4
			variation => N__('Nimzowitsch-Larsen: Spike Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/5P2/8/PPPPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4
			variation => N__('Bird'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'f7f5' => 1,
				'g7g5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 c5
			variation => N__('Bird: 1..c5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/5P2/5N2/PPPPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 c5 2. Nf3
			variation => N__('Bird: 1..c5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 c5 2. Nf3 Nc6
			variation => N__('Bird: 1..c5 2. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 d6
			variation => N__('Bird: 1..d6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5
			variation => N__('Bird: From Gambit'),
			moves => {
				'f4e5' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4P3/8/8/PPPPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5
			variation => N__('Bird: From Gambit Accepted'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4P3/8/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 Nc6
			variation => N__('Bird: From Gambit, Schlecter'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4P3/8/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6
			variation => N__('Bird: From Gambit, 2... d6'),
			moves => {
				'e5d6' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3P4/8/8/8/PPPPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6
			variation => N__('Bird: From Gambit, 3.exd6'),
			moves => {
				'f8d6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqk1nr/ppp2ppp/3b4/8/8/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6
			variation => N__('Bird: From Gambit, 3...Bxd6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp2ppp/3b4/8/8/5N2/PPPPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3
			variation => N__('Bird: From Gambit, 3...Bxd6'),
			moves => {
				'g7g5' => 1,
				'g8h6' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/3b4/6p1/8/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5
			variation => N__('Bird: From Gambit, Lasker Variation'),
			moves => {
				'd2d4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/3b4/6p1/3P4/5N2/PPP1P1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4
			variation => N__('Bird: From Gambit, Lasker, 5. d4'),
			moves => {
				'g5g4' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/3b4/8/3P2p1/5N2/PPP1P1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4 g4
			variation => N__('Bird: From Gambit, Lasker, 5. d4'),
			moves => {
				'f3e5' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/3b4/4N3/3P2p1/8/PPP1P1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4 g4 6. Ne5
			variation => N__('Bird: From Gambit, Lasker, 5. d4'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/8/4b3/3P2p1/8/PPP1P1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4 g4 6. Ne5 Bxe5
			variation => N__('Bird: From Gambit, Lasker, 5. d4'),
			moves => {
				'd4e5' => 1,
			},
		},
		'rnbqk1nr/ppp2p1p/8/4P3/6p1/8/PPP1P1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4 g4 6. Ne5 Bxe5 7. dxe5
			variation => N__('Bird: From Gambit, Lasker, 5. d4'),
			moves => {
				'd8d1' => 1,
			},
		},
		'rnb1k1nr/ppp2p1p/8/4P3/6p1/8/PPP1P1PP/RNBqKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. d4 g4 6. Ne5 Bxe5 7. dxe5 Qxd1+
			variation => N__('Bird: From Gambit, Lasker, Queenswap Line'),
			moves => {
			},
		},
		'rnbqk1nr/ppp2p1p/3b4/6p1/8/5NP1/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 g5 5. g3
			variation => N__('Bird: From Gambit, Lasker, 5. g3'),
			moves => {
			},
		},
		'rnbqk2r/ppp2ppp/3b3n/8/8/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 Nh6
			variation => N__('Bird: From Gambit, 3...Bxd6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqk2r/ppp2ppp/3b3n/8/3P4/5N2/PPP1P1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Bxd6 4. Nf3 Nh6 5. d4
			variation => N__('Bird: From Gambit, Lipke'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3P1n2/8/8/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 e5 2. fxe5 d6 3. exd6 Nf6
			variation => N__('Bird: From Gambit, Langheld Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5
			variation => N__('Bird: Symmetrical'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5 2. e4
			variation => N__('Bird: Swiss Gambit'),
			moves => {
				'f5e4' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/4pP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5 2. e4 fxe4
			variation => N__('Bird: Swiss Gambit'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/4pP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5 2. e4 fxe4 3. Nc3
			variation => N__('Bird: Swiss Gambit'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/4pP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5 2. e4 fxe4 3. Nc3 Nf6
			variation => N__('Bird: Swiss Gambit'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/4pPP1/2N5/PPPP3P/R1BQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 f5 2. e4 fxe4 3. Nc3 Nf6 4. g4
			variation => N__('Bird: Swiss Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g5
			variation => N__('Bird: Hobbs Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g6
			variation => N__('Bird: 1..g6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/5P2/5N2/PPPPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g6 2. Nf3
			variation => N__('Bird: 1..g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g6 2. Nf3 Bg7
			variation => N__('Bird: 1..g6'),
			moves => {
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/5P2/4PN2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g6 2. Nf3 Bg7 3. e3
			variation => N__('Bird: 1..g6'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/5P2/5NP1/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 g6 2. Nf3 Bg7 3. g3
			variation => N__('Bird: 1..g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6
			variation => N__('Bird: 1..Nf6'),
			moves => {
				'b2b3' => 1,
				'e2e3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/5P2/1P6/P1PPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. b3
			variation => N__('Bird: 1..Nf6 2. b3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/5P2/4P3/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. e3
			variation => N__('Bird: 1..Nf6 2. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/5P2/5N2/PPPPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3
			variation => N__('Bird: 1..Nf6 2. Nf3'),
			moves => {
				'c7c5' => 1,
				'd7d6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 c5
			variation => N__('Bird: 1..Nf6 2. Nf3 c5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 d6
			variation => N__('Bird: 1..Nf6 2. Nf3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6
			variation => N__('Bird: 1..Nf6 2. Nf3 g6'),
			moves => {
				'b2b4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/1P3P2/5N2/P1PPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6 3. b4
			variation => N__('Bird: Batavo Polish Attack'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/5P2/5NP1/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6 3. g3
			variation => N__('Bird: 1..Nf6 2. Nf3 g6 3. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/5P2/5NP1/PPPPP2P/RNBQKB1R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6 3. g3 Bg7
			variation => N__('Bird: 1..Nf6 2. Nf3 g6 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/5P2/5NP1/PPPPP1BP/RNBQK2R b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2
			variation => N__('Bird: 1..Nf6 2. Nf3 g6 3. g3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/5P2/5NP1/PPPPP1BP/RNBQK2R w KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 d6
			variation => N__('Bird: 1..Nf6 2. Nf3 g6 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/5P2/6P1/PPPPP2P/RNBQKBNR b KQkq' => {
			eco => 'A02',
			# TRANSLATORS: A02: 1. f4 Nf6 2. g3
			variation => N__('Bird: 1..Nf6 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5P2/8/PPPPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5
			variation => N__('Bird: 1... d5'),
			moves => {
				'b2b3' => 1,
				'c2c4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5P2/1P6/P1PPP1PP/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. b3
			variation => N__('Bird: 1... d5 2. b3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/1P6/P1PPP1PP/RNBQKBNR w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. b3 Nf6
			variation => N__('Bird: 1... d5 2. b3'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/1P6/PBPPP1PP/RN1QKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. b3 Nf6 3. Bb2
			variation => N__('Bird: 1... d5 2. b3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/2P2P2/8/PP1PP1PP/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. c4
			variation => N__('Bird: Sturm Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5P2/4P3/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. e3
			variation => N__('Bird: Lasker Variation'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/4P3/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. e3 Nf6
			variation => N__('Bird: Lasker Variation'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. e4
			variation => N__('Bird: Williams Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5P2/5N2/PPPPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3
			variation => N__('Bird: 1... d5 2. Nf3'),
			moves => {
				'c7c5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 c5
			variation => N__('Bird: 1... d5 2. Nf3 c5'),
			moves => {
				'e2e3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/5P2/4PN2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 c5 3. e3
			variation => N__('Bird: 1... d5 2. Nf3 c5 3. e3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/4PP2/5N2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 c5 3. e4
			variation => N__('Bird: Batavo Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/3p4/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 g6
			variation => N__('Bird: 1... d5 2. Nf3 g6'),
			moves => {
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/3p4/5P2/4PN2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 g6 3. e3
			variation => N__('Bird: 1... d5 2. Nf3 g6 3. e3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/3p4/5P2/5NP1/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 g6 3. g3
			variation => N__('Bird: 1... d5 2. Nf3 g6 3. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/3p4/5P2/5NP1/PPPPP2P/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 g6 3. g3 Bg7
			variation => N__('Bird: 1... d5 2. Nf3 g6 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/3p4/5P2/5NP1/PPPPP1BP/RNBQK2R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 g6 3. g3 Bg7 4. Bg2
			variation => N__('Bird: 1... d5 2. Nf3 g6 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/5N2/PPPPP1PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6
			variation => N__('Bird: 1... d5 2. Nf3 Nf6'),
			moves => {
				'b2b3' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/1P3N2/P1PPP1PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. b3
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. b3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/4PN2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3
			variation => N__('Bird: Lasker Variation'),
			moves => {
				'c7c5' => 1,
				'c8g4' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/5P2/4PN2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3 c5
			variation => N__('Bird: Lasker, 3... c5'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/5P2/1P2PN2/P1PP2PP/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3 c5 4. b3
			variation => N__('Bird: Lasker, 3... c5 4. b3'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3p4/5Pb1/4PN2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3 Bg4
			variation => N__('Bird: Lasker, 3... Bg4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/5P2/4PN2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3 e6
			variation => N__('Bird: Lasker, 3... e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/5P2/4PN2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. e3 g6
			variation => N__('Bird: Lasker, 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/5NP1/PPPPP2P/RNBQKB1R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/5P2/5NP1/PPPPP2P/RNBQKB1R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/5P2/5NP1/PPPPP1BP/RNBQK2R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/5np1/3p4/5P2/5NP1/PPPPP1BP/RNBQK2R w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6'),
			moves => {
				'd2d3' => 1,
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/5np1/3p4/5P2/3P1NP1/PPP1P1BP/RNBQK2R b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. d3
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 5. d3'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/5np1/3p4/5P2/5NP1/PPPPP1BP/RNBQ1RK1 b kq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. O-O
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 5.O-O'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/5np1/3p4/5P2/5NP1/PPPPP1BP/RNBQ1RK1 w -' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. O-O O-O
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 5.O-O O-O'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/5np1/3p4/5P2/3P1NP1/PPP1P1BP/RNBQ1RK1 b -' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. O-O O-O 6. d3
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 5.O-O O-O 6. d3'),
			moves => {
				'c7c5' => 1,
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/5np1/2pp4/5P2/3P1NP1/PPP1P1BP/RNBQ1RK1 w -' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. O-O O-O 6. d3 c5
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 6. d3 c5'),
			moves => {
			},
		},
		'rnbq1rk1/pp2ppbp/2p2np1/3p4/5P2/3P1NP1/PPP1P1BP/RNBQ1RK1 w -' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. Nf3 Nf6 3. g3 g6 4. Bg2 Bg7 5. O-O O-O 6. d3 c6
			variation => N__('Bird: 1... d5 2. Nf3 Nf6 3. g3 g6: 6. d3 c6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5P2/6P1/PPPPP2P/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. g3
			variation => N__('Bird: 1... d5 2. g3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/6P1/PPPPP2P/RNBQKBNR w KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. g3 Nf6
			variation => N__('Bird: 1... d5 2. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/5P2/6P1/PPPPP1BP/RNBQK1NR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. g3 Nf6 3. Bg2
			variation => N__('Bird: 1... d5 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/5PP1/8/PPPPP2P/RNBQKBNR b KQkq' => {
			eco => 'A03',
			# TRANSLATORS: A03: 1. f4 d5 2. g4
			variation => N__('Bird: Dudweiler Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/8/5N2/PPPPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3
			variation => N__('Reti'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c5' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'f7f5' => 1,
				'g7g5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/8/1p6/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 b5
			variation => N__('Reti: 1... b5'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 b6
			variation => N__('Reti: 1... b6'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 Nc6
			variation => N__('Reti: 1... Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5
			variation => N__('Reti: 1... c5'),
			moves => {
				'b2b3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/1P3N2/P1PPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. b3
			variation => N__('Reti: 1... c5, Nimzowitsch-Larsen'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3
			variation => N__('Reti: 1... c5'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 b6
			variation => N__('Reti: 1... c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 b6 3. Bg2
			variation => N__('Reti: 1... c5'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pb1ppppp/1p6/2p5/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 b6 3. Bg2 Bb7
			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 Nc6
			variation => N__('Reti: 1... c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 Nc6 3. Bg2
			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6
			variation => N__('Reti: 1... c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2
			variation => N__('Reti: 1... c5'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7
			variation => N__('Reti: 1... c5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O
			variation => N__('Reti: 1... c5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6
			variation => N__('Reti: 1... c5'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3
			variation => N__('Reti: 1... c5'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 d6
			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 d6 6. e4
			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 e6
			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 e6 6. e4
			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 Nf6
			variation => N__('Reti: 1... c5'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/4P3/3P1NP1/PPP2PBP/RNBQ1RK1 b kq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 c5 2. g3 g6 3. Bg2 Bg7 4. O-O Nc6 5. d3 Nf6 6. e4
			variation => N__('Reti: 1... c5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 d6
			variation => N__('Reti: 1... d6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 e6
			variation => N__('Reti: 1... e6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 e6 2. g3
			variation => N__('Reti: 1... e6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5
			variation => N__('Reti: 1... f5'),
			moves => {
				'd2d3' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/8/3P1N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. d3
			variation => N__('Reti: 1... f5 2. d3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/8/3P1N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. d3 Nf6
			variation => N__('Reti: 1... f5 2. d3 Nf6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/4P3/3P1N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. d3 Nf6 3. e4
			variation => N__('Reti: Lisitsin Deferred'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4
			variation => N__('Reti: Lisitsin'),
			moves => {
				'f5e4' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/4p3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4
			variation => N__('Reti: Lisitsin'),
			moves => {
				'f3g5' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/6N1/4p3/8/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5
			variation => N__('Reti: Lisitsin'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/8/3p2N1/4p3/8/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 d5
			variation => N__('Reti: Lisitsin: 3. Ng5 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/8/4p1N1/4p3/8/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 e5
			variation => N__('Reti: Lisitsin: 3. Ng5 e5'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/6N1/4p3/8/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 Nf6
			variation => N__('Reti: Lisitsin: 3. Ng5 Nf6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/6N1/4p3/3P4/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 Nf6 4. d3
			variation => N__('Reti: Lisitsin: 3. Ng5 Nf6'),
			moves => {
				'e4e3' => 1,
				'e7e5' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/6N1/8/3Pp3/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 Nf6 4. d3 e3
			variation => N__('Reti: Lisitsin: 3. Ng5 Nf6'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/5n2/4p1N1/4p3/3P4/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. e4 fxe4 3. Ng5 Nf6 4. d3 e5
			variation => N__('Reti: Lisitsin: 3. Ng5 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 f5 2. g3
			variation => N__('Reti: 1... f5 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 g5
			variation => N__('Reti: Herrström Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 g6
			variation => N__('Reti: 1... g6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 g6 2. g3
			variation => N__('Reti: 1... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 g6 2. g3 Bg7
			variation => N__('Reti: 1... g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A04',
			# TRANSLATORS: A04: 1. Nf3 g6 2. g3 Bg7 3. Bg2
			variation => N__('Reti: 1... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6
			variation => N__('Reti: 1... Nf6'),
			moves => {
				'b2b3' => 1,
				'b2b4' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/1P3N2/P1PPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. b3
			variation => N__('Reti: 1... Nf6 2. b3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/1P6/5N2/P1PPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. b4
			variation => N__('Reti: Santasiere\'s folly'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/4PN2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. e3
			variation => N__('Reti: 1... Nf6 2. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3
			variation => N__('Reti: KIA'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'c7c5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/5n2/1p6/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 b5
			variation => N__('Reti: KIA, Spassky'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/5n2/1p6/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 b5 3. Bg2
			variation => N__('Reti: KIA, Spassky'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 b6
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 b6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 b6 3. Bg2 Bb7
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 c5
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 c5 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 c5 3. Bg2 Nc6
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6
			variation => N__('Reti: KIA'),
			moves => {
				'b2b4' => 1,
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/1P6/5NP1/P1PPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. b4
			variation => N__('Reti: KIA, Reti-Smyslov Variation'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/1P6/5NP1/P1PPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. b4 Bg7
			variation => N__('Reti: KIA, Reti-Smyslov Variation'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/1P6/5NP1/PBPPPP1P/RN1QKB1R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. b4 Bg7 4. Bb2
			variation => N__('Reti: KIA, Reti-Smyslov Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7
			variation => N__('Reti: KIA'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7 4. O-O
			variation => N__('Reti: KIA'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/8/5NP1/PPPPPPBP/RNBQ1RK1 w -' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7 4. O-O O-O
			variation => N__('Reti: KIA'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b -' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7 4. O-O O-O 5. d3
			variation => N__('Reti: KIA'),
			moves => {
				'c7c5' => 1,
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/pp1pppbp/5np1/2p5/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w -' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7 4. O-O O-O 5. d3 c5
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w -' => {
			eco => 'A05',
			# TRANSLATORS: A05: 1. Nf3 Nf6 2. g3 g6 3. Bg2 Bg7 4. O-O O-O 5. d3 d6
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/5N2/PPPPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5
			variation => N__('Reti: 1... d5'),
			moves => {
				'b2b3' => 1,
				'b2b4' => 1,
				'c2c4' => 1,
				'd2d3' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
				'h1g1' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/1P3N2/P1PPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3
			variation => N__('Reti: Nimzowitsch-Larsen'),
			moves => {
				'c7c5' => 1,
				'c8g4' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/8/1P3N2/P1PPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 c5
			variation => N__('Reti: Nimzowitsch-Larsen, 2... c5'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/6b1/1P3N2/P1PPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Bg4
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Bg4'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/6b1/1P3N2/PBPPPPPP/RN1QKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Bg4 3. Bb2
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Bg4'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkbnr/pppnpppp/8/3p4/6b1/1P3N2/PBPPPPPP/RN1QKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Bg4 3. Bb2 Nd7
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Bg4'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r2qkbnr/pppnpppp/8/3p4/6b1/1P2PN2/PBPP1PPP/RN1QKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Bg4 3. Bb2 Nd7 4. e3
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Bg4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/1P3N2/P1PPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Nf6
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Nf6'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/1P3N2/PBPPPPPP/RN1QKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Nf6 3. Bb2
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Nf6'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/8/1P3N2/PBPPPPPP/RN1QKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Nf6 3. Bb2 e6
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Nf6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/8/1P2PN2/PBPP1PPP/RN1QKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b3 Nf6 3. Bb2 e6 4. e3
			variation => N__('Reti: Nimzowitsch-Larsen, 2... Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/1P6/5N2/P1PPPPPP/RNBQKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b4
			variation => N__('Reti: Santasiere\'s folly'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/1P6/5N2/P1PPPPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. b4 Nf6
			variation => N__('Reti: Santasiere\'s folly'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/3P1N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. d3
			variation => N__('Reti: Old Indian Attack'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/3P1N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. d3 Nf6
			variation => N__('Reti: Old Indian Attack'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/4PN2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. e3
			variation => N__('Reti: 1... d5 2. e3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. e4
			variation => N__('Reti: Tennison/Zukertort Gambit'),
			moves => {
				'd5e4' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/4p3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. e4 dxe4
			variation => N__('Reti: Tennison Gambit Accepted'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/5N2/PPPPPPPP/RNBQKBR1 b Qkq' => {
			eco => 'A06',
			# TRANSLATORS: A06: 1. Nf3 d5 2. Rg1
			variation => N__('Reti: Ampel Variation'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/8/5NP1/PPPPPP1P/RNBQKB1R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3
			variation => N__('Reti: KIA'),
			moves => {
				'b8c6' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'c8g4' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3p4/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3p4/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2n5/3pp3/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2 e5
			variation => N__('Reti: KIA'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2n5/3pp3/8/3P1NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2 e5 4. d3
			variation => N__('Reti: KIA'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n2n2/3pp3/8/3P1NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2 e5 4. d3 Nf6
			variation => N__('Reti: KIA'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n2n2/3pp3/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2 e5 4. d3 Nf6 5. O-O
			variation => N__('Reti: KIA'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/ppp1bppp/2n2n2/3pp3/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nc6 3. Bg2 e5 4. d3 Nf6 5. O-O Be7
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/6b1/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6 3. Bg2 Bg4
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/6b1/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6 3. Bg2 Bg4 4. O-O
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkbnr/pp1npppp/2p5/3p4/6b1/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6 3. Bg2 Bg4 4. O-O Nd7
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r2qkbnr/pp1npppp/2p5/3p4/6b1/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 c6 3. Bg2 Bg4 4. O-O Nd7 5. d3
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/6b1/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Bg4
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/8/3p4/6b1/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Bg4 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkbnr/pppnpppp/8/3p4/6b1/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Bg4 3. Bg2 Nd7
			variation => N__('Reti: KIA, Keres Variation'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/3p4/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/3p4/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/3p4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2 Bg7
			variation => N__('Reti: KIA'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/3p4/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2 Bg7 4. O-O
			variation => N__('Reti: KIA'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqk1nr/ppp2pbp/6p1/3pp3/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2 Bg7 4. O-O e5
			variation => N__('Reti: KIA'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbqk1nr/ppp2pbp/6p1/3pp3/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2 Bg7 4. O-O e5 5. d3
			variation => N__('Reti: KIA'),
			moves => {
				'g8e7' => 1,
			},
		},
		'rnbqk2r/ppp1npbp/6p1/3pp3/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 g6 3. Bg2 Bg7 4. O-O e5 5. d3 Ne7
			variation => N__('Reti: KIA, Pachman'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6
			variation => N__('Reti: KIA'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2
			variation => N__('Reti: KIA'),
			moves => {
				'c7c6' => 1,
				'c8f5' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6
			variation => N__('Reti: KIA'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O
			variation => N__('Reti: KIA'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bf5
			variation => N__('Reti: KIA'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bf5 5. d3
			variation => N__('Reti: KIA'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bf5 5. d3 e6
			variation => N__('Reti: KIA'),
			moves => {
				'b1d2' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bf5 5. d3 e6 6. Nbd2
			variation => N__('Reti: KIA, Petrosian Variation'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/6b1/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/6b1/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4 5. d3
			variation => N__('Reti: KIA, Yugoslav'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkb1r/pp1npppp/2p2n2/3p4/6b1/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4 5. d3 Nbd7
			variation => N__('Reti: KIA, Yugoslav, Main Line'),
			moves => {
				'b1d2' => 1,
			},
		},
		'r2qkb1r/pp1npppp/2p2n2/3p4/6b1/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4 5. d3 Nbd7 6. Nbd2
			variation => N__('Reti: KIA, Yugoslav, Main Line, 6.Nbd2'),
			moves => {
				'e7e5' => 1,
				'e7e6' => 1,
			},
		},
		'r2qkb1r/pp1n1ppp/2p2n2/3pp3/6b1/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4 5. d3 Nbd7 6. Nbd2 e5
			variation => N__('Reti: KIA, Yugoslav, Main Line, 6.Nbd2 e5'),
			moves => {
			},
		},
		'r2qkb1r/pp1n1ppp/2p1pn2/3p4/6b1/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 c6 4. O-O Bg4 5. d3 Nbd7 6. Nbd2 e6
			variation => N__('Reti: KIA, Yugoslav, Main Line, 6.Nbd2 e6'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3p1b2/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 Bf5
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 e6
			variation => N__('Reti: KIA'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A07',
			# TRANSLATORS: A07: 1. Nf3 d5 2. g3 Nf6 3. Bg2 g6
			variation => N__('Reti: KIA, Neo-Grünfeld'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/8/5NP1/PPPPPP1P/RNBQKB1R w KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/8/5NP1/PPPPPPBP/RNBQK2R b KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'b8c6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2pp4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6
			variation => N__('Reti: KIA, 2... c5  3. Bg2 Nc6'),
			moves => {
				'd2d4' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2pp4/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. d4
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4. d4'),
			moves => {
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n1p3/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n1p3/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'f8d6' => 1,
				'g8e7' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk1nr/pp3ppp/2nbp3/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b1d2' => 1,
			},
		},
		'r1bqk1nr/pp3ppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp2nppp/2nbp3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk2r/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7 7. e4
			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w -' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7 7. e4 O-O
			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'f1e1' => 1,
			},
		},
		'r1bq1rk1/pp2nppp/2nbp3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b -' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Bd6 6. Nbd2 Nge7 7. e4 O-O 8. Re1
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp2nppp/2n1p3/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b1d2' => 1,
			},
		},
		'r1bqkb1r/pp2nppp/2n1p3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b7b6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/p3nppp/1pn1p3/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 b6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqkb1r/p3nppp/1pn1p3/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 b6 7. e4
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp2np1p/2n1p1p1/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 g6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqkb1r/pp2np1p/2n1p1p1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 g6 7. e4
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2n1p1p1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nge7 6. Nbd2 g6 7. e4 Bg7
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b1d2' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'b7b6' => 1,
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/p4ppp/1pn1pn2/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 b6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqkb1r/p4ppp/1pn1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 b6 7. e4
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7 7. e4
			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w -' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7 7. e4 O-O
			variation => N__('Reti: KIA, French Variation'),
			moves => {
				'f1e1' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQR1K1 b -' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 Be7 7. e4 O-O 8. Re1
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp3p1p/2n1pnp1/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 g6
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqkb1r/pp3p1p/2n1pnp1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 g6 7. e4
			variation => N__('Reti: KIA, 2... c5, 3. Bg2 Nc6 4.O-O'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk2r/pp3pbp/2n1pnp1/2pp4/4P3/3P1NP1/PPPN1PBP/R1BQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nc6 4. O-O e6 5. d3 Nf6 6. Nbd2 g6 7. e4 Bg7
			variation => N__('Reti: KIA, French Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/6p1/2pp4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/6p1/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'b8c6' => 1,
				'f8g7' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2n3p1/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Nc6
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2n3p1/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Nc6 5. d3
			variation => N__('Reti: KIA, 2... c5'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp2pp1p/2n2np1/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Nc6 5. d3 Nf6
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/6p1/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Bg7
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/6p1/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Bg7 5. d3
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/5np1/2pp4/8/3P1NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Bg7 5. d3 Nf6
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
				'b1d2' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/5np1/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Bg7 5. d3 Nf6 6. Nbd2
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/5np1/2pp4/8/3P1NP1/PPPNPPBP/R1BQ1RK1 w -' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 g6 4. O-O Bg7 5. d3 Nf6 6. Nbd2 O-O
			variation => N__('Reti: KIA, 2... c5 + 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/8/5NP1/PPPPPPBP/RNBQK2R w KQkq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nf6
			variation => N__('Reti: KIA, 2... c5 3. Bg2 Nf6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 b kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nf6 4. O-O
			variation => N__('Reti: KIA, 2... c5 3. Bg2 Nf6 4.O-O'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/2pp4/8/5NP1/PPPPPPBP/RNBQ1RK1 w kq' => {
			eco => 'A08',
			# TRANSLATORS: A08: 1. Nf3 d5 2. g3 c5 3. Bg2 Nf6 4. O-O Nc6
			variation => N__('Reti: KIA, 2... c5 3. Bg2 Nf6 4.O-O Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A09a',
			# TRANSLATORS: A09a: 1. Nf3 d5 2. c4
			variation => N__('Reti: 2. c4'),
			moves => {
				'd5c4' => 1,
				'd5d4' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2Pp4/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A09b',
			# TRANSLATORS: A09b: 1. Nf3 d5 2. c4 d4
			variation => N__('Reti: Advance Variation'),
			moves => {
				'b2b4' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/1PPp4/5N2/P2PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A09c',
			# TRANSLATORS: A09c: 1. Nf3 d5 2. c4 d4 3. b4
			variation => N__('Reti: Advance, Anglo-Polish Attack'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/8/1PPp4/5N2/P2PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A09c',
			# TRANSLATORS: A09c: 1. Nf3 d5 2. c4 d4 3. b4 g6
			variation => N__('Reti: Advance, Anglo-Polish, 3... g6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2Pp4/4PN2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'A09d',
			# TRANSLATORS: A09d: 1. Nf3 d5 2. c4 d4 3. e3
			variation => N__('Reti: Advance, 3. e3'),
			moves => {
				'b8c6' => 1,
				'c7c5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2p5/2Pp4/4PN2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A09e',
			# TRANSLATORS: A09e: 1. Nf3 d5 2. c4 d4 3. e3 c5
			variation => N__('Reti: Advance, 3. e3 c5'),
			moves => {
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/8/2Pp4/4PN2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A09f',
			# TRANSLATORS: A09f: 1. Nf3 d5 2. c4 d4 3. e3 Nc6
			variation => N__('Reti: Advance, 3. e3 Nc6'),
			moves => {
				'e3d4' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/8/2PP4/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'A09f',
			# TRANSLATORS: A09f: 1. Nf3 d5 2. c4 d4 3. e3 Nc6 4. exd4
			variation => N__('Reti: Advance, 3. e3 Nc6'),
			moves => {
				'c6d4' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/8/8/2Pn4/5N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A09f',
			# TRANSLATORS: A09f: 1. Nf3 d5 2. c4 d4 3. e3 Nc6 4. exd4 Nxd4
			variation => N__('Reti: Advance, 3. e3 Nc6 4.exd4 Nxd4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2Pp4/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A09g',
			# TRANSLATORS: A09g: 1. Nf3 d5 2. c4 d4 3. g3
			variation => N__('Reti: Advance, 3. g3'),
			moves => {
				'b8c6' => 1,
				'c7c5' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/8/2Pp4/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A09h',
			# TRANSLATORS: A09h: 1. Nf3 d5 2. c4 d4 3. g3 Nc6
			variation => N__('Reti: Advance, 3. g3 Nc6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/8/2Pp4/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A09h',
			# TRANSLATORS: A09h: 1. Nf3 d5 2. c4 d4 3. g3 Nc6 4. Bg2
			variation => N__('Reti: Advance, 3. g3 Nc6'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2n5/4p3/2Pp4/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A09h',
			# TRANSLATORS: A09h: 1. Nf3 d5 2. c4 d4 3. g3 Nc6 4. Bg2 e5
			variation => N__('Reti: Advance, 3. g3 Nc6 4. Bg2 e5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/8/2Pp4/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A09i',
			# TRANSLATORS: A09i: 1. Nf3 d5 2. c4 d4 3. g3 g6
			variation => N__('Reti: Advance, 3. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/6p1/8/2Pp4/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A09i',
			# TRANSLATORS: A09i: 1. Nf3 d5 2. c4 d4 3. g3 g6 4. Bg2
			variation => N__('Reti: Advance, 3. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/8/2Pp4/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A09i',
			# TRANSLATORS: A09i: 1. Nf3 d5 2. c4 d4 3. g3 g6 4. Bg2 Bg7
			variation => N__('Reti: Advance, 3. g3 g6 4. Bg2 Bg7'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2p5/2Pp4/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A09j',
			# TRANSLATORS: A09j: 1. Nf3 d5 2. c4 d4 3. g3 c5
			variation => N__('Reti: Advance, 3. g3 c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2p5/2Pp4/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A09j',
			# TRANSLATORS: A09j: 1. Nf3 d5 2. c4 d4 3. g3 c5 4. Bg2
			variation => N__('Reti: Advance, 3. g3 c5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2p5/2Pp4/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A09j',
			# TRANSLATORS: A09j: 1. Nf3 d5 2. c4 d4 3. g3 c5 4. Bg2 Nc6
			variation => N__('Reti: Advance, 3. g3 c5 4. Bg2 Nc6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2p5/2Pp4/3P1NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A09j',
			# TRANSLATORS: A09j: 1. Nf3 d5 2. c4 d4 3. g3 c5 4. Bg2 Nc6 5. d3
			variation => N__('Reti: Advance, 3. g3 c5 4. Bg2 Nc6'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n5/2p1p3/2Pp4/3P1NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A09j',
			# TRANSLATORS: A09j: 1. Nf3 d5 2. c4 d4 3. g3 c5 4. Bg2 Nc6 5. d3 e5
			variation => N__('Reti: Advance, 3. g3 c5 4. Bg2 Nc6 5. d3 e5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2p5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A09k',
			# TRANSLATORS: A09k: 1. Nf3 d5 2. c4 dxc4
			variation => N__('Reti: Accepted'),
			moves => {
				'b1a3' => 1,
				'd1a4' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2p5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A09l',
			# TRANSLATORS: A09l: 1. Nf3 d5 2. c4 dxc4 3. g3
			variation => N__('Reti: Accepted, 3. g3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/8/2p5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A09m',
			# TRANSLATORS: A09m: 1. Nf3 d5 2. c4 dxc4 3. g3 e6
			variation => N__('Reti: Accepted, 3. g3 e6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/Q1p5/5N2/PP1PPPPP/RNB1KB1R b KQkq' => {
			eco => 'A09n',
			# TRANSLATORS: A09n: 1. Nf3 d5 2. c4 dxc4 3. Qa4+
			variation => N__('Reti: Accepted, 3. Qa4+'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2p5/N4N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A09o',
			# TRANSLATORS: A09o: 1. Nf3 d5 2. c4 dxc4 3. Na3
			variation => N__('Reti: Accepted, 3. Na3'),
			moves => {
				'a7a6' => 1,
				'c7c5' => 1,
			},
		},
		'rnbqkbnr/1pp1pppp/p7/8/2p5/N4N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A09p',
			# TRANSLATORS: A09p: 1. Nf3 d5 2. c4 dxc4 3. Na3 a6
			variation => N__('Reti: Accepted, 3. Na3 a6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2p5/2p5/N4N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A09q',
			# TRANSLATORS: A09q: 1. Nf3 d5 2. c4 dxc4 3. Na3 c5
			variation => N__('Reti: Accepted, 3. Na3 c5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/2p5/4PN2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'A09r',
			# TRANSLATORS: A09r: 1. Nf3 d5 2. c4 dxc4 3. e3
			variation => N__('Reti: Accepted, 3. e3'),
			moves => {
				'c8e6' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/4b3/8/2p5/4PN2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A09s',
			# TRANSLATORS: A09s: 1. Nf3 d5 2. c4 dxc4 3. e3 Be6
			variation => N__('Reti: Accepted, Keres Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/8/2p5/4PN2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A09t',
			# TRANSLATORS: A09t: 1. Nf3 d5 2. c4 dxc4 3. e3 Nf6
			variation => N__('Reti: Accepted, 3. e3 Nf6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/8/2B5/4PN2/PP1P1PPP/RNBQK2R b KQkq' => {
			eco => 'A09t',
			# TRANSLATORS: A09t: 1. Nf3 d5 2. c4 dxc4 3. e3 Nf6 4. Bxc4
			variation => N__('Reti: Accepted, 3. e3 Nf6'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/8/2B5/4PN2/PP1P1PPP/RNBQK2R w KQkq' => {
			eco => 'A09u',
			# TRANSLATORS: A09u: 1. Nf3 d5 2. c4 dxc4 3. e3 Nf6 4. Bxc4 e6
			variation => N__('Reti: Accepted, 3. e3 Nf6 4.Bxc4 e6'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/2P5/8/PP1PPPPP/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4
			variation => N__('English'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'f7f5' => 1,
				'g7g5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/8/1p6/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b5
			variation => N__('English: Jänisch Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6
			variation => N__('English: 1... b6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3
			variation => N__('English: 1... b6 2. Nc3'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3 Bb7
			variation => N__('English: 1... b6 2. Nc3 Bb7'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3 Bb7 3. e4
			variation => N__('English: 1... b6 2. Nc3 Bb7 3. e4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3 Bb7 3. e4 e6
			variation => N__('English: 1... b6 2. Nc3 Bb7 3. e4 e6'),
			moves => {
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3 e6
			variation => N__('English: 1... b6 2. Nc3 e6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nc3 e6 3. e4
			variation => N__('English: 1... b6 2. Nc3 e6 3. e4'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nf3
			variation => N__('English: 1... b6 2. Nf3'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 b6 2. Nf3 Bb7
			variation => N__('English: 1... b6 2. Nf3 Bb7'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 Nc6
			variation => N__('English: 1... Nc6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 Nc6 2. Nc3
			variation => N__('English: 1... Nc6 2. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 d5
			variation => N__('English: Vector'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 d6
			variation => N__('English: 1... d6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 d6 2. Nc3
			variation => N__('English: 1... d6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 d6 2. Nf3
			variation => N__('English: 1... d6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nc3
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nc3 Nf6
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nc3 Nf6 3. g3
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nc3 Nf6 3. g3 g6
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 e6
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'b1c3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. Nc3
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'c7c6' => 1,
				'f8e7' => 1,
			},
		},
		'rnbqkb1r/pp1p2pp/2p1pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 c6
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp1p2pp/2p1pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 c6 5. O-O
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp4pp/2p1pn2/3p1p2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 c6 5. O-O d5
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 Be7
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 Be7 5. O-O
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w -' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 Be7 5. O-O O-O
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b -' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. Nf3 Nf6 3. g3 e6 4. Bg2 Be7 5. O-O O-O 6. Nc3
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. g3
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. g3 Nf6
			variation => N__('English: Anglo-Dutch'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. g3 Nf6 3. Bg2
			variation => N__('English: Anglo-Dutch'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2P3P1/8/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 f5 2. g4
			variation => N__('English: Wade Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g5
			variation => N__('English: 1... g5'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g5 2. d4
			variation => N__('English: 1... g5 2. d4'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/8/6p1/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g5 2. d4 Bg7
			variation => N__('English: Myers Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6
			variation => N__('English: 1... g6'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. Nc3
			variation => N__('English: 1... g6 2. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. Nc3 Bg7
			variation => N__('English: 1... g6 2. Nc3 Bg7'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. Nc3 Bg7 3. g3
			variation => N__('English: 1... g6 2. Nc3 Bg7 3. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2P1P3/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. e4
			variation => N__('English: 1... g6 2. e4'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pppp1p1p/6p1/4p3/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. e4 e5
			variation => N__('English: Adorjan Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. Nf3
			variation => N__('English: 1... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. Nf3 Bg7
			variation => N__('English: 1... g6'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A10',
			# TRANSLATORS: A10: 1. c4 g6 2. g3
			variation => N__('English: 1... g6 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nc3
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nc3 d5
			variation => N__('English: Caro-Kann Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'd7d5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'b2b3' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P5/4PN2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. e3
			variation => N__('English: Caro-Kann Defence, 3. e3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/4PN2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. e3 Nf6
			variation => N__('English: Caro-Kann Defence, 3. e3 Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/2N1PN2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. e3 Nf6 4. Nc3
			variation => N__('English: Caro-Kann Defence, 3. e3 Nf6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. g3
			variation => N__('English: Caro-Kann Defence, 3. g3'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/2P3b1/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. g3 Bg4
			variation => N__('English: Caro-Kann Defence, 3. g3 Bg4'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/2P3b1/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 d5 3. g3 Bg4 4. Bg2
			variation => N__('English: Caro-Kann Defence, 3. g3 Bg4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. Nf3 Nf6
			variation => N__('English: Caro-Kann Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bf5
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bf5 5. O-O
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bf5 5. O-O e6
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/2P5/3P1NP1/PP2PPBP/RNBQ1RK1 b kq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bf5 5. O-O e6 6. d3
			variation => N__('English: Caro-Kann Defence'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/2P3b1/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bg4
			variation => N__('English: Caro-Kann Defence'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/2P3b1/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A11',
			# TRANSLATORS: A11: 1. c4 c6 2. g3 Nf6 3. Bg2 d5 4. Nf3 Bg4 5. O-O
			variation => N__('English: Caro-Kann Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P5/1P3N2/P2PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3
			variation => N__('English: Caro-Kann Defence, 3. b3'),
			moves => {
				'c8g4' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/2P3b1/1P3N2/P2PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Bg4
			variation => N__('English: Bogoljubow Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/1P3N2/P2PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6
			variation => N__('English: Caro-Kann Defence, 3. b3'),
			moves => {
				'c1b2' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/1P3N2/PB1PPPPP/RN1QKB1R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2
			variation => N__('English: Caro-Kann Defence, 3. b3'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/2P5/1P3N2/PB1PPPPP/RN1QKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2 Bf5
			variation => N__('English: New York/London Defence'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/2P3b1/1P3N2/PB1PPPPP/RN1QKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2 Bg4
			variation => N__('English: Capablanca'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2np1/3p4/2P5/1P3N2/PB1PPPPP/RN1QKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2 g6
			variation => N__('English: Bled Variation'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/2p2np1/3p4/2P5/1P2PN2/PB1P1PPP/RN1QKB1R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2 g6 5. e3
			variation => N__('English: Bled Variation'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2p2np1/3p4/2P5/1P2PN2/PB1P1PPP/RN1QKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. Bb2 g6 5. e3 Bg7
			variation => N__('English: Bled Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P5/1P3NP1/P2PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3
			variation => N__('English: Caro-Kann Defence, 3. b3'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/2P5/1P3NP1/P2PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bf5
			variation => N__('English: London Defence'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p1b2/2P5/1P3NP1/P2PPPBP/RNBQK2R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bf5 5. Bg2
			variation => N__('English: London Defence'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/2P5/1P3NP1/P2PPPBP/RNBQK2R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bf5 5. Bg2 e6
			variation => N__('English: London Defence'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p1b2/2P5/1P3NP1/PB1PPPBP/RN1QK2R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bf5 5. Bg2 e6 6. Bb2
			variation => N__('English: London Defence'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/2P3b1/1P3NP1/P2PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bg4
			variation => N__('English: Torre Defence'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/2P3b1/1P3NP1/P2PPPBP/RNBQK2R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bg4 5. Bg2
			variation => N__('English: Torre Defence'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/2P3b1/1P3NP1/P2PPPBP/RNBQK2R w KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bg4 5. Bg2 e6
			variation => N__('English: Torre Defence'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/2P3b1/1P3NP1/PB1PPPBP/RN1QK2R b KQkq' => {
			eco => 'A12',
			# TRANSLATORS: A12: 1. c4 c6 2. Nf3 d5 3. b3 Nf6 4. g3 Bg4 5. Bg2 e6 6. Bb2
			variation => N__('English: Torre Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A13a',
			# TRANSLATORS: A13a: 1. c4 e6
			variation => N__('English: 1... e6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A13b',
			# TRANSLATORS: A13b: 1. c4 e6 2. g3
			variation => N__('English: 1... e6 2. g3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A13c',
			# TRANSLATORS: A13c: 1. c4 e6 2. g3 d5
			variation => N__('English: 1... e6 2. g3 d5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A13d',
			# TRANSLATORS: A13d: 1. c4 e6 2. g3 d5 3. Bg2
			variation => N__('English: 1... e6 2. g3 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A13e',
			# TRANSLATORS: A13e: 1. c4 e6 2. Nc3
			variation => N__('English: 1... e6 2. Nc3'),
			moves => {
				'd7d5' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/4p3/8/1bP5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A13f',
			# TRANSLATORS: A13f: 1. c4 e6 2. Nc3 Bb4
			variation => N__('English: 1... e6 2. Nc3 Bb4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A13g',
			# TRANSLATORS: A13g: 1. c4 e6 2. Nc3 d5
			variation => N__('English: 1... e6 2. Nc3 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A13h',
			# TRANSLATORS: A13h: 1. c4 e6 2. Nf3
			variation => N__('English: 1... e6 2. Nf3'),
			moves => {
				'd7d5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A13i',
			# TRANSLATORS: A13i: 1. c4 e6 2. Nf3 Nf6
			variation => N__('English: 1... e6 2. Nf3 Nf6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A13j',
			# TRANSLATORS: A13j: 1. c4 e6 2. Nf3 Nf6 3. g3
			variation => N__('English: 1... e6 2. Nf3 Nf6 3. g3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnbqkb1r/1ppp1ppp/p3pn2/8/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A13j',
			# TRANSLATORS: A13j: 1. c4 e6 2. Nf3 Nf6 3. g3 a6
			variation => N__('English: 1... e6 2. Nf3 Nf6 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/1ppp1ppp/p3pn2/8/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A13j',
			# TRANSLATORS: A13j: 1. c4 e6 2. Nf3 Nf6 3. g3 a6 4. Bg2
			variation => N__('English: 1... e6 2. Nf3 Nf6 3. g3'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkb1r/2pp1ppp/p3pn2/1p6/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A13k',
			# TRANSLATORS: A13k: 1. c4 e6 2. Nf3 Nf6 3. g3 a6 4. Bg2 b5
			variation => N__('English: Romanishin Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A13l',
			# TRANSLATORS: A13l: 1. c4 e6 2. Nf3 d5
			variation => N__('English: 1... e6 2. Nf3 d5'),
			moves => {
				'b2b3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/1P3N2/P2PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A13m',
			# TRANSLATORS: A13m: 1. c4 e6 2. Nf3 d5 3. b3
			variation => N__('English: Agincourt Variation'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P5/1P3N2/P2PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A13m',
			# TRANSLATORS: A13m: 1. c4 e6 2. Nf3 d5 3. b3 Nf6
			variation => N__('English: Agincourt Variation'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P5/1P3N2/PB1PPPPP/RN1QKB1R b KQkq' => {
			eco => 'A13m',
			# TRANSLATORS: A13m: 1. c4 e6 2. Nf3 d5 3. b3 Nf6 4. Bb2
			variation => N__('English: Agincourt Variation'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/2pp4/2P5/1P3N2/PB1PPPPP/RN1QKB1R w KQkq' => {
			eco => 'A13m',
			# TRANSLATORS: A13m: 1. c4 e6 2. Nf3 d5 3. b3 Nf6 4. Bb2 c5
			variation => N__('English: Agincourt Variation'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/2pp4/2P5/1P2PN2/PB1P1PPP/RN1QKB1R b KQkq' => {
			eco => 'A13n',
			# TRANSLATORS: A13n: 1. c4 e6 2. Nf3 d5 3. b3 Nf6 4. Bb2 c5 5. e3
			variation => N__('English: Wimpey System'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/2pp4/2P5/1P2PN2/PB1P1PPP/RN1QKB1R w KQkq' => {
			eco => 'A13n',
			# TRANSLATORS: A13n: 1. c4 e6 2. Nf3 d5 3. b3 Nf6 4. Bb2 c5 5. e3 Nc6
			variation => N__('English: Wimpey System'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/4p3/3p4/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A13o',
			# TRANSLATORS: A13o: 1. c4 e6 2. Nf3 d5 3. g3
			variation => N__('English: Agincourt Variation'),
			moves => {
				'c7c6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p1p3/3p4/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A13p',
			# TRANSLATORS: A13p: 1. c4 e6 2. Nf3 d5 3. g3 c6
			variation => N__('English: Kurajica Defence'),
			moves => {
				'd1c2' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p1p3/3p4/2P5/5NP1/PPQPPP1P/RNB1KB1R b KQkq' => {
			eco => 'A13q',
			# TRANSLATORS: A13q: 1. c4 e6 2. Nf3 d5 3. g3 c6 4. Qc2
			variation => N__('English: Kurajica Defence'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A13r',
			# TRANSLATORS: A13r: 1. c4 e6 2. Nf3 d5 3. g3 Nf6
			variation => N__('English: Neo-Catalan'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A13s',
			# TRANSLATORS: A13s: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2
			variation => N__('English: Neo-Catalan'),
			moves => {
				'c7c5' => 1,
				'c7c6' => 1,
				'd5c4' => 1,
				'f8e7' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p1pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A13s',
			# TRANSLATORS: A13s: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 c6
			variation => N__('English: Neo-Catalan, 4... c6'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p1pn2/3p4/2P5/1P3NP1/P2PPPBP/RNBQK2R b KQkq' => {
			eco => 'A13s',
			# TRANSLATORS: A13s: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 c6 5. b3
			variation => N__('English: Neo-Catalan, 4... c6 5. b3'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/2pp4/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A13t',
			# TRANSLATORS: A13t: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 c5
			variation => N__('English: Neo-Catalan, 4... c5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/2pp4/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A13t',
			# TRANSLATORS: A13t: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 c5 5. O-O
			variation => N__('English: Neo-Catalan, 4... c5 5.O-O'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/8/2p5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A13u',
			# TRANSLATORS: A13u: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4
			variation => N__('English: Neo-Catalan Accepted'),
			moves => {
				'd1a4' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/8/Q1p5/5NP1/PP1PPPBP/RNB1K2R b KQkq' => {
			eco => 'A13v',
			# TRANSLATORS: A13v: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/4pn2/8/Q1p5/5NP1/PP1PPPBP/RNB1K2R w KQkq' => {
			eco => 'A13v',
			# TRANSLATORS: A13v: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+ Nbd7
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+ Nbd7'),
			moves => {
				'a4c4' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/4pn2/8/Q1p5/5NP1/PP1PPPBP/RNB2RK1 b kq' => {
			eco => 'A13v',
			# TRANSLATORS: A13v: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+ Nbd7 6. O-O
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+ Nbd7 6.O-O'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/4pn2/8/2Q5/5NP1/PP1PPPBP/RNB1K2R b KQkq' => {
			eco => 'A13w',
			# TRANSLATORS: A13w: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+ Nbd7 6. Qxc4
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+ Nbd7 6.Qxc4'),
			moves => {
				'a7a6' => 1,
				'c7c5' => 1,
			},
		},
		'r1bqkb1r/1ppn1ppp/p3pn2/8/2Q5/5NP1/PP1PPPBP/RNB1K2R w KQkq' => {
			eco => 'A13w',
			# TRANSLATORS: A13w: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+ Nbd7 6. Qxc4 a6
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+ Nbd7 6.Qxc4 a6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1n1ppp/4pn2/2p5/2Q5/5NP1/PP1PPPBP/RNB1K2R w KQkq' => {
			eco => 'A13w',
			# TRANSLATORS: A13w: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 dxc4 5. Qa4+ Nbd7 6. Qxc4 c5
			variation => N__('English: Neo-Catalan Accepted, 5. Qa4+ Nbd7 6.Qxc4 c5'),
			moves => {
			},
		},
		'rnbqk2r/ppp1bppp/4pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7
			variation => N__('English: Neo-Catalan Declined'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppp1bppp/4pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O
			variation => N__('English: Neo-Catalan Declined'),
			moves => {
				'c7c5' => 1,
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/pp2bppp/4pn2/2pp4/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O c5
			variation => N__('English: Neo-Catalan Declined, 5... c5'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/2p1pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O c6
			variation => N__('English: Neo-Catalan Declined, 5... c6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1bppp/4pn2/3p4/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O
			variation => N__('English: Neo-Catalan Declined, 5...O-O'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rnbq1rk1/ppp1bppp/4pn2/3p4/2P5/1P3NP1/P2PPPBP/RNBQ1RK1 b -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3'),
			moves => {
				'b7b6' => 1,
				'c7c5' => 1,
			},
		},
		'rnbq1rk1/p1p1bppp/1p2pn2/3p4/2P5/1P3NP1/P2PPPBP/RNBQ1RK1 w -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 b6
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 b6'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbq1rk1/p1p1bppp/1p2pn2/3p4/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 b -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 b6 7. Bb2
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 b6'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1q1rk1/pbp1bppp/1p2pn2/3p4/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 w -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 b6 7. Bb2 Bb7
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 b6 7. Bb2 Bb7'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rn1q1rk1/pbp1bppp/1p2pn2/3p4/2P5/1P2PNP1/PB1P1PBP/RN1Q1RK1 b -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 b6 7. Bb2 Bb7 8. e3
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 b6 7. Bb2 Bb7 8. e3'),
			moves => {
			},
		},
		'rnbq1rk1/pp2bppp/4pn2/2pp4/2P5/1P3NP1/P2PPPBP/RNBQ1RK1 w -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbq1rk1/pp2bppp/4pn2/2pp4/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 b -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5 7. Bb2
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5 7. Bb2'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 w -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5 7. Bb2 Nc6
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5 7. Bb2 Nc6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2pp4/2P5/1P2PNP1/PB1P1PBP/RN1Q1RK1 b -' => {
			eco => 'A14',
			# TRANSLATORS: A14: 1. c4 e6 2. Nf3 d5 3. g3 Nf6 4. Bg2 Be7 5. O-O O-O 6. b3 c5 7. Bb2 Nc6 8. e3
			variation => N__('English: Neo-Catalan Declined, 5...O-O 6. b3 c5 7. Bb2 Nc6 8. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6
			variation => N__('English: Anglo-Indian'),
			moves => {
				'b1c3' => 1,
				'b2b4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/1PP5/8/P2PPPPP/RNBQKBNR b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. b4
			variation => N__('English: Anglo-Indian, Polish'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3
			variation => N__('English: Anglo-Indian, 2. Nf3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3 g6
			variation => N__('English: Anglo-Indian, 2. Nf3 g6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3 g6 3. g3
			variation => N__('English: Anglo-Indian, 2. Nf3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3 g6 3. g3 Bg7
			variation => N__('English: Anglo-Indian, 2. Nf3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2
			variation => N__('English: Anglo-Indian, 2. Nf3 g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/2P5/5NP1/PP1PPPBP/RNBQK2R w KQ' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O
			variation => N__('English: Anglo-Indian, 2. Nf3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3
			variation => N__('English: Anglo-Indian, 2. g3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3 e6
			variation => N__('English: Anglo-Indian, 2. g3 e6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3 e6 3. Bg2
			variation => N__('English: Anglo-Indian, 2. g3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3 g6
			variation => N__('English: Anglo-Indian, 2. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3 g6 3. Bg2
			variation => N__('English: Anglo-Indian, 2. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A15',
			# TRANSLATORS: A15: 1. c4 Nf6 2. g3 g6 3. Bg2 Bg7
			variation => N__('English: Anglo-Indian, 2. g3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3
			variation => N__('English: Anglo-Indian, 2. Nc3'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pppppppp/2n2n2/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 Nc6
			variation => N__('English: Anglo-Indian, 2. Nc3 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 c6
			variation => N__('English: Anglo-Indian, 2. Nc3 c6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 c6 3. e4
			variation => N__('English: Anglo-Indian, 2. Nc3 c6 3. e4'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3p4/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 c6 3. e4 d5
			variation => N__('English: Anglo-Indian, 2. Nc3 c6 3. e4 d5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/4p3/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 c6 3. e4 e5
			variation => N__('English: Anglo-Indian, 2. Nc3 c6 3. e4 e5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5
			variation => N__('English: Anglo-Grünfeld'),
			moves => {
				'c4d5' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/8/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5
			variation => N__('English: Anglo-Grünfeld, 3.cxd5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5
			variation => N__('English: Anglo-Grünfeld, 3.cxd5'),
			moves => {
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3
			variation => N__('English: Anglo-Grünfeld, 4. Nf3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/8/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6
			variation => N__('English: Anglo-Grünfeld, 4. Nf3 g6'),
			moves => {
				'd1a4' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/Q7/2N2N2/PP1PPPPP/R1B1KB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. Qa4+
			variation => N__('English: Anglo-Grünfeld, 4. Nf3 g6 5. Qa4+'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/4P3/2N2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. e4
			variation => N__('English: Anglo-Grünfeld, Andersson-Böök Variation'),
			moves => {
				'd5c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/8/4P3/2n2N2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. e4 Nxc3
			variation => N__('English: Anglo-Grünfeld, Andersson-Böök Variation'),
			moves => {
				'd2c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/8/4P3/2P2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. e4 Nxc3 6. dxc3
			variation => N__('English: Anglo-Grünfeld, Andersson-Böök Variation'),
			moves => {
				'd8d1' => 1,
			},
		},
		'rnb1kb1r/ppp1pp1p/6p1/8/4P3/2P2N2/PP3PPP/R1BqKB1R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. e4 Nxc3 6. dxc3 Qxd1+
			variation => N__('English: Anglo-Grünfeld, Andersson-Böök Variation'),
			moves => {
				'e1d1' => 1,
			},
		},
		'rnb1kb1r/ppp1pp1p/6p1/8/4P3/2P2N2/PP3PPP/R1BK1B1R b kq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. e4 Nxc3 6. dxc3 Qxd1+ 7. Kxd1
			variation => N__('English: Anglo-Grünfeld, Andersson-Böök, Queenswap'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/8/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. g3
			variation => N__('English: Anglo-Grünfeld, 4. Nf3 g6 5. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/6p1/3n4/8/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. g3 Bg7
			variation => N__('English: Anglo-Grünfeld, 4. Nf3 g6 5. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/6p1/3n4/8/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. g3 Bg7 6. Bg2
			variation => N__('English: Anglo-Grünfeld, 4. Nf3 g6 5. g3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/6p1/3np3/8/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. Nf3 g6 5. g3 Bg7 6. Bg2 e5
			variation => N__('English: Anglo-Grünfeld, Korchnoi Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3
			variation => N__('English: Anglo-Grünfeld, 3.cxd5'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/8/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6
			variation => N__('English: Anglo-Grünfeld, 3.cxd5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/8/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2
			variation => N__('English: Anglo-Grünfeld, 3.cxd5'),
			moves => {
				'd5b6' => 1,
				'd5c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/1n4p1/8/8/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2 Nb6
			variation => N__('English: Anglo-Grünfeld, Czech Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/8/8/2n3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2 Nxc3
			variation => N__('English: Anglo-Grünfeld, Smyslov Variation'),
			moves => {
				'b2c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/8/8/2P3P1/P2PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2 Nxc3 6. bxc3
			variation => N__('English: Anglo-Grünfeld, Smyslov Variation'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/6p1/8/8/2P3P1/P2PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2 Nxc3 6. bxc3 Bg7
			variation => N__('English: Anglo-Grünfeld, Smyslov Variation'),
			moves => {
				'a1b1' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/6p1/8/8/2P3P1/P2PPPBP/1RBQK1NR b Kkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. cxd5 Nxd5 4. g3 g6 5. Bg2 Nxc3 6. bxc3 Bg7 7. Rb1
			variation => N__('English: Anglo-Grünfeld, Smyslov, 7. Rb1'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. Nf3
			variation => N__('English: Anglo-Grünfeld'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. Nf3 g6
			variation => N__('English: Anglo-Grünfeld'),
			moves => {
				'd1a4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/Q1P5/2N2N2/PP1PPPPP/R1B1KB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. Nf3 g6 4. Qa4+
			variation => N__('English: Anglo-Grünfeld'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3p4/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d5 3. Nf3 g6 4. g3
			variation => N__('English: Anglo-Grünfeld'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 d6
			variation => N__('English: Anglo-Indian, 2. Nc3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6
			variation => N__('English: Anglo-Indian, 2. Nc3 g6'),
			moves => {
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. e4
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. e4'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQ' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2 O-O
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQ' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2 O-O 5. e4
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQ' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2 O-O 5. Nf3
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQ' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2 O-O 5. Nf3 d6
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b -' => {
			eco => 'A16',
			# TRANSLATORS: A16: 1. c4 Nf6 2. Nc3 g6 3. g3 Bg7 4. Bg2 O-O 5. Nf3 d6 6. O-O
			variation => N__('English: Anglo-Indian, 2. Nc3 g6 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6
			variation => N__('English: Anglo-Indian, 2. Nc3 e6'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3
			variation => N__('English: Anglo-Indian, 2. Nc3 e6 3. Nf3'),
			moves => {
				'b7b6' => 1,
				'd7d5' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6
			variation => N__('English: Anglo-Queen\'s Indian'),
			moves => {
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/2P1P3/2N2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. e4
			variation => N__('English: Anglo-Queen\'s Indian, 4. e4'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/2P1P3/2N2N2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. e4 Bb7
			variation => N__('English: Anglo-Queen\'s Indian, 4. e4'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/2P1P3/2NB1N2/PP1P1PPP/R1BQK2R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. e4 Bb7 5. Bd3
			variation => N__('English: Anglo-Queen\'s Indian, Romanishin'),
			moves => {
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7 5. Bg2
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rn1qk2r/pbppbppp/1p2pn2/8/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7 5. Bg2 Be7
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qk2r/pbppbppp/1p2pn2/8/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7 5. Bg2 Be7 6. O-O
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/pbppbppp/1p2pn2/8/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7 5. Bg2 Be7 6. O-O O-O
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
				'f1e1' => 1,
			},
		},
		'rn1q1rk1/pbppbppp/1p2pn2/8/2P5/2N2NP1/PP1PPPBP/R1BQR1K1 b -' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 b6 4. g3 Bb7 5. Bg2 Be7 6. O-O O-O 7. Re1
			variation => N__('English: Anglo-Queen\'s Indian, 4. g3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 d5
			variation => N__('English: Anglo-Indian, 2. Nc3 e6 3. Nf3 d5'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/4pn2/8/1bP5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 Bb4
			variation => N__('English: Nimzo-English'),
			moves => {
				'd1c2' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/4pn2/8/1bP5/2N2N2/PPQPPPPP/R1B1KB1R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 Bb4 4. Qc2
			variation => N__('English: Nimzo-English, 4. Qc2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pppp1ppp/4pn2/8/1bP5/2N2N2/PPQPPPPP/R1B1KB1R w KQ' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 Bb4 4. Qc2 O-O
			variation => N__('English: Nimzo-English, 4. Qc2 O-O'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/4pn2/8/1bP5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. Nf3 Bb4 4. g3
			variation => N__('English: Nimzo-English, 4. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A17',
			# TRANSLATORS: A17: 1. c4 Nf6 2. Nc3 e6 3. g3
			variation => N__('English: Anglo-Indian, 2. Nc3 e6 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4
			variation => N__('English: Mikenas'),
			moves => {
				'b8c6' => 1,
				'c7c5' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n1pn2/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 Nc6
			variation => N__('English: Mikenas, Kevitz Defence'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3p4/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5
			variation => N__('English: Mikenas, French Variation'),
			moves => {
				'c4d5' => 1,
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3P4/4P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. cxd5
			variation => N__('English: Mikenas, French, 4.cxd5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3pP3/2P5/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5
			variation => N__('English: Mikenas, Flohr Variation'),
			moves => {
				'd5d4' => 1,
				'f6e4' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/4P3/2Pp4/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4
			variation => N__('English: Mikenas, Flohr, 4... d4'),
			moves => {
				'e5f6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pP2/8/2Pp4/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6
			variation => N__('English: Mikenas, Flohr, 4... d4'),
			moves => {
				'd4c3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pP2/8/2P5/2p5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3
			variation => N__('English: Mikenas, Flohr, 4... d4'),
			moves => {
				'b2c3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4pP2/8/2P5/2P5/P2P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3
			variation => N__('English: Mikenas, Flohr, 4... d4'),
			moves => {
				'd8f6' => 1,
			},
		},
		'rnb1kb1r/ppp2ppp/4pq2/8/2P5/2P5/P2P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3 Qxf6
			variation => N__('English: Mikenas, Flohr, 4... d4'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'rnb1kb1r/ppp2ppp/4pq2/8/2PP4/2P5/P4PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3 Qxf6 7. d4
			variation => N__('English: Mikenas, Flohr, 7. d4'),
			moves => {
				'c7c5' => 1,
				'e6e5' => 1,
			},
		},
		'rnb1kb1r/pp3ppp/4pq2/2p5/2PP4/2P5/P4PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3 Qxf6 7. d4 c5
			variation => N__('English: Mikenas, Flohr, 7. d4 c5'),
			moves => {
			},
		},
		'rnb1kb1r/ppp2ppp/5q2/4p3/2PP4/2P5/P4PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3 Qxf6 7. d4 e5
			variation => N__('English: Mikenas, Flohr, 7. d4 e5'),
			moves => {
			},
		},
		'rnb1kb1r/ppp2ppp/4pq2/8/2P5/2P2N2/P2P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 d4 5. exf6 dxc3 6. bxc3 Qxf6 7. Nf3
			variation => N__('English: Mikenas, Flohr, 7. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4p3/3pP3/2P1n3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d5 4. e5 Ne4
			variation => N__('English: Mikenas, Flohr, 4... Ne4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3ppn2/8/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d6
			variation => N__('English: Mikenas, 3... d6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3ppn2/8/2PPP3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d6 4. d4
			variation => N__('English: Mikenas, 3... d6'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rnbqk2r/ppp1bppp/3ppn2/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A18',
			# TRANSLATORS: A18: 1. c4 Nf6 2. Nc3 e6 3. e4 d6 4. d4 Be7
			variation => N__('English: Mikenas, 3... d6 4. d4 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5
			variation => N__('English: Mikenas, Sicilian Variation'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p1P3/2P5/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5
			variation => N__('English: Mikenas, Sicilian, 4. e5'),
			moves => {
				'f6g8' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p1P3/2P5/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8
			variation => N__('English: Mikenas, Sicilian, 4. e5 Ng8'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p1P3/2PP4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. d4
			variation => N__('English: Mikenas, Nei Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p1P3/2P5/2N2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3
			variation => N__('English: Mikenas, 5. Nf3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p1P3/2P5/2N2N2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3 Nc6
			variation => N__('English: Mikenas, 5. Nf3 Nc6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p1P3/2PP4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3 Nc6 6. d4
			variation => N__('English: Mikenas, 5. Nf3 Nc6 6. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/4P3/2Pp4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3 Nc6 6. d4 cxd4
			variation => N__('English: Mikenas, 5. Nf3 Nc6 6. d4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/4P3/2PN4/2N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3 Nc6 6. d4 cxd4 7. Nxd4
			variation => N__('English: Mikenas, 5. Nf3 Nc6 6. d4'),
			moves => {
				'c6e5' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/4p3/4n3/2PN4/2N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A19',
			# TRANSLATORS: A19: 1. c4 Nf6 2. Nc3 e6 3. e4 c5 4. e5 Ng8 5. Nf3 Nc6 6. d4 cxd4 7. Nxd4 Nxe5
			variation => N__('English: Mikenas, 6. d4 cxd4 7.Nxd4 Nxe5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5
			variation => N__('English: King\'s (1... e5)'),
			moves => {
				'b1c3' => 1,
				'd2d3' => 1,
				'e2e3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/3P4/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. d3
			variation => N__('English: King\'s, 2. d3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/4P3/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. e3
			variation => N__('English: King\'s, 2. e3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. Nf3
			variation => N__('English: King\'s, Nimzowitsch'),
			moves => {
				'b8c6' => 1,
				'e5e4' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. Nf3 Nc6
			variation => N__('English: King\'s, Nimzowitsch, 2... Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/8/2P1p3/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. Nf3 e4
			variation => N__('English: King\'s, Nimzowitsch, Flohr Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3
			variation => N__('English: King\'s, 2. g3'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'd7d6' => 1,
				'f7f5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nc6
			variation => N__('English: King\'s, 2. g3 Nc6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nc6 3. Bg2
			variation => N__('English: King\'s, 2. g3 Nc6 3. Bg2'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/2p5/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 c6
			variation => N__('English: King\'s, 2. g3 c6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/2p5/4p3/2PP4/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 c6 3. d4
			variation => N__('English: King\'s, 2. g3 c6 3. d4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 d6
			variation => N__('English: King\'s, 2. g3 d6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 d6 3. Bg2
			variation => N__('English: King\'s, 2. g3 d6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/8/4pp2/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 f5
			variation => N__('English: King\'s, 2. g3 f5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1p1p/6p1/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 g6
			variation => N__('English: King\'s, 2. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pppp1p1p/6p1/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 g6 3. Bg2
			variation => N__('English: King\'s, 2. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pppp1pbp/6p1/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 g6 3. Bg2 Bg7
			variation => N__('English: King\'s, 2. g3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6
			variation => N__('English: King\'s, 2. g3 Nf6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR b KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6 3. Bg2
			variation => N__('English: King\'s, 2. g3 Nf6 3. Bg2'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'f8c5' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6 3. Bg2 Nc6
			variation => N__('English: King\'s, 2. g3 Nf6 3. Bg2 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/4p3/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6 3. Bg2 c6
			variation => N__('English: King\'s, 2. g3 Nf6 3. Bg2 c6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/3pp3/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6 3. Bg2 d5
			variation => N__('English: King\'s, 2. g3 Nf6 3. Bg2 d5'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/5n2/2b1p3/2P5/6P1/PP1PPPBP/RNBQK1NR w KQkq' => {
			eco => 'A20',
			# TRANSLATORS: A20: 1. c4 e5 2. g3 Nf6 3. Bg2 Bc5
			variation => N__('English: King\'s, 2. g3 Nf6 3. Bg2 Bc5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3
			variation => N__('English: King\'s, 2. Nc3'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'f7f5' => 1,
				'f8b4' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6
			variation => N__('English: King\'s, 2. Nc3 d6'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. d4
			variation => N__('English: King\'s, 2. Nc3 d6 3. d4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3
			variation => N__('English: King\'s, 2. Nc3 d6 3. Nf3'),
			moves => {
				'c8g4' => 1,
				'f7f5' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkbnr/ppp2ppp/3p4/4p3/2P3b1/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 Bg4
			variation => N__('English: Smyslov Defence'),
			moves => {
			},
		},
		'rnbqkbnr/ppp3pp/3p4/4pp2/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5
			variation => N__('English: Lukin Variation'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/ppp3pp/3p4/4pp2/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4
			variation => N__('English: Lukin Variation'),
			moves => {
				'e5e4' => 1,
			},
		},
		'rnbqkbnr/ppp3pp/3p4/5p2/2PPp3/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4
			variation => N__('English: Lukin, 4. d4 e4'),
			moves => {
				'f3d2' => 1,
				'f3g5' => 1,
			},
		},
		'rnbqkbnr/ppp3pp/3p4/5p2/2PPp3/2N5/PP1NPPPP/R1BQKB1R b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Nd2
			variation => N__('English: Lukin, 5. Nd2'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/3p1n2/5p2/2PPp3/2N5/PP1NPPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Nd2 Nf6
			variation => N__('English: Lukin, 5. Nd2'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/3p1n2/5p2/2PPp3/2N1P3/PP1N1PPP/R1BQKB1R b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Nd2 Nf6 6. e3
			variation => N__('English: Lukin, 5. Nd2 Nf6 6. e3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp3pp/3p4/5pN1/2PPp3/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Ng5
			variation => N__('English: Lukin, 5. Ng5'),
			moves => {
				'c7c6' => 1,
				'f8e7' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp4pp/2pp4/5pN1/2PPp3/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Ng5 c6
			variation => N__('English: Lukin, 5. Ng5 c6'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1b1pp/3p4/5pN1/2PPp3/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Ng5 Be7
			variation => N__('English: Lukin, 5. Ng5 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp3pp/3p1n2/5pN1/2PPp3/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 f5 4. d4 e4 5. Ng5 Nf6
			variation => N__('English: Lukin, 5. Ng5 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2p1p/3p2p1/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. Nf3 g6
			variation => N__('English: King\'s, 2. Nc3 d6 3. Nf3 g6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. g3
			variation => N__('English: King\'s, 2. Nc3 d6 3. g3'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2pp4/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. g3 c6
			variation => N__('English: King\'s, Keres Variation'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2pp4/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 d6 3. g3 c6 4. Bg2
			variation => N__('English: King\'s, Keres Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/8/4pp2/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 f5
			variation => N__('English: King\'s, 2. Nc3 f5'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/8/4pp2/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 f5 3. g3
			variation => N__('English: King\'s, 2. Nc3 f5'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/5n2/4pp2/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 f5 3. g3 Nf6
			variation => N__('English: King\'s, 2. Nc3 f5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/5n2/4pp2/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 f5 3. g3 Nf6 4. Bg2
			variation => N__('English: King\'s, 2. Nc3 f5'),
			moves => {
			},
		},
		'rnbqk1nr/pppp1ppp/8/4p3/1bP5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4
			variation => N__('English: Kramnik-Shirov Counterattack'),
			moves => {
				'c3d5' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/8/3Np3/1bP5/8/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5
			variation => N__('English: Kramnik-Shirov, 3. Nd5'),
			moves => {
				'a7a5' => 1,
				'b4a5' => 1,
				'b4c5' => 1,
				'b4e7' => 1,
			},
		},
		'rnbqk1nr/1ppp1ppp/8/p2Np3/1bP5/8/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5 a5
			variation => N__('English: Kramnik-Shirov, 3. Nd5 a5'),
			moves => {
			},
		},
		'rnbqk1nr/pppp1ppp/8/b2Np3/2P5/8/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5 Ba5
			variation => N__('English: Kramnik-Shirov, 3. Nd5 Ba5'),
			moves => {
			},
		},
		'rnbqk1nr/pppp1ppp/8/2bNp3/2P5/8/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5 Bc5
			variation => N__('English: Kramnik-Shirov, 3. Nd5 Bc5'),
			moves => {
			},
		},
		'rnbqk1nr/ppppbppp/8/3Np3/2P5/8/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5 Be7
			variation => N__('English: Kramnik-Shirov, 3. Nd5 Be7'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqk1nr/ppppbppp/8/3Np3/2PP4/8/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. Nd5 Be7 4. d4
			variation => N__('English: Kramnik-Shirov, 3. Nd5 Be7 4. d4'),
			moves => {
			},
		},
		'rnbqk1nr/pppp1ppp/8/4p3/1bP5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. g3
			variation => N__('English: Kramnik-Shirov, 3. g3'),
			moves => {
				'b4c3' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/8/4p3/2P5/2b3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. g3 Bxc3
			variation => N__('English: Kramnik-Shirov, 3. g3 Bxc3'),
			moves => {
				'b2c3' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/8/4p3/2P5/2P3P1/P2PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 Bb4 3. g3 Bxc3 4. bxc3
			variation => N__('English: Kramnik-Shirov, 3. g3 Bxc3 4.bxc3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1p1p/6p1/4p3/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 g6
			variation => N__('English: King\'s, 2. Nc3 g6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pppp1p1p/6p1/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 g6 3. g3
			variation => N__('English: King\'s, 2. Nc3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pppp1pbp/6p1/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 g6 3. g3 Bg7
			variation => N__('English: King\'s, 2. Nc3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk1nr/pppp1pbp/6p1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A21',
			# TRANSLATORS: A21: 1. c4 e5 2. Nc3 g6 3. g3 Bg7 4. Bg2
			variation => N__('English: King\'s, 2. Nc3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6
			variation => N__('English: King\'s, 2. Nc3 Nf6'),
			moves => {
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/2N1P3/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. e3
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. e3'),
			moves => {
				'f8b4' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4p3/1bP5/2N1P3/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. e3 Bb4
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. e3 Bb4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. e4
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. e4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. Nf3
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. Nf3'),
			moves => {
				'd7d6' => 1,
				'e5e4' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. Nf3 d6
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. Nf3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/8/2P1p3/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. Nf3 e4
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. Nf3 e4'),
			moves => {
				'f3g5' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/6N1/2P1p3/2N5/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. Nf3 e4 4. Ng5
			variation => N__('English: King\'s, 2. Nc3 Nf6 3. Nf3 e4'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/5n2/1p4N1/2P1p3/2N5/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. Nf3 e4 4. Ng5 b5
			variation => N__('English: Bellon Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3
			variation => N__('English: Bremen System'),
			moves => {
				'c7c6' => 1,
				'd7d5' => 1,
				'f8b4' => 1,
				'f8c5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/3pp3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 d5
			variation => N__('English: Bremen, Reverse Dragon'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/3Pp3/8/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 d5 4. cxd5
			variation => N__('English: Bremen, Reverse Dragon, 4.cxd5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/8/3np3/8/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5
			variation => N__('English: Bremen, Reverse Dragon, 4.cxd5 Nxd5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/8/3np3/8/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5 5. Bg2
			variation => N__('English: Bremen, Reverse Dragon, 4.cxd5 Nxd5 5. Bg2'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/1n6/4p3/8/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5 5. Bg2 Nb6
			variation => N__('English: Bremen, Reverse Dragon, 4.cxd5 Nxd5 5. Bg2 Nb6'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4p3/1bP5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bb4
			variation => N__('English: Bremen, Smyslov System'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4p3/1bP5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bb4 4. Bg2
			variation => N__('English: Bremen, Smyslov, 4. Bg2'),
			moves => {
				'b4c3' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/5n2/4p3/2P5/2b3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bb4 4. Bg2 Bxc3
			variation => N__('English: Bremen, Smyslov, 4. Bg2 Bxc3'),
			moves => {
			},
		},
		'rnbq1rk1/pppp1ppp/5n2/4p3/1bP5/2N3P1/PP1PPPBP/R1BQK1NR w KQ' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bb4 4. Bg2 O-O
			variation => N__('English: Bremen, Smyslov, 4. Bg2 O-O'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbq1rk1/pppp1ppp/5n2/4p3/1bP1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQ' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bb4 4. Bg2 O-O 5. e4
			variation => N__('English: Bremen, Smyslov, 4. Bg2 O-O 5. e4'),
			moves => {
			},
		},
		'rnbqk2r/pppp1ppp/5n2/2b1p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A22',
			# TRANSLATORS: A22: 1. c4 e5 2. Nc3 Nf6 3. g3 Bc5
			variation => N__('English: Bremen, 3... Bc5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6
			variation => N__('English: Bremen, Keres System'),
			moves => {
				'f1g2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Bg2
			variation => N__('English: Bremen, Keres, 4. Bg2'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p2n2/3pp3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Bg2 d5
			variation => N__('English: Bremen, Keres, 4. Bg2 d5'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p2n2/3Pp3/8/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Bg2 d5 5. cxd5
			variation => N__('English: Bremen, Keres, 4. Bg2 d5 5.cxd5'),
			moves => {
				'c6d5' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/5n2/3pp3/8/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Bg2 d5 5. cxd5 cxd5
			variation => N__('English: Bremen, Keres, 4. Bg2 d5 5.cxd5 cxd5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/4p3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Nf3
			variation => N__('English: Bremen, Keres, 4. Nf3'),
			moves => {
				'd7d6' => 1,
				'e5e4' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2pp1n2/4p3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Nf3 d6
			variation => N__('English: Bremen, Keres, 4. Nf3 d6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2pp1n2/4p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Nf3 d6 5. Bg2
			variation => N__('English: Bremen, Keres, 4. Nf3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/2p2n2/8/2P1p3/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A23',
			# TRANSLATORS: A23: 1. c4 e5 2. Nc3 Nf6 3. g3 c6 4. Nf3 e4
			variation => N__('English: Bremen, Keres, 4. Nf3 e4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1p1p/5np1/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pppp1p1p/5np1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pppp1pbp/5np1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd2d3' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pppp1pbp/5np1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. d3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'rnbqk2r/pppp1pbp/5np1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e3 d6
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'g1e2' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQ' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2 O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'rnbqk2r/pppp1pbp/5np1/4p3/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4 d6
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQ' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQK2R b KQ' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 O-O 7. d3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P1P3/2N2NP1/PP1P1PBP/R1BQK2R b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nf3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'rnbqk2r/pppp1pbp/5np1/4p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'd2d3' => 1,
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3 O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3 O-O 7. O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/2pp1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3 O-O 7. O-O c6
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'rnbqk2r/ppp2pbp/3p1np1/4p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp2pbp/3p1np1/4p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A24',
			# TRANSLATORS: A24: 1. c4 e5 2. Nc3 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O O-O
			variation => N__('English: Bremen, 3... g6'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6
			variation => N__('English: Closed'),
			moves => {
				'e2e3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/2N1P3/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. e3
			variation => N__('English: Closed'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N1P3/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. e3 Nf6
			variation => N__('English: Closed'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3
			variation => N__('English: Closed'),
			moves => {
				'd7d6' => 1,
				'f7f5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2np4/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 d6
			variation => N__('English: Closed'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2np4/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 d6 4. Bg2
			variation => N__('English: Closed'),
			moves => {
				'c8e6' => 1,
			},
		},
		'r2qkbnr/ppp2ppp/2npb3/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 d6 4. Bg2 Be6
			variation => N__('English: Closed, Tröger Defence'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r2qkbnr/ppp2ppp/2npb3/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 d6 4. Bg2 Be6 5. d3
			variation => N__('English: Closed, Tröger, 5. d3'),
			moves => {
			},
		},
		'r1bqkbnr/pppp2pp/2n5/4pp2/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5
			variation => N__('English: Closed'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pppp2pp/2n5/4pp2/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5 4. Bg2
			variation => N__('English: Closed'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pppp2pp/2n2n2/4pp2/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5 4. Bg2 Nf6
			variation => N__('English: Closed'),
			moves => {
				'd2d3' => 1,
				'e2e3' => 1,
			},
		},
		'r1bqkb1r/pppp2pp/2n2n2/4pp2/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5 4. Bg2 Nf6 5. d3
			variation => N__('English: Closed'),
			moves => {
			},
		},
		'r1bqkb1r/pppp2pp/2n2n2/4pp2/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5 4. Bg2 Nf6 5. e3
			variation => N__('English: Closed'),
			moves => {
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pppp3p/2n2np1/4pp2/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 f5 4. Bg2 Nf6 5. e3 g6
			variation => N__('English: Closed'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6
			variation => N__('English: Closed'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2
			variation => N__('English: Closed'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7
			variation => N__('English: Closed'),
			moves => {
				'a1b1' => 1,
				'd2d3' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR b Kkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Rb1
			variation => N__('English: Closed, 5. Rb1'),
			moves => {
				'a7a5' => 1,
				'g8h6' => 1,
			},
		},
		'r1bqk1nr/1ppp1pbp/2n3p1/p3p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR w Kkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Rb1 a5
			variation => N__('English: Closed, 5. Rb1'),
			moves => {
			},
		},
		'r1bqk2r/pppp1pbp/2n3pn/4p3/2P5/2N3P1/PP1PPPBP/1RBQK1NR w Kkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Rb1 Nh6
			variation => N__('English: Closed, 5. Rb1, Taimanov Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3
			variation => N__('English: Closed, 5. d3'),
			moves => {
				'd7d6' => 1,
				'g8e7' => 1,
			},
		},
		'r1bqk2r/ppppnpbp/2n3p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 Nge7
			variation => N__('English: Closed, 5. d3 Nge7'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3
			variation => N__('English: Closed, 5. e3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 d6
			variation => N__('English: Closed, 5. e3'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2
			variation => N__('English: Closed, 5. e3'),
			moves => {
				'c8e6' => 1,
				'g8h6' => 1,
			},
		},
		'r2qk1nr/ppp2pbp/2npb1p1/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2 Be6
			variation => N__('English: Closed, Hort Variation'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r2qk1nr/ppp2pbp/2npb1p1/4p3/2P5/2NPP1P1/PP2NPBP/R1BQK2R b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2 Be6 7. d3
			variation => N__('English: Closed, Hort, 7. d3'),
			moves => {
			},
		},
		'r1bqk2r/ppp2pbp/2np2pn/4p3/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 d6 6. Nge2 Nh6
			variation => N__('English: Closed, Taimanov Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1pbp/2n3p1/4p3/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4
			variation => N__('English: Closed, 5. e4'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6
			variation => N__('English: Closed'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6 4. Bg2
			variation => N__('English: Closed'),
			moves => {
				'f8b4' => 1,
				'f8c5' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/1bP5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6 4. Bg2 Bb4
			variation => N__('English: Closed'),
			moves => {
				'c3d5' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/3Np3/1bP5/6P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6 4. Bg2 Bb4 5. Nd5
			variation => N__('English: Closed'),
			moves => {
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6 4. Bg2 Bc5
			variation => N__('English: Closed'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A25',
			# TRANSLATORS: A25: 1. c4 e5 2. Nc3 Nc6 3. g3 Nf6 4. Bg2 Bc5 5. e3
			variation => N__('English: Closed'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/R1BQK1NR w KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6
			variation => N__('English: Closed, 5. d3 d6'),
			moves => {
				'a1b1' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR b Kkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1
			variation => N__('English: Closed, 5. d3 d6 6. Rb1'),
			moves => {
				'a7a5' => 1,
				'f7f5' => 1,
			},
		},
		'r1bqk1nr/1pp2pbp/2np2p1/p3p3/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1 a5
			variation => N__('English: Closed, 5. d3 d6 6. Rb1 a5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp3bp/2np2p1/4pp2/2P5/2NP2P1/PP2PPBP/1RBQK1NR w Kkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Rb1 f5
			variation => N__('English: Closed, 5. d3 d6 6. Rb1 f5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NPP1P1/PP3PBP/R1BQK1NR b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e3
			variation => N__('English: Closed, 5. d3 d6 6. e3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4
			variation => N__('English: Botvinnik System'),
			moves => {
				'g8e7' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk2r/ppp1npbp/2np2p1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR w KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nge7
			variation => N__('English: Botvinnik System, 6...Nge7'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk2r/ppp1npbp/2np2p1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQK2R b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nge7 7. Nge2
			variation => N__('English: Botvinnik System, 6...Nge7 7.Nge2'),
			moves => {
			},
		},
		'r1bqk2r/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP3PBP/R1BQK1NR w KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqk2r/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQK2R b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nge2
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQK2R w KQ' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nge2 O-O
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nge2 O-O 8. O-O
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
			},
		},
		'r1bqk2r/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQK2R b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nf3
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQK2R w KQ' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nf3 O-O
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P1P3/2NP1NP1/PP3PBP/R1BQ1RK1 b -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. e4 Nf6 7. Nf3 O-O 8. O-O
			variation => N__('English: Botvinnik System, 6... Nf6'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3
			variation => N__('English: Closed, 5. d3 d6 6. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqk2r/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQkq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O'),
			moves => {
				'a1b1' => 1,
			},
		},
		'r1bq1rk1/ppp2pbp/2np1np1/4p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1'),
			moves => {
				'a7a5' => 1,
			},
		},
		'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 w -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5'),
			moves => {
				'a2a3' => 1,
			},
		},
		'r1bq1rk1/1pp2pbp/2np1np1/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 b -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5 9. a3
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5 9. a3'),
			moves => {
				'h7h6' => 1,
			},
		},
		'r1bq1rk1/1pp2pb1/2np1npp/p3p3/2P5/P1NP1NP1/1P2PPBP/1RBQ1RK1 w -' => {
			eco => 'A26',
			# TRANSLATORS: A26: 1. c4 e5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nf3 Nf6 7. O-O O-O 8. Rb1 a5 9. a3 h6
			variation => N__('English: Closed, 5. d3 d6 6. Nf3 Nf6 7.O-O O-O 8. Rb1 a5 9. a3 h6'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3
			variation => N__('English: Three Knights'),
			moves => {
				'd7d6' => 1,
				'f7f5' => 1,
				'f8b4' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/ppp2ppp/2np4/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 d6
			variation => N__('English: Three Knights, 3... d6'),
			moves => {
			},
		},
		'r1bqkbnr/pppp2pp/2n5/4pp2/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 f5
			variation => N__('English: Three Knights, 3... f5'),
			moves => {
				'd2d4' => 1,
			},
		},
		'r1bqkbnr/pppp2pp/2n5/4pp2/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 f5 4. d4
			variation => N__('English: Three Knights, 3... f5 4. d4'),
			moves => {
				'e5e4' => 1,
			},
		},
		'r1bqkbnr/pppp2pp/2n5/5p2/2PPp3/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 f5 4. d4 e4
			variation => N__('English: Three Knights, 3... f5 4. d4 e4'),
			moves => {
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/4p3/1bP5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Bb4
			variation => N__('English: Three Knights, 3... Bb4'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 g6
			variation => N__('English: Three Knights, 3... g6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/4p3/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 g6 4. d4
			variation => N__('English: Three Knights, 3... g6 4. d4'),
			moves => {
				'e5d4' => 1,
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/8/2Pp4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 g6 4. d4 exd4
			variation => N__('English: Three Knights, 3... g6 4. d4 exd4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'r1bqkbnr/pppp1p1p/2n3p1/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A27',
			# TRANSLATORS: A27: 1. c4 e5 2. Nc3 Nc6 3. Nf3 g6 4. d4 exd4 5. Nxd4
			variation => N__('English: Three Knights, 3... g6 4. d4 exd4 5.Nxd4'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6
			variation => N__('English: Four Knights'),
			moves => {
				'a2a3' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/P1N2N2/1P1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. a3
			variation => N__('English: Four Knights, Marini Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2NP1N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d3
			variation => N__('English: Four Knights, Capablanca Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'e5d4' => 1,
				'e5e4' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/8/2Pp4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'f8b4' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/8/1bPN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'c1g5' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/6B1/1bPN4/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'h7h6' => 1,
			},
		},
		'r1bqk2r/pppp1pp1/2n2n1p/6B1/1bPN4/2N5/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5 h6
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'g5h4' => 1,
			},
		},
		'r1bqk2r/pppp1pp1/2n2n1p/8/1bPN3B/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5 h6 7. Bh4
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'b4c3' => 1,
			},
		},
		'r1bqk2r/pppp1pp1/2n2n1p/8/2PN3B/2b5/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5 h6 7. Bh4 Bxc3+
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'b2c3' => 1,
			},
		},
		'r1bqk2r/pppp1pp1/2n2n1p/8/2PN3B/2P5/P3PPPP/R2QKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5 h6 7. Bh4 Bxc3+ 8. bxc3
			variation => N__('English: Four Knights, 4. d4'),
			moves => {
				'c6e5' => 1,
			},
		},
		'r1bqk2r/pppp1pp1/5n1p/4n3/2PN3B/2P5/P3PPPP/R2QKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 exd4 5. Nxd4 Bb4 6. Bg5 h6 7. Bh4 Bxc3+ 8. bxc3 Ne5
			variation => N__('English: Four Knights, Nenarokov Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/8/2PPp3/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4 e4
			variation => N__('English: Four Knights, Bradley Beach Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N1PN2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3
			variation => N__('English: Four Knights, 4. e3'),
			moves => {
				'f8b4' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/1bP5/2N1PN2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4
			variation => N__('English: Four Knights, 4. e3 Bb4'),
			moves => {
				'd1c2' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/1bP5/2N1PN2/PPQP1PPP/R1B1KB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2
			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'b4c3' => 1,
				'e8g8' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/2P5/2b1PN2/PPQP1PPP/R1B1KB1R w KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2 Bxc3
			variation => N__('English: Four Knights, Romanishin Variation'),
			moves => {
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N1PN2/PPQP1PPP/R1B1KB1R w KQ' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2 O-O
			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'c3d5' => 1,
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/3Np3/1bP5/4PN2/PPQP1PPP/R1B1KB1R b KQ' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2 O-O 6. Nd5
			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'f8e8' => 1,
			},
		},
		'r1bqr1k1/pppp1ppp/2n2n2/3Np3/1bP5/4PN2/PPQP1PPP/R1B1KB1R w KQ' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2 O-O 6. Nd5 Re8
			variation => N__('English: Four Knights, 4. e3 Bb4 5. Qc2'),
			moves => {
				'c2f5' => 1,
			},
		},
		'r1bqr1k1/pppp1ppp/2n2n2/3NpQ2/1bP5/4PN2/PP1P1PPP/R1B1KB1R b KQ' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 Bb4 5. Qc2 O-O 6. Nd5 Re8 7. Qf5
			variation => N__('English: Four Knights, Stean Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P1P3/2N2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A28',
			# TRANSLATORS: A28: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. e4
			variation => N__('English: Four Knights, Nimzowitsch Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n2n2/4p3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3
			variation => N__('English: Four Knights, 4. g3'),
			moves => {
				'd7d5' => 1,
				'f8b4' => 1,
				'f8c5' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n2n2/3pp3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5
			variation => N__('English: Four Knights, 4. g3 d5'),
			moves => {
				'c4d5' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n2n2/3Pp3/8/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n5/3np3/8/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2n5/3np3/8/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2'),
			moves => {
				'c8e6' => 1,
				'd5b6' => 1,
			},
		},
		'r2qkb1r/ppp2ppp/2n1b3/3np3/8/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Be6
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Be6'),
			moves => {
			},
		},
		'r1bqkb1r/ppp2ppp/1nn5/4p3/8/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/1nn5/4p3/8/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/ppp1bppp/1nn5/4p3/8/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7'),
			moves => {
				'a1b1' => 1,
				'a2a3' => 1,
				'd2d3' => 1,
			},
		},
		'r1bqk2r/ppp1bppp/1nn5/4p3/8/2N2NP1/PP1PPPBP/1RBQ1RK1 b kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7 8. Rb1
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7 8. Rb1'),
			moves => {
			},
		},
		'r1bqk2r/ppp1bppp/1nn5/4p3/8/P1N2NP1/1P1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7 8. a3
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7 8. a3'),
			moves => {
			},
		},
		'r1bqk2r/ppp1bppp/1nn5/4p3/8/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7 8. d3
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7 8. d3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/ppp1bppp/1nn5/4p3/8/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5 Nxd5 6. Bg2 Nb6 7. O-O Be7 8. d3 O-O
			variation => N__('English: Four Knights, 4. g3 d5 5.cxd5 Nxd5 6. Bg2 Nb6 7.O-O Be7 8. d3'),
			moves => {
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4
			variation => N__('English: Four Knights, 4. g3 Bb4'),
			moves => {
				'c3d5' => 1,
				'f1g2' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/3Np3/1bP5/5NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Nd5
			variation => N__('English: Four Knights, 4. g3 Bb4 5. Nd5'),
			moves => {
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2
			variation => N__('English: Four Knights, 4. g3 Bb4 5. Bg2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQK2R w KQ' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2 O-O
			variation => N__('English: Four Knights, 4. g3 Bb4 5. Bg2'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/4p3/1bP5/2N2NP1/PP1PPPBP/R1BQ1RK1 b -' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2 O-O 6. O-O
			variation => N__('English: Four Knights, Main Line 6.O-O'),
			moves => {
				'e5e4' => 1,
			},
		},
		'r1bq1rk1/pppp1ppp/2n2n2/8/1bP1p3/2N2NP1/PP1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bb4 5. Bg2 O-O 6. O-O e4
			variation => N__('English: Four Knights, Main Line 6.O-O e4'),
			moves => {
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n2n2/2b1p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bqk2r/ppp2ppp/2np1n2/2b1p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/ppp2ppp/2np1n2/2b1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6 6. O-O
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6 6. O-O O-O
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bq1rk1/ppp2ppp/2np1n2/2b1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 Bc5 5. Bg2 d6 6. O-O O-O 7. d3
			variation => N__('English: Four Knights, 4. g3 Bc5'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1p1p/2n2np1/4p3/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 g6
			variation => N__('English: Four Knights, 4. g3 g6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'r1bqkb1r/pppp1p1p/2n2np1/4p3/2PP4/2N2NP1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A29',
			# TRANSLATORS: A29: 1. c4 e5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 g6 5. d4
			variation => N__('English: Four Knights, 4. g3 g6 5. d4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P5/8/PP1PPPPP/RNBQKBNR w KQkq' => {
			eco => 'A30a',
			# TRANSLATORS: A30a: 1. c4 c5
			variation => N__('English: Symmetrical'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P5/1P6/P2PPPPP/RNBQKBNR b KQkq' => {
			eco => 'A30a',
			# TRANSLATORS: A30a: 1. c4 c5 2. b3
			variation => N__('English: Symmetrical, 2. b3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P5/6P1/PP1PPP1P/RNBQKBNR b KQkq' => {
			eco => 'A30a',
			# TRANSLATORS: A30a: 1. c4 c5 2. g3
			variation => N__('English: Symmetrical, 2. g3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P5/6P1/PP1PPP1P/RNBQKBNR w KQkq' => {
			eco => 'A30a',
			# TRANSLATORS: A30a: 1. c4 c5 2. g3 g6
			variation => N__('English: Symmetrical, 2. g3 g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P5/5N2/PP1PPPPP/RNBQKB1R b KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3
			variation => N__('English: Symmetrical, 2. Nf3'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 b6
			variation => N__('English: Symmetrical, 2. Nf3 b6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 b6 3. g3
			variation => N__('English: Symmetrical, 2. Nf3 b6 3. g3'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pb1ppppp/1p6/2p5/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 b6 3. g3 Bb7
			variation => N__('English: Symmetrical, 2. Nf3 b6 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkbnr/pb1ppppp/1p6/2p5/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 b6 3. g3 Bb7 4. Bg2
			variation => N__('English: Symmetrical, 2. Nf3 b6 3. g3 Bb7 4. Bg2'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 Nc6
			variation => N__('English: Symmetrical, 2. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A30b',
			# TRANSLATORS: A30b: 1. c4 c5 2. Nf3 g6
			variation => N__('English: Symmetrical, 2. Nf3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2P5/5N2/PP1PPPPP/RNBQKB1R w KQkq' => {
			eco => 'A30c',
			# TRANSLATORS: A30c: 1. c4 c5 2. Nf3 Nf6
			variation => N__('English: Symmetrical, 2. Nf3 Nf6'),
			moves => {
				'd2d4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2P5/5NP1/PP1PPP1P/RNBQKB1R b KQkq' => {
			eco => 'A30d',
			# TRANSLATORS: A30d: 1. c4 c5 2. Nf3 Nf6 3. g3
			variation => N__('English: Symmetrical, 2. Nf3 Nf6 3. g3'),
			moves => {
				'b7b6' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/2p5/2P5/5NP1/PP1PPP1P/RNBQKB1R w KQkq' => {
			eco => 'A30e',
			# TRANSLATORS: A30e: 1. c4 c5 2. Nf3 Nf6 3. g3 b6
			variation => N__('English: Symmetrical, b6 System'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/2p5/2P5/5NP1/PP1PPPBP/RNBQK2R b KQkq' => {
			eco => 'A30e',
			# TRANSLATORS: A30e: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2
			variation => N__('English: Symmetrical, b6 System'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/2p5/2P5/5NP1/PP1PPPBP/RNBQK2R w KQkq' => {
			eco => 'A30e',
			# TRANSLATORS: A30e: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7
			variation => N__('English: Symmetrical, b6 System'),
			moves => {
				'b1c3' => 1,
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A30e',
			# TRANSLATORS: A30e: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. Nc3
			variation => N__('English: Symmetrical, b6 System'),
			moves => {
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/2p5/2P5/5NP1/PP1PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A30e',
			# TRANSLATORS: A30e: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O
			variation => N__('English: Symmetrical, b6 System'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/pb1ppp1p/1p3np1/2p5/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A30f',
			# TRANSLATORS: A30f: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6
			variation => N__('English: Symmetrical, Double Fianchetto'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
			},
		},
		'rn1qkb1r/pb1ppp1p/1p3np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A30f',
			# TRANSLATORS: A30f: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. Nc3
			variation => N__('English: Symmetrical, Double Fianchetto'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rn1qk2r/pb1pppbp/1p3np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A30f',
			# TRANSLATORS: A30f: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. Nc3 Bg7
			variation => N__('English: Symmetrical, Double Fianchetto'),
			moves => {
			},
		},
		'rn1qkb1r/pb1ppp1p/1p3np1/2p5/2P5/1P3NP1/P2PPPBP/RNBQ1RK1 b kq' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3
			variation => N__('English: Symmetrical, Mutual Double Fianchetto'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rn1qk2r/pb1pppbp/1p3np1/2p5/2P5/1P3NP1/P2PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3 Bg7
			variation => N__('English: Symmetrical, Mutual Double Fianchetto'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rn1qk2r/pb1pppbp/1p3np1/2p5/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 b kq' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3 Bg7 7. Bb2
			variation => N__('English: Symmetrical, Mutual Double Fianchetto'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/pb1pppbp/1p3np1/2p5/2P5/1P3NP1/PB1PPPBP/RN1Q1RK1 w -' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3 Bg7 7. Bb2 O-O
			variation => N__('English: Symmetrical, Mutual Double Fianchetto'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rn1q1rk1/pb1pppbp/1p3np1/2p5/2P5/1PN2NP1/PB1PPPBP/R2Q1RK1 b -' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3 Bg7 7. Bb2 O-O 8. Nc3
			variation => N__('English: Symmetrical, Mutual Double Fianchetto'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rn1q1rk1/pb2ppbp/1p3np1/2pp4/2P5/1PN2NP1/PB1PPPBP/R2Q1RK1 w -' => {
			eco => 'A30g',
			# TRANSLATORS: A30g: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O g6 6. b3 Bg7 7. Bb2 O-O 8. Nc3 d5
			variation => N__('English: Symmetrical, Mutual Double Fianchetto with ...d5'),
			moves => {
			},
		},
		'rn1qkb1r/pb1p1ppp/1p2pn2/2p5/2P5/5NP1/PP1PPPBP/RNBQ1RK1 w kq' => {
			eco => 'A30h',
			# TRANSLATORS: A30h: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6
			variation => N__('English: Symmetrical, Hedgehog System'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rn1qkb1r/pb1p1ppp/1p2pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A30h',
			# TRANSLATORS: A30h: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3
			variation => N__('English: Symmetrical, Hedgehog System'),
			moves => {
				'a7a6' => 1,
				'd7d6' => 1,
				'f8e7' => 1,
			},
		},
		'rn1qkb1r/1b1p1ppp/pp2pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A30i',
			# TRANSLATORS: A30i: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 a6
			variation => N__('English: Symmetrical, Hedgehog, 6... a6'),
			moves => {
			},
		},
		'rn1qkb1r/pb3ppp/1p1ppn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A30j',
			# TRANSLATORS: A30j: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 d6
			variation => N__('English: Symmetrical, Hedgehog, 6... d6'),
			moves => {
				'b2b3' => 1,
				'd2d4' => 1,
			},
		},
		'rn1qkb1r/pb3ppp/1p1ppn2/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A30k',
			# TRANSLATORS: A30k: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 d6 7. b3
			variation => N__('English: Symmetrical, Hedgehog, 6... d6 7. b3'),
			moves => {
			},
		},
		'rn1qkb1r/pb3ppp/1p1ppn2/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A30l',
			# TRANSLATORS: A30l: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 d6 7. d4
			variation => N__('English: Symmetrical, Hedgehog, 6... d6 7. d4'),
			moves => {
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A30m',
			# TRANSLATORS: A30m: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7
			variation => N__('English: Symmetrical, Hedgehog System'),
			moves => {
				'b2b3' => 1,
				'd2d4' => 1,
				'f1e1' => 1,
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQR1K1 b kq' => {
			eco => 'A30n',
			# TRANSLATORS: A30n: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. Re1
			variation => N__('English: Symmetrical, Hedgehog, 7. Re1'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rn1qk2r/pb2bppp/1p2pn2/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQR1K1 w kq' => {
			eco => 'A30o',
			# TRANSLATORS: A30o: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. Re1 d5
			variation => N__('English: Symmetrical, Hedgehog, 7. Re1 d5'),
			moves => {
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A30p',
			# TRANSLATORS: A30p: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. b3
			variation => N__('English: Symmetrical, Hedgehog, 7. b3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/pb1pbppp/1p2pn2/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 w -' => {
			eco => 'A30p',
			# TRANSLATORS: A30p: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. b3 O-O
			variation => N__('English: Symmetrical, Hedgehog, 7. b3'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rn1q1rk1/pb1pbppp/1p2pn2/2p5/2P5/1PN2NP1/PB1PPPBP/R2Q1RK1 b -' => {
			eco => 'A30p',
			# TRANSLATORS: A30p: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. b3 O-O 8. Bb2
			variation => N__('English: Symmetrical, Hedgehog, 7. b3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rn1q1rk1/pb2bppp/1p1ppn2/2p5/2P5/1PN2NP1/PB1PPPBP/R2Q1RK1 w -' => {
			eco => 'A30p',
			# TRANSLATORS: A30p: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. b3 O-O 8. Bb2 d6
			variation => N__('English: Symmetrical, Hedgehog, 7. b3'),
			moves => {
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A30q',
			# TRANSLATORS: A30q: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4
			variation => N__('English: Symmetrical, Hedgehog, 7. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/8/2Pp4/2N2NP1/PP2PPBP/R1BQ1RK1 w kq' => {
			eco => 'A30q',
			# TRANSLATORS: A30q: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4
			variation => N__('English: Symmetrical, Hedgehog, 7. d4'),
			moves => {
				'd1d4' => 1,
			},
		},
		'rn1qk2r/pb1pbppp/1p2pn2/8/2PQ4/2N2NP1/PP2PPBP/R1B2RK1 b kq' => {
			eco => 'A30r',
			# TRANSLATORS: A30r: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/pb1pbppp/1p2pn2/8/2PQ4/2N2NP1/PP2PPBP/R1B2RK1 w -' => {
			eco => 'A30s',
			# TRANSLATORS: A30s: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 O-O
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4 O-O'),
			moves => {
				'f1d1' => 1,
			},
		},
		'rn1q1rk1/pb1pbppp/1p2pn2/8/2PQ4/2N2NP1/PP2PPBP/R1BR2K1 b -' => {
			eco => 'A30s',
			# TRANSLATORS: A30s: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 O-O 9. Rd1
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4 O-O'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2q1rk1/pb1pbppp/1pn1pn2/8/2PQ4/2N2NP1/PP2PPBP/R1BR2K1 w -' => {
			eco => 'A30s',
			# TRANSLATORS: A30s: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 O-O 9. Rd1 Nc6
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4 O-O'),
			moves => {
			},
		},
		'r2qk2r/pb1pbppp/1pn1pn2/8/2PQ4/2N2NP1/PP2PPBP/R1B2RK1 w kq' => {
			eco => 'A30t',
			# TRANSLATORS: A30t: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 Nc6
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4 Nc6'),
			moves => {
			},
		},
		'rn1qk2r/pb2bppp/1p1ppn2/8/2PQ4/2N2NP1/PP2PPBP/R1B2RK1 w kq' => {
			eco => 'A30u',
			# TRANSLATORS: A30u: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6
			variation => N__('English: Symmetrical, Hedgehog, 8.Qxd4 d6'),
			moves => {
				'b2b3' => 1,
				'f1d1' => 1,
			},
		},
		'rn1qk2r/pb2bppp/1p1ppn2/8/2PQ4/1PN2NP1/P3PPBP/R1B2RK1 b kq' => {
			eco => 'A30v',
			# TRANSLATORS: A30v: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. b3
			variation => N__('English: Symmetrical, Hedgehog, 9. b3'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qk2r/pb1nbppp/1p1ppn2/8/2PQ4/1PN2NP1/P3PPBP/R1B2RK1 w kq' => {
			eco => 'A30v',
			# TRANSLATORS: A30v: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. b3 Nbd7
			variation => N__('English: Symmetrical, Hedgehog, 9. b3 Nbd7'),
			moves => {
				'c3b5' => 1,
			},
		},
		'r2qk2r/pb1nbppp/1p1ppn2/1N6/2PQ4/1P3NP1/P3PPBP/R1B2RK1 b kq' => {
			eco => 'A30v',
			# TRANSLATORS: A30v: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. b3 Nbd7 10. Nb5
			variation => N__('English: Symmetrical, Hedgehog, 9. b3 Nbd7 10. Nb5'),
			moves => {
			},
		},
		'rn1qk2r/pb2bppp/1p1ppn2/8/2PQ4/2N2NP1/PP2PPBP/R1BR2K1 b kq' => {
			eco => 'A30w',
			# TRANSLATORS: A30w: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1
			variation => N__('English: Symmetrical, Hedgehog, 9. Rd1'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rn1qk2r/1b2bppp/pp1ppn2/8/2PQ4/2N2NP1/PP2PPBP/R1BR2K1 w kq' => {
			eco => 'A30w',
			# TRANSLATORS: A30w: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6
			variation => N__('English: Symmetrical, Hedgehog, 9. Rd1'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rn1qk2r/1b2bppp/pp1ppn2/8/2PQ4/1PN2NP1/P3PPBP/R1BR2K1 b kq' => {
			eco => 'A30w',
			# TRANSLATORS: A30w: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6 10. b3
			variation => N__('English: Symmetrical, Hedgehog, 9. Rd1'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qk2r/1b1nbppp/pp1ppn2/8/2PQ4/1PN2NP1/P3PPBP/R1BR2K1 w kq' => {
			eco => 'A30x',
			# TRANSLATORS: A30x: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6 10. b3 Nbd7
			variation => N__('English: Symmetrical, Hedgehog, Flexible Formation'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r2qk2r/1b1nbppp/pp1ppn2/8/2PQP3/1PN2NP1/P4PBP/R1BR2K1 b kq' => {
			eco => 'A30y',
			# TRANSLATORS: A30y: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6 10. b3 Nbd7 11. e4
			variation => N__('English: Symmetrical, Hedgehog, Flexible Formation'),
			moves => {
				'd8c7' => 1,
				'e8g8' => 1,
			},
		},
		'r3k2r/1bqnbppp/pp1ppn2/8/2PQP3/1PN2NP1/P4PBP/R1BR2K1 w kq' => {
			eco => 'A30y',
			# TRANSLATORS: A30y: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6 10. b3 Nbd7 11. e4 Qc7
			variation => N__('English: Symmetrical, Hedgehog, Flexible Formation, 11. e4 Qc7'),
			moves => {
			},
		},
		'r2q1rk1/1b1nbppp/pp1ppn2/8/2PQP3/1PN2NP1/P4PBP/R1BR2K1 w -' => {
			eco => 'A30y',
			# TRANSLATORS: A30y: 1. c4 c5 2. Nf3 Nf6 3. g3 b6 4. Bg2 Bb7 5. O-O e6 6. Nc3 Be7 7. d4 cxd4 8. Qxd4 d6 9. Rd1 a6 10. b3 Nbd7 11. e4 O-O
			variation => N__('English: Symmetrical, Hedgehog, Flexible Formation, 11. e4 O-O'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4
			variation => N__('English: Symmetrical, Two Knights'),
			moves => {
				'a7a6' => 1,
				'c5d4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/1p1ppppp/p4n2/2p5/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 a6
			variation => N__('English: Symmetrical, Two Knights, 3... a6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/2Pp4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4
			variation => N__('English: Symmetrical, Two Knights'),
			moves => {
				'f3d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/2PN4/8/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4
			variation => N__('English: Symmetrical, Two Knights'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/8/2PN4/8/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 b6
			variation => N__('English: Symmetrical, Two Knights, 4... b6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 b6 5. Nc3
			variation => N__('English: Symmetrical, Two Knights, 4... b6'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/8/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 b6 5. Nc3 Bb7
			variation => N__('English: Symmetrical, Two Knights, 4... b6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/8/2PN4/8/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 Nc6
			variation => N__('English: Symmetrical, Two Knights, 4... Nc6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3
			variation => N__('English: Symmetrical, Two Knights, 4... Nc6'),
			moves => {
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/2n2np1/8/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 Nc6 5. Nc3 g6
			variation => N__('English: Symmetrical, Two Knights, 4... Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/4p3/2PN4/8/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e5
			variation => N__('English: Symmetrical, Two Knights, 4... e5'),
			moves => {
				'd4b5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/1N2p3/2P5/8/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e5 5. Nb5
			variation => N__('English: Symmetrical, Two Knights, 4... e5 5. Nb5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/8/2PN4/8/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 g6
			variation => N__('English: Symmetrical, Two Knights, 4... g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 g6 5. Nc3
			variation => N__('English: Symmetrical, Two Knights, 4... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp1pppbp/5np1/8/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7
			variation => N__('English: Symmetrical, Two Knights, 4... g6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqk2r/pp1pppbp/5np1/8/2PNP3/2N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. e4
			variation => N__('English: Symmetrical, Two Knights, 4... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 g6
			variation => N__('English: Symmetrical, Two Knights, 3... g6'),
			moves => {
				'b1c3' => 1,
				'd4d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 g6 4. Nc3
			variation => N__('English: Symmetrical, Two Knights, 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2pP4/2P5/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A31',
			# TRANSLATORS: A31: 1. c4 c5 2. Nf3 Nf6 3. d4 g6 4. d5
			variation => N__('English: Symmetrical, Two Knights, 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/8/2PN4/8/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6
			variation => N__('English: Symmetrical, Two Knights, 4... e6'),
			moves => {
				'b1c3' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/8/2PN4/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3'),
			moves => {
				'a7a6' => 1,
				'b8c6' => 1,
				'd7d5' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqkb1r/1p1p1ppp/p3pn2/8/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 a6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 a6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/1p1p1ppp/p3pn2/8/2PN4/2N3P1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 a6 6. g3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 a6 6. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/3p4/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 d5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 d5'),
			moves => {
			},
		},
		'rnbqk2r/pp1p1ppp/4pn2/8/1bPN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4'),
			moves => {
				'c1d2' => 1,
				'd1b3' => 1,
				'd4b5' => 1,
			},
		},
		'rnbqk2r/pp1p1ppp/4pn2/8/1bPN4/2N5/PP1BPPPP/R2QKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4 6. Bd2
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4 6. Bd2'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1bPN4/2N5/PP1BPPPP/R2QKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4 6. Bd2 Nc6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4 6. Bd2 Nc6'),
			moves => {
			},
		},
		'rnbqk2r/pp1p1ppp/4pn2/8/1bPN4/1QN5/PP2PPPP/R1B1KB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4 6. Qb3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4 6. Qb3'),
			moves => {
			},
		},
		'rnbqk2r/pp1p1ppp/4pn2/1N6/1bP5/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Bb4 6. Nb5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Bb4 6. Nb5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/8/2PN4/4P3/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. e3
			variation => N__('English: Symmetrical, Two Knights, 5. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/8/2PN4/6P1/PP2PP1P/RNBQKB1R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3
			variation => N__('English: Symmetrical, Two Knights, 5. g3'),
			moves => {
				'a7a6' => 1,
				'b8c6' => 1,
				'd7d5' => 1,
				'd8b6' => 1,
				'd8c7' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqkb1r/1p1p1ppp/p3pn2/8/2PN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 a6
			variation => N__('English: Symmetrical, Two Knights, 5. g3 a6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/1p1p1ppp/p3pn2/8/2PN4/6P1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 a6 6. Bg2
			variation => N__('English: Symmetrical, Two Knights, 5. g3 a6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Nc6
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Nc6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/6P1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Nc6 6. Bg2
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/3p4/2PN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 d5
			variation => N__('English: Symmetrical, Two Knights, 5. g3 d5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/3p4/2PN4/6P1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 d5 6. Bg2
			variation => N__('English: Symmetrical, Two Knights, 5. g3 d5 6. Bg2'),
			moves => {
				'e6e5' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/5n2/3pp3/2PN4/6P1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 d5 6. Bg2 e5
			variation => N__('English: Symmetrical, Two Knights, 5. g3 d5 6. Bg2'),
			moves => {
				'd4f3' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/5n2/3pp3/2P5/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 d5 6. Bg2 e5 7. Nf3
			variation => N__('English: Symmetrical, Two Knights, 5. g3 d5 6. Bg2 e5 7. Nf3'),
			moves => {
			},
		},
		'rnb1kb1r/pp1p1ppp/1q2pn2/8/2PN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Qb6
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Qb6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnb1kb1r/pp1p1ppp/1q2pn2/8/2PN4/6P1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Qb6 6. Bg2
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Qb6'),
			moves => {
			},
		},
		'rnb1kb1r/ppqp1ppp/4pn2/8/2PN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Qc7
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Qc7'),
			moves => {
			},
		},
		'rnbqk2r/pp1p1ppp/4pn2/8/1bPN4/6P1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A32',
			# TRANSLATORS: A32: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. g3 Bb4+
			variation => N__('English: Symmetrical, Two Knights, 5. g3 Bb4+'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A33a',
			# TRANSLATORS: A33a: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6'),
			moves => {
				'a2a3' => 1,
				'c1f4' => 1,
				'c1g5' => 1,
				'd4b5' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN1B2/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A33b',
			# TRANSLATORS: A33b: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Bf4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. Bf4'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/6B1/2PN4/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A33c',
			# TRANSLATORS: A33c: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Bg5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. Bg5'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/2N1P3/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A33d',
			# TRANSLATORS: A33d: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. e3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. e3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/P1N5/1P2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A33e',
			# TRANSLATORS: A33e: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. a3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. a3'),
			moves => {
				'f8c5' => 1,
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2b5/2PN4/P1N5/1P2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A33f',
			# TRANSLATORS: A33f: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. a3 Bc5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. a3 Bc5'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/1N6/2P5/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A33g',
			# TRANSLATORS: A33g: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5'),
			moves => {
				'd7d5' => 1,
				'f8b4' => 1,
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/1N6/1bP5/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A33h',
			# TRANSLATORS: A33h: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 Bb4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 Bb4'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/1N1p4/2P5/2N5/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'c1f4' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/1N1p4/2P2B2/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'e6e5' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n2n2/1N1pp3/2P2B2/2N5/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'c4d5' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n2n2/1N1Pp3/5B2/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'e5f4' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n2n2/1N1P4/5p2/2N5/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'd5c6' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2P2n2/1N6/5p2/2N5/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4 9. dxc6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'b7c6' => 1,
			},
		},
		'r1bqkb1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4 9. dxc6 bxc6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'r1bQkb1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R3KB1R b KQkq' => {
			eco => 'A33i',
			# TRANSLATORS: A33i: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4 9. dxc6 bxc6 10. Qxd8+
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5'),
			moves => {
				'e8d8' => 1,
			},
		},
		'r1bk1b1r/p4ppp/2p2n2/1N6/5p2/2N5/PP2PPPP/R3KB1R w KQ' => {
			eco => 'A33j',
			# TRANSLATORS: A33j: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. Ndb5 d5 7. Bf4 e5 8. cxd5 exf4 9. dxc6 bxc6 10. Qxd8+ Kxd8
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6.Ndb5 d5 Queenswap'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/8/2PN4/2N3P1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A33k',
			# TRANSLATORS: A33k: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3'),
			moves => {
				'a7a6' => 1,
				'd8b6' => 1,
				'f8b4' => 1,
				'f8c5' => 1,
			},
		},
		'r1bqkb1r/1p1p1ppp/p1n1pn2/8/2PN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A33l',
			# TRANSLATORS: A33l: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 a6
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3 a6'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/8/1bPN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A33m',
			# TRANSLATORS: A33m: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Bb4
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3 Bb4'),
			moves => {
			},
		},
		'r1bqk2r/pp1p1ppp/2n1pn2/2b5/2PN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A33n',
			# TRANSLATORS: A33n: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Bc5
			variation => N__('English: Symmetrical, Two Knights, 5. Nc3 Nc6 6. g3 Bc5'),
			moves => {
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/2PN4/2N3P1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A33o',
			# TRANSLATORS: A33o: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6
			variation => N__('English: Symmetrical, Geller Variation'),
			moves => {
				'd4b3' => 1,
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/2P5/1NN3P1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A33p',
			# TRANSLATORS: A33p: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6 7. Nb3
			variation => N__('English: Symmetrical, Geller, 7. Nb3'),
			moves => {
				'c6e5' => 1,
			},
		},
		'r1b1kb1r/pp1p1ppp/1q2pn2/4n3/2P5/1NN3P1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A33q',
			# TRANSLATORS: A33q: 1. c4 c5 2. Nf3 Nf6 3. d4 cxd4 4. Nxd4 e6 5. Nc3 Nc6 6. g3 Qb6 7. Nb3 Ne5
			variation => N__('English: Symmetrical, Geller, 7. Nb3 Ne5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P5/2N5/PP1PPPPP/R1BQKBNR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3
			variation => N__('English: Symmetrical'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 b6
			variation => N__('English: Symmetrical, 2... b6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 b6 3. Nf3
			variation => N__('English: Symmetrical, 2... b6'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pb1ppppp/1p6/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 b6 3. Nf3 Bb7
			variation => N__('English: Symmetrical, 2... b6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 g6
			variation => N__('English: Symmetrical, 2... g6'),
			moves => {
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 g6 3. Nf3
			variation => N__('English: Symmetrical, 2... g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 g6 3. g3
			variation => N__('English: Symmetrical, 2... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 g6 3. g3 Bg7
			variation => N__('English: Symmetrical, 2... g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 g6 3. g3 Bg7 4. Bg2
			variation => N__('English: Symmetrical, 2... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6
			variation => N__('English: Symmetrical, 2... Nf6'),
			moves => {
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3
			variation => N__('English: Symmetrical, 3 Knights'),
			moves => {
				'b7b6' => 1,
				'd7d5' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 b6
			variation => N__('English: Symmetrical, 3 Knights, 3... b6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5
			variation => N__('English: Symmetrical, 3 Knights'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pP4/8/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5
			variation => N__('English: Symmetrical, 3 Knights'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/8/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5
			variation => N__('English: Symmetrical, 3 Knights, Exchange'),
			moves => {
				'd2d4' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/3P4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. d4
			variation => N__('English: Symmetrical, 3 Knights, 5. d4'),
			moves => {
				'c5d4' => 1,
				'd5c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/3n4/3p4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. d4 cxd4
			variation => N__('English: Symmetrical, 3 Knights, 5. d4 cxd4'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/8/2p5/3P4/2n2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. d4 Nxc3
			variation => N__('English: Symmetrical, 3 Knights, 5. d4 Nxc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/4P3/2N2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4
			variation => N__('English: Symmetrical, 3 Knights, 5. e4'),
			moves => {
				'd5b4' => 1,
				'd5c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2p5/1n2P3/2N2N2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nb4
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nb4'),
			moves => {
				'f1b5' => 1,
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/1Bp5/1n2P3/2N2N2/PP1P1PPP/R1BQK2R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nb4 6. Bb5+
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nb4 6. Bb5+'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/8/2p5/1nB1P3/2N2N2/PP1P1PPP/R1BQK2R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nb4 6. Bc4
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nb4 6. Bc4'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/8/2p5/4P3/2n2N2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nxc3
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nxc3'),
			moves => {
				'd2c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2p5/4P3/2P2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. dxc3
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nxc3'),
			moves => {
				'd8d1' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/8/2p5/4P3/2P2N2/PP3PPP/R1BqKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. dxc3 Qxd1+
			variation => N__('English: Symmetrical, 3 Knights, 5. e4 Nxc3'),
			moves => {
				'e1d1' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/8/2p5/4P3/2P2N2/PP3PPP/R1BK1B1R b kq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. e4 Nxc3 6. dxc3 Qxd1+ 7. Kxd1
			variation => N__('English: Symmetrical, 3 Knights, Queenswap'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/8/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 d5 4. cxd5 Nxd5 5. g3
			variation => N__('English: Symmetrical, 3 Knights, 5. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 e6
			variation => N__('English: Symmetrical, 3 Knights, 3... e6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 e6 4. g3
			variation => N__('English: Symmetrical, 3 Knights, 3... e6 4. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. Nf3 g6
			variation => N__('English: Symmetrical, 3 Knights, 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3
			variation => N__('English: Symmetrical, 2... Nf6 3. g3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3 d5
			variation => N__('English: Symmetrical, 2... Nf6'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pP4/8/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3 d5 4. cxd5
			variation => N__('English: Symmetrical, 2... Nf6'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/8/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5
			variation => N__('English: Symmetrical, 2... Nf6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2pn4/8/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5 5. Bg2
			variation => N__('English: Symmetrical, 2... Nf6 3. g3'),
			moves => {
				'd5c7' => 1,
			},
		},
		'rnbqkb1r/ppn1pppp/8/2p5/8/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A34',
			# TRANSLATORS: A34: 1. c4 c5 2. Nc3 Nf6 3. g3 d5 4. cxd5 Nxd5 5. Bg2 Nc7
			variation => N__('English: Symmetrical, Rubinstein System'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P5/2N5/PP1PPPPP/R1BQKBNR w KQkq' => {
			eco => 'A35a',
			# TRANSLATORS: A35a: 1. c4 c5 2. Nc3 Nc6
			variation => N__('English: Symmetrical'),
			moves => {
				'e2e3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P5/2N1P3/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'A35b',
			# TRANSLATORS: A35b: 1. c4 c5 2. Nc3 Nc6 3. e3
			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. e3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R b KQkq' => {
			eco => 'A35c',
			# TRANSLATORS: A35c: 1. c4 c5 2. Nc3 Nc6 3. Nf3
			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. Nf3'),
			moves => {
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A35d',
			# TRANSLATORS: A35d: 1. c4 c5 2. Nc3 Nc6 3. Nf3 g6
			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. Nf3 g6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/2P5/2N1PN2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A35e',
			# TRANSLATORS: A35e: 1. c4 c5 2. Nc3 Nc6 3. Nf3 g6 4. e3
			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. Nf3 g6 4. e3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N1PN2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A35f',
			# TRANSLATORS: A35f: 1. c4 c5 2. Nc3 Nc6 3. Nf3 g6 4. e3 Bg7
			variation => N__('English: Symmetrical, 2. Nc3 Nc6 3. Nf3 g6 4. e3 Bg7'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/2P5/2N2N2/PP1PPPPP/R1BQKB1R w KQkq' => {
			eco => 'A35g',
			# TRANSLATORS: A35g: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6
			variation => N__('English: Symmetrical, Four Knights'),
			moves => {
				'd2d4' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A35g',
			# TRANSLATORS: A35g: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. d4
			variation => N__('English: Symmetrical, Four Knights, 4. d4'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/2P5/2N1PN2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'A35h',
			# TRANSLATORS: A35h: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3
			variation => N__('English: Symmetrical, Four Knights, 4. e3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/2p1p3/2P5/2N1PN2/PP1P1PPP/R1BQKB1R w KQkq' => {
			eco => 'A35i',
			# TRANSLATORS: A35i: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. e3 e5
			variation => N__('English: Symmetrical, Four Knights, 4. e3 e5'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A35j',
			# TRANSLATORS: A35j: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3
			variation => N__('English: Symmetrical, Four Knights, 4. g3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/2pp4/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A35k',
			# TRANSLATORS: A35k: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5
			variation => N__('English: Symmetrical, Four Knights, 4. g3 d5'),
			moves => {
				'c4d5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/2pP4/8/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A35l',
			# TRANSLATORS: A35l: 1. c4 c5 2. Nc3 Nc6 3. Nf3 Nf6 4. g3 d5 5. cxd5
			variation => N__('English: Symmetrical, Four Knights, 4. g3 d5 5.cxd5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR b KQkq' => {
			eco => 'A36a',
			# TRANSLATORS: A36a: 1. c4 c5 2. Nc3 Nc6 3. g3
			variation => N__('English: Symmetrical, 3. g3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A36b',
			# TRANSLATORS: A36b: 1. c4 c5 2. Nc3 Nc6 3. g3 Nf6
			variation => N__('English: Symmetrical, 3. g3 Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6
			variation => N__('English: Symmetrical, 3. g3 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/2P5/2N2NP1/PP1PPP1P/R1BQKB1R b KQkq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3
			variation => N__('English: Symmetrical, 3. g3 e6 4. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPP1P/R1BQKB1R w KQkq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/pp1pbppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp1pbppp/2n1pn2/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7 6. O-O
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'd7d5' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7 6. O-O d5
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'c4d5' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2pP4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A36c',
			# TRANSLATORS: A36c: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7 6. O-O d5 7. cxd5
			variation => N__('English: Symmetrical, Keres-Parma System'),
			moves => {
				'f6d5' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1p3/2pn4/8/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A36d',
			# TRANSLATORS: A36d: 1. c4 c5 2. Nc3 Nc6 3. g3 e6 4. Nf3 Nf6 5. Bg2 Be7 6. O-O d5 7. cxd5 Nxd5
			variation => N__('English: Symmetrical, Keres-Parma, Main Line Exchange'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/2P5/2N3P1/PP1PPP1P/R1BQKBNR w KQkq' => {
			eco => 'A36e',
			# TRANSLATORS: A36e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6
			variation => N__('English: Symmetrical, 3. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/2P5/2N3P1/PP1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A36e',
			# TRANSLATORS: A36e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2
			variation => N__('English: Symmetrical, 3. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N3P1/PP1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A36e',
			# TRANSLATORS: A36e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7
			variation => N__('English: Symmetrical, 3. g3 g6'),
			moves => {
				'a2a3' => 1,
				'b2b3' => 1,
				'd2d3' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2NP2P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A36f',
			# TRANSLATORS: A36f: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3
			variation => N__('English: Symmetrical, 5. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A36g',
			# TRANSLATORS: A36g: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3
			variation => N__('English: Symmetrical, 5. a3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A36h',
			# TRANSLATORS: A36h: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3 e6
			variation => N__('English: Symmetrical, 5. a3 e6'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/P1N3P1/1P1PPPBP/R1BQK1NR w KQkq' => {
			eco => 'A36i',
			# TRANSLATORS: A36i: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. a3 d6
			variation => N__('English: Symmetrical, 5. a3 d6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/1PN3P1/P2PPPBP/R1BQK1NR b KQkq' => {
			eco => 'A36j',
			# TRANSLATORS: A36j: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. b3
			variation => N__('English: Symmetrical, 5. b3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A36k',
			# TRANSLATORS: A36k: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3
			variation => N__('English: Symmetrical, 5. e3'),
			moves => {
				'e7e5' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A36l',
			# TRANSLATORS: A36l: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e5
			variation => N__('English: Symmetrical, 5. e3 e5 (Botvinnik Reversed)'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A36m',
			# TRANSLATORS: A36m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6
			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A36m',
			# TRANSLATORS: A36m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6 6. Nge2
			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1P1P1/PP1PNPBP/R1BQK2R w KQkq' => {
			eco => 'A36m',
			# TRANSLATORS: A36m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e3 e6 6. Nge2 Nge7
			variation => N__('English: Symmetrical, 5. e3 e6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR b KQkq' => {
			eco => 'A36n',
			# TRANSLATORS: A36n: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4
			variation => N__('English: Symmetrical, Botvinnik System'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A36o',
			# TRANSLATORS: A36o: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 Nf6
			variation => N__('English: Symmetrical, Botvinnik, 5... Nf6'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A36o',
			# TRANSLATORS: A36o: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 Nf6 6. Nge2
			variation => N__('English: Symmetrical, Botvinnik, 5... Nf6 6.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A36p',
			# TRANSLATORS: A36p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6
			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A36p',
			# TRANSLATORS: A36p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2
			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq' => {
			eco => 'A36p',
			# TRANSLATORS: A36p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7
			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq' => {
			eco => 'A36p',
			# TRANSLATORS: A36p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O
			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w -' => {
			eco => 'A36p',
			# TRANSLATORS: A36p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O O-O
			variation => N__('English: Symmetrical, Botvinnik, 5... e6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b -' => {
			eco => 'A36q',
			# TRANSLATORS: A36q: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O O-O 8. d3
			variation => N__('English: Symmetrical, Botvinnik, 5... e6, 8. d3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w -' => {
			eco => 'A36r',
			# TRANSLATORS: A36r: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 e6 6. Nge2 Nge7 7. O-O O-O 8. d3 d6
			variation => N__('English: Symmetrical, Botvinnik, 5... e6, 8. d3 d6'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1P1PBP/R1BQK1NR w KQkq' => {
			eco => 'A36s',
			# TRANSLATORS: A36s: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6
			variation => N__('English: Symmetrical, Botvinnik, 5... d6'),
			moves => {
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R b KQkq' => {
			eco => 'A36t',
			# TRANSLATORS: A36t: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2
			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQK2R w KQkq' => {
			eco => 'A36u',
			# TRANSLATORS: A36u: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6
			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2 Nf6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 b kq' => {
			eco => 'A36u',
			# TRANSLATORS: A36u: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O
			variation => N__('English: Symmetrical, Botvinnik, 5... d6 6.Nge2 Nf6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2N3P1/PP1PNPBP/R1BQ1RK1 w -' => {
			eco => 'A36v',
			# TRANSLATORS: A36v: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O
			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 7.O-O O-O'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 b -' => {
			eco => 'A36w',
			# TRANSLATORS: A36w: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3
			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3'),
			moves => {
				'a7a6' => 1,
				'a8b8' => 1,
				'f6e8' => 1,
			},
		},
		'1rbq1rk1/pp2ppbp/2np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w -' => {
			eco => 'A36x',
			# TRANSLATORS: A36x: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3 Rb8
			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3 Rb8'),
			moves => {
			},
		},
		'r1bqnrk1/pp2ppbp/2np2p1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w -' => {
			eco => 'A36y',
			# TRANSLATORS: A36y: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3 Ne8
			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3 Ne8'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P1P3/2NP2P1/PP2NPBP/R1BQ1RK1 w -' => {
			eco => 'A36z',
			# TRANSLATORS: A36z: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. e4 d6 6. Nge2 Nf6 7. O-O O-O 8. d3 a6
			variation => N__('English: Symmetrical, Botvinnik System, 5... d6, 8. d3 a6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A37a',
			# TRANSLATORS: A37a: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3
			variation => N__('English: Symmetrical, 5. Nf3'),
			moves => {
				'a7a6' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
				'g8h6' => 1,
			},
		},
		'r1bqk1nr/1p1pppbp/p1n3p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A37b',
			# TRANSLATORS: A37b: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 a6
			variation => N__('English: Symmetrical, 5. Nf3 a6'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A37c',
			# TRANSLATORS: A37c: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nh6
			variation => N__('English: Symmetrical, 5. Nf3 Nh6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n3pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A37d',
			# TRANSLATORS: A37d: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nh6 6. O-O
			variation => N__('English: Symmetrical, 5. Nf3 Nh6 6.O-O'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A37e',
			# TRANSLATORS: A37e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6
			variation => N__('English: Symmetrical, 5. Nf3 d6'),
			moves => {
				'd2d3' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A37f',
			# TRANSLATORS: A37f: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. d3
			variation => N__('English: Symmetrical, 5. Nf3 d6 6. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A37g',
			# TRANSLATORS: A37g: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O
			variation => N__('English: Symmetrical, 5. Nf3 d6 6.O-O'),
			moves => {
				'g8h6' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np2pn/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A37h',
			# TRANSLATORS: A37h: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. O-O Nh6
			variation => N__('English: Symmetrical, 5. Nf3 d6 6.O-O Nh6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A37i',
			# TRANSLATORS: A37i: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6
			variation => N__('English: Symmetrical, 5. Nf3 e6'),
			moves => {
				'd2d3' => 1,
				'e1g1' => 1,
				'e2e3' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A37j',
			# TRANSLATORS: A37j: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. d3
			variation => N__('English: Symmetrical, 5. Nf3 e6 6. d3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQK2R b KQkq' => {
			eco => 'A37k',
			# TRANSLATORS: A37k: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. e3
			variation => N__('English: Symmetrical, 5. Nf3 e6 6. e3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A37l',
			# TRANSLATORS: A37l: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A37l',
			# TRANSLATORS: A37l: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7'),
			moves => {
				'd2d3' => 1,
				'e2e3' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2N1PNP1/PP1P1PBP/R1BQ1RK1 b kq' => {
			eco => 'A37m',
			# TRANSLATORS: A37m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. e3
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. e3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A37n',
			# TRANSLATORS: A37n: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A37o',
			# TRANSLATORS: A37o: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3 O-O
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3 O-O'),
			moves => {
				'c1d2' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n1p1p1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b -' => {
			eco => 'A37p',
			# TRANSLATORS: A37p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e6 6. O-O Nge7 7. d3 O-O 8. Bd2
			variation => N__('English: Symmetrical, 5. Nf3 e6 6.O-O Nge7 7. d3 O-O 8. Bd2'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A37q',
			# TRANSLATORS: A37q: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5
			variation => N__('English: Symmetrical, 5. Nf3 e5'),
			moves => {
				'a2a3' => 1,
				'd2d3' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/P1N2NP1/1P1PPPBP/R1BQK2R b KQkq' => {
			eco => 'A37r',
			# TRANSLATORS: A37r: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. a3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6. a3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A37s',
			# TRANSLATORS: A37s: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. d3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6. d3'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQK2R w KQkq' => {
			eco => 'A37t',
			# TRANSLATORS: A37t: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. d3 Nge7
			variation => N__('English: Symmetrical, 5. Nf3 e5 6. d3 Nge7'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A37u',
			# TRANSLATORS: A37u: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O'),
			moves => {
				'd7d6' => 1,
				'g8e7' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A37u',
			# TRANSLATORS: A37u: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A37v',
			# TRANSLATORS: A37v: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w kq' => {
			eco => 'A37v',
			# TRANSLATORS: A37v: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3 Nge7
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3 Nge7'),
			moves => {
				'a2a3' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A37w',
			# TRANSLATORS: A37w: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O d6 7. d3 Nge7 8. a3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O d6 7. d3 Nge7 8. a3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A37x',
			# TRANSLATORS: A37x: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqk2r/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A37y',
			# TRANSLATORS: A37y: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A37y',
			# TRANSLATORS: A37y: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3 O-O
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3 O-O'),
			moves => {
				'a2a3' => 1,
			},
		},
		'r1bq1rk1/pp1pnpbp/2n3p1/2p1p3/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b -' => {
			eco => 'A37y',
			# TRANSLATORS: A37y: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 e5 6. O-O Nge7 7. d3 O-O 8. a3
			variation => N__('English: Symmetrical, 5. Nf3 e5 6.O-O Nge7 7. d3 O-O 8. a3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQK2R w KQkq' => {
			eco => 'A38a',
			# TRANSLATORS: A38a: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6
			variation => N__('English: Symmetrical, Main Line'),
			moves => {
				'd2d4' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A38b',
			# TRANSLATORS: A38b: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. d4
			variation => N__('English: Symmetrical, Main Line, 6. d4'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 b kq' => {
			eco => 'A38c',
			# TRANSLATORS: A38c: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O
			variation => N__('English: Symmetrical, Main Line, 6.O-O'),
			moves => {
				'd7d5' => 1,
				'd7d6' => 1,
				'e8g8' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2n2np1/2pp4/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A38d',
			# TRANSLATORS: A38d: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O d5
			variation => N__('English: Symmetrical, Main Line, 6.O-O d5'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w kq' => {
			eco => 'A38e',
			# TRANSLATORS: A38e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O d6
			variation => N__('English: Symmetrical, Main Line, 6.O-O d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A38f',
			# TRANSLATORS: A38f: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O
			variation => N__('English: Symmetrical, Main Line, 6.O-O O-O'),
			moves => {
				'a1b1' => 1,
				'a2a3' => 1,
				'b2b3' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2N2NP1/PP1PPPBP/1RBQ1RK1 b -' => {
			eco => 'A38g',
			# TRANSLATORS: A38g: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. Rb1
			variation => N__('English: Symmetrical, Main Line, 7. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/1PN2NP1/P2PPPBP/R1BQ1RK1 b -' => {
			eco => 'A38h',
			# TRANSLATORS: A38h: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. b3
			variation => N__('English: Symmetrical, Main Line, 7. b3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 b -' => {
			eco => 'A38i',
			# TRANSLATORS: A38i: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. a3
			variation => N__('English: Symmetrical, Main Line, 7. a3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1N2NP1/1P1PPPBP/R1BQ1RK1 w -' => {
			eco => 'A38j',
			# TRANSLATORS: A38j: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. a3 d6
			variation => N__('English: Symmetrical, Main Line, 7. a3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A38k',
			# TRANSLATORS: A38k: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3
			variation => N__('English: Symmetrical, Main Line, 7. d3'),
			moves => {
				'a7a6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
			},
		},
		'r1bq1rk1/1p1pppbp/p1n2np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A38l',
			# TRANSLATORS: A38l: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 a6
			variation => N__('English: Symmetrical, Main Line, 7. d3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/2pp4/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A38m',
			# TRANSLATORS: A38m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d5
			variation => N__('English: Symmetrical, Main Line, 7. d3 d5'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A38n',
			# TRANSLATORS: A38n: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6
			variation => N__('English: Symmetrical, Main Line, 7. d3 d6'),
			moves => {
				'a1b1' => 1,
				'a2a3' => 1,
				'c1d2' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP2PPBP/1RBQ1RK1 b -' => {
			eco => 'A38o',
			# TRANSLATORS: A38o: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. Rb1
			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/2NP1NP1/PP1BPPBP/R2Q1RK1 b -' => {
			eco => 'A38p',
			# TRANSLATORS: A38p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. Bd2
			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. Bd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 b -' => {
			eco => 'A38q',
			# TRANSLATORS: A38q: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. a3
			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. a3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bq1rk1/1p2ppbp/p1np1np1/2p5/2P5/P1NP1NP1/1P2PPBP/R1BQ1RK1 w -' => {
			eco => 'A38r',
			# TRANSLATORS: A38r: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d3 d6 8. a3 a6
			variation => N__('English: Symmetrical, Main Line, 7. d3 d6 8. a3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/2p5/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A39a',
			# TRANSLATORS: A39a: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4
			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2Pp4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39b',
			# TRANSLATORS: A39b: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4
			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A39c',
			# TRANSLATORS: A39c: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4
			variation => N__('English: Symmetrical, Main Line 7. d4'),
			moves => {
				'a7a6' => 1,
				'c6d4' => 1,
				'd7d6' => 1,
				'd8a5' => 1,
				'd8b6' => 1,
				'f6g4' => 1,
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39d',
			# TRANSLATORS: A39d: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qa5
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qa5'),
			moves => {
			},
		},
		'r1b2rk1/pp1pppbp/1qn2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39e',
			# TRANSLATORS: A39e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qb6
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qb6'),
			moves => {
				'd4c2' => 1,
			},
		},
		'r1b2rk1/pp1pppbp/1qn2np1/8/2P5/2N3P1/PPN1PPBP/R1BQ1RK1 b -' => {
			eco => 'A39e',
			# TRANSLATORS: A39e: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Qb6 9. Nc2
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Qb6'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/p1n2np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39f',
			# TRANSLATORS: A39f: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 a6
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/2PN4/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39g',
			# TRANSLATORS: A39g: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 d6
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39h',
			# TRANSLATORS: A39h: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Ng4
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 b -' => {
			eco => 'A39h',
			# TRANSLATORS: A39h: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Ng4 9. e3
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np2p1/8/2PN2n1/2N1P1P1/PP3PBP/R1BQ1RK1 w -' => {
			eco => 'A39i',
			# TRANSLATORS: A39i: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Ng4 9. e3 d6
			variation => N__('English: Symmetrical, Main Line 7. d4, 8... Ng4 9. e3 d6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/5np1/8/2Pn4/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A39j',
			# TRANSLATORS: A39j: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4
			variation => N__('English: Symmetrical, Main Line 7. d4, 8...Nxd4'),
			moves => {
				'd1d4' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/5np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 b -' => {
			eco => 'A39j',
			# TRANSLATORS: A39j: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4
			variation => N__('English: Symmetrical, Main Line 7. d4, 9.Qxd4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2PQ4/2N3P1/PP2PPBP/R1B2RK1 w -' => {
			eco => 'A39k',
			# TRANSLATORS: A39k: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6
			variation => N__('English: Symmetrical, Main Line 7. d4, 9... d6'),
			moves => {
				'd4d2' => 1,
				'd4d3' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2N3P1/PP1QPPBP/R1B2RK1 b -' => {
			eco => 'A39l',
			# TRANSLATORS: A39l: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd2
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 b -' => {
			eco => 'A39m',
			# TRANSLATORS: A39m: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3'),
			moves => {
				'a7a6' => 1,
				'a8b8' => 1,
				'c8f5' => 1,
			},
		},
		'r2q1rk1/pp2ppbp/3p1np1/5b2/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w -' => {
			eco => 'A39n',
			# TRANSLATORS: A39n: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 Bf5
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 Bf5'),
			moves => {
			},
		},
		'1rbq1rk1/pp2ppbp/3p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w -' => {
			eco => 'A39o',
			# TRANSLATORS: A39o: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 Rb8
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 Rb8'),
			moves => {
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP2PPBP/R1B2RK1 w -' => {
			eco => 'A39p',
			# TRANSLATORS: A39p: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6'),
			moves => {
				'c1d2' => 1,
			},
		},
		'r1bq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 b -' => {
			eco => 'A39q',
			# TRANSLATORS: A39q: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2'),
			moves => {
				'a8b8' => 1,
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/R4RK1 w -' => {
			eco => 'A39q',
			# TRANSLATORS: A39q: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2 Rb8
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2 Rb8'),
			moves => {
				'a1c1' => 1,
			},
		},
		'1rbq1rk1/1p2ppbp/p2p1np1/8/2P5/2NQ2P1/PP1BPPBP/2R2RK1 b -' => {
			eco => 'A39r',
			# TRANSLATORS: A39r: 1. c4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nf3 Nf6 6. O-O O-O 7. d4 cxd4 8. Nxd4 Nxd4 9. Qxd4 d6 10. Qd3 a6 11. Bd2 Rb8 12. Rac1
			variation => N__('English: Symmetrical, Main Line 7. d4, 10. Qd3 a6 11. Bd2 Rb8 12.Rac1'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/3P4/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40a',
			# TRANSLATORS: A40a: 1. d4
			variation => N__('Queen\'s Pawn Game'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'f7f5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40b',
			# TRANSLATORS: A40b: 1. d4 c6
			variation => N__('Queen\'s Pawn: 1... c6'),
			moves => {
				'c2c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40b',
			# TRANSLATORS: A40b: 1. d4 c6 2. c4
			variation => N__('Queen\'s Pawn: 1... c6 2. c4'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/2p5/1p6/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40b',
			# TRANSLATORS: A40b: 1. d4 c6 2. c4 b5
			variation => N__('Queen\'s Pawn: Jadoul'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40b',
			# TRANSLATORS: A40b: 1. d4 c6 2. Nf3
			variation => N__('Queen\'s Pawn: 1... c6 2. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/8/1p6/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40c',
			# TRANSLATORS: A40c: 1. d4 b5
			variation => N__('Queen\'s Pawn: Polish Defence'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/8/1p6/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A40c',
			# TRANSLATORS: A40c: 1. d4 b5 2. e4
			variation => N__('Queen\'s Pawn: Polish Defence'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/8/1p6/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A40c',
			# TRANSLATORS: A40c: 1. d4 b5 2. e4 Bb7
			variation => N__('Queen\'s Pawn: Polish Defence'),
			moves => {
				'f1b5' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/8/1B6/3PP3/8/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'A40c',
			# TRANSLATORS: A40c: 1. d4 b5 2. e4 Bb7 3. Bxb5
			variation => N__('Queen\'s Pawn: Polish Defence, Spassky Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6
			variation => N__('Queen\'s Pawn: English Defence'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6 2. c4
			variation => N__('Queen\'s Pawn: English Defence, 2. c4'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6 2. c4 Bb7
			variation => N__('Queen\'s Pawn: English Defence, 2. c4 Bb7'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6 2. c4 Bb7 3. Nc3
			variation => N__('Queen\'s Pawn: English Defence, 2. c4 Bb7'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6 2. c4 Bb7 3. Nc3 e6
			variation => N__('Queen\'s Pawn: English Defence, 2. c4 Bb7'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/2PPP3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A40d',
			# TRANSLATORS: A40d: 1. d4 b6 2. c4 Bb7 3. Nc3 e6 4. e4
			variation => N__('Queen\'s Pawn: English Defence, 2. c4 Bb7'),
			moves => {
				'f7f5' => 1,
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40e',
			# TRANSLATORS: A40e: 1. d4 b6 2. c4 e6
			variation => N__('Queen\'s Pawn: English Defence, 2. c4 e6'),
			moves => {
				'a2a3' => 1,
				'b1c3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2PP4/P7/1P2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40f',
			# TRANSLATORS: A40f: 1. d4 b6 2. c4 e6 3. a3
			variation => N__('Queen\'s Pawn: English Defence, 3. a3'),
			moves => {
			},
		},
		'rn1qkbnr/pbpp2pp/1p2p3/5p2/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A40g',
			# TRANSLATORS: A40g: 1. d4 b6 2. c4 Bb7 3. Nc3 e6 4. e4 f5
			variation => N__('Queen\'s Pawn: English Defence, Hartlaub Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2PPP3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A40g',
			# TRANSLATORS: A40g: 1. d4 b6 2. c4 e6 3. e4
			variation => N__('Queen\'s Pawn: English Defence, 3. e4'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/2PPP3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A40g',
			# TRANSLATORS: A40g: 1. d4 b6 2. c4 e6 3. e4 Bb7
			variation => N__('Queen\'s Pawn: English Defence, 3. e4'),
			moves => {
				'f2f3' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/2PPP3/5P2/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'A40g',
			# TRANSLATORS: A40g: 1. d4 b6 2. c4 e6 3. e4 Bb7 4. f3
			variation => N__('Queen\'s Pawn: English Defence, 3. e4'),
			moves => {
				'f7f5' => 1,
			},
		},
		'rn1qkbnr/pbpp2pp/1p2p3/5p2/2PPP3/5P2/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'A40g',
			# TRANSLATORS: A40g: 1. d4 b6 2. c4 e6 3. e4 Bb7 4. f3 f5
			variation => N__('Queen\'s Pawn: English Defence, Polli Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40h',
			# TRANSLATORS: A40h: 1. d4 b6 2. c4 e6 3. Nc3
			variation => N__('Queen\'s Pawn: English Defence, 3. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/8/4p3/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5
			variation => N__('Englund Gambit'),
			moves => {
				'd4e5' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/8/4P3/8/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5
			variation => N__('Englund Gambit Accepted'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'f7f6' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4P3/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 Nc6
			variation => N__('Englund Gambit: 2.dxe5 Nc6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4P3/8/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 Nc6 3. Nf3
			variation => N__('Englund Gambit: 2.dxe5 Nc6 3. Nf3'),
			moves => {
				'd8e7' => 1,
				'f7f6' => 1,
				'g8e7' => 1,
			},
		},
		'r1b1kbnr/ppppqppp/2n5/4P3/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 Nc6 3. Nf3 Qe7
			variation => N__('Englund Gambit: 2.dxe5 Nc6 3. Nf3 Qe7'),
			moves => {
			},
		},
		'r1bqkbnr/pppp2pp/2n2p2/4P3/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 Nc6 3. Nf3 f6
			variation => N__('Englund Gambit: Soller Deferred'),
			moves => {
			},
		},
		'r1bqkb1r/ppppnppp/2n5/4P3/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 Nc6 3. Nf3 Nge7
			variation => N__('Englund Gambit: Zilbermints'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4P3/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 d6
			variation => N__('Englund Gambit: Hartlaub'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/5p2/4P3/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40i',
			# TRANSLATORS: A40i: 1. d4 e5 2. dxe5 f6
			variation => N__('Englund Gambit: Soller'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40j',
			# TRANSLATORS: A40j: 1. d4 Nc6
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles Defence'),
			moves => {
				'c1g5' => 1,
				'c2c4' => 1,
				'd4d5' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pppppppp/2n5/6B1/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A40j',
			# TRANSLATORS: A40j: 1. d4 Nc6 2. Bg5
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. Bg5'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/3P4/8/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40j',
			# TRANSLATORS: A40j: 1. d4 Nc6 2. d5
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. d5'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. c4'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4 e5
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. c4'),
			moves => {
				'd4e5' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4P3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4 e5 3. dxe5
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. c4'),
			moves => {
				'c6e5' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/8/4n3/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4 e5 3. dxe5 Nxe5
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. c4'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/8/4n3/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4 e5 3. dxe5 Nxe5 4. Nc3
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. c4'),
			moves => {
				'e5c4' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/8/8/2n5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A40k',
			# TRANSLATORS: A40k: 1. d4 Nc6 2. c4 e5 3. dxe5 Nxe5 4. Nc3 Nxc4
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, Pozarek Gambit'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40l',
			# TRANSLATORS: A40l: 1. d4 Nc6 2. Nf3
			variation => N__('Queen\'s Pawn: Bogoljubow-Miles, 2. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40m',
			# TRANSLATORS: A40m: 1. d4 e6
			variation => N__('Queen\'s Pawn: 1... e6'),
			moves => {
				'c2c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40n',
			# TRANSLATORS: A40n: 1. d4 e6 2. Nf3
			variation => N__('Queen\'s Pawn: 1... e6 2. Nf3'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A40n',
			# TRANSLATORS: A40n: 1. d4 e6 2. Nf3 c5
			variation => N__('Queen\'s Pawn: 1... e6 2. Nf3 c5'),
			moves => {
			},
		},
		'rnbqkbnr/pppp1ppp/4p3/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40o',
			# TRANSLATORS: A40o: 1. d4 e6 2. c4
			variation => N__('Queen\'s Pawn: 1... e6 2. c4'),
			moves => {
				'f8b4' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/4p3/8/1bPP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40p',
			# TRANSLATORS: A40p: 1. d4 e6 2. c4 Bb4+
			variation => N__('Queen\'s Pawn: Keres Defence'),
			moves => {
				'c1d2' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/4p3/8/1bPP4/8/PP1BPPPP/RN1QKBNR b KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2'),
			moves => {
				'a7a5' => 1,
				'b4d2' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/4p3/8/2PP4/8/PP1bPPPP/RN1QKBNR w KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'd1d2' => 1,
			},
		},
		'rnbqk1nr/pppp1ppp/4p3/8/2PP4/8/PP1QPPPP/RN2KBNR b KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'b7b6' => 1,
			},
		},
		'rnbqk1nr/p1pp1ppp/1p2p3/8/2PP4/8/PP1QPPPP/RN2KBNR w KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2 b6
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqk1nr/p1pp1ppp/1p2p3/8/2PP4/2N5/PP1QPPPP/R3KBNR b KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2 b6 5. Nc3
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qk1nr/pbpp1ppp/1p2p3/8/2PP4/2N5/PP1QPPPP/R3KBNR w KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2 b6 5. Nc3 Bb7
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rn1qk1nr/pbpp1ppp/1p2p3/8/2PPP3/2N5/PP1Q1PPP/R3KBNR b KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2 b6 5. Nc3 Bb7 6. e4
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 Bxd2+'),
			moves => {
				'g8h6' => 1,
			},
		},
		'rn1qk2r/pbpp1ppp/1p2p2n/8/2PPP3/2N5/PP1Q1PPP/R3KBNR w KQkq' => {
			eco => 'A40q',
			# TRANSLATORS: A40q: 1. d4 e6 2. c4 Bb4+ 3. Bd2 Bxd2+ 4. Qxd2 b6 5. Nc3 Bb7 6. e4 Nh6
			variation => N__('Queen\'s Pawn: Keres Defence, Kangaroo Variation'),
			moves => {
			},
		},
		'rnbqk1nr/1ppp1ppp/4p3/p7/1bPP4/8/PP1BPPPP/RN1QKBNR w KQkq' => {
			eco => 'A40r',
			# TRANSLATORS: A40r: 1. d4 e6 2. c4 Bb4+ 3. Bd2 a5
			variation => N__('Queen\'s Pawn: Keres Defence, 3. Bd2 a5'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40s',
			# TRANSLATORS: A40s: 1. d4 g6
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'c2c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40t',
			# TRANSLATORS: A40t: 1. d4 g6 2. Nf3
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A40u',
			# TRANSLATORS: A40u: 1. d4 g6 2. c4
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A40u',
			# TRANSLATORS: A40u: 1. d4 g6 2. c4 Bg7
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2PPP3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A40v',
			# TRANSLATORS: A40v: 1. d4 g6 2. c4 Bg7 3. e4
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A40w',
			# TRANSLATORS: A40w: 1. d4 g6 2. c4 Bg7 3. Nf3
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A40w',
			# TRANSLATORS: A40w: 1. d4 g6 2. c4 Bg7 3. Nf3 c5
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40x',
			# TRANSLATORS: A40x: 1. d4 g6 2. c4 Bg7 3. Nc3
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A40x',
			# TRANSLATORS: A40x: 1. d4 g6 2. c4 Bg7 3. Nc3 c5
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'd4d5' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2pP4/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40x',
			# TRANSLATORS: A40x: 1. d4 g6 2. c4 Bg7 3. Nc3 c5 4. d5
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'g7c3' => 1,
			},
		},
		'rnbqk1nr/pp1ppp1p/6p1/2pP4/2P5/2b5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A40x',
			# TRANSLATORS: A40x: 1. d4 g6 2. c4 Bg7 3. Nc3 c5 4. d5 Bxc3+
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'b2c3' => 1,
			},
		},
		'rnbqk1nr/pp1ppp1p/6p1/2pP4/2P5/2P5/P3PPPP/R1BQKBNR b KQkq' => {
			eco => 'A40x',
			# TRANSLATORS: A40x: 1. d4 g6 2. c4 Bg7 3. Nc3 c5 4. d5 Bxc3+ 5. bxc3
			variation => N__('Queen\'s Pawn: Modern'),
			moves => {
				'f7f5' => 1,
			},
		},
		'rnbqk1nr/pp1pp2p/6p1/2pP1p2/2P5/2P5/P3PPPP/R1BQKBNR w KQkq' => {
			eco => 'A40y',
			# TRANSLATORS: A40y: 1. d4 g6 2. c4 Bg7 3. Nc3 c5 4. d5 Bxc3+ 5. bxc3 f5
			variation => N__('Queen\'s Pawn: Modern, Beefeater Defence'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A41a',
			# TRANSLATORS: A41a: 1. d4 d6
			variation => N__('Neo-Old Indian'),
			moves => {
				'c1g5' => 1,
				'c2c4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/6B1/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A41b',
			# TRANSLATORS: A41b: 1. d4 d6 2. Bg5
			variation => N__('Neo-Old Indian: 2. Bg5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/3P4/6P1/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A41c',
			# TRANSLATORS: A41c: 1. d4 d6 2. g3
			variation => N__('Neo-Old Indian: 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A41d',
			# TRANSLATORS: A41d: 1. d4 d6 2. Nf3
			variation => N__('Neo-Old Indian: 2. Nf3'),
			moves => {
				'c8g4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6
			variation => N__('Neo-Old Indian / Modern'),
			moves => {
				'c1f4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6 3. Bf4
			variation => N__('Neo-Old Indian / Modern: 3. Bf4'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6 3. Bf4 Bg7
			variation => N__('Neo-Old Indian / Modern: 3. Bf4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6 3. g3
			variation => N__('Neo-Old Indian / Modern: 3. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6 3. g3 Bg7
			variation => N__('Neo-Old Indian / Modern: 3. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A41e',
			# TRANSLATORS: A41e: 1. d4 d6 2. Nf3 g6 3. g3 Bg7 4. Bg2
			variation => N__('Neo-Old Indian / Modern: 3. g3'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/3P2b1/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41f',
			# TRANSLATORS: A41f: 1. d4 d6 2. Nf3 Bg4
			variation => N__('Neo-Old Indian: Wade Defence'),
			moves => {
				'c2c4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/3P2b1/4PN2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A41f',
			# TRANSLATORS: A41f: 1. d4 d6 2. Nf3 Bg4 3. e3
			variation => N__('Neo-Old Indian: Wade Defence, 3. e3'),
			moves => {
				'b8d7' => 1,
				'g8f6' => 1,
			},
		},
		'r2qkbnr/pppnpppp/3p4/8/3P2b1/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A41f',
			# TRANSLATORS: A41f: 1. d4 d6 2. Nf3 Bg4 3. e3 Nd7
			variation => N__('Neo-Old Indian: Wade Defence, 3. e3 Nd7'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/8/3P2b1/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A41f',
			# TRANSLATORS: A41f: 1. d4 d6 2. Nf3 Bg4 3. e3 Nf6
			variation => N__('Neo-Old Indian: Wade Defence, 3. e3 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/2PP2b1/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4'),
			moves => {
				'b8d7' => 1,
				'e7e5' => 1,
				'g4f3' => 1,
			},
		},
		'rn1qkbnr/ppp2ppp/3p4/4p3/2PP2b1/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4 e5
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 e5'),
			moves => {
				'b1c3' => 1,
				'd4e5' => 1,
			},
		},
		'rn1qkbnr/ppp2ppp/3p4/4p3/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4 e5 4. Nc3
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 e5 4. Nc3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2qkbnr/ppp2ppp/2np4/4p3/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4 e5 4. Nc3 Nc6
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 e5 4. Nc3 Nc6'),
			moves => {
			},
		},
		'rn1qkbnr/ppp2ppp/3p4/4P3/2P3b1/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4 e5 4. dxe5
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 e5 4.dxe5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2qkbnr/ppp2ppp/2np4/4P3/2P3b1/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41g',
			# TRANSLATORS: A41g: 1. d4 d6 2. Nf3 Bg4 3. c4 e5 4. dxe5 Nc6
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 e5 4.dxe5 Nc6 Gambit'),
			moves => {
			},
		},
		'r2qkbnr/pppnpppp/3p4/8/2PP2b1/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41h',
			# TRANSLATORS: A41h: 1. d4 d6 2. Nf3 Bg4 3. c4 Nd7
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 Nd7'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r2qkbnr/pppnpppp/3p4/8/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A41h',
			# TRANSLATORS: A41h: 1. d4 d6 2. Nf3 Bg4 3. c4 Nd7 4. Nc3
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 Nd7 4. Nc3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r2qkbnr/pppn1ppp/3p4/4p3/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A41h',
			# TRANSLATORS: A41h: 1. d4 d6 2. Nf3 Bg4 3. c4 Nd7 4. Nc3 e5
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 Nd7 4. Nc3 e5'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/2PP4/5b2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41i',
			# TRANSLATORS: A41i: 1. d4 d6 2. Nf3 Bg4 3. c4 Bxf3
			variation => N__('Neo-Old Indian: Wade Defence, 3. c4 Bxf3'),
			moves => {
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/3PP1b1/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A41j',
			# TRANSLATORS: A41j: 1. d4 d6 2. Nf3 Bg4 3. e4
			variation => N__('Neo-Old Indian: Wade Defence, 3. e4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/8/3PP1b1/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A41k',
			# TRANSLATORS: A41k: 1. d4 d6 2. Nf3 Bg4 3. e4 Nf6
			variation => N__('Neo-Old Indian: Wade Defence, 3. e4 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A41l',
			# TRANSLATORS: A41l: 1. d4 d6 2. c4
			variation => N__('Neo-Old Indian: 2. c4'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A41m',
			# TRANSLATORS: A41m: 1. d4 d6 2. c4 e5
			variation => N__('Neo-Old Indian: 2. c4 e5'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/3Pp3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A41n',
			# TRANSLATORS: A41n: 1. d4 d6 2. c4 e5 3. d5
			variation => N__('Neo-Old Indian: 2. c4 e5 3. d5'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4P3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A41o',
			# TRANSLATORS: A41o: 1. d4 d6 2. c4 e5 3. dxe5
			variation => N__('Neo-Old Indian: 2. c4 e5 3.dxe5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/4p3/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A41o',
			# TRANSLATORS: A41o: 1. d4 d6 2. c4 e5 3. dxe5 dxe5
			variation => N__('Neo-Old Indian: 2. c4 e5 3.dxe5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'rnbQkbnr/ppp2ppp/8/4p3/2P5/8/PP2PPPP/RNB1KBNR b KQkq' => {
			eco => 'A41o',
			# TRANSLATORS: A41o: 1. d4 d6 2. c4 e5 3. dxe5 dxe5 4. Qxd8+
			variation => N__('Neo-Old Indian: 2. c4 e5 3.dxe5'),
			moves => {
				'e8d8' => 1,
			},
		},
		'rnbk1bnr/ppp2ppp/8/4p3/2P5/8/PP2PPPP/RNB1KBNR w KQ' => {
			eco => 'A41o',
			# TRANSLATORS: A41o: 1. d4 d6 2. c4 e5 3. dxe5 dxe5 4. Qxd8+ Kxd8
			variation => N__('Neo-Old Indian: Queenswap'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/4p3/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A41p',
			# TRANSLATORS: A41p: 1. d4 d6 2. c4 e5 3. Nf3
			variation => N__('Neo-Old Indian: 2. c4 e5 3. Nf3'),
			moves => {
				'e5e4' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/3p4/8/2PPp3/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41p',
			# TRANSLATORS: A41p: 1. d4 d6 2. c4 e5 3. Nf3 e4
			variation => N__('Neo-Old Indian: 2. c4 e5 3. Nf3 e4'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A41q',
			# TRANSLATORS: A41q: 1. d4 d6 2. c4 g6
			variation => N__('Neo-Old Indian: Modern'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/2PPP3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A41q',
			# TRANSLATORS: A41q: 1. d4 d6 2. c4 g6 3. e4
			variation => N__('Neo-Old Indian: Modern, 3. e4'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PPP3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A41q',
			# TRANSLATORS: A41q: 1. d4 d6 2. c4 g6 3. e4 Bg7
			variation => N__('Neo-Old Indian: Modern, 3. e4 Bg7'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A41r',
			# TRANSLATORS: A41r: 1. d4 d6 2. c4 g6 3. Nf3
			variation => N__('Neo-Old Indian: Modern, 3. Nf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A41r',
			# TRANSLATORS: A41r: 1. d4 d6 2. c4 g6 3. Nf3 Bg7
			variation => N__('Neo-Old Indian: Modern, 3. Nf3 Bg7'),
			moves => {
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PP4/5NP1/PP2PP1P/RNBQKB1R b KQkq' => {
			eco => 'A41r',
			# TRANSLATORS: A41r: 1. d4 d6 2. c4 g6 3. Nf3 Bg7 4. g3
			variation => N__('Neo-Old Indian: Modern, 3. Nf3 Bg7 4. g3'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PPP3/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'A41s',
			# TRANSLATORS: A41s: 1. d4 d6 2. c4 g6 3. Nf3 Bg7 4. e4
			variation => N__('Neo-Old Indian: Modern, 3. Nf3 Bg7 4. e4'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qk1nr/ppp1ppbp/3p2p1/8/2PPP1b1/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'A41s',
			# TRANSLATORS: A41s: 1. d4 d6 2. c4 g6 3. Nf3 Bg7 4. e4 Bg4
			variation => N__('Neo-Old Indian: Modern, Rossolimo Variation'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A41t',
			# TRANSLATORS: A41t: 1. d4 d6 2. c4 g6 3. Nc3
			variation => N__('Neo-Old Indian: Modern, 3. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A41t',
			# TRANSLATORS: A41t: 1. d4 d6 2. c4 g6 3. Nc3 Bg7
			variation => N__('Neo-Old Indian: Modern, 3. Nc3 Bg7'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A41u',
			# TRANSLATORS: A41u: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. Nf3
			variation => N__('Neo-Old Indian: Modern, 3. Nc3 Bg7 4. Nf3'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qk1nr/ppp1ppbp/3p2p1/8/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A41v',
			# TRANSLATORS: A41v: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. Nf3 Bg4
			variation => N__('Neo-Old Indian: Modern, 3. Nc3 Bg7 4. Nf3 Bf4'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rn1qk1nr/ppp1ppbp/3p2p1/8/2PP2b1/2N1PN2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A41w',
			# TRANSLATORS: A41w: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. Nf3 Bg4 5. e3
			variation => N__('Neo-Old Indian: Modern, 3. Nc3 Bg7 4. Nf3 Bf4 5. e3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2qk1nr/ppp1ppbp/2np2p1/8/2PP2b1/2N1PN2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A41w',
			# TRANSLATORS: A41w: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. Nf3 Bg4 5. e3 Nc6
			variation => N__('Neo-Old Indian: Modern, 3. Nc3 Bg7 4. Nf3 Bf4 5. e3 Nc6'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A42a',
			# TRANSLATORS: A42a: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4
			variation => N__('Modern: Averbakh'),
			moves => {
				'b8c6' => 1,
				'b8d7' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'e7e5' => 1,
				'f7f5' => 1,
			},
		},
		'rnbqk1nr/ppp1p1bp/3p2p1/5p2/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42b',
			# TRANSLATORS: A42b: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 f5
			variation => N__('Modern: Averbakh, Randspringer Variation'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp1p1bp/3p2p1/5p2/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42b',
			# TRANSLATORS: A42b: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 f5 5. Nf3
			variation => N__('Modern: Averbakh, Randspringer, 5. Nf3'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2p5/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42c',
			# TRANSLATORS: A42c: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c5
			variation => N__('Modern: c4 Pterodactyl'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2p5/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42c',
			# TRANSLATORS: A42c: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c5 5. Nf3
			variation => N__('Modern: c4 Pterodactyl'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1k1nr/pp2ppbp/3p2p1/q1p5/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A42c',
			# TRANSLATORS: A42c: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c5 5. Nf3 Qa5
			variation => N__('Modern: c4 Pterodactyl'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42d',
			# TRANSLATORS: A42d: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c6
			variation => N__('Modern: Averbakh, 4... c6'),
			moves => {
				'c1e3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/2PPP3/2N1B3/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A42d',
			# TRANSLATORS: A42d: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c6 5. Be3
			variation => N__('Modern: Averbakh, 4... c6 5. Be3'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42e',
			# TRANSLATORS: A42e: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 c6 5. Nf3
			variation => N__('Modern: Averbakh, 4... c6 5. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/pppnppbp/3p2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42f',
			# TRANSLATORS: A42f: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nd7
			variation => N__('Modern: Averbakh, 4... Nd7'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/pppnppbp/3p2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42f',
			# TRANSLATORS: A42f: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nd7 5. Nf3
			variation => N__('Modern: Averbakh, 4... Nd7 5. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42g',
			# TRANSLATORS: A42g: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6
			variation => N__('Modern: Averbakh, Kotov Variation'),
			moves => {
				'c1e3' => 1,
				'd4d5' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N5/PP2NPPP/R1BQKB1R b KQkq' => {
			eco => 'A42g',
			# TRANSLATORS: A42g: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Nge2
			variation => N__('Modern: Averbakh, Kotov, 5.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42g',
			# TRANSLATORS: A42g: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Nf3
			variation => N__('Modern: Averbakh, Kotov, 5. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/3P4/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A42h',
			# TRANSLATORS: A42h: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. d5
			variation => N__('Modern: Averbakh, Kotov, 5. d5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/2PPP3/2N1B3/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A42i',
			# TRANSLATORS: A42i: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3
			variation => N__('Modern: Averbakh, Kotov, 5. Be3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'A42j',
			# TRANSLATORS: A42j: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5'),
			moves => {
				'd4d5' => 1,
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/4p3/2PPP3/2N1B3/PP2NPPP/R2QKB1R b KQkq' => {
			eco => 'A42j',
			# TRANSLATORS: A42j: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. Nge2
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/ppp2pbp/2np2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A42k',
			# TRANSLATORS: A42k: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5'),
			moves => {
				'c6e7' => 1,
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P3/2N1B3/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'A42l',
			# TRANSLATORS: A42l: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7'),
			moves => {
				'c4c5' => 1,
				'g2g4' => 1,
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/2PPp3/4P3/2N1B3/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A42l',
			# TRANSLATORS: A42l: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7 7. c5
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7 7. c5'),
			moves => {
			},
		},
		'r1bqk1nr/ppp1npbp/3p2p1/3Pp3/2P1P1P1/2N1B3/PP3P1P/R2QKBNR b KQkq' => {
			eco => 'A42m',
			# TRANSLATORS: A42m: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 Nc6 5. Be3 e5 6. d5 Nce7 7. g4
			variation => N__('Modern: Averbakh, Kotov, 5. Be3 e5 6. d5 Nce7 7. g4'),
			moves => {
			},
		},
		'rnbqk1nr/ppp2pbp/3p2p1/4p3/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42n',
			# TRANSLATORS: A42n: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5
			variation => N__('Modern: Averbakh, 4... e5'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp2pbp/3p2p1/4p3/2PPP3/2N5/PP2NPPP/R1BQKB1R b KQkq' => {
			eco => 'A42o',
			# TRANSLATORS: A42o: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nge2
			variation => N__('Modern: Averbakh, 4... e5 5.Nge2'),
			moves => {
			},
		},
		'rnbqk1nr/ppp2pbp/3p2p1/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A42p',
			# TRANSLATORS: A42p: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3
			variation => N__('Modern: Averbakh, 4... e5 5. Nf3'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A42q',
			# TRANSLATORS: A42q: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3 Nd7
			variation => N__('Modern: Averbakh, 4... e5 5. Nf3 Nd7'),
			moves => {
				'f1e2' => 1,
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'A42r',
			# TRANSLATORS: A42r: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3 Nd7 6. Be2
			variation => N__('Modern: Averbakh, 4... e5 5. Nf3 Nd7 6. Be2'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pppnnpbp/3p2p1/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQkq' => {
			eco => 'A42s',
			# TRANSLATORS: A42s: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. Nf3 Nd7 6. Be2 Ne7
			variation => N__('Modern: Averbakh, 4... e5 5. Nf3 Nd7 6. Be2 Ne7'),
			moves => {
			},
		},
		'rnbqk1nr/ppp2pbp/3p2p1/3Pp3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A42t',
			# TRANSLATORS: A42t: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. d5
			variation => N__('Modern: Averbakh, 4... e5 5. d5'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqk1nr/pppn1pbp/3p2p1/3Pp3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42u',
			# TRANSLATORS: A42u: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. d5 Nd7
			variation => N__('Modern: Averbakh, 4... e5 5. d5 Nd7'),
			moves => {
			},
		},
		'rnbqk1nr/ppp2pbp/3p2p1/4P3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A42v',
			# TRANSLATORS: A42v: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. dxe5
			variation => N__('Modern: Averbakh, 4... e5 5.dxe5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqk1nr/ppp2pbp/6p1/4p3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A42v',
			# TRANSLATORS: A42v: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. dxe5 dxe5
			variation => N__('Modern: Averbakh, 4... e5 5.dxe5 dxe5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'rnbQk1nr/ppp2pbp/6p1/4p3/2P1P3/2N5/PP3PPP/R1B1KBNR b KQkq' => {
			eco => 'A42v',
			# TRANSLATORS: A42v: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. dxe5 dxe5 6. Qxd8+
			variation => N__('Modern: Averbakh, 4... e5 5.dxe5 dxe5'),
			moves => {
				'e8d8' => 1,
			},
		},
		'rnbk2nr/ppp2pbp/6p1/4p3/2P1P3/2N5/PP3PPP/R1B1KBNR w KQ' => {
			eco => 'A42v',
			# TRANSLATORS: A42v: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. dxe5 dxe5 6. Qxd8+ Kxd8
			variation => N__('Modern: Averbakh, Queenswap'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbk2nr/ppp2pbp/6p1/4p3/2P1PP2/2N5/PP4PP/R1B1KBNR b KQ' => {
			eco => 'A42w',
			# TRANSLATORS: A42w: 1. d4 d6 2. c4 g6 3. Nc3 Bg7 4. e4 e5 5. dxe5 dxe5 6. Qxd8+ Kxd8 7. f4
			variation => N__('Modern: Averbakh, Queenswap, 7. f4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43a',
			# TRANSLATORS: A43a: 1. d4 c5
			variation => N__('Old Benoni'),
			moves => {
				'b2b4' => 1,
				'c2c3' => 1,
				'd4c5' => 1,
				'd4d5' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/1P1P4/8/P1P1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43a',
			# TRANSLATORS: A43a: 1. d4 c5 2. b4
			variation => N__('Old Benoni: Nakamura Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2P5/8/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43b',
			# TRANSLATORS: A43b: 1. d4 c5 2. dxc5
			variation => N__('Old Benoni: 2.dxc5'),
			moves => {
				'b7b6' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2P5/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43b',
			# TRANSLATORS: A43b: 1. d4 c5 2. dxc5 b6
			variation => N__('Old Benoni: Cormorant Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/3P4/2P5/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43c',
			# TRANSLATORS: A43c: 1. d4 c5 2. c3
			variation => N__('Old Benoni: 2. c3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/3P4/4P3/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A43d',
			# TRANSLATORS: A43d: 1. d4 c5 2. e3
			variation => N__('Old Benoni: 2. e3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2pP4/8/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43e',
			# TRANSLATORS: A43e: 1. d4 c5 2. d5
			variation => N__('Old Benoni: 2. d5'),
			moves => {
				'b7b5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'f7f5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/p2ppppp/8/1ppP4/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43e',
			# TRANSLATORS: A43e: 1. d4 c5 2. d5 b5
			variation => N__('Old Benoni: 2. d5 b5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2pP4/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43f',
			# TRANSLATORS: A43f: 1. d4 c5 2. d5 e6
			variation => N__('Old Benoni: 2. d5 e6'),
			moves => {
				'c2c4' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2pP4/4P3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A43g',
			# TRANSLATORS: A43g: 1. d4 c5 2. d5 e6 3. e4
			variation => N__('Old Benoni: Franco-Benoni'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2pP4/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43h',
			# TRANSLATORS: A43h: 1. d4 c5 2. d5 e6 3. c4
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'e6d5' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2pp4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43h',
			# TRANSLATORS: A43h: 1. d4 c5 2. d5 e6 3. c4 exd5
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2pP4/8/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43h',
			# TRANSLATORS: A43h: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43h',
			# TRANSLATORS: A43h: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pP4/4P3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A43h',
			# TRANSLATORS: A43h: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. e4
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pP4/8/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A43i',
			# TRANSLATORS: A43i: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. Nc3
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp3p1p/3p2p1/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A43i',
			# TRANSLATORS: A43i: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. Nc3 g6
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp3p1p/3p2p1/2pP4/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A43i',
			# TRANSLATORS: A43i: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. Nc3 g6 6. e4
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp3pbp/3p2p1/2pP4/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A43i',
			# TRANSLATORS: A43i: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. Nc3 g6 6. e4 Bg7
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp3pbp/3p2p1/2pP4/4P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A43i',
			# TRANSLATORS: A43i: 1. d4 c5 2. d5 e6 3. c4 exd5 4. cxd5 d6 5. Nc3 g6 6. e4 Bg7 7. Nf3
			variation => N__('Old Benoni: 2. d5 e6 3. c4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1pp1pp/8/2pP1p2/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43j',
			# TRANSLATORS: A43j: 1. d4 c5 2. d5 f5
			variation => N__('Old Benoni: Clarendon Court Defence'),
			moves => {
				'b1c3' => 1,
				'c2c4' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1pp1pp/8/2pP1p2/8/2N5/PPP1PPPP/R1BQKBNR b KQkq' => {
			eco => 'A43j',
			# TRANSLATORS: A43j: 1. d4 c5 2. d5 f5 3. Nc3
			variation => N__('Old Benoni: Clarendon Court, 3. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1pp1pp/8/2pP1p2/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A43j',
			# TRANSLATORS: A43j: 1. d4 c5 2. d5 f5 3. c4
			variation => N__('Old Benoni: Clarendon Court, 3. c4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1pp1pp/8/2pP1p2/8/6P1/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A43j',
			# TRANSLATORS: A43j: 1. d4 c5 2. d5 f5 3. g3
			variation => N__('Old Benoni: Clarendon Court, 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP4/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43k',
			# TRANSLATORS: A43k: 1. d4 c5 2. d5 Nf6
			variation => N__('Old Benoni: 2. d5 Nf6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP4/8/2N5/PPP1PPPP/R1BQKBNR b KQkq' => {
			eco => 'A43k',
			# TRANSLATORS: A43k: 1. d4 c5 2. d5 Nf6 3. Nc3
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nc3'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/5n2/q1pP4/8/2N5/PPP1PPPP/R1BQKBNR w KQkq' => {
			eco => 'A43k',
			# TRANSLATORS: A43k: 1. d4 c5 2. d5 Nf6 3. Nc3 Qa5
			variation => N__('Old Benoni: Woozle'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP4/8/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A43l',
			# TRANSLATORS: A43l: 1. d4 c5 2. d5 Nf6 3. Nf3
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nf3'),
			moves => {
				'b7b5' => 1,
				'c5c4' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A43l',
			# TRANSLATORS: A43l: 1. d4 c5 2. d5 Nf6 3. Nf3 e6
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nf3 e6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/8/2N2N2/PPP1PPPP/R1BQKB1R b KQkq' => {
			eco => 'A43l',
			# TRANSLATORS: A43l: 1. d4 c5 2. d5 Nf6 3. Nf3 e6 4. Nc3
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nf3 e6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2pP4/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A43m',
			# TRANSLATORS: A43m: 1. d4 c5 2. d5 Nf6 3. Nf3 g6
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nf3 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2pP4/8/2N2N2/PPP1PPPP/R1BQKB1R b KQkq' => {
			eco => 'A43m',
			# TRANSLATORS: A43m: 1. d4 c5 2. d5 Nf6 3. Nf3 g6 4. Nc3
			variation => N__('Old Benoni: 2. d5 Nf6 3. Nf3 g6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A43n',
			# TRANSLATORS: A43n: 1. d4 c5 2. d5 Nf6 3. Nf3 b5
			variation => N__('Old Benoni: Neo-Benko'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP2B1/8/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A43o',
			# TRANSLATORS: A43o: 1. d4 c5 2. d5 Nf6 3. Nf3 b5 4. Bg5
			variation => N__('Old Benoni: Neo-Benko, 4. Bg5'),
			moves => {
				'd7d6' => 1,
				'f6e4' => 1,
			},
		},
		'rnbqkb1r/p3pppp/3p1n2/1ppP2B1/8/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A43p',
			# TRANSLATORS: A43p: 1. d4 c5 2. d5 Nf6 3. Nf3 b5 4. Bg5 d6
			variation => N__('Old Benoni: Neo-Benko, 4. Bg5 d6'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/8/1ppP2B1/4n3/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A43q',
			# TRANSLATORS: A43q: 1. d4 c5 2. d5 Nf6 3. Nf3 b5 4. Bg5 Ne4
			variation => N__('Old Benoni: Neo-Benko, 4. Bg5 Ne4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/3P4/2p5/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A43r',
			# TRANSLATORS: A43r: 1. d4 c5 2. d5 Nf6 3. Nf3 c4
			variation => N__('Old Benoni: Hawk'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/3P4/2p1P3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A43r',
			# TRANSLATORS: A43r: 1. d4 c5 2. d5 Nf6 3. Nf3 c4 4. e4
			variation => N__('Old Benoni: Hawk, 4. e4'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2pP4/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6
			variation => N__('Old Benoni: Schmidt'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2pP4/8/2N5/PPP1PPPP/R1BQKBNR b KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. Nc3
			variation => N__('Old Benoni: Schmidt, 3. Nc3'),
			moves => {
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/3p2p1/2pP4/8/2N5/PPP1PPPP/R1BQKBNR w KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. Nc3 g6
			variation => N__('Old Benoni: Schmidt, 3. Nc3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/8/2N5/PPP1PPPP/R1BQKBNR w KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. Nc3 Nf6
			variation => N__('Old Benoni: Schmidt, 3. Nc3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2pP4/4P3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. e4
			variation => N__('Old Benoni: Schmidt, 3. e4'),
			moves => {
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2pP4/8/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. Nf3
			variation => N__('Old Benoni: Schmidt, 3. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/8/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A43s',
			# TRANSLATORS: A43s: 1. d4 c5 2. d5 d6 3. Nf3 Nf6
			variation => N__('Old Benoni: Schmidt, 3. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/3p2p1/2pP4/4P3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/3p2p1/2pP4/4P3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6 4. Nc3
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2pP4/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6 4. Nc3 Bg7
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/3p2p1/2pP4/4P3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6 4. Nf3
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2pP4/4P3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6 4. Nf3 Bg7
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2pP4/4P3/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'A43t',
			# TRANSLATORS: A43t: 1. d4 c5 2. d5 d6 3. e4 g6 4. Nf3 Bg7 5. Be2
			variation => N__('Old Benoni: Schmidt, 3. e4 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/4P3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A43u',
			# TRANSLATORS: A43u: 1. d4 c5 2. d5 d6 3. e4 Nf6
			variation => N__('Old Benoni: Schmidt, 3. e4 Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/4P3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A43u',
			# TRANSLATORS: A43u: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3
			variation => N__('Old Benoni: Schmidt, 3. e4 Nf6'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/3p1np1/2pP4/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A43u',
			# TRANSLATORS: A43u: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6
			variation => N__('Old Benoni: Schmidt, 3. e4 Nf6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/3p1np1/2pP4/4P3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'A43u',
			# TRANSLATORS: A43u: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3
			variation => N__('Old Benoni: Schmidt, 3. e4 Nf6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/4P3/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'A43u',
			# TRANSLATORS: A43u: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7
			variation => N__('Old Benoni: Schmidt, 3. e4 Nf6'),
			moves => {
				'f1b5' => 1,
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/4P3/2N2N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'A43v',
			# TRANSLATORS: A43v: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. h3
			variation => N__('Old Benoni: Schmidt, 6. h3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/3p1np1/2pP4/4P3/2N2N1P/PPP2PP1/R1BQKB1R w KQ' => {
			eco => 'A43v',
			# TRANSLATORS: A43v: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. h3 O-O
			variation => N__('Old Benoni: Schmidt, 6. h3'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/1BpP4/4P3/2N2N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'A43w',
			# TRANSLATORS: A43w: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Bb5+
			variation => N__('Old Benoni: Schmidt, 6. Bb5+'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQK2R b KQkq' => {
			eco => 'A43x',
			# TRANSLATORS: A43x: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2
			variation => N__('Old Benoni: Schmidt, 6. Be2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/3p1np1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQK2R w KQ' => {
			eco => 'A43x',
			# TRANSLATORS: A43x: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O
			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/3p1np1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 b -' => {
			eco => 'A43x',
			# TRANSLATORS: A43x: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O
			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O'),
			moves => {
				'b8a6' => 1,
				'e7e6' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3ppnp1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 w -' => {
			eco => 'A43x',
			# TRANSLATORS: A43x: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O e6
			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O e6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4P3/2N2N2/PPP1BPPP/R1BQ1RK1 w -' => {
			eco => 'A43y',
			# TRANSLATORS: A43y: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O Na6
			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O Na6'),
			moves => {
				'h2h3' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4P3/2N2N1P/PPP1BPP1/R1BQ1RK1 b -' => {
			eco => 'A43z',
			# TRANSLATORS: A43z: 1. d4 c5 2. d5 d6 3. e4 Nf6 4. Nc3 g6 5. Nf3 Bg7 6. Be2 O-O 7. O-O Na6 8. h3
			variation => N__('Old Benoni: Schmidt, 6. Be2 O-O 7.O-O Na6 8. h3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2pPp3/8/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A44a',
			# TRANSLATORS: A44a: 1. d4 c5 2. d5 e5
			variation => N__('Old Benoni: Czech'),
			moves => {
				'c2c4' => 1,
				'd5e6' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4P3/2p5/8/8/PPP1PPPP/RNBQKBNR b KQkq' => {
			eco => 'A44b',
			# TRANSLATORS: A44b: 1. d4 c5 2. d5 e5 3. dxe6
			variation => N__('Old Benoni: Czech, 3.dxe6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2pPp3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A44c',
			# TRANSLATORS: A44c: 1. d4 c5 2. d5 e5 3. c4
			variation => N__('Old Benoni: Czech, 3. c4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A44d',
			# TRANSLATORS: A44d: 1. d4 c5 2. d5 e5 3. c4 d6
			variation => N__('Old Benoni: Czech, 3. c4 d6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/2P1P3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A44e',
			# TRANSLATORS: A44e: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4'),
			moves => {
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqk1nr/pp2bppp/3p4/2pPp3/2P1P3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A44f',
			# TRANSLATORS: A44f: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 Be7
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 Be7'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqk1nr/pp2bppp/3p4/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A44g',
			# TRANSLATORS: A44g: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 Be7 5. Nc3
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 Be7 5. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/pp3p1p/3p2p1/2pPp3/2P1P3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A44h',
			# TRANSLATORS: A44h: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 g6
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/pp3p1p/3p2p1/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A44h',
			# TRANSLATORS: A44h: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 g6 5. Nc3
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 g6 5. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp3pbp/3p2p1/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A44i',
			# TRANSLATORS: A44i: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 g6 5. Nc3 Bg7
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 g6 5. Nc3 Bg7'),
			moves => {
				'f1d3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp3pbp/3p2p1/2pPp3/2P1P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A44j',
			# TRANSLATORS: A44j: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 g6 5. Nc3 Bg7 6. Nf3
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 g6 5. Nc3 Bg7 6. Nf3'),
			moves => {
			},
		},
		'rnbqk1nr/pp3pbp/3p2p1/2pPp3/2P1P3/2NB4/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A44k',
			# TRANSLATORS: A44k: 1. d4 c5 2. d5 e5 3. c4 d6 4. e4 g6 5. Nc3 Bg7 6. Bd3
			variation => N__('Old Benoni: Czech, 3. c4 d6 4. e4 g6 5. Nc3 Bg7 6. Bd3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2pPp3/4P3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A44l',
			# TRANSLATORS: A44l: 1. d4 c5 2. d5 e5 3. e4
			variation => N__('Old Benoni: Czech, 3. e4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/4P3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A44m',
			# TRANSLATORS: A44m: 1. d4 c5 2. d5 e5 3. e4 d6
			variation => N__('Old Benoni: Czech, 3. e4 d6'),
			moves => {
				'b1c3' => 1,
				'f1b5' => 1,
				'f1d3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/4P3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A44n',
			# TRANSLATORS: A44n: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nf3
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pp3ppp/3p4/1BpPp3/4P3/8/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'A44o',
			# TRANSLATORS: A44o: 1. d4 c5 2. d5 e5 3. e4 d6 4. Bb5+
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Bb5+'),
			moves => {
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/4P3/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'A44p',
			# TRANSLATORS: A44p: 1. d4 c5 2. d5 e5 3. e4 d6 4. Bd3
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Bd3'),
			moves => {
			},
		},
		'rnbqkbnr/pp3ppp/3p4/2pPp3/4P3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A44q',
			# TRANSLATORS: A44q: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3'),
			moves => {
				'a7a6' => 1,
				'f8e7' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pPp3/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A44r',
			# TRANSLATORS: A44r: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 Nf6
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pp3p1p/3p2p1/2pPp3/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A44s',
			# TRANSLATORS: A44s: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 g6
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 g6'),
			moves => {
			},
		},
		'rnbqkbnr/1p3ppp/p2p4/2pPp3/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A44t',
			# TRANSLATORS: A44t: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 a6
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbqkbnr/1p3ppp/p2p4/2pPp3/P3P3/2N5/1PP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A44t',
			# TRANSLATORS: A44t: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 a6 5. a4
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 a6'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rnbqk1nr/1p2bppp/p2p4/2pPp3/P3P3/2N5/1PP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A44u',
			# TRANSLATORS: A44u: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 a6 5. a4 Be7
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 a6 5. a4 Be7'),
			moves => {
			},
		},
		'rnbqk1nr/pp2bppp/3p4/2pPp3/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A44v',
			# TRANSLATORS: A44v: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 Be7
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 Be7'),
			moves => {
				'f1b5' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp2bppp/3p4/1BpPp3/4P3/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'A44w',
			# TRANSLATORS: A44w: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 Be7 5. Bb5+
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 Be7 5. Bb5+'),
			moves => {
			},
		},
		'rnbqk1nr/pp2bppp/3p4/2pPp3/4P3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'A44x',
			# TRANSLATORS: A44x: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 Be7 5. Nf3
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 Be7 5. Nf3'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qk1nr/pp2bppp/3p4/2pPp3/4P1b1/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'A44y',
			# TRANSLATORS: A44y: 1. d4 c5 2. d5 e5 3. e4 d6 4. Nc3 Be7 5. Nf3 Bg4
			variation => N__('Old Benoni: Czech, 3. e4 d6 4. Nc3 Be7 5. Nf3 Bg4'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6
			variation => N__('Queen\'s Pawn: Indian'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'c1f4' => 1,
				'c1g5' => 1,
				'c2c3' => 1,
				'c2c4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'f2f3' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. e4
			variation => N__('Indian: Omega Gambit'),
			moves => {
				'f6e4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/8/3Pn3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. e4 Nxe4
			variation => N__('Indian: Omega Gambit'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/8/3Pn3/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. e4 Nxe4 3. Bd3
			variation => N__('Indian: Omega Gambit'),
			moves => {
				'e4f6' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. e4 Nxe4 3. Bd3 Nf6
			variation => N__('Indian: Omega Gambit'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/3B1N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. e4 Nxe4 3. Bd3 Nf6 4. Nf3
			variation => N__('Indian: Arafat Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/5P2/PPP1P1PP/RNBQKBNR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. f3
			variation => N__('Indian: Paleface Attack'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/3P4/5P2/PPP1P1PP/RNBQKBNR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. f3 d5
			variation => N__('Indian: Paleface Attack'),
			moves => {
				'e2e4' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/3PP3/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. f3 d5 3. e4
			variation => N__('Indian: Blackmar-Diemer Gambit (without Nc3)'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/3P2P1/5P2/PPP1P2P/RNBQKBNR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. f3 d5 3. g4
			variation => N__('Indian: Gedult Attack'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P2P1/8/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. g4
			variation => N__('Indian: Gibbins Gambit'),
			moves => {
				'e7e5' => 1,
				'f6g4' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/3P2P1/8/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. g4 e5
			variation => N__('Indian: Gibbins Gambit, Oshima Defence'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/8/3P2n1/8/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A45a',
			# TRANSLATORS: A45a: 1. d4 Nf6 2. g4 Nxg4
			variation => N__('Indian: Gibbins Gambit Accepted'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P1P2/8/PPP1P1PP/RNBQKBNR b KQkq' => {
			eco => 'A45b',
			# TRANSLATORS: A45b: 1. d4 Nf6 2. f4
			variation => N__('Indian: Canard Opening'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/8/PPPNPPPP/R1BQKBNR b KQkq' => {
			eco => 'A45c',
			# TRANSLATORS: A45c: 1. d4 Nf6 2. Nd2
			variation => N__('Indian: 2. Nd2'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/3P4/8/PPPNPPPP/R1BQKBNR w KQkq' => {
			eco => 'A45c',
			# TRANSLATORS: A45c: 1. d4 Nf6 2. Nd2 e5
			variation => N__('Indian: Lazard Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/4P3/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A45d',
			# TRANSLATORS: A45d: 1. d4 Nf6 2. e3
			variation => N__('Indian: 2. e3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P4/4P3/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A45d',
			# TRANSLATORS: A45d: 1. d4 Nf6 2. e3 e6
			variation => N__('Indian: 2. e3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/4P3/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A45d',
			# TRANSLATORS: A45d: 1. d4 Nf6 2. e3 g6
			variation => N__('Indian: 2. e3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/2P5/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A45e',
			# TRANSLATORS: A45e: 1. d4 Nf6 2. c3
			variation => N__('Indian: 2. c3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/2P5/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A45e',
			# TRANSLATORS: A45e: 1. d4 Nf6 2. c3 g6
			variation => N__('Indian: 2. c3 g6'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/6B1/3P4/2P5/PP2PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45e',
			# TRANSLATORS: A45e: 1. d4 Nf6 2. c3 g6 3. Bg5
			variation => N__('Indian: 2. c3 g6 3. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/2N5/PPP1PPPP/R1BQKBNR b KQkq' => {
			eco => 'A45f',
			# TRANSLATORS: A45f: 1. d4 Nf6 2. Nc3
			variation => N__('Indian: 2. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P1B2/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45g',
			# TRANSLATORS: A45g: 1. d4 Nf6 2. Bf4
			variation => N__('Indian: 2. Bf4'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/6P1/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A45h',
			# TRANSLATORS: A45h: 1. d4 Nf6 2. g3
			variation => N__('Indian: 2. g3'),
			moves => {
				'c7c5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A45h',
			# TRANSLATORS: A45h: 1. d4 Nf6 2. g3 g6
			variation => N__('Indian: 2. g3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/3P4/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A45i',
			# TRANSLATORS: A45i: 1. d4 Nf6 2. g3 c5
			variation => N__('Indian: 2. g3 c5'),
			moves => {
				'd4d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP4/8/6P1/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A45i',
			# TRANSLATORS: A45i: 1. d4 Nf6 2. g3 c5 3. d5
			variation => N__('Indian: 2. g3 c5'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/8/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A45i',
			# TRANSLATORS: A45i: 1. d4 Nf6 2. g3 c5 3. d5 b5
			variation => N__('Indian: 2. g3 c5 3. d5 b5'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/6B1/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45j',
			# TRANSLATORS: A45j: 1. d4 Nf6 2. Bg5
			variation => N__('Trompowsky Opening'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'f6e4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/6B1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45j',
			# TRANSLATORS: A45j: 1. d4 Nf6 2. Bg5 d6
			variation => N__('Trompowsky 2... d6'),
			moves => {
				'b1c3' => 1,
				'g5f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/6B1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A45j',
			# TRANSLATORS: A45j: 1. d4 Nf6 2. Bg5 d6 3. Nc3
			variation => N__('Trompowsky 2... d6 3. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1B2/8/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45j',
			# TRANSLATORS: A45j: 1. d4 Nf6 2. Bg5 d6 3. Bxf6
			variation => N__('Trompowsky 2... d6 3.Bxf6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/6B1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45k',
			# TRANSLATORS: A45k: 1. d4 Nf6 2. Bg5 g6
			variation => N__('Trompowsky 2... g6'),
			moves => {
				'b1c3' => 1,
				'g5f6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/6B1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A45k',
			# TRANSLATORS: A45k: 1. d4 Nf6 2. Bg5 g6 3. Nc3
			variation => N__('Trompowsky 2... g6 3. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5Bp1/8/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45k',
			# TRANSLATORS: A45k: 1. d4 Nf6 2. Bg5 g6 3. Bxf6
			variation => N__('Trompowsky 2... g6 3.Bxf6'),
			moves => {
				'e7f6' => 1,
			},
		},
		'rnbqkb1r/pppp1p1p/5pp1/8/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45k',
			# TRANSLATORS: A45k: 1. d4 Nf6 2. Bg5 g6 3. Bxf6 exf6
			variation => N__('Trompowsky 2... g6 3.Bxf6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pppp1p1p/5pp1/8/3P4/4P3/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'A45l',
			# TRANSLATORS: A45l: 1. d4 Nf6 2. Bg5 g6 3. Bxf6 exf6 4. e3
			variation => N__('Trompowsky 2... g6 3.Bxf6 exf6 4. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45m',
			# TRANSLATORS: A45m: 1. d4 Nf6 2. Bg5 e6
			variation => N__('Trompowsky: 2... e6'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A45m',
			# TRANSLATORS: A45m: 1. d4 Nf6 2. Bg5 e6 3. Nc3
			variation => N__('Trompowsky: 2... e6 3. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3P4/8/PPPNPPPP/R2QKBNR b KQkq' => {
			eco => 'A45m',
			# TRANSLATORS: A45m: 1. d4 Nf6 2. Bg5 e6 3. Nd2
			variation => N__('Trompowsky: 2... e6 3. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3P4/4P3/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'A45m',
			# TRANSLATORS: A45m: 1. d4 Nf6 2. Bg5 e6 3. e3
			variation => N__('Trompowsky: 2... e6 3. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3PP3/8/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'A45n',
			# TRANSLATORS: A45n: 1. d4 Nf6 2. Bg5 e6 3. e4
			variation => N__('Trompowsky: 2... e6 3. e4'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rnbqkb1r/pppp1pp1/4pn1p/6B1/3PP3/8/PPP2PPP/RN1QKBNR w KQkq' => {
			eco => 'A45n',
			# TRANSLATORS: A45n: 1. d4 Nf6 2. Bg5 e6 3. e4 h6
			variation => N__('Trompowsky: 2... e6 3. e4 h6'),
			moves => {
				'g5f6' => 1,
			},
		},
		'rnbqkb1r/pppp1pp1/4pB1p/8/3PP3/8/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'A45n',
			# TRANSLATORS: A45n: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6'),
			moves => {
				'd8f6' => 1,
			},
		},
		'rnb1kb1r/pppp1pp1/4pq1p/8/3PP3/8/PPP2PPP/RN1QKBNR w KQkq' => {
			eco => 'A45n',
			# TRANSLATORS: A45n: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6'),
			moves => {
				'b1c3' => 1,
				'c2c3' => 1,
			},
		},
		'rnb1kb1r/pppp1pp1/4pq1p/8/3PP3/2P5/PP3PPP/RN1QKBNR b KQkq' => {
			eco => 'A45n',
			# TRANSLATORS: A45n: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. c3
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. c3'),
			moves => {
			},
		},
		'rnb1kb1r/pppp1pp1/4pq1p/8/3PP3/2N5/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A45o',
			# TRANSLATORS: A45o: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. Nc3
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. Nc3'),
			moves => {
				'd7d6' => 1,
				'f8b4' => 1,
			},
		},
		'rnb1kb1r/ppp2pp1/3ppq1p/8/3PP3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A45o',
			# TRANSLATORS: A45o: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. Nc3 d6
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. Nc3 d6'),
			moves => {
				'd1d2' => 1,
			},
		},
		'rnb1kb1r/ppp2pp1/3ppq1p/8/3PP3/2N5/PPPQ1PPP/R3KBNR b KQkq' => {
			eco => 'A45o',
			# TRANSLATORS: A45o: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. Nc3 d6 6. Qd2
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. Nc3 d6 6. Qd2'),
			moves => {
				'g7g5' => 1,
			},
		},
		'rnb1kb1r/ppp2p2/3ppq1p/6p1/3PP3/2N5/PPPQ1PPP/R3KBNR w KQkq' => {
			eco => 'A45o',
			# TRANSLATORS: A45o: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. Nc3 d6 6. Qd2 g5
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. Nc3 d6 6. Qd2 g5'),
			moves => {
			},
		},
		'rnb1k2r/pppp1pp1/4pq1p/8/1b1PP3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A45o',
			# TRANSLATORS: A45o: 1. d4 Nf6 2. Bg5 e6 3. e4 h6 4. Bxf6 Qxf6 5. Nc3 Bb4
			variation => N__('Trompowsky: 2... e6 3. e4 h6 4.Bxf6 Qxf6 5. Nc3 Bb4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p3B1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5
			variation => N__('Trompowsky: 2... c5'),
			moves => {
				'b1c3' => 1,
				'd4c5' => 1,
				'd4d5' => 1,
				'g5f6' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p3B1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5 3. Nc3
			variation => N__('Trompowsky: 2... c5 3. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2P3B1/8/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5 3. dxc5
			variation => N__('Trompowsky: 2... c5 3.dxc5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP2B1/8/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5 3. d5
			variation => N__('Trompowsky: 2... c5 3. d5'),
			moves => {
				'd8b6' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/1q3n2/2pP2B1/8/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5 3. d5 Qb6
			variation => N__('Trompowsky: 2... c5 3. d5 Qb6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/1q3n2/2pP2B1/8/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A45p',
			# TRANSLATORS: A45p: 1. d4 Nf6 2. Bg5 c5 3. d5 Qb6 4. Nc3
			variation => N__('Trompowsky: 2... c5 3. d5 Qb6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5B2/2p5/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45q',
			# TRANSLATORS: A45q: 1. d4 Nf6 2. Bg5 c5 3. Bxf6
			variation => N__('Trompowsky: 2... c5 3.Bxf6'),
			moves => {
				'g7f6' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5p2/2p5/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45q',
			# TRANSLATORS: A45q: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6'),
			moves => {
				'd4d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5p2/2pP4/8/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45q',
			# TRANSLATORS: A45q: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5'),
			moves => {
				'd8b6' => 1,
			},
		},
		'rnb1kb1r/pp1ppp1p/1q3p2/2pP4/8/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6'),
			moves => {
				'd1c1' => 1,
			},
		},
		'rnb1kb1r/pp1ppp1p/1q3p2/2pP4/8/8/PPP1PPPP/RNQ1KBNR b KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1'),
			moves => {
				'f6f5' => 1,
			},
		},
		'rnb1kb1r/pp1ppp1p/1q6/2pP1p2/8/8/PPP1PPPP/RNQ1KBNR w KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5'),
			moves => {
				'c2c4' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnb1kb1r/pp1ppp1p/1q6/2pP1p2/2P5/8/PP2PPPP/RNQ1KBNR b KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. c4
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. c4'),
			moves => {
			},
		},
		'rnb1kb1r/pp1ppp1p/1q6/2pP1p2/8/4P3/PPP2PPP/RNQ1KBNR b KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. e3
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. e3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnb1k2r/pp1pppbp/1q6/2pP1p2/8/4P3/PPP2PPP/RNQ1KBNR w KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. e3 Bg7
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. e3 Bg7'),
			moves => {
			},
		},
		'rnb1kb1r/pp1ppp1p/1q6/2pP1p2/8/6P1/PPP1PP1P/RNQ1KBNR b KQkq' => {
			eco => 'A45r',
			# TRANSLATORS: A45r: 1. d4 Nf6 2. Bg5 c5 3. Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. g3
			variation => N__('Trompowsky: 2... c5 3.Bxf6 gxf6 4. d5 Qb6 5. Qc1 f5 6. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/6B1/3Pn3/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4
			variation => N__('Trompowsky: 2... Ne4'),
			moves => {
				'g5f4' => 1,
				'g5h4' => 1,
				'h2h4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/6B1/3Pn2P/8/PPP1PPP1/RN1QKBNR b KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4 3. h4
			variation => N__('Trompowsky: 2... Ne4 3. h4'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p3B1/3Pn2P/8/PPP1PPP1/RN1QKBNR w KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4 3. h4 c5
			variation => N__('Trompowsky: 2... Ne4 3. h4 c5'),
			moves => {
				'd4c5' => 1,
				'd4d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2P3B1/4n2P/8/PPP1PPP1/RN1QKBNR b KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4 3. h4 c5 4. dxc5
			variation => N__('Trompowsky: 2... Ne4 3. h4 c5 4.dxc5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pP2B1/4n2P/8/PPP1PPP1/RN1QKBNR b KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4 3. h4 c5 4. d5
			variation => N__('Trompowsky: 2... Ne4 3. h4 c5 4. d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p2B1/3Pn2P/8/PPP1PPP1/RN1QKBNR w KQkq' => {
			eco => 'A45s',
			# TRANSLATORS: A45s: 1. d4 Nf6 2. Bg5 Ne4 3. h4 d5
			variation => N__('Trompowsky: 2... Ne4 3. h4 d5'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/8/3Pn2B/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45t',
			# TRANSLATORS: A45t: 1. d4 Nf6 2. Bg5 Ne4 3. Bh4
			variation => N__('Trompowsky: 2... Ne4 3. Bh4'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'g7g5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p5/3Pn2B/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45t',
			# TRANSLATORS: A45t: 1. d4 Nf6 2. Bg5 Ne4 3. Bh4 c5
			variation => N__('Trompowsky: 2... Ne4 3. Bh4 c5'),
			moves => {
				'f2f3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p5/3Pn2B/5P2/PPP1P1PP/RN1QKBNR b KQkq' => {
			eco => 'A45t',
			# TRANSLATORS: A45t: 1. d4 Nf6 2. Bg5 Ne4 3. Bh4 c5 4. f3
			variation => N__('Trompowsky: 2... Ne4 3. Bh4 c5 4. f3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p4/3Pn2B/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45t',
			# TRANSLATORS: A45t: 1. d4 Nf6 2. Bg5 Ne4 3. Bh4 d5
			variation => N__('Trompowsky: 2... Ne4 3. Bh4 d5'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/8/6p1/3Pn2B/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45t',
			# TRANSLATORS: A45t: 1. d4 Nf6 2. Bg5 Ne4 3. Bh4 g5
			variation => N__('Trompowsky: 2... Ne4 3. Bh4 g5'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/8/3PnB2/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45u',
			# TRANSLATORS: A45u: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4
			variation => N__('Trompowsky: 2... Ne4 3. Bf4'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'g7g5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p4/3PnB2/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45u',
			# TRANSLATORS: A45u: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5'),
			moves => {
				'b1d2' => 1,
				'e2e3' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p4/3PnB2/8/PPPNPPPP/R2QKBNR b KQkq' => {
			eco => 'A45u',
			# TRANSLATORS: A45u: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5 4. Nd2
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5 4. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/8/6p1/3PnB2/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45u',
			# TRANSLATORS: A45u: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 g5
			variation => N__('Trompowsky: Borg Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p4/3PnB2/5P2/PPP1P1PP/RN1QKBNR b KQkq' => {
			eco => 'A45v',
			# TRANSLATORS: A45v: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5 4. f3
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5 4. f3'),
			moves => {
				'e4f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/3P1B2/5P2/PPP1P1PP/RN1QKBNR w KQkq' => {
			eco => 'A45v',
			# TRANSLATORS: A45v: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5 4. f3 Nf6
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5 4. f3 Nf6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3p4/3PnB2/4P3/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'A45w',
			# TRANSLATORS: A45w: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5 4. e3
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5 4. e3'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/2pp4/3PnB2/4P3/PPP2PPP/RN1QKBNR w KQkq' => {
			eco => 'A45w',
			# TRANSLATORS: A45w: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 d5 4. e3 c5
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 d5 4. e3 c5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p5/3PnB2/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45x',
			# TRANSLATORS: A45x: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5'),
			moves => {
				'd4d5' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pP4/4nB2/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A45x',
			# TRANSLATORS: A45x: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. d5
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. d5'),
			moves => {
				'd8b6' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/1q6/2pP4/4nB2/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A45x',
			# TRANSLATORS: A45x: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. d5 Qb6
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. d5 Qb6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p5/3PnB2/5P2/PPP1P1PP/RN1QKBNR b KQkq' => {
			eco => 'A45y',
			# TRANSLATORS: A45y: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/8/q1p5/3PnB2/5P2/PPP1P1PP/RN1QKBNR w KQkq' => {
			eco => 'A45y',
			# TRANSLATORS: A45y: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3 Qa5+
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3 Qa5+'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/8/q1p5/3PnB2/2P2P2/PP2P1PP/RN1QKBNR b KQkq' => {
			eco => 'A45y',
			# TRANSLATORS: A45y: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3 Qa5+'),
			moves => {
				'e4f6' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/5n2/q1p5/3P1B2/2P2P2/PP2P1PP/RN1QKBNR w KQkq' => {
			eco => 'A45y',
			# TRANSLATORS: A45y: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3 Nf6
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3 Qa5+'),
			moves => {
				'b1d2' => 1,
				'd4d5' => 1,
			},
		},
		'rnb1kb1r/pp1ppppp/5n2/q1pP4/5B2/2P2P2/PP2P1PP/RN1QKBNR b KQkq' => {
			eco => 'A45y',
			# TRANSLATORS: A45y: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3 Nf6 6. d5
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3 Nf6 6. d5'),
			moves => {
			},
		},
		'rnb1kb1r/pp1ppppp/5n2/q1p5/3P1B2/2P2P2/PP1NP1PP/R2QKBNR b KQkq' => {
			eco => 'A45z',
			# TRANSLATORS: A45z: 1. d4 Nf6 2. Bg5 Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3 Nf6 6. Nd2
			variation => N__('Trompowsky: 2... Ne4 3. Bf4 c5 4. f3 Qa5+ 5. c3 Nf6 6. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A46a',
			# TRANSLATORS: A46a: 1. d4 Nf6 2. Nf3
			variation => N__('Indian: 2. Nf3'),
			moves => {
				'b7b5' => 1,
				'b7b6' => 1,
				'c7c5' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'f6e4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/8/3Pn3/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46a',
			# TRANSLATORS: A46a: 1. d4 Nf6 2. Nf3 Ne4
			variation => N__('Indian: Döry Defence'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/5n2/1p6/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46b',
			# TRANSLATORS: A46b: 1. d4 Nf6 2. Nf3 b5
			variation => N__('Indian: 2. Nf3 b5'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/5n2/1p6/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A46b',
			# TRANSLATORS: A46b: 1. d4 Nf6 2. Nf3 b5 3. g3
			variation => N__('Indian: 2. Nf3 b5 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46c',
			# TRANSLATORS: A46c: 1. d4 Nf6 2. Nf3 c5
			variation => N__('Neo-Benoni'),
			moves => {
				'c2c3' => 1,
				'd4c5' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2P5/8/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A46c',
			# TRANSLATORS: A46c: 1. d4 Nf6 2. Nf3 c5 3. dxc5
			variation => N__('Neo-Benoni 3.dxc5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/3P4/4PN2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A46c',
			# TRANSLATORS: A46c: 1. d4 Nf6 2. Nf3 c5 3. e3
			variation => N__('Neo-Benoni: 3. e3'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/3p4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A46c',
			# TRANSLATORS: A46c: 1. d4 Nf6 2. Nf3 c5 3. e3 cxd4
			variation => N__('Neo-Benoni: 3. e3 cxd4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/3P4/2P2N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A46d',
			# TRANSLATORS: A46d: 1. d4 Nf6 2. Nf3 c5 3. c3
			variation => N__('Neo-Benoni: 3. c3'),
			moves => {
				'b7b6' => 1,
				'c5d4' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/1p3n2/2p5/3P4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46d',
			# TRANSLATORS: A46d: 1. d4 Nf6 2. Nf3 c5 3. c3 b6
			variation => N__('Neo-Benoni: 3. c3 b6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/3p4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46d',
			# TRANSLATORS: A46d: 1. d4 Nf6 2. Nf3 c5 3. c3 cxd4
			variation => N__('Neo-Benoni: 3. c3 cxd4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/3P4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46d',
			# TRANSLATORS: A46d: 1. d4 Nf6 2. Nf3 c5 3. c3 g6
			variation => N__('Neo-Benoni: 3. c3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46e',
			# TRANSLATORS: A46e: 1. d4 Nf6 2. Nf3 c5 3. c3 e6
			variation => N__('Neo-Benoni: 3. c3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A46f',
			# TRANSLATORS: A46f: 1. d4 Nf6 2. Nf3 c5 3. g3
			variation => N__('Neo-Benoni: 3. g3'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/3p4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A46f',
			# TRANSLATORS: A46f: 1. d4 Nf6 2. Nf3 c5 3. g3 cxd4
			variation => N__('Neo-Benoni: 3. g3 cxd4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/3N4/6P1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A46f',
			# TRANSLATORS: A46f: 1. d4 Nf6 2. Nf3 c5 3. g3 cxd4 4. Nxd4
			variation => N__('Neo-Benoni: 3. g3 cxd4 4.Nxd4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46g',
			# TRANSLATORS: A46g: 1. d4 Nf6 2. Nf3 d6
			variation => N__('Indian: 2. Nf3 d6'),
			moves => {
				'c1g5' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A46h',
			# TRANSLATORS: A46h: 1. d4 Nf6 2. Nf3 d6 3. g3
			variation => N__('Indian: 2. Nf3 d6 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46i',
			# TRANSLATORS: A46i: 1. d4 Nf6 2. Nf3 d6 3. Bg5
			variation => N__('Indian: 2. Nf3 d6 3. Bg5'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46i',
			# TRANSLATORS: A46i: 1. d4 Nf6 2. Nf3 d6 3. Bg5 Nbd7
			variation => N__('Indian: 2. Nf3 d6 3. Bg5 Nbd7'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46j',
			# TRANSLATORS: A46j: 1. d4 Nf6 2. Nf3 e6
			variation => N__('Indian: 2. Nf3 e6'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'c2c3' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P4/2P2N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A46j',
			# TRANSLATORS: A46j: 1. d4 Nf6 2. Nf3 e6 3. c3
			variation => N__('Indian: 2. Nf3 e6 3. c3'),
			moves => {
				'b7b6' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A46j',
			# TRANSLATORS: A46j: 1. d4 Nf6 2. Nf3 e6 3. c3 b6
			variation => N__('Indian: 2. Nf3 e6 3. c3 b6'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/6B1/3P4/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46j',
			# TRANSLATORS: A46j: 1. d4 Nf6 2. Nf3 e6 3. c3 b6 4. Bg5
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. c3 b6 4. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P4/4PN2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A46k',
			# TRANSLATORS: A46k: 1. d4 Nf6 2. Nf3 e6 3. e3
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. e3'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A46k',
			# TRANSLATORS: A46k: 1. d4 Nf6 2. Nf3 e6 3. e3 c5
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. e3 c5'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P4/3BPN2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'A46l',
			# TRANSLATORS: A46l: 1. d4 Nf6 2. Nf3 e6 3. e3 c5 4. Bd3
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. e3 c5 4. Bd3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A46m',
			# TRANSLATORS: A46m: 1. d4 Nf6 2. Nf3 e6 3. g3
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3'),
			moves => {
				'b7b5' => 1,
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/4pn2/1p6/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A46m',
			# TRANSLATORS: A46m: 1. d4 Nf6 2. Nf3 e6 3. g3 b5
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 b5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/4pn2/1p6/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A46m',
			# TRANSLATORS: A46m: 1. d4 Nf6 2. Nf3 e6 3. g3 b5 4. Bg2
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 b5'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/4pn2/1p6/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A46m',
			# TRANSLATORS: A46m: 1. d4 Nf6 2. Nf3 e6 3. g3 b5 4. Bg2 Bb7
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 b5 4. Bg2 Bb7'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/4pn2/1p6/3P4/5NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A46n',
			# TRANSLATORS: A46n: 1. d4 Nf6 2. Nf3 e6 3. g3 b5 4. Bg2 Bb7 5. O-O
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 b5 4. Bg2 Bb7 5.O-O'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2'),
			moves => {
				'b8c6' => 1,
				'c5d4' => 1,
				'd8a5' => 1,
				'd8c7' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1pn2/2p5/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Nc6
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/8/3p4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 cxd4
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 cxd4'),
			moves => {
			},
		},
		'rnb1kb1r/pp1p1ppp/4pn2/q1p5/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Qa5+
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Qa5+'),
			moves => {
			},
		},
		'rnb1kb1r/ppqp1ppp/4pn2/2p5/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A46o',
			# TRANSLATORS: A46o: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Qc7
			variation => N__('Indian: 1. d4 Nf6 2. Nf3 e6 3. g3 c5 4. Bg2 Qc7'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46p',
			# TRANSLATORS: A46p: 1. d4 Nf6 2. Nf3 e6 3. Bf4
			variation => N__('Indian: London System'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P1B2/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46p',
			# TRANSLATORS: A46p: 1. d4 Nf6 2. Nf3 e6 3. Bf4 c5
			variation => N__('Indian: London, 3... c5'),
			moves => {
				'c2c3' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P1B2/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A46p',
			# TRANSLATORS: A46p: 1. d4 Nf6 2. Nf3 e6 3. Bf4 c5 4. e3
			variation => N__('Indian: London, 3... c5 4. e3'),
			moves => {
				'd8b6' => 1,
			},
		},
		'rnb1kb1r/pp1p1ppp/1q2pn2/2p5/3P1B2/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46p',
			# TRANSLATORS: A46p: 1. d4 Nf6 2. Nf3 e6 3. Bf4 c5 4. e3 Qb6
			variation => N__('Indian: London, 3... c5 4. e3 Qb6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/3P1B2/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46q',
			# TRANSLATORS: A46q: 1. d4 Nf6 2. Nf3 e6 3. Bf4 c5 4. c3
			variation => N__('Indian: London, 3... c5 4. c3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/4pn2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46r',
			# TRANSLATORS: A46r: 1. d4 Nf6 2. Nf3 e6 3. Bg5
			variation => N__('Torre Attack'),
			moves => {
				'b7b6' => 1,
				'c7c5' => 1,
				'f8e7' => 1,
				'h7h6' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46r',
			# TRANSLATORS: A46r: 1. d4 Nf6 2. Nf3 e6 3. Bg5 b6
			variation => N__('Torre Attack: 3... b6'),
			moves => {
			},
		},
		'rnbqk2r/ppppbppp/4pn2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46r',
			# TRANSLATORS: A46r: 1. d4 Nf6 2. Nf3 e6 3. Bg5 Be7
			variation => N__('Torre Attack: 3... Be7'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p3B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46s',
			# TRANSLATORS: A46s: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5
			variation => N__('Torre Attack: 3... c5'),
			moves => {
				'c2c3' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p3B1/3PP3/5N2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A46s',
			# TRANSLATORS: A46s: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e4
			variation => N__('Torre Attack: Wagner Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p3B1/3P4/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46t',
			# TRANSLATORS: A46t: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. c3
			variation => N__('Torre Attack: 3... c5 4. c3'),
			moves => {
				'd8b6' => 1,
				'h7h6' => 1,
			},
		},
		'rnb1kb1r/pp1p1ppp/1q2pn2/2p3B1/3P4/2P2N2/PP2PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46t',
			# TRANSLATORS: A46t: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. c3 Qb6
			variation => N__('Torre Attack: 3... c5 4. c3 Qb6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1pp1/4pn1p/2p3B1/3P4/2P2N2/PP2PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46t',
			# TRANSLATORS: A46t: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. c3 h6
			variation => N__('Torre Attack: 3... c5 4. c3 h6'),
			moves => {
				'g5h4' => 1,
			},
		},
		'rnbqkb1r/pp1p1pp1/4pn1p/2p5/3P3B/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46t',
			# TRANSLATORS: A46t: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. c3 h6 5. Bh4
			variation => N__('Torre Attack: 3... c5 4. c3 h6 5. Bh4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p3B1/3P4/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A46u',
			# TRANSLATORS: A46u: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3
			variation => N__('Torre Attack: 3... c5 4. e3'),
			moves => {
				'c5d4' => 1,
				'd8b6' => 1,
				'f8e7' => 1,
				'h7h6' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/6B1/3p4/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46u',
			# TRANSLATORS: A46u: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3 cxd4
			variation => N__('Torre Attack: 3... c5 4. e3 cxd4'),
			moves => {
			},
		},
		'rnb1kb1r/pp1p1ppp/1q2pn2/2p3B1/3P4/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46u',
			# TRANSLATORS: A46u: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3 Qb6
			variation => N__('Torre Attack: 3... c5 4. e3 Qb6'),
			moves => {
			},
		},
		'rnbqk2r/pp1pbppp/4pn2/2p3B1/3P4/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46v',
			# TRANSLATORS: A46v: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3 Be7
			variation => N__('Torre Attack: 3... c5 4. e3 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1pp1/4pn1p/2p3B1/3P4/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46w',
			# TRANSLATORS: A46w: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3 h6
			variation => N__('Torre Attack: 3... c5 4. e3 h6'),
			moves => {
				'g5h4' => 1,
			},
		},
		'rnbqkb1r/pp1p1pp1/4pn1p/2p5/3P3B/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A46w',
			# TRANSLATORS: A46w: 1. d4 Nf6 2. Nf3 e6 3. Bg5 c5 4. e3 h6 5. Bh4
			variation => N__('Torre Attack: 3... c5 4. e3 h6 5. Bh4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1pp1/4pn1p/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46x',
			# TRANSLATORS: A46x: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6
			variation => N__('Torre Attack: 3... h6'),
			moves => {
				'g5f6' => 1,
				'g5h4' => 1,
			},
		},
		'rnbqkb1r/pppp1pp1/4pn1p/8/3P3B/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46x',
			# TRANSLATORS: A46x: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bh4
			variation => N__('Torre Attack: 3... h6 4. Bh4'),
			moves => {
				'g7g5' => 1,
			},
		},
		'rnbqkb1r/pppp1p2/4pn1p/6p1/3P3B/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46x',
			# TRANSLATORS: A46x: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bh4 g5
			variation => N__('Torre Attack: 3... h6 4. Bh4 g5'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1pp1/4pB1p/8/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A46y',
			# TRANSLATORS: A46y: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6
			variation => N__('Torre Attack: 3... h6 4.Bxf6'),
			moves => {
				'd8f6' => 1,
			},
		},
		'rnb1kb1r/pppp1pp1/4pq1p/8/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A46y',
			# TRANSLATORS: A46y: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6
			variation => N__('Torre Attack: 3... h6 4.Bxf6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnb1kb1r/pppp1pp1/4pq1p/8/3PP3/5N2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A46y',
			# TRANSLATORS: A46y: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4'),
			moves => {
				'b7b6' => 1,
				'c7c5' => 1,
				'd7d6' => 1,
			},
		},
		'rnb1kb1r/p1pp1pp1/1p2pq1p/8/3PP3/5N2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46y',
			# TRANSLATORS: A46y: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 b6
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 b6'),
			moves => {
			},
		},
		'rnb1kb1r/pp1p1pp1/4pq1p/2p5/3PP3/5N2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46y',
			# TRANSLATORS: A46y: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 c5
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 c5'),
			moves => {
			},
		},
		'rnb1kb1r/ppp2pp1/3ppq1p/8/3PP3/5N2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A46z',
			# TRANSLATORS: A46z: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnb1kb1r/ppp2pp1/3ppq1p/8/3PP3/2N2N2/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'A46z',
			# TRANSLATORS: A46z: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3'),
			moves => {
				'b8d7' => 1,
				'g7g6' => 1,
			},
		},
		'r1b1kb1r/pppn1pp1/3ppq1p/8/3PP3/2N2N2/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'A46z',
			# TRANSLATORS: A46z: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7'),
			moves => {
				'd1d2' => 1,
			},
		},
		'r1b1kb1r/pppn1pp1/3ppq1p/8/3PP3/2N2N2/PPPQ1PPP/R3KB1R b KQkq' => {
			eco => 'A46z',
			# TRANSLATORS: A46z: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7 7. Qd2
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3 Nd7 7. Qd2'),
			moves => {
			},
		},
		'rnb1kb1r/ppp2p2/3ppqpp/8/3PP3/2N2N2/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'A46z',
			# TRANSLATORS: A46z: 1. d4 Nf6 2. Nf3 e6 3. Bg5 h6 4. Bxf6 Qxf6 5. e4 d6 6. Nc3 g6
			variation => N__('Torre Attack: 3... h6 4.Bxf6 Qxf6 5. e4 d6 6. Nc3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 Bb7
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P1B2/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 Bb7 4. e3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/3P1B2/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 Bb7 4. e3 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P1B2/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 e6 4. e3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/p2p1ppp/1p2pn2/2p5/3P1B2/4PN2/PPP2PPP/RN1QKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bf4 e6 4. e3 c5
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bg5
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. Bg5 Bb7
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/3P4/4PN2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 Bb7
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P4/3BPN2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 Bb7 4. Bd3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/3P4/3BPN2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 Bb7 4. Bd3 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/3P4/3BPN2/PPP2PPP/RNBQ1RK1 b kq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 Bb7 4. Bd3 e6 5. O-O
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rn1qkb1r/pb1p1ppp/1p2pn2/2p5/3P4/3BPN2/PPP2PPP/RNBQ1RK1 w kq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 Bb7 4. Bd3 e6 5. O-O c5
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P4/3BPN2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 e6 4. Bd3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/p2p1ppp/1p2pn2/2p5/3P4/3BPN2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. e3 e6 4. Bd3 c5
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
				'c7c5' => 1,
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/2p5/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2 c5
			variation => N__('Neo-Queen\'s Indian: Marienbad System'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/2p5/2PP4/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2 c5 5. c4
			variation => N__('Neo-Queen\'s Indian: Marienbad System'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/8/2Pp4/5NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2 c5 5. c4 cxd4
			variation => N__('Neo-Queen\'s Indian: Marienbad System'),
			moves => {
				'd1d4' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/1p3n2/8/2PQ4/5NP1/PP2PPBP/RNB1K2R b KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2 c5 5. c4 cxd4 6. Qxd4
			variation => N__('Neo-Queen\'s Indian: Marienbad System, Berg Variation'),
			moves => {
			},
		},
		'rn1qkb1r/pbpp1ppp/1p2pn2/8/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 Bb7 4. Bg2 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/p1pp1ppp/1p2pn2/8/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A47',
			# TRANSLATORS: A47: 1. d4 Nf6 2. Nf3 b6 3. g3 e6
			variation => N__('Neo-Queen\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A48a',
			# TRANSLATORS: A48a: 1. d4 Nf6 2. Nf3 g6
			variation => N__('Neo-King\'s Indian'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'c1f4' => 1,
				'c1g5' => 1,
				'c2c3' => 1,
				'e2e3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/5N2/PPPNPPPP/R1BQKB1R b KQkq' => {
			eco => 'A48a',
			# TRANSLATORS: A48a: 1. d4 Nf6 2. Nf3 g6 3. Nbd2
			variation => N__('Neo-King\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/2P2N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A48b',
			# TRANSLATORS: A48b: 1. d4 Nf6 2. Nf3 g6 3. c3
			variation => N__('Neo-King\'s Indian'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P4/2P2N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A48b',
			# TRANSLATORS: A48b: 1. d4 Nf6 2. Nf3 g6 3. c3 Bg7
			variation => N__('Neo-King\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/4PN2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'A48c',
			# TRANSLATORS: A48c: 1. d4 Nf6 2. Nf3 g6 3. e3
			variation => N__('Neo-King\'s Indian'),
			moves => {
				'c7c5' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A48c',
			# TRANSLATORS: A48c: 1. d4 Nf6 2. Nf3 g6 3. e3 Bg7
			variation => N__('Neo-King\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/3P4/4PN2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'A48d',
			# TRANSLATORS: A48d: 1. d4 Nf6 2. Nf3 g6 3. e3 c5
			variation => N__('Neo-King\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/2N2N2/PPP1PPPP/R1BQKB1R b KQkq' => {
			eco => 'A48e',
			# TRANSLATORS: A48e: 1. d4 Nf6 2. Nf3 g6 3. Nc3
			variation => N__('Neo-King\'s Indian'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A48f',
			# TRANSLATORS: A48f: 1. d4 Nf6 2. Nf3 g6 3. Bf4
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P1B2/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A48g',
			# TRANSLATORS: A48g: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'b1d2' => 1,
				'c2c3' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P1B2/5N2/PPPNPPPP/R2QKB1R b KQkq' => {
			eco => 'A48h',
			# TRANSLATORS: A48h: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. Nbd2
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P1B2/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A48i',
			# TRANSLATORS: A48i: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. c3
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P1B2/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A48j',
			# TRANSLATORS: A48j: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. e3
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/3P1B2/4PN2/PPP2PPP/RN1QKB1R w KQ' => {
			eco => 'A48j',
			# TRANSLATORS: A48j: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. e3 O-O
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/3P1B2/4PN2/PPP1BPPP/RN1QK2R b KQ' => {
			eco => 'A48k',
			# TRANSLATORS: A48k: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. e3 O-O 5. Be2
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3P1B2/4PN2/PPP1BPPP/RN1QK2R w KQ' => {
			eco => 'A48k',
			# TRANSLATORS: A48k: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. e3 O-O 5. Be2 d6
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
				'h2h3' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3P1B2/4PN1P/PPP1BPP1/RN1QK2R b KQ' => {
			eco => 'A48l',
			# TRANSLATORS: A48l: 1. d4 Nf6 2. Nf3 g6 3. Bf4 Bg7 4. e3 O-O 5. Be2 d6 6. h3
			variation => N__('Neo-King\'s Indian: London System'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A48m',
			# TRANSLATORS: A48m: 1. d4 Nf6 2. Nf3 g6 3. Bg5
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'f6e4' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/6p1/6B1/3Pn3/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A48n',
			# TRANSLATORS: A48n: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Ne4
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/6B1/3P4/5N2/PPP1PPPP/RN1QKB1R w KQkq' => {
			eco => 'A48o',
			# TRANSLATORS: A48o: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'c2c3' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/6B1/3P4/2P2N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A48o',
			# TRANSLATORS: A48o: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. c3
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/6B1/3P4/4PN2/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'A48o',
			# TRANSLATORS: A48o: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. e3
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/6B1/3P4/2N2N2/PPP1PPPP/R2QKB1R b KQkq' => {
			eco => 'A48p',
			# TRANSLATORS: A48p: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nc3
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbqk2r/ppppppbp/5np1/6B1/3P4/5N2/PPPNPPPP/R2QKB1R b KQkq' => {
			eco => 'A48q',
			# TRANSLATORS: A48q: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/6B1/3P4/5N2/PPPNPPPP/R2QKB1R w KQ' => {
			eco => 'A48r',
			# TRANSLATORS: A48r: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'c2c3' => 1,
				'e2e4' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/6B1/3PP3/5N2/PPPN1PPP/R2QKB1R b KQ' => {
			eco => 'A48s',
			# TRANSLATORS: A48s: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. e4
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'd7d5' => 1,
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/6B1/3PP3/5N2/PPPN1PPP/R2QKB1R w KQ' => {
			eco => 'A48t',
			# TRANSLATORS: A48t: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. e4 d6
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/5np1/3p2B1/3PP3/5N2/PPPN1PPP/R2QKB1R w KQ' => {
			eco => 'A48u',
			# TRANSLATORS: A48u: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. e4 d5
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbq1rk1/ppppppbp/5np1/6B1/3P4/2P2N2/PP1NPPPP/R2QKB1R b KQ' => {
			eco => 'A48v',
			# TRANSLATORS: A48v: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. c3
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/6B1/3P4/2P2N2/PP1NPPPP/R2QKB1R w KQ' => {
			eco => 'A48v',
			# TRANSLATORS: A48v: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. c3 d6
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/6B1/3PP3/2P2N2/PP1N1PPP/R2QKB1R b KQ' => {
			eco => 'A48w',
			# TRANSLATORS: A48w: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. c3 d6 6. e4
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/3p1np1/2p3B1/3PP3/2P2N2/PP1N1PPP/R2QKB1R w KQ' => {
			eco => 'A48x',
			# TRANSLATORS: A48x: 1. d4 Nf6 2. Nf3 g6 3. Bg5 Bg7 4. Nbd2 O-O 5. c3 d6 6. e4 c5
			variation => N__('Neo-King\'s Indian: Torre Attack'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppppppbp/5np1/8/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/3P4/5NP1/PPP1PPBP/RNBQK2R w KQ' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'b2b3' => 1,
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/3P4/1P3NP1/P1P1PPBP/RNBQK2R b KQ' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O 5. b3
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3P4/1P3NP1/P1P1PPBP/RNBQK2R w KQ' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O 5. b3 d6
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3P4/1P3NP1/PBP1PPBP/RN1QK2R b KQ' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O 5. b3 d6 6. Bb2
			variation => N__('Neo-King\'s Indian: Double Fianchetto System'),
			moves => {
			},
		},
		'rnbq1rk1/ppppppbp/5np1/8/3P4/5NP1/PPP1PPBP/RNBQ1RK1 b -' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O 5. O-O
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3P4/5NP1/PPP1PPBP/RNBQ1RK1 w -' => {
			eco => 'A49',
			# TRANSLATORS: A49: 1. d4 Nf6 2. Nf3 g6 3. g3 Bg7 4. Bg2 O-O 5. O-O d6
			variation => N__('Neo-King\'s Indian: Fianchetto System'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A50a',
			# TRANSLATORS: A50a: 1. d4 Nf6 2. c4
			variation => N__('Indian: 2. c4'),
			moves => {
				'a7a6' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pppppppp/2n2n2/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A50b',
			# TRANSLATORS: A50b: 1. d4 Nf6 2. c4 Nc6
			variation => N__('Indian: Mexican Defence (Two Knights Tango)'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pppppppp/2n2n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A50c',
			# TRANSLATORS: A50c: 1. d4 Nf6 2. c4 Nc6 3. Nc3
			variation => N__('Indian: Mexican Defence, 3. Nc3'),
			moves => {
			},
		},
		'r1bqkb1r/pppppppp/2n2n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A50d',
			# TRANSLATORS: A50d: 1. d4 Nf6 2. c4 Nc6 3. Nf3
			variation => N__('Indian: Mexican Defence, 3. Nf3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/ppp1pppp/2np1n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A50d',
			# TRANSLATORS: A50d: 1. d4 Nf6 2. c4 Nc6 3. Nf3 d6
			variation => N__('Indian: Mexican Defence, 3. Nf3 d6'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n1pn2/8/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A50e',
			# TRANSLATORS: A50e: 1. d4 Nf6 2. c4 Nc6 3. Nf3 e6
			variation => N__('Indian: Mexican Defence, 3. Nf3 e6'),
			moves => {
				'a2a3' => 1,
				'b1c3' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n1pn2/8/2PP4/P4N2/1P2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A50f',
			# TRANSLATORS: A50f: 1. d4 Nf6 2. c4 Nc6 3. Nf3 e6 4. a3
			variation => N__('Indian: Mexican Defence, 3. Nf3 e6 4. a3'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n1pn2/8/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A50g',
			# TRANSLATORS: A50g: 1. d4 Nf6 2. c4 Nc6 3. Nf3 e6 4. Nc3
			variation => N__('Indian: Mexican Defence, 3. Nf3 e6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A50h',
			# TRANSLATORS: A50h: 1. d4 Nf6 2. c4 c6
			variation => N__('Indian: Slav-Indian'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A50i',
			# TRANSLATORS: A50i: 1. d4 Nf6 2. c4 c6 3. Nf3
			variation => N__('Indian: Slav-Indian, 3. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A50j',
			# TRANSLATORS: A50j: 1. d4 Nf6 2. c4 c6 3. Nc3
			variation => N__('Indian: Slav-Indian, 3. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/1ppppppp/p4n2/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A50k',
			# TRANSLATORS: A50k: 1. d4 Nf6 2. c4 a6
			variation => N__('Indian: 2. c4 a6'),
			moves => {
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A50l',
			# TRANSLATORS: A50l: 1. d4 Nf6 2. c4 b6
			variation => N__('Indian: Queen\'s Indian Accelerated'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/1p3n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A50l',
			# TRANSLATORS: A50l: 1. d4 Nf6 2. c4 b6 3. Nc3
			variation => N__('Indian: Queen\'s Indian Accelerated'),
			moves => {
				'c8b7' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A50m',
			# TRANSLATORS: A50m: 1. d4 Nf6 2. c4 b6 3. Nc3 Bb7
			variation => N__('Indian: Queen\'s Indian Accelerated'),
			moves => {
				'd1c2' => 1,
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/2PP4/2N5/PPQ1PPPP/R1B1KBNR b KQkq' => {
			eco => 'A50n',
			# TRANSLATORS: A50n: 1. d4 Nf6 2. c4 b6 3. Nc3 Bb7 4. Qc2
			variation => N__('Indian: Queen\'s Indian Accelerated'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A51a',
			# TRANSLATORS: A51a: 1. d4 Nf6 2. c4 e5
			variation => N__('Budapest'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4p3/2PP4/4P3/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A51b',
			# TRANSLATORS: A51b: 1. d4 Nf6 2. c4 e5 3. e3
			variation => N__('Budapest: 3. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/3Pp3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A51c',
			# TRANSLATORS: A51c: 1. d4 Nf6 2. c4 e5 3. d5
			variation => N__('Budapest: 3. d5'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/5n2/4P3/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A51d',
			# TRANSLATORS: A51d: 1. d4 Nf6 2. c4 e5 3. dxe5
			variation => N__('Budapest: 3.dxe5'),
			moves => {
				'f6e4' => 1,
				'f6g4' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1n3/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A51d',
			# TRANSLATORS: A51d: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4
			variation => N__('Budapest: Fajarowicz'),
			moves => {
				'a2a3' => 1,
				'b1d2' => 1,
				'd1c2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1n3/8/PPQ1PPPP/RNB1KBNR b KQkq' => {
			eco => 'A51e',
			# TRANSLATORS: A51e: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Qc2
			variation => N__('Budapest: Fajarowicz, Steiner Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1n3/8/PP1NPPPP/R1BQKBNR b KQkq' => {
			eco => 'A51f',
			# TRANSLATORS: A51f: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Nd2
			variation => N__('Budapest: Fajarowicz, 4. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1n3/P7/1P2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A51g',
			# TRANSLATORS: A51g: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. a3
			variation => N__('Budapest: Fajarowicz, 4. a3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1n3/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A51h',
			# TRANSLATORS: A51h: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Nf3
			variation => N__('Budapest: Fajarowicz, 4. Nf3'),
			moves => {
				'b8c6' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/8/4P3/1bP1n3/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A51i',
			# TRANSLATORS: A51i: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Nf3 Bb4+
			variation => N__('Budapest: Fajarowicz, 4. Nf3 Bb4+'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n5/4P3/2P1n3/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A51j',
			# TRANSLATORS: A51j: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Nf3 Nc6
			variation => N__('Budapest: Fajarowicz, 4. Nf3 Nc6'),
			moves => {
				'a2a3' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n5/4P3/2P1n3/P4N2/1P2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A51k',
			# TRANSLATORS: A51k: 1. d4 Nf6 2. c4 e5 3. dxe5 Ne4 4. Nf3 Nc6 5. a3
			variation => N__('Budapest: Fajarowicz, 4. Nf3 Nc6 5. a3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P3n1/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A52a',
			# TRANSLATORS: A52a: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4
			variation => N__('Budapest: 3... Ng4'),
			moves => {
				'c1f4' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P3n1/4P3/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A52b',
			# TRANSLATORS: A52b: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e3
			variation => N__('Budapest: 3... Ng4 4. e3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P1P1n1/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A52c',
			# TRANSLATORS: A52c: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e4
			variation => N__('Budapest: Alekhine Variation'),
			moves => {
				'd7d6' => 1,
				'g4e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p4/4P3/2P1P1n1/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A52c',
			# TRANSLATORS: A52c: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e4 d6
			variation => N__('Budapest: Alekhine, Balogh Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4n3/2P1P3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A52c',
			# TRANSLATORS: A52c: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e4 Nxe5
			variation => N__('Budapest: Alekhine Variation'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/8/4n3/2P1PP2/8/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'A52c',
			# TRANSLATORS: A52c: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e4 Nxe5 5. f4
			variation => N__('Budapest: Alekhine Variation'),
			moves => {
				'e5c6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/2n5/8/2P1PP2/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'A52d',
			# TRANSLATORS: A52d: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. e4 Nxe5 5. f4 Nec6
			variation => N__('Budapest: Alekhine, Abonyi Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P3n1/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A52e',
			# TRANSLATORS: A52e: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Nf3
			variation => N__('Budapest: Adler Variation'),
			moves => {
				'f8c5' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/8/2b1P3/2P3n1/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A52f',
			# TRANSLATORS: A52f: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Nf3 Bc5
			variation => N__('Budapest: Adler, 4... Bc5'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/8/2b1P3/2P3n1/4PN2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'A52f',
			# TRANSLATORS: A52f: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Nf3 Bc5 5. e3
			variation => N__('Budapest: Adler, 4... Bc5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n5/2b1P3/2P3n1/4PN2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'A52f',
			# TRANSLATORS: A52f: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Nf3 Bc5 5. e3 Nc6
			variation => N__('Budapest: Adler, 4... Bc5'),
			moves => {
				'f1e2' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n5/2b1P3/2P3n1/4PN2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'A52g',
			# TRANSLATORS: A52g: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Nf3 Bc5 5. e3 Nc6 6. Be2
			variation => N__('Budapest: Adler, 4... Bc5 5. e3 Nc6 6. Be2'),
			moves => {
			},
		},
		'rnbqkb1r/pppp1ppp/8/4P3/2P2Bn1/8/PP2PPPP/RN1QKBNR b KQkq' => {
			eco => 'A52h',
			# TRANSLATORS: A52h: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4
			variation => N__('Budapest: Rubinstein Variation'),
			moves => {
				'b8c6' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqk2r/pppp1ppp/8/4P3/1bP2Bn1/8/PP2PPPP/RN1QKBNR w KQkq' => {
			eco => 'A52i',
			# TRANSLATORS: A52i: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Bb4+
			variation => N__('Budapest: Rubinstein, 4... Bb4+'),
			moves => {
			},
		},
		'r1bqkb1r/pppp1ppp/2n5/4P3/2P2Bn1/8/PP2PPPP/RN1QKBNR w KQkq' => {
			eco => 'A52j',
			# TRANSLATORS: A52j: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6
			variation => N__('Budapest: Rubinstein, 4... Nc6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pppp1ppp/2n5/4P3/2P2Bn1/5N2/PP2PPPP/RN1QKB1R b KQkq' => {
			eco => 'A52j',
			# TRANSLATORS: A52j: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3
			variation => N__('Budapest: Rubinstein, 4... Nc6'),
			moves => {
				'f8b4' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n5/4P3/1bP2Bn1/5N2/PP2PPPP/RN1QKB1R w KQkq' => {
			eco => 'A52k',
			# TRANSLATORS: A52k: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3 Bb4+
			variation => N__('Budapest: Rubinstein, Main Line'),
			moves => {
				'b1d2' => 1,
			},
		},
		'r1bqk2r/pppp1ppp/2n5/4P3/1bP2Bn1/5N2/PP1NPPPP/R2QKB1R b KQkq' => {
			eco => 'A52l',
			# TRANSLATORS: A52l: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3 Bb4+ 6. Nbd2
			variation => N__('Budapest: Rubinstein, Main Line, 6.Nbd2'),
			moves => {
				'd8e7' => 1,
			},
		},
		'r1b1k2r/ppppqppp/2n5/4P3/1bP2Bn1/5N2/PP1NPPPP/R2QKB1R w KQkq' => {
			eco => 'A52l',
			# TRANSLATORS: A52l: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3 Bb4+ 6. Nbd2 Qe7
			variation => N__('Budapest: Rubinstein, Main Line, 6.Nbd2'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1b1k2r/ppppqppp/2n5/4P3/1bP2Bn1/4PN2/PP1N1PPP/R2QKB1R b KQkq' => {
			eco => 'A52m',
			# TRANSLATORS: A52m: 1. d4 Nf6 2. c4 e5 3. dxe5 Ng4 4. Bf4 Nc6 5. Nf3 Bb4+ 6. Nbd2 Qe7 7. e3
			variation => N__('Budapest: Rubinstein, Main Line, 6.Nbd2 Qe7 7. e3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6
			variation => N__('Old Indian'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3
			variation => N__('Old Indian: 3. Nc3'),
			moves => {
				'b8d7' => 1,
				'c7c6' => 1,
				'c8f5' => 1,
				'e7e5' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7
			variation => N__('Old Indian: 3. Nc3 Nbd7'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/8/2PPP3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7 4. e4
			variation => N__('Old Indian: 3. Nc3 Nbd7 4. e4'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PPP3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7 4. e4 e5
			variation => N__('Old Indian: 3. Nc3 Nbd7 4. e4 e5'),
			moves => {
				'd4d5' => 1,
				'g1e2' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/3Pp3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7 4. e4 e5 5. d5
			variation => N__('Old Indian: 3. Nc3 Nbd7 4. e4 e5 5. d5'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PPP3/2N5/PP2NPPP/R1BQKB1R b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7 4. e4 e5 5. Nge2
			variation => N__('Old Indian: 3. Nc3 Nbd7 4. e4 e5 5.Nge2'),
			moves => {
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/8/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Nbd7 4. Nf3
			variation => N__('Old Indian: 3. Nc3 Nbd7 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2pp1n2/8/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 c6
			variation => N__('Old Indian: 3. Nc3 c6'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/5b2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 Bf5
			variation => N__('Old Indian: Janowski'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5
			variation => N__('Old Indian: Ukrainian'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/3Pp3/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. d5
			variation => N__('Old Indian: Ukrainian, 4. d5'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rnbqk2r/ppp1bppp/3p1n2/3Pp3/2P5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. d5 Be7
			variation => N__('Old Indian: Ukrainian, 4. d5 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4P3/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5
			variation => N__('Old Indian: Ukrainian, 4.dxe5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/4p3/2P5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5 dxe5
			variation => N__('Old Indian: Ukrainian, 4.dxe5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'rnbQkb1r/ppp2ppp/5n2/4p3/2P5/2N5/PP2PPPP/R1B1KBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5 dxe5 5. Qxd8+
			variation => N__('Old Indian: Ukrainian, 4.dxe5'),
			moves => {
				'e8d8' => 1,
			},
		},
		'rnbk1b1r/ppp2ppp/5n2/4p3/2P5/2N5/PP2PPPP/R1B1KBNR w KQ' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5 dxe5 5. Qxd8+ Kxd8
			variation => N__('Old Indian: Ukrainian, Queenswap'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbk1b1r/ppp2ppp/5n2/4p3/2P5/2N2N2/PP2PPPP/R1B1KB1R b KQ' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5 dxe5 5. Qxd8+ Kxd8 6. Nf3
			variation => N__('Old Indian: Ukrainian, Queenswap, 6. Nf3'),
			moves => {
				'f6d7' => 1,
			},
		},
		'rnbk1b1r/pppn1ppp/8/4p3/2P5/2N2N2/PP2PPPP/R1B1KB1R w KQ' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. dxe5 dxe5 5. Qxd8+ Kxd8 6. Nf3 Nfd7
			variation => N__('Old Indian: Ukrainian, Queenswap, 6. Nf3 Nfd7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/2PP4/2N1P3/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. e3
			variation => N__('Old Indian: Ukranian, 4. e3'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PP4/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. e3 Nbd7
			variation => N__('Old Indian: Ukranian, 4. e3'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PP4/2NBP3/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. e3 Nbd7 5. Bd3
			variation => N__('Old Indian: Dus-Khotimirsky'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/2PPP3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. e4
			variation => N__('Old Indian: Ukranian, 4. e4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3
			variation => N__('Old Indian: 3. Nf3'),
			moves => {
				'b8d7' => 1,
				'c7c6' => 1,
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 Nbd7
			variation => N__('Old Indian: 3. Nf3 Nbd7'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2pp1n2/8/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 c6
			variation => N__('Old Indian: 3. Nf3 c6'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/5b2/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 Bf5
			variation => N__('Old Indian: 3. Nf3 Bf5'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/8/2PP2b1/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 Bg4
			variation => N__('Old Indian: 3. Nf3 Bg4'),
			moves => {
				'b1c3' => 1,
				'd1b3' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/8/2PP2b1/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 Bg4 4. Nc3
			variation => N__('Old Indian: 3. Nf3 Bg4 4. Nc3'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p1n2/8/2PP2b1/1Q3N2/PP2PPPP/RNB1KB1R b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. Nf3 Bg4 4. Qb3
			variation => N__('Old Indian: 3. Nf3 Bg4 4. Qb3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/2PP4/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A53',
			# TRANSLATORS: A53: 1. d4 Nf6 2. c4 d6 3. g3
			variation => N__('Old Indian: 3. g3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3
			variation => N__('Old Indian: 4. Nf3'),
			moves => {
				'b8c6' => 1,
				'b8d7' => 1,
				'e5d4' => 1,
				'e5e4' => 1,
			},
		},
		'r1bqkb1r/ppp2ppp/2np1n2/4p3/2PP4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nc6
			variation => N__('Old Indian: 4. Nf3 Nc6'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PP4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7
			variation => N__('Old Indian: 4. Nf3 Nbd7'),
			moves => {
				'c1g5' => 1,
				'e2e3' => 1,
				'e2e4' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p1B1/2PP4/2N2N2/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5
			variation => N__('Old Indian: 5. Bg5'),
			moves => {
				'c7c6' => 1,
				'f8e7' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2pp1n2/4p1B1/2PP4/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 c6
			variation => N__('Old Indian: 5. Bg5 c6'),
			moves => {
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p1B1/2PP4/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7
			variation => N__('Old Indian: 5. Bg5 Be7'),
			moves => {
				'e2e3' => 1,
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3
			variation => N__('Old Indian: 5. Bg5 Be7 6. e3'),
			moves => {
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p1B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 c6
			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 c6'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PP3PPP/R2QKB1R w KQ' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O
			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O'),
			moves => {
				'd1c2' => 1,
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R b KQ' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O 7. Qc2
			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O 7. Qc2'),
			moves => {
				'c7c6' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p1B1/2PP4/2N1PN2/PPQ2PPP/R3KB1R w KQ' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. Bg5 Be7 6. e3 O-O 7. Qc2 c6
			variation => N__('Old Indian: 5. Bg5 Be7 6. e3 O-O 7. Qc2 c6'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PP4/2N1PN2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e3
			variation => N__('Old Indian: 5. e3'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PP4/2N2NP1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3
			variation => N__('Old Indian: 5. g3'),
			moves => {
				'c7c6' => 1,
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2pp1n2/4p3/2PP4/2N2NP1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 c6
			variation => N__('Old Indian: 5. g3 c6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2pp1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 c6 6. Bg2
			variation => N__('Old Indian: 5. g3 c6'),
			moves => {
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7
			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2
			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O
			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O
			variation => N__('Old Indian: 5. g3 Be7'),
			moves => {
				'c7c6' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O c6
			variation => N__('Old Indian: 5. g3 Be7, Main Line'),
			moves => {
				'd1c2' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 Be7 6. Bg2 O-O 7. O-O c6 8. Qc2
			variation => N__('Old Indian: 5. g3 Be7, Main Line, 8. Qc2'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1p1p/3p1np1/4p3/2PP4/2N2NP1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 g6
			variation => N__('Old Indian: 5. g3 g6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqkb1r/pppn1p1p/3p1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 g6 6. Bg2
			variation => N__('Old Indian: 5. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk2r/pppn1pbp/3p1np1/4p3/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. g3 g6 6. Bg2 Bg7
			variation => N__('Old Indian: 5. g3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/8/2Pp4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 exd4
			variation => N__('Old Indian: 4. Nf3 exd4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/8/2PPp3/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 e4
			variation => N__('Old Indian: 4. Nf3 e4'),
			moves => {
				'f3g5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/6N1/2PPp3/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A54',
			# TRANSLATORS: A54: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 e4 5. Ng5
			variation => N__('Old Indian: 4. Nf3 e4 5. Ng5'),
			moves => {
			},
		},
		'r1bqkb1r/pppn1ppp/3p1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A55a',
			# TRANSLATORS: A55a: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4
			variation => N__('Old Indian: 5. e4'),
			moves => {
				'c7c6' => 1,
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pppn1p1p/3p1np1/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A55b',
			# TRANSLATORS: A55b: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 g6
			variation => N__('Old Indian: 5. e4 g6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1n1ppp/2pp1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A55c',
			# TRANSLATORS: A55c: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 c6
			variation => N__('Old Indian: 5. e4 c6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'A55c',
			# TRANSLATORS: A55c: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 c6 6. Be2
			variation => N__('Old Indian: 5. e4 c6 6. Be2'),
			moves => {
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A55d',
			# TRANSLATORS: A55d: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7
			variation => N__('Old Indian: 5. e4 Be7'),
			moves => {
				'f1e2' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p3/2PPP3/2N2NP1/PP3P1P/R1BQKB1R b KQkq' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3
			variation => N__('Old Indian: 5. e4 Be7 6. g3'),
			moves => {
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3P1P/R1BQKB1R w KQkq' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R b KQkq' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQK2R w KQ' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 b -' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O'),
			moves => {
				'a7a6' => 1,
				'd8c7' => 1,
				'f8e8' => 1,
			},
		},
		'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w -' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O Qc7
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O Qc7'),
			moves => {
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PPP3/2N2NP1/PP3P1P/R1BQKB1R w KQ' => {
			eco => 'A55e',
			# TRANSLATORS: A55e: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 O-O
			variation => N__('Old Indian: 5. e4 Be7 6. g3 O-O'),
			moves => {
			},
		},
		'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w -' => {
			eco => 'A55f',
			# TRANSLATORS: A55f: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O Re8
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O Re8'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2NP1/PP3PBP/R1BQ1RK1 w -' => {
			eco => 'A55g',
			# TRANSLATORS: A55g: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O a6
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/P1PPP3/2N2NP1/1P3PBP/R1BQ1RK1 b -' => {
			eco => 'A55h',
			# TRANSLATORS: A55h: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8. O-O a6 9. a4
			variation => N__('Old Indian: 5. e4 Be7 6. g3 c6 7. Bg2 O-O 8.O-O a6 9. a4'),
			moves => {
			},
		},
		'r1bqk2r/pppnbppp/3p1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'A55i',
			# TRANSLATORS: A55i: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2
			variation => N__('Old Indian: 5. e4 Be7 6. Be2'),
			moves => {
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pppnbppp/3p1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQ' => {
			eco => 'A55i',
			# TRANSLATORS: A55i: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 O-O
			variation => N__('Old Indian: 5. e4 Be7 6. Be2 O-O'),
			moves => {
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQK2R w KQkq' => {
			eco => 'A55j',
			# TRANSLATORS: A55j: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6
			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 b kq' => {
			eco => 'A55j',
			# TRANSLATORS: A55j: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O
			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6 7.O-O'),
			moves => {
				'a7a6' => 1,
				'e8g8' => 1,
			},
		},
		'r1bqk2r/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w kq' => {
			eco => 'A55j',
			# TRANSLATORS: A55j: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O a6
			variation => N__('Old Indian: 5. e4 Be7 6. Be2 c6 7.O-O a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A55k',
			# TRANSLATORS: A55k: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O
			variation => N__('Old Indian: Main Line'),
			moves => {
				'c1e3' => 1,
				'd1c2' => 1,
				'f1e1' => 1,
				'h2h3' => 1,
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N1P/PP2BPP1/R1BQ1RK1 b -' => {
			eco => 'A55l',
			# TRANSLATORS: A55l: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. h3
			variation => N__('Old Indian: Main Line, 8. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 b -' => {
			eco => 'A55m',
			# TRANSLATORS: A55m: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Be3
			variation => N__('Old Indian: Main Line, 8. Be3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N1BN2/PP2BPPP/R2Q1RK1 w -' => {
			eco => 'A55n',
			# TRANSLATORS: A55n: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Be3 a6
			variation => N__('Old Indian: Main Line, 8. Be3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 b -' => {
			eco => 'A55o',
			# TRANSLATORS: A55o: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2
			variation => N__('Old Indian: Main Line, 8. Qc2'),
			moves => {
				'a7a6' => 1,
				'd8c7' => 1,
				'f8e8' => 1,
			},
		},
		'r1b2rk1/ppqnbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w -' => {
			eco => 'A55o',
			# TRANSLATORS: A55o: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2 Qc7
			variation => N__('Old Indian: Main Line, 8. Qc2 Qc7'),
			moves => {
			},
		},
		'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w -' => {
			eco => 'A55o',
			# TRANSLATORS: A55o: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2 Re8
			variation => N__('Old Indian: Main Line, 8. Qc2 Re8'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PPQ1BPPP/R1B2RK1 w -' => {
			eco => 'A55p',
			# TRANSLATORS: A55p: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Qc2 a6
			variation => N__('Old Indian: Main Line, 8. Qc2 a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 b -' => {
			eco => 'A55q',
			# TRANSLATORS: A55q: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1
			variation => N__('Old Indian: Main Line, 8. Re1'),
			moves => {
				'a7a6' => 1,
				'f8e8' => 1,
			},
		},
		'r1bqr1k1/pp1nbppp/2pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w -' => {
			eco => 'A55r',
			# TRANSLATORS: A55r: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1 Re8
			variation => N__('Old Indian: Main Line, 8. Re1 Re8'),
			moves => {
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP2BPPP/R1BQR1K1 w -' => {
			eco => 'A55s',
			# TRANSLATORS: A55s: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1 a6
			variation => N__('Old Indian: Main Line, 8. Re1 a6'),
			moves => {
				'e2f1' => 1,
			},
		},
		'r1bq1rk1/1p1nbppp/p1pp1n2/4p3/2PPP3/2N2N2/PP3PPP/R1BQRBK1 b -' => {
			eco => 'A55t',
			# TRANSLATORS: A55t: 1. d4 Nf6 2. c4 d6 3. Nc3 e5 4. Nf3 Nbd7 5. e4 Be7 6. Be2 c6 7. O-O O-O 8. Re1 a6 9. Bf1
			variation => N__('Old Indian: Main Line, 8. Re1 a6 9. Bf1'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56a',
			# TRANSLATORS: A56a: 1. d4 Nf6 2. c4 c5
			variation => N__('Benoni: 2... c5'),
			moves => {
				'd4c5' => 1,
				'd4d5' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/2PP4/4P3/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A56b',
			# TRANSLATORS: A56b: 1. d4 Nf6 2. c4 c5 3. e3
			variation => N__('Benoni: 2... c5 3. e3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2p5/2PP4/4P3/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A56b',
			# TRANSLATORS: A56b: 1. d4 Nf6 2. c4 c5 3. e3 e6
			variation => N__('Benoni: 2... c5 3. e3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/2PP4/4P3/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A56c',
			# TRANSLATORS: A56c: 1. d4 Nf6 2. c4 c5 3. e3 g6
			variation => N__('Benoni: 2... c5 3. e3 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2p5/2PP4/2N1P3/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A56d',
			# TRANSLATORS: A56d: 1. d4 Nf6 2. c4 c5 3. e3 g6 4. Nc3
			variation => N__('Benoni: 2... c5 3. e3 g6 4. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2P5/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A56e',
			# TRANSLATORS: A56e: 1. d4 Nf6 2. c4 c5 3. dxc5
			variation => N__('Benoni: 3.dxc5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2pP4/2P5/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A56f',
			# TRANSLATORS: A56f: 1. d4 Nf6 2. c4 c5 3. d5
			variation => N__('Benoni: 3. d5'),
			moves => {
				'a7a6' => 1,
				'b7b5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'f6e4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/1p1ppppp/p4n2/2pP4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56g',
			# TRANSLATORS: A56g: 1. d4 Nf6 2. c4 c5 3. d5 a6
			variation => N__('Benoni: 3. d5 a6'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2pP4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56h',
			# TRANSLATORS: A56h: 1. d4 Nf6 2. c4 c5 3. d5 g6
			variation => N__('Benoni: 3. d5 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1ppp1p/5np1/2pP4/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A56h',
			# TRANSLATORS: A56h: 1. d4 Nf6 2. c4 c5 3. d5 g6 4. Nc3
			variation => N__('Benoni: 3. d5 g6'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56i',
			# TRANSLATORS: A56i: 1. d4 Nf6 2. c4 c5 3. d5 d6
			variation => N__('Benoni: 3. d5 d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/3p1n2/2pP4/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A56i',
			# TRANSLATORS: A56i: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3
			variation => N__('Benoni: 3. d5 d6'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/3p1np1/2pP4/2P5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A56j',
			# TRANSLATORS: A56j: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3 g6
			variation => N__('Benoni: 3. d5 d6 4. Nc3 g6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/3p1np1/2pP4/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A56j',
			# TRANSLATORS: A56j: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3 g6 5. e4
			variation => N__('Benoni: 3. d5 d6 4. Nc3 g6'),
			moves => {
				'b7b5' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/p3pp1p/3p1np1/1ppP4/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A56j',
			# TRANSLATORS: A56j: 1. d4 Nf6 2. c4 c5 3. d5 g6 4. Nc3 d6 5. e4 b5
			variation => N__('Benoni: Bronstein\'s Gambit'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A56k',
			# TRANSLATORS: A56k: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3 g6 5. e4 Bg7
			variation => N__('Benoni: 3. d5 d6 4. Nc3 g6 5. e4 Bg7'),
			moves => {
				'f1d3' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/2P1P3/2N2P2/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A56l',
			# TRANSLATORS: A56l: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3 g6 5. e4 Bg7 6. f3
			variation => N__('Benoni: 3. d5 d6 4. Nc3 g6 5. e4 Bg7 6. f3'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2pP4/2P1P3/2NB4/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A56m',
			# TRANSLATORS: A56m: 1. d4 Nf6 2. c4 c5 3. d5 d6 4. Nc3 g6 5. e4 Bg7 6. Bd3
			variation => N__('Benoni: 3. d5 d6 4. Nc3 g6 5. e4 Bg7 6. Bd3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pP4/2P1n3/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56n',
			# TRANSLATORS: A56n: 1. d4 Nf6 2. c4 c5 3. d5 Ne4
			variation => N__('Benoni: Vulture'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pPp3/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A56o',
			# TRANSLATORS: A56o: 1. d4 Nf6 2. c4 c5 3. d5 e5
			variation => N__('Benoni: Czech'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pPp3/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A56o',
			# TRANSLATORS: A56o: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3
			variation => N__('Benoni: Czech'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pPp3/2P5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A56p',
			# TRANSLATORS: A56p: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6
			variation => N__('Benoni: Czech, 4. Nc3 d6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A56p',
			# TRANSLATORS: A56p: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4
			variation => N__('Benoni: Czech, 4. Nc3 d6'),
			moves => {
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A56q',
			# TRANSLATORS: A56q: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 g6
			variation => N__('Benoni: Czech, 5. e4 g6'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/3p1n2/2pPp3/2P1P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A56r',
			# TRANSLATORS: A56r: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7
			variation => N__('Benoni: Czech, 5. e4 Be7'),
			moves => {
				'f1d3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqk2r/pp2bppp/3p1n2/2pPp3/2P1P3/2N3P1/PP3P1P/R1BQKBNR b KQkq' => {
			eco => 'A56s',
			# TRANSLATORS: A56s: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. g3
			variation => N__('Benoni: Czech, 5. e4 Be7 6. g3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2bppp/3p1n2/2pPp3/2P1P3/2N3P1/PP3P1P/R1BQKBNR w KQ' => {
			eco => 'A56s',
			# TRANSLATORS: A56s: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. g3 O-O
			variation => N__('Benoni: Czech, 5. e4 Be7 6. g3 O-O'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbq1rk1/pp2bppp/3p1n2/2pPp3/2P1P3/2N3P1/PP3PBP/R1BQK1NR b KQ' => {
			eco => 'A56s',
			# TRANSLATORS: A56s: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. g3 O-O 7. Bg2
			variation => N__('Benoni: Czech, 5. e4 Be7 6. g3 O-O 7. Bg2'),
			moves => {
				'f6e8' => 1,
			},
		},
		'rnbqnrk1/pp2bppp/3p4/2pPp3/2P1P3/2N3P1/PP3PBP/R1BQK1NR w KQ' => {
			eco => 'A56t',
			# TRANSLATORS: A56t: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. g3 O-O 7. Bg2 Ne8
			variation => N__('Benoni: Czech, 5. e4 Be7 6. g3 O-O 7. Bg2 Ne8'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/3p1n2/2pPp3/2P1P3/2NB4/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A56u',
			# TRANSLATORS: A56u: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. Bd3
			variation => N__('Benoni: Czech, 5. e4 Be7 6. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/3p1n2/2pPp3/2P1P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A56v',
			# TRANSLATORS: A56v: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. Nf3
			variation => N__('Benoni: Czech, 5. e4 Be7 6. Nf3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2bppp/3p1n2/2pPp3/2P1P3/2N2N2/PP3PPP/R1BQKB1R w KQ' => {
			eco => 'A56w',
			# TRANSLATORS: A56w: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. Nf3 O-O
			variation => N__('Benoni: Czech, 5. e4 Be7 6. Nf3 O-O'),
			moves => {
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbq1rk1/pp2bppp/3p1n2/2pPp3/2P1P3/2N2N1P/PP3PP1/R1BQKB1R b KQ' => {
			eco => 'A56w',
			# TRANSLATORS: A56w: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. Nf3 O-O 7. h3
			variation => N__('Benoni: Czech, 5. e4 Be7 6. Nf3 O-O 7. h3'),
			moves => {
			},
		},
		'rnbq1rk1/pp2bppp/3p1n2/2pPp3/2P1P3/2N2N2/PP2BPPP/R1BQK2R b KQ' => {
			eco => 'A56x',
			# TRANSLATORS: A56x: 1. d4 Nf6 2. c4 c5 3. d5 e5 4. Nc3 d6 5. e4 Be7 6. Nf3 O-O 7. Be2
			variation => N__('Benoni: Czech, 5. e4 Be7 6. Nf3 O-O 7. Be2'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A57a',
			# TRANSLATORS: A57a: 1. d4 Nf6 2. c4 c5 3. d5 b5
			variation => N__('Benko Gambit'),
			moves => {
				'a2a4' => 1,
				'b1d2' => 1,
				'c4b5' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/P1P5/8/1P2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A57b',
			# TRANSLATORS: A57b: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. a4
			variation => N__('Benko Gambit: 4. a4'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/2P5/8/PP1NPPPP/R1BQKBNR b KQkq' => {
			eco => 'A57c',
			# TRANSLATORS: A57c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nd2
			variation => N__('Benko Gambit: 4. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1ppP4/2P5/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A57d',
			# TRANSLATORS: A57d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3
			variation => N__('Benko Gambit: 4. Nf3'),
			moves => {
				'b5c4' => 1,
				'c8b7' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p2ppppp/5n2/2pP4/2p5/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A57d',
			# TRANSLATORS: A57d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 bxc4
			variation => N__('Benko Gambit: 4. Nf3 bxc4'),
			moves => {
			},
		},
		'rn1qkb1r/pb1ppppp/5n2/1ppP4/2P5/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A57e',
			# TRANSLATORS: A57e: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 Bb7
			variation => N__('Benko Gambit: 4. Nf3 Bb7'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rn1qkb1r/pb1ppppp/5n2/1ppP4/P1P5/5N2/1P2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A57e',
			# TRANSLATORS: A57e: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 Bb7 5. a4
			variation => N__('Benko Gambit: 4. Nf3 Bb7 5. a4'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppp1p/5np1/1ppP4/2P5/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A57f',
			# TRANSLATORS: A57f: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 g6
			variation => N__('Benko Gambit: 4. Nf3 g6'),
			moves => {
				'c4b5' => 1,
			},
		},
		'rnbqkb1r/p2ppp1p/5np1/1PpP4/8/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A57g',
			# TRANSLATORS: A57g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 g6 5. cxb5
			variation => N__('Benko Gambit: 4. Nf3 g6 5.cxb5'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnbqkb1r/3ppp1p/p4np1/1PpP4/8/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A57g',
			# TRANSLATORS: A57g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. Nf3 g6 5. cxb5 a6
			variation => N__('Benko Gambit: 4. Nf3 g6 5.cxb5 a6'),
			moves => {
			},
		},
		'rnbqkb1r/p2ppppp/5n2/1PpP4/8/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A57h',
			# TRANSLATORS: A57h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5
			variation => N__('Benko Gambit: 4.cxb5'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnbqkb1r/3ppppp/p4n2/1PpP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A57h',
			# TRANSLATORS: A57h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6
			variation => N__('Benko Gambit: 4.cxb5 a6'),
			moves => {
				'b1c3' => 1,
				'b5a6' => 1,
				'b5b6' => 1,
				'e2e3' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqkb1r/3ppppp/pP3n2/2pP4/8/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A57i',
			# TRANSLATORS: A57i: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. b6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. b6'),
			moves => {
				'd7d6' => 1,
				'd8b6' => 1,
				'e7e6' => 1,
			},
		},
		'rnb1kb1r/3ppppp/pq3n2/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A57j',
			# TRANSLATORS: A57j: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. b6 Qxb6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. b6 Qxb6'),
			moves => {
			},
		},
		'rnbqkb1r/4pppp/pP1p1n2/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A57k',
			# TRANSLATORS: A57k: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. b6 d6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. b6 d6'),
			moves => {
			},
		},
		'rnbqkb1r/3p1ppp/pP2pn2/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A57l',
			# TRANSLATORS: A57l: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. b6 e6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. b6 e6'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/p4n2/1PpP4/8/4P3/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A57m',
			# TRANSLATORS: A57m: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. e3
			variation => N__('Benko Gambit: 4.cxb5 a6 5. e3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/3ppp1p/p4np1/1PpP4/8/4P3/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'A57n',
			# TRANSLATORS: A57n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. e3 g6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. e3 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/3ppp1p/p4np1/1PpP4/8/2N1P3/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A57n',
			# TRANSLATORS: A57n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. e3 g6 6. Nc3
			variation => N__('Benko Gambit: 4.cxb5 a6 5. e3 g6'),
			moves => {
				'd7d6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/4pp1p/p2p1np1/1PpP4/8/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A57n',
			# TRANSLATORS: A57n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. e3 g6 6. Nc3 d6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. e3 g6 6. Nc3 d6'),
			moves => {
			},
		},
		'rnbqk2r/3pppbp/p4np1/1PpP4/8/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A57n',
			# TRANSLATORS: A57n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. e3 g6 6. Nc3 Bg7
			variation => N__('Benko Gambit: 4.cxb5 a6 5. e3 g6 6. Nc3 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/p4n2/1PpP4/8/5P2/PP2P1PP/RNBQKBNR b KQkq' => {
			eco => 'A57o',
			# TRANSLATORS: A57o: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. f3
			variation => N__('Benko Gambit: 4.cxb5 a6 5. f3'),
			moves => {
				'a6b5' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/3p1ppp/p3pn2/1PpP4/8/5P2/PP2P1PP/RNBQKBNR w KQkq' => {
			eco => 'A57p',
			# TRANSLATORS: A57p: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. f3 e6
			variation => N__('Benko Gambit: 4.cxb5 a6 5. f3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/5n2/1ppP4/8/5P2/PP2P1PP/RNBQKBNR w KQkq' => {
			eco => 'A57q',
			# TRANSLATORS: A57q: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. f3 axb5
			variation => N__('Benko Gambit: 4.cxb5 a6 5. f3 axb5'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/p4n2/1PpP4/8/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A57r',
			# TRANSLATORS: A57r: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3
			variation => N__('Benko Gambit: Zaitsev Variation'),
			moves => {
				'a6b5' => 1,
				'd8a5' => 1,
			},
		},
		'rnb1kb1r/3ppppp/p4n2/qPpP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A57r',
			# TRANSLATORS: A57r: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 Qa5
			variation => N__('Benko Gambit: Zaitsev, 5... Qa5'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/5n2/1ppP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A57s',
			# TRANSLATORS: A57s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5
			variation => N__('Benko Gambit: Zaitsev, 5...axb5'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnbqkb1r/3ppppp/5n2/1ppP4/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A57s',
			# TRANSLATORS: A57s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4
			variation => N__('Benko Gambit: Zaitsev, 5...axb5'),
			moves => {
				'b5b4' => 1,
			},
		},
		'rnbqkb1r/3ppppp/5n2/2pP4/1p2P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A57s',
			# TRANSLATORS: A57s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4
			variation => N__('Benko Gambit: Zaitsev, 5...axb5 6. e4 b4'),
			moves => {
				'c3b5' => 1,
			},
		},
		'rnbqkb1r/3ppppp/5n2/1NpP4/1p2P3/8/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A57s',
			# TRANSLATORS: A57s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5
			variation => N__('Benko Gambit: Zaitsev, 5...axb5 6. e4 b4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkb1r/4pppp/3p1n2/1NpP4/1p2P3/8/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A57s',
			# TRANSLATORS: A57s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5 d6
			variation => N__('Benko Gambit: Zaitsev, 5...axb5 6. e4 b4'),
			moves => {
				'c1f4' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/4pppp/3p1n2/1NpP4/1p2P3/5N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A57t',
			# TRANSLATORS: A57t: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5 d6 8. Nf3
			variation => N__('Benko Gambit: Zaitsev, 8. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/4pppp/3p1n2/1NpP4/1pB1P3/8/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A57u',
			# TRANSLATORS: A57u: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5 d6 8. Bc4
			variation => N__('Benko Gambit: Zaitsev, Nescafe Frappe Attack'),
			moves => {
			},
		},
		'rnbqkb1r/4pppp/3p1n2/1NpP4/1p2PB2/8/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A57v',
			# TRANSLATORS: A57v: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5 d6 8. Bf4
			variation => N__('Benko Gambit: Zaitsev, 8. Bf4'),
			moves => {
				'g7g5' => 1,
			},
		},
		'rnbqkb1r/4pp1p/3p1n2/1NpP2p1/1p2PB2/8/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'A57w',
			# TRANSLATORS: A57w: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. Nc3 axb5 6. e4 b4 7. Nb5 d6 8. Bf4 g5
			variation => N__('Benko Gambit: Zaitsev, 8. Bf4 g5'),
			moves => {
			},
		},
		'rnbqkb1r/3ppppp/P4n2/2pP4/8/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A58a',
			# TRANSLATORS: A58a: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6
			variation => N__('Benko Gambit: 5.bxa6'),
			moves => {
				'c8a6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/3ppp1p/P4np1/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A58b',
			# TRANSLATORS: A58b: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 g6
			variation => N__('Benko Gambit: 5.bxa6 g6'),
			moves => {
			},
		},
		'rn1qkb1r/3ppppp/b4n2/2pP4/8/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A58c',
			# TRANSLATORS: A58c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6
			variation => N__('Benko Gambit: 5.bxa6 Bxa6'),
			moves => {
				'b1c3' => 1,
				'g2g3' => 1,
			},
		},
		'rn1qkb1r/3ppppp/b4n2/2pP4/8/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A58c',
			# TRANSLATORS: A58c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. g3
			variation => N__('Benko Gambit: Accepted, 6. g3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/8/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A58c',
			# TRANSLATORS: A58c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. g3 d6
			variation => N__('Benko Gambit: Accepted, 6. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/8/6P1/PP2PPBP/RNBQK1NR b KQkq' => {
			eco => 'A58c',
			# TRANSLATORS: A58c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. g3 d6 7. Bg2
			variation => N__('Benko Gambit: Accepted, 6. g3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A58d',
			# TRANSLATORS: A58d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. g3 d6 7. Bg2 g6
			variation => N__('Benko Gambit: Accepted, 6. g3 d6 7. Bg2 g6'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/1P4P1/P3PPBP/RNBQK1NR b KQkq' => {
			eco => 'A58d',
			# TRANSLATORS: A58d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. g3 d6 7. Bg2 g6 8. b3
			variation => N__('Benko Gambit: Accepted, 6. g3 d6 7. Bg2 g6 8. b3'),
			moves => {
			},
		},
		'rn1qkb1r/3ppppp/b4n2/2pP4/8/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A58e',
			# TRANSLATORS: A58e: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3
			variation => N__('Benko Gambit: Accepted, 6. Nc3'),
			moves => {
				'd7d6' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/3ppp1p/b4np1/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A58f',
			# TRANSLATORS: A58f: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 g6
			variation => N__('Benko Gambit: Accepted, 6. Nc3 g6'),
			moves => {
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A58g',
			# TRANSLATORS: A58g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6
			variation => N__('Benko Gambit: Accepted, 6. Nc3 d6'),
			moves => {
				'e2e4' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/5P2/2N5/PP2P1PP/R1BQKBNR b KQkq' => {
			eco => 'A58g',
			# TRANSLATORS: A58g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. f4
			variation => N__('Benko Gambit: Accepted, 7. f4'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/5P2/2N5/PP2P1PP/R1BQKBNR w KQkq' => {
			eco => 'A58g',
			# TRANSLATORS: A58g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. f4 g6
			variation => N__('Benko Gambit: Accepted, 7. f4'),
			moves => {
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/8/2N3P1/PP2PP1P/R1BQKBNR b KQkq' => {
			eco => 'A58h',
			# TRANSLATORS: A58h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. g3
			variation => N__('Benko Gambit: Accepted, 7. g3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/2N3P1/PP2PP1P/R1BQKBNR w KQkq' => {
			eco => 'A58h',
			# TRANSLATORS: A58h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. g3 g6
			variation => N__('Benko Gambit: Accepted, 7. g3'),
			moves => {
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A58i',
			# TRANSLATORS: A58i: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3
			variation => N__('Benko Gambit: Accepted, 7. Nf3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A58i',
			# TRANSLATORS: A58i: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6
			variation => N__('Benko Gambit: Accepted, 7. Nf3 g6'),
			moves => {
				'f3d2' => 1,
				'g2g3' => 1,
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/2N5/PP1NPPPP/R1BQKB1R b KQkq' => {
			eco => 'A58j',
			# TRANSLATORS: A58j: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. Nd2
			variation => N__('Benko Gambit: Accepted, 7. Nf3 g6 8. Nd2'),
			moves => {
				'd8a5' => 1,
				'f8g7' => 1,
			},
		},
		'rn2kb1r/4pp1p/b2p1np1/q1pP4/8/2N5/PP1NPPPP/R1BQKB1R w KQkq' => {
			eco => 'A58j',
			# TRANSLATORS: A58j: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. Nd2 Qa5
			variation => N__('Benko Gambit: Accepted, 7. Nf3 g6 8. Nd2 Qa5'),
			moves => {
			},
		},
		'rn1qk2r/4ppbp/b2p1np1/2pP4/8/2N5/PP1NPPPP/R1BQKB1R w KQkq' => {
			eco => 'A58j',
			# TRANSLATORS: A58j: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. Nd2 Bg7
			variation => N__('Benko Gambit: Accepted, 7. Nf3 g6 8. Nd2 Bg7'),
			moves => {
			},
		},
		'rn1qkb1r/4pp1p/b2p1np1/2pP4/8/2N2NP1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A58k',
			# TRANSLATORS: A58k: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3
			variation => N__('Benko Gambit: Fianchetto Variation'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rn1qk2r/4ppbp/b2p1np1/2pP4/8/2N2NP1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A58k',
			# TRANSLATORS: A58k: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7
			variation => N__('Benko Gambit: Fianchetto Variation'),
			moves => {
				'f1g2' => 1,
				'f1h3' => 1,
			},
		},
		'rn1qk2r/4ppbp/b2p1np1/2pP4/8/2N2NPB/PP2PP1P/R1BQK2R b KQkq' => {
			eco => 'A58l',
			# TRANSLATORS: A58l: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bh3
			variation => N__('Benko Gambit: Fianchetto, 9. Bh3'),
			moves => {
			},
		},
		'rn1qk2r/4ppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A58m',
			# TRANSLATORS: A58m: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2
			variation => N__('Benko Gambit: Fianchetto, 9. Bg2'),
			moves => {
				'b8d7' => 1,
				'e8g8' => 1,
			},
		},
		'r2qk2r/3nppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQK2R w KQkq' => {
			eco => 'A58m',
			# TRANSLATORS: A58m: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 Nbd7
			variation => N__('Benko Gambit: Fianchetto, 9...Nbd7'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r2qk2r/3nppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b kq' => {
			eco => 'A58m',
			# TRANSLATORS: A58m: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 Nbd7 10. O-O
			variation => N__('Benko Gambit: Fianchetto, 9...Nbd7'),
			moves => {
				'd7b6' => 1,
			},
		},
		'r2qk2r/4ppbp/bn1p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w kq' => {
			eco => 'A58n',
			# TRANSLATORS: A58n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 Nbd7 10. O-O Nb6
			variation => N__('Benko Gambit: Fianchetto, 9...Nbd7 10.O-O Nb6'),
			moves => {
			},
		},
		'rn1q1rk1/4ppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A58o',
			# TRANSLATORS: A58o: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O
			variation => N__('Benko Gambit: Fianchetto, 9...O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1q1rk1/4ppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A58o',
			# TRANSLATORS: A58o: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O
			variation => N__('Benko Gambit: Fianchetto, 9...O-O'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2q1rk1/3nppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A58p',
			# TRANSLATORS: A58p: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O Nbd7
			variation => N__('Benko Gambit: Fianchetto, Main Line'),
			moves => {
				'a1b1' => 1,
				'd1c2' => 1,
				'f1e1' => 1,
			},
		},
		'r2q1rk1/3nppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/1RBQ1RK1 b -' => {
			eco => 'A58q',
			# TRANSLATORS: A58q: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O Nbd7 11. Rb1
			variation => N__('Benko Gambit: Fianchetto, Main Line, 11. Rb1'),
			moves => {
			},
		},
		'r2q1rk1/3nppbp/b2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQR1K1 b -' => {
			eco => 'A58r',
			# TRANSLATORS: A58r: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O Nbd7 11. Re1
			variation => N__('Benko Gambit: Fianchetto, Main Line, 11. Re1'),
			moves => {
			},
		},
		'r2q1rk1/3nppbp/b2p1np1/2pP4/8/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A58s',
			# TRANSLATORS: A58s: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O Nbd7 11. Qc2
			variation => N__('Benko Gambit: Fianchetto, Main Line, 11. Qc2'),
			moves => {
				'd8b6' => 1,
			},
		},
		'r4rk1/3nppbp/bq1p1np1/2pP4/8/2N2NP1/PPQ1PPBP/R1B2RK1 w -' => {
			eco => 'A58t',
			# TRANSLATORS: A58t: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. Nf3 g6 8. g3 Bg7 9. Bg2 O-O 10. O-O Nbd7 11. Qc2 Qb6
			variation => N__('Benko Gambit: Fianchetto, Main Line, 11. Qc2 Qb6'),
			moves => {
			},
		},
		'rn1qkb1r/4pppp/b2p1n2/2pP4/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A59a',
			# TRANSLATORS: A59a: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4
			variation => N__('Benko Gambit: 7. e4'),
			moves => {
				'a6f1' => 1,
			},
		},
		'rn1qkb1r/4pppp/3p1n2/2pP4/4P3/2N5/PP3PPP/R1BQKbNR w KQkq' => {
			eco => 'A59a',
			# TRANSLATORS: A59a: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1
			variation => N__('Benko Gambit: 7. e4'),
			moves => {
				'e1f1' => 1,
			},
		},
		'rn1qkb1r/4pppp/3p1n2/2pP4/4P3/2N5/PP3PPP/R1BQ1KNR b kq' => {
			eco => 'A59a',
			# TRANSLATORS: A59a: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1
			variation => N__('Benko Gambit: 7. e4'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/4pp1p/3p1np1/2pP4/4P3/2N5/PP3PPP/R1BQ1KNR w kq' => {
			eco => 'A59b',
			# TRANSLATORS: A59b: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6
			variation => N__('Benko Gambit: 7. e4 Bxf1 8.Kxf1 g6'),
			moves => {
				'g1e2' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
			},
		},
		'rn1qkb1r/4pp1p/3p1np1/2pP4/4P3/2N5/PP2NPPP/R1BQ1K1R b kq' => {
			eco => 'A59c',
			# TRANSLATORS: A59c: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. Nge2
			variation => N__('Benko Gambit: 7. e4 Line, 9.Nge2'),
			moves => {
			},
		},
		'rn1qkb1r/4pp1p/3p1np1/2pP4/4P3/2N2N2/PP3PPP/R1BQ1K1R b kq' => {
			eco => 'A59d',
			# TRANSLATORS: A59d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. Nf3
			variation => N__('Benko Gambit: 7. e4 Line, 9. Nf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rn1qk2r/4ppbp/3p1np1/2pP4/4P3/2N2N2/PP3PPP/R1BQ1K1R w kq' => {
			eco => 'A59d',
			# TRANSLATORS: A59d: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. Nf3 Bg7
			variation => N__('Benko Gambit: 7. e4 Line, 9. Nf3'),
			moves => {
				'h2h3' => 1,
			},
		},
		'rn1qk2r/4ppbp/3p1np1/2pP4/4P3/2N2N1P/PP3PP1/R1BQ1K1R b kq' => {
			eco => 'A59e',
			# TRANSLATORS: A59e: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. Nf3 Bg7 10. h3
			variation => N__('Benko Gambit: 7. e4 Line, 9. Nf3 Bg7 10. h3'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qk2r/3nppbp/3p1np1/2pP4/4P3/2N2N1P/PP3PP1/R1BQ1K1R w kq' => {
			eco => 'A59f',
			# TRANSLATORS: A59f: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. Nf3 Bg7 10. h3 Nbd7
			variation => N__('Benko Gambit: 7. e4 Line, 9. Nf3 Bg7 10. h3 Nbd7'),
			moves => {
			},
		},
		'rn1qkb1r/4pp1p/3p1np1/2pP4/4P1P1/2N5/PP3P1P/R1BQ1KNR b kq' => {
			eco => 'A59g',
			# TRANSLATORS: A59g: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g4
			variation => N__('Benko Gambit: 7. e4 Line, 9. g4'),
			moves => {
			},
		},
		'rn1qkb1r/4pp1p/3p1np1/2pP4/4P3/2N3P1/PP3P1P/R1BQ1KNR b kq' => {
			eco => 'A59h',
			# TRANSLATORS: A59h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3
			variation => N__('Benko Gambit: 7. e4 Line, 9. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rn1qk2r/4ppbp/3p1np1/2pP4/4P3/2N3P1/PP3P1P/R1BQ1KNR w kq' => {
			eco => 'A59h',
			# TRANSLATORS: A59h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7
			variation => N__('Benko Gambit: 7. e4 Line, 9. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rn1qk2r/4ppbp/3p1np1/2pP4/4P3/2N3P1/PP3PKP/R1BQ2NR b kq' => {
			eco => 'A59h',
			# TRANSLATORS: A59h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2
			variation => N__('Benko Gambit: 7. e4 Line, 9. g3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/4ppbp/3p1np1/2pP4/4P3/2N3P1/PP3PKP/R1BQ2NR w -' => {
			eco => 'A59h',
			# TRANSLATORS: A59h: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O
			variation => N__('Benko Gambit: 7. e4 Line, 9. g3'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rn1q1rk1/4ppbp/3p1np1/2pP4/4P3/2N2NP1/PP3PKP/R1BQ3R b -' => {
			eco => 'A59i',
			# TRANSLATORS: A59i: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3
			variation => N__('Benko Gambit: 7. e4, Main Line'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2q1rk1/3nppbp/3p1np1/2pP4/4P3/2N2NP1/PP3PKP/R1BQ3R w -' => {
			eco => 'A59j',
			# TRANSLATORS: A59j: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7'),
			moves => {
				'h1e1' => 1,
				'h2h3' => 1,
			},
		},
		'r2q1rk1/3nppbp/3p1np1/2pP4/4P3/2N2NP1/PP3PKP/R1BQR3 b -' => {
			eco => 'A59k',
			# TRANSLATORS: A59k: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. Re1
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. Re1'),
			moves => {
				'd8a5' => 1,
				'f6g4' => 1,
			},
		},
		'r4rk1/3nppbp/3p1np1/q1pP4/4P3/2N2NP1/PP3PKP/R1BQR3 w -' => {
			eco => 'A59l',
			# TRANSLATORS: A59l: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. Re1 Qa5
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. Re1 Qa5'),
			moves => {
			},
		},
		'r2q1rk1/3nppbp/3p2p1/2pP4/4P1n1/2N2NP1/PP3PKP/R1BQR3 w -' => {
			eco => 'A59m',
			# TRANSLATORS: A59m: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. Re1 Ng4
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. Re1 Ng4'),
			moves => {
			},
		},
		'r2q1rk1/3nppbp/3p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQ3R b -' => {
			eco => 'A59n',
			# TRANSLATORS: A59n: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3'),
			moves => {
				'a8a6' => 1,
				'd8a5' => 1,
				'd8b6' => 1,
			},
		},
		'3q1rk1/3nppbp/r2p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQ3R w -' => {
			eco => 'A59o',
			# TRANSLATORS: A59o: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Ra6
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Ra6'),
			moves => {
				'h1e1' => 1,
			},
		},
		'3q1rk1/3nppbp/r2p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQR3 b -' => {
			eco => 'A59o',
			# TRANSLATORS: A59o: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Ra6 13. Re1
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Ra6 13. Re1'),
			moves => {
			},
		},
		'r4rk1/3nppbp/1q1p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQ3R w -' => {
			eco => 'A59p',
			# TRANSLATORS: A59p: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Qb6
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Qb6'),
			moves => {
				'h1e1' => 1,
			},
		},
		'r4rk1/3nppbp/1q1p1np1/2pP4/4P3/2N2NPP/PP3PK1/R1BQR3 b -' => {
			eco => 'A59p',
			# TRANSLATORS: A59p: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Qb6 13. Re1
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Qb6 13. Re1'),
			moves => {
			},
		},
		'r4rk1/3nppbp/3p1np1/q1pP4/4P3/2N2NPP/PP3PK1/R1BQ3R w -' => {
			eco => 'A59q',
			# TRANSLATORS: A59q: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Qa5
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Qa5'),
			moves => {
				'h1e1' => 1,
			},
		},
		'r4rk1/3nppbp/3p1np1/q1pP4/4P3/2N2NPP/PP3PK1/R1BQR3 b -' => {
			eco => 'A59q',
			# TRANSLATORS: A59q: 1. d4 Nf6 2. c4 c5 3. d5 b5 4. cxb5 a6 5. bxa6 Bxa6 6. Nc3 d6 7. e4 Bxf1 8. Kxf1 g6 9. g3 Bg7 10. Kg2 O-O 11. Nf3 Nbd7 12. h3 Qa5 13. Re1
			variation => N__('Benko Gambit: 7. e4, Main Line, 11...Nbd7 12. h3 Qa5 13. Re1'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/2P5/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A60a',
			# TRANSLATORS: A60a: 1. d4 Nf6 2. c4 c5 3. d5 e6
			variation => N__('Benoni: 3. d5 e6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pp4/2P5/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A60b',
			# TRANSLATORS: A60b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nf3 exd5
			variation => N__('Benoni: 4. Nf3'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/2P5/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A60b',
			# TRANSLATORS: A60b: 1. d4 Nf6 2. c4 e6 3. Nf3 c5 4. d5
			variation => N__('Benoni: 4. Nf3'),
			moves => {
				'b7b5' => 1,
				'e6d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pP4/8/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A60c',
			# TRANSLATORS: A60c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nf3 exd5 5. cxd5
			variation => N__('Benoni: 4. Nf3 exd5 5.cxd5'),
			moves => {
				'f8d6' => 1,
			},
		},
		'rnbqk2r/pp1p1ppp/3b1n2/2pP4/8/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A60d',
			# TRANSLATORS: A60d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nf3 exd5 5. cxd5 Bd6
			variation => N__('Benoni: Snake Variation vs. 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/2P5/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A60e',
			# TRANSLATORS: A60e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. g3
			variation => N__('Benoni: 4. g3'),
			moves => {
				'e6d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pp4/2P5/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A60e',
			# TRANSLATORS: A60e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. g3 exd5
			variation => N__('Benoni: 4. g3'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pP4/8/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A60e',
			# TRANSLATORS: A60e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. g3 exd5 5. cxd5
			variation => N__('Benoni: 4. g3'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkb1r/p2p1ppp/5n2/1ppP4/8/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A60f',
			# TRANSLATORS: A60f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. g3 exd5 5. cxd5 b5
			variation => N__('Benoni: 4. g3 exd5 5.cxd5 b5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4pn2/2pP4/2P5/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A60g',
			# TRANSLATORS: A60g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3
			variation => N__('Benoni: 4. Nc3'),
			moves => {
				'e6d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pp4/2P5/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A60g',
			# TRANSLATORS: A60g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5
			variation => N__('Benoni: 4. Nc3'),
			moves => {
				'c3d5' => 1,
				'c4d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pN4/2P5/8/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A60h',
			# TRANSLATORS: A60h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. Nxd5
			variation => N__('Benoni: 4. Nc3 exd5 5.Nxd5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/5n2/2pP4/8/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A60i',
			# TRANSLATORS: A60i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5
			variation => N__('Benoni: 4. Nc3 exd5 5.cxd5'),
			moves => {
				'd7d6' => 1,
				'f8d6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqk2r/pp1p1ppp/3b1n2/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A60j',
			# TRANSLATORS: A60j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6
			variation => N__('Benoni: Snake Variation'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pp1p1ppp/3b1n2/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A60k',
			# TRANSLATORS: A60k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6 6. Nf3
			variation => N__('Benoni: Snake, 6. Nf3'),
			moves => {
				'd6c7' => 1,
			},
		},
		'rnbqk2r/ppbp1ppp/5n2/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A60l',
			# TRANSLATORS: A60l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6 6. Nf3 Bc7
			variation => N__('Benoni: Snake, 6. Nf3 Bc7'),
			moves => {
			},
		},
		'rnbqk2r/pp1p1ppp/3b1n2/2pP4/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A60m',
			# TRANSLATORS: A60m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6 6. e4
			variation => N__('Benoni: Snake, 6. e4'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp1p1ppp/3b1n2/2pP4/4P3/2N5/PP3PPP/R1BQKBNR w KQ' => {
			eco => 'A60m',
			# TRANSLATORS: A60m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6 6. e4 O-O
			variation => N__('Benoni: Snake, 6. e4'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbq1rk1/pp1p1ppp/3b1n2/2pP4/4P3/2N2N2/PP3PPP/R1BQKB1R b KQ' => {
			eco => 'A60n',
			# TRANSLATORS: A60n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 Bd6 6. e4 O-O 7. Nf3
			variation => N__('Benoni: Snake, 6. e4 O-O 7. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1p1p/5np1/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A60o',
			# TRANSLATORS: A60o: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 g6
			variation => N__('Benoni: 4. Nc3 exd5 5.cxd5 g6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pP4/8/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A60p',
			# TRANSLATORS: A60p: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6
			variation => N__('Benoni: 4. Nc3 exd5 5.cxd5 d6'),
			moves => {
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A60q',
			# TRANSLATORS: A60q: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3
			variation => N__('Benoni: 4. Nc3 exd5 5.cxd5 d6 6. Nf3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/8/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6
			variation => N__('Benoni: 6. Nf3 g6'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'f3d2' => 1,
				'g2g3' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/5B2/2N2N2/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bf4
			variation => N__('Benoni: 6. Nf3 g6 7. Bf4'),
			moves => {
				'a7a6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/1p3p1p/p2p1np1/2pP4/5B2/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bf4 a6
			variation => N__('Benoni: 6. Nf3 g6 7. Bf4 a6'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/5B2/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bf4 Bg7
			variation => N__('Benoni: 6. Nf3 g6 7. Bf4 Bg7'),
			moves => {
				'd1a4' => 1,
				'f3d2' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/Q4B2/2N2N2/PP2PPPP/R3KB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bf4 Bg7 8. Qa4+
			variation => N__('Benoni: 6. Nf3 g6 7. Bf4 Bg7 8. Qa4+'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/5B2/2N5/PP1NPPPP/R2QKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bf4 Bg7 8. Nd2
			variation => N__('Benoni: 6. Nf3 g6 7. Bf4 Bg7 8. Nd2'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP2B1/8/2N2N2/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bg5
			variation => N__('Benoni: Uhlmann Variation'),
			moves => {
				'f8g7' => 1,
				'h7h6' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP2B1/8/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bg5 Bg7
			variation => N__('Benoni: Uhlmann, 7... Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p2/3p1npp/2pP2B1/8/2N2N2/PP2PPPP/R2QKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Bg5 h6
			variation => N__('Benoni: Uhlmann, 7... h6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/8/2N5/PP1NPPPP/R1BQKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2
			variation => N__('Benoni: Nimzowitsch Variation'),
			moves => {
				'b8d7' => 1,
				'f8g7' => 1,
			},
		},
		'r1bqkb1r/pp1n1p1p/3p1np1/2pP4/8/2N5/PP1NPPPP/R1BQKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Nbd7
			variation => N__('Benoni: Nimzowitsch, 7...Nbd7'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/8/2N5/PP1NPPPP/R1BQKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Bg7
			variation => N__('Benoni: Nimzowitsch, 7... Bg7'),
			moves => {
				'd2c4' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/2N5/2N5/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Bg7 8. Nc4
			variation => N__('Benoni: Nimzowitsch, 7... Bg7'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/2N5/2N5/PP2PPPP/R1BQKB1R w KQ' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Bg7 8. Nc4 O-O
			variation => N__('Benoni: Nimzowitsch, 7... Bg7'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/2N2B2/2N5/PP2PPPP/R2QKB1R b KQ' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Bg7 8. Nc4 O-O 9. Bf4
			variation => N__('Benoni: Nimzowitsch, 7... Bg7 8. Nc4 O-O 9. Bf4'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP2B1/2N5/2N5/PP2PPPP/R2QKB1R b KQ' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. Nd2 Bg7 8. Nc4 O-O 9. Bg5
			variation => N__('Benoni: Nimzowitsch, 7... Bg7 8. Nc4 O-O 9. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/8/2N2NP1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3
			variation => N__('Benoni: Fianchetto Variation'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/8/2N2NP1/PP2PP1P/R1BQKB1R w KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7
			variation => N__('Benoni: Fianchetto Variation'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2
			variation => N__('Benoni: Fianchetto Variation'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/8/2N2N1P/PP2PPP1/R1BQKB1R b KQkq' => {
			eco => 'A61',
			# TRANSLATORS: A61: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. h3
			variation => N__('Benoni: 6. Nf3 g6 7. h3'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O
			variation => N__('Benoni: Fianchetto, 8. Bg2 O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O
			variation => N__('Benoni: Fianchetto, 9.O-O'),
			moves => {
				'a7a6' => 1,
				'b8a6' => 1,
				'b8d7' => 1,
				'f8e8' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O a6
			variation => N__('Benoni: Fianchetto, 9.O-O a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/P7/2N2NP1/1P2PPBP/R1BQ1RK1 b -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O a6 10. a4
			variation => N__('Benoni: Fianchetto, 9.O-O a6'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Na6
			variation => N__('Benoni: Fianchetto, 9.O-O Na6'),
			moves => {
				'f3d2' => 1,
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Na6 10. Nd2
			variation => N__('Benoni: Fianchetto, 9.O-O Na6'),
			moves => {
				'a6c7' => 1,
			},
		},
		'r1bq1rk1/ppn2pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Na6 10. Nd2 Nc7
			variation => N__('Benoni: Fianchetto, 9.O-O Na6 10. Nd2 Nc7'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A62',
			# TRANSLATORS: A62: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Re8
			variation => N__('Benoni: Fianchetto, 9.O-O Re8'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P7/2N2NP1/1P2PPBP/R1BQ1RK1 w -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O a6 10. a4 Nbd7
			variation => N__('Benoni: Fianchetto, 9... a6 10. a4 Nbd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/8/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7
			variation => N__('Benoni: Fianchetto, 9...Nbd7'),
			moves => {
				'f3d2' => 1,
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2
			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2'),
			moves => {
				'a7a6' => 1,
				'f8e8' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6
			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2 a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P7/2N3P1/1P1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4
			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2 a6 11. a4'),
			moves => {
				'f8e8' => 1,
			},
		},
		'r1bqr1k1/pp1n1pbp/3p1np1/2pP4/8/2N3P1/PP1NPPBP/R1BQ1RK1 w -' => {
			eco => 'A63',
			# TRANSLATORS: A63: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 Re8
			variation => N__('Benoni: Fianchetto, 9...Nbd7 10. Nd2 Re8'),
			moves => {
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P7/2N3P1/1P1NPPBP/R1BQ1RK1 w -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8
			variation => N__('Benoni: Fianchetto, 11... Re8'),
			moves => {
				'a4a5' => 1,
				'd2c4' => 1,
				'h2h3' => 1,
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/P1pP4/8/2N3P1/1P1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. a5
			variation => N__('Benoni: Fianchetto, 11... Re8 12. a5'),
			moves => {
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P1N5/2N3P1/1P2PPBP/R1BQ1RK1 b -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. Nc4
			variation => N__('Benoni: Fianchetto, 11... Re8 12. Nc4'),
			moves => {
				'd7e5' => 1,
			},
		},
		'r1bqr1k1/1p3pbp/p2p1np1/2pPn3/P1N5/2N3P1/1P2PPBP/R1BQ1RK1 w -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. Nc4 Ne5
			variation => N__('Benoni: Fianchetto, 11... Re8 12. Nc4 Ne5'),
			moves => {
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P7/2N3PP/1P1NPPB1/R1BQ1RK1 b -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3
			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3'),
			moves => {
				'a8b8' => 1,
			},
		},
		'1rbqr1k1/1p1n1pbp/p2p1np1/2pP4/P7/2N3PP/1P1NPPB1/R1BQ1RK1 w -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8
			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8'),
			moves => {
				'd2c4' => 1,
			},
		},
		'1rbqr1k1/1p1n1pbp/p2p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 b -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4
			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4'),
			moves => {
				'd7b6' => 1,
				'd7e5' => 1,
			},
		},
		'1rbqr1k1/1p3pbp/pn1p1np1/2pP4/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4 Nb6
			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4 Nb6'),
			moves => {
			},
		},
		'1rbqr1k1/1p3pbp/p2p1np1/2pPn3/P1N5/2N3PP/1P2PPB1/R1BQ1RK1 w -' => {
			eco => 'A64',
			# TRANSLATORS: A64: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. Nf3 g6 7. g3 Bg7 8. Bg2 O-O 9. O-O Nbd7 10. Nd2 a6 11. a4 Re8 12. h3 Rb8 13. Nc4 Ne5
			variation => N__('Benoni: Fianchetto, 11... Re8 12. h3 Rb8 13. Nc4 Ne5'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/3p1n2/2pP4/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A65a',
			# TRANSLATORS: A65a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4
			variation => N__('Benoni: 6. e4'),
			moves => {
				'f8e7' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqk2r/pp2bppp/3p1n2/2pP4/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A65a',
			# TRANSLATORS: A65a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 Be7
			variation => N__('Benoni: 6. e4 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A65b',
			# TRANSLATORS: A65b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6
			variation => N__('Benoni: 6. e4 g6'),
			moves => {
				'c1f4' => 1,
				'f1d3' => 1,
				'f2f3' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4P3/2N4P/PP3PP1/R1BQKBNR b KQkq' => {
			eco => 'A65b',
			# TRANSLATORS: A65b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. h3
			variation => N__('Benoni: 6. e4 g6 7. h3'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4PB2/2N5/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'A65c',
			# TRANSLATORS: A65c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bf4
			variation => N__('Benoni: 6. e4 g6 7. Bf4'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnbqkb1r/1p3p1p/p2p1np1/2pP4/4PB2/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'A65d',
			# TRANSLATORS: A65d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bf4 a6
			variation => N__('Benoni: 6. e4 g6 7. Bf4 a6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/1p3p1p/p2p1np1/2pP4/4PB2/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'A65d',
			# TRANSLATORS: A65d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bf4 a6 8. Nf3
			variation => N__('Benoni: 6. e4 g6 7. Bf4 a6'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqkb1r/5p1p/p2p1np1/1ppP4/4PB2/2N2N2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'A65e',
			# TRANSLATORS: A65e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bf4 a6 8. Nf3 b5
			variation => N__('Benoni: 6. e4 g6 7. Bf4 a6 8. Nf3 b5'),
			moves => {
				'd1e2' => 1,
			},
		},
		'rnbqkb1r/5p1p/p2p1np1/1ppP4/4PB2/2N2N2/PP2QPPP/R3KB1R b KQkq' => {
			eco => 'A65e',
			# TRANSLATORS: A65e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bf4 a6 8. Nf3 b5 9. Qe2
			variation => N__('Benoni: 6. e4 g6 7. Bf4 a6 8. Nf3 b5 9. Qe2'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4P3/2NB4/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'A65f',
			# TRANSLATORS: A65f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3
			variation => N__('Benoni: 6. e4 g6 7. Bd3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2NB4/PP3PPP/R1BQK1NR w KQkq' => {
			eco => 'A65f',
			# TRANSLATORS: A65f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7
			variation => N__('Benoni: 6. e4 g6 7. Bd3'),
			moves => {
				'g1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2NB3P/PP3PP1/R1BQK1NR b KQkq' => {
			eco => 'A65f',
			# TRANSLATORS: A65f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. h3
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8. h3'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQK2R b KQkq' => {
			eco => 'A65g',
			# TRANSLATORS: A65g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQK2R w KQ' => {
			eco => 'A65g',
			# TRANSLATORS: A65g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQ1RK1 b -' => {
			eco => 'A65g',
			# TRANSLATORS: A65g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O 9. O-O
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O 9.O-O'),
			moves => {
				'a7a6' => 1,
				'b7b6' => 1,
				'b8a6' => 1,
			},
		},
		'rnbq1rk1/p4pbp/1p1p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQ1RK1 w -' => {
			eco => 'A65h',
			# TRANSLATORS: A65h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O 9. O-O b6
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O 9.O-O b6'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQ1RK1 w -' => {
			eco => 'A65i',
			# TRANSLATORS: A65i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O 9. O-O Na6
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O 9.O-O Na6'),
			moves => {
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/4P3/2NB4/PP2NPPP/R1BQ1RK1 w -' => {
			eco => 'A65j',
			# TRANSLATORS: A65j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Bd3 Bg7 8. Nge2 O-O 9. O-O a6
			variation => N__('Benoni: 6. e4 g6 7. Bd3 Bg7 8.Nge2 O-O 9.O-O a6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4P3/2N2P2/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A65k',
			# TRANSLATORS: A65k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3
			variation => N__('Benoni: Sämisch (6. e4 g6 7. f3)'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N2P2/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'A65k',
			# TRANSLATORS: A65k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7
			variation => N__('Benoni: Sämisch, 7... Bg7'),
			moves => {
				'c1e3' => 1,
				'c1g5' => 1,
				'g1e2' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N2P2/PP2N1PP/R1BQKB1R b KQkq' => {
			eco => 'A65l',
			# TRANSLATORS: A65l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Nge2
			variation => N__('Benoni: Sämisch, 8.Nge2'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N1BP2/PP4PP/R2QKBNR b KQkq' => {
			eco => 'A65m',
			# TRANSLATORS: A65m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Be3
			variation => N__('Benoni: Sämisch, 8. Be3'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP2B1/4P3/2N2P2/PP4PP/R2QKBNR b KQkq' => {
			eco => 'A65n',
			# TRANSLATORS: A65n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5
			variation => N__('Benoni: Sämisch, 8. Bg5'),
			moves => {
				'e8g8' => 1,
				'h7h6' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP2B1/4P3/2N2P2/PP4PP/R2QKBNR w KQ' => {
			eco => 'A65n',
			# TRANSLATORS: A65n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 O-O
			variation => N__('Benoni: Sämisch, 8. Bg5 O-O'),
			moves => {
				'd1d2' => 1,
				'g1e2' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP2B1/4P3/2N2P2/PP2N1PP/R2QKB1R b KQ' => {
			eco => 'A65o',
			# TRANSLATORS: A65o: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 O-O 9. Nge2
			variation => N__('Benoni: Sämisch, 8. Bg5 O-O 9.Nge2'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP2B1/4P3/2N2P2/PP1Q2PP/R3KBNR b KQ' => {
			eco => 'A65p',
			# TRANSLATORS: A65p: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 O-O 9. Qd2
			variation => N__('Benoni: Sämisch, 8. Bg5 O-O 9. Qd2'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP2B1/4P3/2N2P2/PP1Q2PP/R3KBNR w KQ' => {
			eco => 'A65q',
			# TRANSLATORS: A65q: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 O-O 9. Qd2 a6
			variation => N__('Benoni: Sämisch, 8. Bg5 O-O 9. Qd2 a6'),
			moves => {
			},
		},
		'rnbqk2r/pp3pb1/3p1npp/2pP2B1/4P3/2N2P2/PP4PP/R2QKBNR w KQkq' => {
			eco => 'A65r',
			# TRANSLATORS: A65r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6
			variation => N__('Benoni: Sämisch, 8. Bg5 h6'),
			moves => {
				'g5e3' => 1,
			},
		},
		'rnbqk2r/pp3pb1/3p1npp/2pP4/4P3/2N1BP2/PP4PP/R2QKBNR b KQkq' => {
			eco => 'A65r',
			# TRANSLATORS: A65r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3
			variation => N__('Benoni: Sämisch, 8. Bg5 h6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pb1/3p1npp/2pP4/4P3/2N1BP2/PP4PP/R2QKBNR w KQ' => {
			eco => 'A65r',
			# TRANSLATORS: A65r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O'),
			moves => {
				'd1d2' => 1,
				'g1e2' => 1,
			},
		},
		'rnbq1rk1/pp3pb1/3p1npp/2pP4/4P3/2N1BP2/PP2N1PP/R2QKB1R b KQ' => {
			eco => 'A65s',
			# TRANSLATORS: A65s: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Nge2
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10.Nge2'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pb1/3p1npp/2pP4/4P3/2N1BP2/PP1Q2PP/R3KBNR b KQ' => {
			eco => 'A65t',
			# TRANSLATORS: A65t: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2'),
			moves => {
				'a7a6' => 1,
				'f8e8' => 1,
			},
		},
		'rnbqr1k1/pp3pb1/3p1npp/2pP4/4P3/2N1BP2/PP1Q2PP/R3KBNR w KQ' => {
			eco => 'A65u',
			# TRANSLATORS: A65u: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 Re8
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 Re8'),
			moves => {
			},
		},
		'rnbq1rk1/1p3pb1/p2p1npp/2pP4/4P3/2N1BP2/PP1Q2PP/R3KBNR w KQ' => {
			eco => 'A65v',
			# TRANSLATORS: A65v: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbq1rk1/1p3pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1Q2PP/R3KBNR b KQ' => {
			eco => 'A65v',
			# TRANSLATORS: A65v: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4'),
			moves => {
				'b8d7' => 1,
				'f8e8' => 1,
				'h6h5' => 1,
			},
		},
		'rnbq1rk1/1p3pb1/p2p1np1/2pP3p/P3P3/2N1BP2/1P1Q2PP/R3KBNR w KQ' => {
			eco => 'A65w',
			# TRANSLATORS: A65w: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 h5
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 h5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1Q2PP/R3KBNR w KQ' => {
			eco => 'A65x',
			# TRANSLATORS: A65x: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Nbd7
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Nbd7'),
			moves => {
			},
		},
		'rnbqr1k1/1p3pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1Q2PP/R3KBNR w KQ' => {
			eco => 'A65y',
			# TRANSLATORS: A65y: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8'),
			moves => {
				'g1e2' => 1,
			},
		},
		'rnbqr1k1/1p3pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1QN1PP/R3KB1R b KQ' => {
			eco => 'A65y',
			# TRANSLATORS: A65y: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8 12. Nge2
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bqr1k1/1p1n1pb1/p2p1npp/2pP4/P3P3/2N1BP2/1P1QN1PP/R3KB1R w KQ' => {
			eco => 'A65y',
			# TRANSLATORS: A65y: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f3 Bg7 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8 12. Nge2 Nbd7
			variation => N__('Benoni: Sämisch, 8. Bg5 h6 9. Be3 O-O 10. Qd2 a6 11. a4 Re8 12.Nge2 Nbd7'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4PP2/2N5/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4
			variation => N__('Benoni: Four Pawns Attack'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4PP2/2N5/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7
			variation => N__('Benoni: Four Pawns Attack'),
			moves => {
				'e4e5' => 1,
				'f1b5' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pPP3/5P2/2N5/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5
			variation => N__('Benoni: Four Pawns, Mikenas Attack'),
			moves => {
				'd6e5' => 1,
				'f6d7' => 1,
			},
		},
		'rnbqk2r/pp3pbp/5np1/2pPp3/5P2/2N5/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 dxe5
			variation => N__('Benoni: Mikenas Attack, 8...dxe5'),
			moves => {
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/2pPP3/5P2/2N5/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7
			variation => N__('Benoni: Mikenas Attack, 8...Nfd7'),
			moves => {
				'c3b5' => 1,
				'e5e6' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/1NpPP3/5P2/8/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5
			variation => N__('Benoni: Mikenas Attack, 8...Nfd7 9. Nb5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqk2r/pp1n1pbp/6p1/1NpPp3/5P2/8/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5
			variation => N__('Benoni: Mikenas Attack, 8...Nfd7 9. Nb5'),
			moves => {
				'b5d6' => 1,
			},
		},
		'rnbqk2r/pp1n1pbp/3N2p1/2pPp3/5P2/8/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5 10. Nd6+
			variation => N__('Benoni: Mikenas Attack, 9. Nb5 dxe5 10. Nd6+'),
			moves => {
				'e8e7' => 1,
			},
		},
		'rnbq3r/pp1nkpbp/3N2p1/2pPp3/5P2/8/PP4PP/R1BQKBNR w KQ' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5 10. Nd6+ Ke7
			variation => N__('Benoni: Mikenas Attack, 9. Nb5 dxe5 10. Nd6+'),
			moves => {
				'd6c8' => 1,
			},
		},
		'rnNq3r/pp1nkpbp/6p1/2pPp3/5P2/8/PP4PP/R1BQKBNR b KQ' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5 10. Nd6+ Ke7 11. Nxc8+
			variation => N__('Benoni: Mikenas Attack, 9. Nb5 dxe5 10. Nd6+ Ke7 11.Nxc8+'),
			moves => {
				'd8c8' => 1,
			},
		},
		'rnq4r/pp1nkpbp/6p1/2pPp3/5P2/8/PP4PP/R1BQKBNR w KQ' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5 10. Nd6+ Ke7 11. Nxc8+ Qxc8
			variation => N__('Benoni: Mikenas Attack, 9. Nb5 dxe5 10. Nd6+ Ke7 11.Nxc8+'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnq4r/pp1nkpbp/6p1/2pPp3/5P2/5N2/PP4PP/R1BQKB1R b KQ' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nb5 dxe5 10. Nd6+ Ke7 11. Nxc8+ Qxc8 12. Nf3
			variation => N__('Benoni: Mikenas Attack, 9. Nb5 dxe5 10. Nd6+ Ke7 11.Nxc8+ Qxc8 12. Nf3'),
			moves => {
			},
		},
		'rnbqk2r/pp1n1pbp/3pP1p1/2pP4/5P2/2N5/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. e6
			variation => N__('Benoni: Mikenas Attack, 8...Nfd7 9. e6'),
			moves => {
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/2pPP3/5P2/2N2N2/PP4PP/R1BQKB1R b KQkq' => {
			eco => 'A66',
			# TRANSLATORS: A66: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. e5 Nfd7 9. Nf3
			variation => N__('Benoni: Mikenas Attack, 8...Nfd7 9. Nf3'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/1BpP4/4PP2/2N5/PP4PP/R1BQK1NR b KQkq' => {
			eco => 'A67a',
			# TRANSLATORS: A67a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+
			variation => N__('Benoni: Four Pawns, Taimanov (Alatortsev) Variation'),
			moves => {
				'b8d7' => 1,
				'f6d7' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/3p1np1/1BpP4/4PP2/2N5/PP4PP/R1BQK1NR w KQkq' => {
			eco => 'A67b',
			# TRANSLATORS: A67b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'e4e5' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/3p1np1/1BpPP3/5P2/2N5/PP4PP/R1BQK1NR b KQkq' => {
			eco => 'A67b',
			# TRANSLATORS: A67b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'd6e5' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/5np1/1BpPp3/5P2/2N5/PP4PP/R1BQK1NR w KQkq' => {
			eco => 'A67b',
			# TRANSLATORS: A67b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5 dxe5
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'f4e5' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/5np1/1BpPP3/8/2N5/PP4PP/R1BQK1NR b KQkq' => {
			eco => 'A67b',
			# TRANSLATORS: A67b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5 dxe5 10. fxe5
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'f6h5' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/6p1/1BpPP2n/8/2N5/PP4PP/R1BQK1NR w KQkq' => {
			eco => 'A67b',
			# TRANSLATORS: A67b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5 dxe5 10. fxe5 Nh5
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7'),
			moves => {
				'e5e6' => 1,
			},
		},
		'r1bqk2r/pp1n1pbp/4P1p1/1BpP3n/8/2N5/PP4PP/R1BQK1NR b KQkq' => {
			eco => 'A67c',
			# TRANSLATORS: A67c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nbd7 9. e5 dxe5 10. fxe5 Nh5 11. e6
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nbd7 9. e5 dxe5 10.fxe5 Nh5 11. e6'),
			moves => {
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/1BpP4/4PP2/2N5/PP4PP/R1BQK1NR w KQkq' => {
			eco => 'A67d',
			# TRANSLATORS: A67d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7
			variation => N__('Benoni: Four Pawns, Taimanov, 8...Nfd7'),
			moves => {
				'a2a4' => 1,
				'b5d3' => 1,
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/2pP4/4PP2/2NB4/PP4PP/R1BQK1NR b KQkq' => {
			eco => 'A67e',
			# TRANSLATORS: A67e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp1n1pbp/3p2p1/2pP4/4PP2/2NB4/PP4PP/R1BQK1NR w KQ' => {
			eco => 'A67f',
			# TRANSLATORS: A67f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbq1rk1/pp1n1pbp/3p2p1/2pP4/4PP2/2NB1N2/PP4PP/R1BQK2R b KQ' => {
			eco => 'A67g',
			# TRANSLATORS: A67g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O 10. Nf3
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O 10. Nf3'),
			moves => {
				'a7a6' => 1,
				'b8a6' => 1,
			},
		},
		'rnbq1rk1/1p1n1pbp/p2p2p1/2pP4/4PP2/2NB1N2/PP4PP/R1BQK2R w KQ' => {
			eco => 'A67h',
			# TRANSLATORS: A67h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O 10. Nf3 a6
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O 10. Nf3 a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbq1rk1/1p1n1pbp/p2p2p1/2pP4/P3PP2/2NB1N2/1P4PP/R1BQK2R b KQ' => {
			eco => 'A67i',
			# TRANSLATORS: A67i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O 10. Nf3 a6 11. a4
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O 10. Nf3 a6'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/2pP4/4PP2/2NB1N2/PP4PP/R1BQK2R w KQ' => {
			eco => 'A67j',
			# TRANSLATORS: A67j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. Bd3 O-O 10. Nf3 Na6
			variation => N__('Benoni: Four Pawns, Taimanov, 9. Bd3 O-O 10. Nf3 Na6'),
			moves => {
			},
		},
		'rnbqk2r/pp1n1pbp/3p2p1/1BpP4/P3PP2/2N5/1P4PP/R1BQK1NR b KQkq' => {
			eco => 'A67k',
			# TRANSLATORS: A67k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 (Zaitsev)'),
			moves => {
				'a7a6' => 1,
				'd8h4' => 1,
				'e8g8' => 1,
			},
		},
		'rnb1k2r/pp1n1pbp/3p2p1/1BpP4/P3PP1q/2N5/1P4PP/R1BQK1NR w KQkq' => {
			eco => 'A67l',
			# TRANSLATORS: A67l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 Qh4+
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 Qh4+'),
			moves => {
			},
		},
		'rnbqk2r/1p1n1pbp/p2p2p1/1BpP4/P3PP2/2N5/1P4PP/R1BQK1NR w KQkq' => {
			eco => 'A67m',
			# TRANSLATORS: A67m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 a6
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 a6'),
			moves => {
				'b5d3' => 1,
				'b5e2' => 1,
			},
		},
		'rnbqk2r/1p1n1pbp/p2p2p1/2pP4/P3PP2/2NB4/1P4PP/R1BQK1NR b KQkq' => {
			eco => 'A67n',
			# TRANSLATORS: A67n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 a6 10. Bd3
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 a6 10. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/1p1n1pbp/p2p2p1/2pP4/P3PP2/2N5/1P2B1PP/R1BQK1NR b KQkq' => {
			eco => 'A67o',
			# TRANSLATORS: A67o: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 a6 10. Be2
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 a6 10. Be2'),
			moves => {
			},
		},
		'rnbq1rk1/pp1n1pbp/3p2p1/1BpP4/P3PP2/2N5/1P4PP/R1BQK1NR w KQ' => {
			eco => 'A67p',
			# TRANSLATORS: A67p: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbq1rk1/pp1n1pbp/3p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQK2R b KQ' => {
			eco => 'A67q',
			# TRANSLATORS: A67q: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3'),
			moves => {
				'b8a6' => 1,
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQK2R w KQ' => {
			eco => 'A67r',
			# TRANSLATORS: A67r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3 Na6
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/pp1n1pbp/n2p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 b -' => {
			eco => 'A67r',
			# TRANSLATORS: A67r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3 Na6 11. O-O
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6'),
			moves => {
				'a6c7' => 1,
			},
		},
		'r1bq1rk1/ppnn1pbp/3p2p1/1BpP4/P3PP2/2N2N2/1P4PP/R1BQ1RK1 w -' => {
			eco => 'A67s',
			# TRANSLATORS: A67s: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Bb5+ Nfd7 9. a4 O-O 10. Nf3 Na6 11. O-O Nc7
			variation => N__('Benoni: Four Pawns, Taimanov, 9. a4 O-O 10. Nf3 Na6 11.O-O Nc7'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4PP2/2N2N2/PP4PP/R1BQKB1R b KQkq' => {
			eco => 'A68a',
			# TRANSLATORS: A68a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3
			variation => N__('Benoni: Four Pawns, 8. Nf3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PP2/2N2N2/PP4PP/R1BQKB1R w KQ' => {
			eco => 'A68a',
			# TRANSLATORS: A68a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O
			variation => N__('Benoni: Four Pawns, 8. Nf3 O-O'),
			moves => {
				'f1d3' => 1,
				'f1e2' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PP2/2NB1N2/PP4PP/R1BQK2R b KQ' => {
			eco => 'A68b',
			# TRANSLATORS: A68b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Bd3
			variation => N__('Benoni: Four Pawns, 9. Bd3'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PP2/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A68c',
			# TRANSLATORS: A68c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2
			variation => N__('Benoni: Four Pawns, 9. Be2'),
			moves => {
				'b7b5' => 1,
				'c8g4' => 1,
				'f8e8' => 1,
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1ppP4/4PP2/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A68d',
			# TRANSLATORS: A68d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5
			variation => N__('Benoni: Four Pawns, 9. Be2 b5'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1ppPP3/5P2/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A68e',
			# TRANSLATORS: A68e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5 10. e5
			variation => N__('Benoni: Four Pawns, 9. Be2 b5 10. e5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbq1rk1/p4pbp/5np1/1ppPp3/5P2/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A68f',
			# TRANSLATORS: A68f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5 10. e5 dxe5
			variation => N__('Benoni: Four Pawns, 9. Be2 b5 10. e5 dxe5'),
			moves => {
				'f4e5' => 1,
			},
		},
		'rnbq1rk1/p4pbp/5np1/1ppPP3/8/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A68f',
			# TRANSLATORS: A68f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5 10. e5 dxe5 11. fxe5
			variation => N__('Benoni: Four Pawns, 9. Be2 b5 10. e5 dxe5'),
			moves => {
				'f6g4' => 1,
			},
		},
		'rnbq1rk1/p4pbp/6p1/1ppPP3/6n1/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A68f',
			# TRANSLATORS: A68f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5 10. e5 dxe5 11. fxe5 Ng4
			variation => N__('Benoni: Four Pawns, 9. Be2 b5 10. e5 dxe5'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnbq1rk1/p4pbp/6p1/1ppPP1B1/6n1/2N2N2/PP2B1PP/R2QK2R b KQ' => {
			eco => 'A68g',
			# TRANSLATORS: A68g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 b5 10. e5 dxe5 11. fxe5 Ng4 12. Bg5
			variation => N__('Benoni: Four Pawns, 9. Be2 b5 10. e5 dxe5 11.fxe5 Ng4 12. Bg5'),
			moves => {
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP4/4PPb1/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A68h',
			# TRANSLATORS: A68h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4'),
			moves => {
				'e1g1' => 1,
				'e4e5' => 1,
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pPP3/5Pb1/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A68i',
			# TRANSLATORS: A68i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. e5
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10. e5'),
			moves => {
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP4/4PPb1/2N2N2/PP2B1PP/R1BQ1RK1 b -' => {
			eco => 'A68j',
			# TRANSLATORS: A68j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2q1rk1/pp1n1pbp/3p1np1/2pP4/4PPb1/2N2N2/PP2B1PP/R1BQ1RK1 w -' => {
			eco => 'A68k',
			# TRANSLATORS: A68k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7'),
			moves => {
				'h2h3' => 1,
			},
		},
		'r2q1rk1/pp1n1pbp/3p1np1/2pP4/4PPb1/2N2N1P/PP2B1P1/R1BQ1RK1 b -' => {
			eco => 'A68l',
			# TRANSLATORS: A68l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7 11. h3
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7 11. h3'),
			moves => {
				'g4f3' => 1,
			},
		},
		'r2q1rk1/pp1n1pbp/3p1np1/2pP4/4PP2/2N2b1P/PP2B1P1/R1BQ1RK1 w -' => {
			eco => 'A68l',
			# TRANSLATORS: A68l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7 11. h3 Bxf3
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7 11. h3'),
			moves => {
				'e2f3' => 1,
			},
		},
		'r2q1rk1/pp1n1pbp/3p1np1/2pP4/4PP2/2N2B1P/PP4P1/R1BQ1RK1 b -' => {
			eco => 'A68l',
			# TRANSLATORS: A68l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7 11. h3 Bxf3 12. Bxf3
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7 11. h3'),
			moves => {
				'f8e8' => 1,
			},
		},
		'r2qr1k1/pp1n1pbp/3p1np1/2pP4/4PP2/2N2B1P/PP4P1/R1BQ1RK1 w -' => {
			eco => 'A68m',
			# TRANSLATORS: A68m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7 11. h3 Bxf3 12. Bxf3 Re8
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7 11. h3 Bxf3 12.Bxf3 Re8'),
			moves => {
				'f1e1' => 1,
			},
		},
		'r2qr1k1/pp1n1pbp/3p1np1/2pP4/4PP2/2N2B1P/PP4P1/R1BQR1K1 b -' => {
			eco => 'A68n',
			# TRANSLATORS: A68n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Bg4 10. O-O Nbd7 11. h3 Bxf3 12. Bxf3 Re8 13. Re1
			variation => N__('Benoni: Four Pawns, 9. Be2 Bg4 10.O-O Nbd7 11. h3 Bxf3 12.Bxf3 Re8 13. Re1'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4PP2/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A69a',
			# TRANSLATORS: A69a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8
			variation => N__('Benoni: Four Pawns, Main Line'),
			moves => {
				'e1g1' => 1,
				'e4e5' => 1,
				'f3d2' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4PP2/2N2N2/PP2B1PP/R1BQ1RK1 b -' => {
			eco => 'A69b',
			# TRANSLATORS: A69b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. O-O
			variation => N__('Benoni: Four Pawns, Main Line, 10.O-O'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4PP2/2N5/PP1NB1PP/R1BQK2R b KQ' => {
			eco => 'A69c',
			# TRANSLATORS: A69c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. Nd2
			variation => N__('Benoni: Four Pawns, Main Line, 10. Nd2'),
			moves => {
				'a7a6' => 1,
				'b8d7' => 1,
			},
		},
		'rnbqr1k1/1p3pbp/p2p1np1/2pP4/4PP2/2N5/PP1NB1PP/R1BQK2R w KQ' => {
			eco => 'A69d',
			# TRANSLATORS: A69d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. Nd2 a6
			variation => N__('Benoni: Four Pawns, Main Line, 10. Nd2 a6'),
			moves => {
			},
		},
		'r1bqr1k1/pp1n1pbp/3p1np1/2pP4/4PP2/2N5/PP1NB1PP/R1BQK2R w KQ' => {
			eco => 'A69e',
			# TRANSLATORS: A69e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. Nd2 Nbd7
			variation => N__('Benoni: Four Pawns, Main Line, 10. Nd2 Nbd7'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pPP3/5P2/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A69f',
			# TRANSLATORS: A69f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/5np1/2pPp3/5P2/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A69f',
			# TRANSLATORS: A69f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5'),
			moves => {
				'f4e5' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/5np1/2pPP3/8/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A69f',
			# TRANSLATORS: A69f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5'),
			moves => {
				'f6g4' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/6p1/2pPP3/6n1/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'A69f',
			# TRANSLATORS: A69f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5'),
			moves => {
				'c1g5' => 1,
				'e1g1' => 1,
				'e5e6' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/6p1/2pPP3/6n1/2N2N2/PP2B1PP/R1BQ1RK1 b -' => {
			eco => 'A69g',
			# TRANSLATORS: A69g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. O-O
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12.O-O'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/4P1p1/2pP4/6n1/2N2N2/PP2B1PP/R1BQK2R b KQ' => {
			eco => 'A69h',
			# TRANSLATORS: A69h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. e6
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. e6'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/6p1/2pPP1B1/6n1/2N2N2/PP2B1PP/R2QK2R b KQ' => {
			eco => 'A69i',
			# TRANSLATORS: A69i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5'),
			moves => {
				'd8b6' => 1,
				'f7f6' => 1,
			},
		},
		'rnbqr1k1/pp4bp/5pp1/2pPP1B1/6n1/2N2N2/PP2B1PP/R2QK2R w KQ' => {
			eco => 'A69j',
			# TRANSLATORS: A69j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 f6
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 f6'),
			moves => {
			},
		},
		'rnb1r1k1/pp3pbp/1q4p1/2pPP1B1/6n1/2N2N2/PP2B1PP/R2QK2R w KQ' => {
			eco => 'A69k',
			# TRANSLATORS: A69k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 Qb6
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 Qb6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnb1r1k1/pp3pbp/1q4p1/2pPP1B1/6n1/2N2N2/PP2B1PP/R2Q1RK1 b -' => {
			eco => 'A69k',
			# TRANSLATORS: A69k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 Qb6 13. O-O
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 Qb6'),
			moves => {
				'g4e5' => 1,
			},
		},
		'rnb1r1k1/pp3pbp/1q4p1/2pPn1B1/8/2N2N2/PP2B1PP/R2Q1RK1 w -' => {
			eco => 'A69k',
			# TRANSLATORS: A69k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 Qb6 13. O-O Nxe5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 Qb6'),
			moves => {
				'd5d6' => 1,
				'f3e5' => 1,
			},
		},
		'rnb1r1k1/pp3pbp/1q4p1/2pPN1B1/8/2N5/PP2B1PP/R2Q1RK1 b -' => {
			eco => 'A69l',
			# TRANSLATORS: A69l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 Qb6 13. O-O Nxe5 14. Nxe5
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 Qb6 13.O-O Nxe5 14.Nxe5'),
			moves => {
			},
		},
		'rnb1r1k1/pp3pbp/1q1P2p1/2p1n1B1/8/2N2N2/PP2B1PP/R2Q1RK1 b -' => {
			eco => 'A69m',
			# TRANSLATORS: A69m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. f4 Bg7 8. Nf3 O-O 9. Be2 Re8 10. e5 dxe5 11. fxe5 Ng4 12. Bg5 Qb6 13. O-O Nxe5 14. d6
			variation => N__('Benoni: Four Pawns, Main Line, 10. e5: 12. Bg5 Qb6 13.O-O Nxe5 14. d6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3p1p/3p1np1/2pP4/4P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A70a',
			# TRANSLATORS: A70a: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3
			variation => N__('Benoni: Classical'),
			moves => {
				'a7a6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/1p3p1p/p2p1np1/2pP4/4P3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A70b',
			# TRANSLATORS: A70b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 a6
			variation => N__('Benoni: Classical, 7... a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbqkb1r/1p3p1p/p2p1np1/2pP4/P3P3/2N2N2/1P3PPP/R1BQKB1R b KQkq' => {
			eco => 'A70b',
			# TRANSLATORS: A70b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 a6 8. a4
			variation => N__('Benoni: Classical, 7... a6 8. a4'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn1qkb1r/1p3p1p/p2p1np1/2pP4/P3P1b1/2N2N2/1P3PPP/R1BQKB1R w KQkq' => {
			eco => 'A70b',
			# TRANSLATORS: A70b: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 a6 8. a4 Bg4
			variation => N__('Benoni: Classical, 7... a6 8. a4 Bg4'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qkb1r/1p3p1p/p2p1np1/2pP4/P3P1b1/2N2N2/1P2BPPP/R1BQK2R b KQkq' => {
			eco => 'A70c',
			# TRANSLATORS: A70c: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 a6 8. a4 Bg4 9. Be2
			variation => N__('Benoni: Classical, 7... a6 8. a4 Bg4 9. Be2'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'A70d',
			# TRANSLATORS: A70d: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7
			variation => N__('Benoni: Classical, 7... Bg7'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'd1a4' => 1,
				'f1d3' => 1,
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'A70e',
			# TRANSLATORS: A70e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2
			variation => N__('Benoni: Classical, 8. Be2'),
			moves => {
				'a7a6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/1p3pbp/p2p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQK2R w KQkq' => {
			eco => 'A70e',
			# TRANSLATORS: A70e: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 a6
			variation => N__('Benoni: Classical, 8. Be2 a6'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/Q3P3/2N2N2/PP3PPP/R1B1KB1R b KQkq' => {
			eco => 'A70f',
			# TRANSLATORS: A70f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Qa4+
			variation => N__('Benoni: Classical, 8. Qa4+'),
			moves => {
				'c8d7' => 1,
			},
		},
		'rn1qk2r/pp1b1pbp/3p1np1/2pP4/Q3P3/2N2N2/PP3PPP/R1B1KB1R w KQkq' => {
			eco => 'A70f',
			# TRANSLATORS: A70f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Qa4+ Bd7
			variation => N__('Benoni: Classical, 8. Qa4+'),
			moves => {
				'a4b3' => 1,
			},
		},
		'rn1qk2r/pp1b1pbp/3p1np1/2pP4/4P3/1QN2N2/PP3PPP/R1B1KB1R b KQkq' => {
			eco => 'A70f',
			# TRANSLATORS: A70f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Qa4+ Bd7 9. Qb3
			variation => N__('Benoni: Classical, 8. Qa4+'),
			moves => {
				'd8c7' => 1,
			},
		},
		'rn2k2r/ppqb1pbp/3p1np1/2pP4/4P3/1QN2N2/PP3PPP/R1B1KB1R w KQkq' => {
			eco => 'A70f',
			# TRANSLATORS: A70f: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Qa4+ Bd7 9. Qb3 Qc7
			variation => N__('Benoni: Classical, 8. Qa4+ Bd7 9. Qb3 Qc7'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4PB2/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'A70g',
			# TRANSLATORS: A70g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bf4
			variation => N__('Benoni: Classical, 8. Bf4'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PB2/2N2N2/PP3PPP/R2QKB1R w KQ' => {
			eco => 'A70g',
			# TRANSLATORS: A70g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bf4 O-O
			variation => N__('Benoni: Classical, 8. Bf4'),
			moves => {
				'f3d2' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PB2/2N5/PP1N1PPP/R2QKB1R b KQ' => {
			eco => 'A70g',
			# TRANSLATORS: A70g: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bf4 O-O 9. Nd2
			variation => N__('Benoni: Classical, 8. Bf4 O-O 9. Nd2'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2NB1N2/PP3PPP/R1BQK2R b KQkq' => {
			eco => 'A70h',
			# TRANSLATORS: A70h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bd3
			variation => N__('Benoni: Classical, 8. Bd3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2NB1N2/PP3PPP/R1BQK2R w KQ' => {
			eco => 'A70h',
			# TRANSLATORS: A70h: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bd3 O-O
			variation => N__('Benoni: Classical, 8. Bd3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2NB1N2/PP3PPP/R1BQ1RK1 b -' => {
			eco => 'A70i',
			# TRANSLATORS: A70i: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bd3 O-O 9. O-O
			variation => N__('Benoni: Classical, 8. Bd3 O-O 9.O-O'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP4/4P3/2N2N1P/PP3PP1/R1BQKB1R b KQkq' => {
			eco => 'A70j',
			# TRANSLATORS: A70j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3
			variation => N__('Benoni: Classical, 8. h3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2N2N1P/PP3PP1/R1BQKB1R w KQ' => {
			eco => 'A70j',
			# TRANSLATORS: A70j: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O
			variation => N__('Benoni: Classical, 8. h3'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R b KQ' => {
			eco => 'A70k',
			# TRANSLATORS: A70k: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3'),
			moves => {
				'a7a6' => 1,
				'b7b5' => 1,
				'b8a6' => 1,
				'c8d7' => 1,
				'f8e8' => 1,
			},
		},
		'rn1q1rk1/pp1b1pbp/3p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70l',
			# TRANSLATORS: A70l: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Bd7
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Bd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70m',
			# TRANSLATORS: A70m: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Na6
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Na6'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70n',
			# TRANSLATORS: A70n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Re8
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Re8'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQ1RK1 b -' => {
			eco => 'A70n',
			# TRANSLATORS: A70n: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Re8 10. O-O
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Re8'),
			moves => {
				'c5c4' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/3P4/2p1P3/2NB1N1P/PP3PP1/R1BQ1RK1 w -' => {
			eco => 'A70o',
			# TRANSLATORS: A70o: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 Re8 10. O-O c4
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 Re8 10.O-O c4'),
			moves => {
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70p',
			# TRANSLATORS: A70p: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6'),
			moves => {
				'a2a4' => 1,
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/4P3/2NB1N1P/PP3PP1/R1BQ1RK1 b -' => {
			eco => 'A70p',
			# TRANSLATORS: A70p: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. O-O
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbq1rk1/5pbp/p2p1np1/1ppP4/4P3/2NB1N1P/PP3PP1/R1BQ1RK1 w -' => {
			eco => 'A70q',
			# TRANSLATORS: A70q: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. O-O b5
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10.O-O b5'),
			moves => {
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQK2R b KQ' => {
			eco => 'A70r',
			# TRANSLATORS: A70r: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. a4
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQK2R w KQ' => {
			eco => 'A70s',
			# TRANSLATORS: A70s: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQ1RK1 b -' => {
			eco => 'A70s',
			# TRANSLATORS: A70s: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7 11. O-O
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7'),
			moves => {
				'f8e8' => 1,
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/2NB1N1P/1P3PP1/R1BQ1RK1 w -' => {
			eco => 'A70t',
			# TRANSLATORS: A70t: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7 11. O-O Re8
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 a6 10. a4 Nbd7 11.O-O Re8'),
			moves => {
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1ppP4/4P3/2NB1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70u',
			# TRANSLATORS: A70u: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5'),
			moves => {
				'c3b5' => 1,
				'd3b5' => 1,
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1BpP4/4P3/2N2N1P/PP3PP1/R1BQK2R b KQ' => {
			eco => 'A70v',
			# TRANSLATORS: A70v: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Bxb5
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Bxb5'),
			moves => {
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1NpP4/4P3/3B1N1P/PP3PP1/R1BQK2R b KQ' => {
			eco => 'A70w',
			# TRANSLATORS: A70w: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Nxb5
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Nxb5'),
			moves => {
				'f6e4' => 1,
				'f8e8' => 1,
			},
		},
		'rnbq1rk1/p4pbp/3p2p1/1NpP4/4n3/3B1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70w',
			# TRANSLATORS: A70w: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Nxb5 Nxe4
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Nxb5 Nxe4'),
			moves => {
			},
		},
		'rnbqr1k1/p4pbp/3p1np1/1NpP4/4P3/3B1N1P/PP3PP1/R1BQK2R w KQ' => {
			eco => 'A70x',
			# TRANSLATORS: A70x: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Nxb5 Re8
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Nxb5 Re8'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqr1k1/p4pbp/3p1np1/1NpP4/4P3/3B1N1P/PP3PP1/R1BQ1RK1 b -' => {
			eco => 'A70x',
			# TRANSLATORS: A70x: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Nxb5 Re8 11. O-O
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Nxb5 Re8'),
			moves => {
				'f6e4' => 1,
			},
		},
		'rnbqr1k1/p4pbp/3p2p1/1NpP4/4n3/3B1N1P/PP3PP1/R1BQ1RK1 w -' => {
			eco => 'A70y',
			# TRANSLATORS: A70y: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. h3 O-O 9. Bd3 b5 10. Nxb5 Re8 11. O-O Nxe4
			variation => N__('Benoni: Classical, 8. h3 O-O 9. Bd3 b5 10.Nxb5 Re8 11.O-O Nxe4'),
			moves => {
			},
		},
		'rnbqk2r/pp3pbp/3p1np1/2pP2B1/4P3/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5
			variation => N__('Benoni: Classical, 8. Bg5'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rnbqk2r/pp3pb1/3p1npp/2pP2B1/4P3/2N2N2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6
			variation => N__('Benoni: Classical, 8. Bg5'),
			moves => {
				'g5h4' => 1,
			},
		},
		'rnbqk2r/pp3pb1/3p1npp/2pP4/4P2B/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4
			variation => N__('Benoni: Classical, 8. Bg5'),
			moves => {
				'a7a6' => 1,
				'g6g5' => 1,
			},
		},
		'rnbqk2r/1p3pb1/p2p1npp/2pP4/4P2B/2N2N2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4 a6
			variation => N__('Benoni: Classical, 8. Bg5 h6 9. Bh4 a6'),
			moves => {
				'f3d2' => 1,
			},
		},
		'rnbqk2r/1p3pb1/p2p1npp/2pP4/4P2B/2N5/PP1N1PPP/R2QKB1R b KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4 a6 10. Nd2
			variation => N__('Benoni: Classical, 8. Bg5 h6 9. Bh4 a6 10. Nd2'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqk2r/5pb1/p2p1npp/1ppP4/4P2B/2N5/PP1N1PPP/R2QKB1R w KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4 a6 10. Nd2 b5
			variation => N__('Benoni: Classical, 8. Bg5 h6 9. Bh4 a6 10. Nd2'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnbqk2r/5pb1/p2p1npp/1ppP4/4P2B/2N5/PP1NBPPP/R2QK2R b KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4 a6 10. Nd2 b5 11. Be2
			variation => N__('Benoni: Classical, 8. Bg5 h6 9. Bh4 a6 10. Nd2 b5 11. Be2'),
			moves => {
			},
		},
		'rnbqk2r/pp3pb1/3p1n1p/2pP2p1/4P2B/2N2N2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'A71',
			# TRANSLATORS: A71: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Bg5 h6 9. Bh4 g5
			variation => N__('Benoni: Classical, 8. Bg5 h6 9. Bh4 g5'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQK2R w KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O
			variation => N__('Benoni: Classical, 8. Be2 O-O'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'e1g1' => 1,
				'f3d2' => 1,
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4PB2/2N2N2/PP2BPPP/R2QK2R b KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bf4
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bf4'),
			moves => {
				'a7a6' => 1,
				'b7b5' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/4PB2/2N2N2/PP2BPPP/R2QK2R w KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bf4 a6
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bf4 a6'),
			moves => {
			},
		},
		'rnbq1rk1/p4pbp/3p1np1/1ppP4/4PB2/2N2N2/PP2BPPP/R2QK2R w KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bf4 b5
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bf4 b5'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP2B1/4P3/2N2N2/PP2BPPP/R2QK2R b KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bg5
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bg5'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rnbq1rk1/pp3pb1/3p1npp/2pP2B1/4P3/2N2N2/PP2BPPP/R2QK2R w KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bg5 h6
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bg5'),
			moves => {
				'g5h4' => 1,
			},
		},
		'rnbq1rk1/pp3pb1/3p3p/2pP2pn/4P3/2N3B1/PP1NBPPP/R2QK2R b KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Bg5 h6 10. Bh4 g5 11. Bg3 Nh5 12. Nd2
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Bg5 h6 10. Bh4 g5 11. Bg3 Nh5 12. Nd2'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQK2R b KQ' => {
			eco => 'A72',
			# TRANSLATORS: A72: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. Nd2
			variation => N__('Benoni: Classical, 8. Be2 O-O 9. Nd2'),
			moves => {
			},
		},
		'rnbq1rk1/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 b -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O
			variation => N__('Benoni: Classical, 9.O-O'),
			moves => {
				'a7a6' => 1,
				'b8a6' => 1,
				'b8d7' => 1,
				'c8g4' => 1,
				'f8e8' => 1,
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6
			variation => N__('Benoni: Classical, 9.O-O a6'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6
			variation => N__('Benoni: Classical, 9.O-O Na6'),
			moves => {
				'f3d2' => 1,
			},
		},
		'r1bq1rk1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 b -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6 10. Nd2
			variation => N__('Benoni: Classical, 9.O-O Na6 10. Nd2'),
			moves => {
				'a6c7' => 1,
			},
		},
		'r1bq1rk1/ppn2pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Na6 10. Nd2 Nc7
			variation => N__('Benoni: Classical, 9.O-O Na6 10. Nd2 Nc7'),
			moves => {
			},
		},
		'r1bq1rk1/pp1n1pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Nbd7
			variation => N__('Benoni: Classical, 9.O-O Nbd7'),
			moves => {
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP4/4P1b1/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Bg4
			variation => N__('Benoni: Classical, 9.O-O Bg4'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'h2h3' => 1,
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP4/4PBb1/2N2N2/PP2BPPP/R2Q1RK1 b -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Bg4 10. Bf4
			variation => N__('Benoni: Classical, 9.O-O Bg4 10. Bf4'),
			moves => {
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP2B1/4P1b1/2N2N2/PP2BPPP/R2Q1RK1 b -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Bg4 10. Bg5
			variation => N__('Benoni: Classical, 9.O-O Bg4 10. Bg5'),
			moves => {
			},
		},
		'rn1q1rk1/pp3pbp/3p1np1/2pP4/4P1b1/2N2N1P/PP2BPP1/R1BQ1RK1 b -' => {
			eco => 'A73',
			# TRANSLATORS: A73: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Bg4 10. h3
			variation => N__('Benoni: Classical, 9.O-O Bg4 10. h3'),
			moves => {
			},
		},
		'rnbq1rk1/1p3pbp/p2p1np1/2pP4/P3P3/2N2N2/1P2BPPP/R1BQ1RK1 b -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4'),
			moves => {
				'b8d7' => 1,
				'c8g4' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N2N2/1P2BPPP/R1BQ1RK1 w -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'f3d2' => 1,
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 b -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bf4
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bf4'),
			moves => {
				'd8e7' => 1,
			},
		},
		'r1b2rk1/1p1nqpbp/p2p1np1/2pP4/P3PB2/2N2N2/1P2BPPP/R2Q1RK1 w -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bf4 Qe7
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bf4 Qe7'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP2B1/P3P3/2N2N2/1P2BPPP/R2Q1RK1 b -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Bg5
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N5/1P1NBPPP/R1BQ1RK1 b -' => {
			eco => 'A74',
			# TRANSLATORS: A74: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Nbd7 11. Nd2
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Nbd7 11. Nd2'),
			moves => {
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3P1b1/2N2N2/1P2BPPP/R1BQ1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'f3d2' => 1,
				'h2h3' => 1,
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3PBb1/2N2N2/1P2BPPP/R2Q1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4'),
			moves => {
				'd8e7' => 1,
				'f8e8' => 1,
				'g4f3' => 1,
			},
		},
		'rn3rk1/1p2qpbp/p2p1np1/2pP4/P3PBb1/2N2N2/1P2BPPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Qe7
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Qe7'),
			moves => {
			},
		},
		'rn1qr1k1/1p3pbp/p2p1np1/2pP4/P3PBb1/2N2N2/1P2BPPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Re8
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Re8'),
			moves => {
				'f3d2' => 1,
			},
		},
		'rn1qr1k1/1p3pbp/p2p1np1/2pP4/P3PBb1/2N5/1P1NBPPP/R2Q1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Re8 12. Nd2
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Re8'),
			moves => {
				'g4e2' => 1,
			},
		},
		'rn1qr1k1/1p3pbp/p2p1np1/2pP4/P3PB2/2N5/1P1NbPPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Re8 12. Nd2 Bxe2
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Re8'),
			moves => {
				'd1e2' => 1,
			},
		},
		'rn1qr1k1/1p3pbp/p2p1np1/2pP4/P3PB2/2N5/1P1NQPPP/R4RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Re8 12. Nd2 Bxe2 13. Qxe2
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Re8'),
			moves => {
				'f6h5' => 1,
			},
		},
		'rn1qr1k1/1p3pbp/p2p2p1/2pP3n/P3PB2/2N5/1P1NQPPP/R4RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Re8 12. Nd2 Bxe2 13. Qxe2 Nh5
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Re8 12. Nd2 Bxe2 13.Qxe2 Nh5'),
			moves => {
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3PB2/2N2b2/1P2BPPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3'),
			moves => {
				'e2f3' => 1,
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3PB2/2N2B2/1P3PPP/R2Q1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12. Bxf3
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3'),
			moves => {
				'd8e7' => 1,
			},
		},
		'rn3rk1/1p2qpbp/p2p1np1/2pP4/P3PB2/2N2B2/1P3PPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12. Bxf3 Qe7
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12.Bxf3 Qe7'),
			moves => {
				'f1e1' => 1,
			},
		},
		'rn3rk1/1p2qpbp/p2p1np1/2pP4/P3PB2/2N2B2/1P3PPP/R2QR1K1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12. Bxf3 Qe7 13. Re1
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12.Bxf3 Qe7 13. Re1'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r4rk1/1p1nqpbp/p2p1np1/2pP4/P3PB2/2N2B2/1P3PPP/R2QR1K1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12. Bxf3 Qe7 13. Re1 Nbd7
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12.Bxf3 Qe7 13. Re1'),
			moves => {
				'a4a5' => 1,
			},
		},
		'r4rk1/1p1nqpbp/p2p1np1/P1pP4/4PB2/2N2B2/1P3PPP/R2QR1K1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12. Bxf3 Qe7 13. Re1 Nbd7 14. a5
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bf4 Bxf3 12.Bxf3 Qe7 with 14. a5'),
			moves => {
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP2B1/P3P1b1/2N2N2/1P2BPPP/R2Q1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bg5
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bg5'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rn1q1rk1/1p3pb1/p2p1npp/2pP2B1/P3P1b1/2N2N2/1P2BPPP/R2Q1RK1 w -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Bg5 h6
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Bg5 h6'),
			moves => {
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3P1b1/2N5/1P1NBPPP/R1BQ1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. Nd2
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. Nd2'),
			moves => {
			},
		},
		'rn1q1rk1/1p3pbp/p2p1np1/2pP4/P3P1b1/2N2N1P/1P2BPP1/R1BQ1RK1 b -' => {
			eco => 'A75',
			# TRANSLATORS: A75: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O a6 10. a4 Bg4 11. h3
			variation => N__('Benoni: Classical, 9.O-O a6 10. a4 Bg4 11. h3'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8
			variation => N__('Benoni: Classical, Main Line'),
			moves => {
				'd1c2' => 1,
				'f3d2' => 1,
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4P3/2N2N2/PPQ1BPPP/R1B2RK1 b -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2
			variation => N__('Benoni: Classical, Main Line, 10. Qc2'),
			moves => {
				'b8a6' => 1,
				'c8g4' => 1,
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N2N2/PPQ1BPPP/R1B2RK1 w -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2 Na6
			variation => N__('Benoni: Classical, Main Line, 10. Qc2 Na6'),
			moves => {
				'c1f4' => 1,
				'f1e1' => 1,
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4PB2/2N2N2/PPQ1BPPP/R4RK1 b -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2 Na6 11. Bf4
			variation => N__('Benoni: Classical, Main Line, 10. Qc2 Na6 11. Bf4'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N2N2/PPQ1BPPP/R1B1R1K1 b -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2 Na6 11. Re1
			variation => N__('Benoni: Classical, Main Line, 10. Qc2 Na6 11. Re1'),
			moves => {
				'c8g4' => 1,
			},
		},
		'r2qr1k1/pp3pbp/n2p1np1/2pP4/4P1b1/2N2N2/PPQ1BPPP/R1B1R1K1 w -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2 Na6 11. Re1 Bg4
			variation => N__('Benoni: Classical, Main Line, 10. Qc2 Na6 11. Re1 Bg4'),
			moves => {
			},
		},
		'rn1qr1k1/pp3pbp/3p1np1/2pP4/4P1b1/2N2N2/PPQ1BPPP/R1B2RK1 w -' => {
			eco => 'A76',
			# TRANSLATORS: A76: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Qc2 Bg4
			variation => N__('Benoni: Classical, Main Line, 10. Qc2 Bg4'),
			moves => {
			},
		},
		'rnbqr1k1/pp3pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2
			variation => N__('Benoni: Classical, Main Line, 10. Nd2'),
			moves => {
				'a7a6' => 1,
				'b8a6' => 1,
				'b8d7' => 1,
			},
		},
		'rnbqr1k1/1p3pbp/p2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 a6
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 a6'),
			moves => {
			},
		},
		'r1bqr1k1/pp1n1pbp/3p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bqr1k1/pp1n1pbp/3p1np1/2pP4/P3P3/2N5/1P1NBPPP/R1BQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4'),
			moves => {
				'a7a6' => 1,
				'd7e5' => 1,
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N5/1P1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 a6
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 a6'),
			moves => {
				'a1a3' => 1,
				'd1c2' => 1,
				'h2h3' => 1,
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/R1N5/1P1NBPPP/2BQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 a6 12. Ra3
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 a6 12. Ra3'),
			moves => {
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N5/1PQNBPPP/R1B2RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 a6 12. Qc2
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 a6 12. Qc2'),
			moves => {
			},
		},
		'r1bqr1k1/1p1n1pbp/p2p1np1/2pP4/P3P3/2N4P/1P1NBPP1/R1BQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 a6 12. h3
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 a6 12. h3'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/3p1np1/2pPn3/P3P3/2N5/1P1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 Ne5
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 Ne5'),
			moves => {
				'a1a3' => 1,
				'd1c2' => 1,
				'd2b1' => 1,
				'f1e1' => 1,
			},
		},
		'r1bqr1k1/pp3pbp/3p1np1/2pPn3/P3P3/R1N5/1P1NBPPP/2BQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 Ne5 12. Ra3
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 Ne5 12. Ra3'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/3p1np1/2pPn3/P3P3/2N5/1PQNBPPP/R1B2RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 Ne5 12. Qc2
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 Ne5 12. Qc2'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/3p1np1/2pPn3/P3P3/2N5/1P2BPPP/RNBQ1RK1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 Ne5 12. Ndb1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 Ne5 12.Ndb1'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/3p1np1/2pPn3/P3P3/2N5/1P1NBPPP/R1BQR1K1 b -' => {
			eco => 'A77',
			# TRANSLATORS: A77: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Nbd7 11. a4 Ne5 12. Re1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Nbd7 11. a4 Ne5 12. Re1'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1RK1 w -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6'),
			moves => {
				'a1b1' => 1,
				'f1e1' => 1,
				'f2f3' => 1,
				'f2f4' => 1,
				'g1h1' => 1,
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/1RBQ1RK1 b -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. Rb1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. Rb1'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQR1K1 b -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. Re1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. Re1'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4PP2/2N5/PP1NB1PP/R1BQ1RK1 b -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f4
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f4'),
			moves => {
				'a6c7' => 1,
			},
		},
		'r1bqr1k1/ppn2pbp/3p1np1/2pP4/4PP2/2N5/PP1NB1PP/R1BQ1RK1 w -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f4 Nc7
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f4 Nc7'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N5/PP1NBPPP/R1BQ1R1K b -' => {
			eco => 'A78',
			# TRANSLATORS: A78: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. Kh1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. Kh1'),
			moves => {
			},
		},
		'r1bqr1k1/pp3pbp/n2p1np1/2pP4/4P3/2N2P2/PP1NB1PP/R1BQ1RK1 b -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3'),
			moves => {
				'a6c7' => 1,
			},
		},
		'r1bqr1k1/ppn2pbp/3p1np1/2pP4/4P3/2N2P2/PP1NB1PP/R1BQ1RK1 w -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bqr1k1/ppn2pbp/3p1np1/2pP4/P3P3/2N2P2/1P1NB1PP/R1BQ1RK1 b -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7 12. a4
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7 12. a4'),
			moves => {
				'b7b6' => 1,
				'f6d7' => 1,
			},
		},
		'r1bqr1k1/p1n2pbp/1p1p1np1/2pP4/P3P3/2N2P2/1P1NB1PP/R1BQ1RK1 w -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7 12. a4 b6
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7 12. a4 b6'),
			moves => {
				'd2c4' => 1,
				'g1h1' => 1,
			},
		},
		'r1bqr1k1/p1n2pbp/1p1p1np1/2pP4/P1N1P3/2N2P2/1P2B1PP/R1BQ1RK1 b -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7 12. a4 b6 13. Nc4
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7 12. a4 b6 13. Nc4'),
			moves => {
			},
		},
		'r1bqr1k1/p1n2pbp/1p1p1np1/2pP4/P3P3/2N2P2/1P1NB1PP/R1BQ1R1K b -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7 12. a4 b6 13. Kh1
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7 12. a4 b6 13. Kh1'),
			moves => {
			},
		},
		'r1bqr1k1/ppnn1pbp/3p2p1/2pP4/P3P3/2N2P2/1P1NB1PP/R1BQ1RK1 w -' => {
			eco => 'A79',
			# TRANSLATORS: A79: 1. d4 Nf6 2. c4 c5 3. d5 e6 4. Nc3 exd5 5. cxd5 d6 6. e4 g6 7. Nf3 Bg7 8. Be2 O-O 9. O-O Re8 10. Nd2 Na6 11. f3 Nc7 12. a4 Nd7
			variation => N__('Benoni: Classical, Main Line, 10. Nd2 Na6 11. f3 Nc7 12. a4 Nd7'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/8/PPP1PPPP/RNBQKBNR w KQkq' => {
			eco => 'A80a',
			# TRANSLATORS: A80a: 1. d4 f5
			variation => N__('Dutch'),
			moves => {
				'b1c3' => 1,
				'c1g5' => 1,
				'c2c4' => 1,
				'd1d3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'g2g4' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P2P1/8/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A80b',
			# TRANSLATORS: A80b: 1. d4 f5 2. g4
			variation => N__('Dutch: Krejcik Gambit'),
			moves => {
				'f5g4' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/3P2p1/8/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A80b',
			# TRANSLATORS: A80b: 1. d4 f5 2. g4 fxg4
			variation => N__('Dutch: Krejcik Gambit Accepted'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/7P/PPP1PPP1/RNBQKBNR b KQkq' => {
			eco => 'A80c',
			# TRANSLATORS: A80c: 1. d4 f5 2. h3
			variation => N__('Dutch: Korchnoi Attack'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/7P/PPP1PPP1/RNBQKBNR w KQkq' => {
			eco => 'A80c',
			# TRANSLATORS: A80c: 1. d4 f5 2. h3 Nf6
			variation => N__('Dutch: Korchnoi Attack'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P2P1/7P/PPP1PP2/RNBQKBNR b KQkq' => {
			eco => 'A80c',
			# TRANSLATORS: A80c: 1. d4 f5 2. h3 Nf6 3. g4
			variation => N__('Dutch: Korchnoi, Janzen Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5pB1/3P4/8/PPP1PPPP/RN1QKBNR b KQkq' => {
			eco => 'A80d',
			# TRANSLATORS: A80d: 1. d4 f5 2. Bg5
			variation => N__('Dutch: 2. Bg5'),
			moves => {
				'c7c6' => 1,
				'd7d5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
				'h7h6' => 1,
			},
		},
		'rnbqkbnr/pp1pp1pp/2p5/5pB1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A80d',
			# TRANSLATORS: A80d: 1. d4 f5 2. Bg5 c6
			variation => N__('Dutch: 2. Bg5 c6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1p1pp/8/3p1pB1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A80d',
			# TRANSLATORS: A80d: 1. d4 f5 2. Bg5 d5
			variation => N__('Dutch: 2. Bg5 d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5pB1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A80e',
			# TRANSLATORS: A80e: 1. d4 f5 2. Bg5 Nf6
			variation => N__('Dutch: 2. Bg5 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1p1/7p/5pB1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A80f',
			# TRANSLATORS: A80f: 1. d4 f5 2. Bg5 h6
			variation => N__('Dutch: 2. Bg5 h6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5pB1/3P4/8/PPP1PPPP/RN1QKBNR w KQkq' => {
			eco => 'A80g',
			# TRANSLATORS: A80g: 1. d4 f5 2. Bg5 g6
			variation => N__('Dutch: 2. Bg5 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5pB1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A80h',
			# TRANSLATORS: A80h: 1. d4 f5 2. Bg5 g6 3. Nc3
			variation => N__('Dutch: 2. Bg5 g6 3. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/3Q4/PPP1PPPP/RNB1KBNR b KQkq' => {
			eco => 'A80i',
			# TRANSLATORS: A80i: 1. d4 f5 2. Qd3
			variation => N__('Dutch: Alapin'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/3p4/5p2/3P4/3Q4/PPP1PPPP/RNB1KBNR w KQkq' => {
			eco => 'A80i',
			# TRANSLATORS: A80i: 1. d4 f5 2. Qd3 d6
			variation => N__('Dutch: Alapin'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/3p4/5p2/3P2P1/3Q4/PPP1PP1P/RNB1KBNR b KQkq' => {
			eco => 'A80i',
			# TRANSLATORS: A80i: 1. d4 f5 2. Qd3 d6 3. g4
			variation => N__('Dutch: Alapin, Manhattan Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/3P4/3Q4/PPP1PPPP/RNB1KBNR w KQkq' => {
			eco => 'A80i',
			# TRANSLATORS: A80i: 1. d4 f5 2. Qd3 e6
			variation => N__('Dutch: Alapin'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/3P2P1/3Q4/PPP1PP1P/RNB1KBNR b KQkq' => {
			eco => 'A80i',
			# TRANSLATORS: A80i: 1. d4 f5 2. Qd3 e6 3. g4
			variation => N__('Dutch: Von Pretzel Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/2N5/PPP1PPPP/R1BQKBNR b KQkq' => {
			eco => 'A80j',
			# TRANSLATORS: A80j: 1. d4 f5 2. Nc3
			variation => N__('Dutch: 2. Nc3'),
			moves => {
				'd7d5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/8/3p1p2/3P4/2N5/PPP1PPPP/R1BQKBNR w KQkq' => {
			eco => 'A80k',
			# TRANSLATORS: A80k: 1. d4 f5 2. Nc3 d5
			variation => N__('Dutch: 2. Nc3 d5'),
			moves => {
				'c1g5' => 1,
				'e2e4' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/8/3p1p2/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A80k',
			# TRANSLATORS: A80k: 1. d4 f5 2. Nc3 d5 3. e4
			variation => N__('Dutch: 2. Nc3 d5, Euwe Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1p1pp/8/3p1pB1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A80l',
			# TRANSLATORS: A80l: 1. d4 f5 2. Nc3 d5 3. Bg5
			variation => N__('Dutch: 2. Nc3 d5 3. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/2N5/PPP1PPPP/R1BQKBNR w KQkq' => {
			eco => 'A80m',
			# TRANSLATORS: A80m: 1. d4 f5 2. Nc3 Nf6
			variation => N__('Dutch: 2. Nc3 Nf6'),
			moves => {
				'c1g5' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P2P1/2N5/PPP1PP1P/R1BQKBNR b KQkq' => {
			eco => 'A80m',
			# TRANSLATORS: A80m: 1. d4 f5 2. Nc3 Nf6 3. g4
			variation => N__('Dutch: Spielmann Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5pB1/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A80n',
			# TRANSLATORS: A80n: 1. d4 f5 2. Nc3 Nf6 3. Bg5
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5'),
			moves => {
				'd7d5' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5pB1/3P4/2N5/PPP1PPPP/R2QKBNR w KQkq' => {
			eco => 'A80n',
			# TRANSLATORS: A80n: 1. d4 f5 2. Nc3 Nf6 3. Bg5 e6
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1p1pp/5n2/3p1pB1/3P4/2N5/PPP1PPPP/R2QKBNR w KQkq' => {
			eco => 'A80o',
			# TRANSLATORS: A80o: 1. d4 f5 2. Nc3 Nf6 3. Bg5 d5
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 d5'),
			moves => {
				'g5f6' => 1,
			},
		},
		'rnbqkb1r/ppp1p1pp/5B2/3p1p2/3P4/2N5/PPP1PPPP/R2QKBNR b KQkq' => {
			eco => 'A80o',
			# TRANSLATORS: A80o: 1. d4 f5 2. Nc3 Nf6 3. Bg5 d5 4. Bxf6
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 d5'),
			moves => {
				'e7f6' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/5p2/3p1p2/3P4/2N5/PPP1PPPP/R2QKBNR w KQkq' => {
			eco => 'A80p',
			# TRANSLATORS: A80p: 1. d4 f5 2. Nc3 Nf6 3. Bg5 d5 4. Bxf6 exf6
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 d5 4.Bxf6 exf6'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/5p2/3p1p2/3P4/2N1P3/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A80p',
			# TRANSLATORS: A80p: 1. d4 f5 2. Nc3 Nf6 3. Bg5 d5 4. Bxf6 exf6 5. e3
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 d5 4.Bxf6 exf6 5. e3'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqkb1r/pp4pp/2p2p2/3p1p2/3P4/2N1P3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A80p',
			# TRANSLATORS: A80p: 1. d4 f5 2. Nc3 Nf6 3. Bg5 d5 4. Bxf6 exf6 5. e3 c6
			variation => N__('Dutch: 2. Nc3 Nf6 3. Bg5 d5 4.Bxf6 exf6 5. e3 c6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A80q',
			# TRANSLATORS: A80q: 1. d4 f5 2. Nf3
			variation => N__('Dutch: 2. Nf3'),
			moves => {
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A80r',
			# TRANSLATORS: A80r: 1. d4 f5 2. Nf3 e6
			variation => N__('Dutch: 2. Nf3 e6'),
			moves => {
				'd4d5' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/3P1p2/8/5N2/PPP1PPPP/RNBQKB1R b KQkq' => {
			eco => 'A80s',
			# TRANSLATORS: A80s: 1. d4 f5 2. Nf3 e6 3. d5
			variation => N__('Dutch: 2. Nf3 e6 3. d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/5N2/PPP1PPPP/RNBQKB1R w KQkq' => {
			eco => 'A80t',
			# TRANSLATORS: A80t: 1. d4 f5 2. Nf3 Nf6
			variation => N__('Dutch: 2. Nf3 Nf6'),
			moves => {
				'c1g5' => 1,
				'c2c3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/2P2N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A80u',
			# TRANSLATORS: A80u: 1. d4 f5 2. Nf3 Nf6 3. c3
			variation => N__('Dutch: Barcza System'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5pB1/3P4/5N2/PPP1PPPP/RN1QKB1R b KQkq' => {
			eco => 'A80v',
			# TRANSLATORS: A80v: 1. d4 f5 2. Nf3 Nf6 3. Bg5
			variation => N__('Dutch: 2. Nf3 Nf6 3. Bg5'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3P4/6P1/PPP1PP1P/RNBQKBNR b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3
			variation => N__('Dutch: 2. g3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/3P4/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 e6
			variation => N__('Dutch: 2. g3 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 e6 3. Nf3
			variation => N__('Dutch: 2. g3 e6 3. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/3P4/5NP1/PPP1PP1P/RNBQKB1R w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 e6 3. Nf3 Nf6
			variation => N__('Dutch: 2. g3 e6 3. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5p2/3P4/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6
			variation => N__('Dutch: 2. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5p2/3P4/6P1/PPP1PPBP/RNBQK1NR b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2
			variation => N__('Dutch: 2. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/ppppp1bp/6p1/5p2/3P4/6P1/PPP1PPBP/RNBQK1NR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7
			variation => N__('Dutch: 2. g3'),
			moves => {
				'g1f3' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqk1nr/ppppp1bp/6p1/5p2/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7 4. Nf3
			variation => N__('Dutch: 2. g3'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk1nr/pp1pp1bp/2p3p1/5p2/3P4/5NP1/PPP1PPBP/RNBQK2R w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7 4. Nf3 c6
			variation => N__('Dutch: 2. g3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk1nr/pp1pp1bp/2p3p1/5p2/3P4/5NP1/PPP1PPBP/RNBQ1RK1 b kq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7 4. Nf3 c6 5. O-O
			variation => N__('Dutch: 2. g3'),
			moves => {
				'g8h6' => 1,
			},
		},
		'rnbqk2r/pp1pp1bp/2p3pn/5p2/3P4/5NP1/PPP1PPBP/RNBQ1RK1 w kq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7 4. Nf3 c6 5. O-O Nh6
			variation => N__('Dutch: Leningrad, Basman Variation'),
			moves => {
			},
		},
		'rnbqk1nr/ppppp1bp/6p1/5p2/3P4/6PN/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 g6 3. Bg2 Bg7 4. Nh3
			variation => N__('Dutch: Leningrad, Carlsbad Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/6P1/PPP1PP1P/RNBQKBNR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6
			variation => N__('Dutch: 2. g3 Nf6'),
			moves => {
				'f1g2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/6P1/PPP1PPBP/RNBQK1NR b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Bg2
			variation => N__('Dutch: 2. g3 Nf6 3. Bg2'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/3P4/6P1/PPP1PPBP/RNBQK1NR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Bg2 e6
			variation => N__('Dutch: 2. g3 Nf6 3. Bg2 e6'),
			moves => {
				'g1h3' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/3P4/6PN/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Bg2 e6 4. Nh3
			variation => N__('Dutch: Blackburne Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/3P4/6P1/PPP1PPBP/RNBQK1NR w KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Bg2 g6
			variation => N__('Dutch: 2. g3 Nf6 3. Bg2 g6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/3P4/5NP1/PPP1PPBP/RNBQK2R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Bg2 g6 4. Nf3
			variation => N__('Dutch: 2. g3 Nf6 3. Bg2 g6 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/3P4/5NP1/PPP1PP1P/RNBQKB1R b KQkq' => {
			eco => 'A81',
			# TRANSLATORS: A81: 1. d4 f5 2. g3 Nf6 3. Nf3
			variation => N__('Dutch: 2. g3 Nf6 3. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4
			variation => N__('Dutch: Staunton Gambit'),
			moves => {
				'd7d6' => 1,
				'f5e4' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/3p4/5p2/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 d6
			variation => N__('Dutch: Staunton Gambit, Balogh Defence'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/3Pp3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4
			variation => N__('Dutch: Staunton Gambit Accepted'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/8/3Pp3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3
			variation => N__('Dutch: Staunton Gambit, 3. Nc3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/8/3Pp3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 e6
			variation => N__('Dutch: Staunton Gambit, 3. Nc3 e6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp2p/6p1/8/3Pp3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 g6
			variation => N__('Dutch: Staunton Gambit, 3. Nc3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/3Pp3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6
			variation => N__('Dutch: Staunton Gambit, 3. Nc3 Nf6'),
			moves => {
				'c1g5' => 1,
				'f2f3' => 1,
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/3Pp3/2N2P2/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3
			variation => N__('Dutch: Staunton Gambit, 4. f3'),
			moves => {
				'b8c6' => 1,
				'd7d5' => 1,
				'e4f3' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/ppppp1pp/2n2n2/8/3Pp3/2N2P2/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 Nc6
			variation => N__('Dutch: Staunton Gambit, 4. f3 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1p1pp/5n2/3p4/3Pp3/2N2P2/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 d5
			variation => N__('Dutch: Staunton Gambit, 4. f3 d5'),
			moves => {
				'f3e4' => 1,
			},
		},
		'rnbqkb1r/ppp1p1pp/5n2/3p4/3PP3/2N5/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 d5 5. fxe4
			variation => N__('Dutch: Staunton Gambit, 4. f3 d5 5.fxe4'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/3P4/2N2p2/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 exf3
			variation => N__('Dutch: Staunton Gambit, 4. f3 exf3'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/3P4/2N2N2/PPP3PP/R1BQKB1R b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 exf3 5. Nxf3
			variation => N__('Dutch: Staunton Gambit, 4. f3 exf3 5.Nxf3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/8/3Pp3/2N2P2/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. f3 e6
			variation => N__('Dutch: Staunton Gambit, 4. f3 e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/8/3Pp1P1/2N5/PPP2P1P/R1BQKBNR b KQkq' => {
			eco => 'A82',
			# TRANSLATORS: A82: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. g4
			variation => N__('Dutch: Staunton Gambit, Tartakower Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A83a',
			# TRANSLATORS: A83a: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5
			variation => N__('Dutch: Staunton Gambit, Staunton Variation'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/p1ppp1pp/1p3n2/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83b',
			# TRANSLATORS: A83b: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 b6
			variation => N__('Dutch: Staunton Gambit, Nimzowitsch Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp1pp1pp/2p2n2/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83c',
			# TRANSLATORS: A83c: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 c6
			variation => N__('Dutch: Staunton Gambit, Chigorin Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83d',
			# TRANSLATORS: A83d: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 g6
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 g6'),
			moves => {
				'f2f3' => 1,
				'h2h4' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5np1/6B1/3Pp2P/2N5/PPP2PP1/R2QKBNR b KQkq' => {
			eco => 'A83e',
			# TRANSLATORS: A83e: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 g6 5. h4
			variation => N__('Dutch: Staunton Gambit, Alekhine Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/6B1/3Pp3/2N2P2/PPP3PP/R2QKBNR b KQkq' => {
			eco => 'A83f',
			# TRANSLATORS: A83f: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 g6 5. f3
			variation => N__('Dutch: Staunton Gambit, Lasker Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83g',
			# TRANSLATORS: A83g: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6'),
			moves => {
				'c3e4' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/6B1/3PN3/8/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A83h',
			# TRANSLATORS: A83h: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6 5. Nxe4
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6 5.Nxe4'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/6B1/3PN3/8/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83h',
			# TRANSLATORS: A83h: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6 5. Nxe4 Be7
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6 5.Nxe4'),
			moves => {
				'g5f6' => 1,
			},
		},
		'rnbqk2r/ppppb1pp/4pB2/8/3PN3/8/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A83h',
			# TRANSLATORS: A83h: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6 5. Nxe4 Be7 6. Bxf6
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6 5.Nxe4'),
			moves => {
				'e7f6' => 1,
			},
		},
		'rnbqk2r/pppp2pp/4pb2/8/3PN3/8/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83h',
			# TRANSLATORS: A83h: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6 5. Nxe4 Be7 6. Bxf6 Bxf6
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6 5.Nxe4'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk2r/pppp2pp/4pb2/8/3PN3/5N2/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'A83i',
			# TRANSLATORS: A83i: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 e6 5. Nxe4 Be7 6. Bxf6 Bxf6 7. Nf3
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 e6 5.Nxe4'),
			moves => {
			},
		},
		'r1bqkb1r/ppppp1pp/2n2n2/6B1/3Pp3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83j',
			# TRANSLATORS: A83j: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 Nc6
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 Nc6'),
			moves => {
				'd4d5' => 1,
				'f2f3' => 1,
			},
		},
		'r1bqkb1r/ppppp1pp/2n2n2/6B1/3Pp3/2N2P2/PPP3PP/R2QKBNR b KQkq' => {
			eco => 'A83k',
			# TRANSLATORS: A83k: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 Nc6 5. f3
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 Nc6 5. f3'),
			moves => {
			},
		},
		'r1bqkb1r/ppppp1pp/2n2n2/3P2B1/4p3/2N5/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'A83l',
			# TRANSLATORS: A83l: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 Nc6 5. d5
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 Nc6 5. d5'),
			moves => {
				'c6e5' => 1,
			},
		},
		'r1bqkb1r/ppppp1pp/5n2/3Pn1B1/4p3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'A83l',
			# TRANSLATORS: A83l: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 Nc6 5. d5 Ne5
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 Nc6 5. d5'),
			moves => {
				'd1d4' => 1,
			},
		},
		'r1bqkb1r/ppppp1pp/5n2/3Pn1B1/3Qp3/2N5/PPP2PPP/R3KBNR b KQkq' => {
			eco => 'A83m',
			# TRANSLATORS: A83m: 1. d4 f5 2. e4 fxe4 3. Nc3 Nf6 4. Bg5 Nc6 5. d5 Ne5 6. Qd4
			variation => N__('Dutch: Staunton Gambit, 4. Bg5 Nc6 5. d5 Ne5 6. Qd4'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/2PP4/8/PP2PPPP/RNBQKBNR b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4
			variation => N__('Dutch: 2. c4'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/ppp1p1pp/3p4/5p2/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 d6
			variation => N__('Dutch: 2. c4 d6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6
			variation => N__('Dutch: 2. c4 e6'),
			moves => {
				'b1c3' => 1,
				'e2e4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nc3
			variation => N__('Dutch: Rubinstein Variation'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/ppp3pp/4p3/3p1p2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nc3 d5
			variation => N__('Dutch: Rubinstein, 3... d5'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbqkbnr/ppp3pp/4p3/3p1p2/2PP4/2N1P3/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nc3 d5 4. e3
			variation => N__('Dutch: Rubinstein, 3... d5'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqkbnr/pp4pp/2p1p3/3p1p2/2PP4/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nc3 d5 4. e3 c6
			variation => N__('Dutch: Rubinstein, 3... d5 4. e3 c6'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/2PPP3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. e4
			variation => N__('Dutch: Staunton Gambit Deferred'),
			moves => {
			},
		},
		'rnbqkbnr/pppp2pp/4p3/5p2/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nf3
			variation => N__('Dutch: 2. c4 e6 3. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/5N2/PP2PPPP/RNBQKB1R w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 e6 3. Nf3 Nf6
			variation => N__('Dutch: 2. c4 e6 3. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5p2/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 g6
			variation => N__('Dutch: 2. c4 g6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/ppppp2p/6p1/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 g6 3. Nc3
			variation => N__('Dutch: 2. c4 g6'),
			moves => {
				'g8h6' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/6pn/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 g6 3. Nc3 Nh6
			variation => N__('Dutch: Bladel Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2PP4/8/PP2PPPP/RNBQKBNR w KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 Nf6
			variation => N__('Dutch: 2. c4 Nf6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2PP4/5N2/PP2PPPP/RNBQKB1R b KQkq' => {
			eco => 'A84',
			# TRANSLATORS: A84: 1. d4 f5 2. c4 Nf6 3. Nf3
			variation => N__('Dutch: 2. c4 Nf6'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1p1pp/3p1n2/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 d6
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6'),
			moves => {
				'a2a3' => 1,
				'c1g5' => 1,
				'd1c2' => 1,
				'e2e3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/P1N5/1P2PPPP/R1BQKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. a3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. a3'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5pB1/2PP4/2N5/PP2PPPP/R2QKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. Bg5
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/2N5/PPQ1PPPP/R1B1KBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. Qc2
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. Qc2'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/2N1P3/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. e3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. e3'),
			moves => {
				'd7d5' => 1,
				'f8b4' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. e3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. e3 d5'),
			moves => {
			},
		},
		'rnbqk2r/pppp2pp/4pn2/5p2/1bPP4/2N1P3/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. e3 Bb4
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. e3 Bb4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 e6 4. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 e6 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/2N5/PP2PPPP/R1BQKBNR w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6'),
			moves => {
				'c1g5' => 1,
				'f2f3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5pB1/2PP4/2N5/PP2PPPP/R2QKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Bg5
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Bg5'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/2N2P2/PP2P1PP/R1BQKBNR b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. f3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. f3'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/2N2N2/PP2PPPP/R1BQKB1R b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Nf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/2N2N2/PP2PPPP/R1BQKB1R w KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7'),
			moves => {
				'c1f4' => 1,
				'c1g5' => 1,
				'e2e3' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP1B2/2N2N2/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bf4
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bf4'),
			moves => {
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5pB1/2PP4/2N2N2/PP2PPPP/R2QKB1R b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bg5
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bg5'),
			moves => {
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/2N1PN2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'A85',
			# TRANSLATORS: A85: 1. d4 f5 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. e3
			variation => N__('Dutch: 2. c4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. e3'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp1pp/5n2/5p2/2PP4/6P1/PP2PP1P/RNBQKBNR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3
			variation => N__('Dutch: 2. c4 Nf6 3. g3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1p1pp/3p1n2/5p2/2PP4/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 d6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 d6'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkb1r/ppp1p1pp/3p1n2/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 d6 4. Bg2
			variation => N__('Dutch: 2. c4 Nf6 3. g3 d6'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqkb1r/pp2p1pp/2pp1n2/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 d6 4. Bg2 c6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp2p1pp/2pp1n2/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 d6 4. Bg2 c6 5. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 d6'),
			moves => {
				'd8c7' => 1,
			},
		},
		'rnb1kb1r/ppq1p1pp/2pp1n2/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 d6 4. Bg2 c6 5. Nc3 Qc7
			variation => N__('Dutch: Hort-Antoshin System'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6
			variation => N__('Dutch: Leningrad Variation'),
			moves => {
				'b1c3' => 1,
				'f1g2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/2N3P1/PP2PP1P/R1BQKBNR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Nc3
			variation => N__('Dutch: Leningrad, 4. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/2N3P1/PP2PP1P/R1BQKBNR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Nc3 Bg7
			variation => N__('Dutch: Leningrad, 4. Nc3 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2
			variation => N__('Dutch: Leningrad, 4. Bg2'),
			moves => {
				'd7d6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/ppp1p2p/3p1np1/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 d6
			variation => N__('Dutch: Leningrad, 4. Bg2 d6'),
			moves => {
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3'),
			moves => {
				'd7d6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/ppp1p1bp/3p1np1/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3 d6
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3 d6'),
			moves => {
				'd4d5' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqk2r/ppp1p1bp/3p1np1/3P1p2/2P5/2N3P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3 d6 6. d5
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3 d6 6. d5'),
			moves => {
			},
		},
		'rnbqk2r/ppp1p1bp/3p1np1/5p2/2PP4/2N3PN/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3 d6 6. Nh3
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3 d6 6. Nh3'),
			moves => {
			},
		},
		'rnbq1rk1/ppppp1bp/5np1/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR w KQ' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3 O-O
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3 O-O'),
			moves => {
				'g1h3' => 1,
			},
		},
		'rnbq1rk1/ppppp1bp/5np1/5p2/2PP4/2N3PN/PP2PPBP/R1BQK2R b KQ' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nc3 O-O 6. Nh3
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nc3 O-O 6. Nh3'),
			moves => {
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/6PN/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nh3
			variation => N__('Dutch: Leningrad, 4. Bg2 Bg7 5. Nh3'),
			moves => {
			},
		},
		'rnbqkb1r/ppppp2p/5np1/5p2/2PP4/5NP1/PP2PP1P/RNBQKB1R b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Nf3
			variation => N__('Dutch: Leningrad, 4. Nf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/5NP1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Nf3 Bg7
			variation => N__('Dutch: Leningrad, 4. Nf3 Bg7'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/2N2NP1/PP2PP1P/R1BQKB1R b KQkq' => {
			eco => 'A86',
			# TRANSLATORS: A86: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Nf3 Bg7 5. Nc3
			variation => N__('Dutch: Leningrad, 4. Nf3 Bg7 5. Nc3'),
			moves => {
			},
		},
		'rnbqk2r/ppppp1bp/5np1/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3
			variation => N__('Dutch: Leningrad, Main Line'),
			moves => {
				'd7d6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/ppp1p1bp/3p1np1/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6
			variation => N__('Dutch: Leningrad, Main Line, 5... d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqk2r/ppp1p1bp/3p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 d6 6. Nc3
			variation => N__('Dutch: Leningrad, Main Line, 5... d6 6. Nc3'),
			moves => {
			},
		},
		'rnbq1rk1/ppppp1bp/5np1/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQ' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O
			variation => N__('Dutch: Leningrad, Main Line, 5...O-O'),
			moves => {
				'b1c3' => 1,
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppppp1bp/5np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQ' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. Nc3
			variation => N__('Dutch: Leningrad, Main Line, 6. Nc3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1p1bp/3p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. Nc3 d6
			variation => N__('Dutch: Leningrad, Main Line, 6. Nc3 d6'),
			moves => {
			},
		},
		'rnbq1rk1/ppppp1bp/5np1/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O
			variation => N__('Dutch: Leningrad, Main Line, 6.O-O'),
			moves => {
				'c7c6' => 1,
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/pp1pp1bp/2p2np1/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O c6
			variation => N__('Dutch: Leningrad, Main Line, 6... c6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1p1bp/3p1np1/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6
			variation => N__('Dutch: Leningrad, Main Line, 6... d6'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'd4d5' => 1,
			},
		},
		'rnbq1rk1/ppp1p1bp/3p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3'),
			moves => {
				'b8a6' => 1,
				'b8c6' => 1,
				'c7c6' => 1,
				'd8e8' => 1,
				'e7e6' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/n2p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Na6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Na6'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1p1bp/3p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8'),
			moves => {
				'b2b3' => 1,
				'c3d5' => 1,
				'd4d5' => 1,
				'f1e1' => 1,
			},
		},
		'rnb1qrk1/ppp1p1bp/3p1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. b3'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1p1bp/3p1np1/3N1p2/2PP4/5NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Nd5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. Nd5'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1p1bp/3p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5'),
			moves => {
				'a7a5' => 1,
				'b8a6' => 1,
			},
		},
		'rnb1qrk1/1pp1p1bp/3p1np1/p2P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5 a5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5 a5'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1p1bp/n2p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5 Na6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5 Na6'),
			moves => {
				'a1b1' => 1,
			},
		},
		'r1b1qrk1/ppp1p1bp/n2p1np1/3P1p2/2P5/2N2NP1/PP2PPBP/1RBQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. d5 Na6 9. Rb1
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. d5 Na6 9. Rb1'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1p1bp/3p1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Qe8 8. Re1'),
			moves => {
			},
		},
		'rnbq1rk1/ppp3bp/3ppnp1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 e6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 e6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1p1bp/3p1np1/5p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. b3
			variation => N__('Dutch: Leningrad, Main Line, 7. b3'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/5p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. b3 c6
			variation => N__('Dutch: Leningrad, Main Line, 7. b3 c6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1p1bp/3p1np1/3P1p2/2P5/5NP1/PP2PPBP/RNBQ1RK1 b -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. d5
			variation => N__('Dutch: Leningrad, Main Line, 7. d5'),
			moves => {
				'c7c5' => 1,
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp2p1bp/3p1np1/2pP1p2/2P5/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. d5 c5
			variation => N__('Dutch: Leningrad, Main Line, 7. d5 c5'),
			moves => {
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/3P1p2/2P5/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A87',
			# TRANSLATORS: A87: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. d5 c6
			variation => N__('Dutch: Leningrad, Main Line, 7. d5 c6'),
			moves => {
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6'),
			moves => {
				'b2b3' => 1,
				'd1c2' => 1,
				'd4d5' => 1,
				'f1e1' => 1,
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. b3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. b3'),
			moves => {
				'b8a6' => 1,
				'd8a5' => 1,
				'd8e8' => 1,
			},
		},
		'r1bq1rk1/pp2p1bp/n1pp1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. b3 Na6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. b3 Na6'),
			moves => {
			},
		},
		'rnb2rk1/pp2p1bp/2pp1np1/q4p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. b3 Qa5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. b3 Qa5'),
			moves => {
			},
		},
		'rnb1qrk1/pp2p1bp/2pp1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. b3 Qe8
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. b3 Qe8'),
			moves => {
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/5p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. Qc2
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. Qc2'),
			moves => {
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5'),
			moves => {
				'c6d5' => 1,
				'c8d7' => 1,
				'd8a5' => 1,
				'd8e8' => 1,
				'e7e5' => 1,
			},
		},
		'rnbq1rk1/pp2p1bp/3p1np1/3p1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 cxd5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 cxd5'),
			moves => {
			},
		},
		'rn1q1rk1/pp1bp1bp/2pp1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 Bd7
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 Bd7'),
			moves => {
			},
		},
		'rnb2rk1/pp2p1bp/2pp1np1/q2P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 Qa5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 Qa5'),
			moves => {
			},
		},
		'rnb1qrk1/pp2p1bp/2pp1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 Qe8
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 Qe8'),
			moves => {
			},
		},
		'rnbq1rk1/pp4bp/2pp1np1/3Ppp2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 e5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 e5'),
			moves => {
				'd5e6' => 1,
			},
		},
		'rnbq1rk1/pp4bp/2ppPnp1/5p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 e5 9. dxe6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 e5'),
			moves => {
				'c8e6' => 1,
			},
		},
		'rn1q1rk1/pp4bp/2ppbnp1/5p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 e5 9. dxe6 Bxe6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 e5'),
			moves => {
				'b2b3' => 1,
				'd1d3' => 1,
			},
		},
		'rn1q1rk1/pp4bp/2ppbnp1/5p2/2P5/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 e5 9. dxe6 Bxe6 10. b3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 e5 9.dxe6 Bxe6 10. b3'),
			moves => {
			},
		},
		'rn1q1rk1/pp4bp/2ppbnp1/5p2/2P5/2NQ1NP1/PP2PPBP/R1B2RK1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. d5 e5 9. dxe6 Bxe6 10. Qd3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. d5 e5 9.dxe6 Bxe6 10. Qd3'),
			moves => {
			},
		},
		'rnbq1rk1/pp2p1bp/2pp1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b -' => {
			eco => 'A88',
			# TRANSLATORS: A88: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 c6 8. Re1
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 c6 8. Re1'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6'),
			moves => {
				'b2b3' => 1,
				'd4d5' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. b3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. b3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/2np1np1/3P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5'),
			moves => {
				'c6a5' => 1,
				'c6e5' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5'),
			moves => {
				'd1d3' => 1,
				'f3d2' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2NQ1NP1/PP2PPBP/R1B2RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5 9. Qd3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5 9. Qd3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/n2P1p2/2P5/2N3P1/PP1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Na5 9. Nd2
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Na5 9. Nd2'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/3Pnp2/2P5/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5'),
			moves => {
				'f3e5' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/3p1np1/3PNp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/2N3P1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5 dxe5
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5'),
			moves => {
				'd1b3' => 1,
				'e2e4' => 1,
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P5/1QN3P1/PP2PPBP/R1B2RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5 dxe5 10. Qb3
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5 dxe5 10. Qb3'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1p1bp/5np1/3Ppp2/2P1P3/2N3P1/PP3PBP/R1BQ1RK1 b -' => {
			eco => 'A89',
			# TRANSLATORS: A89: 1. d4 f5 2. c4 Nf6 3. g3 g6 4. Bg2 Bg7 5. Nf3 O-O 6. O-O d6 7. Nc3 Nc6 8. d5 Ne5 9. Nxe5 dxe5 10. e4
			variation => N__('Dutch: Leningrad, Main Line, 7. Nc3 Nc6 8. d5 Ne5 9.Nxe5 dxe5 10. e4'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/6P1/PP2PP1P/RNBQKBNR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6'),
			moves => {
				'b1c3' => 1,
				'f1g2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/2N3P1/PP2PP1P/R1BQKBNR b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Nc3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/2N3P1/PP2PP1P/R1BQKBNR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Nc3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Nc3 d5'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2'),
			moves => {
				'c7c6' => 1,
				'd7d5' => 1,
				'f8b4' => 1,
				'f8e7' => 1,
			},
		},
		'rnbqkb1r/pp1p2pp/2p1pn2/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 c6'),
			moves => {
				'g1f3' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqkb1r/pp1p2pp/2p1pn2/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp4pp/2p1pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5'),
			moves => {
				'b1c3' => 1,
				'd1c2' => 1,
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp4pp/2p1pn2/3p1p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp4pp/2p1pn2/3p1p2/2PP4/5NP1/PPQ1PPBP/RNB1K2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. Qc2
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6. Qc2'),
			moves => {
				'f8d6' => 1,
			},
		},
		'rnbqk2r/pp4pp/2pbpn2/3p1p2/2PP4/5NP1/PPQ1PPBP/RNB1K2R w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. Qc2 Bd6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6. Qc2 Bd6'),
			moves => {
			},
		},
		'rnbqkb1r/pp4pp/2p1pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 b kq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6.O-O'),
			moves => {
				'f8d6' => 1,
			},
		},
		'rnbqk2r/pp4pp/2pbpn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w kq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. O-O Bd6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6.O-O Bd6'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rnbqk2r/pp4pp/2pbpn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b kq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nf3 d5 6. O-O Bd6 7. b3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 5. Nf3 d5 6.O-O Bd6 7. b3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p2pp/2p1pn2/5p2/2PP4/6PN/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nh3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 c6 5. Nh3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 d5'),
			moves => {
				'g1f3' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 d5 5. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 d5 5. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/6PN/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 d5 5. Nh3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 d5 5. Nh3'),
			moves => {
			},
		},
		'rnbqk2r/pppp2pp/4pn2/5p2/1bPP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Bb4+
			variation => N__('Dutch: Dutch-Indian (Nimzo-Dutch) Variation'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'c1d2' => 1,
			},
		},
		'rnbqk2r/pppp2pp/4pn2/5p2/1bPP4/2N3P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Bb4+ 5. Nc3
			variation => N__('Dutch: Dutch-Indian, 5. Nc3'),
			moves => {
			},
		},
		'rnbqk2r/pppp2pp/4pn2/5p2/1bPP4/6P1/PP1NPPBP/R1BQK1NR b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Bb4+ 5. Nd2
			variation => N__('Dutch: Dutch-Indian, 5. Nd2'),
			moves => {
			},
		},
		'rnbqk2r/pppp2pp/4pn2/5p2/1bPP4/6P1/PP1BPPBP/RN1QK1NR b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Bb4+ 5. Bd2
			variation => N__('Dutch: Dutch-Indian, 5. Bd2'),
			moves => {
				'b4e7' => 1,
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2PP4/6P1/PP1BPPBP/RN1QK1NR w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Bb4+ 5. Bd2 Be7
			variation => N__('Dutch: Dutch-Indian, Alekhine Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppp2pp/4pn2/5p2/2PP4/5NP1/PP2PP1P/RNBQKB1R b KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Nf3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/ppp3pp/4pn2/3p1p2/2PP4/5NP1/PP2PP1P/RNBQKB1R w KQkq' => {
			eco => 'A90',
			# TRANSLATORS: A90: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Nf3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Nf3 d5'),
			moves => {
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2PP4/6P1/PP2PPBP/RNBQK1NR w KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
				'g1h3' => 1,
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR b KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3'),
			moves => {
				'd7d5' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/ppp1b1pp/4pn2/3p1p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR w KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3 d5'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/2N3P1/PP2PPBP/R1BQK1NR w KQ' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3 O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3 O-O'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/2N1P1P1/PP3PBP/R1BQK1NR b KQ' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nc3 O-O 6. e3
			variation => N__('Dutch: Botvinnik-Bronstein Variation'),
			moves => {
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3'),
			moves => {
				'd7d5' => 1,
				'd7d6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/ppp1b1pp/4pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqk2r/ppp1b1pp/4pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 b kq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5 6. O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5 6.O-O'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk2r/pp2b1pp/2p1pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w kq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5 6. O-O c6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d5 6.O-O c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1b1pp/3ppn2/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqk2r/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d6 6. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 d6 6. Nc3'),
			moves => {
			},
		},
		'rnbqk2r/ppppb1pp/4pn2/5p2/2PP4/6PN/PP2PPBP/RNBQK2R b KQkq' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/6PN/PP2PPBP/RNBQK2R w KQ' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/6PN/PP2PPBP/RNBQ1RK1 b -' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6. O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6.O-O'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/6PN/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6. O-O d6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6.O-O d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/2N3PN/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6. O-O d6 7. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6.O-O d6 7. Nc3'),
			moves => {
				'c7c6' => 1,
				'd8e8' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2pppn2/5p2/2PP4/2N3PN/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6. O-O d6 7. Nc3 c6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6.O-O d6 7. Nc3 c6'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/2N3PN/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A91',
			# TRANSLATORS: A91: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6. O-O d6 7. Nc3 Qe8
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nh3 O-O 6.O-O d6 7. Nc3 Qe8'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/5NP1/PP2PPBP/RNBQK2R w KQ' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O'),
			moves => {
				'b1c3' => 1,
				'd4d5' => 1,
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R b KQ' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. Nc3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. Nc3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQK2R w KQ' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. Nc3 d6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. Nc3 d6'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/3P1p2/2P5/5NP1/PP2PPBP/RNBQK2R b KQ' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. d5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. d5'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4pn2/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6.O-O'),
			moves => {
				'c7c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'f6e4' => 1,
			},
		},
		'rnbq1rk1/pp1pb1pp/2p1pn2/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O c6
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6.O-O c6'),
			moves => {
				'b2b3' => 1,
			},
		},
		'rnbq1rk1/pp1pb1pp/2p1pn2/5p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O c6 7. b3
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6.O-O c6 7. b3'),
			moves => {
				'a7a5' => 1,
			},
		},
		'rnbq1rk1/1p1pb1pp/2p1pn2/p4p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O c6 7. b3 a5
			variation => N__('Dutch: 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6.O-O c6 7. b3 a5'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1b1pp/4pn2/3p1p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5
			variation => N__('Dutch: Stonewall Variation'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'b2b3' => 1,
				'd1c2' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/4pn2/3p1p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3
			variation => N__('Dutch: Stonewall, 7. Nc3'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/4pn2/3p1p2/2PP4/5NP1/PP1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nbd2
			variation => N__('Dutch: Stonewall, 7.Nbd2'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/5NP1/PP1NPPBP/R1BQ1RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nbd2 c6
			variation => N__('Dutch: Stonewall, 7.Nbd2 c6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1b1pp/4pn2/3p1p2/2PP4/5NP1/PPQ1PPBP/RNB2RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Qc2
			variation => N__('Dutch: Stonewall, 7. Qc2'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/5NP1/PPQ1PPBP/RNB2RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Qc2 c6
			variation => N__('Dutch: Stonewall, 7. Qc2 c6'),
			moves => {
				'b1d2' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/5NP1/PPQNPPBP/R1B2RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Qc2 c6 8. Nbd2
			variation => N__('Dutch: Stonewall, 7. Qc2 c6 8.Nbd2'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4p3/5p2/2PPn3/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O Ne4
			variation => N__('Dutch: Alekhine Variation'),
			moves => {
				'b1d2' => 1,
				'd4d5' => 1,
			},
		},
		'rnbq1rk1/ppppb1pp/4p3/5p2/2PPn3/5NP1/PP1NPPBP/R1BQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O Ne4 7. Nbd2
			variation => N__('Dutch: Alekhine, 7.Nbd2'),
			moves => {
			},
		},
		'rnbq1rk1/ppppb1pp/4p3/3P1p2/2P1n3/5NP1/PP2PPBP/RNBQ1RK1 b -' => {
			eco => 'A92',
			# TRANSLATORS: A92: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O Ne4 7. d5
			variation => N__('Dutch: Alekhine, 7. d5'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1b1pp/4pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3
			variation => N__('Dutch: Stonewall, Botvinnik Variation'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c6' => 1,
			},
		},
		'rnbq1rk1/p1p1b1pp/1p2pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 b6
			variation => N__('Dutch: Stonewall, Botvinnik, 7... b6'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1b1pp/2n1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 Nc6
			variation => N__('Dutch: Stonewall, Botvinnik, 7... Nc6'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6
			variation => N__('Dutch: Stonewall, Botvinnik, 7... c6'),
			moves => {
				'c1a3' => 1,
				'c1b2' => 1,
				'd1c2' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Bb2
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Bb2'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/1P3NP1/P1Q1PPBP/RNB2RK1 b -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Qc2
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Qc2'),
			moves => {
				'f6e4' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1p3/3p1p2/2PPn3/1P3NP1/P1Q1PPBP/RNB2RK1 w -' => {
			eco => 'A93',
			# TRANSLATORS: A93: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Qc2 Ne4
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Qc2 Ne4'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 b -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3'),
			moves => {
				'b8d7' => 1,
				'c8d7' => 1,
				'e7a3' => 1,
			},
		},
		'r1bq1rk1/pp1nb1pp/2p1pn2/3p1p2/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 w -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Nbd7
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Nbd7'),
			moves => {
				'a3e7' => 1,
			},
		},
		'r1bq1rk1/pp1nB1pp/2p1pn2/3p1p2/2PP4/1P3NP1/P3PPBP/RN1Q1RK1 b -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Nbd7 9. Bxe7
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Nbd7 9.Bxe7'),
			moves => {
			},
		},
		'rn1q1rk1/pp1bb1pp/2p1pn2/3p1p2/2PP4/BP3NP1/P3PPBP/RN1Q1RK1 w -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bd7
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bd7'),
			moves => {
			},
		},
		'rnbq1rk1/pp4pp/2p1pn2/3p1p2/2PP4/bP3NP1/P3PPBP/RN1Q1RK1 w -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3'),
			moves => {
				'b1a3' => 1,
			},
		},
		'rnbq1rk1/pp4pp/2p1pn2/3p1p2/2PP4/NP3NP1/P3PPBP/R2Q1RK1 b -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3 9. Nxa3
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3 9.Nxa3'),
			moves => {
				'b8d7' => 1,
				'd8e7' => 1,
			},
		},
		'r1bq1rk1/pp1n2pp/2p1pn2/3p1p2/2PP4/NP3NP1/P3PPBP/R2Q1RK1 w -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3 9. Nxa3 Nbd7
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3 9.Nxa3 Nbd7'),
			moves => {
			},
		},
		'rnb2rk1/pp2q1pp/2p1pn2/3p1p2/2PP4/NP3NP1/P3PPBP/R2Q1RK1 w -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3 9. Nxa3 Qe7
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3 9.Nxa3 Qe7'),
			moves => {
				'd1c1' => 1,
			},
		},
		'rnb2rk1/pp2q1pp/2p1pn2/3p1p2/2PP4/NP3NP1/P3PPBP/R1Q2RK1 b -' => {
			eco => 'A94',
			# TRANSLATORS: A94: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. b3 c6 8. Ba3 Bxa3 9. Nxa3 Qe7 10. Qc1
			variation => N__('Dutch: Stonewall, Botvinnik, 8. Ba3 Bxa3 9.Nxa3 Qe7 10. Qc1'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6
			variation => N__('Dutch: Stonewall, 7. Nc3 c6'),
			moves => {
				'b2b3' => 1,
				'c1f4' => 1,
				'c1g5' => 1,
				'd1c2' => 1,
				'f3e5' => 1,
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. b3
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. b3'),
			moves => {
				'd8e8' => 1,
			},
		},
		'rnb1qrk1/pp2b1pp/2p1pn2/3p1p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. b3 Qe8
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. b3 Qe8'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP1B2/2N2NP1/PP2PPBP/R2Q1RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Bf4
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. Bf4'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1pB1/2PP4/2N2NP1/PP2PPBP/R2Q1RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Bg5
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. Bg5'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3p1p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Qc2
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. Qc2'),
			moves => {
				'd8e8' => 1,
			},
		},
		'rnb1qrk1/pp2b1pp/2p1pn2/3p1p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Qc2 Qe8
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. Qc2'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnb1qrk1/pp2b1pp/2p1pn2/3p1pB1/2PP4/2N2NP1/PPQ1PPBP/R4RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Qc2 Qe8 9. Bg5
			variation => N__('Dutch: Stonewall: Chekhover Variation'),
			moves => {
			},
		},
		'rnbq1rk1/pp2b1pp/2p1pn2/3pNp2/2PP4/2N3P1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A95',
			# TRANSLATORS: A95: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d5 7. Nc3 c6 8. Ne5
			variation => N__('Dutch: Stonewall, 7. Nc3 c6 8. Ne5'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/5NP1/PP2PPBP/RNBQ1RK1 w -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6
			variation => N__('Dutch: Classical'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3
			variation => N__('Dutch: Classical, 7. Nc3'),
			moves => {
				'a7a5' => 1,
				'd8e8' => 1,
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 a5
			variation => N__('Dutch: Classical, 7. Nc3 a5'),
			moves => {
				'b2b3' => 1,
				'd1c2' => 1,
				'f1e1' => 1,
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 a5 8. b3
			variation => N__('Dutch: Classical, 7. Nc3 a5 8. b3'),
			moves => {
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 a5 8. Qc2
			variation => N__('Dutch: Classical, 7. Nc3 a5 8. Qc2'),
			moves => {
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 a5 8. Re1
			variation => N__('Dutch: Classical, 7. Nc3 a5 8. Re1'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1b1pp/3ppn2/5p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. b3
			variation => N__('Dutch: Classical, 7. b3'),
			moves => {
				'a7a5' => 1,
				'd8e8' => 1,
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. b3 a5
			variation => N__('Dutch: Classical, 7. b3 a5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnbq1rk1/1pp1b1pp/3ppn2/p4p2/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. b3 a5 8. Bb2
			variation => N__('Dutch: Classical, 7. b3 a5 8. Bb2'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/1P3NP1/P3PPBP/RNBQ1RK1 w -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. b3 Qe8
			variation => N__('Dutch: Classical, 7. b3 Qe8'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/1P3NP1/PB2PPBP/RN1Q1RK1 b -' => {
			eco => 'A96',
			# TRANSLATORS: A96: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. b3 Qe8 8. Bb2
			variation => N__('Dutch: Classical, 7. b3 Qe8 8. Bb2'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQ1RK1 w -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8
			variation => N__('Dutch: Ilyin-Zhenevsky Variation'),
			moves => {
				'b2b3' => 1,
				'b2b4' => 1,
				'd1c2' => 1,
				'f1e1' => 1,
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/1PPP4/2N2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b4
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b4'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 b -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter Variation'),
			moves => {
				'e8g6' => 1,
				'e8h5' => 1,
				'f6e4' => 1,
			},
		},
		'rnb2rk1/ppp1b1pp/3ppnq1/5p2/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 w -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1 Qg6
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter, 8... Qg6'),
			moves => {
				'e2e4' => 1,
			},
		},
		'rnb2rk1/ppp1b1pp/3ppnq1/5p2/2PPP3/2N2NP1/PP3PBP/R1BQR1K1 b -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1 Qg6 9. e4
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter, 8... Qg6 9. e4'),
			moves => {
				'f5e4' => 1,
			},
		},
		'rnb2rk1/ppp1b1pp/3ppnq1/8/2PPp3/2N2NP1/PP3PBP/R1BQR1K1 w -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1 Qg6 9. e4 fxe4
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter, 8... Qg6 9. e4 fxe4'),
			moves => {
			},
		},
		'rnb2rk1/ppp1b1pp/3ppn2/5p1q/2PP4/2N2NP1/PP2PPBP/R1BQR1K1 w -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1 Qh5
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter, 8... Qh5'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3pp3/5p2/2PPn3/2N2NP1/PP2PPBP/R1BQR1K1 w -' => {
			eco => 'A97',
			# TRANSLATORS: A97: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Re1 Ne4
			variation => N__('Dutch: Ilyin-Zhenevsky, Winter, 8... Ne4'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 b -' => {
			eco => 'A98',
			# TRANSLATORS: A98: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Qc2
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. Qc2'),
			moves => {
				'b8c6' => 1,
				'e8h5' => 1,
			},
		},
		'r1b1qrk1/ppp1b1pp/2nppn2/5p2/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w -' => {
			eco => 'A98',
			# TRANSLATORS: A98: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Qc2 Nc6
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. Qc2 Nc6'),
			moves => {
			},
		},
		'rnb2rk1/ppp1b1pp/3ppn2/5p1q/2PP4/2N2NP1/PPQ1PPBP/R1B2RK1 w -' => {
			eco => 'A98',
			# TRANSLATORS: A98: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. Qc2 Qh5
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. Qc2 Qh5'),
			moves => {
			},
		},
		'rnb1qrk1/ppp1b1pp/3ppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3'),
			moves => {
				'a7a5' => 1,
				'b8c6' => 1,
				'b8d7' => 1,
				'c7c6' => 1,
				'e8h5' => 1,
			},
		},
		'rnb1qrk1/1pp1b1pp/3ppn2/p4p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5'),
			moves => {
				'c1a3' => 1,
				'c1b2' => 1,
				'd1c2' => 1,
			},
		},
		'rnb1qrk1/1pp1b1pp/3ppn2/p4p2/2PP4/BPN2NP1/P3PPBP/R2Q1RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Ba3
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Ba3'),
			moves => {
			},
		},
		'rnb1qrk1/1pp1b1pp/3ppn2/p4p2/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2'),
			moves => {
				'b8a6' => 1,
				'c7c6' => 1,
				'e8h5' => 1,
			},
		},
		'r1b1qrk1/1pp1b1pp/n2ppn2/p4p2/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2 Na6
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2 Na6'),
			moves => {
			},
		},
		'rnb1qrk1/1p2b1pp/2pppn2/p4p2/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2 c6
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2 c6'),
			moves => {
			},
		},
		'rnb2rk1/1pp1b1pp/3ppn2/p4p1q/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2 Qh5
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2 Qh5'),
			moves => {
				'e2e3' => 1,
			},
		},
		'rnb2rk1/1pp1b1pp/3ppn2/p4p1q/2PP4/1PN1PNP1/PB3PBP/R2Q1RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Bb2 Qh5 10. e3
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Bb2 Qh5 10. e3'),
			moves => {
			},
		},
		'rnb1qrk1/1pp1b1pp/3ppn2/p4p2/2PP4/1PN2NP1/P1Q1PPBP/R1B2RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 a5 9. Qc2
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 a5 9. Qc2'),
			moves => {
			},
		},
		'r1b1qrk1/ppp1b1pp/2nppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Nc6
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Nc6'),
			moves => {
			},
		},
		'r1b1qrk1/pppnb1pp/3ppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Nbd7
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Nbd7'),
			moves => {
			},
		},
		'rnb1qrk1/pp2b1pp/2pppn2/5p2/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 c6
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 c6'),
			moves => {
			},
		},
		'rnb2rk1/ppp1b1pp/3ppn2/5p1q/2PP4/1PN2NP1/P3PPBP/R1BQ1RK1 w -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Qh5
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Qh5'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnb2rk1/ppp1b1pp/3ppn2/5p1q/2PP4/1PN2NP1/PB2PPBP/R2Q1RK1 b -' => {
			eco => 'A99',
			# TRANSLATORS: A99: 1. d4 f5 2. c4 Nf6 3. g3 e6 4. Bg2 Be7 5. Nf3 O-O 6. O-O d6 7. Nc3 Qe8 8. b3 Qh5 9. Bb2
			variation => N__('Dutch: Ilyin-Zhenevsky, 8. b3 Qh5 9. Bb2'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/8/4P3/8/PPPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4
			variation => N__('King\'s Pawn'),
			moves => {
				'a7a6' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'c7c5' => 1,
				'c7c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'f7f5' => 1,
				'g7g5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
				'g8h6' => 1,
			},
		},
		'rnbqkbnr/ppppp1pp/8/5p2/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 f5
			variation => N__('King\'s Pawn: Fred'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/7n/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 Nh6
			variation => N__('King\'s Pawn'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/7n/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 Nh6 2. d4
			variation => N__('King\'s Pawn'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/6pn/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 Nh6 2. d4 g6
			variation => N__('King\'s Pawn'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/6pn/8/2PPP3/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 Nh6 2. d4 g6 3. c4
			variation => N__('King\'s Pawn'),
			moves => {
				'f7f6' => 1,
			},
		},
		'rnbqkb1r/ppppp2p/5ppn/8/2PPP3/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B00a',
			# TRANSLATORS: B00a: 1. e4 Nh6 2. d4 g6 3. c4 f6
			variation => N__('King\'s Pawn: Hippopotamus Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00b',
			# TRANSLATORS: B00b: 1. e4 g5
			variation => N__('Reversed Grob (Borg/Basman Defence)'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/8/6p1/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00b',
			# TRANSLATORS: B00b: 1. e4 g5 2. d4
			variation => N__('Reversed Grob (Borg/Basman Defence)'),
			moves => {
			},
		},
		'rnbqkbnr/1ppppppp/p7/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00c',
			# TRANSLATORS: B00c: 1. e4 a6
			variation => N__('St. George Defence'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/1ppppppp/p7/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00c',
			# TRANSLATORS: B00c: 1. e4 a6 2. d4
			variation => N__('St. George Defence'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rnbqkbnr/1pppppp1/p6p/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00c',
			# TRANSLATORS: B00c: 1. e4 a6 2. d4 h6
			variation => N__('Basman\'s Creepy-Crawly System (as Black)'),
			moves => {
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00d',
			# TRANSLATORS: B00d: 1. e4 b6
			variation => N__('Owen Defence'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/p1pppppp/1p6/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00d',
			# TRANSLATORS: B00d: 1. e4 b6 2. d4
			variation => N__('Owen Defence'),
			moves => {
				'c8b7' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/p1pp1ppp/1p2p3/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00e',
			# TRANSLATORS: B00e: 1. e4 b6 2. d4 e6
			variation => N__('Owen Defence: French'),
			moves => {
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00f',
			# TRANSLATORS: B00f: 1. e4 b6 2. d4 Bb7
			variation => N__('Owen Defence: 2. d4 Bb7'),
			moves => {
				'c1g5' => 1,
				'f1d3' => 1,
				'g1f3' => 1,
			},
		},
		'rn1qkbnr/pbpppppp/1p6/6B1/3PP3/8/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'B00f',
			# TRANSLATORS: B00f: 1. e4 b6 2. d4 Bb7 3. Bg5
			variation => N__('Owen Defence: Naselwaus Gambit'),
			moves => {
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B00f',
			# TRANSLATORS: B00f: 1. e4 b6 2. d4 Bb7 3. Nf3
			variation => N__('Owen Defence: Smith Gambit'),
			moves => {
			},
		},
		'rn1qkbnr/pbpppppp/1p6/8/3PP3/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B00g',
			# TRANSLATORS: B00g: 1. e4 b6 2. d4 Bb7 3. Bd3
			variation => N__('Owen Defence: 3. Bd3'),
			moves => {
				'e7e6' => 1,
				'f7f5' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/pbppp1pp/1p6/5p2/3PP3/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B00g',
			# TRANSLATORS: B00g: 1. e4 b6 2. d4 Bb7 3. Bd3 f5
			variation => N__('Owen Defence: Matinovsky Gambit'),
			moves => {
			},
		},
		'rn1qkb1r/pbpppppp/1p3n2/8/3PP3/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B00h',
			# TRANSLATORS: B00h: 1. e4 b6 2. d4 Bb7 3. Bd3 Nf6
			variation => N__('Owen Defence: 3. Bd3 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/3PP3/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B00i',
			# TRANSLATORS: B00i: 1. e4 b6 2. d4 Bb7 3. Bd3 e6
			variation => N__('Owen Defence: 3. Bd3 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rn1qkbnr/pbpp1ppp/1p2p3/8/3PP3/3B1N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B00j',
			# TRANSLATORS: B00j: 1. e4 b6 2. d4 Bb7 3. Bd3 e6 4. Nf3
			variation => N__('Owen Defence: 3. Bd3 e6 4. Nf3'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rn1qkbnr/pb1p1ppp/1p2p3/2p5/3PP3/3B1N2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B00k',
			# TRANSLATORS: B00k: 1. e4 b6 2. d4 Bb7 3. Bd3 e6 4. Nf3 c5
			variation => N__('Owen Defence: 3. Bd3 e6 4. Nf3 c5'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rn1qkbnr/pb1p1ppp/1p2p3/2p5/3PP3/2PB1N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B00k',
			# TRANSLATORS: B00k: 1. e4 b6 2. d4 Bb7 3. Bd3 e6 4. Nf3 c5 5. c3
			variation => N__('Owen Defence: 3. Bd3 e6 4. Nf3 c5 5. c3'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B00l',
			# TRANSLATORS: B00l: 1. e4 Nc6
			variation => N__('Nimzowitsch Defence'),
			moves => {
				'b1c3' => 1,
				'b2b4' => 1,
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/1P2P3/8/P1PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B00l',
			# TRANSLATORS: B00l: 1. e4 Nc6 2. b4
			variation => N__('Nimzowitsch Defence: Wheeler Gambit'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B00m',
			# TRANSLATORS: B00m: 1. e4 Nc6 2. Nc3
			variation => N__('Nimzowitsch Defence: 2. Nc3'),
			moves => {
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n1p3/8/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B00m',
			# TRANSLATORS: B00m: 1. e4 Nc6 2. Nc3 e6
			variation => N__('Nimzowitsch Defence: 2. Nc3 e6'),
			moves => {
			},
		},
		'r1bqkb1r/pppppppp/2n2n2/8/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B00m',
			# TRANSLATORS: B00m: 1. e4 Nc6 2. Nc3 Nf6
			variation => N__('Nimzowitsch Defence: 2. Nc3 Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B00n',
			# TRANSLATORS: B00n: 1. e4 Nc6 2. Nf3
			variation => N__('Nimzowitsch Defence: 2. Nf3'),
			moves => {
				'd7d6' => 1,
				'f7f5' => 1,
			},
		},
		'r1bqkbnr/ppppp1pp/2n5/5p2/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B00n',
			# TRANSLATORS: B00n: 1. e4 Nc6 2. Nf3 f5
			variation => N__('Nimzowitsch Defence: Colorado Counter'),
			moves => {
			},
		},
		'r1bqkbnr/ppp1pppp/2np4/8/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B00o',
			# TRANSLATORS: B00o: 1. e4 Nc6 2. Nf3 d6
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2np4/8/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B00o',
			# TRANSLATORS: B00o: 1. e4 Nc6 2. Nf3 d6 3. d4
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4'),
			moves => {
				'c8g4' => 1,
				'g8f6' => 1,
			},
		},
		'r2qkbnr/ppp1pppp/2np4/8/3PP1b1/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B00o',
			# TRANSLATORS: B00o: 1. e4 Nc6 2. Nf3 d6 3. d4 Bg4
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Bg4'),
			moves => {
			},
		},
		'r1bqkb1r/ppp1pppp/2np1n2/8/3PP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B00p',
			# TRANSLATORS: B00p: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkb1r/ppp1pppp/2np1n2/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B00q',
			# TRANSLATORS: B00q: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3'),
			moves => {
				'c8g4' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/ppp1pp1p/2np1np1/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B00q',
			# TRANSLATORS: B00q: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 g6
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 g6'),
			moves => {
			},
		},
		'r2qkb1r/ppp1pppp/2np1n2/8/3PP1b1/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B00r',
			# TRANSLATORS: B00r: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4'),
			moves => {
				'c1e3' => 1,
				'd4d5' => 1,
				'f1b5' => 1,
				'f1e2' => 1,
			},
		},
		'r2qkb1r/ppp1pppp/2np1n2/3P4/4P1b1/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B00r',
			# TRANSLATORS: B00r: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. d5
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. d5'),
			moves => {
			},
		},
		'r2qkb1r/ppp1pppp/2np1n2/1B6/3PP1b1/2N2N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B00r',
			# TRANSLATORS: B00r: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Bb5
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Bb5'),
			moves => {
			},
		},
		'r2qkb1r/ppp1pppp/2np1n2/8/3PP1b1/2N2N2/PPP1BPPP/R1BQK2R b KQkq' => {
			eco => 'B00r',
			# TRANSLATORS: B00r: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Be2
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Be2'),
			moves => {
			},
		},
		'r2qkb1r/ppp1pppp/2np1n2/8/3PP1b1/2N1BN2/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B00s',
			# TRANSLATORS: B00s: 1. e4 Nc6 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Be3
			variation => N__('Nimzowitsch Defence: 2. Nf3 d6 3. d4 Nf6 4. Nc3 Bg4 5. Be3'),
			moves => {
			},
		},
		'r1bqkbnr/pppppppp/2n5/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00t',
			# TRANSLATORS: B00t: 1. e4 Nc6 2. d4
			variation => N__('Nimzowitsch Defence: 2. d4'),
			moves => {
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2np4/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00t',
			# TRANSLATORS: B00t: 1. e4 Nc6 2. d4 d6
			variation => N__('Nimzowitsch Defence: 2. d4 d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2np4/8/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B00t',
			# TRANSLATORS: B00t: 1. e4 Nc6 2. d4 d6 3. Nc3
			variation => N__('Nimzowitsch Defence: 2. d4 d6 3. Nc3'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4p3/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00u',
			# TRANSLATORS: B00u: 1. e4 Nc6 2. d4 e5
			variation => N__('Nimzowitsch Defence: 2. d4 e5'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/3Pp3/4P3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00u',
			# TRANSLATORS: B00u: 1. e4 Nc6 2. d4 e5 3. d5
			variation => N__('Nimzowitsch Defence: 2. d4 e5 3. d5'),
			moves => {
			},
		},
		'r1bqkbnr/pppp1ppp/2n5/4P3/4P3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00u',
			# TRANSLATORS: B00u: 1. e4 Nc6 2. d4 e5 3. dxe5
			variation => N__('Nimzowitsch Defence: 2. d4 e5 3.dxe5'),
			moves => {
				'f8c5' => 1,
			},
		},
		'r1bqk1nr/pppp1ppp/2n5/2b1P3/4P3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00u',
			# TRANSLATORS: B00u: 1. e4 Nc6 2. d4 e5 3. dxe5 Bc5
			variation => N__('Nimzowitsch Defence: Bielefelder Gambit'),
			moves => {
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3p4/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5
			variation => N__('Nimzowitsch Defence: 2. d4 d5'),
			moves => {
				'b1c3' => 1,
				'e4d5' => 1,
				'e4e5' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3P4/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5 3. exd5
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3.exd5'),
			moves => {
				'c6b4' => 1,
				'd8d5' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/8/3P4/1n1P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5 3. exd5 Nb4
			variation => N__('Nimzowitsch Defence: Aachen Gambit'),
			moves => {
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3.exd5 Qxd5'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5 4. Nc3
			variation => N__('Nimzowitsch Defence: Marshall Gambit'),
			moves => {
			},
		},
		'r1b1kbnr/ppp1pppp/2n5/3q4/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B00v',
			# TRANSLATORS: B00v: 1. e4 Nc6 2. d4 d5 3. exd5 Qxd5 4. Nf3
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3.exd5 Qxd5 4. Nf3'),
			moves => {
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3p4/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B00w',
			# TRANSLATORS: B00w: 1. e4 Nc6 2. d4 d5 3. Nc3
			variation => N__('Nimzowitsch Defence: Bogoljubow Variation'),
			moves => {
				'd5e4' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/8/3Pp3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B00w',
			# TRANSLATORS: B00w: 1. e4 Nc6 2. d4 d5 3. Nc3 dxe4
			variation => N__('Nimzowitsch Defence: Bogoljubow, 3...dxe4'),
			moves => {
			},
		},
		'r1bqkbnr/ppp2ppp/2n5/3pp3/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B00w',
			# TRANSLATORS: B00w: 1. e4 Nc6 2. d4 d5 3. Nc3 e5
			variation => N__('Nimzowitsch Defence: Bogoljubow, 3... e5'),
			moves => {
			},
		},
		'r1bqkb1r/ppp1pppp/2n2n2/3p4/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B00w',
			# TRANSLATORS: B00w: 1. e4 Nc6 2. d4 d5 3. Nc3 Nf6
			variation => N__('Nimzowitsch Defence: Bogoljubow, 3... Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/ppp1pppp/2n5/3pP3/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B00x',
			# TRANSLATORS: B00x: 1. e4 Nc6 2. d4 d5 3. e5
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3. e5'),
			moves => {
				'c8f5' => 1,
			},
		},
		'r2qkbnr/ppp1pppp/2n5/3pPb2/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B00x',
			# TRANSLATORS: B00x: 1. e4 Nc6 2. d4 d5 3. e5 Bf5
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3. e5 Bf5'),
			moves => {
				'c2c3' => 1,
			},
		},
		'r2qkbnr/ppp1pppp/2n5/3pPb2/3P4/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B00x',
			# TRANSLATORS: B00x: 1. e4 Nc6 2. d4 d5 3. e5 Bf5 4. c3
			variation => N__('Nimzowitsch Defence: 2. d4 d5 3. e5 Bf5 4. c3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01a',
			# TRANSLATORS: B01a: 1. e4 d5
			variation => N__('Scandinavian (Centre Counter)'),
			moves => {
				'd2d3' => 1,
				'e4d5' => 1,
				'e4e5' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/8/3p4/4P3/3P4/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B01a',
			# TRANSLATORS: B01a: 1. e4 d5 2. d3
			variation => N__('Scandinavian: 2. d3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3P4/8/8/PPPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B01a',
			# TRANSLATORS: B01a: 1. e4 d5 2. exd5
			variation => N__('Scandinavian: 2.exd5'),
			moves => {
				'd8d5' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/ppp2ppp/8/3Pp3/8/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01a',
			# TRANSLATORS: B01a: 1. e4 d5 2. exd5 e5
			variation => N__('Scandinavian: Böhnke Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/3pP3/8/8/PPPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B01a',
			# TRANSLATORS: B01a: 1. e4 d5 2. e5
			variation => N__('Scandinavian: 2. e5'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/3q4/8/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5
			variation => N__('Scandinavian: 2...Qxd5'),
			moves => {
				'b1c3' => 1,
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'rnb1kbnr/ppp1pppp/8/3q4/8/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. Nc3
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3'),
			moves => {
				'd5a5' => 1,
				'd5d6' => 1,
				'd5d8' => 1,
			},
		},
		'rnb1kbnr/ppp1pppp/8/3q4/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. d4
			variation => N__('Scandinavian: 2...Qxd5 3. d4'),
			moves => {
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnb1kbnr/ppp2ppp/8/3qp3/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. d4 e5
			variation => N__('Scandinavian: 2...Qxd5 3. d4 e5'),
			moves => {
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/3q4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. d4 Nf6
			variation => N__('Scandinavian: 2...Qxd5 3. d4 Nf6'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/3q4/8/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. Nf3
			variation => N__('Scandinavian: 2...Qxd5 3. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/3q4/8/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B01b',
			# TRANSLATORS: B01b: 1. e4 d5 2. exd5 Qxd5 3. Nf3 Nf6
			variation => N__('Scandinavian: 2...Qxd5 3. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/8/8/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B01c',
			# TRANSLATORS: B01c: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd8
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qd8'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/3q4/8/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B01d',
			# TRANSLATORS: B01d: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd6
			variation => N__('Scandinavian: Schiller Defence'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnb1kbnr/ppp1pppp/3q4/8/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B01d',
			# TRANSLATORS: B01d: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd6 4. d4
			variation => N__('Scandinavian: Schiller Defence'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/3q1n2/8/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B01d',
			# TRANSLATORS: B01d: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd6 4. d4 Nf6
			variation => N__('Scandinavian: Schiller, 4. d4 Nf6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/3q1n2/8/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B01d',
			# TRANSLATORS: B01d: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd6 4. d4 Nf6 5. Nf3
			variation => N__('Scandinavian: Schiller, 4. d4 Nf6'),
			moves => {
				'a7a6' => 1,
			},
		},
		'rnb1kb1r/1pp1pppp/p2q1n2/8/3P4/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01d',
			# TRANSLATORS: B01d: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qd6 4. d4 Nf6 5. Nf3 a6
			variation => N__('Scandinavian: Schiller, Bronstein Variation'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B01e',
			# TRANSLATORS: B01e: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5'),
			moves => {
				'b2b4' => 1,
				'd2d4' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/1P6/2N5/P1PP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B01e',
			# TRANSLATORS: B01e: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. b4
			variation => N__('Scandinavian, Mieses Gambit'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/2B5/2N5/PPPP1PPP/R1BQK1NR b KQkq' => {
			eco => 'B01e',
			# TRANSLATORS: B01e: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. Bc4
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. Bc4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/2B5/2N5/PPPP1PPP/R1BQK1NR w KQkq' => {
			eco => 'B01e',
			# TRANSLATORS: B01e: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. Bc4 Nf6
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. Bc4 Nf6'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/8/2N3P1/PPPP1P1P/R1BQKBNR b KQkq' => {
			eco => 'B01e',
			# TRANSLATORS: B01e: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. g3
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. g3'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/8/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B01f',
			# TRANSLATORS: B01f: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. Nf3
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/8/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B01f',
			# TRANSLATORS: B01f: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. Nf3 Nf6
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. Nf3 Nf6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/8/2N2N2/PPPPBPPP/R1BQK2R b KQkq' => {
			eco => 'B01f',
			# TRANSLATORS: B01f: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. Nf3 Nf6 5. Be2
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. Nf3 Nf6 5. Be2'),
			moves => {
			},
		},
		'rnb1kbnr/ppp1pppp/8/q7/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B01g',
			# TRANSLATORS: B01g: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. d4'),
			moves => {
				'c7c6' => 1,
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'rnb1kbnr/ppp2ppp/8/q3p3/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B01g',
			# TRANSLATORS: B01g: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 e5
			variation => N__('Scandinavian: Anderssen Counterattack'),
			moves => {
				'd4e5' => 1,
				'g1f3' => 1,
			},
		},
		'rnb1kbnr/ppp2ppp/8/q3P3/8/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B01g',
			# TRANSLATORS: B01g: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 e5 5. dxe5
			variation => N__('Scandinavian: Anderssen Counterattack, 5.dxe5'),
			moves => {
			},
		},
		'rnb1kbnr/ppp2ppp/8/q3p3/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B01g',
			# TRANSLATORS: B01g: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 e5 5. Nf3
			variation => N__('Scandinavian: Anderssen Counterattack, Goteborg Variation'),
			moves => {
				'c8g4' => 1,
			},
		},
		'rn2kbnr/ppp2ppp/8/q3p3/3P2b1/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01g',
			# TRANSLATORS: B01g: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 e5 5. Nf3 Bg4
			variation => N__('Scandinavian: Anderssen Counterattack, Collijn Variation'),
			moves => {
			},
		},
		'rnb1kbnr/pp2pppp/2p5/q7/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B01h',
			# TRANSLATORS: B01h: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 c6
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. d4 c6'),
			moves => {
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/2p5/q7/2BP4/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B01h',
			# TRANSLATORS: B01h: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 c6 5. Bc4
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. d4 c6 5. Bc4'),
			moves => {
			},
		},
		'rnb1kbnr/pp2pppp/2p5/q7/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B01h',
			# TRANSLATORS: B01h: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 c6 5. Nf3
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. d4 c6 5. Nf3'),
			moves => {
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B01i',
			# TRANSLATORS: B01i: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6
			variation => N__('Scandinavian: 2...Qxd5 3. Nc3 Qa5 4. d4 Nf6'),
			moves => {
				'c1d2' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/3P4/2N5/PPPB1PPP/R2QKBNR b KQkq' => {
			eco => 'B01i',
			# TRANSLATORS: B01i: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Bd2
			variation => N__('Scandinavian: 2...Qxd5, 5. Bd2'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/2p2n2/q7/3P4/2N5/PPPB1PPP/R2QKBNR w KQkq' => {
			eco => 'B01i',
			# TRANSLATORS: B01i: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Bd2 c6
			variation => N__('Scandinavian: 2...Qxd5, 5. Bd2 c6'),
			moves => {
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/2BP4/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B01i',
			# TRANSLATORS: B01i: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Bc4
			variation => N__('Scandinavian: 2...Qxd5, 5. Bc4'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/2p2n2/q7/2BP4/2N5/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B01i',
			# TRANSLATORS: B01i: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Bc4 c6
			variation => N__('Scandinavian: 2...Qxd5, 5. Bc4 c6'),
			moves => {
			},
		},
		'rnb1kb1r/ppp1pppp/5n2/q7/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B01j',
			# TRANSLATORS: B01j: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3'),
			moves => {
				'c7c6' => 1,
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn2kb1r/ppp1pppp/5n2/q7/3P2b1/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01j',
			# TRANSLATORS: B01j: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bg4
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3 Bg4'),
			moves => {
				'h2h3' => 1,
			},
		},
		'rn2kb1r/ppp1pppp/5n2/q7/3P2b1/2N2N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B01k',
			# TRANSLATORS: B01k: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bg4 6. h3
			variation => N__('Scandinavian: Lasker Variation'),
			moves => {
			},
		},
		'rn2kb1r/ppp1pppp/5n2/q4b2/3P4/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01l',
			# TRANSLATORS: B01l: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bf5
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3 Bf5'),
			moves => {
				'f1c4' => 1,
				'f3e5' => 1,
			},
		},
		'rn2kb1r/ppp1pppp/5n2/q4b2/2BP4/2N2N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B01l',
			# TRANSLATORS: B01l: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bf5 6. Bc4
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3 Bf5 6. Bc4'),
			moves => {
			},
		},
		'rn2kb1r/ppp1pppp/5n2/q3Nb2/3P4/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B01l',
			# TRANSLATORS: B01l: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bf5 6. Ne5
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3 Bf5'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rn2kb1r/pp2pppp/2p2n2/q3Nb2/3P4/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01l',
			# TRANSLATORS: B01l: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bf5 6. Ne5 c6
			variation => N__('Scandinavian: 2...Qxd5, 5. Nf3 Bf5'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rn2kb1r/pp2pppp/2p2n2/q3Nb2/3P2P1/2N5/PPP2P1P/R1BQKB1R b KQkq' => {
			eco => 'B01l',
			# TRANSLATORS: B01l: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 Bf5 6. Ne5 c6 7. g4
			variation => N__('Scandinavian: Grünfeld Variation'),
			moves => {
			},
		},
		'rnb1kb1r/pp2pppp/2p2n2/q7/3P4/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B01m',
			# TRANSLATORS: B01m: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6
			variation => N__('Scandinavian: 2...Qxd5, Main Line'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/2p2n2/q7/2BP4/2N2N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B01n',
			# TRANSLATORS: B01n: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 6. Bc4'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn2kb1r/pp2pppp/2p2n2/q7/2BP2b1/2N2N2/PPP2PPP/R1BQK2R w KQkq' => {
			eco => 'B01n',
			# TRANSLATORS: B01n: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bg4
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 6. Bc4 Bg4'),
			moves => {
			},
		},
		'rn2kb1r/pp2pppp/2p2n2/q4b2/2BP4/2N2N2/PPP2PPP/R1BQK2R w KQkq' => {
			eco => 'B01o',
			# TRANSLATORS: B01o: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 6. Bc4 Bf5'),
			moves => {
				'c1d2' => 1,
			},
		},
		'rn2kb1r/pp2pppp/2p2n2/q4b2/2BP4/2N2N2/PPPB1PPP/R2QK2R b KQkq' => {
			eco => 'B01p',
			# TRANSLATORS: B01p: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 6. Bc4 Bf5 7. Bd2'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn2kb1r/pp3ppp/2p1pn2/q4b2/2BP4/2N2N2/PPPB1PPP/R2QK2R w KQkq' => {
			eco => 'B01p',
			# TRANSLATORS: B01p: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 6. Bc4 Bf5 7. Bd2 e6'),
			moves => {
				'c3d5' => 1,
				'd1e2' => 1,
			},
		},
		'rn2kb1r/pp3ppp/2p1pn2/q2N1b2/2BP4/5N2/PPPB1PPP/R2QK2R b KQkq' => {
			eco => 'B01p',
			# TRANSLATORS: B01p: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6 8. Nd5
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 8. Nd5'),
			moves => {
			},
		},
		'rn2kb1r/pp3ppp/2p1pn2/q4b2/2BP4/2N2N2/PPPBQPPP/R3K2R b KQkq' => {
			eco => 'B01q',
			# TRANSLATORS: B01q: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6 8. Qe2
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 8. Qe2'),
			moves => {
				'f8b4' => 1,
			},
		},
		'rn2k2r/pp3ppp/2p1pn2/q4b2/1bBP4/2N2N2/PPPBQPPP/R3K2R w KQkq' => {
			eco => 'B01q',
			# TRANSLATORS: B01q: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6 8. Qe2 Bb4
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 8. Qe2 Bb4'),
			moves => {
				'a2a3' => 1,
				'e1c1' => 1,
			},
		},
		'rn2k2r/pp3ppp/2p1pn2/q4b2/1bBP4/P1N2N2/1PPBQPPP/R3K2R b KQkq' => {
			eco => 'B01q',
			# TRANSLATORS: B01q: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6 8. Qe2 Bb4 9. a3
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 8. Qe2 Bb4 9. a3'),
			moves => {
			},
		},
		'rn2k2r/pp3ppp/2p1pn2/q4b2/1bBP4/2N2N2/PPPBQPPP/2KR3R b kq' => {
			eco => 'B01q',
			# TRANSLATORS: B01q: 1. e4 d5 2. exd5 Qxd5 3. Nc3 Qa5 4. d4 Nf6 5. Nf3 c6 6. Bc4 Bf5 7. Bd2 e6 8. Qe2 Bb4 9. O-O-O
			variation => N__('Scandinavian: 2...Qxd5, Main Line, 8. Qe2 Bb4 9.O-O-O'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/8/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01r',
			# TRANSLATORS: B01r: 1. e4 d5 2. exd5 Nf6
			variation => N__('Scandinavian: 2... Nf6'),
			moves => {
				'c2c4' => 1,
				'd2d4' => 1,
				'f1b5' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/2B5/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B01r',
			# TRANSLATORS: B01r: 1. e4 d5 2. exd5 Nf6 3. Bc4
			variation => N__('Scandinavian: 2... Nf6 3. Bc4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/8/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B01r',
			# TRANSLATORS: B01r: 1. e4 d5 2. exd5 Nf6 3. Nf3
			variation => N__('Scandinavian: 2... Nf6 3. Nf3'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B01r',
			# TRANSLATORS: B01r: 1. e4 d5 2. exd5 Nf6 3. Nf3 Nxd5
			variation => N__('Scandinavian: 2... Nf6 3. Nf3 Nxd5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/1B1P4/8/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+'),
			moves => {
				'b8d7' => 1,
				'c8d7' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/5n2/1B1P4/8/8/PPPP1PPP/RNBQK1NR w KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Nbd7
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Nbd7'),
			moves => {
			},
		},
		'rn1qkb1r/pppbpppp/5n2/1B1P4/8/8/PPPP1PPP/RNBQK1NR w KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7'),
			moves => {
				'b5c4' => 1,
				'b5e2' => 1,
			},
		},
		'rn1qkb1r/pppbpppp/5n2/3P4/2B5/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Bc4
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Bc4'),
			moves => {
				'd7g4' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/2B3b1/8/PPPP1PPP/RNBQK1NR w KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Bc4 Bg4
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Bc4 Bg4'),
			moves => {
			},
		},
		'rn1qkb1r/pppbpppp/5n2/3P4/8/8/PPPPBPPP/RNBQK1NR b KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Be2
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Be2'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rn1qkb1r/pppbpppp/8/3n4/8/8/PPPPBPPP/RNBQK1NR w KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Be2 Nxd5
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Be2'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rn1qkb1r/pppbpppp/8/3n4/3P4/8/PPP1BPPP/RNBQK1NR b KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Be2 Nxd5 5. d4
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Be2'),
			moves => {
				'd7f5' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/8/3n1b2/3P4/8/PPP1BPPP/RNBQK1NR w KQkq' => {
			eco => 'B01s',
			# TRANSLATORS: B01s: 1. e4 d5 2. exd5 Nf6 3. Bb5+ Bd7 4. Be2 Nxd5 5. d4 Bf5
			variation => N__('Scandinavian: 2... Nf6 3. Bb5+ Bd7 4. Be2 Nxd5 5. d4 Bf5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/2P5/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4
			variation => N__('Scandinavian: 2... Nf6 3. c4'),
			moves => {
				'c7c6' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3P4/2P5/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 c6
			variation => N__('Scandinavian: Scandinavian Gambit'),
			moves => {
				'd5c6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2P2n2/8/2P5/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 c6 4. dxc6
			variation => N__('Scandinavian: Scandinavian Gambit Accepted'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/4pn2/3P4/2P5/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 e6
			variation => N__('Scandinavian: Icelandic Gambit'),
			moves => {
				'd5e6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4Pn2/8/2P5/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 e6 4. dxe6
			variation => N__('Scandinavian: Icelandic Gambit Accepted'),
			moves => {
				'c8e6' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/4bn2/8/2P5/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 e6 4. dxe6 Bxe6
			variation => N__('Scandinavian: Icelandic Gambit Accepted'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/4bn2/8/2PP4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 e6 4. dxe6 Bxe6 5. d4
			variation => N__('Scandinavian: Icelandic Gambit, 5. d4'),
			moves => {
			},
		},
		'rn1qkb1r/ppp2ppp/4bn2/8/2P5/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B01t',
			# TRANSLATORS: B01t: 1. e4 d5 2. exd5 Nf6 3. c4 e6 4. dxe6 Bxe6 5. Nf3
			variation => N__('Scandinavian: Icelandic Gambit, 5. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4
			variation => N__('Scandinavian: 2... Nf6 3. d4'),
			moves => {
				'c8g4' => 1,
				'f6d5' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/3P2b1/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4
			variation => N__('Scandinavian: Portuguese Variation'),
			moves => {
				'f1b5' => 1,
				'f1e2' => 1,
				'f2f3' => 1,
				'g1f3' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/1B1P4/3P2b1/8/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Bb5+
			variation => N__('Scandinavian: Portuguese, 4. Bb5+'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/3P2b1/8/PPP1BPPP/RNBQK1NR b KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Be2
			variation => N__('Scandinavian: Portuguese, 4. Be2'),
			moves => {
				'g4e2' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/3P4/8/PPP1bPPP/RNBQK1NR w KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Be2 Bxe2
			variation => N__('Scandinavian: Portuguese, 4. Be2 Bxe2'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/3P2b1/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Nf3
			variation => N__('Scandinavian: Portuguese, 4. Nf3'),
			moves => {
				'd8d5' => 1,
			},
		},
		'rn2kb1r/ppp1pppp/5n2/3q4/3P2b1/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Nf3 Qxd5
			variation => N__('Scandinavian: Portuguese, 4. Nf3 Qxd5'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn2kb1r/ppp1pppp/5n2/3q4/3P2b1/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Nf3 Qxd5 5. Be2
			variation => N__('Scandinavian: Portuguese, 4. Nf3 Qxd5 5. Be2'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r3kb1r/ppp1pppp/2n2n2/3q4/3P2b1/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. Nf3 Qxd5 5. Be2 Nc6
			variation => N__('Scandinavian: Portuguese, 4. Nf3 Qxd5 5. Be2 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/5np1/3P4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B01u',
			# TRANSLATORS: B01u: 1. e4 d5 2. exd5 Nf6 3. d4 g6
			variation => N__('Scandinavian: Richter Variation'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P4/3P2b1/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3
			variation => N__('Scandinavian: Portuguese, 4. f3'),
			moves => {
				'g4f5' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P1b2/3P4/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3 Bf5
			variation => N__('Scandinavian: Portuguese, 4. f3 Bf5'),
			moves => {
				'c2c4' => 1,
				'f1b5' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/3P1b2/2PP4/5P2/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3 Bf5 5. c4
			variation => N__('Scandinavian: Portuguese, 4. f3 Bf5 5. c4'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/5n2/1B1P1b2/3P4/5P2/PPP3PP/RNBQK1NR b KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3 Bf5 5. Bb5+
			variation => N__('Scandinavian: Portuguese, 4. f3 Bf5 5. Bb5+'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkb1r/pppnpppp/5n2/1B1P1b2/3P4/5P2/PPP3PP/RNBQK1NR w KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3 Bf5 5. Bb5+ Nbd7
			variation => N__('Scandinavian: Portuguese, 4. f3 Bf5 5. Bb5+ Nbd7'),
			moves => {
				'c2c4' => 1,
			},
		},
		'r2qkb1r/pppnpppp/5n2/1B1P1b2/2PP4/5P2/PP4PP/RNBQK1NR b KQkq' => {
			eco => 'B01v',
			# TRANSLATORS: B01v: 1. e4 d5 2. exd5 Nf6 3. d4 Bg4 4. f3 Bf5 5. Bb5+ Nbd7 6. c4
			variation => N__('Scandinavian: Portuguese, 4. f3 Bf5 5. Bb5+ Nbd7 6. c4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5
			variation => N__('Scandinavian: Marshall Variation'),
			moves => {
				'c2c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3
			variation => N__('Scandinavian: Marshall, 4. Nf3'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/8/3n1b2/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 Bf5
			variation => N__('Scandinavian: Marshall, 4. Nf3 Bf5'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/8/3n4/3P2b1/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 Bg4
			variation => N__('Scandinavian: Marshall, 4. Nf3 Bg4'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/8/3n4/3P2b1/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 Bg4 5. Be2
			variation => N__('Scandinavian: Marshall, 4. Nf3 Bg4 5. Be2'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/4p3/3n4/3P2b1/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B01w',
			# TRANSLATORS: B01w: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 Bg4 5. Be2 e6
			variation => N__('Scandinavian: Marshall, 4. Nf3 Bg4 5. Be2 e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B01x',
			# TRANSLATORS: B01x: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 g6
			variation => N__('Scandinavian: Marshall, 4. Nf3 g6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3n4/3P4/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B01x',
			# TRANSLATORS: B01x: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 g6 5. Be2
			variation => N__('Scandinavian: Marshall, 4. Nf3 g6 5. Be2'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/6p1/3n4/3P4/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B01x',
			# TRANSLATORS: B01x: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. Nf3 g6 5. Be2 Bg7
			variation => N__('Scandinavian: Marshall, 4. Nf3 g6 5. Be2 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/2PP4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B01y',
			# TRANSLATORS: B01y: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4
			variation => N__('Scandinavian: Marshall, 4. c4'),
			moves => {
				'd5b6' => 1,
				'd5f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/8/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B01y',
			# TRANSLATORS: B01y: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6'),
			moves => {
				'b1c3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/8/2PP4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B01y',
			# TRANSLATORS: B01y: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nc3
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nc3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/1n6/4p3/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B01y',
			# TRANSLATORS: B01y: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nc3 e5
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nc3 e5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/8/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B01y',
			# TRANSLATORS: B01y: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nf6
			variation => N__('Scandinavian: Marshall, 4. c4 Nf6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/8/2PP4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3'),
			moves => {
				'c8g4' => 1,
				'g7g6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/1n6/8/2PP2b1/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 Bg4
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 Bg4'),
			moves => {
				'c4c5' => 1,
				'f1e2' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/1n6/2P5/3P2b1/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 Bg4 6. c5
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 Bg4 6. c5'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/1n6/8/2PP2b1/5N2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 Bg4 6. Be2
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 Bg4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/1n2p3/8/2PP2b1/5N2/PP2BPPP/RNBQK2R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 Bg4 6. Be2 e6
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 Bg4 6. Be2 e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/1n4p1/8/2PP4/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6'),
			moves => {
				'b1c3' => 1,
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/1n4p1/8/2PP4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. Nc3
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/1n4p1/8/2PP4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. Nc3 Bg7
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. Nc3 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/1n4p1/8/2PP4/5N2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. Be2
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. Be2'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/1n4p1/8/2PP4/5N2/PP2BPPP/RNBQK2R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. Be2 Bg7
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. Be2 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/1n4p1/8/2PP4/5N1P/PP3PP1/RNBQKB1R b KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. h3
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. h3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/1n4p1/8/2PP4/5N1P/PP3PP1/RNBQKB1R w KQkq' => {
			eco => 'B01z',
			# TRANSLATORS: B01z: 1. e4 d5 2. exd5 Nf6 3. d4 Nxd5 4. c4 Nb6 5. Nf3 g6 6. h3 Bg7
			variation => N__('Scandinavian: Marshall, 4. c4 Nb6 5. Nf3 g6 6. h3 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02a',
			# TRANSLATORS: B02a: 1. e4 Nf6
			variation => N__('Alekhine Defence'),
			moves => {
				'b1c3' => 1,
				'd2d3' => 1,
				'e4e5' => 1,
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/2B1P3/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B02a',
			# TRANSLATORS: B02a: 1. e4 Nf6 2. Bc4
			variation => N__('Alekhine: Krejcik Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/4P3/3P4/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B02b',
			# TRANSLATORS: B02b: 1. e4 Nf6 2. d3
			variation => N__('Alekhine: Maroczy Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/8/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02c',
			# TRANSLATORS: B02c: 1. e4 Nf6 2. Nc3
			variation => N__('Alekhine: Scandinavian Variation'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3p4/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02c',
			# TRANSLATORS: B02c: 1. e4 Nf6 2. Nc3 d5
			variation => N__('Alekhine: Scandinavian Variation'),
			moves => {
				'e4d5' => 1,
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3pP3/8/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02d',
			# TRANSLATORS: B02d: 1. e4 Nf6 2. Nc3 d5 3. e5
			variation => N__('Alekhine: Scandinavian, 3. e5'),
			moves => {
				'f6d7' => 1,
				'f6e4' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3pP3/4n3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02e',
			# TRANSLATORS: B02e: 1. e4 Nf6 2. Nc3 d5 3. e5 Ne4
			variation => N__('Alekhine: Scandinavian, 3. e5 Ne4'),
			moves => {
			},
		},
		'rnbqkb1r/pppnpppp/8/3pP3/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02f',
			# TRANSLATORS: B02f: 1. e4 Nf6 2. Nc3 d5 3. e5 Nfd7
			variation => N__('Alekhine: Scandinavian, 3. e5 Nfd7'),
			moves => {
				'e5e6' => 1,
			},
		},
		'rnbqkb1r/pppnpppp/4P3/3p4/8/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02f',
			# TRANSLATORS: B02f: 1. e4 Nf6 2. Nc3 d5 3. e5 Nfd7 4. e6
			variation => N__('Alekhine: Spielmann Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/5n2/3P4/8/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02g',
			# TRANSLATORS: B02g: 1. e4 Nf6 2. Nc3 d5 3. exd5
			variation => N__('Alekhine: Scandinavian, Exchange'),
			moves => {
				'c7c6' => 1,
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/3P4/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02g',
			# TRANSLATORS: B02g: 1. e4 Nf6 2. Nc3 d5 3. exd5 c6
			variation => N__('Alekhine: Scandinavian, Geschev Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02g',
			# TRANSLATORS: B02g: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5
			variation => N__('Alekhine: Scandinavian, Exchange'),
			moves => {
				'c3d5' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/8/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B02g',
			# TRANSLATORS: B02g: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Nf3
			variation => N__('Alekhine: Scandinavian, Exchange, 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3N4/8/8/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02h',
			# TRANSLATORS: B02h: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Nxd5
			variation => N__('Alekhine: Scandinavian, Exchange, 4.Nxd5'),
			moves => {
				'd8d5' => 1,
			},
		},
		'rnb1kb1r/ppp1pppp/8/3q4/8/8/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02h',
			# TRANSLATORS: B02h: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Nxd5 Qxd5
			variation => N__('Alekhine: Scandinavian, Exchange, 4.Nxd5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3n4/2B5/2N5/PPPP1PPP/R1BQK1NR b KQkq' => {
			eco => 'B02i',
			# TRANSLATORS: B02i: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Bc4
			variation => N__('Alekhine: Scandinavian, Exchange, 4. Bc4'),
			moves => {
				'd5b6' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/4p3/3n4/2B5/2N5/PPPP1PPP/R1BQK1NR w KQkq' => {
			eco => 'B02j',
			# TRANSLATORS: B02j: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Bc4 e6
			variation => N__('Alekhine: Scandinavian, Exchange, 4. Bc4 e6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/8/2B5/2N5/PPPP1PPP/R1BQK1NR w KQkq' => {
			eco => 'B02k',
			# TRANSLATORS: B02k: 1. e4 Nf6 2. Nc3 d5 3. exd5 Nxd5 4. Bc4 Nb6
			variation => N__('Alekhine: Scandinavian, Exchange, 4. Bc4 Nb6'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/5n2/4P3/8/8/PPPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B02l',
			# TRANSLATORS: B02l: 1. e4 Nf6 2. e5
			variation => N__('Alekhine: 2. e5'),
			moves => {
				'f6d5' => 1,
				'f6e4' => 1,
				'f6g8' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/4P3/4n3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02l',
			# TRANSLATORS: B02l: 1. e4 Nf6 2. e5 Ne4
			variation => N__('Alekhine: Mokele Mbembe (Bücker) Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pppppppp/8/4P3/8/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02l',
			# TRANSLATORS: B02l: 1. e4 Nf6 2. e5 Ng8
			variation => N__('Alekhine: Brooklyn Defence (Retreat Variation)'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/8/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02m',
			# TRANSLATORS: B02m: 1. e4 Nf6 2. e5 Nd5
			variation => N__('Alekhine: 2. e5 Nd5'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'c2c4' => 1,
				'd2d4' => 1,
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/8/1P6/P1PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B02m',
			# TRANSLATORS: B02m: 1. e4 Nf6 2. e5 Nd5 3. b3
			variation => N__('Alekhine: Welling Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/2B5/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B02n',
			# TRANSLATORS: B02n: 1. e4 Nf6 2. e5 Nd5 3. Bc4
			variation => N__('Alekhine: 3. Bc4'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rnbqkb1r/pppppppp/1n6/4P3/2B5/8/PPPP1PPP/RNBQK1NR w KQkq' => {
			eco => 'B02n',
			# TRANSLATORS: B02n: 1. e4 Nf6 2. e5 Nd5 3. Bc4 Nb6
			variation => N__('Alekhine: 3. Bc4'),
			moves => {
				'c4b3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/1n6/4P3/8/1B6/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B02n',
			# TRANSLATORS: B02n: 1. e4 Nf6 2. e5 Nd5 3. Bc4 Nb6 4. Bb3
			variation => N__('Alekhine: 3. Bc4'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/1n6/2p1P3/8/1B6/PPPP1PPP/RNBQK1NR w KQkq' => {
			eco => 'B02n',
			# TRANSLATORS: B02n: 1. e4 Nf6 2. e5 Nd5 3. Bc4 Nb6 4. Bb3 c5
			variation => N__('Alekhine: 3. Bc4'),
			moves => {
				'd2d3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/1n6/2p1P3/8/1B1P4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B02n',
			# TRANSLATORS: B02n: 1. e4 Nf6 2. e5 Nd5 3. Bc4 Nb6 4. Bb3 c5 5. d3
			variation => N__('Alekhine: Kmoch Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/8/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02o',
			# TRANSLATORS: B02o: 1. e4 Nf6 2. e5 Nd5 3. Nc3
			variation => N__('Alekhine: Sämisch Attack'),
			moves => {
				'd5c3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/4P3/8/2n5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B02p',
			# TRANSLATORS: B02p: 1. e4 Nf6 2. e5 Nd5 3. Nc3 Nxc3
			variation => N__('Alekhine: Sämisch Attack, 3...Nxc3'),
			moves => {
				'd2c3' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/4P3/8/2P5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B02q',
			# TRANSLATORS: B02q: 1. e4 Nf6 2. e5 Nd5 3. Nc3 Nxc3 4. dxc3
			variation => N__('Alekhine: Sämisch Attack, 3...Nxc3 4.dxc3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/2P5/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B02r',
			# TRANSLATORS: B02r: 1. e4 Nf6 2. e5 Nd5 3. c4
			variation => N__('Alekhine: Chase Variation'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rnbqkb1r/pppppppp/1n6/4P3/2P5/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02r',
			# TRANSLATORS: B02r: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6
			variation => N__('Alekhine: Chase Variation'),
			moves => {
				'b2b3' => 1,
				'c4c5' => 1,
			},
		},
		'rnbqkb1r/pppppppp/1n6/4P3/2P5/1P6/P2P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B02r',
			# TRANSLATORS: B02r: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. b3
			variation => N__('Alekhine: Chase, Steiner Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/1n6/2P1P3/8/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B02s',
			# TRANSLATORS: B02s: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5
			variation => N__('Alekhine: Two Pawns (Lasker) Attack'),
			moves => {
				'b6d5' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/2PnP3/8/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B02s',
			# TRANSLATORS: B02s: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5
			variation => N__('Alekhine: Two Pawns (Lasker) Attack'),
			moves => {
				'b1c3' => 1,
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pppppppp/8/2PnP3/8/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'B02t',
			# TRANSLATORS: B02t: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5 5. Nc3
			variation => N__('Alekhine: Two Pawns Attack, 5. Nc3'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/2PnP3/2B5/8/PP1P1PPP/RNBQK1NR b KQkq' => {
			eco => 'B02u',
			# TRANSLATORS: B02u: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5 5. Bc4
			variation => N__('Alekhine: Two Pawns Attack, 5. Bc4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4p3/2PnP3/2B5/8/PP1P1PPP/RNBQK1NR w KQkq' => {
			eco => 'B02u',
			# TRANSLATORS: B02u: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5 5. Bc4 e6
			variation => N__('Alekhine: Two Pawns Attack, 5. Bc4'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pppp1ppp/4p3/2PnP3/2B5/2N5/PP1P1PPP/R1BQK1NR b KQkq' => {
			eco => 'B02u',
			# TRANSLATORS: B02u: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5 5. Bc4 e6 6. Nc3
			variation => N__('Alekhine: Two Pawns Attack, 5. Bc4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3pp3/2PnP3/2B5/2N5/PP1P1PPP/R1BQK1NR w KQkq' => {
			eco => 'B02u',
			# TRANSLATORS: B02u: 1. e4 Nf6 2. e5 Nd5 3. c4 Nb6 4. c5 Nd5 5. Bc4 e6 6. Nc3 d6
			variation => N__('Alekhine: Two Pawns Attack, Mikenas Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pppppppp/8/3nP3/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B03a',
			# TRANSLATORS: B03a: 1. e4 Nf6 2. e5 Nd5 3. d4
			variation => N__('Alekhine: 3. d4'),
			moves => {
				'b7b5' => 1,
				'd7d6' => 1,
			},
		},
		'rnbqkb1r/p1pppppp/8/1p1nP3/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B03a',
			# TRANSLATORS: B03a: 1. e4 Nf6 2. e5 Nd5 3. d4 b5
			variation => N__('Alekhine: O\'Sullivan Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p4/3nP3/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B03b',
			# TRANSLATORS: B03b: 1. e4 Nf6 2. e5 Nd5 3. d4 d6
			variation => N__('Alekhine: 3. d4 d6'),
			moves => {
				'c2c4' => 1,
				'e5d6' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3P4/3n4/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B03c',
			# TRANSLATORS: B03c: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. exd6
			variation => N__('Alekhine: 3. d4 d6 4.exd6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p4/3nP3/2BP4/8/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B03d',
			# TRANSLATORS: B03d: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Bc4
			variation => N__('Alekhine: Balogh Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p4/3nP3/2PP4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B03e',
			# TRANSLATORS: B03e: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4
			variation => N__('Alekhine: 4. c4'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/1n1p4/4P3/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B03e',
			# TRANSLATORS: B03e: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6
			variation => N__('Alekhine: 4. c4 Nb6'),
			moves => {
				'e5d6' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/1n1p4/4P3/2PP4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B03f',
			# TRANSLATORS: B03f: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. Nf3
			variation => N__('Alekhine: 4. c4 Nb6 5. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n1p4/4P3/2PP1P2/8/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'B03g',
			# TRANSLATORS: B03g: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4
			variation => N__('Alekhine: Four Pawns Attack'),
			moves => {
				'c8f5' => 1,
				'd6e5' => 1,
				'g7g5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/1n1p4/4P1p1/2PP1P2/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03g',
			# TRANSLATORS: B03g: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 g5
			variation => N__('Alekhine: Four Pawns Attack, Planinc Variation'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/1n1p4/4Pb2/2PP1P2/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03h',
			# TRANSLATORS: B03h: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 Bf5
			variation => N__('Alekhine: Four Pawns Attack, Trifunovic Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/1n1p2p1/4P3/2PP1P2/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03i',
			# TRANSLATORS: B03i: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 g6
			variation => N__('Alekhine: Four Pawns Attack, Fianchetto Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/4p3/2PP1P2/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03j',
			# TRANSLATORS: B03j: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5
			variation => N__('Alekhine: Four Pawns Attack, 5...dxe5'),
			moves => {
				'f4e5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/1n6/4P3/2PP4/8/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'B03j',
			# TRANSLATORS: B03j: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5
			variation => N__('Alekhine: Four Pawns Attack, 5...dxe5'),
			moves => {
				'b8c6' => 1,
				'c7c5' => 1,
				'c8f5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/1n6/2p1P3/2PP4/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03k',
			# TRANSLATORS: B03k: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 c5
			variation => N__('Alekhine: Four Pawns Attack, 6... c5'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/1n6/4Pb2/2PP4/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/1n6/4Pb2/2PP4/2N5/PP4PP/R1BQKBNR b KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/1n2p3/4Pb2/2PP4/2N5/PP4PP/R1BQKBNR w KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/1n2p3/4Pb2/2PP4/2N2N2/PP4PP/R1BQKB1R b KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'f8b4' => 1,
				'f8e7' => 1,
			},
		},
		'rn1qk2r/ppp2ppp/1n2p3/4Pb2/1bPP4/2N2N2/PP4PP/R1BQKB1R w KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Bb4
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rn1qk2r/ppp2ppp/1n2p3/4Pb2/1bPP4/2NB1N2/PP4PP/R1BQK2R b KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Bb4 9. Bd3
			variation => N__('Alekhine: Four Pawns Attack, Fahrni Variation'),
			moves => {
			},
		},
		'rn1qk2r/ppp1bppp/1n2p3/4Pb2/2PP4/2N2N2/PP4PP/R1BQKB1R w KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Be7
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/1n2p3/4Pb2/2PP4/2N2N2/PP2B1PP/R1BQK2R b KQkq' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Be7 9. Be2
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n2p3/4Pb2/2PP4/2N2N2/PP2B1PP/R1BQK2R w KQ' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Be7 9. Be2 O-O
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n2p3/4Pb2/2PP4/2N2N2/PP2B1PP/R1BQ1RK1 b -' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Be7 9. Be2 O-O 10. O-O
			variation => N__('Alekhine: Four Pawns Attack, 6... Bf5'),
			moves => {
				'f7f6' => 1,
			},
		},
		'rn1q1rk1/ppp1b1pp/1n2pp2/4Pb2/2PP4/2N2N2/PP2B1PP/R1BQ1RK1 w -' => {
			eco => 'B03l',
			# TRANSLATORS: B03l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Bf5 7. Nc3 e6 8. Nf3 Be7 9. Be2 O-O 10. O-O f6
			variation => N__('Alekhine: Four Pawns Attack, Korchnoi Variation'),
			moves => {
			},
		},
		'r1bqkb1r/ppp1pppp/1nn5/4P3/2PP4/8/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6
			variation => N__('Alekhine: Four Pawns Attack, 6... Nc6'),
			moves => {
				'c1e3' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/ppp1pppp/1nn5/4P3/2PP4/5N2/PP4PP/RNBQKB1R b KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Nf3
			variation => N__('Alekhine: Four Pawns Attack, 6... Nc6'),
			moves => {
				'c8g4' => 1,
			},
		},
		'r2qkb1r/ppp1pppp/1nn5/4P3/2PP2b1/5N2/PP4PP/RNBQKB1R w KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Nf3 Bg4
			variation => N__('Alekhine: Four Pawns Attack, 6... Nc6'),
			moves => {
				'e5e6' => 1,
			},
		},
		'r2qkb1r/ppp1pppp/1nn1P3/8/2PP2b1/5N2/PP4PP/RNBQKB1R b KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Nf3 Bg4 8. e6
			variation => N__('Alekhine: Four Pawns Attack, 6... Nc6'),
			moves => {
				'f7e6' => 1,
			},
		},
		'r2qkb1r/ppp1p1pp/1nn1p3/8/2PP2b1/5N2/PP4PP/RNBQKB1R w KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Nf3 Bg4 8. e6 fxe6
			variation => N__('Alekhine: Four Pawns Attack, 6... Nc6'),
			moves => {
				'c4c5' => 1,
			},
		},
		'r2qkb1r/ppp1p1pp/1nn1p3/2P5/3P2b1/5N2/PP4PP/RNBQKB1R b KQkq' => {
			eco => 'B03m',
			# TRANSLATORS: B03m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Nf3 Bg4 8. e6 fxe6 9. c5
			variation => N__('Alekhine: Four Pawns Attack, Ilyin-Zhenevsky Variation'),
			moves => {
			},
		},
		'r1bqkb1r/ppp1pppp/1nn5/4P3/2PP4/4B3/PP4PP/RN1QKBNR b KQkq' => {
			eco => 'B03n',
			# TRANSLATORS: B03n: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. f4 dxe5 6. fxe5 Nc6 7. Be3
			variation => N__('Alekhine: Four Pawns Attack, 7. Be3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n1P4/8/2PP4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B03o',
			# TRANSLATORS: B03o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6
			variation => N__('Alekhine: Exchange Variation'),
			moves => {
				'c7d6' => 1,
				'e7d6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/1n1p4/8/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B03o',
			# TRANSLATORS: B03o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 exd6
			variation => N__('Alekhine: Exchange, 5...exd6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/1n1p4/8/2PP4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B03p',
			# TRANSLATORS: B03p: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 exd6 6. Nc3
			variation => N__('Alekhine: Exchange, 5...exd6 6. Nc3'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rnbqk2r/ppp1bppp/1n1p4/8/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B03q',
			# TRANSLATORS: B03q: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 exd6 6. Nc3 Be7
			variation => N__('Alekhine: Exchange, 5...exd6 6. Nc3 Be7'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/1n1p4/8/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B03r',
			# TRANSLATORS: B03r: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 cxd6
			variation => N__('Alekhine: Exchange, 5...cxd6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/1n1p4/8/2PP4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B03s',
			# TRANSLATORS: B03s: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 cxd6 6. Nc3
			variation => N__('Alekhine: Exchange, 5...cxd6 6. Nc3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/1n1p2p1/8/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B03s',
			# TRANSLATORS: B03s: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 cxd6 6. Nc3 g6
			variation => N__('Alekhine: Exchange, 5...cxd6 6. Nc3'),
			moves => {
				'c1e3' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/1n1p2p1/8/2PP4/2N1B3/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'B03t',
			# TRANSLATORS: B03t: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. c4 Nb6 5. exd6 cxd6 6. Nc3 g6 7. Be3
			variation => N__('Alekhine: Exchange, 5...cxd6 6. Nc3 g6 7. Be3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p4/3nP3/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B04a',
			# TRANSLATORS: B04a: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3
			variation => N__('Alekhine: Modern Variation'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'c8g4' => 1,
				'd5b6' => 1,
				'd6e5' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/ppp1pppp/2np4/3nP3/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04b',
			# TRANSLATORS: B04b: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Nc6
			variation => N__('Alekhine: Modern, 4... Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2pp4/3nP3/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04c',
			# TRANSLATORS: B04c: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 c6
			variation => N__('Alekhine: Modern, 4... c6'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2pp4/3nP3/2PP4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B04d',
			# TRANSLATORS: B04d: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 c6 5. c4
			variation => N__('Alekhine: Modern, 4... c6 5. c4'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/1n1p4/4P3/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04e',
			# TRANSLATORS: B04e: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Nb6
			variation => N__('Alekhine: Modern, Schmid Variation'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/8/3np3/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04f',
			# TRANSLATORS: B04f: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5
			variation => N__('Alekhine: Modern, Larsen Variation'),
			moves => {
				'f3e5' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/8/3nN3/3P4/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B04f',
			# TRANSLATORS: B04f: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5 5. Nxe5
			variation => N__('Alekhine: Modern, Larsen Variation'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3nN3/3P4/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04g',
			# TRANSLATORS: B04g: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5 5. Nxe5 g6
			variation => N__('Alekhine: Modern, Larsen, 5.Nxe5 g6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/6p1/3nN3/2BP4/8/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B04h',
			# TRANSLATORS: B04h: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5 5. Nxe5 g6 6. Bc4
			variation => N__('Alekhine: Modern, Larsen, 5.Nxe5 g6 6. Bc4'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/2p3p1/3nN3/2BP4/8/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B04h',
			# TRANSLATORS: B04h: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5 5. Nxe5 g6 6. Bc4 c6
			variation => N__('Alekhine: Modern, Larsen, 5.Nxe5 g6 6. Bc4'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/2p3p1/3nN3/2BP4/8/PPP2PPP/RNBQ1RK1 b kq' => {
			eco => 'B04i',
			# TRANSLATORS: B04i: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 dxe5 5. Nxe5 g6 6. Bc4 c6 7. O-O
			variation => N__('Alekhine: Modern, Larsen, 5.Nxe5 g6 6. Bc4 c6 7.O-O'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p2p1/3nP3/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B04j',
			# TRANSLATORS: B04j: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6
			variation => N__('Alekhine: Modern, Fianchetto Variation'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p2p1/3nP3/2BP4/5N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B04k',
			# TRANSLATORS: B04k: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4
			variation => N__('Alekhine: Modern, Fianchetto, 5. Bc4'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/1n1p2p1/4P3/2BP4/5N2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B04l',
			# TRANSLATORS: B04l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4 Nb6
			variation => N__('Alekhine: Modern, Fianchetto, 5. Bc4 Nb6'),
			moves => {
				'c4b3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/1n1p2p1/4P3/3P4/1B3N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B04l',
			# TRANSLATORS: B04l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4 Nb6 6. Bb3
			variation => N__('Alekhine: Modern, Fianchetto, 5. Bc4 Nb6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/1n1p2p1/4P3/3P4/1B3N2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B04m',
			# TRANSLATORS: B04m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4 Nb6 6. Bb3 Bg7
			variation => N__('Alekhine: Modern, Fianchetto, 5. Bc4 Nb6 6. Bb3 Bg7'),
			moves => {
				'a2a4' => 1,
				'f3g5' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/1n1p2p1/4P3/P2P4/1B3N2/1PP2PPP/RNBQK2R b KQkq' => {
			eco => 'B04n',
			# TRANSLATORS: B04n: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4 Nb6 6. Bb3 Bg7 7. a4
			variation => N__('Alekhine: Modern, Fianchetto, Keres Variation'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/1n1p2p1/4P1N1/3P4/1B6/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B04o',
			# TRANSLATORS: B04o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 g6 5. Bc4 Nb6 6. Bb3 Bg7 7. Ng5
			variation => N__('Alekhine: Modern, Fianchetto, 5. Bc4 Nb6 6. Bb3 Bg7 7. Ng5'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p4/3nP3/3P2b1/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B05a',
			# TRANSLATORS: B05a: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4
			variation => N__('Alekhine: Modern, 4... Bg4'),
			moves => {
				'c2c4' => 1,
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/3p4/3nP3/3P2b1/5N1P/PPP2PP1/RNBQKB1R b KQkq' => {
			eco => 'B05b',
			# TRANSLATORS: B05b: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. h3
			variation => N__('Alekhine: Modern, Panov Variation'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p4/3nP3/2PP2b1/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B05c',
			# TRANSLATORS: B05c: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. c4
			variation => N__('Alekhine: Modern, Alekhine Variation'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/1n1p4/4P3/2PP2b1/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B05c',
			# TRANSLATORS: B05c: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. c4 Nb6
			variation => N__('Alekhine: Modern, Alekhine Variation'),
			moves => {
				'e5d6' => 1,
			},
		},
		'rn1qkb1r/ppp1pppp/1n1P4/8/2PP2b1/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B05d',
			# TRANSLATORS: B05d: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. c4 Nb6 6. exd6
			variation => N__('Alekhine: Modern, Alekhine Variation, 6.exd6'),
			moves => {
				'c7d6' => 1,
				'e7d6' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/1n1p4/8/2PP2b1/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B05d',
			# TRANSLATORS: B05d: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. c4 Nb6 6. exd6 cxd6
			variation => N__('Alekhine: Modern, Alekhine Variation, 6.exd6 cxd6'),
			moves => {
			},
		},
		'rn1qkb1r/ppp2ppp/1n1p4/8/2PP2b1/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B05e',
			# TRANSLATORS: B05e: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. c4 Nb6 6. exd6 exd6
			variation => N__('Alekhine: Modern, Alekhine Variation, 6.exd6 exd6'),
			moves => {
			},
		},
		'rn1qkb1r/ppp1pppp/3p4/3nP3/3P2b1/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B05f',
			# TRANSLATORS: B05f: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2
			variation => N__('Alekhine: Modern, 5. Be2'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'e7e6' => 1,
			},
		},
		'r2qkb1r/ppp1pppp/2np4/3nP3/3P2b1/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B05g',
			# TRANSLATORS: B05g: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 Nc6
			variation => N__('Alekhine: Modern, 5. Be2 Nc6'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2pp4/3nP3/3P2b1/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B05h',
			# TRANSLATORS: B05h: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 c6
			variation => N__('Alekhine: Modern, Flohr Variation'),
			moves => {
				'c2c4' => 1,
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/pp2pppp/2pp4/3nP3/2PP2b1/5N2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B05i',
			# TRANSLATORS: B05i: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 c6 6. c4
			variation => N__('Alekhine: Modern, Flohr, 6. c4'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2pp4/3nP3/3P2b1/5N2/PPP1BPPP/RNBQ1RK1 b kq' => {
			eco => 'B05j',
			# TRANSLATORS: B05j: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 c6 6. O-O
			variation => N__('Alekhine: Modern, Flohr, 6.O-O'),
			moves => {
			},
		},
		'rn1qkb1r/ppp2ppp/3pp3/3nP3/3P2b1/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B05k',
			# TRANSLATORS: B05k: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6
			variation => N__('Alekhine: Modern, 5. Be2 e6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'rn1qkb1r/ppp2ppp/3pp3/3nP3/3P2b1/5N2/PPP1BPPP/RNBQ1RK1 b kq' => {
			eco => 'B05l',
			# TRANSLATORS: B05l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O
			variation => N__('Alekhine: Modern, 5. Be2 e6 6.O-O'),
			moves => {
				'f8e7' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/3pp3/3nP3/3P2b1/5N2/PPP1BPPP/RNBQ1RK1 w kq' => {
			eco => 'B05l',
			# TRANSLATORS: B05l: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7
			variation => N__('Alekhine: Modern, 5. Be2 e6 6.O-O'),
			moves => {
				'c2c4' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/3pp3/3nP3/2PP2b1/5N2/PP2BPPP/RNBQ1RK1 b kq' => {
			eco => 'B05m',
			# TRANSLATORS: B05m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4
			variation => N__('Alekhine: Modern, Main Line'),
			moves => {
				'd5b6' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/1n1pp3/4P3/2PP2b1/5N2/PP2BPPP/RNBQ1RK1 w kq' => {
			eco => 'B05m',
			# TRANSLATORS: B05m: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6
			variation => N__('Alekhine: Modern, Main Line'),
			moves => {
				'b1c3' => 1,
				'e5d6' => 1,
				'h2h3' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/1n1pp3/4P3/2PP2b1/5N1P/PP2BPP1/RNBQ1RK1 b kq' => {
			eco => 'B05n',
			# TRANSLATORS: B05n: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3
			variation => N__('Alekhine: Modern, Main Line, 8. h3'),
			moves => {
				'g4h5' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/1n1pp3/4P2b/2PP4/5N1P/PP2BPP1/RNBQ1RK1 w kq' => {
			eco => 'B05n',
			# TRANSLATORS: B05n: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3 Bh5
			variation => N__('Alekhine: Modern, Main Line, 8. h3 Bh5'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rn1qk2r/ppp1bppp/1n1pp3/4P2b/2PP4/2N2N1P/PP2BPP1/R1BQ1RK1 b kq' => {
			eco => 'B05o',
			# TRANSLATORS: B05o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3 Bh5 9. Nc3
			variation => N__('Alekhine: Modern, Main Line, 8. h3 Bh5 9. Nc3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n1pp3/4P2b/2PP4/2N2N1P/PP2BPP1/R1BQ1RK1 w -' => {
			eco => 'B05o',
			# TRANSLATORS: B05o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3 Bh5 9. Nc3 O-O
			variation => N__('Alekhine: Modern, Main Line, 8. h3 Bh5 9. Nc3 O-O'),
			moves => {
				'c1e3' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n1pp3/4P2b/2PP4/2N1BN1P/PP2BPP1/R2Q1RK1 b -' => {
			eco => 'B05o',
			# TRANSLATORS: B05o: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3 Bh5 9. Nc3 O-O 10. Be3
			variation => N__('Alekhine: Modern, Main Line, 8. h3 Bh5 9. Nc3 O-O 10. Be3'),
			moves => {
				'd6d5' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n2p3/3pP2b/2PP4/2N1BN1P/PP2BPP1/R2Q1RK1 w -' => {
			eco => 'B05p',
			# TRANSLATORS: B05p: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. h3 Bh5 9. Nc3 O-O 10. Be3 d5
			variation => N__('Alekhine: Modern, Main Line, 8. h3 Bh5 9. Nc3 O-O 10. Be3 d5'),
			moves => {
			},
		},
		'rn1qk2r/ppp1bppp/1n1Pp3/8/2PP2b1/5N2/PP2BPPP/RNBQ1RK1 b kq' => {
			eco => 'B05q',
			# TRANSLATORS: B05q: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. exd6
			variation => N__('Alekhine: Modern, Main Line, 8.exd6'),
			moves => {
			},
		},
		'rn1qk2r/ppp1bppp/1n1pp3/4P3/2PP2b1/2N2N2/PP2BPPP/R1BQ1RK1 b kq' => {
			eco => 'B05r',
			# TRANSLATORS: B05r: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n1pp3/4P3/2PP2b1/2N2N2/PP2BPPP/R1BQ1RK1 w -' => {
			eco => 'B05r',
			# TRANSLATORS: B05r: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3 O-O
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3'),
			moves => {
				'c1e3' => 1,
			},
		},
		'rn1q1rk1/ppp1bppp/1n1pp3/4P3/2PP2b1/2N1BN2/PP2BPPP/R2Q1RK1 b -' => {
			eco => 'B05s',
			# TRANSLATORS: B05s: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3 O-O 9. Be3
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3 O-O 9. Be3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2q1rk1/ppp1bppp/1nnpp3/4P3/2PP2b1/2N1BN2/PP2BPPP/R2Q1RK1 w -' => {
			eco => 'B05t',
			# TRANSLATORS: B05t: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3 O-O 9. Be3 Nc6
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3 O-O 9. Be3 Nc6'),
			moves => {
				'e5d6' => 1,
			},
		},
		'r2q1rk1/ppp1bppp/1nnPp3/8/2PP2b1/2N1BN2/PP2BPPP/R2Q1RK1 b -' => {
			eco => 'B05t',
			# TRANSLATORS: B05t: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3 O-O 9. Be3 Nc6 10. exd6
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3 O-O 9. Be3 Nc6'),
			moves => {
				'c7d6' => 1,
			},
		},
		'r2q1rk1/pp2bppp/1nnpp3/8/2PP2b1/2N1BN2/PP2BPPP/R2Q1RK1 w -' => {
			eco => 'B05u',
			# TRANSLATORS: B05u: 1. e4 Nf6 2. e5 Nd5 3. d4 d6 4. Nf3 Bg4 5. Be2 e6 6. O-O Be7 7. c4 Nb6 8. Nc3 O-O 9. Be3 Nc6 10. exd6 cxd6
			variation => N__('Alekhine: Modern, Main Line, 8. Nc3 O-O 9. Be3 Nc6 10.exd6 cxd6'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B06a',
			# TRANSLATORS: B06a: 1. e4 g6
			variation => N__('Modern'),
			moves => {
				'b1c3' => 1,
				'd2d4' => 1,
				'f2f4' => 1,
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B06b',
			# TRANSLATORS: B06b: 1. e4 g6 2. Nc3
			variation => N__('Modern'),
			moves => {
				'd7d6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B06b',
			# TRANSLATORS: B06b: 1. e4 g6 2. Nc3 d6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B06b',
			# TRANSLATORS: B06b: 1. e4 g6 2. Nc3 Bg7
			variation => N__('Modern'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B06b',
			# TRANSLATORS: B06b: 1. e4 g6 2. Nc3 Bg7 3. f4
			variation => N__('Modern'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/4PP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. Nc3 Bg7 3. f4 d6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. f4
			variation => N__('Modern'),
			moves => {
				'd7d6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. f4 d6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. f4 Bg7
			variation => N__('Modern'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/4PP2/5N2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. f4 Bg7 3. Nf3
			variation => N__('Modern'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/4PP2/5N2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'B06c',
			# TRANSLATORS: B06c: 1. e4 g6 2. f4 Bg7 3. Nf3 d6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqkbnr/pppppp1p/6p1/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4
			variation => N__('Modern'),
			moves => {
				'c7c6' => 1,
				'd7d6' => 1,
				'f8g7' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/2p3p1/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 c6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqkb1r/pppppp1p/5np1/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 Nf6
			variation => N__('Modern: Norwegian Defence'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/5np1/4P3/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 Nf6 3. e5
			variation => N__('Modern: Norwegian Defence'),
			moves => {
				'f6h5' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/6p1/4P2n/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 Nf6 3. e5 Nh5
			variation => N__('Modern: Norwegian Defence'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rnbqkb1r/pppppp1p/6p1/4P2n/3P2P1/8/PPP2P1P/RNBQKBNR b KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 Nf6 3. e5 Nh5 4. g4
			variation => N__('Modern: Norwegian Defence'),
			moves => {
				'h5g7' => 1,
			},
		},
		'rnbqkb1r/ppppppnp/6p1/4P3/3P2P1/8/PPP2P1P/RNBQKBNR w KQkq' => {
			eco => 'B06d',
			# TRANSLATORS: B06d: 1. e4 g6 2. d4 Nf6 3. e5 Nh5 4. g4 Ng7
			variation => N__('Modern: Norwegian Defence'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pp1p/3p2p1/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B06e',
			# TRANSLATORS: B06e: 1. e4 g6 2. d4 d6
			variation => N__('Modern'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B06f',
			# TRANSLATORS: B06f: 1. e4 g6 2. d4 Bg7
			variation => N__('Modern'),
			moves => {
				'b1c3' => 1,
				'c2c3' => 1,
				'f1c4' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/2BPP3/8/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B06f',
			# TRANSLATORS: B06f: 1. e4 g6 2. d4 Bg7 3. Bc4
			variation => N__('Modern: 3. Bc4'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/3PP3/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B06g',
			# TRANSLATORS: B06g: 1. e4 g6 2. d4 Bg7 3. c3
			variation => N__('Modern: 3. c3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B06g',
			# TRANSLATORS: B06g: 1. e4 g6 2. d4 Bg7 3. c3 d6
			variation => N__('Modern: 3. c3'),
			moves => {
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PPP2/2P5/PP4PP/RNBQKBNR b KQkq' => {
			eco => 'B06g',
			# TRANSLATORS: B06g: 1. e4 g6 2. d4 Bg7 3. c3 d6 4. f4
			variation => N__('Modern: 3. c3'),
			moves => {
				'c7c5' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2p5/3PPP2/2P5/PP4PP/RNBQKBNR w KQkq' => {
			eco => 'B06g',
			# TRANSLATORS: B06g: 1. e4 g6 2. d4 Bg7 3. c3 d6 4. f4 c5
			variation => N__('Modern: c3 Pterodactyl'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B06h',
			# TRANSLATORS: B06h: 1. e4 g6 2. d4 Bg7 3. c3 d6 4. Nf3
			variation => N__('Modern: Geller System'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B06i',
			# TRANSLATORS: B06i: 1. e4 g6 2. d4 Bg7 3. Nf3
			variation => N__('Modern: 3. Nf3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B06j',
			# TRANSLATORS: B06j: 1. e4 g6 2. d4 Bg7 3. Nf3 d6
			variation => N__('Modern: 3. Nf3 d6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2BPP3/5N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B06k',
			# TRANSLATORS: B06k: 1. e4 g6 2. d4 Bg7 3. Nf3 d6 4. Bc4
			variation => N__('Modern: 3. Nf3 d6 4. Bc4'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B06l',
			# TRANSLATORS: B06l: 1. e4 g6 2. d4 Bg7 3. Nc3
			variation => N__('Modern: 3. Nc3'),
			moves => {
				'c7c5' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B06l',
			# TRANSLATORS: B06l: 1. e4 g6 2. d4 Bg7 3. Nc3 c5
			variation => N__('Modern: Nc3 Pterodactyl'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/6p1/3p4/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B06l',
			# TRANSLATORS: B06l: 1. e4 g6 2. d4 Bg7 3. Nc3 d5
			variation => N__('Modern: Mittenberger Gambit'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B06m',
			# TRANSLATORS: B06m: 1. e4 g6 2. d4 Bg7 3. Nc3 d6
			variation => N__('Modern: 3. Nc3'),
			moves => {
				'c1e3' => 1,
				'f1c4' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/2BPP3/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B06n',
			# TRANSLATORS: B06n: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Bc4
			variation => N__('Modern: 3. Nc3 d6 4. Bc4'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/2BPP3/2N5/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B06n',
			# TRANSLATORS: B06n: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Bc4 c6
			variation => N__('Modern: 3. Nc3 d6 4. Bc4'),
			moves => {
				'd1f3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/2BPP3/2N2Q2/PPP2PPP/R1B1K1NR b KQkq' => {
			eco => 'B06n',
			# TRANSLATORS: B06n: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Bc4 c6 5. Qf3
			variation => N__('Modern: 3. Nc3 d6 4. Bc4'),
			moves => {
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqk1nr/pp3pbp/2ppp1p1/8/2BPP3/2N2Q2/PPP2PPP/R1B1K1NR w KQkq' => {
			eco => 'B06n',
			# TRANSLATORS: B06n: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Bc4 c6 5. Qf3 e6
			variation => N__('Modern: 3. Nc3 d6 4. Bc4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/2BPP3/2N2Q2/PPP2PPP/R1B1K1NR w KQkq' => {
			eco => 'B06n',
			# TRANSLATORS: B06n: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Bc4 c6 5. Qf3 Nf6
			variation => N__('Modern: 3. Nc3 d6 4. Bc4'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'B06o',
			# TRANSLATORS: B06o: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Be3
			variation => N__('Modern: 3. Nc3 d6 4. Be3'),
			moves => {
				'a7a6' => 1,
				'c7c6' => 1,
			},
		},
		'rnbqk1nr/1pp1ppbp/p2p2p1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B06p',
			# TRANSLATORS: B06p: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Be3 a6
			variation => N__('Modern: 3. Nc3 d6 4. Be3 a6'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B06q',
			# TRANSLATORS: B06q: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Be3 c6
			variation => N__('Modern: 3. Nc3 d6 4. Be3 c6'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B06r',
			# TRANSLATORS: B06r: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Nf3
			variation => N__('Modern: Two Knights Variation'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B06r',
			# TRANSLATORS: B06r: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Nf3 c6
			variation => N__('Modern: Two Knights, Suttles Variation'),
			moves => {
				'c1g5' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/6B1/3PP3/2N2N2/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B06r',
			# TRANSLATORS: B06r: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Nf3 c6 5. Bg5
			variation => N__('Modern: Two Knights, Suttles Variation'),
			moves => {
				'd8b6' => 1,
			},
		},
		'rnb1k1nr/pp2ppbp/1qpp2p1/6B1/3PP3/2N2N2/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B06r',
			# TRANSLATORS: B06r: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Nf3 c6 5. Bg5 Qb6
			variation => N__('Modern: Two Knights, Suttles Variation'),
			moves => {
				'd1d2' => 1,
			},
		},
		'rnb1k1nr/pp2ppbp/1qpp2p1/6B1/3PP3/2N2N2/PPPQ1PPP/R3KB1R b KQkq' => {
			eco => 'B06r',
			# TRANSLATORS: B06r: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. Nf3 c6 5. Bg5 Qb6 6. Qd2
			variation => N__('Modern: Two Knights, Suttles, Tal Gambit'),
			moves => {
			},
		},
		'rnbqk1nr/ppp1ppbp/3p2p1/8/3PPP2/2N5/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B06s',
			# TRANSLATORS: B06s: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. f4
			variation => N__('Modern: Pseudo-Austrian Attack'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/3PPP2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B06s',
			# TRANSLATORS: B06s: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. f4 c6
			variation => N__('Modern: Pseudo-Austrian Attack'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/ppp1ppbp/2np2p1/8/3PPP2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B06t',
			# TRANSLATORS: B06t: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. f4 Nc6
			variation => N__('Modern: Pseudo-Austrian Attack'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2pp2p1/8/3PPP2/2N2N2/PPP3PP/R1BQKB1R b KQkq' => {
			eco => 'B06t',
			# TRANSLATORS: B06t: 1. e4 g6 2. d4 Bg7 3. Nc3 d6 4. f4 c6 5. Nf3
			variation => N__('Modern: Pseudo-Austrian Attack'),
			moves => {
			},
		},
		'rnbqk1nr/ppppppbp/6p1/8/3PPP2/8/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B06u',
			# TRANSLATORS: B06u: 1. e4 g6 2. d4 Bg7 3. f4
			variation => N__('Modern: Three Pawns Attack'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B07a',
			# TRANSLATORS: B07a: 1. e4 d6
			variation => N__('Pirc'),
			moves => {
				'b1c3' => 1,
				'c2c4' => 1,
				'd2d4' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B07a',
			# TRANSLATORS: B07a: 1. e4 d6 2. Nc3
			variation => N__('Pirc: 2. Nc3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B07a',
			# TRANSLATORS: B07a: 1. e4 d6 2. Nc3 Nf6
			variation => N__('Pirc: 2. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/2P1P3/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B07a',
			# TRANSLATORS: B07a: 1. e4 d6 2. c4
			variation => N__('Pirc'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/4P3/6P1/PPPP1P1P/RNBQKBNR b KQkq' => {
			eco => 'B07a',
			# TRANSLATORS: B07a: 1. e4 d6 2. g3
			variation => N__('Pirc'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B07b',
			# TRANSLATORS: B07b: 1. e4 d6 2. Nf3
			variation => N__('Pirc: 2. Nf3'),
			moves => {
				'c8g4' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/ppp1pppp/3p4/8/4P1b1/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B07b',
			# TRANSLATORS: B07b: 1. e4 d6 2. Nf3 Bg4
			variation => N__('Pirc/Reti: Wade Defence'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B07b',
			# TRANSLATORS: B07b: 1. e4 d6 2. Nf3 Nf6
			variation => N__('Pirc: 2. Nf3 Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/4P3/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B07b',
			# TRANSLATORS: B07b: 1. e4 d6 2. Nf3 Nf6 3. Nc3
			variation => N__('Pirc: 2. Nf3 Nf6 3. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4
			variation => N__('Pirc: 2. f4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4 Nf6
			variation => N__('Pirc: 2. f4 Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4 Nf6 3. Nc3
			variation => N__('Pirc: 2. f4 Nf6 3. Nc3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/4PP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4 Nf6 3. Nc3 g6
			variation => N__('Pirc: 2. f4 Nf6 3. Nc3'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/4PP2/2N2N2/PPPP2PP/R1BQKB1R b KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4 Nf6 3. Nc3 g6 4. Nf3
			variation => N__('Pirc: 2. f4 Nf6 3. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq' => {
			eco => 'B07c',
			# TRANSLATORS: B07c: 1. e4 d6 2. f4 Nf6 3. Nc3 g6 4. Nf3 Bg7
			variation => N__('Pirc: 2. f4 Nf6 3. Nc3 g6 4. Nf3 Bg7'),
			moves => {
			},
		},
		'rnbqkbnr/ppp1pppp/3p4/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4
			variation => N__('Pirc: 2. d4'),
			moves => {
				'b8d7' => 1,
				'c7c6' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pppnpppp/3p4/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nd7
			variation => N__('Pirc: 2. d4 Nd7'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2pp4/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 c6
			variation => N__('Pirc: 2. d4 c6'),
			moves => {
			},
		},
		'rnbqkbnr/ppp2ppp/3pp3/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 e6
			variation => N__('Pirc: Lengfellner System'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6
			variation => N__('Pirc: 2. d4 Nf6'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'f1d3' => 1,
				'f2f3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/8/PPPN1PPP/R1BQKBNR b KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6 3. Nd2
			variation => N__('Pirc: 3.Nbd2'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/8/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6 3. Nd2 g6
			variation => N__('Pirc: 3.Nbd2 g6'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2P5/PP1N1PPP/R1BQKBNR b KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6 3. Nd2 g6 4. c3
			variation => N__('Pirc: 3.Nbd2 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2P5/PP1N1PPP/R1BQKBNR w KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6 3. Nd2 g6 4. c3 Bg7
			variation => N__('Pirc: 3.Nbd2 g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B07d',
			# TRANSLATORS: B07d: 1. e4 d6 2. d4 Nf6 3. Nf3
			variation => N__('Pirc: Roscher Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B07e',
			# TRANSLATORS: B07e: 1. e4 d6 2. d4 Nf6 3. f3
			variation => N__('Pirc: 3. f3'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/3PP3/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B07f',
			# TRANSLATORS: B07f: 1. e4 d6 2. d4 Nf6 3. f3 e5
			variation => N__('Pirc: 3. f3 e5'),
			moves => {
				'd4d5' => 1,
				'd4e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/3Pp3/4P3/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B07f',
			# TRANSLATORS: B07f: 1. e4 d6 2. d4 Nf6 3. f3 e5 4. d5
			variation => N__('Pirc: 3. f3 e5 4. d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4P3/4P3/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B07f',
			# TRANSLATORS: B07f: 1. e4 d6 2. d4 Nf6 3. f3 e5 4. dxe5
			variation => N__('Pirc: 3. f3 e5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/4p3/4P3/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B07f',
			# TRANSLATORS: B07f: 1. e4 d6 2. d4 Nf6 3. f3 e5 4. dxe5 dxe5
			variation => N__('Pirc: 3. f3 e5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'rnbQkb1r/ppp2ppp/5n2/4p3/4P3/5P2/PPP3PP/RNB1KBNR b KQkq' => {
			eco => 'B07f',
			# TRANSLATORS: B07f: 1. e4 d6 2. d4 Nf6 3. f3 e5 4. dxe5 dxe5 5. Qxd8+
			variation => N__('Pirc: 3. f3 e5 Queenswap'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B07g',
			# TRANSLATORS: B07g: 1. e4 d6 2. d4 Nf6 3. f3 g6
			variation => N__('Pirc: 3. f3 g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3
			variation => N__('Pirc: 3. Bd3'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/3PP3/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5
			variation => N__('Pirc: 3. Bd3 e5'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/3PP3/2PB4/PP3PPP/RNBQK1NR b KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5 4. c3
			variation => N__('Pirc: 3. Bd3 e5'),
			moves => {
				'd6d5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/3pp3/3PP3/2PB4/PP3PPP/RNBQK1NR w KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5 4. c3 d5
			variation => N__('Pirc: 3. Bd3 e5 4. c3 d5'),
			moves => {
				'd4e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/3pP3/4P3/2PB4/PP3PPP/RNBQK1NR b KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5 4. c3 d5 5. dxe5
			variation => N__('Pirc: 3. Bd3 e5 4. c3 d5'),
			moves => {
				'd5e4' => 1,
				'f6e4' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/4P3/4p3/2PB4/PP3PPP/RNBQK1NR w KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5 4. c3 d5 5. dxe5 dxe4
			variation => N__('Pirc: 3. Bd3 e5 4. c3 d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/8/3pP3/4n3/2PB4/PP3PPP/RNBQK1NR w KQkq' => {
			eco => 'B07h',
			# TRANSLATORS: B07h: 1. e4 d6 2. d4 Nf6 3. Bd3 e5 4. c3 d5 5. dxe5 Nxe4
			variation => N__('Pirc: 3. Bd3 e5 4. c3 d5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B07i',
			# TRANSLATORS: B07i: 1. e4 d6 2. d4 Nf6 3. Bd3 g6
			variation => N__('Pirc: 3. Bd3 g6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/3B1N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B07i',
			# TRANSLATORS: B07i: 1. e4 d6 2. d4 Nf6 3. Bd3 g6 4. Nf3
			variation => N__('Pirc: 3. Bd3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/3B1N2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B07i',
			# TRANSLATORS: B07i: 1. e4 d6 2. d4 Nf6 3. Bd3 g6 4. Nf3 Bg7
			variation => N__('Pirc: 3. Bd3 g6'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2PB1N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B07j',
			# TRANSLATORS: B07j: 1. e4 d6 2. d4 Nf6 3. Bd3 g6 4. Nf3 Bg7 5. c3
			variation => N__('Pirc: 3. Bd3 g6 4. Nf3 Bg7 5. c3'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pppp/3p1n2/8/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B07k',
			# TRANSLATORS: B07k: 1. e4 d6 2. d4 Nf6 3. Nc3
			variation => N__('Pirc: 3. Nc3'),
			moves => {
				'b8d7' => 1,
				'c7c6' => 1,
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkb1r/pppnpppp/3p1n2/8/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B07l',
			# TRANSLATORS: B07l: 1. e4 d6 2. d4 Nf6 3. Nc3 Nbd7
			variation => N__('Pirc: 3. Nc3 Nbd7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4p3/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B07m',
			# TRANSLATORS: B07m: 1. e4 d6 2. d4 Nf6 3. Nc3 e5
			variation => N__('Pirc: 3. Nc3 e5'),
			moves => {
				'd4e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/3p1n2/4P3/4P3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B07m',
			# TRANSLATORS: B07m: 1. e4 d6 2. d4 Nf6 3. Nc3 e5 4. dxe5
			variation => N__('Pirc: 3. Nc3 e5'),
			moves => {
				'd6e5' => 1,
			},
		},
		'rnbqkb1r/ppp2ppp/5n2/4p3/4P3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B07m',
			# TRANSLATORS: B07m: 1. e4 d6 2. d4 Nf6 3. Nc3 e5 4. dxe5 dxe5
			variation => N__('Pirc: 3. Nc3 e5'),
			moves => {
				'd1d8' => 1,
			},
		},
		'rnbQkb1r/ppp2ppp/5n2/4p3/4P3/2N5/PPP2PPP/R1B1KBNR b KQkq' => {
			eco => 'B07m',
			# TRANSLATORS: B07m: 1. e4 d6 2. d4 Nf6 3. Nc3 e5 4. dxe5 dxe5 5. Qxd8+
			variation => N__('Pirc: 3. Nc3 e5 Queenswap'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2pp1n2/8/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B07n',
			# TRANSLATORS: B07n: 1. e4 d6 2. d4 Nf6 3. Nc3 c6
			variation => N__('Pirc: Pytel/Czech'),
			moves => {
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2pp1n2/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B07o',
			# TRANSLATORS: B07o: 1. e4 d6 2. d4 Nf6 3. Nc3 c6 4. Nf3
			variation => N__('Pirc: Czech, 4. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2pp1n2/8/3PPP2/2N5/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B07p',
			# TRANSLATORS: B07p: 1. e4 d6 2. d4 Nf6 3. Nc3 c6 4. f4
			variation => N__('Pirc: Czech, 4. f4'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/2pp1n2/q7/3PPP2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B07p',
			# TRANSLATORS: B07p: 1. e4 d6 2. d4 Nf6 3. Nc3 c6 4. f4 Qa5
			variation => N__('Pirc: Czech, 4. f4 Qa5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B07q',
			# TRANSLATORS: B07q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6
			variation => N__('Pirc: 3... g6'),
			moves => {
				'c1e3' => 1,
				'c1g5' => 1,
				'f1c4' => 1,
				'f1e2' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
				'h2h4' => 1,
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N5/PPP1NPPP/R1BQKB1R b KQkq' => {
			eco => 'B07q',
			# TRANSLATORS: B07q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nge2
			variation => N__('Pirc: 4.Nge2'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N5/PPP1NPPP/R1BQKB1R w KQkq' => {
			eco => 'B07q',
			# TRANSLATORS: B07q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nge2 Bg7
			variation => N__('Pirc: 4.Nge2'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP2P/2N5/PPP2PP1/R1BQKBNR b KQkq' => {
			eco => 'B07q',
			# TRANSLATORS: B07q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. h4
			variation => N__('Pirc: 3... g6'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N3P1/PPP2P1P/R1BQKBNR b KQkq' => {
			eco => 'B07r',
			# TRANSLATORS: B07r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. g3
			variation => N__('Pirc: Sveshnikov (4. g3)'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N3P1/PPP2P1P/R1BQKBNR w KQkq' => {
			eco => 'B07r',
			# TRANSLATORS: B07r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. g3 Bg7
			variation => N__('Pirc: Sveshnikov (4. g3)'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N3P1/PPP2PBP/R1BQK1NR b KQkq' => {
			eco => 'B07r',
			# TRANSLATORS: B07r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. g3 Bg7 5. Bg2
			variation => N__('Pirc: Sveshnikov (4. g3)'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N3P1/PPP2PBP/R1BQK1NR w KQkq' => {
			eco => 'B07r',
			# TRANSLATORS: B07r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. g3 Bg7 5. Bg2 c6
			variation => N__('Pirc: Sveshnikov (4. g3)'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/2BPP3/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B07s',
			# TRANSLATORS: B07s: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Bc4
			variation => N__('Pirc: Holmov (4. Bc4)'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/2BPP3/2N5/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B07s',
			# TRANSLATORS: B07s: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Bc4 Bg7
			variation => N__('Pirc: Holmov (4. Bc4)'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'B07t',
			# TRANSLATORS: B07t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3
			variation => N__('Pirc: 4. Be3'),
			moves => {
				'c7c6' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/2pp1np1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B07t',
			# TRANSLATORS: B07t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 c6
			variation => N__('Pirc: 4. Be3 c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N1B3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B07u',
			# TRANSLATORS: B07u: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 Bg7
			variation => N__('Pirc: 4. Be3 Bg7'),
			moves => {
				'd1d2' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N1B3/PPPQ1PPP/R3KBNR b KQkq' => {
			eco => 'B07u',
			# TRANSLATORS: B07u: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 Bg7 5. Qd2
			variation => N__('Pirc: 4. Be3, 150 Attack'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N1B3/PPPQ1PPP/R3KBNR w KQkq' => {
			eco => 'B07v',
			# TRANSLATORS: B07v: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 Bg7 5. Qd2 c6
			variation => N__('Pirc: 4. Be3, 150 Attack'),
			moves => {
				'f2f3' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N1BP2/PPPQ2PP/R3KBNR b KQkq' => {
			eco => 'B07v',
			# TRANSLATORS: B07v: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 Bg7 5. Qd2 c6 6. f3
			variation => N__('Pirc: 4. Be3, 150 Attack'),
			moves => {
				'b7b5' => 1,
			},
		},
		'rnbqk2r/p3ppbp/2pp1np1/1p6/3PP3/2N1BP2/PPPQ2PP/R3KBNR w KQkq' => {
			eco => 'B07v',
			# TRANSLATORS: B07v: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be3 Bg7 5. Qd2 c6 6. f3 b5
			variation => N__('Pirc: 4. Be3, 150 Attack'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/6B1/3PP3/2N5/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'B07w',
			# TRANSLATORS: B07w: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Bg5
			variation => N__('Pirc: Byrne 4. Bg5'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/6B1/3PP3/2N5/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B07x',
			# TRANSLATORS: B07x: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Bg5 Bg7
			variation => N__('Pirc: 4. Bg5 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N5/PPP1BPPP/R1BQK1NR b KQkq' => {
			eco => 'B07y',
			# TRANSLATORS: B07y: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be2
			variation => N__('Pirc: 4. Be2'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N5/PPP1BPPP/R1BQK1NR w KQkq' => {
			eco => 'B07y',
			# TRANSLATORS: B07y: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be2 Bg7
			variation => N__('Pirc: 4. Be2'),
			moves => {
				'g2g4' => 1,
				'h2h4' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP1P1/2N5/PPP1BP1P/R1BQK1NR b KQkq' => {
			eco => 'B07y',
			# TRANSLATORS: B07y: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be2 Bg7 5. g4
			variation => N__('Pirc: Chinese Variation'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP2P/2N5/PPP1BPP1/R1BQK1NR b KQkq' => {
			eco => 'B07z',
			# TRANSLATORS: B07z: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Be2 Bg7 5. h4
			variation => N__('Pirc: Bayonet/Mariotti Attack'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B08a',
			# TRANSLATORS: B08a: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3
			variation => N__('Pirc: Classical'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B08a',
			# TRANSLATORS: B08a: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7
			variation => N__('Pirc: Classical'),
			moves => {
				'a2a4' => 1,
				'c1e3' => 1,
				'f1c4' => 1,
				'f1e2' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N2N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B08b',
			# TRANSLATORS: B08b: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3
			variation => N__('Pirc: Classical, 5. h3'),
			moves => {
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N2N1P/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B08b',
			# TRANSLATORS: B08b: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 c6
			variation => N__('Pirc: Classical, 5. h3'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PP3/2N2N1P/PPP2PP1/R1BQKB1R w KQ' => {
			eco => 'B08b',
			# TRANSLATORS: B08b: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 O-O
			variation => N__('Pirc: Classical, 5. h3'),
			moves => {
				'c1e3' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PP3/2N1BN1P/PPP2PP1/R2QKB1R b KQ' => {
			eco => 'B08c',
			# TRANSLATORS: B08c: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 O-O 6. Be3
			variation => N__('Pirc: Classical, Spassky System'),
			moves => {
				'a7a6' => 1,
				'c7c6' => 1,
				'd6d5' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/5np1/3p4/3PP3/2N1BN1P/PPP2PP1/R2QKB1R w KQ' => {
			eco => 'B08c',
			# TRANSLATORS: B08c: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 O-O 6. Be3 d5
			variation => N__('Pirc: Classical, Spassky System, 6... d5'),
			moves => {
			},
		},
		'rnbq1rk1/1pp1ppbp/p2p1np1/8/3PP3/2N1BN1P/PPP2PP1/R2QKB1R w KQ' => {
			eco => 'B08d',
			# TRANSLATORS: B08d: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 O-O 6. Be3 a6
			variation => N__('Pirc: Classical, Spassky System, 6... a6'),
			moves => {
			},
		},
		'rnbq1rk1/pp2ppbp/2pp1np1/8/3PP3/2N1BN1P/PPP2PP1/R2QKB1R w KQ' => {
			eco => 'B08e',
			# TRANSLATORS: B08e: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. h3 O-O 6. Be3 c6
			variation => N__('Pirc: Classical, Spassky System, 6... c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/P2PP3/2N2N2/1PP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B08f',
			# TRANSLATORS: B08f: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. a4
			variation => N__('Pirc: Classical, 5. a4'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/P2PP3/2N2N2/1PP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B08f',
			# TRANSLATORS: B08f: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. a4 c6
			variation => N__('Pirc: Classical, 5. a4 c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/2BPP3/2N2N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B08g',
			# TRANSLATORS: B08g: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bc4
			variation => N__('Pirc: Classical, 5. Bc4'),
			moves => {
				'c7c6' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/2BPP3/2N2N2/PPP2PPP/R1BQK2R w KQkq' => {
			eco => 'B08g',
			# TRANSLATORS: B08g: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Bc4 c6
			variation => N__('Pirc: Classical, 5. Bc4 c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N1BN2/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B08h',
			# TRANSLATORS: B08h: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be3
			variation => N__('Pirc: Classical, 5. Be3'),
			moves => {
				'a7a6' => 1,
				'c7c6' => 1,
			},
		},
		'rnbqk2r/1pp1ppbp/p2p1np1/8/3PP3/2N1BN2/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B08h',
			# TRANSLATORS: B08h: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be3 a6
			variation => N__('Pirc: Classical, 5. Be3'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbqk2r/1pp1ppbp/p2p1np1/8/P2PP3/2N1BN2/1PP2PPP/R2QKB1R b KQkq' => {
			eco => 'B08h',
			# TRANSLATORS: B08h: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be3 a6 6. a4
			variation => N__('Pirc: Classical, 5. Be3 a6 6. a4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N1BN2/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B08i',
			# TRANSLATORS: B08i: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be3 c6
			variation => N__('Pirc: Classical, 5. Be3 c6'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQK2R b KQkq' => {
			eco => 'B08j',
			# TRANSLATORS: B08j: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2
			variation => N__('Pirc: Classical, 5. Be2'),
			moves => {
				'a7a6' => 1,
				'c7c6' => 1,
				'e8g8' => 1,
			},
		},
		'rnbqk2r/1pp1ppbp/p2p1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQK2R w KQkq' => {
			eco => 'B08j',
			# TRANSLATORS: B08j: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 a6
			variation => N__('Pirc: Classical, 5. Be2'),
			moves => {
				'a2a4' => 1,
			},
		},
		'rnbqk2r/1pp1ppbp/p2p1np1/8/P2PP3/2N2N2/1PP1BPPP/R1BQK2R b KQkq' => {
			eco => 'B08j',
			# TRANSLATORS: B08j: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 a6 6. a4
			variation => N__('Pirc: Classical, 5. Be2 a6 6. a4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/2pp1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQK2R w KQkq' => {
			eco => 'B08k',
			# TRANSLATORS: B08k: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 c6
			variation => N__('Pirc: Classical, 5. Be2 c6'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQK2R w KQ' => {
			eco => 'B08l',
			# TRANSLATORS: B08l: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O
			variation => N__('Pirc: Classical, 5. Be2 O-O'),
			moves => {
				'c1e3' => 1,
				'e1g1' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PP3/2N1BN2/PPP1BPPP/R2QK2R b KQ' => {
			eco => 'B08m',
			# TRANSLATORS: B08m: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. Be3
			variation => N__('Pirc: Classical, 5. Be2 O-O 6. Be3'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQ1RK1 b -' => {
			eco => 'B08n',
			# TRANSLATORS: B08n: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O'),
			moves => {
				'b8c6' => 1,
				'c7c6' => 1,
				'c8g4' => 1,
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQ1RK1 w -' => {
			eco => 'B08o',
			# TRANSLATORS: B08o: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Nc6
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Nc6'),
			moves => {
			},
		},
		'rnbq1rk1/pp2ppbp/2pp1np1/8/3PP3/2N2N2/PPP1BPPP/R1BQ1RK1 w -' => {
			eco => 'B08p',
			# TRANSLATORS: B08p: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O c6
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O c6'),
			moves => {
				'a2a4' => 1,
				'h2h3' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/2pp1np1/8/3PP3/2N2N1P/PPP1BPP1/R1BQ1RK1 b -' => {
			eco => 'B08q',
			# TRANSLATORS: B08q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O c6 7. h3
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O c6 7. h3'),
			moves => {
			},
		},
		'rnbq1rk1/pp2ppbp/2pp1np1/8/P2PP3/2N2N2/1PP1BPPP/R1BQ1RK1 b -' => {
			eco => 'B08r',
			# TRANSLATORS: B08r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O c6 7. a4
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O c6 7. a4'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bq1rk1/pp1nppbp/2pp1np1/8/P2PP3/2N2N2/1PP1BPPP/R1BQ1RK1 w -' => {
			eco => 'B08s',
			# TRANSLATORS: B08s: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O c6 7. a4 Nbd7
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O c6 7. a4 Nbd7'),
			moves => {
			},
		},
		'rn1q1rk1/ppp1ppbp/3p1np1/8/3PP1b1/2N2N2/PPP1BPPP/R1BQ1RK1 w -' => {
			eco => 'B08t',
			# TRANSLATORS: B08t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Bg4
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Bg4'),
			moves => {
				'c1e3' => 1,
			},
		},
		'rn1q1rk1/ppp1ppbp/3p1np1/8/3PP1b1/2N1BN2/PPP1BPPP/R2Q1RK1 b -' => {
			eco => 'B08t',
			# TRANSLATORS: B08t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Bg4 7. Be3
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Bg4'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2q1rk1/ppp1ppbp/2np1np1/8/3PP1b1/2N1BN2/PPP1BPPP/R2Q1RK1 w -' => {
			eco => 'B08t',
			# TRANSLATORS: B08t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Bg4 7. Be3 Nc6
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Bg4'),
			moves => {
				'd1d2' => 1,
			},
		},
		'r2q1rk1/ppp1ppbp/2np1np1/8/3PP1b1/2N1BN2/PPPQBPPP/R4RK1 b -' => {
			eco => 'B08u',
			# TRANSLATORS: B08u: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Bg4 7. Be3 Nc6 8. Qd2
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Bg4 7. Be3 Nc6 8. Qd2'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r2q1rk1/ppp2pbp/2np1np1/4p3/3PP1b1/2N1BN2/PPPQBPPP/R4RK1 w -' => {
			eco => 'B08v',
			# TRANSLATORS: B08v: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. Nf3 Bg7 5. Be2 O-O 6. O-O Bg4 7. Be3 Nc6 8. Qd2 e5
			variation => N__('Pirc: Classical, 5. Be2 O-O 6.O-O Bg4 7. Be3 Nc6 8. Qd2 e5'),
			moves => {
			},
		},
		'rnbqkb1r/ppp1pp1p/3p1np1/8/3PPP2/2N5/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B09a',
			# TRANSLATORS: B09a: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4
			variation => N__('Pirc: Austrian Attack'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PPP2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B09b',
			# TRANSLATORS: B09b: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7
			variation => N__('Pirc: Austrian Attack'),
			moves => {
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/2BPPP2/2N5/PPP3PP/R1BQK1NR b KQkq' => {
			eco => 'B09c',
			# TRANSLATORS: B09c: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Bc4
			variation => N__('Pirc: Austrian, Ljubojevic Variation'),
			moves => {
			},
		},
		'rnbqk2r/ppp1ppbp/3p1np1/8/3PPP2/2N2N2/PPP3PP/R1BQKB1R b KQkq' => {
			eco => 'B09d',
			# TRANSLATORS: B09d: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3
			variation => N__('Pirc: Austrian, 5. Nf3'),
			moves => {
				'c7c5' => 1,
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PPP2/2N2N2/PPP3PP/R1BQKB1R w KQ' => {
			eco => 'B09e',
			# TRANSLATORS: B09e: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O
			variation => N__('Pirc: Austrian, 5...O-O'),
			moves => {
				'c1e3' => 1,
				'e4e5' => 1,
				'f1d3' => 1,
				'f1e2' => 1,
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/4P3/3P1P2/2N2N2/PPP3PP/R1BQKB1R b KQ' => {
			eco => 'B09f',
			# TRANSLATORS: B09f: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. e5
			variation => N__('Pirc: Austrian, 5...O-O 6. e5'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PPP2/2N1BN2/PPP3PP/R2QKB1R b KQ' => {
			eco => 'B09g',
			# TRANSLATORS: B09g: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Be3
			variation => N__('Pirc: Austrian, 5...O-O 6. Be3'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PPP2/2N2N2/PPP1B1PP/R1BQK2R b KQ' => {
			eco => 'B09h',
			# TRANSLATORS: B09h: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Be2
			variation => N__('Pirc: Austrian, 5...O-O 6. Be2'),
			moves => {
			},
		},
		'rnbq1rk1/ppp1ppbp/3p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQK2R b KQ' => {
			eco => 'B09i',
			# TRANSLATORS: B09i: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3'),
			moves => {
				'b8a6' => 1,
				'b8c6' => 1,
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQK2R w KQ' => {
			eco => 'B09j',
			# TRANSLATORS: B09j: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Nc6
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Nc6'),
			moves => {
				'e4e5' => 1,
			},
		},
		'r1bq1rk1/ppp1ppbp/2np1np1/4P3/3P1P2/2NB1N2/PPP3PP/R1BQK2R b KQ' => {
			eco => 'B09k',
			# TRANSLATORS: B09k: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Nc6 7. e5
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Nc6 7. e5'),
			moves => {
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQK2R w KQ' => {
			eco => 'B09l',
			# TRANSLATORS: B09l: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/ppp1ppbp/n2p1np1/8/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 b -' => {
			eco => 'B09l',
			# TRANSLATORS: B09l: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6'),
			moves => {
				'c7c5' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2p5/3PPP2/2NB1N2/PPP3PP/R1BQ1RK1 w -' => {
			eco => 'B09m',
			# TRANSLATORS: B09m: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O c5
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6 7.O-O c5'),
			moves => {
				'd4d5' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/n2p1np1/2pP4/4PP2/2NB1N2/PPP3PP/R1BQ1RK1 b -' => {
			eco => 'B09m',
			# TRANSLATORS: B09m: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O c5 8. d5
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6 7.O-O c5 8. d5'),
			moves => {
				'c8g4' => 1,
			},
		},
		'r2q1rk1/pp2ppbp/n2p1np1/2pP4/4PPb1/2NB1N2/PPP3PP/R1BQ1RK1 w -' => {
			eco => 'B09n',
			# TRANSLATORS: B09n: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 O-O 6. Bd3 Na6 7. O-O c5 8. d5 Bg4
			variation => N__('Pirc: Austrian, 5...O-O 6. Bd3 Na6 7.O-O c5 8. d5 Bg4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2p5/3PPP2/2N2N2/PPP3PP/R1BQKB1R w KQkq' => {
			eco => 'B09o',
			# TRANSLATORS: B09o: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5
			variation => N__('Pirc: Austrian, 5... c5'),
			moves => {
				'd4c5' => 1,
				'f1b5' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/2P5/4PP2/2N2N2/PPP3PP/R1BQKB1R b KQkq' => {
			eco => 'B09p',
			# TRANSLATORS: B09p: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. dxc5
			variation => N__('Pirc: Austrian, 5... c5 6.dxc5'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1k2r/pp2ppbp/3p1np1/q1P5/4PP2/2N2N2/PPP3PP/R1BQKB1R w KQkq' => {
			eco => 'B09p',
			# TRANSLATORS: B09p: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. dxc5 Qa5
			variation => N__('Pirc: Austrian, 5... c5 6.dxc5'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnb1k2r/pp2ppbp/3p1np1/q1P5/4PP2/2NB1N2/PPP3PP/R1BQK2R b KQkq' => {
			eco => 'B09q',
			# TRANSLATORS: B09q: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. dxc5 Qa5 7. Bd3
			variation => N__('Pirc: Austrian, 5... c5 6.dxc5 Qa5 7. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/3p1np1/1Bp5/3PPP2/2N2N2/PPP3PP/R1BQK2R b KQkq' => {
			eco => 'B09r',
			# TRANSLATORS: B09r: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+'),
			moves => {
				'c8d7' => 1,
			},
		},
		'rn1qk2r/pp1bppbp/3p1np1/1Bp5/3PPP2/2N2N2/PPP3PP/R1BQK2R w KQkq' => {
			eco => 'B09s',
			# TRANSLATORS: B09s: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rn1qk2r/pp1bppbp/3p1np1/1Bp1P3/3P1P2/2N2N2/PPP3PP/R1BQK2R b KQkq' => {
			eco => 'B09t',
			# TRANSLATORS: B09t: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+ Bd7 7. e5'),
			moves => {
				'f6g4' => 1,
			},
		},
		'rn1qk2r/pp1bppbp/3p2p1/1Bp1P3/3P1Pn1/2N2N2/PPP3PP/R1BQK2R w KQkq' => {
			eco => 'B09u',
			# TRANSLATORS: B09u: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5 Ng4
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+ Bd7 7. e5 Ng4'),
			moves => {
				'b5d7' => 1,
				'e5e6' => 1,
			},
		},
		'rn1qk2r/pp1Bppbp/3p2p1/2p1P3/3P1Pn1/2N2N2/PPP3PP/R1BQK2R b KQkq' => {
			eco => 'B09v',
			# TRANSLATORS: B09v: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5 Ng4 8. Bxd7+
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+ Bd7 7. e5 Ng4 8.Bxd7+'),
			moves => {
			},
		},
		'rn1qk2r/pp1bppbp/3pP1p1/1Bp5/3P1Pn1/2N2N2/PPP3PP/R1BQK2R b KQkq' => {
			eco => 'B09w',
			# TRANSLATORS: B09w: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5 Ng4 8. e6
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+ Bd7 7. e5 Ng4 8. e6'),
			moves => {
				'd7b5' => 1,
				'f7e6' => 1,
			},
		},
		'rn1qk2r/pp1bp1bp/3pp1p1/1Bp5/3P1Pn1/2N2N2/PPP3PP/R1BQK2R w KQkq' => {
			eco => 'B09w',
			# TRANSLATORS: B09w: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5 Ng4 8. e6 fxe6
			variation => N__('Pirc: Austrian, 5... c5, Seirawan Variation'),
			moves => {
			},
		},
		'rn1qk2r/pp2ppbp/3pP1p1/1bp5/3P1Pn1/2N2N2/PPP3PP/R1BQK2R w KQkq' => {
			eco => 'B09x',
			# TRANSLATORS: B09x: 1. e4 d6 2. d4 Nf6 3. Nc3 g6 4. f4 Bg7 5. Nf3 c5 6. Bb5+ Bd7 7. e5 Ng4 8. e6 Bxb5
			variation => N__('Pirc: Austrian, 5... c5 6. Bb5+ Bd7 7. e5 Ng4 8. e6 Bxb5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10a',
			# TRANSLATORS: B10a: 1. e4 c6
			variation => N__('Caro-Kann'),
			moves => {
				'b1c3' => 1,
				'c2c4' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B10a',
			# TRANSLATORS: B10a: 1. e4 c6 2. f4
			variation => N__('Caro-Kann: 2. f4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4P3/8/PPPPNPPP/RNBQKB1R b KQkq' => {
			eco => 'B10a',
			# TRANSLATORS: B10a: 1. e4 c6 2. Ne2
			variation => N__('Caro-Kann: 2. Ne2'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4P3/3P4/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B10b',
			# TRANSLATORS: B10b: 1. e4 c6 2. d3
			variation => N__('Caro-Kann: Breyer Variation'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/4P3/3P4/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B10b',
			# TRANSLATORS: B10b: 1. e4 c6 2. d3 d5
			variation => N__('Caro-Kann: Breyer Variation'),
			moves => {
				'b1d2' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/4P3/3P4/PPPN1PPP/R1BQKBNR b KQkq' => {
			eco => 'B10c',
			# TRANSLATORS: B10c: 1. e4 c6 2. d3 d5 3. Nd2
			variation => N__('Caro-Kann: Breyer, 3. Nd2'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/4P3/3P4/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B10d',
			# TRANSLATORS: B10d: 1. e4 c6 2. d3 d5 3. Nd2 g6
			variation => N__('Caro-Kann: Breyer, 3. Nd2 g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp3ppp/2p5/3pp3/4P3/3P4/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B10e',
			# TRANSLATORS: B10e: 1. e4 c6 2. d3 d5 3. Nd2 e5
			variation => N__('Caro-Kann: Breyer, 3. Nd2 e5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p5/3pp3/4P3/3P1N2/PPPN1PPP/R1BQKB1R b KQkq' => {
			eco => 'B10e',
			# TRANSLATORS: B10e: 1. e4 c6 2. d3 d5 3. Nd2 e5 4. Ngf3
			variation => N__('Caro-Kann: Breyer, 3. Nd2 e5'),
			moves => {
				'f8d6' => 1,
			},
		},
		'rnbqk1nr/pp3ppp/2pb4/3pp3/4P3/3P1N2/PPPN1PPP/R1BQKB1R w KQkq' => {
			eco => 'B10e',
			# TRANSLATORS: B10e: 1. e4 c6 2. d3 d5 3. Nd2 e5 4. Ngf3 Bd6
			variation => N__('Caro-Kann: Breyer, 3. Nd2 e5'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqk1nr/pp3ppp/2pb4/3pp3/4P3/3P1NP1/PPPN1P1P/R1BQKB1R b KQkq' => {
			eco => 'B10f',
			# TRANSLATORS: B10f: 1. e4 c6 2. d3 d5 3. Nd2 e5 4. Ngf3 Bd6 5. g3
			variation => N__('Caro-Kann: Breyer, Main Line'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/2P1P3/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B10g',
			# TRANSLATORS: B10g: 1. e4 c6 2. c4
			variation => N__('Caro-Kann: English Variation'),
			moves => {
				'd7d5' => 1,
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/2p5/4p3/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10h',
			# TRANSLATORS: B10h: 1. e4 c6 2. c4 e5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10i',
			# TRANSLATORS: B10i: 1. e4 c6 2. c4 d5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
				'c4d5' => 1,
				'e4d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3P4/4P3/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B10i',
			# TRANSLATORS: B10i: 1. e4 c6 2. c4 d5 3. cxd5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
				'c6d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/4P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10i',
			# TRANSLATORS: B10i: 1. e4 c6 2. c4 d5 3. cxd5 cxd5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3P4/2P5/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B10i',
			# TRANSLATORS: B10i: 1. e4 c6 2. c4 d5 3. exd5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
				'c6d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/2P5/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10i',
			# TRANSLATORS: B10i: 1. e4 c6 2. c4 d5 3. exd5 cxd5
			variation => N__('Caro-Kann: English Variation'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/3P4/8/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B10j',
			# TRANSLATORS: B10j: 1. e4 c6 2. c4 d5 3. exd5 cxd5 4. cxd5
			variation => N__('Caro-Kann: English, Exchange'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/3P4/8/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B10k',
			# TRANSLATORS: B10k: 1. e4 c6 2. c4 d5 3. exd5 cxd5 4. cxd5 Nf6
			variation => N__('Caro-Kann: English, Exchange, 4... Nf6'),
			moves => {
				'b1c3' => 1,
				'f1b5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/1B1P4/8/8/PP1P1PPP/RNBQK1NR b KQkq' => {
			eco => 'B10l',
			# TRANSLATORS: B10l: 1. e4 c6 2. c4 d5 3. exd5 cxd5 4. cxd5 Nf6 5. Bb5+
			variation => N__('Caro-Kann: English, Exchange, 4... Nf6 5. Bb5+'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/3P4/8/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'B10m',
			# TRANSLATORS: B10m: 1. e4 c6 2. c4 d5 3. exd5 cxd5 4. cxd5 Nf6 5. Nc3
			variation => N__('Caro-Kann: English, Exchange, 4... Nf6 5. Nc3'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/8/3n4/8/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'B10n',
			# TRANSLATORS: B10n: 1. e4 c6 2. c4 d5 3. exd5 cxd5 4. cxd5 Nf6 5. Nc3 Nxd5
			variation => N__('Caro-Kann: English, Exchange, 4... Nf6 5. Nc3 Nxd5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B10o',
			# TRANSLATORS: B10o: 1. e4 c6 2. Nf3
			variation => N__('Caro-Kann: 2. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B10p',
			# TRANSLATORS: B10p: 1. e4 c6 2. Nc3
			variation => N__('Caro-Kann: 2. Nc3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B10q',
			# TRANSLATORS: B10q: 1. e4 c6 2. Nc3 d5
			variation => N__('Caro-Kann: 2. Nc3 d5'),
			moves => {
				'd1f3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/4P3/2N2Q2/PPPP1PPP/R1B1KBNR b KQkq' => {
			eco => 'B10r',
			# TRANSLATORS: B10r: 1. e4 c6 2. Nc3 d5 3. Qf3
			variation => N__('Caro-Kann: Goldman/Spielmann'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/4P3/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B10s',
			# TRANSLATORS: B10s: 1. e4 c6 2. Nc3 d5 3. Nf3
			variation => N__('Caro-Kann: Two Knights Variation'),
			moves => {
				'c8g4' => 1,
				'd5e4' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/4p3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B10t',
			# TRANSLATORS: B10t: 1. e4 c6 2. Nc3 d5 3. Nf3 dxe4
			variation => N__('Caro-Kann: Two Knights, 3...dxe4'),
			moves => {
				'c3e4' => 1,
				'f3g5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/6N1/4p3/2N5/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B10t',
			# TRANSLATORS: B10t: 1. e4 c6 2. Nc3 d5 3. Nf3 dxe4 4. Ng5
			variation => N__('Caro-Kann: Hector Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/4N3/5N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B10u',
			# TRANSLATORS: B10u: 1. e4 c6 2. Nc3 d5 3. Nf3 dxe4 4. Nxe4
			variation => N__('Caro-Kann: Two Knights, 3...dxe4 4.Nxe4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/8/4N3/5N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B10v',
			# TRANSLATORS: B10v: 1. e4 c6 2. Nc3 d5 3. Nf3 dxe4 4. Nxe4 Nf6
			variation => N__('Caro-Kann: Two Knights, 3...dxe4 4.Nxe4 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/4P1b1/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B11a',
			# TRANSLATORS: B11a: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4
			variation => N__('Caro-Kann: Two Knights, 3... Bg4'),
			moves => {
				'h2h3' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/4P1b1/2N2N1P/PPPP1PP1/R1BQKB1R b KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3'),
			moves => {
				'g4f3' => 1,
				'g4h5' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p3b/4P3/2N2N1P/PPPP1PP1/R1BQKB1R w KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'e4d5' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3P3b/8/2N2N1P/PPPP1PP1/R1BQKB1R b KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'c6d5' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/8/3p3b/8/2N2N1P/PPPP1PP1/R1BQKB1R w KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5 cxd5
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'f1b5' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/8/1B1p3b/8/2N2N1P/PPPP1PP1/R1BQK2R b KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5 cxd5 6. Bb5+
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r2qkbnr/pp2pppp/2n5/1B1p3b/8/2N2N1P/PPPP1PP1/R1BQK2R w KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5 cxd5 6. Bb5+ Nc6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'g2g4' => 1,
			},
		},
		'r2qkbnr/pp2pppp/2n5/1B1p3b/6P1/2N2N1P/PPPP1P2/R1BQK2R b KQkq' => {
			eco => 'B11b',
			# TRANSLATORS: B11b: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5 cxd5 6. Bb5+ Nc6 7. g4
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5'),
			moves => {
				'h5g6' => 1,
			},
		},
		'r2qkbnr/pp2pppp/2n3b1/1B1p4/6P1/2N2N1P/PPPP1P2/R1BQK2R w KQkq' => {
			eco => 'B11c',
			# TRANSLATORS: B11c: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bh5 5. exd5 cxd5 6. Bb5+ Nc6 7. g4 Bg6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bh5, 7. g4 Bg6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/4P3/2N2b1P/PPPP1PP1/R1BQKB1R w KQkq' => {
			eco => 'B11d',
			# TRANSLATORS: B11d: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3'),
			moves => {
				'd1f3' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3p4/4P3/2N2Q1P/PPPP1PP1/R1B1KB1R b KQkq' => {
			eco => 'B11d',
			# TRANSLATORS: B11d: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3'),
			moves => {
				'd5e4' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/8/4p3/2N2Q1P/PPPP1PP1/R1B1KB1R w KQkq' => {
			eco => 'B11e',
			# TRANSLATORS: B11e: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 dxe4
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 dxe4'),
			moves => {
				'c3e4' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/8/4N3/5Q1P/PPPP1PP1/R1B1KB1R b KQkq' => {
			eco => 'B11e',
			# TRANSLATORS: B11e: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 dxe4 6. Nxe4
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 dxe4'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pppp/2p2n2/3p4/4P3/2N2Q1P/PPPP1PP1/R1B1KB1R w KQkq' => {
			eco => 'B11f',
			# TRANSLATORS: B11f: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 Nf6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3p4/4P3/2N2Q1P/PPPP1PP1/R1B1KB1R w KQkq' => {
			eco => 'B11g',
			# TRANSLATORS: B11g: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6'),
			moves => {
				'd2d3' => 1,
				'd2d4' => 1,
				'g2g3' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3p4/3PP3/2N2Q1P/PPP2PP1/R1B1KB1R b KQkq' => {
			eco => 'B11h',
			# TRANSLATORS: B11h: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d4
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d4'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3p4/4P3/2N2QPP/PPPP1P2/R1B1KB1R b KQkq' => {
			eco => 'B11i',
			# TRANSLATORS: B11i: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. g3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. g3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/4P3/2N2QPP/PPPP1P2/R1B1KB1R w KQkq' => {
			eco => 'B11i',
			# TRANSLATORS: B11i: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. g3 Nf6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. g3 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3p4/4P3/2NP1Q1P/PPP2PP1/R1B1KB1R b KQkq' => {
			eco => 'B11j',
			# TRANSLATORS: B11j: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d3'),
			moves => {
				'b8d7' => 1,
				'g8f6' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/2p1p3/3p4/4P3/2NP1Q1P/PPP2PP1/R1B1KB1R w KQkq' => {
			eco => 'B11k',
			# TRANSLATORS: B11k: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d3 Nd7
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d3 Nd7'),
			moves => {
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/4P3/2NP1Q1P/PPP2PP1/R1B1KB1R w KQkq' => {
			eco => 'B11l',
			# TRANSLATORS: B11l: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d3 Nf6
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d3 Nf6'),
			moves => {
				'a2a3' => 1,
				'c1d2' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/4P3/P1NP1Q1P/1PP2PP1/R1B1KB1R b KQkq' => {
			eco => 'B11m',
			# TRANSLATORS: B11m: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d3 Nf6 7. a3
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d3 Nf6 7. a3'),
			moves => {
			},
		},
		'rn1qkb1r/pp3ppp/2p1pn2/3p4/4P3/2NP1Q1P/PPPB1PP1/R3KB1R b KQkq' => {
			eco => 'B11n',
			# TRANSLATORS: B11n: 1. e4 c6 2. Nc3 d5 3. Nf3 Bg4 4. h3 Bxf3 5. Qxf3 e6 6. d3 Nf6 7. Bd2
			variation => N__('Caro-Kann: Two Knights, 3... Bg4 4. h3 Bxf3 5.Qxf3 e6 6. d3 Nf6 7. Bd2'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/2p5/8/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B12a',
			# TRANSLATORS: B12a: 1. e4 c6 2. d4
			variation => N__('Caro-Kann: 2. d4'),
			moves => {
				'b8a6' => 1,
				'd7d5' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/n1p5/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12a',
			# TRANSLATORS: B12a: 1. e4 c6 2. d4 Na6
			variation => N__('Caro-Kann: de Bruycker Defence'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/2p2n2/8/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12a',
			# TRANSLATORS: B12a: 1. e4 c6 2. d4 Nf6
			variation => N__('Caro-Kann: Masi Defence'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12b',
			# TRANSLATORS: B12b: 1. e4 c6 2. d4 d5
			variation => N__('Caro-Kann: 2. d4 d5'),
			moves => {
				'b1c3' => 1,
				'b1d2' => 1,
				'c1e3' => 1,
				'e4d5' => 1,
				'e4e5' => 1,
				'f2f3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B12b',
			# TRANSLATORS: B12b: 1. e4 c6 2. d4 d5 3. Nf3
			variation => N__('Caro-Kann: Ulysses Gambit'),
			moves => {
				'd5e4' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3Pp3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B12b',
			# TRANSLATORS: B12b: 1. e4 c6 2. d4 d5 3. Nf3 dxe4
			variation => N__('Caro-Kann: Ulysses Gambit'),
			moves => {
				'f3g5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/6N1/3Pp3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B12b',
			# TRANSLATORS: B12b: 1. e4 c6 2. d4 d5 3. Nf3 dxe4 4. Ng5
			variation => N__('Caro-Kann: Ulysses Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/4B3/PPP2PPP/RN1QKBNR b KQkq' => {
			eco => 'B12c',
			# TRANSLATORS: B12c: 1. e4 c6 2. d4 d5 3. Be3
			variation => N__('Caro-Kann: Mieses Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/5P2/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B12d',
			# TRANSLATORS: B12d: 1. e4 c6 2. d4 d5 3. f3
			variation => N__('Caro-Kann: Maroczy (Fantasy) Variation'),
			moves => {
				'd5e4' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p1p3/3p4/3PP3/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B12e',
			# TRANSLATORS: B12e: 1. e4 c6 2. d4 d5 3. f3 e6
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3... e6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3Pp3/5P2/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3...dxe4'),
			moves => {
				'f3e4' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3PP3/8/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4 4. fxe4
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3...dxe4'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p5/4p3/3PP3/8/PPP3PP/RNBQKBNR w KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4 4. fxe4 e5
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3...dxe4'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p5/4p3/3PP3/5N2/PPP3PP/RNBQKB1R b KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4 4. fxe4 e5 5. Nf3
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3...dxe4'),
			moves => {
				'e5d4' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p5/8/3pP3/5N2/PPP3PP/RNBQKB1R w KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4 4. fxe4 e5 5. Nf3 exd4
			variation => N__('Caro-Kann: Maroczy (Fantasy), 3...dxe4'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/2p5/8/2BpP3/5N2/PPP3PP/RNBQK2R b KQkq' => {
			eco => 'B12f',
			# TRANSLATORS: B12f: 1. e4 c6 2. d4 d5 3. f3 dxe4 4. fxe4 e5 5. Nf3 exd4 6. Bc4
			variation => N__('Caro-Kann: Maroczy (Fantasy), Gambit line'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/8/PPPN1PPP/R1BQKBNR b KQkq' => {
			eco => 'B12g',
			# TRANSLATORS: B12g: 1. e4 c6 2. d4 d5 3. Nd2
			variation => N__('Caro-Kann: 3. Nd2'),
			moves => {
				'd5e4' => 1,
				'd8b6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3Pp3/8/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B12g',
			# TRANSLATORS: B12g: 1. e4 c6 2. d4 d5 3. Nd2 dxe4
			variation => N__('Caro-Kann: 3. Nd2 dxe4'),
			moves => {
			},
		},
		'rnb1kbnr/pp2pppp/1qp5/3p4/3PP3/8/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B12g',
			# TRANSLATORS: B12g: 1. e4 c6 2. d4 d5 3. Nd2 Qb6
			variation => N__('Caro-Kann: Edinburgh Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/8/PPPN1PPP/R1BQKBNR w KQkq' => {
			eco => 'B12h',
			# TRANSLATORS: B12h: 1. e4 c6 2. d4 d5 3. Nd2 g6
			variation => N__('Caro-Kann: Gurgenidze/Modern: 3. Nd2 g6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/5N2/PPPN1PPP/R1BQKB1R b KQkq' => {
			eco => 'B12h',
			# TRANSLATORS: B12h: 1. e4 c6 2. d4 d5 3. Nd2 g6 4. Ngf3
			variation => N__('Caro-Kann: Gurgenidze/Modern: 3. Nd2 g6 4.Ngf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/5N2/PPPN1PPP/R1BQKB1R w KQkq' => {
			eco => 'B12h',
			# TRANSLATORS: B12h: 1. e4 c6 2. d4 d5 3. Nd2 g6 4. Ngf3 Bg7
			variation => N__('Caro-Kann: Gurgenidze/Modern: 3. Nd2 g6 4.Ngf3 Bg7'),
			moves => {
				'c2c3' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/2P2N2/PP1N1PPP/R1BQKB1R b KQkq' => {
			eco => 'B12h',
			# TRANSLATORS: B12h: 1. e4 c6 2. d4 d5 3. Nd2 g6 4. Ngf3 Bg7 5. c3
			variation => N__('Caro-Kann: Gurgenidze/Modern: 3. Nd2 g6 4.Ngf3 Bg7 5. c3'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/5N1P/PPPN1PP1/R1BQKB1R b KQkq' => {
			eco => 'B12h',
			# TRANSLATORS: B12h: 1. e4 c6 2. d4 d5 3. Nd2 g6 4. Ngf3 Bg7 5. h3
			variation => N__('Caro-Kann: Gurgenidze/Modern: 3. Nd2 g6 4.Ngf3 Bg7 5. h3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3pP3/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B12i',
			# TRANSLATORS: B12i: 1. e4 c6 2. d4 d5 3. e5
			variation => N__('Caro-Kann: Advance Variation'),
			moves => {
				'c6c5' => 1,
				'c8f5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2ppP3/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12j',
			# TRANSLATORS: B12j: 1. e4 c6 2. d4 d5 3. e5 c5
			variation => N__('Caro-Kann: Advance, 3... c5'),
			moves => {
				'd4c5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2PpP3/8/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B12j',
			# TRANSLATORS: B12j: 1. e4 c6 2. d4 d5 3. e5 c5 4. dxc5
			variation => N__('Caro-Kann: Advance, 3... c5 4.dxc5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/2PpP3/8/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12k',
			# TRANSLATORS: B12k: 1. e4 c6 2. d4 d5 3. e5 c5 4. dxc5 Nc6
			variation => N__('Caro-Kann: Advance, 3... c5 4.dxc5 Nc6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B12l',
			# TRANSLATORS: B12l: 1. e4 c6 2. d4 d5 3. e5 Bf5
			variation => N__('Caro-Kann: Advance, 3... Bf5'),
			moves => {
				'b1c3' => 1,
				'c2c3' => 1,
				'f1d3' => 1,
				'g1f3' => 1,
				'g2g4' => 1,
				'h2h4' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P2P1/8/PPP2P1P/RNBQKBNR b KQkq' => {
			eco => 'B12l',
			# TRANSLATORS: B12l: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. g4
			variation => N__('Caro-Kann: Advance, Bayonet Variation'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P3P/8/PPP2PP1/RNBQKBNR b KQkq' => {
			eco => 'B12m',
			# TRANSLATORS: B12m: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. h4
			variation => N__('Caro-Kann: Advance, 4. h4'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P4/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B12n',
			# TRANSLATORS: B12n: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. c3
			variation => N__('Caro-Kann: Advance, 4. c3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B12n',
			# TRANSLATORS: B12n: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. c3 e6
			variation => N__('Caro-Kann: Advance, 4. c3'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P4/2P5/PP2BPPP/RNBQK1NR b KQkq' => {
			eco => 'B12n',
			# TRANSLATORS: B12n: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. c3 e6 5. Be2
			variation => N__('Caro-Kann: Advance, Short Variation'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P4/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B12o',
			# TRANSLATORS: B12o: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Bd3
			variation => N__('Caro-Kann: Advance, 4. Bd3'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B12p',
			# TRANSLATORS: B12p: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3
			variation => N__('Caro-Kann: Advance, 4. Nf3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B12p',
			# TRANSLATORS: B12p: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P4/5N2/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B12q',
			# TRANSLATORS: B12q: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2'),
			moves => {
				'b8d7' => 1,
				'c6c5' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/2p1p3/3pPb2/3P4/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B12r',
			# TRANSLATORS: B12r: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 Nd7
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 Nd7'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/4p3/2ppPb2/3P4/5N2/PPP1BPPP/RNBQK2R w KQkq' => {
			eco => 'B12s',
			# TRANSLATORS: B12s: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5'),
			moves => {
				'c1e3' => 1,
				'e1g1' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/4p3/2ppPb2/3P4/5N2/PPP1BPPP/RNBQ1RK1 b kq' => {
			eco => 'B12t',
			# TRANSLATORS: B12t: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. O-O
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6.O-O'),
			moves => {
				'b8d7' => 1,
				'g8e7' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/4p3/2ppPb2/3P4/5N2/PPP1BPPP/RNBQ1RK1 w kq' => {
			eco => 'B12t',
			# TRANSLATORS: B12t: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. O-O Nd7
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6.O-O'),
			moves => {
			},
		},
		'rn1qkb1r/pp2nppp/4p3/2ppPb2/3P4/5N2/PPP1BPPP/RNBQ1RK1 w kq' => {
			eco => 'B12t',
			# TRANSLATORS: B12t: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. O-O Ne7
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6.O-O'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/4p3/2ppPb2/3P4/4BN2/PPP1BPPP/RN1QK2R b KQkq' => {
			eco => 'B12u',
			# TRANSLATORS: B12u: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. Be3
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6. Be3'),
			moves => {
				'b8d7' => 1,
				'g8e7' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/4p3/2ppPb2/3P4/4BN2/PPP1BPPP/RN1QK2R w KQkq' => {
			eco => 'B12u',
			# TRANSLATORS: B12u: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. Be3 Nd7
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6. Be3'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/4p3/2ppPb2/3P4/4BN2/PPP1BPPP/RN1Q1RK1 b kq' => {
			eco => 'B12u',
			# TRANSLATORS: B12u: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. Be3 Nd7 7. O-O
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6. Be3'),
			moves => {
			},
		},
		'rn1qkb1r/pp2nppp/4p3/2ppPb2/3P4/4BN2/PPP1BPPP/RN1QK2R w KQkq' => {
			eco => 'B12u',
			# TRANSLATORS: B12u: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nf3 e6 5. Be2 c5 6. Be3 Ne7
			variation => N__('Caro-Kann: Advance, 4. Nf3 e6 5. Be2 c5 6. Be3'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/3pPb2/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B12v',
			# TRANSLATORS: B12v: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3
			variation => N__('Caro-Kann: Advance, 4. Nc3'),
			moves => {
				'd8b6' => 1,
				'e7e6' => 1,
				'h7h5' => 1,
			},
		},
		'rn2kbnr/pp2pppp/1qp5/3pPb2/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B12v',
			# TRANSLATORS: B12v: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 Qb6
			variation => N__('Caro-Kann: Advance, 4. Nc3 Qb6'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B12v',
			# TRANSLATORS: B12v: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 e6
			variation => N__('Caro-Kann: Advance, 4. Nc3 e6'),
			moves => {
				'g2g4' => 1,
			},
		},
		'rn1qkbnr/pp2ppp1/2p5/3pPb1p/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B12v',
			# TRANSLATORS: B12v: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 h5
			variation => N__('Caro-Kann: Advance, 4. Nc3 h5'),
			moves => {
			},
		},
		'rn1qkbnr/pp3ppp/2p1p3/3pPb2/3P2P1/2N5/PPP2P1P/R1BQKBNR b KQkq' => {
			eco => 'B12w',
			# TRANSLATORS: B12w: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 e6 5. g4
			variation => N__('Caro-Kann: Advance, Main Line'),
			moves => {
				'f5g6' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p1b1/3pP3/3P2P1/2N5/PPP2P1P/R1BQKBNR w KQkq' => {
			eco => 'B12w',
			# TRANSLATORS: B12w: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 e6 5. g4 Bg6
			variation => N__('Caro-Kann: Advance, Main Line'),
			moves => {
				'g1e2' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p1b1/3pP3/3P2P1/2N5/PPP1NP1P/R1BQKB1R b KQkq' => {
			eco => 'B12w',
			# TRANSLATORS: B12w: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 e6 5. g4 Bg6 6. Nge2
			variation => N__('Caro-Kann: Advance, Main Line, 6.Nge2'),
			moves => {
				'c6c5' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/4p1b1/2ppP3/3P2P1/2N5/PPP1NP1P/R1BQKB1R w KQkq' => {
			eco => 'B12x',
			# TRANSLATORS: B12x: 1. e4 c6 2. d4 d5 3. e5 Bf5 4. Nc3 e6 5. g4 Bg6 6. Nge2 c5
			variation => N__('Caro-Kann: Advance, Main Line, 6.Nge2 c5'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3P4/3P4/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B13a',
			# TRANSLATORS: B13a: 1. e4 c6 2. d4 d5 3. exd5
			variation => N__('Caro-Kann: Exchange Variation'),
			moves => {
				'c6d5' => 1,
				'd8d5' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/2p5/3q4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B13a',
			# TRANSLATORS: B13a: 1. e4 c6 2. d4 d5 3. exd5 Qxd5
			variation => N__('Caro-Kann: Exchange, 3...Qxd5'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/3P4/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B13b',
			# TRANSLATORS: B13b: 1. e4 c6 2. d4 d5 3. exd5 cxd5
			variation => N__('Caro-Kann: Exchange, 3...cxd5'),
			moves => {
				'c2c4' => 1,
				'f1d3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B13c',
			# TRANSLATORS: B13c: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Nf3
			variation => N__('Caro-Kann: Exchange, 4. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/3p4/3P4/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B13d',
			# TRANSLATORS: B13d: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Nf3 Nf6
			variation => N__('Caro-Kann: Exchange, 4. Nf3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/3P4/3B4/PPP2PPP/RNBQK1NR b KQkq' => {
			eco => 'B13e',
			# TRANSLATORS: B13e: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3
			variation => N__('Caro-Kann: Exchange, 4. Bd3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/3p4/3P4/3B4/PPP2PPP/RNBQK1NR w KQkq' => {
			eco => 'B13f',
			# TRANSLATORS: B13f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6
			variation => N__('Caro-Kann: Exchange, 4. Bd3 Nc6'),
			moves => {
				'c2c3' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2n5/3p4/3P4/2PB4/PP3PPP/RNBQK1NR b KQkq' => {
			eco => 'B13f',
			# TRANSLATORS: B13f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3
			variation => N__('Caro-Kann: Exchange, 4. Bd3 Nc6 5. c3'),
			moves => {
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2n3p1/3p4/3P4/2PB4/PP3PPP/RNBQK1NR w KQkq' => {
			eco => 'B13g',
			# TRANSLATORS: B13g: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 g6
			variation => N__('Caro-Kann: Exchange, 4. Bd3 Nc6 5. c3 g6'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p4/3P4/2PB4/PP3PPP/RNBQK1NR w KQkq' => {
			eco => 'B13h',
			# TRANSLATORS: B13h: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6
			variation => N__('Caro-Kann: Exchange, 4. Bd3 Nc6 5. c3 Nf6'),
			moves => {
				'c1f4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p4/3P4/2PB1N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B13i',
			# TRANSLATORS: B13i: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Nf3
			variation => N__('Caro-Kann: Exchange, 4. Bd3 Nc6 5. c3 Nf6 6. Nf3'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p4/3P1B2/2PB4/PP3PPP/RN1QK1NR b KQkq' => {
			eco => 'B13j',
			# TRANSLATORS: B13j: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Bf4
			variation => N__('Caro-Kann: Exchange, Rubinstein Variation'),
			moves => {
				'c8g4' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n2n2/3p4/3P1Bb1/2PB4/PP3PPP/RN1QK1NR w KQkq' => {
			eco => 'B13k',
			# TRANSLATORS: B13k: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Bf4 Bg4
			variation => N__('Caro-Kann: Exchange, Rubinstein, 6... Bg4'),
			moves => {
				'd1b3' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n2n2/3p4/3P1Bb1/1QPB4/PP3PPP/RN2K1NR b KQkq' => {
			eco => 'B13k',
			# TRANSLATORS: B13k: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Bf4 Bg4 7. Qb3
			variation => N__('Caro-Kann: Exchange, Rubinstein, 6... Bg4'),
			moves => {
				'd8c8' => 1,
				'd8d7' => 1,
			},
		},
		'r1q1kb1r/pp2pppp/2n2n2/3p4/3P1Bb1/1QPB4/PP3PPP/RN2K1NR w KQkq' => {
			eco => 'B13l',
			# TRANSLATORS: B13l: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Bf4 Bg4 7. Qb3 Qc8
			variation => N__('Caro-Kann: Exchange, Rubinstein, 7. Qb3 Qc8'),
			moves => {
			},
		},
		'r3kb1r/pp1qpppp/2n2n2/3p4/3P1Bb1/1QPB4/PP3PPP/RN2K1NR w KQkq' => {
			eco => 'B13m',
			# TRANSLATORS: B13m: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. Bd3 Nc6 5. c3 Nf6 6. Bf4 Bg4 7. Qb3 Qd7
			variation => N__('Caro-Kann: Exchange, Rubinstein, 7. Qb3 Qd7'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/2PP4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B13n',
			# TRANSLATORS: B13n: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4
			variation => N__('Caro-Kann: Panov-Botvinnik Attack'),
			moves => {
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/4p3/3p4/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B13n',
			# TRANSLATORS: B13n: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 e6
			variation => N__('Caro-Kann: Panov-Botvinnik, 4... e6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/3p4/2PP4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B13o',
			# TRANSLATORS: B13o: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6
			variation => N__('Caro-Kann: Panov-Botvinnik, 4... Nf6'),
			moves => {
				'b1c3' => 1,
				'c4c5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2Pp4/3P4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B13o',
			# TRANSLATORS: B13o: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. c5
			variation => N__('Caro-Kann: Panov-Botvinnik, Gunderam Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/3p4/2PP4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B13p',
			# TRANSLATORS: B13p: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5. Nc3'),
			moves => {
				'b8c6' => 1,
				'd5c4' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/8/2pP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B13p',
			# TRANSLATORS: B13p: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 dxc4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5...dxc4'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/8/2BP4/2N5/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'B13p',
			# TRANSLATORS: B13p: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 dxc4 6. Bxc4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5...dxc4'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p4/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B13q',
			# TRANSLATORS: B13q: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6'),
			moves => {
				'c1g5' => 1,
				'c4d5' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3P4/3P4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B13q',
			# TRANSLATORS: B13q: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6.cxd5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n5/3n4/3P4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B13q',
			# TRANSLATORS: B13q: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. cxd5 Nxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6.cxd5 Nxd5'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p4/2PP4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B13r',
			# TRANSLATORS: B13r: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3'),
			moves => {
				'c8g4' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n2n2/3p4/2PP2b1/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B13s',
			# TRANSLATORS: B13s: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4'),
			moves => {
				'c4d5' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n2n2/3P4/3P2b1/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B13s',
			# TRANSLATORS: B13s: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4'),
			moves => {
				'f6d5' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n5/3n4/3P2b1/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B13s',
			# TRANSLATORS: B13s: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4'),
			moves => {
				'd1b3' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n5/3n4/3P2b1/1QN2N2/PP3PPP/R1B1KB1R b KQkq' => {
			eco => 'B13t',
			# TRANSLATORS: B13t: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5 8. Qb3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4: 8. Qb3'),
			moves => {
				'g4f3' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n5/3n4/3P4/1QN2b2/PP3PPP/R1B1KB1R w KQkq' => {
			eco => 'B13t',
			# TRANSLATORS: B13t: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5 8. Qb3 Bxf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4: 8. Qb3'),
			moves => {
				'g2f3' => 1,
			},
		},
		'r2qkb1r/pp2pppp/2n5/3n4/3P4/1QN2P2/PP3P1P/R1B1KB1R b KQkq' => {
			eco => 'B13t',
			# TRANSLATORS: B13t: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5 8. Qb3 Bxf3 9. gxf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4: 8. Qb3'),
			moves => {
				'd5b6' => 1,
				'e7e6' => 1,
			},
		},
		'r2qkb1r/pp2pppp/1nn5/8/3P4/1QN2P2/PP3P1P/R1B1KB1R w KQkq' => {
			eco => 'B13t',
			# TRANSLATORS: B13t: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5 8. Qb3 Bxf3 9. gxf3 Nb6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4: 9.gxf3 Nb6'),
			moves => {
			},
		},
		'r2qkb1r/pp3ppp/2n1p3/3n4/3P4/1QN2P2/PP3P1P/R1B1KB1R w KQkq' => {
			eco => 'B13u',
			# TRANSLATORS: B13u: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Nf3 Bg4 7. cxd5 Nxd5 8. Qb3 Bxf3 9. gxf3 e6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Nf3 Bg4: Main Line'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3p2B1/2PP4/2N5/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'B13v',
			# TRANSLATORS: B13v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Bg5'),
			moves => {
				'c8e6' => 1,
				'd5c4' => 1,
				'd8a5' => 1,
				'd8b6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/6B1/2pP4/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13v',
			# TRANSLATORS: B13v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 dxc4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Bg5 dxc4'),
			moves => {
				'd4d5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2n2n2/3P2B1/2p5/2N5/PP3PPP/R2QKBNR b KQkq' => {
			eco => 'B13v',
			# TRANSLATORS: B13v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 dxc4 7. d5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Bg5 dxc4'),
			moves => {
				'c6a5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/5n2/n2P2B1/2p5/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13v',
			# TRANSLATORS: B13v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 dxc4 7. d5 Na5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Herzog Defence'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/1qn2n2/3p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13v',
			# TRANSLATORS: B13v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 Qb6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Spielmann Variation'),
			moves => {
			},
		},
		'r1b1kb1r/pp2pppp/2n2n2/q2p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13w',
			# TRANSLATORS: B13w: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 Qa5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Czerniak Variation'),
			moves => {
			},
		},
		'r2qkb1r/pp2pppp/2n1bn2/3p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13x',
			# TRANSLATORS: B13x: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 Be6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6 6. Bg5 Be6'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/3p2B1/2PP4/2N5/PP3PPP/R2QKBNR w KQkq' => {
			eco => 'B13y',
			# TRANSLATORS: B13y: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/3p2B1/2PP4/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'B13y',
			# TRANSLATORS: B13y: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/3p2B1/2PP4/2N2N2/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'B13y',
			# TRANSLATORS: B13y: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation'),
			moves => {
				'c4c5' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'B13z',
			# TRANSLATORS: B13z: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variaiton, 8. c5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/2Pp2B1/3P4/2N2N2/PP3PPP/R2QKB1R w KQ' => {
			eco => 'B13z',
			# TRANSLATORS: B13z: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5 O-O
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation, 8. c5 O-O'),
			moves => {
				'f1b5' => 1,
			},
		},
		'r1bq1rk1/pp2bppp/2n1pn2/1BPp2B1/3P4/2N2N2/PP3PPP/R2QK2R b KQ' => {
			eco => 'B13z',
			# TRANSLATORS: B13z: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 Nc6 6. Bg5 e6 7. Nf3 Be7 8. c5 O-O 9. Bb5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... Nc6, Normal Variation, 8. c5 O-O 9. Bb5'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/5np1/3p4/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B14a',
			# TRANSLATORS: B14a: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6'),
			moves => {
				'c4d5' => 1,
				'd1b3' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/5np1/3p4/2PP4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B14a',
			# TRANSLATORS: B14a: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Nf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/5np1/3P4/3P4/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B14b',
			# TRANSLATORS: B14b: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6.cxd5'),
			moves => {
				'f6d5' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/6p1/3n4/3P4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B14c',
			# TRANSLATORS: B14c: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. cxd5 Nxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6.cxd5 Nxd5'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/6p1/3n4/2BP4/2N5/PP3PPP/R1BQK1NR b KQkq' => {
			eco => 'B14d',
			# TRANSLATORS: B14d: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. cxd5 Nxd5 7. Bc4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6.cxd5 Nxd5 7. Bc4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/5np1/3P4/3P4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B14e',
			# TRANSLATORS: B14e: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. cxd5 Bg7
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6.cxd5 Bg7'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/5np1/3p4/2PP4/1QN5/PP3PPP/R1B1KBNR b KQkq' => {
			eco => 'B14f',
			# TRANSLATORS: B14f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6. Qb3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/5np1/3p4/2PP4/1QN5/PP3PPP/R1B1KBNR w KQkq' => {
			eco => 'B14f',
			# TRANSLATORS: B14f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6. Qb3'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/5np1/3P4/3P4/1QN5/PP3PPP/R1B1KBNR b KQkq' => {
			eco => 'B14f',
			# TRANSLATORS: B14f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7 7. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6 6. Qb3 Bg7 7.cxd5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/5np1/3P4/3P4/1QN5/PP3PPP/R1B1KBNR w KQ' => {
			eco => 'B14f',
			# TRANSLATORS: B14f: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7 7. cxd5 O-O
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6, Main Line'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rnbq1rk1/pp2ppbp/5np1/3P4/3P4/1QN5/PP2BPPP/R1B1K1NR b KQ' => {
			eco => 'B14g',
			# TRANSLATORS: B14g: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7 7. cxd5 O-O 8. Be2
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6, Main Line, 8. Be2'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r1bq1rk1/pp1nppbp/5np1/3P4/3P4/1QN5/PP2BPPP/R1B1K1NR w KQ' => {
			eco => 'B14h',
			# TRANSLATORS: B14h: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 g6 6. Qb3 Bg7 7. cxd5 O-O 8. Be2 Nbd7
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... g6, Main Line, 8. Be2 Nbd7'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/3p4/2PP4/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B14i',
			# TRANSLATORS: B14i: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/4pn2/3p4/2PP4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B14j',
			# TRANSLATORS: B14j: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3'),
			moves => {
				'b8c6' => 1,
				'f8b4' => 1,
				'f8e7' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2n1pn2/3p4/2PP4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14j',
			# TRANSLATORS: B14j: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Nc6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/4pn2/3p4/2PP4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14k',
			# TRANSLATORS: B14k: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Be7
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Be7'),
			moves => {
				'c4d5' => 1,
				'f1d3' => 1,
			},
		},
		'rnbqk2r/pp2bppp/4pn2/3p4/2PP4/2NB1N2/PP3PPP/R1BQK2R b KQkq' => {
			eco => 'B14l',
			# TRANSLATORS: B14l: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Be7 7. Bd3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Be7 7. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/pp2bppp/4pn2/3P4/3P4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B14m',
			# TRANSLATORS: B14m: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Be7 7. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Be7 7.cxd5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqk2r/pp2bppp/4p3/3n4/3P4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14n',
			# TRANSLATORS: B14n: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Be7 7. cxd5 Nxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Be7 7.cxd5 Nxd5'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rnbqk2r/pp2bppp/4p3/3n4/3P4/2NB1N2/PP3PPP/R1BQK2R b KQkq' => {
			eco => 'B14o',
			# TRANSLATORS: B14o: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Be7 7. cxd5 Nxd5 8. Bd3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Be7 7.cxd5 Nxd5 8. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/pp3ppp/4pn2/3p4/1bPP4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14p',
			# TRANSLATORS: B14p: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4'),
			moves => {
				'c4d5' => 1,
			},
		},
		'rnbqk2r/pp3ppp/4pn2/3P4/1b1P4/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B14p',
			# TRANSLATORS: B14p: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4'),
			moves => {
				'e6d5' => 1,
				'f6d5' => 1,
			},
		},
		'rnbqk2r/pp3ppp/5n2/3p4/1b1P4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14q',
			# TRANSLATORS: B14q: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 exd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 exd5'),
			moves => {
			},
		},
		'rnbqk2r/pp3ppp/4p3/3n4/1b1P4/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B14r',
			# TRANSLATORS: B14r: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5'),
			moves => {
				'c1d2' => 1,
				'd1c2' => 1,
			},
		},
		'rnbqk2r/pp3ppp/4p3/3n4/1b1P4/2N2N2/PPQ2PPP/R1B1KB1R b KQkq' => {
			eco => 'B14s',
			# TRANSLATORS: B14s: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Qc2
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Qc2'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2N2N2/PPQ2PPP/R1B1KB1R w KQkq' => {
			eco => 'B14t',
			# TRANSLATORS: B14t: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Qc2 Nc6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Qc2 Nc6'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PPQ2PPP/R1B1K2R b KQkq' => {
			eco => 'B14u',
			# TRANSLATORS: B14u: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Qc2 Nc6 9. Bd3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Qc2 Nc6 9. Bd3'),
			moves => {
			},
		},
		'rnbqk2r/pp3ppp/4p3/3n4/1b1P4/2N2N2/PP1B1PPP/R2QKB1R b KQkq' => {
			eco => 'B14v',
			# TRANSLATORS: B14v: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2N2N2/PP1B1PPP/R2QKB1R w KQkq' => {
			eco => 'B14w',
			# TRANSLATORS: B14w: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2 Nc6
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1bqk2r/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R b KQkq' => {
			eco => 'B14w',
			# TRANSLATORS: B14w: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2 Nc6 9. Bd3
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp3ppp/2n1p3/3n4/1b1P4/2NB1N2/PP1B1PPP/R2QK2R w KQ' => {
			eco => 'B14x',
			# TRANSLATORS: B14x: 1. e4 c6 2. d4 d5 3. exd5 cxd5 4. c4 Nf6 5. Nc3 e6 6. Nf3 Bb4 7. cxd5 Nxd5 8. Bd2 Nc6 9. Bd3 O-O
			variation => N__('Caro-Kann: Panov-Botvinnik, 5... e6 6. Nf3 Bb4 7.cxd5 Nxd5 8. Bd2 Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/3p4/3PP3/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B15a',
			# TRANSLATORS: B15a: 1. e4 c6 2. d4 d5 3. Nc3
			variation => N__('Caro-Kann: 3. Nc3'),
			moves => {
				'b7b5' => 1,
				'd5e4' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/p3pppp/2p5/1p1p4/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15b',
			# TRANSLATORS: B15b: 1. e4 c6 2. d4 d5 3. Nc3 b5
			variation => N__('Caro-Kann: Gurgenidze Counterattack'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15c',
			# TRANSLATORS: B15c: 1. e4 c6 2. d4 d5 3. Nc3 g6
			variation => N__('Caro-Kann: Gurgenidze/Modern'),
			moves => {
				'c1e3' => 1,
				'e4e5' => 1,
				'g1f3' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/2N1B3/PPP2PPP/R2QKBNR b KQkq' => {
			eco => 'B15d',
			# TRANSLATORS: B15d: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Be3
			variation => N__('Caro-Kann: Gurgenidze: 4. Be3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/2N1B3/PPP2PPP/R2QKBNR w KQkq' => {
			eco => 'B15d',
			# TRANSLATORS: B15d: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Be3 Bg7
			variation => N__('Caro-Kann: Gurgenidze: 4. Be3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/2N4P/PPP2PP1/R1BQKBNR b KQkq' => {
			eco => 'B15e',
			# TRANSLATORS: B15e: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. h3
			variation => N__('Caro-Kann: Gurgenidze: 4. h3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/2N4P/PPP2PP1/R1BQKBNR w KQkq' => {
			eco => 'B15e',
			# TRANSLATORS: B15e: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. h3 Bg7
			variation => N__('Caro-Kann: Gurgenidze: 4. h3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3pP3/3P4/2N5/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B15f',
			# TRANSLATORS: B15f: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5
			variation => N__('Caro-Kann: Gurgenidze: 4. e5'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3pP3/3P4/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15f',
			# TRANSLATORS: B15f: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5 Bg7
			variation => N__('Caro-Kann: Gurgenidze: 4. e5'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3pP3/3P1P2/2N5/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B15g',
			# TRANSLATORS: B15g: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5 Bg7 5. f4
			variation => N__('Caro-Kann: Gurgenidze: 4. e5 Bg7 5. f4'),
			moves => {
				'g8h6' => 1,
				'h7h5' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2p3pn/3pP3/3P1P2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B15g',
			# TRANSLATORS: B15g: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5 Bg7 5. f4 Nh6
			variation => N__('Caro-Kann: Gurgenidze: 4. e5 Bg7 5. f4 Nh6'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppb1/2p3p1/3pP2p/3P1P2/2N5/PPP3PP/R1BQKBNR w KQkq' => {
			eco => 'B15h',
			# TRANSLATORS: B15h: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5 Bg7 5. f4 h5
			variation => N__('Caro-Kann: Gurgenidze: 4. e5 Bg7 5. f4 h5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqk1nr/pp2ppb1/2p3p1/3pP2p/3P1P2/2N2N2/PPP3PP/R1BQKB1R b KQkq' => {
			eco => 'B15i',
			# TRANSLATORS: B15i: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. e5 Bg7 5. f4 h5 6. Nf3
			variation => N__('Caro-Kann: Gurgenidze: 4. e5 Bg7 5. f4 h5 6. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pp1p/2p3p1/3p4/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B15j',
			# TRANSLATORS: B15j: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/2N2N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B15k',
			# TRANSLATORS: B15k: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7'),
			moves => {
				'e4d5' => 1,
				'e4e5' => 1,
				'h2h3' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3P4/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B15k',
			# TRANSLATORS: B15k: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. exd5
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5.exd5'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3pP3/3P4/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B15k',
			# TRANSLATORS: B15k: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. e5
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. e5'),
			moves => {
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/3p4/3PP3/2N2N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B15l',
			# TRANSLATORS: B15l: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. h3
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. h3'),
			moves => {
				'd5e4' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/8/3Pp3/2N2N1P/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B15m',
			# TRANSLATORS: B15m: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. h3 dxe4
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. h3 dxe4'),
			moves => {
				'c3e4' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/2p3p1/8/3PN3/5N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B15m',
			# TRANSLATORS: B15m: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. h3 dxe4 6. Nxe4
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. h3 dxe4'),
			moves => {
			},
		},
		'rnbqk2r/pp2ppbp/2p2np1/3p4/3PP3/2N2N1P/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B15n',
			# TRANSLATORS: B15n: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. h3 Nf6
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. h3 Nf6'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rnbqk2r/pp2ppbp/2p2np1/3pP3/3P4/2N2N1P/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B15o',
			# TRANSLATORS: B15o: 1. e4 c6 2. d4 d5 3. Nc3 g6 4. Nf3 Bg7 5. h3 Nf6 6. e5
			variation => N__('Caro-Kann: Gurgenidze: 4. Nf3 Bg7 5. h3 Nf6 6. e5'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3Pp3/2N5/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15p',
			# TRANSLATORS: B15p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4
			variation => N__('Caro-Kann: 3. Nc3 dxe4'),
			moves => {
				'c3e4' => 1,
				'f1c4' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/2BPp3/2N5/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B15p',
			# TRANSLATORS: B15p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Bc4
			variation => N__('Caro-Kann: von Hennig Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3Pp3/2N2P2/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B15p',
			# TRANSLATORS: B15p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. f3
			variation => N__('Caro-Kann: Rasa-Studier Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/2p5/8/3PN3/8/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B15q',
			# TRANSLATORS: B15q: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4
			variation => N__('Caro-Kann: 4.Nxe4'),
			moves => {
				'b8d7' => 1,
				'c8f5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/8/3PN3/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15r',
			# TRANSLATORS: B15r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6
			variation => N__('Caro-Kann: 4.Nxe4 Nf6'),
			moves => {
				'e4f6' => 1,
				'e4g3' => 1,
				'f1d3' => 1,
				'f2f3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/8/3PN3/3B4/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B15r',
			# TRANSLATORS: B15r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Bd3
			variation => N__('Caro-Kann: Alekhine Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/8/3PN3/5P2/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B15r',
			# TRANSLATORS: B15r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. f3
			variation => N__('Caro-Kann: 4.Nxe4 Nf6 5. f3 Gambit'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2p2n2/8/3P4/6N1/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B15s',
			# TRANSLATORS: B15s: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Ng3
			variation => N__('Caro-Kann: 4.Nxe4 Nf6 5. Ng3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/2p2N2/8/3P4/8/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B15t',
			# TRANSLATORS: B15t: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+
			variation => N__('Caro-Kann: 4.Nxe4 Nf6 5.Nxf6+'),
			moves => {
				'e7f6' => 1,
				'g7f6' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p2p2/8/3P4/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B15t',
			# TRANSLATORS: B15t: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ exf6
			variation => N__('Caro-Kann: Tartakower (Nimzowitsch) Variation'),
			moves => {
				'c2c3' => 1,
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pp3ppp/2p2p2/8/2BP4/8/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B15u',
			# TRANSLATORS: B15u: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ exf6 6. Bc4
			variation => N__('Caro-Kann: Tartakower, Forgacs Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/2p2p2/8/3P4/2P5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B15v',
			# TRANSLATORS: B15v: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ exf6 6. c3
			variation => N__('Caro-Kann: Tartakower, 6. c3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/3P4/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B16a',
			# TRANSLATORS: B16a: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6
			variation => N__('Caro-Kann: Bronstein-Larsen'),
			moves => {
				'c2c3' => 1,
				'd1d3' => 1,
				'f1c4' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/3P4/3Q4/PPP2PPP/R1B1KBNR b KQkq' => {
			eco => 'B16b',
			# TRANSLATORS: B16b: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Qd3
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Qd3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/3P4/8/PPP1NPPP/R1BQKB1R b KQkq' => {
			eco => 'B16c',
			# TRANSLATORS: B16c: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Ne2
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Ne2'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/2BP4/8/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B16d',
			# TRANSLATORS: B16d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Bc4
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Bc4'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/3P4/5N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B16e',
			# TRANSLATORS: B16e: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Nf3
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Nf3'),
			moves => {
				'c8f5' => 1,
				'c8g4' => 1,
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/5b2/3P4/5N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B16f',
			# TRANSLATORS: B16f: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Nf3 Bf5
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Nf3 Bf5'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/8/3P2b1/5N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B16g',
			# TRANSLATORS: B16g: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Nf3 Bg4
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Nf3 Bg4'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/8/3P2b1/5N2/PPP1BPPP/R1BQK2R b KQkq' => {
			eco => 'B16g',
			# TRANSLATORS: B16g: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Nf3 Bg4 7. Be2
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Nf3 Bg4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkb1r/pp3p1p/2p1pp2/8/3P2b1/5N2/PPP1BPPP/R1BQK2R w KQkq' => {
			eco => 'B16h',
			# TRANSLATORS: B16h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. Nf3 Bg4 7. Be2 e6
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. Nf3 Bg4 7. Be2 e6'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pp1p/2p2p2/8/3P4/2P5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B16i',
			# TRANSLATORS: B16i: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3'),
			moves => {
				'c8f5' => 1,
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/5b2/3P4/2P5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B16j',
			# TRANSLATORS: B16j: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3 Bf5
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3 Bf5'),
			moves => {
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/5b2/3P4/2P5/PP2NPPP/R1BQKB1R b KQkq' => {
			eco => 'B16k',
			# TRANSLATORS: B16k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3 Bf5 7. Ne2
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3 Bf5 7. Ne2'),
			moves => {
			},
		},
		'rn1qkb1r/pp2pp1p/2p2p2/5b2/3P4/2P2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B16l',
			# TRANSLATORS: B16l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3 Bf5 7. Nf3
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3 Bf5 7. Nf3'),
			moves => {
				'd8c7' => 1,
				'e7e6' => 1,
			},
		},
		'rn2kb1r/ppq1pp1p/2p2p2/5b2/3P4/2P2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B16m',
			# TRANSLATORS: B16m: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3 Bf5 7. Nf3 Qc7
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3 Bf5 7. Nf3 Qc7'),
			moves => {
			},
		},
		'rn1qkb1r/pp3p1p/2p1pp2/5b2/3P4/2P2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B16n',
			# TRANSLATORS: B16n: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nf6 5. Nxf6+ gxf6 6. c3 Bf5 7. Nf3 e6
			variation => N__('Caro-Kann: Bronstein-Larsen, 6. c3 Bf5 7. Nf3 e6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1npppp/2p5/8/3PN3/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B17a',
			# TRANSLATORS: B17a: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7
			variation => N__('Caro-Kann: Steinitz Variation'),
			moves => {
				'e4g5' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1npppp/2p5/8/3PN3/5N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B17b',
			# TRANSLATORS: B17b: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3
			variation => N__('Caro-Kann: Steinitz, 5. Nf3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/8/3PN3/5N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B17c',
			# TRANSLATORS: B17c: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6'),
			moves => {
				'e4f6' => 1,
				'e4g3' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2N2/8/3P4/5N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B17d',
			# TRANSLATORS: B17d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6 6. Nxf6+
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6 6.Nxf6+'),
			moves => {
				'd7f6' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2p2n2/8/3P4/5N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B17d',
			# TRANSLATORS: B17d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6 6. Nxf6+ Nxf6
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6 6.Nxf6+'),
			moves => {
				'f1c4' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2p2n2/8/2BP4/5N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B17e',
			# TRANSLATORS: B17e: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6 6. Nxf6+ Nxf6 7. Bc4
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6 6.Nxf6+ Nxf6 7. Bc4'),
			moves => {
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/8/3P4/5NN1/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B17f',
			# TRANSLATORS: B17f: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6 6. Ng3
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6 6. Ng3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2p1pn2/8/3P4/5NN1/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B17g',
			# TRANSLATORS: B17g: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Nf3 Ngf6 6. Ng3 e6
			variation => N__('Caro-Kann: Steinitz, 5. Nf3 Ngf6 6. Ng3 e6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1npppp/2p5/6N1/3P4/8/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B17h',
			# TRANSLATORS: B17h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5
			variation => N__('Caro-Kann: Steinitz, 5. Ng5'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/6N1/3P4/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B17i',
			# TRANSLATORS: B17i: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/6N1/3P4/3B4/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B17j',
			# TRANSLATORS: B17j: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2p1pn2/6N1/3P4/3B4/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B17k',
			# TRANSLATORS: B17k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2p1pn2/6N1/3P4/3B1N2/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B17k',
			# TRANSLATORS: B17k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6'),
			moves => {
				'f8d6' => 1,
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/6N1/3P4/3B1N2/PPP2PPP/R1BQK2R w KQkq' => {
			eco => 'B17k',
			# TRANSLATORS: B17k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6'),
			moves => {
				'd1e2' => 1,
			},
		},
		'r1bqk2r/pp1n1ppp/2pbpn2/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R b KQkq' => {
			eco => 'B17l',
			# TRANSLATORS: B17l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'h7h6' => 1,
			},
		},
		'r1bqk2r/pp1n1pp1/2pbpn1p/6N1/3P4/3B1N2/PPP1QPPP/R1B1K2R w KQkq' => {
			eco => 'B17l',
			# TRANSLATORS: B17l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'g5e4' => 1,
			},
		},
		'r1bqk2r/pp1n1pp1/2pbpn1p/8/3PN3/3B1N2/PPP1QPPP/R1B1K2R b KQkq' => {
			eco => 'B17l',
			# TRANSLATORS: B17l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'f6e4' => 1,
			},
		},
		'r1bqk2r/pp1n1pp1/2pbp2p/8/3Pn3/3B1N2/PPP1QPPP/R1B1K2R w KQkq' => {
			eco => 'B17l',
			# TRANSLATORS: B17l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3 e6 7.N1f3 Bd6 8. Qe2'),
			moves => {
				'e2e4' => 1,
			},
		},
		'r1bqk2r/pp1n1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R b KQkq' => {
			eco => 'B17m',
			# TRANSLATORS: B17m: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4'),
			moves => {
				'd7f6' => 1,
				'd8c7' => 1,
			},
		},
		'r1b1k2r/ppqn1pp1/2pbp2p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq' => {
			eco => 'B17n',
			# TRANSLATORS: B17n: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4 Qc7
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4 Qc7'),
			moves => {
			},
		},
		'r1bqk2r/pp3pp1/2pbpn1p/8/3PQ3/3B1N2/PPP2PPP/R1B1K2R w KQkq' => {
			eco => 'B17o',
			# TRANSLATORS: B17o: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Ng5 Ngf6 6. Bd3 e6 7. N1f3 Bd6 8. Qe2 h6 9. Ne4 Nxe4 10. Qxe4 Nf6
			variation => N__('Caro-Kann: Steinitz, 5. Ng5 Ngf6 6. Bd3, Main Line 10.Qxe4 Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1npppp/2p5/8/2BPN3/8/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B17p',
			# TRANSLATORS: B17p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4
			variation => N__('Caro-Kann: Steinitz, 5. Bc4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/8/2BPN3/8/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B17q',
			# TRANSLATORS: B17q: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6'),
			moves => {
				'e4g5' => 1,
			},
		},
		'r1bqkb1r/pp1npppp/2p2n2/6N1/2BP4/8/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B17r',
			# TRANSLATORS: B17r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2p1pn2/6N1/2BP4/8/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B17r',
			# TRANSLATORS: B17r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5'),
			moves => {
				'd1e2' => 1,
			},
		},
		'r1bqkb1r/pp1n1ppp/2p1pn2/6N1/2BP4/8/PPP1QPPP/R1B1K1NR b KQkq' => {
			eco => 'B17r',
			# TRANSLATORS: B17r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6 7. Qe2
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5'),
			moves => {
				'd7b6' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/1np1pn2/6N1/2BP4/8/PPP1QPPP/R1B1K1NR w KQkq' => {
			eco => 'B17r',
			# TRANSLATORS: B17r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6 7. Qe2 Nb6
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5'),
			moves => {
				'c4b3' => 1,
				'c4d3' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/1np1pn2/6N1/3P4/1B6/PPP1QPPP/R1B1K1NR b KQkq' => {
			eco => 'B17s',
			# TRANSLATORS: B17s: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6 7. Qe2 Nb6 8. Bb3
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5, 8. Bb3'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/1np1pn2/6N1/3P4/3B4/PPP1QPPP/R1B1K1NR b KQkq' => {
			eco => 'B17t',
			# TRANSLATORS: B17t: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6 7. Qe2 Nb6 8. Bd3
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5, 8. Bd3'),
			moves => {
				'h7h6' => 1,
			},
		},
		'r1bqkb1r/pp3pp1/1np1pn1p/6N1/3P4/3B4/PPP1QPPP/R1B1K1NR w KQkq' => {
			eco => 'B17u',
			# TRANSLATORS: B17u: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Nd7 5. Bc4 Ngf6 6. Ng5 e6 7. Qe2 Nb6 8. Bd3 h6
			variation => N__('Caro-Kann: Steinitz, 5. Bc4 Ngf6 6. Ng5, 8. Bd3 h6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/5b2/3PN3/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B18a',
			# TRANSLATORS: B18a: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5
			variation => N__('Caro-Kann: Classical'),
			moves => {
				'e4c5' => 1,
				'e4g3' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p5/2N2b2/3P4/8/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B18b',
			# TRANSLATORS: B18b: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Nc5
			variation => N__('Caro-Kann: Classical, 5. Nc5'),
			moves => {
				'b7b6' => 1,
			},
		},
		'rn1qkbnr/p3pppp/1pp5/2N2b2/3P4/8/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B18c',
			# TRANSLATORS: B18c: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Nc5 b6
			variation => N__('Caro-Kann: Classical, 5. Nc5 b6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p5/5b2/3P4/6N1/PPP2PPP/R1BQKBNR b KQkq' => {
			eco => 'B18d',
			# TRANSLATORS: B18d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3
			variation => N__('Caro-Kann: Classical, 5. Ng3'),
			moves => {
				'f5g6' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P4/6N1/PPP2PPP/R1BQKBNR w KQkq' => {
			eco => 'B18e',
			# TRANSLATORS: B18e: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6
			variation => N__('Caro-Kann: Classical, 5. Ng3 Bg6'),
			moves => {
				'f1c4' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
				'g1h3' => 1,
				'h2h4' => 1,
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P1P2/6N1/PPP3PP/R1BQKBNR b KQkq' => {
			eco => 'B18f',
			# TRANSLATORS: B18f: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. f4
			variation => N__('Caro-Kann: Classical, Maroczy Attack'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P4/6NN/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B18g',
			# TRANSLATORS: B18g: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Nh3
			variation => N__('Caro-Kann: Classical, Flohr Variation'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P4/6N1/PPP1NPPP/R1BQKB1R b KQkq' => {
			eco => 'B18h',
			# TRANSLATORS: B18h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. N1e2
			variation => N__('Caro-Kann: Classical, 6.N1e2'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/2BP4/6N1/PPP2PPP/R1BQK1NR b KQkq' => {
			eco => 'B18i',
			# TRANSLATORS: B18i: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Bc4
			variation => N__('Caro-Kann: Classical, 6. Bc4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p1b1/8/2BP4/6N1/PPP2PPP/R1BQK1NR w KQkq' => {
			eco => 'B18i',
			# TRANSLATORS: B18i: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Bc4 e6
			variation => N__('Caro-Kann: Classical, 6. Bc4'),
			moves => {
				'g1e2' => 1,
			},
		},
		'rn1qkbnr/pp3ppp/2p1p1b1/8/2BP4/6N1/PPP1NPPP/R1BQK2R b KQkq' => {
			eco => 'B18j',
			# TRANSLATORS: B18j: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Bc4 e6 7. N1e2
			variation => N__('Caro-Kann: Classical, 6. Bc4 e6 7.N1e2'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rn1qkb1r/pp3ppp/2p1pnb1/8/2BP4/6N1/PPP1NPPP/R1BQK2R w KQkq' => {
			eco => 'B18k',
			# TRANSLATORS: B18k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Bc4 e6 7. N1e2 Nf6
			variation => N__('Caro-Kann: Classical, 6. Bc4 e6 7.N1e2 Nf6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P4/5NN1/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B18l',
			# TRANSLATORS: B18l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Nf3
			variation => N__('Caro-Kann: Classical, 6. Nf3'),
			moves => {
				'b8d7' => 1,
			},
		},
		'r2qkbnr/pp1npppp/2p3b1/8/3P4/5NN1/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B18m',
			# TRANSLATORS: B18m: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Nf3 Nd7
			variation => N__('Caro-Kann: Classical, 6. Nf3 Nd7'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r2qkbnr/pp1npppp/2p3b1/8/3P4/3B1NN1/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B18n',
			# TRANSLATORS: B18n: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Nf3 Nd7 7. Bd3
			variation => N__('Caro-Kann: Classical, 6. Nf3 Nd7 7. Bd3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r2qkbnr/pp1n1ppp/2p1p1b1/8/3P4/3B1NN1/PPP2PPP/R1BQK2R w KQkq' => {
			eco => 'B18o',
			# TRANSLATORS: B18o: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. Nf3 Nd7 7. Bd3 e6
			variation => N__('Caro-Kann: Classical, 6. Nf3 Nd7 7. Bd3 e6'),
			moves => {
			},
		},
		'rn1qkbnr/pp2pppp/2p3b1/8/3P3P/6N1/PPP2PP1/R1BQKBNR b KQkq' => {
			eco => 'B18p',
			# TRANSLATORS: B18p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4
			variation => N__('Caro-Kann: Classical, 6. h4'),
			moves => {
				'h7h6' => 1,
			},
		},
		'rn1qkbnr/pp2ppp1/2p3bp/8/3P3P/6N1/PPP2PP1/R1BQKBNR w KQkq' => {
			eco => 'B18p',
			# TRANSLATORS: B18p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6
			variation => N__('Caro-Kann: Classical, 6. h4 h6'),
			moves => {
				'f2f4' => 1,
				'g1f3' => 1,
				'g1h3' => 1,
				'h4h5' => 1,
			},
		},
		'rn1qkbnr/pp2ppp1/2p3bp/8/3P3P/6NN/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B18q',
			# TRANSLATORS: B18q: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nh3
			variation => N__('Caro-Kann: Classical, 6. h4 h6 7. Nh3'),
			moves => {
			},
		},
		'rn1qkbnr/pp2ppp1/2p3bp/8/3P1P1P/6N1/PPP3P1/R1BQKBNR b KQkq' => {
			eco => 'B18r',
			# TRANSLATORS: B18r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. f4
			variation => N__('Caro-Kann: Classical, 6. h4 h6 7. f4'),
			moves => {
			},
		},
		'rn1qkbnr/pp2ppp1/2p3bp/7P/3P4/6N1/PPP2PP1/R1BQKBNR b KQkq' => {
			eco => 'B18s',
			# TRANSLATORS: B18s: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. h5
			variation => N__('Caro-Kann: Classical, 6. h4 h6 7. h5'),
			moves => {
			},
		},
		'rn1qkbnr/pp2ppp1/2p3bp/8/3P3P/5NN1/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B19a',
			# TRANSLATORS: B19a: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3
			variation => N__('Caro-Kann: Classical, 7. Nf3'),
			moves => {
				'b8d7' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rn1qkbnr/pp3pp1/2p1p1bp/8/3P3P/5NN1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19b',
			# TRANSLATORS: B19b: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 e6
			variation => N__('Caro-Kann: Classical, 7. Nf3 e6'),
			moves => {
			},
		},
		'rn1qkb1r/pp2ppp1/2p2nbp/8/3P3P/5NN1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19c',
			# TRANSLATORS: B19c: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6'),
			moves => {
				'f3e5' => 1,
				'h4h5' => 1,
			},
		},
		'rn1qkb1r/pp2ppp1/2p2nbp/7P/3P4/5NN1/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B19d',
			# TRANSLATORS: B19d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6 8. h5
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6 8. h5'),
			moves => {
				'g6h7' => 1,
			},
		},
		'rn1qkb1r/pp2pppb/2p2n1p/7P/3P4/5NN1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19d',
			# TRANSLATORS: B19d: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6 8. h5 Bh7
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6 8. h5'),
			moves => {
			},
		},
		'rn1qkb1r/pp2ppp1/2p2nbp/4N3/3P3P/6N1/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B19e',
			# TRANSLATORS: B19e: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6 8. Ne5
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6 8. Ne5'),
			moves => {
				'g6h7' => 1,
			},
		},
		'rn1qkb1r/pp2pppb/2p2n1p/4N3/3P3P/6N1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19e',
			# TRANSLATORS: B19e: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6 8. Ne5 Bh7
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6 8. Ne5'),
			moves => {
				'f1d3' => 1,
			},
		},
		'rn1qkb1r/pp2pppb/2p2n1p/4N3/3P3P/3B2N1/PPP2PP1/R1BQK2R b KQkq' => {
			eco => 'B19f',
			# TRANSLATORS: B19f: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nf6 8. Ne5 Bh7 9. Bd3
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nf6 8. Ne5 Bh7 9. Bd3'),
			moves => {
			},
		},
		'r2qkbnr/pp1nppp1/2p3bp/8/3P3P/5NN1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19g',
			# TRANSLATORS: B19g: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7
			variation => N__('Caro-Kann: Classical, 7. Nf3 Nd7'),
			moves => {
				'h4h5' => 1,
			},
		},
		'r2qkbnr/pp1nppp1/2p3bp/7P/3P4/5NN1/PPP2PP1/R1BQKB1R b KQkq' => {
			eco => 'B19h',
			# TRANSLATORS: B19h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5
			variation => N__('Caro-Kann: Classical, Spassky Variation'),
			moves => {
				'g6h7' => 1,
			},
		},
		'r2qkbnr/pp1npppb/2p4p/7P/3P4/5NN1/PPP2PP1/R1BQKB1R w KQkq' => {
			eco => 'B19h',
			# TRANSLATORS: B19h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7
			variation => N__('Caro-Kann: Classical, Spassky Variation'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r2qkbnr/pp1npppb/2p4p/7P/3P4/3B1NN1/PPP2PP1/R1BQK2R b KQkq' => {
			eco => 'B19h',
			# TRANSLATORS: B19h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3
			variation => N__('Caro-Kann: Classical, Spassky Variation'),
			moves => {
				'h7d3' => 1,
			},
		},
		'r2qkbnr/pp1nppp1/2p4p/7P/3P4/3b1NN1/PPP2PP1/R1BQK2R w KQkq' => {
			eco => 'B19h',
			# TRANSLATORS: B19h: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3
			variation => N__('Caro-Kann: Classical, Spassky Variation'),
			moves => {
				'd1d3' => 1,
			},
		},
		'r2qkbnr/pp1nppp1/2p4p/7P/3P4/3Q1NN1/PPP2PP1/R1B1K2R b KQkq' => {
			eco => 'B19i',
			# TRANSLATORS: B19i: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3'),
			moves => {
				'd8c7' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r2qkb1r/pp1nppp1/2p2n1p/7P/3P4/3Q1NN1/PPP2PP1/R1B1K2R w KQkq' => {
			eco => 'B19j',
			# TRANSLATORS: B19j: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Ngf6
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 Ngf6'),
			moves => {
			},
		},
		'r2qkbnr/pp1n1pp1/2p1p2p/7P/3P4/3Q1NN1/PPP2PP1/R1B1K2R w KQkq' => {
			eco => 'B19k',
			# TRANSLATORS: B19k: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 e6
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 e6'),
			moves => {
				'c1f4' => 1,
			},
		},
		'r2qkbnr/pp1n1pp1/2p1p2p/7P/3P1B2/3Q1NN1/PPP2PP1/R3K2R b KQkq' => {
			eco => 'B19l',
			# TRANSLATORS: B19l: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 e6 11. Bf4
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 e6 11. Bf4'),
			moves => {
			},
		},
		'r3kbnr/ppqnppp1/2p4p/7P/3P4/3Q1NN1/PPP2PP1/R1B1K2R w KQkq' => {
			eco => 'B19m',
			# TRANSLATORS: B19m: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7'),
			moves => {
				'c1d2' => 1,
			},
		},
		'r3kbnr/ppqnppp1/2p4p/7P/3P4/3Q1NN1/PPPB1PP1/R3K2R b KQkq' => {
			eco => 'B19n',
			# TRANSLATORS: B19n: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11. Bd2'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r3kbnr/ppqn1pp1/2p1p2p/7P/3P4/3Q1NN1/PPPB1PP1/R3K2R w KQkq' => {
			eco => 'B19o',
			# TRANSLATORS: B19o: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11. Bd2 e6'),
			moves => {
				'e1c1' => 1,
			},
		},
		'r3kbnr/ppqn1pp1/2p1p2p/7P/3P4/3Q1NN1/PPPB1PP1/2KR3R b kq' => {
			eco => 'B19p',
			# TRANSLATORS: B19p: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O
			variation => N__('Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11. Bd2 e6 12.O-O-O'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r3kb1r/ppqn1pp1/2p1pn1p/7P/3P4/3Q1NN1/PPPB1PP1/2KR3R w kq' => {
			eco => 'B19q',
			# TRANSLATORS: B19q: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6
			variation => N__('Caro-Kann: Classical, Spassky, Main Line'),
			moves => {
				'g3e4' => 1,
			},
		},
		'r3kb1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1N2/PPPB1PP1/2KR3R b kq' => {
			eco => 'B19r',
			# TRANSLATORS: B19r: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4
			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 13. Ne4'),
			moves => {
				'e8c8' => 1,
			},
		},
		'2kr1b1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1N2/PPPB1PP1/2KR3R w -' => {
			eco => 'B19s',
			# TRANSLATORS: B19s: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4 O-O-O
			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 13. Ne4 O-O-O'),
			moves => {
				'g2g3' => 1,
			},
		},
		'2kr1b1r/ppqn1pp1/2p1pn1p/7P/3PN3/3Q1NP1/PPPB1P2/2KR3R b -' => {
			eco => 'B19s',
			# TRANSLATORS: B19s: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4 O-O-O 14. g3
			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 13. Ne4 O-O-O'),
			moves => {
				'f6e4' => 1,
			},
		},
		'2kr1b1r/ppqn1pp1/2p1p2p/7P/3Pn3/3Q1NP1/PPPB1P2/2KR3R w -' => {
			eco => 'B19t',
			# TRANSLATORS: B19t: 1. e4 c6 2. d4 d5 3. Nc3 dxe4 4. Nxe4 Bf5 5. Ng3 Bg6 6. h4 h6 7. Nf3 Nd7 8. h5 Bh7 9. Bd3 Bxd3 10. Qxd3 Qc7 11. Bd2 e6 12. O-O-O Ngf6 13. Ne4 O-O-O 14. g3 Nxe4
			variation => N__('Caro-Kann: Classical, Spassky, Main Line, 14. g3 Nxe4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/8/PPPP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5
			variation => N__('Sicilian Defence'),
			moves => {
				'a2a3' => 1,
				'b1c3' => 1,
				'b2b3' => 1,
				'b2b4' => 1,
				'c2c3' => 1,
				'c2c4' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
				'f1c4' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/P7/1PPP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. a3
			variation => N__('Sicilian: Mengarini'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/1P6/P1PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b3
			variation => N__('Sicilian: Snyder'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/1P6/P1PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b3 Nc6
			variation => N__('Sicilian: Snyder, 2... Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4P3/1P6/P1PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b3 d6
			variation => N__('Sicilian: Snyder, 2... d6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/1P6/P1PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b3 e6
			variation => N__('Sicilian: Snyder, 2... e6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/1P2P3/8/P1PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4
			variation => N__('Sicilian: Wing Gambit'),
			moves => {
				'c5b4' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/1p2P3/8/P1PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4
			variation => N__('Sicilian: Wing Gambit'),
			moves => {
				'a2a3' => 1,
				'c2c4' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/1p2P3/P7/2PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3
			variation => N__('Sicilian: Wing Gambit, Marshall Variation'),
			moves => {
				'b4a3' => 1,
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/4P3/p7/2PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3 bxa3
			variation => N__('Sicilian: Wing Gambit, Carlsbad Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/1p2P3/P7/2PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3 d5
			variation => N__('Sicilian: Wing Gambit, Marshall Variation'),
			moves => {
				'e4d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/3P4/1p6/P7/2PP1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3 d5 4. exd5
			variation => N__('Sicilian: Wing Gambit, Marshall Variation'),
			moves => {
				'd8d5' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/3q4/1p6/P7/2PP1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3 d5 4. exd5 Qxd5
			variation => N__('Sicilian: Wing Gambit, Marshall Variation'),
			moves => {
				'c1b2' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/3q4/1p6/P7/1BPP1PPP/RN1QKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. a3 d5 4. exd5 Qxd5 5. Bb2
			variation => N__('Sicilian: Wing Gambit, Marienbad Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/1pP1P3/8/P2P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. b4 cxb4 3. c4
			variation => N__('Sicilian: Wing Gambit, Santasiere Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2P1P3/8/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4
			variation => N__('Sicilian: English (2. c4)'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 Nc6
			variation => N__('Sicilian: English, 2... Nc6'),
			moves => {
				'b1c3' => 1,
				'g1e2' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P1P3/2N5/PP1P1PPP/R1BQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 Nc6 3. Nc3
			variation => N__('Sicilian: English, 2... Nc6 3. Nc3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/2P1P3/2N5/PP1P1PPP/R1BQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 Nc6 3. Nc3 g6
			variation => N__('Sicilian: English, 2... Nc6 3. Nc3 g6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P1P3/8/PP1PNPPP/RNBQKB1R b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 Nc6 3. Ne2
			variation => N__('Sicilian: English, 2... Nc6 3. Ne2'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2P1P3/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 Nc6 3. Nf3
			variation => N__('Sicilian: English, 2... Nc6 3. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 d6
			variation => N__('Sicilian: English, 2... d6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/2P1P3/8/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. c4 e6
			variation => N__('Sicilian: English, 2... e6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/3P4/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. d3
			variation => N__('Sicilian: 2. d3'),
			moves => {
				'b8c6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/3P4/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. d3 Nc6
			variation => N__('Sicilian: 2. d3 Nc6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/3P2P1/PPP2P1P/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. d3 Nc6 3. g3
			variation => N__('Sicilian: 2. d3 Nc6 3. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/3P4/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. d3 e6
			variation => N__('Sicilian: 2. d3 e6'),
			moves => {
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/3P2P1/PPP2P1P/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. d3 e6 3. g3
			variation => N__('Sicilian: 2. d3 e6 3. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/2B1P3/8/PPPP1PPP/RNBQK1NR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. Bc4
			variation => N__('Sicilian: 2. Bc4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/8/PPPPNPPP/RNBQKB1R b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. Ne2
			variation => N__('Sicilian: Keres Variation (2. Ne2)'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/6P1/PPPP1P1P/RNBQKBNR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3
			variation => N__('Sicilian: 2. g3'),
			moves => {
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/6P1/PPPP1P1P/RNBQKBNR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3 g6
			variation => N__('Sicilian: 2. g3'),
			moves => {
				'f1g2' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/6P1/PPPP1PBP/RNBQK1NR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3 g6 3. Bg2
			variation => N__('Sicilian: 2. g3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/4P3/6P1/PPPP1PBP/RNBQK1NR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3 g6 3. Bg2 Bg7
			variation => N__('Sicilian: 2. g3'),
			moves => {
				'f2f4' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/4PP2/6P1/PPPP2BP/RNBQK1NR b KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3 g6 3. Bg2 Bg7 4. f4
			variation => N__('Sicilian: 2. g3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/3p2p1/2p5/4PP2/6P1/PPPP2BP/RNBQK1NR w KQkq' => {
			eco => 'B20',
			# TRANSLATORS: B20: 1. e4 c5 2. g3 g6 3. Bg2 Bg7 4. f4 d6
			variation => N__('Sicilian: 2. g3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4PP2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B21a',
			# TRANSLATORS: B21a: 1. e4 c5 2. f4
			variation => N__('Sicilian: Grand Prix Attack'),
			moves => {
				'b8c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21b',
			# TRANSLATORS: B21b: 1. e4 c5 2. f4 g6
			variation => N__('Sicilian: Grand Prix, 2... g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21c',
			# TRANSLATORS: B21c: 1. e4 c5 2. f4 d6
			variation => N__('Sicilian: Grand Prix, 2... d6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21d',
			# TRANSLATORS: B21d: 1. e4 c5 2. f4 e6
			variation => N__('Sicilian: Grand Prix, 2... e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4PP2/5N2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'B21d',
			# TRANSLATORS: B21d: 1. e4 c5 2. f4 e6 3. Nf3
			variation => N__('Sicilian: Grand Prix, 2... e6 3. Nf3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21e',
			# TRANSLATORS: B21e: 1. e4 c5 2. f4 Nc6
			variation => N__('Sicilian: Grand Prix, 2... Nc6'),
			moves => {
				'd2d3' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4PP2/3P4/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B21e',
			# TRANSLATORS: B21e: 1. e4 c5 2. f4 Nc6 3. d3
			variation => N__('Sicilian: Grand Prix, 2... Nc6 3. d3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4PP2/5N2/PPPP2PP/RNBQKB1R b KQkq' => {
			eco => 'B21f',
			# TRANSLATORS: B21f: 1. e4 c5 2. f4 Nc6 3. Nf3
			variation => N__('Sicilian: Grand Prix, 2... Nc6 3. Nf3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4PP2/5N2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'B21g',
			# TRANSLATORS: B21g: 1. e4 c5 2. f4 Nc6 3. Nf3 e6
			variation => N__('Sicilian: Grand Prix, 2... Nc6 3. Nf3 e6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4PP2/5N2/PPPP2PP/RNBQKB1R w KQkq' => {
			eco => 'B21h',
			# TRANSLATORS: B21h: 1. e4 c5 2. f4 Nc6 3. Nf3 g6
			variation => N__('Sicilian: Grand Prix, 2... Nc6 3. Nf3 g6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/4PP2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21i',
			# TRANSLATORS: B21i: 1. e4 c5 2. f4 d5
			variation => N__('Sicilian: Grand Prix, Tal Defence'),
			moves => {
				'b1c3' => 1,
				'e4d5' => 1,
				'e4e5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B21i',
			# TRANSLATORS: B21i: 1. e4 c5 2. f4 d5 3. Nc3
			variation => N__('Sicilian: Grand Prix, Toilet Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2ppP3/5P2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B21i',
			# TRANSLATORS: B21i: 1. e4 c5 2. f4 d5 3. e5
			variation => N__('Sicilian: Grand Prix, Tal Defence, 3. e5'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pP4/5P2/8/PPPP2PP/RNBQKBNR b KQkq' => {
			eco => 'B21j',
			# TRANSLATORS: B21j: 1. e4 c5 2. f4 d5 3. exd5
			variation => N__('Sicilian: Grand Prix, Tal Defence, 3.exd5'),
			moves => {
				'd8d5' => 1,
				'g8f6' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/2pq4/5P2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21j',
			# TRANSLATORS: B21j: 1. e4 c5 2. f4 d5 3. exd5 Qxd5
			variation => N__('Sicilian: Grand Prix, Tal Defence, 3.exd5 Qxd5'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pP4/5P2/8/PPPP2PP/RNBQKBNR w KQkq' => {
			eco => 'B21k',
			# TRANSLATORS: B21k: 1. e4 c5 2. f4 d5 3. exd5 Nf6
			variation => N__('Sicilian: Grand Prix, Tal Gambit'),
			moves => {
				'f1b5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/1BpP4/5P2/8/PPPP2PP/RNBQK1NR b KQkq' => {
			eco => 'B21l',
			# TRANSLATORS: B21l: 1. e4 c5 2. f4 d5 3. exd5 Nf6 4. Bb5+
			variation => N__('Sicilian: Grand Prix, Tal Gambit, 4. Bb5+'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/3PP3/8/PPP2PPP/RNBQKBNR b KQkq' => {
			eco => 'B21m',
			# TRANSLATORS: B21m: 1. e4 c5 2. d4
			variation => N__('Sicilian: Smith-Morra Gambit'),
			moves => {
				'c5d4' => 1,
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/3pP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B21m',
			# TRANSLATORS: B21m: 1. e4 c5 2. d4 cxd4
			variation => N__('Sicilian: Smith-Morra, 2...cxd4'),
			moves => {
				'c2c3' => 1,
				'f2f4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/3pPP2/8/PPP3PP/RNBQKBNR b KQkq' => {
			eco => 'B21m',
			# TRANSLATORS: B21m: 1. e4 c5 2. d4 cxd4 3. f4
			variation => N__('Sicilian: Halasz Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/3PP3/8/PPP2PPP/RNBQKBNR w KQkq' => {
			eco => 'B21m',
			# TRANSLATORS: B21m: 1. e4 c5 2. d4 d5
			variation => N__('Sicilian: Smith-Morra, 2... d5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/3pP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B21n',
			# TRANSLATORS: B21n: 1. e4 c5 2. d4 cxd4 3. Nf3
			variation => N__('Sicilian: Smith-Morra, Morphy Gambit'),
			moves => {
				'e7e5' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/4p3/3pP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B21n',
			# TRANSLATORS: B21n: 1. e4 c5 2. d4 cxd4 3. Nf3 e5
			variation => N__('Sicilian: Smith-Morra, Morphy Gambit'),
			moves => {
				'c2c3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/4p3/3pP3/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B21n',
			# TRANSLATORS: B21n: 1. e4 c5 2. d4 cxd4 3. Nf3 e5 4. c3
			variation => N__('Sicilian: Smith-Morra, Andreaschek Gambit'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/3pP3/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B21o',
			# TRANSLATORS: B21o: 1. e4 c5 2. d4 cxd4 3. c3
			variation => N__('Sicilian: Smith-Morra, 3. c3'),
			moves => {
				'd4c3' => 1,
				'd4d3' => 1,
				'd7d5' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/8/3pP3/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B21p',
			# TRANSLATORS: B21p: 1. e4 c5 2. d4 cxd4 3. c3 Nf6
			variation => N__('Sicilian: Smith-Morra, 3. c3 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/3p4/3pP3/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B21q',
			# TRANSLATORS: B21q: 1. e4 c5 2. d4 cxd4 3. c3 d5
			variation => N__('Sicilian: Smith-Morra, 3. c3 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/4P3/2Pp4/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B21r',
			# TRANSLATORS: B21r: 1. e4 c5 2. d4 cxd4 3. c3 d3
			variation => N__('Sicilian: Smith-Morra, 3. c3 d3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/4P3/2p5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B21s',
			# TRANSLATORS: B21s: 1. e4 c5 2. d4 cxd4 3. c3 dxc3
			variation => N__('Sicilian: Smith-Morra Accepted'),
			moves => {
				'b1c3' => 1,
			},
		},
		'rnbqkbnr/pp1ppppp/8/8/4P3/2N5/PP3PPP/R1BQKBNR b KQkq' => {
			eco => 'B21s',
			# TRANSLATORS: B21s: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3
			variation => N__('Sicilian: Smith-Morra Accepted'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/3p4/8/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B21t',
			# TRANSLATORS: B21t: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 d6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 d6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/8/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B21u',
			# TRANSLATORS: B21u: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 e6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/8/4P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B21v',
			# TRANSLATORS: B21v: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 e6 5. Nf3
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 e6 5. Nf3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/8/4P3/2N5/PP3PPP/R1BQKBNR w KQkq' => {
			eco => 'B21w',
			# TRANSLATORS: B21w: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/8/4P3/2N2N2/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B21w',
			# TRANSLATORS: B21w: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6 5. Nf3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/8/4P3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B21x',
			# TRANSLATORS: B21x: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 e6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6 5. Nf3 e6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/8/2B1P3/2N2N2/PP3PPP/R1BQK2R b KQkq' => {
			eco => 'B21x',
			# TRANSLATORS: B21x: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 e6 6. Bc4
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6 5. Nf3 e6 6. Bc4'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqkbnr/1p1p1ppp/p1n1p3/8/2B1P3/2N2N2/PP3PPP/R1BQK2R w KQkq' => {
			eco => 'B21x',
			# TRANSLATORS: B21x: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 e6 6. Bc4 a6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6 5. Nf3 e6 6. Bc4 a6'),
			moves => {
			},
		},
		'r1bqkbnr/pp2pppp/2np4/8/4P3/2N2N2/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6
			variation => N__('Sicilian: Smith-Morra, 4.Nxc3 Nc6 5. Nf3 d6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2np4/8/2B1P3/2N2N2/PP3PPP/R1BQK2R b KQkq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4'),
			moves => {
				'a7a6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/1p2pppp/p1np4/8/2B1P3/2N2N2/PP3PPP/R1BQK2R w KQkq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 a6
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 a6'),
			moves => {
			},
		},
		'r1bqkbnr/pp3ppp/2npp3/8/2B1P3/2N2N2/PP3PPP/R1BQK2R w KQkq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2npp3/8/2B1P3/2N2N2/PP3PPP/R1BQ1RK1 b kq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O'),
			moves => {
				'a7a6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/1p3ppp/p1npp3/8/2B1P3/2N2N2/PP3PPP/R1BQ1RK1 w kq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O'),
			moves => {
				'd1e2' => 1,
			},
		},
		'r1bqkbnr/1p3ppp/p1npp3/8/2B1P3/2N2N2/PP2QPPP/R1B2RK1 b kq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6 8. Qe2
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O'),
			moves => {
				'b7b5' => 1,
			},
		},
		'r1bqkbnr/5ppp/p1npp3/1p6/2B1P3/2N2N2/PP2QPPP/R1B2RK1 w kq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6 8. Qe2 b5
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O'),
			moves => {
				'c4b3' => 1,
			},
		},
		'r1bqkbnr/5ppp/p1npp3/1p6/4P3/1BN2N2/PP2QPPP/R1B2RK1 b kq' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6 8. Qe2 b5 9. Bb3
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O'),
			moves => {
				'a8a7' => 1,
			},
		},
		'2bqkbnr/r4ppp/p1npp3/1p6/4P3/1BN2N2/PP2QPPP/R1B2RK1 w k' => {
			eco => 'B21y',
			# TRANSLATORS: B21y: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O a6 8. Qe2 b5 9. Bb3 Ra7
			variation => N__('Sicilian: Smith-Morra Gambit, Chicago Defence'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2nppn2/8/2B1P3/2N2N2/PP3PPP/R1BQ1RK1 w kq' => {
			eco => 'B21z',
			# TRANSLATORS: B21z: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O Nf6
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O Nf6'),
			moves => {
				'd1e2' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2nppn2/8/2B1P3/2N2N2/PP2QPPP/R1B2RK1 b kq' => {
			eco => 'B21z',
			# TRANSLATORS: B21z: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O Nf6 8. Qe2
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O Nf6'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/pp2bppp/2nppn2/8/2B1P3/2N2N2/PP2QPPP/R1B2RK1 w kq' => {
			eco => 'B21z',
			# TRANSLATORS: B21z: 1. e4 c5 2. d4 cxd4 3. c3 dxc3 4. Nxc3 Nc6 5. Nf3 d6 6. Bc4 e6 7. O-O Nf6 8. Qe2 Be7
			variation => N__('Sicilian: Smith-Morra, 5. Nf3 d6 6. Bc4 e6 7.O-O Nf6 8. Qe2 Be7'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B22a',
			# TRANSLATORS: B22a: 1. e4 c5 2. c3
			variation => N__('Sicilian: Alapin'),
			moves => {
				'b8c6' => 1,
				'd7d5' => 1,
				'd7d6' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/8/2p1p3/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22a',
			# TRANSLATORS: B22a: 1. e4 c5 2. c3 e5
			variation => N__('Sicilian: Alapin, 2... e5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22a',
			# TRANSLATORS: B22a: 1. e4 c5 2. c3 g6
			variation => N__('Sicilian: Alapin, 2... g6'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/3PP3/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22a',
			# TRANSLATORS: B22a: 1. e4 c5 2. c3 g6 3. d4
			variation => N__('Sicilian: Alapin, 2... g6 3. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/8/3pP3/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22a',
			# TRANSLATORS: B22a: 1. e4 c5 2. c3 g6 3. d4 cxd4
			variation => N__('Sicilian: Alapin, 2... g6 3. d4 cxd4'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22b',
			# TRANSLATORS: B22b: 1. e4 c5 2. c3 Nc6
			variation => N__('Sicilian: Alapin, 2... Nc6'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/3PP3/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22b',
			# TRANSLATORS: B22b: 1. e4 c5 2. c3 Nc6 3. d4
			variation => N__('Sicilian: Alapin, 2... Nc6 3. d4'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B22b',
			# TRANSLATORS: B22b: 1. e4 c5 2. c3 Nc6 3. Nf3
			variation => N__('Sicilian: Alapin, 2... Nc6 3. Nf3'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22c',
			# TRANSLATORS: B22c: 1. e4 c5 2. c3 e6
			variation => N__('Sicilian: Alapin, 2... e6'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B22c',
			# TRANSLATORS: B22c: 1. e4 c5 2. c3 e6 3. Nf3
			variation => N__('Sicilian: Alapin, 2... e6 3. Nf3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/4p3/2pp4/4P3/2P2N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B22c',
			# TRANSLATORS: B22c: 1. e4 c5 2. c3 e6 3. Nf3 d5
			variation => N__('Sicilian: Alapin, 2... e6 3. Nf3 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/3PP3/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4
			variation => N__('Sicilian: Alapin, 2... e6 3. d4'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/4p3/2pp4/3PP3/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4 d5
			variation => N__('Sicilian: Alapin, 2... e6 3. d4 d5'),
			moves => {
				'e4d5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/4p3/2pP4/3P4/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4 d5 4. exd5
			variation => N__('Sicilian: Alapin, 2... e6 3. d4 d5 4.exd5'),
			moves => {
				'e6d5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/8/2pp4/3P4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4 d5 4. exd5 exd5
			variation => N__('Sicilian: Alapin, 2... e6 3. d4 d5 4.exd5 exd5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/8/2pp4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4 d5 4. exd5 exd5 5. Nf3
			variation => N__('Sicilian: Alapin, 2... e6 3. d4 d5 4.exd5 exd5 5. Nf3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n5/2pp4/3P4/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22d',
			# TRANSLATORS: B22d: 1. e4 c5 2. c3 e6 3. d4 d5 4. exd5 exd5 5. Nf3 Nc6
			variation => N__('Sicilian: Alapin, 2... e6 3. d4 d5 4.exd5 exd5 5. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22e',
			# TRANSLATORS: B22e: 1. e4 c5 2. c3 d6
			variation => N__('Sicilian: Alapin, 2... d6'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/8/2pp4/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5
			variation => N__('Sicilian: Alapin, 2... d5'),
			moves => {
				'e4d5' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/8/2pP4/8/2P5/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5 3. exd5
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5'),
			moves => {
				'd8d5' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/2pq4/8/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5 3. exd5 Qxd5
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/2pq4/3P4/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4'),
			moves => {
				'b8c6' => 1,
				'c5d4' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/3q4/3p4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 cxd4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 cxd4'),
			moves => {
				'c3d4' => 1,
			},
		},
		'rnb1kbnr/pp2pppp/8/3q4/3P4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22f',
			# TRANSLATORS: B22f: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 cxd4 5. cxd4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 cxd4 5.cxd4'),
			moves => {
			},
		},
		'rnb1kbnr/pp3ppp/4p3/2pq4/3P4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22g',
			# TRANSLATORS: B22g: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 e6
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnb1kbnr/pp3ppp/4p3/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22g',
			# TRANSLATORS: B22g: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 e6 5. Nf3
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 e6 5. Nf3'),
			moves => {
			},
		},
		'r1b1kbnr/pp2pppp/2n5/2pq4/3P4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22h',
			# TRANSLATORS: B22h: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1b1kbnr/pp2pppp/2n5/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22h',
			# TRANSLATORS: B22h: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3'),
			moves => {
				'c5d4' => 1,
				'c8g4' => 1,
			},
		},
		'r1b1kbnr/pp2pppp/2n5/3q4/3p4/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22h',
			# TRANSLATORS: B22h: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4'),
			moves => {
				'c3d4' => 1,
			},
		},
		'r1b1kbnr/pp2pppp/2n5/3q4/3P4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22h',
			# TRANSLATORS: B22h: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4 6. cxd4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3 cxd4 6.cxd4'),
			moves => {
			},
		},
		'r3kbnr/pp2pppp/2n5/2pq4/3P2b1/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22i',
			# TRANSLATORS: B22i: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nc6 5. Nf3 Bg4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nc6 5. Nf3 Bg4'),
			moves => {
			},
		},
		'rnb1kb1r/pp2pppp/5n2/2pq4/3P4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22j',
			# TRANSLATORS: B22j: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnb1kb1r/pp2pppp/5n2/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22j',
			# TRANSLATORS: B22j: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3'),
			moves => {
				'c8g4' => 1,
				'e7e6' => 1,
			},
		},
		'rnb1kb1r/pp3ppp/4pn2/2pq4/3P4/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22j',
			# TRANSLATORS: B22j: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 e6
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3'),
			moves => {
				'c1e3' => 1,
				'f1e2' => 1,
			},
		},
		'rn2kb1r/pp2pppp/5n2/2pq4/3P2b1/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22k',
			# TRANSLATORS: B22k: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4'),
			moves => {
				'f1e2' => 1,
			},
		},
		'rn2kb1r/pp2pppp/5n2/2pq4/3P2b1/2P2N2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B22k',
			# TRANSLATORS: B22k: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4 6. Be2
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rn2kb1r/pp3ppp/4pn2/2pq4/3P2b1/2P2N2/PP2BPPP/RNBQK2R w KQkq' => {
			eco => 'B22l',
			# TRANSLATORS: B22l: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4 6. Be2 e6
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3 Bg4 6. Be2 e6'),
			moves => {
			},
		},
		'rnb1kb1r/pp3ppp/4pn2/2pq4/3P4/2P1BN2/PP3PPP/RN1QKB1R b KQkq' => {
			eco => 'B22m',
			# TRANSLATORS: B22m: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 e6 6. Be3
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3 e6 6. Be3'),
			moves => {
			},
		},
		'rnb1kb1r/pp3ppp/4pn2/2pq4/3P4/2P2N2/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B22n',
			# TRANSLATORS: B22n: 1. e4 c5 2. c3 d5 3. exd5 Qxd5 4. d4 Nf6 5. Nf3 e6 6. Be2
			variation => N__('Sicilian: Alapin, 2... d5 3.exd5 Qxd5 4. d4 Nf6 5. Nf3 e6 6. Be2'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/4P3/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22o',
			# TRANSLATORS: B22o: 1. e4 c5 2. c3 Nf6
			variation => N__('Sicilian: Alapin, 2... Nf6'),
			moves => {
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p1P3/8/2P5/PP1P1PPP/RNBQKBNR b KQkq' => {
			eco => 'B22o',
			# TRANSLATORS: B22o: 1. e4 c5 2. c3 Nf6 3. e5
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/8/2P5/PP1P1PPP/RNBQKBNR w KQkq' => {
			eco => 'B22o',
			# TRANSLATORS: B22o: 1. e4 c5 2. c3 Nf6 3. e5 Nd5
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5'),
			moves => {
				'd2d4' => 1,
				'g1f3' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/8/2P3P1/PP1P1P1P/RNBQKBNR b KQkq' => {
			eco => 'B22o',
			# TRANSLATORS: B22o: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. g3
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. g3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/3P4/2P5/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22p',
			# TRANSLATORS: B22p: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/8/2P2N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B22p',
			# TRANSLATORS: B22p: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. Nf3
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. Nf3'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/2pnP3/8/2P2N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B22p',
			# TRANSLATORS: B22p: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. Nf3 Nc6
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. Nf3 Nc6'),
			moves => {
				'b1a3' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/2pnP3/8/N1P2N2/PP1P1PPP/R1BQKB1R b KQkq' => {
			eco => 'B22p',
			# TRANSLATORS: B22p: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. Nf3 Nc6 5. Na3
			variation => N__('Sicilian: Alapin, Heidenfeld Variation'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/3nP3/3p4/2P5/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22q',
			# TRANSLATORS: B22q: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4 cxd4'),
			moves => {
				'c3d4' => 1,
				'd1d4' => 1,
				'f1c4' => 1,
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/3nP3/3Q4/2P5/PP3PPP/RNB1KBNR b KQkq' => {
			eco => 'B22q',
			# TRANSLATORS: B22q: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Qxd4
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.Qxd4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/3nP3/2Bp4/2P5/PP3PPP/RNBQK1NR b KQkq' => {
			eco => 'B22q',
			# TRANSLATORS: B22q: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Bc4
			variation => N__('Sicilian: Alapin, 2... Nf6, 5. Bc4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/3nP3/3p4/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22r',
			# TRANSLATORS: B22r: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3'),
			moves => {
				'b8c6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/3nP3/3p4/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22r',
			# TRANSLATORS: B22r: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 Nc6
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 Nc6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4p3/3nP3/3p4/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22r',
			# TRANSLATORS: B22r: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 e6
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 e6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/3nP3/2Bp4/2P2N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B22s',
			# TRANSLATORS: B22s: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 Nc6 6. Bc4
			variation => N__('Sicilian: Alapin, 2... Nf6 3. e5 Nd5 4. d4 cxd4 5. Nf3 Nc6 6. Bc4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/3nP3/3P4/8/PP3PPP/RNBQKBNR b KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/3nP3/3P4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 Nc6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 Nc6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n5/3nP3/3P4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 Nc6 6. Nf3
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 Nc6 6. Nf3'),
			moves => {
			},
		},
		'rnbqkb1r/pp2pppp/3p4/3nP3/3P4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/3p4/3nP3/3P4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6 6. Nf3
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2np4/3nP3/3P4/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22t',
			# TRANSLATORS: B22t: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6 6. Nf3 Nc6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6 6. Nf3 Nc6'),
			moves => {
				'f1c4' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/2np4/3nP3/2BP4/5N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B22u',
			# TRANSLATORS: B22u: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6 6. Nf3 Nc6 7. Bc4
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6 6. Nf3 Nc6 7. Bc4'),
			moves => {
				'd5b6' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/1nnp4/4P3/2BP4/5N2/PP3PPP/RNBQK2R w KQkq' => {
			eco => 'B22u',
			# TRANSLATORS: B22u: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6 6. Nf3 Nc6 7. Bc4 Nb6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6 6. Nf3 Nc6 7. Bc4 Nb6'),
			moves => {
				'c4b5' => 1,
			},
		},
		'r1bqkb1r/pp2pppp/1nnp4/1B2P3/3P4/5N2/PP3PPP/RNBQK2R b KQkq' => {
			eco => 'B22u',
			# TRANSLATORS: B22u: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 d6 6. Nf3 Nc6 7. Bc4 Nb6 8. Bb5
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 d6 6. Nf3 Nc6 7. Bc4 Nb6 8. Bb5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4p3/3nP3/3P4/8/PP3PPP/RNBQKBNR w KQkq' => {
			eco => 'B22v',
			# TRANSLATORS: B22v: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 e6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4p3/3nP3/3P4/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B22v',
			# TRANSLATORS: B22v: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 e6 6. Nf3
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 e6 6. Nf3'),
			moves => {
				'b7b6' => 1,
				'b8c6' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n1p3/3nP3/3P4/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22v',
			# TRANSLATORS: B22v: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 e6 6. Nf3 Nc6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 e6 6. Nf3 Nc6'),
			moves => {
			},
		},
		'rnbqkb1r/pp3ppp/3pp3/3nP3/3P4/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22w',
			# TRANSLATORS: B22w: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 e6 6. Nf3 d6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 e6 6. Nf3 d6'),
			moves => {
			},
		},
		'rnbqkb1r/p2p1ppp/1p2p3/3nP3/3P4/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B22x',
			# TRANSLATORS: B22x: 1. e4 c5 2. c3 Nf6 3. e5 Nd5 4. d4 cxd4 5. cxd4 e6 6. Nf3 b6
			variation => N__('Sicilian: Alapin, 2... Nf6, 5.cxd4 e6 6. Nf3 b6'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/2N5/PPPP1PPP/R1BQKBNR b KQkq' => {
			eco => 'B23a',
			# TRANSLATORS: B23a: 1. e4 c5 2. Nc3
			variation => N__('Sicilian: Closed'),
			moves => {
				'b8c6' => 1,
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B23b',
			# TRANSLATORS: B23b: 1. e4 c5 2. Nc3 g6
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B23c',
			# TRANSLATORS: B23c: 1. e4 c5 2. Nc3 d6
			variation => N__('Sicilian: Closed'),
			moves => {
				'f2f4' => 1,
				'g1e2' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4P3/2N5/PPPPNPPP/R1BQKB1R b KQkq' => {
			eco => 'B23c',
			# TRANSLATORS: B23c: 1. e4 c5 2. Nc3 d6 3. Nge2
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4P3/2N3P1/PPPP1P1P/R1BQKBNR b KQkq' => {
			eco => 'B23d',
			# TRANSLATORS: B23d: 1. e4 c5 2. Nc3 d6 3. g3
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp2pppp/3p4/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B23e',
			# TRANSLATORS: B23e: 1. e4 c5 2. Nc3 d6 3. f4
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B23f',
			# TRANSLATORS: B23f: 1. e4 c5 2. Nc3 e6
			variation => N__('Sicilian: Closed'),
			moves => {
				'f2f4' => 1,
				'g1e2' => 1,
				'g2g3' => 1,
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B23g',
			# TRANSLATORS: B23g: 1. e4 c5 2. Nc3 e6 3. f4
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/2N5/PPPPNPPP/R1BQKB1R b KQkq' => {
			eco => 'B23h',
			# TRANSLATORS: B23h: 1. e4 c5 2. Nc3 e6 3. Nge2
			variation => N__('Sicilian: Closed'),
			moves => {
			},
		},
		'rnbqkbnr/pp1p1ppp/4p3/2p5/4P3/2N3P1/PPPP1P1P/R1BQKBNR b KQkq' => {
			eco => 'B23i',
			# TRANSLATORS: B23i: 1. e4 c5 2. Nc3 e6 3. g3
			variation => N__('Sicilian: Closed'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkbnr/pp3ppp/4p3/2pp4/4P3/2N3P1/PPPP1P1P/R1BQKBNR w KQkq' => {
			eco => 'B23j',
			# TRANSLATORS: B23j: 1. e4 c5 2. Nc3 e6 3. g3 d5
			variation => N__('Sicilian: Closed, Korchnoi Variation'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2N5/PPPP1PPP/R1BQKBNR w KQkq' => {
			eco => 'B23k',
			# TRANSLATORS: B23k: 1. e4 c5 2. Nc3 Nc6
			variation => N__('Sicilian: Closed, 2... Nc6'),
			moves => {
				'f1b5' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/1Bp5/4P3/2N5/PPPP1PPP/R1BQK1NR b KQkq' => {
			eco => 'B23l',
			# TRANSLATORS: B23l: 1. e4 c5 2. Nc3 Nc6 3. Bb5
			variation => N__('Sicilian: Closed, 2... Nc6 3. Bb5'),
			moves => {
				'c6d4' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/8/1Bp5/3nP3/2N5/PPPP1PPP/R1BQK1NR w KQkq' => {
			eco => 'B23m',
			# TRANSLATORS: B23m: 1. e4 c5 2. Nc3 Nc6 3. Bb5 Nd4
			variation => N__('Sicilian: Closed, 2... Nc6 3. Bb5 Nd4'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2N5/PPPPNPPP/R1BQKB1R b KQkq' => {
			eco => 'B23n',
			# TRANSLATORS: B23n: 1. e4 c5 2. Nc3 Nc6 3. Nge2
			variation => N__('Sicilian: Chameleon Variation'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n5/2p1p3/4P3/2N5/PPPPNPPP/R1BQKB1R w KQkq' => {
			eco => 'B23n',
			# TRANSLATORS: B23n: 1. e4 c5 2. Nc3 Nc6 3. Nge2 e5
			variation => N__('Sicilian: Chameleon, 3... e5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4P3/2N5/PPPPNPPP/R1BQKB1R w KQkq' => {
			eco => 'B23n',
			# TRANSLATORS: B23n: 1. e4 c5 2. Nc3 Nc6 3. Nge2 g6
			variation => N__('Sicilian: Chameleon, 3... g6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR b KQkq' => {
			eco => 'B23o',
			# TRANSLATORS: B23o: 1. e4 c5 2. Nc3 Nc6 3. f4
			variation => N__('Sicilian: Closed, Grand Prix'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2np4/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'B23o',
			# TRANSLATORS: B23o: 1. e4 c5 2. Nc3 Nc6 3. f4 d6
			variation => N__('Sicilian: Closed, Grand Prix, 3... d6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2np4/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R b KQkq' => {
			eco => 'B23o',
			# TRANSLATORS: B23o: 1. e4 c5 2. Nc3 Nc6 3. f4 d6 4. Nf3
			variation => N__('Sicilian: Closed, Grand Prix, 3... d6'),
			moves => {
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2np2p1/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq' => {
			eco => 'B23p',
			# TRANSLATORS: B23p: 1. e4 c5 2. Nc3 Nc6 3. f4 d6 4. Nf3 g6
			variation => N__('Sicilian: Closed, Grand Prix, 3... d6 4. Nf3 g6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'B23q',
			# TRANSLATORS: B23q: 1. e4 c5 2. Nc3 Nc6 3. f4 e6
			variation => N__('Sicilian: Closed, Grand Prix, 3... e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R b KQkq' => {
			eco => 'B23q',
			# TRANSLATORS: B23q: 1. e4 c5 2. Nc3 Nc6 3. f4 e6 4. Nf3
			variation => N__('Sicilian: Closed, Grand Prix, 3... e6 4. Nf3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n1p3/2pp4/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq' => {
			eco => 'B23r',
			# TRANSLATORS: B23r: 1. e4 c5 2. Nc3 Nc6 3. f4 e6 4. Nf3 d5
			variation => N__('Sicilian: Closed, Grand Prix, 3... e6 4. Nf3 d5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4PP2/2N5/PPPP2PP/R1BQKBNR w KQkq' => {
			eco => 'B23s',
			# TRANSLATORS: B23s: 1. e4 c5 2. Nc3 Nc6 3. f4 g6
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R b KQkq' => {
			eco => 'B23s',
			# TRANSLATORS: B23s: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4PP2/2N2N2/PPPP2PP/R1BQKB1R w KQkq' => {
			eco => 'B23s',
			# TRANSLATORS: B23s: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6'),
			moves => {
				'f1b5' => 1,
				'f1c4' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq' => {
			eco => 'B23t',
			# TRANSLATORS: B23t: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bc4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/2B1PP2/2N2N2/PPPP2PP/R1BQK2R w KQkq' => {
			eco => 'B23u',
			# TRANSLATORS: B23u: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4 e6
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bc4 e6'),
			moves => {
				'f4f5' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p2P2/2B1P3/2N2N2/PPPP2PP/R1BQK2R b KQkq' => {
			eco => 'B23v',
			# TRANSLATORS: B23v: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bc4 e6 6. f5
			variation => N__('Sicilian: Closed, Grand Prix, Schofman Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4PP2/2N2N2/PPPP2PP/R1BQK2R b KQkq' => {
			eco => 'B23w',
			# TRANSLATORS: B23w: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5'),
			moves => {
				'c6d4' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQK2R w KQkq' => {
			eco => 'B23w',
			# TRANSLATORS: B23w: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4'),
			moves => {
				'b5d3' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/2p5/3nPP2/2NB1N2/PPPP2PP/R1BQK2R b KQkq' => {
			eco => 'B23x',
			# TRANSLATORS: B23x: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4 6. Bd3
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4 6. Bd3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/6p1/1Bp5/3nPP2/2N2N2/PPPP2PP/R1BQ1RK1 b kq' => {
			eco => 'B23y',
			# TRANSLATORS: B23y: 1. e4 c5 2. Nc3 Nc6 3. f4 g6 4. Nf3 Bg7 5. Bb5 Nd4 6. O-O
			variation => N__('Sicilian: Closed, Grand Prix, 3... g6 4. Nf3 Bg7 5. Bb5 Nd4 6.O-O'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2N3P1/PPPP1P1P/R1BQKBNR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3
			variation => N__('Sicilian: Closed, 3. g3'),
			moves => {
				'e7e6' => 1,
				'g7g6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4P3/2N3P1/PPPP1P1P/R1BQKBNR w KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 e6
			variation => N__('Sicilian: Closed, 3. g3 e6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4P3/2N3P1/PPPP1P1P/R1BQKBNR w KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6
			variation => N__('Sicilian: Closed, 3. g3 g6'),
			moves => {
				'd2d3' => 1,
				'f1g2' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4P3/2NP2P1/PPP2P1P/R1BQKBNR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. d3
			variation => N__('Sicilian: Closed, 3. g3 g6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4P3/2N3P1/PPPP1PBP/R1BQK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2
			variation => N__('Sicilian: Closed, 3. g3 g6'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2N3P1/PPPP1PBP/R1BQK1NR w KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7
			variation => N__('Sicilian: Closed, 3. g3 g6'),
			moves => {
				'd2d3' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3'),
			moves => {
				'd7d6' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c1e3' => 1,
				'f2f4' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. Be3
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c6d4' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. Be3 Nd4
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6'),
			moves => {
				'c3e2' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/4p1p1/2p5/3nP3/3PB1P1/PPP1NPBP/R2QK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. Be3 Nd4 7. Nce2
			variation => N__('Sicilian: Closed, Smyslov Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n1p1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 e6 6. f4
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. d3 e6 6. f4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4PP2/2N3P1/PPPP2BP/R1BQK1NR b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. f4
			variation => N__('Sicilian: Closed, 3. g3 g6, 5. f4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/2p5/4P3/2N3P1/PPPPNPBP/R1BQK2R b KQkq' => {
			eco => 'B24',
			# TRANSLATORS: B24: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. Nge2
			variation => N__('Sicilian: Closed, 3. g3 g6, 5.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP2PBP/R1BQK1NR w KQkq' => {
			eco => 'B25a',
			# TRANSLATORS: B25a: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6
			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6'),
			moves => {
				'c1e3' => 1,
				'f2f4' => 1,
				'g1e2' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R b KQkq' => {
			eco => 'B25b',
			# TRANSLATORS: B25b: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2
			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6 6.Nge2'),
			moves => {
				'e7e5' => 1,
				'e7e6' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq' => {
			eco => 'B25c',
			# TRANSLATORS: B25c: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2 e5
			variation => N__('Sicilian: Closed, Botvinnik'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NP2P1/PPP1NPBP/R1BQK2R w KQkq' => {
			eco => 'B25d',
			# TRANSLATORS: B25d: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Nge2 e6
			variation => N__('Sicilian: Closed, 3. g3, 5. d3 d6 6.Nge2 e6'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR b KQkq' => {
			eco => 'B25e',
			# TRANSLATORS: B25e: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4
			variation => N__('Sicilian: Closed, 6. f4'),
			moves => {
				'a8b8' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQk' => {
			eco => 'B25f',
			# TRANSLATORS: B25f: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Rb8
			variation => N__('Sicilian: Closed, 6. f4 Rb8'),
			moves => {
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq' => {
			eco => 'B25g',
			# TRANSLATORS: B25g: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Nf6
			variation => N__('Sicilian: Closed, 6. f4 Nf6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq' => {
			eco => 'B25h',
			# TRANSLATORS: B25h: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 Nf6 7. Nf3
			variation => N__('Sicilian: Closed, 6. f4 Nf6 7. Nf3'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq' => {
			eco => 'B25i',
			# TRANSLATORS: B25i: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5
			variation => N__('Sicilian: Closed, 6. f4 e5'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq' => {
			eco => 'B25j',
			# TRANSLATORS: B25j: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3
			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq' => {
			eco => 'B25j',
			# TRANSLATORS: B25j: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3 Nge7
			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3 Nge7'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq' => {
			eco => 'B25k',
			# TRANSLATORS: B25k: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e5 7. Nf3 Nge7 8. O-O
			variation => N__('Sicilian: Closed, 6. f4 e5 7. Nf3 Nge7 8.O-O'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP2P1/PPP3BP/R1BQK1NR w KQkq' => {
			eco => 'B25l',
			# TRANSLATORS: B25l: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6
			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'g1f3' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R b KQkq' => {
			eco => 'B25m',
			# TRANSLATORS: B25m: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3
			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQK2R w KQkq' => {
			eco => 'B25n',
			# TRANSLATORS: B25n: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7
			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 b kq' => {
			eco => 'B25o',
			# TRANSLATORS: B25o: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O
			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NP1NP1/PPP3BP/R1BQ1RK1 w -' => {
			eco => 'B25p',
			# TRANSLATORS: B25p: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O
			variation => N__('Sicilian: Closed, 6. f4 e6'),
			moves => {
				'c1e3' => 1,
			},
		},
		'r1bq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 b -' => {
			eco => 'B25q',
			# TRANSLATORS: B25q: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3'),
			moves => {
				'a8b8' => 1,
				'b7b6' => 1,
				'c6d4' => 1,
			},
		},
		'1rbq1rk1/pp2npbp/2npp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w -' => {
			eco => 'B25r',
			# TRANSLATORS: B25r: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Rb8
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Rb8'),
			moves => {
			},
		},
		'r1bq1rk1/p3npbp/1pnpp1p1/2p5/4PP2/2NPBNP1/PPP3BP/R2Q1RK1 w -' => {
			eco => 'B25s',
			# TRANSLATORS: B25s: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 b6
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 b6'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/R2Q1RK1 w -' => {
			eco => 'B25t',
			# TRANSLATORS: B25t: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4'),
			moves => {
				'a1b1' => 1,
				'd1d2' => 1,
				'e4e5' => 1,
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPP3BP/1R1Q1RK1 b -' => {
			eco => 'B25u',
			# TRANSLATORS: B25u: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. Rb1
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. Rb1'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p5/3nPP2/2NPBNP1/PPPQ2BP/R4RK1 b -' => {
			eco => 'B25v',
			# TRANSLATORS: B25v: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. Qd2
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. Qd2'),
			moves => {
			},
		},
		'r1bq1rk1/pp2npbp/3pp1p1/2p1P3/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 b -' => {
			eco => 'B25w',
			# TRANSLATORS: B25w: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. e5
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. e5'),
			moves => {
				'e7f5' => 1,
			},
		},
		'r1bq1rk1/pp3pbp/3pp1p1/2p1Pn2/3n1P2/2NPBNP1/PPP3BP/R2Q1RK1 w -' => {
			eco => 'B25x',
			# TRANSLATORS: B25x: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. f4 e6 7. Nf3 Nge7 8. O-O O-O 9. Be3 Nd4 10. e5 Nef5
			variation => N__('Sicilian: Closed, 6. f4 e6: 9. Be3 Nd4 10. e5 Nef5'),
			moves => {
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR b KQkq' => {
			eco => 'B26a',
			# TRANSLATORS: B26a: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3
			variation => N__('Sicilian: Closed, 6. Be3'),
			moves => {
				'a8b8' => 1,
				'e7e5' => 1,
				'e7e6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq' => {
			eco => 'B26b',
			# TRANSLATORS: B26b: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Nf6
			variation => N__('Sicilian: Closed, 6. Be3 Nf6'),
			moves => {
				'h2h3' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/2np1np1/2p5/4P3/2NPB1PP/PPP2PB1/R2QK1NR b KQkq' => {
			eco => 'B26c',
			# TRANSLATORS: B26c: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Nf6 7. h3
			variation => N__('Sicilian: Closed, 6. Be3 Nf6 7. h3'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq' => {
			eco => 'B26d',
			# TRANSLATORS: B26d: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5
			variation => N__('Sicilian: Closed, 6. Be3 e5'),
			moves => {
				'd1d2' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq' => {
			eco => 'B26e',
			# TRANSLATORS: B26e: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5 7. Qd2
			variation => N__('Sicilian: Closed, 6. Be3 e5 7. Qd2'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqk2r/pp2npbp/2np2p1/2p1p3/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq' => {
			eco => 'B26f',
			# TRANSLATORS: B26f: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e5 7. Qd2 Nge7
			variation => N__('Sicilian: Closed, 6. Be3 e5 7. Qd2 Nge7'),
			moves => {
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQk' => {
			eco => 'B26g',
			# TRANSLATORS: B26g: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8
			variation => N__('Sicilian: Closed, 6. Be3 Rb8'),
			moves => {
				'd1d2' => 1,
			},
		},
		'1rbqk1nr/pp2ppbp/2np2p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQk' => {
			eco => 'B26h',
			# TRANSLATORS: B26h: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2
			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2'),
			moves => {
				'b7b5' => 1,
			},
		},
		'1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk' => {
			eco => 'B26i',
			# TRANSLATORS: B26i: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2 b5
			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2 b5'),
			moves => {
				'g1e2' => 1,
			},
		},
		'1rbqk1nr/p3ppbp/2np2p1/1pp5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk' => {
			eco => 'B26j',
			# TRANSLATORS: B26j: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 Rb8 7. Qd2 b5 8. Nge2
			variation => N__('Sicilian: Closed, 6. Be3 Rb8 7. Qd2 b5 8.Nge2'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPP2PBP/R2QK1NR w KQkq' => {
			eco => 'B26k',
			# TRANSLATORS: B26k: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6
			variation => N__('Sicilian: Closed, 6. Be3 e6'),
			moves => {
				'd1d2' => 1,
			},
		},
		'r1bqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR b KQkq' => {
			eco => 'B26l',
			# TRANSLATORS: B26l: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2'),
			moves => {
				'a8b8' => 1,
				'c6d4' => 1,
				'd8a5' => 1,
				'g8e7' => 1,
			},
		},
		'r1b1k1nr/pp3pbp/2npp1p1/q1p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq' => {
			eco => 'B26m',
			# TRANSLATORS: B26m: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Qa5
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Qa5'),
			moves => {
			},
		},
		'r1bqk1nr/pp3pbp/3pp1p1/2p5/3nP3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq' => {
			eco => 'B26n',
			# TRANSLATORS: B26n: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Nd4
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Nd4'),
			moves => {
			},
		},
		'r1bqk2r/pp2npbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQkq' => {
			eco => 'B26o',
			# TRANSLATORS: B26o: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Nge7
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Nge7'),
			moves => {
			},
		},
		'1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQ1PBP/R3K1NR w KQk' => {
			eco => 'B26p',
			# TRANSLATORS: B26p: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Rb8
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Rb8'),
			moves => {
				'g1e2' => 1,
			},
		},
		'1rbqk1nr/pp3pbp/2npp1p1/2p5/4P3/2NPB1P1/PPPQNPBP/R3K2R b KQk' => {
			eco => 'B26q',
			# TRANSLATORS: B26q: 1. e4 c5 2. Nc3 Nc6 3. g3 g6 4. Bg2 Bg7 5. d3 d6 6. Be3 e6 7. Qd2 Rb8 8. Nge2
			variation => N__('Sicilian: Closed, 6. Be3 e6 7. Qd2 Rb8 8.Nge2'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppppp/8/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B27a',
			# TRANSLATORS: B27a: 1. e4 c5 2. Nf3
			variation => N__('Sicilian: 2. Nf3'),
			moves => {
				'a7a6' => 1,
				'b7b6' => 1,
				'b8c6' => 1,
				'd7d6' => 1,
				'd8a5' => 1,
				'd8c7' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'rnb1kbnr/pp1ppppp/8/q1p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B27b',
			# TRANSLATORS: B27b: 1. e4 c5 2. Nf3 Qa5
			variation => N__('Sicilian: Mongoose Variation'),
			moves => {
			},
		},
		'rnb1kbnr/ppqppppp/8/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B27c',
			# TRANSLATORS: B27c: 1. e4 c5 2. Nf3 Qc7
			variation => N__('Sicilian: Quinteros Variation'),
			moves => {
			},
		},
		'rnbqkbnr/p2ppppp/1p6/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B27d',
			# TRANSLATORS: B27d: 1. e4 c5 2. Nf3 b6
			variation => N__('Sicilian: Katalimov Variation'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B27e',
			# TRANSLATORS: B27e: 1. e4 c5 2. Nf3 g6
			variation => N__('Sicilian: Hungarian Variation'),
			moves => {
				'c2c3' => 1,
				'c2c4' => 1,
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/2P1P3/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B27f',
			# TRANSLATORS: B27f: 1. e4 c5 2. Nf3 g6 3. c4
			variation => N__('Sicilian: Hungarian, 3. c4'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B27g',
			# TRANSLATORS: B27g: 1. e4 c5 2. Nf3 g6 3. c3
			variation => N__('Sicilian: Hungarian, 3. c3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B27g',
			# TRANSLATORS: B27g: 1. e4 c5 2. Nf3 g6 3. c3 Bg7
			variation => N__('Sicilian: Hungarian, 3. c3'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/3PP3/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B27g',
			# TRANSLATORS: B27g: 1. e4 c5 2. Nf3 g6 3. c3 Bg7 4. d4
			variation => N__('Sicilian: Hungarian, 3. c3'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/8/3pP3/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B27g',
			# TRANSLATORS: B27g: 1. e4 c5 2. Nf3 g6 3. c3 Bg7 4. d4 cxd4
			variation => N__('Sicilian: Hungarian, 3. c3'),
			moves => {
				'c3d4' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/8/3PP3/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B27h',
			# TRANSLATORS: B27h: 1. e4 c5 2. Nf3 g6 3. c3 Bg7 4. d4 cxd4 5. cxd4
			variation => N__('Sicilian: Hungarian, 3. c3 Bg7 4. d4 cxd4 5.cxd4'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqk1nr/pp2ppbp/6p1/3p4/3PP3/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B27i',
			# TRANSLATORS: B27i: 1. e4 c5 2. Nf3 g6 3. c3 Bg7 4. d4 cxd4 5. cxd4 d5
			variation => N__('Sicilian: Hungarian, 3. c3 Bg7 4. d4 cxd4 5.cxd4 d5'),
			moves => {
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/2p5/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B27j',
			# TRANSLATORS: B27j: 1. e4 c5 2. Nf3 g6 3. d4
			variation => N__('Sicilian: Hungarian, 3. d4'),
			moves => {
				'c5d4' => 1,
				'f8g7' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/8/3pP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B27k',
			# TRANSLATORS: B27k: 1. e4 c5 2. Nf3 g6 3. d4 cxd4
			variation => N__('Sicilian: Hungarian, 3. d4 cxd4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'rnbqkbnr/pp1ppp1p/6p1/8/3NP3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B27l',
			# TRANSLATORS: B27l: 1. e4 c5 2. Nf3 g6 3. d4 cxd4 4. Nxd4
			variation => N__('Sicilian: Hungarian, 3. d4 cxd4 4.Nxd4'),
			moves => {
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/3PP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B27m',
			# TRANSLATORS: B27m: 1. e4 c5 2. Nf3 g6 3. d4 Bg7
			variation => N__('Sicilian: Accelerated Pterodactyl'),
			moves => {
				'b1c3' => 1,
				'c2c4' => 1,
				'd4c5' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2P5/4P3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B27n',
			# TRANSLATORS: B27n: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. dxc5
			variation => N__('Sicilian: Accelerated Pterodactyl, 4.dxc5'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnb1k1nr/pp1pppbp/6p1/q1P5/4P3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B27o',
			# TRANSLATORS: B27o: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. dxc5 Qa5+
			variation => N__('Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+'),
			moves => {
				'b1c3' => 1,
				'c2c3' => 1,
			},
		},
		'rnb1k1nr/pp1pppbp/6p1/q1P5/4P3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B27p',
			# TRANSLATORS: B27p: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. dxc5 Qa5+ 5. Nc3
			variation => N__('Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5. Nc3'),
			moves => {
			},
		},
		'rnb1k1nr/pp1pppbp/6p1/q1P5/4P3/2P2N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B27q',
			# TRANSLATORS: B27q: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. dxc5 Qa5+ 5. c3
			variation => N__('Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5. c3'),
			moves => {
				'a5c5' => 1,
			},
		},
		'rnb1k1nr/pp1pppbp/6p1/2q5/4P3/2P2N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B27r',
			# TRANSLATORS: B27r: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. dxc5 Qa5+ 5. c3 Qxc5
			variation => N__('Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5. c3'),
			moves => {
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/2PPP3/5N2/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B27s',
			# TRANSLATORS: B27s: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. c4
			variation => N__('Sicilian: Accelerated Pterodactyl, 4. c4'),
			moves => {
				'd8a5' => 1,
			},
		},
		'rnbqk1nr/pp1pppbp/6p1/2p5/3PP3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B27t',
			# TRANSLATORS: B27t: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. Nc3
			variation => N__('Sicilian: Accelerated Pterodactyl, 4. Nc3'),
			moves => {
			},
		},
		'rnb1k1nr/pp1pppbp/6p1/q1p5/2PPP3/5N2/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B27t',
			# TRANSLATORS: B27t: 1. e4 c5 2. Nf3 g6 3. d4 Bg7 4. c4 Qa5+
			variation => N__('Sicilian: Accelerated Pterodactyl, 4. c4 Qa5+'),
			moves => {
			},
		},
		'rnbqkbnr/1p1ppppp/p7/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B28a',
			# TRANSLATORS: B28a: 1. e4 c5 2. Nf3 a6
			variation => N__('Sicilian: O\'Kelly Variation'),
			moves => {
				'b1c3' => 1,
				'c2c3' => 1,
				'c2c4' => 1,
				'd2d4' => 1,
			},
		},
		'rnbqkbnr/1p1ppppp/p7/2p5/4P3/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B28b',
			# TRANSLATORS: B28b: 1. e4 c5 2. Nf3 a6 3. Nc3
			variation => N__('Sicilian: O\'Kelly, 3. Nc3'),
			moves => {
			},
		},
		'rnbqkbnr/1p1ppppp/p7/2p5/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B28c',
			# TRANSLATORS: B28c: 1. e4 c5 2. Nf3 a6 3. d4
			variation => N__('Sicilian: O\'Kelly, 3. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'rnbqkbnr/1p1ppppp/p7/8/3pP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B28c',
			# TRANSLATORS: B28c: 1. e4 c5 2. Nf3 a6 3. d4 cxd4
			variation => N__('Sicilian: O\'Kelly, 3. d4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'rnbqkbnr/1p1ppppp/p7/8/3NP3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B28d',
			# TRANSLATORS: B28d: 1. e4 c5 2. Nf3 a6 3. d4 cxd4 4. Nxd4
			variation => N__('Sicilian: O\'Kelly, 3. d4 cxd4 4.Nxd4'),
			moves => {
				'g8f6' => 1,
			},
		},
		'rnbqkb1r/1p1ppppp/p4n2/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B28e',
			# TRANSLATORS: B28e: 1. e4 c5 2. Nf3 a6 3. d4 cxd4 4. Nxd4 Nf6
			variation => N__('Sicilian: O\'Kelly, 3. d4 cxd4 4.Nxd4 Nf6'),
			moves => {
			},
		},
		'rnbqkbnr/1p1ppppp/p7/2p5/2P1P3/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B28f',
			# TRANSLATORS: B28f: 1. e4 c5 2. Nf3 a6 3. c4
			variation => N__('Sicilian: O\'Kelly, 3. c4'),
			moves => {
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/1p1p1ppp/p3p3/2p5/2P1P3/5N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B28g',
			# TRANSLATORS: B28g: 1. e4 c5 2. Nf3 a6 3. c4 e6
			variation => N__('Sicilian: O\'Kelly, 3. c4 e6'),
			moves => {
			},
		},
		'rnbqkbnr/1p1ppppp/p7/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B28h',
			# TRANSLATORS: B28h: 1. e4 c5 2. Nf3 a6 3. c3
			variation => N__('Sicilian: O\'Kelly, 3. c3'),
			moves => {
				'd7d5' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkbnr/1p1p1ppp/p3p3/2p5/4P3/2P2N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B28i',
			# TRANSLATORS: B28i: 1. e4 c5 2. Nf3 a6 3. c3 e6
			variation => N__('Sicilian: O\'Kelly, 3. c3 e6'),
			moves => {
			},
		},
		'rnbqkbnr/1p2pppp/p7/2pp4/4P3/2P2N2/PP1P1PPP/RNBQKB1R w KQkq' => {
			eco => 'B28j',
			# TRANSLATORS: B28j: 1. e4 c5 2. Nf3 a6 3. c3 d5
			variation => N__('Sicilian: O\'Kelly, 3. c3 d5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B29a',
			# TRANSLATORS: B29a: 1. e4 c5 2. Nf3 Nf6
			variation => N__('Sicilian: Nimzowitsch'),
			moves => {
				'b1c3' => 1,
				'd2d3' => 1,
				'e4e5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/4P3/3P1N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B29b',
			# TRANSLATORS: B29b: 1. e4 c5 2. Nf3 Nf6 3. d3
			variation => N__('Sicilian: Nimzowitsch, 3. d3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p5/4P3/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B29c',
			# TRANSLATORS: B29c: 1. e4 c5 2. Nf3 Nf6 3. Nc3
			variation => N__('Sicilian: Nimzowitsch, 3. Nc3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pp4/4P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B29d',
			# TRANSLATORS: B29d: 1. e4 c5 2. Nf3 Nf6 3. Nc3 d5
			variation => N__('Sicilian: Nimzowitsch, 3. Nc3 d5'),
			moves => {
				'e4d5' => 1,
			},
		},
		'rnbqkb1r/pp2pppp/5n2/2pP4/8/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B29e',
			# TRANSLATORS: B29e: 1. e4 c5 2. Nf3 Nf6 3. Nc3 d5 4. exd5
			variation => N__('Sicilian: Nimzowitsch, 3. Nc3 d5 4.exd5'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/5n2/2p1P3/8/5N2/PPPP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B29f',
			# TRANSLATORS: B29f: 1. e4 c5 2. Nf3 Nf6 3. e5
			variation => N__('Sicilian: Nimzowitsch, 3. e5'),
			moves => {
				'f6d5' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/8/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B29f',
			# TRANSLATORS: B29f: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5
			variation => N__('Sicilian: Nimzowitsch, 3. e5 Nd5'),
			moves => {
				'b1c3' => 1,
				'c2c4' => 1,
				'd2d4' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/2P5/5N2/PP1P1PPP/RNBQKB1R b KQkq' => {
			eco => 'B29g',
			# TRANSLATORS: B29g: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. c4
			variation => N__('Sicilian: Nimzowitsch, 4. c4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/3P4/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B29h',
			# TRANSLATORS: B29h: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. d4
			variation => N__('Sicilian: Nimzowitsch, 4. d4'),
			moves => {
			},
		},
		'rnbqkb1r/pp1ppppp/8/2pnP3/8/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B29i',
			# TRANSLATORS: B29i: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3'),
			moves => {
				'd5c3' => 1,
				'e7e6' => 1,
			},
		},
		'rnbqkb1r/pp1ppppp/8/2p1P3/8/2n2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B29j',
			# TRANSLATORS: B29j: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 Nxc3
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3 Nxc3'),
			moves => {
			},
		},
		'rnbqkb1r/pp1p1ppp/4p3/2pnP3/8/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B29k',
			# TRANSLATORS: B29k: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3 e6'),
			moves => {
				'c3d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/4p3/2pNP3/8/5N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B29l',
			# TRANSLATORS: B29l: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6 5. Nxd5
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3 e6 5.Nxd5'),
			moves => {
				'e6d5' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/8/2ppP3/8/5N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B29l',
			# TRANSLATORS: B29l: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6 5. Nxd5 exd5
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3 e6 5.Nxd5'),
			moves => {
				'd2d4' => 1,
			},
		},
		'rnbqkb1r/pp1p1ppp/8/2ppP3/3P4/5N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B29l',
			# TRANSLATORS: B29l: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6 5. Nxd5 exd5 6. d4
			variation => N__('Sicilian: Nimzowitsch, 4. Nc3 e6 5.Nxd5'),
			moves => {
				'b8c6' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n5/2ppP3/3P4/5N2/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B29m',
			# TRANSLATORS: B29m: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6 5. Nxd5 exd5 6. d4 Nc6
			variation => N__('Sicilian: Nimzowitsch, Rubinstein Countergambit'),
			moves => {
				'd4c5' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n5/2PpP3/8/5N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B29n',
			# TRANSLATORS: B29n: 1. e4 c5 2. Nf3 Nf6 3. e5 Nd5 4. Nc3 e6 5. Nxd5 exd5 6. d4 Nc6 7. dxc5
			variation => N__('Sicilian: Nimzowitsch, Rubinstein Countergambit, 7.dxc5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/5N2/PPPP1PPP/RNBQKB1R w KQkq' => {
			eco => 'B30a',
			# TRANSLATORS: B30a: 1. e4 c5 2. Nf3 Nc6
			variation => N__('Sicilian: 2... Nc6'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'd2d3' => 1,
				'd2d4' => 1,
				'f1b5' => 1,
				'f1c4' => 1,
				'g2g3' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/1P3N2/P1PP1PPP/RNBQKB1R b KQkq' => {
			eco => 'B30b',
			# TRANSLATORS: B30b: 1. e4 c5 2. Nf3 Nc6 3. b3
			variation => N__('Sicilian: 2... Nc6 3. b3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/3P1N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B30c',
			# TRANSLATORS: B30c: 1. e4 c5 2. Nf3 Nc6 3. d3
			variation => N__('Sicilian: 2... Nc6 3. d3'),
			moves => {
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/4P3/3P1N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B30c',
			# TRANSLATORS: B30c: 1. e4 c5 2. Nf3 Nc6 3. d3 Nf6
			variation => N__('Sicilian: 2... Nc6 3. d3 Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/5NP1/PPPP1P1P/RNBQKB1R b KQkq' => {
			eco => 'B30d',
			# TRANSLATORS: B30d: 1. e4 c5 2. Nf3 Nc6 3. g3
			variation => N__('Sicilian: 2... Nc6 3. g3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4P3/5NP1/PPPP1P1P/RNBQKB1R w KQkq' => {
			eco => 'B30d',
			# TRANSLATORS: B30d: 1. e4 c5 2. Nf3 Nc6 3. g3 e6
			variation => N__('Sicilian: 2... Nc6 3. g3'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/2p5/4P3/3P1NP1/PPP2P1P/RNBQKB1R b KQkq' => {
			eco => 'B30e',
			# TRANSLATORS: B30e: 1. e4 c5 2. Nf3 Nc6 3. g3 e6 4. d3
			variation => N__('Sicilian: 2... Nc6 3. g3 e6 4. d3'),
			moves => {
				'd7d5' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2n1p3/2pp4/4P3/3P1NP1/PPP2P1P/RNBQKB1R w KQkq' => {
			eco => 'B30f',
			# TRANSLATORS: B30f: 1. e4 c5 2. Nf3 Nc6 3. g3 e6 4. d3 d5
			variation => N__('Sicilian: 2... Nc6 3. g3 e6 4. d3 d5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/2B1P3/5N2/PPPP1PPP/RNBQK2R b KQkq' => {
			eco => 'B30g',
			# TRANSLATORS: B30g: 1. e4 c5 2. Nf3 Nc6 3. Bc4
			variation => N__('Sicilian: 2... Nc6 3. Bc4'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/4P3/2N2N2/PPPP1PPP/R1BQKB1R b KQkq' => {
			eco => 'B30h',
			# TRANSLATORS: B30h: 1. e4 c5 2. Nf3 Nc6 3. Nc3
			variation => N__('Sicilian: 2... Nc6 3. Nc3'),
			moves => {
				'e7e5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/2p5/4P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B30h',
			# TRANSLATORS: B30h: 1. e4 c5 2. Nf3 Nc6 3. Nc3 Nf6
			variation => N__('Sicilian: 2... Nc6 3. Nc3 Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/2p5/4P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B30i',
			# TRANSLATORS: B30i: 1. e4 c5 2. Nf3 Nc6 3. Nc3 g6
			variation => N__('Sicilian: 2... Nc6 3. Nc3 g6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n5/2p1p3/4P3/2N2N2/PPPP1PPP/R1BQKB1R w KQkq' => {
			eco => 'B30j',
			# TRANSLATORS: B30j: 1. e4 c5 2. Nf3 Nc6 3. Nc3 e5
			variation => N__('Sicilian: 2... Nc6 3. Nc3 e5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq' => {
			eco => 'B30k',
			# TRANSLATORS: B30k: 1. e4 c5 2. Nf3 Nc6 3. Bb5
			variation => N__('Sicilian: Rossolimo'),
			moves => {
				'd8b6' => 1,
				'e7e6' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1b1kbnr/pp1ppppp/1qn5/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq' => {
			eco => 'B30l',
			# TRANSLATORS: B30l: 1. e4 c5 2. Nf3 Nc6 3. Bb5 Qb6
			variation => N__('Sicilian: Rossolimo, 3... Qb6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq' => {
			eco => 'B30m',
			# TRANSLATORS: B30m: 1. e4 c5 2. Nf3 Nc6 3. Bb5 Nf6
			variation => N__('Sicilian: Rossolimo, 3... Nf6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq' => {
			eco => 'B30n',
			# TRANSLATORS: B30n: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6
			variation => N__('Sicilian: Rossolimo, 3... e6'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'b5c6' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/1Bp5/4P3/1P3N2/P1PP1PPP/RNBQK2R b KQkq' => {
			eco => 'B30o',
			# TRANSLATORS: B30o: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. b3
			variation => N__('Sicilian: Rossolimo, 3... e6 4. b3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/1Bp5/4P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq' => {
			eco => 'B30p',
			# TRANSLATORS: B30p: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. Nc3
			variation => N__('Sicilian: Rossolimo, 3... e6 4. Nc3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2B1p3/2p5/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq' => {
			eco => 'B30q',
			# TRANSLATORS: B30q: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. Bxc6
			variation => N__('Sicilian: Rossolimo, 3... e6 4.Bxc6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n1p3/1Bp5/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq' => {
			eco => 'B30r',
			# TRANSLATORS: B30r: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqkb1r/pp1pnppp/2n1p3/1Bp5/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq' => {
			eco => 'B30s',
			# TRANSLATORS: B30s: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7'),
			moves => {
				'b1c3' => 1,
				'b2b3' => 1,
				'c2c3' => 1,
				'f1e1' => 1,
			},
		},
		'r1bqkb1r/pp1pnppp/2n1p3/1Bp5/4P3/1P3N2/P1PP1PPP/RNBQ1RK1 b kq' => {
			eco => 'B30s',
			# TRANSLATORS: B30s: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7 5. b3
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7 5. b3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1pnppp/2n1p3/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq' => {
			eco => 'B30t',
			# TRANSLATORS: B30t: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7 5. Re1
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7 5. Re1'),
			moves => {
			},
		},
		'r1bqkb1r/pp1pnppp/2n1p3/1Bp5/4P3/2N2N2/PPPP1PPP/R1BQ1RK1 b kq' => {
			eco => 'B30u',
			# TRANSLATORS: B30u: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7 5. Nc3
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7 5. Nc3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1pnppp/2n1p3/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq' => {
			eco => 'B30v',
			# TRANSLATORS: B30v: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7 5. c3
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7 5. c3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqkb1r/1p1pnppp/p1n1p3/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 w kq' => {
			eco => 'B30w',
			# TRANSLATORS: B30w: 1. e4 c5 2. Nf3 Nc6 3. Bb5 e6 4. O-O Nge7 5. c3 a6
			variation => N__('Sicilian: Rossolimo, 3... e6 4.O-O Nge7 5. c3 a6'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq' => {
			eco => 'B31a',
			# TRANSLATORS: B31a: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6
			variation => N__('Sicilian: Rossolimo, 3... g6'),
			moves => {
				'b1c3' => 1,
				'b5c6' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/1Bp5/4P3/2N2N2/PPPP1PPP/R1BQK2R b KQkq' => {
			eco => 'B31b',
			# TRANSLATORS: B31b: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Nc3
			variation => N__('Sicilian: Rossolimo, 3... g6 4. Nc3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2B3p1/2p5/4P3/5N2/PPPP1PPP/RNBQK2R b KQkq' => {
			eco => 'B31c',
			# TRANSLATORS: B31c: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6
			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6'),
			moves => {
				'd7c6' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2p3p1/2p5/4P3/5N2/PPPP1PPP/RNBQK2R w KQkq' => {
			eco => 'B31d',
			# TRANSLATORS: B31d: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6 dxc6
			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6'),
			moves => {
				'd2d3' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2p3p1/2p5/4P3/3P1N2/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B31e',
			# TRANSLATORS: B31e: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6 dxc6 5. d3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6 5. d3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2p3p1/2p5/4P3/3P1N2/PPP2PPP/RNBQK2R w KQkq' => {
			eco => 'B31e',
			# TRANSLATORS: B31e: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6 dxc6 5. d3 Bg7
			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6 5. d3'),
			moves => {
				'h2h3' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2p3p1/2p5/4P3/3P1N1P/PPP2PP1/RNBQK2R b KQkq' => {
			eco => 'B31f',
			# TRANSLATORS: B31f: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. Bxc6 dxc6 5. d3 Bg7 6. h3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.Bxc6 dxc6 5. d3 Bg7 6. h3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQ1RK1 b kq' => {
			eco => 'B31g',
			# TRANSLATORS: B31g: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQ1RK1 w kq' => {
			eco => 'B31g',
			# TRANSLATORS: B31g: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O'),
			moves => {
				'c2c3' => 1,
				'f1e1' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 b kq' => {
			eco => 'B31h',
			# TRANSLATORS: B31h: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. c3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. c3'),
			moves => {
				'e7e5' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQ1RK1 w kq' => {
			eco => 'B31i',
			# TRANSLATORS: B31i: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. c3 e5
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. c3 e5'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 b kq' => {
			eco => 'B31j',
			# TRANSLATORS: B31j: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1'),
			moves => {
				'e7e5' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq' => {
			eco => 'B31k',
			# TRANSLATORS: B31k: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6'),
			moves => {
				'c2c3' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq' => {
			eco => 'B31l',
			# TRANSLATORS: B31l: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N2/PP1P1PPP/RNBQR1K1 w -' => {
			eco => 'B31l',
			# TRANSLATORS: B31l: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3 O-O
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3'),
			moves => {
				'd2d4' => 1,
				'h2h3' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/4P3/2P2N1P/PP1P1PP1/RNBQR1K1 b -' => {
			eco => 'B31m',
			# TRANSLATORS: B31m: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. h3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. h3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/1Bp5/3PP3/2P2N2/PP3PPP/RNBQR1K1 b -' => {
			eco => 'B31n',
			# TRANSLATORS: B31n: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. d4
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 Nf6 6. c3 O-O 7. d4'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/5N2/PPPP1PPP/RNBQR1K1 w kq' => {
			eco => 'B31o',
			# TRANSLATORS: B31o: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 e5'),
			moves => {
				'b2b4' => 1,
				'c2c3' => 1,
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/1P2P3/5N2/P1PP1PPP/RNBQR1K1 b kq' => {
			eco => 'B31p',
			# TRANSLATORS: B31p: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5 6. b4
			variation => N__('Sicilian: Rossolimo, Gurgenidze Variation'),
			moves => {
			},
		},
		'r1bqk1nr/pp1p1pbp/2n3p1/1Bp1p3/4P3/2P2N2/PP1P1PPP/RNBQR1K1 b kq' => {
			eco => 'B31q',
			# TRANSLATORS: B31q: 1. e4 c5 2. Nf3 Nc6 3. Bb5 g6 4. O-O Bg7 5. Re1 e5 6. c3
			variation => N__('Sicilian: Rossolimo, 3... g6 4.O-O Bg7 5. Re1 e5'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/2p5/3PP3/5N2/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32a',
			# TRANSLATORS: B32a: 1. e4 c5 2. Nf3 Nc6 3. d4
			variation => N__('Sicilian: 2... Nc6 3. d4'),
			moves => {
				'c5d4' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/8/3pP3/5N2/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32b',
			# TRANSLATORS: B32b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4
			variation => N__('Sicilian: 2... Nc6 3. d4 cxd4'),
			moves => {
				'f3d4' => 1,
			},
		},
		'r1bqkbnr/pp1ppppp/2n5/8/3NP3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32c',
			# TRANSLATORS: B32c: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4
			variation => N__('Sicilian: Open, 2... Nc6'),
			moves => {
				'd7d5' => 1,
				'd7d6' => 1,
				'd8b6' => 1,
				'd8c7' => 1,
				'e7e5' => 1,
				'g7g6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp2pppp/2np4/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32d',
			# TRANSLATORS: B32d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 d6
			variation => N__('Sicilian: Open, 2... Nc6, 4... d6'),
			moves => {
			},
		},
		'r1b1kbnr/pp1ppppp/1qn5/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32e',
			# TRANSLATORS: B32e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qb6
			variation => N__('Sicilian: Open, 2... Nc6, 4... Qb6'),
			moves => {
			},
		},
		'r1b1kbnr/ppqppppp/2n5/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32f',
			# TRANSLATORS: B32f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qc7
			variation => N__('Sicilian: Flohr Variation'),
			moves => {
				'd4b5' => 1,
			},
		},
		'r1b1kbnr/ppqppppp/2n5/1N6/4P3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32g',
			# TRANSLATORS: B32g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Qc7 5. Nb5
			variation => N__('Sicilian: Flohr, 5. Nb5'),
			moves => {
			},
		},
		'r1bqkbnr/pp2pppp/2n5/3p4/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32h',
			# TRANSLATORS: B32h: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 d5
			variation => N__('Sicilian: 2..Nc6, Nimzowitsch Variation'),
			moves => {
			},
		},
		'r1bqkbnr/pp1p1ppp/2n5/4p3/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32i',
			# TRANSLATORS: B32i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5
			variation => N__('Sicilian: Lowenthal'),
			moves => {
				'd4b5' => 1,
			},
		},
		'r1bqkbnr/pp1p1ppp/2n5/1N2p3/4P3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32j',
			# TRANSLATORS: B32j: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5
			variation => N__('Sicilian: Lowenthal, 5. Nb5'),
			moves => {
				'a7a6' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkbnr/1p1p1ppp/p1n5/1N2p3/4P3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32k',
			# TRANSLATORS: B32k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6
			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'b5d6' => 1,
			},
		},
		'r1bqkbnr/1p1p1ppp/p1nN4/4p3/4P3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32k',
			# TRANSLATORS: B32k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+
			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'f8d6' => 1,
			},
		},
		'r1bqk1nr/1p1p1ppp/p1nb4/4p3/4P3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32k',
			# TRANSLATORS: B32k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6
			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd1d6' => 1,
			},
		},
		'r1bqk1nr/1p1p1ppp/p1nQ4/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq' => {
			eco => 'B32k',
			# TRANSLATORS: B32k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6
			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd8f6' => 1,
			},
		},
		'r1b1k1nr/1p1p1ppp/p1nQ1q2/4p3/4P3/8/PPP2PPP/RNB1KB1R w KQkq' => {
			eco => 'B32k',
			# TRANSLATORS: B32k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6 Qf6
			variation => N__('Sicilian: Lowenthal, 5... a6'),
			moves => {
				'd6d1' => 1,
				'd6f6' => 1,
			},
		},
		'r1b1k1nr/1p1p1ppp/p1n2Q2/4p3/4P3/8/PPP2PPP/RNB1KB1R b KQkq' => {
			eco => 'B32l',
			# TRANSLATORS: B32l: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6 Qf6 8. Qxf6
			variation => N__('Sicilian: Lowenthal, 5... a6, Queenswap'),
			moves => {
			},
		},
		'r1b1k1nr/1p1p1ppp/p1n2q2/4p3/4P3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32m',
			# TRANSLATORS: B32m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 a6 6. Nd6+ Bxd6 7. Qxd6 Qf6 8. Qd1
			variation => N__('Sicilian: Lowenthal, 5... a6, 8. Qd1'),
			moves => {
			},
		},
		'r1bqkbnr/pp3ppp/2np4/1N2p3/4P3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B32n',
			# TRANSLATORS: B32n: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6
			variation => N__('Sicilian: Lowenthal, Kalashnikov'),
			moves => {
				'a2a4' => 1,
				'b1c3' => 1,
				'c2c4' => 1,
			},
		},
		'r1bqkbnr/pp3ppp/2np4/1N2p3/P3P3/8/1PP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B32o',
			# TRANSLATORS: B32o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. a4
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. a4'),
			moves => {
			},
		},
		'r1bqkbnr/pp3ppp/2np4/1N2p3/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B32p',
			# TRANSLATORS: B32p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqkbnr/1p3ppp/p1np4/1N2p3/4P3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B32p',
			# TRANSLATORS: B32p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3 a6
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3'),
			moves => {
				'b5a3' => 1,
			},
		},
		'r1bqkbnr/1p3ppp/p1np4/4p3/4P3/N1N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B32p',
			# TRANSLATORS: B32p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3 a6 7. Na3
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3'),
			moves => {
				'b7b5' => 1,
			},
		},
		'r1bqkbnr/5ppp/p1np4/1p2p3/4P3/N1N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B32q',
			# TRANSLATORS: B32q: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3 a6 7. Na3 b5
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3 a6 7. Na3 b5'),
			moves => {
				'c3d5' => 1,
			},
		},
		'r1bqkbnr/5ppp/p1np4/1p1Np3/4P3/N7/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B32q',
			# TRANSLATORS: B32q: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3 a6 7. Na3 b5 8. Nd5
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3 a6 7. Na3 b5'),
			moves => {
				'g8e7' => 1,
			},
		},
		'r1bqkb1r/4nppp/p1np4/1p1Np3/4P3/N7/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B32r',
			# TRANSLATORS: B32r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. N1c3 a6 7. Na3 b5 8. Nd5 Nge7
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6.N1c3 a6 7. Na3 b5 8. Nd5 Nge7'),
			moves => {
			},
		},
		'r1bqkbnr/pp3ppp/2np4/1N2p3/2P1P3/8/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B32s',
			# TRANSLATORS: B32s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/8/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B32t',
			# TRANSLATORS: B32t: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7'),
			moves => {
				'b1c3' => 1,
				'f1e2' => 1,
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/8/PP2BPPP/RNBQK2R b KQkq' => {
			eco => 'B32u',
			# TRANSLATORS: B32u: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. Be2
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7. Be2'),
			moves => {
			},
		},
		'r1bqk1nr/pp2bppp/2np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B32v',
			# TRANSLATORS: B32v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqk1nr/1p2bppp/p1np4/1N2p3/2P1P3/2N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B32v',
			# TRANSLATORS: B32v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3 a6
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'b5a3' => 1,
			},
		},
		'r1bqk1nr/1p2bppp/p1np4/4p3/2P1P3/N1N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B32v',
			# TRANSLATORS: B32v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3 a6 8. Na3
			variation => N__('Sicilian: Lowenthal, Kalashnikov, 6. c4 Be7 7.N1c3'),
			moves => {
				'c8e6' => 1,
			},
		},
		'r2qk1nr/1p2bppp/p1npb3/4p3/2P1P3/N1N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B32w',
			# TRANSLATORS: B32w: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3 a6 8. Na3 Be6
			variation => N__('Sicilian: Lowenthal, Kalashnikov, Main Line'),
			moves => {
				'a3c2' => 1,
				'f1e2' => 1,
			},
		},
		'r2qk1nr/1p2bppp/p1npb3/4p3/2P1P3/2N5/PPN2PPP/R1BQKB1R b KQkq' => {
			eco => 'B32x',
			# TRANSLATORS: B32x: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3 a6 8. Na3 Be6 9. Nc2
			variation => N__('Sicilian: Lowenthal, Kalashnikov, Main Line, 9. Nc2'),
			moves => {
			},
		},
		'r2qk1nr/1p2bppp/p1npb3/4p3/2P1P3/N1N5/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'B32y',
			# TRANSLATORS: B32y: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 e5 5. Nb5 d6 6. c4 Be7 7. N1c3 a6 8. Na3 Be6 9. Be2
			variation => N__('Sicilian: Lowenthal, Kalashnikov, Main Line, 9. Be2'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B33a',
			# TRANSLATORS: B33a: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6
			variation => N__('Sicilian: Open, 2... Nc6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkb1r/pp1ppppp/2n2n2/8/3NP3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33a',
			# TRANSLATORS: B33a: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3
			variation => N__('Sicilian: Open, 2... Nc6'),
			moves => {
				'd8b6' => 1,
				'e7e5' => 1,
			},
		},
		'r1b1kb1r/pp1ppppp/1qn2n2/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33b',
			# TRANSLATORS: B33b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6
			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6'),
			moves => {
				'd4b3' => 1,
			},
		},
		'r1b1kb1r/pp1ppppp/1qn2n2/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33b',
			# TRANSLATORS: B33b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3
			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3'),
			moves => {
				'e7e6' => 1,
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NN5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33b',
			# TRANSLATORS: B33b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3 e6
			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3 e6'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1b1kb1r/pp1p1ppp/1qn1pn2/8/4P3/1NNB4/PPP2PPP/R1BQK2R b KQkq' => {
			eco => 'B33c',
			# TRANSLATORS: B33c: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 Qb6 6. Nb3 e6 7. Bd3
			variation => N__('Sicilian: Open, 2... Nc6, 5... Qb6 6. Nb3 e6 7. Bd3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/4p3/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33d',
			# TRANSLATORS: B33d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5
			variation => N__('Sicilian: Pelikan/Sveshnikov'),
			moves => {
				'd4b3' => 1,
				'd4b5' => 1,
				'd4c6' => 1,
				'd4f3' => 1,
				'd4f5' => 1,
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/4p3/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33d',
			# TRANSLATORS: B33d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Nb3
			variation => N__('Sicilian: Pelikan/Sveshnikov, 6. Nb3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2N2n2/4p3/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33d',
			# TRANSLATORS: B33d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Nxc6
			variation => N__('Sicilian: Pelikan/Sveshnikov, 6.Nxc6'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/4p3/4P3/2N2N2/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33d',
			# TRANSLATORS: B33d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Nf3
			variation => N__('Sicilian: Pelikan/Sveshnikov, 6. Nf3'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/4pN2/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33d',
			# TRANSLATORS: B33d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Nf5
			variation => N__('Sicilian: Pelikan/Sveshnikov, 6. Nf5'),
			moves => {
			},
		},
		'r1bqkb1r/pp1p1ppp/2n2n2/1N2p3/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33e',
			# TRANSLATORS: B33e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5
			variation => N__('Sicilian: Pelikan/Sveshnikov, 6.Nbd5'),
			moves => {
				'd7d6' => 1,
				'h7h6' => 1,
			},
		},
		'r1bqkb1r/pp1p1pp1/2n2n1p/1N2p3/4P3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33e',
			# TRANSLATORS: B33e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 h6
			variation => N__('Sicilian: Pelikan, Haberditz Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2np1n2/1N2p3/4P3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33f',
			# TRANSLATORS: B33f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6
			variation => N__('Sicilian: Pelikan/Sveshnikov, Pelikan Variation'),
			moves => {
				'a2a4' => 1,
				'b5a3' => 1,
				'c1g5' => 1,
				'c3d5' => 1,
			},
		},
		'r1bqkb1r/pp3ppp/2np1n2/4p3/4P3/N1N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33f',
			# TRANSLATORS: B33f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Na3
			variation => N__('Sicilian: Pelikan, 7. Na3'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2np1n2/1N2p3/P3P3/2N5/1PP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33g',
			# TRANSLATORS: B33g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. a4
			variation => N__('Sicilian: Pelikan, 7. a4'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqkb1r/1p3ppp/p1np1n2/1N2p3/P3P3/2N5/1PP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B33g',
			# TRANSLATORS: B33g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. a4 a6
			variation => N__('Sicilian: Pelikan, 7. a4'),
			moves => {
				'b5a3' => 1,
			},
		},
		'r1bqkb1r/1p3ppp/p1np1n2/4p3/P3P3/N1N5/1PP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33g',
			# TRANSLATORS: B33g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. a4 a6 8. Na3
			variation => N__('Sicilian: Pelikan, 7. a4'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2np1n2/1N1Np3/4P3/8/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B33h',
			# TRANSLATORS: B33h: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Nd5
			variation => N__('Sicilian: Pelikan, 7. Nd5'),
			moves => {
			},
		},
		'r1bqkb1r/pp3ppp/2np1n2/1N2p1B1/4P3/2N5/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33i',
			# TRANSLATORS: B33i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5
			variation => N__('Sicilian: Pelikan, 7. Bg5'),
			moves => {
				'a7a6' => 1,
			},
		},
		'r1bqkb1r/1p3ppp/p1np1n2/1N2p1B1/4P3/2N5/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33i',
			# TRANSLATORS: B33i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6
			variation => N__('Sicilian: Pelikan, 7. Bg5 a6'),
			moves => {
				'b5a3' => 1,
				'g5f6' => 1,
			},
		},
		'r1bqkb1r/1p3ppp/p1np1B2/1N2p3/4P3/2N5/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33j',
			# TRANSLATORS: B33j: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Bxf6
			variation => N__('Sicilian: Pelikan, 7. Bg5 a6 8.Bxf6'),
			moves => {
			},
		},
		'r1bqkb1r/1p3ppp/p1np1n2/4p1B1/4P3/N1N5/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33k',
			# TRANSLATORS: B33k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3
			variation => N__('Sicilian: Pelikan, 7. Bg5 a6 8. Na3'),
			moves => {
				'b7b5' => 1,
				'c8e6' => 1,
			},
		},
		'r2qkb1r/1p3ppp/p1npbn2/4p1B1/4P3/N1N5/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33k',
			# TRANSLATORS: B33k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 Be6
			variation => N__('Sicilian: Pelikan, Bird Variation'),
			moves => {
			},
		},
		'r1bqkb1r/5ppp/p1np1n2/1p2p1B1/4P3/N1N5/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33l',
			# TRANSLATORS: B33l: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5
			variation => N__('Sicilian: Pelikan, Chelyabinsk Variation'),
			moves => {
				'c3d5' => 1,
				'g5f6' => 1,
			},
		},
		'r1bqkb1r/5ppp/p1np1n2/1p1Np1B1/4P3/N7/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33l',
			# TRANSLATORS: B33l: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5'),
			moves => {
				'f8e7' => 1,
			},
		},
		'r1bqk2r/4bppp/p1np1n2/1p1Np1B1/4P3/N7/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33m',
			# TRANSLATORS: B33m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'g5f6' => 1,
			},
		},
		'r1bqk2r/4bppp/p1np1B2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33m',
			# TRANSLATORS: B33m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'e7f6' => 1,
			},
		},
		'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33m',
			# TRANSLATORS: B33m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7'),
			moves => {
				'c2c3' => 1,
			},
		},
		'r1bqk2r/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'B33n',
			# TRANSLATORS: B33n: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3'),
			moves => {
				'e8g8' => 1,
				'f6g5' => 1,
			},
		},
		'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/N1P5/PP3PPP/R2QKB1R w KQ' => {
			eco => 'B33o',
			# TRANSLATORS: B33o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O'),
			moves => {
				'a3c2' => 1,
			},
		},
		'r1bq1rk1/5ppp/p1np1b2/1p1Np3/4P3/2P5/PPN2PPP/R2QKB1R b KQ' => {
			eco => 'B33o',
			# TRANSLATORS: B33o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O 12. Nc2
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O'),
			moves => {
				'f6g5' => 1,
			},
		},
		'r1bqk2r/5ppp/p1np4/1p1Np1b1/4P3/N1P5/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'B33p',
			# TRANSLATORS: B33p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 Bg5
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 Bg5'),
			moves => {
			},
		},
		'r1bq1rk1/5ppp/p1np4/1p1Np1b1/4P3/2P5/PPN2PPP/R2QKB1R w KQ' => {
			eco => 'B33q',
			# TRANSLATORS: B33q: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O 12. Nc2 Bg5
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O 12. Nc2 Bg5'),
			moves => {
				'a2a4' => 1,
			},
		},
		'r1bq1rk1/5ppp/p1np4/1p1Np1b1/P3P3/2P5/1PN2PPP/R2QKB1R b KQ' => {
			eco => 'B33r',
			# TRANSLATORS: B33r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Nd5 Be7 10. Bxf6 Bxf6 11. c3 O-O 12. Nc2 Bg5 13. a4
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9. Nd5 Be7, 11. c3 O-O 12. Nc2 Bg5 13. a4'),
			moves => {
			},
		},
		'r1bqkb1r/5ppp/p1np1B2/1p2p3/4P3/N1N5/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33s',
			# TRANSLATORS: B33s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6'),
			moves => {
				'g7f6' => 1,
			},
		},
		'r1bqkb1r/5p1p/p1np1p2/1p2p3/4P3/N1N5/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33s',
			# TRANSLATORS: B33s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6'),
			moves => {
				'c3d5' => 1,
			},
		},
		'r1bqkb1r/5p1p/p1np1p2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B33s',
			# TRANSLATORS: B33s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10. Nd5'),
			moves => {
				'f6f5' => 1,
				'f8g7' => 1,
			},
		},
		'r1bqk2r/5pbp/p1np1p2/1p1Np3/4P3/N7/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33s',
			# TRANSLATORS: B33s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 Bg7
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10. Nd5 Bg7'),
			moves => {
				'f1d3' => 1,
			},
		},
		'r1bqk2r/5pbp/p1np1p2/1p1Np3/4P3/N2B4/PPP2PPP/R2QK2R b KQkq' => {
			eco => 'B33t',
			# TRANSLATORS: B33t: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 Bg7 11. Bd3
			variation => N__('Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10. Nd5 Bg7 11. Bd3'),
			moves => {
			},
		},
		'r1bqkb1r/5p1p/p1np4/1p1Npp2/4P3/N7/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B33u',
			# TRANSLATORS: B33u: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5
			variation => N__('Sicilian: Pelikan, Sveshnikov Variation'),
			moves => {
				'c2c3' => 1,
				'f1d3' => 1,
			},
		},
		'r1bqkb1r/5p1p/p1np4/1p1Npp2/4P3/N1P5/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'B33v',
			# TRANSLATORS: B33v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk2r/5pbp/p1np4/1p1Npp2/4P3/N1P5/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'B33v',
			# TRANSLATORS: B33v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7'),
			moves => {
				'e4f5' => 1,
			},
		},
		'r1bqk2r/5pbp/p1np4/1p1NpP2/8/N1P5/PP3PPP/R2QKB1R b KQkq' => {
			eco => 'B33v',
			# TRANSLATORS: B33v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7 12. exf5
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7'),
			moves => {
				'c8f5' => 1,
			},
		},
		'r2qk2r/5pbp/p1np4/1p1Npb2/8/N1P5/PP3PPP/R2QKB1R w KQkq' => {
			eco => 'B33v',
			# TRANSLATORS: B33v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7 12. exf5 Bxf5
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7 12.exf5 Bxf5'),
			moves => {
				'a3c2' => 1,
			},
		},
		'r2qk2r/5pbp/p1np4/1p1Npb2/8/2P5/PPN2PPP/R2QKB1R b KQkq' => {
			eco => 'B33v',
			# TRANSLATORS: B33v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7 12. exf5 Bxf5 13. Nc2
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7 12.exf5 Bxf5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r2q1rk1/5pbp/p1np4/1p1Npb2/8/2P5/PPN2PPP/R2QKB1R w KQ' => {
			eco => 'B33w',
			# TRANSLATORS: B33w: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. c3 Bg7 12. exf5 Bxf5 13. Nc2 O-O
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. c3 Bg7 12.exf5 Bxf5 13. Nc2 O-O'),
			moves => {
			},
		},
		'r1bqkb1r/5p1p/p1np4/1p1Npp2/4P3/N2B4/PPP2PPP/R2QK2R b KQkq' => {
			eco => 'B33x',
			# TRANSLATORS: B33x: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. Bd3
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. Bd3'),
			moves => {
				'c8e6' => 1,
			},
		},
		'r2qkb1r/5p1p/p1npb3/1p1Npp2/4P3/N2B4/PPP2PPP/R2QK2R w KQkq' => {
			eco => 'B33x',
			# TRANSLATORS: B33x: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. Bd3 Be6
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. Bd3 Be6'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r2qkb1r/5p1p/p1npb3/1p1Npp2/4P3/N2B4/PPP2PPP/R2Q1RK1 b kq' => {
			eco => 'B33y',
			# TRANSLATORS: B33y: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 Nf6 5. Nc3 e5 6. Ndb5 d6 7. Bg5 a6 8. Na3 b5 9. Bxf6 gxf6 10. Nd5 f5 11. Bd3 Be6 12. O-O
			variation => N__('Sicilian: Pelikan, Sveshnikov, 11. Bd3 Be6 12.O-O'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/3NP3/8/PPP2PPP/RNBQKB1R w KQkq' => {
			eco => 'B34a',
			# TRANSLATORS: B34a: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6
			variation => N__('Sicilian: Accelerated Fianchetto'),
			moves => {
				'b1c3' => 1,
				'c1e3' => 1,
				'c2c4' => 1,
				'd4c6' => 1,
				'f1c4' => 1,
				'f1e2' => 1,
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/2BNP3/8/PPP2PPP/RNBQK2R b KQkq' => {
			eco => 'B34b',
			# TRANSLATORS: B34b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Bc4
			variation => N__('Sicilian: Accelerated Fianchetto, 5. Bc4'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2N3p1/8/4P3/8/PPP2PPP/RNBQKB1R b KQkq' => {
			eco => 'B34c',
			# TRANSLATORS: B34c: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nxc6
			variation => N__('Sicilian: Accelerated Fianchetto, Exchange Variation'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/3NP3/8/PPP1BPPP/RNBQK2R b KQkq' => {
			eco => 'B34d',
			# TRANSLATORS: B34d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Be2
			variation => N__('Sicilian: Accelerated Fianchetto, 5. Be2'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/3NP3/4B3/PPP2PPP/RN1QKB1R b KQkq' => {
			eco => 'B34e',
			# TRANSLATORS: B34e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Be3
			variation => N__('Sicilian: Accelerated Fianchetto, 5. Be3'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/3NP3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B34f',
			# TRANSLATORS: B34f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern Variation'),
			moves => {
				'f8g7' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/2n2np1/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B34g',
			# TRANSLATORS: B34g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Nf6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 5... Nf6'),
			moves => {
				'd4c6' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/2N2np1/8/4P3/2N5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B34h',
			# TRANSLATORS: B34h: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Nf6 6. Nxc6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 5... Nf6 6.Nxc6'),
			moves => {
				'b7c6' => 1,
			},
		},
		'r1bqkb1r/p2ppp1p/2p2np1/8/4P3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B34i',
			# TRANSLATORS: B34i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Nf6 6. Nxc6 bxc6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 5... Nf6 6.Nxc6 bxc6'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/3NP3/2N5/PPP2PPP/R1BQKB1R w KQkq' => {
			eco => 'B34j',
			# TRANSLATORS: B34j: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 5... Bg7'),
			moves => {
				'c1e3' => 1,
				'd4b3' => 1,
				'd4e2' => 1,
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/2N5/PPP1NPPP/R1BQKB1R b KQkq' => {
			eco => 'B34k',
			# TRANSLATORS: B34k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Nde2
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6.Nde2'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/4P3/1NN5/PPP2PPP/R1BQKB1R b KQkq' => {
			eco => 'B34l',
			# TRANSLATORS: B34l: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Nb3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Nb3'),
			moves => {
			},
		},
		'r1bqk1nr/pp1pppbp/2n3p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B34m',
			# TRANSLATORS: B34m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3'),
			moves => {
				'd7d6' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B34n',
			# TRANSLATORS: B34n: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 d6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 d6'),
			moves => {
				'd1d2' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/3NP3/2N1B3/PPPQ1PPP/R3KB1R b KQkq' => {
			eco => 'B34o',
			# TRANSLATORS: B34o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 d6 7. Qd2
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 d6 7. Qd2'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B34p',
			# TRANSLATORS: B34p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6'),
			moves => {
				'd4c6' => 1,
				'f1c4' => 1,
				'f1e2' => 1,
				'f2f3' => 1,
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1BP2/PPP3PP/R2QKB1R b KQkq' => {
			eco => 'B34q',
			# TRANSLATORS: B34q: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. f3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. f3'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2N2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B34r',
			# TRANSLATORS: B34r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6'),
			moves => {
				'b7c6' => 1,
			},
		},
		'r1bqk2r/p2pppbp/2p2np1/8/4P3/2N1B3/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B34r',
			# TRANSLATORS: B34r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6'),
			moves => {
				'e4e5' => 1,
			},
		},
		'r1bqk2r/p2pppbp/2p2np1/4P3/8/2N1B3/PPP2PPP/R2QKB1R b KQkq' => {
			eco => 'B34r',
			# TRANSLATORS: B34r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6 8. e5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6 8. e5'),
			moves => {
				'f6g8' => 1,
			},
		},
		'r1bqk1nr/p2pppbp/2p3p1/4P3/8/2N1B3/PPP2PPP/R2QKB1R w KQkq' => {
			eco => 'B34s',
			# TRANSLATORS: B34s: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Nxc6 bxc6 8. e5 Ng8
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7.Nxc6 bxc6 8. e5 Ng8'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R b KQkq' => {
			eco => 'B34t',
			# TRANSLATORS: B34t: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2QK2R w KQ' => {
			eco => 'B34u',
			# TRANSLATORS: B34u: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 b -' => {
			eco => 'B34v',
			# TRANSLATORS: B34v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O'),
			moves => {
				'd7d5' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3p4/3NP3/2N1B3/PPP1BPPP/R2Q1RK1 w -' => {
			eco => 'B34v',
			# TRANSLATORS: B34v: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O d5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O d5'),
			moves => {
				'e4d5' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2n2np1/3P4/3N4/2N1B3/PPP1BPPP/R2Q1RK1 b -' => {
			eco => 'B34w',
			# TRANSLATORS: B34w: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Be2 O-O 8. O-O d5 9. exd5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 6. Be3 Nf6 7. Be2 O-O 8.O-O d5 9.exd5'),
			moves => {
			},
		},
		'r1bqk2r/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R b KQkq' => {
			eco => 'B35a',
			# TRANSLATORS: B35a: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4'),
			moves => {
				'd8a5' => 1,
				'e8g8' => 1,
			},
		},
		'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQkq' => {
			eco => 'B35b',
			# TRANSLATORS: B35b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5'),
			moves => {
				'e1g1' => 1,
			},
		},
		'r1b1k2r/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b kq' => {
			eco => 'B35b',
			# TRANSLATORS: B35b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5'),
			moves => {
				'e8g8' => 1,
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 w -' => {
			eco => 'B35c',
			# TRANSLATORS: B35c: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O'),
			moves => {
				'c4b3' => 1,
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 b -' => {
			eco => 'B35d',
			# TRANSLATORS: B35d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B3/PPP2PPP/R2Q1RK1 w -' => {
			eco => 'B35d',
			# TRANSLATORS: B35d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3 d6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'h2h3' => 1,
			},
		},
		'r1b2rk1/pp2ppbp/2np1np1/q7/3NP3/1BN1B2P/PPP2PP1/R2Q1RK1 b -' => {
			eco => 'B35d',
			# TRANSLATORS: B35d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3 d6 10. h3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'c8d7' => 1,
			},
		},
		'r4rk1/pp1bppbp/2np1np1/q7/3NP3/1BN1B2P/PPP2PP1/R2Q1RK1 w -' => {
			eco => 'B35d',
			# TRANSLATORS: B35d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3 d6 10. h3 Bd7
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3'),
			moves => {
				'f2f4' => 1,
			},
		},
		'r4rk1/pp1bppbp/2np1np1/q7/3NPP2/1BN1B2P/PPP3P1/R2Q1RK1 b -' => {
			eco => 'B35e',
			# TRANSLATORS: B35e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 Qa5 8. O-O O-O 9. Bb3 d6 10. h3 Bd7 11. f4
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 Qa5 8.O-O O-O 9. Bb3 d6 10. h3 Bd7 11. f4'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2QK2R w KQ' => {
			eco => 'B35f',
			# TRANSLATORS: B35f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O'),
			moves => {
				'c4b3' => 1,
				'e1g1' => 1,
				'f2f3' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1BP2/PPP3PP/R2QK2R b KQ' => {
			eco => 'B35g',
			# TRANSLATORS: B35g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. f3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. f3'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/2BNP3/2N1B3/PPP2PPP/R2Q1RK1 b -' => {
			eco => 'B35h',
			# TRANSLATORS: B35h: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. O-O
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8.O-O'),
			moves => {
			},
		},
		'r1bq1rk1/pp1pppbp/2n2np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R b KQ' => {
			eco => 'B35i',
			# TRANSLATORS: B35i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3'),
			moves => {
				'a7a5' => 1,
				'd7d6' => 1,
				'd8a5' => 1,
				'f6g4' => 1,
			},
		},
		'r1bq1rk1/pp1pppbp/2n3p1/8/3NP1n1/1BN1B3/PPP2PPP/R2QK2R w KQ' => {
			eco => 'B35j',
			# TRANSLATORS: B35j: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 Ng4
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 Ng4'),
			moves => {
			},
		},
		'r1b2rk1/pp1pppbp/2n2np1/q7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ' => {
			eco => 'B35k',
			# TRANSLATORS: B35k: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 Qa5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 Qa5'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ' => {
			eco => 'B35l',
			# TRANSLATORS: B35l: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5'),
			moves => {
				'a2a4' => 1,
				'f2f3' => 1,
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/P2NP3/1BN1B3/1PP2PPP/R2QK2R b KQ' => {
			eco => 'B35m',
			# TRANSLATORS: B35m: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5 9. a4
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5 9. a4'),
			moves => {
			},
		},
		'r1bq1rk1/1p1pppbp/2n2np1/p7/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ' => {
			eco => 'B35n',
			# TRANSLATORS: B35n: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 a5 9. f3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 a5 9. f3'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B3/PPP2PPP/R2QK2R w KQ' => {
			eco => 'B35o',
			# TRANSLATORS: B35o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6'),
			moves => {
				'f2f3' => 1,
				'h2h3' => 1,
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1B2P/PPP2PP1/R2QK2R b KQ' => {
			eco => 'B35o',
			# TRANSLATORS: B35o: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. h3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. h3'),
			moves => {
				'c8d7' => 1,
			},
		},
		'r2q1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1B2P/PPP2PP1/R2QK2R w KQ' => {
			eco => 'B35p',
			# TRANSLATORS: B35p: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. h3 Bd7
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. h3 Bd7'),
			moves => {
			},
		},
		'r1bq1rk1/pp2ppbp/2np1np1/8/3NP3/1BN1BP2/PPP3PP/R2QK2R b KQ' => {
			eco => 'B35q',
			# TRANSLATORS: B35q: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. f3
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. f3'),
			moves => {
				'c8d7' => 1,
			},
		},
		'r2q1rk1/pp1bppbp/2np1np1/8/3NP3/1BN1BP2/PPP3PP/R2QK2R w KQ' => {
			eco => 'B35r',
			# TRANSLATORS: B35r: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. Nc3 Bg7 6. Be3 Nf6 7. Bc4 O-O 8. Bb3 d6 9. f3 Bd7
			variation => N__('Sicilian: Accelerated Fianchetto, Modern, 7. Bc4 O-O 8. Bb3 d6 9. f3 Bd7'),
			moves => {
			},
		},
		'r1bqkbnr/pp1ppp1p/2n3p1/8/2PNP3/8/PP3PPP/RNBQKB1R b KQkq' => {
			eco => 'B36a',
			# TRANSLATORS: B36a: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4
			variation => N__('Sicilian: Maroczy Bind'),
			moves => {
				'd7d6' => 1,
				'f8g7' => 1,
				'g8f6' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2np2p1/8/2PNP3/8/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B36b',
			# TRANSLATORS: B36b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 d6
			variation => N__('Sicilian: Maroczy Bind, 5... d6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkbnr/pp2pp1p/2np2p1/8/2PNP3/2N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B36b',
			# TRANSLATORS: B36b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 d6 6. Nc3
			variation => N__('Sicilian: Maroczy Bind, 5... d6 6. Nc3'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk1nr/pp2ppbp/2np2p1/8/2PNP3/2N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B36b',
			# TRANSLATORS: B36b: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 d6 6. Nc3 Bg7
			variation => N__('Sicilian: Maroczy Bind, 5... d6 6. Nc3 Bg7'),
			moves => {
			},
		},
		'r1bqkb1r/pp1ppp1p/2n2np1/8/2PNP3/8/PP3PPP/RNBQKB1R w KQkq' => {
			eco => 'B36c',
			# TRANSLATORS: B36c: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6'),
			moves => {
				'b1c3' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/2n2np1/8/2PNP3/2N5/PP3PPP/R1BQKB1R b KQkq' => {
			eco => 'B36d',
			# TRANSLATORS: B36d: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3'),
			moves => {
				'c6d4' => 1,
				'd7d6' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/5np1/8/2PnP3/2N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B36e',
			# TRANSLATORS: B36e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 Nxd4
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 Nxd4'),
			moves => {
				'd1d4' => 1,
			},
		},
		'r1bqkb1r/pp1ppp1p/5np1/8/2PQP3/2N5/PP3PPP/R1B1KB1R b KQkq' => {
			eco => 'B36e',
			# TRANSLATORS: B36e: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 Nxd4 7. Qxd4
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 Nxd4'),
			moves => {
				'd7d6' => 1,
			},
		},
		'r1bqkb1r/pp2pp1p/3p1np1/8/2PQP3/2N5/PP3PPP/R1B1KB1R w KQkq' => {
			eco => 'B36f',
			# TRANSLATORS: B36f: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 Nxd4 7. Qxd4 d6
			variation => N__('Sicilian: Maroczy Bind, Gurgenidze Variation'),
			moves => {
			},
		},
		'r1bqkb1r/pp2pp1p/2np1np1/8/2PNP3/2N5/PP3PPP/R1BQKB1R w KQkq' => {
			eco => 'B36g',
			# TRANSLATORS: B36g: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6'),
			moves => {
				'f1e2' => 1,
			},
		},
		'r1bqkb1r/pp2pp1p/2np1np1/8/2PNP3/2N5/PP2BPPP/R1BQK2R b KQkq' => {
			eco => 'B36h',
			# TRANSLATORS: B36h: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2'),
			moves => {
				'c6d4' => 1,
			},
		},
		'r1bqkb1r/pp2pp1p/3p1np1/8/2PnP3/2N5/PP2BPPP/R1BQK2R w KQkq' => {
			eco => 'B36i',
			# TRANSLATORS: B36i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4'),
			moves => {
				'd1d4' => 1,
			},
		},
		'r1bqkb1r/pp2pp1p/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B1K2R b KQkq' => {
			eco => 'B36i',
			# TRANSLATORS: B36i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4'),
			moves => {
				'f8g7' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B1K2R w KQkq' => {
			eco => 'B36i',
			# TRANSLATORS: B36i: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8. Qxd4 Bg7
			variation => N__('Sicilian: Maroczy Bind, 5... Nf6 6. Nc3 d6 7. Be2 Nxd4 8.Qxd4 Bg7'),
			moves => {
				'c1e3' => 1,
				'c1g5' => 1,
				'e1g1' => 1,
			},
		},
		'r1bqk2r/pp2ppbp/3p1np1/8/2PQP3/2N5/PP2BPPP/R1B2RK1 b kq' => {
			eco => 'B36j',
			# TRANSLATORS: B36j: 1. e4 c5 2. Nf3 Nc6 3. d4 cxd4 4. Nxd4 g6 5. c4 Nf6 6. Nc3 d6 7. Be2 Nxd4 8