﻿#!/usr/bin/perl -w
# HTML::TocInsertor tests
# http://search.cpan.org/dist/HTML-Toc/Toc.pod#HTML::TocInsertor::insert()
use strict;
use HTML::Toc;
use HTML::TocInsertor;

use Test::More tests => 5;
use Test::Differences;


my $toc = HTML::Toc->new();
my $tocInsertor = HTML::TocInsertor->new();
my $output;



#--- 1. insertionPoint not found ---------------------------------------

my $content = <<HTML;
<body>
<p>preserved p tag</p>
preserved preceding text

ToC will be placed here

preserved following text
<h1>Chapter 1</h1>
Some text here in chapter 1 makes up the first sentence.
HTML

$toc->setOptions({
    insertionPoint => 'this_text_is_not_found',
});
$tocInsertor->insert($toc, $content, {output => \$output});
eq_or_diff($output, <<'HTML', 'insertionPoint not found; create the anchor names only');
<body>
<p>preserved p tag</p>
preserved preceding text

ToC will be placed here

preserved following text
<h1><a name="h-1"></a>Chapter 1</h1>
Some text here in chapter 1 makes up the first sentence.
HTML



#-----------------------------------------------------------------------
#--- text token insertionPoint does not match if also toc token --------
#-----------------------------------------------------------------------


$toc->setOptions({
    insertionPoint => 'r 1',
});
$tocInsertor->insert($toc, $content, {output => \$output});
eq_or_diff($output, <<'HTML', 'text token insertionPoint does not match if also toc token', {max_width => 120});
<body>
<p>preserved p tag</p>
preserved preceding text

ToC will be placed here

preserved following text
<h1><a name="h-1"></a>Chapter 1</h1>
Some text here in chapter 1
<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter 1</a></li>
</ul>
<!-- End of generated Table of Contents -->
 makes up the first sentence.
HTML


#-----------------------------------------------------------------------
#--- start token insertionPoint does match if also toc token -----------
#-----------------------------------------------------------------------

$toc->setOptions({
    insertionPoint => '<h1>',
});
$tocInsertor->insert($toc, $content, {output => \$output});
eq_or_diff($output, <<'HTML', 'start token insertionPoint does match if also toc token', {max_width => 120});
<body>
<p>preserved p tag</p>
preserved preceding text

ToC will be placed here

preserved following text

<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter 1</a></li>
</ul>
<!-- End of generated Table of Contents -->
<h1><a name="h-1"></a>Chapter 1</h1>
Some text here in chapter 1 makes up the first sentence.
HTML



# ------------------------------------------------------------------------
# --- replace insert should preserve the text around the ToC placeholder token
# --- (replicates TestReplaceText() in insert.t
# ------------------------------------------------------------------------
$toc->setOptions({
    insertionPoint => 'replace ToC will be placed here',
});
$tocInsertor->insert($toc, $content, {output => \$output});
eq_or_diff($output, <<'EOT', 'REPLACE should preserve the text around the ToC placeholder token');
<body>
<p>preserved p tag</p>
preserved preceding text


<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter 1</a></li>
</ul>
<!-- End of generated Table of Contents -->


preserved following text
<h1><a name="h-1"></a>Chapter 1</h1>
Some text here in chapter 1 makes up the first sentence.
EOT


# ------------------------------------------------------------------------
# --- BEFORE insert should preserve the text around the ToC placeholder token
# --- (replicates TestReplaceText() in insert.t
# ------------------------------------------------------------------------
$toc->setOptions({
    insertionPoint => 'before ToC will be placed here',
});
$tocInsertor->insert($toc, $content, {output => \$output});
eq_or_diff($output, <<'HTML', 'BEFORE insert should preserve the text around the ToC placeholder token');
<body>
<p>preserved p tag</p>
preserved preceding text


<!-- Table of Contents generated by Perl - HTML::Toc -->
<ul>
   <li><a href="#h-1">Chapter 1</a></li>
</ul>
<!-- End of generated Table of Contents -->
ToC will be placed here

preserved following text
<h1><a name="h-1"></a>Chapter 1</h1>
Some text here in chapter 1 makes up the first sentence.
HTML
