#!perl6

use v6;

unit module Tomtit:ver<0.0.21>;
use File::Directory::Tree;

my %profiles is Map = (
  'perl' => ( 'set-pause', 'make-dist', 'install', 'test', 'changes', 'release' ),
  'perl6' =>  ( 'set-pause', 'test', 'install', 'changes', 'release' ),
  'git' => ( 'set-git', 'commit', 'push', 'pull', 'status' ),
  'ruby' => ( 'rvm' ),
  'ado' => ( 'build', 'list', 'az-resources', 'az-account-set' ),
  'hello' => ( 'world' )
);

# tom cli initializer
  
sub init () is export {

  mkdir ".tom/.cache";
  mkdir ".tom/env";

}

sub tomtit-usage () is export  {
  say 'usage: tom $action|$options $thing'
}

sub tomtit-help () is export  {
  say q:to/DOC/;
  usage:
    tom $action|$options $thing

  run scenario:
    tom $scenario

  remove scenario:
    tom --remove $scenario

  print out scenario:
    tom --cat $scenario

  install profile:
    tom --profile $profile

  set default ennvironment
    tom --env-set $env

  actions:
    tom --bootstrap         # bootstrap system
    tom --list              # list available scenarios
    tom --profile           # list available profiles
    tom --profile $profile  # list profile scenarios
    tom --last              # what is the last run?
    tom --completion        # install Bash completion
    tom --env-set $env      # set current environment
    tom --env-set           # show current environment
    tom --env-list          # list available environments

  options:
    --env=$env  # run scenario for the given environment
    --verbose   # run scenarios in verbose mode
    --quiet,-q  # run scenrios in less verbose mode
  DOC
}

# clean tomtit internal data
# is useful as with time it might grow

sub tomtit-clean ($dir) is export { 

  say "cleaning $dir/.cache ...";

  if "$dir/.cache/".IO ~~ :e {
    empty-directory "$dir/.cache"
  }

}

sub scenario-last ($dir) is export {
  if "$dir/.cache/history".IO ~~ :e {
    my @history =  "$dir/.cache/history".IO.lines;
    say @history[*-1];
  }
}

sub tomtit-bootstrap () is export {

  my $cmd = "sparrowdo --local_mode --bootstrap";

  shell($cmd);

}

sub scenario-run ($dir,$scenario,%args?) is export {

  die "scenario $scenario not found" unless "$dir/$scenario.pl6".IO ~~ :e;

  my $fh = open "$dir/.cache/history", :a;
  $fh.print($scenario,"\n");
  $fh.close;

  my $cmd = "sparrowdo --local_mode --no_sudo --sparrow_root=$dir/.cache --sparrowfile=$dir/$scenario.pl6";

  $cmd ~= " -q" if %args<quiet>;

  if %args<env> {

    my $conf-file = %args<env> eq 'default' ?? "$dir/env/config.pl6" !! "$dir/env/config.{%args<env>}.pl6";

    $cmd ~= " --conf=$conf-file"    

  } elsif ( "$dir/env/config.pl6".IO ~~ :e ) {

    $cmd ~= " --conf=$dir/env/config.pl6"    
    
  }

  if %args<verbose> {
    $cmd ~= " --verbose";
    say "run $cmd ...";
  }

  shell($cmd);

}

sub scenario-remove ($dir,$scenario) is export {

  if "$dir/$scenario.pl6".IO ~~ :e {
    unlink "$dir/$scenario.pl6";
    say "scenario $scenario removed"
  } else {
    say "scenario $scenario not found"
  }

}

sub scenario-cat ($dir,$scenario,%args?) is export {

  if "$dir/$scenario.pl6".IO ~~ :e {
    say "[scenario $scenario]";
    my $i=0;
    
    for "$dir/$scenario.pl6".IO.lines -> $l {
      $i++;
      say %args<lines> ?? "[$i] $l" !! $l;
    }
  } else {
    say "scenario $scenario not found"
  }

}

sub scenario-edit ($dir,$scenario) is export {

    die "you should set EDITOR ENV to run editor" unless  %*ENV<EDITOR>;

    unless "$dir/$scenario.pl6".IO ~~ :e {
      my $confirm = prompt("$dir/$scenario.pl6 does not exit, do you want to create it? (type Y to confirm): ");
      return unless $confirm eq 'Y';
    }

    shell "{%*ENV<EDITOR>} $dir/$scenario.pl6";

}

sub environment-edit ($dir,$env) is export {

    die "you should set EDITOR ENV to run editor" unless  %*ENV<EDITOR>;

    my $conf-file = ( $env eq 'default' ) ?? "$dir/config.pl6" !! "$dir/config.{$env}.pl6";

    unless $conf-file.IO ~~ :e {
      my $confirm = prompt("$conf-file does not exit, do you want to create it? (type Y to confirm): ");
      return unless $confirm eq 'Y';
    }

    shell "{%*ENV<EDITOR>} $conf-file";

}

sub environment-list ($dir) is export {

    say "[environments list]";

    my @list = Array.new;

    my $current = "default";

    if "$dir/current".IO ~~ :e  && "$dir/current".IO.resolve.IO.basename {

      if "$dir/current".IO.resolve.IO.basename ~~ /config\.(.*)\.pl6/ {
        $current = "$0"
      }

    }

    for dir($dir) -> $f {

      next unless "$f".IO ~~ :f;
      next unless $f ~~ /\.pl6$/;

      if $f.basename ~~ /config\.(.*)\.pl6/ {

        @list.push("$0");

      } else {

        @list.push("default")

      }

    }

    for @list.sort -> $l {
      say $current eq $l ?? "$l *" !! $l
    };

}

sub environment-set ($dir,$env) is export {

  my $conf-file = $env eq 'default' ?? "$dir/config.pl6" !! "$dir/config.{$env}.pl6";

  die "environment $conf-file not found" unless $conf-file.IO ~~ :e;

  unlink "$dir/current" if "$dir/current".IO ~~ :e;

  symlink($conf-file,"$dir/current");

}

sub environment-show ($dir) is export {

  if "$dir/current".IO ~~ :e {

    my $current = "$dir/current".IO.resolve.IO.basename;

      if $current ~~ /config\.(.*)\.pl6/ {

        say "current environment: $0"

      } else {

        say "current environment: default"

      }

  } elsif "$dir/config.pl6".IO ~~ :e {

    say "default";

  } else {

    say "default environment is not set, create default configuration file (.tom/env/config.pl6)
or use tom --set-env \$env to set default environments"
  }
  
}


# this is one is implemented, but not presented in public API, as we
# have issues with compiling Sparrowdo scenarios
  
sub scenario-doc ($dir,$scenario) is export {

  die "scenario $scenario not found" unless "$dir/$scenario.pl6".IO ~~ :e;

  run $*EXECUTABLE, '--doc', "$dir/$scenario.pl6";

}

sub scenario-list ($dir) is export {

    say "[scenarios list]";
    my @list = Array.new;

    for dir($dir) -> $f {

      next unless "$f".IO ~~ :f;
      next unless $f ~~ /\.pl6$/;
      my $scenario-name = substr($f.basename,0,($f.basename.chars)-4);
      @list.push($scenario-name);

    }

    say join "\n", @list.sort;

}

multi sub profile-list  is export {

  say "[profiles]";

  for %profiles.keys.sort -> $i {
    say "$i"
  }
}

multi sub profile-list($dir,$profile)  is export {

  unless %profiles{$profile}:exists {
    say "profile $profile does not exist";
    return;
  }

  say "[profile scenarios]";

  for %profiles{$profile}.sort -> $s {

    my $installed = "$dir/$s.pl6".IO ~~ :f;
    say "$profile\@$s\tinstalled: $installed";

  }

}

sub profile-install ($dir, $profile is copy,%args?) is export {

  my @list;

  if $profile ~~ s/ '@' (.*) // {
    my $s1 = $0;
    @list = sort &[cmp], grep { $_ eq $s1 }, %profiles{$profile}
  } else {
    @list = %profiles{$profile}.sort
  }

  unless %profiles{$profile}:exists {
    say "profile $profile does not exist";
    return;
  }

  if @list.elems == 0 {
    say "no scenarios found ...";
    return;
  }

  for @list -> $s {
    if %?RESOURCES{"profiles/$profile/$s.pl6"}.Str.IO ~~ :f {
      say "install $profile\@$s ...";
      my $fh = open "$dir/$s.pl6", :w;
      $fh.print(slurp %?RESOURCES{"profiles/$profile/$s.pl6"}.Str);
      $fh.close;
    } else {
      say "no perl6 resource found for  $profile\@$s scenario ... skipping it";
    }
  }
    
}


sub completion-install () is export {

  say "install completion.sh ...";

  my $fh = open '/home/' ~ %*ENV<USER> ~ '/.tom_completion.sh' , :w;

  $fh.print(slurp %?RESOURCES{"completion.sh"}.Str);

  $fh.close;

  say "to activate completion say: source " ~ '/home/' ~ %*ENV<USER> ~ '/.tom_completion.sh';  
    
}
