######################################################################
#
# make_JEF.pl
#
# Copyright (c) 2018, 2021 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################

use strict;
use FindBin;
use lib $FindBin::Bin;

# JEF
require 'JEF/JEF_by_CP932.pl';
require 'CP932/CP932_by_Unicode.pl';
require 'JIS/JIS78GR_by_JIS83GR.pl';
require 'JIS/JISX0208GR_by_CP932.pl';
require 'JEF/JEF_by_Unicode_CultiCoLtd.pl';

if ($0 eq __FILE__) {
    open(DUMP,">$0.dump") || die;
    binmode(DUMP);
    open(DIFF,">$0.diff") || die;
    binmode(DIFF);
}

# generate array to check for duplicate keys
my @JEF_by_Unicode_OVERRIDE = (

    # added and removed mapping
    # Jacode4e 2.13.6.18 or newer
    # Jacode4e::RoundTrip 2.13.81.12 or newer

    '2000B' => '41A6', # 𠀋
    '20089' => '71FC', # 𠂉
    '201A2' => '71FA', # 𠆢
    '2099D' => '73FC', # 𠦝
    '20B9F' => '',     # 𠮟 (JIS2004)
    '21274' => '47E5', # 𡉴 (国土地理協会住所辞書文字)
    '2131B' => '48C6', # 𡌛
    '215D7' => '73FA', # 𡗗
    '2231E' => '4CD7', # 𢌞
    '235C4' => '70C2', # 𣗄
    '23CFE' => '57C5', # 𣳾
    '23D40' => '56AA', # 𣵀
    '342C' => '41D7', # 㐬
    '342E' => '41D9', # 㐮
    '3468' => '42E5', # 㑨
    '34DB' => '44B7', # 㓛
    '37E2' => '4AFE', # 㟢 (国土地理協会住所辞書文字)
    '37FD' => '4BD5', # 㟽
    '3800' => '4BB6', # 㠀
    '391A' => '4DFB', # 㤚
    '3AD6' => '51AF', # 㫖
    '3B1A' => '51F0', # 㬚
    '3B22' => '51F6', # 㬢
    '3B6D' => '53D4', # 㭭
    '3B88' => '53E3', # 㮈
    '3BB6' => '53F9', # 㮶
    '3BF0' => '54B2', # 㯰
    '3D1E' => '56D6', # 㴞
    '3D64' => '57AC', # 㵤
    '3E66' => '58E0', # 㹦
    '3E68' => '58E4', # 㹨
    '4093' => '5BEB', # 䂓
    '4105' => '5CD2', # 䄅
    '414F' => '5DAA', # 䅏
    '4163' => '5DB5', # 䅣
    '4264' => '5EBA', # 䉤
    '42D6' => '5FA2', # 䋖
    '43EE' => '52C1', # 䏮
    '4453' => '60EA', # 䑓
    '445B' => '60F4', # 䑛
    '447A' => '61AC', # 䑺
    '4491' => '61BA', # 䒑
    '44B3' => '61DF', # 䒳
    '44BE' => '61ED', # 䒾
    '4508' => '62C0', # 䔈
    '4525' => '62CC', # 䔥
    '4543' => '62DB', # 䕃
    '46AF' => '64E1', # 䚯
    '4A29' => '6CAA', # 䨩
    '4ABC' => '6CE6', # 䪼
    '4BC2' => '6DD9', # 䯂
    '4BE8' => '6EA5', # 䯨
    '4C17' => '6EB8', # 䰗
    '4C20' => '6EC5', # 䰠
    '4E12' => '41A8', # 丒
    '4EB9' => '41DB', # 亹
    '4EBB' => '71FB', # 亻
    '4F03' => '41F0', # 伃 (国土地理協会住所辞書文字)
    '4F0B' => '41F2', # 伋 (国土地理協会住所辞書文字)
    '4F60' => '41F8', # 你
    '4F7A' => '42AE', # 佺
    '4F7D' => '42AF', # 佽
    '4F7E' => '42B0', # 佾
    '4FA0' => '43CB', # 侠 (JIS83)
    '4FE6' => '42CD', # 俦
    '4FF1' => '42CF', # 俱 (JIS2004)
    '50D9' => '43B3', # 僙
    '50E1' => '43B5', # 僡
    '51CA' => '43FB', # 凊
    '5213' => '44BC', # 刓
    '5215' => '44B4', # 刕 (国土地理協会住所辞書文字)
    '525D' => 'C7ED', # 剝 (JIS78)
    '5265' => '44D0', # 剥 (JIS2004) (JIS83)
    '52CA' => '44E8', # 勊
    '52D6' => '44EC', # 勖
    '52FB' => '45A2', # 勻
    '533E' => '45BB', # 匾
    '537A' => '4BF2', # 卺
    '537D' => '45D4', # 卽 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5412' => '46AA', # 吒
    '541E' => '',     # 吞 (JIS2004)
    '5516' => '47B9', # 唖 (JIS83)
    '559D' => '47B8', # 喝 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5618' => '47BA', # 嘘 (JIS2004) (JIS83)
    '5630' => '46FD', # 嘰
    '5653' => 'B1B3', # 噓 (JIS78)
    '565B' => '47BC', # 噛 (JIS83)
    '5671' => '47AE', # 噱
    '5699' => 'B3FA', # 嚙 (JIS78)
    '56A2' => '70BA', # 嚢 (JIS83)
    '5721' => '47CC', # 圡
    '579D' => '47F4', # 垝
    '57C7' => '47F6', # 埇 (国土地理協会住所辞書文字)
    '57DE' => '47FC', # 埞 (国土地理協会住所辞書文字)
    '586B' => '47CA', # 填 (JIS83)
    '589E' => '48BB', # 增 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '58E0' => '48C3', # 壠
    '5905' => '48CE', # 夅
    '5946' => '48E1', # 奆
    '59CD' => '49AC', # 姍
    '59E3' => '49B0', # 姣
    '59F8' => '49B5', # 姸 (JIS2004)
    '5A7E' => '49CA', # 婾
    '5A84' => '49CB', # 媄
    '5AF0' => '49DE', # 嫰
    '5B34' => '49EB', # 嬴
    '5BEC' => '4AC6', # 寬 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5C12' => '41E1', # 尒
    '5C4F' => '4AE7', # 屏 (JIS2004) (JIS83)
    '5C5B' => 'D6A2', # 屛 (JIS78)
    '5C61' => '4AE5', # 屡 (JIS83)
    '5C88' => '4BBA', # 岈
    '5CA7' => '4AF2', # 岧
    '5D10' => '4BC9', # 崐
    '5D1D' => '4BCA', # 崝
    '5D20' => '4BCB', # 崠
    '5D47' => '5DAE', # 嵇
    '5D6D' => '4BD3', # 嵭 (国土地理協会住所辞書文字)
    '5DD7' => '4BE3', # 巗
    '5DE2' => '52F1', # 巢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5E6D' => '4CAC', # 幭
    '5E77' => '4CB2', # 幷 (JIS2004)
    '5EAC' => '4CC2', # 庬
    '5FB5' => '4DBF', # 徵 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '5FB7' => '4DC0', # 德 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '61D5' => '4DE6', # 懕
    '623E' => '4EFB', # 戾 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '625A' => '4FAC', # 扚 (国土地理協会住所辞書文字)
    '6285' => '4FB3', # 抅
    '63B4' => '50BB', # 掴 (JIS83)
    '63BB' => '50B9', # 掻 (JIS83)
    '63ED' => '4FEA', # 揭 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '63F7' => '4FE3', # 揷
    '64CA' => '4FA6', # 擊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6505' => '50B1', # 攅 (JIS83)
    '6532' => '50C0', # 攲
    '6544' => '50C6', # 敄
    '6581' => '50D2', # 斁
    '6585' => '49F8', # 斅
    '65B5' => '50E5', # 斵
    '65C9' => '50E9', # 旉
    '65FC' => '51BB', # 旼
    '663A' => '50FD', # 昺
    '6648' => '51D0', # 晈
    '665A' => '51D5', # 晚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6663' => '51D8', # 晣
    '66C6' => '51B6', # 曆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '677B' => '53A7', # 杻 (国土地理協会住所辞書文字)
    '67C0' => '53B4', # 柀 (国土地理協会住所辞書文字)
    '6845' => '53C8', # 桅
    '6849' => '53C9', # 桉
    '68C3' => '52F2', # 棃
    '6935' => '53E4', # 椵 (国土地理協会住所辞書文字)
    '6957' => '53E9', # 楗
    '6980' => '53EE', # 榀 (国土地理協会住所辞書文字)
    '6992' => '53EF', # 榒 (国土地理協会住所辞書文字)
    '69BA' => '52FB', # 榺
    '69C0' => '52F5', # 槀
    '69E9' => '52F7', # 槩
    '69EA' => '53FA', # 槪 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '69F6' => '53FB', # 槶
    '6A49' => '54A6', # 橉
    '6A6B' => '54AD', # 橫 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6A9E' => '54B5', # 檞 (国土地理協会住所辞書文字)
    '6ADC' => '52F9', # 櫜
    '6B65' => '54DC', # 步 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6B77' => '54DF', # 歷 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6BCF' => '54F4', # 每 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6BD7' => '54F9', # 毗
    '6C0E' => '55A1', # 氎
    '6C5B' => '55BC', # 汛
    '6C76' => '55C0', # 汶
    '6C97' => '55B3', # 沗 (国土地理協会住所辞書文字)
    '6D2F' => '55B4', # 洯
    '6D81' => '56A1', # 涁
    '6D89' => '56A5', # 涉 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6D8A' => '56AC', # 涊
    '6D9C' => '70D7', # 涜 (JIS83)
    '6DDA' => '56B4', # 淚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6E1A' => '55AF', # 渚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6E34' => '56C0', # 渴 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6E57' => '56CE', # 湗 (国土地理協会住所辞書文字)
    '6E8C' => '70DA', # 溌 (JIS83)
    '6EAB' => '56DB', # 溫 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '6EBF' => '56DF', # 溿 (国土地理協会住所辞書文字)
    '6ED9' => '56E5', # 滙
    '6F25' => '56F1', # 漥
    '6F51' => 'C8AE', # 潑 (JIS78)
    '6F7E' => '56FD', # 潾
    '6F8D' => '57A3', # 澍 (国土地理協会住所辞書文字)
    '7028' => '57BC', # 瀨 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '710F' => '58A6', # 焏 (国土地理協会住所辞書文字)
    '7114' => '70E0', # 焔 (JIS83)
    '712E' => '57E8', # 焮
    '7151' => '57D3', # 煑
    '7196' => '57F3', # 熖
    '7215' => '57CF', # 爕
    '7238' => '58B3', # 爸
    '7239' => '58B4', # 爹
    '72BE' => '58D6', # 犾 (国土地理協会住所辞書文字)
    '72C0' => '58D7', # 狀 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7352' => '58D4', # 獒
    '739F' => '59B1', # 玟
    '73A8' => '59A4', # 玨
    '7422' => '59A1', # 琢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '744B' => '59E5', # 瑋
    '74A3' => '59FD', # 璣
    '74A9' => '5AA2', # 璩
    '7501' => '5AB8', # 甁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '756C' => '5ACB', # 畬
    '757E' => '5AC7', # 畾 (国土地理協会住所辞書文字)
    '75E9' => '5AF5', # 痩 (JIS2004) (JIS83)
    '7600' => '5AE5', # 瘀
    '76A1' => '5BA7', # 皡
    '784F' => '5CA8', # 硏 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7851' => '5CA9', # 硑
    '792E' => '5CCA', # 礮
    '7958' => '5CD1', # 祘
    '7977' => '5CEC', # 祷 (JIS83)
    '79CC' => '5CF5', # 秌
    '7A5D' => '5DBF', # 穝 (国土地理協会住所辞書文字)
    '7A90' => '5DD0', # 窐
    '7B3B' => '5DF2', # 笻
    '7BAA' => '5EC2', # 箪 (JIS83)
    '7C1E' => 'C3BD', # 簞 (JIS78)
    '7C45' => '5EBB', # 籅
    '7CA0' => '5ECE', # 粠 (国土地理協会住所辞書文字)
    '7CD9' => '5ED8', # 糙 (国土地理協会住所辞書文字)
    '7D5A' => '5FA7', # 絚
    '7D86' => '5FB0', # 綆
    '7DA0' => '5FB6', # 綠 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7DC0' => '5FBA', # 緀
    '7DD6' => '5FBE', # 緖 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7DE3' => '5FC1', # 緣 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '7E4B' => '5FDF', # 繋 (JIS2004) (JIS83)
    '7E4D' => '5FE0', # 繍 (JIS83)
    '8002' => '72FC', # 耂
    '8160' => '52D1', # 腠
    '8184' => '52D7', # 膄
    '81EC' => '52F0', # 臬
    '8279' => '72FA', # 艹
    '828E' => '61BF', # 芎 (国土地理協会住所辞書文字)
    '8330' => '61E3', # 茰 (国土地理協会住所辞書文字)
    '83B1' => '63A2', # 莱 (JIS83)
    '83E1' => '62A6', # 菡
    '845A' => '62B4', # 葚 (国土地理協会住所辞書文字)
    '845F' => '62B5', # 葟
    '848B' => '61B9', # 蒋 (JIS83)
    '853E' => '62CD', # 蔾
    '85AD' => '62E3', # 薭 (国土地理協会住所辞書文字)
    '85B0' => '62E5', # 薰 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '8653' => '63AD', # 虓
    '865B' => '63AA', # 虛 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '8749' => '63B3', # 蝉 (JIS83)
    '874B' => '63B5', # 蝋 (JIS83)
    '87EC' => 'C0E6', # 蟬 (JIS78)
    '880B' => '63DC', # 蠋 (国土地理協会住所辞書文字)
    '8910' => '63EE', # 褐 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '8927' => '63F2', # 褧
    '8989' => '64BC', # 覉
    '8990' => '64CB', # 覐
    '8CF4' => '65F9', # 賴 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '8D0E' => '66B0', # 贎
    '8EAF' => '66EA', # 躯 (JIS83)
    '8EB6' => '66E8', # 躶
    '8EC0' => 'B6ED', # 軀 (JIS78)
    '8FD3' => '67AE', # 迓
    '9088' => '67F1', # 邈
    '90DE' => '68A6', # 郞 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '9115' => '68DF', # 鄕 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '9137' => '68B3', # 鄷
    '9146' => '68B5', # 酆
    '9197' => '69A2', # 醗 (JIS83)
    '91A4' => '69A1', # 醤 (JIS83)
    '9242' => '69E5', # 鉂
    '9256' => '69ED', # 鉖
    '92C6' => '69B0', # 鋆
    '92D3' => '6ABA', # 鋓 (国土地理協会住所辞書文字)
    '9304' => '6ACD', # 錄 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '934A' => '6ADB', # 鍊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '936B' => '69B4', # 鍫
    '93BA' => '6AF5', # 鎺 (国土地理協会住所辞書文字)
    '946D' => '6BCE', # 鑭
    '9471' => '6BCF', # 鑱
    '958D' => '6BDE', # 閍 (国土地理協会住所辞書文字)
    '95D0' => '6BED', # 闐
    '96EF' => '6CA4', # 雯
    '9733' => '6CAC', # 霳 (国土地理協会住所辞書文字)
    '973B' => '6CAD', # 霻 (国土地理協会住所辞書文字)
    '974F' => '6CAE', # 靏 (国土地理協会住所辞書文字)
    '97D1' => '6CCF', # 韑
    '97D9' => '6CD0', # 韙
    '982C' => '6CFD', # 頬 (JIS83)
    '9832' => '6CEA', # 頲
    '983B' => '6CFB', # 頻 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '985B' => '6CFC', # 顛 (JIS83)
    '98F1' => '6DAF', # 飱
    '99A3' => '6DD3', # 馣
    '9A28' => '6DF8', # 騨 (JIS83)
    '9A52' => 'C2CD', # 驒 (JIS78)
    '9A58' => '6DDF', # 驘
    '9B66' => '6ECE', # 魦 (国土地理協会住所辞書文字)
    '9B75' => '6ED0', # 魵 (国土地理協会住所辞書文字)
    '9BC1' => '6EDD', # 鯁
    '9BCE' => '6EDE', # 鯎
    '9C00' => '6EE3', # 鰀 (国土地理協会住所辞書文字)
    '9C27' => '6ECB', # 鰧
    '9C63' => '6EEA', # 鱣 (国土地理協会住所辞書文字)
    '9D0E' => '6FB0', # 鴎 (JIS83)
    '9D1E' => '6EF9', # 鴞 (国土地理協会住所辞書文字)
    '9D47' => '6EFB', # 鵇
    '9D62' => '6EFD', # 鵢 (国土地理協会住所辞書文字)
    '9D70' => '6FA3', # 鵰 (国土地理協会住所辞書文字)
    '9D96' => '6EF1', # 鶖 (国土地理協会住所辞書文字)
    '9E78' => '6FB5', # 鹸 (JIS83)
    '9EB9' => '6FC3', # 麹 (JIS83)
    '9EBA' => '6FC4', # 麺 (JIS83)
    '9EC3' => '6FCC', # 黃 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    '9ED1' => '6FD6', # 黑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F91D' => '54C0', # 欄 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F928' => '4CCA', # 廊 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F929' => '52AD', # 朗 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F936' => '63AB', # 虜 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F970' => '54ED', # 殺 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F9D0' => '6CF5', # 類 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'F9DC' => '68CC', # 隆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA10' => 'C4CD', # 塚 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA16' => '58EF', # 猪 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA19' => '5CDA', # 神 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA1A' => '5CDC', # 祥 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA1B' => '5CE4', # 福 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA22' => '65B7', # 諸 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA26' => '68A8', # 都 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA30' => '42BB', # 侮 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA31' => '43B7', # 僧 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA32' => '43D5', # 免 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA33' => '44E7', # 勉 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA34' => '44EF', # 勤 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA35' => '45C3', # 卑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA36' => 'B3E5', # 喝 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA37' => '46F7', # 嘆 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA38' => '45F4', # 器 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA39' => '48B9', # 塀 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3A' => '47DC', # 墨 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3B' => '4AE0', # 層 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3D' => '4EAD', # 悔 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA3E' => '4ED6', # 慨 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA40' => '4DE8', # 懲 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA41' => '50CA', # 敏 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA42' => '50F4', # 既 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA43' => '51AA', # 暑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA44' => '53D3', # 梅 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4A' => 'C2F6', # 琢 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA4B' => '5CB3', # 碑 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA57' => '5FC8', # 練 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA59' => '5EE3', # 繁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA5F' => '62B3', # 著 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA60' => 'B3EC', # 褐 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA61' => '64C4', # 視 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA62' => '65BB', # 謁 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA63' => '65C4', # 謹 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA64' => '65F4', # 賓 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA65' => '66AF', # 贈 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA67' => '67D5', # 逸 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA68' => '6BFD', # 難 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA69' => '6CD9', # 響 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)
    'FA6A' => 'C9D1', # 頻 (戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字)

    # Jacode4e original mapping

    '00B6' => '7FEF', # ¶
    '2020' => '7FED', # †
    '2021' => '7FEE', # ‡
    '2030' => '76D3', # ‰
    '212B' => '76D1', # Å
    '21D2' => '7FDA', # ⇒
    '21D4' => '7FDB', # ⇔
    '2200' => '7FDC', # ∀
    '2202' => '7FE1', # ∂
    '2203' => '7FDD', # ∃
    '2207' => '7FE2', # ∇
    '2208' => '7FD0', # ∈
    '220B' => '7FD1', # ∋
    '2211' => '',     # ∑
    '221A' => '7FE5', # √
    '221D' => '7FE7', # ∝
    '221F' => '',     # ∟
    '2220' => '7FDE', # ∠
    '2227' => '7FD8', # ∧
    '2228' => '7FD9', # ∨
    '2229' => '7FD7', # ∩
    '222A' => '7FD6', # ∪
    '222B' => '7FE8', # ∫
    '222C' => '7FE9', # ∬
    '222E' => '',     # ∮
    '2235' => '76A8', # ∵
    '223D' => '7FE6', # ∽
    '2252' => '76A9', # ≒
    '2261' => '76AA', # ≡
    '226A' => '7FE3', # ≪
    '226B' => '7FE4', # ≫
    '2282' => '7FD4', # ⊂
    '2283' => '7FD5', # ⊃
    '2286' => '7FD2', # ⊆
    '2287' => '7FD3', # ⊇
    '22A5' => '7FDF', # ⊥
    '22BF' => '',     # ⊿
    '2312' => '7FE0', # ⌒
    '2500' => '7CD1', # ─
    '2501' => '7CF6', # ━
    '2502' => '7CD2', # │
    '2503' => '7CF7', # ┃
    '250C' => '7CC1', # ┌
    '250F' => '7CE6', # ┏
    '2510' => '7CC2', # ┐
    '2513' => '7CE7', # ┓
    '2514' => '7CC4', # └
    '2517' => '7CE9', # ┗
    '2518' => '7CC3', # ┘
    '251B' => '7CE8', # ┛
    '251C' => '7CC7', # ├
    '251D' => '7FA2', # ┝
    '2520' => '7CCA', # ┠
    '2523' => '7CF3', # ┣
    '2524' => '7CC8', # ┤
    '2525' => '7FA4', # ┥
    '2528' => '7CCC', # ┨
    '252B' => '7CF4', # ┫
    '252C' => '7CC5', # ┬
    '252F' => '7CCB', # ┯
    '2530' => '7FA3', # ┰
    '2533' => '7CF1', # ┳
    '2534' => '7CC6', # ┴
    '2537' => '7CC0', # ┷
    '2538' => '7FA1', # ┸
    '253B' => '7CF2', # ┻
    '253C' => '7CC9', # ┼
    '253F' => '7CCE', # ┿
    '2542' => '7CCD', # ╂
    '254B' => '7CF5', # ╋
    '25EF' => '7FF0', # ◯
    '266A' => '7FEC', # ♪
    '266D' => '7FEB', # ♭
    '266F' => '7FEA', # ♯
    '301C' => 'A1C1', # ～
    '301D' => '',     # 〝
    '301F' => '',     # 〟
    '32A4' => '',     # ㊤
    '32A5' => '',     # ㊥
    '32A6' => '',     # ㊦
    '32A7' => '',     # ㊧
    '32A8' => '',     # ㊨
    '3351' => '',     # ㍑
    '337B' => '',     # ㍻
    '337C' => '',     # ㍼
    '337D' => '',     # ㍽
    '337E' => '',     # ㍾
    '51DC' => '44A4', # 凜
    '582F' => 'B6C6', # 堯
    '5C2D' => '47C8', # 尭
    '69C7' => 'CBEA', # 槇
    '69D9' => '54C4', # 槙
    '7199' => '58A8', # 熙
    '7464' => 'E0F6', # 瑤
    '7476' => '59A2', # 瑶
    '9059' => 'CDDA', # 遙
    '9065' => '70EB', # 遥
    'FFE2' => '76A7', # ￢
);

my %JEF_by_Unicode_OVERRIDE = @JEF_by_Unicode_OVERRIDE;
if (scalar(keys %JEF_by_Unicode_OVERRIDE) != (scalar(@JEF_by_Unicode_OVERRIDE) / 2)) {
    my %key = ();
    while (my($key) = splice(@JEF_by_Unicode_OVERRIDE,0,2)) {
        $key{$key}++;
    }
    for my $key (sort keys %key) {
        if ($key{$key} >= 2) {
            print STDERR "[$key], ";
        }
    }
    die "Duplicate hash key(must unique): %JEF_by_Unicode_OVERRIDE\n";
}

my %unicode = map { $_ => 1 } (
    keys_of_CP932_by_Unicode(),
    keys_of_JEF_by_Unicode_CultiCoLtd(),
    keys %JEF_by_Unicode_OVERRIDE,
);

my %JEF_by_Unicode = ();
my %done = ();

for my $unicode (sort { (length($a) <=> length($b)) || ($a cmp $b) } keys %unicode) {

    # override (not defined)
    if (exists($JEF_by_Unicode_OVERRIDE{$unicode}) and ($JEF_by_Unicode_OVERRIDE{$unicode} eq '')) {
        $done{$JEF_by_Unicode_OVERRIDE{$unicode}} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", '----', $unicode, '----', '----', '----', '----', '----';
    }

    # override (defined)
    elsif (($JEF_by_Unicode_OVERRIDE{$unicode} ne '') and not $done{$JEF_by_Unicode_OVERRIDE{$unicode}}) {
        $done{$JEF_by_Unicode{$unicode} = $JEF_by_Unicode_OVERRIDE{$unicode}} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", $JEF_by_Unicode{$unicode}, $unicode, $JEF_by_Unicode{$unicode}, '----', '----', '----', '----';
    }

    # CP932
    elsif ((JEF_by_CP932(CP932_by_Unicode($unicode)) ne '') and not $done{JEF_by_CP932(CP932_by_Unicode($unicode))}) {
        $done{$JEF_by_Unicode{$unicode} = JEF_by_CP932(CP932_by_Unicode($unicode))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", $JEF_by_Unicode{$unicode}, $unicode, '----', $JEF_by_Unicode{$unicode}, '----', '----', '----';
    }

    # JIS78 <-> JIS83
    elsif ((JIS78GR_by_JIS83GR(JISX0208GR_by_CP932(CP932_by_Unicode($unicode))) ne '') and not $done{JIS78GR_by_JIS83GR(JISX0208GR_by_CP932(CP932_by_Unicode($unicode)))}) {
        $done{$JEF_by_Unicode{$unicode} = JIS78GR_by_JIS83GR(JISX0208GR_by_CP932(CP932_by_Unicode($unicode)))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", $JEF_by_Unicode{$unicode}, $unicode, '----', '----', $JEF_by_Unicode{$unicode}, '----', '----';
    }

    # JIS X 0208:1990
    elsif ((JISX0208GR_by_CP932(CP932_by_Unicode($unicode)) ne '') and not $done{JISX0208GR_by_CP932(CP932_by_Unicode($unicode))}) {
        $done{$JEF_by_Unicode{$unicode} = JISX0208GR_by_CP932(CP932_by_Unicode($unicode))} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", $JEF_by_Unicode{$unicode}, $unicode, '----', '----', '----', $JEF_by_Unicode{$unicode}, '----';
    }

    # JIS X 0213:2004
    elsif ((JEF_by_Unicode_CultiCoLtd($unicode) ne '') and not $done{JEF_by_Unicode_CultiCoLtd($unicode)}) {
        $done{$JEF_by_Unicode{$unicode} = JEF_by_Unicode_CultiCoLtd($unicode)} = 1;
printf DUMP "%-4s %-9s %-4s %-4s %-4s %-4s %-4s \n", $JEF_by_Unicode{$unicode}, $unicode, '----', '----', '----', '----', $JEF_by_Unicode{$unicode};
    }

    if (
        ($JEF_by_Unicode{$unicode} ne '') and
        (JEF_by_Unicode_CultiCoLtd($unicode) ne '') and
        ($JEF_by_Unicode{$unicode} ne JEF_by_Unicode_CultiCoLtd($unicode)) and
    1) {
        printf DIFF ("[%s] (%s) (%s) (%s)\n",
            (CP932_by_Unicode($unicode) ne '') ? pack('H*',CP932_by_Unicode($unicode)) : (' ' x 2),
            $unicode || (' ' x 4),
            $JEF_by_Unicode{$unicode} || (' ' x 4),
            JEF_by_Unicode_CultiCoLtd($unicode) || (' ' x 4),
        );
    }

    if ((JEF_by_Unicode_CultiCoLtd($unicode) ne '') and (JEF_by_CP932(CP932_by_Unicode($unicode)) ne '')) {
        if (JEF_by_Unicode_CultiCoLtd($unicode) ne JEF_by_CP932(CP932_by_Unicode($unicode))) {
die sprintf "Unicode=($unicode), CultiCoLtd=(%s) Handmade=(%s)\n", JEF_by_Unicode_CultiCoLtd($unicode), JEF_by_CP932(CP932_by_Unicode($unicode));
        }
    }
}

close(DUMP);
close(DIFF);

my @jis83_compatible_glyph_1 = qw(
43CB
43CC
43CD
43CE
43DA
43DB
43E9
43EE
43F0
44A5
44A6
44D0
45D5
45E4
45EC
47B8
47B9
47BA
47BB
47BC
47C8
47C9
47CA
47CB
48C7
48C8
49EC
49ED
49EE
49EF
4AC8
4AE4
4AE5

4AE6
4AE7
4BF4
4BF5
4CD1
4CD2
4CD3
4DC1
4DC2
4EEA
4EEB
4EEC
4EF9
4FA1
4FA2
4FBB
50B1
50B7
50B8
50B9
50BA
50BB
50BC
50FC
51FD
51FE
52CD
52E2
52E3
52E4
52E5
53BE
54C4

54C5
54C6
54C7
54C8
54C9
55AE
55AF
57A6
57C6
57C7
57C8
57C9
58AA
58AB
58AC
58AD
58AF
58BF
58C5
58F9
59A1
59A2
59A3
5AAB
5AB6
5AB7
5AF4
5AF5
5BAC
5BE3
5BEC
5CCD
5CCF

5CD0
5CEC
5CED
5DC6
5DC7
5DC8
5DD7
5DEA
5DEB
5EC2
5EC3
5EC4
5EC5
5EC6
5EDD
5EDE
5FDF
5FE0
5FE1
5FE2
5FE3
5FFD
5FFE
60B2
60B3
60FE
61B3
61B4
61B5
61B6
61B7
61B8
61B9

61EA
62FC
62FD
62FE
63A1
63A2
63A3
63A4
63A5
63B3
63B4
63B5
63DE
63EE
63EF
64B4
64CC
64D5
64D6
64D7
65D4
65D5
66B2
66B8
66E2
66EA
67F4
67F5
68D2
68D9
68DE
69A1
69A2

69A9
69AA
69AB
6BFE
6CCA
6CCB
6CCC
6CFB
6CFC
6CFD
6DC8
6DC9
6DCA
6DCB
6DCC
6DCD
6DF8
6EC7
6EC8
6EC9
6EEC
6EED
6EF6
6FB0
6FB1
6FB5
6FC3
6FC4
6FC5
6FD3
6FD8
70B6
70B7

70B8
70B9
70BA
70BB
70BC
70BD
70BE
70BF
70C0
70C1
70C3
70C4
70C5
70C6
70C7
70C8
70C9
70CA
70CB
70CC
70D4
70D5
70D6
70D7
70D8
70D9
70DA
70DB
70DC
70DD
70E0
70E4
70E6

70EB
70EC
70ED
70EE
70EF
70F0
70F1
70F2
70F3
70F4
70F5
70F6
70F7
70F8
70FC
);

my @jis83_compatible_glyph_2 = qw(
43CB
43CC
43CD
43CE
43DA
43DB
43E9
43EE
43F0
44A5
44A6
44D0
45D5
45E4
45EC
47B8
47B9
47BA
47BB
47BC
47C8
47C9
47CA
47CB
48C7
48C8
49EC
49ED
49EE
49EF
4AC8
4AE4
4AE5

4AE6
4AE7
4BF4
4BF5
4CD1
4CD2
4CD3
4DC1
4DC2
4EEA
4EEB
4EEC
4EF9
4FA1
4FA2
4FBB
50B1
50B7
50B8
50B9
50BA
50BB
50BC
50FC
51FD
51FE
52CD
52E2
52E3
52E4
52E5
53BE
54C4

54C5
54C6
54C7
54C8
54C9
55AE
55AF
57A6
57C6
57C7
57C8
57C9
58AA
58AB
58AC
58AD
58AF
58BF
58C5
58F9
59A1
59A2
59A3
5AAB
5AB6
5AB7
5AF4
5AF5
5BAC
5BE3
5BEC
5CCD
5CCF

5CD0
5CEC
5CED
5DC6
5DC7
5DC8
5DD7
5DEA
5DEB
5EC2
5EC3
5EC4
5EC5
5EC6
5EDD
5EDE
5FDF
5FE0
5FE1
5FE2
5FE3
5FFD
5FFE
60B2
60B3
60FE
61B3
61B4
61B5
61B6
61B7
61B8
61B9

61EA
62FC
62FD
62FE
63A1
63A2
63A3
63A4
63A5
63B3
63B4
63B5
63DE
63EE
63EF
64B4
64CC
64D5
64D6
64D7
65D4
65D5
66B2
66B8
66E2
66EA
67F4
67F5
68D2
68D9
68DE
69A1
69A2

69A9
69AA
69AB
6BFE
6CCA
6CCB
6CCC
6CFB
6CFC
6CFD
6DC8
6DC9
6DCA
6DCB
6DCC
6DCD
6DF8
6EC7
6EC8
6EC9
6EEC
6EED
6EF6
6FB0
6FB1
6FB5
6FC3
6FC4
6FC5
6FD3
6FD8
70B6
70B7

70B8
70B9
70BA
70BB
70BC
70BD
70BE
70BF
70C0
70C1
70C3
70C4
70C5
70C6
70C7
70C8
70C9
70CA
70CB
70CC
70D4
70D5
70D6
70D7
70D8
70D9
70DA
70DB
70DC
70DD
70E0
70E4
70E6

70EB
70EC
70ED
70EE
70EF
70F0
70F1
70F2
70F3
70F4
70F5
70F6
70F7
70F8
70FC
);

if ("@jis83_compatible_glyph_1" ne"@jis83_compatible_glyph_2") {
    for my $a (@jis83_compatible_glyph_1) {
        my $b = shift @jis83_compatible_glyph_2;
        if ($a ne $b) {
            die "a($a) ne b($b)";
        }
    }
}

if (grep( ! /^[4567][0123456789ABCDEF]{3}$/, @jis83_compatible_glyph_1)) {
    die;
}

if (qq{@{[ sort @jis83_compatible_glyph_1 ]}} ne "@jis83_compatible_glyph_1") {
    die;
}

my %jis83_compatible_glyph_1 = map { $_ => 1 } @jis83_compatible_glyph_1;
if (qq{@{[ sort keys %jis83_compatible_glyph_1 ]}} ne "@jis83_compatible_glyph_1") {
    die;
}

# JEF includes JIS83, JIS90, and JIS2004 fonts as extended character sets, but this software does not use them
for my $unicode (sort { (length($a) <=> length($b)) || ($a cmp $b) } keys %unicode) {

    # JIS83
    if (($unicode =~ /^(?:

        5C2D | # => 47C8 尭
        69D9 | # => 54C4 槙
        7476 | # => 59A2 瑶
        9065 | # => 70EB 遥

        4FA0 | # => 43CB 侠
        5516 | # => 47B9 唖
        565B | # => 47BC 噛
        56A2 | # => 70BA 嚢
        586B | # => 47CA 填
        5C61 | # => 4AE5 屡
        63B4 | # => 50BB 掴
        63BB | # => 50B9 掻
        6505 | # => 50B1 攅
        6D9C | # => 70D7 涜
        6E8C | # => 70DA 溌
        7114 | # => 70E0 焔
        7977 | # => 5CEC 祷
        7BAA | # => 5EC2 箪
        7E4D | # => 5FE0 繍
        83B1 | # => 63A2 莱
        848B | # => 61B9 蒋
        8749 | # => 63B3 蝉
        874B | # => 63B5 蝋
        8EAF | # => 66EA 躯
        9197 | # => 69A2 醗
        91A4 | # => 69A1 醤
        982C | # => 6CFD 頬
        985B | # => 6CFC 顛
        9A28 | # => 6DF8 騨
        9D0E | # => 6FB0 鴎
        9E78 | # => 6FB5 鹸
        9EB9 | # => 6FC3 麹
        9EBA   # => 6FC4 麺

    )$/x) and ($JEF_by_Unicode{$unicode} ne '')) {
        next;
    }

    # JIS2004
    if (($unicode =~ /^(?:

        4FF1  | # => 42CF # 俱
        5265  | # => 44D0 # 剥
        20B9F | # => '',  # 𠮟
        541E  | # => '',  # 吞
        5618  | # => 47BA # 嘘
        59F8  | # => 49B5 # 姸
        5C4F  | # => 4AE7 # 屏
        5E77  | # => 4CB2 # 幷
        75E9  | # => 5AF5 # 痩
        7E4B    # => 5FDF # 繋

    )$/x) and ($JEF_by_Unicode{$unicode} ne '')) {
        next;
    }

    # 戸籍法施行規則付則別表 "人名用漢字許容字体表" (昭和56年法務省令51) 常用漢字表(昭和56年内閣告示第1号)のかっこ書き内の漢字
    if (($unicode =~ /^(?:

        559D  | # => 47B8 # 喝
        585A  | # => 47C9 # 塚
        6602  | # => 50FC # 昂
        6E1A  | # => 55AF # 渚
        7422  | # => 59A1 # 琢
        8910  | # => 63EE # 褐
        983B  | # => 6CFB # 頻

        FA30  | # => 42BB # 侮
        5002  | # => ''   # 倂
        FA31  | # => 43B7 # 僧
        FA32  | # => 43D5 # 免
        FA33  | # => 44E7 # 勉
        FA34  | # => 44EF # 勤
        FA35  | # => 45C3 # 卑
        537D  | # => 45D4 # 卽
        FA36  | # => B3E5 # 喝
        FA37  | # => 46F7 # 嘆
        FA38  | # => 45F4 # 器
        FA10  | # => C4CD # 塚
        FA39  | # => 48B9 # 塀
        589E  | # => 48BB # 增
        FA3A  | # => 47DC # 墨
        5BEC  | # => 4AC6 # 寬
        FA3B  | # => 4AE0 # 層
        5DE2  | # => 52F1 # 巢
        F928  | # => 4CCA # 廊
        5FB5  | # => 4DBF # 徵
        5FB7  | # => 4DC0 # 德
        FA3D  | # => 4EAD # 悔
        FA3E  | # => 4ED6 # 慨
        FA3F  | # => 4ED8 # 憎
        FA40  | # => 4DE8 # 懲
        623E  | # => 4EFB # 戾
        63ED  | # => 4FEA # 揭
        64CA  | # => 4FA6 # 擊
        FA41  | # => 50CA # 敏
        FA42  | # => 50F4 # 既
        665A  | # => 51D5 # 晚
        FA43  | # => 51AA # 暑
        66C6  | # => 51B6 # 曆
        F929  | # => 52AD # 朗
        FA44  | # => 53D3 # 梅
        69EA  | # => 53FA # 槪
        6A6B  | # => 54AD # 橫
        F91D  | # => 54C0 # 欄
        6B65  | # => 54DC # 步
        6B77  | # => 54DF # 歷
        F970  | # => 54ED # 殺
        6BCF  | # => 54F4 # 每
        FA45  | # => 55FB # 海
        6D89  | # => 56A5 # 涉
        6DDA  | # => 56B4 # 淚
        FA46  | # => BDED # 渚
        6E34  | # => 56C0 # 渴
        6EAB  | # => 56DB # 溫
        FA47  | # => 56E6 # 漢
        7028  | # => 57BC # 瀨
        FA48  | # => 58A7 # 煮
        72C0  | # => 58D7 # 狀
        FA16  | # => 58EF # 猪
        FA4A  | # => C2F6 # 琢
        7501  | # => 5AB8 # 甁
        784F  | # => 5CA8 # 硏
        FA4B  | # => 5CB3 # 碑
        FA4C  | # => 5CD3 # 社
        FA4D  | # => 5CD5 # 祉
        FA4E  | # => 5CD4 # 祈
        FA4F  | # => 5CD6 # 祐
        FA50  | # => 5CD7 # 祖
        FA51  | # => 5CD9 # 祝
        FA19  | # => 5CDA # 神
        FA1A  | # => 5CDC # 祥
        FA52  | # => 5CE2 # 禍
        FA53  | # => 5CE3 # 禎
        FA1B  | # => 5CE4 # 福
        FA54  | # => 5CF4 # 穀
        FA55  | # => 5DCD # 突
        FA56  | # => ''   # 節
        7DA0  | # => 5FB6 # 綠
        7DD6  | # => 5FBE # 緖
        7DE3  | # => 5FC1 # 緣
        FA57  | # => 5FC8 # 練
        FA59  | # => 5EE3 # 繁
        FA5A  | # => 5FF0 # 署
        FA5B  | # => 60B5 # 者
        FA5C  | # => 60E3 # 臭
        FA5F  | # => 62B3 # 著
        85B0  | # => 62E5 # 薰
        865B  | # => 63AA # 虛
        F936  | # => 63AB # 虜
        FA60  | # => B3EC # 褐
        FA61  | # => 64C4 # 視
        FA22  | # => 65B7 # 諸
        FA62  | # => 65BB # 謁
        FA63  | # => 65C4 # 謹
        FA64  | # => 65F4 # 賓
        8CF4  | # => 65F9 # 賴
        FA65  | # => 66AF # 贈
        FA67  | # => 67D5 # 逸
        90DE  | # => 68A6 # 郞
        FA26  | # => 68A8 # 都
        9115  | # => 68DF # 鄕
        9304  | # => 6ACD # 錄
        934A  | # => 6ADB # 鍊
        F9DC  | # => 68CC # 隆
        FA68  | # => 6BFD # 難
        FA69  | # => 6CD9 # 響
        FA6A  | # => C9D1 # 頻
        F9D0  | # => 6CF5 # 類
        9EC3  | # => 6FCC # 黃
        9ED1    # => 6FD6 # 黑

    )$/x) and ($JEF_by_Unicode{$unicode} ne '')) {
        next;
    }

    # JIS83 and JIS90
    if (defined $jis83_compatible_glyph_1{$JEF_by_Unicode{$unicode}}) {
        warn "Should not use JIS83 or JIS90 glyph to compatibility of information processing(Unicode=U+$unicode, JEF=$JEF_by_Unicode{$unicode}).\n";
    }
}

sub JEF_by_Unicode {
    my($unicode) = @_;
    return $JEF_by_Unicode{$unicode};
}

sub keys_of_JEF_by_Unicode {
    return keys %JEF_by_Unicode;
}

sub values_of_JEF_by_Unicode {
    return values %JEF_by_Unicode;
}

1;

__END__
