package Module::CPANTS::Kwalitee::Distname;
use warnings;
use strict;
use CPAN::DistnameInfo;

sub order { 15 }

##################################################################
# Analyse
##################################################################

sub analyse {
    my $class=shift;
    my $me=shift;
    
    my $di=CPAN::DistnameInfo->new($me->dist);
    my ($major,$minor);
    if ($di->version) {
        ($major,$minor)=$di->version=~/^(\d+)\.(.*)/;
    }
    $major=0 unless defined($major);
    my $ext=$di->extension || 'unknown';
    
    $me->d->{package}=$di->filename;
    $me->d->{vname}=$di->distvname;
    $me->d->{extension}=$ext;
    $me->d->{version}=$di->version;
    $me->d->{version_major}=$major;
    $me->d->{version_minor}=$minor;
    $me->d->{dist}=$di->dist;
    $me->d->{author}=$di->cpanid;

    unless($me->d->{package}) {
        $me->d->{package}=$me->tarball;
    }
    
    # TODO
    # some authors have dirs on CPAN containing their dist:
    # id/R/RM/RMCFARLA/AI-LibNeural/AI-LibNeural-0.02.tar.gz
    # hack around this...
    #$to=~s|^[\w-]+/||;

    return;
}


##################################################################
# Kwalitee Indicators
##################################################################

sub kwalitee_indicators {
    return [
        {
            name=>'has_version',
            error=>"The package filename (eg. Foo-Bar-1.42.tar.gz) does not include a version number (or something that looks like a reasonable version number to CPAN::DistnameInfo)",
            remedy=>q{Add a version number to the packed distribution. Or use a buildtool ('make dist' or 'Build dist')},
            code=>sub { shift->{version} ? 1 : 0 }
        },
        {
            name=>'has_proper_version',
            error=>"The version number isn't a number. It probably contains letter besides a leading 'v', which it shouldn't",
            remedy=>q{Remove all letters from the version number. If you want to mark a release as a developer release, use the scheme 'Module-1.00_01'},
            code=>sub { my $v=shift->{version};
                 return 0 unless $v;
                 return 1 if ($v=~ /\A v? \d+ (?:\.\d+)* (?:_\d+)? \z/xi );
                 return 0;
            }
        },
    ];
}


q{Favourite record of the moment:
  Fat Freddys Drop: Based on a true story};


__END__

=pod

=head1 NAME

Module::CPANTS::Kwalitee::Distname - Proper Distname layout

=head1 SYNOPSIS

Checks if a distname is well-formed.

A wellformed distname looks like C<Foo-Bar-1.42.tgz>

=head1 DESCRIPTION

=head2 Methods

=head3 order

Defines the order in which Kwalitee tests should be run.

Returns C<20>, as data generated by C<MCK::Distname> is used by other tests.

=head3 analyse

C<MCK::Distname> uses C<CPAN::DistnameInfo> to split a distname into it's components.

=head3 kwalitee_indicators

Returns the Kwalitee Indicators datastructure.

=over

=item * has_version

=item * has_proper_version

=back

=head1 SEE ALSO

L<Module::CPANTS::Analyse>

=head1 AUTHOR

Thomas Klausner, <domm@cpan.org>, http://domm.zsi.at

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2003-2006, 2009  Thomas Klausner

You may use and distribute this module according to the same terms
that Perl is distributed under.

=cut
