/***************************************************************
 * Name:      dxreminders.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#ifdef WX_PRECOMP
#include "wx_pch.h"
#endif

#ifdef __BORLANDC__
#pragma hdrstop
#endif //__BORLANDC__

#include <wx/aboutdlg.h>
#include <wx/listctrl.h>
#include <wx/stdpaths.h>
#include <wx/sysopt.h>
#include <wx/numdlg.h>
#include <random>
#ifdef __WXMSW__
    #include <wx/msw/registry.h>
#endif
#include "dxdefs.h"
#include "config.h"
#include "dxreminders.h"
#include "dxsettings.h"
#include "eventdialog.h"
#include "phdialog.h"
#include "daydialog.h"
#include "dxoptions.h"
#include "dxutils.h"
#include "dxicons.h"
#include "mergedialog.h"
#include "calendarpopup.h"
#include "passdialog.h"
#include "filters/eventfilter.h"
#include "helpdialog.h"
#ifdef HAVE_CURL
    #include <curl/curlver.h>
#endif

#define DEFAULT_PORT 61666

dxremindersFrame* frame = nullptr;
bool frameShown = true;
CalendarPopup* calendar = nullptr;

IMPLEMENT_APP(dxremindersApp)

bool dxremindersApp::OnInit()
{
    m_checker = new wxSingleInstanceChecker;
    if(m_checker->IsAnotherRunning())
    {
        wxLogError(_("dxreminders already running"));
        delete m_checker; // OnExit() won't be called if we return false
        m_checker = nullptr;
        return false;
    }
    bool ipc =false;
    if(this->argc > 1)
    {
        if(this->argv[1].CmpNoCase("-p")==0)
        {
            ipc = true;
        }
    }
    wxInitAllImageHandlers();
    dxsettings.load();
    makeAllIcons("default");
#if defined (__WXMAC__)
    wxSystemOptions::SetOption(_T("mac.listctrl.always_use_generic"),1);
#endif
    wxLocale::AddCatalogLookupPathPrefix(LOCALEDIR);
    m_Locale = new wxLocale();
    m_Locale->Init(wxLANGUAGE_DEFAULT);
    m_Locale->AddCatalog("dxreminders");
    m_Locale->AddCatalog("wxstd");
    dxsettings.setLocale(m_Locale->GetCanonicalName());

    wxSQLite3Database::InitializeSQLite();

    if(ipc)
    {
        dxsettings.setHideTrayIcon(true);
        dxsettings.setStartOnLogin(false);
    }
    frame = new dxremindersFrame(nullptr, "dxReminders", ipc);
    frame->SetIcon(ICO_ICON); // To Set App Icon
    if(dxsettings.startAtTray())
    {
        frame->Show(false);
        frameShown = !frameShown;
    }
    else
    {
        frame->Show();
    }

    if(ipc)
    {
        frame->Hide();
        frameShown = false;
    }

    return true;
}

int dxremindersApp::OnExit()
{
    wxSQLite3Database::ShutdownSQLite();
    delete m_checker;
    return 0;
}

BEGIN_EVENT_TABLE(dxTaskBarIcon, wxTaskBarIcon)
    EVT_TASKBAR_LEFT_DCLICK(dxTaskBarIcon::OnLeftButtonDClick)
    EVT_MENU(ID_TASK_RESTORE, dxTaskBarIcon::OnRestore)
    EVT_MENU(ID_MENU_NEWEVENT, dxTaskBarIcon::OnNewEvent)
    EVT_MENU(wxID_EXIT, dxTaskBarIcon::OnQuit)
    EVT_MENU(ID_TASK_QUIT, dxTaskBarIcon::OnQuit)
    EVT_MENU(wxID_ABOUT, dxTaskBarIcon::OnAbout)
    EVT_MENU(wxID_HELP, dxTaskBarIcon::OnHelp)
    EVT_MENU(ID_TASK_PREFERENCES, dxTaskBarIcon::OnPreferences)
    EVT_MENU(ID_MENU_TEST, dxTaskBarIcon::OnTestEvents)
    EVT_MENU(ID_MENU_LOADDB, dxTaskBarIcon::OnLoadDB)
    EVT_MENU(ID_MENU_SAVEDB, dxTaskBarIcon::OnSaveDB)
    EVT_MENU(ID_MENU_HOLIDAYS, dxTaskBarIcon::OnHolidays)
    EVT_MENU(ID_MENU_SHOWHIDENOTE, dxTaskBarIcon::OnShowHideNote)
    EVT_MENU(ID_MENU_REMOVEALLEVENTS, dxTaskBarIcon::OnRemoveEvents)
    EVT_MENU(ID_MENU_CHECKVERSION, dxTaskBarIcon::OnCheckVersion)
END_EVENT_TABLE()

void dxTaskBarIcon::OnLeftButtonDClick(wxTaskBarIconEvent&)
{
    if(frameShown)
    {
        frame->Show(false);
    }
    else
    {
        frame->Iconize(false);
        frame->SetFocus();
        frame->Raise();
        frame->Show(true);
#if defined (__WXMAC__)
        frame->refreshFrame();
#endif
    }
    frameShown = !frameShown;
}

wxMenu *dxTaskBarIcon::CreatePopupMenu()
{
    wxMenu *menu = new wxMenu;
    if(frameShown)
        menu->Append(ID_TASK_RESTORE, _("Hide"));
    else
        menu->Append(ID_TASK_RESTORE, _("Restore"));
    menu->AppendSeparator();
    menu->Append(ID_MENU_NEWEVENT, _("New event"));
    if(iengine->someEvents())
    {
        menu->AppendSeparator();
        menu->Append(ID_MENU_REMOVEALLEVENTS, _("Remove all events"));
    }
#if !(defined (__WXMAC__))
    menu->AppendSeparator();
    menu->Append(wxID_EXIT);
#else
    menu->AppendSeparator();
    menu->Append(ID_MENU_SAVEDB, _("Backup database"));
    menu->Append(ID_MENU_LOADDB, _("Import database"));
#ifdef DEBUG
    menu->Append(ID_MENU_TEST, "Testovaci nahodna data");
#endif
    menu->AppendSeparator();
    menu->Append(ID_TASK_PREFERENCES, _("Preferences"));
    menu->Append(ID_MENU_HOLIDAYS, _("Public holidays"));
    menu->AppendSeparator();
#ifdef HAVE_CURL
    menu->Append(ID_MENU_CHECKVERSION, _("Check for updates"));
    menu->AppendSeparator();
#endif
    menu->Append(wxID_HELP);
    menu->AppendSeparator();
    menu->Append(wxID_ABOUT);
    menu->AppendSeparator();
    menu->Append(ID_TASK_QUIT, _("Quit"));
#endif
    return menu;
}

void dxTaskBarIcon::OnRestore(wxCommandEvent& /*event*/)
{
    if(frameShown)
    {
        frame->Show(false);
    }
    else
    {
        frame->Iconize(false);
        frame->SetFocus();
        frame->Raise();
        frame->Show(true);
#if defined (__WXMAC__)
        frame->refreshFrame();
#endif
    }
    frameShown = !frameShown;
}

void dxTaskBarIcon::OnQuit(wxCommandEvent& event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnNewEvent(wxCommandEvent& event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnRemoveEvents(wxCommandEvent& event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnCheckVersion(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnAbout(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnHelp(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnPreferences(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnLoadDB(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnSaveDB(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnTestEvents(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnHolidays(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void dxTaskBarIcon::OnShowHideNote(wxCommandEvent &event)
{
    wxPostEvent(frame, event);
}

void EventListCtrl::SetColumnHeaderFiltered(int column, bool filtered)
{
    wxListItem item;
    item.SetMask(wxLIST_MASK_TEXT);
    GetColumn(column, item);
    if(filtered && item.GetText().Right(1) != "*")
    {
        item.SetText(item.GetText()+"*");
    }
    if(!filtered && item.GetText().Right(1) == "*")
    {
        item.SetText(item.GetText().Left(item.GetText().Len()-1));
    }
    SetColumn(column, item);
}

void EventListCtrl::ClearSelection()
{
    for(int i=0; i<GetItemCount(); i++)
    {
        Select(i, false);
    }
}

wxString EventListCtrl::OnGetItemText(long item, long column) const
{
    return frame->getItemText(item, column);
}

int EventListCtrl::OnGetItemImage(long /*item*/) const
{
    return 0;
}

wxListItemAttr *EventListCtrl::OnGetItemAttr(long item) const
{
    return frame->getItemAttr(item);
}

BEGIN_EVENT_TABLE(dxremindersFrame, wxFrame)
    EVT_CLOSE(dxremindersFrame::OnClose)
    EVT_MENU(wxID_EXIT, dxremindersFrame::OnQuit)
    EVT_MENU(ID_TASK_QUIT, dxremindersFrame::OnQuit)
    EVT_MENU(wxID_ABOUT, dxremindersFrame::OnAbout)
    EVT_MENU(wxID_HELP, dxremindersFrame::OnHelp)
    EVT_MENU(ID_MENU_NEWEVENT, dxremindersFrame::OnNewEvent)
    EVT_MENU(ID_MENU_EDITEVENT, dxremindersFrame::OnEditEvent)
    EVT_MENU(ID_MENU_ADVANCE, dxremindersFrame::OnAdvance)
    EVT_MENU(ID_MENU_EDITDATE, dxremindersFrame::OnEditDate)
    EVT_MENU(ID_MENU_REMOVEEVENT, dxremindersFrame::OnRemoveEvent)
    EVT_MENU(ID_BTN_NEWEVENT, dxremindersFrame::OnNewEvent)
    EVT_MENU(ID_BTN_EDITEVENT, dxremindersFrame::OnEditEvent)
    EVT_MENU(ID_BTN_REMOVEEVENT, dxremindersFrame::OnRemoveEvent)
    EVT_LIST_ITEM_RIGHT_CLICK(ID_EVENTLIST, dxremindersFrame::OnItemRightClick)
    EVT_LIST_ITEM_ACTIVATED(ID_EVENTLIST, dxremindersFrame::OnItemActivated)
    EVT_LIST_ITEM_SELECTED(ID_EVENTLIST, dxremindersFrame::OnItemSelected)
    EVT_LIST_ITEM_DESELECTED(ID_EVENTLIST, dxremindersFrame::OnItemDeselected)
    EVT_LIST_END_LABEL_EDIT(ID_EVENTLIST, dxremindersFrame::OnItemEditEnd)
    EVT_TIMER(wxID_ANY, dxremindersFrame::OnTimer)
    EVT_CALENDAR(ID_CALENDAR, dxremindersFrame::OnCalendar)
    EVT_CALENDAR_SEL_CHANGED(ID_CALENDAR, dxremindersFrame::OnCalendarChanged)
    EVT_MENU(wxID_PREFERENCES, dxremindersFrame::OnPreferences)
    EVT_MENU(ID_TASK_PREFERENCES, dxremindersFrame::OnPreferences)
    EVT_KEY_UP(dxremindersFrame::OnKeyUp)
    EVT_MENU(ID_MENU_TEST, dxremindersFrame::OnTestEvents)
    EVT_MENU(ID_MENU_LOADDB, dxremindersFrame::OnLoadDB)
    EVT_MENU(ID_MENU_SAVEDB, dxremindersFrame::OnSaveDB)
    EVT_SOCKET(ID_SERVER, dxremindersFrame::OnServerEvent)
    EVT_SOCKET(ID_SOCKET, dxremindersFrame::OnSocketEvent)
    EVT_MENU(ID_MENU_HOLIDAYS, dxremindersFrame::OnHolidays)
    EVT_MENU(ID_MENU_SHOWHIDENOTE, dxremindersFrame::OnShowHideNote)
    EVT_MENU(ID_BTN_HIDEEVENT, dxremindersFrame::OnHideEvent)
    EVT_MENU(ID_MENU_SHOWEVENTS, dxremindersFrame::OnShowEventsMenu)
    EVT_MENU(ID_BTN_SHOWEVENTS, dxremindersFrame::OnShowEventsBtn)
    EVT_COMMAND(wxID_ANY, ENGINE_UPDATED, dxremindersFrame::OnEngineUpdated)
    EVT_COMMAND(wxID_ANY, ENGINE_MSG, dxremindersFrame::OnEngineMsg)
    EVT_MENU(ID_BTN_CLEARHIDDEN, dxremindersFrame::OnClearHiddenList)
    EVT_MENU(ID_RIGHT_SHOWEVENTS, dxremindersFrame::OnShowEventsRightclick)
    EVT_MENU(ID_MENU_REMOVEHIDDEN, dxremindersFrame::OnRemoveFromHiddenList)
    EVT_MENU(ID_MENU_REMOVEALLEVENTS, dxremindersFrame::OnRemoveEvents)
    EVT_MENU(ID_MENU_FULLSCREEN, dxremindersFrame::OnFullscreen)
    EVT_MENU_RANGE(ID_MENU_HIGHLIGHTEVENT1, ID_MENU_HIGHLIGHTEVENT5, dxremindersFrame::OnHighlightEvent)
    EVT_MENU(ID_MENU_REMOVEHIGHLIGHT, dxremindersFrame::OnUnHighlightEvent)
    EVT_MENU(ID_MENU_CLEARHIGHLIGHT, dxremindersFrame::OnClearHighlightList)
    EVT_MENU(ID_RIGHT_HIDEEVENTS, dxremindersFrame::OnHideEventsOnList)
    EVT_MENU(ID_BTN_FILTER, dxremindersFrame::OnFilter)
    EVT_MENU(ID_BTN_FILTERCLEAR, dxremindersFrame::OnClearFilter)
    EVT_MENU(ID_MENU_SHOWNOTDELETE, dxremindersFrame::OnShowDonotdelete)
    EVT_MENU(ID_MENU_ALWAYSSHOW, dxremindersFrame::OnAlwaysShow)
    EVT_MENU(ID_MENU_CALENDARFOCUS, dxremindersFrame::OnCalendarFocus)
    EVT_MENU(ID_MENU_EVENTLISTFOCUS, dxremindersFrame::OnEventlistFocus)
    EVT_MENU(ID_MENU_CHECKVERSION, dxremindersFrame::OnCheckVersion)
END_EVENT_TABLE()

dxremindersFrame::dxremindersFrame(wxFrame *frame, const wxString& title, bool ipc)
    : wxFrame(frame, -1, title, wxDefaultPosition, wxSize(500,250)), m_taskBarEvents(nullptr), m_selectedIndex(-1), m_timer(this), m_ipc(ipc), m_sock(nullptr), m_msgDialog(nullptr),
      m_ipcString(wxEmptyString)
{
    m_hideDonotdelete = dxsettings.hideDonotdelete();
    m_filter = dxsettings.filter();
    m_hidden= dxsettings.hidden();
    m_filters.Add(Filter(static_cast<FilterAction>(dxsettings.eventfilterAction()), dxsettings.eventfilterUsage(), dxsettings.eventfilterValue()));
    m_filters.Add(Filter(static_cast<FilterAction>(dxsettings.datefilterAction()), dxsettings.datefilterUsage(), dxsettings.datefilterValue()));
    m_filters.Add(Filter(static_cast<FilterAction>(dxsettings.reminderfilterAction()), dxsettings.reminderfilterUsage(), dxsettings.reminderfilterValue()));
    m_filters.Add(Filter(FA_EQUAL, dxsettings.recurrencefilterUsage(), dxsettings.recurrencefilterValue()));
    m_filters.Add(Filter(FA_EQUAL, dxsettings.timefilterUsage(), wxDateSpan(dxsettings.timefilterYears(),dxsettings.timefilterMonths(),dxsettings.timefilterWeeks(),dxsettings.timefilterDays()),
                         wxTimeSpan(dxsettings.timefilterHours(),dxsettings.timefilterMinutes())));
    m_filters.Add(Filter(FA_EQUAL, dxsettings.hiddenfilterUsage(), 1));
    if(m_ipc && dxsettings.notaskbar())
    {
        SetWindowStyleFlag(wxDEFAULT_FRAME_STYLE|wxFRAME_NO_TASKBAR);
    }
#if wxUSE_MENUS
    // create a menu bar
    wxMenuBar* mbar = new wxMenuBar();
    wxMenu* fileMenu = new wxMenu();
#if defined (__WXMAC__)
    fileMenu->Append(wxID_ABOUT);
    fileMenu->Append(wxID_PREFERENCES);
#endif
    fileMenu->Append(ID_MENU_NEWEVENT, _("New event")+"\t"+dxsettings.shortcutKey(ID_MENU_NEWEVENT));
    fileMenu->AppendSeparator();
    m_removeEvents = fileMenu->Append(ID_MENU_REMOVEALLEVENTS, _("Remove all events")+"\t"+dxsettings.shortcutKey(ID_MENU_REMOVEALLEVENTS));
    fileMenu->AppendSeparator();
    fileMenu->Append(ID_MENU_SAVEDB, _("Backup database")+"\t"+dxsettings.shortcutKey(ID_MENU_SAVEDB));
    fileMenu->Append(ID_MENU_LOADDB, _("Import database")+"\t"+dxsettings.shortcutKey(ID_MENU_LOADDB));
#ifdef DEBUG
    fileMenu->Append(ID_MENU_TEST, "Testovaci nahodna data");
#endif
#ifndef DEBUG
    if(!m_ipc)
#endif
    {
        fileMenu->AppendSeparator();
        fileMenu->Append(wxID_EXIT);
    }
    mbar->Append(fileMenu, _("&Events"));

    wxMenu *editMenu = new wxMenu();
#if !defined (__WXMAC__)
    editMenu->Append(wxID_PREFERENCES);
    editMenu->AppendSeparator();
#endif
    editMenu->Append(ID_MENU_HOLIDAYS, _("Public holidays"));
    editMenu->AppendSeparator();
    wxMenuItem *note = editMenu->AppendCheckItem(ID_MENU_SHOWHIDENOTE, _("Show/hide note")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWHIDENOTE));
    if(dxsettings.showNote()) note->Check();
    editMenu->AppendSeparator();
    m_filterMenuItem = editMenu->AppendCheckItem(ID_BTN_FILTER, _("Filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTER));
    m_clearfilterMenuItem = editMenu->Append(ID_BTN_FILTERCLEAR, m_filter?_("Clear filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR):_("Revert filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR));
    editMenu->AppendSeparator();
    m_showHiddenEvents = editMenu->AppendCheckItem(ID_MENU_SHOWEVENTS, _("Hide events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS), _("Show/hide events"));
    m_clearHiddenList = editMenu->Append(ID_BTN_CLEARHIDDEN, _("Clear all hidden events"));
    m_clearHighlightList = editMenu->Append(ID_MENU_CLEARHIGHLIGHT, _("Clear all highlighted events"));
    editMenu->AppendSeparator();
    m_showDonotDelete = editMenu->AppendCheckItem(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete?_("Show all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE):
                                                                                           _("Hide all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE),
                                                  _("Show/hide do not delete events"));
    m_showDonotDelete->Check(m_hideDonotdelete);
    m_showDonotDelete->Enable(false);
#if !(defined (__WXMAC__))
    editMenu->AppendSeparator();
    editMenu->AppendCheckItem(ID_MENU_FULLSCREEN, _("Full &screen")+"\t"+dxsettings.shortcutKey(ID_MENU_FULLSCREEN));
#endif
    mbar->Append(editMenu, _("E&dit"));

#if !(defined (__WXMAC__))
    wxMenu* helpMenu = new wxMenu();
#ifdef HAVE_CURL
    helpMenu->Append(ID_MENU_CHECKVERSION, _("Check for &updates"));
    helpMenu->AppendSeparator();
#endif
    helpMenu->Append(wxID_HELP);
    helpMenu->AppendSeparator();
    helpMenu->Append(wxID_ABOUT);
    mbar->Append(helpMenu, _("&Help"));
#endif

    SetMenuBar(mbar);
#endif // wxUSE_MENUS

#if wxUSE_ACCEL
    wxAcceleratorEntry entries[2];
    entries[0].Set(wxACCEL_CTRL, (int)'R', ID_MENU_CALENDARFOCUS);
    entries[1].Set(wxACCEL_CTRL, (int)'L', ID_MENU_EVENTLISTFOCUS);
    SetAcceleratorTable(wxAcceleratorTable(2, entries));
#endif // // wxUSE_MENUS

#if wxUSE_STATUSBAR
    CreateStatusBar(1);
    SetStatusText("dxReminders",0);
#endif // wxUSE_STATUSBAR

    if(!dxsettings.hideTrayIcon())
    {
#if defined(__WXCOCOA__)
        m_taskBarIcon = new dxTaskBarIcon(wxTaskBarIcon::DOCK);
#else
        m_taskBarIcon = new dxTaskBarIcon();
        m_taskBarIcon->SetIcon(ICO_TRAY);
#endif
    }
    else
    {
        m_taskBarIcon = nullptr;
    }

    wxPanel* panel = new wxPanel(this);

    m_sizer = new wxBoxSizer(wxVERTICAL);

    m_events = new wxPanel(panel);
    wxBoxSizer *esizer = new wxBoxSizer(wxVERTICAL);
#if defined (__WXGTK__) || defined (__WXMAC__)
    wxPanel *panel1= new wxPanel(m_events, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxBORDER_SUNKEN);
#else
    wxPanel *panel1= new wxPanel(m_events);
#endif
    wxBoxSizer *sizer1 = new wxBoxSizer(wxVERTICAL);
    m_eventsList = new EventListCtrl(panel1, ID_EVENTLIST);
    m_eventsList->InsertColumn(0, _("Event"));
    m_eventsList->SetColumnWidth(0, dxsettings.colEventWidth());
    if(dxsettings.colDateShow())
    {
        long col = m_eventsList->AppendColumn(_("Date"));
        m_eventsList->SetColumnWidth(col, dxsettings.colDateWidth());
    }
    if(dxsettings.colReminderShow())
    {
        long col = m_eventsList->AppendColumn(_("Reminder"));
        m_eventsList->SetColumnWidth(col, dxsettings.colReminderWidth());
    }
    if(dxsettings.colRecurrenceShow())
    {
        long col = m_eventsList->AppendColumn(_("Recurrence"));
        m_eventsList->SetColumnWidth(col, dxsettings.colRecurrenceWidth());
    }
    if(m_filter)
    {
        eventslistMarkFiltered();
    }
    sizer1->Add(m_eventsList, 1, wxALL|wxEXPAND, 0);
    panel1->SetSizer(sizer1);
    sizer1->Fit(panel1);
    sizer1->SetSizeHints(panel1);
    esizer->Add(panel1, 1, wxALL|wxEXPAND, 3);
    wxBoxSizer *panelsizer = new wxBoxSizer(wxHORIZONTAL);
    m_bar = new dxToolBar(m_events);
    m_bar->AddTool(ID_BTN_NEWEVENT, wxEmptyString, ICO_EADD, _("New event"));
    m_bar->AddTool(ID_BTN_EDITEVENT, wxEmptyString, ICO_EEDIT, ICO_GEEDIT, wxITEM_NORMAL, _("Edit event"));
    m_bar->AddTool(ID_BTN_REMOVEEVENT, wxEmptyString,ICO_EDELETE, ICO_GEDELETE, wxITEM_NORMAL, _("Remove event"));
    m_bar->AddStretchableSpace();
    m_bar->AddTool(ID_BTN_FILTER, wxEmptyString, ICO_FILTER, ICO_GFILTER, wxITEM_CHECK, _("Filter"));
    m_bar->AddTool(ID_BTN_FILTERCLEAR, wxEmptyString, ICO_FILTERCLEAR, ICO_GFILTERCLEAR, wxITEM_NORMAL, m_filter?_("Clear filter"):_("Revert filter"));
    m_bar->AddSeparator();
    m_bar->AddTool(ID_BTN_HIDEEVENT, wxEmptyString, ICO_HIDE, ICO_GHIDE, wxITEM_NORMAL, _("Hide event"));
    m_bar->AddTool(ID_BTN_SHOWEVENTS, wxEmptyString, ICO_HIDEALL, ICO_GHIDEALL, wxITEM_CHECK, _("Hide events"));
    m_bar->ToggleTool(ID_BTN_SHOWEVENTS, false);
    m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Hide events"));
    m_bar->AddTool(ID_BTN_CLEARHIDDEN, wxEmptyString, ICO_CLEAR, ICO_GCLEAR, wxITEM_NORMAL, _("Clear hidden events"));
    m_bar->AddSeparator();
    m_bar->AddTool(ID_MENU_SHOWNOTDELETE, wxEmptyString, ICO_SHOWDONOTDELETE, ICO_GSHOWDONOTDELETE, wxITEM_CHECK, m_hideDonotdelete?_("Show all do not delete events"):_("Hide all do not delete events"));
    m_bar->AddSeparator();
    m_bar->AddTool(ID_MENU_SHOWHIDENOTE, wxEmptyString, ICO_NOTE, _("Show/hide note"));
    m_bar->Realize();
    panelsizer->Add(m_bar, 1);
    esizer->Add(panelsizer, 0, wxEXPAND, 3);
    m_events->SetSizer(esizer);
    esizer->Fit(m_events);
    esizer->SetSizeHints(m_events);

    m_bar->ToggleTool(ID_BTN_FILTER, m_filter);
    m_filterMenuItem->Check(m_filter);
    m_bar->ToggleTool(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete);
    m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);

    m_calendar = new dxCalendar(panel, ID_CALENDAR);

    if(dxsettings.look() == 0)
    {
        m_sizer->Add(m_calendar, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
    }
    else if(dxsettings.look() == 1)
    {
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
        m_sizer->Add(m_calendar, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
    }
    else if(dxsettings.look() == 2)
    {
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
        m_calendar->Hide();
    }
    else
    {
        m_sizer->AddStretchSpacer();
        m_sizer->Add(m_calendar, 0, wxCENTER, 3);
        m_sizer->AddStretchSpacer();
        m_events->Hide();
    }

    m_note = new wxTextCtrl(panel, ID_NOTE, wxEmptyString, wxDefaultPosition, wxSize(50, 100), wxTE_MULTILINE);
    if(dxsettings.showNote())
    {
        m_sizer->Add(m_note, 0, wxALL|wxEXPAND, 3);
    }
    else
    {
        m_note->Hide();
    }

    loadDatabase();

    m_removeEvents->Enable(m_data->someEvents()?true:false);

    panel->SetSizer(m_sizer);
    m_sizer->Fit(this);
    m_sizer->SetSizeHints(this);

    if(dxsettings.x() == -1 || dxsettings.y() == -1)
    {
        if(wxSystemSettings::GetMetric(wxSYS_SCREEN_X)<dxsettings.w()) dxsettings.setW(750);
        if(wxSystemSettings::GetMetric(wxSYS_SCREEN_Y)<dxsettings.h()) dxsettings.setH(450);
        SetSize(dxsettings.w(), dxsettings.h());
    }
    else
    {
        if(wxSystemSettings::GetMetric(wxSYS_SCREEN_X)<dxsettings.x()+dxsettings.w())
        {
            dxsettings.setX(10);
            dxsettings.setW(750);
        }
        if(wxSystemSettings::GetMetric(wxSYS_SCREEN_Y)<dxsettings.y()+dxsettings.h())
        {
            dxsettings.setY(10);
            dxsettings.setH(450);
        }
        SetSize(dxsettings.x(), dxsettings.y(), dxsettings.w(), dxsettings.h());
    }

    if(dxsettings.iconize()) Iconize();

    m_evenMonth = new wxListItemAttr(wxNullColour, dxsettings.evenMonth(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_oddMonth = new wxListItemAttr(wxNullColour, dxsettings.oddMonth(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_eventFont = new wxListItemAttr(wxNullColour, wxNullColour, dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight1 = new wxListItemAttr(dxsettings.highlightFg(), dxsettings.highlightBg1(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight2 = new wxListItemAttr(dxsettings.highlightFg(), dxsettings.highlightBg2(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight3 = new wxListItemAttr(dxsettings.highlightFg(), dxsettings.highlightBg3(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight4 = new wxListItemAttr(dxsettings.highlightFg(), dxsettings.highlightBg4(), dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight5 = new wxListItemAttr(dxsettings.highlightFg(), dxsettings.highlightBg5(), dxutils::fontFromString(dxsettings.eventsFont()));

    enableTools();

    Bind(wxEVT_CHAR_HOOK, &dxremindersFrame::OnKeyUp, this);

    if(m_ipc)
    {
#ifdef wxHAS_UNIX_DOMAIN_SOCKETS
        wxUNIXaddress addr;
        wxString filename = wxGetHomeDir();
        if(filename.Last() != wxT('/'))
        {
            filename += wxT('/');
        }
        filename << ".dxreminders.socket";
        remove(filename.fn_str());
        addr.Filename(filename);
#else
        wxIPV4address addr;
        addr.Service(DEFAULT_PORT);
#endif
        m_server = new wxSocketServer(addr);
        if(!m_server->IsOk())
        {
#ifdef wxHAS_UNIX_DOMAIN_SOCKETS
            wxLogMessage(wxString::Format(_("Could not listen at the specified socket %s!"), addr.Filename()));
#else
            wxLogMessage(wxString::Format(_("Could not listen at the specified port %d!"), DEFAULT_PORT));
#endif
            return;
        }
#ifndef wxHAS_UNIX_DOMAIN_SOCKETS
        wxIPV4address addrReal;
        if(!m_server->GetLocal(addrReal))
        {
            wxLogDebug("ERROR: couldn't get the address we bound to");
        }
        else
        {
            wxLogDebug("Server listening at %s:%u", addrReal.IPAddress(), addrReal.Service());
        }
#endif
        // Setup the event handler and subscribe to connection events
        m_server->SetEventHandler(*this, ID_SERVER);
        m_server->SetNotify(wxSOCKET_CONNECTION_FLAG);
        m_server->Notify(true);
    }

    m_timer.Start(1000);
}

dxremindersFrame::~dxremindersFrame()
{
    if(m_ipc)
    {
        m_server->Close();
#ifdef wxHAS_UNIX_DOMAIN_SOCKETS
        wxString filename = wxGetHomeDir();
        if(filename.Last() != wxT('/'))
        {
            filename += wxT('/');
        }
        filename << ".dxreminders.socket";
        remove(filename.fn_str());
#endif
    }
#if !(defined (__WXMAC__) || defined (__WXMSW__))
    if(dxsettings.startOnLogin())
    {
        if(!wxFileExists(wxStandardPaths::Get().GetUserConfigDir()+wxFILE_SEP_PATH+".config"+wxFILE_SEP_PATH+"autostart"+wxFILE_SEP_PATH+"dxreminders.desktop"))
            wxCopyFile(wxString(DESKTOPDIR)+wxFILE_SEP_PATH+"dxreminders.desktop",
                                             wxStandardPaths::Get().GetUserConfigDir()+wxFILE_SEP_PATH+".config"+wxFILE_SEP_PATH+"autostart"+wxFILE_SEP_PATH+"dxreminders.desktop");
    }
    else
    {
        if(wxFileExists(wxStandardPaths::Get().GetUserConfigDir()+wxFILE_SEP_PATH+".config"+wxFILE_SEP_PATH+"autostart"+wxFILE_SEP_PATH+"dxreminders.desktop"))
            wxRemoveFile(wxStandardPaths::Get().GetUserConfigDir()+wxFILE_SEP_PATH+".config"+wxFILE_SEP_PATH+"autostart"+wxFILE_SEP_PATH+"dxreminders.desktop");
    }
#endif
#if defined (__WXMSW__)
    wxRegKey key(wxRegKey::HKCU, "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Run");
    if(dxsettings.startOnLogin())
    {
        key.SetValue("dxreminders", wxStandardPaths::Get().GetExecutablePath());
    }
    else
    {
        key.SetValue("dxreminders", "");
    }
#endif
    if(!IsFullScreen())
    {
        dxsettings.setX(GetPosition().x);
        dxsettings.setY(GetPosition().y);
        dxsettings.setW(GetSize().x);
        dxsettings.setH(GetSize().y);
    }
    dxsettings.setIconize(IsIconized());
    saveColumnSize();
    dxsettings.setHideDonotdelete(m_hideDonotdelete);
    dxsettings.setFilter(m_filter);
    dxsettings.setHidden(m_hidden);
    dxsettings.setEventfilterAction(m_filters[0].action());
    dxsettings.setEventfilterValue(m_filters[0].text());
    dxsettings.setEventfilterUsage(m_filters[0].active());
    dxsettings.setDatefilterAction(m_filters[1].action());
    dxsettings.setDatefilterValue(m_filters[1].datum());
    dxsettings.setDatefilterUsage(m_filters[1].active());
    dxsettings.setReminderfilterAction(m_filters[2].action());
    dxsettings.setReminderfilterValue(m_filters[2].reminder());
    dxsettings.setReminderfilterUsage(m_filters[2].active());
    dxsettings.setRecurrencefilterValue(m_filters[3].intvalue());
    dxsettings.setRecurrencefilterUsage(m_filters[3].active());
    dxsettings.setTimefilterYears(m_filters[4].datespan().GetYears());
    dxsettings.setTimefilterMonths(m_filters[4].datespan().GetMonths());
    dxsettings.setTimefilterWeeks(m_filters[4].datespan().GetWeeks());
    dxsettings.setTimefilterDays(m_filters[4].datespan().GetDays());
    dxsettings.setTimefilterHours(m_filters[4].timespan().GetHours());
    dxsettings.setTimefilterMinutes(m_filters[4].timespan().GetMinutes()-m_filters[4].timespan().GetHours()*60);
    dxsettings.setTimefilterUsage(m_filters[4].active());
    dxsettings.setHiddenfilterUsage(m_filters[5].active());
    dxsettings.save();
    if(m_taskBarIcon) delete m_taskBarIcon;
    if(m_taskBarEvents) delete  m_taskBarEvents;
    if(m_timer.IsRunning())
        m_timer.Stop();
}

void dxremindersFrame::hideShowTrayIcon()
{
    if(m_taskBarIcon == nullptr && !dxsettings.hideTrayIcon())
    {
#if defined(__WXCOCOA__)
        m_taskBarIcon = new dxTaskBarIcon(wxTaskBarIcon::DOCK);
#else
        m_taskBarIcon = new dxTaskBarIcon();
        m_taskBarIcon->SetIcon(ICO_TRAY);
#endif
    }
    if(m_taskBarIcon != nullptr && dxsettings.hideTrayIcon())
    {
        m_taskBarIcon->Destroy();
        m_taskBarIcon = nullptr;
    }
}

void dxremindersFrame::applyPreference(bool resize)
{
    recreateColumns();
    m_evenMonth->SetBackgroundColour(dxsettings.evenMonth());
    m_evenMonth->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_oddMonth->SetBackgroundColour(dxsettings.oddMonth());
    m_oddMonth->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight1->SetTextColour(dxsettings.highlightFg());
    m_highlight1->SetBackgroundColour(dxsettings.highlightBg1());
    m_highlight1->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight2->SetTextColour(dxsettings.highlightFg());
    m_highlight2->SetBackgroundColour(dxsettings.highlightBg2());
    m_highlight2->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight3->SetTextColour(dxsettings.highlightFg());
    m_highlight3->SetBackgroundColour(dxsettings.highlightBg3());
    m_highlight3->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight4->SetTextColour(dxsettings.highlightFg());
    m_highlight4->SetBackgroundColour(dxsettings.highlightBg4());
    m_highlight4->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight5->SetTextColour(dxsettings.highlightFg());
    m_highlight5->SetBackgroundColour(dxsettings.highlightBg5());
    m_highlight5->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_eventFont->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_calendar->updateCalendar();
    if(calendar) calendar->updateCalendar();
    reLayout(resize);
    if(m_filter)
    {
        updateEventsList();
    }
}

void dxremindersFrame::applyFilter(const dxFilterArray &filters)
{
    m_filters = filters;
    m_filter = false;
    for(size_t i=0; i<m_filters.GetCount(); i++)
    {
        if(m_filters[i].active())
        {
            m_filter = true;
            break;
        }
    }
    eventslistMarkFiltered();
    m_eventsList->Update();
    updateEventsList();
}

void dxremindersFrame::refreshFrame()
{
#if defined (__WXMAC__)
    m_calendar->refreshFrame();
#endif
}

bool dxremindersFrame::noteShown()
{
    return m_note->IsShown();
}

wxString dxremindersFrame::getItemText(long item, long column) const
{
    long index = 0;
    std::vector<EventView>::const_iterator i;
    for(i = m_eventViews.begin(); i != m_eventViews.end(); ++i)
    {
        if(item == index)
        {
            if(column == 0)
                return (*i).name();
            if(column == 1)
            {
                if(dxsettings.colDateShow())
                {
                    if((*i).recurrencetype()==R_NONE) return wxEmptyString;
                    else return dxutils::formatDateTime((*i).date(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout());
                }
                else
                {
                    if(dxsettings.colReminderShow())
                    {
                        if((*i).recurrencetype()==R_NONE) return wxEmptyString;
                        else return (*i).reminder();
                    }
                    else return (*i).recurrence();
                }
            }
            if(column == 2)
            {
                if(dxsettings.colReminderShow() && dxsettings.colDateShow())
                {
                    if((*i).recurrencetype()==R_NONE) return wxEmptyString;
                    else return (*i).reminder();
                }
                else return (*i).recurrence();
            }
            if(column == 3)
            {
                return (*i).recurrence();
            }
        }
        index++;
    }
    return wxEmptyString;
}

wxListItemAttr *dxremindersFrame::getItemAttr(long item) const
{
    if(item < 0 || item >= static_cast<long>(m_eventViews.size()))
        return nullptr;
    if(m_eventViews[static_cast<size_t>(item)].highlight())
    {
        switch(m_eventViews[static_cast<size_t>(item)].highlight()) {
        case 1: return m_highlight1;
        case 2: return m_highlight2;
        case 3: return m_highlight3;
        case 4: return m_highlight4;
        default: return m_highlight5;
        }
    }
    if(m_eventViews[static_cast<size_t>(item)].recurrencetype()==R_NONE || m_eventViews[static_cast<size_t>(item)].recurrencetype()==R_ONCENOTDELETE)
    {
        return m_eventFont;
    }
    if(dxsettings.disableColors())
        return m_eventFont;
    if((m_eventViews[static_cast<size_t>(item)].date().GetMonth()+1)%2==0) //even month
    {
        return m_evenMonth;
    }
    else
    {
        return m_oddMonth;
    }
}

/**
 * @brief dxremindersFrame::showEvents - show events to remind on silent mode
 */
void dxremindersFrame::showEvents()
{
    if(m_msgDialog == nullptr)
    {
        if(m_data->someEventsByReminder(wxDateTime::Now()))
        {
#if defined (__WXMAC__)
            m_msgDialog = new EventMsgDialog(nullptr);
#else
            m_msgDialog = new EventMsgDialog(this);
#endif
            m_msgDialog->ShowModal();
            markDays();
            updateEventsList();
            sendTooltip();
        }
        if(m_taskBarEvents != nullptr)
        {
            m_taskBarEvents->Destroy();
            m_taskBarEvents = nullptr;
        }
        if(m_data->someEventsByReminder(wxDateTime::Now()))
        {
#if defined (__WXMAC__)
            m_msgDialog = new EventMsgDialog(nullptr);
#else
            m_msgDialog = new EventMsgDialog(this);
#endif
            m_msgDialog->ShowModal();
            markDays();
            updateEventsList();
            sendTooltip();
        }
        m_msgDialog = nullptr;
    }
}

void dxremindersFrame::loadDatabase()
{
    m_data = new Engine(this);
    if(!wxDirExists(wxStandardPaths::Get().GetUserDataDir()))
    {
        wxMkdir(wxStandardPaths::Get().GetUserDataDir());
    }
    if(!m_data->loadData())
    {
        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db"), PACKAGE);
        Destroy();
        return;
    }
    m_data->loadCalendarData(m_hidden);
    if(!m_hidden && m_data->someEventOnHiddenList())
    {
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, true);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
        m_showHiddenEvents->Check();
        m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    if(!m_data->someDonotdeleteEvent(m_hidden))
    {
        m_hideDonotdelete = false;
    }
    m_showDonotDelete->Check(m_hideDonotdelete);
    m_showDonotDelete->SetItemLabel(m_hideDonotdelete?_("Show all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE):
                                                      _("Hide all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE));
    m_showDonotDelete->Enable(!m_hideDonotdelete);
    m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, !m_hideDonotdelete);
    m_bar->ToggleTool(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete);
    m_bar->SetToolShortHelp(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete?_("Show all do not delete events"):_("Hide all do not delete events"));
    m_note->ChangeValue(m_data->noteText());
}

void dxremindersFrame::reLayout(bool resize)
{
    m_sizer->Clear(false);
    if(dxsettings.look() == 0)
    {
        m_calendar->Hide();
        m_events->Hide();
        m_sizer->Add(m_calendar, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
        m_sizer->Show(m_calendar);
        m_sizer->Show(m_events);
    }
    else if(dxsettings.look() == 1)
    {
        m_calendar->Hide();
        m_events->Hide();
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
        m_sizer->Add(m_calendar, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 3);
        m_sizer->Show(m_events);
        m_sizer->Show(m_calendar);
    }
    else if(dxsettings.look() == 2)
    {
        m_calendar->Hide();
        m_sizer->Add(m_events, 1, wxALL|wxEXPAND, 3);
        m_sizer->Show(m_events);
    }
    else
    {
        m_events->Hide();
        m_sizer->AddStretchSpacer();
        m_sizer->Add(m_calendar, 0, wxCENTER, 3);
        m_sizer->AddStretchSpacer();
        m_sizer->Show(m_calendar);
    }
    if(dxsettings.showNote())
    {
        m_note->Hide();
        m_sizer->Add(m_note, 1, wxALL|wxEXPAND, 3);
        m_sizer->Show(m_note);
    }
    else
    {
        m_note->Hide();
    }
    m_sizer->Layout();
    if(resize)
    {
        m_sizer->Fit(this);
        m_sizer->SetSizeHints(this);
    }
}

void dxremindersFrame::recreateColumns()
{
    int listWidth = dxsettings.colEventWidth();
    m_eventsList->SetColumnWidth(0, dxsettings.colEventWidth());
    for(int i=m_eventsList->GetColumnCount()-1; i>0; i--)
    {
        m_eventsList->DeleteColumn(i);
    }
    if(dxsettings.colDateShow())
    {
        long col = m_eventsList->AppendColumn(_("Date"));
        m_eventsList->SetColumnWidth(col, dxsettings.colDateWidth());
        listWidth += dxsettings.colDateWidth();
    }
    if(dxsettings.colReminderShow())
    {
        long col = m_eventsList->AppendColumn(_("Reminder"));
        m_eventsList->SetColumnWidth(col, dxsettings.colReminderWidth());
        listWidth += dxsettings.colReminderWidth();
    }
    if(dxsettings.colRecurrenceShow())
    {
        long col = m_eventsList->AppendColumn(_("Recurrence"));
        m_eventsList->SetColumnWidth(col, dxsettings.colRecurrenceWidth());
        listWidth += dxsettings.colRecurrenceWidth();
    }
    m_eventsList->SetMinClientSize(wxSize(listWidth,-1));
    eventslistMarkFiltered();
}

void dxremindersFrame::enableTools()
{
    if(m_selectedIndex == -1)
    {
        m_bar->EnableTool(ID_BTN_EDITEVENT, false);
        m_bar->EnableTool(ID_BTN_REMOVEEVENT, false);
        m_bar->EnableTool(ID_BTN_HIDEEVENT, false);
        m_calendar->setEventID(-1);
    }
    else
    {
        m_bar->EnableTool(ID_BTN_EDITEVENT, true);
        m_bar->EnableTool(ID_BTN_REMOVEEVENT, true);
        m_bar->EnableTool(ID_BTN_HIDEEVENT, true);
        m_calendar->setEventID(m_eventViews[static_cast<size_t>(m_selectedIndex)].id());
    }
}

void dxremindersFrame::selectDate()
{
    if(m_selectedIndex != -1)
    {
        if(m_eventViews[static_cast<size_t>(m_selectedIndex)].recurrencetype()==R_NONE)
        {
            return;
        }
        if(m_calendar->daytypeForDay() == 1 && m_calendar->getValue().IsSameDate(m_eventViews[static_cast<size_t>(m_selectedIndex)].date()))
        {
            return;
        }
        if(m_calendar->daytypeForDay() == 2)
        {
            if(m_calendar->hasEventForDay(m_eventViews[static_cast<size_t>(m_selectedIndex)].id()))
            {
                return;
            }
        }
        m_calendar->setValue(m_eventViews[static_cast<size_t>(m_selectedIndex)].date());
    }
}

void dxremindersFrame::saveColumnSize()
{
    dxsettings.setColEventWidth(m_eventsList->GetColumnWidth(0));
    int i = 1;
    if(dxsettings.colDateShow())
    {
        dxsettings.setColDateWidth(m_eventsList->GetColumnWidth(i));
        i++;
    }
    else
    {
        dxsettings.setColDateWidth(-1);
    }
    if(dxsettings.colReminderShow())
    {
        dxsettings.setColReminderWidth(m_eventsList->GetColumnWidth(i));
        i++;
    }
    else
    {
        dxsettings.setColReminderWidth(-1);
    }
    if(dxsettings.colRecurrenceShow())
    {
        dxsettings.setColRecurrenceWidth(m_eventsList->GetColumnWidth(i));
    }
    else
    {
        dxsettings.setColRecurrenceWidth(-1);
    }
}

void dxremindersFrame::fireHoliday()
{
    if(m_data->someHolidayToFire())
    {
        wxMessageBox(m_data->holidayTextsForDate(), _("Public holiday"));
    }
}

void dxremindersFrame::updateEventsList()
{
    wxInt64 lastid = m_data->getLastid();
    size_t oldcnt = m_eventViews.size();
    if(m_filter)
    {
        m_eventViews.clear();
        std::vector<EventView> events = m_data->eventViews(m_filters[5].active(), m_hidden);
        for(const EventView &event : events)
        {
            if(lastid == event.id() && m_data->isEventOnHiddenList(lastid))
            {
                continue;
            }
            if(event.recurrencetype()==R_NONE)
            {
                if(event.always())
                {
                    m_eventViews.push_back(event);
                    continue;
                }
                if(m_hideDonotdelete)
                {
                    continue;
                }
                if(dxsettings.alwaysNone())
                {
                    m_eventViews.push_back(event);
                }
                else
                {
                    if(m_filters[0].matchFilter(event.name()) && !m_filters[1].active() && !m_filters[2].active()
                            && m_filters[3].matchFilter(m_data->getEventRecurrence(event.id())) && !m_filters[4].active())
                    {
                        m_eventViews.push_back(event);
                    }
                }
            }
            else if(event.recurrencetype()==R_ONCENOTDELETE)
            {
                if(event.always())
                {
                    m_eventViews.push_back(event);
                    continue;
                }
                if(m_hideDonotdelete)
                {
                    continue;
                }
                if(dxsettings.alwaysOncenotdelete())
                {
                    m_eventViews.push_back(event);
                }
                else
                {
                    if(m_filters[0].matchFilter(event.name()) && !m_filters[1].active() && !m_filters[2].active()
                            && m_filters[3].matchFilter(m_data->getEventRecurrence(event.id())) && !m_filters[4].active())
                    {
                        m_eventViews.push_back(event);
                    }
                }
            }
            else
            {
                if(m_filters[0].matchFilter(event.name()) && m_filters[1].matchFilter(event.date()) && m_filters[2].matchFilter(Reminder(m_data->getEventReminder(event.id())))
                        && m_filters[3].matchFilter(m_data->getEventRecurrence(event.id())) && m_filters[4].matchFilter(event.date()))
                {
                    m_eventViews.push_back(event);
                }
            }
        }
    }
    else
    {
        m_eventViews.clear();
        std::vector<EventView> events = m_data->eventViews(false, m_hidden);
        for(const EventView &event : events)
        {
            if(lastid == event.id() && m_data->isEventOnHiddenList(lastid))
            {
                continue;
            }
            if(event.recurrencetype()==R_NONE || event.recurrencetype()==R_ONCENOTDELETE)
            {
                if(event.always())
                {
                    m_eventViews.push_back(event);
                    continue;
                }
                if(m_hideDonotdelete)
                {
                    continue;
                }
                m_eventViews.push_back(event);
            }
            else
                m_eventViews.push_back(event);
        }
    }
    size_t cnt = m_eventViews.size();
    m_eventsList->SetItemCount(cnt);
    if(cnt)
    {
        if(m_selectedIndex >= static_cast<long>(cnt))
        {
            m_eventsList->SetItemState(0, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
            m_selectedIndex = 0;
        }
        if(oldcnt <= cnt) m_eventsList->RefreshItems(0, --cnt);
        else m_eventsList->Refresh();
        m_bar->EnableTool(ID_BTN_FILTER, true);
        m_bar->EnableTool(ID_BTN_FILTERCLEAR, true);
        m_filterMenuItem->Enable();
        m_clearfilterMenuItem->Enable();
    }
    else
    {
        m_eventsList->Refresh();
        m_selectedIndex = -1;
        m_bar->EnableTool(ID_BTN_FILTER, m_filter?true:false);
        m_bar->EnableTool(ID_BTN_FILTERCLEAR, m_filter?true:false);
        m_filterMenuItem->Enable(m_filter?true:false);
        m_clearfilterMenuItem->Enable(m_filter?true:false);
        if(lastid != -1)
        {
            m_hidden = false;
            m_bar->ToggleTool(ID_BTN_SHOWEVENTS, true);
            m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
            m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
            m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
            m_showHiddenEvents->Check();
            m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
        }
    }
    enableTools();
}

void dxremindersFrame::OnCheckVersion(wxCommandEvent &/*event*/)
{
    wxString availableVersion = dxutils::availableVersion();
    int availableVersionInt = dxutils::versionToInt(availableVersion);
    if(availableVersionInt == 0)
    {
        wxMessageDialog dialog(this, _("The update could not be verified.\nPlease try again later."), _("Information"), wxCENTER|wxOK_DEFAULT|wxOK|wxICON_INFORMATION);
        dialog.ShowModal();
        return;
    }
    if(dxutils::versionToInt(VERSION) < availableVersionInt)
    {
        wxMessageDialog dialog(this, wxString::Format(_("Newer version %s is available."), availableVersion), _("Information"), wxCENTER|wxOK_DEFAULT|wxOK|wxICON_INFORMATION);
        dialog.ShowModal();
    }
    else if(dxutils::versionToInt(VERSION) > availableVersionInt)
    {
        wxMessageDialog dialog(this, wxString::Format(_("You have development version %s."), VERSION), _("Information"), wxCENTER|wxOK_DEFAULT|wxOK|wxICON_INFORMATION);
        dialog.ShowModal();
    }
    else
    {
        wxMessageDialog dialog(this, wxString::Format(_("You have current version %s."), availableVersion), _("Information"), wxCENTER|wxOK_DEFAULT|wxOK|wxICON_INFORMATION);
        dialog.ShowModal();
    }
}

void dxremindersFrame::sendTooltip()
{
    if(m_ipc && m_sock && m_sock->IsOk())
    {
        const wxWX2MBbuf buf = m_data->getTooltipPlugin().mb_str();
        unsigned int len = strlen(buf);
        m_sock->Write(buf,len);
    }
    if(m_taskBarIcon) m_taskBarIcon->SetIcon(ICO_TRAY, m_data->getTooltipPlugin().AfterFirst('>').BeforeFirst('\r'));
}

void dxremindersFrame::sendSilentmode()
{
    if(m_ipc && m_sock && m_sock->IsOk())
    {
        if(dxsettings.silentMode()) m_sock->Write("<silenton>\r\n",13);
        else m_sock->Write("<silentoff>\r\n",14);
    }
}

void dxremindersFrame::parseReceived(const wxString &text)
{
    if(text.CmpNoCase("showhide")==0)
    {
        if(frameShown)
        {
            Hide();
            frameShown = false;
        }
        else
        {
            Show(true);
            if(dxsettings.calendarToday()) m_calendar->setValue(wxDateTime::Now());
            frameShown = true;
        }
    }
    if(text.CmpNoCase("calendar")==0)
    {
        if(calendar == nullptr)
        {
            calendar = new CalendarPopup(this);
        }
        calendar->setPosition(wxGetMousePosition());
        if(calendar->shown())
        {
            calendar->Hide();
            calendar->setShown(false);
        }
        else
        {
            calendar->Show(true);
            if(dxsettings.calendarToday()) calendar->setValue(wxDateTime::Now());
            calendar->setShown(true);
        }
    }
    if(text.CmpNoCase("newevent")==0)
    {
        EventDialog dialog(this);
        dialog.ShowModal();
        if(m_data->someDonotdeleteEvent(m_hidden))
        {
            m_showDonotDelete->Enable();
            m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
        }
        else
        {
            m_showDonotDelete->Enable(false);
            m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
        }
    }
    if(text.CmpNoCase("saveini")==0)
    {
        if(!IsFullScreen())
        {
            dxsettings.setX(GetPosition().x);
            dxsettings.setY(GetPosition().y);
            dxsettings.setW(GetSize().x);
            dxsettings.setH(GetSize().y);
        }
        dxsettings.setIconize(IsIconized());
        saveColumnSize();
        dxsettings.setFilter(m_filter);
        dxsettings.setEventfilterAction(m_filters[0].action());
        dxsettings.setEventfilterValue(m_filters[0].text());
        dxsettings.setEventfilterUsage(m_filters[0].active());
        dxsettings.setDatefilterAction(m_filters[1].action());
        dxsettings.setDatefilterValue(m_filters[1].datum());
        dxsettings.setDatefilterUsage(m_filters[1].active());
        dxsettings.setReminderfilterAction(m_filters[2].action());
        dxsettings.setReminderfilterValue(m_filters[2].reminder());
        dxsettings.setReminderfilterUsage(m_filters[2].active());
        dxsettings.setRecurrencefilterValue(m_filters[3].intvalue());
        dxsettings.setRecurrencefilterUsage(m_filters[3].active());
        dxsettings.setTimefilterYears(m_filters[4].datespan().GetYears());
        dxsettings.setTimefilterMonths(m_filters[4].datespan().GetMonths());
        dxsettings.setTimefilterWeeks(m_filters[4].datespan().GetWeeks());
        dxsettings.setTimefilterDays(m_filters[4].datespan().GetDays());
        dxsettings.setTimefilterHours(m_filters[4].timespan().GetHours());
        dxsettings.setTimefilterMinutes(m_filters[4].timespan().GetMinutes()-m_filters[4].timespan().GetHours()*60);
        dxsettings.setTimefilterUsage(m_filters[4].active());
        dxsettings.setHiddenfilterUsage(m_filters[5].active());
        dxsettings.save();
        if(m_ipc && m_sock && m_sock->IsOk())
        {
            m_sock->Write("<gorestart>\r\n",14);
        }
    }
    if(text.CmpNoCase("tooltip")==0)
    {
        sendTooltip();
    }
    if(text.CmpNoCase("silentmode")==0)
    {
        sendSilentmode();
    }
    if(text.CmpNoCase("silenton")==0)
    {
        dxsettings.setSilentMode(true);
    }
    if(text.CmpNoCase("silentoff")==0)
    {
        dxsettings.setSilentMode(false);
    }
    if(text.CmpNoCase("quit")==0)
    {
        m_data->updateNote(m_note->GetValue());
        Destroy();
    }
}

void dxremindersFrame::OnClose(wxCloseEvent& /*event*/)
{
    if(m_taskBarIcon || m_ipc)
    {
        Show(false);
        frameShown = !frameShown;
    }
    else
    {
        m_data->updateNote(m_note->GetValue());
        Destroy();
    }
}

void dxremindersFrame::OnQuit(wxCommandEvent& /*event*/)
{
    m_data->updateNote(m_note->GetValue());
    Destroy();
}

void dxremindersFrame::OnAbout(wxCommandEvent& /*event*/)
{
    wxString description = wxString() << wxVERSION_STRING
#ifdef __WXGTK__
                            << " (gtk " << GTKV << ")\n"
#else
                            << "\n"
#endif
                            << wxSQLite3Database::GetWrapperVersion() << "\n"
                            << "SQLite3 " << wxSQLite3Database::GetVersion() << "\n"
#ifdef HAVE_CURL
                            << "LibCurl " << LIBCURL_VERSION << "\n"
#endif
                            << _("Some icons by Yusuke Kamiyamane,\nlicensed under a Creative Commons Attribution 3.0 License.");
    wxAboutDialogInfo info;
    info.SetIcon(ICO_ABOUT);
    info.SetName(PACKAGE);
    info.SetVersion(VERSION);
    info.SetDescription(description);
    info.SetCopyright("(c) 2013 ~");
    info.SetWebSite("https://dxreminders.dxsolutions.org");
    info.AddDeveloper("David Vachulka <arch_dvx@users.sourceforge.net>");
    info.SetLicence(_(
                "This program is free software; you can redistribute it and/or modify it under the terms of\n"
                "the GNU General Public License as published by the Free Software Foundation;\n"
                "either version 3 of the License, or (at your option) any later version.\n\n"

                "This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;\n"
                "without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
                "See the GNU General Public License for more details.\n\n"

                "You should have received a copy of the GNU General Public License along with this program;\n"
                "if not, write to the Free Software Foundation, Inc., 51 Franklin Street,\n"
                "Fifth Floor, Boston, MA 02110-1301, USA"));
    wxAboutBox(info);
}

void dxremindersFrame::OnHelp(wxCommandEvent &/*event*/)
{
#if defined (__WXMAC__)
    HelpDialog dialog(nullptr);
#else
    HelpDialog dialog(this);
#endif
    dialog.ShowModal();
}

void dxremindersFrame::OnNewEvent(wxCommandEvent& /*event*/)
{
#if defined (__WXMAC__)
    EventDialog dialog(nullptr);
#else
    EventDialog dialog(this);
#endif
    dialog.ShowModal();
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnEditEvent(wxCommandEvent& /*event*/)
{
    if(m_selectedIndex == -1) return;
    EventDialog dialog(this, m_data->getEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id()));
    dialog.ShowModal();
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnEditDate(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex == -1) return;
    DateDialog dialog(this, m_eventViews[static_cast<size_t>(m_selectedIndex)].date());
    if(dialog.ShowModal() == wxID_OK)
    {
        wxDateTime date = dialog.date();
        date.SetHour(m_eventViews[static_cast<size_t>(m_selectedIndex)].date().GetHour());
        date.SetMinute(m_eventViews[static_cast<size_t>(m_selectedIndex)].date().GetMinute());
        date.SetSecond(0);
        m_data->editEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id(), date);
        m_calendar->setValue(date);
    }
}

void dxremindersFrame::OnAdvance(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex == -1) return;
    Event event = m_data->getEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id());
    wxDateTime date = m_data->recurrentEventDate(event,m_eventViews[static_cast<size_t>(m_selectedIndex)].date()+wxTimeSpan::Seconds(3));
    event.setDate(date);
    m_eventViews[static_cast<size_t>(m_selectedIndex)].setDate(date);
    m_data->editEvent(event);
    m_calendar->setValue(date);
}

void dxremindersFrame::OnRemoveEvent(wxCommandEvent& /*event*/)
{
    if(m_selectedIndex == -1) return;
    wxMessageDialog dialog(this, _("Do you really want to delete the Event?")+"\n"+m_data->getEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id()).tipText(true),
            _("Confirm Event Deletion"), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
    if(dialog.ShowModal() == wxID_YES)
    {
        m_data->removeEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id());
    }
}

void dxremindersFrame::OnItemRightClick(wxListEvent& event)
{
    m_selectedIndex = event.GetIndex();
    wxMenu popup;
    popup.Append(ID_MENU_NEWEVENT, _("New event"));
    if(m_selectedIndex != -1)
    {
        popup.AppendSeparator();
        popup.Append(ID_MENU_EDITEVENT, _("Edit event"));
        if(m_eventViews[static_cast<size_t>(m_selectedIndex)].recurrencetype()!=R_NONE)
        {
            popup.Append(ID_MENU_EDITDATE, _("Edit date"));
        }
        if(m_eventViews[static_cast<size_t>(m_selectedIndex)].recurrencetype()!=R_NONE && m_eventViews[static_cast<size_t>(m_selectedIndex)].date().IsLaterThan(wxDateTime::Now()))
        {
            popup.Append(ID_MENU_ADVANCE, _("Advance to next reminder"));
        }
        popup.Append(ID_MENU_REMOVEEVENT, _("Remove event"));
        popup.AppendSeparator();
        long highlighted = 0;
        long selected = 0;
        long hidden = 0;
        long nodelete = 0;
        long index = m_eventsList->GetFirstSelected();
        if(index != -1)
        {
            selected++;
            if(m_data->eventHighlightColor(m_eventViews[static_cast<size_t>(index)].id()))
                highlighted++;
            if(m_data->isEventOnHiddenList(m_eventViews[static_cast<size_t>(index)].id()))
                hidden++;
            if(m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_NONE || m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_ONCENOTDELETE)
                nodelete++;
        }
        do
        {
            index = m_eventsList->GetNextSelected(index);
            if(index != -1)
            {
                selected++;
                if(m_data->eventHighlightColor(m_eventViews[static_cast<size_t>(index)].id()))
                    highlighted++;
                if(m_data->isEventOnHiddenList(m_eventViews[static_cast<size_t>(index)].id()))
                    hidden++;
                if(m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_NONE || m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_ONCENOTDELETE)
                    nodelete++;
            }
        } while(index > 0);
        popup.Append(ID_BTN_HIDEEVENT, selected>1?_("Hide events"):_("Hide event"));
        if(hidden)
        {
            popup.AppendSeparator();
            popup.Append(ID_MENU_REMOVEHIDDEN, _("Remove from hidden events list"));
        }
        popup.AppendSeparator();
        if(nodelete)
        {
            popup.Append(ID_MENU_ALWAYSSHOW, nodelete>1?_("Always show (Do not delete) events"):_("Always show (Do not delete) event"));
            popup.AppendSeparator();
        }
        wxMenu *high = new wxMenu();
        wxMenuItem *color1 = new wxMenuItem(NULL, ID_MENU_HIGHLIGHTEVENT1, _("Color #1"));
        color1->SetBitmap(drawColorBitmap(1));
        high->Append(color1);
        wxMenuItem *color2 = new wxMenuItem(NULL, ID_MENU_HIGHLIGHTEVENT2, _("Color #2"));
        color2->SetBitmap(drawColorBitmap(2));
        high->Append(color2);
        wxMenuItem *color3 = new wxMenuItem(NULL, ID_MENU_HIGHLIGHTEVENT3, _("Color #3"));
        color3->SetBitmap(drawColorBitmap(3));
        high->Append(color3);
        wxMenuItem *color4 = new wxMenuItem(NULL, ID_MENU_HIGHLIGHTEVENT4, _("Color #4"));
        color4->SetBitmap(drawColorBitmap(4));
        high->Append(color4);
        wxMenuItem *color5 = new wxMenuItem(NULL, ID_MENU_HIGHLIGHTEVENT5, _("Color #5"));
        color5->SetBitmap(drawColorBitmap(5));
        high->Append(color5);
        if(selected > 1)
        {
            if(highlighted)
            {
                popup.AppendSubMenu(high, _("Edit highlighted events"));
                popup.Append(ID_MENU_REMOVEHIGHLIGHT, _("Remove from highlighted events list"));
            }
            else
            {
                popup.AppendSubMenu(high, _("Highlight events"));
            }
        }
        else
        {
            if(highlighted)
            {
                popup.AppendSubMenu(high, _("Edit highlighted event"));
                popup.Append(ID_MENU_REMOVEHIGHLIGHT, _("Remove from highlighted events list"));
            }
            else
            {
                popup.AppendSubMenu(high, _("Highlight event"));
            }
        }
    }
    if(!m_filter && m_data->someEventOnHiddenList() && m_hidden)
    {
        popup.AppendSeparator();
        popup.Append(ID_RIGHT_HIDEEVENTS, _("Hide all events"));
    }
    if(!m_filter && m_data->someEventOnHiddenList() && !m_hidden)
    {
        popup.AppendSeparator();
        popup.Append(ID_RIGHT_SHOWEVENTS, _("Show all events"));
    }
    if(!m_filter && m_data->someEventOnHiddenList())
    {
        popup.AppendSeparator();
        popup.Append(ID_BTN_CLEARHIDDEN, _("Clear all hidden events"));
    }
    if(!m_filter && m_data->someEventOnHighlightList())
    {
        popup.AppendSeparator();
        popup.Append(ID_MENU_CLEARHIGHLIGHT, _("Clear all highlighted events"));
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        popup.AppendSeparator();
        popup.Append(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete?_("Show all do not delete events"):_("Hide all do not delete events"));
    }
    PopupMenu(&popup, event.GetPoint()+m_events->GetPosition());
    selectDate();
}

void dxremindersFrame::OnItemActivated(wxListEvent& event)
{
    m_selectedIndex = event.GetIndex();
    EventDialog dialog(this, m_data->getEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id()));
    dialog.ShowModal();
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
    selectDate();
}

void dxremindersFrame::OnItemSelected(wxListEvent& event)
{
    m_selectedIndex = event.GetIndex();
    enableTools();
    selectDate();
}

void dxremindersFrame::OnItemDeselected(wxListEvent& /*event*/)
{
    m_selectedIndex = -1;
    enableTools();
}

void dxremindersFrame::OnItemEditEnd(wxListEvent &event)
{
    if(!event.GetLabel().IsEmpty())
    {
        m_data->editEvent(m_eventViews[static_cast<size_t>(event.GetIndex())].id(), event.GetLabel());
    }
}

void dxremindersFrame::OnTimer(wxTimerEvent& /*event*/)
{
    if(dxsettings.remindHoliday() && dxsettings.remindHolidayLast().IsEarlierThan(wxDateTime::Today())
            && wxDateTime::Now().GetHour()*3600+wxDateTime::Now().GetMinute()*60+wxDateTime::Now().GetSecond() >=
            dxsettings.remindHolidayTime().GetHour()*3600+dxsettings.remindHolidayTime().GetMinute()*60+dxsettings.remindHolidayTime().GetSecond())
    {
        fireHoliday();
        dxsettings.setRemindHolidayLast(wxDateTime::Now());
    }
    if(wxDateTime::Now().GetSecond() != 0) return;
    fireEvents();
    if(m_filter && m_filters[4].active())
    {
        updateEventsList();
    }
}

void dxremindersFrame::OnCalendar(wxCalendarEvent &event)
{
    if(m_data->hasEventsForCalendarDate(event.GetDate().GetDateOnly()))
    {
        DayDialog dialog(this, event.GetDate());
        dialog.ShowModal();
    }
    else
    {
        EventDialog dialog(this, event.GetDate());
        dialog.ShowModal();
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnCalendarChanged(wxCalendarEvent &event)
{
    if(m_eventViews.size() == 0) return;
    if(m_calendar->daytypeForDay() == 0) return;
    else if(m_calendar->daytypeForDay() == 1)
    {
        m_eventsList->ClearSelection();
        size_t row = 0;
        std::vector<EventView>::const_iterator i;
        for(i = m_eventViews.begin(); i != m_eventViews.end(); ++i, ++row)
        {
            if((*i).date().IsSameDate(event.GetDate()))
            {
                m_eventsList->EnsureVisible(row);
                m_eventsList->Select(row);
//                m_eventsList->SetItemState(row, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
//                break;
            }
        }
    }
    else
    {
        m_eventsList->ClearSelection();
        size_t row = 0;
        std::vector<EventView>::const_iterator i;
        for(i = m_eventViews.begin(); i != m_eventViews.end(); ++i, ++row)
        {
            if(m_calendar->hasEventForDay((*i).id()))
            {
                m_eventsList->EnsureVisible(row);
                m_eventsList->Select(row);
//                m_eventsList->SetItemState(row, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
//                break;
            }
        }
    }
}

void dxremindersFrame::OnPreferences(wxCommandEvent &/*event*/)
{
    saveColumnSize();
#if defined (__WXMAC__)
    OptionsDialog dialog(nullptr, m_ipc);
#else
    OptionsDialog dialog(this, m_ipc);
#endif
    if(dialog.ShowModal() == wxID_OK) dxsettings.save();
    if(m_ipc)
    {
        bool notaskbar = (GetWindowStyleFlag()&wxFRAME_NO_TASKBAR) != 0;
        if(dxsettings.notaskbar())
        {
            if(!notaskbar) SetWindowStyleFlag(wxDEFAULT_FRAME_STYLE|wxFRAME_NO_TASKBAR);
        }
        else
        {
            if(notaskbar) SetWindowStyleFlag(wxDEFAULT_FRAME_STYLE);
        }
    }
    recreateColumns();
    m_evenMonth->SetBackgroundColour(dxsettings.evenMonth());
    m_evenMonth->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_oddMonth->SetBackgroundColour(dxsettings.oddMonth());
    m_oddMonth->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight1->SetTextColour(dxsettings.highlightFg());
    m_highlight1->SetBackgroundColour(dxsettings.highlightBg1());
    m_highlight1->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight2->SetTextColour(dxsettings.highlightFg());
    m_highlight2->SetBackgroundColour(dxsettings.highlightBg2());
    m_highlight2->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight3->SetTextColour(dxsettings.highlightFg());
    m_highlight3->SetBackgroundColour(dxsettings.highlightBg3());
    m_highlight3->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight4->SetTextColour(dxsettings.highlightFg());
    m_highlight4->SetBackgroundColour(dxsettings.highlightBg4());
    m_highlight4->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_highlight5->SetTextColour(dxsettings.highlightFg());
    m_highlight5->SetBackgroundColour(dxsettings.highlightBg5());
    m_highlight5->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_eventFont->SetFont(dxutils::fontFromString(dxsettings.eventsFont()));
    m_calendar->updateCalendar();
    if(calendar)
    {
        bool notaskbar = (calendar->GetWindowStyleFlag()&wxFRAME_NO_TASKBAR) != 0;
        if(dxsettings.notaskbar())
        {
            if(!notaskbar) calendar->SetWindowStyleFlag(wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER|wxFRAME_NO_TASKBAR);
        }
        else
        {
            if(notaskbar) calendar->SetWindowStyleFlag(wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER);
        }
        calendar->updateCalendar();
    }
    reLayout(dxsettings.resize());
    m_calendar->markDays(m_hidden);
    hideShowTrayIcon();
    m_data->reEncrypt();
    sendSilentmode();
    if(m_filter)
    {
        updateEventsList();
    }
}

void dxremindersFrame::OnKeyUp(wxKeyEvent &event)
{
    if(m_eventsList->HasFocus() && m_selectedIndex != -1 && event.GetKeyCode() == WXK_DELETE)
    {
        wxCommandEvent ev;
        OnRemoveEvent(ev);
    }
    if(event.GetKeyCode() == WXK_ESCAPE)
    {
        if(m_taskBarIcon || m_ipc)
        {
            Show(false);
            frameShown = !frameShown;
        }
    }
    event.Skip();
}

void dxremindersFrame::OnLoadDB(wxCommandEvent &/*event*/)
{
    wxString pk = "";
    wxArrayString choices;
    choices.Add(_("Overwrite all (replaces all events)"));
    choices.Add(_("Merge all (combines all events)"));
    choices.Add(_("Merge selected (combines one or more events)"));
#if defined (__WXMAC__)
    wxWindow *parent = nullptr;
#else
    wxWindow *parent = this;
#endif
    wxSingleChoiceDialog vyber(parent, _("Select action"), _("Select"), choices);
    vyber.SetMinSize(wxSize(-1,250));
    vyber.SetSelection(0);
    if(vyber.ShowModal() == wxID_OK)
    {
        wxFileDialog dialog(parent, _("Open database"), wxEmptyString, wxEmptyString,
                            wxString::Format( _("Database files (*.db)|*.db|All files (%s)|%s"), wxFileSelectorDefaultWildcardStr, wxFileSelectorDefaultWildcardStr), wxFD_OPEN|wxFD_FILE_MUST_EXIST);
        dialog.SetDirectory(wxGetHomeDir());
        if(dialog.ShowModal() == wxID_OK)
        {
            wxString path = dialog.GetPath();
            int id = wxID_NO; //for encrypted db
            if(!path.empty())
            {
                if(vyber.GetSelection() == 0)
                {
                    wxMessageDialog dialog(parent, _("Is database encrypted?"), PACKAGE, wxCENTER | wxNO_DEFAULT | wxYES_NO | wxICON_QUESTION);
                    id = dialog.ShowModal();
                    wxString pk = wxEmptyString;
                    if(id==wxID_YES)
                    {
                        wxPasswordEntryDialog pass(parent, _("Database password"));
                        if(pass.ShowModal() == wxID_OK)
                        {
                            pk = pass.GetValue();
                        }
                    }
                    if(!m_data->overwriteEvents(path, id==wxID_YES, pk))
                    {
                        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), path), PACKAGE);
                        loadDatabase();
                        return;
                    }
                }
                if(vyber.GetSelection() == 2)
                {
                    wxMessageDialog dialog(parent, _("Is database encrypted?"), PACKAGE, wxCENTER | wxNO_DEFAULT | wxYES_NO | wxICON_QUESTION);
                    id = dialog.ShowModal();
                    wxString pk = wxEmptyString;
                    if(id==wxID_YES)
                    {
                        wxPasswordEntryDialog pass(parent, _("Database password"));
                        if(pass.ShowModal() == wxID_OK)
                        {
                            pk = pass.GetValue();
                        }
                    }
                    if(!m_data->mergeEvents(path, id==wxID_YES, pk))
                    {
                        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), path), PACKAGE);
                        loadDatabase();
                        return;
                    }
                    else
                    {
                        MergeEventsDialog merge(parent);
                        if(merge.ShowModal() == wxID_OK)
                        {
                            loadDatabase();
                        }
                    }
                }
                if(vyber.GetSelection() == 1)
                {
                    wxMessageDialog dialog(parent, _("Is database encrypted?"), PACKAGE, wxCENTER | wxNO_DEFAULT | wxYES_NO | wxICON_QUESTION);
                    id = dialog.ShowModal();
                    wxString pk = wxEmptyString;
                    if(id==wxID_YES)
                    {
                        wxPasswordEntryDialog pass(parent, _("Database password"));
                        if(pass.ShowModal() == wxID_OK)
                        {
                            pk = pass.GetValue();
                        }
                    }
                    if(!m_data->mergeAllEvents(path, id==wxID_YES, dxsettings.pk()))
                    {
                        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), path), PACKAGE);
                        loadDatabase();
                        return;
                    }
                }
                wxMessageDialog holiday(parent, _("Overwrite Public holidays?"), PACKAGE, wxCENTER | wxNO_DEFAULT | wxYES_NO | wxICON_QUESTION);
                if(holiday.ShowModal() == wxID_YES)
                {
                    if(!m_data->overwriteHolidays(path, id==wxID_YES, pk))
                    {
                        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), path), _T(PACKAGE));
                    }
                    else
                    {
                        markDays();
                    }
                }
                wxMessageDialog note(parent, _("Overwrite note?"), PACKAGE, wxCENTER | wxNO_DEFAULT | wxYES_NO | wxICON_QUESTION);
                if(note.ShowModal() == wxID_YES)
                {
                    if(!m_data->overwriteNote(path, id==wxID_YES, pk))
                    {
                        wxMessageBox(wxString::Format(_("Couldn't open database file\n%s"), path), PACKAGE);
                    }
                    else
                    {
                        m_note->ChangeValue(m_data->noteText());;
                    }
                }
            }
        }
    }
}

void dxremindersFrame::OnSaveDB(wxCommandEvent &/*event*/)
{
    if(dxsettings.encrypt())
    {
        wxArrayString choices;
        choices.Add(_("Backup encrypted"));
        choices.Add(_("Backup non-encrypted"));
        wxSingleChoiceDialog vyber(this, _("Database is encrypted\nSelect action:"), _("Select"), choices);
        vyber.SetSelection(0);
        if(vyber.ShowModal() == wxID_OK)
        {
            if(vyber.GetSelection() == 0)
            {
                PassDialog(this).ShowModal();
                wxFileDialog dialog(this, _("Backup database"), wxStandardPaths::Get().GetDocumentsDir(), "dxreminders.db", _("Database files (*.db)|*.db"), wxFD_SAVE|wxFD_OVERWRITE_PROMPT);
                if(dialog.ShowModal() == wxID_OK)
                {
                    if(!dialog.GetFilename().IsEmpty()) wxCopyFile(wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db", dialog.GetPath());
                }
            }
            else
            {
                wxFileDialog dialog(this, _("Backup database"), wxStandardPaths::Get().GetDocumentsDir(), "dxreminders.db", _("Database files (*.db)|*.db"), wxFD_SAVE|wxFD_OVERWRITE_PROMPT);
                if(dialog.ShowModal() == wxID_OK)
                {
                    if(!dialog.GetFilename().IsEmpty())
                    {
                        wxCopyFile(wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db", dialog.GetPath());
                        wxSQLite3Database db;
                        db.Open(dialog.GetPath(), dxsettings.pk());
                        db.ReKey("");
                        db.Close();
                    }
                }
            }
        }
        return;
    }
    wxFileDialog dialog(this, _("Backup database"), wxStandardPaths::Get().GetDocumentsDir(), "dxreminders.db", _("Database files (*.db)|*.db"), wxFD_SAVE|wxFD_OVERWRITE_PROMPT);
    if(dialog.ShowModal() == wxID_OK)
    {
        if(!dialog.GetFilename().IsEmpty()) wxCopyFile(wxStandardPaths::Get().GetUserDataDir()+wxFILE_SEP_PATH+"reminders.db", dialog.GetPath());
    }
}

void dxremindersFrame::OnTestEvents(wxCommandEvent &/*event*/)
{
#if defined (__WXMAC__)
    long events = wxGetNumberFromUser("Pocet udalosti", "", "Pocet udalosti", 2, 2, 1000, nullptr);
#else
    long events = wxGetNumberFromUser("Pocet udalosti", "", "Pocet udalosti", 2, 2, 1000, this);
#endif
    if(events != -1)
    {
        std::random_device random_device;
        std::uniform_int_distribution<> distribution(0, 23);
        std::mt19937 generator(random_device());
        for(long i=1; i<=events; i++)
        {
            wxDateTime date = wxDateTime::Today()+wxDateSpan::Days(distribution(generator))+wxTimeSpan::Hours(distribution(generator));
            int day = date.GetDay();
            int recurrence = distribution(generator);
            if(recurrence == R_WEEKLYBUSINESS || recurrence == R_WEEKLYPLUSBUSINESSDAY || recurrence == R_WEEKLYGIVENBUSINESSDAY
                    || recurrence == R_2WEEKLYBUSINESS || recurrence == R_2WEEKLYPLUSBUSINESSDAY || recurrence == R_2WEEKLYGIVENBUSINESSDAY)
            {
                day = date.GetWeekDay();
            }
            int month = static_cast<int>(date.GetMonth())+1;
            int minutes = 0;
            if(recurrence == R_MINUTES || recurrence == R_OWN)
            {
                minutes = distribution(generator);
            }
            int hours = 0;
            if(recurrence == R_HOURLY || recurrence == R_OWN)
            {
                hours = distribution(generator);
            }
            int days = 0;
            if(recurrence == R_OWN)
            {
                days = distribution(generator);
            }
            int months = 0;
            if(recurrence == R_OWN)
            {
                months = distribution(generator)%3;
            }
            if(recurrence == R_MONTHLY || recurrence == R_MONTHLYATDAY || recurrence == R_MONTHLYBUSINESS || recurrence == R_MONTHLYGIVENBUSINESSDAY || recurrence == R_MONTHLYPLUSBUSINESSDAY)
            {
                months = 1;
            }
            if(recurrence == R_YEARLY)
            {
                months = 12;
            }
            int mday = 0;
            int mweek = 0;
            if(recurrence == R_MONTHLYATDAY)
            {
                mday = distribution(generator)%7;
                mweek = distribution(generator)%3+1;
            }
            if(recurrence == R_MONTHLYPLUSBUSINESSDAY || recurrence == R_QUARTERLYPLUSBUSINESSDAY || recurrence == R_MONTHLYGIVENBUSINESSDAY || recurrence == R_QUARTERLYGIVENBUSINESSDAY)
            {
                mday = distribution(generator);
            }
            wxInt64 reminder = distribution(generator)%5;
            if(recurrence == R_MINUTES)
            {
                reminder = 0;
            }
            m_data->addEvent(Event(-1,wxString::Format("Event#%ld",i),wxEmptyString, date,
                                   reminder,false,
                                   day,month,
                                   false,
                                   recurrence,
                                   minutes,
                                   hours,
                                   days,
                                   months,
                                   mday,
                                   mweek,
                                   reminder,
                                   INFINITY_DATE,
                                   false,
                                   distribution(generator)%5));
        }
        m_data->fireUpdated();
    }
}

void dxremindersFrame::OnRemoveEvents(wxCommandEvent &/*event*/)
{
    wxMessageDialog dialog(this, _("Do you really want to delete all Events?"), _("Confirm Events Deletion"), wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
    if(dialog.ShowModal() == wxID_YES)
    {
        m_data->removeEvents();
    }
}

wxBitmap dxremindersFrame::drawColorBitmap(int color)
{
    wxBitmap ico(16,16);
    if(ico.IsOk())
    {
        wxMemoryDC memdc;
        memdc.SelectObject(ico);
        memdc.SetPen(wxPen(wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOWTEXT)));
        if(color == 1)
        {
            memdc.SetBrush(wxBrush(dxsettings.highlightBg1()));
        }
        else if(color == 2)
        {
            memdc.SetBrush(wxBrush(dxsettings.highlightBg2()));
        }
        else if(color == 3)
        {
            memdc.SetBrush(wxBrush(dxsettings.highlightBg3()));
        }
        else if(color == 4)
        {
            memdc.SetBrush(wxBrush(dxsettings.highlightBg4()));
        }
        else
        {
            memdc.SetBrush(wxBrush(dxsettings.highlightBg5()));
        }
        memdc.DrawRectangle(0,0,16,16);
    }
    return ico;
}

void dxremindersFrame::eventslistMarkFiltered()
{
    m_eventsList->SetColumnHeaderFiltered(0, m_filter?m_filters[0].active()||m_filters[5].active():false);
    if(m_eventsList->GetColumnCount()>1)
    {
        if(dxsettings.colDateShow()) m_eventsList->SetColumnHeaderFiltered(1, m_filter?m_filters[1].active()||m_filters[4].active():false);
        else
        {
            if(dxsettings.colReminderShow()) m_eventsList->SetColumnHeaderFiltered(1, m_filter?m_filters[2].active():false);
            else m_eventsList->SetColumnHeaderFiltered(1, m_filter?m_filters[3].active():false);
        }
    }
    if(m_eventsList->GetColumnCount()>2)
    {
        if(dxsettings.colReminderShow() && dxsettings.colDateShow()) m_eventsList->SetColumnHeaderFiltered(2, m_filter?m_filters[2].active():false);
        else m_eventsList->SetColumnHeaderFiltered(2, m_filter?m_filters[3].active():false);
    }
    if(m_eventsList->GetColumnCount()>3)
    {
        m_eventsList->SetColumnHeaderFiltered(3, m_filter?m_filters[3].active():false);
    }
}

void dxremindersFrame::OnServerEvent(wxSocketEvent &/*event*/)
{
    m_sock = m_server->Accept(false);
    if(m_sock)
    {
#ifdef wxHAS_UNIX_DOMAIN_SOCKETS
        wxLogDebug("New connection from client accepted.");
#else
        wxIPV4address addr;
        if(!m_sock->GetPeer(addr))
        {
            wxLogDebug("New connection from unknown client accepted.");
        }
        else
        {
            wxLogDebug("New client connection from %s:%u accepted", addr.IPAddress(), addr.Service());
        }
#endif
    }
    else
    {
        wxLogMessage(_("Error: couldn't accept a new connection"));
        return;
    }
    m_sock->SetEventHandler(*this, ID_SOCKET);
    m_sock->SetNotify(wxSOCKET_INPUT_FLAG | wxSOCKET_LOST_FLAG);
    m_sock->Notify(true);
}

void dxremindersFrame::OnSocketEvent(wxSocketEvent &event)
{
    wxSocketBase *sock = event.GetSocket();
    if(!m_sock->IsOk()) m_sock = sock;
    switch(event.GetSocketEvent())
    {
        case wxSOCKET_INPUT:
        {
            sock->SetNotify(wxSOCKET_LOST_FLAG);
            char tmpbuf[1024];
            sock->Read(tmpbuf, sizeof(tmpbuf) - 1);
            tmpbuf[sock->LastCount()] = '\0';
            sock->SetNotify(wxSOCKET_LOST_FLAG | wxSOCKET_INPUT_FLAG);
            m_ipcString.Append(tmpbuf);
            while(m_ipcString.Find('>') != wxNOT_FOUND)
            {
                wxString line = m_ipcString.BeforeFirst('>');
                line = line.AfterFirst('<');
                parseReceived(line);
                m_ipcString = m_ipcString.AfterFirst('>');
            }
            break;
        }
        case wxSOCKET_LOST:
        {
            wxLogDebug("Deleting socket.");
            sock->Destroy();
            break;
        }
        default: ;
    }
}

void dxremindersFrame::OnHolidays(wxCommandEvent &/*event*/)
{
    int days = dxsettings.remindHolidayDays();
#if defined (__WXMAC__)
    PublicholidayDialog dialog(nullptr);
#else
    PublicholidayDialog dialog(this);
#endif
    if(dialog.ShowModal() == wxID_OK)
    {
        if(dxsettings.remindHolidayDays() > days) dxsettings.setRemindHolidayLast(wxDateTime(21, wxDateTime::Mar, 1976));
        dxsettings.save();
    }
}

void dxremindersFrame::OnShowHideNote(wxCommandEvent &/*event*/)
{
    dxsettings.setShowNote(!noteShown());
    saveColumnSize();
    recreateColumns();
    reLayout(true);
}

void dxremindersFrame::OnHideEvent(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex == -1) return;
    bool hidden = false;
    long index = m_eventsList->GetFirstSelected();
    if(index != -1)
    {
        m_data->hideEvent(m_eventViews[static_cast<size_t>(index)].id());
        hidden = true;
    }
    do
    {
        index = m_eventsList->GetNextSelected(index);
        if(index != -1)
        {
            m_data->hideEvent(m_eventViews[static_cast<size_t>(index)].id());
            hidden = true;
        }
    } while(index > 0);
    if(!hidden)
    {
        m_data->hideEvent(m_eventViews[static_cast<size_t>(m_selectedIndex)].id());
        hidden = true;
    }
    if(hidden && !m_showHiddenEvents->IsChecked())
    {
        m_hidden = false;
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, true);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
        m_showHiddenEvents->Check(true);
        m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnHideEventsOnList(wxCommandEvent &/*event*/)
{
    if(!m_showHiddenEvents->IsChecked())
    {
        m_hidden = false;
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, true);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
        m_showHiddenEvents->Check(true);
        m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
        m_data->setLastid(-1);
        markDays();
        updateEventsList();
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnShowEventsMenu(wxCommandEvent &/*event*/)
{
    if(m_showHiddenEvents->IsChecked())
    {
        m_hidden = false;
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, true);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
        m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    else
    {
        m_hidden = true;
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, false);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Hide events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_HIDEALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GHIDEALL);
        m_showHiddenEvents->SetItemLabel(_("Hide events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
    m_data->setLastid(-1);
    markDays();
    updateEventsList();
}

void dxremindersFrame::OnShowEventsBtn(wxCommandEvent &/*event*/)
{
    if(m_bar->GetToolState(ID_BTN_SHOWEVENTS))
    {
        m_hidden = false;
        m_showHiddenEvents->Check();
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Show events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_SHOWALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GSHOWALL);
        m_showHiddenEvents->SetItemLabel(_("Show events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    else
    {
        m_hidden = true;
        m_showHiddenEvents->Check(false);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Hide events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_HIDEALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GHIDEALL);
        m_showHiddenEvents->SetItemLabel(_("Hide events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
    m_data->setLastid(-1);
    markDays();
    updateEventsList();
}

void dxremindersFrame::OnShowEventsRightclick(wxCommandEvent &event)
{
    m_bar->ToggleTool(ID_BTN_SHOWEVENTS, false);
    OnShowEventsBtn(event);
}

void dxremindersFrame::OnEngineUpdated(wxCommandEvent &event)
{
    if(event.GetId()==1)
    {
        sendTooltip();
    }
    if(m_data->someEventOnHiddenList())
    {
        m_bar->EnableTool(ID_BTN_CLEARHIDDEN, true);
        m_clearHiddenList->Enable();
        m_showHiddenEvents->Enable();
        m_bar->EnableTool(ID_BTN_SHOWEVENTS, true);
    }
    else
    {
        m_hidden = false;
        m_bar->EnableTool(ID_BTN_CLEARHIDDEN, false);
        m_clearHiddenList->Enable(false);
        m_showHiddenEvents->Enable(false);
        m_bar->EnableTool(ID_BTN_SHOWEVENTS, false);
    }
    if(m_data->someEventOnHighlightList())
    {
        m_clearHighlightList->Enable();
    }
    else
    {
        m_clearHighlightList->Enable(false);
    }
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
    if(dxsettings.silentMode())
    {
        fireEvents();
        markDays();
    }
    else
    {
        if(!fireEvents()) markDays();
    }
    m_removeEvents->Enable(m_data->someEvents()?true:false);
    updateEventsList();
}

void dxremindersFrame::OnEngineMsg(wxCommandEvent &event)
{
    wxMessageDialog dialog(this, event.GetString(), _("Information"), wxCENTER|wxOK_DEFAULT|wxOK|wxICON_INFORMATION);
    dialog.ShowModal();
}

void dxremindersFrame::OnClearHiddenList(wxCommandEvent &/*event*/)
{
    if(m_showHiddenEvents->IsChecked())
    {
        m_hidden = true;
        m_bar->ToggleTool(ID_BTN_SHOWEVENTS, false);
        m_showHiddenEvents->Check(false);
        m_bar->SetToolShortHelp(ID_BTN_SHOWEVENTS, _("Hide events"));
        m_bar->SetToolNormalBitmap(ID_BTN_SHOWEVENTS, ICO_HIDEALL);
        m_bar->SetToolDisabledBitmap(ID_BTN_SHOWEVENTS, ICO_GHIDEALL);
        m_showHiddenEvents->SetItemLabel(_("Hide events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWEVENTS));
    }
    m_data->clearListOfHiddenEvents();
    if(m_data->someDonotdeleteEvent(m_hidden))
    {
        m_showDonotDelete->Enable();
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, true);
    }
    else
    {
        m_showDonotDelete->Enable(false);
        m_bar->EnableTool(ID_MENU_SHOWNOTDELETE, false);
    }
}

void dxremindersFrame::OnRemoveFromHiddenList(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex != -1)
    {
        long index = m_eventsList->GetFirstSelected();
        if(index != -1)
        {
            m_data->removeEventFromHiddenList(m_eventViews[static_cast<size_t>(index)].id());
        }
        do
        {
            index = m_eventsList->GetNextSelected(index);
            if(index != -1)
            {
                m_data->removeEventFromHiddenList(m_eventViews[static_cast<size_t>(index)].id());
            }
        } while(index > 0);
    }
}

void dxremindersFrame::OnFullscreen(wxCommandEvent &/*event*/)
{
    if(!IsFullScreen())
    {
        dxsettings.setX(GetPosition().x);
        dxsettings.setY(GetPosition().y);
        dxsettings.setW(GetSize().x);
        dxsettings.setH(GetSize().y);
    }
    ShowFullScreen(!IsFullScreen());
}

void dxremindersFrame::OnAlwaysShow(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex != -1)
    {
        long index = m_eventsList->GetFirstSelected();
        if(index != -1)
        {
            if(m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_NONE || m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_ONCENOTDELETE)
                m_data->alwaysshowEvent(m_eventViews[static_cast<size_t>(index)].id(), true);
        }
        do
        {
            index = m_eventsList->GetNextSelected(index);
            if(index != -1)
            {
                if(m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_NONE || m_eventViews[static_cast<size_t>(index)].recurrencetype()==R_ONCENOTDELETE)
                    m_data->alwaysshowEvent(m_eventViews[static_cast<size_t>(index)].id(), true);
            }
        } while(index > 0);
    }
}

void dxremindersFrame::OnHighlightEvent(wxCommandEvent &event)
{
    if(m_selectedIndex != -1)
    {
        int color;
        switch(event.GetId()) {
        case ID_MENU_HIGHLIGHTEVENT1: color = 1; break;
        case ID_MENU_HIGHLIGHTEVENT2: color = 2; break;
        case ID_MENU_HIGHLIGHTEVENT3: color = 3; break;
        case ID_MENU_HIGHLIGHTEVENT4: color = 4; break;
        default: color = 5;
        }
        long index = m_eventsList->GetFirstSelected();
        if(index != -1)
        {
            m_data->highlightEvent(m_eventViews[static_cast<size_t>(index)].id(), color);
        }
        do
        {
            index = m_eventsList->GetNextSelected(index);
            if(index != -1)
            {
                m_data->highlightEvent(m_eventViews[static_cast<size_t>(index)].id(), color);
            }
        } while(index > 0);
    }
}

void dxremindersFrame::OnUnHighlightEvent(wxCommandEvent &/*event*/)
{
    if(m_selectedIndex != -1)
    {
        long index = m_eventsList->GetFirstSelected();
        if(index != -1)
        {
            m_data->removeEventFromHighlightList(m_eventViews[static_cast<size_t>(index)].id());
        }
        do
        {
            index = m_eventsList->GetNextSelected(index);
            if(index != -1)
            {
                m_data->removeEventFromHighlightList(m_eventViews[static_cast<size_t>(index)].id());
            }
        } while(index > 0);
    }
}

void dxremindersFrame::OnClearHighlightList(wxCommandEvent &/*event*/)
{
    m_data->clearListOfHighlightEvents();
}

void dxremindersFrame::OnFilter(wxCommandEvent &/*event*/)
{
    bool oldfilter = m_filter;
    dxFilterArray oldfilters = m_filters;
    EventFilter filter(this,m_filters);
    int result = filter.ShowModal();
    if(result == wxID_OK)
    {
        m_filters = filter.filters();
        m_filter = false;
        for(size_t i=0; i<m_filters.GetCount(); i++)
        {
            if(m_filters[i].active())
            {
                m_filter = true;
                break;
            }
        }
        m_bar->ToggleTool(ID_BTN_FILTER, m_filter);
        m_filterMenuItem->Check(m_filter);
        m_bar->SetToolShortHelp(ID_BTN_FILTERCLEAR, m_filter?_("Clear filter"):_("Revert filter"));
        m_clearfilterMenuItem->SetItemLabel(m_filter?_("Clear filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR):_("Revert filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR));
        eventslistMarkFiltered();
        m_eventsList->Update();
    }
    if(result == wxID_CANCEL)
    {
        m_filters = oldfilters;
        m_filter = oldfilter;
        m_bar->ToggleTool(ID_BTN_FILTER, m_filter);
        m_filterMenuItem->Check(m_filter);
        eventslistMarkFiltered();
        m_eventsList->Update();
    }
    updateEventsList();
}

void dxremindersFrame::OnClearFilter(wxCommandEvent &/*event*/)
{
    m_filter = !m_filter;
    m_bar->ToggleTool(ID_BTN_FILTER, m_filter);
    m_bar->SetToolShortHelp(ID_BTN_FILTERCLEAR, m_filter?_("Clear filter"):_("Revert filter"));
    m_clearfilterMenuItem->SetItemLabel(m_filter?_("Clear filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR):_("Revert filter")+"\t"+dxsettings.shortcutKey(ID_BTN_FILTERCLEAR));
    m_filterMenuItem->Check(m_filter);
    eventslistMarkFiltered();
    m_eventsList->Update();
    updateEventsList();
}

void dxremindersFrame::OnShowDonotdelete(wxCommandEvent &/*event*/)
{
    m_hideDonotdelete = !m_hideDonotdelete;
    m_showDonotDelete->Check(m_hideDonotdelete);
    m_showDonotDelete->SetItemLabel(m_hideDonotdelete?_("Show all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE):
                                                      _("Hide all do not delete events")+"\t"+dxsettings.shortcutKey(ID_MENU_SHOWNOTDELETE));
    m_bar->ToggleTool(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete);
    m_bar->SetToolShortHelp(ID_MENU_SHOWNOTDELETE, m_hideDonotdelete?_("Show all do not delete events"):_("Hide all do not delete events"));
    m_eventsList->Update();
    updateEventsList();
}

void dxremindersFrame::OnCalendarFocus(wxCommandEvent &/*event*/)
{
    if(dxsettings.look() == 2) return; //No calendar shown
    m_calendar->SetFocus();
}

void dxremindersFrame::OnEventlistFocus(wxCommandEvent &/*event*/)
{
    if(dxsettings.look() == 3) return; //No eventslist shown
    m_eventsList->SetFocus();
}

bool dxremindersFrame::fireEvents()
{
    if(m_data->someEventsByReminder(wxDateTime::Now()))
    {
        if(dxsettings.silentMode())
        {
            if(m_taskBarEvents == nullptr)
            {
                m_taskBarEvents = new dxTaskBarEvents();
                m_taskBarEvents->SetIcon(ICO_ETRAY, _("Some events to remind"));
            }
            return true;
        }
        if(m_msgDialog == nullptr)
        {
#if defined (__WXMAC__)
            m_msgDialog = new EventMsgDialog(nullptr);
#else
            m_msgDialog = new EventMsgDialog(this);
#endif
            m_msgDialog->ShowModal();
            markDays();
            updateEventsList();
            sendTooltip();
            if(m_data->someEventsByReminder(wxDateTime::Now()))
            {
#if defined (__WXMAC__)
                m_msgDialog = new EventMsgDialog(nullptr);
#else
                m_msgDialog = new EventMsgDialog(this);
#endif
                m_msgDialog->ShowModal();
                markDays();
                updateEventsList();
                sendTooltip();
            }
            m_msgDialog = nullptr;
            return true;
        }
    }
    return false;
}

void dxremindersFrame::markDays()
{
    m_calendar->markDays(m_hidden);
    if(calendar != nullptr) calendar->markDays(m_hidden);
}

BEGIN_EVENT_TABLE(dxTaskBarEvents, wxTaskBarIcon)
    EVT_TASKBAR_LEFT_DCLICK(dxTaskBarEvents::OnLeftButtonDClick)
    EVT_MENU(ID_TASK_EVENTS, dxTaskBarEvents::OnShowEvents)
END_EVENT_TABLE()

void dxTaskBarEvents::OnLeftButtonDClick(wxTaskBarIconEvent &)
{
    frame->showEvents();
}

wxMenu *dxTaskBarEvents::CreatePopupMenu()
{
    wxMenu *menu = new wxMenu;
    menu->Append(ID_TASK_EVENTS, _("Show events"));
    return menu;
}

void dxTaskBarEvents::OnShowEvents(wxCommandEvent &/*event*/)
{
    frame->showEvents();
}
