#!/bin/bash

# --- KONFIGURATION DER KONFIGURATIONSDATEI ---
# Pfad zur benutzerspezifischen Konfigurationsdatei im Home-Verzeichnis.
CONFIG_FILE="$HOME/.htmjoypadcomborc"
# Pfad zum systemweiten Template der Konfigurationsdatei.
TEMPLATE_FILE="/etc/htmjoypadcombo.template"

# --- FUNKTION ZUM ERSTELLEN DER KONFIGURATION AUS DEM TEMPLATE ---
# Diese Funktion wird aufgerufen, wenn die benutzerspezifische Konfigurationsdatei nicht gefunden wird.
create_config_from_template() {
    echo "Konfigurationsdatei '$CONFIG_FILE' nicht gefunden."
    
    # Prüfe, ob das Template überhaupt existiert
    if [ ! -f "$TEMPLATE_FILE" ]; then
        echo "Fehler: Das systemweite Template '$TEMPLATE_FILE' wurde nicht gefunden!"
        echo "Bitte erstelle diese Datei manuell oder stelle sicher, dass das Paket korrekt installiert ist."
        exit 1
    fi

    echo "Kopiere Template von '$TEMPLATE_FILE' nach '$CONFIG_FILE'."
    cp "$TEMPLATE_FILE" "$CONFIG_FILE"
    echo "Die Template-Datei wurde kopiert. BITTE BEARBEITE '$CONFIG_FILE' MIT DEINEN WERTEN!"
    echo "Starte das Skript danach neu."
    exit 1 # Skript beenden, damit der Benutzer die Datei bearbeiten kann
}

# --- PRÜFEN UND LADEN DER KONFIGURATION ---
# Wenn die benutzerspezifische Konfigurationsdatei NICHT existiert, erstelle sie aus dem Template.
if [ ! -f "$CONFIG_FILE" ]; then
    create_config_from_template
fi

# Lade die Konfiguration aus der benutzerspezifischen Datei in die Umgebung des Skripts.
source "$CONFIG_FILE"

# Überprüfe, ob alle benötigten Variablen aus der Konfigurationsdatei gesetzt wurden.
if [ -z "$JOYPAD_DEVICE" ] || [ -z "$BUTTON1_CODE" ] || [ -z "$BUTTON2_CODE" ] || [ -z "$COMBO_TIMEOUT" ] || [ -z "$COMMAND_TO_EXECUTE" ]; then
    echo "Fehler: Nicht alle notwendigen Konfigurationsvariablen in '$CONFIG_FILE' sind gesetzt oder sind leer."
    echo "Bitte überprüfe JOYPAD_DEVICE, BUTTON1_CODE, BUTTON2_CODE, COMBO_TIMEOUT und COMMAND_TO_EXECUTE in der Datei."
    exit 1
fi

# --- HAUPTPROGRAMMLOGIK ---
LAST_BUTTON_PRESS_TIME=0
LAST_BUTTON_CODE=""

echo "Joypad Combo Monitor gestartet. Überwache $JOYPAD_DEVICE"
echo "Drücke die Tasten (Codes: $BUTTON1_CODE und $BUTTON2_CODE) nacheinander innerhalb von $COMBO_TIMEOUT Sekunden."
echo "Ausgeführter Befehl bei Kombi: '$COMMAND_TO_EXECUTE'"

# Die Endlosschleife, die kontinuierlich auf Joypad-Ereignisse wartet.
while true; do
    # Prüfen, ob das Joypad-Gerät existiert und bereit ist.
    if [ ! -c "$JOYPAD_DEVICE" ]; then
        echo "Fehler: Joypad-Gerät $JOYPAD_DEVICE nicht gefunden oder nicht bereit. Warte 5 Sekunden..."
        sleep 5
        continue # Springe zum Anfang der Schleife und versuche es erneut.
    fi

    # Lese rohe Event-Daten vom Joypad-Gerät.
    # 'cat' liest die Binärdaten, 'hexdump' formatiert sie in Hexadezimal.
    # Die innere Schleife 'while read -r line' verarbeitet jede Zeile des hexdump-Outputs.
    cat "$JOYPAD_DEVICE" | hexdump -v -e '16/1 "%02x " "|\n"' | \
    while read -r line; do
        # Suche nach einem "Key Down" Event: Typ (01) und Wert (1).
        if [[ "$line" =~ \ 01\ 00\ ([0-9a-f]{2})\ ([0-9a-f]{2})\ 01\ 00\ 00\ 00 ]]; then
            # Extrahiere die Hex-Bytes des Codes und konvertiere sie in Dezimal.
            CODE_HEX_LSB="${BASH_REMATCH[1]}"
            CODE_HEX_MSB="${BASH_REMATCH[2]}"
            DEC_CODE=$((16#$CODE_HEX_MSB$CODE_HEX_LSB)) 

            CURRENT_TIME=$(date +%s.%N) # Erfasse die aktuelle Zeit für die Kombinationsprüfung.

            # Prüfe, ob der gedrückte Code einer der beiden definierten Tasten ist.
            if [ "$DEC_CODE" -eq "$BUTTON1_CODE" ] || [ "$DEC_CODE" -eq "$BUTTON2_CODE" ]; then
                echo "Taste gedrückt: Code $DEC_CODE"
                
                if [ -z "$LAST_BUTTON_CODE" ]; then
                    # Dies ist die erste Taste einer möglichen Kombination.
                    LAST_BUTTON_CODE="$DEC_CODE"
                    LAST_BUTTON_PRESS_TIME="$CURRENT_TIME"
                elif [ "$DEC_CODE" -ne "$LAST_BUTTON_CODE" ]; then
                    # Dies ist die zweite, *andere* Taste der Kombination.
                    TIME_DIFF=$(echo "$CURRENT_TIME - $LAST_BUTTON_PRESS_TIME" | bc -l) 
                    
                    if (( $(echo "$TIME_DIFF < $COMBO_TIMEOUT" | bc -l) )); then
                        echo "KOMBINATION ($LAST_BUTTON_CODE + $DEC_CODE) ERKANNT!"
                        
                        # --- BEFEHL AUS DER KONFIGURATION AUSFÜHREN ---
                        echo "Führe Befehl aus: $COMMAND_TO_EXECUTE"
                        eval "$COMMAND_TO_EXECUTE" # 'eval' führt den String als Shell-Befehl aus.
                        # -----------------------------------------------
                    fi
                    # Setze die Statusvariablen zurück, um auf die nächste Kombination zu warten.
                    LAST_BUTTON_CODE=""
                    LAST_BUTTON_PRESS_TIME=0
                else
                    # Die gleiche Taste wurde zweimal hintereinander gedrückt – dies zählt nicht als Kombination.
                    LAST_BUTTON_CODE=""
                    LAST_BUTTON_PRESS_TIME=0
                fi
            fi
        fi
    done
    # Diese Zeile wird erreicht, wenn 'cat' endet (z.B. wenn das Joypad getrennt wird).
    echo "Joypad-Leseschleife beendet (Gerät möglicherweise getrennt). Versuche Reconnect..."
    sleep 2 # Kurze Pause, bevor ein erneuter Versuch gestartet wird.
done
