##############################################################################
#
# Copyright (c) 2006 Zope Foundation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################


import shutil
import os
import doctest
import errno
import sys

def rmtree (path):
    """
    A variant of shutil.rmtree which tries hard to be successful.
    On windows shutil.rmtree aborts when it tries to delete a
    read only file or a file which is still handled by another
    process (e.g. antivirus scanner). This tries to chmod the
    file to writeable and retries 10 times before giving up.

    Also it tries to remove symlink itself if a symlink as passed as
    path argument.

    Finally, it tries to make parent directory writable.
    >>> from tempfile import mkdtemp

    Let's make a directory ...

    >>> d = mkdtemp()

    and make sure it is actually there

    >>> os.path.isdir (d)
    1

    Now create a file ...

    >>> foo = os.path.join (d, 'foo')
    >>> with open (foo, 'w') as f: _ = f.write ('huhu')
    >>> bar = os.path.join (d, 'bar')
    >>> os.symlink(bar, bar)

    and make it unwriteable

    >>> os.chmod (foo, 0o400)

    and make parent dir unwritable

    >>> os.chmod (d, 0o400)

    rmtree should be able to remove it:

    >>> rmtree (d)

    and now the directory is gone

    >>> os.path.isdir (d)
    0

    Let's make a directory ...

    >>> d = mkdtemp()

    and make sure it is actually there

    >>> os.path.isdir (d)
    1

    Now create a broken symlink ...

    >>> foo = os.path.join (d, 'foo')
    >>> os.symlink(foo + '.not_exist', foo)

    rmtree should be able to remove it:

    >>> rmtree (foo)

    and now the directory is gone

    >>> os.path.isdir (foo)
    0

    cleanup directory

    >>> rmtree (d)

    and now the directory is gone

    >>> os.path.isdir (d)
    0
    """
    def chmod_retry(func, failed_path, exc_info):
        """Make sure the directories are executable and writable.
        """
        if func is os.path.islink:
            os.unlink(path)
        elif func is os.lstat or func is os.open:
            if not os.path.islink(path):
                raise
            os.unlink(path)
        else:
            # Depending on the Python version, the following items differ.
            if sys.version_info >= (3, ):
                expected_error_type = PermissionError
                expected_func_tuple = (os.lstat, os.open)
            else:
                expected_error_type = OSError
                expected_func_tuple = (os.listdir, )
            e = exc_info[1]
            if isinstance(e, expected_error_type):
                if e.errno == errno.ENOENT:
                    # because we are calling again rmtree on listdir errors, this path might
                    # have been already deleted by the recursive call to rmtree.
                    return
                if e.errno == errno.EACCES:
                    if func in expected_func_tuple:
                        os.chmod(failed_path, 0o700)
                        # corner case to handle errors in listing directories.
                        # https://bugs.python.org/issue8523
                        return shutil.rmtree(failed_path, onerror=chmod_retry)
                    # If parent directory is not writable, we still cannot delete the file.
                    # But make sure not to change the parent of the folder we are deleting.
                    if failed_path != path:
                        os.chmod(os.path.dirname(failed_path), 0o700)
                        return func(failed_path)
            raise

    shutil.rmtree(path, onerror=chmod_retry)

def test_suite():
    return doctest.DocTestSuite()

if "__main__" == __name__:
    doctest.testmod()
