/**
 * @file src/prg/c/src/srv-prg/swocserverd/main.c
 *
 * Server Wait On Clients server daemon.
 * Daemon to enable a server to manage client locks and wait on the removal of
 * those locks prior to further server processing.
 *
 * @author Copyright (C) 2016-2022  Mark Grant
 *
 * Released under the GPLv3 only.\n
 * SPDX-License-Identifier: GPL-3.0-only
 *
 * @version _v1.0.22 ==== 17/09/2022_
 */

/* **********************************************************************
 *									*
 * Changelog								*
 *									*
 * Date		Author	Version	Description				*
 *									*
 * 19/11/2016	MG	1.0.1	First release.				*
 * 17/12/2016	MG	1.0.2	Re-distribute some functions over	*
 *				libswocserver and new common library	*
 *				libswoccommon.				*
 * 08/01/2017	MG	1.0.3	Add creation / deletion of pid file.	*
 *				(Recommended for a systemd service of	*
 *				type forking.)				*
 * 02/02/2017	MG	1.0.4	Migrate to use of mge_errno in libmgec.	*
 * 13/02/2017	MG	1.0.5	Implement config file reload		*
 *				functionality.				*
 * 				Correct validate config file failure	*
 *				exiting with 0. swsd_err was not	*
 *				assigned an error value on failure.	*
 * 22/04/2017	MG	1.0.6	Change to use new bstree struct.	*
 * 27/05/2017	MG	1.0.7	Give daemon its own config file.	*
 * 				Use new local validateconfig.		*
 * 				Add support for temporary include	*
 *				directory.				*
 * 04/06/2017	MG	1.0.8	Split comms functions out of main into	*
 *				their own source file.			*
 *				Tidy up unnecessary include statements.	*
 * 				Use more meaningful name for client	*
 *				lock bstree.				*
 * 07/06/2017	MG	1.0.9	Implement epoll controlled use of	*
 *				multiple ports.				*
 * 16/09/2017	MG	1.0.10	Add full file path to error message if	*
 *				pid file creation fails.		*
 * 				Change close to re-direct for stdin,	*
 *				stdout & stderr during daemonisation as	*
 *				epoll_wait silently does not work as	*
 *				expected if the standard fd's are 	*
 *				re-used.				*
 * 18/11/2017	MG	1.0.11	Add Doxygen comments.			*
 *				Add SPDX license tag.			*
 * 19/11/2017	MG	1.0.12	Make program exit with EXIT_SUCCESS or	*
 *				EXIT_FAILURE only.			*
 * 22/03/2018	MG	1.0.13	Remove unnecessary libsoccommon.h	*
 * 10/05/2018	MG	1.0.14	Add support for blocked clients list.	*
 *				Add support for server locking.		*
 * 22/05/2018	MG	1.0.15	Change from swocserverd.h to internal.h	*
 * 18/05/2019	MG	1.0.16	Merge sub-projects into one.		*
 * 01/06/2019	MG	1.0.17	Use standard GNU ifdeffery around use	*
 *				of AC_HEADER_STDBOOL.			*
 * 09/03/2020	MG	1.0.18	Initialise client.			*
 * 13/10/2021	MG	1.0.19	Eliminate -Wunused-result warnings.	*
 * 08/12/2021	MG	1.0.20	Tighten SPDX tag.			*
 * 11/06/2022	MG	1.0.21	Replace sprintf with safer snprintf.	*
 * 17/09/2022	MG	1.0.22	Rename bstree.h				*
 *				Use pkginclude location.		*
 *				Correct included headers.		*
 *									*
 ************************************************************************
 */

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <syslog.h>
#include <unistd.h>

/* Standard GNU AC_HEADER_STDBOOL ifdeffery. */
#ifdef HAVE_STDBOOL_H
	#include <stdbool.h>
#else
	#ifndef HAVE__BOOL
		#ifdef __cplusplus
typedef bool _Bool;
		#else
			#define _Bool signed char
		#endif
	#endif
	#define bool _Bool
	#define false 0
	#define true 1
	#define __bool_true_false_are_defined 1
#endif

#include <configmake.h>

#include "internal.h"
#include <libmgec/libmgec.h>
#include <libmgec/mge-bstree.h>
#include <libmgec/mge-errno.h>
#include <swoc/cmdlineargs.h>
#include <swoc/signalhandle.h>

int swsd_err;			   /**< swoc daemon error number. */
char client[_POSIX_HOST_NAME_MAX]; /**< Client name. */
int debug;			   /**< Debug - 0 false, 1 true. */
int end;			   /**< End pending. */
int cursockfd;			   /**< Socket file descriptor in use. */
struct comm_spec *port_spec;	   /**< Port / socket config mappings. */
bool srv_blocked;		   /**< Server is blocked? */
struct bstree *cli_locks;	   /**< Clients and locks. */
struct bstree *cli_blocked;	   /**< Blocked client list. */
struct bstree *port_sock;	   /**< Port / socket actual mappings. */

static void daemonise(void);
static int csscmp(const struct comm_spec *first, const struct comm_spec *last);

/**
 * Program entry point.
 * @param argc Standard CLA argc.
 * @param argv Standard CLA argv.
 * @return EXIT_SUCCESS on success, EXIT_FAILURE on error.
 */
int main(int argc, char **argv)
{
	char *pidfile = RUNSTATEDIR "/swocserverd.pid";
	FILE *fp;

	client[0] = '\0';
	swsd_err = 0;
	debug = 0;
	end = 0;

	/* Initialise signal handling. */
	init_sig_handle();

	/* Check not already running. */
	fp = fopen(pidfile, "r");
	if (fp != NULL) {
		syslog((int)(LOG_USER | LOG_NOTICE), "Daemon already "
						     "running.");
		fclose(fp);
		exit(EXIT_FAILURE);
	}

	/* Validate the config file. */
	port_spec = calloc(MAX_LISTEN_PORTS, sizeof(*port_spec));
	if (port_spec == NULL) {
		swsd_err = errno;
		if (debug)
			perror("ERROR allocating port_spec");
		syslog((int)(LOG_USER | LOG_NOTICE),
		       "ERROR allocating "
		       "port_spec - %s",
		       strerror(swsd_err));
		exit(EXIT_FAILURE);
	}
	swsd_err = swsd_validate_config();
	if (swsd_err)
		goto b4_cli_locks;

	/* Process command line. */
	swsd_err = process_cla(argc, argv);
	if (swsd_err)
		goto b4_cli_locks;

	/* Daemonise if not in debug mode. */
	if (!debug)
		daemonise();

	srv_blocked = false;

	cli_locks = cre_bst(BST_NODES_DUPLICATES,
			    (int (*)(const void *, const void *))strcmp);
	if (cli_locks == NULL) {
		if (debug)
			fprintf(stderr, "BST creation errored with %i.\n",
				mge_errno);
		syslog((int)(LOG_USER | LOG_NOTICE),
		       "BST creation errored "
		       "with %i.",
		       mge_errno);
		goto b4_cli_locks;
	}

	cli_blocked = cre_bst(BST_NODES_UNIQUE,
			      (int (*)(const void *, const void *))strcmp);
	if (cli_blocked == NULL) {
		if (debug)
			fprintf(stderr, "BST creation errored with %i.\n",
				mge_errno);
		syslog((int)(LOG_USER | LOG_NOTICE),
		       "BST creation errored "
		       "with %i.",
		       mge_errno);
		goto b4_cli_blocked;
	}

	/* Prepare sockets. */
	port_sock = cre_bst(BST_NODES_UNIQUE,
			    (int (*)(const void *, const void *))csscmp);
	if (port_sock == NULL) {
		if (debug)
			fprintf(stderr, "BST creation errored with %i.\n",
				mge_errno);
		syslog((int)(LOG_USER | LOG_NOTICE),
		       "BST creation errored "
		       "with %i.",
		       mge_errno);
		goto b4_port_sock;
	}
	swsd_err = prepare_sockets();
	if (swsd_err)
		goto comms_fail;

	/* Wait and perform comms. */
	swsd_err = process_comms();

comms_fail:
	port_sock = del_bst(port_sock);

b4_port_sock:
	cli_blocked = del_bst(cli_blocked);

b4_cli_blocked:
	cli_locks = del_bst(cli_locks);

b4_cli_locks:
	free(port_spec);
	if (!debug)
		remove(pidfile);
	if (swsd_err)
		exit(EXIT_FAILURE);
	exit(EXIT_SUCCESS);
}

/*
 * Daemonise the process.
 */
static void daemonise(void)
{
	char *pidfile = RUNSTATEDIR "/swocserverd.pid";
	pid_t pid, sid;
	char spid[256];
	FILE *fp;

	syslog((int)(LOG_USER | LOG_NOTICE), "Starting daemon.");

	/*
	 * Fork off the parent process.
	 * < 0 Error
	 * == 0 This is the child
	 * > 0 is pid of child, ie this is the parent.
	 */
	pid = fork();
	if (pid < 0) {
		syslog((int)(LOG_USER | LOG_NOTICE), "Error forking.");
		exit(EXIT_FAILURE);
	}
	/* This is the parent process, so exit after creating pid file. */
	if (pid > 0) {
		/* Create pid file. */
		if ((fp = fopen(pidfile, "w")) == NULL) {
			syslog((int)(LOG_USER | LOG_NOTICE),
			       "Cannot create "
			       "pid file - %s",
			       pidfile);
			exit(EXIT_FAILURE);
		}
		snprintf(spid, ARRAY_SIZE(spid), "%i\n", (int)pid);
		fputs(spid, fp);
		fclose(fp);
		exit(EXIT_SUCCESS);
	}

	/* Set the file mode creation mask for this new process. */
	umask(0);

	/*
	 * Create a new session and sets this process as process group leader
	 * in new process group.
	 */
	if ((sid = setsid()) < 0) {
		syslog((int)(LOG_USER | LOG_NOTICE), "setsid error.");
		exit(EXIT_FAILURE);
	}

	/* Change the current working directory. */
	if ((chdir("/")) < 0) {
		syslog((int)(LOG_USER | LOG_NOTICE), "Error on cd /.");
		exit(EXIT_FAILURE);
	}

	/*
	 * Rather than closing the standard file descriptors we shall
	 * re-direct them to /dev/null. This is because epoll_wait silently
	 * does not work as intended if these file descriptors are re-used.
	 */
	if (freopen("/dev/null", "r", stdin) == NULL)
		syslog((int)(LOG_USER | LOG_NOTICE), "stdin freopen error.");
	if (freopen("/dev/null", "w", stdout) == NULL)
		syslog((int)(LOG_USER | LOG_NOTICE), "stdout freopen error.");
	if (freopen("/dev/null", "w", stderr) == NULL)
		syslog((int)(LOG_USER | LOG_NOTICE), "stderr freopen error.");
	syslog((int)(LOG_USER | LOG_NOTICE), "Daemon started successfully.");
}

/*
 * comm_spec struct (css) comparison function.
 * Tree should be indexed by socket file descriptor so compare the socket value.
 */
static int csscmp(const struct comm_spec *first, const struct comm_spec *last)
{
	if (first->socketfd == last->socketfd)
		return 0;
	return first->socketfd < last->socketfd ? -1 : 1;
}
