#include "dsCommandMessage_c.h"
#include "CommandSetCustomData.h"
#include <sstream>

#define MARKER_STRING 1000
#define MARKER_INT16 1060
#define MARKER_INT32 1071
#define MARKER_DOUBLE 1020
#define MARKER_POINT 1010

bool CommandSetCustomData::ExecuteNotify(void)
{
    // Get DraftSight command line interface
    dsCommandMessage_ptr commandline;
    application->GetCommandMessage( &commandline );

    // Get the active document and its selection manager.
    dsDocument_ptr doc;
    application->GetActiveDocument(&doc);
    dsSelectionManager_ptr selectionmanager;
    doc->GetSelectionManager(&selectionmanager);

    // Clear the filter, since it may have
    // been set somewhere else beforehand.
    dsInterfacePtr<dsSelectionFilter_c> filter;
    selectionmanager->GetSelectionFilter(&filter);
    filter->Clear();

    // Prompt for selection, if no entity selected yet.
    bool result;
    commandline->PromptForSelection(
        false, L"Select one or more entities", L"", &result);

    // Get the selected objects
    long count_selection_set;
    selectionmanager->GetSelectedObjectCount(
        dsSelectionSetType_Previous, &count_selection_set);
	std::wostringstream prompt;
	prompt << L"Selected objects: "
           << count_selection_set;
    commandline->PrintLine( prompt.str().c_str() );

    dsStringArray globalKeywords;
    globalKeywords.add(L"_String");
    globalKeywords.add(L"_int16");
    globalKeywords.add(L"_int32");
    globalKeywords.add(L"_Double");
    globalKeywords.add(L"_Point");
    globalKeywords.add(L"_Next");
    dsString keyword;

    dsStringArray localKeywords;
    localKeywords.add(L"String");
    localKeywords.add(L"int16");
    localKeywords.add(L"int32");
    localKeywords.add(L"Double");
    localKeywords.add(L"Point");
    localKeywords.add(L"Next");

    // Iterate over the selected objects
    for (long o = 0; o < count_selection_set; ++o)
    {
        // Retrieve object type
        dsObjectType_e type;
        dsObject *object;
        selectionmanager->GetSelectedObject(
            dsSelectionSetType_Previous, o, &type, &object);
        std::wostringstream prompt;
		prompt << L"Object type ="
               << type
			   << L":"
			   << d2s(type);
        commandline->PrintLine( prompt.str().c_str() );

        // Retrieve the object's custom data
        dsEntityHelper_ptr helper;
        application->GetEntityHelper(&helper);
        dsCustomData_ptr data;
        helper->GetCustomData(object, L"BlockCustomData", &data);

        // Keep inserting custom data to the current object
        // until keyword "NEXT" was entered.
        bool next = false;
        while (!next)
        {
            dsPromptResultType_e res;
            // Choose what data to enter
            long last_index;
            data->GetDataCount(&last_index);

            commandline->PromptForKeyword2(
                L"Add custom data to current object or advance to next object",
                L"",
                globalKeywords,
                localKeywords,
                0,
                globalKeywords[5],
                &keyword,
                &res);

            if( res == dsPromptResultType_Cancel )
                return true;

            if( res != dsPromptResultType_Keyword )
                return false;

            // Add a string
            if( keyword.CompareNoCase( dsString(L"_STRING") ) == 0 )
            {
                dsString s;
                commandline->PromptForString(
                    true, L"Specify String", L"", &s, &result);
                data->InsertStringData(last_index, MARKER_STRING, s);
            }
            // Add a 16 bit integer
            else if( keyword.CompareNoCase( dsString(L"_INT16") ) == 0 )
            {
                long i16 = 42;
                commandline->PromptForInteger(
                    L"Specify 16 bit integer", i16, &i16, &result);
                data->InsertInteger16Data(last_index, MARKER_INT16, i16);
            }
            // Add a 32 bit integer
            else if( keyword.CompareNoCase( dsString(L"_INT32") ) == 0 )
            {
                long i32 = 23;
                commandline->PromptForInteger(
                    L"Specify 32 bit integer", i32, &i32, &result);
                data->InsertInteger32Data(last_index, MARKER_INT32, i32);
            }
            // Add a double precision number
            else if( keyword.CompareNoCase( dsString(L"_DOUBLE") ) == 0 )
            {
                double d = 3.1415926535897932384626433832795;
                commandline->PromptForDouble(
                    L"Specify double precision number", d, &d, &result);
                data->InsertDoubleData(last_index, MARKER_DOUBLE, d);
            }
            // Add a point
            else if( keyword.CompareNoCase( dsString(L"_POINT") ) == 0 )
            {
                double x = 0.0, y = 0.0, z = 0.0;
                commandline->PromptForPoint(
                    L"Specify point", &x, &y, &z, &result);
                data->InsertPointData(last_index, MARKER_POINT, x, y, z);
            }
            // Go ahead to the next object
            else if( keyword.CompareNoCase( dsString(L"_NEXT") ) == 0 )
            {
                next = true;
            }

            // Add the data to the object.
            helper->SetCustomData(object, L"BlockCustomData", data);
        }
    } // iterate over all selected objects
    return true; 
}
