#ifndef DS_ADDIN_CONNECTION_H
#define DS_ADDIN_CONNECTION_H

/*
Sample Name: BlockLibraryManager

Sample Description:
    This sample is a block library manager, in which we can insert blocks
    from a given library. It will show a preview of all the blocks within
    the selected folder.

Usage:
    This example has four commands, which can be invoked using either the
    command line or by using the menu "BlockLibraryManager".

    _BLMSETLIBRARYROOTPATH "Set library root path"
        Use this command to set the root path of the block library.
        Its default location is
            %ProgramData%/Dassault Systemes/DraftSight/BlockLibraryRoot
        It is specified in a singleton class called "Options" to be found
        in "Options.h".
        
    _BLMDIALOG "Open dialog"
        This commands opens a modal dialog showing the block library. You
        can select blocks, which will be inserted into th drawing upon
        clicking OK.

    _BLMSHOW "Show block library"
        This command shows the block library in a dock widget, initially on
        the right hand side of DraftSight.

    _BLMHIDE "Hide block library"
        Use this command to hide the dock widget again.

Overview:
    Put in a few words, this add-in consists of the following parts:

    Graphical User Interface:
        The class BlockLibraryWidget (.h, .cpp, .ui) is the GUI of this
        add-in. It is used in two ways. Firstly, it appears as a local
        variable in CommandDialog::ExecuteNotify(), embedded in a modal
        dialog.
        Secondly, it appears as a member of BlockLibraryManagerApplication,
        embedded in the dock widget. The preview images are generated using
        dsApplication_c::GetPreviewBitmapBytes().

    Functionality:
        The actual job is done by the class BlockLibraryManager. Like the
        BlockLibraryWidget, this class is used twice. Firstly, as a local
        variable in CommandDialog::ExecuteNotify() and secondly as a local
        variable in BlockLibraryManagerApplication::on_insert().
        This class's task is to take the list of selected blocks from the
        BlockLibraryWidget as well as some values for scaling and rotation
        and do the insertion job, performed by BlockLibraryManager::insert().
        This method uses dsCommandMessage_c::PromptForBlockInsertion() for
        user interaction followed by dsCommandMessage_c::InsertBlock2().

    Command classes:
        All four available commands derive from CommandBase. This base
        class provides an interface for both, registering a command to be
        used from the command line via CommandBase::registerCommand() as
        well as putting a command into a menu item via CommandBase::
        createUserCommand(), so done in BlockLibraryManagerApplication::
        CreateUserInterfaceAndCommands().

    Globally available options:
        This singleton Options class holds the block library root path and
        a pointer to the dock widget, so the commands SHOW and HIDE can
        have access to it. Furthermore, there is the miscellaneous.h, a
        collection of useful functions for converting between dsAPI and Qt
        classes. These functions are used throughout the add-in.

Notable dsAPI functions used:
    dsApplication_c::GetPreviewBitmapBytes()
        Used in BlockLibraryWidget::on_tree_libraries_clicked() for
        generating preview images from a given list of drawing files.

    dsCommandMessage_c::PromptForBlockInsertion()
        Used in BlockLibraryManager::insert() prompting the user to specify
        position, scale and rotation of a block.

    dsCommandMessage_c::InsertBlock2()
        Used in BlockLibraryManager::insert() following the user prompt.
*/

#include <QObject>

#include "dsApplication_c.h"
#include "dsCustomPropertyControl_c.h"
#include "dsInterfacePtr.h"

#include "BlockLibraryManager_global.h"
#include "ui/BlockLibraryWidget.h"

// You can append new commands like this:
// 1 - Append the header file here.
// 2 - Add the command as a private member of BlockLibraryManagerApplication.
// 3 - Go to bool BlockLibraryManagerApplication::CreateUserInterfaceAndCommands() and:
//     1 - Register your command using the registerCommand()
//         (then the command can be invoked from the command line).
//     2 - If it makes sense, add it to the user menu
//         (then the command can be invoked from the user menu as well).
#include "commands/CommandSetLibraryRootPath.h"        
#include "commands/CommandDialog.h"        
#include "commands/CommandShow.h"        
#include "commands/CommandHide.h"        

class BlockLibraryManager_EXPORT BlockLibraryManagerApplication : public QObject
{
    Q_OBJECT
public:
    BlockLibraryManagerApplication(dsApplication_c *dsApp)
        : QObject(0)
        , m_dsApp(dsApp)
        , m_dockwidget(0)
        , m_block_library_widget(0)
    {}

    ~BlockLibraryManagerApplication()
    {
        if (m_dockwidget)
            m_dockwidget->Destroy();
    }

    dsApplication_c* getDsApp() { return m_dsApp; }  

    // Registers commands to the command line, to the menu and to a
    // toolbar. Furthermore it invokes CreateDockWidget().
    bool CreateUserInterfaceAndCommands();

    // Creates the dock widget containing a BlockLibraryWidget and connects
    // its insert button with the private method on_insert().
    bool CreateDockWidget();

    static const dsString appID;

private:
    dsApplication_ptr m_dsApp;

    // GUI
    dsCustomPropertyControl_c *m_dockwidget;
    BlockLibraryWidget *m_block_library_widget;

    // Commands
    CommandSetLibraryRootPath m_CommandSetLibraryRootPath;
    CommandDialog m_CommandDialog;
    CommandShow m_CommandShow;
    CommandHide m_CommandHide;

private slots:
    void on_insert();
};

#endif // DS_ADDIN_CONNECTION_H
