#ifndef COMMANDBASE_H
#define COMMANDBASE_H

#include "dsString.h"
#include "dsInterfacePtr.h"
#include "dsCommand_c.h"
#include "dsUserCommand_c.h"
#include "dsApplication_c.h"
#include "dsMenuItem_c.h"
#include "dsToolbarItem_c.h"
#include "dsToolbar_c.h"
#include "dsContextMenuItem_c.h"

// Derive commands you want to add to your add-in from this class, defining
// the abstract methods groupName(), globalName(), localName(), and
// ExecuteNotify(). This suffices if you want to invoke your command from
// the command line only.
// 
// If you want your command to be available in menus or on toolbars as
// well, you should also redefine ItemName() and Description(). If you have
// an icon available for this command, you use it redefining SmallIcon() or
// LargeIcon().
//
// This abstract base class contains only dsAPI code and is thus
// independent from this project. Hence you can use it, as is, in other
// add-in projects as well.
class CommandBase
{
public:
    CommandBase(dsApplication_c *application)
        : application(application), m_command(0) {}
    virtual ~CommandBase() { removeCommand(); }

    // The groupName is used to group a set of commands within an add-in.
    // Usually there is only one group name and that is the add-in's UUID.
    // You can define other group names, but in this case you have to take
    // care of the disposal of those commands yourself. You can remove a
    // group using dsApplication_c::RemoveUserInterface(), passing the
    // groupName as argument.
    virtual const dsString groupName() const = 0;

    // The globalName is the untranslated name of the command. Usually it
    // starts with an underscore '_'.
    // It has to be unique within a group.
    virtual const dsString globalName() const = 0;

    // The localName is the translated name of the command.
    // It has to be unique within a group.
    virtual const dsString localName() const = 0;

    // This method is the command body. It will be
    // called when this command was invoked.
    virtual bool ExecuteNotify() = 0;

    // The ItemName is the caption of this
    // command in a menu or on a toolbar.
    virtual const dsString ItemName() const { return L""; };

    // The Description appears on DraftSight's status bar.
    virtual const dsString Description() const { return L""; };

    // This command's small icon that appears in a menu or on a toolbar.
    virtual const dsString SmallIcon() const { return L""; }

    // This command's large icon that appears in a menu or on a toolbar.
    virtual const dsString LargeIcon() const { return L""; }

    // Creates the command. Then it can registered using registerCommand()
    // to make this command available on the command line.
    dsCreateCommandError_e createCommand();

    // Registers the command, so it can be used from the
    // command line by typing the command's global or local name.
    // NOTE: The command needs to be created first,
    //       so invoke createCommand() beforehand.
    void registerCommand();

    // Unregisters the command, so it cannot be invoked anymore.
    void unregisterCommand();

    // Unregisters and removes the command, but not its user command
    // representation.
    // Returns DSRESULT_True, if the command was successfully removed and
    // DSRESULT_Disconnected, if the command was removed already.
    DSRESULT removeCommand();

    // A function for convenience invoking createCommand()
    // followed by registerCommand().
    dsCreateCommandError_e createAndRegisterCommand();

    // Creates a 'user' command, i.e. a GUI command, that enables the
    // user to invoke the command from a menu or a toolbar.
    // NOTE: The command needs to be created first,
    //       so invoke createCommand() beforehand.
    dsCreateCommandError_e createUserCommand(dsUIState_e uiState = dsUIState_Document);

    // True, if the command exists.
    bool isValid() const { return m_command; }

    // Once createUserCommand() has been successfully invoked, this command
    // is available as a so-called user command. This means, it can appear
    // in a menu or on a toolbar. In that case, UserCommandID is a
    // non-empty string.
    dsString UserCommandID() const { return m_user_command_id; }

    // Inserts the command into the given menu at the given position.
    // NOTE: The command needs to be created first, so invoke both
    //       createCommand() and createUserCommand() beforehand.
    dsMenuItem_c* insert(dsMenuItem_c *menu, const int &position);

    // Inserts the command into the given toolbar at the given position.
    // NOTE: The command needs to be created first, so invoke both
    //       createCommand() and createUserCommand() beforehand.
    dsToolbarItem_c* insert(dsToolbar_c *toolbar, const int &position);

    // Inserts the command into the given context menu at the given
    // position.
    // NOTE: The command needs to be created first, so invoke both
    //       createCommand() and createUserCommand() beforehand.
    dsContextMenuItem_c* insert(
        dsContextMenuItem_c *context_menu, const int &position);
    
    DeclareCommandExecuteNotifyHook

protected:
    dsApplication_c *application;
    dsCommand_c *m_command;
    dsUserCommand_ptr m_user_command;
    dsString m_user_command_id;
};

#endif // COMMANDBASE_H
