//========================================================================//
//							SOURCE CODE LICENSE							  //
//  Copyright 2012 Dassault Systmes SolidWorks Corporation               //
//  All rights reserved.                                                  //
//  This software and related documentation are proprietary to            // 
//  Dassault Systmes SolidWorks Corporation                              //
//========================================================================//
// dsObject.h, C++ API dsDocument interface
//
#pragma once

#ifdef DS_WIN
#include <malloc.h>
#else //DS_WIN
#include <stdlib.h>
#endif //DS_WIN 

#include <memory.h>
 
template<class DT>
class dsArray
{
public:
    dsArray(); // constructor 
    dsArray(const dsArray &a); // copy constructor 
    ~dsArray(); // distructor 
    dsArray& operator = (const dsArray &a); // assignment operator 
 
    DT& operator [] (unsigned int index) const; // get array item 
    void add(const DT &item); // Add item to the end of array 
 
    unsigned int getSize() const; // get size of array (DTements)
    void setSize(unsigned int newsize); // set size of array (DTements)
    void clear(); // clear array
    bool removeAt(unsigned int pos); // delete array item 
	bool remove(const DT &item);
	void* getptr(); // get void* pointer to array data
 
    enum exception { MEMFAIL, WRONG_INDEX }; // exception enum 
 
private:
    DT *array; // pointer for array's memory 
    unsigned int size; // size of array (DTemets)
    unsigned int realsize; // actual size of allocated memory  
 
	const static int dyn_array_step = 128; // initial size of array memory (DTements)
	const static int dyn_array_mult = 2; // multiplier (enlarge array memory 
										 // dyn_array_mult times  )
};
 
//////////////////////////////////////////////////////////////////////
 
template <class DT>
inline dsArray<DT>::dsArray()
{
    realsize = dyn_array_step; // First, allocate step 
							   // for dyn_array_step items
    size = 0;
    array = (DT *)malloc(realsize*sizeof(DT));
 
    if (array == 0)
		throw MEMFAIL;
}
 
 
template <class DT>
inline dsArray<DT>::~dsArray()
{
    if (array)
    {
    	free(array); // Freeing memory 
    	array = 0;
    }
}
 
 
template <class DT>
inline dsArray<DT>::dsArray(const dsArray &a)
{
    array = (DT *)malloc(sizeof(DT)*a.realsize);
    if (array == 0)
		throw MEMFAIL;
 
    memcpy(array, a.array, sizeof(DT)*a.realsize);
	// memcpy call -- coping memory contents 
    realsize = a.realsize;
    size = a.size;
}
 
 
template <class DT>
inline dsArray<DT>& dsArray<DT>::operator = (const dsArray &a)
{
    if (this == &a) // in case somebody tries assign array to itsDTf 
		return *this;
 
	if (a.size == 0) // is other array is empty -- clear this array 
		clear();
 
	setSize(a.size); // set size 
 
    memcpy(array, a.array, sizeof(DT)*a.size);
 
    return *this;
}
 
template <class DT>
inline unsigned int dsArray<DT>::getSize() const
{
    return size; // simply return size
}
 
 
template <class DT>
inline void dsArray<DT>::setSize(unsigned int newsize)
{
    size = newsize;
 
	if (size != 0)
	{
		// change array memory size 
		// if new size is larger than current 
		// or new size is less then half of the current 
		if ((size > realsize) || (size < realsize/2))
		{
	    	realsize = size;
	    	array = (DT *)realloc(array, sizeof(DT)*size);
 
			if (array == 0)
				throw MEMFAIL;
		}
	}
	else
		clear();
}
 
template <class DT>
inline bool dsArray<DT>::removeAt(unsigned int pos)
{
	if ( pos > size-1 )
		return false;

	if (size == 1) // If array has only one element 
		clear(); // than we clear it, since it will be deleted 
	else
	{
		// otherwise, shift array DTements 
		for(unsigned int i=pos; i<size-1; i++) 
			array[i] = array[i+1];
 
		// decrease array size
		size--;
	}
	return true;
}
 
template <class DT>
inline bool dsArray<DT>::remove(const DT &item)
{
	bool found = false;
	// otherwise, shift array DTements 
	for(unsigned int i=0; i<size; i++) 
	{
		if ( item == array[i] )
			found = true;

		if ( found && i < size-1)
			array[i] = array[i+1];
	}
	// decrease array size
	if ( found )
		size--;

	return found;
}


template <class DT>
inline void dsArray<DT>::clear() // clear array memory 
{
    size = 0; 
	array = (DT *)realloc(array, sizeof(DT)*dyn_array_step); 
				  // set initial memory size again 
	realsize = dyn_array_step;
}
 
template <class DT>
inline void* dsArray<DT>::getptr()
{
    return array; // return void* pointer 
}
 
template <class DT>
inline DT& dsArray<DT>::operator [] (unsigned int index) const
{
    if( index >= getSize() )
		throw WRONG_INDEX;

	return array[index]; // return array DTement 
}
 
template <class DT>
inline void dsArray<DT>::add(const DT &item)
{
    size++;
 
    if (size > realsize)
    {
		realsize *= dyn_array_mult;
 
		array = (DT *)realloc(array, sizeof(DT)*realsize);
 
		if (array == 0)
			throw MEMFAIL;
    }
 
	array[size-1] = item;
}
 

