//SPDX-License-Identifier: LGPL-2.0-or-later
/*

   Copyright (c) 2019-2025 Cyril Hrubis <metan@ucw.cz>

 */

#include <stdlib.h>
#include <string.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <inttypes.h>
#include <pthread.h>
#include <sys/mman.h>
#include <core/gp_debug.h>
#include <core/gp_pixmap.h>
#include <backends/gp_backend.h>
#include <backends/gp_proxy_proto.h>
#include <backends/gp_proxy_conn.h>
#include <backends/gp_proxy.h>

#define UPDATE_COOKIES_MAX 16

struct proxy_priv {
	struct gp_proxy_buf buf;
	gp_pixmap dummy;

	gp_pixmap shm_pixmap;

	gp_ev_queue ev_queue;

	gp_fd fd;

	unsigned int visible:1;
	unsigned int sys_quit_requested:1;
	unsigned int unmap_requested:1;

	/* array to store udpate rect cookies */
	uint32_t update_cookies[UPDATE_COOKIES_MAX];
	unsigned int update_cookies_cnt;

	/* mapped memory backing the pixmap */
	void *map;
	size_t map_size;
};

static enum gp_backend_ret proxy_set_attr(gp_backend *self,
                                          enum gp_backend_attr attr,
                                          const void *vals)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);
	uint32_t val;

	switch (attr) {
	case GP_BACKEND_ATTR_TITLE:
//		gp_proxy_send(priv->fd.fd, GP_PROXY_NAME, vals);
	break;
	case GP_BACKEND_ATTR_CURSOR:
		val = *(enum gp_backend_cursor_req *)vals;
		gp_proxy_send(priv->fd.fd, GP_PROXY_CURSOR, &val);
	break;
	default:
	break;
	}

	return GP_BACKEND_NOTSUPP;
}

static void proxy_exit(gp_backend *self)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	gp_proxy_send(priv->fd.fd, GP_PROXY_EXIT, NULL);
}

static void map_buffer(gp_backend *self, union gp_proxy_msg *msg)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);
	void *p;
	int fd;

	GP_DEBUG(1, "Mapping buffer '%s' size %zu",
	         msg->map.map.path, msg->map.map.size);

	fd = open(msg->map.map.path, O_RDWR);
	if (!fd) {
		GP_WARN("Invalid path for map event");
		return;
	}

	size_t size = msg->map.map.size;

	p = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	close(fd);

	if (p == MAP_FAILED) {
		GP_WARN("mmap() failed :-(");
		return;
	}

	priv->map = p;
	priv->map_size = msg->map.size;

	gp_proxy_send(priv->fd.fd, GP_PROXY_MAP, NULL);
}

static void visible(gp_backend *self)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	self->pixmap = &priv->shm_pixmap;

	priv->visible = 1;

	gp_ev_queue_push_resize(self->event_queue, self->pixmap->w, self->pixmap->h, 0);
}

static void hidden(gp_backend *self)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	priv->visible = 0;
}

static void unmap_buffer(gp_backend *self)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	if (!priv->map) {
		GP_WARN("Buffer unmap requested but buffer is not mapped!");
		return;
	}

	priv->unmap_requested = 1;

	GP_DEBUG(1, "Buffer unmap requested");

	gp_ev_queue_push_resize(self->event_queue, 0, 0, 0);
}

static void init_pixmap(gp_backend *self, union gp_proxy_msg *msg)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	if (!priv->map) {
		GP_WARN("Buffer not mapped!");
		return;
	}

	priv->shm_pixmap = msg->pix.pix;
	priv->shm_pixmap.pixels = priv->map;

	GP_DEBUG(1, "Pixmap %ux%u initialized", msg->pix.pix.w, msg->pix.pix.h);

	//TODO: check that the buffer is large enough!

	gp_ev_queue_set_screen_size(self->event_queue, msg->pix.pix.w, msg->pix.pix.h);
}

/*
 * Inserts update cookie into the array of cookies.
 */
static void update_cookie_add(struct proxy_priv *priv, uint32_t cookie)
{
	if (priv->update_cookies_cnt >= UPDATE_COOKIES_MAX) {
		GP_WARN("Too many update cookies!");
		return;
	}

	GP_DEBUG(4, "Storing update cookie %"PRIu32" cookie cnt %u",
	         cookie, priv->update_cookies_cnt);

	priv->update_cookies[priv->update_cookies_cnt++] = cookie;
}

/*
 * Looks up an update cookie, returns index if found, -1 if not.
 */
static int update_cookie_get(struct proxy_priv *priv, uint32_t cookie)
{
	unsigned int i;

	for (i = 0; i < priv->update_cookies_cnt; i++) {
		if (priv->update_cookies[i] == cookie) {
			GP_DEBUG(4, "Update cookie %u found", (unsigned int)cookie);
			return i;
		}
	}

	GP_DEBUG(4, "Update cookie %u not found", (unsigned int)cookie);

	return -1;
}

/*
 * Looks up an update cookie, clears it if found.
 */
static int update_cookie_clear(struct proxy_priv *priv, uint32_t cookie)
{
	int i = update_cookie_get(priv, cookie);

	/* Fake completion events if we are hidden */
	if (!priv->visible)
		return 1;

	if (i < 0)
		return 0;

	uint32_t last_cookie = priv->update_cookies[--priv->update_cookies_cnt];
	priv->update_cookies[i] = last_cookie;

	return 1;
}

/**
 * @brief Receives data and parses them into event(s).
 *
 * Reads data from the client socket and parses them into event(s).
 *
 * @param self A fd.
 * @block If set the call blocks until data are received.
 * @return A number of bytes read or -1 on a failure.
 */
static int proxy_recv_events(gp_fd *self, int block)
{
	gp_backend *backend = self->priv;
	struct proxy_priv *priv = GP_BACKEND_PRIV(backend);
	union gp_proxy_msg *msg;
	int ret;

	ret = gp_proxy_buf_recv(priv->fd.fd, &priv->buf, block);

	if (ret == 0) {
		if (priv->sys_quit_requested) {
			GP_FATAL("Application still alive after GP_EV_SYS_QUIT event!");
			gp_backend_exit(backend);
			exit(1);
		}
		GP_WARN("Connection closed");
		gp_ev_queue_push(backend->event_queue, GP_EV_SYS, GP_EV_SYS_QUIT, 0, 0);
		priv->sys_quit_requested = 1;

		return 0;
	}

	if (ret < 0)
		return ret;

	while (gp_proxy_next(&priv->buf, &msg)) {
		switch (msg->type) {
		case GP_PROXY_CLI_INIT:
			GP_DEBUG(4, "Got GP_PROXY_CLI_INIT");
			priv->dummy.pixel_type = msg->cli_init.cli_init.pixel_type;
			backend->dpi = msg->cli_init.cli_init.dpi;
		break;
		case GP_PROXY_EVENT:
			GP_DEBUG(4, "Got GP_PROXY_EVENT");
			gp_ev_queue_put(backend->event_queue, &msg->ev.ev);
		break;
		case GP_PROXY_MAP:
			GP_DEBUG(4, "Got GP_PROXY_MAP");
			map_buffer(backend, msg);
		break;
		case GP_PROXY_UNMAP:
			GP_DEBUG(4, "Got GP_PROXY_UNMAP");
			unmap_buffer(backend);
		break;
		case GP_PROXY_PIXMAP:
			GP_DEBUG(4, "Got GP_PROXY_PIXMAP");
			init_pixmap(backend, msg);
		break;
		case GP_PROXY_SHOW:
			GP_DEBUG(4, "Got GP_PROXY_SHOW");
			visible(backend);
		break;
		case GP_PROXY_HIDE:
			GP_DEBUG(4, "Got GP_PROXY_HIDE");
			hidden(backend);
		break;
		case GP_PROXY_CURSOR_POS:
			GP_DEBUG(4, "Got GP_PROXY_CURSOR_POS");
			gp_ev_queue_set_cursor_pos(backend->event_queue,
			                           msg->cursor_pos.pos.x,
			                           msg->cursor_pos.pos.y);
		break;
		case GP_PROXY_EXIT:
			GP_DEBUG(4, "Got GP_PROXY_EXIT");
			gp_ev_queue_push(backend->event_queue, GP_EV_SYS,
			                 GP_EV_SYS_QUIT, 0, 0);
			priv->sys_quit_requested = 1;
		break;
		case GP_PROXY_UPDATE:
			GP_DEBUG(4, "Got GP_PROXY_UPDATE %i %i %i %i cookie = %u",
				 msg->rect.rect.x, msg->rect.rect.y, msg->rect.rect.w, msg->rect.rect.h,
				 msg->rect.rect.cookie);
			update_cookie_add(priv, msg->rect.rect.cookie);
		break;
		}
	}

	return ret;
}

static int proxy_recv_events_mt(gp_backend *self, int block, int check_cookie, uint32_t cookie)
{
	static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

	struct proxy_priv *priv = GP_BACKEND_PRIV(self);
	int ret = 0;

	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
	pthread_mutex_lock(&mutex);
	if (!check_cookie || update_cookie_get(priv, cookie) < 0)
		ret = proxy_recv_events(&priv->fd, block);
	pthread_mutex_unlock(&mutex);
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

	return ret;
}

static enum gp_poll_event_ret proxy_process_fd(gp_fd *self)
{
	gp_backend *backend = self->priv;

	while (proxy_recv_events_mt(backend, 0, 0, 0) > 0);

	return 0;
}

static void proxy_update_rect(gp_backend *self, gp_coord x0, gp_coord y0,
                             gp_coord x1, gp_coord y1)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);
	static uint32_t cookie;

	if (!priv->visible) {
		GP_DEBUG(4, "Not visible!");
		return;
	}

	struct gp_proxy_rect rect = {
		.x = x0,
		.y = y0,
		.w = x1 - x0 + 1,
		.h = y1 - y0 + 1,
		.cookie = ++cookie,
	};

	GP_DEBUG(4, "Sending GP_PROXY_UPDATE cookie %" PRIu32, cookie);

	gp_proxy_send(priv->fd.fd, GP_PROXY_UPDATE, &rect);

	while (!update_cookie_clear(priv, cookie))
		proxy_recv_events_mt(self, 1, 1, cookie);
}

static void proxy_flip(gp_backend *self)
{
	proxy_update_rect(self, 0, 0, self->pixmap->w-1, self->pixmap->h-1);
}

static int proxy_resize_ack(gp_backend *self)
{
	struct proxy_priv *priv = GP_BACKEND_PRIV(self);

	if (priv->unmap_requested) {
		munmap(priv->map, priv->map_size);

		priv->map = NULL;
		priv->map_size = 0;

		self->pixmap = &priv->dummy;

		gp_proxy_send(priv->fd.fd, GP_PROXY_UNMAP, NULL);

		priv->unmap_requested = 0;

		GP_DEBUG(1, "Buffer unmap completed");
	}

	return 0;
}

gp_backend *gp_proxy_init(const char *path, const char *title)
{
	int fd;
	gp_backend *ret;
	size_t size = sizeof(gp_backend) + sizeof(struct proxy_priv);

	ret = malloc(size);
	if (!ret) {
		GP_WARN("Malloc failed :-(");
		return NULL;
	}

	memset(ret, 0, size);

	if (!path)
		path = getenv("GP_PROXY_PATH");

	fd = gp_proxy_client_connect(NULL);
	if (fd < 0) {
		free(ret);
		return NULL;
	}

	struct proxy_priv *priv = GP_BACKEND_PRIV(ret);

	priv->fd = (gp_fd) {
		.fd = fd,
		.event = proxy_process_fd,
		.events = GP_POLLIN,
		.priv = ret,
	};

	if (gp_poll_add(&ret->fds, &priv->fd)) {
		close(fd);
		free(ret);
		return NULL;
	}

	ret->name = "proxy";
	ret->set_attr = proxy_set_attr;
	ret->exit = proxy_exit;
	ret->update_rect = proxy_update_rect;
	ret->flip = proxy_flip;
	ret->resize_ack = proxy_resize_ack;

	priv->map = NULL;
	priv->map_size = 0;
	priv->visible = 0;
	priv->sys_quit_requested = 0;
	priv->unmap_requested = 0;

	gp_proxy_buf_init(&priv->buf);

	ret->event_queue = &priv->ev_queue;

	gp_ev_queue_init(ret->event_queue, 1, 1, 0, NULL, NULL, 0);

	ret->pixmap = &priv->dummy;
	ret->pixmap->pixel_type = 0;

	gp_proxy_send(fd, GP_PROXY_NAME, title);

	/* Wait for the pixel type */
	while (!priv->dummy.pixel_type)
		gp_poll_wait(&ret->fds, -1);

	gp_pixmap_init(&priv->dummy, 0, 0, priv->dummy.pixel_type, NULL, 0);

	return ret;
}
