//SPDX-License-Identifier: LGPL-2.0-or-later

/*

   Copyright (c) 2014-2024 Cyril Hrubis <metan@ucw.cz>

 */

/**
 * @file gp_widgets_color_scheme.h
 * @brief Colors and color scheme.
 *
 * Color theme provies a mapping from an abstract color names such as text
 * color or background color to indexes into a global color table
 * #gp_widget_render_ctx::colors.
 *
 * JSON Theme colors
 * -----------------
 *
 * | Color name  | Color definition                 |
 * |-------------|----------------------------------|
 * | "text"      | Text color                       |
 * | "fg"        | Foreground color                 |
 * | "bg         | Background color                 |
 * | "highlight" | Highlighted background           |
 * | "alert"     | Alert (error) color, usually red |
 * | "warn"      | Warning color, usually yellow    |
 * | "accept"    | Accept color, usually green      |
 *
 * JSON 16 colors names
 * --------------------
 *
 * | Color names    |                  |                |                 |
 * |----------------|------------------|----------------|-----------------|
 * | "black"        | "red"            | "green"        | "yellow"        |
 * | "blue"         | "magenta"        | "cyan"         | "gray"          |
 * | "bright black" | "bright red"     | "bright green" | "bright yellow" |
 * | "bright blue"  | "bright magenta" | "bright cyan"  | "white"         |
 *
 * Label colors example
 * --------------------
 *
 * @image html label_colors.png
 * @include{json} demos/widgets/test_layouts/label_colors.json
 */

#ifndef GP_WIDGETS_COLOR_SCHEME_H
#define GP_WIDGETS_COLOR_SCHEME_H

#include <widgets/gp_widget_types.h>

/**
 * @brief A color scheme.
 *
 * Decides if the pixmaps generated by application should have dark or light
 * background. There is also default light and dark scheme compiled in the
 * library.
 */
enum gp_widgets_color_scheme {
	/** @brief Default color scheme. */
	GP_WIDGET_COLOR_SCHEME_DEFAULT,
	/** @brief Light color scheme. */
	GP_WIDGET_COLOR_SCHEME_LIGHT,
	/** @brief Dark color scheme. */
	GP_WIDGET_COLOR_SCHEME_DARK,
};

/**
 * @brief Widget theme colors.
 *
 * The first 10 colors are theme colors, such as default text color, foreground
 * and background color and the next are standard 16 colors.
 */
enum gp_widgets_color {
	/**
	 * @brief Default text color.
	 *
	 * Text color is also used for widget frames.
	 */
	GP_WIDGETS_COL_TEXT = 0,
	/**
	 * @brief Default foreground color.
	 *
	 * Foreground is usually used to paint inner parts of the widgets.
	 */
	GP_WIDGETS_COL_FG = 1,
	/**
	 * @brief Default background color.
	 *
	 * Background is used to fill in areas between widgets.
	 */
	GP_WIDGETS_COL_BG = 2,
	/**
	 * @brief Default highlighted color.
	 */
	GP_WIDGETS_COL_HIGHLIGHT = 3,
	/**
	 * @brief Default selection color.
	 *
	 * Selected text, active widget, etc.
	 */
	GP_WIDGETS_COL_SELECT = 4,
	/**
	 * @brief Default alert color.
	 *
	 * Used to highlight errors, usually red.
	 */
	GP_WIDGETS_COL_ALERT = 5,
	/**
	 * @brief Default warning color.
	 *
	 * Used to highlight warnings, usually yelow.
	 */
	GP_WIDGETS_COL_WARN = 6,
	/**
	 * @brief Default accept color.
	 *
	 * Used to highlight accepted input, agreement etc., usually green.
	 */
	GP_WIDGETS_COL_ACCEPT = 7,
	/**
	 * @brief Default fill color.
	 *
	 * This is used to fill in background when application is smaller than backend.
	 */
	GP_WIDGETS_COL_FILL = 8,
	/**
	 * @brief Default disabled widget color.
	 *
	 * This color replaces text color when widget is disabled.
	 */
	GP_WIDGETS_COL_DISABLED = 9,
	/** @brief The number of theme colors. */
	GP_WIDGETS_THEME_COLORS =  10,

	/** @brief Black in standard 16 colors. */
	GP_WIDGETS_COL_BLACK = 10,
	/** @brief Red in standard 16 colors. */
	GP_WIDGETS_COL_RED = 11,
	/** @brief Green in standard 16 colors. */
	GP_WIDGETS_COL_GREEN = 12,
	/** @brief Yellow in standard 16 colors. */
	GP_WIDGETS_COL_YELLOW = 13,
	/** @brief Blue in standard 16 colors. */
	GP_WIDGETS_COL_BLUE = 14,
	/** @brief Magenta in standard 16 colors. */
	GP_WIDGETS_COL_MAGENTA = 15,
	/** @brief Cyan in standard 16 colors. */
	GP_WIDGETS_COL_CYAN = 16,
	/** @brief Gray in standard 16 colors. */
	GP_WIDGETS_COL_GRAY = 17,
	/** @brief Bright black in standard 16 colors. */
	GP_WIDGETS_COL_BR_BLACK = 18,
	/** @brief Bright red in standard 16 colors. */
	GP_WIDGETS_COL_BR_RED = 19,
	/** @brief Bright green in standard 16 colors. */
	GP_WIDGETS_COL_BR_GREEN = 20,
	/** @brief Bright yellow in standard 16 colors. */
	GP_WIDGETS_COL_BR_YELLOW = 21,
	/** @brief Bright blue in standard 16 colors. */
	GP_WIDGETS_COL_BR_BLUE = 22,
	/** @brief Bright magenta in standard 16 colors. */
	GP_WIDGETS_COL_BR_MAGENTA = 23,
	/** @brief Bright cyan in standard 16 colors. */
	GP_WIDGETS_COL_BR_CYAN = 24,
	/** @brief White in standard 16 colors. */
	GP_WIDGETS_COL_WHITE = 25,
	/** @brief The number of widget colors. */
	GP_WIDGETS_COL_CNT,
};

/**
 * @brief Converts a color name into a color index.
 *
 * @param name a color name
 *
 * @return An enum gp_widgets_color or -1 on a failure.
 */
enum gp_widgets_color gp_widgets_color_name_idx(const char *name);

/**
 * @brief Returns a pixel given an color index.
 *
 * @param ctx A widget rendering context.
 * @param color A color index.
 *
 * @return A pixel value.
 */
gp_pixel gp_widgets_color(const gp_widget_render_ctx *ctx, enum gp_widgets_color color);

/**
 * @brief Toggles current color scheme.
 *
 * Reloads colors, redraws application.
 */
void gp_widgets_color_scheme_toggle(void);

/**
 * @brief Sets a color scheme.
 *
 * Reloads colors, redraws application.
 *
 * @param color_scheme New color scheme.
 */
void gp_widgets_color_scheme_set(enum gp_widgets_color_scheme color_scheme);

/**
 * @brief Returns current color scheme.
 *
 * @return Current color scheme.
 */
enum gp_widgets_color_scheme gp_widgets_color_scheme_get(void);

/**
 * @brief Creates a color scheme switch widget.
 *
 * @return TODO!
 */
gp_widget *gp_widget_color_scheme_switch(void);

#endif /* GP_WIDGETS_COLOR_SCHEME_H */
