#ifdef TEST_MEMORY_USAGE
long                 LINETYPES[6];
#endif
/*
 *  L3Input.cpp, part of the L3 project for handling LDraw *.dat files
 *  Copyright (C) 1997-1999  Lars C. Hassing (lch@ccieurope.com)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
                                                                             */

/*****************************************************************************
  Please do not edit this file. In stead contact me (lch@ccieurope.com)
  to get your changes integrated and you will receive an up-to-date version.
******************************************************************************/

/* L3Input.cpp  Various input functions for L3 program */
/*
970918 lch First version, both for TurboC 2.0 and Visual C++ 5.0
980926 lch Version released for ldlite
981004 lch File search also in directory of model
981005 lch Support for MPD files
981111 lch Skip collinear lines if color is not 16
981218 lch Better message for moved parts: tell who referenced it
981226 lch Added BBoxES: BBox exclusive studs
990113 lch Skip header before first "0 FILE xx.dat" of MPD files
990114 lch Also look for files with .MPD extension
990212 lch Added support for TRANSLATE, ROTATE, SCALE, TRANSFORM meta commands
990221 lch Treat any header before first valid command as comments
990324 lch Default Part Color added
990408 lch L3P will now look in ldraw.ini if LDRAWDIR is not defined
990416 lch FilteType DAT0, don't complain about comments before "0 FILE"
990419 lch Prepare for L3P32, changing #ifdef __TURBOC__ to #ifdef L3P
990424 lch #if (STUDS) added
990603 lch ModelTitle and ModelAuthor added
990803 lch Use getenv(windir) for ldraw.ini, and remove trailing backslash
991020 lch Crash in ErrorInInput if Filename+InputStr exceeded 200 characters
991130 lch Added LineNo to L3LineS if not TurboC
000317 lch Recursion check added
                                                                             */
/*
12345678901234567890123456789012345678901234567890123456789012345678901234567890
problem with bbox for mpd files...
                                                                             */
#ifdef USE_OPENGL
#include "math.h"
//extern "C" {
#endif

#ifdef USE_OPENGL
// Gotta prepare for case sensitive file systems.
#include "StdAfx.h"
#else
#include "stdafx.h"
#endif

#ifndef L3P
#ifndef USE_OPENGL
#include "resource.h"                     /* main symbols                    */
#include "LogDialog.h"
#endif
#endif

#include "stdio.h"
#include "stdlib.h"
#include "L3Def.h"
#include "math.h"
#include "string.h"

#ifdef USE_OPENGL
#include "platform.h"
#include "LDrawIni.h"
extern LDrawIniBoolT platform_fixcase(char *);

extern char pathname[256];
extern char primitivepath[256];
extern char partspath[256];
extern char modelspath[256];
extern char datfilepath[256];
extern int use_uppercase;

extern char ldconfig[256];
extern char ldconfigfilename[256];

extern int ldlite_parse_fade_meta(char *s);
extern int ldlite_parse_silhouette_meta(char *s);
#endif

#ifdef LOGFP
static int           nLevel = 0;
#endif
#ifndef L3P
#ifndef USE_OPENGL
CLogDialog           LogDia("Warnings and errors during input");
struct L3StatS       L3Stat;
#endif
#endif

int                  DoCheck = 0;
#ifdef L3P
int                  ReadLineTypeMask = BIT(0) + BIT(1) + BIT(3) + BIT(4);
#else
int                  ReadLineTypeMask = BIT(0) + BIT(1) + BIT(2) + BIT(3)
+ BIT(4) + BIT(5);
#endif

#ifndef L3P
char                 ModelTitle[400];     /* Title of model file             */
char                 ModelAuthor[400];    /* Author of model file            */
#endif
char                 LDrawDir[_MAX_PATH]; /* Directory of P, PARTS, MODELS   */
char                 ModelDir[_MAX_PATH]; /* Directory of model              */
struct L3ColorS      Colors[MAX_COLORS];
int                  nColors;
struct L3LightS     *Lights;
int                  nLights;
int                  nLightsAlloc;
struct L3PartS       Parts[MAX_PARTS];
int                  nParts;              /* Number of parts in Parts[]      */
int                  WarningLevel;
float                DetThreshold = 0.0;
float                DistThreshold = 0.0;
int                  ShowAllDists = 0;
int                  LightDotDat;         /* 1: use light.dat, 0: ignore     */
/* TurboC has a limited stack size (default _stklen=4096)
   Although I set it to e.g. 16000, it is best to keep automatic variables at a minimum.
   Therefore these variables here.
   Be careful that they are used in limited code to avoid side-effects.
                                                                             */
static struct InputInfo
{
   char                 InputStr[1000];
   char                *Filename;
   int                  LineNo;
}                    IInfo;
static char          Path[_MAX_PATH];
static char          SubPartDatName[_MAX_PATH];
static char          ErrStr[400];
#ifdef USE_OPENGL
char                *Dirs[] = {"\\P\\", "\\Parts\\", "\\Models\\", 
			       "\\Unofficial\\P\\", "\\Unofficial\\Parts\\",
                               "\\Unofficial\\Lsynth\\"};

struct LDrawIniS *LDrawIni;
int LDrawIniErrorCode;

void GetLDrawSearchDirs(int *ErrorCode)
{
  int i;
  UNUSED(ErrorCode);

  //LDrawIniSetFileCaseCallback(&platform_fixcase);

  LDrawIni = LDrawIniGet(NULL, NULL, &LDrawIniErrorCode);
  if (!LDrawIni)
  {
    if (LDrawIniErrorCode == LDRAWINI_ERROR_LDRAWDIR_NOT_SET)
    {
      /* Neither environment variable, nor ldraw.ini, simply try current dir */
      if (DirHasPandPARTS("."))
	LDrawIni = LDrawIniGet(".", NULL, &LDrawIniErrorCode);
    }
  }
  if (!LDrawIni)
  {
    printf("\
Environment variable LDRAWDIR must be set to a directory with p,parts subdirs.\n"
#ifdef _WIN32
"e.g.  'set LDRAWDIR=c:\\LDraw'       (don't use long names)\n\
You may type the set command at the DOS prompt or put it into C:\\AUTOEXEC.BAT\n"
#else
#endif
);
    exit(1);
  }
  else {
    extern int LDSearchDirsGet(struct LDrawIniS * LDrawIni);
    
    i = LDSearchDirsGet(LDrawIni);
    if (!i){
      printf("Out of memory adding search dirs.\n");
      exit(1);
    }
  }

  strcpy(pathname, LDrawIni->LDrawDir); // Needed for old ldglite code.
  
  // That NULL at the end is the modelpath.  Do we have it yet?
  // If not, then we may need to redo this when we do have it.
  i = LDrawIniComputeRealDirs(LDrawIni, 1, 0, NULL);
  if (!i){
    printf("Failed to compute search dirs.\n");
    exit(1);
  }
}

#else
char                *Dirs[] = {"\\P\\", "\\Parts\\", "\\Models\\"};
#endif
struct L3PovS        L3Pov = {
   {                                      /* Camera position                 */
      30.0,                               /* Latitude                        */
      45.0,                               /* Longitude                       */
      0.0,                                /* Radius                          */
      0.0
   },
   {0.0, 0.0, 0.0, 0.0},                  /* BackgroundColor                 */
   7,                                     /* DefaultPartColorNumber          */
   {0.0, 0.0, 0.0, 0.0},                  /* DefaultPartColor                */
   0,                                     /* CameraAngle, 0=default to 67.38 */
   0.5,                                   /* SeamWidth                       */
   0,                                     /* FloorY                          */
   0,                                     /* FloorType                       */
   2,                                     /* Quality                         */
   0,                                     /* Debug                           */
   1,                                     /* Globe format for camera
                                             position                        */
   0,                                     /* FloorYspecified                 */
   1,                                     /* UsePovParts                     */
   0,                                     /* UseDefaultLights                */
   0                                      /* Bumps                           */
};


static void          PromptUser(char *Str)
{
#ifdef L3P
   printf("%s\n", Str);
#else
#ifdef USE_OPENGL
   printf("%s\n", Str);
#else
   AfxMessageBox(Str, MB_OK | MB_APPLMODAL | MB_ICONERROR, 0);
#endif
#endif
}

/* Severity: 0=error, 1=warning, 2=det, 3=pedantic */
#define II_WARNING  1
#define II_SKIPPING 2
static void          ErrorInInput(int Severity, int UseII, char *Str)
{
   char                 TmpStr[2000];     /* Must be at least size of
                                             IInfo.InputStr+Filename         */

   if (Severity > WarningLevel)
      return;
   if (UseII)
   {
      sprintf(TmpStr, "%s \"%s\" Line %d: %s: %s",
              (UseII == II_WARNING ? "WARNING" : "SKIPPING"),
              IInfo.Filename, IInfo.LineNo, Str, IInfo.InputStr);
      Str = TmpStr;
   }
#ifdef L3P
   printf("%s\n", Str);
#else
#ifdef USE_OPENGL
   printf("%s\n", Str);
#else
   extern void          Cprintf(char *Format,...);
   /* Cprintf("%s\n",Str); */
   /* AfxMessageBox(Str, MB_OK|MB_APPLMODAL|MB_ICONERROR,0); */
   LogDia.m_LogStr += Str;
   LogDia.m_LogStr += "\r\n";
#endif
#endif
}


/* Strdup, not strdup, to be able to use farmalloc in TurboC */
char                *Strdup(const char *Str)
{
   char                *Copy;

   Copy = (char *) malloc(strlen(Str) + 1);  /* Maybe farmalloc              */
   if (Copy)
      strcpy(Copy, Str);
   return (Copy);
}

#ifndef USE_OPENGL
static 
#endif
void          FreePart(struct L3PartS * PartPtr);

static void          FreeLines(struct L3PartS * PartPtr)
{
   register struct L3LineS *LinePtr;
   struct L3LineS      *NextLine;

   LinePtr = PartPtr->FirstLine;
   PartPtr->FirstLine = NULL;
   while (LinePtr)
   {
      if (LinePtr->Comment && LinePtr->Comment != (char *) LinePtr->v)
         free(LinePtr->Comment);
      if (LinePtr->LineType == 1 && LinePtr->PartPtr->Internal)
         FreePart(LinePtr->PartPtr);
      NextLine = LinePtr->NextLine;
      free(LinePtr);
      LinePtr = NextLine;
   }
}

#ifndef USE_OPENGL
static 
#endif
void          FreePart(struct L3PartS * PartPtr)
{
   free(PartPtr->DatName);
   FreeLines(PartPtr);
   if (PartPtr->Internal)
      free(PartPtr);
   else
      memset(PartPtr, 0, sizeof(struct L3PartS));  /* Clear all flags        */
}

#ifndef L3P
void                 FreeParts(void)
{
   register int         i;
   for (i = 0; i < nParts; i++)
      FreePart(&Parts[i]);
   nParts = 0;
}
#endif

struct L3LightS     *AddLight(void)
{
   if (nLights == nLightsAlloc)
   {
      nLightsAlloc += 10;
      Lights = (struct L3LightS *)
         realloc(Lights, nLightsAlloc * sizeof(struct L3LightS));
      if (!Lights)
      {
         PromptUser("Out of memory for more lights");
         /* If this small malloc fails, we might as well exit! */
         exit(1);
      }
   }
   return (&Lights[nLights++]);
}

#ifndef L3P
void                 FreeLights(void)
{
   if (nLights)
   {
      free(Lights);
      nLights = nLightsAlloc = 0;
   }
}
#endif


static char         *StudPrimitives[] = {
   "stud5.dat",
   "stu2.dat",
   "stu22.dat",
   "stu22a.dat",
   "stu23.dat",
   "stu23a.dat",
   "stu24.dat",
   "stu24a.dat",
   "stud2.dat",
   "stud2a.dat",
   "stud3a.dat",
   "stud4a.dat",
   "studline.dat",
   "stu2p01.dat",
   "studp01.dat",
   "studel.dat",
   "stud.dat",
   "stud3.dat",
   "stud4.dat",
};



/* If this code works, it was written by Lars C. Hassing. */
/* If not, I don't know who wrote it.                     */

/* Like fgets, except that 1) any line ending is accepted (\n (unix),
\r\n (DOS/Windows), \r (Mac (OS9)) and 2) Str is ALWAYS zero terminated
(even if no line ending was found) */
char *L3fgets(char *Str, int n, FILE *fp)
{
   register int   c;
   int            nextc;
   register char *s = Str;

   while (--n > 0)
   {
      if ((c = getc(fp)) == EOF)
         break;
      if (c == '\032')
         continue;              /* Skip CTRL+Z                               */
      if (c == '\r' || c == '\n')
      {
         *s++ = '\n';
         /* We got CR or LF, eat next character if LF or CR respectively */
         if ((nextc = getc(fp)) == EOF)
            break;
         if (nextc == c || (nextc != '\r' && nextc != '\n'))
            ungetc(nextc, fp);  /* CR-CR or LF-LF or ordinary character      */
         break;
      }
      *s++ = c;
   }
   *s = 0;

   /* if (ferror(fp)) return NULL; if (s == Str) return NULL; */
   if (s == Str)
      return NULL;

   return Str;
}

#ifndef USE_OPENGL
static 
#else
void StripQuotes(char *s, char *SubPartDatName)
{
  char *sub;

  if (SubPartDatName[0] == '\"')
  {
    if (s = strchr(s, '\"'))
    {
      strcpy(SubPartDatName, s+1); /* Strip leading quotes */
      if (s = strchr(SubPartDatName, '\"'))
	*s = 0; /* Strip trailing quotes */
    }
  }
  // NOTE:  This is kinda lame.
  else if ((sub = strstr(s, SubPartDatName)) == strstr(s, SubPartDatName))
  {
    strcpy(SubPartDatName, sub); 
  }
}
#endif

/* Modifies the string: lower case and OS-correct slashes */
void          FixDatName(register char *DatName)
{
   register int         i;
   UNUSED(DatName);
   UNUSED(i);

#ifdef USE_OPENGL
   platform_fixcase(SubPartDatName); // Should NOT do this for model files.
   localize_path(SubPartDatName);
#else
   _strlwr(DatName);
   for (i = strlen(DatName); --i >= 0;)
      if (DatName[i] == '/')
         DatName[i] = '\\';
#endif
}

static void          TrimRight(char *Str)
{
   register char       *s;

   /* 971114/lch. Win95: If a file ends with a line not terminated by CR/LF
      fgets will not append a newline. So Line[Len-1] may not be a newline but
      a valid character.                                                     */
   /* TurboC huge pointers for pointer comparison? */
   s = Str + strlen(Str) - 1;
   while (s >= Str && (*s == '\n' || *s == '\r' || *s == '\t' || *s == ' '))
      *s-- = '\0';                        /* Clear newline and trailing tabs
                                             and spaces                      */
}

void                 DeleteTrailingBackslash(char *Str)
{
   register char       *s;

   s = Str + strlen(Str) - 1;
   while (s >= Str && *s == '\\')
      *s-- = '\0';
}

static char         *FirstNonBlank(char *s)
{
   while (*s == '\t' || *s == ' ')
      s++;
   return (s);
}

#ifndef USE_OPENGL
static 
#endif
struct L3PartS *FindPart(int Internal, char *DatName)
{
   register int         i;
   struct L3PartS      *PartPtr;

   if (strlen(DatName) >= _MAX_PATH)
   {
      sprintf(ErrStr, "Name too long '%s'.", DatName);
      PromptUser(ErrStr);
      return (NULL);
   }
   if (Internal)
   {
      PartPtr = (struct L3PartS *) malloc(sizeof(struct L3PartInternalS));
      if (!PartPtr)
      {
         PromptUser("Out of memory for new internal part");
         return (NULL);
      }
   }
   else
   {
      /* First check if we have already registered it. */
      for (i = 0; i < nParts; i++)
      {
         if (strcmp(Parts[i].DatName, DatName) == 0)
            return (&Parts[i]);
      }
      /* No, allocate new part. */
      if (nParts >= MAX_PARTS)
      {
         sprintf(ErrStr, "Sorry, max %d parts.", MAX_PARTS);
         PromptUser(ErrStr);
         return (NULL);
      }
      PartPtr = &Parts[nParts++];
   }
   memset(PartPtr, 0, sizeof(struct L3PartS));  /* Clear all flags           */
   PartPtr->Internal = Internal;
   PartPtr->DatName = Strdup(DatName);
   if (!PartPtr->DatName)
   {
      PromptUser("Out of memory for new part name");
      free(PartPtr);
      return (NULL);
   }
   for (i = sizeof(StudPrimitives) / sizeof(StudPrimitives[0]); --i >= 0;)
   {
#ifdef USE_OPENGL
      if (stricmp(DatName, StudPrimitives[i]) == 0)
#else
      if (strcmp(DatName, StudPrimitives[i]) == 0)
#endif
      {
         PartPtr->IsStud = 1;
         break;
      }
   }
   return (PartPtr);
}                                         /* FindPart                        */

/* Return 0 if OK */
int                  SaveLine(struct L3LineS *** LinePtrPtrPtr,
                              struct L3LineS * Data, char *Comment)
{
   register struct L3LineS *LinePtr;
   int                  Len;

   LinePtr = (struct L3LineS *) malloc(sizeof(struct L3LineS));
   if (!LinePtr)
      return (1);
   *LinePtr = *Data;                      /* Copy data                       */
   if (LinePtr->LineType == 0)
   {
      /* Save comment */
      Len = strlen(Comment) + 1;
      if (Len > sizeof(LinePtr->v))
      {
         LinePtr->Comment = Strdup(Comment);
         if (!LinePtr->Comment)
         {
            free(LinePtr);
            return (1);
         }
      }
      else
      {
         /* Reuse the 64 bytes of float v[4][4] */
         LinePtr->Comment = (char *) LinePtr->v;
         strcpy(LinePtr->Comment, Comment);
      }
   }

   **LinePtrPtrPtr = LinePtr;
   *LinePtrPtrPtr = &(LinePtr->NextLine);
   return (0);
}

static int           FileIsFromP;
static int           FileIsFromPARTS;
static FILE         *OpenDatFile2(char *DatName, char *Extension)
{
   FILE                *fp;
   register int         i;

   strcpy(Path, DatName);
   strcat(Path, Extension);
   FileIsFromP = 0;
   FileIsFromPARTS = 0;
   fp = fopen(Path, "rt");
   if (!fp)
   {
      /* If not in current directory then look in P, PARTS and MODELS */
#if 1     
     for (i = 0; i < LDrawIni->nSearchDirs; i++)
     {
       concat_path(LDrawIni->SearchDirs[i].Dir, DatName, Path);
       strcat(Path, Extension);
       FileIsFromP = (LDrawIni->SearchDirs[i].Flags & LDSDF_DEFPRIM) ? 1 : 0;
       FileIsFromPARTS = (LDrawIni->SearchDirs[i].Flags & LDSDF_DEFPART) ? 1 : 0;
       fp = fopen(Path, "rt");
       if (fp)
         return fp;
     }
#else
      for (i = 0; i < sizeof(Dirs) / sizeof(char *); i++)
      {
#ifdef USE_OPENGL
	 if (stricmp(Dirs[i], "\\P\\") == 0)
	   concat_path(primitivepath, DatName, Path);
	 if (stricmp(Dirs[i], "\\Parts\\") == 0)
	   concat_path(partspath, DatName, Path);
	 if (stricmp(Dirs[i], "\\Models\\") == 0)
	   concat_path(modelspath, DatName, Path);
	 if (stricmp(Dirs[i], "\\Unofficial\\P\\") == 0)
	 {
	   char PrePath[_MAX_PATH];
	   concat_path(LDrawDir, Dirs[i], PrePath);
	   concat_path(PrePath, DatName, Path);
	 }
	 if (stricmp(Dirs[i], "\\Unofficial\\Parts\\") == 0)
	 {
	   char PrePath[_MAX_PATH];
	   concat_path(LDrawDir, Dirs[i], PrePath);
	   concat_path(PrePath, DatName, Path);
	 }
	 if (stricmp(Dirs[i], "\\Unofficial\\LSynth\\") == 0)
	 {
	   char PrePath[_MAX_PATH];
	   concat_path(LDrawDir, Dirs[i], PrePath);
	   concat_path(PrePath, DatName, Path);
	 }
#else	
         strcpy(Path, LDrawDir);
         strcat(Path, Dirs[i]);
         strcat(Path, DatName);
#endif
         strcat(Path, Extension);
         FileIsFromP = (i == 0);
         FileIsFromPARTS = i & 1;
         fp = fopen(Path, "rt");
         if (fp)
            break;
      }
      if (!fp && ModelDir[0])
      {
         /* If still not found, try directory of the model itself */
#ifdef USE_OPENGL
	 concat_path(datfilepath, DatName, Path);
#else
         strcpy(Path, ModelDir);
         strcat(Path, DatName);
#endif
         strcat(Path, Extension);
         fp = fopen(Path, "rt");
      }
#endif
      /* If still not found, try LDConfig.ldr */
      if (!fp && (stricmp(DatName, ldconfigfilename) == 0))
      {
         strcpy(Path, ldconfig);
         fp = fopen(Path, "rt");
         if (!fp)
           printf("Could not open LDConfig (%s)\n", ldconfig);
      }
   }
   return (fp);
}

FILE                *OpenDatFile(char *DatName)
{
   FILE                *fp;

   fp = OpenDatFile2(DatName, "");
#ifdef USE_OPENGL
   if (use_uppercase)
   {
     if (!fp)
      fp = OpenDatFile2(DatName, ".DAT");
     if (!fp)
      fp = OpenDatFile2(DatName, ".MPD");
   }
   else
   {
     if (!fp)
      fp = OpenDatFile2(DatName, ".dat");
     if (!fp)
      fp = OpenDatFile2(DatName, ".mpd");
   }
#else
   if (!fp)
      fp = OpenDatFile2(DatName, ".dat");
   if (!fp)
      fp = OpenDatFile2(DatName, ".mpd");
#endif
   return (fp);
}
char                *GetOpenDatFilePath(void)
{
   return (Path);
}


/*
// Bounding boxes of linetype 1,2,3,4,5 D:\Lego\Ldraw\LDRAW\P 19980818/LCH
// - 1-4ccyli.dat:          2 <0,-1,0> <3.5,1,3.5>
// - 1-4cyli.dat :          1 <0,0,0> <1,1,1>
// - 1-4cyls.dat :          1 <0,0,0> <1,1,1>
// + 1-4disc.dat :          0 <0,0,0> <1,0,1>
// + 1-4edge.dat :          0 <0,0,0> <1,0,1>
// + 1-4ndis.dat :          0 <0,0,0> <1,0,1>
// - 1-8cyli.dat :          1 <0.7071,0,0> <1,1,0.7071>
// + 1-8disc.dat :          0 <0,0,0> <1,0,0.7>
// + 1-8edge.dat :          0 <0.7071,0,0> <1,0,0.7071>
// + 1-8ndis.dat :          0 <0.7071,0,0> <1,0,1>
// - 1-8sphe.dat :          1 <0,0,0> <1,1,1>
// - 2-4cyli.dat :          1 <-1,0,0> <1,1,1>
// - 2-4cyls.dat :          2 <-1,0,0> <1,2,1>
// + 2-4disc.dat :          0 <-1,0,0> <1,0,1>
// + 2-4edge.dat :          0 <-1,0,0> <1,0,1>
// + 2-4ndis.dat :          0 <-1,0,0> <1,0,1>
// + 2-4ring3.dat:          0 <-4,0,0> <4,0,4>
// - 2-8sphe.dat :          1 <-1,0,0> <1,1,1>
// - 3-4cyli.dat :          1 <-1,0,-1> <1,1,1>
// + 3-4disc.dat :          0 <-1,0,-1> <1,0,1>
// + 3-4edge.dat :          0 <-1,0,-1> <1,0,1>
// - 3-8cyli.dat :          1 <-0.7071,0,0> <1,1,1>
// - 3-8cyls.dat :        1.7 <-0.7,0,0> <1,1.7,1>
// + 3-8edge.dat :          0 <-0.7071,0,0> <1,0,1>
// - 4-4cyli.dat :          1 <-1,0,-1> <1,1,1>
// + 4-4disc.dat :          0 <-1,0,-1> <1,0,1>
// + 4-4edge.dat :          0 <-1,0,-1> <1,0,1>
// + 4-4ndis.dat :          0 <-1,0,-1> <1,0,1>
// - 4-8sphe.dat :          1 <-1,0,-1> <1,1,1>
// - 8-8sphe.dat :          2 <-1,-1,-1> <1,1,1>
// - arm1.dat    :         20 <-10,-10,-20> <10,10,0>
// - arm2.dat    :         20 <-10,-10,-20> <10,10,0>
// - axle.dat    :          1 <-6,0,-6> <6,1,6>
// + axlehol2.dat:          0 <-6,0,-6> <6,0,6>
// + axlehol3.dat:          0 <-5.58,0,-5.58> <5.58,0,5.58>
// - axlehol4.dat:          1 <-6,0,-6> <6,1,5.58>
// - axlehol5.dat:          1 <-6,0,-5.58> <6,1,5.58>
// - axlehole.dat:          1 <-6,0,-6> <6,1,6>
// - box.dat     :          2 <-1,-1,-1> <1,1,1>
// - box4.dat    :          1 <-1,0,-1> <1,1,1>
// - box5.dat    :          1 <-1,0,-1> <1,1,1>
// - bushloc2.dat:          2 <-9,0,-9> <9,2,9>
// - bushloc3.dat:          2 <-9,0,-9> <9,2,9>
// - bushloc4.dat:          2 <-9,0,-9> <9,2,9>
// - bushlock.dat:          2 <-10,0,-10> <10,2,10>
// - clip3.dat   :         16 <-4,-6,-4> <4,10,10>
// - connect.dat :         20 <-8,-20,-8> <8,0,8>
// - connect2.dat:         20 <-6,-20,-6> <6,0,6>
// - connect3.dat:         10 <-8,-10,-8> <8,0,8>
// - connect4.dat:         10 <-6,-10,-6> <6,0,6>
// - h1.dat      :         14 <-4,0,-10> <4,14,10>
// - h2.dat      :         14 <-4,0,-10> <4,14,10>
// - stud5.dat   :          4 <-18.48,-4,-18.48> <18.48,0,18.48>
// + rect.dat    :          0 <-1,0,-1> <1,0,1>
// + ring10.dat  :          0 <-11,0,-11> <11,0,11>
// + ring2.dat   :          0 <-3,0,-3> <3,0,3>
// + ring3.dat   :          0 <-4,0,-4> <4,0,4>
// + ring4.dat   :          0 <-5,0,-5> <5,0,5>
// + ring7.dat   :          0 <-8,0,-8> <8,0,8>
// - steerend.dat:          8 <-10,0,-10> <10,8,10>
// - stu2.dat    :          4 <-6,-4,-6> <6,0,6>
// - stu22.dat   :          4 <-6,-4,-6> <6,0,6>
// - stu22a.dat  :          4 <-6,-4,-6> <6,0,6>
// - stu23.dat   :          4 <-4,-4,-4> <4,0,4>
// - stu23a.dat  :          4 <-4,-4,-4> <4,0,4>
// - stu24.dat   :          4 <-8,-4,-8> <8,0,8>
// - stu24a.dat  :          4 <-8,-4,-8> <8,0,8>
// - stud2.dat   :          4 <-6,-4,-6> <6,0,6>
// - stud2a.dat  :          4 <-6,-4,-6> <6,0,6>
// - stud3a.dat  :          4 <-4,-4,-4> <4,0,4>
// - stud4a.dat  :          4 <-8,-4,-8> <8,0,8>
// - studline.dat:          4 <0,-4,0> <0,0,0>
// - tri3.dat    :          1 <0,0,0> <1,1,1>
// - stu2p01.dat :          4 <-6,-4,-6> <6,0,6>
// - studp01.dat :          4 <-6,-4,-6> <6,0,6>
// + ring1.dat   :          0 <-2,0,-2> <2,0,2>
// + 3-4ndis.dat :          0 <-1,0,-1> <1,0,1>
// - studel.dat  :          4 <-6,-4,-6> <6,0,6>
// - box4t.dat   :          1 <-1,0,-1> <1,1,1>
// - stud.dat    :          4 <-6,-4,-6> <6,0,6>
// - stud3.dat   :          4 <-4,-4,-4> <4,0,4>
// - stud4.dat   :          4 <-8,-4,-8> <8,0,8>
// - peghole.dat :          2 <-8,0,-8> <8,2,8>
// - 4-4con12.dat:          1 <-2,0,-2> <2,1,2>
// - box4-7a.dat :          1 <-1,0,-1> <1,1,1>
// - box4-1.dat  :          1 <-1,0,-1> <1,1,1>
// - box4-4a.dat :          1 <-1,0,-1> <1,1,1>

More primitives 19990217/LCH
// - 1-4con3.dat :          1 <0,0,0> <4,1,4>
// - 1-4con4.dat :          1 <0,0,0> <5,1,5>
// - box3#8.dat  :          1 <-1,0,-1> <1,1,1>
// - 1-4cyls2.dat:          1 <-1,0,0> <0,1,1>
                                                                             */
static char         *xzParts[] = {
   "1-4disc.dat",
   "1-4edge.dat",
   "1-4ndis.dat",
   "1-8disc.dat",
   "1-8edge.dat",
   "1-8ndis.dat",
   "2-4disc.dat",
   "2-4edge.dat",
   "2-4ndis.dat",
   "2-4ring3.dat",
   "3-4disc.dat",
   "3-4edge.dat",
   "3-4ndis.dat",
   "3-8edge.dat",
   "4-4disc.dat",
   "4-4edge.dat",
   "4-4ndis.dat",
   "axlehol2.dat",
   "axlehol3.dat",
   "rect.dat",
   "ring1.dat",
   "ring2.dat",
   "ring3.dat",
   "ring4.dat",
   "ring7.dat",
   "ring10.dat",
};

#define  POL 1e-4
static float         PointsOnLine(float p1[4], float p2[4], float p3[4])
{
   float                v1[4],
                        v2[4],
                        cp[4],
                        dp;

   V3Sub(v1, p1, p3);
   V3Sub(v2, p2, p3);
   V3Cross(cp, v1, v2);
   dp = V3Dot(cp, cp);
   if (dp < 0.0)
      dp = -dp;
   return (dp);
#ifdef   TOO_INACCURATE
   dp = V3Dot(v1, v2);
   dp = dp * dp / V3Dot(v1, v1) / V3Dot(v2, v2);
   return (dp >= 0.999924 ? dp : 0.0);    /* cos^2(0.5 degrees)              */
#endif
}

void                 Resequence(float v[4][4], int a, int b, int c, int d)
{
   float                o[4][4];

   memcpy(o, v, sizeof(o));
   memcpy(v[0], o[a], sizeof(o[0]));
   memcpy(v[1], o[b], sizeof(o[0]));
   memcpy(v[2], o[c], sizeof(o[0]));
   memcpy(v[3], o[d], sizeof(o[0]));
}

#define SamePoint(i,j) (LinePtr->v[i][0]==LinePtr->v[j][0] && LinePtr->v[i][1]==LinePtr->v[j][1] && LinePtr->v[i][2]==LinePtr->v[j][2])


#define LINE_ERROR 1
static int           CheckLine(struct L3LineS * LinePtr,
                               char *DatName,
                               struct L3PartS * PartPtr)
{
   register int         i;
   register int         j;
   float                vec[4][4];
   float                det;
   float                maxdist;
   float                dist[4];
   float                cp1[4];
   float                cp2[4];
   float                dp;
   float                v01[4];
   float                v02[4];
   float                v03[4];
   float                v12[4];
   float                v13[4];
   float                v23[4];
   float                dotA,
                        dotB,
                        dotC;
   int                  A,
                        B,
                        C;
    UNUSED(PartPtr);

   /* Now do some argument checking */
   switch (LinePtr->LineType)
   {
      case 1:                             /* Part command                    */
/* 1 16 16   -36 24.75   4 0 0   0 0 -4   0 0 0.25   1-4disc.dat */
         det = M3Det(LinePtr->v);
         if (!det)
         {
            /* The determinant is zero. We will try to fix the transformation
               matrix if the part is one of the XZ-plane primitives. Many part
               authors are lazy or ignorant and specify zero y values.       */
            for (i = sizeof(xzParts) / sizeof(xzParts[0]); --i >= 0;)
#ifdef USE_OPENGL
               if (stricmp(DatName, xzParts[i]) == 0)
#else
               if (strcmp(DatName, xzParts[i]) == 0)
#endif
                  break;
            if (i < 0)
            {
               ErrorInInput(0, II_SKIPPING, "Singular matrix");
               return (LINE_ERROR);
            }
            /* A known xz-only primitive. Try to fix singular matrix. Typically
               one of the rows are all zeros, and it doesn't affect the
               transformation to set the y value to 1.0 (what the part author
               should have done). Also the second column may be all zeros. Here
               we may set one of the y values to 1.0 at a time. Setting all
               three y values to 1.0 at the same time may give two rows that
               are linear dependent.                                         */
            for (i = 0; i < 3; i++)
            {
               if (!LinePtr->v[i][0] && !LinePtr->v[i][1] && !LinePtr->v[i][2])
               {
                  /* Row i all zeros, try fixing by setting the y value */
                  LinePtr->v[i][1] = 1.0;
                  det = M3Det(LinePtr->v);/* Calculate new determinant       */
                  if (det)
                  {
                     sprintf(ErrStr, "Row %d all zeros", i);
                     ErrorInInput(3, II_WARNING, ErrStr);
                     break;
                  }
               }
            }
            if (!det && !LinePtr->v[0][1] && !LinePtr->v[1][1] && !LinePtr->v[2][1])
            {
               for (i = 0; i < 3; i++)
               {
                  /* Y column all zeros, try fixing one y at the time */
                  LinePtr->v[i][1] = 1.0;
                  det = M3Det(LinePtr->v);/* Calculate new determinant       */
                  if (det)
                  {
                     ErrorInInput(3, II_WARNING, "Y column all zeros");
                     break;
                  }
               }
            }
            if (!det)
            {
               ErrorInInput(0, II_SKIPPING, "Singular matrix (unfixable)");
               return (LINE_ERROR);
            }
         }
         break;

      case 2:                             /* Line command                    */
         if (SamePoint(0, 1))
         {
            ErrorInInput(1, II_WARNING, "Identical vertices");
         }
         LinePtr->v[0][3] = LinePtr->v[1][3] = 1.0;
         break;

      case 3:                             /* Triangle command                */
         if (SamePoint(0, 1) || SamePoint(0, 2) || SamePoint(1, 2))
         {
            ErrorInInput(1, II_WARNING, "Identical vertices");
            if (LinePtr->Color != 16)
               return (LINE_ERROR);
         }
         else
         {
            /* If "Identical vertices" then no need for further checks. */
            dp = PointsOnLine(LinePtr->v[0], LinePtr->v[1], LinePtr->v[2]);
            if (dp < POL)
            {
               sprintf(ErrStr, "Collinear vertices (%g)", dp);
               ErrorInInput(1, II_WARNING, ErrStr);
               if (LinePtr->Color != 16)
                  return (LINE_ERROR);
            }
         }
         LinePtr->v[0][3] = LinePtr->v[1][3] = LinePtr->v[2][3] = 1.0;
         break;

      case 4:                             /* Quadrilateral command           */
         if (SamePoint(0, 1) || SamePoint(0, 2) || SamePoint(0, 3) ||
             SamePoint(1, 2) || SamePoint(1, 3) || SamePoint(2, 3))
         {
/* Error reported by PovRay:

// Slope Brick 45  2 x  2 with computer screen front
// name 3039p23.dat
...
   triangle { <-16,2.93,-12.93>, <0,2.93,-12.93>, <-16,2.93,-12.93> texture { Color15 } }
...

POV:
   triangle { <-16,2.93,-12.93>, <0,2.93,-12.93>, <-16,2.93,-12.93> texture  <----ERROR
  star-ftr.pov:2998: error: No matching } in triangle, texture found instead.

Apparently only if triangle also has texture, no problems with 40.dat !
                                                                             */
            ErrorInInput(1, II_WARNING, "Identical vertices");
            if (LinePtr->Color != 16)
               return (LINE_ERROR);
         }
         else
         {
            /* If "Identical vertices" then no need for further checks. */
            /* Maybe use the three non-identical vertices as linetype 3 ? */
            dp = PointsOnLine(LinePtr->v[0], LinePtr->v[1], LinePtr->v[2]);
            if (dp < POL)
            {
               sprintf(ErrStr, "Collinear vertices 012 (%g)", dp);
               ErrorInInput(1, II_WARNING, ErrStr);
               if (LinePtr->Color != 16)
                  return (LINE_ERROR);
            }
            dp = PointsOnLine(LinePtr->v[0], LinePtr->v[1], LinePtr->v[3]);
            if (dp < POL)
            {
               sprintf(ErrStr, "Collinear vertices 013 (%g)", dp);
               ErrorInInput(1, II_WARNING, ErrStr);
               if (LinePtr->Color != 16)
                  return (LINE_ERROR);
            }
            dp = PointsOnLine(LinePtr->v[0], LinePtr->v[2], LinePtr->v[3]);
            if (dp < POL)
            {
               sprintf(ErrStr, "Collinear vertices 023 (%g)", dp);
               ErrorInInput(1, II_WARNING, ErrStr);
               if (LinePtr->Color != 16)
                  return (LINE_ERROR);
            }
            dp = PointsOnLine(LinePtr->v[1], LinePtr->v[2], LinePtr->v[3]);
            if (dp < POL)
            {
               sprintf(ErrStr, "Collinear vertices 123 (%g)", dp);
               ErrorInInput(1, II_WARNING, ErrStr);
               if (LinePtr->Color != 16)
                  return (LINE_ERROR);
            }


/*

           \         /
            \  II   /
             \     /
              \   /
               \ /
      III       1       IV
               / \
              /   \
             /     \
            /   V   \
           /         \
   -------0-----------2--------

                I

    __   __   __   __
A: (01 x 02).(02 x 03) > 0 then 3 is in I else in II, III, IV or V
    __   __   __   __
B: (12 x 10).(10 x 13) > 0 then 3 is in II or III else in IV or V
    __   __   __   __
C: (20 x 21).(21 x 23) > 0 then 3 is in II or IV else in III or V

if (A)
{
   3 is in I
   OK: 0123 D02 (convex/concave)
   if (!B && !C) OK: 0123 D02/13 (convex)
   OK: abcd=0123 Dac
}
else
{
   3 is in II, III, IV or V
   if (B)
   {
      3 is in II or III
      if (C)
      {
         3 is in II
         OK: 0123 D13 (concave)
         OK: abcd=1230 Dac
      }
      else
      {
         3 is in III
         Error, use 0312 D01/D23 (convex)
         Error, use abcd=0312 Dac
      }
   }
   else
   {
      3 is in IV or V
      if (C)
      {
         3 is in IV
         Error, use 0132 D12/D03 (convex)
         Error, use abcd=0132 Dac
      }
      else
      {
         3 is in V
         OK: 0123 D13 (concave)
         OK: abcd=1230 Dac
      }
   }
}

                                                                             */


            V3Sub(v01, LinePtr->v[1], LinePtr->v[0]);
            V3Sub(v02, LinePtr->v[2], LinePtr->v[0]);
            V3Sub(v03, LinePtr->v[3], LinePtr->v[0]);
            V3Sub(v12, LinePtr->v[2], LinePtr->v[1]);
            V3Sub(v13, LinePtr->v[3], LinePtr->v[1]);
            V3Sub(v23, LinePtr->v[3], LinePtr->v[2]);
            V3Cross(cp1, v01, v02);
            V3Cross(cp2, v02, v03);
            dotA = V3Dot(cp1, cp2);
            V3Cross(cp1, v12, v01);
            V3Cross(cp2, v01, v13);
            dotB = V3Dot(cp1, cp2);
            V3Cross(cp1, v02, v12);
            V3Cross(cp2, v12, v23);
            dotC = -V3Dot(cp1, cp2);
            A = (dotA > 0.0);
            B = (dotB > 0.0);
            C = (dotC > 0.0);

/* dot[ABC] may falsely be zero for these:
WARNING "30033.DAT" Line 11: Collinear vertices A (-0): 4 16 0 4 20 0 8 20 -7.65 4 18.48 7.65 4 18.48
WARNING "6084.DAT" Line 458: Collinear vertices C (0): 4 16 -57 -29 15 -57 -29 40 20 -53 40 -20 -53 40
but they do not have three points on a line. The det is very big however.    */

            if (A)
            {
               /* 3 is in I */
#ifdef XXXXXXX
               Str = "OK: 0123 D02 (convex/concave)";
               if (!B && !C)
                  Str = "OK: 0123 D02/13 (CONVEX)";
#endif
            }
            else
            {
               /* 3 is in II, III, IV or V */
               if (B)
               {
                  /* 3 is in II or III */
                  if (C)
                  {
                     /* 3 is in II */
#ifdef XXXXXXX
                     Str = "OK: 0123 D13 (concave) abcd=1230";
#endif
                     Resequence(LinePtr->v, 1, 2, 3, 0);
                  }
                  else
                  {
                     /* 3 is in III */
#ifdef XXXXXXX
                     Str = "Error, use 0312 D01/D23 (convex)";
#endif
                     ErrorInInput(2, II_WARNING, "Bad vertex sequence, 0312 used");
                     Resequence(LinePtr->v, 0, 3, 1, 2);
                  }
               }
               else
               {
                  /* 3 is in IV or V */
                  if (C)
                  {
                     /* 3 is in IV */
#ifdef XXXXXXX
                     Str = "Error, use 0132 D12/D03 (convex)";
#endif
                     ErrorInInput(2, II_WARNING, "Bad vertex sequence, 0132 used");
                     Resequence(LinePtr->v, 0, 1, 3, 2);
                  }
                  else
                  {
                     /* 3 is in V */
#ifdef XXXXXXX
                     Str = "OK: 0123 D13 (concave) abcd=1230";
#endif
                     Resequence(LinePtr->v, 1, 2, 3, 0);
                  }
               }
            }
            if (DetThreshold > 0.0)
            {
               for (j = 0; j < 3; j++)
                  V3Sub(vec[j], LinePtr->v[j], LinePtr->v[3]);
               det = M3Det(vec);
               if (det < 0.0)
                  det = -det;
               /* Should I use unit vectors for determinat ? */
               if (det > DetThreshold)
               {
                  /* for (j=0; j<3; j++) for (i=0; i<3; i++) printf("%g ",
                     vec[j][i]); printf("\n");                               */
                  sprintf(ErrStr, "Vertices not coplanar (%g)", det);
                  ErrorInInput(2, II_WARNING, ErrStr);
#ifdef CHECK_PROPER_ROUNDING
                  for (j = 0; j < 3; j++)
                     for (i = 0; i < 3; i++)
                        for (n = 0; n < 2; n++)
                        {
                           vec[j][i] += n ? 0.01 : -0.01;
                           det = M3Det(vec);
                           if (det < 0.0)
                              det = -det;
                           printf(" %.3f", det);
                           vec[j][i] -= n ? 0.01 : -0.01;
                        }
                  printf("\n");
#endif
               }
            }
            if (ShowAllDists || DistThreshold > 0.0)
            {
               /* We need to calculate */
               for (j = 0; j < 3; j++)
                  V3Sub(vec[j], LinePtr->v[j], LinePtr->v[3]);
               det = M3Det(vec);
               /* if (det == 0.0) return; */
               if (det < 0.0)
                  det = -det;
               for (j = 0; j < 4; j++)
                  V3Sub(vec[j], LinePtr->v[j], LinePtr->v[(j + 1) & 3]);
               for (j = 0; j < 4; j++)
               {
                  V3Cross(cp1, vec[j], vec[(j + 3) & 3]);
                  dist[j] = V3Length(cp1);
               }
            }
            if (ShowAllDists && det)
            {
               printf("%15.11f %12.8f %12.8f %12.8f %12.8f \"%s\" Line %3d: %s\n",
                      det, dist[0], dist[1], dist[2], dist[3],
                      IInfo.Filename, IInfo.LineNo, IInfo.InputStr);
               if (dist[0] && dist[1] && dist[2] && dist[3])
                  printf("%15.11f %12.8f %12.8f %12.8f %12.8f \"%s\" Line %3d: %s\n",
                         det, det / dist[0], det / dist[1], det / dist[2], det / dist[3],
                         IInfo.Filename, IInfo.LineNo, IInfo.InputStr);
               printf("\n");
            }
            if (DistThreshold > 0.0)
            {
               /* Find max dist */
               maxdist = 0.0;
               for (j = 0; j < 4; j++)
               {
                  /* if dist[j] is zero, det should also be zero...? */
                  if (dist[j] == 0.0)
                     continue;
                  if (det / dist[j] > maxdist)
                     maxdist = det / dist[j];
               }
               if (maxdist > DistThreshold)
               {
                  sprintf(ErrStr, "Vertices not coplanar (%g)", maxdist);
                  ErrorInInput(2, II_WARNING, ErrStr);
               }
            }
         }
         LinePtr->v[0][3] = LinePtr->v[1][3] = LinePtr->v[2][3] = LinePtr->v[3][3] = 1.0;
         break;

      case 5:                             /* Optional-Line command           */
         if (SamePoint(0, 1) || SamePoint(2, 3))
         {
            ErrorInInput(1, II_WARNING, "Identical vertices");
            return (LINE_ERROR);
         }
         LinePtr->v[0][3] = LinePtr->v[1][3] = LinePtr->v[2][3] = LinePtr->v[3][3] = 1.0;
         break;
   }                                      /* switch (LinePtr->LineType)      */


   if (LinePtr->LineType != 0)
   {
      if (LinePtr->Color != 16)
      {
         if (LinePtr->Color == 24)
         {
            if (LinePtr->LineType == 1 ||
                LinePtr->LineType == 3 ||
                LinePtr->LineType == 4)
            {
               ErrorInInput(0, II_WARNING,
                            "Sorry, cannot handle color 24, using 16");
               LinePtr->Color = 16;
            }
         }
         else
         {
#ifndef USE_OPENGL
            if (Color2rgb(LinePtr->Color, NULL) < 0)
            {
               ErrorInInput(0, II_WARNING, "Unknown color, using 7");
               LinePtr->Color = 7;
            }
#endif
         }
      }
   }
   return 0;
}                                         /* CheckLine                       */



#define METATYPE_TRANSLATE 1
#define METATYPE_ROTATE    2
#define METATYPE_SCALE     3
#define METATYPE_TRANSFORM 4
#define METATYPE_TEXMAP    5

static char         *MetaKeywords[] = {
   NULL,
   "TRANSLATE",
   "ROTATE",
   "SCALE",
   "TRANSFORM",
   "!TEXMAP",
};
/* Return MetaType, negative if END */
static int           ReadMetaLine(struct L3LineS * LinePtr)
{
   int                  n;
   static int           nArgumentsExpected[] = {0, 3, 4, 1, 12, 12};
   register int         i;
   static char          MetaStr[200];     /* static to save stack            */
   int                  MetaType;
   float                vec[4];
   float                c;
   float                s;
   float                t;
   float                a;
   float                b;

     static char          TexStr[200];     /* static to save stack            */
     static char          TexTyp[200];     /* static to save stack            */
     static char          TexNam[200];     /* static to save stack            */


/*
    0 TRANSLATE x y z
    surrounded lines
    0 TRANSLATE end
        This command shifts all surrounded lines in the x, y, and
        z directions

    0 ROTATE angle_degrees axis_x axis_y axis_z
    surrounded lines
    0 ROTATE END
        This command rotates all surrounded lines around the line
        through the origin and the point (axis_x, axis_y,
        axis_z).

    0 SCALE scale_factor
    surrounded lines
    0 SCALE END
        This scales all surrounded lines by the scale factor.

    0 TRANSFORM x y z a b c d e f g h i
    surrounded lines
    0 TRANSFORM END
        This command accomplishes the same sort of thing as
        putting the lines in a subfile and using a Type-1 line to
        include them. x, y, and z specify translation values, and
        a,b,c,d,e,f,g,h,i specify the scale/rotation matrix.

    0 !TEXMAP x1 y1 z1 x2 y2 z2 x3 y3 z3 filename
    0 !: geometry for mapping texture 
    0 !TEXMAP FALLBACK
    surrounded lines
    0 !TEXMAP END
        This command optionally replaces all surrounded lines by the texture.

                                                                             */
   if (LinePtr->LineType != 0)
      return (0);
   if (sscanf(IInfo.InputStr, "%*d %s", MetaStr) != 1)
      return (0);

   for (i = sizeof(MetaKeywords) / sizeof(MetaKeywords[0]); --i > 0;)
      if (strcmp(MetaStr, MetaKeywords[i]) == 0)
         break;
   if (i == 0)
      return (0);
   MetaType = i;
   if (sscanf(IInfo.InputStr, "%*d %*s %s", MetaStr) == 1
       && strcmp(MetaStr, "END") == 0)
      return (-MetaType);

   // Hmmm, FALLBACK is sorta between START and END.  What to do here...
   if (sscanf(IInfo.InputStr, "%*d %*s %s", MetaStr) == 1
       && strcmp(MetaStr, "FALLBACK") == 0)
      return (0);

   memset(LinePtr, 0, sizeof(struct L3LineS));

   if (MetaType == METATYPE_TEXMAP)
   {
     n = sscanf(IInfo.InputStr,
		"%*d %*s %s %s %f %f %f %f %f %f %f %f %f %s",
		TexStr, TexTyp,
		&LinePtr->v[0][0], &LinePtr->v[0][1], &LinePtr->v[0][2],
		&LinePtr->v[1][0], &LinePtr->v[1][1], &LinePtr->v[1][2],
		&LinePtr->v[2][0], &LinePtr->v[2][1], &LinePtr->v[2][2],
		TexNam);
   }
   else

   n = sscanf(IInfo.InputStr,
              "%*d %*s %f %f %f %f %f %f %f %f %f %f %f %f",
              &LinePtr->v[0][3], &LinePtr->v[1][3], &LinePtr->v[2][3],
              &LinePtr->v[0][0], &LinePtr->v[0][1], &LinePtr->v[0][2],
              &LinePtr->v[1][0], &LinePtr->v[1][1], &LinePtr->v[1][2],
              &LinePtr->v[2][0], &LinePtr->v[2][1], &LinePtr->v[2][2]);
   if (n != nArgumentsExpected[MetaType])
   {
      ErrorInInput(0, II_SKIPPING, "Illegal syntax");
      return (0);
   }
   switch (MetaType)
   {
      case METATYPE_TRANSLATE:
         LinePtr->v[0][0] = LinePtr->v[1][1] = LinePtr->v[2][2] = 1.0;
         break;
      case METATYPE_ROTATE:
         a = LinePtr->v[0][3];
         vec[0] = LinePtr->v[1][3];
         vec[1] = LinePtr->v[2][3];
         vec[2] = LinePtr->v[0][0];
         LinePtr->v[0][3] = LinePtr->v[1][3] = LinePtr->v[2][3] = 0.0;
/* See Foley+vanDam "Computer Graphics, Principles and Practice 2nd Edition"
Exercise 5.15 p. 227. There is a sign error in the book!                     */
#ifndef M_PI
#define M_PI      3.14159265358979323846
#endif
         a = M_PI * a / 180.0;
         c = cos(a);
         s = sin(a);
         t = 1 - c;
         if (V3Unit(vec, vec))
            ErrorInInput(0, II_WARNING, "Zero vector");
         LinePtr->v[0][0] = t * vec[0] * vec[0] + c;
         a = t * vec[0] * vec[1];
         b = s * vec[2];
         LinePtr->v[0][1] = a - b;        /* Book erroneously says +         */
         LinePtr->v[1][0] = a + b;
         a = t * vec[0] * vec[2];
         b = s * vec[1];
         LinePtr->v[0][2] = a + b;
         LinePtr->v[2][0] = a - b;
         LinePtr->v[1][1] = t * vec[1] * vec[1] + c;
         a = t * vec[1] * vec[2];
         b = s * vec[0];
         LinePtr->v[1][2] = a - b;
         LinePtr->v[2][1] = a + b;
         LinePtr->v[2][2] = t * vec[2] * vec[2] + c;
         break;
      case METATYPE_SCALE:
         LinePtr->v[0][0] = LinePtr->v[1][1] = LinePtr->v[2][2] = LinePtr->v[0][3];
         LinePtr->v[0][3] = 0.0;
         break;
      case METATYPE_TRANSFORM:
         break;
      case METATYPE_TEXMAP:
	 {
#if 0
	   int texture;
	   int w, h;
	   extern int loadTexture(char *, int *, int *);
	   
	   sprintf(MetaStr, "textures\\%s", TexNam);
	   printf("TEXMAP %s\n", MetaStr);
	   texture = loadTexture(MetaStr, &w, &h);
	   printf("TEXMAP %s = %d (%dx%d pixels)\n", TexNam, texture, w, h);
#endif
	 }
	 // Fake an identity transform for now.  Gotta store the texture number somewhere.
	 memset(LinePtr, 0, sizeof(struct L3LineS));
         LinePtr->v[0][0] = LinePtr->v[1][1] = LinePtr->v[2][2] = LinePtr->v[3][3] = 1;
         break;
   }
   LinePtr->v[3][3] = 1.0;
#ifdef USE_OPENGL
   // Preserve leading whitespace in unused RandomColor field. 
   LinePtr->RandomColor = strspn(IInfo.InputStr, " \t"); // Was wiped by memset above.
   // Save a copy of the original ldlite macro for LEDIT mode saves.
   LinePtr->Comment = Strdup(IInfo.InputStr);
#endif
   return (MetaType);
}

static int           InternID;
/*
Initial state is HEADER.
             ___________Type of input line________
Oldstate     Bad line    0 FILE    0 xxx     valid
HEADER       HEADER      MPD        DAT0      DAT
MPD          E+MPD       MPD        MPD       MPD
DAT0         E+DAT0      MPD        DAT0      DAT
DAT          E+DAT     W+MPD        DAT       DAT
E: Skipping bad line
W: LDraw commands preceded this 0 FILE

Valid lines are error-free linetypes 1-5 plus certain meta commands.
This pragmatic approach fails to report error in a line (e.g. bad number of
arguments in a linetype 4) if the line has not been preceded by a valid line
(so that FileType has been set to DAT).
Comment lines (0 xxx) are allowed before first 0 FILE, but not valid lines.
                                                                             */
static int           ReadDatFile(FILE *fp, struct L3PartS * PartPtr,
                                 char *ReferencingDatfile, int IsModel)
{
   struct L3LineS       Data;
   struct L3LineS     **LinePtrPtr;
   int                  n;
   char                *s;
   register int         i;
   int                 FirstFILE;
   enum
   {
      HEADER, MPD, DAT0, DAT
   }                    FileType;
   static int           nArgumentsExpected[] = {0, 15, 8, 11, 14, 14};
   int                  MetaType;
   int                  ExpectedEnd;
   struct L3PartS      *NewPartPtr;

   FileType = HEADER;
   FirstFILE = true;
   IInfo.Filename = PartPtr->DatName;     /* Save in case of MPD file        */
   IInfo.LineNo = 0;
   ExpectedEnd = 0;
   PartPtr->FirstLine = NULL;
   LinePtrPtr = &(PartPtr->FirstLine);
   while (L3fgets(IInfo.InputStr, sizeof(IInfo.InputStr), fp))
   {
      ++IInfo.LineNo;
      TrimRight(IInfo.InputStr);
#ifndef USE_OPENGL
      if (!IInfo.InputStr[0])
         continue;                        /* Empty line                      */
#endif

      MetaType = 0;
      memset(&Data, 0, sizeof(Data));
#ifndef __TURBOC__
      Data.LineNo = IInfo.LineNo;
#endif
      s = IInfo.InputStr;
      /* double: n = sscanf(IInfo.InputStr,"%d %d %lf %lf... */
#ifndef USE_OPENGL
      n = sscanf(IInfo.InputStr, "%d %d %f %f %f %f %f %f %f %f %f %f %f %f %s",
#else
      /* Allow hex notation for extended colors */
      n = sscanf(IInfo.InputStr, "%d %i %f %f %f %f %f %f %f %f %f %f %f %f %s",
#endif
                 &Data.LineType, &Data.Color,
                 &Data.v[0][0], &Data.v[0][1], &Data.v[0][2],
                 &Data.v[1][0], &Data.v[1][1], &Data.v[1][2],
                 &Data.v[2][0], &Data.v[2][1], &Data.v[2][2],
                 &Data.v[3][0], &Data.v[3][1], &Data.v[3][2], SubPartDatName);
      if (n < 1)
      {
#ifndef USE_OPENGL
         if (FileType != HEADER)
         {
            ErrorInInput(0, II_SKIPPING, "Bad line");
            continue;
         }
#else
	 // Save ALL empty lines as comments (for LEDIT mode).
         //Data.Comment = Strdup(" ");
         /* Reuse the 64 bytes of float v[4][4] */
         Data.Comment = (char *) Data.v;
         strcpy(Data.Comment, " ");
#endif
         Data.LineType = 0;               /* HEADER,save bad line as comment */
      }
      else
      {
#ifdef USE_OPENGL
         // Preserve leading whitespace in unused RandomColor field.
         Data.RandomColor = strspn(IInfo.InputStr, " \t");  
#endif
         switch (Data.LineType)
         {
            case 0:                       /* Comment                         */
               while (*s != '0')
                  s++;
               s++;                       /* Skip '0'                        */
               if (*s)                    /* If character after '0'          */
               {
                  if (*s == ' ' || *s == '\t')
                     s++;                 /* Also skip space after '0'       */
                  else
                     ErrorInInput(3, II_WARNING, "Comment lines should have space after 0");
               }
               /* Watch out for uppercase and (back)slashes in filename... */
               if (sscanf(s, "FILE %s", SubPartDatName) == 1)
               {
                  if (FileType == DAT)
                  {
                     ErrorInInput(0, II_WARNING, "LDraw command(s) preceded this line");
                  }
#ifndef L3P
                  /* Clear any title collected before the first 0 FILE */
                  if (FileType != MPD)
                     ModelTitle[0] = '\0';
#endif
                  FileType = MPD;
                  if (!FirstFILE)
                  {
                     /* Here comes a new part. Check like when at EOF: */
                     while (PartPtr->Internal)
                     {
                        sprintf(ErrStr, "%s END expected",
                                MetaKeywords[ExpectedEnd]);
                        ErrorInInput(0, II_WARNING, ErrStr);
                        PartPtr->FileRead = 1;
                        ExpectedEnd = ((struct L3PartInternalS *) PartPtr)->ExpectedEnd;
                        PartPtr = ((struct L3PartInternalS *) PartPtr)->Father;
                     }
                     PartPtr->FileRead = 1;
#ifdef USE_OPENGL
		     /* Allow quotes around filename... */
		     StripQuotes(s, SubPartDatName);
#endif
                     FixDatName(SubPartDatName);
                     NewPartPtr = FindPart(0, SubPartDatName);
                     if (!NewPartPtr)
                        continue;
                     PartPtr = NewPartPtr;
                     PartPtr->FirstLine = NULL;
#ifdef USE_OPENGL
                     PartPtr->IsMPD = 1;
#endif
                     LinePtrPtr = &(PartPtr->FirstLine);
                     continue;
                  }
                  /* First occurrence of FILE in the MPD file, the file itself.
                     However, keep this "0 FILE file.dat" as a comment, as it
                     may be different from IInfo.Filename                    */
                  FirstFILE = false;
                  break;
               }
               MetaType = ReadMetaLine(&Data);
               if (MetaType)
               {
                  if (FileType != MPD)
                     FileType = DAT;
                  if (MetaType < 0)       /* END                             */
                  {
                     if (ExpectedEnd == 0)
                     {
                        ErrorInInput(0, II_SKIPPING, "END not expected");
                        continue;
                     }
                     if (ExpectedEnd != -MetaType)
                     {
                        sprintf(ErrStr, "%s END expected", MetaKeywords[ExpectedEnd]);
                        ErrorInInput(0, II_SKIPPING, ErrStr);
                        continue;
                     }
                     PartPtr->FileRead = 1;
                     ExpectedEnd = ((struct L3PartInternalS *) PartPtr)->ExpectedEnd;
                     LinePtrPtr = ((struct L3PartInternalS *) PartPtr)->LinePtrPtr;
                     PartPtr = ((struct L3PartInternalS *) PartPtr)->Father;
                     MetaType = 0;        /* Just save the xx END as comment */
                  }
                  break;
               }
	       // ReadMetaLine is really only for transforms
	       // But we need to notice some other metas at parse time.
	       // Need to enable fading render path for runtime FADE cmds to work.
	       else if (ldlite_parse_fade_meta(s)) {
	       }
	       else if (ldlite_parse_silhouette_meta(s)) {
	       }
               if (FileType == HEADER)
                  FileType = DAT0;
               if (strncmp(s, "~Moved to ", 10) == 0)
               {
                  if (ReferencingDatfile)
                     sprintf(ErrStr, "NOTABENE \"%s\" referenced by \"%s\": %s",
                             IInfo.Filename, ReferencingDatfile, s + 1);
                  else
                     sprintf(ErrStr, "NOTABENE \"%s\" Line %d: %s",
                             IInfo.Filename, IInfo.LineNo, IInfo.InputStr);
                  ErrorInInput(0, 0, ErrStr);
               }
#ifndef L3P
               if (IsModel)
               {
                  if (strncmp(s, "Author:", 7) == 0)
                  {
                     if (!ModelAuthor[0])
                     {
                        strncpy(ModelAuthor, FirstNonBlank(s + 7), sizeof(ModelAuthor));
                        ModelAuthor[sizeof(ModelAuthor) - 1] = '\0';
                     }
                  }
                  else
                  {
                     if (!ModelTitle[0])
                     {
                        strncpy(ModelTitle, FirstNonBlank(s), sizeof(ModelTitle));
                        ModelTitle[sizeof(ModelTitle) - 1] = '\0';
                     }
                  }
               }
#endif
               break;
            case 1:                       /* Part command                    */
            case 2:                       /* Line command                    */
            case 3:                       /* Triangle command                */
            case 4:                       /* Quadrilateral command           */
            case 5:                       /* Optional-Line command           */
               if (n != nArgumentsExpected[Data.LineType])
               {
                  if (FileType != HEADER)
                  {
                     sprintf(ErrStr, "%d arguments expected",
                             nArgumentsExpected[Data.LineType]);
                     ErrorInInput(0, II_SKIPPING, ErrStr);
                     continue;
                  }
                  Data.LineType = 0;      /* HEADER,save bad line as comment */
                  break;
               }
               if (FileType != MPD)
                  FileType = DAT;
               break;
            default:
               if (FileType != HEADER)
               {
                  ErrorInInput(0, II_SKIPPING, "Unknown line type");
                  continue;
               }
               Data.LineType = 0;         /* HEADER,save bad line as comment */
               break;
         }



         /* Special care for linetype 1 */
         if (Data.LineType == 1)
         {
#ifdef USE_OPENGL
            /* Allow quotes around filename... */
	    StripQuotes(IInfo.InputStr, SubPartDatName);
#endif
            /* Watch out for uppercase and (back)slashes in filename... */
            FixDatName(SubPartDatName);
            for (i = 0; i < 3; i++)
            {
               Data.v[i][3] = Data.v[0][i];
               Data.v[0][i] = Data.v[1][i];
               Data.v[1][i] = Data.v[2][i];
               Data.v[2][i] = Data.v[3][i];
               Data.v[3][i] = 0.0;
            }
            Data.v[3][3] = 1.0;
#ifdef USE_OPENGL
            if (LightDotDat && stricmp(SubPartDatName, "light.dat") == 0)
#else
            if (LightDotDat && strcmp(SubPartDatName, "light.dat") == 0)
#endif
            {
               struct L3LightS     *LightPtr;

               LightPtr = AddLight();
               LightPtr->Globe = 0;
               for (i = 0; i < 3; i++)
                  LightPtr->Position[i] = Data.v[i][3];
               if (Data.Color == 16 || Data.Color == 24)
               {
                  sprintf(ErrStr, "Sorry, cannot handle color %d, using 15",
                          Data.Color);
                  ErrorInInput(0, II_WARNING, ErrStr);
                  Data.Color = 15;
               }
/*
               if (Color2rgb(Data.Color, LightPtr->LightColor) < 0)
               {
                  ErrorInInput(0, II_WARNING, "Unknown color, using 15");
                  Color2rgb(15, LightPtr->LightColor);
               }
*/
               continue;
            }
         }

         /* Now do some argument checking */
         if (DoCheck && !IsModel)
         {
            /* Only serious errors for subparts, when checking parts */
            int SaveWarningLevel;
            
            SaveWarningLevel = WarningLevel;
            WarningLevel = 0;
            i = CheckLine(&Data, SubPartDatName, PartPtr);
            WarningLevel = SaveWarningLevel;
         }
         else
         {
            i = CheckLine(&Data, SubPartDatName, PartPtr);
         }
         if (i)
            continue;

         /* Special care for linetype 1 */
         if (Data.LineType == 1)
         {
            Data.PartPtr = FindPart(0, SubPartDatName);
            if (!Data.PartPtr)
               continue;
         }
      }                                   /* End of "if (n<1) else..."       */


#ifdef TEST_MEMORY_USAGE
      ++LINETYPES[Data.LineType];
#endif
      if (ReadLineTypeMask & BIT(Data.LineType))
      {
         /* Store input line in memory */
         if (Data.LineType != 0 && Data.Color != 16 && Data.Color != 24)
         {
            /* First check if we have already registered it. */
            for (i = nColors; --i >= 0;)
               if (Data.Color == Colors[i].Color)
                  break;
            if (i < 0)
            {
               /* No, allocate new color. */
               if (nColors >= MAX_COLORS)
               {
                  sprintf(ErrStr, "Sorry, max %d colors.", MAX_COLORS);
                  PromptUser(ErrStr);
                  Data.Color = 16;
               }
               else
               {
                  Colors[nColors++].Color = Data.Color;
               }
            }
         }
         if (SaveLine(&LinePtrPtr, &Data, s))
         {
            sprintf(ErrStr,
                    "Out of memory in \"%s\" line %d, skipping rest of file",
                    IInfo.Filename, IInfo.LineNo);
            PromptUser(ErrStr);
            fclose(fp);
            return false;
         }
      }
      if (MetaType)
      {
         /* Fake a linetype 1 holding the transformation */
         Data.LineType = 1;
         Data.Color = 16;
#ifndef __TURBOC__
         Data.LineNo = 0;
#endif
#ifndef USE_OPENGL
         sprintf(SubPartDatName, "L3Transf%d", InternID++);
         Data.PartPtr = FindPart(1, SubPartDatName);
#else
	 // Make it easier in LEDIT mode to tell what metatype is in the internal file.
	 if (MetaType == METATYPE_ROTATE)
	   sprintf(SubPartDatName, "L3_%s_%d", MetaKeywords[MetaType],InternID++);
	 else if (MetaType < METATYPE_TEXMAP)
	   sprintf(SubPartDatName, "L3_%s_%d", MetaKeywords[MetaType],InternID++);
	 else
	   sprintf(SubPartDatName, "L3_TEXMAP_%d", MetaKeywords[MetaType],InternID++);
         Data.PartPtr = FindPart(MetaType, SubPartDatName);
#endif
         if (!Data.PartPtr)
            continue;
         if (SaveLine(&LinePtrPtr, &Data, NULL))
         {
            sprintf(ErrStr,
                    "Out of memory in \"%s\" line %d, skipping rest of file",
                    IInfo.Filename, IInfo.LineNo);
            PromptUser(ErrStr);
            fclose(fp);
            return false;
         }
         ((struct L3PartInternalS *) Data.PartPtr)->ExpectedEnd = ExpectedEnd;
         ((struct L3PartInternalS *) Data.PartPtr)->LinePtrPtr = LinePtrPtr;
         ((struct L3PartInternalS *) Data.PartPtr)->Father = PartPtr;
         PartPtr = Data.PartPtr;
         LinePtrPtr = &(PartPtr->FirstLine);
         ExpectedEnd = MetaType;
      }
   }                                      /* while (fgets(,,))               */
   fclose(fp);
   while (PartPtr->Internal)
   {
      sprintf(ErrStr, "WARNING \"%s\": End-Of-File, %s END expected",
              IInfo.Filename, MetaKeywords[ExpectedEnd]);
      ErrorInInput(0, 0, ErrStr);
      PartPtr->FileRead = 1;
      ExpectedEnd = ((struct L3PartInternalS *) PartPtr)->ExpectedEnd;
      PartPtr = ((struct L3PartInternalS *) PartPtr)->Father;
   }
   PartPtr->FileRead = 1;
   return true;
}                                         /* ReadDatFile                     */


void                 CalcPartBBox(struct L3PartS * PartPtr, int DoBBox, int DoCamera)
{
   struct L3LineS      *LinePtr;
   int                  i;
   int                  n;
   register int         xyz;
   float                r[4];
   float                BBcorner[4];
   UNUSED(DoCamera);

   if (DoBBox)
   {
      V3Load(PartPtr->BBox[0], 1e16, 1e16, 1e16);
      V3Load(PartPtr->BBox[1], -1e16, -1e16, -1e16);
#ifndef L3P
      memcpy(PartPtr->BBoxES, PartPtr->BBox, sizeof(PartPtr->BBox));
#endif
      PartPtr->Empty = true;
   }
   for (LinePtr = PartPtr->FirstLine; LinePtr; LinePtr = LinePtr->NextLine)
   {
      n = LinePtr->LineType;
      switch (n)
      {
         case 1:
            if (LinePtr->PartPtr->Empty)
               break;
            if (DoBBox)
               PartPtr->Empty = false;
            BBcorner[3] = 1.0;
            for (i = 0; i < 8; i++)
            {
               for (xyz = 0; xyz < 3; xyz++)
                  BBcorner[xyz] = LinePtr->PartPtr->BBox[(i >> xyz) & 1][xyz];
               M4V4Mul(r, LinePtr->v, BBcorner);
               if (DoBBox)
                  CheckPointAgainstBBox(r, PartPtr->BBox);
/*
               if (DoCamera)
                  CalcCameraAddPoint(r);
*/
            }
#ifndef L3P
            if (LinePtr->PartPtr->IsStud)
               break;
            for (i = 0; i < 8; i++)
            {
               for (xyz = 0; xyz < 3; xyz++)
                  BBcorner[xyz] = LinePtr->PartPtr->BBoxES[(i >> xyz) & 1][xyz];
               M4V4Mul(r, LinePtr->v, BBcorner);
               if (DoBBox)
                  CheckPointAgainstBBox(r, PartPtr->BBoxES);
            }
#endif
            break;
         case 5:
            n = 2;
         case 2:
         case 3:
         case 4:
            if (DoBBox)
               PartPtr->Empty = false;
            for (i = 0; i < n; i++)
            {
               if (DoBBox)
               {
                  CheckPointAgainstBBox(LinePtr->v[i], PartPtr->BBox);
#ifndef L3P
                  CheckPointAgainstBBox(LinePtr->v[i], PartPtr->BBoxES);
#endif
               }
/*
               if (DoCamera)
                  CalcCameraAddPoint(LinePtr->v[i]);
*/
            }
            break;
      }
   }
   /* If part only contains studs (like 313.dat) BBoxES is never set! */
#ifndef L3P
   if (PartPtr->BBoxES[0][0] > 1e15)
      memcpy(PartPtr->BBoxES, PartPtr->BBox, sizeof(PartPtr->BBox));
#endif
}


/* List of parts with only linetype 2, no need to load in L3P. */
#ifdef L3P
static char         *LineType2Primitives[] = {
   "1-4edge.dat",
   "1-8edge.dat",
   "2-4edge.dat",
   "3-4edge.dat",
   "3-8edge.dat",
   "4-4edge.dat",
   "axlehol2.dat",
   "axlehol3.dat",
   "studline.dat",
};
#endif
/* Read part (if necessary - it may already be read if MPD) and
resolve any linetype 1 parts                                                 */
/* Return: 0=OK, 1=NotFound, 2=Recursion */
#ifndef USE_OPENGL
static 
#endif
int          LoadPart(struct L3PartS * PartPtr, int IsModel, char *ReferencingDatfile)
{
   FILE                *fp;
   struct L3LineS      *LinePtr;
   struct PovPartS     *PovPartPtr;
#ifdef L3P
   int                  i;
#endif

#ifdef LOGFP
   ++nLevel;
   fprintf(LogFp, "%*sLoadPart     %s from %s %ld\n", nLevel, "",
           PartPtr->DatName, (ReferencingDatfile ? ReferencingDatfile : "*"),
           farcoreleft0 - farcoreleft());
#endif
   if (PartPtr->Recursion)
   {
      sprintf(ErrStr, "Recursive references: %s", PartPtr->DatName);
      ErrorInInput(0, 0, ErrStr);
#ifdef LOGFP
      --nLevel;
#endif
#ifdef USE_OPENGL_RECURSION_HACK
      PartPtr->Investigated = 1; // Mark this part for when we pop out.
#endif      
      return (2);                         /* Recursion                       */
   }
   PartPtr->Recursion = 1;                /* Candidate for recursion         */

   /* Don't use POV equivalent (yet, maybe later on output (L3)) */
   PovPartPtr = NULL;

#ifdef L3P
   if (!DoCheck)
   {
      for (i = sizeof(LineType2Primitives) / sizeof(LineType2Primitives[0]); --i >= 0;)
      {
         if (strcmp(PartPtr->DatName, LineType2Primitives[i]) == 0)
         {
            PartPtr->Empty = true;
            PartPtr->FileRead = 1;
            PartPtr->Resolved = 1;
#ifdef LOGFP
            --nLevel;
#endif
            PartPtr->Recursion = 0;       /* No longer suspected             */
            return (0);
         }
      }
      if (L3Pov.UsePovParts && (PovPartPtr = FindPovPart(PartPtr->DatName)))
      {
         memcpy(PartPtr->BBox, PovPartPtr->BBox, sizeof(PartPtr->BBox));
#ifndef L3P
         memcpy(PartPtr->BBoxES, PovPartPtr->BBox, sizeof(PartPtr->BBox));
#endif
         PartPtr->Empty = false;
         PartPtr->FileRead = 1;
         PartPtr->Resolved = 1;
      }
   }
#endif
   if (!PovPartPtr)
   {
      /* Don't use POV equivalent (yet, maybe later on output (L3)) */
      if (!PartPtr->FileRead)
      {
         fp = OpenDatFile(PartPtr->DatName);
         if (!fp)
         {
            sprintf(ErrStr,
#ifdef L3P
                    "Could not find %s referenced by %s",
#else
                    "Could not find %s\nReferenced by %s",
#endif
                    PartPtr->DatName,
                    (ReferencingDatfile ? ReferencingDatfile : "*"));
            PromptUser(ErrStr);
            PartPtr->Empty = true;
            PartPtr->FileRead = 1;        /* Only show error message once    */
            PartPtr->Resolved = 1;
            PartPtr->Recursion = 0;       /* No longer suspected             */
#ifdef LOGFP
            --nLevel;
#endif
            return (1);                   /* Not found                       */
         }
         PartPtr->FromP = FileIsFromP;
         PartPtr->FromPARTS = FileIsFromPARTS;
         ReadDatFile(fp, PartPtr, ReferencingDatfile, IsModel);
#ifdef LOGFP
         fprintf(LogFp, "%*sLoadPart     %s after read  %ld\n", nLevel, "", PartPtr->DatName, farcoreleft0 - farcoreleft());
#endif
      }
      /* Now load datfiles newly referenced in part. If more parts were read in
         case of MPD file, these parts will be processed (LoadPart) when (or if
         at all) someone references them.                                    */
      for (LinePtr = PartPtr->FirstLine; LinePtr; LinePtr = LinePtr->NextLine)
      {
         if (LinePtr->LineType == 1 && !LinePtr->PartPtr->Resolved)
         {
            if (LoadPart(LinePtr->PartPtr,
                         LinePtr->PartPtr->Internal ? IsModel : false,
                         PartPtr->DatName) == 2)
            {
               /* Recursion */
               sprintf(ErrStr, "referenced by %s"
#ifndef __TURBOC__
                       " line %d"
#endif
                       ,PartPtr->DatName
#ifndef __TURBOC__
                       ,LinePtr->LineNo
#endif
                  );
               ErrorInInput(0, 0, ErrStr);
#ifdef USE_OPENGL_RECURSION_HACK
	       if (PartPtr->Investigated)
	       {
		 ErrorInInput(0, 0, "Skipping part");
		 // Pretend we could not find recursive part and move on.
		 PartPtr->FileRead = 1;   /* Only show error message once    */
		 PartPtr->Resolved = 1;
		 PartPtr->Recursion = 0;  /* No longer suspected             */
		 return (1);              /* Not found */
	       }
	       else
#endif
               return (2);                /* Recursion                       */
            }
         }
      }
      /* Now calculate bounding box of part */
      /* Only lines in model (scene) are used for camera calculation */
      if (!DoCheck)
         CalcPartBBox(PartPtr, true, IsModel);
   }                                      /* !PovPartPtr                     */

#ifdef L3P
   if (!DoCheck)
      PrintPovPart(PovFp, PartPtr);
#ifdef LOGFP
   fprintf(LogFp, "%*sLoadPart     %s before free %ld\n", nLevel, "",
           PartPtr->DatName, farcoreleft0 - farcoreleft());
#endif
   CheckMemoryUsage();
   FreeLines(PartPtr);
#ifdef LOGFP
   fprintf(LogFp, "%*sLoadPart     %s  after free %ld\n", nLevel, "",
           PartPtr->DatName, farcoreleft0 - farcoreleft());
   --nLevel;
#endif
#endif
   PartPtr->Resolved = 1;
   PartPtr->Recursion = 0;                /* No longer suspected             */
   return (0);                            /* OK                              */
}

int                  InitLDrawDir(void)
{
   char                *e;

   e = getenv("LDRAWDIR");
   if (!e)
   {
#ifdef L3P
      FILE                *fp;
      char                 Str[400];
      int                  InLDrawSection;

      e = getenv("windir");
      strcpy(Str, e ? e : "c:\\windows");
      DeleteTrailingBackslash(Str);
      strcat(Str, "\\ldraw.ini");
      fp = fopen(Str, "rt");
      if (fp)
      {
         InLDrawSection = 0;
         while (L3fgets(Str, sizeof(Str), fp))
         {
            if (Str[0] == '[')
            {
               if (strncmp(Str, "[LDraw]", 7) == 0)
                  InLDrawSection = 1;
               else if (InLDrawSection)
                  break;                  /* End of [LDraw] section          */
               continue;
            }
            if (InLDrawSection && strncmp(Str, "BaseDirectory=", 14) == 0)
            {
               fclose(fp);
               TrimRight(Str);
               strcpy(LDrawDir, Str + 14);
               DeleteTrailingBackslash(LDrawDir);
               return (true);
            }
         }
         fclose(fp);
      }
#else
      PromptUser("LDRAWDIR not set.\nSpecify directory in the Edit menu.");
#endif
      return (false);
   }
   strcpy(LDrawDir, e);
   DeleteTrailingBackslash(LDrawDir);
   return (true);
}

/* Return 0 if error, 1 if OK */
int                  LoadModel(const char *lpszPathName)
{
   struct L3PartS      *PartPtr;
   int                  i;
   UNUSED(i);

#ifdef USE_OPENGL
   if (!LDrawDir[0])
     strcpy(LDrawDir, pathname);
   strcpy(ModelDir, datfilepath);
#else
   if (!LDrawDir[0] && !InitLDrawDir())
      return (0);

   strcpy(ModelDir, lpszPathName);
   for (i = strlen(ModelDir); --i >= 0;)
   {
      if (ModelDir[i] == '/' || ModelDir[i] == '\\')
         break;                           /* Leave trailing slash in
                                             ModelDir                        */
      ModelDir[i] = '\0';
   }
#endif
   PartPtr = &Parts[0];
   if (nParts == 0)
      nParts = 1;                         /* First time a model is loaded    */
   else
      FreePart(PartPtr);                  /* Free old model                  */
   PartPtr->DatName = Strdup(lpszPathName);
   if (!PartPtr->DatName)
   {
      PromptUser("Out of memory for model name");
      return (0);
   }
   PartPtr->FirstLine = NULL;
   nColors = 0;
   /* Be sure to register default part color */
   Colors[nColors++].Color = L3Pov.DefaultPartColorNumber;
   if (LoadPart(PartPtr, true, NULL) == 2)
   {
#ifndef L3P
      /* Safest to free all parts since recursion is in cached parts (i.e. not
         enough to free model)                                               */
      PromptUser("Recursion detected. See log, fix problem, try again");
      FreeParts();
      FreeLights();
#endif
      return (0);
   }
#ifdef LOGPC
   for (i = 0; i < nColors; i++)
   {
      fprintf(LogFp, "Color[%d] = %d\n", i, Colors[i].Color);
   }
#endif
#ifdef TEST_MEMORY_USAGE
   {
      long                 sum = 0;
      long                 le = 0;
      int                  siz[6] = {0, 60, 32, 44, 56, 56};
      for (i = 0; i < 6; i++)
      {
         printf("Linetype %d: %4ld %6ld\n", i, LINETYPES[i], siz[i] * LINETYPES[i]);
         le += LINETYPES[i];
         sum += siz[i] * LINETYPES[i];
      }
      printf("Total:       %4ld %6ld\n", le, sum);
   }
#endif
   return (1);
}

#ifndef L3P
void                 LoadModelPre(void)
{
#ifndef USE_OPENGL
   LogDia.m_LogStr.Empty();
#endif
   ModelTitle[0] = '\0';
   ModelAuthor[0] = '\0';
}

void                 LoadModelPost(void)
{
#ifndef USE_OPENGL
   if (!LogDia.m_LogStr.IsEmpty())
      LogDia.DoModal();
#endif
}
#endif

#ifdef USE_OPENGL
//}
#endif /* __cplusplus */

