/******************************************************************************
 *
 * ISO16845 Compliance tests
 * Copyright (C) 2021-present Ondrej Ille
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this SW component and associated documentation files (the "Component"),
 * to use, copy, modify, merge, publish, distribute the Component for
 * educational, research, evaluation, self-interest purposes. Using the
 * Component for commercial purposes is forbidden unless previously agreed with
 * Copyright holder.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Component.
 *
 * THE COMPONENT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHTHOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE COMPONENT OR THE USE OR OTHER DEALINGS
 * IN THE COMPONENT.
 *
 * @author Ondrej Ille, <ondrej.ille@gmail.com>
 * @date 31.7.2020
 *
 *****************************************************************************/

/******************************************************************************
 *
 * @test ISO16845 8.4.1
 *
 * @brief This test verifies that an IUT acting as transmitter generates an
 *        overload frame when it detects a dominant bit on one of the 2 first
 *        recessive bits of the intermission field following a data frame it
 *        is transmitting.
 *
 * @version Classical CAN, CAN FD tolerant, CAN FD enabled
 *
 * Test variables:
 *      Classical CAN, CAN FD Tolerant, CAN FD Enabled:
 *          FDF = 0
 *      CAN FD Enabled:
 *          FDF = 1
 *
 * Elementary test cases:
 *      Elementary tests to perform:
 *          #1 dominant bit on first bit of the intermission field;
 *          #2 dominant bit on second bit of the intermission field.
 *
 * Setup:
 *  The IUT is set to the TEC passive state.
 *
 * Execution:
 *  The LT causes the IUT to transmit a frame. Then, the LT forces one of the
 *  bits of the intermission field to the dominant state according to
 *  elementary test-cases.
 *
 * Response:
 *  The IUT shall generate an overload frame starting at the bit position
 *  following the dominant bit generated by the LT.
 *****************************************************************************/

#include <iostream>
#include <unistd.h>
#include <chrono>
#include <cmath>

#include "TestBase.h"

using namespace can;
using namespace test;

class TestIso_8_4_1 : public test::TestBase
{
    public:

        void ConfigureTest()
        {
            FillTestVariants(VariantMatchType::CommonAndFd);
            for (size_t i = 0; i < 2; i++)
            {
                AddElemTest(TestVariant::Common, ElemTest(i + 1, FrameKind::Can20));
                AddElemTest(TestVariant::CanFdEna, ElemTest(i + 1, FrameKind::CanFd));
            }

            /* Standard settings for tests where IUT is transmitter */
            SetupMonitorTxTests();
            CanAgentConfigureTxToRxFeedback(true);

            /*
             * Set TEC, so that IUT becomes error passive. Keep sufficient
             * reserve from 128 for decrements due to test frames!
             */
            dut_ifc->SetTec(200);
        }

        int RunElemTest([[maybe_unused]] const ElemTest &elem_test,
                        [[maybe_unused]] const TestVariant &test_variant)
        {
            frm_flags = std::make_unique<FrameFlags>(elem_test.frame_kind_, EsiFlag::ErrPas);
            gold_frm = std::make_unique<Frame>(*frm_flags);
            RandomizeAndPrint(gold_frm.get());

            drv_bit_frm = ConvBitFrame(*gold_frm);
            mon_bit_frm = ConvBitFrame(*gold_frm);

            /**************************************************************************************
             * Modify test frames:
             *  1. Turn driven frame as received.
             *  2. Force first or second bit of intermission to dominant. Insert Overload frame on
             *     monitored frame from next bit. Insert Passive Error frame to driven frame. This
             *     also lasts the same length as Overload frame and it consists of 14 recessive
             *     bits. Therefore LT will not affect IUT!
             *  3. Append 15 more recessive bits at the end to both driven and monitored frames.
             *     This should cover intermission, suspend and some reserve and check that IUT does
             *     not retransmitt the frame!
             *************************************************************************************/
            drv_bit_frm->ConvRXFrame();

            Bit *interm_bit = drv_bit_frm->GetBitOf(elem_test.index_ - 1, BitKind::Interm);
            drv_bit_frm->FlipBitAndCompensate(interm_bit, dut_input_delay);
            mon_bit_frm->InsertOvrlFrm(elem_test.index_, BitKind::Interm);
            drv_bit_frm->InsertPasErrFrm(elem_test.index_, BitKind::Interm);

            for (size_t k = 0; k < 15; k++)
            {
                drv_bit_frm->AppendBit(BitKind::Idle, BitVal::Recessive);
                mon_bit_frm->AppendBit(BitKind::Idle, BitVal::Recessive);
            }

            drv_bit_frm->Print(true);
            mon_bit_frm->Print(true);

            /*****************************************************************************
             * Execute test
             *****************************************************************************/
            PushFramesToLT(*drv_bit_frm, *mon_bit_frm);
            StartDrvAndMon();
            this->dut_ifc->SendFrame(gold_frm.get());
            WaitForDrvAndMon();
            CheckLTResult();

            return FinishElemTest();
        }

};