/*
    This file is part of peda.
    Copyright (C) 2000-2010  Petr Porazil <porazil@volny.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Outputs.h"
#include "Error.h"

#include <math.h>
#include <stdlib.h>
#include <QTextStream>
#include <QFileDialog>
#include <QFile>
#include <QPrinter>
#include <QPainter>

#include <QHBoxLayout>
#include <QVBoxLayout>

extern Errors gErr;

const char * OutType[]={
  "OFF",
  "PDF",
  "GERBER-274X",
  "EXCELON",
  "PARTLST",
  "PLACE_TOP",
  "PLACE_BOT",
  "BOM_JLC",
  "CPL_JLC",
  0
};

#define OUTPDF 1
#define OUTGERBER 2
#define OUTEXCELON 3
#define OUTPARTL 4
#define OUTPLTOP 5
#define OUTPLBOT 6
#define BOMJLC 7
#define CPLJLC 8

struct gbr_char{
  char c;
  int * data;
};

int char_A[]={0,0,0, 8,20,1, 16,0,1, 3,7,0, 13,7,1,-1};
int char_B[]={0,0,0, 0,20,1, 12,20,1, 16,16,1, 16,14,1, 12,10,1, 0,10,1, 
              12,10,0, 16,6,1, 16,4,1, 12,0,1, 0,0,1,-1};
int char_C[]={16,4,0, 12,0,1, 4,0,1, 0,4,1, 0,16,1, 4,20,1, 12,20,1,
              16,16,1, -1};
int char_D[]={0,0,0, 0,20,1, 12,20,1, 16,16,1, 16,4,1, 12,0,1, 0,0,1,-1};
int char_E[]={16,0,0, 0,0,1, 0,20,1, 16,20,1, 0,10,0, 14,10,1,-1};
int char_F[]={0,0,0, 0,20,1, 16,20,1, 0,10,0, 14,10,1,-1};
int char_G[]={12,5,0, 16,5,1, 16,4,1, 12,0,1, 4,0,1, 0,4,1, 0,16,1, 
              4,20,1, 12,20,1, 16,16,1, -1};
int char_H[]={0,0,0, 0,20,1, 0,10,0, 16,10,1, 16,0,0, 16,20,1,-1};
int char_I[]={8,0,0, 8,20,1, 6,0,0, 10,0,1, 6,20,0, 10,20,1,-1};
int char_J[]={16,20,0, 16,4,1, 12,0,1, 4,0,1, 0,4,1, -1};
int char_K[]={0,0,0, 0,20,1, 0,10,0, 16,0,1, 0,10,0, 16,20,1, -1};
int char_L[]={0,20,0, 0,0,1, 16,0,1, -1};
int char_M[]={0,0,0, 0,20,1, 8,12,1, 16,20,1, 16,0,1, -1};
int char_N[]={0,0,0, 0,20,1, 16,0,1, 16,20,1, -1};
int char_O[]={16,4,0, 12,0,1, 4,0,1, 0,4,1, 0,16,1, 4,20,1, 12,20,1,
              16,16,1, 16,4,1, -1};
int char_P[]={0,0,0, 0,20,1, 12,20,1, 16,16,1, 16,14,1, 12,10,1, 0,10,1,-1};
int char_Q[]={16,4,0, 12,0,1, 4,0,1, 0,4,1, 0,16,1, 4,20,1, 12,20,1,
              16,16,1, 16,4,1, 12,4,0, 16,0,1, -1};
int char_R[]={0,0,0, 0,20,1, 12,20,1, 16,16,1, 16,14,1, 12,10,1, 0,10,1,
              12,10,0, 16,0,1,-1};
int char_S[]={0,4,0,  4,0,1, 12,0,1, 16,4,1, 0,16,1, 4,20,1, 12,20,1,
              16,16,1, -1};
int char_T[]={8,0,0, 8,20,1, 0,20,0, 16,20,1,-1};
int char_U[]={0,20,0, 0,4,1, 4,0,1, 12,0,1, 16,4,1, 16,20,1, -1};
int char_V[]={0,20,0, 8,0,1, 16,20,1, -1};
int char_W[]={0,20,0, 0,0,1, 8,10,1, 16,0,1, 16,20,1, -1};
int char_X[]={0,20,0, 16,0,1, 0,0,0, 16,20,1, -1};
int char_Y[]={0,20,0, 8,10,1, 8,0,1, 8,10,0, 16,20,1, -1};
int char_Z[]={0,20,0, 16,20,1, 0,0,1, 16,0,1, -1};
int char_0[]={16,16,0, 16,4,1, 12,0,1, 4,0,1, 0,4,1, 0,16,1, 4,20,1, 12,20,1,
              16,16,1, 0,4,1, -1};
int char_1[]={8,0,0, 8,20,1, 4,16,1, 6,0,0, 10,0,1,-1};
int char_2[]={0,16,0, 4,20,1, 12,20,1, 16,16,1, 16,12,1, 0,0,1, 16,0,1, -1};
int char_3[]={0,16,0, 4,20,1, 12,20,1, 16,16,1, 16,14,1, 12,10,1, 8,10,1, 
              12,10,0, 16,6,1, 16,4,1, 12,0,1, 4,0,1, 0,4,1, -1};
int char_4[]={10,0,0, 10,20,1, 0,8,1, 16,8,1, -1};
int char_5[]={16,20,0, 0,20,1, 0,10,1, 12,10,1, 16,6,1,
              16,4,1, 12,0,1, 4,0,1, 0,4,1, -1};
int char_6[]={12,20,0, 4,20,1, 0,16,1, 0,10,1, 12,10,1, 16,6,1,
              16,4,1, 12,0,1, 4,0,1, 0,4,1, 0,10,1, -1};
int char_7[]={0,20,0, 16,20,1, 8,0,1, -1};
int char_8[]={0,16,0, 4,20,1, 12,20,1, 16,16,1, 16,14,1, 12,10,1, 4,10,1, 
              12,10,0, 16,6,1, 16,4,1, 12,0,1, 4,0,1, 0,4,1,
	      0,6,1, 4,10,1, 0,14,1, 0,16,1, -1};
int char_9[]={16,10,0, 4,10,1, 0,14,1, 0,16,1, 4,20,1, 12,20,1, 16,16,1, 
              16,4,1, 12,0,1, 4,0,1, -1};
int char_minus[]={4,10,0, 12,10,1, -1};
int char_eq[]={4,7,0, 12,7,1, 4,13,0, 12,13,1, -1};
int char_plus[]={4,10,0, 12,10,1, 8,6,0, 8,14,1, -1};
int char_uline[]={0,-1,0, 16,-1,1, -1};
int char_slash[]={0,-1,0, 16,20,1, -1};
int char_point[]={7,0,0, 9,0,1, 9,2,1, 7,2,1, 7,0,1, -1};
int char_a[]={16,4,0, 12,0,1, 3,0,1, 0,3,1, 0,4,1, 3,7,1, 13,7,1,
              16,5,1, 0,12,0, 12,12,1, 16,8,1, 16,0,1, -1};
int char_b[]={0,2,0, 4,0,1, 12,0,1, 16,4,1, 16,8,1, 12,12,1, 4,12,1,
              0,10,1, 0,20,0, 0,0,1, -1};
int char_c[]={16,2,0, 12,0,1, 4,0,1, 0,4,1, 0,8,1, 4,12,1, 12,12,1,
              16,10,1, -1};
int char_d[]={16,2,0, 12,0,1, 4,0,1, 0,4,1, 0,8,1, 4,12,1, 12,12,1,
              16,10,1, 16,20,0, 16,0,1, -1};
int char_e[]={16,2,0, 12,0,1, 4,0,1, 0,4,1, 0,8,1, 4,12,1, 12,12,1,
              16,8,1, 16,7,1, 0,7,1, -1};
int char_g[]={16,4,0, 12,0,1, 4,0,1, 0,4,1, 0,8,1, 4,12,1, 12,12,1,
              16,8,1, 16,12,0, 16,-4,1, 12,-8,1, 4,-8,1, -1};
int char_i[]={8,0,0, 8,12,1, 6,0,0, 10,0,1, 6,10,0, 8,12,1, 7,19,0,
              7,18,1, 8,18,1, 8,19,1, 7,19,1, -1};
int char_k[]={0,0,0, 0,20,1, 0,6,0, 16,0,1, 0,6,0, 16,12,1, -1};
int char_l[]={8,0,0, 8,20,1, 6,18,1, 6,0,0, 10,0,1,-1};
int char_m[]={0,0,0, 0,12,1, 0,8,0, 2,12,1, 6,12,1, 8,8,1, 8,0,1, 8,8,0,
              10,12,1, 14,12,1, 16,8,1, 16,0,1, -1};
int char_n[]={0,0,0, 0,12,1, 0,8,0, 4,12,1, 12,12,1,
              16,8,1, 16,0,1, -1};
int char_o[]={16,8,0, 16,4,1, 12,0,1, 4,0,1, 0,4,1, 0,8,1, 4,12,1, 12,12,1,
              16,8,1, -1};
int char_p[]={0,-6,0, 0,12,1, 12,12,1, 16,8,1, 16,4,1, 12,0,1, 0,0,1,-1};
int char_r[]={0,0,0, 0,12,1, 0,8,0, 4,12,1, 16,12,1, -1};
int char_s[]={0,3,0,  4,0,1, 12,0,1, 16,3,1, 0,9,1, 4,12,1, 12,12,1,
              16,9,1, -1};
int char_t[]={16,4,0, 12,0,1, 4,0,1, 4,20,1, 0,12,0, 16,12,1, -1};
int char_u[]={16,12,0, 16,0,1, 16,4,0, 12,0,1, 4,0,1,
              0,4,1, 0,12,1, -1};
int char_w[]={0,12,0, 4,0,1, 8,10,1, 12,0,1, 16,12,1, -1};
int char_x[]={0,12,0, 16,0,1, 0,0,0, 16,12,1, -1};
int char_y[]={16,4,0, 12,0,1, 4,0,1, 0,4,1, 0,12,1,
              16,12,0, 16,-4,1, 12,-8,1, 4,-8,1, -1};


gbr_char gbr_font[]={
{'A',char_A},
{'B',char_B},
{'C',char_C},
{'D',char_D},
{'E',char_E},
{'F',char_F},
{'G',char_G},
{'H',char_H},
{'I',char_I},
{'J',char_J},
{'K',char_K},
{'L',char_L},
{'M',char_M},
{'N',char_N},
{'O',char_O},
{'P',char_P},
{'Q',char_Q},
{'R',char_R},
{'S',char_S},
{'T',char_T},
{'U',char_U},
{'V',char_V},
{'W',char_W},
{'X',char_X},
{'Y',char_Y},
{'Z',char_Z},
{'0',char_0},
{'1',char_1},
{'2',char_2},
{'3',char_3},
{'4',char_4},
{'5',char_5},
{'6',char_6},
{'7',char_7},
{'8',char_8},
{'9',char_9},
{'-',char_minus},
{'=',char_eq},
{'+',char_plus},
{'_',char_uline},
{'/',char_slash},
{'.',char_point},
{'a',char_a},
{'b',char_b},
{'c',char_c},
{'d',char_d},
{'e',char_e},
{'g',char_g},
{'i',char_i},
{'k',char_k},
{'l',char_l},
{'m',char_m},
{'n',char_n},
{'o',char_o},
{'p',char_p},
{'r',char_r},
{'s',char_s},
{'t',char_t},
{'u',char_u},
{'w',char_w},
{'x',char_x},
{'y',char_y},
{0,NULL}
};

Outputs::Outputs()
{
  int i;
  for(i=0;i<OUTNUM;i++){
    Type[i]=0;
    Mirror[i]=false;
  }
  FillApr=100;
  XOffs=0;
  YOffs=0;
}

Outputs::~Outputs()
{
}

void Outputs::DoOutputs(PicGrp * Sch,PicGrp * Pcb, QString n, QString v)
{
  int i;
  QString s;
  QString s1;
  QString s2;
  Layers * Lay;
  unsigned long int L;
  PicGrp * D;
  QDir Dir;
  int fl=0;

  n.remove(".tdb");
  while(n.contains('/')) n.remove(0,n.indexOf('/')+1);

  s=OutDir;
  s.replace("%f",n);
  s.replace("%v",v);
  Dir.mkpath(s);

  for(i=0;i<OUTNUM;i++){
    fl=0;
    if(Type[i]){
      s=OutDir+FN[i];
      s.replace("%f",n);
      s.replace("%v",v);
      if(Sch){
	Lay=Sch->GetLayers();
	if(Ly[i].contains(',')) s1=Ly[i].left(Ly[i].indexOf(','));
	else s1=Ly[i];
	if(Lay->IsDef(&s1)) D=Sch;
	else {
          D=Pcb;
	  fl=1;
	}
      }
      else {
        D=Pcb;
	fl=1;
      }
      if(D){
        Lay=D->GetLayers();
	s1=Ly[i];
	L=0;
	while(s1.contains(',')){
	  s2=s1.left(s1.indexOf(','));
	  if(Lay->IsDef(&s2)) L|=Lay->GetLy(&s2);
	  s1.remove(0,s1.indexOf(',')+1);
	}
	if(Lay->IsDef(&s1)) L|=Lay->GetLy(&s1);
	if(Mirror[i]){
	  D->SetMirror(1-D->GetMirror());
	  L=Lay->MirLy(L);
	}
	if(Type[i]==OUTPDF) PdfOutput(D,L,s,fl);
	else if(Type[i]==OUTGERBER) GerberOutput(D,L,s);
	else if(Type[i]==OUTEXCELON) ExcelonOutput(D,L,s);
	else if(Type[i]==OUTPARTL){
	  if(Sch && (Sch->GetIden()==kDbSch))  PartLOutput((DbSch *)Sch,Pcb,L,s);
	}
	else if(Type[i]==OUTPLTOP){
	  if(Sch && (Sch->GetIden()==kDbSch))  PlaceOutput((DbSch *)Sch,Pcb,L,s,1);
	}
	else if(Type[i]==OUTPLBOT){
	  if(Sch && (Sch->GetIden()==kDbSch))  PlaceOutput((DbSch *)Sch,Pcb,L,s,2);
	}
	else if(Type[i]==BOMJLC){
	  if(Sch && (Sch->GetIden()==kDbSch))  BomOutput((DbSch *)Sch,Pcb,L,s);
	}
	else if(Type[i]==CPLJLC){
	  if(Sch && (Sch->GetIden()==kDbSch))  CplOutput((DbSch *)Sch,Pcb,L,s);
	}

	if(Mirror[i]) D->SetMirror(1-D->GetMirror());
	D->UpdORect();
      }
    }
  }
  if(!DescFile.isEmpty()) DescOutput(n,v);
}

void Outputs::DescOutput(QString n,QString v)
{
  QString s;
  int i;
  s=OutDir+DescFile;
  s.replace("%f",n);
  s.replace("%v",v);
  printf("DescOutput %s\n",s.toLatin1().data());
  QFile F(s);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);
  
  S << "NAME          : " << n.toUpper() << "\n";
  s.sprintf("%3.1fmm x %3.1fmm\n",(double)(MaxX-MinX)*0.00254,(double)(MaxY-MinY)*0.00254);
  S << "SIZE          : " << s;
  S << "THICKNESS     : " << Th <<"mm\n";
  S << "BASE MATERIAL : " << Mat <<"\n";
  S << "CU THICKNESS  : " << Cu <<"um\n\n";
  S << "FILES:\n";
  for(i=0;i<OUTNUM;i++){
    if(!Desc[i].isEmpty()){
      s=FN[i];
      s.replace("%f",n);
      s.replace("%v",v);
      S <<" "<< s << " : " << Desc[i] <<"\n";
    }
  }
  F.close();
}

void Outputs::PartLOutput(DbSch *D,PicGrp * Pcb,unsigned long int  L,QString n)
{
  printf("Part List to %s \n",n.toLatin1().data());
  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  Layers * Lay=Pcb->GetLayers();
  Lay->Store();
  Lay->SetAct((unsigned long int)0);
  Lay->SetVis(L);

  Part * P=new Part;
//  P->SetFName(fname);
  D->GetPart(P);
  if(Pcb->GetIden()==kDbPcb){
    ((DbPcb *)Pcb)->GetPart(P);
    P->AddCenter((DbPcb *)Pcb,XOffs,YOffs);
  }
  else{
    D->GetPart(P);
  }
  P->SetType(1);
  P->Sort();
//  P->List();
  P->WriteCsv(S,0);
  S<<"\n\n";

  P->SetType(0);
  P->Sort();
//  P->List();
  P->WriteCsv(S,0);
  S<<"\n\n";

  P->SetType(4);
  P->Sort();
//  P->List();
  P->WriteCsv(S,0);
  Lay->ReStore();
  F.close();
}

void Outputs::BomOutput(DbSch *D,PicGrp * Pcb,unsigned long int  L,QString n)
{
  printf("BOM to %s \n",n.toLatin1().data());
  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  Layers * Lay=Pcb->GetLayers();
  Lay->Store();
  Lay->SetAct((unsigned long int)0);
  Lay->SetVis(L);

  Part * P=new Part;
//  P->SetFName(fname);
  D->GetPart(P);
  if(Pcb->GetIden()==kDbPcb){
    ((DbPcb *)Pcb)->GetPart(P);
    P->AddCenter((DbPcb *)Pcb,XOffs,YOffs);
  }
  else{
    D->GetPart(P);
  }
  P->SetType(1);
  P->Sort();
//  P->List();
  P->SetType(5);
  P->WriteCsv(S,0);
  Lay->ReStore();
  F.close();
}

void Outputs::CplOutput(DbSch *D,PicGrp * Pcb,unsigned long int  L,QString n)
{
  printf("BOM to %s \n",n.toLatin1().data());
  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  Layers * Lay=Pcb->GetLayers();
  Lay->Store();
  Lay->SetAct((unsigned long int)0);
  Lay->SetVis(L);

  Part * P=new Part;
//  P->SetFName(fname);
  D->GetPart(P);
  if(Pcb->GetIden()==kDbPcb){
    ((DbPcb *)Pcb)->GetPart(P);
    P->AddCenter((DbPcb *)Pcb,XOffs,YOffs);
  }
  else{
    D->GetPart(P);
  }
  P->SetType(0);
  P->Sort();
//  P->List();
  P->SetType(6);
  P->WriteCsv(S,0);
  Lay->ReStore();
  F.close();
}

void Outputs::PlaceOutput(DbSch *D,PicGrp * Pcb,unsigned long int  L,QString n, int fl)
{
  printf("Place to %s \n",n.toLatin1().data());
  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  Layers * Lay=Pcb->GetLayers();
  Lay->Store();
  Lay->SetAct((unsigned long int)0);
  Lay->SetVis(L);

  Part * P=new Part;
//  P->SetFName(fname);
  D->GetPart(P);
  if(Pcb->GetIden()==kDbPcb){
    ((DbPcb *)Pcb)->GetPart(P);
    P->AddCenter((DbPcb *)Pcb,XOffs,YOffs);
  }
  else{
    D->GetPart(P);
  }

  P->SetType(0);
  P->Sort();
//  P->List();
  P->WriteCsv(S,fl);
  S<<"\n";

  Lay->ReStore();
  F.close();
}

void Outputs::PdfOutput(PicGrp *D,unsigned long int L,QString n,int fl)
{
  printf("PdfOutput to %s layers %lx\n",n.toLatin1().data(),L);

  QPrinter prt;
  Layers * Lay=D->GetLayers();
  QPainter p;
  QRect r;
  QRect r1;
  int width=595;
  int height=841;

  int ID;
  QPoint P1;
  QPoint P2;
  int W;
  int R;
  int Lo;
  QPainterPath Po;
  QString N;
  PicEl * E;
  bool pad;

  int XPos;
  int YPos;

  float scale;
  
  int dfl=kDrawAct | kDrawOth | kDrawFillPoly;

  int pn=0;
  unsigned long int L1=Lay->GetPrint();

  Lay->Store();
  Lay->SetAct((unsigned long int)0);
  Lay->SetVis(L);

  D->UpdORect();
  D->GetORect(&r1);

  prt.setColorMode(QPrinter::Color);
  prt.setOutputFileName(n);
//  prt.setOutputToFile(true);
  prt.setOutputFormat(QPrinter::PdfFormat);
  prt.setOrientation(QPrinter::Landscape);

  if(!p.begin(&prt)) return;

  E=D->GetEl(ID,P1,P2,W,R,Lo,N,Po,L1,pad);
  while(E||(pn==0)){
   if((pn==0)||(ID==kPicRect)){
     if(pn>0) prt.newPage();

     if(E){
       r1.setTopLeft(P1);
       r1.setBottomRight(P2);
       r1=r1.normalized();
     }

     if(r1.width()>=r1.height()) prt.setOrientation(QPrinter::Landscape);
     else prt.setOrientation(QPrinter::Portrait);

     printf("Resolution %d\n",prt.resolution());

     p.resetTransform();
     r=p.viewport();
     printf("Wiewport x=%d y=%d w=%d h=%d\n",r.left(),r.top(),r.width(),r.height());

     width=r.width();
     height=r.height();

     if(fl){
       ((DbPcb*)D)->SelXPartL();
     }  
     else {    
       ((DbSch*)D)->SelXPartL();
       dfl|= kDrawMirrorTxt;
     }


     if ((float)width/r1.width()<(float)height/r1.height()){
      scale=(float)width/r1.width();
     } else {
      scale=(float)height/r1.height();
     }
   //   scale=scale*0.9;
   //   if(scale<0.03) scale=0.03;
     if(fl && (scale >0.03)) scale=0.03;
     if(scale>1) scale=1;
     QPoint p1=r1.center();
     XPos=-p1.x()+(int)((width/2)/scale);
     YPos=-p1.y()+(int)((height/2)/scale);

     printf("Scale %f X %d Y %d x %d y %d\n",scale,XPos,YPos,r1.x(),r1.y());

     p.scale(scale,scale);
     r=r1;
     r.moveTo(r1.x()+XPos,r1.y()+YPos);
     p.setClipRect(r);

 //    if(fl){
       Lay->SetAlpha(47);
       D->DrawTo(&p,XPos,YPos,0,kDrawSel | kDrawFillPoly | kDrawMirrorTxt);
       printf("Draw Sel\n");
       Lay->SetAlpha(255);
 //    }
     Lay->SetVis(L & (Lay->GetAtrLy()^AllLy));
     D->DrawTo(&p,XPos,YPos,0,dfl);
     Lay->SetVis(L & Lay->GetAtrLy());
     D->DrawTo(&p,XPos,YPos,0,dfl);
     
   }
   pn++;
   if(E) E=D->GetNextEl(ID,P1,P2,W,R,Lo,N,Po,L1,pad);
  }

  p.end();
  D->DeSelect();
  Lay->ReStore();
}

#define MAXAPR 990

#define APROUND 1
#define APROUND1 2
#define APSQUARE 3

void Outputs::AddApr(Aperture * apr,int & num,int type,int x,int y)
{
  int i;
  if(x<0) x=-x;
  if(y<0) y=-y;
  for(i=0;i<num;i++){
    if((apr[i].t==type) && (apr[i].x==x) && (apr[i].y==y)) return;
  }
  apr[num].t=type;
  apr[num].x=x;
  apr[num].y=y;
  num++;
}

int Outputs::FindApr(Aperture * apr,int & num,int type,int x,int y)
{
  int i;
  if(x<0) x=-x;
  if(y<0) y=-y;
  for(i=0;i<num;i++){
    if((apr[i].t==type) && (apr[i].x==x) && (apr[i].y==y)) return i+10;
  }
  return 0;
}

void Outputs::WriteApr(Aperture * apr,int num,QTextStream & S)
{
  double x,y;
  int i;
  
  for(i=0;i<num;i++){
    x=apr[i].x/10000.0;
    y=apr[i].y/10000.0;
    if(apr[i].t==APROUND) 
       S << "%ADD" << i+10 << "C," << x << "*%\n";
    if(apr[i].t==APROUND1) 
       S << "%ADD" << i+10 << "C," << x << "X" << y << "*%\n";
    if(apr[i].t==APSQUARE) 
       S << "%ADD" << i+10 << "R," << x << "X" << y << "*%\n";
  }
}

void Outputs::WriteAprEx(Aperture * apr,int num,QTextStream & S)
{
  double x;
  int i;
  QString s;
  for(i=0;i<num;i++){
    x=apr[i].x/10000.0;
    if(apr[i].t==APROUND){
      s.sprintf("T%02dC%5.4f\n",i+10,x);
      S << s;
    }
  }
}

void Outputs::SortApr(Aperture * apr,int & num)
{
  int i;
  int j;
  int t;
  for(i=0;i<num-1;i++){
    for(j=0;j<num-i-1;j++){
      if(apr[j].x>apr[j+1].x){
        t=apr[j].t;
	apr[j].t=apr[j+1].t;
	apr[j+1].t=t;

        t=apr[j].x;
	apr[j].x=apr[j+1].x;
	apr[j+1].x=t;

        t=apr[j].y;
	apr[j].y=apr[j+1].y;
	apr[j+1].y=t;
      }
    }
  }
}

bool Outputs::XPoint(QPoint p1, QPoint p2, QPoint p3, QPoint p4, QPoint &x)
{
  double Ax,Ay,Bx,By,Cx,Cy,Dx,Dy;

  double t;
  double t1;

  double d;
  double d1;

  Ax=p1.x();
  Ay=p1.y();

  Bx=p2.x();
  By=p2.y();

  Cx=p3.x();
  Cy=p3.y();

  Dx=p4.x();
  Dy=p4.y();

  if(fabs((Dx-Cx)*(By-Ay)-(Dy-Cy)*(Bx-Ax))< 0.0000001){
    if((Ay==Cy) && (By==Dy) && (Ay==By)){
      if(Ax>Bx){
        t=Ax;
	Ax=Bx;
	Bx=t;
      }
      if(Cx>Dx){
        t=Cx;
	Cx=Dx;
	Dx=t;
      }
      if((Cx>Bx) || (Ax>Dx)) return false;
      if(Ax>Cx) t=Ax;
      else t=Cx;
      if(Bx<Dx) t1=Bx;
      else t1=Dx;
      x.setX((t+t1)/2);
      x.setY(Ay);
      return true;
    }
    return false;
  }  


  t1=((Cy-Ay)*(Bx-Ax)+(Ax-Cx)*(By-Ay))/((Dx-Cx)*(By-Ay)-(Dy-Cy)*(Bx-Ax));

  if (fabs(Bx-Ax)>fabs(By-Ay)) t=(Cx-Ax+t1*(Dx-Cx))/(Bx-Ax);
  else t=(Cy-Ay+t1*(Dy-Cy))/(By-Ay);

  d=1/(fabs(Ax-Bx)+fabs(Ay-By));
  d1=1/(fabs(Cx-Dx)+fabs(Cy-Dy));

  if (t1>=-d1 && t1<=(1+d1) && t>=-d && t<=(1+d)){
    x.setX(Ax+t*(Bx-Ax));
    x.setY(Ay+t*(By-Ay));
    return true;
  }
  return false;
}

/*
bool Outputs::FindSubPoly(QPainterPath &Po,int &i,int &j)
{
  QPoint p;
  while(i<Po.elementCount()-1){
    j=i+2;
    while(j<Po.elementCount()){
      if(j==Po.elementCount()) p=Po.point(0);
      else p=Po.point(j+1);
      if((Po.point(i)==p) && (Po.point(i+1)==Po.point(j))){
        return true;
      }
      j++;
    }
    i++;
  }
  return false;
}

void  Outputs::SplitSubPoly(QPainterPath Po,PolyLst** PLst)
{
  int i,j;
  int ii;
  int f=1;
  PolyLst * PL=new PolyLst;
  PL->Next=NULL;
  PolyLst * PL1;
  PolyLst * PL2=PL;
  PL->P=Po;

  while(f){
    f=0;
    PL2=PL;
    while(PL2){
      i=0;
      while(FindSubPoly(PL2->P,i,j)){
 //       printf("Find Sub Poly %d %d\n",i,j);
        f=1;
	PL1=new PolyLst;

	PL1->Next=PL;
	PL=PL1;

	ii=i+1;
	while(ii<=j){
	  PL1->P.append(PL2->P.point(ii));
	  ii++;
	}
	PL2->P.remove(i,j-i+1);
      }
      PL2=PL2->Next;
    }
  }
  *PLst=PL;
}

void Outputs::ReducePoly(QPainterPath &Po, QPainterPath &SP, QPainterPath &DP)
{
  QPoint p,p1;
  double l;
  QPoint A,B,C,D,X;
  int i,j;
  int k=0;
//  printf("ReducePoly size=%d\n",SP.size());
  
  p=SP.point(SP.size()-1);
  for(i=-1;i<SP.size();i++){
    if((i+1)==SP.size()) p1=SP.point(0);
    else p1=SP.point(i+1);
    if(i>=0) p=SP.point(i);
    l=sqrt(((double)(p1-p).x())*((p1-p).x())+((double)(p1-p).y())*((p1-p).y()));
    if(i<0){
      j=SP.size()-2;
      while((l<0.00001) && j){
        p=SP.point(j);
        l=sqrt(((double)(p1-p).x())*((p1-p).x())+((double)(p1-p).y())*((p1-p).y()));
        j--;
      }
    }
    if(l>0.00001){
      A.setX((p1.x()+p.x())/2+3*(p1.y()-p.y())/l);
      A.setY((p1.y()+p.y())/2-3*(p1.x()-p.x())/l);
      if(Po.containsPoint(A,Qt::OddEvenFill)){
	A.setX(p.x()-FillApr*0.5*(p1.x()-p.x())/l+FillApr*0.5*(p1.y()-p.y())/l);
	A.setY(p.y()-FillApr*0.5*(p1.x()-p.x())/l-FillApr*0.5*(p1.y()-p.y())/l);
	B.setX(p1.x()+FillApr*0.5*(p1.x()-p.x())/l+FillApr*0.5*(p1.y()-p.y())/l);
	B.setY(p1.y()-FillApr*0.5*(p1.x()-p.x())/l+FillApr*0.5*(p1.y()-p.y())/l);
      }
      else{
	A.setX(p.x()-FillApr*0.5*(p1.x()-p.x())/l-FillApr*0.5*(p1.y()-p.y())/l);
	A.setY(p.y()+FillApr*0.5*(p1.x()-p.x())/l-FillApr*0.5*(p1.y()-p.y())/l);
	B.setX(p1.x()+FillApr*0.5*(p1.x()-p.x())/l-FillApr*0.5*(p1.y()-p.y())/l);
	B.setY(p1.y()+FillApr*0.5*(p1.x()-p.x())/l+FillApr*0.5*(p1.y()-p.y())/l);
      }
//      printf("ReducePoly p=%d,%d p1=%d,%d l=%3.2f A=%d,%d  D=%d,%d\n",
//                   p.x(),p.y(),p1.x(),p1.y(),l,A.x(),A.y(),D.x(),D.y());
      if(k){
        if(XPoint(A,B,C,D,X)){
          DP.append(X);
	}
	else{
          DP.append(D);
          DP.append(A);
	}
//	DP.append(D);
//	DP.append(A);
      }
      k=1;
      C=A;
      D=B;
    }
  }
  p=DP.point(DP.size()-1);
  for(i=0;i<DP.size();i++){
    if(p==DP.point(i)){
      DP.remove(i);
      i--;
    }
    else p=DP.point(i);
  }

  A=DP.point(DP.size()-1);
  for(i=0;i<DP.size()-1;i++){
    B=DP.point(i);
    j=i+DP.size()/2-1;
    while(j>i){
      if(j<DP.size()) C=DP.point(j);
      else C=DP.point(j-DP.size());
      if((j+1)<DP.size()) D=DP.point(j+1);
      else D=DP.point(j+1-DP.size());
      if(XPoint(A,B,C,D,X)){
        DP.replace(i,X);
	if(j<DP.size()) DP.remove(i+1,j-i);
	else{
	  if((i+1)<DP.size()-1) DP.remove(i+1,DP.size()-1-i);
	  DP.remove(0,j-DP.size()+1);
	  i-=j-DP.size()+1;
	}
	
        j=0;
      }
      j--;
    }
    A=DP.point(i);
  }

}
*/

void Outputs::FillPoly(QPainterPath &Po,QTextStream &S,int W,Aperture * apr,int a)
{
  int i;
  QPointF p;
//  QPainterPath P;
  int y;
  int x[200];
  int j,t;
  int e;
  int cnt;
  QPointF A,B,C,D;
  QPoint X;
  
  
  
  if(W){
//    SplitSubPoly(Po,&PL);
/*
  PL1=PL;
  
  while(PL1){
    ReducePoly(Po,PL1->P,P);
    PL1->P=P;
    P.clear();
    PL1=PL1->Next;
  }
*/

    i=FindApr(apr,a,APROUND,W,0);
    S<<"G54D"<<i<<"*";
  
    for(i=0;i<Po.elementCount();i++){
      p=Po.elementAt(i);
      if(Po.elementAt(i).isMoveTo()) S<<"X"<<p.x()-XOffs<<"Y"<<-p.y()-YOffs<<"D02*\n";
      else S<<"X"<<p.x()-XOffs<<"Y"<<-p.y()-YOffs<<"D01*";
    }
  }
/*  p=Po.point(Po.size()-1);
  S<<"X"<<p.x()<<"Y"<<-p.y()<<"D02*\n";
  for(i=0;i<Po.size();i++){
    p=Po.point(i);
    S<<"X"<<p.x()<<"Y"<<-p.y()<<"D01*";
  }*/

  if((FillApr) && (W)){
    i=FindApr(apr,a,APROUND,FillApr,0);
    S<<"G54D"<<i<<"*";
  
    y=Po.boundingRect().top();
    C.setX(Po.boundingRect().left()-10000);
    D.setX(Po.boundingRect().right()+10000);
    while(y<Po.boundingRect().bottom()){
      cnt=0;
      C.setY(y);
      D.setY(y);

      e=Po.elementCount();
//      A=Po.elementAt(0);
      for(i=0;i<e;i++){
        B=Po.elementAt(i);
	if(Po.elementAt(i).isLineTo()){
	  if((A.y()==B.y())){
//	    if(fabs(y-A.y())<2){
//	      x[cnt]=A.x();
//	      cnt++;
//	    }
	  }
	  else if(((A.y()<=y)&&(B.y()>y))||((A.y()>y)&&(B.y()<=y))){
	    if(XPoint(A.toPoint(),B.toPoint(),C.toPoint(),D.toPoint(),X)){
	      x[cnt]=X.x();
	      cnt++;
	    }
	  }
	}
        A=Po.elementAt(i);
      }

      for(i=0;i<cnt-1;i++){
	for(j=i+1;j<cnt;j++){
          if(x[i]>x[j]){
	    t=x[i];
	    x[i]=x[j];
	    x[j]=t;
	  }
	}
      }
/*
      for(i=0;i<(cnt-1);i++){
        if(fabs(x[i+1]-x[i])<2){
	  printf("FILL remove in %d: %d %d \n",y,x[i],x[i+1]);
	  for(j=i+1;j<(cnt-1);j++){
	    x[j]=x[j+1];
	  }
	  cnt--;
	}
      }
*/
      i=0;
      while(i<cnt){
	if((i==cnt-1) || ((x[i+1]-x[i])>(FillApr-10))){
	  S<<"X"<<x[i]-XOffs<<"Y"<<-y-YOffs<<"D02*\n";
	  i++;
	  if(i<cnt) S<<"X"<<x[i]-XOffs<<"Y"<<-y-YOffs<<"D01*";
	  else S<<"X"<<D.x()-XOffs<<"Y"<<-y-YOffs<<"D01*";
	}
	else i++;
	i++;
      }
      if(cnt%2){
	printf("FILL Error in %d ",y);
	for(i=0;i<cnt;i++){
          printf("%d ",x[i]);
	}
	printf("\n");
      }
      y+=FillApr-10;
    }
  }
  else{
    S<<"G36*\n";
    for(i=0;i<Po.elementCount();i++){
      p=Po.elementAt(i);
      if(Po.elementAt(i).isMoveTo()) S<<"X"<<p.x()-XOffs<<"Y"<<-p.y()-YOffs<<"D02*\n";
      else S<<"X"<<p.x()-XOffs<<"Y"<<-p.y()-YOffs<<"D01*";
    }
    S<<"G37*\n";
  }
}

void Outputs::GerberOutput(PicGrp *D,unsigned long int L,QString n)
{
  printf("GerberOutput to %s layers %lx\n",n.toLatin1().data(),L);

  int ID;
  QPoint P1;
  QPoint P2;
  int W;
  int R;
  int Lo;
  QPainterPath Po;
  QString N;
  PicEl * E;
  bool pad;

  int i;
  int la;
  
  Aperture apr[MAXAPR];
  int a=0;

  QPoint p;
  
  int x=0;
  int y;

  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  E=D->GetEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  while(E){
   if((ID==kPicRect) || (ID==kPicLine)){
     if(x==0){
       if(P1.x()<P2.x()){
	 MinX=P1.x();
	 MaxX=P2.x();
       }
       else{
	 MinX=P2.x();
	 MaxX=P1.x();
       }
       if(P1.y()<P2.y()){
	 MinY=P1.y();
	 MaxY=P2.y();
       }
       else{
	 MinY=P2.y();
	 MaxY=P1.y();
       }
       x=1;
     }
     else{
       if(P1.x()<MinX) MinX=P1.x();
       if(P2.x()<MinX) MinX=P2.x();
       if(P1.x()>MaxX) MaxX=P1.x();
       if(P2.x()>MaxX) MaxX=P2.x();
       if(P1.y()<MinY) MinY=P1.y();
       if(P2.y()<MinY) MinY=P2.y();
       if(P1.y()>MaxY) MaxY=P1.y();
       if(P2.y()>MaxY) MaxY=P2.y();
     }
   }
   if(ID==kPicRect){
     if(W<1000) AddApr(apr,a,APROUND,W,0);
     else AddApr(apr,a,APSQUARE,P2.x()-P1.x(),P2.y()-P1.y());
   }
   else if(ID==kPicCir){
     if(W<1000) AddApr(apr,a,APROUND1,R*2+W,2*R-W);
     else AddApr(apr,a,APROUND,2*R,0);
   }
   else if(ID==kPicLine){
     AddApr(apr,a,APROUND,W,0);
   }
   else if(ID==kPicText){
     if(W<500) AddApr(apr,a,APROUND,50,0);
     else AddApr(apr,a,APROUND,W/10,0);
   }
   else if(ID==kPicPoly){
     if(FillApr) AddApr(apr,a,APROUND,FillApr,0);
     if(W) AddApr(apr,a,APROUND,W,0);
     if(W && (FillApr>W)){
      gErr.Err("Fill apperture is bigger then polygon width in "+n);
      F.close();
      return;
     }
   }
   E=D->GetNextEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  }
  S << "%FSLAX24Y24*%\n";
  S << "%MOIN*%\n";
  S << "G01*\n";
  printf("GerberOutput use %d apertures\n",a);
  WriteApr(apr,a,S);

  x=0;
  y=0;
  la=0;
  
  E=D->GetEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  while(E){
   if(ID==kPicRect){
     if(W<1000){
       i=FindApr(apr,a,APROUND,W,0);
       if(i!=la){
         S<<"G54D"<<i<<"*";
	 la=i;
       }
       S<<"X"<<P1.x()-XOffs<<"Y"<<-P1.y()-YOffs<<"D02*";
       S<<"X"<<P1.x()-XOffs<<"Y"<<-P2.y()-YOffs<<"D01*";
       S<<"X"<<P2.x()-XOffs<<"Y"<<-P2.y()-YOffs<<"D01*";
       S<<"X"<<P2.x()-XOffs<<"Y"<<-P1.y()-YOffs<<"D01*";
       S<<"X"<<P1.x()-XOffs<<"Y"<<-P1.y()-YOffs<<"D01*\n";
     }
     else{
       i=FindApr(apr,a,APSQUARE,P2.x()-P1.x(),P2.y()-P1.y());
       if(i!=la){
         S<<"G54D"<<i<<"*";
	 la=i;
       }
       S<<"X"<<(P1.x()+P2.x())/2-XOffs<<"Y"<<-(P1.y()+P2.y())/2-YOffs<<"D03*\n";
     }
   }
   else if(ID==kPicCir){
     if(R>2){
       if(W<1000) i=FindApr(apr,a,APROUND1,R*2+W,2*R-W);
       else i=FindApr(apr,a,APROUND,2*R,0);
       if(i!=la){
	 S<<"G54D"<<i<<"*";
	 la=i;
       }
       S<<"X"<<P1.x()-XOffs<<"Y"<<-P1.y()-YOffs<<"D03*\n";
     }
   }
   else if(ID==kPicLine){
     i=FindApr(apr,a,APROUND,W,0);
     if(i!=la){
       S<<"G54D"<<i<<"*";
       la=i;
     }
     S<<"X"<<P1.x()-XOffs<<"Y"<<-P1.y()-YOffs<<"D02*";
     S<<"X"<<P2.x()-XOffs<<"Y"<<-P2.y()-YOffs<<"D01*\n";
   }
   else if(ID==kPicPoly){
     FillPoly(Po,S,W,apr,a);
     la=0;
   }
   else if(ID==kPicText){
     if(!N.isEmpty()){
       int xx,xy,yx,yy;
       int xo,yo;
       int j,k;
       int o;
       if(W<500) i=FindApr(apr,a,APROUND,50,0);
       else i=FindApr(apr,a,APROUND,W/10,0);
       if(i!=la){
	 S<<"G54D"<<i<<"*";
	 la=i;
       }
       p=P2-P1;
       if(p.x()>=0) x=p.x();
       else x=-p.x();
       if(p.y()>=0) y=p.y();
       else y=-p.y();

       if((x*3)>(y*2)){
	 xy=0;
	 yx=0;
	 xx=p.x()/(N.size()*24);
	 yy=-p.y()/25;
	 o=1;
       }
       else{
	 xy=-p.x()/25;
	 yx=+p.y()/(N.size()*24);
	 xx=0;
	 yy=0;
	 o=0;
       }

       for(i=0;i<N.size();i++){
	 if(o){
	   xo=P1.x()+i*p.x()/N.size()+p.x()/N.size()/5;
	   yo=P1.y()+p.y()*21/25;
	 }
	 else{
	   xo=P1.x()+p.x()*21/25;
	   yo=P1.y()+i*p.y()/N.size()+p.y()/N.size()/5;
	 }
	 j=0;
	 while(gbr_font[j].c){
           if(gbr_font[j].c==N.at(i).toLatin1()){
  //           printf("GerberOutput found char %d at %d %c %c\n",i,j,gbr_font[j].c,
  //	           N.at(j).ascii());
	     k=0;
	     while(gbr_font[j].data[k]>=0){
	       x=gbr_font[j].data[k]*xx+gbr_font[j].data[k+1]*xy+xo;
	       y=gbr_font[j].data[k]*yx+gbr_font[j].data[k+1]*yy+yo;
	       S<<"X"<<x-XOffs<<"Y"<<-y-YOffs<<"D0"<<2-gbr_font[j].data[k+2]<<"*";
	       k+=3;
	     }
             S<<"\n";
	     break;
	   }
	   j++;
	 }
       }
  //     printf("GerberOutput text %s\n",N.ascii());
     } 
   }
//   printf("get next el\n");
   E=D->GetNextEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  }
  S << "G02*M02*\n";
  F.close();
}

void Outputs::ExcelonOutput(PicGrp *D,unsigned long int L,QString n)
{
  printf("ExcelonOutput to %s layers %lx\n",n.toLatin1().data(),L);
  int ID;
  QPoint P1;
  QPoint P2;
  int W;
  int R;
  int Lo;
  QPainterPath Po;
  QString N;
  PicEl * E;
  bool pad;

  int i;
  
  Aperture apr[MAXAPR];
  int a=0;
  int mill=-1;

  QPoint p;
  
  QString s;

  QFile F(n);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);

  E=D->GetEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  while(E){
   if(ID==kPicCir){
     AddApr(apr,a,APROUND,2*R,0);
   }
   if(ID==kPicLine){
     AddApr(apr,a,APROUND,W,0);
   }
   E=D->GetNextEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
  }
  S << "M48\n";
  S << "M72\n";
  S << "INCH,TZ\n";
  printf("ExcelonOutput use %d tools\n",a);
  SortApr(apr,a);
  WriteAprEx(apr,a,S);
  S << "%\n";
  for(i=10;i<a+10;i++){
    S << "T" <<i <<"\n";
    E=D->GetEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
    while(E){
     if(ID==kPicCir){
       if(i==FindApr(apr,a,APROUND,2*R,0)){
	 if(mill!=0){
           S << "G05\n";
	   mill=0;
	 }
         s.sprintf("X%05dY%05d\n",P1.x()-XOffs,-P1.y()-YOffs);
         S << s;
       }
     }
     if(ID==kPicLine){
       if(i==FindApr(apr,a,APROUND,W,0)){
	 if(mill!=1){
	   mill=1;
	 }
         S << "G00\n";
         s.sprintf("X%05dY%05d\n",P1.x()-XOffs,-P1.y()-YOffs);
         S << s;
	 S << "M15\n";
         s.sprintf("X%05dY%05d\n",P2.x()-XOffs,-P2.y()-YOffs);
         S << s;
	 S << "M16\n";
       }
     }
     E=D->GetNextEl(ID,P1,P2,W,R,Lo,N,Po,L,pad);
    }
  }
  S << "T00\nM30\n";
  F.close();
}

int Outputs::GetType(int i)
{
  return Type[i];
}

bool Outputs::GetMirror(int i)
{
  return Mirror[i];
}

QString Outputs::GetLayers(int i)
{
  return Ly[i];
}

QString Outputs::GetFName(int i)
{
  return FN[i];
}

QString Outputs::GetDesc(int i)
{
  return Desc[i];
}

QString Outputs::GetOutDir()
{
  return OutDir;
}

void Outputs::GetPCB(QString & th,QString & cu,QString & mat)
{
  th=Th;
  cu=Cu;
  mat=Mat;
}

int Outputs::GetFillApr()
{
  return FillApr;
}
QString Outputs::GetDescFile()
{
  return DescFile;
}

int Outputs::GetXOffs()
{
  return XOffs;
}
int Outputs::GetYOffs()
{
  return YOffs;
}

void Outputs::SetOffs(int x,int y)
{
  XOffs=x;
  YOffs=y;
}

void Outputs::SetType(int i,int t)
{
  Type[i]=t;
}
void Outputs::SetOutDir(QString OD)
{
  OutDir=OD;
}
void Outputs::SetFName(int i,QString n)
{
  FN[i]=n;
}
void Outputs::SetDesc(int i,QString n)
{
  Desc[i]=n;
}
void Outputs::SetLayers(int i,QString l)
{
  Ly[i]=l;
}
void Outputs::SetMirror(int i,bool m)
{
  Mirror[i]=m;
}
void Outputs::SetPCB(QString th,QString cu,QString mat)
{
  Th=th;
  Cu=cu;
  Mat=mat;
}
void Outputs::SetFillApr(int a)
{
  FillApr=a;
}
void Outputs::SetDescFile(QString n)
{
  DescFile=n;
}

void Outputs::TextSave(QTextStream * S)
{
  int i;
  *S << "{OutSetting\n";
  *S << " {OutDir " << OutDir << " }\n";
  *S << " {DescFile " << DescFile << " }\n";
  *S << " {FillApr " << FillApr << " }\n";
  *S << " {Offs " << XOffs << " " << YOffs << " }\n";
  *S << " {PCBParam " << Th << " " << Cu << " " << Mat << " }\n";
  for(i=0;i<OUTNUM;i++){
    if(Type[i]){
      *S << " {OutSet " << OutType[Type[i]] << " \n";
      *S << "  {Ly " << Ly[i] <<" }\n";
      *S << "  {Mir " << Mirror[i] <<" }\n";
      *S << "  {FName " << FN[i] <<" }\n";
      *S << "  {Desc " << Desc[i] <<" }\n";
      *S << " }\n";
    } 
  }
  *S << "}\n"; 
}

void Outputs::TextLoad(QTextStream * S)
{
  QString s;
  int i;
  int l=0;
  *S >> s;
  while (!s.contains("}")){
    if(s.contains("OutSet")){
      *S >> s;
      i=0;
      while(OutType[i]){
        if(s.contains(OutType[i])) Type[l]=i;
        i++;
      }
      *S >> s;
      while (!s.contains("}")){
	if (s.contains("Ly")){
	  *S >> s;
	  if(!s.contains("}")){
	    Ly[l]=s;
	    *S >> s;
	  }  
	}
	if (s.contains("FName")){
	  *S >> s;
	  if(!s.contains("}")){
	    FN[l]=s;
	    *S >> s;
	  }
	}
	if (s.contains("Desc")){
	  *S >> s;
	  Desc[l]="";
	  while(!s.contains("}")){
	    Desc[l]+=s+" ";
	    *S >> s;
	  }
	}
	if (s.contains("Mir")){
	  *S >> s;
	  Mirror[l]=s.toInt();
	  *S >> s;
	}
	while (!s.contains("}")) *S >> s;
	*S >> s;
      }
      l++;
    }
    if(s.contains("OutDir")){
      *S >> s;
      if(!s.contains("}")){
	OutDir=s;
	*S >> s;
      }
    }
    if(s.contains("DescFile")){
      *S >> s;
      if(!s.contains("}")){
	DescFile=s;
	*S >> s;
      }
    }
    if(s.contains("FillApr")){
      *S >> s;
      if(!s.contains("}")){
	FillApr=s.toInt();
	*S >> s;
      }
    }
    if(s.contains("Offs")){
      *S >> s;
      if(!s.contains("}")){
	XOffs=s.toInt();
	*S >> s;
      }
      if(!s.contains("}")){
	YOffs=s.toInt();
	*S >> s;
      }
    }
    if(s.contains("PCBParam")){
      *S >> s;
      if(!s.contains("}")){
	Th=s;
	*S >> s;
      }
      if(!s.contains("}")){
	Cu=s;
	*S >> s;
      }
      if(!s.contains("}")){
	Mat=s;
	*S >> s;
      }
    }
    while (!s.contains("}")) *S >> s;
    *S >> s;
  }
  for(i=l;i<OUTNUM;i++){
    FN[i]="";
    Ly[i]="";
    Type[i]=0;
    Mirror[i]=0;
  }
  
}

//============================================================================


OutDlg::OutDlg(Outputs * O, QWidget *parent )
        : QDialog( parent)
{
  QVBoxLayout * MLayout;
  QHBoxLayout * BLayout;


  MLayout = new QVBoxLayout(this);
  MLayout->setSpacing(6);
  MLayout->setMargin(11);

  Out=O;

  QPushButton* Pb;
  QLabel* Lb;

  BLayout = new QHBoxLayout();
  BLayout->setSpacing(6);
  BLayout->setMargin(4);

  Lb=new QLabel("OUTPUT SETTINGS",this);
  BLayout->addWidget( Lb );

  QSpacerItem * spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding , QSizePolicy::Minimum);
  BLayout->addItem( spacer );

  Lb=new QLabel("Output Directory:",this);
  BLayout->addWidget( Lb );

  OutDir=new QLineEdit(this);
  BLayout->addWidget( OutDir );

  Lb=new QLabel("Desc. File:",this);
  BLayout->addWidget( Lb );

  DescFile=new QLineEdit(this);
  BLayout->addWidget( DescFile );


  MLayout->addLayout( BLayout );

  BLayout = new QHBoxLayout();
  BLayout->setSpacing(2);
  BLayout->setMargin(4);

  Lb=new QLabel("PCB PARAMETERS:   Thickness:",this);
  BLayout->addWidget( Lb );

  Th=new QLineEdit(this);
  BLayout->addWidget( Th );

  Lb=new QLabel("Material:",this);
  BLayout->addWidget( Lb );

  Mat=new QLineEdit(this);
  BLayout->addWidget( Mat );

  Lb=new QLabel("Cu:",this);
  BLayout->addWidget( Lb );

  Cu=new QLineEdit(this);
  BLayout->addWidget( Cu );

  BLayout->addSpacing( 100 );
  Lb=new QLabel("Fill Apr.:",this);
  BLayout->addWidget( Lb );

  FillApr=new QLineEdit(this);
  BLayout->addWidget( FillApr );

  Lb=new QLabel("Offset X:",this);
  BLayout->addWidget( Lb );

  XOffs=new QLineEdit(this);
  BLayout->addWidget( XOffs );

  Lb=new QLabel("Y:",this);
  BLayout->addWidget( Lb );

  YOffs=new QLineEdit(this);
  BLayout->addWidget( YOffs );

  MLayout->addLayout( BLayout );
  MLayout->addSpacing( 10 );

  QString s;
  int i;
  for(i=0;i<OUTNUM;i++){
    BLayout = new QHBoxLayout();
    BLayout->setSpacing(1);
    BLayout->setMargin(1);
    s.sprintf("%02d.  Type:",i+1);
    Lb=new QLabel(s,this);
    BLayout->addWidget( Lb );
    
    CBTyp[i]=new QComboBox(this);
    BLayout->addWidget( CBTyp[i] );
    CBTyp[i]->addItem("OFF");
    CBTyp[i]->addItem("PDF");
    CBTyp[i]->addItem("GERBER-274X");
    CBTyp[i]->addItem("EXCELON");
    CBTyp[i]->addItem("PART LIST csv");
    CBTyp[i]->addItem("PLACEMENT TOP");
    CBTyp[i]->addItem("PLACEMENT BOT");
    CBTyp[i]->addItem("BOM (JLC)");
    CBTyp[i]->addItem("CPL (JLC)");

    Lb=new QLabel("Layers:",this);
    BLayout->addWidget( Lb );

    Ly[i]=new QLineEdit(this);
    BLayout->addWidget( Ly[i] );

    CbMir[i]=new QCheckBox("Mirror",this);
    BLayout->addWidget( CbMir[i] );

    Lb=new QLabel("Out File:",this);
    BLayout->addWidget( Lb );

    FN[i]=new QLineEdit(this);
    BLayout->addWidget( FN[i] );

    Lb=new QLabel("Desc.:",this);
    BLayout->addWidget( Lb );

    Desc[i]=new QLineEdit(this);
    BLayout->addWidget( Desc[i] );
    
    MLayout->addLayout( BLayout );
  }

  BLayout = new QHBoxLayout();
  BLayout->setSpacing(6);
  BLayout->setMargin(11);

  Pb = new QPushButton( "Load", this );
  connect( Pb, SIGNAL(clicked()),this, SLOT(Load()) );
  BLayout->addWidget( Pb );

  Pb = new QPushButton( "Save", this );
  connect( Pb, SIGNAL(clicked()),this, SLOT(Save()) );
  BLayout->addWidget( Pb );

  spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding , QSizePolicy::Minimum);
  BLayout->addItem( spacer );

  Pb = new QPushButton( "Cancel", this );
  connect( Pb, SIGNAL(clicked()),this, SLOT(Cancel()) );
  BLayout->addWidget( Pb );

  Pb = new QPushButton( "Revert", this );
  connect( Pb, SIGNAL(clicked()),this, SLOT(Revert()) );
  BLayout->addWidget( Pb );

  Pb = new QPushButton( "OK", this );
  connect( Pb, SIGNAL(clicked()),this, SLOT(OK()) );
  BLayout->addWidget( Pb );

  MLayout->addLayout( BLayout );
  Revert();
}

OutDlg::~OutDlg()
{
}

void OutDlg::Set()
{
  int i;
  Out->SetOutDir(OutDir->text());
  Out->SetDescFile(DescFile->text());

  Out->SetPCB(Th->text(),Cu->text(),Mat->text());

  Out->SetFillApr(FillApr->text().toInt()*10);
  Out->SetOffs(XOffs->text().toInt()*10,YOffs->text().toInt()*10);

  for(i=0;i<OUTNUM;i++){
    Out->SetType(i,CBTyp[i]->currentIndex());
    Out->SetFName(i,FN[i]->text());
    Out->SetDesc(i,Desc[i]->text());
    Out->SetLayers(i,Ly[i]->text());
    Out->SetMirror(i,CbMir[i]->checkState());
  }
}

void OutDlg::OK()
{
  Set();
  close();
}

void OutDlg::Cancel()
{
  close();
}

void OutDlg::Revert()
{
  int i;
  QString s1,s2,s3;
  
  OutDir->setText(Out->GetOutDir());
  DescFile->setText(Out->GetDescFile());
  Out->GetPCB(s1,s2,s3);
  Th->setText(s1);
  Cu->setText(s2);
  Mat->setText(s3);
  s1.setNum(Out->GetFillApr()/10);
  FillApr->setText(s1);

  s1.setNum(Out->GetXOffs()/10);
  XOffs->setText(s1);
  s1.setNum(Out->GetYOffs()/10);
  YOffs->setText(s1);
  
  for(i=0;i<OUTNUM;i++){
    CBTyp[i]->setCurrentIndex(Out->GetType(i));
    if(Out->GetMirror(i)) CbMir[i]->setCheckState(Qt::Checked);
    else CbMir[i]->setCheckState(Qt::Unchecked);
    
    Ly[i]->setText(Out->GetLayers(i));
    FN[i]->setText(Out->GetFName(i));
    Desc[i]->setText(Out->GetDesc(i));
  }
}

void OutDlg::Save()
{
  QString s;
  s = QFileDialog::getSaveFileName(this, tr("Save File"),0,tr("*"));
  if (s.isEmpty()) return;

  Set();
  QFile F(s);
  F.open(QIODevice::WriteOnly);
  QTextStream S;
  S.setDevice(&F);
  
  Out->TextSave(&S);
  
  F.close();
}

void OutDlg::Load()
{
  QString s;
  int i;
  s = QFileDialog::getOpenFileName(this, tr("Open File"),0,tr("*"));
  if (s.isEmpty()) return;

  QFile F(s);
  F.open(QIODevice::ReadOnly);
  QTextStream S;
  S.setDevice(&F);
  
  S >> s;
  
  while(!s.isEmpty()){
    if (s.contains("OutSetting")){
      Out->TextLoad(&S);
      break;
    }
    else{
      i=1;
      while(i){
        S >> s;
        if(s.contains("{")) i++;
        if(s.contains("}")) i--;
        if(s.isEmpty()) break;
      }
    }
    S >> s;
  }
  
  F.close();
  Revert();
}


