/*
    This file is part of peda.
    Copyright (C) 2000-2010  Petr Porazil <porazil@volny.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "DRC.h"
#include "PicObj.h"

#include <math.h>
#include "Error.h"

extern Errors  gErr;

extern const char *sPic[];

DRC::DRC()
{
 LE=NULL;

 PP=4.89;
 PW=4.89;
 PO=4.89;
 WW=4.89;
 WO=4.89;
 OO=4.89;
}

DRC::~DRC()
{
 Clear();
}

void DRC::GetParam(double & pp,double & pw,double & po,
                   double & ww,double & wo,double & oo)
{
  pp=PP;
  pw=PW;
  po=PO;
  ww=WW;
  wo=WO;
  oo=OO;
}

void DRC::SetParam(double pp,double pw,double po,
                   double ww,double wo,double oo)
{
  PP=pp;
  PW=pw;
  PO=po;
  WW=ww;
  WO=wo;
  OO=oo;
}


bool DRC::Check(PicGrp * Db)
{
 DEl * L=LE;
 DEl * L1=LE;
 bool x=true;

 double d;
 double lim;

 QString s;
 int i=0;
 unsigned long int le=0;
 s="DRCERR";
 if (L) le=L->E->GetLayers()->GetLy(&s);

 while(L){
  if(!L->Lo) L->E->SetLayer(L->E->GetLayer() & (le^AllLy));
  L=L->Next;
 }

 L=LE;

 while(L){
  L1=L->Next;
  while(L1){
//   if(L->N!=L1->N || L->N.isEmpty()){
   if((L->N!=L1->N) || L->N.isEmpty()){
    d=Distance(L->Iden,L->P1,L->P2,L->W,L->R,L->Po,L1->Iden,L1->P1,L1->P2,L1->W,L1->R,L1->Po);
    if(L->N.isEmpty()){
      if(L1->N.isEmpty()) lim=OO;
      else if((L1->Iden==kPicLine) || (L1->Iden==kPicPoly)) lim=WO;
      else lim=PO;
    }
    else if((L->Iden==kPicLine) || (L->Iden==kPicPoly)){
      if(L1->N.isEmpty()) lim=WO;
      else if((L1->Iden==kPicLine) || (L1->Iden==kPicPoly)) lim=WW;
      else lim=PW;
    }
    else{
      if(L1->N.isEmpty()) lim=PO;
      else if((L1->Iden==kPicLine) || (L1->Iden==kPicPoly)) lim=PW;
      else lim=PP;
    }
    lim=lim*10;
    if(d<lim){
//     printf("DRC ERROR D:%d %s:%d,%d,W:%d,R:%d  %s:%d,%d\n",d,
//                    sPic[L->Iden],L->P1.x(),L->P1.y(),L->W,L->R,
//                    sPic[L1->Iden],L1->P1.x(),L1->P1.y());
     if(i<4){
      s.sprintf(" DRC ERROR D:%3.2f %s:%d,%d,W:%d,N:%s  %s:%d,%d,W:%d,N:%s",d,
                     sPic[L->Iden],L->P1.x(),-L->P1.y(),L->W,
                     L->N.toLatin1().data(),
                     sPic[L1->Iden],L1->P1.x(),-L1->P1.y(),L1->W,
                     L1->N.toLatin1().data());
      gErr.ToLst(s);
     }
     i++;
     x=false;
     if(!L->Lo) L->E->SetLayer(L->E->GetLayer() | le);
     else AddErr(Db,L,le);
     if(!L1->Lo) L1->E->SetLayer(L1->E->GetLayer() | le);
     else AddErr(Db,L1,le);
    }
    else{
//     printf("DRC OK D:%d %s:%d,%d,W:%d,R:%d  %s:%d,%d\n",d,
//                    sPic[L->Iden],L->P1.x(),L->P1.y(),L->W,L->R,
//                    sPic[L1->Iden],L1->P1.x(),L1->P1.y());
    }
   }
   L1=L1->Next;
  }
  L=L->Next;
 }
 if(i){
  s.sprintf(" %d DRC ERRORS",i);
  gErr.ToLst(s);
 }
 return x;
}


void DRC::AddErr(PicGrp * Db, DEl * L,unsigned long int le)
{
 PicEl * E=NULL;
 if(L && Db){
  if(L->Iden==kPicLine){
   E=new PicLine(Db->GetLayers());
   E->SetLayer(le);
   E->SetWidth(L->W);
   ((PicLine *)E)->Begin(L->P1);
   ((PicLine *)E)->End(L->P2);
  }
  else if(L->Iden==kPicCir){
   E=new PicCir(Db->GetLayers());
   E->SetLayer(le);
   E->SetWidth(L->W);
   ((PicCir *)E)->Circle(L->P1,L->R);
  }
  else if(L->Iden==kPicRect){
   int x1;
   int x2;
   int y1;
   int y2;

   if(L->P1.x()<L->P2.x()){
    x1=L->P1.x();
    x2=L->P2.x();
   }
   else{
    x1=L->P2.x();
    x2=L->P1.x();
   }

   if(L->P1.y()<L->P2.y()){
    y1=L->P1.y();
    y2=L->P2.y();
   }
   else{
    y1=L->P2.y();
    y2=L->P1.y();
   }

   E=new PicRect(Db->GetLayers());
   E->SetLayer(le);
   E->SetWidth(L->W);
   ((PicRect *)E)->Rect(x1,y1,x2-x1,y2-y1);
  }
  else{
   int x1;
   int x2;
   int y1;
   int y2;

   if(L->P1.x()<L->P2.x()){
    x1=L->P1.x();
    x2=L->P2.x();
   }
   else{
    x1=L->P2.x();
    x2=L->P1.x();
   }

   if(L->P1.y()<L->P2.y()){
    y1=L->P1.y();
    y2=L->P2.y();
   }
   else{
    y1=L->P2.y();
    y2=L->P1.y();
   }

   E=new PicRect(Db->GetLayers());
   E->SetLayer(le);
   E->SetWidth(1);
   ((PicRect *)E)->Rect(x1,y1,x2-x1,y2-y1);
  }
  if(E) Db->AddItem(E);
 }
}

double DRC::Distance(int IdenA,QPoint P1A,QPoint P2A,int WA,int RA,QPainterPath PoA,
                  int IdenB,QPoint P1B,QPoint P2B,int WB,int RB,QPainterPath PoB)
{
 double d;
 if(IdenA==kPicLine){
  if(IdenB==kPicLine){
   return DLinLin(P1A,P2A,P1B,P2B)-(WA+WB)/2;
  }
  else if(IdenB==kPicCir){
   d=DPointLin(P1B,P1A,P2A);
   d=d-WA/2-RB;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicRect){
   if(WB<2000){
    d=DLinRec(P1A,P2A,P1B,P2B);
    d-=WB/2;
   }
   else{
    d=DLinRecF(P1A,P2A,P1B,P2B);
   }
   d-=WA/2;
   return d;
  }
  else if(IdenB==kPicPoly){
   d=DLinPoly(P1A,P2A,PoB);
   d-=WA/2;
   d-=WB/2;
   return d;
  }
  else{
   d=DLinRecF(P1A,P2A,P1B,P2B);
   d-=WA/2;
   return d;
  }
 }
 else if(IdenA==kPicCir){
  if(IdenB==kPicLine){
   d=DPointLin(P1A,P1B,P2B);
   d=d-WB/2-RA;
   if(WA<2000) d=d-WA/2;
   return d;
  }
  else if(IdenB==kPicCir){
   d=DPointPoint(P1A,P1B);
   d=d-(RA+RB);
   if(WA<2000) d=d-WA/2;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicRect){
   if(WB<2000){
    d=DPointRec(P1A,P1B,P2B);
    d-=WB/2;
   }
   else{
    d=DPointRecF(P1A,P1B,P2B);
   }
   d-=RA;
   if(WA<2000) d=d-WA/2;
   return d;
  }
  else if(IdenB==kPicPoly){
   d=DPointPoly(P1A,PoB);
   d-=RA;
   if(WA<2000) d=d-WA/2;
   d-=WB/2;
   return d;
  }
  else{
   d=DPointRecF(P1A,P1B,P2B);
   d-=RA;
   if(WA<2000) d=d-WA/2;
   return d;
  }
 }
 else if(IdenA==kPicRect){
  if(IdenB==kPicLine){
   if(WA<2000){
    d=DLinRec(P1B,P2B,P1A,P2A);
    d-=WA/2;
   }
   else{
    d=DLinRecF(P1B,P2B,P1A,P2A);
   }
   d-=WB/2;
   return d;
  }
  else if(IdenB==kPicCir){
   if(WA<2000){
    d=DPointRec(P1B,P1A,P2A);
    d-=WA/2;
   }
   else{
    d=DPointRecF(P1B,P1A,P2A);
   }
   d-=RB;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicRect){
   if(WA<2000){
    d=DRecRec(P1B,P2B,P1A,P2A);
   }
   else{
    if(WB<2000){
     d=DRecRec(P1B,P2B,P1A,P2A);
    }
    else{
     d=DRecFRecF(P1B,P2B,P1A,P2A);
    }
   }
   if(WA<2000) d=d-WA/2;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicPoly){
   if(WA<2000){
    d=DRecPoly(P1A,P2A,PoB);
    d-=WA/2;
    d-=WB/2;
   }
   else{
    d=DRecFPoly(P1A,P2A,PoB);
    d-=WB/2;
   }
   return d;
  }
  else{
   if(WA<2000){
    d=DRecRec(P1B,P2B,P1A,P2A);
    d=d-WA/2;
   }
   else{
    d=DRecFRecF(P1B,P2B,P1A,P2A);
   } 
   return d;
  }
 }
 else if(IdenA==kPicPoly){
  if(IdenB==kPicLine){
   d=DLinPoly(P1B,P2B,PoA);
   d-=WA/2;
   d-=WB/2;
   return d;
  }
  else if(IdenB==kPicCir){
   d=DPointPoly(P1B,PoA);
   d-=WA/2;
   d-=RB;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicRect){
   if(WB<2000){
    d=DRecPoly(P1B,P2B,PoA);
    d-=WA/2;
    d-=WB/2;
   }
   else{
    d=DRecFPoly(P1B,P2B,PoA);
    d-=WA/2;
   }
   return d;
  }
  else if(IdenB==kPicPoly){
   d=DPolyPoly(PoA,PoB);
   d-=WA/2;
   d-=WB/2;
   return d;
  }
  else{
   d=DRecFPoly(P1B,P2B,PoA);
   return d;
  }
 }
 else{
  if(IdenB==kPicLine){
   d=DLinRecF(P1B,P2B,P1A,P2A);
   d-=WB/2;
   return d;
  }
  else if(IdenB==kPicCir){
   d=DPointRecF(P1B,P1A,P2A);
   d-=RB;
   if(WB<2000) d=d-WB/2;
   return d;
  }
  else if(IdenB==kPicRect){
   if(WB<2000){
    d=DRecRec(P1B,P2B,P1A,P2A);
    d=d-WB/2;
   }
   else d=DRecFRecF(P1B,P2B,P1A,P2A);
   return d;
  }
  else if(IdenB==kPicPoly){
   d=DRecFPoly(P1A,P2A,PoB);
   d-=WB/2;
   return d;
  }
  else{
   d=DRecFRecF(P1B,P2B,P1A,P2A);
   return d;
  }
 }
 return 100.0;
}


double DRC::DLinLin(QPoint P1A,QPoint P2A,QPoint P1B,QPoint P2B)
{
 double Ax=P1A.x();
 double Ay=P1A.y();
 double Bx=P2A.x();
 double By=P2A.y();

 double Cx=P1B.x();
 double Cy=P1B.y();
 double Dx=P2B.x();
 double Dy=P2B.y();

 double t;
 double t1;

 double d1;
 double d2;

 if((Ax==Cx && Ay==Cy) || (Bx==Cx && By==Cy) ||
    (Ax==Dx && Ay==Dy) || (Bx==Dx && By==Dy)) return 0;
 if((Dx-Cx)*(By-Ay)-(Dy-Cy)*(Bx-Ax)!=0){
  t1=((Cy-Ay)*(Bx-Ax)+(Ax-Cx)*(By-Ay))/
     ((Dx-Cx)*(By-Ay)-(Dy-Cy)*(Bx-Ax));

  if (Bx-Ax) t=(Cx-Ax+t1*(Dx-Cx))/(Bx-Ax);
  else t=(Cy-Ay+t1*(Dy-Cy))/(By-Ay);

  if (t1>=0 && t1<=1 && t>=0 && t<=1) return 0;
 }

 d1=DPointLin(P1A,P1B,P2B);
 d2=DPointLin(P2A,P1B,P2B);
 if(d2<d1) d1=d2;
 d2=DPointLin(P1B,P1A,P2A);
 if(d2<d1) d1=d2;
 d2=DPointLin(P2B,P1A,P2A);
 if(d2<d1) d1=d2;

 return d1;
}

double DRC::DPointLin(QPoint P1A,QPoint P1B,QPoint P2B)
{
 double k;
 P2B=P2B-P1B;
 P1A=P1A-P1B;
 k=((double)P1A.x()*(double)P2B.x()+(double)P1A.y()*(double)P2B.y())/
   ((double)P2B.x()*(double)P2B.x()+(double)P2B.y()*(double)P2B.y());

 if(k<0){
 }
 else if(k>1){
  P1A=P2B-P1A;
 }
 else{
  P1A=(P2B*k)-P1A;
 }
 return sqrt((long long)P1A.x()*P1A.x()+(long long)P1A.y()*P1A.y());
}



double DRC::DPointPoint(QPoint P1A,QPoint P1B)
{
 return sqrt((long long)(P1A-P1B).x()*(P1A-P1B).x()+(long long)(P1A-P1B).y()*(P1A-P1B).y());
}

double DRC::DPointRec(QPoint P1A,QPoint P1B,QPoint P2B)
{
 double d1;
 double d2;
 QPoint p1;
 QPoint p2;

 p1.setX(P1B.x());
 p1.setY(P1B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d1=DPointLin(P1A,p1,p2);

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DPointLin(P1A,p1,p2);
 if(d2<d1) d1=d2;

 p1.setX(P2B.x());
 p1.setY(P2B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d2=DPointLin(P1A,p1,p2);
 if(d2<d1) d1=d2;

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DPointLin(P1A,p1,p2);
 if(d2<d1) d1=d2;


 return d1;

}

double DRC::DLinRec(QPoint P1A,QPoint P2A,QPoint P1B,QPoint P2B)
{
 double d1;
 double d2;
 QPoint p1;
 QPoint p2;

 p1.setX(P1B.x());
 p1.setY(P1B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d1=DLinLin(P1A,P2A,p1,p2);

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DLinLin(P1A,P2A,p1,p2);
 if(d2<d1) d1=d2;

 p1.setX(P2B.x());
 p1.setY(P2B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d2=DLinLin(P1A,P2A,p1,p2);
 if(d2<d1) d1=d2;

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DLinLin(P1A,P2A,p1,p2);
 if(d2<d1) d1=d2;


 return d1;

}

double DRC::DLinRecF(QPoint P1A,QPoint P2A,QPoint P1B,QPoint P2B)
{
 int x1;
 int x2;
 int y1;
 int y2;

 if(P1B.x()<P2B.x()){
  x1=P1B.x();
  x2=P2B.x();
 }
 else{
  x1=P2B.x();
  x2=P1B.x();
 }

 if(P1B.y()<P2B.y()){
  y1=P1B.y();
  y2=P2B.y();
 }
 else{
  y1=P2B.y();
  y2=P1B.y();
 }

 if((x1<=P1A.x() && P1A.x()<=x2 && y1<=P1A.y() && P1A.y()<=y2) ||
    (x1<=P2A.x() && P2A.x()<=x2 && y1<=P2A.y() && P2A.y()<=y2)) return 0;

 return DLinRec(P1A,P2A,P1B,P2B);
}

double DRC::DPointRecF(QPoint P1A,QPoint P1B,QPoint P2B)
{
 int x1;
 int x2;
 int y1;
 int y2;
 int x=P1A.x();
 int y=P1A.y();

 if(P1B.x()<P2B.x()){
  x1=P1B.x();
  x2=P2B.x();
 }
 else{
  x1=P2B.x();
  x2=P1B.x();
 }

 if(P1B.y()<P2B.y()){
  y1=P1B.y();
  y2=P2B.y();
 }
 else{
  y1=P2B.y();
  y2=P1B.y();
 }

 if(x<x1){
  if(y<y1){
   return sqrt((long long)(x-x1)*(x-x1)+(long long)(y-y1)*(y-y1));
  }
  else if(y<=y2){
   return (double)x1-x;
  }
  else{
   return sqrt((long long)(x-x1)*(x-x1)+(long long)(y-y2)*(y-y2));
  }
 }
 else if(x<=x2){
  if(y<y1){
   return (double)y1-y;
  }
  else if(y<=y2){
   return 0;
  }
  else{
   return (double)y-y2;
  }
 }
 else{
  if(y<y1){
   return sqrt((long long)(x-x2)*(x-x2)+(long long)(y-y1)*(y-y1));
  }
  else if(y<=y2){
   return (double)x-x2;
  }
  else{
   return sqrt((long long)(x-x2)*(x-x2)+(long long)(y-y2)*(y-y2));
  }
 }

}

double DRC::DRecRec(QPoint P1A,QPoint P2A,QPoint P1B,QPoint P2B)
{
 double d1;
 double d2;
 QPoint p1;
 QPoint p2;

 p1.setX(P1B.x());
 p1.setY(P1B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d1=DLinRec(p1,p2,P1A,P2A);

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DLinRec(p1,p2,P1A,P2A);
 if(d2<d1) d1=d2;

 p1.setX(P2B.x());
 p1.setY(P2B.y());

 p2.setX(P1B.x());
 p2.setY(P2B.y());
 d2=DLinRec(p1,p2,P1A,P2A);
 if(d2<d1) d1=d2;

 p2.setX(P2B.x());
 p2.setY(P1B.y());
 d2=DLinRec(p1,p2,P1A,P2A);
 if(d2<d1) d1=d2;

 return d1;
}

double DRC::DRecFRecF(QPoint P1A,QPoint P2A,QPoint P1B,QPoint P2B)
{
 int x1A;
 int x2A;
 int x1B;
 int x2B;
 double dx;
 double dy;

 if(P1A.x()<P2A.x()){
  x1A=P1A.x();
  x2A=P2A.x();
 }
 else{
  x1A=P2A.x();
  x2A=P1A.x();
 }

 if(P1B.x()<P2B.x()){
  x1B=P1B.x();
  x2B=P2B.x();
 }
 else{
  x1B=P2B.x();
  x2B=P1B.x();
 }

 if(x2A<x1B){
  dx=x1B-x2A;
 }
 else{
  dx=x1A-x2B;
 }
 if(dx<=0) dx=0;

 if(P1A.y()<P2A.y()){
  x1A=P1A.y();
  x2A=P2A.y();
 }
 else{
  x1A=P2A.y();
  x2A=P1A.y();
 }

 if(P1B.y()<P2B.y()){
  x1B=P1B.y();
  x2B=P2B.y();
 }
 else{
  x1B=P2B.y();
  x2B=P1B.y();
 }

 if(x2A<x1B){
  dy=x1B-x2A;
 }
 else{
  dy=x1A-x2B;
 }
 if(dy<=0) dy=0;

 if(dx<dy) return dy;
 else return dx;
}

double DRC::DLinPoly(QPoint P1A,QPoint P2A,QPainterPath Po)
{
  int i;
  double d;
  double m=100000000;
  QPointF p;
  QPoint p1,p2;
  QPainterPath::Element el;
  if(Po.contains(P1A)) return 0;
  if(Po.contains(P2A)) return 0;
  el = Po.elementAt(0);
  p=el;
  p1=p.toPoint();
  for(i=1;i<Po.elementCount();i++){
    el = Po.elementAt(i);
    p=el;
    p2=p.toPoint();
    if(el.isLineTo()){
      d=DLinLin(P1A,P2A,p1,p2);
      if(d<m) m=d;
    }
    p1=p2;
  }
  return m;
}

double DRC::DPointPoly(QPoint P1A,QPainterPath Po)
{
  int i;
  double d;
  double m=100000000;
  QPointF p;
  QPoint p1,p2;
  QPainterPath::Element el;
  if(Po.contains(P1A)) return 0;
  el = Po.elementAt(0);
  p=el;
  p1=p.toPoint();
  for(i=1;i<Po.elementCount();i++){
    el = Po.elementAt(i);
    p=el;
    p2=p.toPoint();
    if(el.isLineTo()){
      d=DPointLin(P1A,p1,p2);
      if(d<m) m=d;
    }
    p1=p2;
  }
  return m;
}

double DRC::DRecPoly(QPoint P1A,QPoint P2A,QPainterPath Po)
{
  int i;
  double d;
  double m=100000000;
  QPointF p;
  QPoint p1,p2;
  QPainterPath::Element el;

  if(Po.contains(P1A)) return 0;
  if(Po.contains(P2A)) return 0;
  p1.setX(P1A.x());
  p1.setY(P2A.y());
  if(Po.contains(p1)) return 0;
  p1.setX(P2A.x());
  p1.setY(P1A.y());
  if(Po.contains(p1)) return 0;

  el = Po.elementAt(0);
  p=el;
  p1=p.toPoint();
  for(i=1;i<Po.elementCount();i++){
    el = Po.elementAt(i);
    p=el;
    p2=p.toPoint();
    if(el.isLineTo()){
      d=DLinRec(p1,p2,P1A,P2A);
      if(d<m) m=d;
    }
    p1=p2;
  }
  return m;
}

double DRC::DRecFPoly(QPoint P1A,QPoint P2A,QPainterPath Po)
{
  double d;
  double m;
  QPointF p;
  QPoint p1,p2;
  QPainterPath::Element el;

  el=Po.elementAt(0);
  p=el;
  p1=p.toPoint();

  m=DPointRecF(p1,P1A,P2A);
  d=DRecPoly(P1A,P2A,Po);
  if(d<m) m=d;
  return m;
}

double DRC::DPolyPoly(QPainterPath PoA,QPainterPath PoB)
{
  int i;
  double d;
  double m=100000000;
  QPointF p;
  QPoint p1,p2;
  QPainterPath::Element el;

  el=PoB.elementAt(0);
  p=el;
  p1=p.toPoint();
  if(PoA.contains(p1)) return 0;

  el=PoA.elementAt(0);
  p=el;
  p1=p.toPoint();
  for(i=1;i<PoA.elementCount();i++){
    el = PoA.elementAt(i);
    p=el;
    p2=p.toPoint();
    if(el.isLineTo()){
      d=DLinPoly(p1,p2,PoB);
      if(d<m) m=d;
    }
    p1=p2;
  }
  return m;
}

void DRC::AddEl(PicEl * E,int &Iden,QPoint &P1,QPoint &P2,int &W,int &R,
                        int &Lo,QString &N,QPainterPath Po)
{
 DEl * L;
 L=new DEl;
 L->E=E;
 L->Iden=Iden;
 L->P1=P1;
 L->P2=P2;
 L->W=W;
 L->R=R;
 L->Lo=Lo;
 L->Po=Po;
 L->N=N;

 L->Next=LE;
 LE=L;
// printf("DRC ADD %s:%d,%d net:%s\n",sPic[L->Iden],L->P1.x(),L->P1.y(),
//        (const char *)N);
}

void DRC::Clear()
{
 DEl * L;
 while(LE){
  L=LE;
  LE=LE->Next;
  delete L;
 }
}

void DRC::TextSave(QTextStream * S)
{
  *S << "{DrcSetting\n";
  *S << " {PP " << PP << " }\n";
  *S << " {PW " << PW << " }\n";
  *S << " {PO " << PO << " }\n";
  *S << " {WW " << WW << " }\n";
  *S << " {WO " << WO << " }\n";
  *S << " {OO " << OO << " }\n";
  *S << "}\n";
}

void DRC::TextLoad(QTextStream * S)
{
  QString s;
  *S >> s;
  while (!s.contains("}")){
    if(s.contains("PP")){
      *S >> PP;
    }
    else if(s.contains("PW")){
      *S >> PW;
    }
    else if(s.contains("PO")){
      *S >> PO;
    }
    else if(s.contains("WW")){
      *S >> WW;
    }
    else if(s.contains("WO")){
      *S >> WO;
    }
    else if(s.contains("OO")){
      *S >> OO;
    }
    while (!s.contains("}")) *S >> s;
    *S >> s;
  }
}
