/*
 * Author: Petre Eftime <petre.p.eftime@intel.com>
 * Copyright (c) 2015 Intel Corporation.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#pragma once
#include "BluetoothObject.hpp"
#include "BluetoothEvent.hpp"
#include <vector>
#include <list>

class tinyb::BluetoothManager: public BluetoothObject
{
friend class BluetoothAdapter;
friend class BluetoothDevice;
friend class BluetoothGattService;
friend class BluetoothGattCharacteristic;
friend class BluetoothGattDescriptor;
friend class BluetoothEventManager;

private:
    std::unique_ptr<BluetoothAdapter> default_adapter;
    static BluetoothManager *bluetooth_manager;
    std::list<std::shared_ptr<BluetoothEvent>> event_list;

    BluetoothManager();
    BluetoothManager(const BluetoothManager &object);

protected:

    void handle_event(BluetoothType type, std::string *name,
        std::string *identifier, BluetoothObject *parent, BluetoothObject &object);

public:

    static std::string java_class() {
        return std::string(JAVA_PACKAGE "/BluetoothManager");
    }

    static std::string get_api_version();
    static std::string get_library_version();

    virtual std::string get_java_class() const;
    virtual std::string get_class_name() const;
    virtual std::string get_object_path() const;
    virtual BluetoothType get_bluetooth_type() const;

    ~BluetoothManager();
    /** Returns an instance of BluetoothManager, to be used instead of constructor.
      * @return An initialized BluetoothManager instance.
      */
    static BluetoothManager *get_bluetooth_manager();

    /** Add event to checked against events generated by BlueZ. If an the event
      * matches an incoming event its' callback will be triggered. Events can be
      * the addition of a new Device, GattService, GattCharacteristic, etc. */
    void add_event(std::shared_ptr<BluetoothEvent> &event) {
        event_list.push_back(event);
    }

    /** Remove event to checked against events generated by BlueZ.
      */
    void remove_event(std::shared_ptr<BluetoothEvent> &event) {
        event_list.remove(event);
    }

    void remove_event(BluetoothEvent &event) {
        for(auto it = event_list.begin(); it != event_list.end(); ++it) {
            if ((*it).get() == &event) {
                event_list.remove(*it);
                break;
            }
        }
    }


    /** Find a BluetoothObject of type T. If parameters name, identifier and
      * parent are not null, the returned object will have to match them.
      * It will first check for existing objects. It will not turn on discovery
      * or connect to devices.
      * @parameter name optionally specify the name of the object you are
      * waiting for (for Adapter or Device)
      * @parameter identifier optionally specify the identifier of the object you are
      * waiting for (UUID for GattService, GattCharacteristic or GattDescriptor, address
      * for Adapter or Device)
      * @parameter parent optionally specify the parent of the object you are
      * waiting for
      * @parameter timeout the function will return after timeout time, a
      * value of zero means wait forever. If object is not found during this time null will be returned.
      * @return An object matching the name, identifier, parent or null if not found before
      * timeout expires or event is canceled.
      */
    template<class T>
    std::unique_ptr<T> find(std::string *name,
        std::string* identifier, BluetoothObject *parent,
        std::chrono::milliseconds timeout = std::chrono::milliseconds::zero())
    {
        std::unique_ptr<BluetoothObject> obj = find(T::class_type(), name, identifier, parent, timeout);
        T *t = dynamic_cast<T *>(obj.release());
        return std::unique_ptr<T>(t);
    }

    /** Find a BluetoothObject of a type matching type. If parameters name,
      * identifier and parent are not null, the returned object will have to
      * match them.
      * It will first check for existing objects. It will not turn on discovery
      * or connect to devices.
      * @parameter type specify the type of the object you are
      * waiting for, NONE means anything.
      * @parameter name optionally specify the name of the object you are
      * waiting for (for Adapter or Device)
      * @parameter identifier optionally specify the identifier of the object you are
      * waiting for (UUID for GattService, GattCharacteristic or GattDescriptor, address
      * for Adapter or Device)
      * @parameter parent optionally specify the parent of the object you are
      * waiting for
      * @parameter timeout the function will return after timeout time, a
      * value of zero means wait forever. If object is not found during this time null will be returned.
      * @return An object matching the name, identifier, parent or null if not found before
      * timeout expires or event is canceled.
      */
    std::unique_ptr<BluetoothObject> find(BluetoothType type, std::string *name,
        std::string* identifier, BluetoothObject *parent,
        std::chrono::milliseconds timeout = std::chrono::milliseconds::zero());

    /** Find a BluetoothObject of a type matching type. If parameters name,
      * identifier and parent are not null, the found object will have to
      * match them. The callback cb will be triggered when a match is made.
      * Only new objects will trigger the callback.
      * @parameter type specify the type of the object you are
      * waiting for, NONE means anything.
      * @parameter name optionally specify the name of the object you are
      * waiting for (for Adapter or Device)
      * @parameter identifier optionally specify the identifier of the object you are
      * waiting for (UUID for GattService, GattCharacteristic or GattDescriptor, address
      * for Adapter or Device)
      * @parameter parent optionally specify the parent of the object you are
      * waiting for
      * @parameter timeout the function will return after timeout time, a
      * value of zero means wait forever. If object is not found during this time null will be returned.
      * @return It returns the BluetoothEvent generated by this function, allowing to manage the parameters or cancel the event.
      */
    std::weak_ptr<BluetoothEvent> find(BluetoothType type, std::string *name,
        std::string* identifier, BluetoothObject *parent, BluetoothCallback cb,
        bool execute_once = true,
        std::chrono::milliseconds timeout = std::chrono::milliseconds::zero());

    /** Return a BluetoothObject of a type matching type. If parameters name,
      * identifier and parent are not null, the returned object will have to
      * match them. Only objects which are already in the system will be returned.
      * @parameter type specify the type of the object you are
      * waiting for, NONE means anything.
      * @parameter name optionally specify the name of the object you are
      * waiting for (for Adapter or Device)
      * @parameter identifier optionally specify the identifier of the object you are
      * waiting for (UUID for GattService, GattCharacteristic or GattDescriptor, address
      * for Adapter or Device)
      * @parameter parent optionally specify the parent of the object you are
      * waiting for
      * @return An object matching the name, identifier, parent or null if not found.
      */
    std::unique_ptr<BluetoothObject> get_object(BluetoothType type,
        std::string *name, std::string *identifier, BluetoothObject *parent);

    /** Return a vector of BluetoothObject of a type matching type. If parameters name,
      * identifier and parent are not null, the returned object will have to
      * match them. Only objects which are already in the system will be returned.
      * @parameter type specify the type of the object you are
      * waiting for, NONE means anything.
      * @parameter name optionally specify the name of the object you are
      * waiting for (for Adapter or Device)
      * @parameter identifier optionally specify the identifier of the object you are
      * waiting for (UUID for GattService, GattCharacteristic or GattDescriptor, address
      * for Adapter or Device)
      * @parameter parent optionally specify the parent of the object you are
      * waiting for
      * @return A vector of object matching the name, identifier, parent.
      */
    std::vector<std::unique_ptr<BluetoothObject>> get_objects(
        BluetoothType type = BluetoothType::NONE,
        std::string *name = nullptr, std::string *identifier = nullptr,
        BluetoothObject *parent = nullptr);

    /** Returns a list of BluetoothAdapters available in the system
      * @return A list of BluetoothAdapters available in the system
      */
    std::vector<std::unique_ptr<BluetoothAdapter>> get_adapters(
    );

    /** Returns a list of discovered BluetoothDevices
      * @return A list of discovered BluetoothDevices
      */
    std::vector<std::unique_ptr<BluetoothDevice>> get_devices(
    );

    /** Returns a list of available BluetoothGattServices
      * @return A list of available BluetoothGattServices
      */
    std::vector<std::unique_ptr<BluetoothGattService>> get_services(
    );

    /** Sets a default adapter to use for discovery.
      * @return TRUE if the device was set
      */
    bool set_default_adapter(
        BluetoothAdapter &adapter
    );

    std::unique_ptr<BluetoothAdapter> get_default_adapter();

    /** Turns on device discovery on the default adapter if it is disabled.
      * @return TRUE if discovery was successfully enabled
      */
    bool start_discovery(
    );

    /** Turns off device discovery on the default adapter if it is enabled.
      * @return TRUE if discovery was successfully disabled
      */
    bool stop_discovery(
    );

    /** Returns if the discovers is running or not.
      * @return TRUE if discovery is running
      */
    bool get_discovering(
    );
};
