/*
 * Copyright (C) 2012-2025 Robin Haberkorn
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#pragma once

#include <glib.h>

#include "sciteco.h"
#include "parser.h"
#include "view.h"
#include "undo.h"

/** Indicator number used for the rubbed out part of the command line */
#define INDICATOR_RUBBEDOUT (INDICATOR_CONTAINER+0)

typedef struct {
	/**
	 * State machine used for interactive mode (commandline macro).
	 * It is initialized on-demand in main.c.
	 * This is a global variable instead of being passed down the call stack
	 * since some process_edit_cmd_cb will be for nested state machines
	 * but we must "step" only the toplevel state machine.
	 */
	teco_machine_main_t machine;

	/**
	 * Command-line Scintilla view.
	 * It's document contains the current command line macro.
	 * The current position (cursor) marks the end of the
	 * "effective" command line, while everything afterwards
	 * is the rubbed out part of the command line.
	 * The rubbed out part should be highlighted with an indicator.
	 */
	teco_view_t *view;

	/** Height of the command line view in lines */
	guint height;

	/** Program counter within the command-line macro */
	gsize pc;

	/**
	 * Specifies whether the immediate editing modifier
	 * is enabled/disabled.
	 * It can be toggled with the ^G immediate editing command
	 * and influences the undo/redo direction and function of the
	 * TAB key.
	 */
	gboolean modifier_enabled;
} teco_cmdline_t;

extern teco_cmdline_t teco_cmdline;

void teco_cmdline_init(void);

static inline sptr_t
teco_cmdline_ssm(unsigned int iMessage, uptr_t wParam, sptr_t lParam)
{
	return teco_view_ssm(teco_cmdline.view, iMessage, wParam, lParam);
}

/**
 * Update scroll beavior on command line after window resizes.
 *
 * This should ensure that the caret jumps to the middle of the command line.
 *
 * @param width Window (command line view) width in pixels or columns.
 *
 * @fixme
 * On the other hand this limits how you can customize the scroll behavior.
 */
static inline void
teco_cmdline_resized(guint width)
{
	teco_cmdline_ssm(SCI_SETXCARETPOLICY, CARET_SLOP | CARET_EVEN, width/2);
}

gboolean teco_cmdline_keypress(const gchar *data, gsize len, GError **error);

typedef enum {
	TECO_KEYMACRO_ERROR = 0,	/**< GError occurred */
	TECO_KEYMACRO_SUCCESS,		/**< key macro found and inserted */
	TECO_KEYMACRO_UNDEFINED		/**< no key macro found */
} teco_keymacro_status_t;

teco_keymacro_status_t teco_cmdline_keymacro(const gchar *name, gssize name_len, GError **error);

static inline gboolean
teco_cmdline_keymacro_c(gchar key, GError **error)
{
	switch (teco_cmdline_keymacro(&key, sizeof(key), error)) {
	case TECO_KEYMACRO_ERROR:
		return FALSE;
	case TECO_KEYMACRO_SUCCESS:
		break;
	case TECO_KEYMACRO_UNDEFINED:
		return teco_cmdline_keypress(&key, sizeof(key), error);
	}
	return TRUE;
}

void teco_cmdline_cleanup(void);

/*
 * Command states
 */

TECO_DECLARE_STATE(teco_state_save_cmdline);
