#include <dbus/dbus.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>

// Handle the PrepareForSleep signal
void handle_prepare_for_sleep(DBusMessage* msg) {
    DBusError error;
    dbus_error_init(&error);

    dbus_bool_t going_to_sleep;

    if (!dbus_message_get_args(msg, &error, DBUS_TYPE_BOOLEAN, &going_to_sleep, DBUS_TYPE_INVALID)) {
        fprintf(stderr, "Failed to parse signal arguments: %s\n", error.message);
        dbus_error_free(&error);
        return;
    }

    if (going_to_sleep) {
        printf("System is going to sleep.\n");
    } else {
        printf("System has resumed from sleep.\n");
    }
}

int main() {
    DBusError error;
    DBusConnection* connection;
    dbus_error_init(&error);

    // Connect to the system bus
    connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
    if (dbus_error_is_set(&error)) {
        fprintf(stderr, "Connection Error (%s)\n", error.message);
        dbus_error_free(&error);
        return EXIT_FAILURE;
    }

    if (!connection) {
        fprintf(stderr, "Failed to connect to the D-Bus system bus\n");
        return EXIT_FAILURE;
    }

    // Add a match rule for the PrepareForSleep signal
    const char* match_rule = 
        "type='signal',"
        "interface='org.freedesktop.login1.Manager',"
        "member='PrepareForSleep'";

    dbus_bus_add_match(connection, match_rule, &error);
    dbus_connection_flush(connection);

    if (dbus_error_is_set(&error)) {
        fprintf(stderr, "Match Error (%s)\n", error.message);
        dbus_error_free(&error);
        return EXIT_FAILURE;
    }

    printf("Listening for system power state changes...\n");

    // Main loop
    while (true) {
        dbus_connection_read_write(connection, 0);
        DBusMessage* msg = dbus_connection_pop_message(connection);

        if (msg == NULL) {
            usleep(100000); // Sleep for 100ms
            continue;
        }

        if (dbus_message_is_signal(msg, "org.freedesktop.login1.Manager", "PrepareForSleep")) {
            handle_prepare_for_sleep(msg);
        }

        dbus_message_unref(msg);
    }

    return EXIT_SUCCESS;
}

