/*
  $Id: RenderFuncs.c,v 1.171 2014/01/16 15:47:50 dug9 Exp $

  FreeWRL support library.
  Scenegraph rendering.

*/

/****************************************************************************
    This file is part of the FreeWRL/FreeX3D Distribution.

    Copyright 2009 CRC Canada. (http://www.crc.gc.ca)

    FreeWRL/FreeX3D is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    FreeWRL/FreeX3D is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with FreeWRL/FreeX3D.  If not, see <http://www.gnu.org/licenses/>.
****************************************************************************/



#include <config.h>
#include <system.h>
#include <display.h>
#include <internal.h>

#include <libFreeWRL.h>

#include "../vrml_parser/Structs.h"
#include "../main/headers.h"
#include "../opengl/Textures.h"
#include "../scenegraph/Component_ProgrammableShaders.h"

#include "Polyrep.h"
#include "Collision.h"
#include "../scenegraph/quaternion.h"
#include "Viewer.h"
#include "LinearAlgebra.h"
#include "../input/SensInterps.h"
#include "system_threads.h"
#include "threads.h"

#include "../opengl/OpenGL_Utils.h"
#include "../scenegraph/Component_Shape.h"
#include "RenderFuncs.h"


typedef float shaderVec4[4];


typedef struct pRenderFuncs{
	float light_linAtten[MAX_LIGHTS];
	float light_constAtten[MAX_LIGHTS];
	float light_quadAtten[MAX_LIGHTS];
	float light_spotCutoffAngle[MAX_LIGHTS];
	float light_spotBeamWidth[MAX_LIGHTS];
	shaderVec4 light_amb[MAX_LIGHTS];
	shaderVec4 light_dif[MAX_LIGHTS];
	shaderVec4 light_pos[MAX_LIGHTS];
	shaderVec4 light_spec[MAX_LIGHTS];
	shaderVec4 light_spotDir[MAX_LIGHTS];
    float light_radius[MAX_LIGHTS];
	GLint lightType[MAX_LIGHTS]; //0=point 1=spot 2=directional
	/* Rearrange to take advantage of headlight when off */
	int nextFreeLight;// = 0;

	/* lights status. Light HEADLIGHT_LIGHT is the headlight */
	GLint lightOnOff[MAX_LIGHTS];
	GLint last_lightOnOff[MAX_LIGHTS]; //optimization
	GLint lastShader;
	int cur_hits;//=0;
	void *empty_group;//=0;
	//struct point_XYZ ht1, ht2; not used
	struct point_XYZ hyper_r1,hyper_r2; /* Transformed ray for the hypersensitive node */
	struct currayhit rayph;
	struct X3D_Group *rootNode;//=NULL;	/* scene graph root node */
	struct X3D_Anchor *AnchorsAnchor;// = NULL;
	struct currayhit rayHit,rayHitHyper;
	struct trenderstate renderstate;
	int renderLevel;

	// which Shader is currently in use?
	GLint currentShader;

}* ppRenderFuncs;
void *RenderFuncs_constructor(){
	void *v = malloc(sizeof(struct pRenderFuncs));
	memset(v,0,sizeof(struct pRenderFuncs));
	return v;
}
void RenderFuncs_init(struct tRenderFuncs *t){
	//public

	t->BrowserAction = FALSE;
	//	t->hitPointDist; /* distance in ray: 0 = r1, 1 = r2, 2 = 2*r2-r1... */
	///* used to save rayhit and hyperhit for later use by C functions */
	//t->hyp_save_posn;
	//t->hyp_save_norm;t->ray_save_posn;
	t->hypersensitive = 0;
	t->hyperhit = 0;
	t->have_transparency=FALSE;/* did any Shape have transparent material? */
	/* material node usage depends on texture depth; if rgb (depth1) we blend color field
	   and diffusecolor with texture, else, we dont bother with material colors */
	t->last_texture_type = NOTEXTURE;

	//private
	t->prv = RenderFuncs_constructor();
	{
		ppRenderFuncs p = (ppRenderFuncs)t->prv;
		/* which arrays are enabled, and defaults for each array */
        
		/* Rearrange to take advantage of headlight when off */
		p->nextFreeLight = 0;

		p->cur_hits=0;
		p->empty_group=0;
		p->rootNode=NULL;	/* scene graph root node */
		p->AnchorsAnchor = NULL;
		t->rayHit = (void *)&p->rayHit;
		t->rayHitHyper = (void *)&p->rayHitHyper;
		p->renderLevel = 0;
		p->lastShader = -1;
	}

	//setLightType(HEADLIGHT_LIGHT,2); // ensure that this is a DirectionalLight.
}


/* we assume max MAX_LIGHTS lights. The max light is the Headlight, so we go through 0-HEADLIGHT_LIGHT for Lights */
int nextlight() {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	int rv = p->nextFreeLight;
	if(p->nextFreeLight == HEADLIGHT_LIGHT) { return -1; }
	p->nextFreeLight ++;
	return rv;
}

/* lightType 0=point 1=spot 2=directional */
void setLightType(GLint light, int type) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
    p->lightType[light] = type;
}

/* keep track of lighting */
void setLightState(GLint light, int status) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
    //ConsoleMessage ("start lightState, light %d, status %d\n",light,status);

    
    PRINT_GL_ERROR_IF_ANY("start lightState");

	if (light<0) return; /* nextlight will return -1 if too many lights */
	p->lightOnOff[light] = status;
    PRINT_GL_ERROR_IF_ANY("end lightState");
}

/* for local lights, we keep track of what is on and off */
void saveLightState(int *ls) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	memcpy (ls,p->lightOnOff,sizeof(int)*HEADLIGHT_LIGHT);
} 

void restoreLightState(int *ls) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	memcpy (p->lightOnOff,ls,sizeof(int)*HEADLIGHT_LIGHT);
}

void transformLightToEye(float *pos, float* dir)
{
	int i;
    GLDOUBLE modelMatrix[16], *b;
	float *a;
	shaderVec4 aux, auxt;
    FW_GL_GETDOUBLEV(GL_MODELVIEW_MATRIX, modelMatrix);

/*
ConsoleMessage ("nvm %4.2f %4.2f %4.2f %4.2f",modelMatrix[0],modelMatrix[1],modelMatrix[2],modelMatrix[3]);
ConsoleMessage ("nvm %4.2f %4.2f %4.2f %4.2f",modelMatrix[4],modelMatrix[5],modelMatrix[6],modelMatrix[7]);
ConsoleMessage ("nvm %4.2f %4.2f %4.2f %4.2f",modelMatrix[8],modelMatrix[9],modelMatrix[10],modelMatrix[11]);
ConsoleMessage ("nvm %4.2f %4.2f %4.2f %4.2f",modelMatrix[12],modelMatrix[13],modelMatrix[14],modelMatrix[15]);
*/

    /* pre-multiply the light position, as per the orange book, page 216,
     "OpenGL specifies that light positions are transformed by the modelview
     matrix when they are provided to OpenGL..." */
    /* DirectionalLight?  PointLight, SpotLight? */

	// assumes pos[3] = 0.0; only use first 3 of these numbers
	transformf(auxt,pos,modelMatrix);
	auxt[3] = 0.0;

/*
ConsoleMessage("LightToEye, after transformf, auxt %4.2f %4.2f %4.2f %4.2f, pos %4.2f %4.2f %4.2f %4.2f",
auxt[0],auxt[1],auxt[2],auxt[3],
pos[0],pos[1],pos[2],pos[3]);
*/

	for(i=0;i<4;i++){
		pos[i] = auxt[i];
	}
	b = modelMatrix;
	a = dir;
	aux[0] = (float) (b[0]*a[0] +b[4]*a[1] +b[8]*a[2] );
	aux[1] = (float) (b[1]*a[0] +b[5]*a[1] +b[9]*a[2] );
	aux[2] = (float) (b[2]*a[0] +b[6]*a[1] +b[10]*a[2]);
	for(i=0;i<3;i++)
		dir[i] = aux[i];

	// just initialize this to 0.0
	dir[3] = 0.0;

}

void fwglLightfv (int light, int pname, GLfloat *params) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	/*printf ("fwglLightfv light: %d ",light);
	switch (pname) {
		case GL_AMBIENT: printf ("GL_AMBIENT"); break;
		case GL_DIFFUSE: printf ("GL_DIFFUSE"); break;
		case GL_POSITION: printf ("GL_POSITION"); break;
		case GL_SPECULAR: printf ("GL_SPECULAR"); break;
		case GL_SPOT_DIRECTION: printf ("GL_SPOT_DIRECTION"); break;
        case GL_LIGHT_RADIUS: printf ("GL_LIGHT_RADIUS"); break;
	}
	printf (" %f %f %f %f\n",params[0], params[1],params[2],params[3]);
     */
    
	//printLTDebug(__FILE__,__LINE__);


	switch (pname) {
		case GL_AMBIENT:
			memcpy ((void *)p->light_amb[light],(void *)params,sizeof(shaderVec4));
			break;
		case GL_DIFFUSE:
			memcpy ((void *)p->light_dif[light],(void *)params,sizeof(shaderVec4));
			break;
		case GL_POSITION:
			memcpy ((void *)p->light_pos[light],(void *)params,sizeof(shaderVec4));
			//the following function call assumes spotdir has already been set - set it first from render_light

/*
ConsoleMessage("fwglLightfv - NOT transforming pos %3.2f %3.2f %3.2f %3.2f spd %3.2f %3.2f %3.2f %3.2f",
			p->light_pos[light][0],
			p->light_pos[light][1],
			p->light_pos[light][2],
			p->light_pos[light][3],
			p->light_spotDir[light][0],
			p->light_spotDir[light][1],
			p->light_spotDir[light][2],
			p->light_spotDir[light][3]);
*/
			if (light != HEADLIGHT_LIGHT)  transformLightToEye(p->light_pos[light], p->light_spotDir[light]);
			break;
		case GL_SPECULAR:
			memcpy ((void *)p->light_spec[light],(void *)params,sizeof(shaderVec4));
			break;
		case GL_SPOT_DIRECTION:
			//call spot_direction before spot_position, so direction gets transformed above in spot position
			memcpy ((void *)p->light_spotDir[light],(void *)params,sizeof(shaderVec4));
			break;
		default: {printf ("help, unknown fwgllightfv param %d\n",pname);}
	}
}

void fwglLightf (int light, int pname, GLfloat param) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;

#ifdef RENDERVERBOSE
	printf ("fwglLightf light: %d ",light);
	switch (pname) {
		case GL_CONSTANT_ATTENUATION: printf ("GL_CONSTANT_ATTENUATION"); break;
		case GL_LINEAR_ATTENUATION: printf ("GL_LINEAR_ATTENUATION"); break;
		case GL_QUADRATIC_ATTENUATION: printf ("GL_QUADRATIC_ATTENUATION"); break;
		case GL_SPOT_CUTOFF: printf ("GL_SPOT_CUTOFF"); break;
		case GL_SPOT_BEAMWIDTH: printf ("GL_SPOT_BEAMWIDTH"); break;
	}
	printf (" %f\n",param);
#endif
	
    
	switch (pname) {
		case GL_CONSTANT_ATTENUATION:
			p->light_constAtten[light] = param;
			break;
		case GL_LINEAR_ATTENUATION:
			p->light_linAtten[light] = param;
			break;
		case GL_QUADRATIC_ATTENUATION:
			p->light_quadAtten[light] = param;
			break;
		case GL_SPOT_CUTOFF:
			p->light_spotCutoffAngle[light] = param;
            //ConsoleMessage ("setting light_spotCutoffAngle for %d to %f\n",light,param);
			break;
		case GL_SPOT_BEAMWIDTH:
			p->light_spotBeamWidth[light] = param;
            //ConsoleMessage ("setting light_spotBeamWidth for %d to %f\n",light,param);

			break;
        case GL_LIGHT_RADIUS:
            p->light_radius[light] = param;
            break;

		default: {printf ("help, unknown fwgllightfv param %d\n",pname);}
	}
}


/* send light info into Shader. if OSX gets glGetUniformBlockIndex calls, we can do this with 1 call 
	On old pentium with old board in old PCI slot, 8 lights take 1050bytes and 12% of mainloop load
	3 optimizations reduce the light sending traffic:
	1. send only active lights 
		-Android had a problem on startup with this, I changed the flavor a bit - lets see if it works now
		-cuts from 12% of loop to 4%
	2. for an active light, send only parameters that light type needs in the shader calc
		-cuts from 4% of loop to 3%
	3. if the active light set and shader haven't changed since last shape, don't resend any lights.
		(active light sets can change during scene graph traversal. But a light itself doesn't change
		settings during traversal. Just during routing and scripting. So in render_heir 
		lastShader is set to -1 to trigger a fresh send.
		- cuts from 3% of loop to .5%

*/
void sendLightInfo (s_shader_capabilities_t *me) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
    int i,j, lightcount, lightsChanged;
    		
	// in case we are trying to render a node that has just been killed...
	if (me==NULL) return;

	PRINT_GL_ERROR_IF_ANY("BEGIN sendLightInfo");
	/* if one of these are equal to -1, we had an error in the shaders... */
	lightcount = 0;
	//Optimization 3>> if the shader and lights haven't changed since the last shape,
	//then don't resend the lights to the shader
	lightsChanged = FALSE;
	for(i=0;i<MAX_LIGHTS;i++){
		if(p->lightOnOff[i]) lightcount++;
		if(p->lightOnOff[i] != p->last_lightOnOff[i]) lightsChanged = TRUE;
		p->last_lightOnOff[i] = p->lightOnOff[i];
	}
	if(!lightsChanged && (p->currentShader == p->lastShader)) return;
	p->lastShader = p->currentShader;
	//<<end optimization 3
	profile_start("sendlight");
	GLUNIFORM1I(me->lightcount,lightcount);
	//GLUNIFORM1IV(me->lightState,MAX_LIGHTS,p->lightOnOff); //don't need with lightcount
	//GLUNIFORM1IV(me->lightType,MAX_LIGHTS,p->lightType); //need to pack into light struct
    //GLUNIFORM1FV(me->lightRadius,MAX_LIGHTS,p->light_radius); //need to pack into lightstruct
	PRINT_GL_ERROR_IF_ANY("MIDDLE1 sendLightInfo");
    
    // send in lighting info, but only for lights that are "on"
	// reason: at 1100+ bytes per shape for 8 lights, it takes up 11.2% of mainloop activity on an old pentium
	// so this cuts it down to about 200 bytes per shape if you have a headlight and another light.
    for (i=0,j=0; i<MAX_LIGHTS; i++) {
		if (p->lightOnOff[i]) { // this causes initial screen on Android to fail.
								// dug9 - I added another parameter lightcount above and in ADSL shader
								// and pack the lights ie. moving headlight up here so its at 
								// lightcount-1 instead of MAX_LIGHTS-1 on the GPU.
								// LMK if breaks android
            //0 - pointlight
            //1 - spotlight
            //2 - directionlight
			//save a bit of bandwidth by not sending unused parameters for a light type
			if(p->lightType[i]<2 ){ //not direction
				shaderVec4 light_Attenuations;
				light_Attenuations[0] = p->light_constAtten[i];
				light_Attenuations[1] = p->light_linAtten[i];
				light_Attenuations[2] = p->light_quadAtten[i];
				GLUNIFORM3FV(me->lightAtten[j],1,light_Attenuations);
				//GLUNIFORM1F (me->lightConstAtten[j], p->light_constAtten[i]);
				//GLUNIFORM1F (me->lightLinAtten[j], p->light_linAtten[i]);
				//GLUNIFORM1F(me->lightQuadAtten[j], p->light_quadAtten[i]);
			}
			if(p->lightType[i]==1 ){ //spot
				GLUNIFORM1F(me->lightSpotCutoffAngle[j], p->light_spotCutoffAngle[i]);
				GLUNIFORM1F(me->lightSpotBeamWidth[j], p->light_spotBeamWidth[i]);
			}
			if(p->lightType[i]==0){ //point
	            GLUNIFORM1F(me->lightRadius[j],p->light_radius[i]);
			}
			GLUNIFORM4FV(me->lightSpotDir[j],1, p->light_spotDir[i]);
            GLUNIFORM4FV(me->lightPosition[j],1,p->light_pos[i]);
            GLUNIFORM4FV(me->lightAmbient[j],1,p->light_amb[i]);
            GLUNIFORM4FV(me->lightDiffuse[j],1,p->light_dif[i]);
            GLUNIFORM4FV(me->lightSpecular[j],1,p->light_spec[i]);
			GLUNIFORM1I(me->lightType[j],p->lightType[i]);
			j++;
        }
    }
	profile_end("sendlight");
    PRINT_GL_ERROR_IF_ANY("END sendLightInfo");
}

/* finished rendering thisshape. */
void finishedWithGlobalShader(void) {
    //printf ("finishedWithGlobalShader\n");


    /* get rid of the shader */
    getAppearanceProperties()->currentShaderProperties = NULL;
FW_GL_BINDBUFFER(GL_ARRAY_BUFFER, 0);

FW_GL_BINDBUFFER(GL_ELEMENT_ARRAY_BUFFER, 0);

}


/* should the system need to rebuild the OpenGL system (eg, Android, 
on restore of screen, iPhone?? Blackberry???) we ensure that the system
state is such that new information will get cached */

void resetGlobalShader() {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;

	//ConsoleMessage ("resetGlobalShader called");

	/* no shader currently active */
	p->currentShader = 0;
}

void restoreGlobalShader(){
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
    if (p->currentShader)
		USE_SHADER(p->currentShader);
}
/* choose and turn on a shader for this geometry */

void enableGlobalShader(s_shader_capabilities_t *myShader) {
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;

    //ConsoleMessage ("enableGlobalShader, have myShader %d",myShader->myShaderProgram);
    if (myShader == NULL) {
        finishedWithGlobalShader(); 
        return;
    };
    
    
    getAppearanceProperties()->currentShaderProperties = myShader;
    if (myShader->myShaderProgram != p->currentShader) {
		USE_SHADER(myShader->myShaderProgram);
		p->currentShader = myShader->myShaderProgram;
	}
}


/* send in vertices, normals, etc, etc... to either a shader or via older opengl methods */
void sendAttribToGPU(int myType, int dataSize, int dataType, int normalized, int stride, float *pointer, char *file, int line){

    s_shader_capabilities_t *me = getAppearanceProperties()->currentShaderProperties;

	// checking to see that we really have the data
	if (me==NULL) return;

#ifdef RENDERVERBOSE

ConsoleMessage ("sendAttribToGPU, getAppearanceProperties()->currentShaderProperties %p\n",getAppearanceProperties()->currentShaderProperties);
ConsoleMessage ("myType %d, dataSize %d, dataType %d, stride %d\n",myType,dataSize,dataType,stride);
	if (me != NULL) {
		switch (myType) {
			case FW_NORMAL_POINTER_TYPE:
				ConsoleMessage ("glVertexAttribPointer  Normals %d at %s:%d\n",me->Normals,file,line);
				break;
			case FW_VERTEX_POINTER_TYPE:
				ConsoleMessage ("glVertexAttribPointer  Vertexs %d at %s:%d\n",me->Vertices,file,line);
				break;
			case FW_COLOR_POINTER_TYPE:
				ConsoleMessage ("glVertexAttribPointer  Colours %d at %s:%d\n",me->Colours,file,line);
				break;
			case FW_TEXCOORD_POINTER_TYPE:
				ConsoleMessage ("glVertexAttribPointer  TexCoords %d at %s:%d\n",me->TexCoords,file,line);
				break;

			default : {ConsoleMessage ("sendAttribToGPU, unknown type in shader\n");}
		}
	}
#endif
#undef RENDERVERBOSE

	switch (myType) {
		case FW_NORMAL_POINTER_TYPE:
		if (me->Normals != -1) {
			glEnableVertexAttribArray(me->Normals);
			glVertexAttribPointer(me->Normals, 3, dataType, normalized, stride, pointer);
		}
			break;
		case FW_VERTEX_POINTER_TYPE:
		if (me->Vertices != -1) {
			glEnableVertexAttribArray(me->Vertices);
			glVertexAttribPointer(me->Vertices, dataSize, dataType, normalized, stride, pointer);
		}
			break;
		case FW_COLOR_POINTER_TYPE:
		if (me->Colours != -1) {
			glEnableVertexAttribArray(me->Colours);
			glVertexAttribPointer(me->Colours, dataSize, dataType, normalized, stride, pointer);
		}
			break;
		case FW_TEXCOORD_POINTER_TYPE:
		if (me->TexCoords != -1) {
			glEnableVertexAttribArray(me->TexCoords);
			glVertexAttribPointer(me->TexCoords, dataSize, dataType, normalized, stride, pointer);
		}
			break;

		default : {printf ("sendAttribToGPU, unknown type in shader\n");}
	}
}


void sendBindBufferToGPU (GLenum target, GLuint buffer, char *file, int line) {

	
/*
	if (target == GL_ARRAY_BUFFER_BINDING) printf ("glBindBuffer, GL_ARRAY_BUFFER_BINDING %d at %s:%d\n",buffer,file,line);
	else if (target == GL_ARRAY_BUFFER) printf ("glBindBuffer, GL_ARRAY_BUFFER %d at %s:%d\n",buffer,file,line);
	else if (target == GL_ELEMENT_ARRAY_BUFFER) printf ("glBindBuffer, GL_ELEMENT_ARRAY_BUFFER %d at %s:%d\n",buffer,file,line);
	else printf ("glBindBuffer, %d %d at %s:%d\n",target,buffer,file,line);
	
*/

	glBindBuffer(target,buffer);
}



static bool setupShader() {

    s_shader_capabilities_t *mysp = getAppearanceProperties()->currentShaderProperties;

PRINT_GL_ERROR_IF_ANY("BEGIN setupShader");
	if (mysp == NULL) return FALSE;
        

		/* if we had a shader compile problem, do not draw */
		if (!(mysp->compiledOK)) {
#ifdef RENDERVERBOSE
			printf ("shader compile error\n");
#endif
			PRINT_GL_ERROR_IF_ANY("EXIT(false) setupShader");
			return false;
		}
       
#ifdef RENDERVERBOSE
        printf ("setupShader, we have Normals %d Vertices %d Colours %d TexCoords %d \n",
                mysp->Normals,
                mysp->Vertices,
                mysp->Colours,
                mysp->TexCoords);
           
#endif

        
        /* send along lighting, material, other visible properties */
        sendMaterialsToShader(mysp);
        sendMatriciesToShader(mysp);
    
    return true;
    
}


void sendArraysToGPU (int mode, int first, int count) {
#ifdef RENDERVERBOSE
	printf ("sendArraysToGPU start\n"); 
#endif


	// when glDrawArrays bombs it's usually some function left an array
	// enabled that's not supposed to be - try disabling something
//glDisableClientState(GL_VERTEX_ARRAY);
//glDisableClientState(GL_NORMAL_ARRAY);
//glDisableClientState(GL_INDEX_ARRAY);
//glDisableClientState(GL_COLOR_ARRAY);
//glDisableClientState(GL_SECONDARY_COLOR_ARRAY);
//glDisableClientState(GL_FOG_COORDINATE_ARRAY);
//glDisableClientState(GL_TEXTURE_COORD_ARRAY);
//glDisableClientState(GL_EDGE_FLAG_ARRAY);

	if (setupShader()){
		profile_start("draw_arr");
		glDrawArrays(mode,first,count);
		profile_end("draw_arr");
	}
    #ifdef RENDERVERBOSE
	printf ("sendArraysToGPU end\n"); 
    #endif
}



void sendElementsToGPU (int mode, int count, ushort *indices) {
    #ifdef RENDERVERBOSE
	printf ("sendElementsToGPU start\n"); 
    #endif
    
	if (setupShader()){
		profile_start("draw_el");
        glDrawElements(mode,count,GL_UNSIGNED_SHORT,indices);
		profile_end("draw_el");
	}

	#ifdef RENDERVERBOSE
	printf ("sendElementsToGPU finish\n"); 
	#endif
}


void initializeLightTables() {
	int i;
        float pos[] = { 0.0f, 0.0f, 1.0f, 0.0f };
        float dif[] = { 1.0f, 1.0f, 1.0f, 1.0f };
        float shin[] = { 0.0f, 0.0f, 0.0f, 1.0f }; /* light defaults - headlight is here, too */
        float As[] = { 0.0f, 0.0f, 0.0f, 1.0f };
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;

      PRINT_GL_ERROR_IF_ANY("start of initializeightTables");

	for(i=0; i<MAX_LIGHTS; i++) {
                p->lightOnOff[i] = TRUE;
                setLightState(i,FALSE);
            
		FW_GL_LIGHTFV(i, GL_SPOT_DIRECTION, pos);
       		FW_GL_LIGHTFV(i, GL_POSITION, pos);
       		FW_GL_LIGHTFV(i, GL_AMBIENT, As);
       		FW_GL_LIGHTFV(i, GL_DIFFUSE, dif);
       		FW_GL_LIGHTFV(i, GL_SPECULAR, shin);
         	FW_GL_LIGHTF(i, GL_CONSTANT_ATTENUATION,1.0f);
       		FW_GL_LIGHTF(i, GL_LINEAR_ATTENUATION,0.0f);
       		FW_GL_LIGHTF(i, GL_QUADRATIC_ATTENUATION,0.0f);
       		FW_GL_LIGHTF(i, GL_SPOT_CUTOFF,0.0f);
       		FW_GL_LIGHTF(i, GL_SPOT_BEAMWIDTH,0.0f);
           	FW_GL_LIGHTF(i, GL_LIGHT_RADIUS, 100000.0); /* just make it large for now*/ 
            
            	PRINT_GL_ERROR_IF_ANY("initizlizeLight2.10");
        }
        setLightState(HEADLIGHT_LIGHT, TRUE);

#ifdef OLDCODE
OLDCODE	#ifndef GL_ES_VERSION_2_0
OLDCODE        FW_GL_LIGHTMODELI(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);
OLDCODE        FW_GL_LIGHTMODELI(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
OLDCODE        FW_GL_LIGHTMODELI(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
OLDCODE        FW_GL_LIGHTMODELI(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
OLDCODE        FW_GL_LIGHTMODELFV(GL_LIGHT_MODEL_AMBIENT,As);
OLDCODE	#else
OLDCODE	//printf ("skipping light setups\n");
OLDCODE	#endif
#endif //OLDCODE

    LIGHTING_INITIALIZE
	

    PRINT_GL_ERROR_IF_ANY("end initializeLightTables");
}


ttrenderstate renderstate()
{
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	return &p->renderstate;
}


//true statics:
GLint viewport[4] = {-1,-1,2,2};  //doesn't change, used in glu unprojects
/* These two points define a ray in window coordinates */
struct point_XYZ r1 = {0,0,-1},r2 = {0,0,0},r3 = {0,1,0};


struct X3D_Anchor *AnchorsAnchor()
{
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	return p->AnchorsAnchor;
}
void setAnchorsAnchor(struct X3D_Anchor* anchor)
{
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	p->AnchorsAnchor = anchor;
}


//static struct currayhit rayph;
//struct currayhit rayHit,rayHitHyper;
/* used to test new hits */



//struct X3D_Group *_rootNode=NULL;	/* scene graph root node */
struct X3D_Group *rootNode()
{
	// ConsoleMessage ("rootNode called");
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;	
	if (p==NULL) {
		ConsoleMessage ("rootNode, p null");
		return NULL;
	}
	return p->rootNode;
}
void setRootNode(struct X3D_Group *rn)
{
	ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
	p->rootNode = rn;
}
//void *empty_group=0;

/*******************************************************************************/

/* Sub, rather than big macro... */
void rayhit(float rat, float cx,float cy,float cz, float nx,float ny,float nz,
	    float tx,float ty, char *descr)  {
	GLDOUBLE modelMatrix[16];
	GLDOUBLE projMatrix[16];
	ppRenderFuncs p;
	ttglobal tg = gglobal();
	p = (ppRenderFuncs)tg->RenderFuncs.prv;

	/* Real rat-testing */
#ifdef RENDERVERBOSE
	//printf("RAY HIT %s! %f (%f %f %f) (%f %f %f)\n\tR: (%f %f %f) (%f %f %f)\n",
	//       descr, rat,cx,cy,cz,nx,ny,nz,
	//       t_r1.x, t_r1.y, t_r1.z,
	//       t_r2.x, t_r2.y, t_r2.z
	//	);
#endif

	if(rat<0 || (rat>tg->RenderFuncs.hitPointDist && tg->RenderFuncs.hitPointDist >= 0)) {
		return;
	}
	FW_GL_GETDOUBLEV(GL_MODELVIEW_MATRIX, modelMatrix);
	FW_GL_GETDOUBLEV(GL_PROJECTION_MATRIX, projMatrix);
	FW_GLU_PROJECT(cx,cy,cz, modelMatrix, projMatrix, viewport, &tg->RenderFuncs.hp.x, &tg->RenderFuncs.hp.y, &tg->RenderFuncs.hp.z);
	tg->RenderFuncs.hitPointDist = rat;
	p->rayHit=p->rayph;
	p->rayHitHyper=p->rayph;
#ifdef RENDERVERBOSE 
//	printf ("Rayhit, hp.x y z: - %f %f %f rat %f hitPointDist %f\n",hp.x,hp.y,hp.z, rat, tg->RenderFuncs.hitPointDist);
#endif
}


/* Call this when modelview and projection modified */
void upd_ray() {
	struct point_XYZ t_r1,t_r2,t_r3;
	GLDOUBLE modelMatrix[16];
	GLDOUBLE projMatrix[16];
	ttglobal tg = gglobal();
	FW_GL_GETDOUBLEV(GL_MODELVIEW_MATRIX, modelMatrix);
	FW_GL_GETDOUBLEV(GL_PROJECTION_MATRIX, projMatrix);
/*

{int i; printf ("\n"); 
printf ("upd_ray, pm %p\n",projMatrix);
for (i=0; i<16; i++) printf ("%4.3lf ",modelMatrix[i]); printf ("\n"); 
for (i=0; i<16; i++) printf ("%4.3lf ",projMatrix[i]); printf ("\n"); 
} 
*/


	FW_GLU_UNPROJECT(r1.x,r1.y,r1.z,modelMatrix,projMatrix,viewport,
		     &t_r1.x,&t_r1.y,&t_r1.z);
	FW_GLU_UNPROJECT(r2.x,r2.y,r2.z,modelMatrix,projMatrix,viewport,
		     &t_r2.x,&t_r2.y,&t_r2.z);
	FW_GLU_UNPROJECT(r3.x,r3.y,r3.z,modelMatrix,projMatrix,viewport,
		     &t_r3.x,&t_r3.y,&t_r3.z);

	VECCOPY(tg->RenderFuncs.t_r1,t_r1);
	VECCOPY(tg->RenderFuncs.t_r2,t_r2);
	VECCOPY(tg->RenderFuncs.t_r3,t_r3);

/*
	printf("Upd_ray: (%f %f %f)->(%f %f %f) == (%f %f %f)->(%f %f %f)\n",
	r1.x,r1.y,r1.z,r2.x,r2.y,r2.z,
	t_r1.x,t_r1.y,t_r1.z,t_r2.x,t_r2.y,t_r2.z);
*/

}


/* if a node changes, void the display lists */
/* Courtesy of Jochen Hoenicke */

void update_node(struct X3D_Node *node) {
	int i;
	
#ifdef VERBOSE
	printf ("update_node for %d %s nparents %d renderflags %x\n",node, stringNodeType(node->_nodeType),node->_nparents, node->_renderFlags); 
	if (node->_nparents == 0) {
		if (node == rootNode) printf ("...no parents, this IS the rootNode\n"); 
		else printf ("...no parents, this IS NOT the rootNode\n");
	}



	for (i = 0; i < node->_nparents; i++) {
		struct X3D_Node *n = X3D_NODE(node->_parents[i]);
		if( n != 0 ) {
			printf ("	parent %u is %s\n",n,stringNodeType(n->_nodeType));
		} else {
			printf ("	parent %d is NULL\n",i);
		}
	}
#endif

	node->_change ++;

	/* parentVector here yet?? */
	if (node->_parentVector == NULL) {
		return;
	}

	for (i = 0; i < vectorSize(node->_parentVector); i++) {
		struct X3D_Node *n = vector_get(struct X3D_Node *, node->_parentVector,i);
		if(n == node) {
			fprintf(stderr, "Error: self-referential node structure! (node:'%s')\n", stringNodeType(node->_nodeType));
			vector_set(struct X3D_Node*, node->_parentVector, i,NULL);
		} else if( n != 0 ) {
			update_node(n);
		}
	}
}

/*********************************************************************
 *********************************************************************
 *
 * render_node : call the correct virtual functions to render the node
 * depending on what we are doing right now.
 */

//#ifdef RENDERVERBOSE
//static int renderLevel = 0;
//#endif

#define PRINT_NODE(_node, _v)  do {					\
		if (gglobal()->internalc.global_print_opengl_errors && (gglobal()->display._global_gl_err != GL_NO_ERROR)) { \
			printf("Render_node_v %p (%s) PREP: %p REND: %p CH: %p FIN: %p RAY: %p HYP: %p\n",_v, \
			       stringNodeType(_node->_nodeType),	\
			       _v->prep,				\
			       _v->rend,				\
			       _v->children,			\
			       _v->fin,				\
			       _v->rendray,			\
			       gglobal()->RenderFuncs.hypersensitive);			\
			printf("Render_state geom %d light %d sens %d\n", \
			       renderstate()->render_geom,				\
			       renderstate()->render_light,				\
			       renderstate()->render_sensitive);			\
			printf("pchange %d pichange %d \n", _node->_change, _node->_ichange); \
		}							\
	} while (0)

//static int renderLevel = 0;
//#define RENDERVERBOSE


/* poor-man's performance profiler:
   wrap a section of code like this
     profile_start("section1");
	 ...code...
	 profile_end("section1");
	then let the browser loop for 10 seconds 
	and hit period '.' on the keyboard to get a printout
*/

static double Time1970sec(void) {
		static struct timeval mytime;
        gettimeofday(&mytime, NULL);
        return (double) mytime.tv_sec + (double)mytime.tv_usec/1000000.0;
}

static int profile_entry_count = 0;
struct profile_entry {
	char *name;
	double start;
	double accum;
	int hits;
};
static struct profile_entry profile_entries[100];

void profile_start(char *name){
	int ifound, i;
	double sec = Time1970sec();
	ifound = -1;
	for(i=0;i<profile_entry_count;i++){
		if(!strcmp(name,profile_entries[i].name)){
			ifound = i;
			break;
		}
	}
	if(ifound == -1){
		profile_entries[profile_entry_count].name = name;
		profile_entries[profile_entry_count].hits = 0;
		ifound = profile_entry_count;
		profile_entry_count++;
	}
	profile_entries[ifound].start = Time1970sec();
}
void profile_end(char *name){
	int i, ifound = -1;
	for(i=0;i<profile_entry_count;i++){
		if(!strcmp(name,profile_entries[i].name)){
			ifound = i;
			break;
		}
	}
	if(ifound > -1){
		profile_entries[ifound].accum += Time1970sec() - profile_entries[ifound].start;
		profile_entries[ifound].hits++;
	}
}
void profile_print_all(){
	//hit '.' in the graphics window 
	int i;
	printf("frame rate: %9.3f  number of items tracked: %d\n",gglobal()->Mainloop.BrowserFPS, profile_entry_count);
	printf("%15s %10s %15s %10s\n","profile name","hits","time(sec)","% of 1st");
	for(i=0;i<profile_entry_count;i++){
		printf("%15s %10d %15.3lf %10.2lf\n",profile_entries[i].name,profile_entries[i].hits,profile_entries[i].accum,profile_entries[i].accum/profile_entries[0].accum*100.0);
	}
}
void render_node(struct X3D_Node *node) {
	struct X3D_Virt *virt;

	int srg = 0;
	int sch = 0;
	int justGeom = 0;
	struct currayhit *srh;
	ppRenderFuncs p;
	ttglobal tg = gglobal();
	p = (ppRenderFuncs)tg->RenderFuncs.prv;

	X3D_NODE_CHECK(node);

#ifdef RENDERVERBOSE
	p->renderLevel ++;
#endif

	if(!node) {
#ifdef RENDERVERBOSE
		DEBUG_RENDER("%d no node, quick return\n", renderLevel);
		p->renderLevel--;
#endif
		return;
	}

	virt = virtTable[node->_nodeType];

#ifdef RENDERVERBOSE 
	//printf("%d =========================================NODE RENDERED===================================================\n",renderLevel);
	{
		int i;
		for(i=0;i<p->renderLevel;i++) printf(" ");
	}
	printf("%d node %u (%s) , v %u renderFlags %x ",p->renderLevel, node,stringNodeType(node->_nodeType),virt,node->_renderFlags);

	if ((node->_renderFlags & VF_Viewpoint) == VF_Viewpoint) printf (" VF_Viewpoint");
	if ((node->_renderFlags & VF_Geom )== VF_Geom) printf (" VF_Geom");
	if ((node->_renderFlags & VF_localLight )== VF_localLight) printf (" VF_localLight");
	if ((node->_renderFlags & VF_Sensitive) == VF_Sensitive) printf (" VF_Sensitive");
	if ((node->_renderFlags & VF_Blend) == VF_Blend) printf (" VF_Blend");
	if ((node->_renderFlags & VF_Proximity) == VF_Proximity) printf (" VF_Proximity");
	if ((node->_renderFlags & VF_Collision) == VF_Collision) printf (" VF_Collision");
	if ((node->_renderFlags & VF_globalLight) == VF_globalLight) printf (" VF_globalLight");
	if ((node->_renderFlags & VF_hasVisibleChildren) == VF_hasVisibleChildren) printf (" VF_hasVisibleChildren");
	if ((node->_renderFlags & VF_shouldSortChildren) == VF_shouldSortChildren) printf (" VF_shouldSortChildren");
	if ((node->_renderFlags & VF_Other) == VF_Other) printf (" VF_Other");
	/*
	if ((node->_renderFlags & VF_inPickableGroup == VF_inPickableGroup) printf (" VF_inPickableGroup");
	if ((node->_renderFlags & VF_PickingSensor == VF_PickingSensor) printf (" VF_PickingSensor");
	*/
	printf ("\n");

	//printf("PREP: %d REND: %d CH: %d FIN: %d RAY: %d HYP: %d\n",virt->prep, virt->rend, virt->children, virt->fin,
	//       virt->rendray, hypersensitive);
	//printf("%d state: vp %d geom %d light %d sens %d blend %d prox %d col %d ", renderLevel, 
 //        	render_vp,render_geom,render_light,render_sensitive,render_blend,render_proximity,render_collision); 
	//printf("change %d ichange %d \n",node->_change, node->_ichange);
#endif

        /* if we are doing Viewpoints, and we don't have a Viewpoint, don't bother doing anything here */ 
        //if (renderstate()->render_vp == VF_Viewpoint) { 
        if (p->renderstate.render_vp == VF_Viewpoint) { 
                if ((node->_renderFlags & VF_Viewpoint) != VF_Viewpoint) { 
#ifdef RENDERVERBOSE
                        printf ("doing Viewpoint, but this  node is not for us - just returning\n"); 
			p->renderLevel--;
#endif
                        return; 
                } 
        }

	/* are we working through global PointLights, DirectionalLights or SpotLights, but none exist from here on down? */
        if (p->renderstate.render_light == VF_globalLight) { 
                if ((node->_renderFlags & VF_globalLight) != VF_globalLight) { 
#ifdef RENDERVERBOSE
                        printf ("doing globalLight, but this  node is not for us - just returning\n"); 
			p->renderLevel--;
#endif
                        return; 
                }
        }
	justGeom = p->renderstate.render_geom && !p->renderstate.render_sensitive && !p->renderstate.render_blend;
	if(virt->prep) {
		DEBUG_RENDER("rs 2\n");
		profile_start("prep");
		if(justGeom)
			profile_start("prepgeom");
		virt->prep(node);
		profile_end("prep");
		if(justGeom)
			profile_end("prepgeom");
		if(p->renderstate.render_sensitive && !tg->RenderFuncs.hypersensitive) {
			upd_ray();
		}
		PRINT_GL_ERROR_IF_ANY("prep"); PRINT_NODE(node,virt);
	}
	if(p->renderstate.render_proximity && virt->proximity) {
		DEBUG_RENDER("rs 2a\n");
		profile_start("proximity");
		virt->proximity(node);
		profile_end("proximity");
		PRINT_GL_ERROR_IF_ANY("render_proximity"); PRINT_NODE(node,virt);
	}
	
	if(p->renderstate.render_collision && virt->collision) {
		DEBUG_RENDER("rs 2b\n");
		profile_start("collision");
		virt->collision(node);
		profile_end("collision");
		PRINT_GL_ERROR_IF_ANY("render_collision"); PRINT_NODE(node,virt);
	}

	if(p->renderstate.render_geom && !p->renderstate.render_sensitive && virt->rend) {
		DEBUG_RENDER("rs 3\n");
		PRINT_GL_ERROR_IF_ANY("BEFORE render_geom"); PRINT_NODE(node,virt);
		profile_start("rend");
		virt->rend(node);
		profile_end("rend");
		PRINT_GL_ERROR_IF_ANY("render_geom"); PRINT_NODE(node,virt);
	}

	if(p->renderstate.render_other && virt->other )
	{
#ifdef DJTRACK_PICKSENSORS
		DEBUG_RENDER("rs 4a\n");
		virt->other(node); //other() is responsible for push_renderingState(VF_inPickableGroup);
		PRINT_GL_ERROR_IF_ANY("render_other"); PRINT_NODE(node,virt);
#endif
	} //other

	if(p->renderstate.render_sensitive && (node->_renderFlags & VF_Sensitive)) {
		DEBUG_RENDER("rs 5\n");
		profile_start("sensitive");
		srg = p->renderstate.render_geom;
		p->renderstate.render_geom = 1;
		DEBUG_RENDER("CH1 %d: %d\n",node, p->cur_hits, node->_hit);
		sch = p->cur_hits;
		p->cur_hits = 0;
		/* HP */
		srh = malloc(sizeof(struct currayhit));
		//srh = p->rayph;
		memcpy(srh,&p->rayph,sizeof(struct currayhit));
		p->rayph.hitNode = node;
		FW_GL_GETDOUBLEV(GL_MODELVIEW_MATRIX, p->rayph.modelMatrix);
		FW_GL_GETDOUBLEV(GL_PROJECTION_MATRIX, p->rayph.projMatrix);
		PRINT_GL_ERROR_IF_ANY("render_sensitive"); PRINT_NODE(node,virt);
		profile_end("sensitive");
	}

	if(p->renderstate.render_geom && p->renderstate.render_sensitive && !tg->RenderFuncs.hypersensitive && virt->rendray) {
		DEBUG_RENDER("rs 6\n");
		profile_start("rendray");
		virt->rendray(node);
		profile_end("rendray");
		PRINT_GL_ERROR_IF_ANY("rs 6"); PRINT_NODE(node,virt);
	}

    if((p->renderstate.render_sensitive) && (tg->RenderFuncs.hypersensitive == node)) {
		DEBUG_RENDER("rs 7\n");
		p->hyper_r1 = tg->RenderFuncs.t_r1;
		p->hyper_r2 = tg->RenderFuncs.t_r2;
		tg->RenderFuncs.hyperhit = 1;
    }

	/* start recursive section */
    if(virt->children) { 
		DEBUG_RENDER("rs 8 - has valid child node pointer\n");
		virt->children(node);
		PRINT_GL_ERROR_IF_ANY("children"); PRINT_NODE(node,virt);
    }
	/* end recursive section */

	if(p->renderstate.render_other && virt->other)
	{
#ifdef DJTRACK_PICKSENSORS
		//pop_renderingState(VF_inPickableGroup);
#endif
	}

	if(p->renderstate.render_sensitive && (node->_renderFlags & VF_Sensitive)) {
		DEBUG_RENDER("rs 9\n");
		profile_start("sensitive2");

		p->renderstate.render_geom = srg;
		p->cur_hits = sch;
		DEBUG_RENDER("CH3: %d %d\n",p->cur_hits, node->_hit);
		/* HP */
		//p->rayph = srh;
		memcpy(&p->rayph,srh,sizeof(struct currayhit));
		free(srh);
		profile_end("sensitive2");
	}

	if(virt->fin) {
		DEBUG_RENDER("rs A\n");
		profile_start("fin");
		if(justGeom)
			profile_start("fingeom");

		virt->fin(node);
		profile_end("fin");
		if(justGeom)
			profile_end("fingeom");
		if(p->renderstate.render_sensitive && virt == &virt_Transform) {
			upd_ray();
		}
		PRINT_GL_ERROR_IF_ANY("fin"); PRINT_NODE(node,virt);
	}

#ifdef RENDERVERBOSE 
	{
		int i;
		for(i=0;i<p->renderLevel;i++)printf(" ");
	}
	printf("%d (end render_node)\n",p->renderLevel);
	p->renderLevel--;
#endif
}
//#undef RENDERVERBOSE

/*
 * The following code handles keeping track of the parents of a given
 * node. This enables us to traverse the scene on C level for optimizations.
 *
 * We use this array code because VRML nodes usually don't have
 * hundreds of children and don't usually shuffle them too much.
 */
//dug9 dec 13 >>
struct X3D_Node* getTypeNode(struct X3D_Node *node);

void add_parent(struct X3D_Node *node, struct X3D_Node *parent, char *file, int line) {
	struct X3D_Node* itype;
	if(!node) return;
	//if(node->_nodeType == NODE_PlaneSensor)
	//	printf("hi from add_parent, have a Planesensor");
#ifdef CHILDVERBOSE
	printf ("add_parent; adding node %u ,to parent %u at %s:%d\n",node,  parent,file,line);
	printf ("add_parent; adding node %x ,to parent %x (hex) at %s:%d\n",node,  parent,file,line);
	printf ("add_parent; adding node %p ,to parent %p (ptr) at %s:%d\n",node,  parent,file,line);


	printf ("add_parent; adding node %u (%s) to parent %u (%s) at %s:%d\n",node, stringNodeType(node->_nodeType), 
		parent, stringNodeType(parent->_nodeType),file,line);
#endif

	parent->_renderFlags = parent->_renderFlags | node->_renderFlags;

	/* add it to the parents list */
	vector_pushBack (struct X3D_Node*,node->_parentVector, parent);
	/* tie in sensitive nodes */
	itype = getTypeNode(node);
	if(itype)
		setSensitive (parent, itype );
}

void remove_parent(struct X3D_Node *child, struct X3D_Node *parent) {
	int i;
	int pi;

	if(!child) return;
	if(!parent) return;
	
#ifdef CHILDVERBOSE
	printf ("remove_parent, parent %u (%s) , child %u (%s)\n",parent, stringNodeType(parent->_nodeType),
		child, stringNodeType(child->_nodeType));
#endif

		pi = -1;
		for (i=0; i<vectorSize(child->_parentVector); i++) {
			struct X3D_Node *n = vector_get(struct X3D_Node *, child->_parentVector,i);
			if (n==parent) pi = i;
		}

		if (pi >=0) {
			struct X3D_Node *n = vector_get(struct X3D_Node *, child->_parentVector,vectorSize(child->_parentVector)-1);

			/* get the last entry, and overwrite the entry found */
			vector_set(struct X3D_Node*, child->_parentVector, pi,n);

			/* take that last entry off the vector */
			vector_popBack(struct X3D_Node*, child->_parentVector);
	}
}

void
render_hier(struct X3D_Group *g, int rwhat) {
	/// not needed now - see below struct point_XYZ upvec = {0,1,0};
	/// not needed now - see below GLDOUBLE modelMatrix[16];

#ifdef render_pre_profile
	/*  profile */
	double xx,yy,zz,aa,bb,cc,dd,ee,ff;
	struct timeval mytime;
	struct timezone tz; /* unused see man gettimeofday */
#endif
	ppRenderFuncs p;
	ttglobal tg = gglobal();
	ttrenderstate rs;
	p = (ppRenderFuncs)tg->RenderFuncs.prv;
	rs = renderstate();

	rs->render_vp = rwhat & VF_Viewpoint;
	rs->render_geom =  rwhat & VF_Geom;
	rs->render_light = rwhat & VF_globalLight;
	rs->render_sensitive = rwhat & VF_Sensitive;
	rs->render_blend = rwhat & VF_Blend;
	rs->render_proximity = rwhat & VF_Proximity;
	rs->render_collision = rwhat & VF_Collision;
	rs->render_other = rwhat & VF_Other;
#ifdef DJTRACK_PICKSENSORS
	rs->render_picksensors = rwhat & VF_PickingSensor;
	rs->render_pickables = rwhat & VF_inPickableGroup;
#endif
	p->nextFreeLight = 0;
	p->lastShader = -1; //in sendLights,and optimization
	tg->RenderFuncs.hitPointDist = -1;


#ifdef render_pre_profile
	if (rs->render_geom) {
		gettimeofday (&mytime,&tz);
		aa = (double)mytime.tv_sec+(double)mytime.tv_usec/1000000.0;
	}
#endif
#ifdef RENDERVERBOSE
	 printf ("render_hier vp %d geom %d light %d sens %d blend %d prox %d col %d\n",
	   rs->render_vp,rs->render_geom,rs->render_light,rs->render_sensitive,rs->render_blend,rs->render_proximity,rs->render_collision);  
#endif

	if (!g) {
		/* we have no geometry yet, sleep for a tiny bit */
		usleep(1000);
		return;
	}

#ifdef RENDERVERBOSE
	printf("Render_hier node=%d what=%d\n", g, rwhat);
#endif

#ifdef render_pre_profile
	if (rs->render_geom) {
		gettimeofday (&mytime,&tz);
		bb = (double)mytime.tv_sec+(double)mytime.tv_usec/1000000.0;
	}
#endif

	if (rs->render_sensitive) {
		upd_ray();
	}

#ifdef render_pre_profile
	if (rs->render_geom) {
		gettimeofday (&mytime,&tz);
		cc = (double)mytime.tv_sec+(double)mytime.tv_usec/1000000.0;
	}
#endif
	//if(!rs->render_geom || rs->render_blend || rs->render_sensitive)
	//if(!rs->render_geom || rs->render_blend )
	//if(!rs->render_geom || rs->render_sensitive)
	//if(!rs->render_geom || rs->render_blend || rs->render_sensitive)
	//if(!rs->render_blend && !rs->render_sensitive)
	profile_start("render_hier");
	render_node(X3D_NODE(g));
	profile_end("render_hier");
#ifdef render_pre_profile
	if (rs->render_geom) {
		gettimeofday (&mytime,&tz);
		dd = (double)mytime.tv_sec+(double)mytime.tv_usec/1000000.0;
		printf ("render_geom status %f ray %f geom %f\n",bb-aa, cc-bb, dd-cc);
	}
#endif
}


/******************************************************************************
 *
 * shape compiler "thread"
 *
 ******************************************************************************/

void compileNode (void (*nodefn)(void *, void *, void *, void *, void *), void *node, void *Icoord, void *Icolor, void *Inormal, void *ItexCoord) {
	void *coord; void *color; void *normal; void *texCoord;

	/* are any of these SFNodes PROTOS? If so, get the underlying real node, as PROTOS are handled like Groups. */
	POSSIBLE_PROTO_EXPANSION(void *, Icoord,coord)
		POSSIBLE_PROTO_EXPANSION(void *, Icolor,color)
		POSSIBLE_PROTO_EXPANSION(void *, Inormal,normal)
		POSSIBLE_PROTO_EXPANSION(void *, ItexCoord,texCoord)

	nodefn(node, coord, color, normal, texCoord);
}


/* for CRoutes, we need to have a function pointer to an interpolator to run, if we
   route TO an interpolator */
void *returnInterpolatorPointer (const char *x) {
	if (strcmp("OrientationInterpolator",x)==0) { return (void *)do_Oint4;
	} else if (strcmp("CoordinateInterpolator2D",x)==0) { return (void *)do_OintCoord2D;
	} else if (strcmp("PositionInterpolator2D",x)==0) { return (void *)do_OintPos2D;
	} else if (strcmp("ScalarInterpolator",x)==0) { return (void *)do_OintScalar;
	} else if (strcmp("ColorInterpolator",x)==0) { return (void *)do_ColorInterpolator;
	} else if (strcmp("PositionInterpolator",x)==0) { return (void *)do_PositionInterpolator;
	} else if (strcmp("CoordinateInterpolator",x)==0) { return (void *)do_OintCoord;
	} else if (strcmp("NormalInterpolator",x)==0) { return (void *)do_OintNormal;
	} else if (strcmp("GeoPositionInterpolator",x)==0) { return (void *)do_GeoPositionInterpolator;
	} else if (strcmp("BooleanFilter",x)==0) { return (void *)do_BooleanFilter;
	} else if (strcmp("BooleanSequencer",x)==0) { return (void *)do_BooleanSequencer;
	} else if (strcmp("BooleanToggle",x)==0) { return (void *)do_BooleanToggle;
	} else if (strcmp("BooleanTrigger",x)==0) { return (void *)do_BooleanTrigger;
	} else if (strcmp("IntegerTrigger",x)==0) { return (void *)do_IntegerTrigger;
	} else if (strcmp("IntegerSequencer",x)==0) { return (void *)do_IntegerSequencer;
	} else if (strcmp("TimeTrigger",x)==0) { return (void *)do_TimeTrigger;
	
	} else {
		return 0;
	}
}




void checkParentLink (struct X3D_Node *node,struct X3D_Node *parent) {
        int n;

	int *offsetptr;
	char *memptr;
	struct Multi_Node *mfn;
	uintptr_t *voidptr;

        if (node == NULL) return;

	/* printf ("checkParentLink for node %u parent %u type %s\n",node,parent,stringNodeType(node->_nodeType)); */
 
        if (parent != NULL) ADD_PARENT(node, parent);

	if ((node->_nodeType<0) || (node->_nodeType>NODES_COUNT)) {
		ConsoleMessage ("checkParentLink - %d not a valid nodeType",node->_nodeType);
		return;
	}

	/* find all the fields of this node */
	offsetptr = (int *) NODE_OFFSETS[node->_nodeType];

	/* FIELDNAMES_bboxCenter, offsetof (struct X3D_Group, bboxCenter),  FIELDTYPE_SFVec3f, KW_field, */
	while (*offsetptr >= 0) {

		/* 
		   printf ("	field %s",FIELDNAMES[offsetptr[0]]); 
		   printf ("	offset %d",offsetptr[1]);
		   printf ("	type %s",FIELDTYPES[offsetptr[2]]);
		   printf ("	kind %s\n",KEYWORDS[offsetptr[3]]);
		*/

		/* worry about SFNodes and MFNodes */
		if ((offsetptr[2] == FIELDTYPE_SFNode) || (offsetptr[2] == FIELDTYPE_MFNode)) {
			if ((offsetptr[3] == KW_initializeOnly) || (offsetptr[3] == KW_inputOutput)) {

				/* create a pointer to the actual field */
				memptr = (char *) node;
				memptr += offsetptr[1];

				if (offsetptr[2] == FIELDTYPE_SFNode) {
					/* get the field as a POINTER VALUE, not just a pointer... */
					voidptr = (uintptr_t *) memptr;
					voidptr = (uintptr_t *) *voidptr;

					/* is there a node here? */
					if (voidptr != NULL) {
						checkParentLink(X3D_NODE(voidptr),node);
					}
				} else {
					mfn = (struct Multi_Node*) memptr;
					/* printf ("MFNode has %d children\n",mfn->n); */
					for (n=0; n<mfn->n; n++) {
				                checkParentLink(mfn->p[n],node);
					}
				}
			}

		}
		offsetptr+=5;
	}
}

#define X3D_COORD(node) ((struct X3D_Coordinate*)node)
#define X3D_GEOCOORD(node) ((struct X3D_GeoCoordinate*)node)

/* get a coordinate array - (SFVec3f) from either a NODE_Coordinate or NODE_GeoCoordinate */
struct Multi_Vec3f *getCoordinate (struct X3D_Node *innode, char *str) {
	struct X3D_Coordinate * xc;
	struct X3D_GeoCoordinate *gxc;
	struct X3D_Node *node;

	POSSIBLE_PROTO_EXPANSION (struct X3D_Node *, innode,node)

		xc = X3D_COORD(node);
	/* printf ("getCoordinate, have a %s\n",stringNodeType(xc->_nodeType)); */

	if (xc->_nodeType == NODE_Coordinate) {
		return &(xc->point);
	} else if (xc->_nodeType == NODE_GeoCoordinate) {
		COMPILE_IF_REQUIRED_RETURN_NULL_ON_ERROR;
		gxc = X3D_GEOCOORD(node);
		return &(gxc->__movedCoords);
	} else {
		ConsoleMessage ("%s - coord expected but got %s\n", stringNodeType(xc->_nodeType));
	}
	return NULL;
}



/*
void printLTDebug(char * fn, int ln)
{
 ppRenderFuncs p = (ppRenderFuncs)gglobal()->RenderFuncs.prv;
ConsoleMessage ("headlight pos %f %f %f %f at %s:%d",
p->light_pos[HEADLIGHT_LIGHT][0],p->light_pos[HEADLIGHT_LIGHT][1],
                p->light_pos[HEADLIGHT_LIGHT][2],p->light_pos[HEADLIGHT_LIGHT][3],fn,ln);
}
*/

