/* like cube, an exact reproducation of egl command stream
 * but try to generate the command stream ourselves using etna_XXX */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <stdarg.h>
#include <string.h>

#include <etnaviv/common.xml.h>
#include <etnaviv/state.xml.h>
#include <etnaviv/state_3d.xml.h>
#include <etnaviv/cmdstream.xml.h>
#include "viv_raw.h"
#include "write_bmp.h"
#include "esTransform.h"

/* Print generated command buffer */
//#define CMD_DEBUG
/* Compare generated with stored command buffers generated by Vivante driver */
#define CMD_COMPARE

#ifdef CMD_COMPARE
#include "cube_cmd_gc880.h"
#endif
/* TODO: should actually update context as we go,
   a context switch would currently revert state and likely result in corrupted rendering.
 */

float vVertices[] = {
  // front
  -1.0f, -1.0f, +1.0f, // point blue
  +1.0f, -1.0f, +1.0f, // point magenta
  -1.0f, +1.0f, +1.0f, // point cyan
  +1.0f, +1.0f, +1.0f, // point white
  // back
  +1.0f, -1.0f, -1.0f, // point red
  -1.0f, -1.0f, -1.0f, // point black
  +1.0f, +1.0f, -1.0f, // point yellow
  -1.0f, +1.0f, -1.0f, // point green
  // right
  +1.0f, -1.0f, +1.0f, // point magenta
  +1.0f, -1.0f, -1.0f, // point red
  +1.0f, +1.0f, +1.0f, // point white
  +1.0f, +1.0f, -1.0f, // point yellow
  // left
  -1.0f, -1.0f, -1.0f, // point black
  -1.0f, -1.0f, +1.0f, // point blue
  -1.0f, +1.0f, -1.0f, // point green
  -1.0f, +1.0f, +1.0f, // point cyan
  // top
  -1.0f, +1.0f, +1.0f, // point cyan
  +1.0f, +1.0f, +1.0f, // point white
  -1.0f, +1.0f, -1.0f, // point green
  +1.0f, +1.0f, -1.0f, // point yellow
  // bottom
  -1.0f, -1.0f, -1.0f, // point black
  +1.0f, -1.0f, -1.0f, // point red
  -1.0f, -1.0f, +1.0f, // point blue
  +1.0f, -1.0f, +1.0f  // point magenta
};

float vColors[] = {
  // front
  0.0f,  0.0f,  1.0f, // blue
  1.0f,  0.0f,  1.0f, // magenta
  0.0f,  1.0f,  1.0f, // cyan
  1.0f,  1.0f,  1.0f, // white
  // back
  1.0f,  0.0f,  0.0f, // red
  0.0f,  0.0f,  0.0f, // black
  1.0f,  1.0f,  0.0f, // yellow
  0.0f,  1.0f,  0.0f, // green
  // right
  1.0f,  0.0f,  1.0f, // magenta
  1.0f,  0.0f,  0.0f, // red
  1.0f,  1.0f,  1.0f, // white
  1.0f,  1.0f,  0.0f, // yellow
  // left
  0.0f,  0.0f,  0.0f, // black
  0.0f,  0.0f,  1.0f, // blue
  0.0f,  1.0f,  0.0f, // green
  0.0f,  1.0f,  1.0f, // cyan
  // top
  0.0f,  1.0f,  1.0f, // cyan
  1.0f,  1.0f,  1.0f, // white
  0.0f,  1.0f,  0.0f, // green
  1.0f,  1.0f,  0.0f, // yellow
  // bottom
  0.0f,  0.0f,  0.0f, // black
  1.0f,  0.0f,  0.0f, // red
  0.0f,  0.0f,  1.0f, // blue
  1.0f,  0.0f,  1.0f  // magenta
};

float vNormals[] = {
  // front
  +0.0f, +0.0f, +1.0f, // forward
  +0.0f, +0.0f, +1.0f, // forward
  +0.0f, +0.0f, +1.0f, // forward
  +0.0f, +0.0f, +1.0f, // forward
  // back
  +0.0f, +0.0f, -1.0f, // backbard
  +0.0f, +0.0f, -1.0f, // backbard
  +0.0f, +0.0f, -1.0f, // backbard
  +0.0f, +0.0f, -1.0f, // backbard
  // right
  +1.0f, +0.0f, +0.0f, // right
  +1.0f, +0.0f, +0.0f, // right
  +1.0f, +0.0f, +0.0f, // right
  +1.0f, +0.0f, +0.0f, // right
  // left
  -1.0f, +0.0f, +0.0f, // left
  -1.0f, +0.0f, +0.0f, // left
  -1.0f, +0.0f, +0.0f, // left
  -1.0f, +0.0f, +0.0f, // left
  // top
  +0.0f, +1.0f, +0.0f, // up
  +0.0f, +1.0f, +0.0f, // up
  +0.0f, +1.0f, +0.0f, // up
  +0.0f, +1.0f, +0.0f, // up
  // bottom
  +0.0f, -1.0f, +0.0f, // down
  +0.0f, -1.0f, +0.0f, // down
  +0.0f, -1.0f, +0.0f, // down
  +0.0f, -1.0f, +0.0f  // down
};
#define COMPONENTS_PER_VERTEX (3)
#define NUM_VERTICES (6*4)
#define VERTICES_PER_DRAW 4
#define DRAW_COUNT 6

#ifdef CMD_COMPARE
char is_padding[0x8000 / 4];
#endif

/* macro for MASKED() (multiple can be &ed) */
#define VIV_MASKED(NAME, VALUE) (~(NAME ## _MASK | NAME ## __MASK) | ((VALUE)<<(NAME ## __SHIFT)))
/* for boolean bits */
#define VIV_MASKED_BIT(NAME, VALUE) (~(NAME ## _MASK | NAME) | ((VALUE) ? NAME : 0))
/* for inline enum bit fields 
 * XXX in principle headergen could simply generate these fields prepackaged 
 */
#define VIV_MASKED_INL(NAME, VALUE) (~(NAME ## _MASK | NAME ## __MASK) | (NAME ## _ ## VALUE))

/* XXX store state changes
 * group consecutive states 
 * make LOAD_STATE commands, add to current command buffer
 */
static inline void etna_set_state(gcoCMDBUF commandBuffer, uint32_t address, uint32_t value)
{
#ifdef CMD_DEBUG
    printf("%05x := %08x\n", address, value);
#endif
    uint32_t *tgt = (uint32_t*)((size_t)commandBuffer->logical + commandBuffer->offset);
    tgt[0] = VIV_FE_LOAD_STATE_HEADER_OP_LOAD_STATE |
            VIV_FE_LOAD_STATE_HEADER_COUNT(1) |
            VIV_FE_LOAD_STATE_HEADER_OFFSET(address >> 2);
    tgt[1] = value;
    commandBuffer->offset += 8;
}

/* this can be inlined, though would likely be even faster to return a pointer and let the client write to
 * the buffer directly */
static inline void etna_set_state_multi(gcoCMDBUF commandBuffer, uint32_t base, uint32_t num, uint32_t *values)
{
    uint32_t *tgt = (uint32_t*)((size_t)commandBuffer->logical + commandBuffer->offset);
    tgt[0] = VIV_FE_LOAD_STATE_HEADER_OP_LOAD_STATE |
            VIV_FE_LOAD_STATE_HEADER_COUNT(num & 0x3ff) |
            VIV_FE_LOAD_STATE_HEADER_OFFSET(base >> 2);
#ifdef CMD_DEBUG
    for(uint32_t idx=0; idx<num; ++idx)
    {
        printf("%05x := %08x\n", base, values[idx]);
        base += 4;
    }
#endif
    memcpy(&tgt[1], values, 4*num);
    commandBuffer->offset += 4 + num*4;
    if(commandBuffer->offset & 4) /* PAD */
    {
#ifdef CMD_COMPARE
        is_padding[commandBuffer->offset / 4] = 1;
#endif
        commandBuffer->offset += 4;
    }
}
static inline void etna_set_state_multi_fixp(gcoCMDBUF commandBuffer, uint32_t base, uint32_t num, uint32_t *values)
{
    uint32_t *tgt = (uint32_t*)((size_t)commandBuffer->logical + commandBuffer->offset);
    tgt[0] = VIV_FE_LOAD_STATE_HEADER_OP_LOAD_STATE |
    VIV_FE_LOAD_STATE_HEADER_FIXP |
    VIV_FE_LOAD_STATE_HEADER_COUNT(num & 0x3ff) |
    VIV_FE_LOAD_STATE_HEADER_OFFSET(base >> 2);
    #ifdef CMD_DEBUG
    for(uint32_t idx=0; idx<num; ++idx)
    {
        printf("%05x := %08x\n", base, values[idx]);
        base += 4;
    }
    #endif
    memcpy(&tgt[1], values, 4*num);
    commandBuffer->offset += 4 + num*4;
    if(commandBuffer->offset & 4) /* PAD */
    {
        #ifdef CMD_COMPARE
        is_padding[commandBuffer->offset / 4] = 1;
        #endif
        commandBuffer->offset += 4;
    }
}
static inline void etna_set_state_f32(gcoCMDBUF commandBuffer, uint32_t address, float value)
{
    union {
        uint32_t i32;
        float f32;
    } x = { .f32 = value };
    etna_set_state(commandBuffer, address, x.i32);
}
static inline void etna_set_state_fixp(gcoCMDBUF commandBuffer, uint32_t address, uint32_t value)
{
#ifdef CMD_DEBUG
    printf("%05x := %08x (fixp)\n", address, value);
#endif
    uint32_t *tgt = (uint32_t*)((size_t)commandBuffer->logical + commandBuffer->offset);
    tgt[0] = VIV_FE_LOAD_STATE_HEADER_OP_LOAD_STATE |
            VIV_FE_LOAD_STATE_HEADER_COUNT(1) |
            VIV_FE_LOAD_STATE_HEADER_FIXP |
            VIV_FE_LOAD_STATE_HEADER_OFFSET(address >> 2);
    tgt[1] = value;
    commandBuffer->offset += 8;
}
static inline void etna_draw_primitives(gcoCMDBUF cmdPtr, uint32_t primitive_type, uint32_t start, uint32_t count)
{
#ifdef CMD_DEBUG
    printf("draw_primitives %08x %08x %08x %08x\n", 
            VIV_FE_DRAW_PRIMITIVES_HEADER_OP_DRAW_PRIMITIVES,
            primitive_type, start, count);
#endif
    uint32_t *tgt = (uint32_t*)((size_t)cmdPtr->logical + cmdPtr->offset);
    tgt[0] = VIV_FE_DRAW_PRIMITIVES_HEADER_OP_DRAW_PRIMITIVES;
    tgt[1] = primitive_type;
    tgt[2] = start;
    tgt[3] = count;
    cmdPtr->offset += 16;
}
static inline void etna_stall_9(gcoCMDBUF commandBuffer)
{
    uint32_t *tgt = (uint32_t*)((size_t)commandBuffer->logical + commandBuffer->offset);
    tgt[0] = 0x48000000;
    tgt[1] = 0x00000701;
    commandBuffer->offset += 8;
}

#ifdef CMD_COMPARE
static int cmdbuffer_compare(gcoCMDBUF cmdPtr, uint32_t *cmdbuf, uint32_t cmdbuf_size)
{
    /* Count differences between generated and stored command buffer */
    int diff = 0;
    for(int idx=8; idx<cmdbuf_size/4; ++idx)
    {
        uint32_t cmdbuf_word = cmdbuf[idx];
        uint32_t my_word = *(uint32_t*)((size_t)cmdPtr->logical + cmdPtr->startOffset + idx*4);
        printf("/*%03x*/ %08x ref:%08x ", idx, my_word, cmdbuf_word);
        if(is_padding[cmdPtr->startOffset/4 + idx])
        {
            printf("PAD");
        } else if(cmdbuf_word != my_word)
        {
            diff += 1;
            printf("DIFF");
        }
        printf("\n");
    }
    printf("Number of differences: %i\n", diff);
    uint32_t size_cmd = cmdbuf_size/4;
    uint32_t size_my = (cmdPtr->offset - cmdPtr->startOffset)/4;
    printf("Sizes: %i %i\n", size_cmd, size_my);
    return diff != 0 || size_cmd != size_my;
}
#endif

static inline uint32_t align_up(uint32_t value, uint32_t granularity)
{
    return (value + (granularity-1)) & (~(granularity-1));
}

int main(int argc, char **argv)
{
    int rv;
    int width = 400;
    int height = 240;
    int padded_width = align_up(width, 64);
    int padded_height = align_up(height, 64);
    printf("padded_width %i padded_height %i\n", padded_width, padded_height);
    struct viv_conn *conn = 0;
    rv = viv_open(VIV_HW_3D, &conn);
    if(rv!=0)
    {
        fprintf(stderr, "Error opening device\n");
        exit(1);
    }
    printf("Succesfully opened device\n");
    viv_show_chip_info(conn);
    
    gcsHAL_INTERFACE id = {};
    id.command = gcvHAL_ATTACH;
    if((viv_invoke(conn, &id)) != gcvSTATUS_OK)
    {
        #ifdef DEBUG
        fprintf(stderr, "Error attaching to GPU\n");
        #endif
        exit(1);
    }
    gckCONTEXT context = id.u.Attach.context;

    /* allocate command buffer (blob uses four command buffers, but we don't even fill one) */
    viv_addr_t buf0_physical = 0;
    void *buf0_logical = 0;
    if(viv_alloc_contiguous(conn, 0x20000, &buf0_physical, &buf0_logical, NULL)!=0)
    {
        fprintf(stderr, "Error allocating host memory\n");
        exit(1);
    }
    printf("Allocated buffer: phys=%08x log=%08x\n", (uint32_t)buf0_physical, (uint32_t)buf0_logical);

    /* allocate main render target */
    gcuVIDMEM_NODE_PTR rt_node = 0;
    if(viv_alloc_linear_vidmem(conn, (padded_width * padded_height * 4), 0x40, gcvSURF_RENDER_TARGET, gcvPOOL_DEFAULT, &rt_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating render target buffer memory\n");
        exit(1);
    }
    printf("Allocated render target node: node=%08x\n", (uint32_t)rt_node);
    
    viv_addr_t rt_physical = 0; /* ADDR_A */
    void *rt_logical = 0;
    if(viv_lock_vidmem(conn, rt_node, &rt_physical, &rt_logical)!=0)
    {
        fprintf(stderr, "Error locking render target memory\n");
        exit(1);
    }
    printf("Locked render target: phys=%08x log=%08x\n", (uint32_t)rt_physical, (uint32_t)rt_logical);
    memset(rt_logical, 0xff, padded_width * padded_height * 4); /* clear previous result just in case, test that clearing works */

    /* allocate tile status for main render target */
    gcuVIDMEM_NODE_PTR rt_ts_node = 0;
    uint32_t rt_ts_size = align_up((padded_width * padded_height * 4)/0x100, 0x100);
    if(viv_alloc_linear_vidmem(conn, rt_ts_size, 0x40, gcvSURF_TILE_STATUS, gcvPOOL_DEFAULT, &rt_ts_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating render target tile status memory\n");
        exit(1);
    }
    printf("Allocated render target tile status node: node=%08x size=%08x\n", (uint32_t)rt_ts_node, rt_ts_size);
    
    viv_addr_t rt_ts_physical = 0; /* ADDR_B */
    void *rt_ts_logical = 0;
    if(viv_lock_vidmem(conn, rt_ts_node, &rt_ts_physical, &rt_ts_logical)!=0)
    {
        fprintf(stderr, "Error locking render target memory\n");
        exit(1);
    }
    printf("Locked render target ts: phys=%08x log=%08x\n", (uint32_t)rt_ts_physical, (uint32_t)rt_ts_logical);

    /* allocate depth for main render target */
    gcuVIDMEM_NODE_PTR z_node = 0;
    if(viv_alloc_linear_vidmem(conn, padded_width * padded_height * 2, 0x40, gcvSURF_DEPTH, gcvPOOL_DEFAULT, &z_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating depth memory\n");
        exit(1);
    }
    printf("Allocated depth node: node=%08x\n", (uint32_t)z_node);
    
    viv_addr_t z_physical = 0; /* ADDR_C */
    void *z_logical = 0;
    if(viv_lock_vidmem(conn, z_node, &z_physical, &z_logical)!=0)
    {
        fprintf(stderr, "Error locking depth target memory\n");
        exit(1);
    }
    printf("Locked depth target: phys=%08x log=%08x\n", (uint32_t)z_physical, (uint32_t)z_logical);

    /* allocate depth ts for main render target */
    gcuVIDMEM_NODE_PTR z_ts_node = 0;
    uint32_t z_ts_size = align_up((padded_width * padded_height * 2)/0x100, 0x100);
    if(viv_alloc_linear_vidmem(conn, z_ts_size, 0x40, gcvSURF_TILE_STATUS, gcvPOOL_DEFAULT, &z_ts_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating depth memory\n");
        exit(1);
    }
    printf("Allocated depth ts node: node=%08x size=%08x\n", (uint32_t)z_ts_node, z_ts_size);
    
    viv_addr_t z_ts_physical = 0; /* ADDR_D */
    void *z_ts_logical = 0;
    if(viv_lock_vidmem(conn, z_ts_node, &z_ts_physical, &z_ts_logical)!=0)
    {
        fprintf(stderr, "Error locking depth target ts memory\n");
        exit(1);
    }
    printf("Locked depth ts target: phys=%08x log=%08x\n", (uint32_t)z_ts_physical, (uint32_t)z_ts_logical);

    /* allocate vertex buffer */
    gcuVIDMEM_NODE_PTR vtx_node = 0;
    if(viv_alloc_linear_vidmem(conn, 0x100000, 0x40, gcvSURF_VERTEX, gcvPOOL_DEFAULT, &vtx_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating vertex memory\n");
        exit(1);
    }
    printf("Allocated vertex node: node=%08x\n", (uint32_t)vtx_node);
    
    viv_addr_t vtx_physical = 0; /* ADDR_E */
    void *vtx_logical = 0;
    if(viv_lock_vidmem(conn, vtx_node, &vtx_physical, &vtx_logical)!=0)
    {
        fprintf(stderr, "Error locking vertex memory\n");
        exit(1);
    }
    printf("Locked vertex memory: phys=%08x log=%08x\n", (uint32_t)vtx_physical, (uint32_t)vtx_logical);

    /* allocate aux render target */
    gcuVIDMEM_NODE_PTR aux_rt_node = 0;
    if(viv_alloc_linear_vidmem(conn, padded_width * padded_height * 4, 0x40, gcvSURF_BITMAP, gcvPOOL_DEFAULT, &aux_rt_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating aux render target buffer memory\n");
        exit(1);
    }
    printf("Allocated aux render target node: node=%08x\n", (uint32_t)aux_rt_node);
    
    viv_addr_t aux_rt_physical = 0; /* ADDR_F */
    void *aux_rt_logical = 0;
    if(viv_lock_vidmem(conn, aux_rt_node, &aux_rt_physical, &aux_rt_logical)!=0)
    {
        fprintf(stderr, "Error locking aux render target memory\n");
        exit(1);
    }
    printf("Locked aux render target: phys=%08x log=%08x\n", (uint32_t)aux_rt_physical, (uint32_t)aux_rt_logical);

    /* Phew, now we got all the memory we need.
     * Write interleaved attribute vertex stream.
     * Unlike the GL example we only do this once, not every time glDrawArrays is called, the same would be accomplished
     * from GL by using a vertex buffer object.
     */
    int dest_idx = 0;
    int v_src_idx = 0;
    int n_src_idx = 0;
    int c_src_idx = 0;
    for(int jj=0; jj<DRAW_COUNT; jj++)
    {
        for(int vert=0; vert<VERTICES_PER_DRAW*3; ++vert)
        {
            ((float*)vtx_logical)[dest_idx] = vVertices[v_src_idx];
            dest_idx++;
            v_src_idx++;
        }
        for(int vert=0; vert<VERTICES_PER_DRAW*3; ++vert)
        {
            ((float*)vtx_logical)[dest_idx] = vNormals[n_src_idx];
            dest_idx++;
            n_src_idx++;
        }
        for(int vert=0; vert<VERTICES_PER_DRAW*3; ++vert)
        {
            ((float*)vtx_logical)[dest_idx] = vColors[c_src_idx];
            dest_idx++;
            c_src_idx++;
        }
    }
    /*
    for(int idx=0; idx<NUM_VERTICES*3*3; ++idx)
    {
        printf("%i %f\n", idx, ((float*)vtx_logical)[idx]);
    }*/

    struct _gcoCMDBUF commandBuffer = {
        .object = {
            .type = gcvOBJ_COMMANDBUFFER
        },
        //.os = (_gcoOS*)0xbf7488,
        //.hardware = (_gcoHARDWARE*)0x402694e0,
        .physical = (void*)buf0_physical,
        .logical = (void*)buf0_logical,
        .bytes = 0x8000,
        .startOffset = 0x0,
        //.offset = 0xac0,
        //.free = 0x7520,
        //.hintTable = (unsigned int*)0x0, // Used when gcdSECURE
        //.hintIndex = (unsigned int*)0x58,  // Used when gcdSECURE
        //.hintCommit = (unsigned int*)0xffffffff // Used when gcdSECURE
    };
    struct _gcoCMDBUF *cmdPtr = &commandBuffer;
    
    commandBuffer.free = commandBuffer.bytes - 0x8; /* Always keep 0x8 at end of buffer for kernel driver */
    commandBuffer.startOffset = 0;
    commandBuffer.offset = commandBuffer.startOffset + 8*4;
#ifdef CMD_COMPARE
    memset(is_padding, 0, 0x8000/4); /* keep track of what words in the command buffer are padding; just for debugging / comparing */
#endif

    /* XXX how important is the ordering? I suppose we could group states (except the flushes, kickers, semaphores etc)
     * and simply submit them at once. Especially for consecutive states and masked stated this could be a big win
     * in DMA command buffer size. */

    /* Build first command buffer */
    etna_set_state(cmdPtr, VIVS_GL_VERTEX_ELEMENT_CONFIG, 0x1);
    etna_set_state(cmdPtr, VIVS_RA_CONTROL, 0x1);

    etna_set_state(cmdPtr, VIVS_PA_W_CLIP_LIMIT, 0x34000001);
    etna_set_state(cmdPtr, VIVS_PA_SYSTEM_MODE, 0x11);    
    etna_set_state(cmdPtr, VIVS_GL_API_MODE, VIVS_GL_API_MODE_OPENGL);
    for(int i=0; i<2; i++)
    {
        etna_set_state(cmdPtr, VIVS_SE_CONFIG, 0x0); //LAST_PIXEL_ENABLE=0
        etna_set_state(cmdPtr, VIVS_SE_DEPTH_SCALE, 0x0);
        etna_set_state(cmdPtr, VIVS_SE_DEPTH_BIAS, 0x0);
    }
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR);
    etna_set_state(cmdPtr, VIVS_DUMMY_DUMMY, 0);
    etna_set_state(cmdPtr, VIVS_TS_DEPTH_AUTO_DISABLE_COUNT, 0x1cc0);
    etna_set_state_multi(cmdPtr, VIVS_TS_COLOR_STATUS_BASE, 3,
                         (uint32_t[]){rt_ts_physical, rt_physical, 0});
    etna_set_state(cmdPtr, VIVS_TS_MEM_CONFIG,
                   VIVS_TS_MEM_CONFIG_COLOR_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_COLOR_AUTO_DISABLE);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_DUMMY_DUMMY, 0);
    etna_set_state_multi(cmdPtr, VIVS_TS_DEPTH_STATUS_BASE, 3,
                         (uint32_t[]){z_ts_physical, z_physical, 0xffffffff});
    etna_set_state(cmdPtr, VIVS_TS_MEM_CONFIG,
                   VIVS_TS_MEM_CONFIG_DEPTH_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_COLOR_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_DEPTH_16BPP |
                   VIVS_TS_MEM_CONFIG_COLOR_AUTO_DISABLE |
                   VIVS_TS_MEM_CONFIG_DEPTH_COMPRESSION);
    etna_set_state(cmdPtr, VIVS_SE_DEPTH_SCALE, 0);
    etna_set_state(cmdPtr, VIVS_SE_DEPTH_BIAS, 0);
    
    for(int i=0; i<2; i++)
    {
        etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE,
                       VIVS_GL_FLUSH_CACHE_DEPTH | VIVS_GL_FLUSH_CACHE_COLOR);
        etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE,
                       VIVS_GL_FLUSH_CACHE_DEPTH | VIVS_GL_FLUSH_CACHE_COLOR);
        etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE,
                       VIVS_GL_FLUSH_CACHE_DEPTH | VIVS_GL_FLUSH_CACHE_COLOR);
        etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE,
                       VIVS_GL_FLUSH_CACHE_DEPTH | VIVS_GL_FLUSH_CACHE_COLOR);
        etna_set_state(cmdPtr, VIVS_TS_FLUSH_CACHE,
                       VIVS_TS_FLUSH_CACHE_FLUSH);
    }
    
    /* semaphore time */
    etna_set_state(cmdPtr, VIVS_GL_SEMAPHORE_TOKEN, 
                   (SYNC_RECIPIENT_RA<<VIVS_GL_SEMAPHORE_TOKEN_FROM__SHIFT)|
                   (SYNC_RECIPIENT_PE<<VIVS_GL_SEMAPHORE_TOKEN_TO__SHIFT));
    etna_set_state(cmdPtr, VIVS_GL_STALL_TOKEN, 
                   (SYNC_RECIPIENT_RA<<VIVS_GL_STALL_TOKEN_FROM__SHIFT)|
                   (SYNC_RECIPIENT_PE<<VIVS_GL_STALL_TOKEN_TO__SHIFT));
    /* Set up the resolve to clear tile status for main render target 
     * Regard the TS as an image of width 16 with 4 bytes per pixel (64 bytes per row)
     * XXX need to clear the depth ts too.
     * */
    etna_set_state(cmdPtr, VIVS_RS_CONFIG,
                   (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_SOURCE_FORMAT__SHIFT) |
                   (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_DEST_FORMAT__SHIFT));
    etna_set_state_multi(cmdPtr, VIVS_RS_DITHER(0), 2, (uint32_t[]){0xffffffff, 0xffffffff});
    etna_set_state(cmdPtr, VIVS_RS_DEST_ADDR, rt_ts_physical); /* ADDR_B */
    etna_set_state(cmdPtr, VIVS_RS_DEST_STRIDE, 0x200);
    etna_set_state(cmdPtr, VIVS_RS_FILL_VALUE(0), 0x55555555);
    etna_set_state(cmdPtr, VIVS_RS_CLEAR_CONTROL, 
                   VIVS_RS_CLEAR_CONTROL_MODE_ENABLED1 |
                   (0xffff << VIVS_RS_CLEAR_CONTROL_BITS__SHIFT));
    etna_set_state(cmdPtr, VIVS_RS_EXTRA_CONFIG, 
                   0); /* no AA, no endian switch */
    etna_set_state(cmdPtr, VIVS_RS_WINDOW_SIZE, 
                   (4 << VIVS_RS_WINDOW_SIZE_HEIGHT__SHIFT) |
                   (128 << VIVS_RS_WINDOW_SIZE_WIDTH__SHIFT));
    etna_set_state(cmdPtr, VIVS_RS_KICKER, 0xbadabeeb);
    
    etna_set_state(cmdPtr, VIVS_TS_COLOR_CLEAR_VALUE, 0xff7f7f7f);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR);
    etna_set_state(cmdPtr, VIVS_DUMMY_DUMMY, 0);
    etna_set_state(cmdPtr, VIVS_TS_DEPTH_AUTO_DISABLE_COUNT, 0x1cc0);
    etna_set_state_multi(cmdPtr, VIVS_TS_COLOR_STATUS_BASE, 3,
                         (uint32_t[]){rt_ts_physical, rt_physical, 0xff7f7f7f});
    etna_set_state(cmdPtr, VIVS_TS_MEM_CONFIG,
                   VIVS_TS_MEM_CONFIG_DEPTH_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_COLOR_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_DEPTH_16BPP |
                   VIVS_TS_MEM_CONFIG_COLOR_AUTO_DISABLE |
                   VIVS_TS_MEM_CONFIG_DEPTH_COMPRESSION);
    
    
    /*   Compute transform matrices in the same way as cube egl demo */ 
    ESMatrix modelview;
    esMatrixLoadIdentity(&modelview);
    esTranslate(&modelview, 0.0f, 0.0f, -8.0f);
    esRotate(&modelview, 45.0f, 1.0f, 0.0f, 0.0f);
    esRotate(&modelview, 45.0f, 0.0f, 1.0f, 0.0f);
    esRotate(&modelview, 10.0f, 0.0f, 0.0f, 1.0f);

    float aspect = (float)(height) / (float)(width);

    ESMatrix projection;
    esMatrixLoadIdentity(&projection);
    esFrustum(&projection, -2.8f, +2.8f, -2.8f * aspect, +2.8f * aspect, 6.0f, 10.0f);

    ESMatrix modelviewprojection;
    esMatrixLoadIdentity(&modelviewprojection);
    esMatrixMultiply(&modelviewprojection, &modelview, &projection);

    float normal[9]; // normal transformation matrix
    normal[0] = modelview.m[0][0];
    normal[1] = modelview.m[0][1];
    normal[2] = modelview.m[0][2];
    normal[3] = modelview.m[1][0];
    normal[4] = modelview.m[1][1];
    normal[5] = modelview.m[1][2];
    normal[6] = modelview.m[2][0];
    normal[7] = modelview.m[2][1];
    normal[8] = modelview.m[2][2];
    
    etna_set_state_multi(cmdPtr, VIVS_VS_UNIFORMS(0), 16, (uint32_t*)&modelviewprojection.m[0][0]);
    etna_set_state_multi(cmdPtr, VIVS_VS_UNIFORMS(16), 3, (uint32_t*)&normal[0]); /* u4.xyz */
    etna_set_state_multi(cmdPtr, VIVS_VS_UNIFORMS(20), 3, (uint32_t*)&normal[3]); /* u5.xyz */
    etna_set_state_multi(cmdPtr, VIVS_VS_UNIFORMS(24), 3, (uint32_t*)&normal[6]); /* u6.xyz */
    etna_set_state_multi(cmdPtr, VIVS_VS_UNIFORMS(28), 16, (uint32_t*)&modelview.m[0][0]);

    //TYPE=FLOAT,ENDIAN=NO_SWAP,NONCONSECUTIVE=1,STREAM=TO_BE_DEFINED,NUM=0x3,NORMALIZE=OFF,START=0x0,END=0xc
    unsigned fe_vert_elem_conf_base = 
    VIVS_FE_VERTEX_ELEMENT_CONFIG_TYPE_FLOAT |
    (ENDIAN_MODE_NO_SWAP << VIVS_FE_VERTEX_ELEMENT_CONFIG_ENDIAN__SHIFT) |
    VIVS_FE_VERTEX_ELEMENT_CONFIG_NONCONSECUTIVE |
    (3 <<VIVS_FE_VERTEX_ELEMENT_CONFIG_NUM__SHIFT) |
    VIVS_FE_VERTEX_ELEMENT_CONFIG_NORMALIZE_OFF |
    (0x0 << VIVS_FE_VERTEX_ELEMENT_CONFIG_START__SHIFT) |
    (0xc << VIVS_FE_VERTEX_ELEMENT_CONFIG_END__SHIFT);
    //(xx << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT) |

    etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_ELEMENT_CONFIG(0), 3, (uint32_t[]){
        fe_vert_elem_conf_base | (0 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT),
        fe_vert_elem_conf_base | (1 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT),
        fe_vert_elem_conf_base | (2 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT)});
    etna_set_state(cmdPtr, VIVS_VS_INPUT(0), VIVS_VS_INPUT_I0(0) |
        VIVS_VS_INPUT_I1(1) | VIVS_VS_INPUT_I2(2) | VIVS_VS_INPUT_I3(0));
    
    //0x30 = 12floats (4 vertices) per position, normals, colors
    etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_STREAMS_BASE_ADDR(0), 3, (uint32_t[])
        {vtx_physical, vtx_physical+0x30, vtx_physical+0x60});
    etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_STREAMS_CONTROL(0), 3, (uint32_t[])
        {0xc, 0xc, 0xc});
    etna_set_state(cmdPtr, VIVS_PA_W_CLIP_LIMIT, 0);
    
    etna_set_state_multi_fixp(cmdPtr, VIVS_PA_VIEWPORT_SCALE_X, 2, (uint32_t[]){200<<16, 120<<16});
    etna_set_state_multi_fixp(cmdPtr, VIVS_PA_VIEWPORT_OFFSET_X, 2, (uint32_t[]){200<<16, 120<<16});
    etna_set_state/*_f32*/(cmdPtr, VIVS_PA_VIEWPORT_UNK00A80, 0x37c81905); //0.000024f but loads 37c9539c not 37c81905
    etna_set_state_fixp(cmdPtr, VIVS_PA_VIEWPORT_UNK00A84, 8192<<16);
    etna_set_state/*_f32*/(cmdPtr, VIVS_PA_VIEWPORT_UNK00A8C, 0x38000000); //0.000031f but load 380205ff not 38000000
    etna_set_state_multi_fixp(cmdPtr, VIVS_SE_SCISSOR_LEFT, 4, (uint32_t[]){0, 0, (400<<16)|0x1000, (240<<16)|0x1000});//0x1000=1/16
    etna_set_state_fixp(cmdPtr, VIVS_SE_CLIP_RIGHT, (400<<16)|0xffff);
    etna_set_state_fixp(cmdPtr, VIVS_SE_CLIP_BOTTOM, (240<<16)|0xffff);
    etna_set_state_multi(cmdPtr, VIVS_PE_ALPHA_OP, 3, (uint32_t[]){0, 0,
        VIVS_PE_ALPHA_CONFIG_SRC_FUNC_COLOR(1) |
        VIVS_PE_ALPHA_CONFIG_SRC_FUNC_ALPHA(1)});
    etna_set_state(cmdPtr, VIVS_PE_STENCIL_CONFIG_EXT, 0x0000fdff);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR);
    etna_set_state(cmdPtr, VIVS_PE_COLOR_FORMAT, RS_FORMAT_A8R8G8B8 |
        VIVS_PE_COLOR_FORMAT_COMPONENTS(0xf) |
        VIVS_PE_COLOR_FORMAT_OVERWRITE |
        VIVS_PE_COLOR_FORMAT_SUPER_TILED);
    etna_set_state(cmdPtr, VIVS_PE_PIPE_COLOR_ADDR(0), rt_physical);
    etna_set_state(cmdPtr, VIVS_PE_COLOR_ADDR, rt_physical);
    etna_set_state(cmdPtr, VIVS_PE_COLOR_STRIDE, 0x700);
    etna_set_state_multi(cmdPtr, VIVS_PE_DEPTH_CONFIG, 4, (uint32_t[]){
        VIVS_PE_DEPTH_CONFIG_DEPTH_MODE_Z |
        VIVS_PE_DEPTH_CONFIG_DEPTH_FUNC(COMPARE_FUNC_ALWAYS) |
        VIVS_PE_DEPTH_CONFIG_EARLY_Z |
        VIVS_PE_DEPTH_CONFIG_SUPER_TILED |
        VIVS_PE_DEPTH_CONFIG_DISABLE_ZS,
        0x00000000/*0.0*/, 0x3f800000/*1.0*/, 0x477fff00/*65535.0*/});
    etna_set_state(cmdPtr, VIVS_PE_PIPE_DEPTH_ADDR(0), z_physical);
    etna_set_state(cmdPtr, VIVS_PE_DEPTH_ADDR, z_physical);
    etna_set_state(cmdPtr, VIVS_PE_DEPTH_STRIDE, 0x380);
    etna_set_state(cmdPtr, VIVS_PE_HDEPTH_CONTROL, 0);
    etna_set_state_f32(cmdPtr, VIVS_PA_VIEWPORT_SCALE_Z, 1.0);
    etna_set_state_f32(cmdPtr, VIVS_PA_VIEWPORT_OFFSET_Z, 0.0);
    etna_set_state(cmdPtr, VIVS_PE_STENCIL_CONFIG, 0);
    etna_set_state(cmdPtr, VIVS_GL_MULTI_SAMPLE_CONFIG, 0);
    etna_set_state_f32(cmdPtr, VIVS_PA_LINE_WIDTH, 0.5);
    etna_set_state_multi(cmdPtr, VIVS_PA_CONFIG, 3, (uint32_t[]){
       VIVS_PA_CONFIG_CULL_FACE_MODE_CCW |
       VIVS_PA_CONFIG_FILL_MODE_SOLID |
       VIVS_PA_CONFIG_SHADE_MODEL_SMOOTH |
       VIVS_PA_CONFIG_UNK22, 
       0x3f000000/*0.5*/, 0x3f000000/*0.5*/
    });
    etna_set_state(cmdPtr, VIVS_GL_SEMAPHORE_TOKEN, 
                   (SYNC_RECIPIENT_FE<<VIVS_GL_SEMAPHORE_TOKEN_FROM__SHIFT)|
                   (SYNC_RECIPIENT_PE<<VIVS_GL_SEMAPHORE_TOKEN_TO__SHIFT));
    etna_stall_9(cmdPtr);
    
    /* shader setup */
    etna_set_state_multi(cmdPtr, VIVS_VS_INPUT_COUNT, 3, (uint32_t[]){
        VIVS_VS_INPUT_COUNT_COUNT(3) | VIVS_VS_INPUT_COUNT_UNK8(1),
        VIVS_VS_TEMP_REGISTER_CONTROL_NUM_TEMPS(6),
        VIVS_VS_OUTPUT_O0(4) | VIVS_VS_OUTPUT_O1(0) |
        VIVS_VS_OUTPUT_O2(0) | VIVS_VS_OUTPUT_O3(0)});
    etna_set_state(cmdPtr, VIVS_VS_RANGE, VIVS_VS_RANGE_HIGH(0x15));
    etna_set_state(cmdPtr, VIVS_VS_START_PC, 0);
    etna_set_state(cmdPtr, VIVS_VS_END_PC, 0x16);
    
    //vec4 lightSource = vec4(2.0, 2.0, 20.0, 0.0);
    //VS.UNIFORMS[14] = 2.0, VS.UNIFORMS[19]=2.0, VS.UNIFORMS[23]=20.0 VS.UNIFORMS[27]=0.0 u[3-6].w
    etna_set_state_f32(cmdPtr, VIVS_VS_UNIFORMS(44), 1.0);
    etna_set_state_f32(cmdPtr, VIVS_VS_UNIFORMS(27), 0.0);
    etna_set_state_f32(cmdPtr, VIVS_VS_UNIFORMS(23), 20.0);
    etna_set_state_f32(cmdPtr, VIVS_VS_UNIFORMS(19), 2.0);
    
    /* Now load the shader itself */
    uint32_t vs[] = {
        0x01831009, 0x00000000, 0x00000000, 0x203fc048,
        0x02031009, 0x00000000, 0x00000000, 0x203fc058,
        0x07841003, 0x39000800, 0x00000050, 0x00000000,
        0x07841002, 0x39001800, 0x00aa0050, 0x00390048,
        0x07841002, 0x39002800, 0x01540050, 0x00390048,
        0x07841002, 0x39003800, 0x01fe0050, 0x00390048,
        0x03851003, 0x29004800, 0x000000d0, 0x00000000,
        0x03851002, 0x29005800, 0x00aa00d0, 0x00290058,
        0x03811002, 0x29006800, 0x015400d0, 0x00290058,
        0x07851003, 0x39007800, 0x00000050, 0x00000000,
        0x07851002, 0x39008800, 0x00aa0050, 0x00390058,
        0x07851002, 0x39009800, 0x01540050, 0x00390058,
        0x07801002, 0x3900a800, 0x01fe0050, 0x00390058,
        0x0401100c, 0x00000000, 0x00000000, 0x003fc008,
        0x03801002, 0x69000800, 0x01fe00c0, 0x00290038,
        0x03831005, 0x29000800, 0x01480040, 0x00000000,
        0x0383100d, 0x00000000, 0x00000000, 0x00000038,
        0x03801003, 0x29000800, 0x014801c0, 0x00000000,
        0x00801005, 0x29001800, 0x01480040, 0x00000000,
        0x0080108f, 0x3fc06800, 0x00000050, 0x203fc068,
        0x03801003, 0x00000800, 0x01480140, 0x00000000,
        0x04001009, 0x00000000, 0x00000000, 0x200000b8
    };
    uint32_t ps[] = {
        0x00000000, 0x00000000, 0x00000000, 0x00000000
    };
    etna_set_state_multi(cmdPtr, VIVS_SH_INST_MEM(0), sizeof(vs)/4, vs);
    etna_set_state(cmdPtr, VIVS_RA_CONTROL, 0x1);
    etna_set_state(cmdPtr, VIVS_PS_OUTPUT_REG, 0x1);
    etna_set_state(cmdPtr, VIVS_PA_SHADER_ATTRIBUTES(0), 0x200);
    etna_set_state(cmdPtr, VIVS_GL_VARYING_NUM_COMPONENTS,  /* one varying, with four components */
                   (4 << VIVS_GL_VARYING_NUM_COMPONENTS_VAR0__SHIFT)
    );
    etna_set_state(cmdPtr, VIVS_GL_UNK03834, 0x40000);
    etna_set_state_multi(cmdPtr, VIVS_GL_VARYING_COMPONENT_USE(0), 2, (uint32_t[]){
        VIVS_GL_VARYING_COMPONENT_USE_COMP0(1) |
        VIVS_GL_VARYING_COMPONENT_USE_COMP1(1) |
        VIVS_GL_VARYING_COMPONENT_USE_COMP2(1) |
        VIVS_GL_VARYING_COMPONENT_USE_COMP3(1),
        0});
    etna_set_state(cmdPtr, VIVS_GL_UNK03838, 0x0);
    etna_set_state(cmdPtr, VIVS_PS_RANGE, VIVS_PS_RANGE_LOW(0x100) | VIVS_PS_RANGE_HIGH(0x100));
    etna_set_state(cmdPtr, VIVS_PS_START_PC, 0x0);
    etna_set_state(cmdPtr, VIVS_PS_END_PC, 0x1);
    

    etna_set_state_multi(cmdPtr, VIVS_SH_INST_MEM(0x100*4), sizeof(ps)/4, ps);
    etna_set_state_multi(cmdPtr, VIVS_PS_INPUT_COUNT, 3, (uint32_t[]){
        (31<<8)|2,
        (2 << VIVS_PS_TEMP_REGISTER_CONTROL_NUM_TEMPS__SHIFT),
        VIVS_PS_CONTROL_UNK1});

    etna_set_state(cmdPtr, VIVS_PA_ATTRIBUTE_ELEMENT_COUNT, 0x100);
    etna_set_state(cmdPtr, VIVS_GL_VARYING_TOTAL_COMPONENTS,  /* one varying, with four components */
            (4 << VIVS_GL_VARYING_TOTAL_COMPONENTS_NUM__SHIFT)
            );
    etna_set_state(cmdPtr, VIVS_VS_LOAD_BALANCING, 0xf3f0582);
    etna_set_state(cmdPtr, VIVS_VS_OUTPUT_COUNT, 2);
    etna_set_state(cmdPtr, VIVS_GL_SEMAPHORE_TOKEN, 
                   (SYNC_RECIPIENT_RA<<VIVS_GL_SEMAPHORE_TOKEN_FROM__SHIFT)|
                   (SYNC_RECIPIENT_PE<<VIVS_GL_SEMAPHORE_TOKEN_TO__SHIFT));
    etna_set_state(cmdPtr, VIVS_GL_STALL_TOKEN, 
                   (SYNC_RECIPIENT_RA<<VIVS_GL_STALL_TOKEN_FROM__SHIFT)|
                   (SYNC_RECIPIENT_PE<<VIVS_GL_STALL_TOKEN_TO__SHIFT));
    etna_draw_primitives(cmdPtr, PRIMITIVE_TYPE_TRIANGLE_STRIP, 0, 2);
    

    //first draw is little butchered 
    for(int drawNr = 1; drawNr<6; drawNr++)
    {
        unsigned fe_vert_elem_conf_base = 
        VIVS_FE_VERTEX_ELEMENT_CONFIG_TYPE_FLOAT |
        (ENDIAN_MODE_NO_SWAP << VIVS_FE_VERTEX_ELEMENT_CONFIG_ENDIAN__SHIFT) |
        VIVS_FE_VERTEX_ELEMENT_CONFIG_NONCONSECUTIVE |
        (3 <<VIVS_FE_VERTEX_ELEMENT_CONFIG_NUM__SHIFT) |
        VIVS_FE_VERTEX_ELEMENT_CONFIG_NORMALIZE_OFF |
        (0x0 << VIVS_FE_VERTEX_ELEMENT_CONFIG_START__SHIFT) |
        (0xc << VIVS_FE_VERTEX_ELEMENT_CONFIG_END__SHIFT);
        //(xx << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT) |
        
        etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_ELEMENT_CONFIG(0), 3, (uint32_t[]){
            fe_vert_elem_conf_base | (0 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT),
            fe_vert_elem_conf_base | (1 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT),
            fe_vert_elem_conf_base | (2 << VIVS_FE_VERTEX_ELEMENT_CONFIG_STREAM__SHIFT)});
        etna_set_state(cmdPtr, VIVS_VS_INPUT(0), VIVS_VS_INPUT_I0(0) |
        VIVS_VS_INPUT_I1(1) | VIVS_VS_INPUT_I2(2) | VIVS_VS_INPUT_I3(0));
        
        //0x30 = 12floats (4 vertices) per position, normals, colors
        etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_STREAMS_BASE_ADDR(0), 3, (uint32_t[])
        {vtx_physical+(0x60*drawNr), vtx_physical+(0x60*drawNr)+0x30, vtx_physical+(0x60*drawNr)+0x60});
        etna_set_state_multi(cmdPtr, VIVS_FE_VERTEX_STREAMS_CONTROL(0), 3, (uint32_t[])
        {0xc, 0xc, 0xc});
        etna_set_state(cmdPtr, VIVS_PA_W_CLIP_LIMIT, 0);
        etna_set_state(cmdPtr, VIVS_PA_CONFIG,
            VIVS_PA_CONFIG_CULL_FACE_MODE_CCW |
            VIVS_PA_CONFIG_FILL_MODE_SOLID |
            VIVS_PA_CONFIG_SHADE_MODEL_SMOOTH |
            VIVS_PA_CONFIG_UNK22);
        etna_draw_primitives(cmdPtr, PRIMITIVE_TYPE_TRIANGLE_STRIP, drawNr*4, 2);
    }
    for(int idx=0; idx<2; idx++)
        etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    
#ifdef CMD_COMPARE
    /* Set addresses in first command buffer */
    cmdbuf1[37] = cmdbuf1[87] = cmdbuf1[109] = rt_ts_physical;
    cmdbuf1[38] = cmdbuf1[110] = cmdbuf1[213] = cmdbuf1[215] = rt_physical;
    cmdbuf1[47] = z_ts_physical; //ADDR_J */   0x500 gcvSURF_TILE_STATUS
    cmdbuf1[48] = cmdbuf1[225] = cmdbuf1[227] = z_physical; //DDR_I */       0x45000 gcvSURF_DEPTH
    cmdbuf1[169] = vtx_physical;
    cmdbuf1[170] = vtx_physical + 0x030;
    cmdbuf1[171] = vtx_physical + 0x060;
    
    cmdbuf1[413] = vtx_physical + 0x060;
    cmdbuf1[414] = vtx_physical + 0x090;
    cmdbuf1[415] = vtx_physical + 0x0c0;
    
    cmdbuf1[435] = vtx_physical + 0x0c0;
    cmdbuf1[436] = vtx_physical + 0x0f0;
    cmdbuf1[437] = vtx_physical + 0x120;
    
    cmdbuf1[457] = vtx_physical + 0x120;
    cmdbuf1[458] = vtx_physical + 0x150;
    cmdbuf1[459] = vtx_physical + 0x180;
    
    cmdbuf1[479] = vtx_physical + 0x180;
    cmdbuf1[480] = vtx_physical + 0x1b0;
    cmdbuf1[481] = vtx_physical + 0x1e0;
    
    cmdbuf1[501] = vtx_physical + 0x1e0;
    cmdbuf1[502] = vtx_physical + 0x210;
    cmdbuf1[503] = vtx_physical + 0x240;
    
    /* Must exactly match */
    if(cmdbuffer_compare(cmdPtr, cmdbuf1, sizeof(cmdbuf1)))
        exit(1);
#endif

    /* Submit first command buffer */
    commandBuffer.free = commandBuffer.bytes - commandBuffer.offset - 0x8;
    printf("[1] startOffset=%08x, offset=%08x, free=%08x\n", (uint32_t)commandBuffer.startOffset, (uint32_t)commandBuffer.offset, (uint32_t)commandBuffer.free);
    if(viv_commit(conn, &commandBuffer, context) != 0)
    {
        fprintf(stderr, "Error committing first command buffer\n");
        exit(1);
    }


    /* Build second command buffer */
    commandBuffer.startOffset = commandBuffer.offset + 0x18;
    commandBuffer.offset = commandBuffer.startOffset + 8*4;

    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_RS_CONFIG,
            (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_SOURCE_FORMAT__SHIFT) |
            VIVS_RS_CONFIG_SOURCE_TILED |
            (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_DEST_FORMAT__SHIFT) |
            VIVS_RS_CONFIG_DEST_TILED);
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_STRIDE, (padded_width * 4 * 4) | VIVS_RS_SOURCE_STRIDE_TILING);
    etna_set_state(cmdPtr, VIVS_RS_DEST_STRIDE, (padded_width * 4 * 4) | VIVS_RS_DEST_STRIDE_TILING);
    etna_set_state_multi(cmdPtr, VIVS_RS_DITHER(0), 2, (uint32_t[]){0xffffffff, 0xffffffff});
    etna_set_state(cmdPtr, VIVS_RS_CLEAR_CONTROL, VIVS_RS_CLEAR_CONTROL_MODE_DISABLED);
    etna_set_state(cmdPtr, VIVS_RS_EXTRA_CONFIG, 0); /* no AA, no endian switch */
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_ADDR, rt_physical); /* ADDR_A */
    etna_set_state(cmdPtr, VIVS_RS_DEST_ADDR, rt_physical); /* ADDR_A */
    etna_set_state(cmdPtr, VIVS_RS_WINDOW_SIZE, 
            (padded_height << VIVS_RS_WINDOW_SIZE_HEIGHT__SHIFT) |
            (padded_width << VIVS_RS_WINDOW_SIZE_WIDTH__SHIFT));
    etna_set_state(cmdPtr, VIVS_RS_KICKER, 0xbadabeeb);

#ifdef CMD_COMPARE
    /* Submit second command buffer, with updated context.
     * Second command buffer fills the background.
     */
    cmdbuf2[35] = cmdbuf2[37] = rt_physical;
    
    /* Must exactly match */
    if(cmdbuffer_compare(cmdPtr, cmdbuf2, sizeof(cmdbuf2)))
        exit(1);
#endif
    commandBuffer.free = commandBuffer.bytes - commandBuffer.offset - 0x8;
    printf("[2] startOffset=%08x, offset=%08x, free=%08x\n", (uint32_t)commandBuffer.startOffset, (uint32_t)commandBuffer.offset, (uint32_t)commandBuffer.free);
    if(viv_commit(conn, &commandBuffer, context) != 0)
    {
        fprintf(stderr, "Error committing second command buffer\n");
        exit(1);
    }

    /* Build third command buffer 
     * Third command buffer does some cache flush trick?
     * It can be left out without any visible harm.
     **/
    commandBuffer.startOffset = commandBuffer.offset + 0x18;
    commandBuffer.offset = commandBuffer.startOffset + 8*4;
    etna_set_state(cmdPtr, VIVS_TS_FLUSH_CACHE, VIVS_TS_FLUSH_CACHE_FLUSH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR);
    etna_set_state(cmdPtr, VIVS_DUMMY_DUMMY, 0);
    etna_set_state(cmdPtr, VIVS_TS_MEM_CONFIG,
                   VIVS_TS_MEM_CONFIG_DEPTH_FAST_CLEAR |
                   VIVS_TS_MEM_CONFIG_DEPTH_16BPP | 
                   VIVS_TS_MEM_CONFIG_DEPTH_COMPRESSION);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_RS_CONFIG,
                   (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_SOURCE_FORMAT__SHIFT) |
                   VIVS_RS_CONFIG_SOURCE_TILED |
                   (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_DEST_FORMAT__SHIFT) |
                   VIVS_RS_CONFIG_SWAP_RB);
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_STRIDE, (padded_width * 4 * 4) | VIVS_RS_SOURCE_STRIDE_TILING);
    etna_set_state(cmdPtr, VIVS_RS_DEST_STRIDE, (padded_width * 4));
    etna_set_state_multi(cmdPtr, VIVS_RS_DITHER(0), 2, (uint32_t[]){0xffffffff, 0xffffffff});
    etna_set_state(cmdPtr, VIVS_RS_CLEAR_CONTROL, VIVS_RS_CLEAR_CONTROL_MODE_DISABLED);
    etna_set_state(cmdPtr, VIVS_RS_EXTRA_CONFIG, 0); /* no AA, no endian switch */
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_ADDR, rt_physical); /* ADDR_A */
    etna_set_state(cmdPtr, VIVS_RS_DEST_ADDR, aux_rt_physical); /* ADDR_A */
    etna_set_state(cmdPtr, VIVS_RS_WINDOW_SIZE, 
                   (padded_height << VIVS_RS_WINDOW_SIZE_HEIGHT__SHIFT) |
                   (padded_width << VIVS_RS_WINDOW_SIZE_WIDTH__SHIFT));
    etna_set_state(cmdPtr, VIVS_RS_KICKER, 0xbadabeeb);

    /* Submit third command buffer */
#ifdef CMD_COMPARE
    cmdbuf3[35] = rt_physical;
    cmdbuf3[37] = aux_rt_physical;
    /* Must exactly match */
    if(cmdbuffer_compare(cmdPtr, cmdbuf3, sizeof(cmdbuf3)))
        exit(1);
#endif
    commandBuffer.free = commandBuffer.bytes - commandBuffer.offset - 0x8;
    printf("[3] startOffset=%08x, offset=%08x, free=%08x\n", (uint32_t)commandBuffer.startOffset, (uint32_t)commandBuffer.offset, (uint32_t)commandBuffer.free);
    if(viv_commit(conn, &commandBuffer, context) != 0)
    {
        fprintf(stderr, "Error committing third command buffer\n");
        exit(1);
    }

    /* Submit event queue with SIGNAL, fromWhere=gcvKERNEL_PIXEL (wait for pixel engine to finish) */
    int sig_id = 0;
    if(viv_user_signal_create(conn, 0, &sig_id) != 0) /* automatic resetting signal */
    {
        fprintf(stderr, "Cannot create user signal\n");
        exit(1);
    }
    printf("Created user signal %i\n", sig_id);
    if(viv_event_queue_signal(conn, sig_id, gcvKERNEL_PIXEL) != 0)
    {
        fprintf(stderr, "Cannot queue GPU signal\n");
        exit(1);
    }

    /* Wait for signal */
    if(viv_user_signal_wait(conn, sig_id, VIV_WAIT_INDEFINITE) != 0)
    {
        fprintf(stderr, "Cannot wait for signal\n");
        exit(1);
    }

    /* Allocate video memory for BITMAP, lock */
    gcuVIDMEM_NODE_PTR bmp_node = 0;
    if(viv_alloc_linear_vidmem(conn, width*height*4, 0x40, gcvSURF_BITMAP, gcvPOOL_DEFAULT, &bmp_node, NULL)!=0)
    {
        fprintf(stderr, "Error allocating bitmap status memory\n");
        exit(1);
    }
    printf("Allocated bitmap node: node=%08x\n", (uint32_t)bmp_node);
    
    viv_addr_t bmp_physical = 0;
    void *bmp_logical = 0;
    if(viv_lock_vidmem(conn, bmp_node, &bmp_physical, &bmp_logical)!=0)
    {
        fprintf(stderr, "Error locking bmp memory\n");
        exit(1);
    }
    memset(bmp_logical, 0xff, width*height*4); /* clear previous result */
    printf("Locked bmp: phys=%08x log=%08x\n", (uint32_t)bmp_physical, (uint32_t)bmp_logical);

    /* Start building fourth command buffer
     * Fourth command buffer copies render result to bitmap, detiling along the way. 
     */
    commandBuffer.startOffset = commandBuffer.offset + 0x18;
    commandBuffer.offset = commandBuffer.startOffset + 8*4;
    etna_set_state(cmdPtr, VIVS_GL_FLUSH_CACHE, VIVS_GL_FLUSH_CACHE_COLOR | VIVS_GL_FLUSH_CACHE_DEPTH);
    etna_set_state(cmdPtr, VIVS_RS_CONFIG,
            (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_SOURCE_FORMAT__SHIFT) |
            (RS_FORMAT_A8R8G8B8 << VIVS_RS_CONFIG_DEST_FORMAT__SHIFT) /*|
            VIVS_RS_CONFIG_SWAP_RB*/);
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_STRIDE, (padded_width * 4));
    etna_set_state(cmdPtr, VIVS_RS_DEST_STRIDE, width * 4);
    etna_set_state(cmdPtr, VIVS_RS_DITHER(0), 0xffffffff);
    etna_set_state(cmdPtr, VIVS_RS_DITHER(1), 0xffffffff);
    etna_set_state(cmdPtr, VIVS_RS_CLEAR_CONTROL, VIVS_RS_CLEAR_CONTROL_MODE_DISABLED);
    etna_set_state(cmdPtr, VIVS_RS_EXTRA_CONFIG, 
            0); /* no AA, no endian switch */
    etna_set_state(cmdPtr, VIVS_RS_SOURCE_ADDR, aux_rt_physical); /* ADDR_A */
    etna_set_state(cmdPtr, VIVS_RS_DEST_ADDR, bmp_physical); /* ADDR_J */
    etna_set_state(cmdPtr, VIVS_RS_WINDOW_SIZE, 
            (height << VIVS_RS_WINDOW_SIZE_HEIGHT__SHIFT) |
            (width << VIVS_RS_WINDOW_SIZE_WIDTH__SHIFT));
    etna_set_state(cmdPtr, VIVS_RS_KICKER, 0xbeebbeeb);

    /* Submit fourth command buffer
     */
#ifdef CMD_COMPARE
    cmdbuf4[0x19] = aux_rt_physical;
    cmdbuf4[0x1b] = bmp_physical;
    /* Must exactly match */
    if(cmdbuffer_compare(cmdPtr, cmdbuf4, sizeof(cmdbuf4)))
        exit(1);
#endif
    commandBuffer.free = commandBuffer.bytes - commandBuffer.offset - 0x8;
    printf("[4] startOffset=%08x, offset=%08x, free=%08x\n", (uint32_t)commandBuffer.startOffset, (uint32_t)commandBuffer.offset, (uint32_t)commandBuffer.free);
    if(viv_commit(conn, &commandBuffer, context) != 0)
    {
        fprintf(stderr, "Error committing fourth command buffer\n");
        exit(1);
    }

    /* Submit event queue with SIGNAL, fromWhere=gcvKERNEL_PIXEL */
    if(viv_event_queue_signal(conn, sig_id, gcvKERNEL_PIXEL) != 0)
    {
        fprintf(stderr, "Cannot queue GPU signal\n");
        exit(1);
    }

    /* Wait for signal */
    if(viv_user_signal_wait(conn, sig_id, VIV_WAIT_INDEFINITE) != 0)
    {
        fprintf(stderr, "Cannot wait for signal\n");
        exit(1);
    }
    bmp_dump32(bmp_logical, width, height, false, "/home/linaro/replay.bmp");
    /* Unlock video memory */
    if(viv_unlock_vidmem(conn, bmp_node, gcvSURF_BITMAP, 1) != 0)
    {
        fprintf(stderr, "Cannot unlock vidmem\n");
        exit(1);
    }

    viv_close(conn);
    return 0;
}

