# -*- coding: utf-8 -*-

# This file is part of the desktop management solution opsi http://www.opsi.org
# Copyright (c) 2023-2024 uib GmbH <info@uib.de>
# This code is owned by the uib GmbH, Mainz, Germany (uib.de). All rights reserved.
# License: AGPL-3.0

from __future__ import annotations

import re
from pathlib import Path
from typing import IO, TYPE_CHECKING, Generator

from opsicommon.logging import get_logger

if TYPE_CHECKING:
	from opsiagent.opsiagent import OpsiAgent


logger = get_logger("opsiagent.plugin.boot_server.bootfs")

UUID_RE = re.compile(r"^[0-9a-f]{8}-([0-9a-f]{4}-){3}[0-9a-f]{12}$", re.IGNORECASE)
MAC_RE = re.compile(r"^01-([0-9a-f]{2}[:-]){5}[0-9a-f]{2}$", re.IGNORECASE)


class BootFilesystem:
	def __init__(self, opsi_agent: OpsiAgent, base_dir: Path) -> None:
		self.opsi_agent = opsi_agent
		self._base_dir = base_dir
		self._config_dir = self._base_dir / "cfg"
		self._service_client = self.opsi_agent.opsi_service.get_service_client()

	def open_file(self, filename: Path) -> Generator[IO, None, None]:
		filename = (self._base_dir / filename).resolve()
		if not filename.is_relative_to(self._base_dir):
			raise PermissionError(f"Access to '{filename}' is forbidden")
		if filename.is_relative_to(self._config_dir):
			hardware_address: str | None = None
			system_uuid: str | None = None
			if UUID_RE.match(filename.name):
				system_uuid = filename.name.lower()
			elif MAC_RE.match(filename.name):
				hardware_address = filename.name.lower().removeprefix("01-").replace("-", ":")
			if system_uuid or hardware_address:
				self._service_client.boot_getConfig(  # type: ignore[attr-defined]
					ip_version=4,
					architecture="x64",
					bios_type="UEFI",
					protocol="TFTP",
					system_uuid=system_uuid,
					hardware_address=hardware_address,
				)

		with filename.open("rb") as file:
			yield file
