# This file is part of the desktop management solution opsi http://www.opsi.org
# Copyright (c) 2023-2024 uib GmbH <info@uib.de>
# This code is owned by the uib GmbH, Mainz, Germany (uib.de). All rights reserved.
# License: AGPL-3.0

from __future__ import annotations

from ipaddress import IPv4Address
from pathlib import Path
from typing import TYPE_CHECKING

from confz import BaseConfig
from opsicommon.logging import get_logger

from opsiagent.plugin import OpsiAgentPlugin, OpsiAgentPluginConfig
from opsiagent.utils import get_primary_ipv4_interface

from .dhcp_server import ProxyDHCPServer, ProxyDHCPServerConfig
from .tftp_server import TFTPServer, TFTPServerConfig

if TYPE_CHECKING:
	from opsiagent.config import Config
	from opsiagent.opsiagent import OpsiAgent

logger = get_logger("opsiagent.plugin.boot_server")
logger.context_name = "boot_server"


class PXEBootConfig(BaseConfig):  # type: ignore[metaclass]
	boot_server_address: IPv4Address = IPv4Address(get_primary_ipv4_interface().address)
	boot_filename_x86_bios: str = "opsi/loader/grub.x86.bios"  # path according to otw21 (0.3)
	boot_filename_x64_uefi: str = "opsi/loader/grub-shim.x64.efi"  # path according to otw21 (0.3)


class BootServerPluginConfig(OpsiAgentPluginConfig):  # type: ignore[metaclass]
	proxy_dhcp_server: ProxyDHCPServerConfig
	tftp_server: TFTPServerConfig
	pxe_boot: PXEBootConfig


class BootServerPlugin(OpsiAgentPlugin[BootServerPluginConfig]):
	id = "boot_server"
	name = "Boot server"
	version = "0.2"

	def __init__(self, opsi_agent: OpsiAgent, path: Path) -> None:
		super().__init__(opsi_agent, path)
		self._proxy_dhcp_server = ProxyDHCPServer(self.opsi_agent)
		self._tftp_server = TFTPServer(self.opsi_agent)
		self.opsi_agent.register_config_changed_callback(self.on_config_change, ["plugin.boot_server.*"])

	def on_config_change(self, new_config: Config, changed_config_keys: set[str]) -> None:
		logger.info("Boot server config changed")
		if (
			"plugin.boot_server.proxy_dhcp_server.enabled" in changed_config_keys
			or "plugin.boot_server.proxy_dhcp_server.port" in changed_config_keys
		):
			self._proxy_dhcp_server.stop()
			if self._proxy_dhcp_server.config.enabled:
				self._proxy_dhcp_server = ProxyDHCPServer(self.opsi_agent)
				self._proxy_dhcp_server.start()
		if "plugin.boot_server.tftp_server.enabled" in changed_config_keys or "plugin.boot_server.tftp_server.port" in changed_config_keys:
			self._tftp_server.stop()
			if self._tftp_server.config.enabled:
				self._tftp_server = TFTPServer(self.opsi_agent)
				self._tftp_server.start()

	def on_load(self) -> None:
		"""Called after loading the plugin"""
		if self._proxy_dhcp_server.config.enabled:
			self._proxy_dhcp_server.start()
		if self._tftp_server.config.enabled:
			self._tftp_server.start()

	def on_unload(self) -> None:
		"""Called before unloading the plugin"""
		self._proxy_dhcp_server.stop()
		self._tftp_server.stop()
