# This file is part of the desktop management solution opsi http://www.opsi.org
# Copyright (c) 2023-2024 uib GmbH <info@uib.de>
# This code is owned by the uib GmbH, Mainz, Germany (uib.de). All rights reserved.
# License: AGPL-3.0

from __future__ import annotations

import asyncio
import traceback
from pathlib import Path
from threading import Event, Thread
from typing import TYPE_CHECKING

from opsicommon.client.opsiservice import Messagebus, MessagebusListener
from opsicommon.logging import get_logger
from opsicommon.messagebus.file_transfer import (
	process_messagebus_message as process_messagebus_filetransfer_message,
)
from opsicommon.messagebus.file_transfer import stop_running_file_transfers
from opsicommon.messagebus.message import (
	Error,
	FileTransferMessage,
	FileUploadRequestMessage,
	GeneralErrorMessage,
	Message,
	ProcessMessage,
	TerminalMessage,
)
from opsicommon.messagebus.process import (
	process_messagebus_message as process_messagebus_process_message,
)
from opsicommon.messagebus.process import stop_running_processes
from opsicommon.messagebus.terminal import (
	process_messagebus_message as process_messagebus_terminal_message,
)
from opsicommon.messagebus.terminal import stop_running_terminals, terminals

from opsiagent.plugin import OpsiAgentPlugin, OpsiAgentPluginConfig

if TYPE_CHECKING:
	from opsiagent.opsi_service import ServiceClient
	from opsiagent.opsiagent import OpsiAgent

logger = get_logger("opsiagent.plugin.remote_control")
logger.context_name = "remote_control"


class RemoteControlPluginConfig(OpsiAgentPluginConfig):  # type: ignore[misc]
	pass


class RemoteControlMessagebusListener(Thread, MessagebusListener):
	def __init__(self, messagebus: Messagebus) -> None:
		Thread.__init__(self, name="RemoteControlMessagebusListener", daemon=True)
		MessagebusListener.__init__(self, messagebus)
		self._loop = asyncio.new_event_loop()
		self._stopped = Event()

	def run(self) -> None:
		self._loop.run_forever()
		self._stopped.set()

	def stop(self) -> None:
		future = asyncio.run_coroutine_threadsafe(self.async_stop(), self._loop)
		future.result(5)
		self._loop.call_soon_threadsafe(self._loop.stop)
		self._stopped.wait(3)

	async def async_stop(self) -> None:
		await stop_running_terminals()
		await stop_running_file_transfers()
		await stop_running_processes()

	def message_received(self, message: Message) -> None:
		assert self.messagebus
		try:
			asyncio.run_coroutine_threadsafe(self._process_message(message), self._loop)
		except Exception as err:
			logger.error(err, exc_info=True)
			response = GeneralErrorMessage(
				sender="@",
				channel=message.response_channel,
				ref_id=message.id,
				error=Error(code=0, message=str(err), details=str(traceback.format_exc())),
			)
			self.messagebus.send_message(response)

	async def _process_message(self, message: Message) -> None:
		assert self.messagebus
		if isinstance(message, TerminalMessage):
			await process_messagebus_terminal_message(message, self.messagebus.async_send_message)
		elif isinstance(message, FileTransferMessage):
			if isinstance(message, FileUploadRequestMessage):
				if message.terminal_id and not message.destination_dir:
					terminal = terminals.get(message.terminal_id)
					if terminal:
						destination_dir = terminal.get_cwd()
						message.destination_dir = str(destination_dir)
			await process_messagebus_filetransfer_message(message, self.messagebus.async_send_message)
		elif isinstance(message, ProcessMessage):
			await process_messagebus_process_message(message, self.messagebus.async_send_message)


class RemoteControlPlugin(OpsiAgentPlugin[RemoteControlPluginConfig]):
	id = "remote_control"
	name = "Remote control"
	version = "0.1"

	def __init__(self, opsi_agent: OpsiAgent, path: Path) -> None:
		super().__init__(opsi_agent, path)
		self._service_client: ServiceClient | None = None
		self._messagebus_listener: RemoteControlMessagebusListener | None = None

	def on_load(self) -> None:
		"""Called after loading the plugin"""
		self._service_client = self.opsi_agent.opsi_service.get_service_client()
		self._messagebus_listener = RemoteControlMessagebusListener(messagebus=self._service_client.messagebus)
		self._messagebus_listener.start()
		self._service_client.messagebus.register_messagebus_listener(self._messagebus_listener)

	def on_unload(self) -> None:
		"""Called before unloading the plugin"""
		if self._messagebus_listener:
			self._messagebus_listener.stop()
		self.opsi_agent.opsi_service.release_service_client("remote_control")
