-- creative/init.lua

-- Load support for MT game translation.
local S

if minetest.get_translator ~= nil then
	S = minetest.get_translator("creative") -- 5.x translation function
else
	if minetest.get_modpath("intllib") then
		dofile(minetest.get_modpath("intllib") .. "/init.lua")
		if intllib.make_gettext_pair then
			S = intllib.make_gettext_pair() -- new gettext method
		else
			S = intllib.Getter() -- old text file method
		end
	else -- boilerplate function
		S = function(str, ...)
			local args = {...}
			return str:gsub("@%d+", function(match)
				return args[tonumber(match:sub(2))]
			end)
		end
	end
end

local is_50 = minetest.has_feature("object_use_texture_alpha")
local is_53 = minetest.has_feature("object_step_has_moveresult")
local is_54 = minetest.has_feature("direct_velocity_on_players")

creative = {}
creative.get_translator = S
creative.is_50 = is_50
creative.is_53 = is_53
creative.is_54 = is_54

local privs_description = "Allow player to use creative inventory"
local privs_definition = {}

local function update_sfinv(name)
	minetest.after(0, function()
		local player = minetest.get_player_by_name(name)
		if player then
			if sfinv.get_page(player):sub(1, 9) == "creative:" then
				sfinv.set_page(player, sfinv.get_homepage_name(player))
			else
				sfinv.set_player_inventory_formspec(player)
			end
		end
	end)
end

if creative.is_50 then 
	privs_definition = {
		description = privs_description,
		give_to_singleplayer = false,
		give_to_admin = false,
		on_grant = update_sfinv,
		on_revoke = update_sfinv
	}
else
	privs_definition = {
		description = privs_description,
		give_to_singleplayer = false
	}
end

minetest.register_privilege("creative", privs_definition)

local creative_mode_cache = minetest.settings:get_bool("creative_mode")

function minetest.is_creative_enabled(name)
	if name == "" then
		return creative_mode_cache
	else
		return minetest.check_player_privs(name, {creative = true}) or creative_mode_cache
	end
end

-- backguard compatibility 
function creative.is_creative(name)
	if name == "" then
		return creative_mode_cache
	else
		return minetest.check_player_privs(name, {creative = true}) or creative_mode_cache
	end
end


-- For backwards compatibility:
function creative.is_enabled_for(name)
	return creative.is_creative(name)
end

local player_inventory = {}
local inventory_cache = {}

local function init_creative_cache(items)
	inventory_cache[items] = {}
	local i_cache = inventory_cache[items]

	for name, def in pairs(items) do
		if def.groups and def.groups.not_in_creative_inventory ~= 1 and
				def.description and def.description ~= "" then
			i_cache[name] = def
		end
	end
	table.sort(i_cache)
	return i_cache
end

function creative.init_creative_inventory(player)
	local player_name = player:get_player_name()
	player_inventory[player_name] = {
		size = 0,
		filter = "",
		start_i = 0,
		old_filter = nil, -- use only for caching in update_creative_inventory
		old_content = nil
	}

	minetest.create_detached_inventory("creative_" .. player_name, {
		allow_move = function(inv, from_list, from_index, to_list, to_index, count, player2)
			local name = player2 and player2:get_player_name() or ""
			if not creative.is_enabled_for(name) or
					to_list == "main" then
				return 0
			end
			return count
		end,
		allow_put = function(inv, listname, index, stack, player2)
			return 0
		end,
		allow_take = function(inv, listname, index, stack, player2)
			local name = player2 and player2:get_player_name() or ""
			if not creative.is_enabled_for(name) then
				return 0
			end
			return -1
		end,
		on_move = function(inv, from_list, from_index, to_list, to_index, count, player2)
		end,
		on_take = function(inv, listname, index, stack, player2)
			if stack and stack:get_count() > 0 then
				minetest.log("action", player_name .. " takes " .. stack:get_name().. " from creative inventory")
			end
		end,
	}, player_name)

	return player_inventory[player_name]
end

local NO_MATCH = 999
local function match(s, filter)
	if filter == "" then
		return 0
	end
	if s:lower():find(filter, 1, true) then
		return #s - #filter
	end
	return NO_MATCH
end

local function description(def, lang_code)
	local s = def.description
	if creative.is_53 then
		if lang_code then
			s = minetest.get_translated_string(lang_code, s)
		end
	end
	return s:gsub("\n.*", "") -- First line only
end

function creative.update_creative_inventory(player_name, tab_content)
	local inv = player_inventory[player_name] or
			creative.init_creative_inventory(minetest.get_player_by_name(player_name))
	local player_inv = minetest.get_inventory({type = "detached", name = "creative_" .. player_name})

	if inv.filter == inv.old_filter and tab_content == inv.old_content then
		return
	end
	inv.old_filter = inv.filter
	inv.old_content = tab_content

	local items = inventory_cache[tab_content] or init_creative_cache(tab_content)

	local lang = minetest.settings:get("language") or "en"
	local player_info = minetest.get_player_information(player_name)

	if creative.is_53 then 
		if player_info and player_info.lang_code ~= "" then
			lang = player_info.lang_code
		end
	end

	local creative_list = {}
	local order = {}
	for name, def in pairs(items) do
		local m = match(description(def), inv.filter)
		if m > 0 then
			m = math.min(m, match(description(def, lang), inv.filter))
		end
		if m > 0 then
			m = math.min(m, match(name, inv.filter))
		end

		if m < NO_MATCH then
			creative_list[#creative_list+1] = name
			-- Sort by match value first so closer matches appear earlier
			order[name] = string.format("%02d", m) .. name
		end
	end

	table.sort(creative_list, function(a, b) return order[a] < order[b] end)

	player_inv:set_size("main", #creative_list)
	player_inv:set_list("main", creative_list)
	inv.size = #creative_list
end

-- Create the trash field
local trash = minetest.create_detached_inventory("trash", {
	-- Allow the stack to be placed and remove it in on_put()
	-- This allows the creative inventory to restore the stack
	allow_put = function(inv, listname, index, stack, player)
		return stack:get_count()
	end,
	on_put = function(inv, listname)
		inv:set_list(listname, {})
	end,
})
trash:set_size("main", 1)

creative.formspec_add = ""

function creative.register_tab(name, title, items)
	sfinv.register_page("creative:" .. name, {
		title = title,
		is_in_nav = function(self, player, context)
			return creative.is_enabled_for(player:get_player_name())
		end,
		get = function(self, player, context)
			local player_name = player:get_player_name()
			creative.update_creative_inventory(player_name, items)
			local inv = player_inventory[player_name]
			local start_i = inv.start_i or 0
			local pagenum = math.floor(inv.start_i / (4*8) + 1)
			local pagemax = math.max(math.ceil(inv.size / (4*8)), 1)
			local esc = minetest.formspec_escape
			return sfinv.make_formspec(player, context,
				(inv.size == 0 and ("label[3,2;>_<]") or "") ..
				"label[5.8,4.15;" .. minetest.colorize("#FFFF00", tostring(pagenum)) .. " / " .. tostring(pagemax) .. "]" ..
				[[
					image[4,3.9;0.8,0.8;creative_trash_icon.png]
					listcolors[#00000069;#5A5A5A;#141318;#30434C;#FFF]
					list[current_player;main;0,4.7;8,1;]
					list[current_player;main;0,5.85;8,3;8]
					list[detached:trash;main;4,3.9;1,1;]
					listring[]
					button[5,4.05;0.8,0.8;creative_prev;<]
					button[7.25,4.05;0.8,0.8;creative_next;>]
					button[2.63,4.05;0.8,0.8;creative_search;?]
					button[3.25,4.05;0.8,0.8;creative_clear;X]
				]] ..
				"listring[current_player;main]" ..
				"field_close_on_enter[creative_filter;false]" ..
				"field[0.3,4.2;2.8,1.2;creative_filter;;" .. esc(inv.filter) .. "]" ..
				"listring[detached:creative_" .. player_name .. ";main]" ..
				"list[detached:creative_" .. player_name .. ";main;0,0;8,4;" .. tostring(inv.start_i) .. "]" ..
				default.get_hotbar_bg(0,4.7) ..
				default.gui_bg .. default.gui_bg_img .. default.gui_slots
				.. creative.formspec_add, false)
		end,
		on_enter = function(self, player, context)
			local player_name = player:get_player_name()
			local inv = player_inventory[player_name]
			if inv then
				inv.start_i = 0
			end
		end,
		on_player_receive_fields = function(self, player, context, fields)
			local player_name = player:get_player_name()
			local inv = player_inventory[player_name]
			assert(inv)

			if fields.creative_clear then
				inv.start_i = 0
				inv.filter = ""
				--creative.update_creative_inventory(player_name, items)
				sfinv.set_player_inventory_formspec(player, context)
			elseif (fields.creative_search or
					fields.key_enter_field == "creative_filter")
					and fields.creative_filter then
				inv.start_i = 0
				inv.filter = fields.creative_filter:sub(1, 128) -- truncate to a sane length
						:gsub("[%z\1-\8\11-\31\127]", "") -- strip naughty control characters (keeps \t and \n)
						:lower() -- search is case insensitive
				--creative.update_creative_inventory(player_name, items)
				sfinv.set_player_inventory_formspec(player, context)
			elseif not fields.quit then
				local start_i = inv.start_i or 0

				if fields.creative_prev then
					start_i = start_i - 4*8
					if start_i < 0 then
						start_i = inv.size - (inv.size % (4*8))
						if inv.size == start_i then
							start_i = math.max(0, inv.size - (4*8))
						end
					end
				elseif fields.creative_next then
					start_i = start_i + 4*8
					if start_i >= inv.size then
						start_i = 0
					end
				end

				inv.start_i = start_i
				sfinv.set_player_inventory_formspec(player, context)
			end
		end
	})
end

-- Sort registered items
local registered_nodes = {}
local registered_tools = {}
local registered_craftitems = {}

local tab_items = function()
	for name, def in pairs(minetest.registered_items) do
		local group = def.groups or {}

		local nogroup = not (group.node or group.tool or group.craftitem)
		if group.node or (nogroup and minetest.registered_nodes[name]) then
			registered_nodes[name] = def
		elseif group.tool or (nogroup and minetest.registered_tools[name]) then
			registered_tools[name] = def
		elseif group.craftitem or (nogroup and minetest.registered_craftitems[name]) then
			registered_craftitems[name] = def
		end
	end
end

if minetest.register_on_mods_loaded then
	minetest.register_on_mods_loaded(tab_items)
else
	minetest.after(0.1, tab_items)
end

creative.register_tab("all", "Creative", minetest.registered_items)
--creative.register_tab("nodes", "Nodes", minetest.registered_nodes)
--creative.register_tab("tools", "Tools", minetest.registered_tools)
--creative.register_tab("craftitems", "Items", minetest.registered_craftitems)

local old_homepage_name = sfinv.get_homepage_name
function sfinv.get_homepage_name(player)
	if creative.is_enabled_for(player:get_player_name()) then
		return "creative:all"
	else
		return old_homepage_name(player)
	end
end


if minetest.is_creative_enabled("") then
	local hand_hack = function()
		-- Dig time is modified according to difference (leveldiff) between tool
		-- 'maxlevel' and node 'level'. Digtime is divided by the larger of
		-- leveldiff and 1.
		-- To speed up digging in creative, hand 'maxlevel' and 'digtime' have been
		-- increased such that nodes of differing levels have an insignificant
		-- effect on digtime.
	local digtime = 42
	local caps = {times = {digtime, digtime, digtime}, uses = 0, maxlevel = 256}

	minetest.register_item(":", {
		type = "none",
		wield_image = "wieldhand.png",
		wield_scale = {x = 1, y = 1, z = 2.5},
		range = 10,
		tool_capabilities = {
			full_punch_interval = 0.5,
			max_drop_level = 3,
			groupcaps = {
				crumbly = caps,
				cracky  = caps,
				snappy  = caps,
				choppy  = caps,
				oddly_breakable_by_hand = caps,
					-- dig_immediate group doesn't use value 1. Value 3 is instant dig
				dig_immediate = {times = {[2] = digtime, [3] = 0}, uses = 0, maxlevel = 256},
			},
			damage_groups = {fleshy = 10},
		}
	})
	end
	if minetest.register_on_mods_loaded then
		minetest.register_on_mods_loaded(hand_hack)
	else
		minetest.after(0, hand_hack)
	end
end

-- Unlimited node placement
minetest.register_on_placenode(function(pos, newnode, placer, oldnode, itemstack)
	if placer then
		if placer:is_player() then
			return creative.is_creative(placer:get_player_name())
		end
	end
end)

-- Don't pick up if the item is already in the inventory only in hard creative single
if creative_mode_cache then
	local old_handle_node_drops = minetest.handle_node_drops
	function minetest.handle_node_drops(pos, drops, digger)
		if not digger then
			if not digger:is_player() then
				if not creative.is_creative(digger:get_player_name()) then
					return old_handle_node_drops(pos, drops, digger)
				end
			end
		end
		local inv = digger:get_inventory()
		if inv then
			for _, item in ipairs(drops) do
				if not inv:contains_item("main", item, true) then
					inv:add_item("main", item)
				end
			end
		end
	end
end

print("[creative] mod loaded")

