// break-kat corrupts a known-answer-test input in a binary and writes the
// corrupted binary to stdout. This is used to demonstrate that the KATs in the
// binary notice the error.
package main

import (
	"bytes"
	"encoding/hex"
	"flag"
	"fmt"
	"os"
	"sort"
)

var (
	kats = map[string]string{
		"HMAC-SHA-256":                     "dad91293dfcf2a7c8ecd13fe353fa75b",
		"AES-CBC-encrypt":                  "078609a6c5ac2544699adf682fa377f9be8ab6aef563e8c56a36b84f557fadd3",
		"AES-CBC-decrypt":                  "347aa5a024b28257b36510be583d4f47adb7bbeedc6005bbbd0d0a9f06bb7b10",
		"AES-GCM-encrypt":                  "8fcc4099808e75caaff582898848a88d808b55ab4e9370797d940be8cc1d7884",
		"AES-GCM-decrypt":                  "35f3058f875760ff09d3120f70c4bc9ed7a86872e13452202176f7371ae04faae1dd391920f5d13953d896785994823c",
		"DRBG":                             "c4da0740d505f1ee280b95e58c4931ac6de846a0152fbb4a3f174cf4787a4f1a40c2b50babe14aae530be5886d910a27",
		"DRBG-reseed":                      "c7161ca36c2309b716e9859bb96c6d49bdc8352103a18cd24ef42ec97ef46bf446eb1a4576c186e9351803763a7912fe",
		"SHA-1":                            "132fd9bad5c1826263bafbb699f707a5",
		"SHA-256":                          "ff3b857da7236a2baa0f396b51522217",
		"SHA-512":                          "212512f8d2ad8322781c6c4d69a9daa1",
		"SHA3-256":                         "d83c721ee51b060c5a41438a8221e040",
		"HKDF-SHA-256":                     "ca5e6410e7a52332fe0ab3601212a7d3dbdf55a162af42a5daf38b94f24523477e880dd711508684cc21",
		"TLS-KDF":                          "abc3657b094c7628a0b282996fe75a75f4984fd94d4ecc2fcf53a2c469a3f731",
		"PBKDF2":                           "4157532d4c434649505370617373776f7264",
		"SSKDF":                            "39a1e2b3899e87efecf6271282d8f8008f252686dd35bfc39a0f71478da48c691565cee431254dd50cab7462c6cf199be9bf5c",
		"KBKDF":                            "dd1d91b7d90b2bd3138533ce92b272fbf8a369316aefe242e659cc0ae238afe0",
		"RSA-sign":                         "d2b56e53306f720d7929d8708bf46f1c22300305582b115bedcac722d8aa5ab2",
		"RSA-verify":                       "c5ef030d00a13e3a705b23e1e3de3f2c8e84b2e82d1bec14116f8245e5b6fa4b207f12afe72c8d36675acb7d670f6a5c590e44716c3df31104bea89f61becd6cc188814801d308ceec2a843ec7f25ebcdde588dd8980326928b10843c4b3190338b0a07c5a94a53f6d84de7947f3db3d9f730610bf463c291fd901ab8a54e47dbb196d8af53ef15fd06b7ad0ddb65c83c8fee8d0a708334733cbe7b03322695d9a406a687867b38294d8634805d4cb68737355d776484df3cd8b8feb51fd94608218d88eb2f2d04064312ad0412e7a96d49a9d71d76606ab7a5bd99bc31c37593c837f15ba8601926a1f6919d1110b55906e18e29e2a94045ae9218bc6c8da74",
		"ECDSA-sign":                       "1e35930be860d0942ca7bbd6f6ded87f157e4de24f81ed4b875c0e018e89a81f",
		"ECDSA-verify":                     "6780c5fc70275e2c7061a0e7877bb174deadeb9887027f3fa83654158ba7f50c3c77d1b6e09e747bc5ab5501d75e618d8e5b272e159ff3413cb71a81408d5605",
		"Z-computation":                    "e7604491269afb5b102d6ea52cb59feb70aede6ce3bfb3e0105485abd861d77b",
		"FFDH":                             "a14f8ad36be37b18b8f35864392f150ab7ee22c47e1870052a3f17918274af18aaeaf4cf6aacfde96c9d586eb7ebaff6b03fe3b79a8e2ff9dd6df34caaf2ac70fd3771d026b41a561ee90e4337d0575f8a0bd160c868e7e3cef88aa1d88448b1e4742ba11480a9f8a8b737347c408d74a7d57598c48875629df0c85327a124ddec1ad50cd597a985588434ce19c6f044a1696b5f244b899b7e77d4f6f20213ae8eb15d37eb8e67e6c8bdbc4fd6e17426283da96f23a897b210058c7c70fb126a5bf606dbeb1a6d5cca04184c4e95c2e8a70f50f5c1eabd066bd79c180456316ac02d366eb3b0e7ba82fb70dcbd737ca55734579dd250fffa8e0584be99d32b35",
		"ED25519-sign":                     "1961d1d52d8c045fdfc1c682b35f07aae1d3b6e548639830eed929bc122d799f",
		"ED25519-verify":                   "44f238f7ea7154cedd95631144078ffec655938f73e29676728b400f8f46c8048d5bf3ab124342ebae54b6e04f3f167b5ee8bdcfec9be6ff65bcc69a7889670a",
		"ED25519ph-sign":                   "f9bfec63c4e073fa971a80499147d00acd26aae1ff0364db20f8a7a4954db387",
		"ED25519ph-verify":                 "101bcca256ef620bb087592e917336d8542b71728e2a2748c55c719b825dad4521bfb375624b27fff81cbf7165e5ba4a98e8c451b3c3c2fa23278f2bb7458107",
		"ML-KEM-keyGen-decaps":             "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",
		"ML-KEM-keyGen-encaps":             "a7cc68f8d02110ca5720223b9e2a8987c8a24835a20dabcbefa430e74a85af80b9b74b74574c5e5f585459ca3610940f0b57b33344ceacccc135557b82f4968688a0168c1aa2940e5604482bf8b900a4343096446330cfee10917c0338181b7fe8d30f4816087d6299f225417f533ee40473894847bd45291367be6b1a7dee55bb21d60e3828552f4c8a6f3c54fc74cf67a614eeab002a076851879cef2218fdb3766123c2de32a269b6aa4661b69370314c004446f7258c40b2ea789f40ca023bbb1217c2c44b380c6e3194edd129d039218d9b75194a386d944acce7a9720ab026362004e95bf9229290b53613416082e82ba8a42a5ff14759f57712395706b307f7635ecee42317a48eb3b90683f3ab53d17c50f53c6cfb1c0cf59d6f2a981021428a7ac5edf13b26844d83c31d3608710e623aabb24b6c5b48baebc1b3078972589201b7a30bc09315612b067655bec403a69c89eb137c31157971098cb693ba4ae9ae40a8031cec92580bcc1b5ab3ecd1aa5f79aa2cd69249d138c8965a81c87a07eb59a4612e60658f4df028cef8af1b837e0ab0bfedb726904290d0bc41df6a67f7a4166609952439631960648e229a21f2a4d82abad3ec8135dc9bb43c703d3b33e437c9ef7bca91c3465676740125a15ad1707088b101b4273d3c4bf30181b4b2575de75ccfc13312a2a6bcebc477a9668e751629b569bfa20beca09800992c63f04b6b4a7df977a00131c2f8722e5138775235b517a709852167c1d415fdc7ad32f2aaca437e9cc6b248d9ca7c65b405e68d24e81b8688caa22b3cf5c9b147f0cc27e667a80b83ccbb4f4161a5ffd0194b1b5720e68ea0f59997f26740972d2c8124d7a6ad8327c2075a3f76b968d2aaad19c00697599e0be49fa6d65b4088b0be692dcda028095852a0d7205e4417409c0317780305a878fb582963b6953f6b8f0880b050178301d659be3a4db7e0bf2587129164178f32707d40392d85713dea82913999aa6c35aa94b3547abe40e2b0ba82b5b78319182a5a47d173176ba6fa3a4d70a8130b310743fa8aaae314381c9b7f49991f19a4ff8369638de380b5d828b7b5fcfdd91fe68efe16d4cd9eed66d65c1d2d8caf5af4a692",
		"ML-KEM-encapsulate-ciphertext":    "431a4f1b2d2c6c00f1690bbe482541ef3d563774daff83207f96de7e5e4a59d5d936d9443ad422e645793e7a60a9b0a76cd672d20c69b82a5563df52d96f9a6cdfc56fbd4fd8d5a8afeb2a09d92ec854094794b4ed2db381f04c68439608aa9902a4d1689e2eb1e5f07a4a1c709262d7c2ff2f81f6eeaab2a86a41ba210eb1bf8e75febccd1a15b4d7a7b60257c89d00bd81d39fcb8d1ce3278102595dd652f7fb7d5584874f3327b174043b350ebd4d41fe08bd0e854d41cbb027c481da64dc6151b88dececcf022ddac2e22736c147e0773294231c0589967154c526b0b7cdd59568eeff5749a40cb100c60c6480897655d96e9f64d61684c0b3150646732c19409fe565540a31894703cf0179cae85bc8c1a5732649836e48e676405b9591b65ba25f9b489b9e5772aa1ed5a00143cb9f5449fd013457a3c13874cb58c75b52c9b6a9ae495ccb504a89cb5f145695b921632fb85b0316b30d4ad17fef0862d6b1e6ca6a611c8a6a7234b4362c5ca0ad9f7697687798cf624dc9f35fbb376e09953156532a9033709df755b46cc6d83de3a111e19a76b361e0ef14c91db8d91c6c6d9e3e46f42291fd6cbf5cfd122716fb0675698e602ab39ee98e0d8145eebaaa9374f5b3bb0df4d0fd83a40e0d25038c39e9bee01cf79c86f3086158d031d5c5e86bc7e7eb16e622505f2888213884c0b5252289b11fce5bfeebfbef0a32ceaf9c14c6250090028463db6f8d19684f541108fe934d88e7ef5cce9daebb32700b9397691a684298c9bf1b7c22d1bcec3fcacfbb17f2ed2b98b85e6a8fe2482996b5e099e9d0211cb9412614de87dc18d23613ed7f6c29cc37b727116dd901c2817938c29fcd026089336addc09eca90de9a25a6374fee86bcdd06ae3daaf0b1bc5b3b2790d4d9f759bef8ac743612a2bbf6e45de8b22efa61226625d4c39f346b844c5ebec5355866c00b726cc1640cb237c34a20a7c603d251f46e6b3b0fa71b3276835e3e9da5b9485e789614af49f1e9504db2528631fbe1cd7dbee85164e4c099a27a4583e9247d078f8830b46874c1b010bf3cd90eb0774961f239ba",
		"ML-KEM-encapsulate-shared-secret": "a772df2de250ac7d896bbb820b57f2ae05f9a412ab55baa421d4af6dac62662a",
		"ML-DSA-sign":                      "4a414b45204d415353494d4f20415753323032352e0102030405060708090a0b",
		"ML-DSA-verify":                    "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",
	}

	listTests = flag.Bool("list-tests", false, "List known test values and exit")
)

func main() {
	flag.Parse()

	if *listTests {
		for _, kat := range sortedKATs() {
			fmt.Println(kat)
		}
		os.Exit(0)
	}

	if flag.NArg() != 2 || kats[flag.Arg(1)] == "" {
		fmt.Fprintln(os.Stderr, "Usage: break-kat <binary path> <test to break> > output")
		fmt.Fprintln(os.Stderr, "Possible values for <test to break>:")
		for _, kat := range sortedKATs() {
			fmt.Fprintln(os.Stderr, " ", kat)
		}
		os.Exit(1)
	}

	inPath := flag.Arg(0)
	test := flag.Arg(1)
	testInputValue, err := hex.DecodeString(kats[test])
	if err != nil {
		panic("invalid kat data: " + err.Error())
	}

	binaryContents, err := os.ReadFile(inPath)
	if err != nil {
		fmt.Fprintln(os.Stderr, err)
		os.Exit(2)
	}

	i := bytes.Index(binaryContents, testInputValue)
	if i < 0 {
		fmt.Fprintln(os.Stderr, "Expected test input value was not found in binary.")
		os.Exit(3)
	}

	// Zero out the entire value because the compiler may produce code
	// where parts of the value are embedded in the instructions.
	for j := range testInputValue {
		binaryContents[i+j] = 0
	}

	if bytes.Index(binaryContents, testInputValue) >= 0 {
		fmt.Fprintln(os.Stderr, "Test input value was still found after erasing it. Second copy?")
		os.Exit(4)
	}

	os.Stdout.Write(binaryContents)
}

func sortedKATs() []string {
	var ret []string
	for kat := range kats {
		ret = append(ret, kat)
	}
	sort.Strings(ret)
	return ret
}
