/*****************************************************************************
 *
 * $Id$
 *
 * Copyright (C) 2009 - 2012  Richard Hacker <lerich@gmx.net>
 *                            Florian Pose <fp@igh-essen.com>
 *
 * This file is part of the PdCom library.
 *
 * The PdCom library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The PdCom library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the PdCom Library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "pdcom/Exception.h"

#include "MsrParam.h"
#include "MsrProtocolHandler.h"

#include <cstdlib>
#include <sstream>

using namespace MSRProto;

#define DEBUG 0

#if DEBUG
#include <iostream>
#endif

const char *Param::hexvalues[] = {
    "00", "01", "02", "03", "04", "05", "06", "07",
    "08", "09", "0A", "0B", "0C", "0D", "0E", "0F",
    "10", "11", "12", "13", "14", "15", "16", "17",
    "18", "19", "1A", "1B", "1C", "1D", "1E", "1F",
    "20", "21", "22", "23", "24", "25", "26", "27",
    "28", "29", "2A", "2B", "2C", "2D", "2E", "2F",
    "30", "31", "32", "33", "34", "35", "36", "37",
    "38", "39", "3A", "3B", "3C", "3D", "3E", "3F",
    "40", "41", "42", "43", "44", "45", "46", "47",
    "48", "49", "4A", "4B", "4C", "4D", "4E", "4F",
    "50", "51", "52", "53", "54", "55", "56", "57",
    "58", "59", "5A", "5B", "5C", "5D", "5E", "5F",
    "60", "61", "62", "63", "64", "65", "66", "67",
    "68", "69", "6A", "6B", "6C", "6D", "6E", "6F",
    "70", "71", "72", "73", "74", "75", "76", "77",
    "78", "79", "7A", "7B", "7C", "7D", "7E", "7F",
    "80", "81", "82", "83", "84", "85", "86", "87",
    "88", "89", "8A", "8B", "8C", "8D", "8E", "8F",
    "90", "91", "92", "93", "94", "95", "96", "97",
    "98", "99", "9A", "9B", "9C", "9D", "9E", "9F",
    "A0", "A1", "A2", "A3", "A4", "A5", "A6", "A7",
    "A8", "A9", "AA", "AB", "AC", "AD", "AE", "AF",
    "B0", "B1", "B2", "B3", "B4", "B5", "B6", "B7",
    "B8", "B9", "BA", "BB", "BC", "BD", "BE", "BF",
    "C0", "C1", "C2", "C3", "C4", "C5", "C6", "C7",
    "C8", "C9", "CA", "CB", "CC", "CD", "CE", "CF",
    "D0", "D1", "D2", "D3", "D4", "D5", "D6", "D7",
    "D8", "D9", "DA", "DB", "DC", "DD", "DE", "DF",
    "E0", "E1", "E2", "E3", "E4", "E5", "E6", "E7",
    "E8", "E9", "EA", "EB", "EC", "ED", "EE", "EF",
    "F0", "F1", "F2", "F3", "F4", "F5", "F6", "F7",
    "F8", "F9", "FA", "FB", "FC", "FD", "FE", "FF",
};

/****************************************************************************/

Param::Param(
        ProtocolHandler *protocolhandler,
        std::ostream* os,
        const char *name,
        unsigned int index,
        unsigned int rnum,
        unsigned int cnum,
        const char *orientation,
        unsigned int msr_flags,
        unsigned int datasize,
        const char *unit,
        const char *datatype):
    Variable(protocolhandler, os, name, "", index, rnum, cnum,
            orientation, datatype, datasize, 0.0, "")
{
    writeable = true;
    requestActive = false;
    pollActive = false;
    subscribed = false;
    binparameters = msrHandler->hasFeature("binparameters");
    pushparameters = msrHandler->hasFeature("pushparameters");
    pmtime = msrHandler->hasFeature("pmtime");

    allocateMemory();
}

/****************************************************************************/

Param::~Param()
{
}

/****************************************************************************/

void Param::valueChanged(const char *src, size_t len)
{
#if DEBUG
    std::cerr << "Set value for parameter " << index << "." << std::endl;
#endif

    if (binparameters) {
#if DEBUG
        std::cerr << "binparameters." << std::endl;
#endif
        unsigned char* end = (unsigned char *) dataPtr + memSize;
        *os << "<wp index=\"" << index << "\" hexvalue=\"";
        for (unsigned char* p = (unsigned char *) dataPtr; p != end; p++) {
            *os << hexvalues[*p];
        }
        *os << "\" />\n";
    } else {
#if DEBUG
        std::cerr << "no binparameters." << std::endl;
#endif
        double value[elementCount];
        getValue(value, elementCount);
        *os << "<wp index=\"" << index << "\" value=\"";
        for (size_t i = 0; i < elementCount; i++) {
            *os << value[i];
        }
        *os << "\" />\n";
    }

    os->flush();
}

/****************************************************************************/

void Param::newValues(const char *asc_time, const char *buf)
{
    double value[elementCount];
    char delim;
    unsigned int bufPos = 0;

#if DEBUG
    std::cerr << "New values for parameter " << index << ": "
        << buf << std::endl;
#endif

    std::stringstream str;
    str.imbue(std::locale("C")); // use '.' as a decimal separator!
    str << buf;

    do {
        str >> value[bufPos];
#if DEBUG
        std::cerr << "value[" << bufPos << "]=" << value[bufPos] << std::endl;
#endif
        bufPos++;
        str >> delim;

        if (transpose_skip) {
            bufPos += transpose_skip;

            if (bufPos == elementCount + transpose_skip) {
                bufPos = 0;
            }
            else if (bufPos >= elementCount) {
                bufPos -= elementCount - 1;
            }
        }
        else if (bufPos >= elementCount) {
            bufPos = 0;
        }
    } while (bufPos);

    pushValue(value, elementCount);

    if (asc_time && pmtime) {
        std::stringstream str;
        double t;
        str.imbue(std::locale("C")); // use '.' as a decimal separator!
        str << asc_time;
        str >> t;
        mtime = t;
    }

    if (subscribed) {
        notifySubscribers(0);
    }

    if (pollActive) {
        notifyPoll();
        pollActive = false;
    }

    requestActive = false;
}

/****************************************************************************/

unsigned int Param::addTransmissionInterval(double interval)
{
    if (interval) {
        throw PdCom::VariableException("Variable is only available as event");
    }

    if (!pushparameters) {
        throw PdCom::VariableException(
                "MSR Protocol does not have \"pushparameters\" feature.");
    }

    subscribed = true;
    requestParameter();

    return 0;
}

/****************************************************************************/

void Param::rmTransmissionDecimation(int decimation)
{
    subscribed = false;
}

/****************************************************************************/

void Param::parameterUpdate()
{
#if DEBUG
    std::cerr << "Parameter " << index << " update." << std::endl;
#endif
    if (subscribed) {
        requestParameter();
    }
}

/****************************************************************************/
void Param::pollVariable()
{
    requestParameter();
    pollActive = true;
}

/****************************************************************************/

void Param::requestParameter()
{
#if DEBUG
    std::cerr << "Requesting parameter " << index << "." << std::endl;
#endif
    if (!requestActive) {
        requestActive = true;
        *os << "<rp index=\"" << index << "\" />\n";
    }
}

/****************************************************************************/
