/*****************************************************************************
 *
 * Copyright (C) 2017  Florian Pose <fp@igh-essen.com>
 *
 * This file is part of the DLS widget library.
 *
 * The DLS widget library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * The DLS widget library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the DLS widget library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "FilterDialog.h"

#include <QPushButton>

#define PCRE2_CODE_UNIT_WIDTH 8
#include <pcre2.h>

using DLS::FilterDialog;

/****************************************************************************/

static QString get_pcre2_error_message(int error_code)
{
    PCRE2_SIZE buf_size = 256;
    std::vector<PCRE2_UCHAR> buffer(buf_size);

    int len = pcre2_get_error_message(error_code, buffer.data(), buf_size);
    if (len < 0) {
        return "Unknown PCRE2 error code " + QString().setNum(error_code);
    }

    return QString::fromLocal8Bit(
            reinterpret_cast<char *>(buffer.data()));
}

/****************************************************************************/

/** Constructor.
 */
FilterDialog::FilterDialog(
        QWidget *parent,
        const QString &re
        ):
    QDialog(parent),
    valid(true) // empty regex is valid
{
    setupUi(this);

    connect(lineEdit, SIGNAL(textChanged(const QString &)),
            this, SLOT(textChanged(const QString &)));

    lineEdit->setText(re);
}

/****************************************************************************/

/** Destructor.
 */
FilterDialog::~FilterDialog()
{
}

/****************************************************************************/

void FilterDialog::accept()
{
    done(Accepted);
}

/****************************************************************************/

void FilterDialog::textChanged(const QString &text)
{
    regex = text;

    if (text.isEmpty()) {
        valid = true;
        updateButtons();
        return;
    }

    int pcre_errorcode{0};
    PCRE2_SIZE pcre_erroroffset{0};

    pcre2_code *re = pcre2_compile((PCRE2_SPTR) text.toUtf8().constData(),
            PCRE2_ZERO_TERMINATED, 0, &pcre_errorcode, &pcre_erroroffset,
            NULL);
    if (re == NULL) {
        QString message(get_pcre2_error_message(pcre_errorcode));
        QString err;
        err = tr("At %1: %2").arg(pcre_erroroffset).arg(message);
        labelStatus->setText(err);
        valid = false;
        updateButtons();
    }
    else {
        pcre2_code_free(re);
        valid = true;
        updateButtons();
    }
}

/****************************************************************************/

void FilterDialog::updateButtons()
{
    if (valid) {
        labelStatus->setText(tr("Pattern valid."));
    }
    buttonBox->button(QDialogButtonBox::Ok)->setEnabled(valid);
}

/****************************************************************************/
