classdef el2535 < EtherCATSlave

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    methods
        %====================================================================
        function obj = el2535(id)
            if nargin > 0
                obj.slave = obj.find(id);
            end
        end

        %====================================================================
        function rv = configure(obj,channels,extra_output, sdo_config)

            rv.SlaveConfig.vendor = 2;
            rv.SlaveConfig.description = obj.slave{1};
            rv.SlaveConfig.product  = obj.slave{2};
            rv.OutputAmpere = obj.slave{4};
            % PDO entries have changed since release 0x0017
            pdos = el2535.pdo(obj.slave{5});

            extra_output_1 = ~isequal(extra_output(1:2), [1; 1]);
            extra_output_2 = ~isequal(extra_output(3:4), [1; 1]);
            rx_select = find([channels(1), channels(2), 0, 0, 0, 0]);
            tx_select = find([0, 0, channels(1), extra_output_1, channels(2), extra_output_2]);

            % enable dithering input port conditionally
            rx_entry_select = [ ...
                sdo_config( 1), ones(1, size(pdos{1}{3}, 1) - 1);
                sdo_config(16), ones(1, size(pdos{2}{3}, 1) - 1);
            ];

            % x1600 and x1601 are mandatory in old revisions,
            % so always map both channels.
            rv.SlaveConfig.sm = {...
                 {2,0, arrayfun(@(x) {pdos{x}{1}, pdos{x}{2}},...
                                 [1,2], 'UniformOutput', false)},...
                 {3,1, arrayfun(@(x) {pdos{x}{1}, pdos{x}{2}},...
                                 tx_select, 'UniformOutput', false)}
            };

            inputs = arrayfun(@(i) {...
                arrayfun(@(o) ...
                    struct('pdo', [0, i-1, pdos{i}{3}{o, 1}, 0], ...
                           'pdo_data_type',  pdos{i}{3}{o, 3}, ...
                           'portname',    pdos{i}{3}{o, 2}, ...
                           'full_scale',  pdos{i}{3}{o, 4} ...
                           ) ...
                    , find(rx_entry_select(i, :))) ...
                }, rx_select);

            % modify pdo info according to selected extra outputs
            for i = 1:4
                pdoentryidx = el2535.extra_output_popup_pdo_entry(i, 1);
                idx = el2535.extra_output_popup_pdo_entry(i, 2);
                pdos{pdoentryidx}{3}{idx, 2} = ...
                    el2535.extra_output_details{extra_output(i), 1};

                pdos{pdoentryidx}{3}{idx, 4} = ...
                    el2535.extra_output_details{extra_output(i), 2};
                if el2535.extra_output_details{extra_output(i), 3}
                    % relative to rated current
                    pdos{pdoentryidx}{3}{idx, 4} = ...
                        pdos{pdoentryidx}{3}{idx, 4} / obj.slave{4};
                end
            end


            outputs = arrayfun(@(i) {...
                arrayfun(@(o) ...
                    struct('pdo', [1, i-1, pdos{tx_select(i)}{3}{o, 1}, 0], ...
                           'pdo_data_type',  pdos{tx_select(i)}{3}{o, 3}, ...
                           'portname',    pdos{tx_select(i)}{3}{o, 2}, ...
                           'full_scale',  pdos{tx_select(i)}{3}{o, 4} ...
                           ) ...
                    , 1:size(pdos{tx_select(i)}{3}, 1)) ...
                }, 1:length(tx_select));

            rv.PortConfig.input = horzcat(inputs{:});
            rv.PortConfig.output = horzcat(outputs{:});
            rv.SlaveConfig.sdo = num2cell(...
                horzcat(el2535.sdo, sdo_config'));
        end
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    methods (Static)
        %====================================================================
        function modelChanged()
            obj = el2535(get_param(gcbh,'model'));
            obj.updateRevision();
        end
        %====================================================================
        function updateChannelEnable()
            names = get_param(gcbh,'MaskNames');
            values = get_param(gcbh,'MaskValues');

            ch1_sdos = arrayfun(@(x) sprintf('sdo_%d', x),  1:13, ...
                'UniformOutput', false);
            ch2_sdos = arrayfun(@(x) sprintf('sdo_%d', x), 16:28, ...
                'UniformOutput', false);

            enable_channels = values([...
                find(strncmp(names, 'ch1enable', 9)); ...
                find(strncmp(names, 'ch2enable', 9))]);
            ch_vars = [...
                find(ismember(names, ch1_sdos)), ...
                find(ismember(names, ch2_sdos))];
            ch_enable = cell2mat(...
                arrayfun(@(i) repmat(...
                        strcmp(enable_channels(i), 'on'), ...
                        size(ch_vars(:,i))...
                    ), 1:length(enable_channels), ...
                'UniformOutput', false));

            EtherCATSlave.setEnable(...
                vertcat(names(ch_vars, :)), ...
                vertcat(ch_enable(:)));

        end
        %====================================================================

        %====================================================================
        function test(p)
            ei = EtherCATInfo(fullfile(p,'Beckhoff EL25xx.xml'));
            models = el2535.models(strncmp(el2535.models(:,1), 'EL', 2), :);
            el2535.testModels(ei, models);
        end

        %====================================================================
        function testModels(ei, models)
            for i = 1:size(models,1)
                fprintf('Testing %s\n', models{i,1});
                slave = ei.getSlave(models{i,2},...
                        'revision', models{i,3});

                default_sdo = zeros(1, 30);

                rv = el2535(models{i,1}).configure([1; 1], [2, 3, 4, 5], default_sdo);
                slave.testConfig(rv.SlaveConfig,rv.PortConfig);

                rv = el2535(models{i,1}).configure([1; 0], [2, 3, 4, 5], default_sdo);
                slave.testConfig(rv.SlaveConfig,rv.PortConfig);

                rv = el2535(models{i,1}).configure([0; 1], [6, 7, 8, 9], default_sdo);
                slave.testConfig(rv.SlaveConfig,rv.PortConfig);
            end
        end
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    properties (Access = private, Constant)

        %     Pdo             [EntryIdx, EntrySubidx, Bitlen]
        pdo = {
            {hex2dec('1600'), [
                                hex2dec('7000'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   4; ...
                                hex2dec('7000'), hex2dec('06'),   1; ...
                                hex2dec('7000'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   8; ...
                                hex2dec('7000'), hex2dec('11'),  16; ...
            ], {             % {idx, portname, datatype, full_scale}
                                0, 'Ch.1 Dithering', uint(1), []; ...
                                2, 'Ch.1 Enable', uint(1), []; ...
                                3, 'Ch.1 Reset', uint(1), []; ...
                                6, 'Ch.1 PWM Output [-1;1]', sint(16), hex2dec('7fff')}}, ...
            {hex2dec('1601'), [
                                hex2dec('7010'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   4; ...
                                hex2dec('7010'), hex2dec('06'),   1; ...
                                hex2dec('7010'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   8; ...
                                hex2dec('7010'), hex2dec('11'),  16; ...
            ], {
                                0, 'Ch.2 Dithering', uint(1), []; ...
                                2, 'Ch.2 Enable', uint(1), []; ...
                                3, 'Ch.2 Reset', uint(1), []; ...
                                6, 'Ch.2 PWM Output [-1;1]', sint(16), hex2dec('7fff')}}, ...
            {hex2dec('1a00'), [
                                hex2dec('6000'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6000'), hex2dec('05'),   1; ...
                                hex2dec('6000'), hex2dec('06'),   1; ...
                                hex2dec('6000'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('6000'), hex2dec('10'),   1; ...
            ], {                0, 'Ch.1 Digital Input', uint(1), []; ...
                                2, 'Ch.1 Ready', uint(1), [];...
                                3, 'Ch.1 Warning', uint(1), []; ...
                                4, 'Ch.1 Error', uint(1), []; ...
                                7, 'Ch.1 TxPDO Toggle', uint(1), []}}, ...
            {hex2dec('1a01'), [
                                hex2dec('6000'), 17,   16; ...
                                hex2dec('6000'), 18,   16; ...
            ], {                0, '', sint(16), [];
                                1, '', sint(16), [];}}, ...
            {hex2dec('1a02'), [
                                hex2dec('6010'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6010'), hex2dec('05'),   1; ...
                                hex2dec('6010'), hex2dec('06'),   1; ...
                                hex2dec('6010'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('6010'), hex2dec('10'),   1; ...
            ], {
                                0, 'Ch.2 Digital Input', uint(1), []; ...
                                2, 'Ch.2 Ready', uint(1), []; ...
                                3, 'Ch.2 Warning', uint(1), []; ...
                                4, 'Ch.2 Error', uint(1), []; ...
                                7, 'Ch.2 TxPDO Toggle', uint(1), [];}}, ...
            {hex2dec('1a03'), [
                                hex2dec('6010'), 17,   16; ...
                                hex2dec('6010'), 18,   16; ...
            ], {                0, '', sint(16), [];
                                1, '', sint(16), [];}}, ...
            {hex2dec('1600'), [
                                hex2dec('7000'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   4; ...
                                hex2dec('7000'), hex2dec('06'),   1; ...
                                hex2dec('7000'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   9; ...
                                hex2dec('7000'), hex2dec('11'),  16; ...
            ], {
                                0, 'Ch.1 Dithering', uint(1), []; ...
                                2, 'Ch.1 Enable', uint(1), []; ...
                                3, 'Ch.1 Reset', uint(1), []; ...
                                5, 'Ch.1 PWM Output [-1;1]', sint(16), hex2dec('7fff')}}, ...
            {hex2dec('1601'), [
                                hex2dec('7010'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   4; ...
                                hex2dec('7010'), hex2dec('06'),   1; ...
                                hex2dec('7010'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   9; ...
                                hex2dec('7010'), hex2dec('11'),  16; ...
            ], {
                                0, 'Ch.2 Dithering', uint(1), []; ...
                                2, 'Ch.2 Enable', uint(1), []; ...
                                3, 'Ch.2 Reset', uint(1), []; ...
                                5, 'Ch.2 PWM Output [-1;1]', sint(16), hex2dec('7fff')}}, ...
            {hex2dec('1a00'), [
                                hex2dec('6000'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6000'), hex2dec('05'),   1; ...
                                hex2dec('6000'), hex2dec('06'),   1; ...
                                hex2dec('6000'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('1800'), hex2dec('09'),   1; ...
            ], {                0, 'Ch.1 Digital Input', uint(1), []; ...
                                2, 'Ch.1 Ready', uint(1), [];...
                                3, 'Ch.1 Warning', uint(1), []; ...
                                4, 'Ch.1 Error', uint(1), []; ...
                                7, 'Ch.1 TxPDO Toggle', uint(1), []}}, ...
            {hex2dec('1a02'), [
                                hex2dec('6010'), hex2dec('01'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6010'), hex2dec('05'),   1; ...
                                hex2dec('6010'), hex2dec('06'),   1; ...
                                hex2dec('6010'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('1802'), hex2dec('09'),   1; ...
            ], {
                                0, 'Ch.2 Digital Input', uint(1), []; ...
                                2, 'Ch.2 Ready', uint(1), []; ...
                                3, 'Ch.2 Warning', uint(1), []; ...
                                4, 'Ch.2 Error', uint(1), []; ...
                                7, 'Ch.2 TxPDO Toggle', uint(1), [];}}, ...
            {hex2dec('1a00'), [
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6000'), hex2dec('05'),   1; ...
                                hex2dec('6000'), hex2dec('06'),   1; ...
                                hex2dec('6000'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('6000'), hex2dec('10'),   1; ...
            ], {                2, 'Ch.1 Ready', uint(1), [];...
                                3, 'Ch.1 Warning', uint(1), []; ...
                                4, 'Ch.1 Error', uint(1), []; ...
                                7, 'Ch.1 TxPDO Toggle', uint(1), []}}, ...
            {hex2dec('1a02'), [
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   3; ...
                                hex2dec('6010'), hex2dec('05'),   1; ...
                                hex2dec('6010'), hex2dec('06'),   1; ...
                                hex2dec('6010'), hex2dec('07'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   1; ...
                                hex2dec('0000'), hex2dec('00'),   7; ...
                                hex2dec('6010'), hex2dec('10'),   1; ...
            ], {
                                2, 'Ch.2 Ready', uint(1), []; ...
                                3, 'Ch.2 Warning', uint(1), []; ...
                                4, 'Ch.2 Error', uint(1), []; ...
                                7, 'Ch.1 TxPDO Toggle', uint(1), [];}}, ...
        }

        extra_output_popup_pdo_entry = [
            4, 1;
            4, 2;
            6, 1;
            6, 2;
        ]

        extra_output_details = {...
            % name, full_scale, divide by rated current
            'Channel 1 Actual Current [A]', 1024, 1;
            'Channel 1 Actual Current [A]', 1024, 1;
            'Channel 1 Set Current [A]',    1024, 1;
            'Channel 1 Duty Cycle [-1;1]',  1000, 0;
            'Channel 1 Supply Voltage [V]', 1000, 0;
            'Channel 2 Actual Current [A]', 1024, 1;
            'Channel 2 Set Current [A]',    1024, 1;
            'Channel 2 Duty Cycle [-1;1]',  1000, 0;
            'Channel 2 Supply Voltage [V]', 1000, 0;
        }

        sdo = [
            hex2dec('8000'), hex2dec('03'),   8; %  1 Dithering
            hex2dec('8000'), hex2dec('04'),   8; %  2 Invert Polarity
            hex2dec('8000'), hex2dec('05'),   8; %  3 Watchdog
            hex2dec('8000'), hex2dec('0B'), -16; %  4 Offset
            hex2dec('8000'), hex2dec('0C'), -32; %  5 Gain
            hex2dec('8000'), hex2dec('0D'), -16; %  6 Default Output
            hex2dec('8000'), hex2dec('0E'),  16; %  7 Default Output Ramp
            hex2dec('8000'), hex2dec('10'),   8; %  8 Max current
            hex2dec('8000'), hex2dec('12'),  16; %  9 Kp factor
            hex2dec('8000'), hex2dec('13'),  16; % 10 Ki factor
            hex2dec('8000'), hex2dec('14'),  16; % 11 Kd factor
            hex2dec('8000'), hex2dec('1E'),  16; % 12 Dithering frequency
            hex2dec('8000'), hex2dec('1F'),   8; % 13 Dithering amplitude
            hex2dec('8000'), hex2dec('21'),   8; % 14 Extra Output 1
            hex2dec('8000'), hex2dec('22'),   8; % 15 Extra Output 2

            hex2dec('8010'), hex2dec('03'),   8; % 16 Dithering
            hex2dec('8010'), hex2dec('04'),   8; % 17 Invert Polarity
            hex2dec('8010'), hex2dec('05'),   8; % 18 Watchdog
            hex2dec('8010'), hex2dec('0B'), -16; % 19 Offset
            hex2dec('8010'), hex2dec('0C'), -32; % 20 Gain
            hex2dec('8010'), hex2dec('0D'), -16; % 21 Default Output
            hex2dec('8010'), hex2dec('0E'),  16; % 22 Default Output Ramp
            hex2dec('8010'), hex2dec('10'),   8; % 23 Max current
            hex2dec('8010'), hex2dec('12'),  16; % 24 Kp factor
            hex2dec('8010'), hex2dec('13'),  16; % 25 Ki factor
            hex2dec('8010'), hex2dec('14'),  16; % 26 Kd factor
            hex2dec('8010'), hex2dec('1E'),  16; % 27 Dithering frequency
            hex2dec('8010'), hex2dec('1F'),   8; % 28 Dithering amplitude
            hex2dec('8010'), hex2dec('21'),   8; % 29 Extra Output 1
            hex2dec('8010'), hex2dec('22'),   8; % 30 Extra Output 2
        ]
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    properties (Constant)
        extra_output_popup_sdo_map = [0, 0, 1, 2, 3, ...
            hex2dec('20'), hex2dec('21'), hex2dec('22'), hex2dec('23')];

        %   Model          ProductCode          Revision             Amp.
        %   Pdo
        models = {...
            'EL2535',           hex2dec('09e73052'), hex2dec('00170000'),  1.0, (1:6)';
            'EL2535-0002',      hex2dec('09e73052'), hex2dec('00170002'),  2.0, (1:6)';
            'EL2535-0005',      hex2dec('09e73052'), hex2dec('00170005'),  5.0, (1:6)';
            'EL2535-0050',      hex2dec('09e73052'), hex2dec('00170032'), 0.05, (1:6)';
            'EL2535-0100',      hex2dec('09e73052'), hex2dec('00170064'),  0.1, (1:6)';
            'EL2535-0103',      hex2dec('09e73052'), hex2dec('00170067'),  3.5, [1;2;11;4;12;6];
            'EL2535-0000-0016', hex2dec('09e73052'), hex2dec('00100000'),  1.0, [7;8; 9;4;10;6];
            'EL2535-0000-0017', hex2dec('09e73052'), hex2dec('00110000'),  1.0, [7;8; 9;4;10;6];
            'EL2535-0000-0018', hex2dec('09e73052'), hex2dec('00120000'),  1.0, [7;8; 9;4;10;6];
            'EL2535-0000-0019', hex2dec('09e73052'), hex2dec('00130000'),  1.0, [7;8; 9;4;10;6];
            'EL2535-0000-0021', hex2dec('09e73052'), hex2dec('00150000'),  1.0, [7;8; 3;4; 5;6];
            'EL2535-0000-0022', hex2dec('09e73052'), hex2dec('00160000'),  1.0, [7;8; 3;4; 5;6];
            'EL2535-0002-0016', hex2dec('09e73052'), hex2dec('00100002'),  2.0, [7;8; 9;4;10;6];
            'EL2535-0002-0017', hex2dec('09e73052'), hex2dec('00110002'),  2.0, [7;8; 9;4;10;6];
            'EL2535-0002-0018', hex2dec('09e73052'), hex2dec('00120002'),  2.0, [7;8; 9;4;10;6];
            'EL2535-0002-0019', hex2dec('09e73052'), hex2dec('00130002'),  2.0, [7;8; 9;4;10;6];
            'EL2535-0002-0020', hex2dec('09e73052'), hex2dec('00140002'),  2.0, [7;8; 9;4;10;6];
            'EL2535-0002-0021', hex2dec('09e73052'), hex2dec('00150002'),  2.0, [7;8; 3;4; 5;6];
            'EL2535-0002-0022', hex2dec('09e73052'), hex2dec('00160002'),  2.0, [7;8; 3;4; 5;6];
            'EL2535-0005-0020', hex2dec('09e73052'), hex2dec('00160005'),  5.0, [7;8; 3;4; 5;6];
            'EL2535-0050-0016', hex2dec('09e73052'), hex2dec('00100032'), 0.05, [7;8; 9;4;10;6];
            'EL2535-0050-0017', hex2dec('09e73052'), hex2dec('00110032'), 0.05, [7;8; 9;4;10;6];
            'EL2535-0050-0018', hex2dec('09e73052'), hex2dec('00120032'), 0.05, [7;8; 9;4;10;6];
            'EL2535-0050-0019', hex2dec('09e73052'), hex2dec('00130032'), 0.05, [7;8; 9;4;10;6];
            'EL2535-0050-0020', hex2dec('09e73052'), hex2dec('00140032'), 0.05, [7;8; 9;4;10;6];
            'EL2535-0050-0021', hex2dec('09e73052'), hex2dec('00150032'), 0.05, [7;8; 3;4; 5;6];
            'EL2535-0050-0022', hex2dec('09e73052'), hex2dec('00160032'), 0.05, [7;8; 3;4; 5;6];
            'EL2535-0100-0019', hex2dec('09e73052'), hex2dec('00130064'),  0.1, [7;8; 9;4;10;6];
            'EL2535-0100-0020', hex2dec('09e73052'), hex2dec('00140064'),  0.1, [7;8; 9;4;10;6];
            'EL2535-0100-0021', hex2dec('09e73052'), hex2dec('00150064'),  0.1, [7;8; 3;4; 5;6];
            'EL2535-0100-0022', hex2dec('09e73052'), hex2dec('00160064'),  0.1, [7;8; 3;4; 5;6];
        };

    end
end
