/*****************************************************************************
 *
 * Testmanager - Graphical Automation and Visualisation Tool
 *
 * Copyright (C) 2018 - 2019  Florian Pose <fp@igh.de>
 *
 * This file is part of Testmanager.
 *
 * Testmanager is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Testmanager is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with Testmanager. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include <Python.h>  // must be first!

#include "MainWindow.h"

#include "AboutDialog.h"
#include "ConnectDialog.h"
#include "DataModel.h"
#include "DataSource.h"
#include "DetachableTabWidget.h"
#include "DetachedDockWidget.h"
#include "LoginDialog.h"
#include "MessageDialog.h"
#include "Parameter.h"
#include "ParameterItemDelegate.h"
#include "ParameterModel.h"
#include "ParameterSaveDialog.h"
#include "ParameterSet.h"
#include "ParameterSetModel.h"
#include "ParameterTableModel.h"
#include "Property.h"
#include "PropertyDelegate.h"
#include "PropertyModel.h"
#include "ReplaceDialog.h"
#include "SlotViewFilter.h"
#include "SourceDelegate.h"
#include "StyleDialog.h"
#include "TabPage.h"
#include "WidgetContainer.h"

#include <QtPdCom1/LoginManager.h>
#include <QtPdCom1/Message.h>
#include <QtPdCom1/MessageModelUnion.h>

// manually load plugins
#include "plugins/BarPlugin.h"
#include "plugins/CheckBoxPlugin.h"
#include "plugins/DialPlugin.h"
#include "plugins/DigitalPlugin.h"
#include "plugins/DoubleSpinBoxPlugin.h"
#include "plugins/GraphPlugin.h"
#include "plugins/LedPlugin.h"
#include "plugins/ParameterSetWidgetPlugin.h"
#include "plugins/PushButtonPlugin.h"
#include "plugins/QLabelPlugin.h"
#include "plugins/RadioButtonPlugin.h"
#include "plugins/RotorPlugin.h"
#include "plugins/SendBroadcastPlugin.h"
#include "plugins/TableViewPlugin.h"
#include "plugins/TextPlugin.h"
#include "plugins/TouchEditPlugin.h"
#include "plugins/XYGraphPlugin.h"

#include <QAbstractItemModel>
#include <QAbstractItemView>
#include <QClipboard>
#include <QDebug>
#include <QElapsedTimer>
#include <QFileDialog>
#include <QItemSelectionModel>
#include <QJsonArray>
#include <QJsonDocument>
#include <QKeyEvent>
#include <QListView>
#include <QMessageBox>
#include <QMimeData>
#include <QObject>
#include <QScrollArea>
#include <QSettings>
#include <QTableView>

#include <cstddef>
#include <memory>
#include <stdexcept>

#include <sys/types.h>  // socketpair()
#include <signal.h>     // SIGINT, SIGTERM

#ifndef _WIN32
#include <sys/socket.h>
#else
#include <io.h>
#include <fcntl.h>
#include <windows.h>  // MAX_PATH
#endif

#ifdef TM_MODEL_TEST
#include "modeltest.h"
#endif

/****************************************************************************/

int multiply(int a, int b)
{
    return a * b;
}

/****************************************************************************/

MainWindow *MainWindow::singleton = NULL;

/****************************************************************************/

static QColor color[] = {
        QColor(38, 139, 210),  // blue
        QColor(220, 50, 47),   // red
        QColor(133, 153, 0),   // green
        QColor(181, 137, 0),   // yellow
        QColor(211, 54, 130)   // magenta
};

/****************************************************************************/

MainWindow::MainWindow(
        const QString &fileName,
        bool newView,
        bool autoConnect,
        QWidget *parent):
    QMainWindow(parent),
    autoConnect(autoConnect),
    pluginMap {new PluginMap()},
    dataModel(new DataModel(this)),
    sourceDelegate(new SourceDelegate(dataModel)),
    slotModelCollection(new SlotModelCollection(dataModel, this)),
    slotViewFilter(new SlotViewFilter(*this, this)),
    tabWidget(new DetachableTabWidget(this)),
    propertyModel(new PropertyModel()),
    propertyDelegate(new PropertyDelegate()),
    propertyNode(nullptr),
    messageModelUnion(new QtPdCom::MessageModelUnion(this)),
    showHistoricMessages {false},
    editMode(false),
    gridStep(10),
    restore(false),
    loadedLayoutIsOldVersion {false},
    layoutChanged {false},
    closeWithoutAsking {false},
    afterSaveFunction {[] {}},
    pythonStdOutNotifier(nullptr),
    pythonStdErrNotifier(nullptr),
    pythonHistoryIndex(0),
    scriptVariable {nullptr},
    parameterSetModel(new ParameterSetModel(this)),
    defaultParameterModel(new ParameterModel(this)),
    labelIn(new QLabel(this)),
    labelOut(new QLabel(this)),
    statusMessageIcon(new QLabel(this)),
    statusMessageText(new QLabel(this))
{
    if (singleton) {
        throw(std::runtime_error("Only one instance allowed!"));
    }
    singleton = this;

    setupUi(this);

    verticalLayoutCentral->addWidget(tabWidget);

    setWindowIcon(QPixmap(":/images/testmanager.svg"));

#ifdef TM_MODEL_TEST
    qDebug() << "DataModel test enabled.";
    new ModelTest(dataModel, this);
#endif
    sourceTree->setModel(dataModel);
    sourceTree->header()->setSectionResizeMode(0, QHeaderView::Stretch);
    sourceTree->header()->resizeSection(1, 100);
    sourceTree->setItemDelegateForColumn(0, sourceDelegate);
    sourceTree->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(sourceTree,
            SIGNAL(customContextMenuRequested(const QPoint &)),
            this,
            SLOT(sourceTreeCustomContextMenu(const QPoint &)));
    connect(dataModel,
            SIGNAL(connectionEstablished()),
            this,
            SLOT(connectDataSlots()));
    connect(dataModel,
            SIGNAL(connectionStateChanged(DataModel::ConnectionState)),
            this,
            SLOT(connectionStateChanged(DataModel::ConnectionState)));
    connect(dataModel, SIGNAL(statsChanged()), this, SLOT(statsChanged()));
    connectionStateChanged(DataModel::NoSources);

#ifdef TM_MODEL_TEST
    qDebug() << "PropertyModel test enabled.";
    new ModelTest(propertyModel, this);
#endif
    propertyTree->setModel(propertyModel);
    propertyTree->header()->resizeSection(0, 192);
    propertyTree->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(propertyTree,
            SIGNAL(customContextMenuRequested(const QPoint &)),
            this,
            SLOT(propertyTreeCustomContextMenu(const QPoint &)));

    connect(propertyModel,
            SIGNAL(rowsInserted(const QModelIndex &, int, int)),
            this,
            SLOT(expandProperties()));

    propertyTree->setItemDelegateForColumn(1, propertyDelegate);
    propertyTree->updateBrightness();

    tableViewScriptVariables->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(tableViewScriptVariables,
            SIGNAL(customContextMenuRequested(const QPoint &)),
            this,
            SLOT(scriptVariablesCustomContextMenu(const QPoint &)));

    ParameterItemDelegate *pathDelegate =
            new ParameterItemDelegate(tableViewParameters);
    parameterSplitter->setSizes({100, 700});

    tableViewParameters->setModel(defaultParameterModel);
    tableViewParameters->setContextMenuPolicy(Qt::CustomContextMenu);
    tableViewParameters->setItemDelegateForColumn(2, pathDelegate);
    tableViewParameters->horizontalHeader()->setStretchLastSection(true);
    connect(tableViewParameters,
            SIGNAL(customContextMenuRequested(const QPoint &)),
            this,
            SLOT(tableViewParametersCustomContextMenu(const QPoint &)));

    listViewParameterSets->setModel(parameterSetModel);
    listViewParameterSets->setResizeMode(QListView::Adjust);

    tableViewMessages->setModel(messageModelUnion);
    QHeaderView *hdr = tableViewMessages->horizontalHeader();
    hdr->setSectionResizeMode(0, QHeaderView::Stretch);
    hdr->setSectionResizeMode(1, QHeaderView::ResizeToContents);
    hdr->setSectionResizeMode(2, QHeaderView::ResizeToContents);
    hdr->setSectionResizeMode(3, QHeaderView::ResizeToContents);
    tableViewMessages->verticalHeader()->hide();
    connect(messageModelUnion,
            &QtPdCom::MessageModelUnion::currentMessage,
            this,
            &MainWindow::currentMessage);

    connect(listViewParameterSets,
            &QAbstractItemView::clicked,
            this,
            &MainWindow::changeParameterSet);
    connect(listViewParameterSets->selectionModel(),
            &QItemSelectionModel::selectionChanged,
            this,
            &MainWindow::updateRemoveSetButton);
    connect(tableViewParameters->selectionModel(),
            &QItemSelectionModel::selectionChanged,
            this,
            &MainWindow::updateParameterButtons);
    connect(parameterSetModel,
            &QAbstractItemModel::dataChanged,
            this,
            &MainWindow::changeParameterSet);
    connect(pushButtonRemove,
            SIGNAL(clicked()),
            this,
            SLOT(removeParameter()));
    connect(pushButtonClear,
            SIGNAL(clicked()),
            this,
            SLOT(clearParameters()));
    connect(pushButtonNewSet,
            SIGNAL(clicked()),
            this,
            SLOT(newParameterSet()));
    connect(pushButtonRemoveSet,
            SIGNAL(clicked()),
            this,
            SLOT(removeParameterSet()));
    connect(pushButtonSaveParameters,
            SIGNAL(clicked()),
            this,
            SLOT(saveParameters()));

    // fill status bar
    labelIn->setAlignment(Qt::AlignRight);
    statusBar()->addWidget(labelIn);
    labelOut->setAlignment(Qt::AlignRight);
    statusBar()->addWidget(labelOut);
    QFontMetrics fm(labelIn->font());
    auto num = QLocale().toString(10000.0, 'f', 1);
    auto minSize(fm.boundingRect(tr("%1 KB/s").arg(num)).size());
    labelIn->setMinimumSize(minSize);
    labelOut->setMinimumSize(minSize);
    statusMessageIcon->setMinimumSize(QSize(16, 16));
    statusBar()->addWidget(statusMessageIcon);
    statusBar()->addWidget(statusMessageText, 2);

    initPython();

    WidgetContainer::registerPythonType();
    ScriptVariable::registerPythonType();

    // fill list with available widgets
    Plugin *plugin;
    plugin = new BarPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new CheckBoxPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new RadioButtonPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new DialPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new DigitalPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new DoubleSpinBoxPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new GraphPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new LedPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new ParameterSetWidgetPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new PushButtonPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new QLabelPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new RotorPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new SendBroadcastPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new TableViewPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new TextPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new TouchEditPlugin();
    pluginMap->insert(plugin->type(), plugin);
    plugin = new XYGraphPlugin();
    pluginMap->insert(plugin->type(), plugin);

    newLayout();

    updateWindowTitle();

#if QT_VERSION >= 0x050600
    QList<QDockWidget *> docks;
    docks << dockWidgetSources << dockWidgetProperties;
    QList<int> sizes;
    sizes << 256 << 350;
    resizeDocks(docks, sizes, Qt::Horizontal);
#endif

    QSettings settings;
    restore = settings.value("restore", restore).toBool();
    recentFiles = settings.value("recentFiles").toStringList();
    restoreGeometry(settings.value("geometry").toByteArray());
    restoreState(settings.value("windowState").toByteArray());
    showHistoricMessages =
            settings.value("showHistoricMessages", showHistoricMessages)
                    .toBool();

    for (int i = 0; i < MaxRecentFiles; ++i) {
        QAction *action = new QAction(this);
        action->setVisible(false);
        connect(action, SIGNAL(triggered()), this, SLOT(openRecentFile()));
        menuRecentFiles->addAction(action);
        recentFileActions.push_back(action);
    }

    updateRecentFileActions();

    menuWindows->addAction(dockWidgetSources->toggleViewAction());
    menuWindows->addAction(dockWidgetProperties->toggleViewAction());
    menuWindows->addAction(dockWidgetShell->toggleViewAction());
    menuWindows->addAction(dockWidgetScript->toggleViewAction());
    menuWindows->addAction(dockWidgetParameters->toggleViewAction());
    menuWindows->addAction(dockWidgetBroadcasts->toggleViewAction());
    menuWindows->addAction(dockWidgetDataSlots->toggleViewAction());
    menuWindows->addAction(dockWidgetMessages->toggleViewAction());
    menuWindows->addAction(toolBar->toggleViewAction());

    broadcastDialog->setDataModel(*dataModel);
    slotViewFilter->setSourceModel(slotModelCollection);
    connect(tabWidget,
            &QTabWidget::currentChanged,
            slotViewFilter,
            &SlotViewFilter::widgetSelectionChanged);
    treeViewDataSlots->setModel(slotViewFilter);
    treeViewDataSlots->header()->setSectionResizeMode(
            0,
            QHeaderView::Stretch);
    treeViewDataSlots->header()->resizeSection(1, 120);

#if 0
    pythonBenchmark();
#endif

    QString fileToLoad;

    if (newView) {
        // pass
    }
    else if (!fileName.isEmpty()) {
        fileToLoad = fileName;
    }
    else if (restore && recentFiles.size() > 0) {
        fileToLoad = recentFiles.front();
    }

    if (!fileToLoad.isEmpty()) {
        loadLayout(fileToLoad);

        if (autoConnect) {
            dataModel->connectAll();
        }
    }

    updateEditMode();

    connect(tabWidget->tabBar(),
            SIGNAL(tabCloseRequested(int)),
            this,
            SLOT(tabCloseRequested(int)));

    QClipboard *clipboard(QApplication::clipboard());
    connect(clipboard,
            SIGNAL(changed(QClipboard::Mode)),
            this,
            SLOT(clipboardChanged(QClipboard::Mode)));

    QFont f(textEditScript->font());
    f.setStyleHint(QFont::Monospace);
    textEditScript->setFont(f);

    updateRemoveSetButton();
    updateSaveButton();

    treeViewDataSlots->setMainWindow(this);

    updateHistoricMessages();
    actionHistoricMessages->setChecked(showHistoricMessages);
}

/****************************************************************************/

MainWindow::~MainWindow()
{
    clearTabs();
    sourceTree->setModel(nullptr);
    dataModel->clear();

    tableViewParameters->setModel(nullptr);
    delete defaultParameterModel;

    listViewParameterSets->setModel(nullptr);
    delete parameterSetModel;

    tableViewMessages->setModel(nullptr);
    delete messageModelUnion;

    delete propertyDelegate;
    delete propertyModel;
    delete tabWidget;
    delete slotViewFilter;
    delete slotModelCollection;
    delete sourceDelegate;
    delete dataModel;

    auto map = std::move(*pluginMap);
    for (const auto &plugin : map) {
        delete plugin;
    }
    delete pluginMap;

    Py_Finalize();
}

/****************************************************************************/

PyObject *MainWindow::pythonMethodTabCount(PyObject *, PyObject *args)
{
    if (!PyArg_ParseTuple(args, ":tabCount")) {
        return NULL;
    }
    return PyLong_FromLong(singleton->tabPages().count());
}

/****************************************************************************/

PyObject *MainWindow::pythonMethodMaximize(PyObject *, PyObject *args)
{
    bool max(true);

    if (!PyArg_ParseTuple(args, "|b:maximize", &max)) {
        return NULL;
    }
    Qt::WindowStates states(singleton->windowState());
    if (max) {
        states |= Qt::WindowMaximized;
    }
    else {
        states &= ~Qt::WindowMaximized;
    }
    singleton->setWindowState(states);
    return Py_None;
}

/****************************************************************************/

QSet<WidgetContainer *> MainWindow::containers(Filter filter) const
{
    QList<WidgetContainer *> all;
    for (auto tabPage : tabPages()) {
        all += tabPage->findChildren<WidgetContainer *>();
    }

    if (filter == Selected || filter == SelectedOrCurrentTab) {
        QSet<WidgetContainer *> selectedContainers;
        for (WidgetContainer *container : all) {
            if (container->isSelected()) {
                selectedContainers.insert(container);
            }
        }
        if (filter == Selected || !selectedContainers.empty()) {
            return selectedContainers;
        }
        if (const auto w = tabWidget->currentWidget()) {
            all = w->findChildren<WidgetContainer *>();
        }
    }

#if QT_VERSION >= QT_VERSION_CHECK(5, 14, 0)
    return QSet<WidgetContainer *>(all.begin(), all.end());
#else
    return all.toSet();
#endif
}

/****************************************************************************/

QSet<WidgetContainer *> MainWindow::containers(const QUrl &url) const
{
    QSet<QUrl> urls;
    urls << url;
    return containers(urls);
}

/****************************************************************************/

QSet<WidgetContainer *> MainWindow::containers(const QSet<QUrl> &urls) const
{
    QList<WidgetContainer *> all;
    for (auto tabPage : tabPages()) {
        all += tabPage->findChildren<WidgetContainer *>();
    }

    QSet<WidgetContainer *> ret;
    for (WidgetContainer *container : all) {
        for (const QUrl &url : urls) {
            if (container->uses(url)) {
                ret.insert(container);
            }
        }
    }
    return ret;
}

/****************************************************************************/

void MainWindow::deselectAll() const
{
    foreach (WidgetContainer *container, containers(MainWindow::All)) {
        container->deselect();
    }
}

/****************************************************************************/

void MainWindow::selectionChanged()
{
    auto selected = propertyModel->getContainers();
    actionCopy->setEnabled(selected.size() > 0);
    slotViewFilter->widgetSelectionChanged();
}

/****************************************************************************/

QJsonArray MainWindow::checkPaste()
{
    QClipboard *clipboard = QGuiApplication::clipboard();
    const QMimeData *mimeData(clipboard->mimeData(QClipboard::Clipboard));

    if (not mimeData->hasText()) {
        return QJsonArray();
    }

    QString jsonStr(mimeData->text());
    QByteArray ba = jsonStr.toUtf8();

    QJsonParseError err;
    QJsonDocument doc(QJsonDocument::fromJson(ba, &err));
    if (err.error != QJsonParseError::NoError) {
        return QJsonArray();
    }

    return doc.array();
}

/****************************************************************************/

namespace {

QJsonObject updateBar(QJsonObject container)
{
    QJsonArray stacks;

    for (const auto var : container["slots"].toArray()) {
        QJsonArray stack;
        stack.append(var);
        stacks.append(stack);
    }
    QJsonObject v2slots;
    v2slots["stacks"] = stacks;

    container["slots"] = v2slots;
    return container;
}

QJsonObject updateDial(QJsonObject container)
{
    QJsonObject v2slots;
    const auto slotArray = container["slots"].toArray();
    if (slotArray.size() > 0) {
        v2slots["value"] = slotArray[0];
    }
    if (slotArray.size() > 1) {
        v2slots["setpoint"] = slotArray[1];
    }
    container["slots"] = v2slots;
    return container;
}

QJsonObject updateGraph(QJsonObject container)
{
    QJsonObject v2slots;
    v2slots["graphs"] = container["slots"].toArray();
    container["slots"] = v2slots;
    return container;
}

QJsonObject updateXYGraph(QJsonObject container)
{
    QJsonObject v2slots;
    const auto slotArray = container["slots"].toArray();
    if (slotArray.size() > 0) {
        QJsonArray a;
        a.append(slotArray[0]);
        v2slots["xAxis"] = a;
    }
    if (slotArray.size() > 1) {
        QJsonArray a;
        a.append(slotArray[1]);
        v2slots["yAxis"] = a;
    }
    container["slots"] = v2slots;
    return container;
}

QJsonObject updateTableView(QJsonObject container)
{
    const auto columns = container["private"].toObject()["columns"].toArray();
    const auto slotArray = container["slots"].toArray();

    QJsonArray columnArray;
    for (int i = 0; i < columns.size(); ++i) {
        auto column = columns[i].toObject();

        QJsonArray columnSlots;
        if (i < slotArray.size()) {
            columnSlots.append(slotArray[i]);
        }
        column["columns"] = columnSlots;
        columnArray.append(column);
    }
    QJsonObject slotObject;
    slotObject["columns"] = columnArray;
    container["slots"] = slotObject;
    return container;
}

QJsonObject updateContainer(QJsonObject container)
{
    const auto type = container["type"].toString();
    if (type == "QtPdWidgets::Bar") {
        return updateBar(container);
    }
    if (type == "QtPdWidgets::Dial") {
        return updateDial(container);
    }
    if (type == "QtPdWidgets::Graph") {
        return updateGraph(container);
    }
    if (type == "QtPdWidgets::XYGraph") {
        return updateXYGraph(container);
    }
    if (type == "QtPdWidgets2::TableView") {
        return updateTableView(container);
    }
    return container;
}

}  // namespace

/****************************************************************************/

QJsonArray MainWindow::updateV1TmlToV2(const QJsonArray &tabs)
{
    QJsonArray ans;
    for (const auto _tab : tabs) {
        auto tab = _tab.toObject();
        QJsonArray containers;
        for (const auto c : tab["containers"].toArray()) {
            containers.append(updateContainer(c.toObject()));
        }
        tab["containers"] = containers;
        ans.append(tab);
    }
    return ans;
}

/****************************************************************************/

void MainWindow::setLayoutChanged()
{
    layoutChanged = true;
    updateLayoutChanged();
}

/****************************************************************************/

void MainWindow::appendDetachedTab(
        DetachedDockWidget *dockWidget,
        Qt::DockWidgetArea area)
{
    detachedTabs.append(dockWidget);
    addDockWidget(area, dockWidget);

    connect(dockWidget,
            &QDockWidget::topLevelChanged,
            this,
            [this, dockWidget](bool) { setLayoutChanged(); });

    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::removeDetachedTab(DetachedDockWidget *dockWidget)
{
    detachedTabs.removeAll(dockWidget);
    removeDockWidget(dockWidget);
    setLayoutChanged();
}

/****************************************************************************/

QSet<TabPage *> MainWindow::tabPages() const
{
    auto list {tabWidget->findChildren<TabPage *>()};
    QSet<TabPage *> tabPages(list.begin(), list.end());
    for (auto dockWidget : detachedTabs) {
        auto list {dockWidget->findChildren<TabPage *>()};
        QSet<TabPage *> detachedTabPages(list.begin(), list.end());
        tabPages.unite(detachedTabPages);
    }
    return tabPages;
}

/*****************************************************************************
 * public slots
 ****************************************************************************/

void MainWindow::signalRaised(int signum)
{
    switch (signum) {
        case SIGINT:
        case SIGTERM:
            qDebug() << "Closing application.";
            close();
            break;
    }
}

/****************************************************************************/

void MainWindow::updateParameterButtons()
{
    updateRemoveButton();
    updateSaveButton();
}

/*****************************************************************************
 * private
 ****************************************************************************/

void MainWindow::clearTabs()
{
    for (int i = tabWidget->count(); i > 0; i--) {
        const auto w = tabWidget->widget(i - 1);
        tabWidget->removeTab(i - 1);
        delete w;
    }
    for (auto dockWidget : detachedTabs) {
        dockWidget->setAttachOnClose(false);
        dockWidget->close();
        delete dockWidget;
    }
    detachedTabs.clear();
}

/****************************************************************************/

static PyMethodDef pythonModuleMethods[] = {
        {
                "tabCount",                        // ml_name
                MainWindow::pythonMethodTabCount,  // ml_meth
                METH_VARARGS,                      // ml_flags
                "Return the number of tab pages."  // ml_doc
        },
        {
                "maximize",                        // ml_name
                MainWindow::pythonMethodMaximize,  // ml_meth
                METH_VARARGS,                      // ml_flags
                "Maximize the window."             // ml_doc
        },
        {NULL, NULL, 0, NULL}};

/****************************************************************************/

static PyModuleDef pythonModuleDef = {
        PyModuleDef_HEAD_INIT,  // m_base
        "tm",                   // m_name
        NULL,                   // m_doc
        -1,                     // m_size: global state
        pythonModuleMethods,    // methods
        NULL,                   // slots
        NULL,                   // m_traverse
        NULL,                   // m_clear
        NULL                    // m_free
};

/****************************************************************************/

static PyObject *initPythonModule(void)
{
    return PyModule_Create(&pythonModuleDef);
}

/****************************************************************************/

void MainWindow::initPython()
{
    qInfo() << __func__ << "Initializing Python environment.";

    if (PyImport_AppendInittab("tm", &initPythonModule)) {
        throw std::runtime_error("Failed to append python module.");
    }

#if PY_VERSION_HEX >= 0x03080000
    {
        qInfo() << __func__ << "Python version is >= 3.8" << Py_GetVersion();

        PyStatus status;

        PyPreConfig preconfig;
        PyPreConfig_InitPythonConfig(&preconfig);

        status = Py_PreInitialize(&preconfig);
        if (PyStatus_Exception(status)) {
            QString msg;
            QTextStream str(&msg);
            str << "Py_PreInitialize failed with error: "
                << (status.err_msg != nullptr ? status.err_msg : "Unknown");
            Py_ExitStatusException(status);
            throw std::runtime_error(msg.toLocal8Bit());
        }

        PyConfig config;
        PyConfig_InitPythonConfig(&config);

        config.verbose = 0;
        config.parse_argv = 1;
        config.site_import = 1;      // allow site.py to run
        config.use_environment = 1;  // allow PYTHONPATH and friends

        QString exePath = QCoreApplication::applicationFilePath();

#ifdef _WIN32
        QDir exeDir = QFileInfo(exePath).dir();
        QString pythonHome = exeDir.absolutePath() + "/..";
        pythonHome = QDir(pythonHome).absolutePath();
        std::wstring pythonHomeW = pythonHome.toStdWString();
        config.home = wcsdup(pythonHomeW.c_str());
        qInfo() << "Python home set to"
                << QString::fromWCharArray(config.home);
        QFileInfo homePath(QString::fromWCharArray(config.home));
        if (!homePath.exists()) {
            qWarning() << "Home path does not exist!";
        }

        wchar_t program_path[MAX_PATH];
        GetModuleFileNameW(NULL, program_path, MAX_PATH);
        config.program_name = wcsdup(program_path);
#else
        std::wstring programPathW = exePath.toStdWString();
        config.program_name = wcsdup(programPathW.c_str());
#endif

        qInfo() << "Python program_name set to"
                << QString::fromWCharArray(config.program_name);
        QFileInfo programFile(QString::fromWCharArray(config.program_name));
        if (!programFile.exists()) {
            qWarning() << "Program name path does not exist!";
        }

        // Initialize Python with config
        status = Py_InitializeFromConfig(&config);
        if (PyStatus_Exception(status)) {
            QString msg;
            QTextStream str(&msg);
            str << "Py_InitializeFromConfig failed with error: "
                << (status.err_msg != nullptr ? status.err_msg : "Unknown");
            Py_ExitStatusException(status);
            PyConfig_Clear(&config);
            throw std::runtime_error(msg.toLocal8Bit());
        }

        PyConfig_Clear(&config);
    }
#else
    qInfo() << __func__ << "Python version is < 3.8" << Py_GetVersion();

#ifdef _WIN32
    static const std::wstring pythonHome =
            (QCoreApplication::applicationDirPath() + "/../").toStdWString();
    Py_SetPythonHome(pythonHome.data());
#endif

    // argument must be a wide-character string with static storage
    // older Pythons take the version without const, so avoid a warning
    Py_SetProgramName(L"testmanager");  // Deprecated in 3.11
    Py_Initialize();
#endif

    PyObject *sys = PyImport_ImportModule("sys");
    if (!sys) {
        throw std::runtime_error("Failed to import sys module.");
    }

    // ----- redirect stdout -----

#ifndef _WIN32
    if (::socketpair(AF_UNIX, SOCK_STREAM, 0, pythonStdOut)) {
#else
    if (_pipe(pythonStdOut, 512, _O_BINARY)) {
#endif
        QString msg;
        QTextStream str(&msg);
        str << "Could not create stdout socket pair:" << strerror(errno);
        throw std::runtime_error(msg.toLocal8Bit());
    }

    pythonStdOutNotifier =
            new QSocketNotifier(pythonStdOut[1], QSocketNotifier::Read, this);
    connect(pythonStdOutNotifier,
            SIGNAL(activated(int)),
            this,
            SLOT(pythonStdOutDataAvailable(int)));

    PyObject *out =
            PyFile_FromFd(pythonStdOut[0], NULL, "w", 1, NULL, NULL, NULL, 1);
    if (!out) {
        throw std::runtime_error(
                "Failed to create python file from stdout socket.");
    }

    if (PyObject_SetAttrString(sys, "stdout", out)) {
        throw std::runtime_error("Failed to replace python stdout.");
    }

    // ----- redirect stderr -----

#ifndef _WIN32
    if (::socketpair(AF_UNIX, SOCK_STREAM, 0, pythonStdErr)) {
#else
    if (_pipe(pythonStdErr, 512, _O_BINARY)) {
#endif
        QString msg;
        QTextStream str(&msg);
        str << "Couldn't create stderr socket pair:" << strerror(errno);
        throw std::runtime_error(msg.toLocal8Bit());
    }

    pythonStdErrNotifier =
            new QSocketNotifier(pythonStdErr[1], QSocketNotifier::Read, this);
    connect(pythonStdErrNotifier,
            SIGNAL(activated(int)),
            this,
            SLOT(pythonStdErrDataAvailable(int)));

    PyObject *err =
            PyFile_FromFd(pythonStdErr[0], NULL, "w", 1, NULL, NULL, NULL, 1);
    if (!err) {
        throw std::runtime_error(
                "Failed to create python file from stderr socket.");
    }

    if (PyObject_SetAttrString(sys, "stderr", err)) {
        throw std::runtime_error("Failed to replace python stderr.");
    }

    QString banner(QString("Python %1 on %2\n")
                           .arg(Py_GetVersion())
                           .arg(Py_GetPlatform()));
    textEditPython->setTextColor(color[0]);
    textEditPython->append(banner);

#if 0
    ::write(pythonStdOut[0], "stdout\n", 7);
    ::write(pythonStdErr[0], "stderr\n", 7);
#endif

    qInfo() << __func__ << "Python initialized.";
}

/****************************************************************************/

void MainWindow::newLayout()
{
    filePath = QString();
    loadedLayoutIsOldVersion = false;
    QDir::setSearchPaths("layout", QStringList());
    updateWindowTitle();

    parameterSetModel->clear();
    tableViewParameters->setModel(defaultParameterModel);
    updateRemoveSetButton();
    updateRemoveButton();
    updateSaveButton();

    clearTabs();
    tabWidget->setStyleSheet(QString());

    on_actionAddTab_triggered();

    layoutChanged = false;
    updateLayoutChanged();
}

/****************************************************************************/

void MainWindow::openLoadDialog()
{
    auto *dialog = new QFileDialog(this);
    dialog->setAcceptMode(QFileDialog::AcceptOpen);

    QStringList filters;
    filters << tr("Testmanager Layouts (*.tml)") << tr("Any files (*)");

    dialog->setNameFilters(filters);
    dialog->setDefaultSuffix("tml");

    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    connect(dialog, &QDialog::accepted, this, [this, dialog]() {
        QString path = dialog->selectedFiles()[0];
        loadLayout(path);
    });

    dialog->open();
}

/****************************************************************************/

void MainWindow::loadLayout(const QString &path)
{
    QFile file(path);
    if (!file.open(QIODevice::ReadOnly)) {
        QString msg(tr("Failed to open %1.").arg(path));
        statusBar()->showMessage(msg, 5000);
        qWarning() << msg;
        return;
    }

    QByteArray ba = file.readAll();
    file.close();

    QJsonParseError err;
    QJsonDocument doc(QJsonDocument::fromJson(ba, &err));

    if (err.error != QJsonParseError::NoError) {
        qCritical() << "Layout parsing error (" << err.error << ") at offset "
                    << err.offset << ": " << err.errorString();
        return;
    }

    filePath = path;
    QFileInfo fi(filePath);
    QString searchPath(fi.absoluteDir().path());
    QDir::setSearchPaths("layout", QStringList(searchPath));

    QJsonObject layoutObject(doc.object());

    QJsonArray tabArray(layoutObject["tabs"].toArray());

    /* Layout file version history:
     *
     * 1: Initial JSON format
     * 2: Changed slot storage of Bar, Dial, Graph, XYGraph and TableView
     *    (see updateV1TmlToV2).
     * 3: Additional fields (detached, ...) in tab structure. Increased file
     *    version, because older software versions could load tabs, but would
     *    omit new fields on save.
     */
    int version = layoutObject["version"].toInt();

    /* Update old layout file version step-by-step. */
    if (version == 1) {
        tabArray = updateV1TmlToV2(tabArray);
        layoutObject["tabs"] = tabArray;
        loadedLayoutIsOldVersion = true;
        version = 2;
    }
    if (version == 2) {
        // no changes, just additional fields in tab (see above).
        loadedLayoutIsOldVersion = true;
        version = 3;
    }
    if (version != 3) {
        auto mb = new QMessageBox(
                QMessageBox::Critical,
                tr("Import error"),
                tr("Layout File version %1 is not implemented.").arg(version),
                QMessageBox::NoButton,
                this);
        connect(mb, &QDialog::finished, mb, &QObject::deleteLater);
        mb->open();
        return;
    }

    clearTabs();

    foreach (QJsonValue tabValue, tabArray) {
        QJsonObject tabObject(tabValue.toObject());
        QString name(tabObject["name"].toString());
        appendTab(name, tabObject);
    }

    QString styleSheet;
    if (layoutObject.contains("styleSheet")) {
        styleSheet = layoutObject["styleSheet"].toString();
    }
    tabWidget->setStyleSheet(styleSheet);
    for (auto dockWidget : detachedTabs) {
        dockWidget->setStyleSheet(styleSheet);
    }

    QJsonArray dataSourceArray(layoutObject["dataSources"].toArray());
    dataModel->read(dataSourceArray);

    parameterSetModel->clear();
    if (layoutObject.contains("parameterSets")) {
        parameterSetModel->fromJson(layoutObject["parameterSets"].toArray());
        parameterSetModel->appendDataSources(dataModel);

        // select first parameter set, if existing
        if (parameterSetModel->rowCount()) {
            QModelIndex index {parameterSetModel->index(0)};
            listViewParameterSets->setCurrentIndex(index);
            tableViewParameters->setModel(
                    parameterSetModel->getParameterSet(index)
                            ->getParameterModel());
        }
    }

    updateWindowTitle();
    addRecentFile(path);

    layoutChanged = false;
    updateLayoutChanged();
}

/****************************************************************************/

void MainWindow::saveLayout()
{
    QJsonArray tabArray;

    for (int i = 0; i < tabWidget->count(); i++) {
        QScrollArea *scrollArea =
                static_cast<QScrollArea *>(tabWidget->widget(i));
        TabPage *tabPage = static_cast<TabPage *>(scrollArea->widget());
        QJsonObject tabObject;
        tabObject["name"] = tabWidget->tabText(i);
        tabObject["detached"] = false;
        tabPage->write(tabObject);
        tabArray.append(tabObject);
    }
    for (auto dockWidget : detachedTabs) {
        for (auto tabPage : dockWidget->findChildren<TabPage *>()) {
            QJsonObject tabObject;
            tabObject["name"] = tabPage->getName();
            tabObject["detached"] = true;
            tabObject["floating"] = dockWidget->isFloating();
            if (dockWidget->isFloating()) {
                tabObject["x"] = dockWidget->geometry().x();
                tabObject["y"] = dockWidget->geometry().y();
                tabObject["width"] = dockWidget->geometry().width();
                tabObject["height"] = dockWidget->geometry().height();
            }
            else {
                tabObject["dockArea"] = dockWidgetArea(dockWidget);
            }
            tabPage->write(tabObject);
            tabArray.append(tabObject);
        }
    }

    QJsonObject layoutObject;
    layoutObject["version"] = 3;
    layoutObject["tabs"] = tabArray;

    if (not tabWidget->styleSheet().isEmpty()) {
        layoutObject["styleSheet"] = tabWidget->styleSheet();
    }

    QJsonArray dataSourceArray;
    dataModel->write(dataSourceArray);
    if (!dataSourceArray.isEmpty()) {
        layoutObject["dataSources"] = dataSourceArray;
    }

    QJsonArray parameterSetArray;
    if (parameterSetModel) {
        parameterSetArray = parameterSetModel->toJson();
    }
    if (not parameterSetArray.isEmpty()) {
        layoutObject["parameterSets"] = parameterSetArray;
    }

    QJsonDocument saveDoc(layoutObject);

    QFile saveFile(filePath);
    if (!saveFile.open(QIODevice::WriteOnly)) {
        statusBar()->showMessage(tr("Failed to open %1").arg(filePath), 2000);
        afterSaveFunction = [] {};
        return;
    }

    saveFile.write(saveDoc.toJson());
    addRecentFile(filePath);

    layoutChanged = false;
    updateLayoutChanged();

    afterSaveFunction();
    afterSaveFunction = [] {};
}

/****************************************************************************/

void MainWindow::updateWindowTitle()
{
    if (filePath.isEmpty()) {
        setWindowFilePath("");
        setWindowTitle(QCoreApplication::applicationName());
    }
    else {
        setWindowTitle("");
        setWindowFilePath(filePath);
    }
}

/****************************************************************************/

void MainWindow::pythonBenchmark()
{
    // add local directory to python path
    PyObject *sysPath = PySys_GetObject("path");
    PyObject *dirString = PyUnicode_FromString(".");
    PyList_Append(sysPath, dirString);

    PyObject *module = PyImport_ImportModule("mult");
    if (!module) {
        PyErr_Print();
        qWarning() << "Failed to import module.";
        return;
    }

    PyObject *func = PyObject_GetAttrString(module, "multiply");
    if (!func || !PyCallable_Check(func)) {
        if (PyErr_Occurred()) {
            PyErr_Print();
        }
        qWarning() << "Failed to find func.";
        Py_DECREF(module);
        return;
    }

    PyObject *args = PyTuple_New(2);
    for (int i = 0; i < 2; ++i) {
        PyObject *value = PyLong_FromLong(i + 3);
        if (!value) {
            Py_DECREF(args);
            Py_DECREF(func);
            Py_DECREF(module);
            qWarning() << "Cannot convert argument.";
            return;
        }
        /* pValue reference
         * stolen here: */
        PyTuple_SetItem(args, i, value);
        Py_DECREF(value);
    }


    PyObject *call = NULL;

    QElapsedTimer timer;
    timer.start();

    for (int i = 0; i < 1000000; i++) {
        if (call) {
            Py_DECREF(call);
        }
        call = PyObject_CallObject(func, args);
    }

    qDebug() << "The slow operation took" << timer.elapsed()
             << "milliseconds";

    Py_DECREF(args);

    if (call) {
        qDebug() << "Result of call:" << PyLong_AsLong(call);
        Py_DECREF(call);
    }
    else {
        Py_DECREF(func);
        Py_DECREF(module);
        PyErr_Print();
        qWarning() << "Call failed";
        return;
    }

    Py_DECREF(func);
    Py_DECREF(module);

    timer.start();

    int (*ref)(int, int) = &multiply;

    for (int i = 0; i < 1000000; i++) {
        int res = ref(3, 4);
        res = res;
    }

    qDebug() << "The fast operation took" << timer.elapsed()
             << "milliseconds";
}

/********************** HACK: QTBUG-16507 workaround ************************/

void MainWindow::showEvent(QShowEvent *event)
{
    QMainWindow::showEvent(event);
    QString filePath = windowFilePath();
    if (!filePath.isEmpty()) {
        setWindowFilePath(filePath + "x");
        setWindowFilePath(filePath);
    }
}

/****************************************************************************/

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (closeWithoutAsking) {
        QSettings settings;

        // remove dock widgets, otherwise Qt would try to store their
        // positions in saveGeometry(). Their positions are saved in the
        // layout file instead.
        for (auto dockWidget : detachedTabs) {
            removeDockWidget(dockWidget);
            dockWidget->setAttachOnClose(false);
            dockWidget->close();
        }

        settings.setValue("restore", restore);
        settings.setValue("recentFiles", recentFiles);
        settings.setValue("geometry", saveGeometry());
        settings.setValue("windowState", saveState());
        settings.setValue("showHistoricMessages", showHistoricMessages);
        qDebug() << "Storing settings in" << settings.fileName();

        event->accept();
    }
    else {
        event->ignore();  // will be done later
        askToSaveChanges([this]() {
            closeWithoutAsking = true;
            close();
        });
    }
}

/****************************************************************************/

void MainWindow::addRecentFile(const QString &path)
{
    QDir cur = QDir::currentPath();
    QString absPath = cur.absoluteFilePath(path);
    absPath = QDir::cleanPath(absPath);
    recentFiles.removeAll(absPath);
    recentFiles.prepend(absPath);
    updateRecentFileActions();
}

/****************************************************************************/

void MainWindow::updateRecentFileActions()
{
    int numRecentFiles = qMin(recentFiles.size(), (int) MaxRecentFiles);

    for (int i = 0; i < numRecentFiles; ++i) {
        QString text = QString("&%1 %2").arg(i + 1).arg(recentFiles[i]);
        recentFileActions[i]->setText(text);
        recentFileActions[i]->setData(recentFiles[i]);
        recentFileActions[i]->setVisible(true);
    }

    for (int j = numRecentFiles; j < MaxRecentFiles; ++j) {
        recentFileActions[j]->setVisible(false);
    }

    menuRecentFiles->setEnabled(numRecentFiles > 0);
}

/****************************************************************************/

void MainWindow::expandChildren(
        const QModelIndex &index,
        QTreeView *view,
        int depth)
{
    if (!depth or !view->model()) {
        return;
    }

    if (!view->isExpanded(index)) {
        view->expand(index);
    }

    if (depth > 0) {
        depth--;
    }

    int rowCount = 0;

    rowCount = view->model()->rowCount(index);

    for (int i = 0; i < rowCount; i++) {
        const QModelIndex &child = view->model()->index(i, 0, index);
        expandChildren(child, view, depth);
    }
}

/****************************************************************************/

void MainWindow::updateEditMode()
{
    dockWidgetProperties->setVisible(editMode);
    tabWidget->setTabsClosable(editMode);
    tabWidget->setMovable(editMode);
    actionAddTab->setEnabled(editMode);

    for (auto tabPage : tabPages()) {
        tabPage->editModeChanged();
    }

    for (auto dockWidget : detachedTabs) {
        dockWidget->editModeChanged();
    }

    updatePaste();
}

/****************************************************************************/

void MainWindow::updatePaste()
{
    QJsonArray array = checkPaste();
    actionPaste->setEnabled(
            array.size() > 0 && editMode && tabWidget->currentIndex() != -1);
    actionPaste->setText(tr("Paste %Ln widgets", "", array.size()));
}

/****************************************************************************/

void MainWindow::replaceUrl(const QUrl &oldUrl, const QUrl &newUrl)
{
    for (auto tabPage : tabPages()) {
        tabPage->replaceUrl(oldUrl, newUrl);
    }

    parameterSetModel->replaceUrl(oldUrl, newUrl);
}

/****************************************************************************/

void MainWindow::appendTab(const QString &name, const QJsonObject &config)
{
    const bool detached {config["detached"].toBool()};

    QWidget *scrollParent {this};
    DetachedDockWidget *dockWidget {nullptr};

    if (detached) {
        dockWidget = new DetachedDockWidget(this);
        scrollParent = dockWidget;
    }

    QScrollArea *scrollArea(new QScrollArea(scrollParent));
    TabPage *tabPage = new TabPage(this, scrollArea);
    scrollArea->setWidget(tabPage);
    scrollArea->setWidgetResizable(true);
    tabPage->setName(name);
    tabPage->read(config);

    if (detached) {
        Qt::DockWidgetArea area = Qt::RightDockWidgetArea;

        dockWidget->setWidget(scrollArea);

        const bool floating {config["floating"].toBool()};
        dockWidget->setFloating(floating);
        if (floating) {
            dockWidget->setGeometry(
                    config["x"].toInt(),
                    config["y"].toInt(),
                    config["width"].toInt(),
                    config["height"].toInt());
        }
        else {
            area = (Qt::DockWidgetArea) config["dockArea"].toInt();
        }

        appendDetachedTab(dockWidget, area);
        dockWidget->setWindowTitle(name);
        dockWidget->show();
    }
    else {
        tabWidget->addTab(scrollArea, name);
    }
}

/****************************************************************************/

QSet<WidgetContainer *> MainWindow::usingSelected() const
{
    QSet<QUrl> urls;

    foreach (
            QModelIndex index,
            sourceTree->selectionModel()->selectedIndexes()) {
        auto dataNode(dataModel->getDataNode(index));
        if (dataNode) {
            urls.insert(dataNode->nodeUrl());
        }
    }

    return containers(urls);
}

/****************************************************************************/

void MainWindow::askToSaveChanges(std::function<void()> proceed)
{
    if (not layoutChanged) {
        QTimer::singleShot(0, this, proceed);
        return;
    }

    auto mb = new QMessageBox(
            QMessageBox::Question,
            tr("Save changes?"),
            tr("Layout has been changed. Should these changes be saved?"),
            QMessageBox::Save | QMessageBox::Discard | QMessageBox::Abort,
            this);
    connect(mb, &QDialog::finished, mb, &QObject::deleteLater);

    QObject::connect(
            mb,
            &QMessageBox::finished,
            this,
            [this, mb, proceed](int) {
                QMessageBox::StandardButton result =
                        mb->standardButton(mb->clickedButton());

                switch (result) {
                    case QMessageBox::Save:
                        afterSaveFunction = proceed;
                        on_actionSave_triggered();
                        break;

                    case QMessageBox::Discard:
                        QTimer::singleShot(0, this, proceed);
                        break;

                    case QMessageBox::Cancel:
                    default:
                        break;
                }
            });

    mb->open();
}

/****************************************************************************/

void MainWindow::updateLayoutChanged()
{
    actionSave->setEnabled(layoutChanged);
}

/****************************************************************************/

void MainWindow::updateHistoricMessages()
{
    dataModel->showHistoricMessages(showHistoricMessages);

    const int column {QtPdCom::MessageModelUnion::TimeResetColumn};
    if (showHistoricMessages) {
        tableViewMessages->showColumn(column);
    }
    else {
        tableViewMessages->hideColumn(column);
    }
}

/*****************************************************************************
 * private slots
 ****************************************************************************/

void MainWindow::on_actionNew_triggered()
{
    askToSaveChanges([this]() { this->newLayout(); });
}

/****************************************************************************/

void MainWindow::on_actionLoad_triggered()
{
    askToSaveChanges([this]() { this->openLoadDialog(); });
}

/****************************************************************************/

void MainWindow::on_actionLoadParameterFile_triggered()
{
    ParameterTableModel *model = new ParameterTableModel(this);
    model->loadParameters();
    delete model;
}

/****************************************************************************/

void MainWindow::openRecentFile()
{
    QAction *action = qobject_cast<QAction *>(sender());
    if (!action) {
        return;
    }

    QString path = action->data().toString();
    askToSaveChanges([this, path]() { loadLayout(path); });
}

/****************************************************************************/

void MainWindow::on_actionSave_triggered()
{
    if (filePath.isEmpty()) {
        on_actionSaveAs_triggered();
        return;
    }

    if (not loadedLayoutIsOldVersion) {
        saveLayout();
        return;
    }

    auto mb = new QMessageBox(
            QMessageBox::Warning,
            tr("Layout File format update"),
            tr("The Layout file format will be updated to a newer "
               "Testmanager version. The file won't be readable with "
               "older Testmanager versions."),
            QMessageBox::Ok | QMessageBox::Cancel,
            this);
    connect(mb, &QDialog::finished, mb, &QObject::deleteLater);
    connect(mb, &QDialog::accepted, this, [this]() {
        loadedLayoutIsOldVersion = false;
        saveLayout();
    });
    mb->open();
}

/****************************************************************************/

void MainWindow::on_actionSaveAs_triggered()
{
    auto *dialog = new QFileDialog(this);
    dialog->setAcceptMode(QFileDialog::AcceptSave);

    QStringList filters;
    filters << tr("Testmanager Layouts (*.tml)") << tr("Any files (*)");

    dialog->setNameFilters(filters);
    dialog->setDefaultSuffix("tml");

    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    connect(dialog, &QDialog::accepted, this, [this, dialog]() {
        filePath = dialog->selectedFiles()[0];
        QFileInfo fi(filePath);
        QString searchPath(fi.absoluteDir().path());
        QDir::setSearchPaths("layout", QStringList(searchPath));
        updateWindowTitle();

        saveLayout();
    });

    dialog->open();
}

/****************************************************************************/

void MainWindow::on_actionClose_triggered()
{
    close();
}

/****************************************************************************/

void MainWindow::on_actionCopy_triggered()
{
    auto selected(propertyModel->getContainers());
    QJsonArray array;
    foreach (WidgetContainer *container, selected) {
        QJsonObject obj;
        container->write(obj);
        array.append(obj);
    }

    QJsonDocument doc(array);
    QByteArray json = doc.toJson(QJsonDocument::Indented);
    QString jsonStr = QString::fromUtf8(json);
    QClipboard *clipboard = QGuiApplication::clipboard();
    clipboard->setText(jsonStr);
}

/****************************************************************************/

void MainWindow::clipboardChanged(QClipboard::Mode mode)
{
    if (mode != QClipboard::Clipboard) {
        return;
    }

    updatePaste();
}

/****************************************************************************/

void MainWindow::on_actionPaste_triggered()
{
    QJsonArray array = checkPaste();
    if (array.size() == 0) {
        return;
    }

    int tabIdx = tabWidget->currentIndex();
    if (tabIdx == -1) {
        return;
    }

    QScrollArea *scrollArea =
            static_cast<QScrollArea *>(tabWidget->widget(tabIdx));
    TabPage *tabPage = static_cast<TabPage *>(scrollArea->widget());

    deselectAll();

    tabPage->addContainerArray(array, true /* select */);
    setLayoutChanged();

    if (not editMode) {
        editMode = true;
        updateEditMode();
    }
}

/****************************************************************************/

void MainWindow::on_actionConnectAll_triggered()
{
    dataModel->connectAll();
}

/****************************************************************************/

void MainWindow::on_actionConnect_triggered()
{
    auto *dialog = new ConnectDialog(dataModel, this);
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);

    connect(dialog, &QDialog::accepted, this, [this, dialog] {
        DataSource *source = dialog->adoptSource();
        dataModel->append(source);
        connectDataSlots();
    });

    dialog->open();
}

/****************************************************************************/

void MainWindow::on_actionEditMode_toggled(bool checked)
{
    editMode = checked;
    updateEditMode();
}

/****************************************************************************/

void MainWindow::on_actionGlobalStyleSheet_triggered()
{
    auto *dialog = new StyleDialog(tabWidget->styleSheet(), this);
    connect(dialog, &StyleDialog::styleChanged, this, [this](QString style) {
        tabWidget->setStyleSheet(style);
        for (auto dockWidget : detachedTabs) {
            dockWidget->setStyleSheet(style);
        }
        setLayoutChanged();
    });
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    dialog->open();
}

/****************************************************************************/

void MainWindow::on_actionHistoricMessages_toggled(bool checked)
{
    showHistoricMessages = checked;
    updateHistoricMessages();
}

/****************************************************************************/

void MainWindow::on_actionAddTab_triggered()
{
    appendTab(tr("New Tab"));
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::on_actionAboutTestManager_triggered()
{
    auto *dialog = new AboutDialog(this);
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    dialog->open();
}

/****************************************************************************/

void MainWindow::on_lineEditFilter_textChanged(const QString &text)
{
    if (text.isEmpty()) {
        dataModel->filter(QRegExp());
        return;
    }

    QRegExp re(text, Qt::CaseInsensitive);

    if (re.isValid()) {
        lineEditFilter->setStyleSheet(QString());
        lineEditFilter->setToolTip(QString());
        dataModel->filter(re);
    }
    else {
        lineEditFilter->setStyleSheet("color: red;");
        lineEditFilter->setToolTip(re.errorString());
    }
}

/****************************************************************************/

void MainWindow::on_toolButtonClearFilter_clicked()
{
    lineEditFilter->setText(QString());
}

/****************************************************************************/

void MainWindow::sourceTreeCustomContextMenu(const QPoint &point)
{
    std::unique_ptr<QMenu> menu(new QMenu());
    dataIndex = sourceTree->indexAt(point);
    int rows(dataModel->rowCount(dataIndex));
    DataSource *dataSource(dataModel->getDataSource(dataIndex));

    QAction *action;
    if (dataSource) {
        action = menu->addAction(
                tr("Connect"),
                this,
                SLOT(connectDataSource()));
        action->setEnabled(!dataSource->isConnected());
        action->setIcon(QIcon(":/images/network-transmit-receive.svg"));

        action = menu->addAction(
                tr("Disconnect"),
                this,
                SLOT(disconnectDataSource()));
        action->setIcon(QIcon(":/images/network-offline.svg"));
        action->setEnabled(dataSource->isConnected());

        action = menu->addAction(
                tr("Login"),
                this,
                &MainWindow::loginDataSource);
        action->setIcon(QIcon(":/images/login.svg"));
        action->setEnabled(
                dataSource->isConnected()
                && !dataSource->lastLoginSuccessful());

        action = menu->addAction(
                tr("Logout"),
                this,
                &MainWindow::logoutDataSource);
        action->setIcon(QIcon(":/images/logout.svg"));
        action->setEnabled(
                dataSource->isConnected()
                && dataSource->lastLoginSuccessful());

        action =
                menu->addAction(tr("Remove"), this, SLOT(removeDataSource()));
        action->setIcon(QIcon(":/images/list-remove.svg"));

        action = menu->addAction(
                tr("Replace..."),
                this,
                SLOT(replaceDataSource()));
        action->setIcon(QIcon(":/images/replace-datasource.svg"));

        menu->addSeparator();  // -------------------------------

        action = menu->addAction(
                tr("Show messages..."),
                this,
                SLOT(dataSourceShowMessages()));
        action->setIcon(QIcon(":/images/messages.svg"));

        action = menu->addAction(
                tr("Configure messages..."),
                this,
                SLOT(dataSourceConfigMessages()));
        action->setIcon(QIcon(":/images/messages.svg"));

        menu->addSeparator();  // -------------------------------
    }

    action = menu->addAction(
            tr("Add datasource..."),
            this,
            SLOT(on_actionConnect_triggered()));
    action->setIcon(QIcon(":/images/list-add.svg"));

    if (dataIndex.isValid() and not dataSource) {
        menu->addSeparator();  // -------------------------------
        action = menu->addAction(
                tr("Copy variable URL"),
                this,
                SLOT(copyVariableUrl()));
        action->setIcon(QIcon(":/images/edit-copy.svg"));

        auto containers(usingSelected());
        action = menu->addAction(
                tr("Select %Ln subscriber(s)", "", containers.size()),
                this,
                SLOT(selectSubscribers()));
        action->setEnabled(containers.size() > 0);
    }

    if (dataIndex.isValid()) {
        menu->addSeparator();  // -------------------------------
        action = menu->addAction(
                tr("Expand complete subtree"),
                this,
                SLOT(expandData()));
        action->setEnabled(rows > 0);
        action->setIcon(QIcon(":/images/view-fullscreen.svg"));
    }

    menu->exec(sourceTree->viewport()->mapToGlobal(point));
}

/****************************************************************************/

void MainWindow::connectDataSource()
{
    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    if (!dataSource->isConnected()) {
        ensureLoginDialog(dataSource);
        dataSource->connectToHost();
    }
    parameterSetModel->connectSets();
}

/****************************************************************************/

void MainWindow::ensureLoginDialog(DataSource *dataSource)
{
    using QtPdCom::LoginManager;

    auto manager = dataSource->getLoginManager();
    if (dataSource->credentialsNeededAction) {
        QObject::disconnect(dataSource->credentialsNeededAction);
    }
    dataSource->credentialsNeededAction =
            connect(manager,
                    &LoginManager::needCredentials,
                    this,
                    [this, manager, dataSource]() {
                        auto dialog = new LoginDialog(*manager, this);
                        connect(dialog,
                                &QDialog::finished,
                                dialog,
                                &QObject::deleteLater);
                        // abort connect to host in case user cancels dialog
                        connect(dialog,
                                &QDialog::rejected,
                                dataSource,
                                &DataSource::disconnectFromHost);
                        connect(manager,
                                &LoginManager::loginFailed,
                                this,
                                &MainWindow::onLoginFailed,
                                Qt::UniqueConnection);
                        dialog->setModal(true);
                        dialog->open();
                    });
}

/****************************************************************************/

void MainWindow::disconnectDataSource()
{
    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    if (dataSource->isConnected()) {
        dataSource->disconnectFromHost();
        dataSource->getLoginManager()->clearCredentials();
    }
}
/****************************************************************************/

void MainWindow::loginDataSource()
{
    using QtPdCom::LoginManager;

    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    if (dataSource->isConnected()) {
        auto dialog = new LoginDialog(*dataSource->getLoginManager(), this);
        dialog->setModal(true);
        auto manager = dataSource->getLoginManager();
        if (dataSource->credentialsNeededAction) {
            QObject::disconnect(dataSource->credentialsNeededAction);
        }
        dataSource->credentialsNeededAction =
                connect(manager,
                        &LoginManager::needCredentials,
                        dialog,
                        &QDialog::show);
        connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
        connect(manager,
                &LoginManager::loginFailed,
                this,
                &MainWindow::onLoginFailed,
                Qt::UniqueConnection);
        manager->login();
    }
}

/****************************************************************************/

void MainWindow::logoutDataSource()
{
    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    dataSource->logout();
}

/****************************************************************************/

void MainWindow::removeDataSource()
{
    dataModel->remove(dataIndex);
}

/****************************************************************************/

void MainWindow::onLoginFailed()
{
    auto *loginManager = qobject_cast<QtPdCom::LoginManager *>(sender());
    QString errorMessage(tr("Unknown error"));
    if (loginManager) {
        errorMessage = loginManager->getErrorMessage();
    }

#if QT_VERSION >= QT_VERSION_CHECK(6, 2, 0)
    auto *mb = new QMessageBox(
            QMessageBox::Critical,
            tr("Login failed"),
            errorMessage,
            QMessageBox::Ok,
            this);
#else
    auto *mb = new QMessageBox(
            tr("Login failed"),
            errorMessage,
            QMessageBox::Critical,
            QMessageBox::Ok,
            QMessageBox::NoButton,
            QMessageBox::NoButton,
            this);
#endif
    connect(mb, &QDialog::finished, mb, &QObject::deleteLater);
    mb->open();
}

/****************************************************************************/

void MainWindow::replaceDataSource()
{
    DataSource *oldSource(dataModel->getDataSource(dataIndex));
    if (not oldSource) {
        return;
    }

    auto *dialog = new ReplaceDialog(dataModel, oldSource, this);
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    connect(dialog, &QDialog::accepted, this, [this, oldSource, dialog] {
        DataSource *newSource = dialog->adoptSource();

        if (dialog->getMode() == ReplaceDialog::Permanent) {
            newSource->setUrl(newSource->getConnectUrl());
            replaceUrl(oldSource->getUrl(), newSource->getUrl());
            setLayoutChanged();
        }
        // oldSource invalid now!
        dataModel->replace(oldSource, newSource);

        connectDataSlots();
    });

    dialog->open();
}

/****************************************************************************/

void MainWindow::dataSourceShowMessages()
{
    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    dataSource->showMessages(this);
}

/****************************************************************************/

void MainWindow::dataSourceConfigMessages()
{
    DataSource *dataSource(dataModel->getDataSource(dataIndex));
    if (!dataSource) {
        return;
    }

    auto *dialog = new MessageDialog(dataSource, this);
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    dialog->open();

    setLayoutChanged();  // FIXME only on changed
}

/****************************************************************************/

void MainWindow::copyVariableUrl()
{
    QSet<QUrl> seen;
    QList<QUrl> urls;
    QString plainUrls;

    foreach (
            QModelIndex index,
            sourceTree->selectionModel()->selectedIndexes()) {
        DataNode *dataNode(dataModel->getDataNode(index));
        if (dataNode) {
            QUrl url(dataNode->nodeUrl());
            if (not seen.contains(url)) {
                urls.append(url);
                seen.insert(url);
                if (plainUrls.isEmpty()) {
                    plainUrls += "\n";
                }
                plainUrls += url.toString();
            }
        }
    }

    QClipboard *clipboard = QGuiApplication::clipboard();
    QMimeData *mimeData;
    QByteArray encoded = plainUrls.toUtf8();

    mimeData = new QMimeData();
    mimeData->setText(plainUrls);
    mimeData->setData("text/plain;charset=utf-8", encoded);
    mimeData->setUrls(urls);
    clipboard->setMimeData(mimeData, QClipboard::Clipboard);

    mimeData = new QMimeData();
    mimeData->setText(plainUrls);
    mimeData->setData("text/plain;charset=utf-8", encoded);
    mimeData->setUrls(urls);
    clipboard->setMimeData(mimeData, QClipboard::Selection);
}

/****************************************************************************/

void MainWindow::selectSubscribers()
{
    deselectAll();
    foreach (WidgetContainer *container, usingSelected()) {
        container->select();
    }
}

/****************************************************************************/

void MainWindow::expandData()
{
    expandChildren(dataIndex, sourceTree, -1);
}

/****************************************************************************/

void MainWindow::expandProperties()
{
    expandChildren(propertyTree->rootIndex(), propertyTree, 2);
}

/****************************************************************************/

void MainWindow::propertyTreeCustomContextMenu(const QPoint &point)
{
    QMenu *menu = new QMenu(propertyTree);
    menu->setAttribute(Qt::WA_DeleteOnClose);
    QModelIndex index = propertyTree->indexAt(point);

    if (index.isValid()) {
        propertyNode = (PropertyNode *) index.internalPointer();
    }
    else {
        propertyNode = NULL;
    }

    QAction *a = menu->addAction(tr("Reset"), this, SLOT(resetProperty()));
    // a->setIcon(QIcon(":/images/view-refresh.svg"));
    Property *property = dynamic_cast<Property *>(propertyNode);
    if (property) {
        a->setEnabled(property->isSet(propertyModel));
    }
    else {
        a->setEnabled(false);
    }

    menu->exec(propertyTree->viewport()->mapToGlobal(point));
}

/****************************************************************************/

void MainWindow::resetProperty()
{
    if (!propertyNode) {
        return;
    }

    Property *property = dynamic_cast<Property *>(propertyNode);
    if (property) {
        property->reset(propertyModel);
        setLayoutChanged();
    }
}

/****************************************************************************/

void MainWindow::scriptVariablesCustomContextMenu(const QPoint &point)
{
    ScriptVariableModel *model(dynamic_cast<ScriptVariableModel *>(
            tableViewScriptVariables->model()));
    if (not model) {
        return;
    }

    QMenu *menu = new QMenu(this);
    menu->setAttribute(Qt::WA_DeleteOnClose);
    QModelIndex index(tableViewScriptVariables->indexAt(point));
    scriptVariable = model->getScriptVariable(index);

    QAction *action;
    action = menu->addAction(
            tr("Remove"),
            this,
            SLOT(removeScriptVariables()));
    action->setEnabled(scriptVariable);
    action->setIcon(QIcon(":/images/list-remove.svg"));

    menu->addSeparator();  // -------------------------------

    action = menu->addAction(
            tr("Clear list"),
            this,
            SLOT(clearScriptVariables()));
    action->setEnabled(not model->isEmpty());
    action->setIcon(QIcon(":/images/edit-clear.svg"));

    menu->exec(tableViewScriptVariables->viewport()->mapToGlobal(point));
}

/****************************************************************************/

void MainWindow::removeScriptVariables()
{
    if (!scriptVariable) {
        return;
    }

    ScriptVariableModel *model(dynamic_cast<ScriptVariableModel *>(
            tableViewScriptVariables->model()));
    if (not model) {
        return;
    }

    model->remove(scriptVariable);
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::clearScriptVariables()
{
    ScriptVariableModel *model(dynamic_cast<ScriptVariableModel *>(
            tableViewScriptVariables->model()));
    if (not model) {
        return;
    }

    model->clear();
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::tableViewParametersCustomContextMenu(const QPoint &point)
{
    ParameterModel *model(
            dynamic_cast<ParameterModel *>(tableViewParameters->model()));
    if (not model) {
        return;
    }

    QMenu *menu = new QMenu(tableViewParameters);
    menu->setAttribute(Qt::WA_DeleteOnClose);
    QModelIndex index(tableViewParameters->indexAt(point));
    Parameter *parameter = model->getParameter(index);

    QAction *action;
    action = menu->addAction(
            tr("Remove parameter"),
            this,
            SLOT(removeParameter()));
    action->setEnabled(parameter);
    action->setIcon(QIcon(":/images/list-remove.svg"));

    menu->addSeparator();  // -------------------------------

    action = menu->addAction(
            tr("Clear parameter list"),
            this,
            SLOT(clearParameters()));
    action->setEnabled(not model->isEmpty());
    action->setIcon(QIcon(":/images/edit-clear.svg"));


    menu->exec(tableViewParameters->viewport()->mapToGlobal(point));
}

/****************************************************************************/

void MainWindow::removeParameter()
{
    ParameterModel *model(
            dynamic_cast<ParameterModel *>(tableViewParameters->model()));
    if (not model) {
        return;
    }

    const QModelIndexList indexes =
            tableViewParameters->selectionModel()->selectedIndexes();
    QList<Parameter *> parameters;

    for (const QModelIndex &index : indexes) {
        if (not parameters.contains(model->getParameter(index))) {
            Parameter *parameter = model->getParameter(index);
            parameters.append(parameter);
        }
    }
    foreach (Parameter *parameter, parameters) {
        model->remove(parameter);
    }
    updateRemoveButton();
    updateSaveButton();
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::clearParameters()
{
    ParameterModel *model(
            dynamic_cast<ParameterModel *>(tableViewParameters->model()));
    if (not model) {
        return;
    }

    model->clear();
    updateRemoveButton();
    updateSaveButton();
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::newParameterSet()
{
    ParameterSetModel *model(dynamic_cast<ParameterSetModel *>(
            listViewParameterSets->model()));
    if (not model) {
        return;
    }

    model->newParameterSet();
    updateRemoveSetButton();
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::removeParameterSet()
{
    ParameterSetModel *model(dynamic_cast<ParameterSetModel *>(
            listViewParameterSets->model()));
    if (not model) {
        return;
    }

    const QModelIndexList indexes =
            listViewParameterSets->selectionModel()->selectedIndexes();
    QList<ParameterSet *> parameterSets;
    QModelIndex setIndex =
            listViewParameterSets->selectionModel()->currentIndex();
    if (setIndex.isValid()) {
        ParameterSet *parameterSet = model->getParameterSet(setIndex);
        model->remove(parameterSet);
    }

    listViewParameterSets->clearSelection();

    tableViewParameters->setModel(defaultParameterModel);

    updateParameterButtons();
    updateRemoveSetButton();
    setLayoutChanged();
}

/****************************************************************************/

void MainWindow::changeParameterSet()
{
    ParameterSet *parameterSet;
    QModelIndex index;

    QObject *sender = QObject::sender();
    if (sender) {
        if (listViewParameterSets->selectionModel()->hasSelection()) {
            index = listViewParameterSets->currentIndex();
        }
        else {
            index = parameterSetModel->index(0);
        }
        parameterSet = parameterSetModel->getParameterSet(index);
        if (parameterSet) {
            parameterSet->getParameterModel();
            tableViewParameters->setModel(parameterSet->getParameterModel());
        }
    }

    updateRemoveSetButton();
    updateSaveButton();
    updateRemoveButton();
    setLayoutChanged();

    if (tableViewParameters->selectionModel()) {
        connect(tableViewParameters->selectionModel(),
                &QItemSelectionModel::selectionChanged,
                this,
                &MainWindow::updateRemoveButton);
    }

    if (tableViewParameters->model()) {
        connect(tableViewParameters->model(),
                &QAbstractTableModel::dataChanged,
                this,
                &MainWindow::updateSaveButton);
    }
}

/****************************************************************************/

void MainWindow::saveParameters()
{
    ParameterModel *model(
            dynamic_cast<ParameterModel *>(tableViewParameters->model()));
    if (not model) {
        return;
    }

    ParameterSaveDialog *dialog(new ParameterSaveDialog(model));

    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);

    connect(dialog, &QDialog::accepted, this, [model]() {
        model->updateParameters();
        // store directory for opening the dialog next time
        QFileInfo file(model->getPath());
        QString searchPath(file.absoluteDir().path());
        QDir::setSearchPaths("parameters", QStringList(searchPath));

        model->saveParameters();
    });

    dialog->open();
}

/****************************************************************************/

void MainWindow::updateRemoveSetButton()
{
    pushButtonRemoveSet->setEnabled(
            listViewParameterSets->selectionModel()->hasSelection());
}

/****************************************************************************/

void MainWindow::updateRemoveButton()
{
    QItemSelectionModel *selectionModel(dynamic_cast<QItemSelectionModel *>(
            tableViewParameters->selectionModel()));

    if (not selectionModel) {
        qWarning() << "invalid selection model";
        return;
    }

    pushButtonRemove->setEnabled(selectionModel->hasSelection());
}

/****************************************************************************/

void MainWindow::updateSaveButton()
{
    ParameterModel *model(
            dynamic_cast<ParameterModel *>(tableViewParameters->model()));

    if (not model) {
        qWarning() << "invalid model";
        return;
    }

    pushButtonSaveParameters->setEnabled(not model->isEmpty());
    pushButtonClear->setEnabled(not model->isEmpty());
}

/****************************************************************************/

void MainWindow::tabCloseRequested(int index)
{
    tabWidget->widget(index)->deleteLater();
    tabWidget->removeTab(index);
}

/****************************************************************************/

void MainWindow::pythonStdOutDataAvailable(int)
{
    pythonStdOutNotifier->setEnabled(false);

    QByteArray data;
    data.reserve(1024);
    int ret = ::read(pythonStdOut[1], data.data(), data.capacity());
    if (ret == -1) {
        qWarning() << "Failed to read from python stdout:" << strerror(errno);
        return;
    }
    data.resize(ret);

    textEditPython->moveCursor(QTextCursor::End);
    textEditPython->setTextColor(color[0]);
    textEditPython->insertPlainText(data);
    textEditPython->moveCursor(QTextCursor::End);

    pythonStdOutNotifier->setEnabled(true);
}

/****************************************************************************/

void MainWindow::pythonStdErrDataAvailable(int)
{
    pythonStdErrNotifier->setEnabled(false);

    QByteArray data;
    data.reserve(1024);
    int ret = ::read(pythonStdErr[1], data.data(), data.capacity());
    if (ret == -1) {
        qWarning() << "Failed to read from python stderr:" << strerror(errno);
        return;
    }
    data.resize(ret);

    textEditPython->moveCursor(QTextCursor::End);
    textEditPython->setTextColor(color[1]);
    textEditPython->insertPlainText(data);
    textEditPython->moveCursor(QTextCursor::End);

    pythonStdErrNotifier->setEnabled(true);
}

/****************************************************************************/

void MainWindow::connectionStateChanged(DataModel::ConnectionState state)
{
    actionConnectAll->setEnabled(
            state == DataModel::NoneConnected
            || state == DataModel::SomeConnected);
}

/****************************************************************************/

void MainWindow::connectDataSlots()
{
    for (auto tabPage : tabPages()) {
        tabPage->connectDataSlots();
    }

    parameterSetModel->connectSets();
}

/****************************************************************************/

void MainWindow::statsChanged()
{
    QLocale loc;
    QString num;

    num = loc.toString(dataModel->getInRate() / 1024.0, 'f', 1);
    labelIn->setText(tr("%1 KB/s").arg(num));
    num = loc.toString(dataModel->getOutRate() / 1024.0, 'f', 1);
    labelOut->setText(tr("%1 KB/s").arg(num));
}

/****************************************************************************/

void MainWindow::currentMessage(const QtPdCom::Message *msg)
{
    if (msg) {
        QIcon icon;
        switch (msg->getType()) {
            case QtPdCom::Message::Critical:
            case QtPdCom::Message::Error:
                icon = QIcon(":/images/dialog-error.svg");
                break;
            case QtPdCom::Message::Warning:
                icon = QIcon(":/images/dialog-warning.svg");
                break;
            default:
                break;
        }
        statusMessageIcon->setPixmap(icon.pixmap(QSize(16, 16)));

        QString lang = QLocale().name().left(2);
        if (lang == "C") {
            lang = "en";
        }

        statusMessageText->setText(msg->getText(lang));
        statusMessageText->setToolTip(msg->getDescription(lang));
    }
    else {
        statusMessageIcon->setPixmap(QPixmap());
        statusMessageText->setText("");
        statusMessageText->setToolTip(QString());
    }
}

/****************************************************************************/
