# CiA 402 Support

DISCLAIMER: LinuxCNC-Ethercat's CiA 402 support is still very young
and is not complete yet.  Expect rough edges, bugs, and undocumented
requirements.  Please file bugs as you find them, and it'll hopefully
mature rapidly.

## About

[CiA 402](https://www.can-cia.org/can-knowledge/canopen/cia402/) is a
standard for servo drives, stepper drives, VFDs, and similar motion
controllers.  It is part of the CanOPEN Industry Association, and can
be implemented on top of EtherCAT and other common fieldbusses.

Most Ethercat servo and stepper drives support CiA 402, including
Beckhoff's AX8000 series, EtherCAT drives from RTelligent, Leadshine,
Delta, and other vendors at all points of the price spectrum.

Notable *non* CiA 402 devices are Beckhoff EL7xxx and AX5xxx series
drives.

## How to use

To use a CiA 402 EtherCAT device with LinuxCNC, you need two things:

1.  A way to get the EtherCAT layer talking to your device.  This can
    be done with a `generic` driver and a bunch of XML, or a
    device-specific driver that (at least potentially) makes
    everything much less trouble-prone.  This document covers the
    device-specific driver case, but some of the details can be used
    with generic as well.  See the LinuxCNC forums for help with
    generic configs.
2.  The CiA 402 HAL component for LinuxCNC, from https://github.com/dbraun1981/hal-cia402

### Configuring XML

Here's a simple example for an RTelligent ECT60:

```xml
<masters>
  <master idx="0" appTimePeriod="1000000" refClockSyncCycles="1000">
    ...
    <slave idx="22" type="ECT60" name="xaxis">
      <modParam name="peakCurrent_amps" value="3.5"/>
      <modParam name="homeOffset" value="1"/>
    </slave>
	...
  </master>
</masters>
```

or the same device using the `basic_cia402` driver:

```xml
<masters>
  <master idx="0" appTimePeriod="1000000" refClockSyncCycles="1000">
    ...
    <slave idx="22" vid="0x00000a88" pid="0x0a880002" type="basic_cia402" name="xaxis">
      <modParam name="homeOffset" value="1"/>
    </slave>
	...
  </master>
</masters>
```

Notice that you need to specify the `vid` and `pid` parameters for
`basic_cia402` devices, and that `peakCurrent_amps` is no longer
allowed as it's an ECT60-specific configuration setting.

See below for more details on all of the options available.

### Installing the HAL component

To use the CiA HAL component, you'll need to download it, make sure
that you have the LinuxCNC dev package installed, and run `sudo halcompile
--install hal-cia402.comp`.  This should install the component needed.

### Configuring HAL for CiA 402 devices

TODO: explain setting up the HAL settings for devices.  For now,
follow the [hal-cia402](https://github.com/dbraun1981/hal-cia402)
instructions.

## Supported devices and drivers

- The LinuxCNC forums are full of examples of generic configs for various devices.
- The [DeASDA](deasda.md) driver supports Delta A2, A3, and B3
  devices, but doesn't (currently) use the CiA 402 component.
- The [rtec](rtec.md) driver supports RTelligent ECTxx devices, a line
  of inexpensive open- and closed-loop stepper drivers.
- The [basic_cia402](TBD) driver provides basic support for CiA 402
  devices, but lacks device-specific features like setting motor
  current limits.

Generally, for new hardware, the best way to support it is to begin
with a generic XML config but then create a dedicated driver using
`basic_cia402` as template, adding device-specific features over time.

## Common CiA 402 Features

### Operation Modes

CiA 402 defines a number of different operation modes.  Devices may
support one or more of these modes.  These names appear throughout
this document.

- `csp` Cyclic Synchronous Position mode.  The device and the
        controller (LinuxCNC) work together to control position,
        velocity, and torque.  The controller manages the feedback
        loop between these variables.  This is probably the best mode
        for general CNC position control.
- `cst` Cyclic Synchronous Torque mode.  The device and the controller
        (LinuxCNC) work together to control torque.
- `csv` Cyclic Synchronous Velocity mode.  The device and the
        controller (LinuxCNC) work together to control velocity and
        torque.
- `hm` Homing mode.
- `it` Interpolation mode.
- `pp` Profile Position mode.  You tell the motor what position to
       move to, it controls velocity and acceleration.
- `pt` Profile Torque mode.  You tell the motor how much torque to
       use.
- `pv` Profile Velocity mode.  You tell the motor what speed to move
       at, it controls acceleration.
- `vl` Velocity mode.  You tell the motor to run at a constant speed.
       Used for VFDs and similar devices.

### `<modParam>`s

The CiA 402 framework defines a number of generic parameters that can
be configured in the XML file.  Here's an overly-complex example:

```xml
    <slave idx="22" type="ECT60" name="xaxis">
      <modParam name="peakCurrent_amps" value="6.0"/>
      <modParam name="controlMode" value="closedloop"/>
      <modParam name="encoderResolution" value="1000"/>
      <modParam name="input3Func" value="home"/>
      <modParam name="input6Func" value="emergency-stop"/>
      <modParam name="output1Func" value="alarm"/>
      <modParam name="homeMode" value="17"/>
      <modParam name="homeOffset" value="0"/>
      <modParam name="homeAccel" value="500"/>
      <modParam name="homeSpeedFast" value="2500"/>
      <modParam name="homeSpeedSlow" value="500"/>
    </slave>
```

Not all devices support all modparams.  Trying to set a parameter that
your device doesn't support will trigger an error and halt LinuxCNC.
Since this only happens at startup, this is relatively safe.  

You can verify which modparams your device supports with a bit of
work.  Look at the table below under "Implementation Notes", find the
modparam that you're interested in, find the SDO address (`6085:00`,
for example, and then look in `ethercat sdos` to see if your device
supports it.  If it doesn't appear in `ethercat sdos`, then it won't
work.

#### Option Codes

CiA 402 provides a framework for telling the device what to do in the
case of certain exceptional events.  These `*OptionCode` settings
control how it handles these events.

##### `<modParam name="quickStopOptionCode" value=?>`

This configures behavior when a CiA 402 "quick stop" event happens.
Standard values include:

- `0`: disable drive function.
- `1`: slow down using the default deceleration (see `profileDecel`, below) and go to state "switch on disabled."
- `2`: slow down using the "quick stop ramp" (see `quickDecel`, below) and go to state "switch on disabled."
- `3`: slow down using the current limit and go to "switch on disabled."
- `4`: slow down using the voltage limit and go to "switch on disabled."
- `5`: slow down like `1`, but go stay in "quick stop active"
- `6`: slow down like `2`, but go stay in "quick stop active"
- `7`: slow down like `3`, but go stay in "quick stop active"
- `8`: slow down like `4`, but go stay in "quick stop active"

Consult your device documentation to see which of these are supported,
what device-specific options are available, and what your device's
default it (likely `2`).

##### `<modParam name="shutdownOptionCode" value=?>`

This configures behavior when a CiA 402 device goes from "Operation
Enabled" to "Ready To Switch On".  Standard values include:

- `0`: Disable drive function and switch off the drive power.
- `1`: Slow down using the default deceleration (see `profileDecel`, below), then disable the drive function.

Consult your device documentation to see which of these are supported,
what device-specific options are available, and what your device's
default it (likely `0`).

##### `<modParam name="disableOptionCode" value=?>`

This configures behavior when a CiA 402 device goes from "Operation
Enabled" to "Switched On."  Standard values include:

- `0`: Disable drive function and switch off the drive power.
- `1`: Slow down using the default deceleration (see `profileDecel`, below), then disable the drive function.

Consult your device documentation to see which of these are supported,
what device-specific options are available, and what your device's
default it (likely `1`).

##### `<modParam name="haltOptionCode" value=?>`

This configures behavior when the halt function is triggered.
Standard values include:

- `0`: reserved
- `1`: Slow down using the default deceleration (see `profileDecel`, below), then stay in "Operation Enabled."
- `2`: Slow down using the quick stop deceleration (see `quickDecel`, below), then stay in "Operation Enabled."
- `3`: Slow down using current limiting, then stay in "Operation Enabled."
- `4`: Slow down using voltage limiting, then stay in "Operation Enabled."

Consult your device documentation to see which of these are supported,
what device-specific options are available, and what your device's
default it (likely `1`).

##### `<modParam name="faultOptionCode" value=?>`

This configures behavior when a fault occurs.

Standard values include:

- `0`: Disable the drive, possibly leaving the motor free to rotate.
- `1`: Slow down using the default deceleration (see `profileDecel`, below).
- `2`: Slow down using the quick stop deceleration (see `quickDecel`, below).
- `3`: Slow down using current limiting.
- `4`: Slow down using voltage limiting.

Consult your device documentation to see which of these are supported,
what device-specific options are available, and what your device's
default it (likely `2`).

#### Position Limits

CiA 402 provides two ways to tell the device what the limits are to its range of motion:

- `swPositionLimit*`: sets the software limits on how far the axis is allowed to travel.  You probably want to set this for most CNC axes.
- `positionLimit`: sets the limit on how far the measurement for the axis goes, and rolls over when it's exceeded.

Units are implementation-defined, see your manufacturer's documentation.

##### `<modParam name="positionLimitMax" value=?>`

Sets the maximum value allowed for the position.  When this happens,
the position value will wrap around to the minimum limit (see below).

##### `<modParam name="positionLimitMin" value=?>`

Sets the minimum value allowed for the position.  When this happens,
the position value will wrap around to the maximum limit (see above).


##### `<modParam name="swPositionLimitMax" value=?>`

Sets the maximum value for the "software position limit".  This is the
largest value that you can request that the motor moves to.  This may
be offset by `homeOffset`, below.

##### `<modParam name="swPositionLimitMin" value=?>`

Sets the minimum value for the "software position limit".  This is the
smallest value that you can request that the motor moves to.  This may
be offset by `homeOffset`, below.

#### Velocity and Acceleration

CiA 402 has a number of speed, velocity, and acceleration settings.

Many devices will also have device-specific settings that interact with these.

Note that the units for most of these are undefined in the CiA 402
standard, you'll need to see your device's documentation.

##### `<modParam name="profileMaxVelocity" value=?>`

This is the maximum velocity allowed during movement in `pv`, `pp`, and `it` modes.

##### `<modParam name="motorMaxSpeed_RPM" value=?>`

This is the maximum motor speed allowed, in RPM.  This controls the
same basic setting as `profileMaxVelocity`, above, but in different
units.

##### `<modParam name="profileVelocity" value=?>`

This controls the maximum velocity used when moving in `pp` and `ip` modes.

##### `<modParam name="profileEndVelocity" value=?>`

This controls the *ending* velocity when motion reaches the target
position.  It should generally be 0.  It's used in `it` and `pp`
modes.

##### `<modParam name="profileAccel" value=?>`

This controls the normal acceleration in `pp`, `pv`, and `it` modes.

##### `<modParam name="profileDecel" value=?>`

This controls the normal deceleration in `pp`, `pv`, and `it` modes.

##### `<modParam name="quickDeccel" value=?>`

This controls the deceleration when a quick stop is needed, see the
"Option Code" settings above to see when quick stops are used.

#### Homing

CiA 402 provides a bunch of homing settings.  In general, CiA 402 homing isn't yet complete in LinuxCNC.

##### `<modParam name="homeOffset" value=?>`

TODO

##### `<modParam name="homeMethod" value=?>`

TODO

##### `<modParam name="homeVelocityFast" value=?>`

TODO

##### `<modParam name="homeVelocitySlow" value=?>`

TODO

##### `<modParam name="homeAccel" value=?>`

TODO


#### Probing

CiA 402 provides several probing settings.  In general, CiA 402 probing isn't yet complete in LinuxCNC.

##### `<modParam name="probeFunction" value=?>`

TODO

##### `<modParam name="probe1Positive" value=?>`

TODO

##### `<modParam name="probe1Negative" value=?>`

TODO

##### `<modParam name="probe2Positive" value=?>`

TODO

##### `<modParam name="probe2Negative" value=?>`

TODO

### Pins

The CiA 402 framework defines a number of HAL pins, depending on which optional features are enabled.
 
- `srv-cia-controlword` -- CiA 402 state machine control word, used by cia402 HAL component.
- `srv-cia-statusword` -- CiA 402 state machine status word, used by cia402 HAL component.
- `srv-opmode` -- CiA 402 opmode, used by cia402 HAL component.
- `srv-opmode-display` -- CiA 402 current op mode, used by cia402 HAL component.
- `srv-actual-position` -- actual position, in device-dependent units.
- `srv-actual-velocity` -- actual velocity, in device-dependent units.
- `srv-actual-torque` -- actual torque, in device-dependent units.
- `srv-target-position` -- target position for `pp` and `csp` modes.
- `srv-target-velocity` -- target velocity for `pv` and `csv` modes.
- `srv-supported-modes`  -- Modes supported by this device, from 0x6502:00.
- `srv-supports-mode-csp` -- True if this device supports `csp` mode.
- `srv-supports-mode-cst` -- True if this device supports `cst` mode.
- `srv-supports-mode-csv` -- True if this device supports `csv` mode.
- `srv-supports-mode-hm` -- True if this device supports `hm` mode.
- `srv-supports-mode-ip` -- True if this device supports `ip` mode.
- `srv-supports-mode-pp` -- True if this device supports `pp` mode.
- `srv-supports-mode-pv` -- True if this device supports `pv` mode.
- `srv-supports-mode-tq` -- True if this device supports `pq` mode.
- `srv-supports-mode-vl` -- True if this device supports `vl` mode.

## Supporting new devices

The CiA 402 standard covers a wide variety of motion controllers with
a variety of operating modes.  Very few of the features and settings
that the CiA 402 standard describes are required in all devices; the
bulk of them are either completely optional or only required if the
controller supports a specific operating mode.

Given this, the LinuxCNC CiA 402 driver framework has to be very
flexible, and this means that each specific device driver will have to
do more setup work than might be expected. 

To create a new driver, start with the driver in
`src/devices/lcec_basic_cia402.c`.  Make a copy of it with a new name
(`lcec_my_devicename.c`) and start modifying it.  As a first step, do
a search-and-replace for `lcec_basic_cia402` and replace that with
`lcec_my_devicename` through the whole file.

Search for `XXXX`, this should flag things that will need to be
changed, and things that you may wish to modify for your device.  use
`lcec_rtec.c` as an example of a more fleshed-out CiA 402 driver, it
should have examples of most of the features that are currently
supported.

There are 4 basic classes of things that will need to be handled:

1.  Configure CiA 402 featues.  Look for the block that includes
    `options->enable_pv` and set values as appropriate for your
    device.  Feel free to delete lines rather than setting them to
    `0`.  There is only one `options->enable_` setting that defaults
    to true; if somehow you have a CiA 402 device that doesn't support
    the `opmode` PDO, then you'll want to set
    `options->enable_opmode=0`.  Then please file a bug to let us know
    that you actually found an opmode-less device.
2.  Add device-specific `<modParam>`s.  See `lcec_rtec.c` for an
    example here; you can define parameters that let users control
    device-specific settings (motor current, encoder type, etc) from
    the XML configuration file.  To do this, you'll need to assign the
    modparam a number (add a `#define M_MYNEWMODPARAM x` entry, x
    needs to be unique but less than 0x1000.)  Then add your new param
    to the list just below this, and add code for setting it to
    `handle_modparams()`.  Again, see `lcec_rtec.c` for an example.
3.  Add device-specific pins.  This is a multi-step process, but isn't
    particularly CiA 402-specific:
    1.  Add variables for the pins and PDOs to
        `lcec_my_devicename_data_t`.  You'll want a `hal_pin_*`
        variable for each pin.  Each PDO will need an `int pdoname_os`
        entry, and boolean PDOs will also need an `int pdoname_bp`
        variable.
    2.  Add the pins to `slave_pins`.  See `lcec_rtec.c` for the format.
	3.  Add the PDO entries to the sync mapping.  Look for
        `lcec_syncs_add_pdo_entry` and follow the instructions.  Make
        sure that you get the variable sizes correct, and add gaps
        around boolean entries.
	4.  Register the PDOs with `lcec_pdo_init()`.  There's an example
        commented out.  For boolean PDOs (or really anything smaller
        than 8 bits), you'll need to use `&hal_data->pdoname_bp`
        instead of `NULL` at the end of the line.
	5.  Add the PDOs and pins to the read and write functions.  There
        should be an example provided, if you need more help look at
        almost any other driver's read or write functions, and you'll
        probably find an example.
4. Deal with digital in/out pins.  There are examples in place now,
   commented out.  At some point, this may be moved fully into the CiA
   402 framework, but it's somewhat complicated right now.

Run `make -j` from `src/` to compile the driver, and `sudo make
install` to install it.  Then test, find bugs, and iterate.

Feel free to open issues in Github asking for help.

## Implementation Notes

I haven't yet found a current copy of the CiA 402 spec that is
available without paying hundreds of dollars.  So, for the moment,
we're implementing this somewhat blind, by following a 2007 draft plus
the manufacturer documentation for multiple devices, and trying to
infer what the actual spec says by what devices implement.

Here's a full list of objects defined across multiple vendors'
allegedly CiA 402-compliant EtherCAT hardware plus entries from a 2007
draft.  Note that almost all devices will *also* provide
device-specific objects, and some of these will overlap with CiA 402
objects.  Also, it's not always clear which objects should be mapped
as PDOs and which should just be set via SDOs before startup.  We will
try to document problem cases here as they arise.

In addition, many objects are only used in specific opmodes, and a
very similar (but not quite identical) set of objects control similar
features in other modes.  For example, `6042:00` is target velocity
for `vl` mode, while `60ff:00` is target velocity for `pv` mode.

All addresses are hex, presented without the leading `0x`.  Addresses
are provided for the first channel on a slave; additional channels on
multi-axis devices will start at `0x6800`, `0x7000`, and `0x7800`
instead of `0x6000`.

Items listed as "add" should be added soon.  Items listed as "add if
hardware supports" or "add?" can be added if hardware that supports
the feature is added.

Most objects are marked as "optional" in the standard.  The
"Mandatory?" column records which objects are required and when.  Many
objects are only required if specific opmodes are supported, these
should all be called out.


| Index   | Type | Mandatory?             | Used in Modes                         | Description                        | Pin/modParam                         | Notes                                                                                               |
|---------|------|------------------------|---------------------------------------|------------------------------------|--------------------------------------|-----------------------------------------------------------------------------------------------------|
| 6007:00 | S16  |                        |                                       | Abort connection option code       | ADD MODPARAM                         |                                                                                                     |
| 603f:00 | U16  |                        |                                       | Fault Code                         | pin: `src-error-code`                | Error code of the last error.  Optional.                                                            |
| 6040:00 | U16  | Yes                    | All                                   | Control Word                       | pin: `srv-cia-controlword`           | Controls CiA 402 device.  Required.                                                                 |
| 6041:00 | U16  | Yes                    | All                                   | Status Word                        | pin: `srv-cia-statusword`            | Status of CiA 402 device. Required.                                                                 |
| 6042:00 | S16  | `vl`                   | `vl`                                  | Target vl velocity                 | pin: `srv-target-vl`                 | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6043:00 | S16  | `vl`                   | `vl`                                  | vl Velocity demand                 | pin: `srv-demand-vl`                 | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6044:00 | S16  | `vl`                   | `vl`                                  | Actual vl velocity                 | pin: `srv-actual-vl`                 | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6046:01 | S16  | `vl`                   | `vl`                                  | vl velocity minimum                | pin: `srv-vl-minimum`                | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6046:02 | S16  | `vl`                   | `vl`                                  | vl velocity maximum                | pin: `srv-vl-maximum`                | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6048:00 | U32  | `vl`                   | `vl`                                  | vl velocity acceleration           | pin: `srv-vl-accel`                  | Not on ECT60 (for VFDs, mostly)                                                                     |
| 6049:00 | U32  | `vl`                   | `vl`                                  | vl velocity deceleration           | pin: `srv-vl-decel`                  | Not on ECT60 (for VFDs, mostly)                                                                     |
| 604a:01 | U32  | `vl`                   | `vl`                                  | vl velocity quick stop numerator   | mp: `vlQuickStopRatio`               |                                                                                                     |
| 604a:02 | U16  | `vl`                   | `vl`                                  | vl velocity quick stop denominator | mp: `vlQuickStopRatio`               |                                                                                                     |
| 604b:01 | S16  | `vl`                   | `vl`                                  | vl velocity set-point numerator    | mp: `vlSetPoint`                     |                                                                                                     |
| 604b:02 | S16  | `vl`                   | `vl`                                  | vl velocity set-point denominator  | mp: `vlSetPoint`                     |                                                                                                     |
| 604c:00 | S32  | `vl`                   | `vl`                                  | vl velocity dimension numerator    | mp: `vlDimensionFactor`              |                                                                                                     |
| 604c:00 | S32  | `vl`                   | `vl`                                  | vl velocity dimension denominator  | mp: `vlDimensionFactor`              |                                                                                                     |
| 605a:00 | S16  |                        | ~all                                  | Quick stop option code             | mp: `quickStopOptionCode`            |                                                                                                     |
| 605b:00 | S16  |                        |                                       | Shutdown option code               | mp: `shutdownOptionCode`             |                                                                                                     |
| 605c:00 | S16  |                        |                                       | Disable operation option code      | mp: `disableOptionCode`              |                                                                                                     |
| 605d:00 | S16  |                        |                                       | Halt option code                   | mp: `haltOptionCode`                 |                                                                                                     |
| 605e:00 | S16  |                        |                                       | Fault Reaction Option Code         | mp: `faultOptionCode`                | Controls what happens when a fault is detected.                                                     |
| 6060:00 | U16  | optional!              | All                                   | Operating Mode                     | pin: `srv-opmode`                    | Controls the operating mode of the device.  Profile/velocity/torque/etc.  See `cia402` component.   |
| 6061:00 | U16  | optional!              | All                                   | Operating Mode Display             | pin: `srv-opmode-display`            | Shows the current operating mode.  See `cia402` component.                                          |
| 6062:00 | S32  |                        | `pp`, `ip`, others?                   | Position demand value              | add pin: `srv-position-demand`       | Input, showing state of motion controller                                                           |
| 6063:00 | S32  |                        |                                       | Actual location internal           | NOT PLANNED                          | Current position in internal units.  Encoder position?                                              |
| 6064:00 | S32  | `pp` `csp`             | `pp`, `csp`, `csv`?, `cst`?           | Actual location                    | pin: `srv-actual-position`           | Current position.  See `cia402` component.                                                          |
| 6065:00 | U32  |                        | `pp`?, `csp`                          | Following error window             | pin: `srv-following-error-window`    | Error window that will trigger a fault is the motor position is outside of the window. -1 disables. |
| 6066:00 | U16  |                        | `pp`?, `csp`                          | Following error timeout            | pin: `srv-following-error-timeout`   | Time window (in ms) for a following error.                                                          |
| 6067:00 | U32  |                        |                                       | Position error window              | ADD PIN                              | Position window that will trigger a fault                                                           |
| 6068:00 | U16  |                        |                                       | Position error timeout             | ADD PIN                              | Time window (in ms) for a position error.                                                           |
| 6069:00 | S32  |                        | `pv`, `csp`, `csv`                    | Velocity sensor actual value       | pin: `srv-actual-velocity-sensor`    |                                                                                                     |
| 606a:00 | S16  |                        | `pv` (also `csp`, `csv`?)             | Velocity sensor selection code     | pin: `srv-velocity-sensor-selector`  |                                                                                                     |
| 606b:00 | S32  |                        | `pv`                                  | Velocity demand value              | pin: `srv-velocity-demand`           | Outputs internal "velocity demand" between trajectory generator and velocity controller.            |
| 606c:00 | S32  | `pv` `csv`             | `pv`, `csp`, `csv`, `cst`             | Actual velocity                    | pin: `srv-actual-velocity`           | Current velocity.  See `cia402` component.                                                          |
| 606d:00 | U16  |                        | `pv`                                  | Velocity error window              | pin: `srv-velocity-error-window`     | Velocity window for calculating a velocity error                                                    |
| 606e:00 | U16  |                        | `pv`                                  | Velocity error timeout             | pin: `srv-velocity-error-time`       | Time for calculating a velocity error                                                               |
| 606f:00 | U16  |                        | `pv`                                  | Velocity threshold                 | pin: `srv-velocity-threshold-window` |                                                                                                     |
| 6070:00 | U16  |                        | `pv`                                  | Velocity threshold time            | pin: `srv-velocity-threshold-time`   |                                                                                                     |
| 6071:00 | S16  | `tq` `cst`             | `tq`, `cst`                           | Target torque                      | pin: `srv-target-torque`             |                                                                                                     |
| 6072:00 | U16  |                        | `tq`, `csp`, `csv`, `cst`             | Maximum torque                     | pin: `srv-maximum-torque`            |                                                                                                     |
| 6073:00 | U16  |                        | `tq`                                  | Maximum current                    | pin: `srv-maximum-current`           | In torque mode, in mA.                                                                              |
| 6074:00 | S16  |                        | `tq`                                  | Torque demand                      | pin: `srv-torque-demand`             | Outputs internal "torque demand" between trajectory generator and torque controller.                |
| 6075:00 | U32  |                        | `tq`                                  | Motor rated current                | pin: `srv-motor-rated-current`       | Only for torque mode?  In mA.                                                                       |
| 6076:00 | U32  |                        | `tq`, `csp`, `csv`, `cst`             | Motor rated torque                 | pin: `srv-motor-rated-torque`        | In `csp`, others?  In mN or mN-m                                                                    |
| 6077:00 | S16  | `cst`                  | `tq`, `csp`, `csv`, `cst`             | Actual torque                      | pin: `srv-actual-torque`             | Current torque.  See `cia402` component.                                                            |
| 6078:00 | S16  |                        | `tq`                                  | Actual current                     | pin: `srv-actual-current`            | Current current, in mA.                                                                             |
| 6079:00 | U32  |                        | `tq`                                  | Actual voltage                     | pin: `srv-actual-voltage`            | Current voltage, in mV.                                                                             |
| 607a:00 | S32  | `pp` `pc`(?) `csp`     | `pp`, `csp`                           | Target location                    | pin: `srv-target-position`           | Target position.  See `cia402` component.                                                           |
| 607b:01 | S32  |                        | `pp`, `ip`, `csp`                     | Position range limit (min)         | mp: `positionLimitMin`               | Range limit for position, wraps around on overflow.  Not on ECT60.                                  |
| 607b:01 | S32  |                        | `pp`, `ip`, `csp`                     | Position range limit (max)         | mp: `positionLimitMax`               | Range limit for position, wraps around on overflow.  Not on ECT60.                                  |
| 607c:00 | S32  |                        | `hm`, `ip`, other `*p`?               | Home Offset                        | mp: `homeOffset`                     | ECT60 says it's in units of 'pulses'.                                                               |
| 607d:01 | S32  |                        | `pp`, `ip`, `csp`                     | Min position limit                 | mp: `swPositionLimitMin`             | Range limit for position, does not wrap.                                                            |
| 607d:02 | S32  |                        | `pp`, `ip`, `csp`                     | Max position limit                 | mp: `swPositionLimitMax`             | Range limit for position, does not wrap.                                                            |
| 607e:00 | U8   |                        | `pv`, `pp`, `csv`, `csp`              | Polarity                           | pin: `srv-polarity`                  | Controls direction +/-.  Functionally 2 different settings in one register.                         |
| 607f:00 | U32  |                        | `pp`, `ip`, `pv`                      | Max profile velocity               | pin: `srv-profile-max-velocity`      | Maximum speed during a profiled motion.                                                             |
| 6080:00 | U32  |                        | `pp`, `ip`, `pv`, `csp`, `csv`, `cst` | Max motor speed                    | pin: `srv-maximum-motor-rpm`         | Max allowed speed of motor, in RPM.                                                                 |
| 6081:00 | U32  |                        | `pp`, `ip`                            | Profile Velocity                   | pin: `srv-profile-velocity`          | Normal limit for velocity during a move operation.                                                  |
| 6082:00 | U32  |                        | `pp`, `ip`                            | End velocity                       | pin: `srv-profile-end-velocity`      | Velocity at the end of a move.  Usually 0.                                                          |
| 6083:00 | S32  |                        | `pp`, `ip`, `pv`                      | Profile/Track Acceleration         | pin: `srv-profile-accel`             |                                                                                                     |
| 6084:00 | S32  |                        | `pp`, `ip`, `pv`                      | Profile/Track deceleration         | pin: `srv-profile-decel`             |                                                                                                     |
| 6085:00 | S32  |                        | `pp`, `ip`, `pv`                      | Quick stop deceleration            | mp: `quickDecel`                     | Used for "quick stops" (errors, etc).                                                               |
| 6086:00 | S16  |                        | `pp`, `pv`, `csp`, `csv`              | Motion profile type                | pin: `srv-motion-profile`            | Not supported on ECT60.  Linear ramp, sin^2 ramp, jerk ramps, etc.                                  |
| 6087:00 | U32  | `tq`                   | `tq`                                  | Torque slope                       | pin: `srv-torque-slope`              |                                                                                                     |
| 6088:00 | S16  |                        | `tq`                                  | Torque profile type                | pin: `srv-torque-profile-type`       |                                                                                                     |
| 608f:01 | U32  |                        |                                       | Position encoder increments        | ADD RATIO MODPARAM                   | Not supported on ECT60.  Number of encoder steps per revolultion is ratio with below.               |
| 608f:02 | U32  |                        |                                       | Position motor revolutions         | ADD RATIO MODPARAM                   | Not supported on ECT60.                                                                             |
| 6090:01 | U32  |                        |                                       | Velocity encoder increments        | ADD RATIO MODPARAM                   | for "velocity encoders".                                                                            |
| 6090:02 | U32  |                        |                                       | Velocity motor revolutions         | ADD RATIO MODPARAM                   |                                                                                                     |
| 6091:01 | U32  |                        |                                       | Gear ratio motor revolutions       | ADD RATIO MODPARAM                   | Not on ECT60                                                                                        |
| 6091:02 | U32  |                        |                                       | Gear ratio shaft revolutions       | ADD RATIO MODPARAM                   | From Beckhoff AX8xxx, not on ECT60                                                                  |
| 6092:01 | U32  |                        |                                       | Feed constant numerator            | ADD RATIO MODPARAM                   | According to Leadshine, this is steps per revolution                                                |
| 6092:02 | U32  |                        |                                       | Feed constant denominator          | ADD RATIO MODPARAM                   | According to Leadshine, this is steps per revolution                                                |
| 6093:01 | U32  |                        |                                       | E-Gear ratio numerator             | ADD RATIO MODPARAM                   | From Delta ASDA                                                                                     |
| 6093:02 | U32  |                        |                                       | E-Gear ratio denominator           | ADD RATIO MODPARAM                   | From Delta ASDA                                                                                     |
| 6096:00 | ?    |                        |                                       | Velocity factor                    |                                      | Not in my CiA 402 draft                                                                             |
| 6097:00 | ?    |                        |                                       | Acceleration factor                |                                      | Not in my CiA 402 draft                                                                             |
| 6098:00 | S8   | `hm`                   | `hm`                                  | Homing method                      | pin: `srv-home-method`               | Defines which of 36 standard homing modes (plus up to 128 manufacturer-specific modes) to use.      |
| 6099:01 | U32  | `hm`                   | `hm`                                  | Homing velocity fast               | pin: `srv-home-velocity-fast`        |                                                                                                     |
| 6099:02 | U32  | `hm`                   | `hm`                                  | Homing velocity slow               | pin: `srv-home-velocity-slow`        |                                                                                                     |
| 609a:00 | U32  |                        | `hm`                                  | Homing acceleration                | pin: `srv-home-accel`                |                                                                                                     |
| 60a2:00 | ?    |                        |                                       | Jerk factor                        |                                      | Not in my CiA 402 draft                                                                             |
| 60a3:00 | U8   |                        | `pp`                                  | Profile jerk use                   |                                      | Not supported on ECT60 (?)                                                                          |
| 60a4:0x | U32  |                        | `pp`                                  | Profile jerk (x=1..6)              |                                      | Not supported on ECT60                                                                              |
| 60b0:00 | S32  |                        | `csp`                                 | Position offset                    | NOT PLANNED                          | Offset for position in `csp` mode.  Probably not useful in LinuxCNC.                                |
| 60b1:00 | S32  |                        | `csp`, `csv`                          | Velocity offset                    | NOT PLANNED                          | Offset for velocity in `csp` and `csv` modes.  Probably not useful in LinuxCNC.                     |
| 60b2:00 | S16  |                        | `csp`, `csv`, `cst`                   | Torque offset                      | NOT PLANNED                          | Offset for torque in `cs*` modes.  Actually a multiplier, in units of 1/1000 max torque.            |
| 60b8:00 | U16  |                        | `hm`                                  | Probe function                     | mp: `probeFunction`                  | Broken into 10 sub-fields.  Implement as an integer for now?                                        |
| 60b9:00 | U16  |                        | `hm`                                  | Probe status                       | ADD PIN                              |                                                                                                     |
| 60ba:00 | S32  |                        | `hm`                                  | Touch probe 1 positive value       | mp: `probe1Positive`                 |                                                                                                     |
| 60bb:00 | S32  |                        | `hm`                                  | Touch probe 1 negative value       | mp: `probe1Negative`                 |                                                                                                     |
| 60bc:00 | S32  |                        | `hm`                                  | Touch probe 2 positive value       | mp: `probe2Positive`                 |                                                                                                     |
| 60bd:00 | S32  |                        | `hm`                                  | Touch probe 2 negative value       | mp: `probe2Negative`                 |                                                                                                     |
| 60c0:00 | S16  |                        | `ip`                                  | Interpolated sub mode select       | NOT PLANNED                          |                                                                                                     |
| 60c1:xx | S32  |                        | `ip`                                  | Interpolation data record          | NOT PLANNED                          | Data for interpolation mode                                                                         |
| 60c2:01 | U8   | `ip` `csp` `csv` `cst` | `ip`, `csp`, `csv`, `cst`             | Interpolation time period          | pin: `srv-interpolation-time-period` |                                                                                                     |
| 60c2:02 | S8   | `ip` `csp` `csv` `cst` | `ip`, `csp`, `csv`, `cst`             | Interpolation index                | NOT PLANNED                          |                                                                                                     |
| 60c4:01 | U32  |                        | `ip`                                  | Interpolation max buffer size      | NOT PLANNED                          |                                                                                                     |
| 60c4:02 | U32  |                        | `ip`                                  | Interpolation act buffer size      | NOT PLANNED                          |                                                                                                     |
| 60c4:03 | bit  |                        | `ip`                                  | Interpolation buffer organ.        | NOT PLANNED                          |                                                                                                     |
| 60c4:04 | U16  |                        | `ip`                                  | Interpolation buffer position      | NOT PLANNED                          |                                                                                                     |
| 60c4:05 | U8   |                        | `ip`                                  | Interpolation record size          | NOT PLANNED                          |                                                                                                     |
| 60c4:06 | bit  |                        | `ip`                                  | Interpolation buffer clear         | NOT PLANNED                          |                                                                                                     |
| 60c5:00 | U32  |                        | `pp`, `ip`, `pv`                      | Max acceleration                   | pin: `srv-maximum-acceleration`      | Overall max accel, not on ECT60                                                                     |
| 60c6:00 | U32  |                        | `pp`, `ip`, `pv`                      | Max deceleration                   | pin: `srv-maximum-deceleration`      | Overall max decel, not on ECT60                                                                     |
| 60d0:00 | ?    |                        |                                       | Touch probe source                 |                                      | Not in my CiA 402 draft                                                                             |
| 60d1:00 | ?    |                        |                                       | Touch probe timestamp 1 + edge     |                                      | Not in my CiA 402 draft                                                                             |
| 60d2:00 | ?    |                        |                                       | Touch probe timestamp 1 - edge     |                                      | Not in my CiA 402 draft                                                                             |
| 60d3:00 | ?    |                        |                                       | Touch probe timestamp 2 + edge     |                                      | Not in my CiA 402 draft                                                                             |
| 60d4:00 | ?    |                        |                                       | Touch probe timestamp 2 - edge     |                                      | Not in my CiA 402 draft                                                                             |
| 60d5:00 | U16  |                        |                                       | Touch probe counter 1 + edge       |                                      | Not in my CiA 402 draft                                                                             |
| 60d6:00 | U16  |                        |                                       | Touch probe counter 1 - edge       |                                      | Not in my CiA 402 draft                                                                             |
| 60d7:00 | U16  |                        |                                       | Touch probe counter 2 + edge       |                                      | Not in my CiA 402 draft                                                                             |
| 60d8:00 | U16  |                        |                                       | Touch probe counter 2 - edge       |                                      | Not in my CiA 402 draft                                                                             |
| 60f2:00 | U16  |                        | `pp`, `ip`                            | Positioning option code            | ADD MODPARAM                         |                                                                                                     |
| 60f4:00 | U32  |                        | `csp`                                 | Following error actual value       | pin: `srv-actual-following-error`    |                                                                                                     |
| 60f8:00 | S32  |                        | `pv`                                  | Max slipage                        | ADD PIN                              |                                                                                                     |
| 60fa:00 | S32  |                        | `pp`                                  | Control effort                     | ADD PIN                              | "Control effort" measured in control loop, in undefined units.                                      |
| 60fc:00 | S32  |                        | `pp`, `hm`, `ip`                      | Position demand internal value     | skip for now                         |                                                                                                     |
| 60fd:00 | U32  |                        |                                       | Digital inputs                     | added                                |                                                                                                     |
| 60fe:01 | U32  |                        |                                       | Digital outputs                    | added                                | Undocumented on ECT60                                                                               |
| 60fe:02 | U32  |                        |                                       | Digital outputs mask               | unsure what to do with these today   | Undocumented on ECT60                                                                               |
| 60ff:00 | S32  | `pv` `csv`             | `pv`, `csv`                           | Target velocity                    | pin: `srv-target-velocity`           |                                                                                                     |
| 6402:00 | U16  |                        |                                       | Motor type                         | NOT PLANNED                          |                                                                                                     |
| 6403:00 | str  |                        |                                       | Motor catalog number               | NOT PLANNED                          |                                                                                                     |
| 6404:00 | str  |                        |                                       | Motor manufacturer                 | NOT PLANNED                          |                                                                                                     |
| 6405:00 | str  |                        |                                       | Motor catalog URL                  | NOT PLANNED                          |                                                                                                     |
| 6406:00 | time |                        |                                       | Motor calibration date             | NOT PLANNED                          |                                                                                                     |
| 6407:00 | U32  |                        |                                       | Motor service period               | NOT PLANNED                          |                                                                                                     |
| 6502:00 | U32  | yes                    |                                       | Supported opmodes                  | pin: `srv-supported-modes`           | See also `srv-supports-mode-*`                                                                      |
| 6503:00 | str  |                        |                                       | Drive catalog number               | NOT PLANNED                          |                                                                                                     |
| 6505:00 | str  |                        |                                       | Drive catalog url                  | NOT PLANNED                          |                                                                                                     |
