/*****************************************************************************
 *
 * $Id$
 *
 * Copyright (C) 2009 - 2012  Richard Hacker <lerich@gmx.net>
 *                            Florian Pose <fp@igh-essen.com>
 *
 * This file is part of the PdCom library.
 *
 * The PdCom library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The PdCom library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the PdCom Library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "pdcom/Time.h"

#include <sstream>

using namespace PdCom;

/****************************************************************************/

#define my_timerclear(tvp) ((tvp)->tv_sec = (tvp)->tv_usec = 0)

#define my_timercmp(a, b, CMP) \
    (((a)->tv_sec == (b)->tv_sec) ? \
     ((a)->tv_usec CMP (b)->tv_usec) : \
     ((a)->tv_sec CMP (b)->tv_sec))

#define my_timeradd(a, b, result) \
    do { \
        (result)->tv_sec = (a)->tv_sec + (b)->tv_sec; \
        (result)->tv_usec = (a)->tv_usec + (b)->tv_usec; \
        if ((result)->tv_usec >= 1000000) { \
            ++(result)->tv_sec; \
            (result)->tv_usec -= 1000000; \
        } \
    } while (0)

#define my_timersub(a, b, result) \
    do { \
        (result)->tv_sec = (a)->tv_sec - (b)->tv_sec; \
        (result)->tv_usec = (a)->tv_usec - (b)->tv_usec; \
        if ((result)->tv_usec < 0) { \
            --(result)->tv_sec; \
            (result)->tv_usec += 1000000; \
        } \
    } while (0)

/****************************************************************************/

Time::Time()
{
    my_timerclear(&t);
}

/****************************************************************************/

Time::Time(const Time &o)
{
    t = o.t;
}

/****************************************************************************/

Time::Time(double dbltime)
{
    *this = dbltime;
}

/****************************************************************************/

Time &Time::operator=(double dbltime)
{
    t.tv_sec = (typeof(t.tv_sec)) dbltime;
    t.tv_usec = (typeof(t.tv_usec)) ((dbltime - t.tv_sec) * 1e6);
    return *this;
}

/****************************************************************************/

Time &Time::operator+=(const Time &dt)
{
    my_timeradd(&t, &dt.t, &t);
    return *this;
}

/****************************************************************************/

Time &Time::operator-=(const Time &dt)
{
    my_timersub(&t, &dt.t, &t);
    return *this;
}

/****************************************************************************/

Time Time::operator*(double d) const
{
    return double(*this) * d;
}

/****************************************************************************/

Time Time::operator+(const Time &dt) const
{
    Time res;
    my_timeradd(&t, &dt.t, &res.t);
    return res;
}

/****************************************************************************/

Time Time::operator-(const Time &dt) const
{
    Time res;
    my_timersub(&t, &dt.t, &res.t);
    return res;
}

/****************************************************************************/

bool Time::operator>=(const Time &o) const
{
    return my_timercmp(&t, &o.t, >=);
}

/****************************************************************************/

bool Time::operator>(const Time &o) const
{
    return my_timercmp(&t, &o.t, >);
}

/****************************************************************************/

bool Time::operator<=(const Time &o) const
{
    return my_timercmp(&t, &o.t, <=);
}

/****************************************************************************/

bool Time::operator<(const Time &o) const
{
    return my_timercmp(&t, &o.t, <);
}

/****************************************************************************/

Time::operator double() const
{
    return (double) t.tv_sec + t.tv_usec / 1e6;
}

/****************************************************************************/

Time::operator bool() const
{
	return !(t.tv_sec == 0 && t.tv_usec == 0);
}

/****************************************************************************/

bool Time::operator==(const Time &o) const
{
	return t.tv_sec == o.t.tv_sec && t.tv_usec == o.t.tv_usec;
}

/****************************************************************************/

std::string Time::str() const
{
    std::ostringstream os;
    os << t.tv_sec << ".";
    os.fill('0');
    os.width(6);
    os << t.tv_usec;
    return os.str();
}

/****************************************************************************/
