/***********************************************************************************************************************************
Ini Handler
***********************************************************************************************************************************/
#ifndef COMMON_INI_H
#define COMMON_INI_H

/***********************************************************************************************************************************
Ini object
***********************************************************************************************************************************/
typedef struct Ini Ini;

#include "common/io/read.h"
#include "common/io/write.h"
#include "common/type/object.h"
#include "common/type/variant.h"

/***********************************************************************************************************************************
Constructors
***********************************************************************************************************************************/
Ini *iniNew(void);

/***********************************************************************************************************************************
Functions
***********************************************************************************************************************************/
// Move to a new parent mem context
__attribute__((always_inline)) static inline Ini *
iniMove(Ini *const this, MemContext *const parentNew)
{
    return objMove(this, parentNew);
}

// Parse ini from a string. Comments are ignored and additional whitespace around sections, keys, and values is trimmed. Should be
// used *only* to read user-generated config files, for code-generated info files see iniLoad().
void iniParse(Ini *this, const String *content);

// Set an ini value
void iniSet(Ini *this, const String *section, const String *key, const String *value);

/***********************************************************************************************************************************
Getters/Setters
***********************************************************************************************************************************/
// Get an ini value -- error if it does not exist
const String *iniGet(const Ini *this, const String *section, const String *key);

// Get an ini value -- if it does not exist then return specified default
const String *iniGetDefault(const Ini *this, const String *section, const String *key, const String *defaultValue);

// Ini key list
StringList *iniGetList(const Ini *this, const String *section, const String *key);

// The key's value is a list
bool iniSectionKeyIsList(const Ini *this, const String *section, const String *key);

// List of keys for a section
StringList *iniSectionKeyList(const Ini *this, const String *section);

// List of sections
StringList *iniSectionList(const Ini *this);

/***********************************************************************************************************************************
Destructor
***********************************************************************************************************************************/
__attribute__((always_inline)) static inline void
iniFree(Ini *const this)
{
    objFree(this);
}

/***********************************************************************************************************************************
Helper Functions
***********************************************************************************************************************************/
// Load an ini file and return data to a callback. Intended to read info files that were generated by code so do not have comments
// or extraneous spaces, and where all values are valid JSON. This allows syntax characters such as [, =, #, and whitespace to be
// used in keys.
void iniLoad(
    IoRead *read, void (*callbackFunction)(void *data, const String *section, const String *key, const String *value),
    void *callbackData);

/***********************************************************************************************************************************
Macros for function logging
***********************************************************************************************************************************/
#define FUNCTION_LOG_INI_TYPE                                                                                                      \
    Ini *
#define FUNCTION_LOG_INI_FORMAT(value, buffer, bufferSize)                                                                         \
    objToLog(value, "Ini", buffer, bufferSize)

#endif
