/***********************************************************************************************************************************
Test Build Config
***********************************************************************************************************************************/
#include "common/harnessStorage.h"

/***********************************************************************************************************************************
Test Run
***********************************************************************************************************************************/
static void
testRun(void)
{
    FUNCTION_HARNESS_VOID();

    // Create default storage object for testing
    Storage *storageTest = storagePosixNewP(TEST_PATH_STR, .write = true);

    // *****************************************************************************************************************************
    if (testBegin("bldCfgParse() and bldCfgRender()"))
    {
        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("render with label");

        TEST_RESULT_STR_Z(
            bldCfgRenderLabel(STRDEF("line"), true, STRDEF("label")),
            "line                                                                                                                  "
                "      // label",
            "render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("render without label due to long line");

        TEST_RESULT_STR_Z(
            bldCfgRenderLabel(
                STRDEF(
                    "12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                        "12345678901234567890"),
                true, STRDEF("label")),
            "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678"
                "901234567890",
            "render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("command parse errors");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            "command:\n"
            "  archive-get:\n"
            "    bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown command definition 'bogus'");

        #define TEST_COMMAND_VALID                                                                                                 \
            "command:\n"                                                                                                           \
            "  archive-get:\n"                                                                                                     \
            "    internal: true\n"                                                                                                 \
            "\n"

        #define TEST_OPTION_GROUP_VALID                                                                                            \
            "optionGroup:\n"                                                                                                       \
            "  repo: {}\n"                                                                                                         \
            "\n"

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("option parse errors");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown option definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    section: global\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'config' requires 'type'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    depend:\n"
            "      bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown depend definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    command:\n"
            "      backup:\n"
            "        bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown option command definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    depend: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency inherited from option 'bogus' before it is defined");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    command:\n"
            "      bogus: {}\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "invalid command 'bogus' in option 'config' command list");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency on undefined option 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    group: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'config' has invalid group 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  stanza:\n"
            "    type: string\n"
            "  config:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: online\n"
            "  online:\n"
            "    type: boolean\n"
            "    depend:\n"
            "      option: config\n");

        TEST_ERROR(
            bldCfgParse(storageTest), FormatError,
            "unable to resolve dependencies for option(s) 'config, online'\n"
            "HINT: are there circular dependencies?");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' must exist");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "  stanza:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: config\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' may not depend on other option");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "  stanza:\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' command 'archive-get' may not depend on other option");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: bool\n"
            "  stanza:\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n"
            "          default: false\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency default invalid for non-boolean option 'stanza'");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("parse and render config");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            "command:\n"
            "  archive-get:\n"
            "    command-role:\n"
            "      async: {}\n"
            "      local: {}\n"
            "      remote: {}\n"
            "    lock-type: archive\n"
            "    log-file: false\n"
            "    log-level-default: debug\n"
            "    parameter-allowed: true\n"
            "\n"
            "  backup:\n"
            "    internal: true\n"
            "    command-role:\n"
            "      local: {}\n"
            "      remote: {}\n"
            "    lock-type: backup\n"
            "    lock-required: true\n"
            "    lock-remote-required: true\n"
            "\n"
            "  help: {}\n"
            "\n"
            "  version: {}\n"
            "\n"
            "optionGroup:\n"
            "  pg: {}\n"
            "  repo: {}\n"
            "\n"
            "option:\n"
            "  timeout:\n"
            "    type: time\n"
            "    default: 10\n"
            "    command:\n"
            "      backup:\n"
            "        default: 20\n"
            "      archive-get:\n"
            "        default: 30\n"
            "    allow-range: [5, 50]\n"
            "\n"
            "  buffer-size:\n"
            "    section: global\n"
            "    type: size\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        default: 32768\n"
            "        allow-list:\n"
            "          - 32768\n"
            "    allow-list:\n"
            "      - 8192\n"
            "      - 16384\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  compress-network:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    command:\n"
            "      backup:\n"
            "        allow-list:\n"
            "          - gz\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  compress-type:\n"
            "    section: global\n"
            "    type: string\n"
            "    default: gz\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n"
            "    command-role: {}\n"
            "    deprecate:\n"
            "      compress: {}\n"
            "\n"
            "  compress-level:\n"
            "    section: global\n"
            "    type: integer\n"
            "    required: false\n"
            "    allow-range: [0, 9]\n"
            "    command: compress-type\n"
            "    depend: compress-type\n"
            "\n"
            "  compress-level-network:\n"
            "    inherit: compress-level\n"
            "    internal: true\n"
            "    secure: true\n"
            "    default: ~\n"
            "    depend:\n"
            "      option: compress-type\n"
            "      list:\n"
            "        - none\n"
            "        - gz\n"
            "\n"
            "  config:\n"
            "    type: string\n"
            "    default: CFGOPTDEF_CONFIG_PATH \"/\" PROJECT_CONFIG_FILE\n"
            "    default-literal: true\n"
            "    negate: true\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        default: CFGOPTDEF_CONFIG_PATH \"/.\" PROJECT_CONFIG_FILE\n"
            "\n"
            "  config-include:\n"
            "    section: global\n"
            "    type: path\n"
            "    default: /include\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  log-level-console:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    default: warn\n"
            "    allow-list:\n"
            "      - off\n"
            "      - error\n"
            "      - warn\n"
            "      - debug1\n"
            "\n"
            "  log-level-file:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    default: info\n"
            "    allow-list: log-level-console\n"
            "    command:\n"
            "      backup:\n"
            "        internal: true\n"
            "        required: false\n"
            "        default: warn\n"
            "        allow-list:\n"
            "          - off\n"
            "          - warn\n"
            "        depend:\n"
            "          option: log-level-console\n"
            "          list:\n"
            "            - warn\n"
            "        command-role:\n"
            "          main: {}\n"
            "      help: {}\n"
            "      archive-get:\n"
            "        default: ~\n"
            "\n"
            "  stanza:\n"
            "    type: list\n"
            "    command:\n"
            "      archive-get: {}\n"
            "      backup: {}\n"
            "\n"
            "  online:\n"
            "    type: boolean\n"
            "    default: true\n"
            "    command:\n"
            "      backup:\n"
            "        default: false\n"
            "    command-role:\n"
            "      main: {}\n"
            "    deprecate:\n"
            "      online-old: {}\n"
            "\n"
            "  backup-standby:\n"
            "    section: global\n"
            "    type: boolean\n"
            "    default: false\n"
            "    depend:\n"
            "      option: online\n"
            "      default: true\n"
            "      list:\n"
            "        - true\n"
            "    command:\n"
            "      backup: {}\n"
            "    command-role:\n"
            "      async: {}\n"
            "      main: {}\n"
            "    deprecate:\n"
            "      backup-standby-old: {}\n"
            "\n"
            "  pg-path:\n"
            "    section: stanza\n"
            "    group: pg\n"
            "    type: hash\n"
            "    deprecate:\n"
            "      pg-path: {}\n"
            "      pg?-path-indexed: {}\n"
            "      db-path: {}\n"
            "      db?-path: {}\n"
            "\n"
            "  pg-host:\n"
            "    section: stanza\n"
            "    group: pg\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        default: host1\n"
            "      backup: {}\n"
            "    deprecate:\n"
            "      pg?-host: {}\n");

        TEST_RESULT_VOID(bldCfgRender(storageTest, bldCfgParse(storageTest), false), "parse and render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("bldCfgRenderVar128Size()");

        TEST_RESULT_UINT(bldCfgRenderVar128Size(10000), 2, "check size");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("check config.auto.h");

        TEST_STORAGE_GET(
            storageTest,
            "src/config/config.auto.h",
            COMMENT_BLOCK_BEGIN "\n"
            "Command and Option Configuration\n"
            "\n"
            "Automatically generated by 'make build-config' -- do not modify directly.\n"
            COMMENT_BLOCK_END "\n"
            "#ifndef CONFIG_CONFIG_AUTO_H\n"
            "#define CONFIG_CONFIG_AUTO_H\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGCMD_ARCHIVE_GET                                          \"archive-get\"\n"
            "#define CFGCMD_BACKUP                                               \"backup\"\n"
            "#define CFGCMD_HELP                                                 \"help\"\n"
            "#define CFGCMD_VERSION                                              \"version\"\n"
            "\n"
            "#define CFG_COMMAND_TOTAL                                           4\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFG_OPTION_GROUP_TOTAL                                      2\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGOPT_BACKUP_STANDBY                                       \"backup-standby\"\n"
            "#define CFGOPT_BUFFER_SIZE                                          \"buffer-size\"\n"
            "#define CFGOPT_COMPRESS_LEVEL                                       \"compress-level\"\n"
            "#define CFGOPT_COMPRESS_LEVEL_NETWORK                               \"compress-level-network\"\n"
            "#define CFGOPT_COMPRESS_NETWORK                                     \"compress-network\"\n"
            "#define CFGOPT_COMPRESS_TYPE                                        \"compress-type\"\n"
            "#define CFGOPT_CONFIG                                               \"config\"\n"
            "#define CFGOPT_CONFIG_INCLUDE                                       \"config-include\"\n"
            "#define CFGOPT_LOG_LEVEL_CONSOLE                                    \"log-level-console\"\n"
            "#define CFGOPT_LOG_LEVEL_FILE                                       \"log-level-file\"\n"
            "#define CFGOPT_ONLINE                                               \"online\"\n"
            "#define CFGOPT_STANZA                                               \"stanza\"\n"
            "#define CFGOPT_TIMEOUT                                              \"timeout\"\n"
            "\n"
            "#define CFG_OPTION_TOTAL                                            15\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option value constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGOPTVAL_COMPRESS_NETWORK_GZ                               STRID5(\"gz\", 0x3470)\n"
            "#define CFGOPTVAL_COMPRESS_NETWORK_GZ_Z                             \"gz\"\n"
            "\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_DEBUG1                          STRID6(\"debug1\", 0x7475421441)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_DEBUG1_Z                        \"debug1\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_ERROR                           STRID5(\"error\", 0x127ca450)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_ERROR_Z                         \"error\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_OFF                             STRID5(\"off\", 0x18cf0)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_OFF_Z                           \"off\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_WARN                            STRID5(\"warn\", 0x748370)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_WARN_Z                          \"warn\"\n"
            "\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_DEBUG1                             STRID6(\"debug1\", 0x7475421441)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_DEBUG1_Z                           \"debug1\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_ERROR                              STRID5(\"error\", 0x127ca450)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_ERROR_Z                            \"error\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_OFF                                STRID5(\"off\", 0x18cf0)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_OFF_Z                              \"off\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_WARN                               STRID5(\"warn\", 0x748370)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_WARN_Z                             \"warn\"\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgCmdArchiveGet,\n"
            "    cfgCmdBackup,\n"
            "    cfgCmdHelp,\n"
            "    cfgCmdVersion,\n"
            "    cfgCmdNone,\n"
            "} ConfigCommand;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgOptGrpPg,\n"
            "    cfgOptGrpRepo,\n"
            "} ConfigOptionGroup;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgOptBackupStandby,\n"
            "    cfgOptBufferSize,\n"
            "    cfgOptCompressLevel,\n"
            "    cfgOptCompressLevelNetwork,\n"
            "    cfgOptCompressNetwork,\n"
            "    cfgOptCompressType,\n"
            "    cfgOptConfig,\n"
            "    cfgOptConfigInclude,\n"
            "    cfgOptLogLevelConsole,\n"
            "    cfgOptLogLevelFile,\n"
            "    cfgOptOnline,\n"
            "    cfgOptPgHost,\n"
            "    cfgOptPgPath,\n"
            "    cfgOptStanza,\n"
            "    cfgOptTimeout,\n"
            "} ConfigOption;\n"
            "\n"
            "#endif\n");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("check parse.auto.c.inc");

        TEST_STORAGE_GET(
            storageTest,
            "src/config/parse.auto.c.inc",
            COMMENT_BLOCK_BEGIN "\n"
            "Config Parse Rules\n"
            "\n"
            "Automatically generated by 'make build-config' -- do not modify directly.\n"
            COMMENT_BLOCK_END "\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Strings\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_STR(value)                                   PARSE_RULE_U32_1(value)\n"
            "\n"
            "static const StringPub parseRuleValueStr[] =\n"
            "{\n"
            "    PARSE_RULE_STRPUB(\"/include\"),\n"
            "    PARSE_RULE_STRPUB(\"10\"),\n"
            "    PARSE_RULE_STRPUB(\"20\"),\n"
            "    PARSE_RULE_STRPUB(\"30\"),\n"
            "    PARSE_RULE_STRPUB(\"32768\"),\n"
            "    PARSE_RULE_STRPUB(\"gz\"),\n"
            "    PARSE_RULE_STRPUB(\"host1\"),\n"
            "    PARSE_RULE_STRPUB(\"info\"),\n"
            "    PARSE_RULE_STRPUB(\"warn\"),\n"
            "    PARSE_RULE_STRPUB(CFGOPTDEF_CONFIG_PATH \"/\" PROJECT_CONFIG_FILE),\n"
            "    PARSE_RULE_STRPUB(CFGOPTDEF_CONFIG_PATH \"/.\" PROJECT_CONFIG_FILE),\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValStrQT_FS_include_QT,\n"
            "    parseRuleValStrQT_10_QT,\n"
            "    parseRuleValStrQT_20_QT,\n"
            "    parseRuleValStrQT_30_QT,\n"
            "    parseRuleValStrQT_32768_QT,\n"
            "    parseRuleValStrQT_gz_QT,\n"
            "    parseRuleValStrQT_host1_QT,\n"
            "    parseRuleValStrQT_info_QT,\n"
            "    parseRuleValStrQT_warn_QT,\n"
            "    parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_QT_SP_PROJECT_CONFIG_FILE,\n"
            "    parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_DT_QT_SP_PROJECT_CONFIG_FILE,\n"
            "} ParseRuleValueStr;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule StringIds\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_STRID(value)                                 PARSE_RULE_U32_1(value)\n"
            "\n"
            "static const StringId parseRuleValueStrId[] =\n"
            "{\n"
            "    STRID6(\"debug1\", 0x7475421441),\n"
            "    STRID5(\"error\", 0x127ca450),\n"
            "    STRID5(\"gz\", 0x3470),\n"
            "    STRID5(\"info\", 0x799c90),\n"
            "    STRID5(\"off\", 0x18cf0),\n"
            "    STRID5(\"warn\", 0x748370),\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValStrIdDebug1,\n"
            "    parseRuleValStrIdError,\n"
            "    parseRuleValStrIdGz,\n"
            "    parseRuleValStrIdInfo,\n"
            "    parseRuleValStrIdOff,\n"
            "    parseRuleValStrIdWarn,\n"
            "} ParseRuleValueStrId;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Ints\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_INT(value)                                   PARSE_RULE_U32_1(value)\n"
            "\n"
            "static const int64_t parseRuleValueInt[] =\n"
            "{\n"
            "    0,\n"
            "    9,\n"
            "    5000,\n"
            "    8192,\n"
            "    10000,\n"
            "    16384,\n"
            "    20000,\n"
            "    30000,\n"
            "    32768,\n"
            "    50000,\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValInt0,\n"
            "    parseRuleValInt9,\n"
            "    parseRuleValInt5000,\n"
            "    parseRuleValInt8192,\n"
            "    parseRuleValInt10000,\n"
            "    parseRuleValInt16384,\n"
            "    parseRuleValInt20000,\n"
            "    parseRuleValInt30000,\n"
            "    parseRuleValInt32768,\n"
            "    parseRuleValInt50000,\n"
            "} ParseRuleValueInt;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command parse data\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_CMD(value)                                   PARSE_RULE_U32_1(value)\n"
            "\n"
            "static const ParseRuleCommand parseRuleCommand[CFG_COMMAND_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"archive-get\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(lockTypeArchive),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(logLevelDebug),\n"
            "        PARSE_RULE_COMMAND_PARAMETER_ALLOWED(true),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleAsync)\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleLocal)\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleMain)\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleRemote)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"backup\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_REQUIRED(true),\n"
            "        PARSE_RULE_COMMAND_LOCK_REMOTE_REQUIRED(true),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(lockTypeBackup),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(logLevelInfo),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleLocal)\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleMain)\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleRemote)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"help\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(lockTypeNone),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(logLevelInfo),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleMain)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"version\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(lockTypeNone),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(logLevelInfo),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(cfgCmdRoleMain)\n"
            "        ),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group parse data\n"
            COMMENT_BLOCK_END "\n"
            "static const ParseRuleOptionGroup parseRuleOptionGroup[CFG_OPTION_GROUP_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_OPTION_GROUP\n"
            "    (\n"
            "        PARSE_RULE_OPTION_GROUP_NAME(\"pg\"),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION_GROUP\n"
            "    (\n"
            "        PARSE_RULE_OPTION_GROUP_NAME(\"repo\"),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option parse data\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_OPT(value)                                   PARSE_RULE_U32_1(value)\n"
            "\n"
            "static const ParseRuleOption parseRuleOption[CFG_OPTION_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"backup-standby\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeBoolean),\n"
            "        PARSE_RULE_OPTION_NEGATE(true),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_OPTIONAL_DEPEND_DEFAULT(PARSE_RULE_VAL_BOOL_TRUE),\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptOnline),\n"
            "                    PARSE_RULE_VAL_BOOL_TRUE,\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_FALSE,\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"buffer-size\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeSize),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt32768),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt32768),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_32768_QT),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt8192),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt16384),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-level\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeInteger),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(false),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptConfig),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt0),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt9),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt0),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt9),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-level-network\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeInteger),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(false),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "        PARSE_RULE_OPTION_SECURE(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptConfig),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt0),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt9),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptCompressType),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdNone),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdGz),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt0),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt9),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-network\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeStringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdBackup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdGz),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-type\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeString),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptConfig),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_gz_QT),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_gz_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"config\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeString),\n"
            "        PARSE_RULE_OPTION_NEGATE(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionCommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_DT_QT_SP_PROJECT_CONFIG_FILE),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_QT_SP_PROJECT_CONFIG_FILE),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"config-include\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypePath),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_FS_include_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"log-level-console\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeStringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdOff),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdError),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdDebug1),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_warn_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"log-level-file\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeStringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionGlobal),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdHelp)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdBackup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(cfgOptLogLevelConsole),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdOff),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_warn_QT),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_NOT_REQUIRED(),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdOff),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdError),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdWarn),\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdDebug1),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(parseRuleValStrIdInfo),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_info_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"online\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeBoolean),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionCommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdBackup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_FALSE,\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_TRUE,\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"pg-host\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeString),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionStanza),\n"
            "        PARSE_RULE_OPTION_GROUP_MEMBER(true),\n"
            "        PARSE_RULE_OPTION_GROUP_ID(cfgOptGrpPg),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_host1_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"pg-path\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeHash),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionStanza),\n"
            "        PARSE_RULE_OPTION_MULTI(true),\n"
            "        PARSE_RULE_OPTION_GROUP_MEMBER(true),\n"
            "        PARSE_RULE_OPTION_GROUP_ID(cfgOptGrpPg),\n"
            "        PARSE_RULE_OPTION_DEPRECATE_MATCH(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"stanza\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeList),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionCommandLine),\n"
            "        PARSE_RULE_OPTION_MULTI(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"timeout\"),\n"
            "        PARSE_RULE_OPTION_TYPE(cfgOptTypeTime),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(cfgSectionCommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(cfgCmdBackup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt5000),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt50000),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt30000),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_30_QT),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(cfgCmdBackup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt5000),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt50000),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt20000),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_20_QT),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt5000),\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt50000),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(parseRuleValInt10000),\n"
            "                    PARSE_RULE_VAL_STR(parseRuleValStrQT_10_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option deprecations\n"
            COMMENT_BLOCK_END "\n"
            "#define CFG_OPTION_DEPRECATE_TOTAL                                  7\n"
            "\n"
            "static const ParseRuleOptionDeprecate parseRuleOptionDeprecate[CFG_OPTION_DEPRECATE_TOTAL] =\n"
            "{\n"
            "    {\n"
            "        .name = \"backup-standby-old\",\n"
            "        .id = cfgOptBackupStandby,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"compress\",\n"
            "        .id = cfgOptCompressType,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"db-path\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .indexed = true,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"online-old\",\n"
            "        .id = cfgOptOnline,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-host\",\n"
            "        .id = cfgOptPgHost,\n"
            "        .indexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-path\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-path-indexed\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .indexed = true,\n"
            "    },\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Order for option parse resolution\n"
            COMMENT_BLOCK_END "\n"
            "static const uint8_t optionResolveOrder[] =\n"
            "{\n"
            "    cfgOptStanza,\n"
            "    cfgOptBufferSize,\n"
            "    cfgOptCompressNetwork,\n"
            "    cfgOptConfig,\n"
            "    cfgOptConfigInclude,\n"
            "    cfgOptLogLevelConsole,\n"
            "    cfgOptLogLevelFile,\n"
            "    cfgOptOnline,\n"
            "    cfgOptPgHost,\n"
            "    cfgOptPgPath,\n"
            "    cfgOptTimeout,\n"
            "    cfgOptBackupStandby,\n"
            "    cfgOptCompressLevel,\n"
            "    cfgOptCompressType,\n"
            "    cfgOptCompressLevelNetwork,\n"
            "};\n");
    }

    FUNCTION_HARNESS_RETURN_VOID();
}
