/*
   Copyright (c) 2003, 2023, Oracle and/or its affiliates.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License, version 2.0,
   as published by the Free Software Foundation.

   This program is also distributed with certain software (including
   but not limited to OpenSSL) that is licensed under separate terms,
   as designated in a particular file or component or in included license
   documentation.  The authors of MySQL hereby grant you an additional
   permission to link the program and your derivative works with the
   separately licensed software that they have included with MySQL.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License, version 2.0, for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
*/

#define DBLQH_C
#include <ndb_limits.h>
#include <algorithm>
#include <cstring>
#include <md5_hash.hpp>
#include "Dblqh.hpp"

#include <ndb_version.h>
#include <signaldata/AccFrag.hpp>
#include <signaldata/AccKeyReq.hpp>
#include <signaldata/AccScan.hpp>
#include <signaldata/CopyActive.hpp>
#include <signaldata/CopyFrag.hpp>
#include <signaldata/CreateTrigImpl.hpp>
#include <signaldata/DropTrigImpl.hpp>
#include <signaldata/DumpStateOrd.hpp>
#include <signaldata/EventReport.hpp>
#include <signaldata/ExecFragReq.hpp>
#include <signaldata/GCP.hpp>
#include <signaldata/LqhFrag.hpp>
#include <signaldata/LqhKey.hpp>
#include <signaldata/LqhTransReq.hpp>
#include <signaldata/NFCompleteRep.hpp>
#include <signaldata/NextScan.hpp>
#include <signaldata/NodeFailRep.hpp>
#include <signaldata/NodeRecoveryStatusRep.hpp>
#include <signaldata/PackedSignal.hpp>
#include <signaldata/ReadNodesConf.hpp>
#include <signaldata/RelTabMem.hpp>
#include <signaldata/ScanFrag.hpp>
#include <signaldata/SrFragidConf.hpp>
#include <signaldata/StartFragReq.hpp>
#include <signaldata/StartRec.hpp>
#include <signaldata/TcKeyRef.hpp>
#include <signaldata/TupCommit.hpp>
#include <signaldata/TupFrag.hpp>
#include <signaldata/TupKey.hpp>
#include <signaldata/TuxBound.hpp>

#include <signaldata/CreateTab.hpp>
#include <signaldata/CreateTable.hpp>
#include <signaldata/DropTab.hpp>
#include <signaldata/DropTable.hpp>
#include <signaldata/PrepDropTab.hpp>

#include <signaldata/AlterTab.hpp>
#include <signaldata/AlterTable.hpp>
#include <signaldata/DictTabInfo.hpp>

#include <NdbEnv.h>
#include <NdbSpin.h>
#include <Checksum.hpp>
#include <DebuggerNames.hpp>
#include <KeyDescriptor.hpp>
#include <SectionReader.hpp>
#include <signaldata/AttrInfo.hpp>
#include <signaldata/BackupImpl.hpp>
#include <signaldata/DbinfoScan.hpp>
#include <signaldata/FireTrigOrd.hpp>
#include <signaldata/FsReadWriteReq.hpp>
#include <signaldata/FsRef.hpp>
#include <signaldata/IsolateOrd.hpp>
#include <signaldata/KeyInfo.hpp>
#include <signaldata/LCP.hpp>
#include <signaldata/LocalSysfile.hpp>
#include <signaldata/RestoreImpl.hpp>
#include <signaldata/RouteOrd.hpp>
#include <signaldata/SignalDroppedRep.hpp>
#include <signaldata/SystemError.hpp>
#include <signaldata/TransIdAI.hpp>
#include <signaldata/UndoLogLevel.hpp>

#include "../suma/Suma.hpp"
#include "DblqhCommon.hpp"
#include "portlib/mt-asm.h"

#include "../backup/Backup.hpp"
#include "../dbtux/Dbtux.hpp"

/**
 * overload handling...
 * TODO: cleanup...from all sorts of perspective
 */
#include <TransporterRegistry.hpp>

#include <EventLogger.hpp>

#if (defined(VM_TRACE) || defined(ERROR_INSERT))
//#define DEBUG_COPY_ACTIVE 1
//#define DEBUG_LOCAL_LCP 1
//#define DEBUG_EMPTY_LCP 1
//#define DEBUG_LAST_LCP 1
//#define DEBUG_NEWEST_GCI 1
//#define ABORT_TRACE 1
//#define DO_TRANSIENT_POOL_STAT 1
//#define DEBUG_EXTRA_LCP 1
//#define DEBUG_LCP 1
//#define DEBUG_LCP_RESTORE
//#define DEBUG_COPY 1
//#define DEBUG_GCP 1
//#define DEBUG_CUT_REDO 1
//#define DEBUG_LOCAL_LCP_EXTRA 1
//#define DEBUG_REDO_FLAG 1
//#define DEBUG_TRANSACTION_TIMEOUT 1
//#define DEBUG_SCHEMA_VERSION 1
//#define DEBUG_EARLY_LCP 1
//#define DEBUG_NODE_STATUS 1
//#define DEBUG_INDEX_BUILD 1
//#define DEBUG_RESTART_SYNCH 1
//#define DEBUG_START_PHASE9 1
//#define DEBUG_COMMITTED_WORDS 1
//#define DEBUG_RESTORE 1
//#define NDB_DEBUG_REDO_EXEC 1
//#define NDB_DEBUG_REDO_REC 1
#endif

#ifdef DEBUG_NEWEST_GCI
#define DEB_NEWEST_GCI(arglist)  \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_NEWEST_GCI(arglist) \
  do {                          \
  } while (0)
#endif

#ifdef DEBUG_LAST_LCP
#define DEB_LAST_LCP(arglist)    \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_LAST_LCP(arglist) \
  do {                        \
  } while (0)
#endif

#ifdef DEBUG_COPY_ACTIVE
#define DEB_COPY_ACTIVE(arglist) \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_COPY_ACTIVE(arglist) \
  do {                           \
  } while (0)
#endif

#ifdef DEBUG_EMPTY_LCP
#define DEB_EMPTY_LCP(arglist)   \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_EMPTY_LCP(arglist) \
  do {                         \
  } while (0)
#endif

#ifdef DEBUG_RESTORE
#define DEB_RESTORE(arglist)     \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_RESTORE(arglist) \
  do {                       \
  } while (0)
#endif

#ifdef DEBUG_COMMITTED_WORDS
#define DEB_COMMITTED_WORDS(arglist) \
  do {                               \
    g_eventLogger->info arglist;     \
  } while (0)
#else
#define DEB_COMMITTED_WORDS(arglist) \
  do {                               \
  } while (0)
#endif

#ifdef DEBUG_START_PHASE9
#define DEB_START_PHASE9(arglist) \
  do {                            \
    g_eventLogger->info arglist;  \
  } while (0)
#else
#define DEB_START_PHASE9(arglist) \
  do {                            \
  } while (0)
#endif

#ifdef DEBUG_RESTART_SYNCH
#define DEB_RESTART_SYNCH(arglist) \
  do {                             \
    g_eventLogger->info arglist;   \
  } while (0)
#else
#define DEB_RESTART_SYNCH(arglist) \
  do {                             \
  } while (0)
#endif

#ifdef DEBUG_INDEX_BUILD
#define DEB_INDEX_BUILD(arglist) \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_INDEX_BUILD(arglist) \
  do {                           \
  } while (0)
#endif

#ifdef DEBUG_NODE_STATUS
#define DEB_NODE_STATUS(arglist) \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_NODE_STATUS(arglist) \
  do {                           \
  } while (0)
#endif

#ifdef DEBUG_EARLY_LCP
#define DEB_EARLY_LCP(arglist)   \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_EARLY_LCP(arglist) \
  do {                         \
  } while (0)
#endif

#ifdef DEBUG_SCHEMA_VERSION
#define DEB_SCHEMA_VERSION(arglist) \
  do {                              \
    g_eventLogger->info arglist;    \
  } while (0)
#else
#define DEB_SCHEMA_VERSION(arglist) \
  do {                              \
  } while (0)
#endif

#ifdef DEBUG_EXTRA_LCP
#define DEB_EXTRA_LCP(arglist)   \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_EXTRA_LCP(arglist) \
  do {                         \
  } while (0)
#endif

#ifdef DEBUG_LCP
#define DEB_LCP(arglist)         \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_LCP(arglist) \
  do {                   \
  } while (0)
#endif

#ifdef DEBUG_LCP_RESTORE
#define DEB_LCP_RESTORE(arglist) \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_LCP_RESTORE(arglist) \
  do {                           \
  } while (0)
#endif

#ifdef DEBUG_COPY
#define DEB_COPY(arglist)        \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_COPY(arglist) \
  do {                    \
  } while (0)
#endif

#ifdef DEBUG_GCP
#define DEB_GCP(arglist)         \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_GCP(arglist) \
  do {                   \
  } while (0)
#endif

#ifdef DEBUG_CUT_REDO
#define DEB_CUT_REDO(arglist)    \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_CUT_REDO(arglist) \
  do {                        \
  } while (0)
#endif

#ifdef DEBUG_LOCAL_LCP
#define DEB_LOCAL_LCP(arglist)   \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_LOCAL_LCP(arglist) \
  do {                         \
  } while (0)
#endif

#ifdef DEBUG_LOCAL_LCP_EXTRA
#define DEB_LOCAL_LCP_EXTRA(arglist) \
  do {                               \
    g_eventLogger->info arglist;     \
  } while (0)
#else
#define DEB_LOCAL_LCP_EXTRA(arglist) \
  do {                               \
  } while (0)
#endif

#ifdef DEBUG_REDO_FLAG
#define DEB_REDO(arglist)        \
  do {                           \
    g_eventLogger->info arglist; \
  } while (0)
#else
#define DEB_REDO(arglist) \
  do {                    \
  } while (0)
#endif

// Use LQH_DEBUG to print messages that should be
// seen only when we debug the product
//#define USE_LQH_DEBUG
#ifdef USE_LQH_DEBUG
#define LQH_DEBUG(x) ndbout << "DBLQH: " << x << endl;
static NdbOut &operator<<(NdbOut &out,
                          Dblqh::TcConnectionrec::TransactionState state) {
  out << (int)state;
  return out;
}

static NdbOut &operator<<(NdbOut &out,
                          Dblqh::TcConnectionrec::LogWriteState state) {
  out << (int)state;
  return out;
}

static NdbOut &operator<<(NdbOut &out,
                          Dblqh::TcConnectionrec::AbortState state) {
  out << (int)state;
  return out;
}

static NdbOut &operator<<(NdbOut &out, Dblqh::ScanRecord::ScanState state) {
  out << (int)state;
  return out;
}

#ifdef DEB_TRANSACTION_TIMEOUT
static NdbOut &operator<<(NdbOut &out,
                          Dblqh::LogFileOperationRecord::LfoState state) {
  out << (int)state;
  return out;
}
#endif
static NdbOut &operator<<(NdbOut &out, Dblqh::ScanRecord::ScanType state) {
  out << (int)state;
  return out;
}

static NdbOut &operator<<(NdbOut &out, Operation_t op) {
  switch (op) {
    case ZREAD:
      out << "READ";
      break;
    case ZREAD_EX:
      out << "READ-EX";
      break;
    case ZINSERT:
      out << "INSERT";
      break;
    case ZUPDATE:
      out << "UPDATE";
      break;
    case ZDELETE:
      out << "DELETE";
      break;
    case ZWRITE:
      out << "WRITE";
      break;
    case ZUNLOCK:
      out << "UNLOCK";
      break;
    case ZREFRESH:
      out << "REFRESH";
      break;
  }
  return out;
}

#else
#define LQH_DEBUG(x)
#endif

//#define MARKER_TRACE 0
//#define TRACE_SCAN_TAKEOVER 1

#ifdef NDB_DEBUG_REDO_REC
static int DEBUG_REDO_REC = 1;
#else
#define DEBUG_REDO_REC 0
#endif

#ifdef NDB_DEBUG_REDO_EXEC
static int DEBUG_REDO_EXEC = 1;
#else
#define DEBUG_REDO_EXEC 0
#endif

#define DELAY_CHECK_SYSTEM_SCANS 10000
/**
 * System reserved scan ids. Scan ids 0-11 are specific for ACC scans.
 * Scan ids from 12 and up to a maximum of 133 (configurable) are used
 * for TUX range scans and finally scan ids of from last TUX range scan
 * id up to a maximum of 252 is used for TUP full table scans. Scan ids
 * 253, 254 and 255 are reserved for LCP scans, Backup scans and NR scans.
 */
const Uint32 NR_ScanNo = 253;
const Uint32 LCP_ScanNo = 254;
const Uint32 Backup_ScanNo = 255;

#ifndef NDBD_TRACENR
#if defined VM_TRACE
#define NDBD_TRACENR
#endif
#endif

#ifdef NDBD_TRACENR
#include <NdbConfig.h>
static NdbOut *tracenrout = 0;
static int TRACENR_FLAG = 0;
#define TRACENR(x) (*tracenrout) << x
#define SET_TRACENR_FLAG TRACENR_FLAG = 1
#define CLEAR_TRACENR_FLAG TRACENR_FLAG = 0
#else
#define TRACENR_FLAG 0
#define TRACENR(x) \
  do {             \
  } while (0)
#define SET_TRACENR_FLAG
#define CLEAR_TRACENR_FLAG
#endif

#define JAM_FILE_ID 451

#ifdef NDBD_TRACENR
static NdbOut *traceopout = 0;
#define TRACE_OP(regTcPtr, place)                                 \
  do {                                                            \
    if (TRACE_OP_CHECK(regTcPtr)) TRACE_OP_DUMP(regTcPtr, place); \
  } while (0)
#else
#define TRACE_OP(x, y) \
  { (void)x; }
#endif

static NdbMutex m_restart_synch_mutex;
static enum RestartSynchState {
  NO_SYNCH_ONGOING = 0,
  START_PHASE3_SYNCH = 1,
  START_LOG_LIMITS_SYNCH = 2,
  COMPLETED_REDO_LOG_EXEC_SYNCH = 3,
  COMPLETED_REDO_FOURTH_PHASE_SYNCH = 4,
  START_SEND_EXEC_CONF = 5,
} m_restart_synch_state;
static Uint32 m_restart_synch_ready;

struct LogPosition {
  Uint32 m_file_no;
  Uint32 m_mbyte;
};

int cmp(const LogPosition &pos1, const LogPosition &pos2) {
  if (pos1.m_file_no > pos2.m_file_no) return 1;
  if (pos1.m_file_no < pos2.m_file_no) return -1;
  if (pos1.m_mbyte > pos2.m_mbyte) return 1;
  if (pos1.m_mbyte < pos2.m_mbyte) return -1;

  return 0;
}

/**
 * head - tail
 */
static Uint64 free_log(const LogPosition &head, const LogPosition &tail,
                       Uint32 cnt, Uint32 size) {
  Uint64 headmb = head.m_file_no * Uint64(size) + head.m_mbyte;
  Uint64 tailmb = tail.m_file_no * Uint64(size) + tail.m_mbyte;
  if (headmb >= tailmb) {
    return (cnt * Uint64(size)) - headmb + tailmb;
  } else {
    return tailmb - headmb;
  }
}

/* ------------------------------------------------------------------------- */
/* -------               SEND SYSTEM ERROR                           ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::systemError(Signal *signal, int line) {
  signal->theData[0] = DumpStateOrd::LqhSystemError;
  execDUMP_STATE_ORD(signal);
  progError(line, NDBD_EXIT_NDBREQUIRE);
}  // Dblqh::systemError()

/* *************** */
/*  ACCSEIZEREF  > */
/* *************** */
void Dblqh::execACCSEIZEREF(Signal *signal) {
  jamEntry();
  ndbabort();
}  // Dblqh::execACCSEIZEREF()

void Dblqh::send_handle_queued_log_write(Signal *signal,
                                         TcConnectionrec *tcPtrP,
                                         LogPartRecord *logPartPtrP,
                                         Uint32 instanceNo) {
  BlockReference ref = numberToRef(DBLQH, instanceNo, getOwnNodeId());
  signal->theData[0] = ZCONTINUE_WRITE_LOG;
  signal->theData[1] = tcPtrP->ptrI;
  signal->theData[2] = tcPtrP->transid[0];
  signal->theData[3] = tcPtrP->transid[1];
  switch (tcPtrP->transactionState) {
    case TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL:
    case TcConnectionrec::LOG_COMMIT_QUEUED: {
      jam();
      signal->theData[4] = ZCOMMIT_LOG_SIZE;
      logPartPtrP->m_booked_redo_log_space += ZCOMMIT_LOG_SIZE;
      break;
    }
    case TcConnectionrec::LOG_ABORT_QUEUED: {
      jam();
      signal->theData[4] = ZABORT_LOG_SIZE;
      logPartPtrP->m_booked_redo_log_space += ZABORT_LOG_SIZE;
      break;
    }
    case TcConnectionrec::LOG_QUEUED: {
      jam();
      Uint32 booked_space =
          tcPtrP->currTupAiLen + tcPtrP->primKeyLen + ZLOG_HEAD_SIZE;
      signal->theData[4] = booked_space;
      logPartPtrP->m_booked_redo_log_space += booked_space;
      break;
    }
    default: {
      ndbabort();
    }
  }
  sendSignal(ref, GSN_CONTINUEB, signal, 5, JBB);
}

void Dblqh::prepare_queued_log_write(Signal *signal, TcConnectionrec *tcPtrP,
                                     LogPartRecord *logPartPtrP,
                                     Uint32 booked_space) {
  /* Holds logPartPtrP->m_log_part_mutex at entry of this function */
  ndbrequire(logPartPtrP->m_booked_redo_log_space >= booked_space);
  logPartPtrP->m_booked_redo_log_space -= booked_space;

  if (logPartPtrP->m_log_prepare_queue.isEmpty() &&
      logPartPtrP->m_log_complete_queue.isEmpty() &&
      (logPartPtrP->waitWriteGciLog != LogPartRecord::WWGL_TRUE) &&
      logPartPtrP->m_booked_redo_log_space == 0) {
    jam();
    logPartPtrP->logPartState = LogPartRecord::IDLE;
  }
}

void Dblqh::handle_queued_log_write(Signal *signal, LogPartRecord *logPartPtrP,
                                    TcConnectionrecPtr tcConnectptr) {
  /* Holds logPartPtrP->m_log_part_mutex at entry of this function */
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  logFilePtr.i = logPartPtrP->currentLogfile;
  ptrCheckGuard(logFilePtr, logPartPtrP->my_block->clogFileFileSize,
                logPartPtrP->my_block->logFileRecord);
  logPagePtr.i = logFilePtr.p->currentLogpage;
  ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                logPartPtrP->logPageRecord);

  fragptr.i = tcConnectptr.p->fragmentptr;
  c_fragment_pool.getPtr(fragptr);

  // so that operation can continue...
  switch (tcConnectptr.p->transactionState) {
    case TcConnectionrec::LOG_QUEUED: {
      if (tcConnectptr.p->abortState != TcConnectionrec::ABORT_IDLE) {
        jam();
        /**
         * No need to log the ABORT in the REDO log since no PREPARE
         * log record have been written.
         */
        unlock_log_part(logPartPtrP);
        abortCommonLab(signal, tcConnectptr);
      } else {
        jam();
        doWritePrepareLog(signal, tcConnectptr);
      }
      return;
    }
    case TcConnectionrec::LOG_ABORT_QUEUED: {
      jam();
      writeAbortLog(signal, tcConnectptr.p, logPagePtr, logFilePtr,
                    logPartPtrP);
      removeLogTcrec(tcConnectptr.p, logPartPtrP);
      unlock_log_part(logPartPtrP);
      continueAfterLogAbortWriteLab(signal, tcConnectptr);
      break;
    }
    case TcConnectionrec::LOG_COMMIT_QUEUED:
    case TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL: {
      jam();
      writeCommitLog(signal, tcConnectptr.p, logPartPtrP);
      removeLogTcrec(tcConnectptr.p, logPartPtrP);
      unlock_log_part(logPartPtrP);
      if (tcConnectptr.p->transactionState ==
          TcConnectionrec::LOG_COMMIT_QUEUED) {
        if (tcConnectptr.p->seqNoReplica == 0 ||
            tcConnectptr.p->activeCreat == Fragrecord::AC_NR_COPY) {
          jam();
          localCommitLab(signal, tcConnectptr);
        } else {
          jam();
          commitReplyLab(signal, tcConnectptr.p);
        }
      } else {
        jam();
        tcConnectptr.p->transactionState =
            TcConnectionrec::LOG_COMMIT_WRITTEN_WAIT_SIGNAL;
      }
      return;
    }
    default: {
      ndbabort();
      return;
    }
  }  // switch
}

Uint32 Dblqh::count_free_log_pages(LogPartRecord *logPartPtrP) {
  /**
   * Since a log part owner can send signals to other LDM instances
   * instructing them to write in the REDO log we introduce a
   * booking mechanism for this. The owning log part books space
   * in the REDO log. The space booked is the size of the REDO log
   * entry.
   *
   * The sum of those bookings is used to calculate the number of
   * booked log pages. In this calculation we use the number of
   * words in a page minus the page header. In addition we deduct
   * pages that are lost due to end of MByte. Log entries cannot
   * span over MByte boundaries. Thus we can skip over up to the
   * size of a page at the end of a MByte. This is calculated as
   * the page size divided by the number of pages in a MByte. Thus
   * we deduct 256 words from each page. In addition we add one
   * more page as booked to ensure we don't get any chance of
   * assuming there is space that isn't there due to these calculations.
   * This could happen if we are close to the end of a MByte and
   * the lost page is close to us.
   *
   * By using this booking scheme we can ensure that an operation will
   * always be able to perform its writing in the REDO log and thus
   * once removed from the REDO log queue it will write into the
   * REDO log. This ensures that we can retain the order of writing
   * into the REDO log. The only prioritisation is that COMMIT and
   * ABORT log messages are prioritised over PREPARE REDO log
   * messages.
   */
  Uint64 noOfFreeLogPages = (Uint64)logPartPtrP->noOfFreeLogPages;
  Uint64 booked_log_words = logPartPtrP->m_booked_redo_log_space;
  Uint64 space_per_page =
      ZPAGE_SIZE - (ZPAGE_HEADER_SIZE + (ZPAGE_SIZE / ZPAGES_IN_MBYTE));
  Uint64 booked_log_pages = booked_log_words / space_per_page + 1;
  if (noOfFreeLogPages <= booked_log_pages) {
    jam();
    return 0;
  } else {
    jam();
    Uint64 free_pages = noOfFreeLogPages - booked_log_pages;
    return Uint32(free_pages);
  }
}

void Dblqh::queued_log_write(Signal *signal, LogPartRecord *logPartPtrP) {
  lock_log_part(logPartPtrP);
  Uint32 noOfFreeLogPages = count_free_log_pages(logPartPtrP);
  if (noOfFreeLogPages == 0) {
    jam();
    unlock_log_part(logPartPtrP);
    /**
     * REDO log buffer is full, wait 10ms before coming back to write more
     * REDO log entries to give the REDO log file system a chance to catch
     * up with the REDO log write speed.
     */
    sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 10, 2);
    return;
  }  // if

  logPartPtrP->LogLqhKeyReqSent = ZFALSE;

  do {
    if (logPartPtrP->waitWriteGciLog == LogPartRecord::WWGL_TRUE) {
      jam();
      unlock_log_part(logPartPtrP);
      /**
       * We need to write a Completed GCI entry, perform this before any
       * other action is performed. This happens in logNextStart.
       */
      break;
    }
    if (logPartPtrP->m_log_complete_queue.isEmpty()) {
      jam();
      if (logPartPtrP->m_log_prepare_queue.isEmpty()) {
        /**
         * We have already removed all entries from both queues (this can
         * happen if aborts arrive and remove entries from the prepare
         * queue). We stop checking the log queues until they fill up
         * again.
         */
        jam();
        if (logPartPtrP->m_booked_redo_log_space == 0) {
          jam();
          /**
           * Removal of entries from the Prepare REDO log queue put us into
           * an IDLE state since there is also no outstanding booked REDO
           * log writes.
           */
          logPartPtrP->logPartState = LogPartRecord::IDLE;
        }
        unlock_log_part(logPartPtrP);
        return;
      }
      /**
       * prepare is first in queue...check that it's ok to rock'n'roll
       */
      if (logPartPtrP->m_log_problems != 0 || ERROR_INSERTED(5083)) {
        /**
         * It will be restarted when problems are cleared...
         */
        jam();
        unlock_log_part(logPartPtrP);
        return;
      }
      if (noOfFreeLogPages < ZMIN_LOG_PAGES_OPERATION) {
        jam();
        logPartPtrP->LogLqhKeyReqSent = ZTRUE;
        unlock_log_part(logPartPtrP);
        /**
         * We still have a bit of room to write in the REDO log buffer.
         * But we don't write Prepare REDO log entries unless there is
         * room left for COMMIT messages to be written. We leave room
         * for almost a 1000 COMMIT log records to be written at all
         * times. COMMIT and ABORT log records are only stopped when
         * there is no room at all in the REDO log buffer.
         *
         * We will wait for 10ms to give the REDO log file system a chance
         * to catch up. During this time we can still write COMMIT and
         * ABORT messages until the REDO log buffer is full.
         */
        sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 10, 2);
        return;
      }
    }
    {
      /**
       * We use get_table_frag_record here only to get the instance
       * to ensure we call handle_queued_log_write in the correct
       * instance. We don't use tabptr and fragptr after this call.
       */
      TcConnectionrec *tcPtrP = 0;
      getFirstInLogQueue(signal, &tcPtrP, logPartPtrP);
      Uint32 instanceNo;
      ndbrequire(get_table_frag_record(tcPtrP->tableref, tcPtrP->fragmentid,
                                       tabptr, fragptr, instanceNo));
      if (instanceNo == instance()) {
        jam();
        TcConnectionrecPtr tcConnectptr;
        tcConnectptr.i = tcPtrP->ptrI;
        tcConnectptr.p = tcPtrP;
        handle_queued_log_write(signal, logPartPtrP, tcConnectptr);
      } else {
        jam();
        send_handle_queued_log_write(signal, tcPtrP, logPartPtrP, instanceNo);
        unlock_log_part(logPartPtrP);
      }
    }
  } while (0);
  {
    logNextStart(signal, logPartPtrP);
    return;
  }
}
/* ******************************************************>> */
/* THIS SIGNAL IS USED TO HANDLE REAL-TIME                  */
/* BREAKS THAT ARE NECESSARY TO ENSURE REAL-TIME            */
/* OPERATION OF LQH.                                        */
/* This signal is also used for signal loops, for example   */
/* the timeout handling for writing logs every second.      */
/* ******************************************************>> */
void Dblqh::execCONTINUEB(Signal *signal) {
  jamEntry();
  Uint32 tcase = signal->theData[0];
  Uint32 data0 = signal->theData[1];
  Uint32 data1 = signal->theData[2];
  Uint32 data2 = signal->theData[3];
  TcConnectionrecPtr tcConnectptr;
  switch (tcase) {
    case ZPRINT_MUTEX_STATS: {
      jam();
      print_fragment_mutex_stats(signal);
      return;
    }
    case ZSTART_SEND_EXEC_CONF: {
      jam();
      start_send_exec_conf(signal);
      return;
    }
    case ZCONTINUE_WRITE_LOG: {
      /**
       * While in state LOG_COMMIT_QUEUED, LOG_COMMIT_QUEUED_WAIT_SIGNAL,
       * LOG_ABORT_QUEUED, LOG_QUEUED AND also not in any log queue, no
       * one should be able to change the state of the operation.
       * We verify this here.
       */
      jam();
      TcConnectionrecPtr tcPtr;
      tcPtr.i = data0;
      ndbrequire(tcConnect_pool.getValidPtr(tcPtr));
      ndbrequire(is_same_trans(tcPtr.i, data1, data2));
      LogPartRecord *regLogPartPtr;
      regLogPartPtr = tcPtr.p->m_log_part_ptr_p;
      lock_log_part(regLogPartPtr);
      prepare_queued_log_write(signal, tcPtr.p, regLogPartPtr,
                               signal->theData[4]);
      handle_queued_log_write(signal, regLogPartPtr, tcPtr);
      return;
    }
    case ZCONTINUE_PHASE3_START: {
      jam();
      continue_srPhase3Start(signal);
      return;
    }
    case ZCONTINUE_SR_GCI_LIMITS: {
      jam();
      continue_srGciLimits(signal);
      return;
    }
    case ZCONTINUE_REDO_LOG_EXEC_COMPLETED: {
      jam();
      continue_execSrCompletedLab(signal);
      return;
    }
    case ZCONTINUE_SR_FOURTH_COMP: {
      jam();
      continue_srFourthComp(signal);
      return;
    }
    case ZPGMAN_PREP_LCP_ACTIVE_CHECK: {
      if (data1 == 0) {
        jam();
        check_pgman_prep_lcp_active_prep_drop_tab(signal, data0);
      } else {
        jam();
        check_pgman_prep_lcp_active_drop_tab(signal, data0);
      }
      return;
    }
    case ZLQH_SHRINK_TRANSIENT_POOLS: {
      jam();
      Uint32 pool_index = signal->theData[1];
      ndbassert(signal->getLength() == 2);
      shrinkTransientPools(pool_index);
      return;
    }
#if (defined(VM_TRACE) || defined(ERROR_INSERT)) && \
    defined(DO_TRANSIENT_POOL_STAT)

    case ZLQH_TRANSIENT_POOL_STAT: {
      for (Uint32 pool_index = 0; pool_index < c_transient_pool_count;
           pool_index++) {
        g_eventLogger->info(
            "DBLQH %u: Transient slot pool %u %p: Entry size %u:"
            " Free %u: Used %u: Used high %u: Size %u: For shrink %u",
            instance(), pool_index, c_transient_pools[pool_index],
            c_transient_pools[pool_index]->getEntrySize(),
            c_transient_pools[pool_index]->getNoOfFree(),
            c_transient_pools[pool_index]->getUsed(),
            c_transient_pools[pool_index]->getUsedHi(),
            c_transient_pools[pool_index]->getSize(),
            c_transient_pools_shrinking.get(pool_index));
      }
      sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 5000, 1);
      break;
    }
#endif

    case ZSTART_QUEUED_SCAN: {
      jamDebug();
      ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
      restart_queued_scan(signal, data0);
      return;
    }
    case ZCHECK_SYSTEM_SCANS: {
      handle_check_system_scans(signal);
      signal->theData[0] = ZCHECK_SYSTEM_SCANS;
      sendSignalWithDelay(cownref, GSN_CONTINUEB, signal,
                          DELAY_CHECK_SYSTEM_SCANS, 1);
      break;
    }
    case ZLOG_LQHKEYREQ: {
      LogPartRecordPtr logPartPtr;

      ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
      logPartPtr.i = data0;
      ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
      queued_log_write(signal, logPartPtr.p);
      return;
    }
    case ZSR_GCI_LIMITS:
      jam();
      signal->theData[0] = data0;
      srGciLimits(signal);
      return;
      break;
    case ZSR_LOG_LIMITS:
      jam();
      signal->theData[0] = data0;
      signal->theData[1] = data1;
      signal->theData[2] = data2;
      srLogLimits(signal);
      return;
      break;
    case ZSEND_EXEC_CONF:
      jam();
      signal->theData[0] = data0;
      sendExecConf(signal);
      return;
      break;
    case ZEXEC_SR:
      jam();
      signal->theData[0] = data0;
      execSr(signal);
      return;
      break;
    case ZSR_FOURTH_COMP:
      jam();
      signal->theData[0] = data0;
      srFourthComp(signal);
      return;
      break;
    case ZINIT_FOURTH:
      jam();
      signal->theData[0] = data0;
      initFourth(signal);
      return;
      break;
    case ZTIME_SUPERVISION:
      jam();
      signal->theData[0] = data0;
      timeSup(signal);
      return;
      break;
    case ZSR_PHASE3_START:
      jam();
      srPhase3Start(signal);
      return;
      break;
    case ZLQH_TRANS_NEXT: {
      jam();
      ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
      TcNodeFailRecordPtr tcNodeFailPtr;
      tcNodeFailPtr.i = data0;
      ptrCheckGuard(tcNodeFailPtr, ctcNodeFailrecFileSize, tcNodeFailRecord);
      lqhTransNextLab(signal, tcNodeFailPtr);
      return;
      break;
    }
    case ZSCAN_TC_CONNECT:
      jam();
      ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
      tabptr.i = data1;
      ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
      scanTcConnectLab(signal, data0, data2);
      return;
      break;
    case ZINITIALISE_RECORDS:
      jam();
      initialiseRecordsLab(signal, data0, data2, signal->theData[4]);
      return;
      break;
    case ZINIT_GCP_REC:
      jam();
      gcpPtr.i = 0;
      ptrAss(gcpPtr, gcpRecord);
      initGcpRecLab(signal);
      return;
      break;
    case ZCHECK_LCP_STOP_BLOCKED: {
      jam();
      checkLcpStopBlockedLab(signal, data0);
      return;
    }
    case ZSCAN_MARKERS:
      jam();
      scanMarkers(signal, data0, data1);
      return;
      break;

    case ZOPERATION_EVENT_REP:
      jam();
      /* Send counter event report */
      {
        const Uint32 len = c_Counters.build_event_rep(signal);
        sendSignal(CMVMI_REF, GSN_EVENT_REP, signal, len, JBB);
      }

      {
        const Uint32 report_interval = 5000;
        const Uint32 len = c_Counters.build_continueB(signal);
        signal->theData[0] = ZOPERATION_EVENT_REP;
        sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, report_interval,
                            len);
      }
      break;
    case ZDROP_TABLE_WAIT_USAGE:
      jam();
      ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
      dropTab_wait_usage(signal);
      return;
      break;
    case ZENABLE_EXPAND_CHECK: {
      jam();
      fragptr.i = signal->theData[1];
      if (fragptr.i != RNIL) {
        jam();
        c_lcp_complete_fragments.getPtr(fragptr);
        Ptr<Fragrecord> save = fragptr;

        c_lcp_complete_fragments.next(fragptr);
        signal->theData[0] = ZENABLE_EXPAND_CHECK;
        signal->theData[1] = fragptr.i;
        sendSignal(reference(), GSN_CONTINUEB, signal, 2, JBB);

        c_lcp_complete_fragments.remove(save);
        return;
      } else {
        jam();
        cstartRecReq = SRR_REDO_COMPLETE;
        ndbrequire(c_lcp_complete_fragments.isEmpty());

        rebuildOrderedIndexes(signal, 0);
        return;
      }
    }
    case ZRETRY_TCKEYREF: {
      jam();
      Uint32 cnt = signal->theData[1];
      Uint32 ref = signal->theData[2];
      if (cnt < (10 * 60 * 5)) {
        jam();
        /**
         * Only retry for 5 minutes...then hope that API has handled it..somehow
         */
        memmove(signal->theData, signal->theData + 3,
                4 * TcKeyRef::SignalLength);
        sendTCKEYREF(signal, ref, 0, cnt);
      }
      return;
    }
    case ZWAIT_REORG_SUMA_FILTER_ENABLED:
      jam();
      wait_reorg_suma_filter_enabled(signal);
      return;
    case ZREBUILD_ORDERED_INDEXES: {
      Uint32 tableId = signal->theData[1];
      rebuildOrderedIndexes(signal, tableId);
      return;
    }
    case ZWAIT_READONLY: {
      jam();
      wait_readonly(signal);
      return;
    }
    case ZLCP_FRAG_WATCHDOG: {
      jam();
      checkLcpFragWatchdog(signal);
      return;
    }
    case ZSTART_LOCAL_LCP: {
      jam();
      start_lcp_on_table(signal);
      return;
    }
    default:

#if defined ERROR_INSERT
      // ERROR_INSERT 5090
      Uint32 compact = signal->theData[0];
      if (compact >> 16 == ZDELAY_FS_OPEN) {
        jam();
        // Remove ZDELAY_FS_OPEN from compacted theData[0] and
        // restore logFilePtr.i as it was in the original FSOPENCONF signal
        signal->theData[0] = (Uint16)compact;
        sendSignalWithDelay(cownref, GSN_FSOPENCONF, signal, 10, 2);
        return;
      }
#endif

      ndbabort();
  }  // switch
}  // Dblqh::execCONTINUEB()

/* *********************************************************> */
/*  Request from DBDIH to include a new node in the node list */
/*  and so forth.                                             */
/* *********************************************************> */
void Dblqh::execINCL_NODEREQ(Signal *signal) {
  jamEntry();
  BlockReference retRef = signal->theData[0];
  Uint32 nodeId = signal->theData[1];
  cnewestGci = signal->theData[2];
  DEB_NEWEST_GCI(("(%u)(%u) cnewestGci = %u, line: %u", instance(),
                  m_is_query_block, cnewestGci, __LINE__));
  cnewestCompletedGci = signal->theData[2] - 1;
  ndbrequire(cnoOfNodes < MAX_NDB_NODES);
  for (Uint32 i = 0; i < cnoOfNodes; i++) {
    jam();
    if (cnodeData[i] == nodeId) {
      jam();
      cnodeStatus[i] = ZNODE_UP;
    }  // if
  }    // for

  {
    HostRecordPtr Thostptr;
    Thostptr.i = nodeId;
    ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
    Thostptr.p->nodestatus = ZNODE_UP;
    DEB_NODE_STATUS(
        ("(%u,%u) Node %u UP", instance(), m_is_query_block, nodeId));
  }

  signal->theData[0] = nodeId;
  signal->theData[1] = cownref;
  sendSignal(retRef, GSN_INCL_NODECONF, signal, 2, JBB);
  return;
}  // Dblqh::execINCL_NODEREQ()

void Dblqh::execTUPSEIZEREF(Signal *signal) {
  jamEntry();
  ndbabort();
}  // Dblqh::execTUPSEIZEREF()

bool Dblqh::is_first_instance() {
  if (!isNdbMtLqh() || instance() == 1) return true;
  return false;
}

/* ######################################################################## */
/* #######                START / RESTART MODULE                    ####### */
/* ######################################################################## */
/* **********************************************************************>> */
/**
  This is first signal that arrives in a start / restart.
  Sender is NDBCNTR_REF.
*/
/* **********************************************************************>> */
void Dblqh::sttor_startphase1(Signal *signal) {
  UintR Ti, Tj;
  HostRecordPtr ThostPtr;
#if defined NDBD_TRACENR
  FILE *out = 0;
  char *name;
#endif
  Uint32 tstartPhase = signal->theData[1];
  cstartPhase = tstartPhase;
  if (m_is_query_block) {
    c_tup = (Dbtup *)globalData.getBlock(DBQTUP, instance());
    c_tux = (Dbtux *)globalData.getBlock(DBQTUX, instance());
    c_acc = (Dbacc *)globalData.getBlock(DBQACC, instance());
    c_backup = (Backup *)globalData.getBlock(QBACKUP, instance());
    c_restore = (Restore *)globalData.getBlock(QRESTORE, instance());
  } else {
    c_tup = (Dbtup *)globalData.getBlock(DBTUP, instance());
    c_tux = (Dbtux *)globalData.getBlock(DBTUX, instance());
    c_acc = (Dbacc *)globalData.getBlock(DBACC, instance());
    c_backup = (Backup *)globalData.getBlock(BACKUP, instance());
    c_restore = (Restore *)globalData.getBlock(RESTORE, instance());
  }
  c_lgman = (Lgman *)globalData.getBlock(LGMAN);
  c_pgman = (Pgman *)globalData.getBlock(PGMAN, instance());
  ndbrequire(c_tup != 0 && c_tux != 0 && c_acc != 0 && c_backup != 0 &&
             c_pgman != 0 && c_lgman != 0 && c_restore != 0);
  if (isNdbMtLqh() && !m_is_query_block) {
    jam();
    c_restore_mutex_lqh = (Dblqh *)globalData.getBlock(DBLQH, 1);
    m_lock_acc_page_mutex = NdbMutex_Create();
    m_lock_tup_page_mutex = NdbMutex_Create();
  }
  if (instance() == 1 && !m_is_query_block &&
      (globalData.ndbMtRecoverThreads + globalData.ndbMtQueryThreads) > 0 &&
      isNdbMtLqh()) {
    jam();
    m_restore_mutex = NdbMutex_Create();
    ndbrequire(m_restore_mutex != 0);
    m_num_recover_active =
        (Uint32 *)ndbd_malloc(sizeof(Uint32) * (MAX_NDBMT_QUERY_THREADS + 1));
    memset(m_num_recover_active, 0,
           sizeof(Uint32) * (MAX_NDBMT_QUERY_THREADS + 1));
    m_instance_recover_active =
        (Uint32 *)ndbd_malloc(sizeof(Uint32) * (MAX_NDBMT_LQH_THREADS + 1));
    memset(m_instance_recover_active, 0,
           sizeof(Uint32) * (MAX_NDBMT_LQH_THREADS + 1));
    /**
     * The number of active instances is tracked when we are using query
     * threads and recover threads. The reason is that we want to enable
     * reporting of restore rates also from all recover threads.
     *
     * This variable is decremented when all LDM threads have completed
     * the restore processing and only then do we ask the recover threads
     * report the restore rates.
     *
     * We use the m_instance_recover_active array to ensure that each LDM
     * thread only reports ready once.
     */
    for (Uint32 i = 1; i <= globalData.ndbMtLqhWorkers; i++) {
      m_instance_recover_active[i] = 1;
    }
    m_num_instances_active = globalData.ndbMtLqhWorkers;
  }
  if (!m_is_query_block) {
    if (isNdbMtLqh()) {
      jam();
      Uint32 num_restore_threads =
          1 +
          (((globalData.ndbMtRecoverThreads + globalData.ndbMtQueryThreads) +
            (globalData.ndbMtLqhWorkers - 1)) /
           globalData.ndbMtLqhWorkers);
      m_num_restore_threads = num_restore_threads;
    } else {
      jam();
      m_num_restore_threads = 1;
    }
  }

#ifdef NDBD_TRACENR
#ifdef VM_TRACE
  out = globalSignalLoggers.getOutputStream();
#endif
  if (out == 0) {
    name = NdbConfig_SignalLogFileName(getOwnNodeId());
    out = fopen(name, "a");
  }
  tracenrout = new NdbOut(*new FileOutputStream(out));
#endif

#ifdef NDBD_TRACENR
  traceopout = &ndbout;
#endif

  /* Which bits in request info should 'pass through' replicas */
  preComputedRequestInfoMask = 0;
  // Dont setDisableFkconstraints - handled on primary
  LqhKeyReq::setNoTriggersFlag(preComputedRequestInfoMask, 1);
  LqhKeyReq::setUtilFlag(preComputedRequestInfoMask, 1);
  // Dont setNoWaitFlag - handled on primary
  LqhKeyReq::setLastReplicaNo(preComputedRequestInfoMask,
                              LqhKeyReq::RI_LAST_REPL_MASK);
  // Dont LqhKeyReq::setApplicationAddressFlag
  LqhKeyReq::setDirtyFlag(preComputedRequestInfoMask, 1);
  // Dont LqhKeyReq::setInterpretedFlag
  LqhKeyReq::setSimpleFlag(preComputedRequestInfoMask, 1);
  LqhKeyReq::setOperation(preComputedRequestInfoMask,
                          LqhKeyReq::RI_OPERATION_MASK);
  LqhKeyReq::setGCIFlag(preComputedRequestInfoMask, 1);
  LqhKeyReq::setNrCopyFlag(preComputedRequestInfoMask, 1);
  // Dont setAIInLqhKeyReq
  // Dont setSeqNoReplica
  // Dont setSameClientAndTcFlag
  // Dont setReturnedReadLenAIFlag
  LqhKeyReq::setMarkerFlag(preComputedRequestInfoMask, 1);
  LqhKeyReq::setQueueOnRedoProblemFlag(preComputedRequestInfoMask, 1);
  // preComputedRequestInfoMask = 0x003d7fff;

  /* ------- INITIATE ALL RECORDS ------- */
  cownNodeid = getOwnNodeId();
  caccBlockref = calcInstanceBlockRef(getDBACC());
  ctupBlockref = calcInstanceBlockRef(getDBTUP());
  ctuxBlockref = calcInstanceBlockRef(getDBTUX());
  cownref = reference();

  {
    /* Start counter activity event reporting. */
    const Uint32 len = c_Counters.build_continueB(signal);
    signal->theData[0] = ZOPERATION_EVENT_REP;
    sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 10, len);
  }

  for (Uint32 i = 0; i <= ZCOPY_FRAGREQ_CHECK_INDEX; i++) {
    c_check_scanptr_i[i] = RNIL;
    c_check_scanptr_save_line[i] = __LINE__;
    c_check_scanptr_save_timer[i] = 0;
  }

  for (Ti = 0; Ti < chostFileSize; Ti++) {
    ThostPtr.i = Ti;
    ptrCheckGuard(ThostPtr, chostFileSize, hostRecord);
    /*
     * Valid only if receiver has same number of LQH workers.
     * In general full instance key of fragment must be used.
     */
    ThostPtr.p->inPackedList = false;
    ThostPtr.p->lqh_pack_mask.clear();
    ThostPtr.p->tc_pack_mask.clear();
    for (Tj = 0; Tj < NDB_ARRAY_SIZE(ThostPtr.p->lqh_pack); Tj++) {
      ThostPtr.p->lqh_pack[Tj].noOfPackedWords = 0;
      ThostPtr.p->lqh_pack[Tj].hostBlockRef =
          numberToRef(getDBLQH(), Tj, ThostPtr.i);
    }
    for (Tj = 0; Tj < NDB_ARRAY_SIZE(ThostPtr.p->tc_pack); Tj++) {
      ThostPtr.p->tc_pack[Tj].noOfPackedWords = 0;
      ThostPtr.p->tc_pack[Tj].hostBlockRef = numberToRef(DBTC, Tj, ThostPtr.i);
    }
    ThostPtr.p->nodestatus = ZNODE_DOWN;
    DEB_NODE_STATUS(
        ("(%u,%u) Node %u DOWN", instance(), m_is_query_block, ThostPtr.i));
  }  // for
  cpackedListIndex = 0;
}

/**
 * This is first signal that arrives in a start / restart.
 * Sender is NDBCNTR_REF.
 */
void Dblqh::execSTTOR(Signal *signal) {
  UintR tstartPhase;

  jamEntry();
  /* START CASE */
  tstartPhase = signal->theData[1];
  /* SYSTEM RESTART RANK */
  csignalKey = signal->theData[6];
  switch (tstartPhase) {
    case ZSTART_PHASE1:
      jam();
      sttor_startphase1(signal);
      set_use_mutex_for_log_parts();
      sendsttorryLab(signal);
      return;
    case 2:
      jam();
      startphase2Lab(signal, /* dummy */ ~0);
      return;
    case 3:
      jam();
#if (defined(VM_TRACE) || defined(ERROR_INSERT)) && \
    defined(DO_TRANSIENT_POOL_STAT)

      /* Start reporting statistics for transient pools */
      signal->theData[0] = ZLQH_TRANSIENT_POOL_STAT;
      sendSignal(reference(), GSN_CONTINUEB, signal, 1, JBB);
#endif
      if (m_is_query_block) {
        /**
         * These variables are the same in all LDM instances and they are
         * not changed while the data node is running. Thus we set them
         * here instead of when calling setup_query_thread* methods.
         */
        ndbrequire(isNdbMtLqh());
        Dblqh *lqh_block = (Dblqh *)globalData.getBlock(DBLQH, 1);
        this->c_fragment_pool.setNewSize(lqh_block->c_fragment_pool.getSize());
        this->ctabrecFileSize = lqh_block->ctabrecFileSize;
        Dbtup *tup_block = (Dbtup *)globalData.getBlock(DBTUP, 1);
        c_tup->cnoOfFragrec = tup_block->cnoOfFragrec;
        c_tup->cnoOfTablerec = tup_block->cnoOfTablerec;
        Dbacc *acc_block = (Dbacc *)globalData.getBlock(DBACC, 1);
        c_acc->cfragmentsize = acc_block->cfragmentsize;
        Dbtux *tux_block = (Dbtux *)globalData.getBlock(DBTUX, 1);
        c_tux->c_fragPool.setNewSize(tux_block->c_fragPool.getSize());
        c_tux->c_indexPool.setNewSize(tux_block->c_indexPool.getSize());
        c_tux->c_descPagePool.setNewSize(tux_block->c_descPagePool.getSize());
        signal->theData[0] = cownref;
        sendSignal(NDBCNTR_REF, GSN_READ_NODESREQ, signal, 1, JBB);
        return;
      } else {
        send_read_local_sysfile(signal);
      }
      return;
    case 4:
      jam();
      if (!m_is_query_block) {
        define_backup(signal);
      } else {
        jam();
        sendsttorryLab(signal);
      }
      break;
    case 6:
      c_elapsed_time_millis = 0;
      init_elapsed_time(signal, c_latestTIME_SIGNAL);
      sendsttorryLab(signal);
      break;
    case 9:
      jam();
      /**
       * We add this wait phase to avoid having to handle multiple
       * writers of the Local sysfile. We check here if we have an
       * outstanding WRITE_LOCAL_SYSFILE_REQ signal, if that is the
       * case we set a flag that we are waiting for this and send
       * STTORRY when this is returned.
       *
       * WRITE_LOCAL_SYSFILE_REQ is only sent from first instance, so
       * need to handle this for other instances.
       */
      ndbrequire(m_is_query_block || cstartRecReq == SRR_FIRST_LCP_DONE);
      if (!m_is_query_block && is_first_instance()) {
        c_start_phase_9_waiting = true;
        /**
         * Restart is completed, we need to wait until this has been
         * reflected in the local sysfile. It becomes reflected in
         * local sysfile in the next processing of GCP_SAVEREQ. This
         * avoids complex interaction handling of writes to the
         * local sysfile.
         */
        DEB_LCP(("(%u)Start phase 9 wait started", instance()));
        DEB_START_PHASE9(("(%u)Start phase 9 wait started", instance()));
      } else {
        jam();
        /**
         * Restart is done, record this fact and move on in restart
         * processing.
         */
        write_local_sysfile_restart_complete_done(signal);
        DEB_START_PHASE9(("(%u)Start phase 9 wait completed", instance()));
      }
      send_print_mutex_stats(signal);
      return;
    default:
      jam();
      /*empty*/;
      sendsttorryLab(signal);
      return;
      break;
  }  // switch
}  // Dblqh::execSTTOR()

void Dblqh::write_local_sysfile_restart_complete_done(Signal *signal) {
  cstartPhase = ZNIL;
  cstartType = ZNIL;
  c_start_phase_9_waiting = false;
  sendsttorryLab(signal);
}

void Dblqh::send_read_local_sysfile(Signal *signal) {
  ReadLocalSysfileReq *req = (ReadLocalSysfileReq *)signal->getDataPtrSend();
  req->userPointer = 0;
  req->userReference = reference();
  sendSignal(NDBCNTR_REF, GSN_READ_LOCAL_SYSFILE_REQ, signal,
             ReadLocalSysfileReq::SignalLength, JBB);
}

void Dblqh::execREAD_LOCAL_SYSFILE_CONF(Signal *signal) {
  ReadLocalSysfileConf *conf = (ReadLocalSysfileConf *)signal->getDataPtr();
  c_local_sysfile.m_node_restorable_on_its_own = conf->nodeRestorableOnItsOwn;
  c_local_sysfile.m_max_restorable_gci = conf->maxGCIRestorable;
  c_backup->setRestorableGci(c_local_sysfile.m_max_restorable_gci);
  sendsttorryLab(signal);
}

void Dblqh::define_backup(Signal *signal) {
  DefineBackupReq *req = (DefineBackupReq *)signal->getDataPtrSend();
  req->backupId = 0;
  req->clientRef = 0;
  req->clientData = 0;
  req->senderRef = reference();
  req->masterRef = reference();
  req->backupPtr = 0;
  req->backupKey[0] = 0;
  req->backupKey[1] = 0;
  req->backupDataLen = ~0;
  req->nodes.clear();  // Use nodes in section
  req->flags = 0;
  req->senderData = 0;

  NdbNodeBitmask nodes;
  nodes.set(getOwnNodeId());

  ndbassert(!m_is_query_block);
  BlockReference backupRef = calcInstanceBlockRef(BACKUP);
  Uint32 packed_length = nodes.getPackedLengthInWords();

  // Backup is not allowed for mixed versions of data nodes
  ndbrequire(ndbd_send_node_bitmask_in_section(
      getNodeInfo(refToNode(backupRef)).m_version));

  LinearSectionPtr lsptr[3];
  lsptr[0].p = nodes.rep.data;
  lsptr[0].sz = packed_length;
  sendSignal(backupRef, GSN_DEFINE_BACKUP_REQ, signal,
             DefineBackupReq::SignalLength_v1, JBB, lsptr, 1);
}

void Dblqh::execDEFINE_BACKUP_REF(Signal *signal) {
  jamEntry();
  m_backup_ptr = RNIL;
  DefineBackupRef *ref = (DefineBackupRef *)signal->getDataPtrSend();
  int err_code = 0;
  char *extra_msg = NULL;

  switch (ref->errorCode) {
    case DefineBackupRef::Undefined:
    case DefineBackupRef::FailedToSetupFsBuffers:
    case DefineBackupRef::FailedToAllocateBuffers:
    case DefineBackupRef::FailedToAllocateTables:
    case DefineBackupRef::FailedAllocateTableMem:
    case DefineBackupRef::FailedToAllocateFileRecord:
    case DefineBackupRef::FailedToAllocateAttributeRecord:
    case DefineBackupRef::FailedInsertFileHeader:
    case DefineBackupRef::FailedInsertTableList:
      jam();
      err_code = NDBD_EXIT_INVALID_CONFIG;
      extra_msg =
        (char*) "Probably Backup parameters configuration error,"
                " Please consult the manual";
      progError(__LINE__, err_code, extra_msg);
  }

  sendsttorryLab(signal);
}

void Dblqh::execDEFINE_BACKUP_CONF(Signal *signal) {
  jamEntry();
  DefineBackupConf *conf = (DefineBackupConf *)signal->getDataPtrSend();
  m_backup_ptr = conf->backupPtr;
  sendsttorryLab(signal);
}

/* ***************************************> */
/*  Restart phases 1 - 6, sender is Ndbcntr */
/* ***************************************> */
void Dblqh::execNDB_STTOR(Signal *signal) {
  jamEntry();
  Uint32 ownNodeId = signal->theData[1]; /* START PHASE*/
  cstartPhase = signal->theData[2];      /* MY NODE ID */
  cstartType = signal->theData[3];       /* START TYPE */

  switch (cstartPhase) {
    case ZSTART_PHASE1:
      jam();
      startphase1Lab(signal, /* dummy */ ~0, ownNodeId);
      return;
      break;
    case ZSTART_PHASE2:
      jam();
      /*empty*/;
      sendNdbSttorryLab(signal);
      return;
      break;
    case ZSTART_PHASE3:
      jam();
      startphase3Lab(signal);
      return;
      break;
    case ZSTART_PHASE4:
      jam();
      /*empty*/;
      sendNdbSttorryLab(signal);
      return;
      break;
    case ZSTART_PHASE6:
      jam();
      startphase6Lab(signal);
      return;
      break;
    default:
      jam();
      /*empty*/;
      sendNdbSttorryLab(signal);
      return;
      break;
  }  // switch
}  // Dblqh::execNDB_STTOR()

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
/* +++++++                         START PHASE 2                    +++++++ */
/*                                                                          */
/*             INITIATE ALL RECORDS WITHIN THE BLOCK                        */
/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void Dblqh::startphase1Lab(Signal *signal, Uint32 _dummy, Uint32 ownNodeId) {
  bool do_init = (cstartType == NodeState::ST_INITIAL_START) ||
                 (cstartType == NodeState::ST_INITIAL_NODE_RESTART);

  LogFileRecordPtr prevLogFilePtr;
  LogFileRecordPtr zeroLogFilePtr;

  if (do_init) {
    g_eventLogger->info("LDM(%u): Starting REDO log initialisation",
                        instance());
  }
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  ndbrequire(cnoLogFiles != 0);
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    initLogpart(signal, logPartPtr);
    for (Uint32 fileNo = 0; fileNo < cnoLogFiles; fileNo++) {
      seizeLogfile(logFilePtr, logPartPtr.p);
      if (fileNo != 0) {
        jam();
        prevLogFilePtr.p->nextLogFile = logFilePtr.i;
        logFilePtr.p->prevLogFile = prevLogFilePtr.i;
      } else {
        jam();
        logPartPtr.p->firstLogfile = logFilePtr.i;
        logPartPtr.p->currentLogfile = logFilePtr.i;
        zeroLogFilePtr.i = logFilePtr.i;
        zeroLogFilePtr.p = logFilePtr.p;
      }  // if
      prevLogFilePtr.i = logFilePtr.i;
      prevLogFilePtr.p = logFilePtr.p;
      initLogfile(logFilePtr, logPartPtr.p->logPartNo, fileNo, logPartPtr.i);
      if (do_init) {
        jam();
        if (logFilePtr.i == zeroLogFilePtr.i) {
          jam();
          /* -------------------------------------------------------------------------
           */
          /*IN AN INITIAL START WE START BY CREATING ALL LOG FILES AND SETTING
           * THEIR   */
          /*PROPER SIZE AND INITIALISING PAGE ZERO IN ALL FILES. */
          /*WE START BY CREATING FILE ZERO IN EACH LOG PART AND THEN PROCEED */
          /*SEQUENTIALLY THROUGH ALL LOG FILES IN THE LOG PART. */
          /* -------------------------------------------------------------------------
           */
          if (m_use_om_init == 0 || logPartPtr.i == 0) {
            /**
             * initialize one file at a time if using OM_INIT
             */
            jam();
#if defined(USE_INIT_GLOBAL_VARIABLES)
            if (m_use_om_init) {
              jam();
              /**
               * FSWRITEREQ does cross-thread execute-direct
               *   which makes the clear_global_variables "unsafe"
               *   disable it until we're finished with init log-files
               */
              disable_global_variables();
            }
#endif
            openLogfileInit(signal, logFilePtr);
          }
        }  // if
      }    // if
    }      // for
    zeroLogFilePtr.p->prevLogFile = logFilePtr.i;
    logFilePtr.p->nextLogFile = zeroLogFilePtr.i;
  }

  if (clogPartFileSize == 0) {
    /**
     * No Redo Log part in this LDM thread. Continue restart
     * immediately.
     */
    g_eventLogger->info(
        "LDM(%u): Completed LDM Restart phase 1, no REDO"
        " log parts to initiate",
        instance());
    if (do_init) {
      jam();
      csrExecUndoLogState = EULS_COMPLETED;
    }
    sendNdbSttorryLab(signal);
    return;
  }

  initReportStatus(signal);
  if (!do_init) {
    jam();
    g_eventLogger->info(
        "LDM(%u): Started LDM restart phase 1"
        " (read REDO log page headers to init"
        " REDO log data)",
        instance());
    sendNdbSttorryLab(signal);
  } else {
    reportStatus(signal);
  }

  return;
}  // Dblqh::startphase1Lab()

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
/* +++++++                         START PHASE 2                    +++++++ */
/*                                                                          */
/* CONNECT LQH WITH ACC AND TUP.                                            */
/* EVERY CONNECTION RECORD IN LQH IS ASSIGNED TO ONE ACC CONNECTION RECORD  */
/*       AND ONE TUP CONNECTION RECORD.                                     */
/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void Dblqh::startphase2Lab(Signal *signal, Uint32 _dummy) {
  cmaxWordsAtNodeRec = MAX_NO_WORDS_OUTSTANDING_COPY_FRAGMENT;
  /* -- ACC AND TUP CONNECTION PROCESS -- */
  TcConnectionrecPtr tcConnectptr;
  ndbrequire(tcConnect_pool.seize(tcConnectptr));
  tcConnectptr.p->ptrI = tcConnectptr.i;
  ctcConnectReservedCount = 0;
  cfirstfreeTcConrec = RNIL;
  moreconnectionsLab(signal, tcConnectptr);
  signal->theData[0] = ZCHECK_SYSTEM_SCANS;
  sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, DELAY_CHECK_SYSTEM_SCANS,
                      1);
  return;
}  // Dblqh::startphase2Lab()

void Dblqh::moreconnectionsLab(Signal *signal,
                               const TcConnectionrecPtr tcConnectptr) {
  // set TUX block here (no operation is seized in TUX)
  /* NO STATE CHECKING IS PERFORMED, ASSUMED TO WORK */
  /* *************** */
  /*  ACCSEIZEREQ  < */
  /* *************** */
  signal->theData[0] = tcConnectptr.i;
  signal->theData[1] = cownref;
  sendSignal(caccBlockref, GSN_ACCSEIZEREQ, signal, 2, JBB);
  return;
}  // Dblqh::moreconnectionsLab()

/* ***************> */
/*  ACCSEIZECONF  > */
/* ***************> */
void Dblqh::execACCSEIZECONF(Signal *signal) {
  jamEntry();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  tcConnectptr.p->accConnectrec = signal->theData[1];
  tcConnectptr.p->accConnectPtrP = c_acc->get_operation_ptr(signal->theData[1]);

  /* *************** */
  /*  TUPSEIZEREQ  < */
  /* *************** */
  signal->theData[0] = tcConnectptr.i;
  signal->theData[1] = cownref;
  sendSignal(ctupBlockref, GSN_TUPSEIZEREQ, signal, 2, JBB);
  return;
}  // Dblqh::execACCSEIZECONF()

/* ***************> */
/*  TUPSEIZECONF  > */
/* ***************> */
void Dblqh::execTUPSEIZECONF(Signal *signal) {
  jamEntry();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  tcConnectptr.p->tupConnectrec = signal->theData[1];
  tcConnectptr.p->tupConnectPtrP = c_tup->get_operation_ptr(signal->theData[1]);
  /* ------- CHECK IF THERE ARE MORE CONNECTIONS TO BE CONNECTED ------- */
  Uint32 prevFirst = cfirstfreeTcConrec;
  tcConnectptr.p->nextTcConnectrec = prevFirst;
  cfirstfreeTcConrec = tcConnectptr.i;
  ctcConnectReservedCount++;
  if (ctcConnectReservedCount < ctcConnectReserved) {
    jam();
    ndbrequire(tcConnect_pool.seize(tcConnectptr));
    tcConnectptr.p->ptrI = tcConnectptr.i;
    moreconnectionsLab(signal, tcConnectptr);
    return;
  }  // if
  /* ALL LQH_CONNECT RECORDS ARE CONNECTED TO ACC AND TUP ---- */
  sendsttorryLab(signal);
}  // Dblqh::execTUPSEIZECONF()

/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
/* +++++++                    START PHASE 4                          +++++++ */
/*                                                                           */
/*       CONNECT LQH WITH LQH.                                               */
/*       CONNECT EACH LQH WITH EVERY LQH IN THE DATABASE SYSTEM.             */
/*       IF INITIAL START THEN CREATE THE FRAGMENT LOG FILES                 */
/*IF SYSTEM RESTART OR NODE RESTART THEN OPEN THE FRAGMENT LOG FILES AND     */
/*FIND THE END OF THE LOG FILES.                                             */
/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
/*        WAIT UNTIL ADD NODE PROCESSES ARE COMPLETED                        */
/*        IF INITIAL START ALSO WAIT FOR LOG FILES TO INITIALISED            */
/*START TIME SUPERVISION OF LOG FILES. WE HAVE TO WRITE LOG PAGES TO DISK    */
/*EVEN IF THE PAGES ARE NOT FULL TO ENSURE THAT THEY COME TO DISK ASAP.      */
/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void Dblqh::startphase3Lab(Signal *signal) {
  caddNodeState = ZTRUE;
  /* ***************<< */
  /*  READ_NODESREQ  < */
  /* ***************<< */
  cinitialStartOngoing = ZTRUE;
  LogPartRecordPtr logPartPtr;

  switch (cstartType) {
    case NodeState::ST_NODE_RESTART:
    case NodeState::ST_SYSTEM_RESTART: {
      jam();
      for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
        jam();
        LogFileRecordPtr locLogFilePtr;
        ptrAss(logPartPtr, logPartRecord);
        locLogFilePtr.i = logPartPtr.p->firstLogfile;
        ptrCheckGuard(locLogFilePtr, clogFileFileSize, logFileRecord);
        locLogFilePtr.p->logFileStatus = LogFileRecord::OPEN_SR_FRONTPAGE;
        openFileRw(signal, locLogFilePtr, false); /* No write buffering */
      }                                           // for
      break;
    }
    case NodeState::ST_INITIAL_START:
    case NodeState::ST_INITIAL_NODE_RESTART: {
      jam();
      for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
        jam();
        signal->theData[0] = ZINIT_FOURTH;
        signal->theData[1] = logPartPtr.i;
        sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
      }
      break;
    }
  }

  signal->theData[0] = cownref;
  sendSignal(NDBCNTR_REF, GSN_READ_NODESREQ, signal, 1, JBB);
  return;
}  // Dblqh::startphase3Lab()

/* ****************** */
/*  READ_NODESCONF  > */
/* ****************** */
void Dblqh::execREAD_NODESCONF(Signal *signal) {
  jamEntry();

  ReadNodesConf *const readNodes = (ReadNodesConf *)&signal->theData[0];
  cnoOfNodes = readNodes->noOfNodes;

  {
    ndbrequire(signal->getNoOfSections() == 1);
    SegmentedSectionPtr ptr;
    SectionHandle handle(this, signal);
    ndbrequire(handle.getSection(ptr, 0));
    ndbrequire(ptr.sz == 5 * NdbNodeBitmask::Size);
    copy((Uint32 *)&readNodes->definedNodes.rep.data, ptr);
    releaseSections(handle);
  }

  unsigned ind = 0;
  unsigned i = 0;
  for (i = 1; i < MAX_NDB_NODES; i++) {
    jam();
    if (readNodes->definedNodes.get(i)) {
      jam();
      cnodeData[ind] = i;
      cnodeStatus[ind] = readNodes->inactiveNodes.get(i);
      {
        HostRecordPtr Thostptr;
        Thostptr.i = i;
        ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
        Thostptr.p->nodestatus = cnodeStatus[ind];
        DEB_NODE_STATUS(("(%u,%u) Node %u %u", instance(), m_is_query_block,
                         Thostptr.i, cnodeStatus[ind]));
      }
      // readNodes->getVersionId(i, readNodes->theVersionIds) not used
      if (!readNodes->inactiveNodes.get(i)) {
        jam();
        m_sr_nodes.set(i);
      }
      ind++;
    }  // if
  }    // for
  ndbrequire(ind == cnoOfNodes);
  ndbrequire(cnoOfNodes >= 1 && cnoOfNodes < MAX_NDB_NODES);
  ndbrequire(!(cnoOfNodes == 1 && cstartType == NodeState::ST_NODE_RESTART));

#ifdef ERROR_INSERT
  c_master_node_id = readNodes->masterNodeId;
#endif

  caddNodeState = ZFALSE;
  if (cstartType != NodeState::ST_SYSTEM_RESTART &&
      cstartType != NodeState::ST_NODE_RESTART) {
    jam();
    if (!m_is_query_block) {
      if (clogPartFileSize == 0) {
        jam();
        g_eventLogger->info(
            "LDM(%u): No REDO log parts in this LDM, "
            "thus REDO actions completed now",
            instance());
        restart_synch_state(signal, COMPLETED_REDO_FOURTH_PHASE_SYNCH,
                            ZCONTINUE_SR_FOURTH_COMP);
      }
      return;
    }
  } else {
    if (clogPartFileSize == 0) {
      g_eventLogger->info("(%u)No REDO log parts, send ZSR_PHASE1_COMPLETED",
                          instance());
      signal->theData[0] = ZSR_PHASE3_START;
      signal->theData[1] = ZSR_PHASE1_COMPLETED;
      sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
    }
    if (cstartType == NodeState::ST_NODE_RESTART) {
      jam();
      m_sr_nodes.clear();
      m_sr_nodes.set(getOwnNodeId());
    }
  }
  SET_TRACENR_FLAG;
  if (m_is_query_block) {
    jam();
    sendsttorryLab(signal);
  } else {
    jam();
    sendNdbSttorryLab(signal);
  }
  return;
}  // Dblqh::execREAD_NODESCONF()

void Dblqh::checkStartCompletedLab(Signal *signal) {
  if (caddNodeState == ZFALSE) {
    if (cinitialStartOngoing == ZFALSE) {
      jam();
      sendNdbSttorryLab(signal);
      return;
    }  // if
  }    // if
  return;
}  // Dblqh::checkStartCompletedLab()

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
/* SET CONCURRENCY OF LOCAL CHECKPOINTS TO BE USED AFTER SYSTEM RESTART.      */
/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void Dblqh::startphase6Lab(Signal *signal) {
  CLEAR_TRACENR_FLAG;
  sendNdbSttorryLab(signal);
  return;
}  // Dblqh::startphase6Lab()

void Dblqh::sendNdbSttorryLab(Signal *signal) {
  signal->theData[0] = cownref;
  ndbassert(!m_is_query_block);
  BlockReference cntrRef = !isNdbMtLqh() ? NDBCNTR_REF : DBLQH_REF;
  sendSignal(cntrRef, GSN_NDB_STTORRY, signal, 1, JBB);
  return;
}  // Dblqh::sendNdbSttorryLab()

void Dblqh::sendsttorryLab(Signal *signal) {
  /* *********<< */
  /*  STTORRY  < */
  /* *********<< */
  BlockReference cntrRef;
  signal->theData[0] = csignalKey; /* SIGNAL KEY */
  signal->theData[1] = 3;          /* BLOCK CATEGORY */
  signal->theData[2] = 2;          /* SIGNAL VERSION NUMBER */
  signal->theData[3] = ZSTART_PHASE1;
  signal->theData[4] = 2;
  signal->theData[5] = 3;
  signal->theData[6] = 4;
  signal->theData[7] = 6;
  signal->theData[8] = 9;
  signal->theData[9] = 255;
  if (m_is_query_block) {
    cntrRef = DBQLQH_REF;
  } else {
    cntrRef = !isNdbMtLqh() ? NDBCNTR_REF : DBLQH_REF;
  }
  sendSignal(cntrRef, GSN_STTORRY, signal, 10, JBB);
  return;
}  // Dblqh::sendsttorryLab()

/* ***************>> */
/*  READ_NODESREF  > */
/* ***************>> */
void Dblqh::execREAD_NODESREF(Signal *signal) {
  jamEntry();
  ndbabort();
}  // Dblqh::execREAD_NODESREF()

/* ****************** */
/*  READ_CONFIG_REQ > */
/* ****************** */
void Dblqh::execREAD_CONFIG_REQ(Signal *signal) {
  const ReadConfigReq *req = (ReadConfigReq *)signal->getDataPtr();
  Uint32 ref = req->senderRef;
  Uint32 senderData = req->senderData;
  ndbrequire(req->noOfParameters == 0);

  jamEntry();

  const ndb_mgm_configuration_iterator *p =
      m_ctx.m_config.getOwnConfigIterator();
  ndbrequire(p != 0);

  ndbrequire(globalData.ndbLogParts <= NDB_MAX_LOG_PARTS);
  if (globalData.ndbLogParts != 4 && globalData.ndbLogParts != 6 &&
      globalData.ndbLogParts != 8 && globalData.ndbLogParts != 10 &&
      globalData.ndbLogParts != 12 && globalData.ndbLogParts != 16 &&
      globalData.ndbLogParts != 20 && globalData.ndbLogParts != 24 &&
      globalData.ndbLogParts != 32) {
    char buf[255];
    BaseString::snprintf(
        buf, sizeof(buf),
        "Trying to start with %d log parts, number of log parts can"
        " only be set to 4, 6, 8, 10, 12, 16, 20, 24 or 32.",
        globalData.ndbLogParts);
    progError(__LINE__, NDBD_EXIT_INVALID_CONFIG, buf);
  }

  cnoLogFiles = 8;
  ndbrequire(
      !ndb_mgm_get_int_parameter(p, CFG_DB_NO_REDOLOG_FILES, &cnoLogFiles));
  ndbrequire(cnoLogFiles > 0);

  Uint32 log_page_size = 0;
  ndb_mgm_get_int_parameter(p, CFG_DB_REDO_BUFFER, &log_page_size);

  c_max_scan_direct_count = ZMAX_SCAN_DIRECT_COUNT;
  ndb_mgm_get_int_parameter(p, CFG_DB_SCHED_SCAN_PRIORITY,
                            &c_max_scan_direct_count);

  /**
   * Always set page size in half MBytes
   */
  Uint64 logPageFileSize = (log_page_size / sizeof(LogPageRecord));
  Uint32 mega_byte_part = logPageFileSize & 15;
  if (mega_byte_part != 0) {
    jam();
    logPageFileSize += (16 - mega_byte_part);
  }
  /**
   * We use one REDO log buffer per log part, thus LDM instances
   * with no REDO log part need no buffer and instances with
   * multiple log parts need more REDO buffer.
   */
  logPageFileSize *= clogPartFileSize;

  /* maximum number of log file operations */
  clfoFileSize = logPageFileSize;
  if (clfoFileSize < ZLFO_MIN_FILE_SIZE && clfoFileSize != 0) {
    jam();
    clfoFileSize = ZLFO_MIN_FILE_SIZE;
  }

  ndbrequire(!ndb_mgm_get_int_parameter(p, CFG_LQH_TABLE, &ctabrecFileSize));
  clogFileFileSize = clogPartFileSize * cnoLogFiles;

  if (m_is_query_block) {
    clfoFileSize = 0;
    logPageFileSize = 0;
    clogFileFileSize = 0;
  }

  ndbrequire(!ndb_mgm_get_int_parameter(p, CFG_DB_DISCLESS, &c_diskless));
  c_o_direct = true;
  ndb_mgm_get_int_parameter(p, CFG_DB_O_DIRECT, &c_o_direct);

  Uint32 encrypted_filesystem = 0;
  ndb_mgm_get_int_parameter(p, CFG_DB_ENCRYPTED_FILE_SYSTEM,
                            &encrypted_filesystem);
  c_encrypted_filesystem = encrypted_filesystem;

  m_use_om_init = 0;
  {
    const char *conf = 0;
    if (!ndb_mgm_get_string_parameter(p, CFG_DB_INIT_REDO, &conf) && conf) {
      jam();
      if (native_strcasecmp(conf, "sparse") == 0) {
        jam();
        m_use_om_init = 0;
      } else if (native_strcasecmp(conf, "full") == 0) {
        jam();
        m_use_om_init = 1;
      }
    }
  }

  c_o_direct_sync_flag = false;
  ndb_mgm_get_int_parameter(p, CFG_DB_O_DIRECT_SYNC_FLAG,
                            &c_o_direct_sync_flag);
#ifdef WIN32
  /**
   * Windows currently has no support for O_DIRECT and in
   * O_DIRECT_SYNC mode we optimise away needed FSYNCs
   * So avoid doing that by accident on Win
   */
  if (c_o_direct_sync_flag) {
    g_eventLogger->warning(
        "ODirectSyncFlag not supported on Windows,"
        " ignored");
    c_o_direct_sync_flag = false;
  } else if (c_o_direct_sync_flag && m_use_om_init == 0) {
    g_eventLogger->warning(
        "ODirectSyncFlag not supported"
        "without setting InitFragmentLogFiles=full");
    c_o_direct_sync_flag = false;
  }
#endif

  if (!m_is_query_block) {
    Uint32 tmp = 0;
    ndbrequire(!ndb_mgm_get_int_parameter(p, CFG_LQH_FRAG, &tmp));
    c_fragment_pool.setSize(tmp);
  }
  if (!m_is_query_block) {
    c_copy_fragment_pool.setSize(ZCOPYFRAGREC_FILE_SIZE);
    c_copy_active_pool.setSize(ZCOPYACTIVEREC_FILE_SIZE);
  }

  if (!ndb_mgm_get_int_parameter(p, CFG_DB_REDOLOG_FILE_SIZE, &clogFileSize)) {
    // convert to mbyte
    clogFileSize = (clogFileSize + 1024 * 1024 - 1) / (1024 * 1024);
    ndbrequire(clogFileSize >= 4 && clogFileSize <= 1024);
  }

  m_startup_report_frequency = 0;
  ndb_mgm_get_int_parameter(p, CFG_DB_STARTUP_REPORT_FREQUENCY,
                            &m_startup_report_frequency);

  /**
   * Initialise variables used to report status of REDO log initialisation.
   */
  c_totalLogFiles = clogFileFileSize;
  c_totallogMBytes = c_totalLogFiles * clogFileSize;

  cmaxLogFilesInPageZero =
      (ZPAGE_SIZE - ZPAGE_HEADER_SIZE - 128) / (ZFD_MBYTE_SIZE * clogFileSize);

  /**
   * "Old" cmaxLogFilesInPageZero was 40
   * Each FD need 3 words per mb, require that they can fit into 1 page
   *   (at least 1 FD)
   * Is also checked in ConfigInfo.cpp (max FragmentLogFileSize = 1Gb)
   *   1Gb = 1024Mb => 3(ZFD_MBYTE_SIZE) * 1024 < 8192 (ZPAGE_SIZE)
   */
  if (cmaxLogFilesInPageZero > 40) {
    jam();
    cmaxLogFilesInPageZero = 40;
  } else {
    ndbrequire(cmaxLogFilesInPageZero);
  }

#if defined VM_TRACE || defined ERROR_INSERT
  if (cmaxLogFilesInPageZero_DUMP != 0) {
    if (cmaxLogFilesInPageZero_DUMP > cmaxLogFilesInPageZero) {
      g_eventLogger->info("LQH DUMP 2396 %u: max allowed is %d",
                          cmaxLogFilesInPageZero_DUMP, cmaxLogFilesInPageZero);
      // do not continue with useless test
      ndbabort();
    } else {
      g_eventLogger->info("LQH DUMP 2396 %u", cmaxLogFilesInPageZero_DUMP);
    }
    cmaxLogFilesInPageZero = cmaxLogFilesInPageZero_DUMP;
  }
#endif

  /* How many file's worth of info is actually valid? */
  cmaxValidLogFilesInPageZero = cmaxLogFilesInPageZero - 1;

  /* Must be at least 1 */
  ndbrequire(cmaxValidLogFilesInPageZero > 0);

  {
    /* TODO RONM: Check how LCPs are affected by REDO log parts independence.*/
    Uint32 config_val = 20;
    ndb_mgm_get_int_parameter(p, CFG_DB_LCP_INTERVAL, &config_val);
    config_val = config_val > 31 ? 31 : config_val;

    const Uint32 mb = 1024 * 1024;

    // perform LCP after this amount of mbytes written
    const Uint64 config_mbytes = ((Uint64(4) << config_val) + mb - 1) / mb;
    const Uint64 totalmb = Uint64(cnoLogFiles) * Uint64(clogFileSize);
    if (totalmb > config_mbytes) {
      c_free_mb_force_lcp_limit = Uint32(totalmb - config_mbytes);
    } else {
      c_free_mb_force_lcp_limit = 0;
    }

    // No less than 33%
    Uint32 limit = Uint32(totalmb / 3);
    if (c_free_mb_force_lcp_limit < limit) {
      c_free_mb_force_lcp_limit = limit;
    }
  }
  c_free_mb_tail_problem_limit = 4;  // If less than 4Mb set TAIL_PROBLEM

  ndb_mgm_get_int_parameter(p, CFG_DB_TRANSACTION_DEADLOCK_TIMEOUT,
                            &cTransactionDeadlockDetectionTimeout);

  initRecords(p, logPageFileSize);
  initialiseRecordsLab(signal, 0, ref, senderData);

  c_max_redo_lag = 30;
  ndb_mgm_get_int_parameter(p, CFG_DB_REDO_OVERCOMMIT_LIMIT, &c_max_redo_lag);

  c_max_redo_lag_counter = 3;
  ndb_mgm_get_int_parameter(p, CFG_DB_REDO_OVERCOMMIT_COUNTER,
                            &c_max_redo_lag_counter);

  c_max_parallel_scans_per_frag = 32;
  ndb_mgm_get_int_parameter(p, CFG_DB_PARALLEL_SCANS_PER_FRAG,
                            &c_max_parallel_scans_per_frag);

  if (c_max_parallel_scans_per_frag > (256 - MAX_PARALLEL_SCANS_PER_FRAG) / 2) {
    jam();
    c_max_parallel_scans_per_frag = (256 - MAX_PARALLEL_SCANS_PER_FRAG) / 2;
  }

  {
    Uint32 param = 60;
    ndb_mgm_get_int_parameter(p, CFG_DB_LCP_SCAN_WATCHDOG_LIMIT, &param);

    /* LCP fail when LCP_SCAN_WATCHDOG_LIMIT exceeded */
    param *= 1000;  // Convert to milliseconds
    c_lcpFragWatchdog.MaxElapsedWithNoProgressMillis = param;

    /* Warn when stalled for roughly 1/3 time, */
    c_lcpFragWatchdog.WarnElapsedWithNoProgressMillis = (param + 2) / 3;

    ndbrequire(c_lcpFragWatchdog.MaxElapsedWithNoProgressMillis >=
               c_lcpFragWatchdog.WarnElapsedWithNoProgressMillis);
    c_lcpFragWatchdog.MaxGcpWaitLimitMillis = 0;  // Will be set by DIH

    if (!m_is_query_block) {
      /* Dump LCPFragWatchdog parameter values */
      signal->theData[0] = 2395;
      execDUMP_STATE_ORD(signal);
    }
  }
}

void Dblqh::init_restart_synch() {
  if (!m_is_query_block) {
    if (!isNdbMtLqh() || instance() == 1) {
      NdbMutex_Init(&m_restart_synch_mutex);
      m_restart_synch_state = NO_SYNCH_ONGOING;
      m_restart_synch_ready = 0;
    }
  }
}

void Dblqh::deinit_restart_synch() {
  if (!m_is_query_block) {
    if (!isNdbMtLqh() || instance() == 1) {
      NdbMutex_Deinit(&m_restart_synch_mutex);
    }
  }
}

Dblqh::LogPartRecord *Dblqh::get_log_part_record(Uint32 instanceKey) {
  LogPartRecord *logPartPtrP;
  Uint32 part_instance = (instanceKey - 1) % globalData.ndbLogParts;
  if (!isNdbMtLqh()) {
    logPartPtrP = &logPartRecord[part_instance];
  } else {
    Uint32 num_ldm_instances = globalData.ndbMtLqhWorkers;
    Uint32 ldm_instance = part_instance % num_ldm_instances;
    Dblqh *lqh_block = (Dblqh *)globalData.getBlock(DBLQH, ldm_instance + 1);
    Uint32 local_part_instance = part_instance / num_ldm_instances;
    logPartPtrP = lqh_block->getLogPart(local_part_instance);
  }
  ndbrequire(logPartPtrP != nullptr);
  ndbrequire(logPartPtrP->logPartNo == part_instance);
  return logPartPtrP;
}

void Dblqh::restart_synch_state(Signal *signal, Uint32 synch_state,
                                Uint32 continueb_case) {
  DEB_RESTART_SYNCH(("(%u) restart_synch_state, CONTINUEB_CASE: %u", instance(),
                     continueb_case));
  NdbMutex_Lock(&m_restart_synch_mutex);
  if (m_restart_synch_state == NO_SYNCH_ONGOING) {
    ndbrequire(m_restart_synch_ready == 0);
    m_restart_synch_state = (RestartSynchState)synch_state;
    m_restart_synch_ready = 1;
  } else {
    ndbrequire(m_restart_synch_state == synch_state);
    m_restart_synch_ready++;
  }
  Uint32 num_ready = m_restart_synch_ready;
  Uint32 num_expecting = isNdbMtLqh() ? globalData.ndbMtLqhWorkers : 1;
  if (num_ready == num_expecting) {
    m_restart_synch_state = NO_SYNCH_ONGOING;
    m_restart_synch_ready = 0;
    send_CONTINUEB_all(signal, continueb_case);
    DEB_RESTART_SYNCH(
        ("(%u) restart_synch_state, CONTINUEB_CASE: %u"
         ", CONTINUEB sent",
         instance(), continueb_case));
  } else {
    DEB_RESTART_SYNCH(
        ("(%u) restart_synch_state, CONTINUEB_CASE: %u"
         ", ready: %u, expecting: %u",
         instance(), continueb_case, num_ready, num_expecting));
  }
  NdbMutex_Unlock(&m_restart_synch_mutex);
}

void Dblqh::send_CONTINUEB_all(Signal *signal, Uint32 continueb_case) {
  signal->theData[0] = continueb_case;
  if (!isNdbMtLqh()) {
    sendSignal(DBLQH_REF, GSN_CONTINUEB, signal, 1, JBB);
  } else {
    for (Uint32 instance = 1; instance <= globalData.ndbMtLqhWorkers;
         instance++) {
      BlockReference ref = numberToRef(DBLQH, instance, getOwnNodeId());
      sendSignal(ref, GSN_CONTINUEB, signal, 1, JBB);
    }
  }
}

/* ######################################################################## */
/* #######                        ADD/DELETE FRAGMENT MODULE        ####### */
/*     THIS MODULE IS USED BY DICTIONARY TO CREATE NEW FRAGMENTS AND DELETE */
/*     OLD FRAGMENTS.                                                       */
/*                                                                          */
/* ######################################################################## */
/* -------------------------------------------------------------- */
/*            FRAG REQ                                            */
/* -------------------------------------------------------------- */
/* *********************************************************> */
/*  LQHFRAGREQ: Create new fragments for a table. Sender DICT */
/* *********************************************************> */

// this unbelievable mess could be replaced by one signal to LQH
// and execute direct to local DICT to get everything at once
void Dblqh::execCREATE_TAB_REQ(Signal *signal) {
  CreateTabReq *req = (CreateTabReq *)signal->getDataPtr();
  tabptr.i = req->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  Uint32 senderRef = req->senderRef;
  Uint32 senderData = req->senderData;

  if (tabptr.p->tableStatus != Tablerec::NOT_DEFINED) {
    jam();
    CreateTabRef *ref = (CreateTabRef *)signal->getDataPtrSend();
    ref->senderData = senderData;
    ref->senderRef = reference();
    ref->errorCode = CreateTableRef::TableAlreadyExist;
    sendSignal(senderRef, GSN_CREATE_TAB_REF, signal,
               CreateTabRef::SignalLength, JBB);
    return;
  }

  seizeAddfragrec(signal);
  addfragptr.p->m_createTabReq = *req;
  req = &addfragptr.p->m_createTabReq;

  DEB_SCHEMA_VERSION(
      ("(%u)tab: %u tableStatus = ADD_TABLE_ONGOING", instance(), tabptr.i));
  tabptr.p->tableStatus = Tablerec::ADD_TABLE_ONGOING;
  tabptr.p->tableType = req->tableType;
  tabptr.p->m_addfragptr_i = RNIL;
  tabptr.p->primaryTableId =
      (req->primaryTableId == RNIL ? tabptr.i : req->primaryTableId);
  tabptr.p->schemaVersion = req->tableVersion;
  DEB_SCHEMA_VERSION(("(%u)tab(%u): %u tableStatus = ADD_TABLE_ONGOING",
                      instance(), tabptr.p->schemaVersion, tabptr.i));
  tabptr.p->m_disk_table = 0;

  addfragptr.p->addfragStatus = AddFragRecord::WAIT_TUP;
  sendCreateTabReq(signal, addfragptr);
}

void Dblqh::sendCreateTabReq(Signal *signal, AddFragRecordPtr addfragptr) {
  TablerecPtr tabPtr;
  tabPtr.i = addfragptr.p->m_createTabReq.tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  ndbassert(!m_is_query_block);
  CreateTabReq *req = (CreateTabReq *)signal->getDataPtrSend();
  *req = addfragptr.p->m_createTabReq;

  req->senderRef = reference();
  req->senderData = addfragptr.i;

  Uint32 ref = calcInstanceBlockRef(DBTUP);
  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::WAIT_TUP:
      if (DictTabInfo::isOrderedIndex(tabPtr.p->tableType)) {
        jam();
        req->noOfAttributes = 1;
        req->noOfKeyAttr = 1;
        req->noOfNullAttributes = 0;
      }
      break;
    case AddFragRecord::WAIT_TUX:
      jam();
      ndbrequire(req->noOfAttributes >= 2);
      req->noOfAttributes--;
      ref = calcInstanceBlockRef(DBTUX);
      break;
    default:
      jamLine(addfragptr.p->addfragStatus);
      ndbabort();
  }

  sendSignal(ref, GSN_CREATE_TAB_REQ, signal, CreateTabReq::SignalLengthLDM,
             JBB);
}

void Dblqh::execCREATE_TAB_REF(Signal *signal) {
  jamEntry();

  CreateTabRef *ref = (CreateTabRef *)signal->getDataPtr();
  addfragptr.i = ref->senderData;
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);

  abortAddFragOps(signal);

  ref->senderRef = reference();
  ref->senderData = addfragptr.p->m_createTabReq.senderData;
  sendSignal(addfragptr.p->m_createTabReq.senderRef, GSN_CREATE_TAB_REF, signal,
             CreateTabConf::SignalLength, JBB);

  releaseAddfragrec(signal);
}

void Dblqh::execCREATE_TAB_CONF(Signal *signal) {
  jamEntry();
  CreateTabConf *conf = (CreateTabConf *)signal->getDataPtr();
  addfragptr.i = conf->senderData;
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);

  TablerecPtr tabPtr;
  tabPtr.i = addfragptr.p->m_createTabReq.tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::WAIT_TUP:
      jam();
      addfragptr.p->tupConnectptr = conf->tupConnectPtr;
      if (DictTabInfo::isOrderedIndex(tabPtr.p->tableType)) {
        jam();
        addfragptr.p->addfragStatus = AddFragRecord::WAIT_TUX;
        sendCreateTabReq(signal, addfragptr);
        return;
      }
      break;
    case AddFragRecord::WAIT_TUX:
      jam();
      addfragptr.p->tuxConnectptr = conf->tuxConnectPtr;
      break;
    default:
      jamLine(addfragptr.p->addfragStatus);
      ndbabort();
  }

  addfragptr.p->addfragStatus = AddFragRecord::WAIT_ADD_ATTR;

  conf->senderRef = reference();
  conf->senderData = addfragptr.p->m_createTabReq.senderData;
  conf->lqhConnectPtr = addfragptr.i;
  sendSignal(addfragptr.p->m_createTabReq.senderRef, GSN_CREATE_TAB_CONF,
             signal, CreateTabConf::SignalLength, JBB);
}

/* ************************************************************************> */
/*  LQHADDATTRREQ: Request from DICT to create attributes for the new table. */
/* ************************************************************************> */
void Dblqh::execLQHADDATTREQ(Signal *signal) {
  jamEntry();
  LqhAddAttrReq *req = (LqhAddAttrReq *)signal->getDataPtr();

  addfragptr.i = req->lqhFragPtr;
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);

  addfragptr.p->m_addAttrReq = *req;

  const Uint32 tnoOfAttr = req->noOfAttributes;
  const Uint32 numSections = signal->getNoOfSections();
  bool isLongReq = (numSections != 0);
  addfragptr.p->defValSectionI = RNIL;
  addfragptr.p->defValNextPos = 0;

  if (isLongReq) {
    SectionHandle handle(this, signal);
    SegmentedSectionPtr defValSection;
    ndbrequire(handle.getSection(defValSection,
                                 LqhAddAttrReq::DEFAULT_VALUE_SECTION_NUM));
    addfragptr.p->defValSectionI = defValSection.i;
    addfragptr.p->defValNextPos = 0;
    // Don't free Section here. Section is freed after default values are
    // trasfered to TUP
    handle.clear();
  }

  ndbrequire(addfragptr.p->addfragStatus == AddFragRecord::WAIT_ADD_ATTR);
  ndbrequire((tnoOfAttr != 0) && (tnoOfAttr <= LqhAddAttrReq::MAX_ATTRIBUTES));
  addfragptr.p->totalAttrReceived += tnoOfAttr;
  ndbrequire(addfragptr.p->totalAttrReceived <=
             addfragptr.p->m_createTabReq.noOfAttributes);

  addfragptr.p->attrReceived = tnoOfAttr;

  TablerecPtr tabPtr;
  tabPtr.i = addfragptr.p->m_createTabReq.tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  for (Uint32 i = 0; i < tnoOfAttr; i++) {
    if (AttributeDescriptor::getDiskBased(req->attributes[i].attrDescriptor)) {
      jam();
      tabPtr.p->m_disk_table = 1;
    }
  }  // for

  addfragptr.p->attrSentToTup = 0;
  addfragptr.p->addfragStatus = AddFragRecord::TUP_ATTR_WAIT;
  sendAddAttrReq(signal);
}  // Dblqh::execLQHADDATTREQ()

/* *********************>> */
/*  TUP_ADD_ATTCONF      > */
/* *********************>> */
void Dblqh::execTUP_ADD_ATTCONF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  // implies that operation was released on the other side
  const bool lastAttr = signal->theData[1];
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);

  tabptr.i = addfragptr.p->m_createTabReq.tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  Uint32 noOfAttr = addfragptr.p->m_createTabReq.noOfAttributes;

  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::TUP_ATTR_WAIT:
      if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
        addfragptr.p->addfragStatus = AddFragRecord::TUX_ATTR_WAIT;
        sendAddAttrReq(signal);
        break;
      }
      goto done_with_attr;
      break;
    case AddFragRecord::TUX_ATTR_WAIT:
      jam();
      if (lastAttr) addfragptr.p->tuxConnectptr = RNIL;
      goto done_with_attr;
      break;
    done_with_attr:
      addfragptr.p->attrSentToTup = addfragptr.p->attrSentToTup + 1;
      ndbrequire(addfragptr.p->attrSentToTup <= addfragptr.p->attrReceived);
      ndbrequire(addfragptr.p->totalAttrReceived <= noOfAttr);
      if (addfragptr.p->attrSentToTup < addfragptr.p->attrReceived) {
        // more in this batch
        jam();
        addfragptr.p->addfragStatus = AddFragRecord::TUP_ATTR_WAIT;
        sendAddAttrReq(signal);
        return;
      }

      if (addfragptr.p->defValSectionI != RNIL) {
        releaseSection(addfragptr.p->defValSectionI);
        addfragptr.p->defValNextPos = 0;
        addfragptr.p->defValSectionI = RNIL;
      }

      {  // Reply
        LqhAddAttrConf *const conf = (LqhAddAttrConf *)signal->getDataPtrSend();
        conf->senderData = addfragptr.p->m_addAttrReq.senderData;
        conf->senderAttrPtr = addfragptr.p->m_addAttrReq.senderAttrPtr;
        sendSignal(addfragptr.p->m_createTabReq.senderRef, GSN_LQHADDATTCONF,
                   signal, LqhAddAttrConf::SignalLength, JBB);
      }
      if (addfragptr.p->totalAttrReceived < noOfAttr) {
        jam();
        addfragptr.p->addfragStatus = AddFragRecord::WAIT_ADD_ATTR;
      } else {
        jam();
        releaseAddfragrec(signal);
      }
      break;
    default:
      ndbabort();
  }
}

/* **********************>> */
/*  TUX_ADD_ATTRCONF      > */
/* **********************>> */
void Dblqh::execTUX_ADD_ATTRCONF(Signal *signal) {
  jamEntry();
  execTUP_ADD_ATTCONF(signal);
}  // Dblqh::execTUX_ADD_ATTRCONF

/* *********************> */
/*  TUP_ADD_ATTREF      > */
/* *********************> */
void Dblqh::execTUP_ADD_ATTRREF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  const Uint32 errorCode = terrorCode = signal->theData[1];

  abortAddFragOps(signal);

  // operation was released on the other side
  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::TUP_ATTR_WAIT:
      jam();
      break;
    case AddFragRecord::TUX_ATTR_WAIT:
      jam();
      break;
    default:
      ndbabort();
  }

  if (addfragptr.p->defValSectionI != RNIL) {
    releaseSection(addfragptr.p->defValSectionI);
    addfragptr.p->defValNextPos = 0;
    addfragptr.p->defValSectionI = RNIL;
  }

  const Uint32 Ref = addfragptr.p->m_createTabReq.senderRef;
  const Uint32 senderData = addfragptr.p->m_addAttrReq.senderData;

  releaseAddfragrec(signal);

  LqhAddAttrRef *const ref = (LqhAddAttrRef *)signal->getDataPtrSend();
  ref->senderData = senderData;
  ref->errorCode = errorCode;
  sendSignal(Ref, GSN_LQHADDATTREF, signal, LqhAddAttrRef::SignalLength, JBB);
}  // Dblqh::execTUP_ADD_ATTRREF()

/* **********************> */
/*  TUX_ADD_ATTRREF      > */
/* **********************> */
void Dblqh::execTUX_ADD_ATTRREF(Signal *signal) {
  jamEntry();
  execTUP_ADD_ATTRREF(signal);
}  // Dblqh::execTUX_ADD_ATTRREF

/*
 * Add attribute in TUP or TUX.  Called up to 4 times.
 */
void Dblqh::sendAddAttrReq(Signal *signal) {
  arrGuard(addfragptr.p->attrSentToTup, LqhAddAttrReq::MAX_ATTRIBUTES);
  LqhAddAttrReq::Entry &entry =
      addfragptr.p->m_addAttrReq.attributes[addfragptr.p->attrSentToTup];

  ndbassert(!m_is_query_block);
  const Uint32 attrId = entry.attrId & 0xffff;
  const Uint32 primaryAttrId = entry.attrId >> 16;

  tabptr.i = addfragptr.p->m_createTabReq.tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  if (addfragptr.p->addfragStatus == AddFragRecord::TUP_ATTR_WAIT) {
    if (DictTabInfo::isTable(tabptr.p->tableType) ||
        DictTabInfo::isHashIndex(tabptr.p->tableType) ||
        (DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
         primaryAttrId == ZNIL)) {
      jam();
      TupAddAttrReq *const tupreq = (TupAddAttrReq *)signal->getDataPtrSend();
      tupreq->tupConnectPtr = addfragptr.p->tupConnectptr;
      tupreq->attrId = attrId;
      tupreq->attrDescriptor = entry.attrDescriptor;
      tupreq->extTypeInfo = entry.extTypeInfo;
      BlockReference tupRef = calcInstanceBlockRef(DBTUP);

      Uint32 sectionLen = 0;
      Uint32 startIndex = TupAddAttrReq::SignalLength;
      if (addfragptr.p->defValSectionI != RNIL) {
        SegmentedSectionPtr defValSection;
        getSection(defValSection, addfragptr.p->defValSectionI);

        SectionReader defValueReader(defValSection, getSectionSegmentPool());

        ndbrequire(defValueReader.step(addfragptr.p->defValNextPos));

        Uint32 defValueHeader;
        ndbrequire(defValueReader.peekWord(&defValueHeader));

        AttributeHeader ah(defValueHeader);
        Uint32 defValueLen = ah.getByteSize();
        Uint32 defValueWords = ((defValueLen + 3) / 4) + 1;
        Uint32 *dst = &signal->theData[startIndex];
        ndbrequire(defValueReader.getWords(dst, defValueWords));
        addfragptr.p->defValNextPos += defValueWords;
        sectionLen = defValueWords;
      }

      // A long section is attached when a default value is sent.
      if (sectionLen != 0) {
        LinearSectionPtr ptr[3];
        ptr[0].p = &signal->theData[startIndex];
        ptr[0].sz = sectionLen;
        sendSignal(tupRef, GSN_TUP_ADD_ATTRREQ, signal,
                   TupAddAttrReq::SignalLength, JBB, ptr, 1);
      } else
        sendSignal(tupRef, GSN_TUP_ADD_ATTRREQ, signal,
                   TupAddAttrReq::SignalLength, JBB);

      return;
    }
    if (DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
        primaryAttrId != ZNIL) {
      // this attribute is not for TUP
      jam();
      TupAddAttrConf *tupconf = (TupAddAttrConf *)signal->getDataPtrSend();
      tupconf->userPtr = addfragptr.i;
      tupconf->lastAttr = false;
      sendSignal(reference(), GSN_TUP_ADD_ATTCONF, signal,
                 TupAddAttrConf::SignalLength, JBB);
      return;
    }
  }

  if (addfragptr.p->addfragStatus == AddFragRecord::TUX_ATTR_WAIT) {
    jam();
    if (DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
        primaryAttrId != ZNIL) {
      jam();
      TuxAddAttrReq *const tuxreq = (TuxAddAttrReq *)signal->getDataPtrSend();
      tuxreq->tuxConnectPtr = addfragptr.p->tuxConnectptr;
      tuxreq->notused1 = 0;
      tuxreq->attrId = attrId;
      tuxreq->attrDescriptor = entry.attrDescriptor;
      tuxreq->extTypeInfo = entry.extTypeInfo;
      tuxreq->primaryAttrId = primaryAttrId;
      BlockReference tuxRef = calcInstanceBlockRef(DBTUX);
      sendSignal(tuxRef, GSN_TUX_ADD_ATTRREQ, signal,
                 TuxAddAttrReq::SignalLength, JBB);
      return;
    }
    if (DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
        primaryAttrId == ZNIL) {
      // this attribute is not for TUX
      jam();
      TuxAddAttrConf *tuxconf = (TuxAddAttrConf *)signal->getDataPtrSend();
      tuxconf->userPtr = addfragptr.i;
      tuxconf->lastAttr = false;
      sendSignal(reference(), GSN_TUX_ADD_ATTRCONF, signal,
                 TuxAddAttrConf::SignalLength, JBB);
      return;
    }
  }
  ndbabort();
}  // Dblqh::sendAddAttrReq

/**
 * Return the schemaVersion-part that changes when table is created.
 * This function can be called during restart from thread where
 * TSMAN belongs, so it is important to not use any block variables
 * here. The table object should not change here during this phase
 * since it is in a very specific restart phase.
 */
Uint32 Dblqh::getCreateSchemaVersion(Uint32 tableId) {
  TablerecPtr tabPtr;
  tabPtr.i = tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  Uint32 schemaVersion;
  if (tabPtr.p->tableStatus == Tablerec::TABLE_DEFINED ||
      tabPtr.p->tableStatus == Tablerec::TABLE_READ_ONLY) {
    schemaVersion = (tabPtr.p->schemaVersion & 0xFFFFFF);
  } else {
    schemaVersion = 0;
  }
#ifdef DEBUG_SCHEMA_VERSION
  if (schemaVersion == 0) {
    g_eventLogger->info("(%u) table: %u, schemaVersion: %u, tableStatus: %u",
                        instance(), tableId, tabPtr.p->schemaVersion,
                        tabPtr.p->tableStatus);
  }
#endif
  return schemaVersion;
}

void Dblqh::execLQHFRAGREQ(Signal *signal) {
  jamEntry();
  {
    LqhFragReq *req = (LqhFragReq *)signal->getDataPtr();
    if (signal->length() == LqhFragReq::OldestSignalLength) {
      jam();
      ndbabort(); /* Not supported to upgrade from < 7.2 */
      /**
       * Upgrade support to specify partitionId
       */
      req->partitionId = req->fragmentId;
      /**
       * Upgrade support to specify createGci
       */
      req->createGci = 0;
    }
    if (signal->length() == LqhFragReq::OldSignalLength) {
      jam();
      ndbabort(); /* Not supported to upgrade from < 7.2 */
      /**
       * Upgrade support to specify createGci
       */
      req->createGci = 0;
    }
  }

  c_num_fragments_created_since_restart++;

  LqhFragReq copy = *(LqhFragReq *)signal->getDataPtr();
  LqhFragReq *req = &copy;

  tabptr.i = req->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  if (tabptr.p->tableStatus != Tablerec::ADD_TABLE_ONGOING &&
      (AlterTableReq::getAddFragFlag(req->changeMask) == 0)) {
    jam();
    fragrefLab(signal, ZTAB_STATE_ERROR, req);
    return;
  }  // if

  if (getFragmentrec(req->fragId)) {
    jam();
    fragrefLab(signal, terrorCode, req);
    return;
  }  // if

  if (!insertFragrec(signal, req->fragId)) {
    jam();
    fragrefLab(signal, terrorCode, req);
    return;
  }  // if

  Uint32 copyType = req->requestInfo & 3;
  bool tempTable = ((req->requestInfo & LqhFragReq::TemporaryTable) != 0);
  initFragrec(signal, tabptr.i, req->fragId, copyType);
  fragptr.p->createGci = req->createGci;
  fragptr.p->startGci = req->startGci;
  fragptr.p->newestGci = req->startGci;
  DEB_NEWEST_GCI(("%u) execLQHFRAGREQ: tab(%u,%u) fragptr.p->newestGci = %u",
                  instance(), fragptr.p->tabRef, fragptr.p->fragId,
                  fragptr.p->newestGci));
  DEB_LCP(
      ("(%u)LQHFRAGREQ: tab(%u,%u) createGci: %u, startGci: %u,"
       " newestGci: %u",
       instance(), tabptr.i, req->fragId, fragptr.p->createGci,
       fragptr.p->startGci, fragptr.p->newestGci));
  set_min_keep_gci(fragptr.p->createGci);

  if (fragptr.p->newestGci < req->createGci) {
    jam();
    fragptr.p->newestGci = req->createGci;
    DEB_NEWEST_GCI(
        ("%u) execLQHFRAGREQ(2): tab(%u,%u)"
         " fragptr.p->newestGci = %u",
         instance(), fragptr.p->tabRef, fragptr.p->fragId,
         fragptr.p->newestGci));
  }
  ndbrequire(tabptr.p->tableType < 256);
  fragptr.p->tableType = (Uint8)tabptr.p->tableType;

  {
    fragptr.p->lqhInstanceKey = getInstanceKey(tabptr.i, req->fragId);
    ndbrequire(fragptr.p->lqhInstanceKey == (req->logPartId + 1));
    LogPartRecord *logPartPtrP = get_log_part_record(fragptr.p->lqhInstanceKey);
    fragptr.p->m_log_part_ptr_p = logPartPtrP;
    DEB_COMMITTED_WORDS(
        ("(%u) FragId: %u, instKey: %u, logPart: %u,"
         " phys_log_part: %u, logPartPtrI: %u",
         instance(), req->fragId, fragptr.p->lqhInstanceKey,
         logPartPtrP->logPartNo, logPartPtrP->logPartNo, logPartPtrP->ptrI));
  }

  /* Init per-frag op counters */
  fragptr.p->m_useStat.init();

  if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    // find corresponding primary table fragment
    TablerecPtr tTablePtr;
    tTablePtr.i = tabptr.p->primaryTableId;
    ptrCheckGuard(tTablePtr, ctabrecFileSize, tablerec);
    FragrecordPtr tFragPtr;
    tFragPtr.i = RNIL;
    for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tTablePtr.p->fragid); i++) {
      if (tTablePtr.p->fragid[i] == fragptr.p->fragId) {
        jam();
        tFragPtr.i = tTablePtr.p->fragrec[i];
        break;
      }
    }
    ndbrequire(tFragPtr.i != RNIL);
    // store it
    fragptr.p->tableFragptr = tFragPtr.i;
  } else {
    jam();
    fragptr.p->tableFragptr = fragptr.i;
  }

  if (tempTable) {
    //--------------------------------------------
    // reqinfo bit 3-4 = 2 means temporary table
    // without logging or checkpointing.
    //--------------------------------------------
    jam();
    fragptr.p->logFlag = Fragrecord::STATE_FALSE;
    fragptr.p->lcpFlag = Fragrecord::LCP_STATE_FALSE;
  }  // if

  seizeAddfragrec(signal);
  addfragptr.p->m_lqhFragReq = *req;
  addfragptr.p->fragmentPtr = fragptr.i;

  if (DictTabInfo::isTable(tabptr.p->tableType) ||
      DictTabInfo::isHashIndex(tabptr.p->tableType)) {
    jam();
    AccFragReq *const accreq = (AccFragReq *)signal->getDataPtrSend();
    accreq->userPtr = addfragptr.i;
    accreq->userRef = cownref;
    accreq->tableId = tabptr.i;
    accreq->reqInfo = 0;
    accreq->fragId = req->fragId;
    accreq->localKeyLen = addfragptr.p->m_lqhFragReq.localKeyLength;
    accreq->maxLoadFactor = addfragptr.p->m_lqhFragReq.maxLoadFactor;
    accreq->minLoadFactor = addfragptr.p->m_lqhFragReq.minLoadFactor;
    accreq->kValue = addfragptr.p->m_lqhFragReq.kValue;
    accreq->lhFragBits = addfragptr.p->m_lqhFragReq.lh3DistrBits;
    accreq->lhDirBits = addfragptr.p->m_lqhFragReq.lh3PageBits;
    accreq->keyLength = addfragptr.p->m_lqhFragReq.keyLength;
    /* --------------------------------------------------------------------- */
    /* Send ACCFRAGREQ, when confirmation is received send 2 * TUPFRAGREQ to */
    /* create 2 tuple fragments on this node.                                */
    /* --------------------------------------------------------------------- */
    addfragptr.p->addfragStatus = AddFragRecord::ACC_ADDFRAG;
    sendSignal(fragptr.p->accBlockref, GSN_ACCFRAGREQ, signal,
               AccFragReq::SignalLength, JBB);
    return;
  }
  if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    addfragptr.p->addfragStatus = AddFragRecord::WAIT_TUP;
    sendAddFragReq(signal);
    return;
  }
  ndbabort();
}  // Dblqh::execLQHFRAGREQ()

/* *************** */
/*  ACCFRAGCONF  > */
/* *************** */
void Dblqh::execACCFRAGCONF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  Uint32 taccConnectptr = signal->theData[1];
  // Uint32 fragId1 = signal->theData[2];
  Uint32 accFragPtr1 = signal->theData[4];
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  ndbrequire(addfragptr.p->addfragStatus == AddFragRecord::ACC_ADDFRAG);

  addfragptr.p->accConnectptr = taccConnectptr;
  fragptr.i = addfragptr.p->fragmentPtr;
  c_fragment_pool.getPtr(fragptr);
  fragptr.p->accFragptr = accFragPtr1;

  addfragptr.p->addfragStatus = AddFragRecord::WAIT_TUP;
  sendAddFragReq(signal);
}  // Dblqh::execACCFRAGCONF()

/* *************** */
/*  TUPFRAGCONF  > */
/* *************** */
void Dblqh::execTUPFRAGCONF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  Uint32 tupConnectptr = signal->theData[1];
  Uint32 tupFragPtr = signal->theData[2]; /* TUP FRAGMENT POINTER */
  // Uint32 localFragId = signal->theData[3];  /* LOCAL FRAGMENT ID    */
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  fragptr.i = addfragptr.p->fragmentPtr;
  c_fragment_pool.getPtr(fragptr);
  tabptr.i = fragptr.p->tabRef;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::WAIT_TUP:
      jam();
      fragptr.p->tupFragptr = tupFragPtr;
      addfragptr.p->tupConnectptr = tupConnectptr;
      if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
        addfragptr.p->addfragStatus = AddFragRecord::WAIT_TUX;
        sendAddFragReq(signal);
        break;
      }
      c_acc->set_tup_fragptr(fragptr.p->accFragptr, tupFragPtr);
      goto done_with_frag;
      break;
    case AddFragRecord::WAIT_TUX:
      jam();
      fragptr.p->tuxFragptr = tupFragPtr;
      addfragptr.p->tuxConnectptr = tupConnectptr;
      goto done_with_frag;
      break;
    done_with_frag:
      /* ---------------------------------------------------------------- */
      /* Finished create of fragments. Now ready for creating attributes. */
      /* ---------------------------------------------------------------- */
      fragptr.p->fragStatus = Fragrecord::FSACTIVE;
      {
        LqhFragConf *conf = (LqhFragConf *)signal->getDataPtrSend();
        conf->senderData = addfragptr.p->m_lqhFragReq.senderData;
        conf->lqhFragPtr = RNIL;
        conf->tableId = addfragptr.p->m_lqhFragReq.tableId;
        conf->fragId = fragptr.p->fragId;
        conf->changeMask = addfragptr.p->m_lqhFragReq.changeMask;
        sendSignal(addfragptr.p->m_lqhFragReq.senderRef, GSN_LQHFRAGCONF,
                   signal, LqhFragConf::SignalLength, JBB);
      }
      releaseAddfragrec(signal);
      break;
    default:
      ndbabort();
  }
}  // Dblqh::execTUPFRAGCONF()

/* *************** */
/*  TUXFRAGCONF  > */
/* *************** */
void Dblqh::execTUXFRAGCONF(Signal *signal) {
  jamEntry();
  execTUPFRAGCONF(signal);
}  // Dblqh::execTUXFRAGCONF

/*
 * Add fragment in TUP or TUX.  Called up to 4 times.
 */
void Dblqh::sendAddFragReq(Signal *signal) {
  fragptr.i = addfragptr.p->fragmentPtr;
  c_fragment_pool.getPtr(fragptr);
  tabptr.i = fragptr.p->tabRef;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  if (addfragptr.p->addfragStatus == AddFragRecord::WAIT_TUP) {
    TupFragReq *const tupFragReq = (TupFragReq *)signal->getDataPtrSend();
    tupFragReq->userPtr = addfragptr.i;
    tupFragReq->userRef = cownref;
    tupFragReq->reqInfo = 0; /* ADD TABLE */
    tupFragReq->tableId = tabptr.i;
    tupFragReq->fragId = addfragptr.p->m_lqhFragReq.fragId;
    tupFragReq->tablespaceid = addfragptr.p->m_lqhFragReq.tablespace_id;
    tupFragReq->maxRowsHigh = addfragptr.p->m_lqhFragReq.maxRowsHigh;
    tupFragReq->maxRowsLow = addfragptr.p->m_lqhFragReq.maxRowsLow;
    tupFragReq->minRowsHigh = addfragptr.p->m_lqhFragReq.minRowsHigh;
    tupFragReq->minRowsLow = addfragptr.p->m_lqhFragReq.minRowsLow;
    tupFragReq->changeMask = addfragptr.p->m_lqhFragReq.changeMask;
    tupFragReq->partitionId = addfragptr.p->m_lqhFragReq.partitionId;
    sendSignal(fragptr.p->tupBlockref, GSN_TUPFRAGREQ, signal,
               TupFragReq::SignalLength, JBB);
    return;
  }
  if (addfragptr.p->addfragStatus == AddFragRecord::WAIT_TUX) {
    jam();
    ndbrequire(DictTabInfo::isOrderedIndex(tabptr.p->tableType));
    TuxFragReq *const tuxreq = (TuxFragReq *)signal->getDataPtrSend();
    tuxreq->userPtr = addfragptr.i;
    tuxreq->userRef = cownref;
    tuxreq->reqInfo = 0; /* ADD TABLE */
    tuxreq->tableId = tabptr.i;
    tuxreq->fragId = addfragptr.p->m_lqhFragReq.fragId;
    tuxreq->primaryTableId = tabptr.p->primaryTableId;
    // pointer to index fragment in TUP
    tuxreq->tupIndexFragPtrI = fragptr.p->tupFragptr;
    // pointers to table fragments in TUP and ACC
    FragrecordPtr tFragPtr;
    tFragPtr.i = fragptr.p->tableFragptr;
    c_fragment_pool.getPtr(tFragPtr);
    tuxreq->tupTableFragPtrI = tFragPtr.p->tupFragptr;
    tuxreq->accTableFragPtrI = tFragPtr.p->accFragptr;
    sendSignal(fragptr.p->tuxBlockref, GSN_TUXFRAGREQ, signal,
               TuxFragReq::SignalLength, JBB);
    return;
  }
}  // Dblqh::sendAddFragReq

/* ************************************************************************>> */
/*  TAB_COMMITREQ: Commit the new table for use in transactions. Sender DICT. */
/* ************************************************************************>> */
void Dblqh::insert_new_fragments_into_lcp(Signal *signal) {
  /**
   * When a disk data table is recovered its recovery will start by executing
   * the UNDO. The idea is that each fragment LCP records the LCP identity
   * that the fragment LCP will assist to restore. This means that the UNDO
   * log will at least be executed until the UNDO log record indicating the
   * start of this LCP id is found in the UNDO log. If no LCP have yet been
   * executed the UNDO log will be executed until the beginning of the UNDO
   * log.
   *
   * Thus we are safe that all UNDO logs will be executed to restore exactly
   * the page state at the moment the checkpoint was started. The LCP code in
   * the BACKUP block ensures that for each fragment LCP this is a well
   * defined time to ensure that the disk data record is synchronized with the
   * in-memory records. In particular this refers to the pointers from disk
   * data record to in-memory record and pointer from in-memory record to disk
   * data record.
   *
   * When a crash occurs and there is no fragment LCP yet executed for the
   * table we have a some what more complicated problem to handle. In this
   * case we want to UNDO everything until the table was created.
   *
   * If we do nothing special we cannot trust that the UNDO log is executed
   * to this point. As an example we could be creating the table while LCP
   * 14 is ongoing. However since the table didn't exist when LCP 14 was
   * started in the master DIH, there will not be any LCP_FRAG_ORD sent to
   * execute this fragment LCP. This means that the first LCP the table will
   * participate in is LCP 15.
   *
   * Now if the node crashes when all fragment LCPs from LCP 14 have completed
   * but not all of them have completed LCP 15 we are still safe since at least
   * one fragment LCP will require restore from LCP 14. Since the table was
   * created after the start of LCP 14 we know that the UNDO log execution will
   * undo all rows inserted since the table was created.
   *
   * However one case exists that we cannot handle. This is when all fragment
   * LCPs have completed  LCP 15 except the fragment LCPs of the new table.
   *
   * This is not necessarily a rare event since the new table is likely to get
   * the highest table id of all tables and thus its fragment LCPs are executed
   * after all other fragment LCPs have completed.
   *
   * There are many ways to solve this problem. One problem would be to ensure
   * that a fragment LCP has been executed before the API users are allowed to
   * execute any writes to the table.
   *
   * Another approach would be to know that all extents belonging to the table
   * must be returned at recovery and thus ensuring that we create an empty
   * set of disk data records which is the desired outcome.
   *
   * None of those approaches are well suited to the current code in NDB, they
   * would both require substantial changes to the LCP code and UNDO log
   * recovery code.
   *
   * The approach we have followed here is to ensure that the LCP is started
   * from LQH independent of when DIH decides to start it. This approach is
   * well aligned with the current development that makes LCPs more and more
   * local to LQH. LQH already maintains a queue of fragments to execute
   * fragment LCPs on.
   *
   * So the basic idea of our approach is to insert the fragment into the
   * queue of fragment LCPs. By so doing the above problem is no longer an
   * issue since we will execute the first fragment LCP already in LCP 14
   * and thus there is no chance that we can find ourself in the situation
   * described above.
   *
   * So now we need to describe a proof that our approach works.
   * The basis of this proof is to consider how the insert into the LCP
   * fragment queue is handled dependent on the progress of the LCPs.
   *
   * Case 1:
   * We create the table during execution of an LCP, say for example LCP 14.
   * We will insert the table last in the LCP fragment queue in this
   * function. This function is called from execTAB_COMMITREQ, this signal
   * is called before any write operation can start against the table.
   *
   * We conclude that since we insert the fragment into the LCP queue before
   * the LCP 14 is completed we know that the first fragment LCP of the new
   * table will happen in LCP 14. Thus recovery after LCP 15 have started is
   * trivial since there will a fragment LCP to restore.
   *
   * Thus the only case to consider is when the node crashes before the
   * fragment LCP has been made restorable. This happens before LCP 14 is
   * completed and thus any recovery will at least execute UNDO log records
   * until the start of LCP 14.
   *
   * One exception is that some fragment LCP might even require execution of
   * UNDO log records back to LCP 13 (or beginning of UNDO log if LCP 14 was
   * the first LCP in the UNDO log).
   *
   * Another very rare exception is that during recovery the new table is the
   * only fragment to restore (or that all fragments are new fragments
   * without an LCP to execute). This can happen in nodes added where not
   * even SYSTAB_0 is available.
   *
   * In this special case the LCP id that we will execute UNDO log unto is
   * LCP (0,0). There is a rare case that could cause issues here. This is
   * when we have had older tables that made us fill the UNDO log at least
   * once. In this case we might execute the UNDO log and never find the
   * UNDO log record referring to the end of the UNDO log and neither will
   * we find the LCP record of the LCP (0,0).
   *
   * In this we get the LCP identity from DIH by using SYSFILE->latestLCP_ID.
   * If SYSFILE->latestLCP_ID is 15 it means that LCP 14 was fully completed
   * before the crash. It is important that we grab the SYSFILE->latestLCP_ID
   * from the local node and not from the master DIH. This number is sent to
   * NDBCNTR in the early phases. We make sure that this LCP id is also
   * present in DBLQH to handle this rare special case.
   *
   * If NDBCNTR reports this number as 15, we know that LCP 14 was fully
   * completed before the crash. The only reason why LCP 14 might not be fully
   * completed is if we had a cluster restart that failed, but in that case we
   * have completed executing the UNDO log fully before we crashed the cluster
   * restart. Thus there will be no UNDO log execution required at all in this
   * case for the new tables since they have already been undone.
   *
   * Thus we have proven that with those two extra measures, first to insert
   * the fragment into the LCP queue before the CREATE TABLE completes and
   * by ensuring that DBLQH receives the latest LCP id we are sure that the
   * UNDO recovery of the new table will be performed in a correct manner.
   *
   * Case 2:
   * The second option is that the table is created after an LCP has
   * completed and before the next LCP has started.
   *
   * In this case we need to ensure that the fragment LCPs are not
   * executed at the end of the LCP. This comes natural by inserting
   * them into the LCP queue before any other fragment has been
   * inserted into the LCP queue.
   *
   * However we need another step to be safe in this case. The problem
   * here is that checkpoints are not completed one at a time, they are
   * executed in parallel. Especially the next fragment LCP is executed
   * while earlier fragment LCPs are still waiting for the UNDO log to
   * be written up to a certain LSN.
   *
   * To handle this in a safe we introduce a synchronisation point. All
   * fragment LCPs that are inserted when no LCP was ongoing are first
   * in the queue of fragment scheduled to execute an LCP. We introduce
   * a new signal that ensures that all of those fragment LCPs are
   * completed before we start with any fragment LCPs requested by DIH.
   *
   * This method ensures that the fragment LCPs for new tables are not
   * placed at the end of the LCP, rather it is placed even before any
   * other fragment LCP is executed. Thus a stronger condition than
   * required, but certainly strong enough.
   *
   * With this we are safe that if the table is created between LCP
   * 14 and 15, then the table have been checkpointed if any normal
   * table has its LCP executed for LCP 15. Thus we know that if
   * the recovery has a new table without a fragment LCP, then we know
   * that no fragment LCP has been executed in LCP 15. Thus we are
   * certain that the UNDO log will be executed back to the start of
   * LCP 14.
   *
   * Again the same reasoning as above on Case 1: also applies to
   * Case 2: when it comes to the case of no fragment with any
   * LCP executed on it.
   *
   * Thus we have proven the thesis that disk data recovery will be
   * ok for new table fragments that haven't executed their first LCP
   * at a node restart.
   *
   * A new table have been created. We must ensure that the table is
   * inserted into the list of waiting LCPs. The
   * m_create_table_flag_lcp_frag_ord indicates that there is no LCP_FRAG_ORD
   * outstanding for this fragment. The fragment is new and it should not be
   * possible for it to be part of any checkpoint yet. It will not be executed
   * until a checkpoint is started.
   *
   * m_create_table_insert_lcp indicates that this checkpoint requested has
   * been requested by a table creation. This only needs to be signalled when
   * no LCP is ongoing at the time when it is inserted. It is used to ensure
   * that the set of checkpoints on new tables between LCPs are executed AND
   * completed before any other checkpoints on any other fragments are
   * performed as described above.
   */
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabptr.p->fragid); i++) {
    FragrecordPtr curr_fragptr;
    curr_fragptr.i = tabptr.p->fragrec[i];
    if (curr_fragptr.i != RNIL) {
      jam();
      c_fragment_pool.getPtr(curr_fragptr);
      jamLine(Uint16(curr_fragptr.p->fragId));
      ndbrequire(curr_fragptr.p->lcp_frag_ord_state ==
                 Fragrecord::LCP_EXECUTED);
      curr_fragptr.p->m_create_table_flag_lcp_frag_ord = true;
      curr_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_QUEUED;

      DEB_EARLY_LCP(
          ("(%u)Create table tab(%u,%u), insert into LCP, "
           "LCP running: %u",
           instance(), curr_fragptr.p->tabRef, curr_fragptr.p->fragId,
           clcpCompletedState == LCP_RUNNING));

      c_queued_lcp_frag_ord.addLast(curr_fragptr);
      if (clcpCompletedState != LCP_RUNNING) {
        jam();
        curr_fragptr.p->m_create_table_insert_lcp = true;
      } else {
        jam();
        /**
         * It is possible that we have received all LCP_FRAG_ORD except
         * the last one containing lastFragmentFlag. Thus it is vital to
         * ensure that we kick-start the next fragment checkpoint here if
         * there is no LCP being neither prepared nor running.
         */
        if (is_lcp_idle(lcpPtr.p)) {
          jam();
          ndbrequire(lcpPtr.p->lastFragmentFlag == false);
          prepare_next_fragment_checkpoint(signal, false);
        }
      }
    }
  }
}

void Dblqh::execTAB_COMMITREQ(Signal *signal) {
  jamEntry();
  Uint32 dihPtr = signal->theData[0];
  BlockReference dihBlockref = signal->theData[1];
  tabptr.i = signal->theData[2];

  if (tabptr.i >= ctabrecFileSize) {
    jam();
    terrorCode = ZTAB_FILE_SIZE;
    signal->theData[0] = dihPtr;
    signal->theData[1] = cownNodeid;
    signal->theData[2] = tabptr.i;
    signal->theData[3] = terrorCode;
    sendSignal(dihBlockref, GSN_TAB_COMMITREF, signal, 4, JBB);
    return;
  }  // if
  ptrAss(tabptr, tablerec);
  if (tabptr.p->tableStatus != Tablerec::ADD_TABLE_ONGOING) {
    jam();
    terrorCode = ZTAB_STATE_ERROR;
    signal->theData[0] = dihPtr;
    signal->theData[1] = cownNodeid;
    signal->theData[2] = tabptr.i;
    signal->theData[3] = terrorCode;
    signal->theData[4] = tabptr.p->tableStatus;
    sendSignal(dihBlockref, GSN_TAB_COMMITREF, signal, 5, JBB);
    ndbabort();
    return;
  }  // if
  if (cstartPhase == ZNIL &&
      !DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    insert_new_fragments_into_lcp(signal);
  }
  tabptr.p->usageCountR = 0;
  tabptr.p->usageCountW = 0;
  tabptr.p->tableStatus = Tablerec::TABLE_DEFINED;
  DEB_SCHEMA_VERSION(
      ("(%u)tab: %u tableStatus = TABLE_DEFINED", instance(), tabptr.i));
  c_pgman->set_table_ready_for_prep_lcp_writes(tabptr.i, true);
  signal->theData[0] = dihPtr;
  signal->theData[1] = cownNodeid;
  signal->theData[2] = tabptr.i;
  sendSignal(dihBlockref, GSN_TAB_COMMITCONF, signal, 3, JBB);

  return;
}  // Dblqh::execTAB_COMMITREQ()

void Dblqh::fragrefLab(Signal *signal, Uint32 errorCode,
                       const LqhFragReq *req) {
  LqhFragRef *ref = (LqhFragRef *)signal->getDataPtrSend();
  ref->senderData = req->senderData;
  ref->errorCode = errorCode;
  ref->requestInfo = req->requestInfo;
  ref->tableId = req->tableId;
  ref->fragId = req->fragId;
  ref->changeMask = req->changeMask;
  sendSignal(req->senderRef, GSN_LQHFRAGREF, signal, LqhFragRef::SignalLength,
             JBB);
  return;
}  // Dblqh::fragrefLab()

/*
 * Abort on-going ops.
 */
void Dblqh::abortAddFragOps(Signal *signal) {
  if (addfragptr.p->tupConnectptr != RNIL) {
    jam();
    TupFragReq *const tupFragReq = (TupFragReq *)signal->getDataPtrSend();
    tupFragReq->userPtr = (Uint32)-1;
    tupFragReq->userRef = addfragptr.p->tupConnectptr;
    sendSignal(ctupBlockref, GSN_TUPFRAGREQ, signal, 2, JBB);
    addfragptr.p->tupConnectptr = RNIL;
  }
  if (addfragptr.p->tuxConnectptr != RNIL) {
    jam();
    TuxFragReq *const tuxFragReq = (TuxFragReq *)signal->getDataPtrSend();
    tuxFragReq->userPtr = (Uint32)-1;
    tuxFragReq->userRef = addfragptr.p->tuxConnectptr;
    sendSignal(ctuxBlockref, GSN_TUXFRAGREQ, signal, 2, JBB);
    addfragptr.p->tuxConnectptr = RNIL;
  }
}

/* ************>> */
/*  ACCFRAGREF  > */
/* ************>> */
void Dblqh::execACCFRAGREF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  Uint32 errorCode = terrorCode = signal->theData[1];
  ndbrequire(addfragptr.p->addfragStatus == AddFragRecord::ACC_ADDFRAG);

  fragrefLab(signal, errorCode, &addfragptr.p->m_lqhFragReq);
  releaseAddfragrec(signal);

  return;
}  // Dblqh::execACCFRAGREF()

/* ************>> */
/*  TUPFRAGREF  > */
/* ************>> */
void Dblqh::execTUPFRAGREF(Signal *signal) {
  jamEntry();
  addfragptr.i = signal->theData[0];
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  Uint32 errorCode = terrorCode = signal->theData[1];
  fragptr.i = addfragptr.p->fragmentPtr;
  c_fragment_pool.getPtr(fragptr);

  // no operation to release, just add some jams
  switch (addfragptr.p->addfragStatus) {
    case AddFragRecord::WAIT_TUP:
      jam();
      break;
    case AddFragRecord::WAIT_TUX:
      jam();
      break;
    default:
      ndbabort();
  }

  fragrefLab(signal, errorCode, &addfragptr.p->m_lqhFragReq);
  releaseAddfragrec(signal);

}  // Dblqh::execTUPFRAGREF()

void Dblqh::execDROP_FRAG_REQ(Signal *signal) {
  DropFragReq *req = (DropFragReq *)signal->getDataPtr();
  seizeAddfragrec(signal);
  addfragptr.p->m_dropFragReq = *req;

  /**
   * 1 - self
   * 2 - acc
   * 3 - tup
   * 4 - tux (optional)
   */
  tabptr.i = req->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  deleteFragrec(req->fragId);

  ndbassert(!m_is_query_block);
  Uint32 ref = calcInstanceBlockRef(DBACC);
  if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    ref = calcInstanceBlockRef(DBTUP);
  }

  req->senderRef = reference();
  req->senderData = addfragptr.i;
  sendSignal(ref, GSN_DROP_FRAG_REQ, signal, DropFragReq::SignalLength, JBB);
}

void Dblqh::execDROP_FRAG_REF(Signal *signal) { ndbabort(); }

void Dblqh::execDROP_FRAG_CONF(Signal *signal) {
  DropFragConf *conf = (DropFragConf *)signal->getDataPtr();
  addfragptr.i = conf->senderData;
  ptrCheckGuard(addfragptr, caddfragrecFileSize, addFragRecord);
  ndbassert(!m_is_query_block);

  Uint32 ref = RNIL;
  switch (refToMain(conf->senderRef)) {
    case DBACC:
      jam();
      ref = calcInstanceBlockRef(DBTUP);
      break;
    case DBTUP: {
      tabptr.i = addfragptr.p->m_dropFragReq.tableId;
      ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
      if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
        jam();
        ref = calcInstanceBlockRef(DBTUX);
      }
      break;
    }
    case DBTUX:
      break;
    default:
      ndbabort();
  }

  if (ref != RNIL) {
    DropFragReq *req = (DropFragReq *)signal->getDataPtrSend();
    *req = addfragptr.p->m_dropFragReq;
    req->senderRef = reference();
    req->senderData = addfragptr.i;
    sendSignal(ref, GSN_DROP_FRAG_REQ, signal, DropFragReq::SignalLength, JBB);
    return;
  }

  conf->senderRef = reference();
  conf->senderData = addfragptr.p->m_dropFragReq.senderData;
  conf->tableId = addfragptr.p->m_dropFragReq.tableId;
  conf->fragId = addfragptr.p->m_dropFragReq.fragId;
  sendSignal(addfragptr.p->m_dropFragReq.senderRef, GSN_DROP_FRAG_CONF, signal,
             DropFragConf::SignalLength, JBB);

  releaseAddfragrec(signal);
}

/* ************>> */
/*  TUXFRAGREF  > */
/* ************>> */
void Dblqh::execTUXFRAGREF(Signal *signal) {
  jamEntry();
  execTUPFRAGREF(signal);
}  // Dblqh::execTUXFRAGREF

void Dblqh::execPREP_DROP_TAB_REQ(Signal *signal) {
  jamEntry();

  PrepDropTabReq *req = (PrepDropTabReq *)signal->getDataPtr();

  Uint32 senderRef = req->senderRef;
  Uint32 senderData = req->senderData;

  TablerecPtr tabPtr;
  tabPtr.i = req->tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  Uint32 errCode = 0;
  switch (tabPtr.p->tableStatus) {
    case Tablerec::TABLE_DEFINED:
      jam();
      break;
    case Tablerec::NOT_DEFINED:
      jam();
      errCode = PrepDropTabRef::NoSuchTable;
      break;
    case Tablerec::ADD_TABLE_ONGOING:
      jam();
      errCode = PrepDropTabRef::NoSuchTable;
      break;
    case Tablerec::PREP_DROP_TABLE_DONE:
      jam();
      errCode = PrepDropTabRef::DropInProgress;
      break;
    case Tablerec::DROP_TABLE_WAIT_USAGE:
    case Tablerec::DROP_TABLE_WAIT_DONE:
    case Tablerec::DROP_TABLE_ACC:
    case Tablerec::DROP_TABLE_TUP:
    case Tablerec::DROP_TABLE_TUX:
      jam();
      errCode = PrepDropTabRef::DropInProgress;
      break;
    case Tablerec::TABLE_READ_ONLY:
      jam();
      errCode = PrepDropTabRef::InvalidTableState;
      break;
  }

  if (errCode != 0) {
    jam();

    PrepDropTabRef *ref = (PrepDropTabRef *)signal->getDataPtrSend();
    ref->senderRef = reference();
    ref->senderData = senderData;
    ref->tableId = tabPtr.i;
    ref->errorCode = errCode;
    sendSignal(senderRef, GSN_PREP_DROP_TAB_REF, signal,
               PrepDropTabRef::SignalLength, JBB);
    return;
  }

  tabPtr.p->m_senderData = senderData;
  tabPtr.p->m_senderRef = senderRef;
  c_pgman->set_table_ready_for_prep_lcp_writes(tabPtr.i, false);
  check_pgman_prep_lcp_active_prep_drop_tab(signal, tabPtr.i);
}

/**
 * In PGMAN we have a feature that writes data pages before the actual
 * checkpoint happens. These pages are tagged with PREP_LCP as the state.
 * We need to ensure that no such writes are outstanding for a table
 * that is being prepared to drop. We will only write such pages when
 * the table is active in PGMAN. We declare it as inactive when we start
 * preparing to drop the table.
 */
void Dblqh::check_pgman_prep_lcp_active_prep_drop_tab(Signal *signal,
                                                      Uint32 tabPtrI) {
  if (c_pgman->is_prep_lcp_writes_outstanding(tabPtrI)) {
    jam();
    signal->theData[0] = ZPGMAN_PREP_LCP_ACTIVE_CHECK;
    signal->theData[1] = tabPtrI;
    signal->theData[2] = 0;
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 10, 3);
    return;
  }

  TablerecPtr tabPtr;
  tabPtr.i = tabPtrI;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  tabPtr.p->tableStatus = Tablerec::PREP_DROP_TABLE_DONE;
  DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = PREP_DROP_TABLE_DONE(2)",
                      instance(), tabPtr.i));
  PrepDropTabConf *conf = (PrepDropTabConf *)signal->getDataPtrSend();
  conf->tableId = tabPtrI;
  conf->senderRef = reference();
  conf->senderData = tabPtr.p->m_senderData;
  sendSignal(tabPtr.p->m_senderRef, GSN_PREP_DROP_TAB_CONF, signal,
             PrepDropTabConf::SignalLength, JBB);
}

void Dblqh::execINFORM_BACKUP_DROP_TAB_CONF(Signal *signal) {
  TablerecPtr tabPtr;
  tabPtr.i = signal->theData[0];
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  tabPtr.p->m_informed_backup_drop_tab = true;
}

void Dblqh::check_pgman_prep_lcp_active_drop_tab(Signal *signal,
                                                 Uint32 tabPtrI) {
  if (c_pgman->is_prep_lcp_writes_outstanding(tabPtrI)) {
    jam();
    signal->theData[0] = ZPGMAN_PREP_LCP_ACTIVE_CHECK;
    signal->theData[1] = tabPtrI;
    signal->theData[2] = 1;
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 10, 3);
    return;
  }
  TablerecPtr tabPtr;
  tabPtr.i = tabPtrI;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  tabPtr.p->tableStatus = Tablerec::DROP_TABLE_WAIT_USAGE;
  DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = DROP_TABLE_WAIT_USAGE(2)",
                      instance(), tabPtr.i));
  signal->theData[0] = ZDROP_TABLE_WAIT_USAGE;
  signal->theData[1] = tabPtrI;
  signal->theData[2] = tabPtr.p->m_senderRef;
  signal->theData[3] = tabPtr.p->m_addfragptr_i;
  dropTab_wait_usage(signal);
}

void Dblqh::dropTab_wait_usage(Signal *signal) {
  TablerecPtr tabPtr;
  FragrecordPtr loc_fragptr;
  tabPtr.i = signal->theData[1];
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  ndbassert(!m_is_query_block);
  Uint32 senderRef = signal->theData[2];
  Uint32 senderData = signal->theData[3];

  ndbrequire(tabPtr.p->tableStatus == Tablerec::DROP_TABLE_WAIT_USAGE);

  if (tabPtr.p->usageCountR > 0 || tabPtr.p->usageCountW > 0) {
    jam();
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 10, 4);
    return;
  }

  bool lcpDone = true;
  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);
  if (!is_lcp_idle(lcpPtr.p)) {
    jam();

    for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabPtr.p->fragrec); i++) {
      jam();
      loc_fragptr.i = tabPtr.p->fragrec[i];
      if (loc_fragptr.i != RNIL) {
        jam();
        c_fragment_pool.getPtr(loc_fragptr);
        if ((loc_fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_QUEUED) &&
            (!ERROR_INSERTED(5089))) {
          /**
           * The fragment is queued up for an LCP scan, but it hasn't
           * started yet. In this case the LCP scan will be faked anyways,
           * so we will remove it from the queue immediately and fake its
           * completion. The only reason to send this signal is to ensure
           * that DIH and other blocks that wait for this REP signal can
           * keep track of the outstanding number of outstanding signals.
           * It will be dropped immediately after that when received in
           * DIH since the table is being dropped.
           */
          LcpRecord::FragOrd fragOrd;
          jam();
          CLEAR_ERROR_INSERT_VALUE;
          c_queued_lcp_frag_ord.remove(loc_fragptr);
          loc_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTED;

          if (!loc_fragptr.p->m_create_table_flag_lcp_frag_ord) {
            jam();
            fragOrd.lcpFragOrd.lcpNo = loc_fragptr.p->lcp_frag_ord_lcp_no;
            fragOrd.lcpFragOrd.lcpId = loc_fragptr.p->lcp_frag_ord_lcp_id;
            fragOrd.lcpFragOrd.fragmentId = loc_fragptr.p->fragId;
            fragOrd.lcpFragOrd.tableId = loc_fragptr.p->tabRef;
            sendLCP_FRAG_REP(signal, fragOrd, loc_fragptr.p);
          } else {
            DEB_EARLY_LCP(("(%u)Remove tab(%u,%u) with flag for lcp_frag_ord",
                           instance(), loc_fragptr.p->tabRef,
                           loc_fragptr.p->fragId));
          }
          loc_fragptr.p->m_create_table_flag_lcp_frag_ord = false;
          loc_fragptr.p->m_create_table_insert_lcp = false;
        } else if (loc_fragptr.p->lcp_frag_ord_state ==
                   Fragrecord::LCP_EXECUTING) {
          /**
           * The LCP scan is ongoing, we need to make sure it has completed
           * before we can drop the table. Thus we need to continue the
           * wait for a while longer.
           */
          jam();
          CLEAR_ERROR_INSERT_VALUE;
          lcpDone = false;
        } else if (ERROR_INSERTED(5088) || ERROR_INSERTED(5089)) {
          /**
           * Delay drop table until we reach either LCP_QUEUED or
           * LCP_EXECUTING.
           */
          jam();
          lcpDone = false;
        }
      }
    }
  } else {
    for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabPtr.p->fragrec); i++) {
      loc_fragptr.i = tabPtr.p->fragrec[i];
      if (loc_fragptr.i != RNIL) {
        c_fragment_pool.getPtr(loc_fragptr);
        if (loc_fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_QUEUED) {
          jam();
          jamLine(Uint16(loc_fragptr.p->fragId));
          ndbrequire(loc_fragptr.p->m_create_table_flag_lcp_frag_ord == true);
          c_queued_lcp_frag_ord.remove(loc_fragptr);
          loc_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTED;
          loc_fragptr.p->m_create_table_flag_lcp_frag_ord = false;
          loc_fragptr.p->m_create_table_insert_lcp = false;
          DEB_EARLY_LCP(
              ("(%u)Remove tab(%u,%u) with flag for lcp_frag_ord, "
               "No LCP running",
               instance(), loc_fragptr.p->tabRef, loc_fragptr.p->fragId));
        }
      }
    }
    if (ERROR_INSERTED(5088) || ERROR_INSERTED(5089)) {
      jam();
      CLEAR_ERROR_INSERT_VALUE;
    }
  }

  if (!lcpDone) {
    jam();
    signal->theData[0] = ZDROP_TABLE_WAIT_USAGE;
    signal->theData[1] = tabPtr.i;
    signal->theData[2] = senderRef;
    signal->theData[3] = senderData;
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 10, 4);
    return;
  }
  if (!tabPtr.p->m_informed_backup_drop_tab) {
    jam();
    signal->theData[0] = ZDROP_TABLE_WAIT_USAGE;
    signal->theData[1] = tabPtr.i;
    signal->theData[2] = senderRef;
    signal->theData[3] = senderData;
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 1, 4);

    signal->theData[0] = tabPtr.i;
    signal->theData[1] = reference();
    BlockReference backupRef = calcInstanceBlockRef(BACKUP);
    sendSignal(backupRef, GSN_INFORM_BACKUP_DROP_TAB_REQ, signal, 2, JBB);
    return;
  }
  tabPtr.p->tableStatus = Tablerec::DROP_TABLE_WAIT_DONE;
  DEB_SCHEMA_VERSION(
      ("(%u)tab: %u tableStatus = DROP_TABLE_WAIT_DONE", instance(), tabPtr.i));

  if (tabPtr.p->m_addfragptr_i == RNIL) {
    jam();
    DropTabConf *conf = (DropTabConf *)signal->getDataPtrSend();
    conf->tableId = tabPtr.i;
    conf->senderRef = reference();
    conf->senderData = senderData;
    sendSignal(senderRef, GSN_DROP_TAB_CONF, signal, DropTabConf::SignalLength,
               JBB);
  } else {
    jam();
    Ptr<AddFragRecord> addFragPtr;
    addFragPtr.i = senderData;
    ptrCheckGuard(addFragPtr, caddfragrecFileSize, addFragRecord);
    dropTable_nextStep(signal, addFragPtr);
  }
}

void Dblqh::execDROP_TAB_REQ(Signal *signal) {
  jamEntry();
  if (ERROR_INSERTED(5076)) {
    /**
     * This error insert simulates a situation where it takes a long time
     * to execute DROP_TAB_REQ, such that we can crash the (dict) master
     * while there is an outstanding DROP_TAB_REQ.
     */
    jam();
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(reference(), GSN_DROP_TAB_REQ, signal, 10000,
                        signal->getLength());
    return;
  }
  if (ERROR_INSERTED(5077)) {
    jam();
    CLEAR_ERROR_INSERT_VALUE;
    /**
     * Kill this node 2 seconds from now. We wait for two seconds to make sure
     * that DROP_TAB_REQ messages have reached other nodes before this one
     * dies.
     */
    signal->theData[0] = 9999;
    sendSignalWithDelay(CMVMI_REF, GSN_NDB_TAMPER, signal, 2000, 1);
    return;
  }
  DropTabReq reqCopy = *(DropTabReq *)signal->getDataPtr();
  DropTabReq *req = &reqCopy;

  TablerecPtr tabPtr;
  tabPtr.i = req->tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

  Uint32 errCode = 0;
  switch ((DropTabReq::RequestType)req->requestType) {
    case DropTabReq::RestartDropTab:
      jam();
      [[fallthrough]];
    case DropTabReq::CreateTabDrop:
      if (tabPtr.p->tableStatus == Tablerec::TABLE_DEFINED) {
        jam();
        ndbrequire(tabPtr.p->usageCountR == 0 && tabPtr.p->usageCountW == 0);
        seizeAddfragrec(signal);
        tabPtr.p->m_addfragptr_i = addfragptr.i;
        addfragptr.p->m_dropTabReq = *req;
        tabPtr.p->m_informed_backup_drop_tab = false;
        tabPtr.p->m_senderRef = req->senderRef;
        c_pgman->set_table_ready_for_prep_lcp_writes(tabPtr.i, false);
        check_pgman_prep_lcp_active_drop_tab(signal, tabPtr.i);
        return;
      } else {
        jam();
        tabPtr.p->tableStatus = Tablerec::DROP_TABLE_WAIT_DONE;
        DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = DROP_TABLE_WAIT_DONE(2)",
                            instance(), tabPtr.i));
      }
      break;
    case DropTabReq::OnlineDropTab:
      jam();
      switch (tabPtr.p->tableStatus) {
        case Tablerec::TABLE_DEFINED:
          jam();
          errCode = DropTabRef::DropWoPrep;
          break;
        case Tablerec::NOT_DEFINED:
          jam();
          errCode = DropTabRef::NoSuchTable;
          break;
        case Tablerec::ADD_TABLE_ONGOING:
          jam();
          ndbassert(false);
          [[fallthrough]];
        case Tablerec::PREP_DROP_TABLE_DONE:
          jam();
          tabPtr.p->m_informed_backup_drop_tab = false;
          tabPtr.p->tableStatus = Tablerec::DROP_TABLE_WAIT_USAGE;
          DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = DROP_TABLE_WAIT_USAGE",
                              instance(), tabPtr.i));
          signal->theData[0] = ZDROP_TABLE_WAIT_USAGE;
          signal->theData[1] = tabPtr.i;
          signal->theData[2] = req->senderRef;
          signal->theData[3] = req->senderData;
          dropTab_wait_usage(signal);
          return;
          break;
        case Tablerec::DROP_TABLE_WAIT_USAGE:
        case Tablerec::DROP_TABLE_ACC:
        case Tablerec::DROP_TABLE_TUP:
        case Tablerec::DROP_TABLE_TUX:
          ndbabort();
        case Tablerec::DROP_TABLE_WAIT_DONE:
          jam();
          break;
        case Tablerec::TABLE_READ_ONLY:
          jam();
          errCode = DropTabRef::InvalidTableState;
          break;
      }
  }

  if (errCode) {
    jam();
    DropTabRef *ref = (DropTabRef *)signal->getDataPtrSend();
    ref->tableId = tabPtr.i;
    ref->senderRef = reference();
    ref->senderData = req->senderData;
    ref->errorCode = errCode;
    sendSignal(req->senderRef, GSN_DROP_TAB_REF, signal,
               DropTabRef::SignalLength, JBB);
    return;
  }

  ndbrequire(tabPtr.p->usageCountR == 0 && tabPtr.p->usageCountW == 0);
  seizeAddfragrec(signal);
  addfragptr.p->m_dropTabReq = *req;
  dropTable_nextStep(signal, addfragptr);
}

void Dblqh::execDROP_TAB_REF(Signal *signal) {
  jamEntry();
  DropTabRef *ref = (DropTabRef *)signal->getDataPtr();

#if defined ERROR_INSERT || defined VM_TRACE
  jamLine(ref->errorCode);
  ndbabort();
#endif

  Ptr<AddFragRecord> addFragPtr;
  addFragPtr.i = ref->senderData;
  ptrCheckGuard(addFragPtr, caddfragrecFileSize, addFragRecord);
  dropTable_nextStep(signal, addFragPtr);
}

void Dblqh::execDROP_TAB_CONF(Signal *signal) {
  jamEntry();
  DropTabConf *conf = (DropTabConf *)signal->getDataPtr();

  Ptr<AddFragRecord> addFragPtr;
  addFragPtr.i = conf->senderData;
  ptrCheckGuard(addFragPtr, caddfragrecFileSize, addFragRecord);
  dropTable_nextStep(signal, addFragPtr);
}

void Dblqh::check_no_active_scans_at_drop_table(TablerecPtr tabPtr) {
  /*
   * Verify that LQH has terminated scans.  (If not, then drop order
   * must change from TUP,TUX to TUX,TUP and we must wait for scans).
   */
  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabPtr.p->fragid); i++) {
    FragrecordPtr curr_fragptr;
    curr_fragptr.i = tabPtr.p->fragrec[i];
    if (curr_fragptr.i != RNIL) {
      jam();
      jamLine(Uint16(tabPtr.p->fragid[i]));
      c_fragment_pool.getPtr(curr_fragptr);
      ndbrequire(curr_fragptr.p->m_activeScans == 0);
      ndbrequire(
          Local_ScanRecord_fifo(c_scanRecordPool, curr_fragptr.p->m_queuedScans)
              .getCount() == 0);
      ndbrequire(Local_ScanRecord_fifo(c_scanRecordPool,
                                       curr_fragptr.p->m_queuedTupScans)
                     .getCount() == 0);
      ndbrequire(Local_ScanRecord_fifo(c_scanRecordPool,
                                       curr_fragptr.p->m_queuedAccScans)
                     .getCount() == 0);
    }
  }
}

void Dblqh::dropTable_nextStep(Signal *signal, Ptr<AddFragRecord> addFragPtr) {
  jam();

  TablerecPtr tabPtr;
  tabPtr.i = addFragPtr.p->m_dropTabReq.tableId;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  ndbassert(!m_is_query_block);

  Uint32 ref = 0;
  if (tabPtr.p->tableStatus == Tablerec::DROP_TABLE_WAIT_DONE) {
    jam();
    check_no_active_scans_at_drop_table(tabPtr);
    if (DictTabInfo::isTable(tabPtr.p->tableType) ||
        DictTabInfo::isHashIndex(tabPtr.p->tableType)) {
      jam();
      ref = calcInstanceBlockRef(DBACC);
      tabPtr.p->tableStatus = Tablerec::DROP_TABLE_ACC;
      DEB_SCHEMA_VERSION(
          ("(%u)tab: %u tableStatus = DROP_TABLE_ACC", instance(), tabPtr.i));
    } else {
      jam();
      ref = calcInstanceBlockRef(DBTUP);
      tabPtr.p->tableStatus = Tablerec::DROP_TABLE_TUP;
      DEB_SCHEMA_VERSION(
          ("(%u)tab: %u tableStatus = DROP_TABLE_TUP", instance(), tabPtr.i));
    }
  } else if (tabPtr.p->tableStatus == Tablerec::DROP_TABLE_ACC) {
    jam();
    ref = calcInstanceBlockRef(DBTUP);
    tabPtr.p->tableStatus = Tablerec::DROP_TABLE_TUP;
    DEB_SCHEMA_VERSION(
        ("(%u)tab: %u tableStatus = DROP_TABLE_TUP(2)", instance(), tabPtr.i));
  } else if (tabPtr.p->tableStatus == Tablerec::DROP_TABLE_TUP) {
    jam();
    if (DictTabInfo::isOrderedIndex(tabPtr.p->tableType)) {
      jam();
      ref = calcInstanceBlockRef(DBTUX);
      tabPtr.p->tableStatus = Tablerec::DROP_TABLE_TUX;
      DEB_SCHEMA_VERSION(
          ("(%u)tab: %u tableStatus = DROP_TABLE_TUX", instance(), tabPtr.i));
    }
  }

  if (ref) {
    jam();
    DropTabReq *req = (DropTabReq *)signal->getDataPtrSend();
    req->senderData = addFragPtr.i;
    req->senderRef = reference();
    req->tableId = tabPtr.i;
    req->tableVersion = tabPtr.p->schemaVersion;
    req->requestType = addFragPtr.p->m_dropTabReq.requestType;
    sendSignal(ref, GSN_DROP_TAB_REQ, signal, DropTabReq::SignalLength, JBB);
    return;
  }

  removeTable(tabPtr.i);
  tabPtr.p->m_addfragptr_i = RNIL;
  tabPtr.p->tableStatus = Tablerec::NOT_DEFINED;
  DEB_SCHEMA_VERSION(
      ("(%u)tab: %u tableStatus = NOT_DEFINED", instance(), tabPtr.i));

  DropTabConf *conf = (DropTabConf *)signal->getDataPtrSend();
  conf->senderRef = reference();
  conf->senderData = addFragPtr.p->m_dropTabReq.senderData;
  conf->tableId = tabPtr.i;
  sendSignal(addFragPtr.p->m_dropTabReq.senderRef, GSN_DROP_TAB_CONF, signal,
             DropTabConf::SignalLength, JBB);

  addfragptr = addFragPtr;
  releaseAddfragrec(signal);
}

void Dblqh::removeTable(Uint32 tableId) {
  tabptr.i = tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabptr.p->fragid); i++) {
    jam();
    if (tabptr.p->fragid[i] != ZNIL) {
      jam();
      deleteFragrec(tabptr.p->fragid[i]);
    }  // if
  }    // for
}  // Dblqh::removeTable()

/**
 * When adding a set of new columns to a table the row size grows.
 * This can have a bad effect on ongoing LCP scans. So therefore
 * we need to wait to change the table metadata until we are sure
 * that it is safe to change this parameter.
 *
 * It is safe if no LCP execution is ongoing on the table.
 *
 * It is safe when returning from executing an LCP since only one
 * at a time can execute an LCP (we can have another LCP in prepare
 * phase, but only one at a time in execution phase).
 *
 * We have also made it safe as soon as the LCP scan returns
 * a SCAN_FRAGCONF. We will check if it is necessary to change
 * the max record size of the table before we decide whether to
 * continue executing an LCP scan.
 *
 * The max record size is used to ensure that the LCP scan have
 * buffer space to receive at least 16 rows with maximum size.
 * This is checked before executing the next SCAN_NEXTREQ or SCAN_FRAGREQ,
 * so changing the max record size immediately after receiving SCAN_FRAGCONF
 * is a working solution. This means that at most we have to wait
 * for a scan of 16 rows, so normally the wait here should be very
 * small and practically unnoticeable for all practical purposes.
 *
 * The real-time break should not constitute any issue here since
 * we don't perform any action until it is safe to execute all
 * actions.
 */
bool Dblqh::handleLCPSurfacing(Signal *signal) {
  ndbassert(!m_is_query_block);
  if (!c_wait_lcp_surfacing) {
    jam();
    return false;
  }
  jam();
  DEB_LCP(("(%u)LCP surfaced, continue", instance()));
  TablerecPtr tablePtr;
  tablePtr.i = c_keep_alter_tab_req.tableId;
  ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);

  lock_table_exclusive(tablePtr.p);
  c_wait_lcp_surfacing = false;

  AlterTabReq *req = (AlterTabReq *)signal->getDataPtr();
  *req = c_keep_alter_tab_req;
  const Uint32 newTableVersion = req->newTableVersion;
  const Uint32 senderRef = req->senderRef;
  const Uint32 senderData = req->senderData;

  tablePtr.p->schemaVersion = newTableVersion;
  DEB_SCHEMA_VERSION(("(%u)tab(%u): %u tableStatus = %u", instance(),
                      tablePtr.p->schemaVersion, tablePtr.i,
                      tablePtr.p->tableStatus));
  if (AlterTableReq::getReorgFragFlag(req->changeMask)) {
    jam();
    commit_reorg(tablePtr);
  }
  Uint32 len = c_keep_alter_tab_req_len;
  EXECUTE_DIRECT(getDBTUP(), GSN_ALTER_TAB_REQ, signal, len);
  jamEntry();

  Uint32 errCode = signal->theData[0];
  Uint32 connectPtr = signal->theData[1];
  ndbrequire(errCode == 0);
  unlock_table_exclusive(tablePtr.p);

  AlterTabConf *conf = (AlterTabConf *)signal->getDataPtrSend();
  conf->senderRef = reference();
  conf->senderData = senderData;
  conf->connectPtr = connectPtr;
  sendSignal(senderRef, GSN_ALTER_TAB_CONF, signal, AlterTabConf::SignalLength,
             JBB);
  return true;
}

void Dblqh::execALTER_TAB_REQ(Signal *signal) {
  jamEntry();

  if (!assembleFragments(signal)) return;

  ndbassert(!m_is_query_block);
  AlterTabReq copy = *(AlterTabReq *)signal->getDataPtr();
  const AlterTabReq *req = &copy;
  const Uint32 senderRef = req->senderRef;
  const Uint32 senderData = req->senderData;
  const Uint32 tableId = req->tableId;
  const Uint32 tableVersion = req->tableVersion;
  const Uint32 newTableVersion = req->newTableVersion;
  AlterTabReq::RequestType requestType =
      (AlterTabReq::RequestType)req->requestType;

  TablerecPtr tablePtr;
  tablePtr.i = tableId;
  ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);

  bool locked_table = false;

  D("ALTER_TAB_REQ(LQH): requestType: " << requestType);
  Uint32 len = signal->getLength();
  switch (requestType) {
    case AlterTabReq::AlterTablePrepare:
      jam();
      break;
    case AlterTabReq::AlterTableRevert:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      tablePtr.p->schemaVersion = tableVersion;
      DEB_SCHEMA_VERSION(("(%u)tab(%u): %u tableStatus = %u (2)", instance(),
                          tablePtr.p->schemaVersion, tablePtr.i,
                          tablePtr.p->tableStatus));
      break;
    case AlterTabReq::AlterTableCommit:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      if (AlterTableReq::getAddAttrFlag(req->changeMask)) {
        jam();
        lcpPtr.i = 0;
        ptrAss(lcpPtr, lcpRecord);
        if (lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING &&
            tableId == lcpPtr.p->currentRunFragment.lcpFragOrd.tableId) {
          jam();
          /* See comment above on handleLCPSurfacing */
          DEB_LCP(("(%u)Wait for LCP to surface again", instance()));
          ndbrequire(!c_wait_lcp_surfacing);
          c_wait_lcp_surfacing = true;
          c_keep_alter_tab_req = copy;
          c_keep_alter_tab_req_len = len;
          unlock_table_exclusive(tablePtr.p);
          return;
        }
      }
      tablePtr.p->schemaVersion = newTableVersion;
      DEB_SCHEMA_VERSION(("(%u)tab(%u): %u tableStatus = %u (3)", instance(),
                          tablePtr.p->schemaVersion, tablePtr.i,
                          tablePtr.p->tableStatus));
      if (AlterTableReq::getReorgFragFlag(req->changeMask)) {
        jam();
        commit_reorg(tablePtr);
      }
      break;
    case AlterTabReq::AlterTableComplete:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      break;
    case AlterTabReq::AlterTableSumaEnable:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      break;
    case AlterTabReq::AlterTableSumaFilter:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      signal->theData[len++] = cnewestGci + 3;
      break;
    case AlterTabReq::AlterTableReadOnly:
      jam();
      lock_table_exclusive(tablePtr.p);
      ndbrequire(tablePtr.p->tableStatus == Tablerec::TABLE_DEFINED);
      tablePtr.p->tableStatus = Tablerec::TABLE_READ_ONLY;
      DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = TABLE_READ_ONLY",
                          instance(), tablePtr.i));
      signal->theData[0] = ZWAIT_READONLY;
      signal->theData[1] = tablePtr.i;
      signal->theData[2] = senderRef;
      signal->theData[3] = senderData;
      sendSignal(reference(), GSN_CONTINUEB, signal, 4, JBB);
      unlock_table_exclusive(tablePtr.p);
      return;
    case AlterTabReq::AlterTableReadWrite:
      jam();
      locked_table = true;
      lock_table_exclusive(tablePtr.p);
      ndbrequire(tablePtr.p->tableStatus == Tablerec::TABLE_READ_ONLY);
      tablePtr.p->tableStatus = Tablerec::TABLE_DEFINED;
      DEB_SCHEMA_VERSION(("(%u)tab: %u tableStatus = TABLE_DEFINED(2)",
                          instance(), tablePtr.i));
      break;
    default:
      ndbabort();
  }

  EXECUTE_DIRECT(getDBTUP(), GSN_ALTER_TAB_REQ, signal, len);
  jamEntry();

  Uint32 errCode = signal->theData[0];
  Uint32 connectPtr = signal->theData[1];
  if (errCode == 0) {
    // Request handled successfully
    AlterTabConf *conf = (AlterTabConf *)signal->getDataPtrSend();
    conf->senderRef = reference();
    conf->senderData = senderData;
    conf->connectPtr = connectPtr;
    sendSignal(senderRef, GSN_ALTER_TAB_CONF, signal,
               AlterTabConf::SignalLength, JBB);
  } else if (errCode == ~Uint32(0)) {
    /**
     * Wait
     */
    ndbrequire(requestType == AlterTabReq::AlterTableSumaFilter);
    signal->theData[0] = ZWAIT_REORG_SUMA_FILTER_ENABLED;
    signal->theData[1] = cnewestGci + 3;
    signal->theData[2] = senderData;
    signal->theData[3] = connectPtr;
    signal->theData[4] = senderRef;
    wait_reorg_suma_filter_enabled(signal);
  } else {
    jam();
    AlterTabRef *ref = (AlterTabRef *)signal->getDataPtrSend();
    ref->senderRef = reference();
    ref->senderData = senderData;
    ref->connectPtr = connectPtr;
    ref->errorCode = errCode;
    sendSignal(senderRef, GSN_ALTER_TAB_REF, signal, AlterTabRef::SignalLength,
               JBB);
  }
  if (locked_table) {
    jam();
    unlock_table_exclusive(tablePtr.p);
  }
}

void Dblqh::wait_reorg_suma_filter_enabled(Signal *signal) {
  if (cnewestCompletedGci >= signal->theData[1]) {
    jam();
    Uint32 senderData = signal->theData[2];
    Uint32 connectPtr = signal->theData[3];
    Uint32 senderRef = signal->theData[4];

    D("ALTER_TAB_CONF after suma filter enabled");
    AlterTabConf *conf = (AlterTabConf *)signal->getDataPtrSend();
    conf->senderRef = reference();
    conf->senderData = senderData;
    conf->connectPtr = connectPtr;
    sendSignal(senderRef, GSN_ALTER_TAB_CONF, signal,
               AlterTabConf::SignalLength, JBB);
    return;
  }
  sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 500, 5);
}

void Dblqh::commit_reorg(TablerecPtr tablePtr) {
  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tablePtr.p->fragrec); i++) {
    jam();
    /**
     * This update of distribution key only happens in LDM threads, so there
     * can never be a concurrency conflict in its update. Multiple query
     * threads can read this value, but they ignore it since they are always
     * dirty read operations. Distribution keys are there to ensure that
     * we abort updates that do not take all replicas into account. Thus it
     * is not an issue for query threads as long as they only perform
     * dirty read operations.
     */
    Ptr<Fragrecord> fragPtr;
    if ((fragPtr.i = tablePtr.p->fragrec[i]) != RNIL) {
      jam();
      c_fragment_pool.getPtr(fragPtr);
      fragPtr.p->fragDistributionKey =
          (fragPtr.p->fragDistributionKey + 1) & 0xFF;
    }
  }
}

void Dblqh::wait_readonly(Signal *signal) {
  jam();

  Uint32 tableId = signal->theData[1];

  TablerecPtr tablePtr;
  tablePtr.i = tableId;
  ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);
  ndbrequire(tablePtr.p->tableStatus == Tablerec::TABLE_READ_ONLY);

  if (tablePtr.p->usageCountW > 0) {
    jam();
    sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 3000,
                        signal->getLength());
    return;
  }

  Uint32 senderRef = signal->theData[2];
  Uint32 senderData = signal->theData[3];

  // Request handled successfully
  AlterTabConf *conf = (AlterTabConf *)signal->getDataPtrSend();
  conf->senderRef = reference();
  conf->senderData = senderData;
  sendSignal(senderRef, GSN_ALTER_TAB_CONF, signal, AlterTabConf::SignalLength,
             JBB);
}

/* ************************************************************************>>
 * TIME_SIGNAL: Handles time-out of local operations. This is a clean-up
 * handler. If no other measure has succeeded in cleaning up after time-outs
 * or else then this routine will remove the transaction after 120 seconds of
 * inactivity. The check is performed once per 40 milliseconds.
 * ************************************************************************>> */
#define LQH_TIME_SIGNAL_DELAY 40
void Dblqh::execTIME_SIGNAL(Signal *signal) {
  jamEntry();

  const NDB_TICKS currentTime = NdbTick_getCurrentTicks();
  Uint64 num_ms_elapsed = elapsed_time(signal, currentTime, c_latestTIME_SIGNAL,
                                       Uint32(LQH_TIME_SIGNAL_DELAY));
  sendTIME_SIGNAL(signal, currentTime, Uint32(LQH_TIME_SIGNAL_DELAY));

  /**
   * timer_handling will effectively call tick in the IOTracker and
   * this in turn will only do something every 128ms, so this means
   * that with a maximum of 8 runs per execTIME_SIGNAL we can at most
   * have one call to tick that actually does something useful.
   * We avoid using the same delay as in DBTC to avoid running the
   * delay handling code at the same time.
   *
   * The idea of calling timer_handling logically once per 10ms is
   * a remnant from when TIME_SIGNAL was generated by QMGR, this could
   * be changed in the future.
   */
  c_elapsed_time_millis += num_ms_elapsed;
  while (c_elapsed_time_millis > Uint64(10)) {
    jam();
    c_elapsed_time_millis -= Uint64(10);
    timer_handling(signal);
  }
}

bool Dblqh::getNextTcConRec(Uint32 &next, TcConnectionrecPtr &tcConnectptr,
                            Uint32 max_loops) {
  Uint32 found = 0;
  Uint32 loop_count = 0;
  while (found == 0 && next != RNIL &&
         (max_loops == 0 || loop_count < max_loops)) {
    found = tcConnect_pool.getUncheckedPtrs(&next, &tcConnectptr, 1);
    if (found > 0 && !Magic::check_ptr(tcConnectptr.p)) found = 0;
    loop_count++;
  }
  return (found > 0);
}

bool Dblqh::getNextScanRec(Uint32 &next, ScanRecordPtr &loc_scanptr,
                           Uint32 max_loops) {
  Uint32 found = 0;
  Uint32 loop_count = 0;
  while (found == 0 && next != RNIL &&
         (max_loops == 0 || loop_count < max_loops)) {
    found = c_scanRecordPool.getUncheckedPtrs(&next, &loc_scanptr, 1);
    if (found > 0 && !Magic::check_ptr(loc_scanptr.p)) found = 0;
    loop_count++;
  }
  return (found > 0);
}

bool Dblqh::getNextCommitAckMarker(Uint32 &next,
                                   CommitAckMarkerPtr &commitAckMarkerPtr,
                                   Uint32 max_loops) {
  Uint32 found = 0;
  Uint32 loop_count = 0;
  while (found == 0 && next != RNIL &&
         (max_loops == 0 || loop_count < max_loops)) {
    found =
        m_commitAckMarkerPool.getUncheckedPtrs(&next, &commitAckMarkerPtr, 1);
    if (found > 0 && !Magic::check_ptr(commitAckMarkerPtr.p)) found = 0;
    loop_count++;
  }
  return (found > 0);
}

void Dblqh::timer_handling(Signal *signal) {
  cLqhTimeOutCount++;
  cLqhTimeOutCheckCount++;

  LogPartRecordPtr logPartPtr;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
    int ret = logPartPtr.p->m_io_tracker.tick(
        10 * cLqhTimeOutCount, c_max_redo_lag, c_max_redo_lag_counter);
    if (ret < 0) {
      /**
       * set problem
       */
      update_log_problem(signal, logPartPtr.p, LogPartRecord::P_REDO_IO_PROBLEM,
                         true);
    } else if (ret > 0) {
      /**
       * clear
       */
      update_log_problem(signal, logPartPtr.p, LogPartRecord::P_REDO_IO_PROBLEM,
                         false);
    }
  }

  if (cLqhTimeOutCheckCount < 1000) {
    jam();
    return;
  }  // if

  cLqhTimeOutCheckCount = 0;
#ifdef DEBUG_TRANSACTION_TIMEOUT
#ifdef VM_TRACE
  TcConnectionrecPtr tTcConptr;
  Uint32 next = 0;
  do {
    bool found = getNextTcConRec(next, tTcConptr, 0);
    if (found && (tTcConptr.p->tcTimer != 0) &&
        ((tTcConptr.p->tcTimer + 12000) < cLqhTimeOutCount)) {
      ndbout << "Dblqh::execTIME_SIGNAL" << endl
             << "Timeout found in tcConnectRecord " << tTcConptr.i << endl
             << " cLqhTimeOutCount = " << cLqhTimeOutCount << endl
             << " tcTimer=" << tTcConptr.p->tcTimer << endl
             << " tcTimer+12000=" << tTcConptr.p->tcTimer + 12000 << endl;

      signal->theData[0] = DumpStateOrd::LqhDumpAllTcRec;
      signal->theData[1] = tTcConptr.i;
      execDUMP_STATE_ORD(signal);

      // Reset the timer
      tTcConptr.p->tcTimer = 0;
    }  // if
  } while (next != RNIL);
#endif
#ifdef VM_TRACE
  LogFileOperationRecordPtr lfoPtr;
  for (lfoPtr.i = 0; lfoPtr.i < clfoFileSize; lfoPtr.i++) {
    ptrAss(lfoPtr, logFileOperationRecord);
    if ((lfoPtr.p->lfoTimer != 0) &&
        ((lfoPtr.p->lfoTimer + 12000) < cLqhTimeOutCount)) {
      ndbout << "We have lost LFO record" << endl;
      ndbout << "index = " << lfoPtr.i;
      ndbout << "State = " << lfoPtr.p->lfoState;
      ndbout << " Page No = " << lfoPtr.p->lfoPageNo;
      ndbout << " noPagesRw = " << lfoPtr.p->noPagesRw;
      ndbout << "lfoWordWritten = " << lfoPtr.p->lfoWordWritten << endl;
      lfoPtr.p->lfoTimer = cLqhTimeOutCount;
    }  // if
  }    // for

#endif
#endif
#if 0
  LcpRecordPtr TlcpPtr;
  // Print information about the current local checkpoint
  TlcpPtr.i = 0;
  ptrAss(TlcpPtr, lcpRecord);
  ndbout << "Information about LCP in this LQH" << endl
	 << "  lcpState=" << TlcpPtr.p->lcpState << endl
	 << "   firstLcpLocAcc=" << TlcpPtr.p->firstLcpLocAcc << endl
	 << "   firstLcpLocTup=" << TlcpPtr.p->firstLcpLocTup << endl
	 << "   lcpAccptr=" << TlcpPtr.p->lcpAccptr << endl
	 << "   lastFragmentFlag=" << TlcpPtr.p->lastFragmentFlag << endl
#endif
}  // Dblqh::execTIME_SIGNAL()

/* ######################################################################### */
/* #######                  EXECUTION MODULE                         ####### */
/* THIS MODULE HANDLES THE RECEPTION OF LQHKEYREQ AND ALL PROCESSING         */
/* OF OPERATIONS ON BEHALF OF THIS REQUEST. THIS DOES ALSO INVOLVE           */
/* RECEPTION OF VARIOUS TYPES OF ATTRINFO AND KEYINFO. IT DOES ALSO          */
/* INVOLVE COMMUNICATION WITH ACC AND TUP.                                   */
/* ######################################################################### */

/**
 * earlyKeyReqAbort
 *
 * Exit early from handling an LQHKEYREQ request.
 * Method determines which resources (if any) need freed, then
 * signals requestor with error response.
 * * Verify all required resources are freed if adding new callers *
 */
void Dblqh::earlyKeyReqAbort(Signal *signal, const LqhKeyReq *lqhKeyReq,
                             Uint32 errCode,
                             const TcConnectionrecPtr tcConnectptr) {
  jamEntry();
  const Uint32 transid1 = lqhKeyReq->transId1;
  const Uint32 transid2 = lqhKeyReq->transId2;
  Uint32 treqInfo = lqhKeyReq->requestInfo;

  const Uint32 reqInfo = treqInfo;

  bool tcConnectRecAllocated = (tcConnectptr.i != RNIL);

  if (tcConnectRecAllocated) {
    jam();

    /* Could have a commit-ack marker allocated. */
    remove_commit_marker(tcConnectptr.p);

    /* Could have long key/attr sections linked */
    ndbrequire(tcConnectptr.p->m_dealloc_state == TcConnectionrec::DA_IDLE);
    ndbrequire(tcConnectptr.p->m_dealloc_data.m_unused == RNIL);
    releaseOprec(signal, tcConnectptr);

    /*
     * Free the TcConnectRecord, ensuring that the
     * table reference counts have not been incremented and
     * so will not be decremented.
     * Also verify that we're not present in the transid
     * hash
     */
    ndbrequire(tcConnectptr.p->tableref == RNIL);
    /* Following is not 100% check, but a reasonable guard */
    ndbrequire(tcConnectptr.p->nextHashRec == RNIL);
    ndbrequire(tcConnectptr.p->prevHashRec == RNIL);
    releaseTcrec(signal, tcConnectptr);
  }

  /* Now perform signalling */

  if (LqhKeyReq::getDirtyFlag(reqInfo) &&
      LqhKeyReq::getOperation(reqInfo) == ZREAD &&
      !LqhKeyReq::getNormalProtocolFlag(reqInfo)) {
    jam();
    /* Dirty read sends TCKEYREF direct to client, and nothing to TC */
    ndbrequire(LqhKeyReq::getApplicationAddressFlag(reqInfo));
    const Uint32 apiRef = lqhKeyReq->variableData[0];
    const Uint32 apiOpRec = lqhKeyReq->variableData[1];

    TcKeyRef *const tcKeyRef = (TcKeyRef *)signal->getDataPtrSend();

    tcKeyRef->connectPtr = apiOpRec;
    tcKeyRef->transId[0] = transid1;
    tcKeyRef->transId[1] = transid2;
    tcKeyRef->errorCode = errCode;
    sendTCKEYREF(signal, apiRef, lqhKeyReq->tcBlockref, 0);
  } else {
    jam();
    /* All ops apart from dirty read send LQHKEYREF to TC
     * (This includes simple read)
     */

    const Uint32 clientPtr = lqhKeyReq->clientConnectPtr;
    Uint32 TcOprec = clientPtr;
    if (LqhKeyReq::getSameClientAndTcFlag(reqInfo) == 1) {
      if (LqhKeyReq::getApplicationAddressFlag(reqInfo))
        TcOprec = lqhKeyReq->variableData[2];
      else
        TcOprec = lqhKeyReq->variableData[0];
    }

    LqhKeyRef *const ref = (LqhKeyRef *)signal->getDataPtrSend();
    ref->userRef = clientPtr;
    ref->connectPtr = TcOprec;
    ref->errorCode = errCode;
    ref->transId1 = transid1;
    ref->transId2 = transid2;
    ref->flags = 0;
    Uint32 block = refToMain(signal->senderBlockRef());
    if (block != getRESTORE()) {
      sendSignal(signal->senderBlockRef(), GSN_LQHKEYREF, signal,
                 LqhKeyRef::SignalLength, JBB);
    } else {
      ndbrequire(refToNode(signal->senderBlockRef()) == cownNodeid &&
                 refToInstance(signal->senderBlockRef()) == instance());
      EXECUTE_DIRECT(getRESTORE(), GSN_LQHKEYREF, signal,
                     LqhKeyRef::SignalLength);
    }
  }  // if
  return;
}  // Dblqh::earlyKeyReqAbort()

Uint32 Dblqh::get_table_state_error(Ptr<Tablerec> tabPtr) const {
  switch (tabPtr.p->tableStatus) {
    case Tablerec::NOT_DEFINED:
      jam();
      return ZTABLE_NOT_DEFINED;
      break;
    case Tablerec::ADD_TABLE_ONGOING:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::PREP_DROP_TABLE_DONE:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::DROP_TABLE_WAIT_USAGE:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::DROP_TABLE_WAIT_DONE:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::DROP_TABLE_ACC:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::DROP_TABLE_TUP:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::DROP_TABLE_TUX:
      jam();
      return ZDROP_TABLE_IN_PROGRESS;
    case Tablerec::TABLE_DEFINED:
    case Tablerec::TABLE_READ_ONLY:
      ndbabort();
      return ZTABLE_NOT_DEFINED;
  }
  ndbabort();
  return ~Uint32(0);
}

int Dblqh::check_tabstate(Signal *signal, const Tablerec *tablePtrP, Uint32 op,
                          const TcConnectionrecPtr tcConnectptr) {
  if (tabptr.p->tableStatus == Tablerec::TABLE_READ_ONLY) {
    jam();
    if (op == ZREAD || op == ZREAD_EX || op == ZUNLOCK) {
      jam();
      return 0;
    }
    terrorCode = ZTABLE_READ_ONLY;
  } else {
    jam();
    terrorCode = get_table_state_error(tabptr);
  }
  abortErrorLab(signal, tcConnectptr);
  return 1;
}

void Dblqh::LQHKEY_abort(Signal *signal, int errortype,
                         const TcConnectionrecPtr tcConnectptr) {
  switch (errortype) {
    case 0:
      jam();
      terrorCode = ZCOPY_NODE_ERROR;
      break;
    case 1:
      jam();
      terrorCode = ZNO_FREE_LQH_CONNECTION;
      break;
    case 2:
      jam();
      terrorCode = signal->theData[1];
      break;
    case 3:
      jam();
      ndbrequire(tcConnectptr.p->transactionState ==
                 TcConnectionrec::WAIT_ACC_ABORT);
      return;
    case 4:
      jam();
      terrorCode = get_table_state_error(tabptr);
      break;
    case 5:
      jam();
      terrorCode = ZINVALID_SCHEMA_VERSION;
      break;
    case 6:
      jam();
      terrorCode = ZNO_SUCH_FRAGMENT_ID;
      break;
    default:
      ndbabort();
  }  // switch
  abortErrorLab(signal, tcConnectptr);
}  // Dblqh::LQHKEY_abort()

void Dblqh::LQHKEY_error(Signal *signal, int errortype,
                         TcConnectionrecPtr tcConnectptr) {
  switch (errortype) {
    case 0:
      jam();
      break;
    case 1:
      jam();
      break;
    case 2:
      jam();
      break;
    case 3:
      jam();
      break;
    case 4:
      jam();
      break;
    case 5:
      jam();
      break;
    case 6:
      jam();
      break;
    default:
      jam();
      break;
  }  // switch
  g_eventLogger->info("(%u)Protocol error in LQHKEYREQ: %u", instance(),
                      errortype);
  terrorCode = ZLQHKEY_PROTOCOL_ERROR;
  abortErrorLab(signal, tcConnectptr);
}  // Dblqh::LQHKEY_error()

void Dblqh::execLQHKEYREF(Signal *signal) {
  jamEntry();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  Uint32 tcOprec = signal->theData[1];
  terrorCode = signal->theData[2];
  Uint32 transid1 = signal->theData[3];
  Uint32 transid2 = signal->theData[4];
  if (!tcConnect_pool.getValidPtr(tcConnectptr)) {
    jam();
    warningReport(signal, 13);
    return;
  }  // if
  m_tc_connect_ptr = tcConnectptr;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;

  if (likely(!((regTcPtr->connectState == TcConnectionrec::LOG_CONNECTED) ||
               (regTcPtr->connectState == TcConnectionrec::COPY_CONNECTED)))) {
    /**
     * This...is unpleasant...
     *   LOG_CONNECTED and COPY_CONNECTED will not release there tcConnectptr
     *   before all outstanding is finished.
     *
     *   CONNECTED on the other hand can, (in ::execABORT)
     *     which means that findTransaction *should* be used
     *     to make sure that correct tcConnectptr is accessed.
     *
     *   However, as LOG_CONNECTED & COPY_CONNECTED only uses 1 tcConnectptr
     *     (and fiddles) with transid and other stuff, I could
     *     not find an easy way to modify the code so that findTransaction
     *     is usable also for them
     */
    if (findTransaction(transid1, transid2, tcOprec, regTcPtr->tcHashKeyHi,
                        true, false, tcConnectptr) != ZOK) {
      jam();
      warningReport(signal, 14);
      return;
    }
  } else if (regTcPtr->connectState == TcConnectionrec::COPY_CONNECTED) {
    jam();
    if ((regTcPtr->transid[0] != transid1) ||
        (regTcPtr->transid[1] != transid2)) {
      jam();
      /**
       * A previous normal operation was aborted from DBTC, but the
       * LQHKEYREQ was also aborted, when we arrive here the operation
       * recorded was already released by the abort operation, the
       * operation record have been taken over by a copy operation.
       * Ensure that we don't touch the operation record since it is
       * in use by another operation. So this signal should be
       * ignored.
       */
      warningReport(signal, 17);
      return;
    }
  } else {
    ndbrequire((regTcPtr->transid[0] == transid1) &&
               (regTcPtr->transid[1] == transid2));
  }

  switch (regTcPtr->connectState) {
    case TcConnectionrec::CONNECTED:
      jam();
      if (regTcPtr->abortState != TcConnectionrec::ABORT_IDLE) {
        warningReport(signal, 15);
        return;
      }  // if
      /* Mark abort due to replica issue */
      regTcPtr->abortState = TcConnectionrec::ABORT_FROM_LQH_REPLICA;
      regTcPtr->errorCode = terrorCode;
      abortCommonLab(signal, tcConnectptr);
      return;
    case TcConnectionrec::LOG_CONNECTED:
      jam();
      logLqhkeyrefLab(signal, tcConnectptr);
      return;
    case TcConnectionrec::COPY_CONNECTED:
      jam();
      setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
      copyLqhKeyRefLab(signal, tcConnectptr);
      release_frag_access(prim_tab_fragptr.p);
      return;
    default:
      warningReport(signal, 16);
      return;
  }  // switch
}  // Dblqh::execLQHKEYREF()

/* -------------------------------------------------------------------------- */
/* -------                       ENTER PACKED_SIGNAL                  ------- */
/* Execution of packed signal. The packed signal can contain COMMIT, COMPLETE */
/* or LQHKEYCONF signals. These signals will be executed by their resp. exec  */
/* functions.                                                                 */
/* -------------------------------------------------------------------------- */
void Dblqh::execPACKED_SIGNAL(Signal *signal) {
  Uint32 Tstep = 0;
  Uint32 Tlength;
  Uint32 TpackedData[28];
  Uint32 sig0, sig1, sig2, sig3, sig4, sig5, sig6;

  jamEntry();
  Tlength = signal->length();
  Uint32 TsenderRef = signal->getSendersBlockRef();
  Uint32 TcommitLen = 5;
  Uint32 Tgci_lo_mask = ~(Uint32)0;

#ifdef ERROR_INSERT
  Uint32 senderBlockRef = signal->getSendersBlockRef();
#endif

  ndbrequire(Tlength <= 25);
  MEMCOPY_NO_WORDS(&TpackedData[0], &signal->theData[0], Tlength);

  if (VERIFY_PACKED_RECEIVE) {
    ndbrequire(PackedSignal::verify(&TpackedData[0], Tlength, cownref,
                                    LQH_RECEIVE_TYPES, TcommitLen));
  }

  while (Tlength > Tstep) {
    switch (TpackedData[Tstep] >> 28) {
      case ZCOMMIT:
        jam();
        sig0 = TpackedData[Tstep + 0] & 0x0FFFFFFF;
        sig1 = TpackedData[Tstep + 1];
        sig2 = TpackedData[Tstep + 2];
        sig3 = TpackedData[Tstep + 3];
        sig4 = TpackedData[Tstep + 4];
        signal->theData[0] = sig0;
        signal->theData[1] = sig1;
        signal->theData[2] = sig2;
        signal->theData[3] = sig3;
        signal->theData[4] = sig4 & Tgci_lo_mask;
        signal->header.theLength = TcommitLen;
        jamBuffer()->markEndOfSigExec();
        execCOMMIT(signal);
        Tstep += TcommitLen;
        break;
      case ZCOMPLETE:
        jam();
        sig0 = TpackedData[Tstep + 0] & 0x0FFFFFFF;
        sig1 = TpackedData[Tstep + 1];
        sig2 = TpackedData[Tstep + 2];
        signal->theData[0] = sig0;
        signal->theData[1] = sig1;
        signal->theData[2] = sig2;
        signal->header.theLength = 3;
        jamBuffer()->markEndOfSigExec();
        execCOMPLETE(signal);
        Tstep += 3;
        break;
      case ZLQHKEYCONF: {
        jam();
        LqhKeyConf *lqhKeyConf = CAST_PTR(LqhKeyConf, signal->theData);
        sig0 = TpackedData[Tstep + 0] & 0x0FFFFFFF;
        sig1 = TpackedData[Tstep + 1];
        sig2 = TpackedData[Tstep + 2];
        sig3 = TpackedData[Tstep + 3];
        sig4 = TpackedData[Tstep + 4];
        sig5 = TpackedData[Tstep + 5];
        sig6 = TpackedData[Tstep + 6];
        lqhKeyConf->connectPtr = sig0;
        lqhKeyConf->opPtr = sig1;
        lqhKeyConf->userRef = sig2;
        lqhKeyConf->readLen = sig3;
        lqhKeyConf->transId1 = sig4;
        lqhKeyConf->transId2 = sig5;
        lqhKeyConf->numFiredTriggers = sig6;
        jamBuffer()->markEndOfSigExec();
        execLQHKEYCONF(signal);
        Tstep += LqhKeyConf::SignalLength;
        break;
      }
      case ZREMOVE_MARKER:
        jam();
        sig0 = TpackedData[Tstep + 1];
        sig1 = TpackedData[Tstep + 2];
        signal->theData[0] = sig0;
        signal->theData[1] = sig1;
        if ((TpackedData[Tstep] & 1) == 0) {
          /**
           * This is the normal path where we remove a marker
           * after commit.
           */
          signal->header.theLength = 2;
        } else {
          /**
           * This is a new path that is used when removing a marker
           * after an API node failure. We indicate this in packed
           * signal by setting one of the 28 unused bits in the
           * packed signal (the first word only uses the last 4 bits
           * in the first 32-bit word.
           *
           * We indicate this to the execREMOVE_MARKER_ORD method
           * by setting the Length of the signal to 3 (we cannot
           * add an extra parameter since the signal can be sent
           * directly and not through the packed signal interface.
           */
          signal->header.theLength = 3;
        }
        jamBuffer()->markEndOfSigExec();
        execREMOVE_MARKER_ORD(signal);
        Tstep += 3;
        break;
      case ZFIRE_TRIG_REQ:
        jam();
        ndbassert(FireTrigReq::SignalLength == 4);
        sig0 = TpackedData[Tstep + 0] & 0x0FFFFFFF;
        sig1 = TpackedData[Tstep + 1];
        sig2 = TpackedData[Tstep + 2];
        sig3 = TpackedData[Tstep + 3];
        signal->theData[0] = sig0;
        signal->theData[1] = sig1;
        signal->theData[2] = sig2;
        signal->theData[3] = sig3;
        signal->header.theLength = FireTrigReq::SignalLength;
        signal->header.theSendersBlockRef = TsenderRef;
        jamBuffer()->markEndOfSigExec();
        execFIRE_TRIG_REQ(signal);
        Tstep += FireTrigReq::SignalLength;
        break;
      default:
        ndbabort();
        return;
    }  // switch
#ifdef ERROR_INSERT
    signal->header.theSendersBlockRef = senderBlockRef;
#endif
  }  // while
  ndbrequire(Tlength == Tstep);
  return;
}  // Dblqh::execPACKED_SIGNAL()

void Dblqh::execREMOVE_MARKER_ORD(Signal *signal) {
  CommitAckMarker key;
  key.transid1 = signal->theData[0];
  key.transid2 = signal->theData[1];
  bool removed_by_fail_api = (signal->header.theLength == 3);
  jamEntry();

  CommitAckMarkerPtr removedPtr;
  if (m_commitAckMarkerHash.remove(removedPtr, key)) {
    jam();
    ndbrequire(removedPtr.p->in_hash);
    removedPtr.p->in_hash = false;
    removedPtr.p->reference_count = 0;
    removedPtr.p->removed_by_fail_api = removed_by_fail_api;
    m_commitAckMarkerPool.release(removedPtr);
    checkPoolShrinkNeed(DBLQH_COMMIT_ACK_MARKER_TRANSIENT_POOL_INDEX,
                        m_commitAckMarkerPool);
  } else {
    /**
     * This can happen in a special situation. This is when a large transaction
     * commits. As it decides to commit it sends of the commit decision to the
     * API. As soon as the API receives this it will send a TC_COMMIT_ACK
     * message back to DBTC. This message could arrive before the transaction
     * is fully completed. When the TC_COMMIT_ACK it is received it is
     * immediately transferred to the DBLQH's of the transaction owning the
     * commit ack markers. Next if the node where DBTC resides fails after
     * completing the sending of the TC_COMMIT_ACK, but before the transaction
     * is completed. This can happen in cases with a large transaction doing a
     * commit.
     *
     * Next step that happens is that a new DBTC takes over the transaction to
     * complete it. It will complete the parts remaining and since all parts
     * heard of were in the Committed state, the transaction will be committed
     * and a TCKEY_FAILCONF will be sent to the API. This TCKEY_FAILCONF will
     * trigger a new TC_COMMIT_ACK which will be passed through the new DBTC
     * and sent onwards to the participating DBLQH's. The REMOVE_MARKER_ORD
     * signal is received from DBTC here in DBLQH can thus in some cases be
     * received when multiple times, possibly even more than twice since there
     * could be multiple failures serially. The second and further times will
     * end up in this else-branch. There is no longer any record in DBLQH
     * with the received transaction id.
     *
     * This only happens as part of normal TC_COMMIT_ACK reception, so not when
     * the flag removed_by_fail_api is set.
     *
     * It can also happen in a situation where we are performing a TC takeover.
     * If an LQH instance reports having a marker, then the new TC inserts all
     * LQH instances as having the marker. So this means that a lot of LQH
     * instances will receive this message even when they haven't claimed to
     * have the marker. This is so since the protocol in LQH_TRANSCONF doesn't
     * specify which LQH instances that sent the LQH_TRANSCONF. Actually this
     * isn't entirely true since the senders block reference is always received
     * as part of Protocol6, thus we can actually find the LQH instance from
     * this. But for now we take this safe approach and send too many signals.
     * These signals will also end up in this branch.
     */
#if (defined VM_TRACE || defined ERROR_INSERT) && defined(wl4391_todo)
    g_eventLogger->info(
        "%u Rem marker failed[%.8x %.8x] remove_by_fail_api = %u", instance(),
        key.transid1, key.transid2, removed_by_fail_api);
#endif
  }
#ifdef MARKER_TRACE
  g_eventLogger->info("%u Rem marker[%.8x %.8x]", instance(), key.transid1,
                      key.transid2);
#endif
}

/* -------------------------------------------------------------------------- */
/* -------                 ENTER SEND_PACKED                          ------- */
/* Used to force a packed signal to be sent if local signal buffer is not     */
/* empty.                                                                     */
/* -------------------------------------------------------------------------- */
void Dblqh::execSEND_PACKED(Signal *signal) {
  HostRecordPtr Thostptr;
  UintR TpackedListIndex = cpackedListIndex;
  jamEntry();
  for (Uint32 i = 0; i < TpackedListIndex; i++) {
    Thostptr.i = cpackedList[i];
    ptrAss(Thostptr, hostRecord);
    jam();
    ndbrequire(Thostptr.i - 1 < MAX_NDB_NODES - 1);

    // LQH pack and send
    for (Uint32 j = Thostptr.p->lqh_pack_mask.find_first();
         j != BitmaskImpl::NotFound;
         j = Thostptr.p->lqh_pack_mask.find_next(j + 1)) {
      jamDebug();
      struct PackedWordsContainer *container = &Thostptr.p->lqh_pack[j];
      ndbassert(container->noOfPackedWords > 0);
      sendPackedSignal(signal, container);
    }
    // TC pack and send
    for (Uint32 j = Thostptr.p->tc_pack_mask.find_first();
         j != BitmaskImpl::NotFound;
         j = Thostptr.p->tc_pack_mask.find_next(j + 1)) {
      jamDebug();
      struct PackedWordsContainer *container = &Thostptr.p->tc_pack[j];
      ndbassert(container->noOfPackedWords > 0);
      sendPackedSignal(signal, container);
    }

    Thostptr.p->lqh_pack_mask.clear();
    Thostptr.p->tc_pack_mask.clear();
    Thostptr.p->inPackedList = false;
  }  // for
  cpackedListIndex = 0;
  return;
}  // Dblqh::execSEND_PACKED()

void Dblqh::updatePackedList(Signal *signal, HostRecord *ahostptr,
                             Uint16 hostId) {
  Uint32 TpackedListIndex = cpackedListIndex;
  if (ahostptr->inPackedList == false) {
    jamDebug();
    ahostptr->inPackedList = true;
    ahostptr->lqh_pack_mask.clear();
    ahostptr->tc_pack_mask.clear();
    cpackedList[TpackedListIndex] = hostId;
    cpackedListIndex = TpackedListIndex + 1;
  }  // if
}  // Dblqh::updatePackedList()

void Dblqh::execREAD_PSEUDO_REQ(Uint32 opPtrI, Uint32 attrId, Uint32 *out,
                                Uint32 out_words) {
  jamEntryDebug();
  TcConnectionrecPtr regTcPtr;
  regTcPtr.i = opPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));

  switch (attrId) {
    case AttributeHeader::RANGE_NO:
      ndbassert(1 <= out_words);
      out[0] = regTcPtr.p->m_scan_curr_range_no;
      break;
    case AttributeHeader::RECORDS_IN_RANGE:
    case AttributeHeader::INDEX_STAT_KEY:
    case AttributeHeader::INDEX_STAT_VALUE: {
      jam();
      // scanptr gets reset somewhere within the timeslice
      ScanRecordPtr tmp;
      tmp.i = regTcPtr.p->tcScanRec;
      ndbrequire(c_scanRecordPool.getValidPtr(tmp));
      c_tux->execREAD_PSEUDO_REQ(tmp.p->scanAccPtr, attrId, out, out_words);
      break;
    }
    case AttributeHeader::LOCK_REF: {
      /* Return 3x 32-bit words
       *  - LQH instance info
       *  - TC operation index
       *  - Bottom 32-bits of LQH-local key-request id (for uniqueness)
       */
      jam();
      ndbassert(3 <= out_words);
      out[0] = (getOwnNodeId() << 16) | regTcPtr.p->fragmentid;
      out[1] = regTcPtr.p->tcOprec;
      out[2] = (Uint32)regTcPtr.p->lqhKeyReqId;
      break;
    }
    case AttributeHeader::OP_ID: {
      jam();
      ndbassert(8 <= out_words * 4);
      memcpy(out, &regTcPtr.p->lqhKeyReqId, 8);
      break;
    }
    case AttributeHeader::CORR_FACTOR64: {
      Uint32 add = 0;
      ScanRecordPtr tmp;
      tmp.i = regTcPtr.p->tcScanRec;
      if (tmp.i != RNIL) {
        ndbrequire(c_scanRecordPool.getValidPtr(tmp));
        add = tmp.p->m_curr_batch_size_rows;
      }

      ndbassert(2 <= out_words);
      out[0] = regTcPtr.p->m_corrFactorLo + add;
      out[1] = regTcPtr.p->m_corrFactorHi;
      break;
    }
    default:
      ndbabort();
  }
}

/* ************>> */
/*  TUPKEYCONF  > */
/* ************>> */
void Dblqh::execTUPKEYCONF(Signal *signal) {
  TcConnectionrecPtr regTcPtr = m_tc_connect_ptr;
  switch (regTcPtr.p->transactionState) {
    case TcConnectionrec::SCAN_TUPKEY: {
      jamDebug();
      scanTupkeyConfLab(signal, regTcPtr.p);
      return;
    }
    case TcConnectionrec::WAIT_TUP: {
      FragrecordPtr regFragptr = fragptr;
      const TupKeyConf *const tupKeyConf = (TupKeyConf *)signal->getDataPtr();
      jamDebug();
      if (regTcPtr.p->seqNoReplica == 0)  // Primary replica
        regTcPtr.p->numFiredTriggers = tupKeyConf->numFiredTriggers;

      if (!m_is_in_query_thread) {
        Fragrecord::UsageStat &useStat = regFragptr.p->m_useStat;
        useStat.m_keyReqWordsReturned += tupKeyConf->readLength;
        useStat.m_keyInstructionCount += tupKeyConf->noExecInstructions;
      }
      tupkeyConfLab(signal, regTcPtr);
      return;
    }
    case TcConnectionrec::COPY_TUPKEY: {
      jam();
      copyTupkeyConfLab(signal, regTcPtr);
      return;
    }
    case TcConnectionrec::WAIT_TUP_TO_ABORT: {
      Uint32 activeCreat = regTcPtr.p->activeCreat;
      jam();
      /* -------------------------------------------------------------------------
       */
      // Abort was not ready to start until this signal came back. Now we are
      // ready to start the abort.
      /* -------------------------------------------------------------------------
       */
      if (unlikely(activeCreat == Fragrecord::AC_NR_COPY)) {
        jam();
        ndbrequire(regTcPtr.p->m_nr_delete.m_cnt);
        regTcPtr.p->m_nr_delete.m_cnt--;
        if (regTcPtr.p->m_nr_delete.m_cnt) {
          jam();
          /**
           * Let operation wait for pending NR operations
           *   even for before writing log...(as it's simpler)
           */

#ifdef VM_TRACE
          /**
           * Only disk table can have pending ops...
           */
          TablerecPtr tablePtr;
          tablePtr.i = regTcPtr.p->tableref;
          ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);
          ndbrequire(tablePtr.p->m_disk_table);
#endif
          return;
        }
      }

      abortCommonLab(signal, regTcPtr);
      return;
    }
    case TcConnectionrec::WAIT_ACC_ABORT: {
      jam();
      /* -------------------------------------------------------------------------
       */
      /*      IGNORE SINCE ABORT OF THIS OPERATION IS ONGOING ALREADY. */
      /* -------------------------------------------------------------------------
       */
      return;
    }
    default: {
      jamLine(regTcPtr.p->transactionState);
      ndbabort();
    }
  }  // switch
}  // Dblqh::execTUPKEYCONF()

/* ************> */
/*  TUPKEYREF  > */
/* ************> */
void Dblqh::execTUPKEYREF(Signal *signal) {
  const TupKeyRef *const tupKeyRef = (TupKeyRef *)signal->getDataPtr();
  jamEntryDebug();
  TcConnectionrecPtr tcConnectptr = m_tc_connect_ptr;
  terrorCode = tupKeyRef->errorCode;
  TRACE_OP(tcConnectptr.p, "TUPKEYREF");

#ifdef VM_TRACE
  if (unlikely(tcConnectptr.p->activeCreat == Fragrecord::AC_NR_COPY)) {
    ndbassert(tcConnectptr.p->transactionState == TcConnectionrec::WAIT_TUP ||
              tcConnectptr.p->transactionState ==
                  TcConnectionrec::WAIT_TUP_TO_ABORT);
  }
#endif
  switch (tcConnectptr.p->transactionState) {
    case TcConnectionrec::SCAN_TUPKEY: {
      jamDebug();
      scanTupkeyRefLab(signal, tcConnectptr);
      return;
    }
    case TcConnectionrec::WAIT_TUP: {
      const TupKeyRef *const tupKeyRef = (TupKeyRef *)signal->getDataPtr();
      jamDebug();
      if (unlikely(tcConnectptr.p->activeCreat == Fragrecord::AC_NR_COPY)) {
        jam();
        ndbrequire(tcConnectptr.p->m_nr_delete.m_cnt);
        tcConnectptr.p->m_nr_delete.m_cnt--;
      }
      if (!m_is_in_query_thread) {
        Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
        useStat.m_keyRefCount++;
        useStat.m_keyInstructionCount += tupKeyRef->noExecInstructions;
      }
      abortErrorLab(signal, tcConnectptr);
      return;
    }
    case TcConnectionrec::COPY_TUPKEY: {
      copyTupkeyRefLab(signal, tcConnectptr);
      return;
    }
    case TcConnectionrec::WAIT_TUP_TO_ABORT: {
      jam();
      if (unlikely(tcConnectptr.p->activeCreat == Fragrecord::AC_NR_COPY)) {
        jam();
        ndbrequire(tcConnectptr.p->m_nr_delete.m_cnt);
        tcConnectptr.p->m_nr_delete.m_cnt--;
      }
      /* -------------------------------------------------------------------------
       */
      // Abort was not ready to start until this signal came back. Now we are
      // ready to start the abort.
      /* -------------------------------------------------------------------------
       */
      abortCommonLab(signal, tcConnectptr);
      return;
    }
    case TcConnectionrec::WAIT_ACC_ABORT: {
      jam();
      /* -------------------------------------------------------------------------
       */
      /*       IGNORE SINCE ABORT OF THIS OPERATION IS ONGOING ALREADY. */
      /* -------------------------------------------------------------------------
       */
      return;
    }
    default:
      jamLine(tcConnectptr.p->transactionState);
      ndbabort();
  }  // switch
}  // Dblqh::execTUPKEYREF()

void Dblqh::sendPackedSignal(Signal *signal,
                             struct PackedWordsContainer *container) {
  Uint32 noOfWords = container->noOfPackedWords;
  BlockReference hostRef = container->hostBlockRef;
  container->noOfPackedWords = 0;
  MEMCOPY_NO_WORDS(&signal->theData[0], &container->packedWords[0], noOfWords);
  if (VERIFY_PACKED_SEND) {
    int receiveTypes = (refToMain(hostRef) == getDBLQH()) ? LQH_RECEIVE_TYPES
                                                          : TC_RECEIVE_TYPES;
    ndbrequire(PackedSignal::verify(&signal->theData[0], noOfWords, hostRef,
                                    receiveTypes,
                                    5)); /* Commit signal length */
  }
  sendSignal(hostRef, GSN_PACKED_SIGNAL, signal, noOfWords, JBB);
}

void Dblqh::sendCommitLqh(Signal *signal, BlockReference alqhBlockref,
                          const TcConnectionrec *regTcPtr) {
  ndbassert(!m_is_in_query_thread);
  Uint32 instanceKey = refToInstance(alqhBlockref);
  ndbassert(refToMain(alqhBlockref) == getDBLQH());

  if (instanceKey > MAX_NDBMT_LQH_THREADS) {
    /* No send packed support in these cases */
    jam();
    signal->theData[0] = regTcPtr->clientConnectrec;
    signal->theData[1] = regTcPtr->transid[0];
    signal->theData[2] = regTcPtr->transid[1];
    sendSignal(alqhBlockref, GSN_COMMIT, signal, 3, JBB);
    return;
  }

  HostRecordPtr Thostptr;

  Thostptr.i = refToNode(alqhBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  struct PackedWordsContainer *container = &Thostptr.p->lqh_pack[instanceKey];

  Uint32 Tdata[5];
  Tdata[0] = regTcPtr->clientConnectrec;
  Tdata[1] = regTcPtr->gci_hi;
  Tdata[2] = regTcPtr->transid[0];
  Tdata[3] = regTcPtr->transid[1];
  Tdata[4] = regTcPtr->gci_lo;
  Uint32 len = 5;

  if (container->noOfPackedWords > 25 - len) {
    jam();
    sendPackedSignal(signal, container);
  } else {
    jam();
    updatePackedList(signal, Thostptr.p, Thostptr.i);
  }

  Tdata[0] |= (ZCOMMIT << 28);
  Uint32 pos = container->noOfPackedWords;
  container->noOfPackedWords = pos + len;
  Thostptr.p->lqh_pack_mask.set(instanceKey);
  memcpy(&container->packedWords[pos], &Tdata[0], len << 2);
}

void Dblqh::sendCompleteLqh(Signal *signal, BlockReference alqhBlockref,
                            const TcConnectionrec *regTcPtr) {
  Uint32 instanceKey = refToInstance(alqhBlockref);
  ndbassert(refToMain(alqhBlockref) == getDBLQH());

  if (instanceKey > MAX_NDBMT_LQH_THREADS) {
    /* No send packed support in these cases */
    jam();
    signal->theData[0] = regTcPtr->clientConnectrec;
    signal->theData[1] = regTcPtr->transid[0];
    signal->theData[2] = regTcPtr->transid[1];
    sendSignal(alqhBlockref, GSN_COMPLETE, signal, 3, JBB);
    return;
  }

  HostRecordPtr Thostptr;

  Thostptr.i = refToNode(alqhBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  struct PackedWordsContainer *container = &Thostptr.p->lqh_pack[instanceKey];

  Uint32 Tdata[3];
  Tdata[0] = regTcPtr->clientConnectrec;
  Tdata[1] = regTcPtr->transid[0];
  Tdata[2] = regTcPtr->transid[1];
  Uint32 len = 3;

  if (container->noOfPackedWords > 22) {
    jam();
    sendPackedSignal(signal, container);
  } else {
    jam();
    updatePackedList(signal, Thostptr.p, Thostptr.i);
  }

  Tdata[0] |= (ZCOMPLETE << 28);
  Uint32 pos = container->noOfPackedWords;
  container->noOfPackedWords = pos + len;
  Thostptr.p->lqh_pack_mask.set(instanceKey);
  memcpy(&container->packedWords[pos], &Tdata[0], len << 2);
}

void Dblqh::sendCommittedTc(Signal *signal, BlockReference atcBlockref,
                            const TcConnectionrec *regTcPtr) {
  Uint32 instanceKey = refToInstance(atcBlockref);

  ndbassert(refToMain(atcBlockref) == DBTC);
  if (instanceKey > MAX_NDBMT_TC_THREADS) {
    /* No send packed support in these cases */
    jam();
    signal->theData[0] = regTcPtr->clientConnectrec;
    signal->theData[1] = regTcPtr->transid[0];
    signal->theData[2] = regTcPtr->transid[1];
    sendSignal(atcBlockref, GSN_COMMITTED, signal, 3, JBB);
    return;
  }

  HostRecordPtr Thostptr;
  Thostptr.i = refToNode(atcBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  struct PackedWordsContainer *container = &Thostptr.p->tc_pack[instanceKey];

  Uint32 Tdata[3];
  Tdata[0] = regTcPtr->clientConnectrec;
  Tdata[1] = regTcPtr->transid[0];
  Tdata[2] = regTcPtr->transid[1];
  Uint32 len = 3;

  if (container->noOfPackedWords > 22) {
    jam();
    sendPackedSignal(signal, container);
  } else {
    jam();
    updatePackedList(signal, Thostptr.p, Thostptr.i);
  }

  Tdata[0] |= (ZCOMMITTED << 28);
  Uint32 pos = container->noOfPackedWords;
  container->noOfPackedWords = pos + len;
  Thostptr.p->tc_pack_mask.set(instanceKey);
  memcpy(&container->packedWords[pos], &Tdata[0], len << 2);
}

void Dblqh::sendCompletedTc(Signal *signal, BlockReference atcBlockref,
                            const TcConnectionrec *regTcPtr) {
  Uint32 instanceKey = refToInstance(atcBlockref);

  ndbassert(refToMain(atcBlockref) == DBTC);
  if (instanceKey > MAX_NDBMT_TC_THREADS) {
    /* No handling of send packed in those cases */
    jam();
    signal->theData[0] = regTcPtr->clientConnectrec;
    signal->theData[1] = regTcPtr->transid[0];
    signal->theData[2] = regTcPtr->transid[1];
    sendSignal(atcBlockref, GSN_COMPLETED, signal, 3, JBB);
    return;
  }

  HostRecordPtr Thostptr;
  Thostptr.i = refToNode(atcBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  struct PackedWordsContainer *container = &Thostptr.p->tc_pack[instanceKey];

  Uint32 Tdata[3];
  Tdata[0] = regTcPtr->clientConnectrec;
  Tdata[1] = regTcPtr->transid[0];
  Tdata[2] = regTcPtr->transid[1];
  Uint32 len = 3;

  if (container->noOfPackedWords > 22) {
    jam();
    sendPackedSignal(signal, container);
  } else {
    jam();
    updatePackedList(signal, Thostptr.p, Thostptr.i);
  }

  Tdata[0] |= (ZCOMPLETED << 28);
  Uint32 pos = container->noOfPackedWords;
  container->noOfPackedWords = pos + len;
  Thostptr.p->tc_pack_mask.set(instanceKey);
  memcpy(&container->packedWords[pos], &Tdata[0], len << 2);
}

void Dblqh::sendLqhkeyconfTc(Signal *signal, BlockReference atcBlockref,
                             const TcConnectionrecPtr tcConnectptr) {
  LqhKeyConf *lqhKeyConf;
  struct PackedWordsContainer *container = nullptr;
  bool send_packed = true;
  HostRecordPtr Thostptr;
  Thostptr.i = refToNode(atcBlockref);
  Uint32 instanceKey = refToInstance(atcBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  Uint32 block = refToMain(atcBlockref);

  if (block == getDBLQH()) {
    if (instanceKey <= MAX_NDBMT_LQH_THREADS) {
      container = &Thostptr.p->lqh_pack[instanceKey];
    } else {
      send_packed = false;
    }
  } else if (block == DBTC) {
    if (instanceKey <= MAX_NDBMT_TC_THREADS) {
      container = &Thostptr.p->tc_pack[instanceKey];
    } else {
      send_packed = false;
    }
  } else {
    send_packed = false;
  }

  /*******************************************************************
  // Normal path
  // This signal was intended for DBTC as part of the normal transaction
  // execution.
  // More unusual path
  // This signal was intended for DBLQH as part of log execution or
  // node recovery.
  // Yet another path
  // Intended for DBSPJ as part of join processing
  ********************************************************************/
  if (send_packed) {
    if (container->noOfPackedWords > (25 - LqhKeyConf::SignalLength)) {
      jamDebug();
      sendPackedSignal(signal, container);
    } else {
      jamDebug();
      updatePackedList(signal, Thostptr.p, Thostptr.i);
    }  // if
    lqhKeyConf =
        (LqhKeyConf *)&container->packedWords[container->noOfPackedWords];
    container->noOfPackedWords += LqhKeyConf::SignalLength;

    if (block == getDBLQH()) {
      if (instanceKey <= MAX_NDBMT_LQH_THREADS)
        Thostptr.p->lqh_pack_mask.set(instanceKey);
    } else if (block == DBTC) {
      if (instanceKey <= MAX_NDBMT_TC_THREADS)
        Thostptr.p->tc_pack_mask.set(instanceKey);
    }
  } else {
    lqhKeyConf = (LqhKeyConf *)&signal->theData[0];
  }

  Uint32 ptrAndType = tcConnectptr.i | (ZLQHKEYCONF << 28);
  Uint32 tcOprec = tcConnectptr.p->tcOprec;
  Uint32 ownRef = cownref;
  lqhKeyConf->connectPtr = ptrAndType;
  lqhKeyConf->opPtr = tcOprec;
  lqhKeyConf->userRef = ownRef;

  Uint32 readlenAi = tcConnectptr.p->readlenAi;
  Uint32 transid1 = tcConnectptr.p->transid[0];
  Uint32 transid2 = tcConnectptr.p->transid[1];
  Uint32 numFiredTriggers = tcConnectptr.p->numFiredTriggers;
  lqhKeyConf->readLen = readlenAi;
  lqhKeyConf->transId1 = transid1;
  lqhKeyConf->transId2 = transid2;
  lqhKeyConf->numFiredTriggers = numFiredTriggers;

  if (!send_packed) {
    lqhKeyConf->connectPtr = tcConnectptr.i;
    if (block == getRESTORE()) {
      ndbrequire(refToNode(atcBlockref) == cownNodeid &&
                 refToInstance(atcBlockref) == instance());
      EXECUTE_DIRECT(getRESTORE(), GSN_LQHKEYCONF, signal,
                     LqhKeyConf::SignalLength);
    } else {
      sendSignal(atcBlockref, GSN_LQHKEYCONF, signal, LqhKeyConf::SignalLength,
                 JBB);
    }
  }
}  // Dblqh::sendLqhkeyconfTc()

/* ------------------------------------------------------------------------- */
/* -------                HANDLE ATTRINFO SIGNALS                    ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
/* ************************************************************************>>*/
/*  TUP_ATTRINFO: Interpreted execution in DBTUP generates redo-log info     */
/*  which is sent back to DBLQH for logging. This is because the decision    */
/*  to execute or not is made in DBTUP and thus we cannot start logging until*/
/*  DBTUP part has been run.                                                 */
/* ************************************************************************>>*/
void Dblqh::execTUP_ATTRINFO(Signal *signal) {
  jamEntryDebug();
  TcConnectionrecPtr tcConnectptr = m_tc_connect_ptr;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;

  ndbrequire(regTcPtr->transactionState == TcConnectionrec::WAIT_TUP);

  /* TUP_ATTRINFO signal is unrelated to ATTRINFO
   * It just transports a section IVAL from TUP back to
   * LQH
   */
  Uint32 tupAttrInfoWords = signal->theData[1];
  Uint32 tupAttrInfoIVal = signal->theData[2];

  ndbassert(tupAttrInfoWords > 0);
  ndbassert(tupAttrInfoIVal != RNIL);

  /* If we have stored ATTRINFO that we sent to TUP,
   * free it now
   */
  if (regTcPtr->attrInfoIVal != RNIL) {
    /* We should be expecting to receive attrInfo back */
    ndbassert(!(regTcPtr->m_flags & TcConnectionrec::OP_SAVEATTRINFO));
    releaseSection(regTcPtr->attrInfoIVal);
    regTcPtr->attrInfoIVal = RNIL;
  }

  /* Store reference to ATTRINFO from TUP */
  regTcPtr->attrInfoIVal = tupAttrInfoIVal;
  regTcPtr->currTupAiLen = tupAttrInfoWords;
}  // Dblqh::execTUP_ATTRINFO()

/* ------------------------------------------------------------------------- */
/* ------         FIND TRANSACTION BY USING HASH TABLE               ------- */
/*                                                                           */
/* We keep a hash structure of TcConnectionrec which are identified by:      */
/*  - Id of Transaction owning this TcConnectionrec.                         */
/*  - A 'tcOpRec' id which uniquely(*below) identify this TcConnectionRec    */
/*    within this specific transaction.                                      */
/*  - An optional 'hashHi' id used for SCANREQs in cases where 'tcOpRec'     */
/*    on its own can't provide uniqueness.                                    */
/*    This is required in cases where there are multiple (internal) clients  */
/*    producing REQs where the uniqueness is only guaranteed within          */
/*    each client. Currently the only such client is the SPJ block.          */
/*                                                                           */
/* Hash lookup of TcConnectionrecPtr might be required for TcConnectionRecs  */
/* having a lifetime beyond the initial REQ. That is:                        */
/*  - Short requests awaiting for a later ATTR- or KEYINFO.                  */
/*  - SCANREQ which may need a later NEXTREQ to fetch more or close scan     */
/*  - Transactional (non-DirtyOp) REQs which need a later abort, commit      */
/*    or unlock request.                                                     */
/*                                                                           */
/* TcConnectionrec's identified as not requiring hash lookup are not         */
/* inserted in the hash table!                                               */
/*                                                                           */
/* 'tcOpRec' ids comes from TC. Where TC has released the record (dirtyOp),  */
/* the id can be reused. Therefore it cannot be considered 'unique' beyond   */
/* the reception of the request signal (train). For non dirty operations it  */
/* is unique for the lifecycle of the operation at TC.                       */
/*                                                                           */
/* We also include the TC block reference in the hash lookup. This is a      */
/* requirement for SCAN operations, for key operations it makes it possible  */
/* to use the same transaction id from different TC blocks and still be able */
/* to distinguish the correct operation.                                     */
/*                                                                           */
/* There is one case where we might pick the wrong operation record. This    */
/* is when it comes from a TC takeover operation. In this case we only       */
/* know about the TC node and the TC operation record. We don't know the     */
/* TC instance used. However it is not of any major concern, even if we      */
/* did support multi-TC instance transactions they would still be committed  */
/* or aborted as one unit and thus as long as all operations are treated     */
/* the same way it is ok to sort of pick the wrong operation. The operation  */
/* not picked will be picked by another operation instead.                   */
/*                                                                           */
/* The partial_fit_ok flag indicates that this is a search from TC takeover  */
/* thread. In this case we only compare the node and block from the          */
/* block reference, we ignore the block instance.                            */
/*                                                                           */
/* NOTE:                                                                     */
/*   The internal clients of NDB does *not* guarantee hash uniqueness        */
/*   for LQHKEYREQs as described above (SPJ, node restart ..). However,      */
/*   these requests are all 'long', 'dirtyOp'-requests and thus neither      */
/*   inserted nor searched after in the hash table.                          */
/*                                                                           */
/* ------------------------------------------------------------------------- */
int Dblqh::findTransaction(UintR Transid1, UintR Transid2, UintR TcOprec,
                           Uint32 hi, bool is_key_operation,
                           bool partial_fit_ok,
                           TcConnectionrecPtr &tcConnectptr) {
  TcConnectionrecPtr locTcConnectptr;

  ndbassert(partial_fit_ok == false || is_key_operation == true);
  const Uint32 ThashIndex = getHashIndex(Transid1, Transid2, TcOprec);
  locTcConnectptr.i = ctransidHash[ThashIndex];
  while (locTcConnectptr.i != RNIL) {
    ndbrequire(tcConnect_pool.getUncheckedPtrRW(locTcConnectptr));
    if ((locTcConnectptr.p->transid[0] == Transid1) &&
        (locTcConnectptr.p->transid[1] == Transid2) &&
        (locTcConnectptr.p->tcOprec == TcOprec)) {
      if (((partial_fit_ok == false) && locTcConnectptr.p->tcHashKeyHi == hi) ||
          (partial_fit_ok == true)) {
        if ((is_key_operation && locTcConnectptr.p->tcScanRec == RNIL) ||
            (!is_key_operation && locTcConnectptr.p->tcScanRec != RNIL)) {
          /* FIRST PART OF TRANSACTION CORRECT */
          /* SECOND PART ALSO CORRECT */
          /* THE OPERATION RECORD POINTER IN TC WAS ALSO CORRECT */
          jam();
          tcConnectptr = locTcConnectptr;
          ndbrequire(Magic::check_ptr(locTcConnectptr.p));
          ndbassert(tcConnectptr.p->hashIndex == ThashIndex);
          return (int)ZOK;
        }
      }
    }  // if
    jam();
    /* THIS WAS NOT THE TRANSACTION WHICH WAS SOUGHT */
    locTcConnectptr.i = locTcConnectptr.p->nextHashRec;
    ndbrequire(Magic::check_ptr(locTcConnectptr.p));
  }  // while
  /* WE DID NOT FIND THE TRANSACTION, REPORT NOT FOUND */
  return (int)ZNOT_FOUND;
}  // Dblqh::findTransaction()

inline Uint32 Dblqh::getHashKey(UintR Transid1, UintR Transid2 [[maybe_unused]],
                                UintR TcOprec) {
  return (Transid1 ^ TcOprec);
}

inline Uint32 Dblqh::getHashIndex(UintR Transid1,
                                  UintR Transid2 [[maybe_unused]],
                                  UintR TcOprec) {
  return getHashKey(Transid1, Transid2, TcOprec) % ctransidHashSize;
}

inline Uint32 Dblqh::getHashIndex(TcConnectionrec *regTcPtr) {
  return getHashIndex(regTcPtr->transid[0], regTcPtr->transid[1],
                      regTcPtr->tcOprec);
}

inline static void prefetch_op_record_3(Uint32 *op_ptr) {
  NDB_PREFETCH_WRITE(op_ptr);
  NDB_PREFETCH_WRITE(op_ptr + 16);
  NDB_PREFETCH_WRITE(op_ptr + 32);
}

inline static void prefetch_op_record_4(Uint32 *op_ptr) {
  // First cacheline prefetched in TransientPool
  NDB_PREFETCH_WRITE(op_ptr + 16);
  NDB_PREFETCH_WRITE(op_ptr + 32);
  NDB_PREFETCH_WRITE(op_ptr + 48);
  NDB_PREFETCH_WRITE(op_ptr + 64);
}

bool Dblqh::seize_op_rec(TcConnectionrecPtr &tcConnectptr) {
  if (ERROR_INSERTED(5031)) {
    jam();
    return false;
  }
  TcConnectionrecPtr opPtr;
  if (unlikely(!tcConnect_pool.seize(opPtr))) {
    jam();
    return false;
  }
  opPtr.p->ptrI = opPtr.i;
  prefetch_op_record_4((Uint32 *)opPtr.p);
  ndbrequire(opPtr.i >= ctcConnectReserved);
  if (unlikely(!c_acc->seize_op_rec(opPtr.i, reference(),
                                    opPtr.p->accConnectrec,
                                    &opPtr.p->accConnectPtrP))) {
    goto acc_fail;
  }
  if (unlikely(!c_tup->seize_op_rec(opPtr.i, reference(),
                                    opPtr.p->tupConnectrec,
                                    &opPtr.p->tupConnectPtrP))) {
    goto tup_fail;
  }
  opPtr.p->tcTimer = cLqhTimeOutCount;
  c_tup->prepare_op_pointer(opPtr.p->tupConnectrec, opPtr.p->tupConnectPtrP);
  tcConnectptr = opPtr;
  m_tc_connect_ptr = opPtr;
  ndbrequire(Magic::check_ptr(opPtr.p->accConnectPtrP));
  ndbrequire(Magic::check_ptr(opPtr.p->tupConnectPtrP));
  return true;

tup_fail:
  jam();
  ndbrequire(Magic::check_ptr(opPtr.p));
  c_acc->release_op_rec(opPtr.p->accConnectrec, opPtr.p->accConnectPtrP);

acc_fail:
  jam();
  ndbrequire(Magic::check_ptr(opPtr.p));
  tcConnect_pool.release(opPtr);
  checkPoolShrinkNeed(DBLQH_OPERATION_RECORD_TRANSIENT_POOL_INDEX,
                      tcConnect_pool);
  return false;
}

void Dblqh::release_op_rec(TcConnectionrecPtr opPtr) {
  c_tup->release_op_rec(opPtr.p->tupConnectrec, opPtr.p->tupConnectPtrP);
  c_acc->release_op_rec(opPtr.p->accConnectrec, opPtr.p->accConnectPtrP);
  tcConnect_pool.release(opPtr);
  checkPoolShrinkNeed(DBLQH_OPERATION_RECORD_TRANSIENT_POOL_INDEX,
                      tcConnect_pool);
}

/* ==========================================================================
 * =======                        SEIZE TC CONNECT RECORD             =======
 *
 *       GETS A NEW TC CONNECT RECORD FROM FREELIST.
 * ========================================================================= */
void Dblqh::seizeTcrec(TcConnectionrecPtr &tcConnectptr) {
  TcConnectionrecPtr locTcConnectptr;

  locTcConnectptr.i = cfirstfreeTcConrec;

  Uint32 numFree = ctcNumFree;
  Uint32 timeOutCount = cLqhTimeOutCount;

  ndbrequire(tcConnect_pool.getUncheckedPtrRW(locTcConnectptr));

  /**
   * We are going to write to most of the operation object which is
   * likely to cover at least 240 bytes and thus at least 4
   * cache lines ahead. We will prefetch those immediately now
   * to ensure that memory accesses are started as soon as possible.
   */
  prefetch_op_record_4((Uint32 *)locTcConnectptr.p);

  Uint32 nextTc = locTcConnectptr.p->nextTcConnectrec;

  locTcConnectptr.p->nextTcConnectrec = RNIL;
  locTcConnectptr.p->clientConnectrec = RNIL;
  locTcConnectptr.p->clientBlockref = RNIL;
  locTcConnectptr.p->tableref = RNIL;
  locTcConnectptr.p->hashIndex = RNIL;
  locTcConnectptr.p->nextSeqNoReplica = 0;
  locTcConnectptr.p->seqNoReplica = 0;
  locTcConnectptr.p->m_committed_log_space = 0;
  locTcConnectptr.p->m_dealloc_state = TcConnectionrec::DA_IDLE;
  locTcConnectptr.p->m_dealloc_data.m_unused = RNIL;
  locTcConnectptr.p->tcScanRec = RNIL;
  ndbrequire(Magic::check_ptr(locTcConnectptr.p));

  ctcNumFree = numFree - 1;
  cfirstfreeTcConrec = nextTc;

  locTcConnectptr.p->tcTimer = timeOutCount;
  locTcConnectptr.p->abortState = TcConnectionrec::ABORT_IDLE;
  locTcConnectptr.p->connectState = TcConnectionrec::CONNECTED;
  locTcConnectptr.p->savePointId = 0;
  locTcConnectptr.p->gci_hi = 0;
  locTcConnectptr.p->gci_lo = 0;
  locTcConnectptr.p->errorCode = 0;
  c_tup->prepare_op_pointer(locTcConnectptr.p->tupConnectrec,
                            locTcConnectptr.p->tupConnectPtrP);

  tcConnectptr = locTcConnectptr;
  m_tc_connect_ptr = locTcConnectptr;
  ndbrequire(Magic::check_ptr(locTcConnectptr.p->tupConnectPtrP));
}  // Dblqh::seizeTcrec()

bool Dblqh::checkTransporterOverloaded(Signal *signal, const NodeBitmask &all,
                                       const LqhKeyReq *req) {
  /* FC : Quick exit if the mask is clear? */
  // nodes likely to be affected by this op
  NodeBitmask mask;
  // tc
  Uint32 tc_node = refToNode(req->tcBlockref);
  if (tc_node < MAX_NODES)  // not worth to crash here
    mask.set(tc_node);
  const Uint8 op = LqhKeyReq::getOperation(req->requestInfo);
  if (op == ZREAD || op == ZREAD_EX || op == ZUNLOCK) {
    // the receiver
    Uint32 api_node = refToNode(req->variableData[0]);
    if (api_node < MAX_NODES)  // not worth to crash here
      mask.set(api_node);
  } else {
    // next replica
    Uint32 replica_node = LqhKeyReq::getNextReplicaNodeId(req->fragmentData);
    if (replica_node < MAX_NODES)  // could be ZNIL
      mask.set(replica_node);
    // event subscribers
    const Suma *suma = (Suma *)globalData.getBlock(SUMA);
    mask.bitOR(suma->getSubscriberNodes());
  }
  mask.bitAND(all);
  if (likely(mask.isclear())) {
    return false;
  }

  jam();
  /* Overloaded, do some accounting */
  c_keyOverloads++;

  if (tc_node < MAX_NODES && all.get(tc_node)) {
    jam();
    c_keyOverloadsTcNode++;
  }

  if (op == ZREAD || op == ZREAD_EX || op == ZUNLOCK) {
    jam();
    // the receiver
    Uint32 api_node = refToNode(req->variableData[0]);
    if ((api_node < MAX_NODES) &&  // not worth to crash here
        (all.get(api_node))) {
      jam();
      c_keyOverloadsReaderApi++;
    }
  } else {
    jam();
    // write
    // next replica
    Uint32 replica_node = LqhKeyReq::getNextReplicaNodeId(req->fragmentData);
    if ((replica_node < MAX_NODES) && (all.get(replica_node))) {
      jam();
      c_keyOverloadsPeerNode++;
    }

    // event subscribers
    const Suma *suma = (Suma *)globalData.getBlock(SUMA);
    NodeBitmask subscribers = suma->getSubscriberNodes();
    subscribers.bitAND(all);
    if (!subscribers.isclear()) {
      jam();
      c_keyOverloadsSubscriber++;
    }
  }

  return true;
}

void Dblqh::execSIGNAL_DROPPED_REP(Signal *signal) {
  /* An incoming signal was dropped, handle it
   * Dropped signal really means that we ran out of
   * long signal buffering to store its sections
   */
  jamEntry();

  if (!assembleDroppedFragments(signal)) {
    jam();
    return;
  }

  const SignalDroppedRep *rep = (SignalDroppedRep *)&signal->theData[0];
  Uint32 originalGSN = rep->originalGsn;

  LQH_DEBUG("SignalDroppedRep received for GSN " << originalGSN);

  switch (originalGSN) {
    case GSN_LQHKEYREQ: {
      jam();
      /* Get original signal data - unfortunately it may
       * have been truncated.  We must not read beyond
       * word # 22
       * We will notify the client that their LQHKEYREQ
       * failed
       */
      TcConnectionrecPtr tcConnectptr;
      tcConnectptr.i = RNIL;
      const LqhKeyReq *const truncatedLqhKeyReq =
          (LqhKeyReq *)&rep->originalData[0];

      earlyKeyReqAbort(signal, truncatedLqhKeyReq, ZGET_DATAREC_ERROR,
                       tcConnectptr);

      break;
    }
    case GSN_SCAN_FRAGREQ: {
      jam();
      /* Get original signal data - unfortunately it may
       * have been truncated.  We must not read beyond
       * word # 22
       * We will notify the client that their SCAN_FRAGREQ
       * failed
       */
      // TODO : Handle fragmented failure
      const ScanFragReq *const truncatedScanFragReq =
          (ScanFragReq *)&rep->originalData[0];
      const Uint32 senderData = truncatedScanFragReq->senderData;
      const Uint32 transid1 = truncatedScanFragReq->transId1;
      const Uint32 transid2 = truncatedScanFragReq->transId2;

      /* Send SCAN_FRAGREF back to the client */
      ScanFragRef *ref = (ScanFragRef *)&signal->theData[0];
      ref->senderData = senderData;
      ref->transId1 = transid1;
      ref->transId2 = transid2;
      ref->errorCode = ZGET_ATTRINBUF_ERROR;
      ref->senderRef = reference();

      sendSignal(signal->senderBlockRef(), GSN_SCAN_FRAGREF, signal,
                 ScanFragRef::SignalLength, JBB);
      break;
    }
    default:
      jam();
      /* Don't expect dropped signals for other GSNs,
       * default handling
       */
      SimulatedBlock::execSIGNAL_DROPPED_REP(signal);
  };

  return;
}

// Get size of interpreted program, in words.
static inline Uint32 getProgramWordCount(SegmentedSectionPtr attrInfo) {
  /*
    The second word of 'attrinfo' contains the length of the interpreted
    program, and the fifth contains the length of associated subroutines.
    (There should be a header of 5 length fields at the start of
    'attrinfo'.)
  */
  assert(attrInfo.sz >= 5);
  SectionReader aiReader(attrInfo, g_sectionSegmentPool);
  Uint32 header[5];
  const bool ok = aiReader.getWords(header, sizeof(header) / sizeof(header[0]));
  assert(ok);
  (void)ok;  // Prevent compiler warning.
  return header[1] + header[4];
}

/**
 * Concurrent fragment access module
 * ---------------------------------
 * The introduction of query threads means that we can have multiple
 * threads accessing the same fragment.
 *
 * We have 4 different basic access variants:
 * 1) Exclusive access
 *    - Currently used for ALTER TABLE operations, filling in data into
 *      the page map in DBTUP, commit and abort operations and finally
 *      Unlock operations.
 *      This mode of operations is normally used by the LDM thread, but
 *      can in rare cases happen as well for Query threads. This always
 *      happens through an lock upgrade.
 * 2) Write key access
 *    - Currently used for INSERT, UPDATE, REFRESH prepare operations
 *      Can run concurrently with shared key access.
 *      This mode is always executing in the LDM thread currently.
 * 3) Shared key access
 *    - Used for read operations, both Prepare and Commit and Dirty.
 *      Also used for Prepare of DELETE operations.
 *      Can run concurrently with Write key access and shared scan
 *      access
 * 4) Shared scan access
 *    - Used for scan operations, range scans in DBTUX and table scans
 *      in DBTUP. It can run concurrently with shared key accesses, but
 *      not with any other access type.
 *
 * When we need to perform metadata operations on all fragments in a
 * table, we will perform 1) above for all fragments that resides in
 * the LDM thread.
 *
 * Finally we have some dirty scan operations that require a short
 * term upgrade to an exclusive lock. This happens when we have holes
 * in the page map in DBTUP that needs to be filled in while scanning
 * the entire fragment in a TUP scan using a dirty read.
 *
 * In addition we have an array of mutexes on DBACC hash index and on the
 * tuple manager in DBTUP per fragment.
 *
 * The lock in DBACC ensures that the query thread will always see a
 * consistent view of the hash index, no changes of the hash index will
 * happen unless we are in exclusive access or we own the lock the proper
 * mutex in the array of mutexes. We use an array to minimise lock conflicts
 * on the hash index. The index is calculated based on the lowest part of
 * the hash value on the keys in the hash index.
 *
 * The DBACC lock is held during shrink and expand of the hash index.
 *
 * The lock in DBTUP enables us to install a new row version in a controlled
 * manner. Commits and aborts requires a consistent update of both DBACC and
 * DBTUP and is currently managed by exclusive locks that are required at
 * commit time.
 *
 * Normal read operations that are not dirty reads, thus shared read lock
 * access, exclusive read lock access and simple reads with either shared
 * or exclusive lock is always executed in the LDM thread, but there is not
 * much hindering it from being possible to execute in the query threads.
 * One thing that would have to be solved in that case is to ensure that
 * the list of locked operations in DBACC is a multi-threaded list, currently
 * it is a list of operation records from the owning LDM thread always. This is
 * similar to linked lists implemented for shared log parts in DBLQH.
 *
 * An important question to consider to make query threads behave correctly
 * in the presence of multiple threads executing at the fragment
 * concurrently is: When exactly does a READ COMMITTED key lookup or
 * READ COMMITTED scan (often referred to as dirty read for key lookups
 * and read committed scans for scans) see the result of a committed
 * change.
 *
 * The answer is the following:
 * If the READ COMMITTED comes from the same transaction as the one
 * currently executing it will see exactly the view that is currently
 * installed.
 *
 * This means that if the last operation on the row was a delete the
 * READ COMMITTED will not see the row after finding the current
 * operation we should look at.
 * If the operation is anything else we will use the current operations
 * copy row as the current row to read for the READ COMMITTED operation.
 *
 * If the READ COMMITTED operation doesn't belong to the transaction
 * currently updating the row it will look at the last committed result.
 * The exact timing of when the READ COMMITTED sees the commit differs a
 * bit, but for each change there is a well-defined point when the
 * READ COMMITTED operation sees the change.
 *
 * First of all for any update and insert the point is when the last
 * operation on the tuple is performed in DBTUP. This point in time can
 * be long after the commit of the operations on this row started. When
 * multiple operations are executed on a row there can be several operations
 * that commit on the row before the actual committed change is installed.
 *
 * The above is the case for all UPDATEs and all INSERTs. This time is
 * safe from a concurrency point of view since the fragment is used in
 * exclusive mode while performing a COMMIT in DBTUP of an UPDATE and an
 * INSERT.
 *
 * However for DELETEs the time of the visibility of the COMMIT differs.
 * If the row is a disk data row it is actually deleted already on the
 * very first COMMIT operation on the row. Actually it also becomes
 * visible even before the COMMIT has been fully performed, before the
 * retrieve of the disk data page has been performed.
 *
 * Thus again the DELETE becomes visible at a very distinct point in time
 * when we hold an exclusive lock on the row and thus there are no issues
 * from a concurrency point of view.
 *
 * Finally DELETE of a in-memory row. This also happens on the first operation
 * to commit. However it differs in that the DELETE is made visible in
 * DBACC rather than in DBTUP. This is what makes it necessary to use the
 * operation pointer to check whether the tuple exists in DBTUP. This causes
 * a concurrency problem for query threads.
 *
 * We can circumvent this problem by moving the discovery to DBTUP. By
 * setting the DELETE_WAIT flag on the tuple on the very first commit
 * also for rows without disk data AND by checking this flag in setup_read
 * we can avoid checking for tuple existence in DBACC for dirty reads.
 *
 * One can then of course wonder if the row always exists when the row
 * exists in DBACC. There is lots of logic that ensures that we don't
 * deallocate the row in DBTUP until the last operation that holds a lock
 * on the row is deleted.
 *
 * This approach will however still not work, this is due to a dependency
 * on the operation record to contain important information required in
 * retrieving an element from the hash index. This happens since locking a
 * row will remove some data from the hash element and place it in the
 * operation record. Thus without this operation record we cannot complete
 * the hash lookup.
 *
 * The actual commit of the delete still happens in a state when the
 * operation holds an exclusive lock on the fragment.
 *
 * So the only concurrency issue in DBACC is that we need to ensure that
 * the operation record that is inserted into the hash element is not
 * removed while we are executing the lookup in DBACC. The operation
 * owning the lock can be of any type, it can be a Key Read, a scan read
 * and an UPDATE, DELETE or an INSERT.
 *
 * Thus we need to protect dirty read operations from commits also of
 * Key Read and of Scan read operations. This is accomplished using the
 * above DBACC lock.
 *
 * Interestingly it is sufficient to protect against changes on the operation
 * record of the op_bits, the localdata and the release of this record.
 *
 * Thus the conclusion is the following:
 * 1) Introduce an array of mutexes on the DBACC fragment.
 * 2) Acquire this mutex while performing SHRINKCHECK2 and EXPANDCHECK2 to
 *    protect query threads from the impact of changes the hash index
 *    structure.
 * 3) Acquire this mutex in query threads while performing the lookup
 *    in execACCKEYREQ. It is not required to acquire this mutex for
 *    LDM threads performing dirty read or READ COMMITTED scans since
 *    only LDM threads so far can own locks in DBACC. If we are to allow
 *    the query thread to also execute generic READ queries, this needs to
 *    changed.
 *    The mutex can be released when execACCKEYREQ processing is done.
 *    If a tuple is found the row for it will remain in place since no
 *    DELETE operation have committed and will not while the query thread
 *    is active.
 * 4) Acquire the mutex when executing an ACC_COMMITREQ for a read operation
 *    This is only required when the operation is the lock owner. The
 *    mutex can be unlocked when the element header have been updated and
 *    the data is consistent for use by the query thread again.
 *
 * 5) Same for ACC_ABORTREQ as for ACC_COMMITREQ.
 *
 * 6) Update of the element header when inserting oneself as the lock owner
 *    has to be done in a secure manner and using a memory barrier to
 *    ensure that the query thread sees the correct view of the important
 *    parts of the operation record inserted as lock owner.
 *
 * 7) We need to lock the mutex in DBACC when inserting a new element, this
 *    will still have an invalid local key and thus be considered by dirty
 *    reads as a non-existing row.
 *
 * 8) We need to lock the mutex when inserts of new rows have completed and
 *    the local key is updated using the ACCMINUPDATE signal. After this
 *    update the inserted row is seen by dirty reads from the same transaction.
 *
 * There is no need to protect against releaseFragPage happening while
 * committing a read operation. This can only happen when all rows of the
 * page have been removed and thus are no longer reachable from any elements
 * in the hash index.
 *
 * With those changes it is sufficient to acquire a shared access for
 * reads and commits of reads. It is still required to acquire shared
 * access since the query thread potentially can request exclusive access.
 * There is no need to acquire exclusive access from DBACC when reorganising
 * the hash index using SHRINKCHECK2 and EXPANDCHECK2.
 *
 * The following changes are required to make it possible to handle
 * also generic read queries from query threads.
 * 1) Ensure that lock owner and lock operations can be operation records
 *    in query threads, thus ensuring that the list is multi-threaded.
 * 2) The DBACC fragment mutex must be held for all ACCKEYREQ accesses
 *    and also for all ACC_COMMITREQ accesses.
 *
 * Here is a description of the concurrent access issues to handle in
 * NDB data nodes:
 *
 * Notes on Concurrency of Query Threads
 * -------------------------------------
 *
 * DBACC Notes:
 * ------------
 * A query thread only access DBACC through the calls
 * execACCKEY_REQ, execACCKEY_ORD, and execACC_ABORTREQ.
 * for key operations and not at all for SCAN operations since
 * all scans from Query Threads are either TUX scans or
 * TUP scans. Dirty reads specifically doesn't call execACC_COMMITREQ.
 * Since Query threads do not use disk data currently there is also no
 * real-time breaks in handling DBACC since it is only used from
 * LQHKEYREQ.
 *
 * Query threads only performs Dirty Read operations and in DBACC these only
 * look at a few operation bits in execACC_ABORTREQ, so there is no problems
 * with concurrency for this signal. The operation record belongs to the
 * Query thread and is never placed in any lock queue such that other threads
 * can see or touch it. Thus execACC_ABORTREQ is safe from concurrency point of
 * view.
 *
 * Similarly execACCKEY_ORD only touch the operation record for Query threads.
 * Thus there is no concurrency problem at all here.
 *
 * So the concurrency problem in DBACC is only about the execACCKEYREQ call.
 *
 * Obviously execACCKEYREQ uses the entire data structure of the hash index
 * in DBACC for a fragment. For dirty reads DBACC is used to enter with a
 * primary key and coming back with a logical pointer to the data in DBTUP.
 * Dirty reads do not enter any lock queues and thus have no interaction with
 * other threads through locking operation.
 *
 * So there are two concurrency problems for Query threads
 * 1) The hash index cannot be updated during the time that the query thread
 * is using.
 * The hash index calculates a logical page id and container id using a hash
 * of the primary key.
 * The map from the hash value to the actual container address is changed
 * through calls to execSHRINKCHECK2 and execEXPANDCHECK2. Both of these calls
 * are made in their own signals and are never executed as part of any user
 * operation.
 *
 * Clearly both this signals need to execute such that no query thread are
 * executing in the fragment at the same time.
 *
 * A read and a update in execACCKEYREQ will only read the hash index and
 * manipulate the lock queue.
 *
 * An insert will insert a new entry element into the hash index. This is
 * done at the level of the container. So one could construe mutexes that
 * protect only down to the container level here. Obviously the memory overhead
 * would make this not such a good idea. But it is easy enough to construe
 * a mutex scheme where the logical page id and container id is mapped to a
 * mutex thus protecting sufficient amount of detail to ensure that any
 * other thread performing actions in DBACC will not be bothered.
 *
 * In the first implementation inserts will be performed with an exclusive
 * lock on the fragment and will thus not constitute any issue for query
 * threads since they cannot execute when someone owns an exclusive lock
 * on the fragment.
 *
 * A delete operation actually doesn't perform anything other than the map
 * that is also done by reads and updates. The actual delete happens later
 * as part of the commit process.
 *
 * Thus delete operations only interact through lock queues as well.
 *
 * So how does locking affect Query threads?
 *
 * The lock queues only interact with query threads in one distinct manner.
 * The query thread will see if there is a lock owner and needs to
 * check 2 variables in the lock owner operation to ensure that the row
 * isn't deleted.
 *
 * The reason is that the row must remain in DBACC until all locked operations
 * have been handled before the row can be actually deleted.
 *
 * The following check is done by Query threads in DBACC:
 *
 * if (! (lockOwnerPtr.p->m_op_bits & Operationrec::OP_ELEMENT_DISAPPEARED) &&
 *    ! lockOwnerPtr.p->localdata.isInvalid())
 *
 * Thus there are three things that effect a Query thread.
 * 1) Any changes to the m_op_bits that sets or resets the bit
 * OP_ELEMENT_DISAPPEARED must be protected and not done while query threads
 * can access the operation record.
 * 2) Any changes that sets the localdata variable to invalid must be
 *    protected.
 * 3) Any release of the operation record that makes the whole record
 *    invalid must be protected.
 *
 * Executing all reads, updates, deletes and inserts as exclusive operations
 * safeguards these things.
 *
 * If e.g. we want to be able to execute concurrent read operations with the
 * query thread that could use locking we need to follow the following
 * principle.
 *
 * 1) Before writing the operation record into the hash element we need to
 * ensure that the above variables are properly set followed by a memory
 * barrier.
 * 2) Any change to a new element pointer must follow the same principle
 * and ensure that things are safe before issuing a memory barrier.
 *
 * If any of the above 3 conditions are to be changed such as could be the
 * case at commit of a delete or a commit that removes the lock owner and
 * releases the operation must either be done with exclusive fragment
 * access or with a more fine-grained lock on the ACC fragment protecting
 * the part where the change is made.
 *
 * DBTUP Notes:
 * ------------
 * DBTUP is very similar to DBACC in its concurrency.
 * Query threads use DBTUP through execTUPKEYREQ, execTUP_ABORTREQ.
 * Query threads also use DBTUP in performing table scans through
 * execNEXT_SCANREQ and a number of other functions in DBTUP.
 *
 * When accessing a row one starts with a row id. The row id consists
 * of a logical page id and a logical page index. DBTUP uses a page
 * map implemented using DynArr256 objects to map the logical page
 * id and logical page index to a row address.
 *
 * The DynArr256 map consists of some state information and an i-value
 * of a 32 kB page where the fixed size part of the row is stored.
 * There can also be variable sized parts and disk data parts. These
 * parts are referenced directly with physical page references that
 * uses i-values that can be used by the global NDB memory manager to
 * get the physical address of those parts.
 *
 * Normally all pages are in the DynArr256 map, but after deleting full
 * pages and a restart one can get holes in the DynArr256 map. The
 * query thread scan can hit one of those holes and be forced to fill
 * in the hole with a new DynArr256 object. This is a rare case where
 * Query threads have to acquire exclusive access to the fragment.
 *
 * Adding information to the DynArr256 map happens when a new page is
 * added or a page is deleted. This only happens when an insert or a
 * delete is committed. Updates can change the record layout by
 * changing the variable sized part, but it will not move around the
 * fixed size part.
 *
 * Thus any such changes to the DynArr256 map requires exclusive access
 * to the fragment. This exclusivity covers all access types.
 *
 * DBTUP also have an operation record linked into the row. This operation
 * record points to ongoing updates. A query thread needs to be able to
 * read this list since it might come from the same transaction as one of
 * the updaters. If a query thread comes from the same transaction as a
 * writer it must use the list of operation records on the fixed size part
 * of the row to find which row to use in the read. It can mean that I should
 * use a copy row of one of those operations (even from a query thread).
 *
 * Thus updating this list must be protected from access by query threads.
 * This list is updated by update, insert and delete operations and also
 * by their commit. However read operations are not affecting this list.
 * Thus read operations do not interact with dirty read operations in
 * the query thread. This also means that all scan operations that access
 * rows in DBTUP from a query thread has no interaction with the reads
 * performing locks (shared read access, exclusive read access and
 * simple reads). Actually reads don't even perform TUP_COMMITREQ and
 * only initialise the operation record in TUP_ABORTREQ.
 *
 * Thus we can conclude that if read operations and commits of read operations
 * protect against DBACC access from query thread that will be sufficient.
 *
 * Higher levels of concurrency can be achieved. these higher levels of
 * concurrency requires splitting the rows into segments that are
 * protected by a mutex. The nice property with the NDB implementation
 * is that we handle operations on different rows individually and thus
 * we can protect the part that the current operation is currently dealing
 * with.
 *
 * One problem to consider in such an implementation is how to handle DBACC
 * and DBTUP since they require different partitions and thus one has to
 * consider how the DBACC and the DBTUP mutex would cooperate.
 *
 * DBTUX Notes:
 * ------------
 * Making it possible to execute writes in parallel on the TUX index is a
 * hard nut to solve. This requires some variant where we lock in a tree.
 * There are solutions to this problem, but are distinctly more complex
 * compared to the above problems in DBACC and DBTUP.
 *
 * Given that both UPDATE and INSERT insert into the TUX index in the
 * prepare phase it means that it is difficult to concurrently scan
 * the TUX index while performing the LQHKEYREQ signals. So it is not
 * even enough with exclusive access during the commit phase.
 *
 * Interestingly we also have a special problem where dirty scans interact
 * with each other. A scan on a TUX index needs to insert the operation
 * into the TUX tree before a real-time break, the tree can be modified
 * before the next real-time break happens. Thus we need to move around
 * this operation record based on changes in the TUX index.
 *
 * This is solved with a mutex on the ordered index fragment, this is only
 * held while performing insert and remove of those scan records in the
 * TUX nodes.
 *
 * DBTUX improvement ideas
 * -----------------------
 * One potential solution to the problem of requiring exclusive access
 * during UPDATE and INSERTs is the following:
 * The idea is to avoid inserting the new entries into the TUX index in
 * the prepare phase. This has no consequence for scanners that are not
 * part of updating transactions since they are not interested in looking
 * at data from ongoing transactions. However if a scan comes around that
 * is part of an updating transaction, it will first have to merge all the
 * UPDATEs and INSERTs performed by this transaction into the TUX index
 * before it executes its work.
 *
 * One method to handle this efficiently is to track updates for a
 * transaction, when they reach a certain level we acquire exclusive
 * access to the fragment and merge all entries into the TUX index.
 * The same mechanism is used by scanners that are part of updating
 * transactions. When the transaction commits, the commit operation
 * will have to know whether the entry has been added to the index yet
 * or not.
 *
 * As an example one can look at the transactions in the TPC-C benchmark.
 * In this transactions the stock level and order status transactions are
 * pure readers, the new order transaction performs a lot of updates, but
 * it doesn't perform any range scans on updated tables. The delivery
 * transaction does however use range scans on updated tables as a
 * technique to handle all deliveries of all new orders for a customer.
 * Neither the payment transaction uses any range scans on updated tables.
 *
 * Thus it is evident that this approach might be an interesting step to
 * decrease the requirement on exclusive access to a fragment.
 *
 * DELETEs do not perform any changes to the index in the prepare phase,
 * this happens in the commit phase. But even DELETEs can interact with
 * READ COMMITTED scans and key lookups if they come from the same
 * transaction. If a row is not currently being updated (m_operation_ptr_i
 * in the row header is RNIL), then the row can be read without risk of
 * interacting with changes happening in the prepare phase.
 * If the row is being processed with updates currently AND if we have not
 * exclusive access during UPDATE, INSERT and DELETE operations, then we
 * need to use a TUP fragment mutex. Normally this mutex is only acquired
 * by the updater and thus should not cause any contention, but when a
 * scanner needs to ensure that the list of active operations is stable
 * it needs to acquire this mutex, it should be sufficient to do this
 * when m_operation_ptr_i != RNIL. This mutex can easily be divided on
 * a simple hash on logical page id to ensure that multiple operations
 * can happen in parallel on a fragment.
 */

/**
 * We always print mutex statistics, but only by setting the
 * DEBUG_MUTEX_STATS do we print it unconditionally, otherwise it is
 * printed only if activating debug printouts from command line.
 */
//#define DEBUG_MUTEX_STATS 1
#ifdef DEBUG_MUTEX_STATS
#define DEB_PRINT_MUTEX_STATS(arglist) \
  do {                                 \
    g_eventLogger->info arglist;       \
  } while (0)
#else
#define DEB_PRINT_MUTEX_STATS(arglist) \
  do {                                 \
    g_eventLogger->debug arglist;      \
  } while (0)
#endif

void Dblqh::print_fragment_mutex_stats(Signal *signal) {
  DEB_PRINT_MUTEX_STATS(
      ("%s[%u]SCAN access: %u"
       ", SCAN contention: %u"
       ", SCAN spintime: %llu"
       ", SCAN spinloops: %llu"
       ", SCAN cond waits: %u",
       m_is_query_block ? "query" : "ldm", instance(), m_scan_frag_access,
       m_scan_frag_access_contended, m_scan_frag_access_spintime,
       m_scan_frag_access_spinloops, m_scan_frag_access_cond_waits));
  DEB_PRINT_MUTEX_STATS(
      ("%s[%u]RKey access: %u"
       ", RKey contention: %u"
       ", RKey spintime: %llu"
       ", RKey spinloops: %llu"
       ", RKey cond waits: %u",
       m_is_query_block ? "query" : "ldm", instance(), m_read_key_frag_access,
       m_read_key_frag_access_contended, m_read_key_frag_access_spintime,
       m_read_key_frag_access_spinloops, m_read_key_frag_access_cond_waits));
  DEB_PRINT_MUTEX_STATS(
      ("%s[%u]WKey access: %u"
       ", WKey contention: %u"
       ", WKey spintime: %llu"
       ", WKey spinloops: %llu"
       ", WKey cond waits: %u",
       m_is_query_block ? "query" : "ldm", instance(), m_write_key_frag_access,
       m_write_key_frag_access_contended, m_write_key_frag_access_spintime,
       m_write_key_frag_access_spinloops, m_write_key_frag_access_cond_waits));
  DEB_PRINT_MUTEX_STATS(
      ("%s[%u]EXCL access: %u"
       ", EXCL contention: %u"
       ", UP->EXCL access: %u"
       ", EXCL spintime: %llu"
       ", EXCL spinloops: %llu"
       ", EXCL cond waits: %u",
       m_is_query_block ? "query" : "ldm", instance(), m_exclusive_frag_access,
       m_exclusive_frag_access_contended, m_upgrade_frag_access,
       m_exclusive_frag_access_spintime, m_exclusive_frag_access_spinloops,
       m_exclusive_frag_access_cond_waits));
  send_print_mutex_stats(signal);
}

void Dblqh::send_print_mutex_stats(Signal *signal) {
  m_scan_frag_access = 0;
  m_scan_frag_access_contended = 0;
  m_scan_frag_access_spintime = 0;
  m_scan_frag_access_spinloops = 0;
  m_scan_frag_access_cond_waits = 0;
  m_read_key_frag_access = 0;
  m_read_key_frag_access_contended = 0;
  m_read_key_frag_access_spintime = 0;
  m_read_key_frag_access_spinloops = 0;
  m_read_key_frag_access_cond_waits = 0;
  m_write_key_frag_access = 0;
  m_write_key_frag_access_contended = 0;
  m_write_key_frag_access_spintime = 0;
  m_write_key_frag_access_spinloops = 0;
  m_write_key_frag_access_cond_waits = 0;
  m_exclusive_frag_access = 0;
  m_exclusive_frag_access_contended = 0;
  m_exclusive_frag_access_spintime = 0;
  m_exclusive_frag_access_spinloops = 0;
  m_exclusive_frag_access_cond_waits = 0;
  m_upgrade_frag_access = 0;

  signal->theData[0] = ZPRINT_MUTEX_STATS;
  sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 1000, 1);
}

/**
 * Fragment lock logic
 * -------------------
 * We use a sort of Read-Write Fragment lock to handle concurrent access to
 * a fragment.
 *
 * There are 4 starting points:
 * 1) Prepare key operation
 * 2) Scan operation
 * 3) Commit operation
 * 4) Abort operation
 *
 * We support 4 different lock modes.
 * 1) Exclusive access, only one thread can access the fragment
 * 2) Write Key access, no scans are allowed access to the fragment
 * 3) Read Key access, no exclusive access are allowed until done
 * 4) Scan access, no exclusive or write key access allowed until done
 *
 * As mentioned below we can set some extra variables when using
 * Read Key access. So actually Read Key access has 3 variants.
 *
 * 1) Read Key plain
 * 2) Read Key prepared for Write Key upgrade
 * 3) Read Key prepared for Exclusive upgrade (used by REFRESH)
 *
 * 1) is used by READ and DELETE
 * 2) is used by WRITE, INSERT, UPDATE and UNLOCK
 * 3) is used by REFRESH operations
 *
 * Prepare operations
 * ------------------
 * A prepare operation always calls acquire_frag_prepare_key_access.
 * There are a number of variants here:
 * - Write operation (INSERT, UPDATE)
 * - Read/Delete operation
 * - Refresh operation
 *
 * A write operation will normally make writes to the TUX index. Thus when
 * starting a prepare write operation that will normally require an upgrade
 * to Write Key access we will set m_cond_write_key_waiters and
 * m_spin_write_key_waiters to 1. This indicates to query threads that it
 * is ok to start Read Key operations, but it isn't ok to start new scan
 * operations. However we will only set the Read Key lock. Later when we
 * require the Write Key lock we will have to wait for a shorter time
 * since we blocked scanners from starting for a while.
 *
 * When starting a Refresh operation we will later have to acquire exclusive
 * access to the fragment. We prepare for this by setting
 * m_cond_exclusive_waiters, m_spin_exclusive_waiters to 1. This means that
 * no new scans and read key operations can start, but those ongoing will
 * run in parallel until we upgrade to exclusive lock. At upgrade time we
 * will eventually have to wait for the other operations to complete, but
 * we save some wait time by preparing for the upgrade early on.
 *
 * When starting read key and delete operations it is enough to set the
 * Read Key lock. If this happens in the LDM thread it won't block any
 * query threads at all. If a query thread sets Read Key locks it means
 * that the LDM thread will have to wait for exclusive access but it can
 * proceed with write key access.
 *
 * Query threads can only use Read Key operations of type 1) above since
 * it only performs Read operations.
 *
 * Commit operations
 * -----------------
 * Commit operations have two phases, the first phase is to write the
 * TUX indexes and the second phase is to perform the actual update
 * where the new row version is installed.
 *
 * This starts with acquiring Write Key access to the fragment, but also
 * preparing for exclusive access to the fragment similar to how the
 * Refresh operation above prepared for exclusive access.
 *
 * The Write Key lock is only acquired if the table has TUX indexes,
 * otherwise one goes directly to exclusive access. Since it is the
 * commit phase, no aborts or other code paths can happen here.
 *
 * Scan operations
 * ---------------
 * Scan operations need to ensure that no one is updating the TUX indexes
 * while the scan is ongoing. This is true for range scans. However
 * full table scans that are using TUP scans will not require this.
 * TUP scans can run in parallel with everything except exclusive access.
 *
 * Range scans thus need to avoid running concurrently with Write Key
 * locks and Exclusive locks. These locks also have higher priority than
 * the scans. To implement this we will stop a scan after completing the
 * row currently being scanned.
 *
 * One complication is that TUP scans might have to upgrade to exclusive
 * access in some rare situations. Other than that scans require no upgrade
 * to higher lock levels.
 *
 * Abort operations
 * ----------------
 * Aborts can either happen as an external operation decided by DBTC or by
 * application code. In this case we will immediately set the lock level
 * to Read Key for Reads and Deletes. All other aborts will use the
 * exclusive access. It is likely that one can analyse things a bit more
 * to find more concurrency here.
 *
 * The other case is when an operation is aborted, in this case we will
 * keep the lock level for Reads and Delete operations and we will upgrade
 * to exclusive access level for all other operations. After this upgrade
 * we will release the fragment lock.
 *
 * Upgrade lock situations
 * -----------------------
 * It would be great if the row was undisturbed by Prepare operations and
 * that the row is only changed at commit time. This is however not the
 * case. There are cases when we perform OPTIMISE TABLE and there are
 * cases when the variable sized part of the row has to grow where we
 * actually will change the row while running the Prepare operation.
 * To accommodate for this we have the ability to upgrade to exclusive
 * locks for a short time. This is not an expected situation normally
 * and thus we perform no special preparation for this event.
 *
 * Another upgrade case is when we have to write into the fragment page
 * map in DBTUP. This can happen even to scanners.
 *
 * Both of those upgrade will always be immediately followed by a
 * downgrade.
 *
 * Another situation for an upgrade is when an error occurs and we
 * start an abort. In this case we don't perform any downgrade,
 * we will continue in exclusive mode until the lock is released.
 *
 * Finally we have upgrade to write key which is an expected operation.
 * This operation is always followed by either an upgrade to exclusive
 * due to an abort, or followed by a release of the lock. So no
 * downgrade will ever happen here.
 */

#ifdef DEBUG_FRAGMENT_LOCK
#define DEB_FRAGMENT_LOCK(frag) debug_fragment_lock(frag, __LINE__)

#else
#define DEB_FRAGMENT_LOCK(frag)
#endif

#ifdef DEBUG_FRAGMENT_LOCK
void Dblqh::debug_fragment_lock(Fragrecord *fragPtrP, Uint32 line) {
  Uint32 instance_no = instance();
  if (m_is_query_block) {
    instance_no += getNumLDMInstances();
  }
  fragPtrP->lock_line_index = (fragPtrP->lock_line_index + 1) & LOCK_LINE_MASK;
  fragPtrP->lock_line[fragPtrP->lock_line_index] = Uint16(line);
  fragPtrP->lock_line_index = (fragPtrP->lock_line_index + 1) & LOCK_LINE_MASK;
  fragPtrP->lock_line[fragPtrP->lock_line_index] = Uint16(instance_no);
}
#endif

Dblqh::Fragrecord *Dblqh::get_fragptr(Uint32 tableId, Uint32 fragId) {
  TablerecPtr tabPtr;
  FragrecordPtr fragPtr;
  FragrecordPtr fragPrimPtr;
  ndbrequire(getTableFragmentrec(tableId, fragId, tabPtr, fragPtr));
  return fragPtr.p;
}

void Dblqh::lock_table_exclusive(Tablerec *tablePtrP) {
  TablerecPtr tabPtr;
  if (tablePtrP->primaryTableId != RNIL) {
    jam();
    tabPtr.i = tablePtrP->primaryTableId;
    ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  } else {
    jam();
    tabPtr.p = tablePtrP;
  }
  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabPtr.p->fragid); i++) {
    if (tabPtr.p->fragrec[i] != RNIL) {
      FragrecordPtr fragPtr;
      fragPtr.i = tabPtr.p->fragrec[i];
      c_fragment_pool.getPtr(fragPtr);
      handle_acquire_exclusive_frag_access(fragPtr.p, false);
    }
  }
}

void Dblqh::unlock_table_exclusive(Tablerec *tablePtrP) {
  TablerecPtr tabPtr;
  if (tablePtrP->primaryTableId != RNIL) {
    jam();
    tabPtr.i = tablePtrP->primaryTableId;
    ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  } else {
    jam();
    tabPtr.p = tablePtrP;
  }
  for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabPtr.p->fragid); i++) {
    if (tabPtr.p->fragrec[i] != RNIL) {
      FragrecordPtr fragPtr;
      fragPtr.i = tabPtr.p->fragrec[i];
      c_fragment_pool.getPtr(fragPtr);
      handle_release_exclusive_frag_access(fragPtr.p);
    }
  }
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::downgrade_exclusive_to_scan() {
  Fragrecord *fragPtrP;
  if (DictTabInfo::isOrderedIndex(fragptr.p->tableType)) {
    fragPtrP = prim_tab_fragptr.p;
  } else {
    fragPtrP = fragptr.p;
  }
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbassert(!m_is_query_block);
  ndbrequire(fragPtrP->m_exclusive_locked == true);
  ndbrequire(fragPtrP->m_cond_exclusive_waiters == 0);
  ndbrequire(fragPtrP->m_cond_write_key_waiters == 0);
  ndbrequire(fragPtrP->m_write_key_locked == false);
  ndbrequire(fragPtrP->m_concurrent_scan_count == 0);
  ndbrequire(fragPtrP->m_concurrent_read_key_count == 0);
  fragPtrP->m_exclusive_locked = false;
  fragPtrP->m_concurrent_scan_count = 1;
  if (fragPtrP->m_cond_read_waiters > 0) {
    jam();
    ndbrequire(is_scan_condition_ready(fragPtrP));
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Broadcast(&fragPtrP->frag_read_cond);
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
}

void Dblqh::downgrade_exclusive_to_write_key(Fragrecord *fragPtrP) {
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbrequire(fragPtrP->m_exclusive_locked == true);
  ndbrequire(fragPtrP->m_write_key_locked == false);
  fragPtrP->m_exclusive_locked = false;
  fragPtrP->m_write_key_locked = true;
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
}

void Dblqh::downgrade_exclusive_to_read_key(Fragrecord *fragPtrP) {
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbrequire(fragPtrP->m_exclusive_locked == true);
  fragPtrP->m_exclusive_locked = false;
  fragPtrP->m_concurrent_read_key_count++;
  if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_WK_MODE) {
    fragPtrP->m_cond_write_key_waiters = 1;
    fragPtrP->m_spin_write_key_waiters = 1;
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_REFRESH_MODE) {
    fragPtrP->m_cond_exclusive_waiters = 1;
    fragPtrP->m_spin_exclusive_waiters = 1;
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
}

void Dblqh::handle_acquire_scan_frag_access(Fragrecord *fragPtrP) {
  Uint64 elapsed = 0;
  Uint64 spin_loops = 0;
  Uint32 num_spins = 0;
  bool first = true;
  NDB_TICKS now;
  NDB_TICKS start_time;
  (void)now;
  (void)start_time;
  (void)num_spins;
  m_scan_frag_access++;
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  while (!is_scan_condition_ready(fragPtrP)) {
    if (unlikely(!first)) {
      jam();
      goto scan_cond_wait;
    }
    m_scan_frag_access_contended++;
#ifdef NDB_HAVE_CPU_PAUSE
    NdbMutex_Unlock(&fragPtrP->frag_mutex);
    start_time = NdbTick_getCurrentTicks();
    NdbSpin();
    /**
     * We wait a bit extra before moving into the spin loop.
     * Thus first wait is a bit longer.
     */
    do {
      /**
       * Inside of the spin loop we wake up about once per
       * microsecond. We extend the num_spins-wait if we
       * had to wait multiple times.
       */
      spin_loops++;
      num_spins++;
      for (Uint32 i = 0; i < num_spins; i++) {
        NdbSpin();  // ~1us
      }
      /**
       * Acquire mutex in a less active manner to ensure that
       * the waiters for exclusive access have a bit of
       * precedence to get the mutex. This also protects the
       * mutex cache line from too many busy waiters.
       */
      NdbMutex_Lock(&fragPtrP->frag_mutex);
      if (is_scan_condition_ready(fragPtrP)) {
        goto got_lock;
      }
      NdbMutex_Unlock(&fragPtrP->frag_mutex);
      now = NdbTick_getCurrentTicks();
      elapsed = NdbTick_Elapsed(start_time, now).microSec();
      if (elapsed > LOCK_READ_SPIN_TIME) {
        break;
      }
    } while (true);
    /**
     * We have spun for more than 30 microseconds and still
     * haven't got access. We will go to sleep and wait to
     * be woken up instead.
     *
     * Before we get into the condition wait we have to
     * verify that the wait condition is still true. If not
     * we can break out of loop and continue with our read.
     */
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    if (is_scan_condition_ready(fragPtrP)) {
      goto got_lock;
    }
#endif
  scan_cond_wait:
    jam();
    first = false;
    fragPtrP->m_cond_read_waiters++;
    m_scan_frag_access_cond_waits++;
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Wait(&fragPtrP->frag_read_cond, &fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    jam();
    fragPtrP->m_cond_read_waiters--;
  }
#ifdef NDB_HAVE_CPU_PAUSE
got_lock:
#endif
  fragPtrP->m_concurrent_scan_count++;
  DEB_FRAGMENT_LOCK(fragPtrP);
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_LOCKED_IN_SCAN_MODE;
  m_scan_frag_access_spintime += elapsed;
  m_scan_frag_access_spinloops += spin_loops;
}

void Dblqh::handle_acquire_read_key_frag_access(Fragrecord *fragPtrP,
                                                bool hold_lock,
                                                bool check_exclusive_waiters) {
  Uint64 elapsed = 0;
  Uint64 spin_loops = 0;
  Uint32 num_spins = 0;
  bool first = true;
  NDB_TICKS now;
  NDB_TICKS start_time;
  (void)now;
  (void)start_time;
  (void)num_spins;
  m_read_key_frag_access++;
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  if (!hold_lock) {
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
  }
  while (!is_read_key_condition_ready(fragPtrP, check_exclusive_waiters)) {
    if (unlikely(!first)) {
      goto read_key_cond_wait;
    }
    m_read_key_frag_access_contended++;
#ifdef NDB_HAVE_CPU_PAUSE
    NdbMutex_Unlock(&fragPtrP->frag_mutex);
    start_time = NdbTick_getCurrentTicks();
    NdbSpin();
    do {
      spin_loops++;
      num_spins++;
      for (Uint32 i = 0; i < num_spins; i++) {
        NdbSpin();
      }
      NdbMutex_Lock(&fragPtrP->frag_mutex);
      if (is_read_key_condition_ready(fragPtrP, check_exclusive_waiters)) {
        goto got_lock;
      }
      NdbMutex_Unlock(&fragPtrP->frag_mutex);
      now = NdbTick_getCurrentTicks();
      elapsed = NdbTick_Elapsed(start_time, now).microSec();
      if (elapsed > LOCK_READ_SPIN_TIME) {
        break;
      }
    } while (true);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    if (is_read_key_condition_ready(fragPtrP, check_exclusive_waiters)) {
      goto got_lock;
    }
#endif
  read_key_cond_wait:
    jam();
    first = false;
    fragPtrP->m_cond_read_waiters++;
    m_read_key_frag_access_cond_waits++;
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Wait(&fragPtrP->frag_read_cond, &fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    fragPtrP->m_cond_read_waiters--;
    jam();
  }
#ifdef NDB_HAVE_CPU_PAUSE
got_lock:
#endif
  fragPtrP->m_concurrent_read_key_count++;
  DEB_FRAGMENT_LOCK(fragPtrP);
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_read_key_frag_access_spintime += elapsed;
  m_read_key_frag_access_spinloops += spin_loops;
}

void Dblqh::handle_acquire_write_key_frag_access(Fragrecord *fragPtrP,
                                                 bool hold_lock) {
  m_write_key_frag_access++;
  Uint64 elapsed = 0;
  Uint64 spin_loops = 0;
  Uint32 num_spins = 0;
  NDB_TICKS now;
  NDB_TICKS start_time;
  (void)now;
  (void)start_time;
  (void)num_spins;
  bool first = true;
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  if (!hold_lock) {
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
  }
  /**
   * Both write key and exclusive is restricted to happen from owning
   * LDM thread, thus cannot happen concurrently.
   */
  while (!is_write_key_condition_ready(fragPtrP)) {
    if (unlikely(!first)) {
      goto cond_write_key_wait;
    }
    m_write_key_frag_access_contended++;
    ndbrequire(fragPtrP->m_cond_write_key_waiters == 0);
    fragPtrP->m_cond_write_key_waiters = 1;
#ifdef NDB_HAVE_CPU_PAUSE
    ndbrequire(fragPtrP->m_spin_write_key_waiters == 0);
    fragPtrP->m_spin_write_key_waiters = 1;
    NdbMutex_Unlock(&fragPtrP->frag_mutex);
    start_time = NdbTick_getCurrentTicks();
    NdbSpin();
    do {
      num_spins++;
      spin_loops++;
      for (Uint32 i = 0; i < num_spins; i++) {
        NdbSpin();
      }
      NdbMutex_Lock(&fragPtrP->frag_mutex);
      if (is_write_key_condition_ready(fragPtrP)) {
        goto got_lock;
      }
      NdbMutex_Unlock(&fragPtrP->frag_mutex);
      now = NdbTick_getCurrentTicks();
      elapsed = NdbTick_Elapsed(start_time, now).microSec();
      if (elapsed > LOCK_WRITE_SPIN_TIME) {
        /**
         * Time out is discovered outside of mutex, before
         * diving into the condition wait we must recheck
         * the condition to enter the condition again.
         * After acquiring the mutex again of course.
         */
        NdbMutex_Lock(&fragPtrP->frag_mutex);
        if (is_write_key_condition_ready(fragPtrP)) {
          goto got_lock;
        }
        break;
      }
    } while (true);
#else
    m_write_key_frag_access_contended++;
#endif
  cond_write_key_wait:
    /* Holding the mutex when arriving here. */
    jam();
    fragPtrP->m_spin_write_key_waiters = 0;
    m_write_key_frag_access_cond_waits++;
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Wait(&fragPtrP->frag_write_cond, &fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    jam();
    first = false;
  }
#ifdef NDB_HAVE_CPU_PAUSE
got_lock:
#endif
  fragPtrP->m_spin_write_key_waiters = 0;
  fragPtrP->m_cond_write_key_waiters = 0;
  ndbrequire(fragPtrP->m_write_key_locked == false);
  fragPtrP->m_write_key_locked = true;
  DEB_FRAGMENT_LOCK(fragPtrP);
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_LOCKED_IN_WRITE_KEY_MODE;
  m_write_key_frag_access_spintime += elapsed;
  m_write_key_frag_access_spinloops += spin_loops;
}

void Dblqh::handle_acquire_exclusive_frag_access(Fragrecord *fragPtrP,
                                                 bool hold_lock) {
  m_exclusive_frag_access++;
  Uint64 elapsed = 0;
  Uint64 spin_loops = 0;
  Uint32 num_spins = 0;
  NDB_TICKS now;
  NDB_TICKS start_time;
  (void)now;
  (void)start_time;
  (void)num_spins;
  bool first = true;
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  if (!hold_lock) {
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
  }
  while (!is_exclusive_condition_ready(fragPtrP)) {
    /**
     * There are readers currently busy with the fragment. We need to
     * wait for those to complete their task before we can get our
     * exclusive access to the fragment. Given that other readers are
     * blocked while we are waiting and while we are performing our
     * task we avoid going to sleep to start with, we will spin for
     * about 50 microseconds before we go to sleep. This should decrease
     * the lock hold times significantly.
     *
     * We only perform spinning on platforms that actually support
     * spinning in some fashion.
     */
    if (unlikely(!first)) {
      goto cond_exclusive_wait;
    }
    m_exclusive_frag_access_contended++;
    ndbrequire(fragPtrP->m_cond_exclusive_waiters == 0);
    fragPtrP->m_cond_exclusive_waiters = 1;
#ifdef NDB_HAVE_CPU_PAUSE
    ndbrequire(fragPtrP->m_spin_exclusive_waiters == 0);
    fragPtrP->m_spin_exclusive_waiters = 1;
    NdbMutex_Unlock(&fragPtrP->frag_mutex);
    start_time = NdbTick_getCurrentTicks();
    NdbSpin();
    do {
      /**
       * Initially we spin two 1us-cycles to give the readers a fair chance
       * to complete their task. If this was not sufficient, we will extend
       * spin_loops for the next wait.
       *
       * When we come here we will not hold the mutex.
       */
      num_spins++;
      spin_loops++;
      for (Uint32 i = 0; i < num_spins; i++) {
        NdbSpin();  // ~1us
      }
      NdbMutex_Lock(&fragPtrP->frag_mutex);
      if (is_exclusive_condition_ready(fragPtrP)) {
        goto got_lock;
      }
      NdbMutex_Unlock(&fragPtrP->frag_mutex);
      now = NdbTick_getCurrentTicks();
      elapsed = NdbTick_Elapsed(start_time, now).microSec();
      if (elapsed > LOCK_WRITE_SPIN_TIME) {
        /**
         * Time out is discovered outside of mutex, before
         * diving into the condition wait we must recheck
         * the condition to enter the condition again.
         * After acquiring the mutex again of course.
         */
        NdbMutex_Lock(&fragPtrP->frag_mutex);
        if (is_exclusive_condition_ready(fragPtrP)) {
          goto got_lock;
        }
        break;
      }
    } while (true);
#else
    m_exclusive_frag_access_contended++;
#endif
  cond_exclusive_wait:
    /* Holding the mutex when arriving here. */
    jam();
    fragPtrP->m_spin_exclusive_waiters = 0;
    m_exclusive_frag_access_cond_waits++;
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Wait(&fragPtrP->frag_write_cond, &fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    jam();
    first = false;
  }
#ifdef NDB_HAVE_CPU_PAUSE
got_lock:
#endif
  ndbrequire(fragPtrP->m_exclusive_locked == false);
  fragPtrP->m_exclusive_locked = true;
  fragPtrP->m_cond_exclusive_waiters = 0;
  fragPtrP->m_spin_exclusive_waiters = 0;
  DEB_FRAGMENT_LOCK(fragPtrP);
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_LOCKED_IN_EXCLUSIVE_MODE;
  m_exclusive_frag_access_spintime += elapsed;
  m_exclusive_frag_access_spinloops += spin_loops;
}

void Dblqh::handle_acquire_frag_abort_access(Fragrecord *fragPtrP,
                                             TcConnectionrec *regTcPtr) {
  if (m_fragment_lock_status == FRAGMENT_UNLOCKED) {
    if (is_read_key_frag_access(regTcPtr) ||
        is_read_key_delete_frag_access(regTcPtr)) {
      jamDebug();
      handle_acquire_read_key_frag_access(fragPtrP, false, true);
      m_fragment_lock_status = FRAGMENT_LOCKED_IN_READ_KEY_MODE;
    } else {
      jamDebug();
      handle_acquire_exclusive_frag_access(fragPtrP, false);
    }
  } else {
    jamDebug();
    if (is_read_key_frag_access(regTcPtr) ||
        is_read_key_delete_frag_access(regTcPtr)) {
      /**
       * No need to upgrade when aborting reads and deletes
       * Already at sufficient level
       */
      if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_READ_KEY_MODE &&
          is_read_key_delete_frag_access(regTcPtr)) {
        /**
         * A Prepare DELETE operation requires an abort.
         * This can happen immediately due to row is missing in DBACC, in
         * this case it is ok to proceed in shared mode.
         * It can happen due to a failure within execTUPKEYREQ, in this
         * the failure is immediately taken care of and the call to
         * execTUP_ABORTREQ will find an operation of type ZREAD and will
         * discard the abort before proceeding to do anything useful.
         * Thus we can continue in SHARE mode also for this abort reason.
         *
         * We verify that this is the case by checking that the operation
         * in DBTUP is a ZREAD and thus will not cause any real ABORT action
         * to happen here.
         */
        return;
      }
    } else {
      jamDebug();
      upgrade_to_exclusive_frag_access();
    }
  }
}

void Dblqh::handle_upgrade_to_exclusive_frag_access(Fragrecord *fragPtrP) {
  ndbrequire(m_old_fragment_lock_status == FRAGMENT_UNLOCKED);
  m_old_fragment_lock_status = m_fragment_lock_status;
  if (m_fragment_lock_status == FRAGMENT_UNLOCKED) {
    jamDebug();
    handle_acquire_exclusive_frag_access(fragPtrP, false);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_WRITE_KEY_MODE) {
    jamDebug();
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    fragPtrP->m_write_key_locked = false;
    handle_acquire_exclusive_frag_access(fragPtrP, true);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_READ_KEY_MODE) {
    jamDebug();
    ndbassert(!m_is_in_query_thread);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbrequire(fragPtrP->m_concurrent_read_key_count > 0);
    fragPtrP->m_concurrent_read_key_count--;
    handle_acquire_exclusive_frag_access(fragPtrP, true);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_WK_MODE) {
    jamDebug();
    ndbassert(!m_is_in_query_thread);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbrequire(fragPtrP->m_concurrent_read_key_count > 0);
    fragPtrP->m_concurrent_read_key_count--;
    fragPtrP->m_cond_write_key_waiters = 0;
    fragPtrP->m_spin_write_key_waiters = 0;
    handle_acquire_exclusive_frag_access(fragPtrP, true);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_REFRESH_MODE) {
    jamDebug();
    ndbassert(!m_is_in_query_thread);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbrequire(fragPtrP->m_concurrent_read_key_count > 0);
    fragPtrP->m_concurrent_read_key_count--;
    fragPtrP->m_cond_exclusive_waiters = 0;
    fragPtrP->m_spin_exclusive_waiters = 0;
    handle_acquire_exclusive_frag_access(fragPtrP, true);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_EXCLUSIVE_MODE) {
    jamDebug(); /* Already at exclusive level */
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_SCAN_MODE) {
    jamDebug();
    ndbassert(!m_is_query_block);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbrequire(fragPtrP->m_concurrent_scan_count > 0);
    fragPtrP->m_concurrent_scan_count--;
    handle_acquire_exclusive_frag_access(fragPtrP, true);
  } else {
    ndbabort();
  }
}

void Dblqh::handle_upgrade_to_write_key_frag_access(Fragrecord *fragPtrP) {
  /**
   * No need to prepare for downgrade here, we will either release
   * after this or upgrade to exclusive, we will never go below
   * write key level again before releasing the lock.
   */
  ndbrequire(m_old_fragment_lock_status == FRAGMENT_UNLOCKED);
  if (m_fragment_lock_status == FRAGMENT_UNLOCKED) {
    jamDebug();
    handle_acquire_write_key_frag_access(fragPtrP, false);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_READ_KEY_MODE) {
    ndbabort();  // No such case should exist
    jamDebug();
    ndbassert(!m_is_in_query_thread);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbassert(fragPtrP->m_concurrent_read_key_count > 0);
    fragPtrP->m_concurrent_read_key_count--;
    handle_acquire_write_key_frag_access(fragPtrP, false);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_WK_MODE) {
    jamDebug();
    ndbassert(!m_is_in_query_thread);
    NdbMutex_Lock(&fragPtrP->frag_mutex);
    DEB_FRAGMENT_LOCK(fragPtrP);
    ndbrequire(fragPtrP->m_concurrent_read_key_count > 0);
    fragPtrP->m_concurrent_read_key_count--;
    fragPtrP->m_cond_write_key_waiters = 0;
    fragPtrP->m_spin_write_key_waiters = 0;
    handle_acquire_write_key_frag_access(fragPtrP, true);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_WRITE_KEY_MODE) {
    ndbabort();  // No such case should exist
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_REFRESH_MODE) {
    ndbabort();  // No such case should exist
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_SCAN_MODE) {
    ndbabort();  // No such case should exist
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_EXCLUSIVE_MODE) {
    ndbabort();  // No such case should exist
  } else {
    ndbabort();  // No such lock state
  }
}

void Dblqh::handle_downgrade_from_exclusive_frag_access(Fragrecord *fragPtrP) {
  if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_WRITE_KEY_MODE) {
    jamDebug();
    downgrade_exclusive_to_write_key(fragPtrP);
  } else if (m_old_fragment_lock_status == FRAGMENT_UNLOCKED) {
    jamDebug();
    release_frag_access(fragPtrP);
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_EXCLUSIVE_MODE) {
    jamDebug(); /* Already at exclusive level */
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_READ_KEY_MODE) {
    jamDebug();
    downgrade_exclusive_to_read_key(fragPtrP);
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_WK_MODE) {
    jamDebug();
    downgrade_exclusive_to_read_key(fragPtrP);
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_REFRESH_MODE) {
    jamDebug();
    downgrade_exclusive_to_read_key(fragPtrP);
  } else if (m_old_fragment_lock_status == FRAGMENT_LOCKED_IN_SCAN_MODE) {
    jamDebug();
    downgrade_exclusive_to_scan();
  } else {
    ndbabort();  // No such lock state
  }
  m_fragment_lock_status = m_old_fragment_lock_status;
  m_old_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::handle_release_frag_access(Fragrecord *fragPtrP) {
  if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_SCAN_MODE) {
    jamDebug();
    handle_release_scan_frag_access(fragPtrP);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_READ_KEY_MODE) {
    jamDebug();
    handle_release_read_key_frag_access(fragPtrP);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_WRITE_KEY_MODE) {
    jamDebug();
    handle_release_write_key_frag_access(fragPtrP);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_EXCLUSIVE_MODE) {
    jamDebug();
    handle_release_exclusive_frag_access(fragPtrP);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_WK_MODE) {
    jamDebug();
    handle_release_read_key_frag_access(fragPtrP);
  } else if (m_fragment_lock_status == FRAGMENT_LOCKED_IN_RK_REFRESH_MODE) {
    jamDebug();
    handle_release_read_key_frag_access(fragPtrP);
  } else {
    ndbabort();
  }
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
  m_old_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::handle_release_scan_frag_access(Fragrecord *fragPtrP) {
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  fragPtrP->m_concurrent_scan_count--;
  if (fragPtrP->m_concurrent_scan_count == 0) {
    jamDebug();
    if ((fragPtrP->m_spin_write_key_waiters == 0) &&
        (fragPtrP->m_cond_write_key_waiters > 0)) {
      /**
       * A Prepare write operation is waiting and no Upgrade is waiting,
       * Prepare write operations can run concurrently with read key, so
       * no need to wait for them to go to 0.
       */
      jamDebug();
      DEB_FRAGMENT_LOCK(fragPtrP);
      NdbCondition_Signal(&fragPtrP->frag_write_cond);
    } else if (fragPtrP->m_concurrent_read_key_count == 0) {
      /**
       * All readers have finished, now we can start exclusive access
       * and Upgrade to exclusive lock.
       */
      if (((fragPtrP->m_spin_exclusive_waiters == 0) &&
           (fragPtrP->m_cond_exclusive_waiters > 0))) {
        jamDebug();
        DEB_FRAGMENT_LOCK(fragPtrP);
        NdbCondition_Signal(&fragPtrP->frag_write_cond);
      }
    }
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::handle_release_read_key_frag_access(Fragrecord *fragPtrP) {
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbassert(fragPtrP->m_concurrent_read_key_count > 0);
  fragPtrP->m_concurrent_read_key_count--;
  if (fragPtrP->m_concurrent_read_key_count == 0 &&
      fragPtrP->m_concurrent_scan_count == 0) {
    jamDebug();
    /**
     * Only exclusive access and Upgrade access can be stalled waiting
     * for a read key event to complete, so no need to check for write
     * key events waiting since they can start even with read key events
     * ongoing.
     */
    if ((fragPtrP->m_cond_exclusive_waiters > 0 &&
         fragPtrP->m_spin_exclusive_waiters == 0)) {
      jamDebug();
      DEB_FRAGMENT_LOCK(fragPtrP);
      NdbCondition_Signal(&fragPtrP->frag_write_cond);
    }
  }
  if (!m_is_in_query_thread) {
    if (fragPtrP->m_cond_write_key_waiters > 0 ||
        fragPtrP->m_cond_exclusive_waiters > 0) {
      /**
       * We performed a prepare key operation access where we were
       * performing a write operation, but we didn't upgrade before
       * releasing the lock. We need to remove the variables we set
       * in this case.
       */
      jamDebug();
      fragPtrP->m_cond_write_key_waiters = 0;
      fragPtrP->m_spin_write_key_waiters = 0;
      fragPtrP->m_cond_exclusive_waiters = 0;
      fragPtrP->m_spin_exclusive_waiters = 0;
      if (fragPtrP->m_cond_read_waiters > 0) {
        jamDebug();
        DEB_FRAGMENT_LOCK(fragPtrP);
        NdbCondition_Broadcast(&fragPtrP->frag_read_cond);
      }
    } else if (fragPtrP->m_cond_read_waiters > 0) {
      /**
       * We are executing in an LDM thread, this means that before arriving
       * here we could have upgraded the lock to an exclusive lock for a
       * time, when downgrading back to read key lock we will not yet start
       * the waiting readers. Since we are the LDM thread we cannot wait here
       * for other than query threads. Query threads cannot use write key
       * or exclusive locks. Arriving here in an LDM thread means that we
       * cannot hold any exclusive lock or write key lock, thus it is ok
       * here to wake up any waiters.
       */
      jamDebug();
      DEB_FRAGMENT_LOCK(fragPtrP);
      NdbCondition_Broadcast(&fragPtrP->frag_read_cond);
    }
    ndbrequire(is_scan_condition_ready(fragPtrP));
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::handle_release_write_key_frag_access(Fragrecord *fragPtrP) {
  jamDebug();
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  ndbrequire(fragPtrP->m_write_key_locked);
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbrequire(fragPtrP->m_spin_exclusive_waiters == 0);
  ndbrequire(fragPtrP->m_spin_write_key_waiters == 0);
  fragPtrP->m_write_key_locked = false;
  ndbrequire(is_scan_condition_ready(fragPtrP));
  if (fragPtrP->m_cond_read_waiters > 0) {
    jamDebug();
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Broadcast(&fragPtrP->frag_read_cond);
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
}

void Dblqh::handle_release_exclusive_frag_access(Fragrecord *fragPtrP) {
  jamDebug();
  ndbrequire(!DictTabInfo::isOrderedIndex(fragPtrP->tableType));
  NdbMutex_Lock(&fragPtrP->frag_mutex);
  DEB_FRAGMENT_LOCK(fragPtrP);
  ndbrequire(fragPtrP->m_exclusive_locked);
  ndbrequire(fragPtrP->m_spin_exclusive_waiters == 0);
  ndbrequire(fragPtrP->m_spin_write_key_waiters == 0);
  ndbrequire(fragPtrP->m_cond_write_key_waiters == 0);
  ndbrequire(fragPtrP->m_cond_exclusive_waiters == 0);
  ndbrequire(fragPtrP->m_write_key_locked == false);
  fragPtrP->m_exclusive_locked = false;
  ndbrequire(is_scan_condition_ready(fragPtrP));
  if (fragPtrP->m_cond_read_waiters > 0) {
    jamDebug();
    DEB_FRAGMENT_LOCK(fragPtrP);
    NdbCondition_Broadcast(&fragPtrP->frag_read_cond);
  }
  NdbMutex_Unlock(&fragPtrP->frag_mutex);
  m_fragment_lock_status = FRAGMENT_UNLOCKED;
}

/* ------------------------------------------------------------------------- */
/* -------                TAKE CARE OF LQHKEYREQ                     ------- */
/* LQHKEYREQ IS THE SIGNAL THAT STARTS ALL OPERATIONS IN THE LQH BLOCK       */
/* THIS SIGNAL CONTAINS A LOT OF INFORMATION ABOUT WHAT TYPE OF OPERATION,   */
/* KEY INFORMATION, ATTRIBUTE INFORMATION, NODE INFORMATION AND A LOT MORE   */
/* ------------------------------------------------------------------------- */
void Dblqh::execLQHKEYREQ(Signal *signal) {
  if (unlikely(!assembleFragments(signal))) {
    jam();
    return;
  }
  UintR sig0, sig1, sig2, sig3, sig4, sig5;
  Uint8 tfragDistKey;

  const LqhKeyReq *const lqhKeyReq = (LqhKeyReq *)signal->getDataPtr();
  SectionHandle handle(this, signal);
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = RNIL;
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  {
    const NodeBitmask &all = globalTransporterRegistry.get_status_overloaded();
    if (unlikely(!all.isclear())) {
      if (checkTransporterOverloaded(signal, all, lqhKeyReq)) {
        /* Overloaded, reject new work */
        jam();
        releaseSections(handle);
        earlyKeyReqAbort(signal, lqhKeyReq, ZTRANSPORTER_OVERLOADED_ERROR,
                         tcConnectptr);
        return;
      }
    }
  }

  const UintR Treqinfo = lqhKeyReq->requestInfo;

  if (ERROR_INSERTED(5078) &&
      refToMain(signal->header.theSendersBlockRef) == DBSPJ &&
      LqhKeyReq::getDirtyFlag(Treqinfo) &&
      !LqhKeyReq::getNormalProtocolFlag(Treqinfo)) {
    /**
     * This is used to trigger Bug#16187976 "NDBD NODE FAILS TO START WITH
     * ILLEGAL SIGNAL RECEIVED (GSN 121 NOT ADDED)". This bug occurs if a
     * ROUTE_ORD signal carrying a TCKEYREC signal is sent via the SPJ block.
     * ROUTE_ORD signals should always be sent via TC, which unlike SPJ should
     * be connected to the API. (Otherwise, the API will initiate its own
     * error handling which will compensate for TCKEYREC and other missing
     * signals.) The tests above check that we use the short-circuited protocol,
     * meaning that LQH wants to send TCKEYREC directly to the API, instead
     * of sending LQHKEYREC to SPJ (or TC).
     * Here we enable a different error insert (5079) which we test for in
     * Dblqh::sendTCKEYREF() below. It is done this way since in
     * sendTCKEYREF() we would otherwise not have sufficient context to tell
     * when to send the ROUTE_ORD signal.
     */
    SET_ERROR_INSERT_VALUE(5079);
  }

  if (ERROR_INSERTED_CLEAR(5047) || ERROR_INSERTED_CLEAR(5108) ||
      ERROR_INSERTED(5079) ||
      (ERROR_INSERTED(5102) && LqhKeyReq::getNoTriggersFlag(Treqinfo)) ||
      (ERROR_INSERTED(5103) && LqhKeyReq::getOperation(Treqinfo) == ZDELETE) ||
      (ERROR_INSERTED(5104) && LqhKeyReq::getOperation(Treqinfo) == ZINSERT) ||
      (ERROR_INSERTED(5105) && LqhKeyReq::getOperation(Treqinfo) == ZUPDATE) ||
      ERROR_INSERTED(5098)) {
    jam();
    releaseSections(handle);
    earlyKeyReqAbort(signal, lqhKeyReq, ZTRANSPORTER_OVERLOADED_ERROR,
                     tcConnectptr);
    return;
  }

  sig0 = lqhKeyReq->clientConnectPtr;
  if (likely((ctcNumFree > ZNUM_RESERVED_UTIL_CONNECT_RECORDS &&
              !ERROR_INSERTED(5031)) ||
             (ctcNumFree > ZNUM_RESERVED_TC_CONNECT_RECORDS &&
              LqhKeyReq::getUtilFlag(Treqinfo)))) {
    jamEntryDebug();
    seizeTcrec(tcConnectptr);
  } else {
    jamEntry();
    if (unlikely(!seize_op_rec(tcConnectptr))) {
      jam();
      /* -------------------------------------------------------------------------
       */
      /* NO FREE TC RECORD AVAILABLE, THUS WE CANNOT HANDLE THE REQUEST. */
      /* -------------------------------------------------------------------------
       */
      releaseSections(handle);
      earlyKeyReqAbort(signal, lqhKeyReq, ZNO_TC_CONNECT_ERROR, tcConnectptr);
      return;
    }
  }  // if

  if (ERROR_INSERTED(5038) &&
      refToNode(signal->getSendersBlockRef()) != getOwnNodeId()) {
    jam();
    releaseSections(handle);
    SET_ERROR_INSERT_VALUE(5039);
    return;
  }

  Uint32 tot_lqh_key_req_count = cTotalLqhKeyReqCount;
  Uint32 num_operations = c_Counters.operations;

  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  c_Counters.operations = num_operations + 1;
  cTotalLqhKeyReqCount = tot_lqh_key_req_count + 1;

  Uint32 senderRef = regTcPtr->clientBlockref = signal->senderBlockRef();
  regTcPtr->clientConnectrec = sig0;
  regTcPtr->tcOprec = sig0;
  regTcPtr->lqhKeyReqId = cTotalLqhKeyReqCount;
  regTcPtr->commitAckMarker = RNIL;
  regTcPtr->m_flags = 0;
  regTcPtr->m_flags |= TcConnectionrec::OP_ISLONGREQ;

  UintR attrLenFlags = lqhKeyReq->attrLen;
  sig1 = lqhKeyReq->savePointId;
  sig2 = lqhKeyReq->hashValue;
  sig4 = lqhKeyReq->tableSchemaVersion;
  sig5 = lqhKeyReq->tcBlockref;
  // Ensure that ROUTE_ORD (carrying TCKEYREF) will not be sent to SPJ.
  ndbassert(refToNode(signal->getSendersBlockRef()) == getOwnNodeId() ||
            !LqhKeyReq::getDirtyFlag(lqhKeyReq->requestInfo) ||
            LqhKeyReq::getNormalProtocolFlag(lqhKeyReq->requestInfo) ||
            LqhKeyReq::getOperation(lqhKeyReq->requestInfo) != ZREAD ||
            refToMain(lqhKeyReq->tcBlockref) == DBTC);

  regTcPtr->savePointId = sig1;
  regTcPtr->hashValue = sig2;
  const Uint32 schemaVersion = regTcPtr->schemaVersion =
      LqhKeyReq::getSchemaVersion(sig4);
  tabptr.i = LqhKeyReq::getTableId(sig4);
  regTcPtr->tcBlockref = sig5;
  regTcPtr->tcHashKeyHi = sig5;

  const Uint8 op = LqhKeyReq::getOperation(Treqinfo);
  if (ERROR_INSERTED(5080) ||
      (unlikely((op == ZREAD || op == ZREAD_EX) && !getAllowRead()))) {
    jam();
    releaseSections(handle);
    earlyKeyReqAbort(signal, lqhKeyReq, ZNODE_SHUTDOWN_IN_PROGRESS,
                     tcConnectptr);
    return;
  }

  if (ERROR_INSERTED(5081) ||
      unlikely(get_node_status(refToNode(sig5)) != ZNODE_UP)) {
    jam();
    releaseSections(handle);
    earlyKeyReqAbort(signal, lqhKeyReq, ZNODE_SHUTDOWN_IN_PROGRESS,
                     tcConnectptr);
    return;
  }

  Uint32 senderVersion = getNodeInfo(refToNode(senderRef)).m_version;

  regTcPtr->tcScanInfo = lqhKeyReq->scanInfo;
  regTcPtr->indTakeOver = LqhKeyReq::getScanTakeOverFlag(attrLenFlags);
  regTcPtr->m_reorg = LqhKeyReq::getReorgFlag(attrLenFlags);

  regTcPtr->readlenAi = 0;
  regTcPtr->currTupAiLen = 0;
  regTcPtr->logWriteState = TcConnectionrec::NOT_STARTED;
  regTcPtr->fragmentptr = RNIL;

  sig0 = lqhKeyReq->fragmentData;
  sig1 = lqhKeyReq->transId1;
  sig2 = lqhKeyReq->transId2;
  sig3 = lqhKeyReq->variableData[0];
  sig4 = lqhKeyReq->variableData[1];

  regTcPtr->fragmentid = LqhKeyReq::getFragmentId(sig0);
  regTcPtr->nextReplica = LqhKeyReq::getNextReplicaNodeId(sig0);
  regTcPtr->transid[0] = sig1;
  regTcPtr->transid[1] = sig2;
  regTcPtr->applRef = sig3;
  regTcPtr->applOprec = sig4;

  if (LqhKeyReq::getMarkerFlag(Treqinfo)) {
    struct CommitAckMarker check;
    CommitAckMarkerPtr markerPtr;
    jamDebug();
    check.transid1 = regTcPtr->transid[0];
    check.transid2 = regTcPtr->transid[1];

    if (m_commitAckMarkerHash.find(markerPtr, check)) {
      /*
        A commit ack marker was already placed here for this transaction.
        We increase the reference count to ensure we don't remove the
        commit ack marker prematurely.
      */
      ndbrequire(markerPtr.p->in_hash == true);
      ndbrequire(markerPtr.p->reference_count > 0);
      markerPtr.p->reference_count++;
#ifdef MARKER_TRACE
      g_eventLogger->info("Inc marker[%.8x %.8x] op: %u ref: %u",
                          markerPtr.p->transid1, markerPtr.p->transid2,
                          tcConnectptr.i, markerPtr.p->reference_count);
#endif
    } else {
      if (ERROR_INSERTED(5082) ||
          unlikely(!m_commitAckMarkerPool.seize(markerPtr))) {
        jam();
        releaseSections(handle);
        earlyKeyReqAbort(signal, lqhKeyReq, ZNO_FREE_MARKER_RECORDS_ERROR,
                         tcConnectptr);
        return;
      }
      markerPtr.p->transid1 = sig1;
      markerPtr.p->transid2 = sig2;
      markerPtr.p->apiRef = sig3;
      markerPtr.p->apiOprec = sig4;
      markerPtr.p->tcRef = sig5;
      markerPtr.p->reference_count = 1;
      markerPtr.p->in_hash = true;
      markerPtr.p->removed_by_fail_api = false;
      m_commitAckMarkerHash.add(markerPtr);

#ifdef MARKER_TRACE
      g_eventLogger->info("%u Add marker[%.8x %.8x] op: %u", instance(),
                          markerPtr.p->transid1, markerPtr.p->transid2,
                          tcConnectptr.i);
#endif
    }
    regTcPtr->commitAckMarker = markerPtr.i;
  }

  regTcPtr->reqinfo = Treqinfo;
  regTcPtr->lastReplicaNo = LqhKeyReq::getLastReplicaNo(Treqinfo);
  regTcPtr->dirtyOp = LqhKeyReq::getDirtyFlag(Treqinfo);
  regTcPtr->opExec = LqhKeyReq::getInterpretedFlag(Treqinfo);
  regTcPtr->opSimple = LqhKeyReq::getSimpleFlag(Treqinfo);
  regTcPtr->seqNoReplica = LqhKeyReq::getSeqNoReplica(Treqinfo);
  regTcPtr->m_use_rowid = LqhKeyReq::getRowidFlag(Treqinfo);
  regTcPtr->m_dealloc_state = TcConnectionrec::DA_IDLE;
  regTcPtr->m_query_thread = 0;
  regTcPtr->m_dealloc_data.m_dealloc_ref_count = RNIL;
  {
    regTcPtr->operation = (Operation_t)op == ZREAD_EX ? ZREAD : (Operation_t)op;
    regTcPtr->lockType =
        op == ZREAD_EX
            ? ZUPDATE
            : (Operation_t)op == ZWRITE
                  ? ZINSERT
                  : (Operation_t)op == ZREFRESH
                        ? ZINSERT
                        : (Operation_t)op == ZUNLOCK
                              ? ZREAD
                              :  // lockType not relevant for unlock req
                              (Operation_t)op;
  }
  if (LqhKeyReq::getNoWaitFlag(Treqinfo)) {
    /* Check sender version before processing - older versions sent junk */
    if (likely(senderVersion >= NDBD_NOWAIT_KEYREQ)) {
      ndbassert(!regTcPtr->dirtyOp);
      ndbrequire((op == ZREAD) || (op == ZREAD_EX));  // For now
      regTcPtr->m_flags |= TcConnectionrec::OP_NOWAIT;
    }
  }
  if (regTcPtr->dirtyOp) {
    ndbrequire(regTcPtr->opSimple);
  }

  CRASH_INSERTION2(5041,
                   (op == ZREAD && (regTcPtr->opSimple || regTcPtr->dirtyOp) &&
                    refToNode(signal->senderBlockRef()) != cownNodeid));

  regTcPtr->numFiredTriggers = lqhKeyReq->numFiredTriggers;

  UintR TapplAddressInd = LqhKeyReq::getApplicationAddressFlag(Treqinfo);
  UintR nextPos = (TapplAddressInd << 1);
  UintR TsameClientAndTcOprec = LqhKeyReq::getSameClientAndTcFlag(Treqinfo);
  if (TsameClientAndTcOprec == 1) {
    regTcPtr->tcOprec = lqhKeyReq->variableData[nextPos];
    nextPos++;
  }  // if
  UintR TnextReplicasIndicator =
      regTcPtr->lastReplicaNo - regTcPtr->seqNoReplica;
  if (TnextReplicasIndicator > 1) {
    regTcPtr->nodeAfterNext[0] = lqhKeyReq->variableData[nextPos] & 0xFFFF;
    regTcPtr->nodeAfterNext[1] = lqhKeyReq->variableData[nextPos] >> 16;
    nextPos++;
  }  // if
  UintR TreadLenAiIndicator = LqhKeyReq::getReturnedReadLenAIFlag(Treqinfo);
  if (TreadLenAiIndicator == 1) {
    regTcPtr->readlenAi = lqhKeyReq->variableData[nextPos] & ZNIL;
    nextPos++;
  }  // if

  Uint32 TanyValueFlag = LqhKeyReq::getCorrFactorFlag(Treqinfo);
  if (TanyValueFlag == 1) {
    /**
     * For short lqhkeyreq, ai-length in-signal is stored in same pos...
     */
    regTcPtr->m_corrFactorLo = lqhKeyReq->variableData[nextPos + 0];
    regTcPtr->m_corrFactorHi = lqhKeyReq->variableData[nextPos + 1];
    nextPos += 2;
  }

  regTcPtr->m_fire_trig_pass = 0;
  Uint32 Tdeferred = LqhKeyReq::getDeferredConstraints(Treqinfo);
  if (Tdeferred) {
    regTcPtr->m_flags |= TcConnectionrec::OP_DEFERRED_CONSTRAINTS;
  }

  Uint32 TdisableFk = LqhKeyReq::getDisableFkConstraints(Treqinfo);
  if (TdisableFk) {
    regTcPtr->m_flags |= TcConnectionrec::OP_DISABLE_FK;
  }

  Uint32 TnormalProtocolFlag = LqhKeyReq::getNormalProtocolFlag(Treqinfo);
  if (TnormalProtocolFlag) {
    /**
     * Only set normal protocol flag if long request.
     * As above, short lqhKeyReq ai-length in-signal overlaps the bit.
     * bug#14702377
     */
    regTcPtr->m_flags |= TcConnectionrec::OP_NORMAL_PROTOCOL;
  }

  if (LqhKeyReq::getNoTriggersFlag(Treqinfo)) {
    regTcPtr->m_flags |= TcConnectionrec::OP_NO_TRIGGERS;
  }

  UintR TitcKeyLen = 0;
  UintR TreclenAiLqhkey = 0;

  if (likely(handle.m_cnt > 0)) {
    jamDebug();
    /* Long LQHKEYREQ indicates Key and AttrInfo presence and
     * size via section lengths
     */
    SegmentedSectionPtr keyInfoSection, attrInfoSection;

    ndbrequire(handle.getSection(keyInfoSection, LqhKeyReq::KeyInfoSectionNum));

    ndbassert(keyInfoSection.i != RNIL);

    regTcPtr->keyInfoIVal = keyInfoSection.i;
    TitcKeyLen = keyInfoSection.sz;

    Uint32 totalAttrInfoLen = 0;
    if (handle.getSection(attrInfoSection, LqhKeyReq::AttrInfoSectionNum)) {
      regTcPtr->attrInfoIVal = attrInfoSection.i;
      totalAttrInfoLen = attrInfoSection.sz;
    }

    regTcPtr->reclenAiLqhkey = 0;
    regTcPtr->currReclenAi = totalAttrInfoLen;
    regTcPtr->totReclenAi = totalAttrInfoLen;
    regTcPtr->primKeyLen = TitcKeyLen;

    /* Detach sections from the handle, we are now responsible
     * for freeing them when appropriate
     */
    handle.clear();
  } else {
    /**
     * Only node restart copy allowed to send no KeyInfo.
     *
     * Only allowed use case for no primary key is DELETE by ROWID.
     * This is used by COPY Fragment and Restore fragment.
     */
    if (refToMain(senderRef) == DBSPJ) {
      jam();
      ndbassert(!LqhKeyReq::getNrCopyFlag(Treqinfo));
      /* Reply with NO_TUPLE_FOUND */
      earlyKeyReqAbort(signal, lqhKeyReq, ZNO_TUPLE_FOUND, tcConnectptr);
      return;
    }
    jamDebug();
    ndbrequire((LqhKeyReq::getRowidFlag(Treqinfo) != 0) &&
               (LqhKeyReq::getNrCopyFlag(Treqinfo) != 0) &&
               (LqhKeyReq::getAIInLqhKeyReq(Treqinfo) == 0));
    regTcPtr->reclenAiLqhkey = 0;
    regTcPtr->currReclenAi = 0;
    regTcPtr->totReclenAi = 0;
    regTcPtr->primKeyLen = 0;

    if (unlikely(!LqhKeyReq::getNrCopyFlag(Treqinfo))) {
      LQHKEY_error(signal, 3, tcConnectptr);
      return;
    }  // if
  }

  sig0 = lqhKeyReq->variableData[nextPos + 0];
  sig1 = lqhKeyReq->variableData[nextPos + 1];
  regTcPtr->m_row_id.m_page_no = sig0;
  regTcPtr->m_row_id.m_page_idx = sig1;
  nextPos += 2 * LqhKeyReq::getRowidFlag(Treqinfo);

  jamLineDebug(Uint16(nextPos));

  sig2 = lqhKeyReq->variableData[nextPos + 0];
  sig3 = cnewestGci;
  /* If gci_hi provided, take it and set gci_lo to max value
   * Otherwise, it will be decided by TUP at commit time as normal
   */
  regTcPtr->gci_hi = LqhKeyReq::getGCIFlag(Treqinfo) ? sig2 : sig3;
  regTcPtr->gci_lo = LqhKeyReq::getGCIFlag(Treqinfo) ? ~Uint32(0) : 0;
  nextPos += LqhKeyReq::getGCIFlag(Treqinfo);

  if (LqhKeyReq::getRowidFlag(Treqinfo)) {
    ndbassert(refToMain(senderRef) != DBTC);
  } else if (op == ZINSERT) {
    ndbassert(refToMain(senderRef) == DBTC);
  }

  if (unlikely((LqhKeyReq::FixedSignalLength + nextPos + TreclenAiLqhkey) !=
               signal->length())) {
    g_eventLogger->info("nextPos: %u, TreclenAiLqhkey: %u, siglen: %u", nextPos,
                        TreclenAiLqhkey, signal->length());
    LQHKEY_error(signal, 2, tcConnectptr);
    return;
  }  // if
  UintR TseqNoReplica = regTcPtr->seqNoReplica;
  UintR TlastReplicaNo = regTcPtr->lastReplicaNo;
  if (TseqNoReplica == TlastReplicaNo) {
    jamDebug();
    regTcPtr->nextReplica = ZNIL;
  } else {
    if (likely(TseqNoReplica < TlastReplicaNo)) {
      jamDebug();
      regTcPtr->nextSeqNoReplica = TseqNoReplica + 1;
      if (unlikely((regTcPtr->nextReplica == 0) ||
                   (regTcPtr->nextReplica == cownNodeid))) {
        LQHKEY_error(signal, 0, tcConnectptr);
        return;
      }  // if
    } else {
      LQHKEY_error(signal, 4, tcConnectptr);
      return;
    }  // if
  }    // if

  /**
   * If this is a 'dirtyOp' we dont care about transaction semantics.
   * There will then be no further abort, commit or unlock requests for
   * this operation. Thus, we will never have to find this operation
   * in the hashlist by calling findTransaction().
   * If also all ATTR- and KEYINFOs has been received, there will be no
   * ::execKEY- or ATTRINFO. (Long request, or all INFO fit in the REQ.)
   *
   * Thus we skip insertion in hashlist whenever not required.
   */
  if (regTcPtr->dirtyOp == ZFALSE)  // Transactional operation
  {
    jamDebug();
    /* Check that no equal element exists */
    ndbrequire(findTransaction(regTcPtr->transid[0], regTcPtr->transid[1],
                               regTcPtr->tcOprec, regTcPtr->tcHashKeyHi, true,
                               false, tcConnectptr) == ZNOT_FOUND);
    TcConnectionrecPtr localNextTcConnectptr;
    const Uint32 hashIndex = getHashIndex(regTcPtr);
    localNextTcConnectptr.i = ctransidHash[hashIndex];
    ctransidHash[hashIndex] = tcConnectptr.i;
    regTcPtr->prevHashRec = RNIL;
    regTcPtr->nextHashRec = localNextTcConnectptr.i;
    regTcPtr->hashIndex = hashIndex;
    if (localNextTcConnectptr.i != RNIL) {
      jamDebug();
      ndbrequire(tcConnect_pool.getValidPtr(localNextTcConnectptr));
      ndbassert(localNextTcConnectptr.p->prevHashRec == RNIL);
      localNextTcConnectptr.p->prevHashRec = tcConnectptr.i;
    }  // if
  }    // if
  /**
   * Up until this point in execLQHKEYREQ all we have done is setting up
   * data structures that are local to this thread. Now we are ready to
   * setup parts that can be concurrently used by multiple LDM and query
   * threads.
   */
  if (qt_likely(m_is_query_block)) {
    /**
     * We are a query thread or a recover thread. We need to set up
     * the references to the LDM instances meta data objects.
     * This is a key access, so there is no need to setup TUX objects.
     */
    Uint32 instanceNo = getInstanceNoCanFail(tabptr.i, regTcPtr->fragmentid);
    if (unlikely(instanceNo == RNIL)) {
      jam();
      LQHKEY_abort(signal, 5, tcConnectptr);
      return;
    }
    regTcPtr->ldmInstance = instanceNo;
    setup_query_thread_for_key_access(instanceNo);
  }
  if (unlikely(tabptr.i >= ctabrecFileSize)) {
    LQHKEY_error(signal, 5, tcConnectptr);
    return;
  }  // if
  ptrAss(tabptr, tablerec);
  if (unlikely(table_version_major_lqhkeyreq(tabptr.p->schemaVersion) !=
               table_version_major_lqhkeyreq(schemaVersion))) {
    LQHKEY_abort(signal, 5, tcConnectptr);
    return;
  }

  if (unlikely(tabptr.p->tableStatus != Tablerec::TABLE_DEFINED)) {
    if (check_tabstate(signal, tabptr.p, op, tcConnectptr)) return;
  }
  if (unlikely(!getFragmentrec(regTcPtr->fragmentid))) {
    LQHKEY_abort(signal, 6, tcConnectptr);
    return;
  }  // if

  regTcPtr->tableref = tabptr.i;
  regTcPtr->m_disk_table = tabptr.p->m_disk_table;
  Uint32 senderBlockNo = refToMain(signal->senderBlockRef());
  if (senderBlockNo == getRESTORE())
    regTcPtr->m_disk_table &= !LqhKeyReq::getNoDiskFlag(Treqinfo);
  else if (op == ZREAD || op == ZREAD_EX || op == ZUPDATE)
    regTcPtr->m_disk_table &= !LqhKeyReq::getNoDiskFlag(Treqinfo);

  if (op == ZREAD || op == ZREAD_EX || op == ZUNLOCK) {
    tabptr.p->usageCountR++;
  } else {
    tabptr.p->usageCountW++;
  }

  if (LqhKeyReq::getNrCopyFlag(Treqinfo) &&
      refToMain(senderRef) != getRESTORE()) {
    ndbassert(refToMain(senderRef) == getDBLQH());
    ndbassert(LqhKeyReq::getRowidFlag(Treqinfo));
    if (!(fragptr.p->fragStatus == Fragrecord::ACTIVE_CREATION)) {
      g_eventLogger->info("fragptr.p->fragStatus: %d", fragptr.p->fragStatus);
      CRASH_INSERTION(5046);
    }
    /**
     * We discover start of Node recovery phase in starting node
     * by seeing the first LQHKEYREQ arrive with getNrCopyFlag set.
     * We will set it on every LQHKEYREQ, only the first is really
     * needed. We set state to Fragrecord::AC_IGNORED in the
     * PREPARE_COPY_FRAGREQ. We could participate in transactions
     * even before the first copy row has been received. In this
     * case we can safely ignore the row, so this code ensures that
     * we won't ignore rows later rows after the first copy row
     * has been received. When this row has been received we need
     * to check if the UPDATE/DELETEs received from normal transactions
     * have to be applied since the row could have arrived before
     * the transaction then.
     */
    ndbassert(fragptr.p->fragStatus == Fragrecord::ACTIVE_CREATION);
    fragptr.p->m_copy_started_state = Fragrecord::AC_NR_COPY;

    if (op == ZDELETE)
      c_fragCopyRowsDel++;
    else
      c_fragCopyRowsIns++;

    c_fragBytesCopied += (signal->length() << 2);
  } else if (!m_is_in_query_thread && refToMain(senderRef) != getRESTORE()) {
    Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
    /**
     * Don't count for NR fragcopy, just 'normal' operation.
     * No stats produced during restore operation.
     * No stats produced by Query threads.
     */
    switch (op) {
      case ZREAD:
      case ZREAD_EX:
      case ZUNLOCK:
        useStat.m_readKeyReqCount++;
        break;

      case ZUPDATE:
        useStat.m_updKeyReqCount++;
        break;

      case ZINSERT:
        useStat.m_insKeyReqCount++;
        break;

      case ZWRITE:
        useStat.m_writeKeyReqCount++;
        break;

      case ZDELETE:
        useStat.m_delKeyReqCount++;
        break;

      default:
        // ZREFRESH is not counted.
        break;
    }
    useStat.m_keyReqAttrWords += regTcPtr->totReclenAi;
    useStat.m_keyReqKeyWords += TitcKeyLen;
    if (unlikely(LqhKeyReq::getInterpretedFlag(Treqinfo))) {
      /*
        Complete attrinfo may not have been received yet for short-signal
        lookups. We ignore these, since they only happen during online
        upgrade.
      */
      ndbassert(regTcPtr->attrInfoIVal != RNIL);
      SegmentedSectionPtr attrInfo;
      getSection(attrInfo, regTcPtr->attrInfoIVal);
      useStat.m_keyProgramWords += getProgramWordCount(attrInfo);
    }
  }

  Uint8 TcopyType = fragptr.p->fragCopy;
  LogPartRecord *logPart = fragptr.p->m_log_part_ptr_p;
  tfragDistKey = fragptr.p->fragDistributionKey;
  if (fragptr.p->fragStatus == Fragrecord::ACTIVE_CREATION) {
    jam();
    /**
     * Starting node in active creation mode, we set activeCreat to
     * either AC_IGNORED (before first copy row arrived, or to
     * AC_NR_COPY after first copy row arrived. We set activeCreat
     * to AC_IGNORED also when we discover that we should ignore
     * the row since it updates a row which we haven't received
     * a copy row for yet.
     */
    regTcPtr->activeCreat = fragptr.p->m_copy_started_state;
    CRASH_INSERTION(5002);
    CRASH_INSERTION2(5042, tabptr.i == c_error_insert_table_id);
  } else {
    regTcPtr->activeCreat = Fragrecord::AC_NORMAL;
  }  // if
  c_tup->prepare_tab_pointers(fragptr.p->tupFragptr);
  regTcPtr->replicaType = TcopyType;
  regTcPtr->fragmentptr = fragptr.i;
  regTcPtr->m_log_part_ptr_p = logPart;
  Uint8 TdistKey = LqhKeyReq::getDistributionKey(attrLenFlags);
  if (unlikely((tfragDistKey != TdistKey) && (regTcPtr->seqNoReplica == 0) &&
               (regTcPtr->dirtyOp == ZFALSE))) {
    /* ----------------------------------------------------------------------
     * WE HAVE DIFFERENT OPINION THAN THE DIH THAT STARTED THE TRANSACTION.
     * THE REASON COULD BE THAT THIS IS AN OLD DISTRIBUTION WHICH IS NO LONGER
     * VALID TO USE. THIS MUST BE CHECKED.
     * ONE IS ADDED TO THE DISTRIBUTION KEY EVERY TIME WE ADD A NEW REPLICA.
     * FAILED REPLICAS DO NOT AFFECT THE DISTRIBUTION KEY. THIS MEANS THAT THE
     * MAXIMUM DEVIATION CAN BE 3 BETWEEN THOSE TWO VALUES SINCE WITH 4 REPLICAS
     * WE CAN ADD 3 REPLICAS IN ONE GO.
     * --------------------------------------------------------------------- */
    Int8 tmp = (TdistKey - tfragDistKey);
    tmp = (tmp < 0 ? -tmp : tmp);
    if (unlikely((tmp <= (MAX_REPLICAS - 1)) || (tfragDistKey == 0))) {
      LQHKEY_abort(signal, 0, tcConnectptr);
      return;
    }  // if
    LQHKEY_error(signal, 1, tcConnectptr);
    return;
  }  // if

  /*
   * Interpreted updates and deletes may require different AttrInfo in
   * different replicas, as only the primary executes the interpreted
   * program, and the effect of the program rather than the program
   * should be logged.
   * Non interpreted inserts, updates, writes and deletes use the same
   * AttrInfo in all replicas.
   * All reads only run on one replica, and are not logged.
   * The AttrInfo section is passed to TUP attached to the TUPKEYREQ
   * signal below.
   *
   * Normal processing :
   *   - LQH passes ATTRINFO section to TUP attached to direct TUPKEYREQ
   *     signal
   *   - TUP processes request and sends direct TUPKEYCONF back to LQH
   *   - LQH continues processing (logging, forwarding LQHKEYREQ to other
   *     replicas as necessary)
   *   - LQH frees ATTRINFO section
   *   Note that TUP is not responsible for freeing the passed ATTRINFO
   *   section, LQH is.
   *
   * Interpreted Update / Delete processing
   *   - LQH passes ATTRINFO section to TUP attached to direct TUPKEYREQ
   *     signal
   *   - TUP processes request, generating new ATTRINFO data
   *   - If new AttrInfo data is > 0 words, TUP sends it back to LQH as
   *     a long section attached to a single ATTRINFO signal.
   *     - LQH frees the original AttrInfo section and stores a ref to
   *       the new section
   *   - TUP sends direct TUPKEYCONF back to LQH with new ATTRINFO length
   *   - If the new ATTRINFO is > 0 words,
   *       - LQH continues processing with it (logging, forwarding
   *         LQHKEYREQ to other replicas as necessary)
   *       - LQH frees the new ATTRINFO section
   *   - If the new ATTRINFO is 0 words, LQH frees the original ATTRINFO
   *     section and continues processing (logging, forwarding LQHKEYREQ
   *     to other replicas as necessary)
   *
   */
  bool attrInfoToPropagate =
      (regTcPtr->totReclenAi != 0) && (regTcPtr->operation != ZREAD) &&
      (regTcPtr->operation != ZDELETE) && (regTcPtr->operation != ZUNLOCK);
  bool tupCanChangePropagatedAttrInfo = (regTcPtr->opExec == 1);

  bool saveAttrInfo = attrInfoToPropagate && (!tupCanChangePropagatedAttrInfo);

  if (saveAttrInfo) regTcPtr->m_flags |= TcConnectionrec::OP_SAVEATTRINFO;

  /* Handle any AttrInfo we received with the LQHKEYREQ */
  if (regTcPtr->currReclenAi != 0) {
    /* Long LQHKEYREQ */
    jamDebug();
    regTcPtr->currTupAiLen = saveAttrInfo ? regTcPtr->totReclenAi : 0;
  }  // if
  ndbassert(regTcPtr->totReclenAi == regTcPtr->currReclenAi);
  if (qt_likely((globalData.ndbMtQueryThreads > 0) &&
                refToMain(regTcPtr->clientBlockref) != getRESTORE())) {
    acquire_frag_prepare_key_access(fragptr.p, regTcPtr);
  }
  /* ---------------------------------------------------------------------- */
  /*       NOW RECEPTION OF LQHKEYREQ IS COMPLETED THE NEXT STEP IS TO START*/
  /*       PROCESSING THE MESSAGE. IF THE MESSAGE IS TO A STAND-BY NODE     */
  /*       WITHOUT NETWORK REDUNDANCY OR PREPARE-TO-COMMIT ACTIVATED THE    */
  /*       PREPARATION TO SEND TO THE NEXT NODE WILL START IMMEDIATELY.     */
  /*                                                                        */
  /*       OTHERWISE THE PROCESSING WILL START AFTER SETTING THE PROPER     */
  /*       STATE. HOWEVER BEFORE PROCESSING THE MESSAGE                     */
  /*       IT IS NECESSARY TO CHECK THAT THE FRAGMENT IS NOT PERFORMING     */
  /*       A CHECKPOINT. THE OPERATION SHALL ALSO BE LINKED INTO THE        */
  /*       FRAGMENT QUEUE OR LIST OF ACTIVE OPERATIONS.                     */
  /*                                                                        */
  /*       THE FIRST STEP IN PROCESSING THE MESSAGE IS TO CONTACT DBACC.    */
  /*------------------------------------------------------------------------*/
  switch (fragptr.p->fragStatus) {
    case Fragrecord::FSACTIVE:
    case Fragrecord::CRASH_RECOVERING:
    case Fragrecord::ACTIVE_CREATION:
      prepareContinueAfterBlockedLab(signal, tcConnectptr);
      release_frag_access(fragptr.p);
      return;
    case Fragrecord::FREE:
      ndbabort();
    case Fragrecord::DEFINED:
      ndbabort();
    case Fragrecord::REMOVING:
      ndbabort();
    default:
      ndbabort();
  }  // switch
}  // Dblqh::execLQHKEYREQ()

void Dblqh::prepareContinueAfterBlockedLab(
    Signal *signal, const TcConnectionrecPtr tcConnectptr) {
  UintR ttcScanOp;

  /* --------------------------------------------------------------------------
   */
  /*       INPUT:          TC_CONNECTPTR           ACTIVE CONNECTION RECORD */
  /*                       FRAGPTR                 FRAGMENT RECORD */
  /* --------------------------------------------------------------------------
   */
  /* --------------------------------------------------------------------------
   */
  /*  CONTINUE HERE AFTER BEING BLOCKED FOR A WHILE DURING LOCAL CHECKPOINT. */
  /* --------------------------------------------------------------------------
   */
  /*       ALSO AFTER NORMAL PROCEDURE WE CONTINUE HERE */
  /* --------------------------------------------------------------------------
   */
  Uint32 tc_ptr_i = tcConnectptr.i;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  Uint32 activeCreat = regTcPtr->activeCreat;
  if (regTcPtr->operation == ZUNLOCK) {
    jam();
    ndbassert(!m_is_in_query_thread);
    handleUserUnlockRequest(signal, tcConnectptr);
    return;
  }

  if (unlikely(regTcPtr->indTakeOver == ZTRUE)) {
    jam();
    ndbassert(!m_is_query_block);
    ttcScanOp = KeyInfo20::getScanOp(regTcPtr->tcScanInfo);
    scanptr.i = RNIL;
    {
      ScanRecord key;
      key.scanNumber = KeyInfo20::getScanNo(regTcPtr->tcScanInfo);
      key.fragPtrI = fragptr.i;
      c_scanTakeOverHash.find(scanptr, key);
#ifdef TRACE_SCAN_TAKEOVER
      if (scanptr.i == RNIL)
        g_eventLogger->info("not finding (%d %d)", key.scanNumber,
                            key.fragPtrI);
#endif
    }
    if (unlikely(scanptr.i == RNIL)) {
      jam();
      takeOverErrorLab(signal, tcConnectptr);
      return;
    }  // if
    regTcPtr->accOpPtr =
        get_acc_ptr_from_scan_record(scanptr.p, ttcScanOp, true);
    if (unlikely(regTcPtr->accOpPtr == RNIL)) {
      jam();
      takeOverErrorLab(signal, tcConnectptr);
      return;
    }  // if
  }    // if
  /*-------------------------------------------------------------------*/
  /*       IT IS NOW TIME TO CONTACT ACC. THE TUPLE KEY WILL BE SENT   */
  /*       AND THIS WILL BE TRANSLATED INTO A LOCAL KEY BY USING THE   */
  /*       LOCAL PART OF THE LH3-ALGORITHM. ALSO PROPER LOCKS ON THE   */
  /*       TUPLE WILL BE SET. FOR INSERTS AND DELETES THE MESSAGE WILL */
  /*       START AN INSERT/DELETE INTO THE HASH TABLE.                 */
  /*                                                                   */
  /*       BEFORE SENDING THE MESSAGE THE REQUEST INFORMATION IS SET   */
  /*       PROPERLY.                                                   */
  /* ----------------------------------------------------------------- */
  if (TRACENR_FLAG) {
    TRACE_OP(regTcPtr, "RECEIVED");
    switch (regTcPtr->operation) {
      case ZREAD:
        TRACENR("READ");
        break;
      case ZUPDATE:
        TRACENR("UPDATE");
        break;
      case ZWRITE:
        TRACENR("WRITE");
        break;
      case ZINSERT:
        TRACENR("INSERT");
        break;
      case ZDELETE:
        TRACENR("DELETE");
        break;
      case ZUNLOCK:
        TRACENR("UNLOCK");
        break;
      case ZREFRESH:
        TRACENR("REFRESH");
        break;
      default:
        TRACENR("<Unknown: " << regTcPtr->operation << ">");
        break;
    }

    TRACENR(" tab: " << regTcPtr->tableref << " frag: " << regTcPtr->fragmentid
                     << " activeCreat: " << (Uint32)activeCreat);
    if (LqhKeyReq::getNrCopyFlag(regTcPtr->reqinfo)) TRACENR(" NrCopy");
    if (LqhKeyReq::getRowidFlag(regTcPtr->reqinfo))
      TRACENR(" rowid: " << regTcPtr->m_row_id);
    TRACENR(" key: " << getKeyInfoWordOrZero(regTcPtr, 0));
  }

  if (likely(activeCreat == Fragrecord::AC_NORMAL)) {
    if (TRACENR_FLAG) TRACENR(endl);
    if (likely(!LqhKeyReq::getNrCopyFlag(regTcPtr->reqinfo))) {
      /* Normal path */
      exec_acckeyreq(signal, tcConnectptr);
    } else {
      jam();
      /**
       * Delete by ROWID from RESTORE
       */
      ndbrequire(LqhKeyReq::getRowidFlag(regTcPtr->reqinfo));
      ndbrequire(regTcPtr->operation == ZDELETE);
      handle_nr_copy(signal, tcConnectptr);
    }
  } else if (activeCreat == Fragrecord::AC_NR_COPY) {
    /* Node restart do not use scan lock take over */
    /**
     * This is always the code path taken after the first copy row has
     * arrived, both for copy rows and for normal transactions. We are
     * in the starting node and the fragment isn't yet up to date, so we
     * need to be careful with all variants of how we deal with syncing
     * this starting fragment with the live fragment.
     */
    jam();
    ndbrequire(!regTcPtr->indTakeOver);
    regTcPtr->totSendlenAi = regTcPtr->totReclenAi;
    handle_nr_copy(signal, tcConnectptr);
  } else {
    /* Aborts can not use scan lock take over.
     * And scan lock take over can not be aborted.
     *
     * First copy row hasn't arrived yet, we will ignore any row updates,
     * but to the other nodes we will act as if we have applied the
     * changes.
     */
    jam();
    ndbrequire(!regTcPtr->indTakeOver);
    ndbassert(activeCreat == Fragrecord::AC_IGNORED);
    if (TRACENR_FLAG) TRACENR(" IGNORING (activeCreat == 2)" << endl);

    signal->theData[0] = tc_ptr_i;
    regTcPtr->transactionState = TcConnectionrec::WAIT_ACC_ABORT;

    signal->theData[0] = regTcPtr->tupConnectrec;
    c_tup->do_tup_abortreq(signal, 0);
    jamEntryDebug();
    regTcPtr->totSendlenAi = regTcPtr->totReclenAi;
    packLqhkeyreqLab(signal, tcConnectptr);
  }
}

void Dblqh::exec_acckeyreq(Signal *signal, TcConnectionrecPtr regTcPtr) {
  /* ************ */
  /*  ACCKEYREQ < */
  /* ************ */
  prefetch_op_record_3((Uint32 *)regTcPtr.p->accConnectPtrP);
  jam();
  {
    Uint32 taccreq = 0;
    taccreq = AccKeyReq::setOperation(taccreq, regTcPtr.p->operation);
    taccreq = AccKeyReq::setLockType(taccreq, regTcPtr.p->lockType);
    taccreq = AccKeyReq::setDirtyOp(taccreq, regTcPtr.p->dirtyOp);
    taccreq = AccKeyReq::setReplicaType(taccreq, regTcPtr.p->replicaType);
    taccreq = AccKeyReq::setTakeOver(taccreq, regTcPtr.p->indTakeOver);
    taccreq = AccKeyReq::setNoWait(
        taccreq, ((regTcPtr.p->m_flags & TcConnectionrec::OP_NOWAIT) != 0));
    taccreq = AccKeyReq::setLockReq(taccreq, false);

    AccKeyReq *const req = reinterpret_cast<AccKeyReq *>(&signal->theData[0]);
    req->fragmentPtr = fragptr.p->accFragptr;
    req->requestInfo = taccreq;
    req->hashValue = regTcPtr.p->hashValue;
    req->keyLen = regTcPtr.p->primKeyLen;
    req->transId1 = regTcPtr.p->transid[0];
    req->transId2 = regTcPtr.p->transid[1];
    req->lockConnectPtr = regTcPtr.p->indTakeOver ? regTcPtr.p->accOpPtr : RNIL;
    ndbrequire(req->keyLen > 0);
    memcpy(req->keyInfo, &req, AccKeyReq::SignalLength_keyInfo);

    regTcPtr.p->transactionState = TcConnectionrec::WAIT_ACC;

    /* Copy KeyInfo to end of ACCKEYREQ signal, starting at offset 7 */
    copy(req->keyInfo, regTcPtr.p->keyInfoIVal);
    static_assert(AccKeyReq::SignalLength_keyInfo == 8);
  }
  TRACE_OP(regTcPtr.p, "ACC");

  signal->setLength(AccKeyReq::SignalLength_keyInfo + regTcPtr.p->primKeyLen);
  c_acc->execACCKEYREQ(signal, regTcPtr.p->accConnectrec,
                       regTcPtr.p->accConnectPtrP);
  jamEntryDebug();
  m_tc_connect_ptr = regTcPtr;
  if (signal->theData[0] < RNIL) {
    jamDebug();
    continueACCKEYCONF(signal, signal->theData[3], signal->theData[4],
                       regTcPtr);
    return;
  } else if (signal->theData[0] == RNIL) {
    ndbassert(!m_is_query_block);
    return;
  } else {
    ndbassert(signal->theData[0] == (UintR)-1);
    /* Dbacc scan take over error */
    if (signal->theData[1] == ZTO_OP_STATE_ERROR) {
      ndbassert(!m_is_query_block);
      execACC_TO_REF(signal, regTcPtr);
    } else {
      terrorCode = signal->theData[1];
      continueACCKEYREF(signal, regTcPtr);
    }
  }  // if
}  // Dblqh::prepareContinueAfterBlockedLab()

void Dblqh::handle_nr_copy(Signal *signal, Ptr<TcConnectionrec> regTcPtr) {
  jam();
  Uint32 fragPtr = fragptr.p->tupFragptr;
  Uint32 op = regTcPtr.p->operation;

  const bool nrCopyFlag = LqhKeyReq::getNrCopyFlag(regTcPtr.p->reqinfo);

  if (!LqhKeyReq::getRowidFlag(regTcPtr.p->reqinfo)) {
    /**
     * Rowid not set, that mean that primary has finished copying...
     * This effectively means that our fragment is up-to-date and
     * synchronised with the primary replica. There is still work
     * needed to make the fragment durable, but from the point of
     * view of executing LQHKEYREQ we're a normal fragment now.
     */
    jam();
    if (TRACENR_FLAG) TRACENR(" Waiting for COPY_ACTIVEREQ" << endl);
    ndbassert(!LqhKeyReq::getNrCopyFlag(regTcPtr.p->reqinfo));
    regTcPtr.p->activeCreat = Fragrecord::AC_NORMAL;
    exec_acckeyreq(signal, regTcPtr);
    return;
  }

  /* Signal header was counted for when receiving LQHKEYREQ */
  c_fragBytesCopied +=
      ((regTcPtr.p->primKeyLen + ((regTcPtr.p->attrInfoIVal == RNIL)
                                      ? 0
                                      : getSectionSz(regTcPtr.p->attrInfoIVal)))
       << 2);

  regTcPtr.p->m_nr_delete.m_cnt = 1;  // Wait for real op as well
  Uint32 *dst = signal->theData + 24;
  bool uncommitted;
  const int len =
      c_tup->nr_read_pk(fragPtr, &regTcPtr.p->m_row_id, dst, uncommitted);
  const bool match = (len > 0) ? compare_key(regTcPtr.p, dst, len) == 0 : false;

  if (TRACENR_FLAG)
    TRACENR(" len: " << len << " match: " << match
                     << " uncommitted: " << uncommitted);

  /**
   * len == 0 here means that the row id had no record attached to it.
   * len > 0 means that we returned a primary key from nr_read_pk.
   * len == 0 >> match = false
   *
   * DELETE by ROWID means regTcPtr.p->primKeyLen is 0 and thus compare_key
   * will not return true and thus match = false
   *
   * When len > 0 we will check if the primary key sent from the live node
   * is equal to the primary key we store here. If it is equal match = true
   * otherwise match = false
   *
   * The DELETE by ROWID case is reused also for delete row from RESTORE when
   * restoring changes in an LCP. In this we set the NrCopyFlag.
   */
  TcConnectionrecPtr tcConnectptr = regTcPtr;
  if (nrCopyFlag) {
    /**
     * This is a copy row sent from live node to starting node.
     * It is either an INSERT with the full row and with row id.
     * Otherwise it is a DELETE by ROWID without primary key.
     * This signal comes with the GCI set on the row at the primary
     * replica.
     *
     * It can also be a DELETE_BY_ROWID sent from RESTORE.
     * In this case the operation is always DELETE.
     */
    ndbassert(LqhKeyReq::getGCIFlag(regTcPtr.p->reqinfo));
    if (match) {
      /**
       * Case 1
       * ------
       * An INSERT is used to copy the row from the live node to the
       * starting node. The starting node already had the row and the
       * primary key was correct. So we simply translate the INSERT
       * into an UPDATE and perform the update. After this the row
       * is up to date.
       */
      jam();
      ndbrequire(op == ZINSERT);
      if (TRACENR_FLAG) TRACENR(" Changing from INSERT to ZUPDATE" << endl);
      regTcPtr.p->operation = ZUPDATE;
      goto run;
    } else if (op == ZDELETE) {
      ndbrequire(regTcPtr.p->primKeyLen == 0);
      if (len > 0) {
        /**
         * Case 4
         * ------
         *   We are performing DELETE by ROWID and the row id had an already
         *   existing, we need to delete the row in this position.
         */
        jam();
        if (TRACENR_FLAG) TRACENR(" performing DELETE key: " << dst[0] << endl);

        if (refToMain(regTcPtr.p->tcBlockref) == getRESTORE()) {
          jam();
          c_restore->delete_by_rowid_succ(regTcPtr.p->tcOprec);
        }
        DEB_LCP_RESTORE(("(%u)tab(%u,%u) row(%u,%u), set GCI = %u", instance(),
                         regTcPtr.p->tableref, regTcPtr.p->fragmentid,
                         regTcPtr.p->m_row_id.m_page_no,
                         regTcPtr.p->m_row_id.m_page_idx, regTcPtr.p->gci_hi));
        c_tup->nr_update_gci(fragPtr, &regTcPtr.p->m_row_id, regTcPtr.p->gci_hi,
                             true);
        nr_copy_delete_row(signal, regTcPtr, &regTcPtr.p->m_row_id, len);
        ndbassert(regTcPtr.p->m_nr_delete.m_cnt);
        regTcPtr.p->m_nr_delete.m_cnt--;  // No real op is run
        if (regTcPtr.p->m_nr_delete.m_cnt) {
          jam();
          /* Only happens with disk data in copy fragment phase */
          ndbrequire(regTcPtr.p->activeCreat == Fragrecord::AC_NR_COPY);
          return;
        }
        packLqhkeyreqLab(signal, regTcPtr);
        return;
      } else if (len == 0 && op == ZDELETE) {
        /**
         * Case 7
         * ------
         * We are performing a DELETE by ROWID and there was no row at this
         * row id. We set the correct GCI in this row id.
         */
        jam();
        if (TRACENR_FLAG) TRACENR(" UPDATE_GCI" << endl);
        if (refToMain(regTcPtr.p->tcBlockref) == getRESTORE()) {
          jam();
          c_restore->delete_by_rowid_fail(regTcPtr.p->tcOprec);
        }
        c_tup->nr_update_gci(fragPtr, &regTcPtr.p->m_row_id, regTcPtr.p->gci_hi,
                             false);
        goto update_gci_ignore;
      }
    }
    /* !match && op != ZDELETE */

    /**
     * If we come here we are receiving a copy row (an INSERT), the
     * row id position either had an existing row at this position or not,
     * but if it had it has a different primary key.
     *
     * Perform the following action:
     * 1) Delete row at specified rowid (if len > 0)
     * 2) Delete specified row at different rowid (if exists)
     * 3) Run insert
     */
    if (len > 0) {
      /**
       * 1) Delete row at specified rowid (if len > 0)
       * A row existed but it was different so we delete the row at this
       * row id position.
       */
      jam();
      nr_copy_delete_row(signal, regTcPtr, &regTcPtr.p->m_row_id, len);
    }
    /**
     * 2) Delete specified row at different rowid (if exists)
     * It is technically possible that a row with the same primary key
     * also exists. This record then has a different row id. This is an
     * interesting case which can happen if the given primary key and then
     * later inserted again. We have to handle this case now even though it
     * would be handled later as the hash index is unique and cannot have
     * two records with the same primary key.
     *
     * We will soon reinsert a record with this primary key, so the primary
     * key is simply moved to another row id. The row id it is currently
     * placed should have a higher row id since the copy process goes from
     * low row ids to higher row ids.
     */
    jam();
    nr_copy_delete_row(signal, regTcPtr, 0, 0);
    if (TRACENR_FLAG) TRACENR(" RUN INSERT" << endl);
    goto run;
  } else {
    /**
     * nrCopyFlag == false
     * This is a normal operation in a starting node which is currently being
     * synchronised with the live node.
     */
    if (!match && op != ZINSERT) {
      /**
       * We are performing an UPDATE or a DELETE and the row id position
       * doesn't contain the correct primary key.
       *
       * Either there was no row in this row id, or it is an old row which
       * which haven't yet seen the copy row. We can safely ignore this
       * one.
       */
      jam();
      if (TRACENR_FLAG) TRACENR(" IGNORE " << endl);
      goto ignore;
    }
    if (match) {
      /**
       * An INSERT/UPDATE/DELETE/REFRESH on a record where we have the correct
       * primary key in this row id position. We convert the INSERT to a write
       * to speed things up a bit rather than first deleting row and then
       * inserting it. UPDATE is also converted to WRITE, but this has no real
       * effect when the row is already there.
       */
      jam();
      if (op != ZDELETE && op != ZREFRESH) {
        if (TRACENR_FLAG)
          TRACENR(" Changing from INSERT/UPDATE to ZWRITE" << endl);
        regTcPtr.p->operation = ZWRITE;
      }
      goto run;
    }

    /**
     * This is a normal operation that does an insert in a row id position
     * which either has a different primary key or no record in the row
     * id position.
     *
     * We cannot ignore this one. If it is inserted before the current row
     * id position in the live node, then we will not see any copy row for
     * this row. Since we don't know we will perform the insert now in the
     * same manner as if it was a copy row coming. It might be redone later
     * but this is not a problem with consistency.
     */
    ndbassert(!match && op == ZINSERT);

    /**
     * Perform the following action (same as above for copy row case)
     * 1) Delete row at specified rowid (if len > 0)
     * 2) Delete specified row at different rowid (if exists)
     * 3) Run insert
     */
    if (len > 0) {
      /**
       * 1) Delete row at specified rowid (if len > 0)
       */
      jam();
      nr_copy_delete_row(signal, regTcPtr, &regTcPtr.p->m_row_id, len);
    }

    /**
     * 2) Delete specified row at different rowid (if exists)
     */
    jam();
    nr_copy_delete_row(signal, regTcPtr, 0, 0);
    if (TRACENR_FLAG) TRACENR(" RUN op: " << op << endl);
    goto run;
  }

run:
  jam();
  exec_acckeyreq(signal, regTcPtr);
  return;

ignore:
  jam();
  ndbassert(!LqhKeyReq::getNrCopyFlag(regTcPtr.p->reqinfo));
update_gci_ignore:
  upgrade_to_exclusive_frag_access();
  regTcPtr.p->activeCreat = Fragrecord::AC_IGNORED;
  signal->theData[0] = regTcPtr.p->tupConnectrec;
  c_tup->do_tup_abortreq(signal, 0);
  jamEntryDebug();
  packLqhkeyreqLab(signal, tcConnectptr);
}

/**
 * Compare received key data with the data supplied
 * returning 0 if they are the same, 1 otherwise
 */
int Dblqh::compare_key(const TcConnectionrec *regTcPtr, const Uint32 *ptr,
                       Uint32 len) {
  ndbassert(len > 0);
  if (regTcPtr->keyInfoIVal == RNIL) return 1;

  const Uint32 tableId = regTcPtr->tableref;
  if (g_key_descriptor_pool.getPtr(tableId)->hasCharAttr) {
    /**
     * Need to do a collation aware compare.
     * Note that we could always have taken this code path.
     * However, doing a binary compare when possible, is likely more efficient.
     */
    jam();

    // Copy key into linear space.
    Uint64 reqKey[(MAX_KEY_SIZE_IN_WORDS + 1) >> 1];
    copy((Uint32 *)reqKey, regTcPtr->keyInfoIVal);

    return cmp_key(tableId, ptr, (Uint32 *)reqKey);
  } else {
    // A binary compare is sufficient.
    if (regTcPtr->primKeyLen != len) return 1;

    SectionReader keyInfoReader(regTcPtr->keyInfoIVal, getSectionSegmentPool());
    ndbassert(regTcPtr->primKeyLen == keyInfoReader.getSize());

    while (len != 0) {
      const Uint32 *keyChunk = nullptr;
      Uint32 chunkSize = 0;

      /* Get a ptr to a chunk of contiguous words to compare */
      bool ok = keyInfoReader.getWordsPtr(len, keyChunk, chunkSize);
      ndbrequire(ok);

      if (memcmp(ptr, keyChunk, chunkSize << 2)) return 1;

      ptr += chunkSize;
      len -= chunkSize;
    }
    return 0;
  }
}

void Dblqh::nr_copy_delete_row(Signal *signal, Ptr<TcConnectionrec> regTcPtr,
                               Local_key *rowid, Uint32 len) {
  Ptr<Fragrecord> fragPtr = fragptr;

  Uint32 tableId = regTcPtr.p->tableref;
  Uint32 siglen;

  prefetch_op_record_3((Uint32 *)regTcPtr.p->accConnectPtrP);

  Uint32 accreq = 0;
  accreq = AccKeyReq::setOperation(accreq, ZDELETE);
  accreq = AccKeyReq::setLockType(accreq, ZDELETE);
  accreq = AccKeyReq::setDirtyOp(accreq, false);
  accreq = AccKeyReq::setReplicaType(accreq, 0);  // ?
  accreq = AccKeyReq::setTakeOver(accreq, false);
  accreq = AccKeyReq::setLockReq(accreq, false);

  AccKeyReq *const req = reinterpret_cast<AccKeyReq *>(&signal->theData[0]);
  req->fragmentPtr = fragptr.p->accFragptr;
  req->requestInfo = accreq;
  req->transId1 = regTcPtr.p->transid[0];
  req->transId2 = regTcPtr.p->transid[1];
  req->lockConnectPtr = RNIL;

  if (rowid) {
    jam();
    if (g_key_descriptor_pool.getPtr(tableId)->hasCharAttr) {
      req->hashValue = calculateHash(tableId, signal->theData + 24);
    } else {
      req->hashValue = md5_hash(signal->theData + 24, len);
    }
    req->keyLen = 0;  // search by local key
    req->localKey[0] = rowid->m_page_no;
    req->localKey[1] = rowid->m_page_idx;
    siglen = AccKeyReq::SignalLength_localKey;
    static_assert(AccKeyReq::SignalLength_localKey == 10);
  } else {
    jam();
    Uint32 keylen = regTcPtr.p->primKeyLen;
    req->hashValue = regTcPtr.p->hashValue;
    req->keyLen = keylen;

    /* Copy KeyInfo inline into the ACCKEYREQ signal,
     * starting at word 7
     */
    copy(req->keyInfo, regTcPtr.p->keyInfoIVal);
    siglen = AccKeyReq::SignalLength_keyInfo + keylen;
    static_assert(AccKeyReq::SignalLength_keyInfo == 8);
  }
  signal->setLength(siglen);
  c_acc->execACCKEYREQ(signal, regTcPtr.p->accConnectrec,
                       regTcPtr.p->accConnectPtrP);
  jamEntry();

  Uint32 retValue = signal->theData[0];
  ndbrequire(retValue != RNIL);  // This should never block...

  if (retValue == (Uint32)-1) {
    /**
     * Only delete by pk, may fail
     */
    jam();
    ndbrequire(rowid == 0);
    c_acc->execACC_ABORTREQ(signal, regTcPtr.p->accConnectrec,
                            regTcPtr.p->accConnectPtrP, 0);
    jamEntry();
    return;
  }

  /**
   * We found row (and have it locked in ACC)
   */
  ndbrequire(regTcPtr.p->m_dealloc_state == TcConnectionrec::DA_IDLE);
  ndbrequire(regTcPtr.p->m_dealloc_data.m_dealloc_ref_count == RNIL);
  Local_key save = regTcPtr.p->m_row_id;

  c_acc->execACCKEY_ORD(signal, regTcPtr.p->accConnectrec,
                        regTcPtr.p->accConnectPtrP);
  c_acc->execACC_COMMITREQ(signal, regTcPtr.p->accConnectrec,
                           regTcPtr.p->accConnectPtrP);
  jamEntry();

  ndbrequire(regTcPtr.p->m_dealloc_state == TcConnectionrec::DA_DEALLOC_COUNT);
  ndbrequire(regTcPtr.p->m_dealloc_data.m_dealloc_ref_count == 1);
  int ret = c_tup->nr_delete(signal, regTcPtr.i, fragPtr.p->tupFragptr,
                             &regTcPtr.p->m_row_id, regTcPtr.p->gci_hi);
  jamEntry();

  if (ret) {
    ndbassert(ret == 1);
    ndbrequire(regTcPtr.p->activeCreat == Fragrecord::AC_NR_COPY);
    Uint32 pos = regTcPtr.p->m_nr_delete.m_cnt - 1;
    memcpy(regTcPtr.p->m_nr_delete.m_disk_ref + pos, signal->theData,
           sizeof(Local_key));
    regTcPtr.p->m_nr_delete.m_page_id[pos] = RNIL;
    regTcPtr.p->m_nr_delete.m_cnt = pos + 2;
    if (0)
      ndbout << "PENDING DISK DELETE: "
             << regTcPtr.p->m_nr_delete.m_disk_ref[pos] << endl;
  }

  TRACENR("DELETED: " << regTcPtr.p->m_row_id << endl);

  regTcPtr.p->m_dealloc_state = TcConnectionrec::DA_IDLE;
  regTcPtr.p->m_dealloc_data.m_dealloc_ref_count = RNIL;
  regTcPtr.p->m_row_id = save;
  fragptr = fragPtr;
}

void Dblqh::get_nr_op_info(Nr_op_info *op, Uint32 page_id) {
  Ptr<TcConnectionrec> tcPtr;
  tcPtr.i = op->m_ptr_i;

  ndbrequire(tcConnect_pool.getValidPtr(tcPtr));
  Ptr<Fragrecord> fragPtr;
  ndbrequire(c_fragment_pool.getPtr(fragPtr, tcPtr.p->fragmentptr));

  ndbassert(!m_is_in_query_thread);
  op->m_gci_hi = tcPtr.p->gci_hi;
  op->m_gci_lo = tcPtr.p->gci_lo;
  op->m_row_id = tcPtr.p->m_row_id;
  op->m_tup_frag_ptr_i = fragPtr.p->tupFragptr;

  ndbrequire(tcPtr.p->activeCreat == Fragrecord::AC_NR_COPY);
  ndbrequire(tcPtr.p->m_nr_delete.m_cnt);

  if (page_id == RNIL) {
    // get log buffer callback
    for (Uint32 i = 0; i < 2; i++) {
      if (tcPtr.p->m_nr_delete.m_page_id[i] != RNIL) {
        op->m_page_id = tcPtr.p->m_nr_delete.m_page_id[i];
        op->m_disk_ref = tcPtr.p->m_nr_delete.m_disk_ref[i];
        return;
      }
    }
  } else {
    // get page callback
    for (Uint32 i = 0; i < 2; i++) {
      Local_key key = tcPtr.p->m_nr_delete.m_disk_ref[i];
      if (op->m_disk_ref.m_page_no == key.m_page_no &&
          op->m_disk_ref.m_file_no == key.m_file_no &&
          tcPtr.p->m_nr_delete.m_page_id[i] == RNIL) {
        op->m_disk_ref = key;
        tcPtr.p->m_nr_delete.m_page_id[i] = page_id;
        return;
      }
    }
  }
  ndbabort();
}

void Dblqh::nr_delete_complete(Signal *signal, Nr_op_info *op) {
  jamEntry();
  Ptr<TcConnectionrec> tcPtr;
  tcPtr.i = op->m_ptr_i;
  ndbrequire(tcConnect_pool.getValidPtr(tcPtr));

  ndbrequire(tcPtr.p->activeCreat == Fragrecord::AC_NR_COPY);
  ndbrequire(tcPtr.p->m_nr_delete.m_cnt);

  tcPtr.p->m_nr_delete.m_cnt--;
  if (tcPtr.p->m_nr_delete.m_cnt == 0) {
    jam();
    const TcConnectionrecPtr tcConnectptr = tcPtr;
    ndbrequire(c_fragment_pool.getPtr(fragptr, tcPtr.p->fragmentptr));

    if (tcPtr.p->abortState != TcConnectionrec::ABORT_IDLE) {
      jam();
      tcPtr.p->activeCreat = Fragrecord::AC_NORMAL;
      abortCommonLab(signal, tcConnectptr);
    } else if (tcPtr.p->operation == ZDELETE &&
               LqhKeyReq::getNrCopyFlag(tcPtr.p->reqinfo)) {
      /**
       * This is run directly in handle_nr_copy
       */
      jam();
      packLqhkeyreqLab(signal, tcConnectptr);
    } else {
      jam();
      rwConcludedLab(signal, tcConnectptr);
    }
    return;
  }

  if (memcmp(&tcPtr.p->m_nr_delete.m_disk_ref[0], &op->m_disk_ref,
             sizeof(Local_key)) == 0) {
    jam();
    ndbassert(tcPtr.p->m_nr_delete.m_page_id[0] != RNIL);
    tcPtr.p->m_nr_delete.m_page_id[0] = tcPtr.p->m_nr_delete.m_page_id[1];
    tcPtr.p->m_nr_delete.m_disk_ref[0] = tcPtr.p->m_nr_delete.m_disk_ref[1];
  }
}

/**
 * getKeyInfoWordOrZero
 * Get given word of KeyInfo, or zero if it's not available
 * Used for tracing
 */
Uint32 Dblqh::getKeyInfoWordOrZero(const TcConnectionrec *regTcPtr,
                                   Uint32 offset) {
  if (regTcPtr->keyInfoIVal != RNIL) {
    SectionReader keyInfoReader(regTcPtr->keyInfoIVal, g_sectionSegmentPool);

    if (keyInfoReader.getSize() > offset) {
      if (offset) keyInfoReader.step(offset);

      Uint32 word;
      keyInfoReader.getWord(&word);
      return word;
    }
  }
  return 0;
}

void Dblqh::unlockError(Signal *signal, Uint32 error,
                        const TcConnectionrecPtr tcConnectptr) {
  terrorCode = error;
  abortErrorLab(signal, tcConnectptr);
}

/**
 * handleUserUnlockRequest
 *
 * This method handles an LQHKEYREQ unlock request from
 * TC.
 */
void Dblqh::handleUserUnlockRequest(Signal *signal,
                                    TcConnectionrecPtr tcConnectptr) {
  jam();
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  Uint32 tcPtrI = tcConnectptr.i;

  /* Request to unlock (abort) an existing read operation
   *
   * 1) Get user's LOCK_REF from KeyInfo
   *
   * 2) Lookup TC_OP_REF in hash
   *
   * 3) Check state of found op : TransId, state, type, lock
   *
   * 4) Check op_id portion
   *
   * 5) Abort locking op in ACC
   *
   * 6) Clean up locking op in LQH
   *
   * 7) Send LQHKEYCONF to TC for user unlock op
   *
   * 8) Clean up user unlock op
   */
  if (unlikely(regTcPtr->primKeyLen != LqhKeyReq::UnlockKeyLen)) {
    jam();
    unlockError(signal, 4109,
                tcConnectptr); /* Faulty primary key attribute length */
    return;
  }

  SectionReader keyInfoReader(regTcPtr->keyInfoIVal, getSectionSegmentPool());

  ndbrequire(keyInfoReader.getSize() == regTcPtr->primKeyLen);

  /* Extract components of user lock reference */
  Uint32 tcOpRecIndex;
  Uint32 lqhOpIdWord;
  ndbrequire(keyInfoReader.getWord(&tcOpRecIndex));  // Locking op TC index
  ndbrequire(keyInfoReader.getWord(&lqhOpIdWord));  // Part of Locking op LQH id

  /* Use TC operation record index to find the operation record
   * This requires that this operation and the referenced
   * operation are part of the same transaction.
   * On success this sets tcConnectptr.i and .p to the
   * operation-to-unlock's record.
   */
  if (unlikely(findTransaction(regTcPtr->transid[0], regTcPtr->transid[1],
                               tcOpRecIndex, regTcPtr->tcHashKeyHi, true, false,
                               tcConnectptr) != ZOK)) {
    jam();
    unlockError(signal, ZBAD_OP_REF, tcConnectptr);
    return;
  }

  TcConnectionrec *const regLockTcPtr = tcConnectptr.p;

  /* Validate that the bottom 32-bits of the operation id reference
   * we were given are in alignment
   */
  Uint32 lockOpKeyReqId = (Uint32)regLockTcPtr->lqhKeyReqId;
  if (unlikely(lockOpKeyReqId != lqhOpIdWord)) {
    jam();
    unlockError(signal, ZBAD_OP_REF, tcConnectptr);
    return;
  }

  /* Validate the state of the locking operation */
  bool lockingOpValid =
      ((regLockTcPtr->operation == ZREAD) &&
       // ZREAD_EX mapped to ZREAD above
       (!regLockTcPtr->dirtyOp) && (!regLockTcPtr->opSimple) &&
       ((regLockTcPtr->lockType == ZREAD) ||  // LM_Read
        (regLockTcPtr->lockType == ZUPDATE))  // LM_Exclusive
       && (regLockTcPtr->transactionState == TcConnectionrec::PREPARED) &&
       (regLockTcPtr->commitAckMarker == RNIL) &&
       // No commit ack marker
       (regLockTcPtr->logWriteState ==
        TcConnectionrec::NOT_STARTED));  // No log written

  if (unlikely(!lockingOpValid)) {
    jam();
    unlockError(signal, ZBAD_UNLOCK_STATE, tcConnectptr);
    return;
  }

  /* Ok, now we're ready to start 'aborting' this operation, to get the
   * effect of unlocking it
   */
  c_acc->execACC_ABORTREQ(signal, regLockTcPtr->accConnectrec,
                          regLockTcPtr->accConnectPtrP, 0);
  jamEntry();

  /* Would be nice to handle non-success case somehow */
  ndbrequire(signal->theData[1] == 0);

  /* Now we want to release LQH resources associated with the
   * locking operation
   */
  cleanUp(signal, tcConnectptr);

  /* Now that the locking operation has been 'disappeared', we need to
   * send an LQHKEYCONF for the unlock operation and then 'disappear' it
   * as well
   */
  tcConnectptr.i = tcPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));

  ndbrequire(regTcPtr == tcConnectptr.p);

  /* Set readlenAi to the unlocked operation's TC operation ref */
  regTcPtr->readlenAi = tcOpRecIndex;

  /* Clear number of fired triggers */
  regTcPtr->numFiredTriggers = 0;

  /* Now send the LQHKEYCONF to TC */
  sendLqhkeyconfTc(signal, regTcPtr->tcBlockref, tcConnectptr);

  /* Finally, clean up the unlock operation itself */
  cleanUp(signal, tcConnectptr);

  return;
}

/**
 * TUPle deallocation
 *
 * ACC informs LQH via TUP_DEALLOCREQ when a TUPle (ROWID)
 * is no longer needed by active ACC operations (Key ops,
 * scans with locks etc)
 * LQH then informs TUP via TUP_DEALLOCREQ when TUP should
 * release the storage, making it available for some other
 * insert to the fragment.
 *
 * It is important that ROWIDs are released on Backup
 * replicas before the Primary replica.
 *
 * To ensure this:
 *   ACC :
 *   - Informs LQH of each of the operations involved in releasing
 *     a ROWID at commit time
 *   - Informs LQH when the last operation involved in releasing
 *     a ROWID has committed
 *
 *   LQH :
 *   - Tracks these details as a reference count on one of the
 *     operations involved in releasing the ROWID (the dealloc op).
 *   - Marks the other operations to point to the dealloc op
 *   - Decrements the reference count when :
 *       - ACC informs LQH that the last involved operation
 *         has committed
 *       - Referring operations complete locally
 *       - The dealloc op completes locally
 *   - When the reference count hits zero, TUP is told to release
 *     the row storage
 *
 * Using a reference count avoids problems with premature release when
 * the order of completion of the involved operations varies.
 *
 * Since the dealloc operation can complete before the ref count hits
 * zero, it supports a 'zombie' state where it is not yet deallocated
 * as it is hosting a count
 */

/**
 * incrDeallocRefCount
 *
 * Called when ACC notifies LQH of operations involved in TUPle
 * deallocation
 */
void Dblqh::incrDeallocRefCount(Signal *signal, Uint32 opPtrI,
                                Uint32 countOpPtrI) {
  jam();
  ndbrequire(opPtrI != RNIL);
  ndbrequire(countOpPtrI != RNIL);

  TcConnectionrecPtr opPtr;
  opPtr.i = opPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(opPtr));

  TcConnectionrecPtr countOpPtr;
  countOpPtr.i = countOpPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(countOpPtr));

  const bool referring_op = (opPtrI != countOpPtrI);

  if (referring_op) {
    jam();
    ndbrequire(opPtr.p->m_dealloc_state == TcConnectionrec::DA_IDLE);
    ndbrequire(opPtr.p->m_dealloc_data.m_dealloc_ref_count == RNIL);
    opPtr.p->m_dealloc_state = TcConnectionrec::DA_DEALLOC_REFERENCE;
    opPtr.p->m_dealloc_data.m_dealloc_op_id = countOpPtr.i;
  }

  if (countOpPtr.p->m_dealloc_state == TcConnectionrec::DA_IDLE) {
    jam();
    ndbrequire(countOpPtr.p->m_dealloc_data.m_dealloc_op_id == RNIL);

    // init count to 1 so that final refcount = op count + 1
    // this ensures that dealloc cannot happen until ACC sends
    // additional signal to decrement refcount to 0.
    countOpPtr.p->m_dealloc_state = TcConnectionrec::DA_DEALLOC_COUNT;
    countOpPtr.p->m_dealloc_data.m_dealloc_ref_count = 1;
  }

  /* Increment count */
  ndbrequire(countOpPtr.p->m_dealloc_state ==
                 TcConnectionrec::DA_DEALLOC_COUNT ||
             countOpPtr.p->m_dealloc_state ==
                 TcConnectionrec::DA_DEALLOC_COUNT_ZOMBIE);
  ndbrequire(countOpPtr.p->m_dealloc_data.m_dealloc_ref_count != RNIL);

  countOpPtr.p->m_dealloc_data.m_dealloc_ref_count++;
}

/**
 * decrDeallocRefCount
 *
 * Called when ACC triggers deallocation, and when involved
 * operations complete.
 *
 * Returns the new count of references on the rowID
 */
Uint32 Dblqh::decrDeallocRefCount(Signal *signal, Uint32 opPtrI) {
  jam();
  ndbrequire(opPtrI != RNIL);

  TcConnectionrecPtr opPtr;
  opPtr.i = opPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(opPtr));

  TcConnectionrecPtr countOpPtr = opPtr;

  if (opPtr.p->m_dealloc_state == TcConnectionrec::DA_DEALLOC_REFERENCE) {
    jam();
    ndbrequire(opPtr.p->m_dealloc_data.m_dealloc_op_id != RNIL);
    countOpPtr.i = opPtr.p->m_dealloc_data.m_dealloc_op_id;
    ndbrequire(tcConnect_pool.getValidPtr(countOpPtr));
  }

  ndbrequire(countOpPtr.p->m_dealloc_state ==
                 TcConnectionrec::DA_DEALLOC_COUNT ||
             countOpPtr.p->m_dealloc_state ==
                 TcConnectionrec::DA_DEALLOC_COUNT_ZOMBIE);
  ndbrequire(countOpPtr.p->m_dealloc_data.m_dealloc_ref_count != RNIL);
  ndbrequire(countOpPtr.p->m_dealloc_data.m_dealloc_ref_count > 0);

  const Uint32 newCount = --countOpPtr.p->m_dealloc_data.m_dealloc_ref_count;

  if (newCount == 0) {
    jam();
#if defined(VM_TRACE) || defined(ERROR_INSERT)
    /* Verify that fragment used is a primary table fragment */
    FragrecordPtr fragPtr;
    ndbrequire(c_fragment_pool.getPtr(fragPtr, countOpPtr.p->fragmentptr));
    ndbrequire(fragPtr.p->tableFragptr == fragPtr.i);
#endif
    /**
     * If we are running a SCAN query, decrease the amount of rows to
     * scan during the scan.
     */
    m_scan_direct_count += 8;
    /* Dealloc TUPle now */
    signal->theData[0] = countOpPtr.p->fragmentid;
    signal->theData[1] = countOpPtr.p->tableref;
    signal->theData[2] = countOpPtr.p->m_row_id.m_page_no;
    signal->theData[3] = countOpPtr.p->m_row_id.m_page_idx;
    signal->theData[4] = RNIL;

    ndbassert(!m_is_in_query_thread);
    c_tup->execTUP_DEALLOCREQ(signal);

    bool countOpIsZombie = (countOpPtr.p->m_dealloc_state ==
                            TcConnectionrec::DA_DEALLOC_COUNT_ZOMBIE);

    countOpPtr.p->m_dealloc_state = TcConnectionrec::DA_IDLE;
    countOpPtr.p->m_dealloc_data.m_dealloc_ref_count = RNIL;

    if (countOpIsZombie) {
      jamDebug();
      /**
       * Op was not released during COMPLETE because it is a zombie dealloc op
       * Release it now.
       */
      releaseTcrec(signal, countOpPtr);
    }
  }

  return newCount;
}

/**
 * handleDeallocOp
 *
 * Called when LQH is releasing an operation record which has
 * a non idle m_dealloc_state state
 *
 * The relevant counting operation is found, the deallocation
 * ref count is decremented, the tuple is freed if the count
 * hits zero, and the counting op is freed if it is a zombie.
 *
 * If the counting op is released before the count is zero
 * it becomes a zombie.
 */
void Dblqh::handleDeallocOp(Signal *signal, TcConnectionrecPtr regTcPtr) {
  jam();

  ndbrequire(regTcPtr.p->m_dealloc_state != TcConnectionrec::DA_IDLE);
  const bool referringOpReleased =
      (regTcPtr.p->m_dealloc_state == TcConnectionrec::DA_DEALLOC_REFERENCE);

  const Uint32 newCount = decrDeallocRefCount(signal, regTcPtr.i);

  if (referringOpReleased) {
    jam();
    regTcPtr.p->m_dealloc_state = TcConnectionrec::DA_IDLE;
    regTcPtr.p->m_dealloc_data.m_dealloc_op_id = RNIL;
  } else {
    jam();
    // The dealloc op hosts the refcount, so it cannot be released until
    // the refcount reaches zero. Mark the dealloc op as a zombie op to
    // prevent its release.
    if (newCount != 0) {
      jam();
      regTcPtr.p->m_dealloc_state = TcConnectionrec::DA_DEALLOC_COUNT_ZOMBIE;
    }
  }
}

/**
 * execTUP_DEALLOCREQ
 *
 * Receive notification from ACC that a TUPle is no longer needed.
 * This can be due to transaction COMMIT or ABORT.
 * ACC informs LQH of each operation involved in deallocation
 * so that LQH can determine when it is safe to ask TUP to release
 * the storage, even when operations complete in an arbitrary order.
 *
 * 3 cases :
 *   i)   Op notification : theData[4] != RNIL, theData[5] != RNIL
 *   ii)  Trigger         : theData[4] != RNIL, theData[5] == RNIL
 *   iii) Immediate       : theData[4] == RNIL
 *
 * For transaction commit resulting in deallocation, there will be
 * one or more invocations of i) Notification, followed by one
 * invocation of ii) Trigger
 *
 * For transaction abort, there will either be :
 *   One invocation of ii)  Trigger
 *     or
 *   One invocation of iii) Immediate
 *
 * From LQH's point of view :
 * 1) ACC informs LQH of a set of operations involved in deallocating a tuple
 *   - LQH should not release the tuple before *all* of those operations are
 *     complete
 *   - For each tuple + set of operations involved in deallocating it, ACC
 *     informs LQH of a single designated 'counting op' which is a member of
 *     the set and which might help LQH track the state of the set + the tuple
 * 2) ACC still requires that LQH does not deallocate the tuple before
 *    ACC gives permission to do so (in the Trigger notification)
 * 3) Special cases (aborts) :
 *    Either :
 *    - Just a single trigger notification - LQH should deallocate on completion
 *      of this operation.
 *    - Just a single immediate notification - LQH should deallocate immediately
 *      on receiving this signal.
 *
 * LQH : dealloc_iff ((All notified ops complete && Trigger from ACC received)
 * || (Immediate deallocation requested))
 */
void Dblqh::execTUP_DEALLOCREQ(Signal *signal) {
  jamEntryDebug();

  if (TRACENR_FLAG) {
    Local_key tmp;
    tmp.m_page_no = signal->theData[2];
    tmp.m_page_idx = signal->theData[3];

    TRACENR("TUP_DEALLOC: "
            << tmp
            << (signal->theData[4] == RNIL
                    ? " IMMEDIATE"
                    : (signal->theData[5] == RNIL ? " TRIGGER"
                                                  : "NOTIFICATION"))
            << endl);
  }

  if (signal->theData[4] == RNIL) {
    /* Immediate Dealloc Request, used for abort of insert
     * triggered by commit of scan op.
     * No delete or counting operation specified.
     */
    jam();
    ndbrequire(signal->theData[5] == RNIL);
    /* FragId, TableRef, PageNo + PageIdx set by ACC, pass on */
    c_tup->execTUP_DEALLOCREQ(signal);
    return;
  }

  TcConnectionrecPtr regTcPtr;

  regTcPtr.i = signal->theData[4];
  ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));

  regTcPtr.p->m_row_id.m_page_no = signal->theData[2];
  regTcPtr.p->m_row_id.m_page_idx = signal->theData[3];

  if (signal->theData[5] != RNIL) {
    jam();
    /**
     * Notification of Op involved in deallocation
     */
    incrDeallocRefCount(signal, signal->theData[4], signal->theData[5]);
  } else {
    /**
     * Deallocation triggered by ACC
     * Now LQH decides what to do next.
     */
    if (regTcPtr.p->m_dealloc_state == TcConnectionrec::DA_IDLE) {
      jam();
      /**
       * ABORT case
       * Set count to 1 for quick dealloc on complete of this op
       */
      ndbrequire(regTcPtr.p->m_dealloc_data.m_dealloc_ref_count == RNIL);
      regTcPtr.p->m_dealloc_state = TcConnectionrec::DA_DEALLOC_COUNT;
      /* Init count to 1 as we are processing report_dealloc*/
      regTcPtr.p->m_dealloc_data.m_dealloc_ref_count = 1;
    } else {
      jam();
      /**
       * [Multi-op] COMMIT case
       * Decrement count to indicate 'permission to dealloc'
       * from ACC
       */
      decrDeallocRefCount(signal, signal->theData[4]);
    }
  }
}  // Dblqh::execTUP_DEALLOCREQ()

/* ************>> */
/*  ACCKEYCONF  > */
/* ************>> */
void Dblqh::execACCKEYCONF(Signal *signal) {
  jamEntry();
  ndbassert(!m_is_query_block);
  if (ERROR_INSERTED(5095)) {
    jam();
    g_eventLogger->info("LQH %u : ERRINS 5095 Delaying ACCKEYCONF", instance());
    sendSignalWithDelay(reference(), GSN_ACCKEYCONF, signal, 10, 5);
    return;
  }

  if (ERROR_INSERTED(5094)) {
    jam();
    g_eventLogger->info(
        "LQH %u : ERRINS 5094 Passing ACCKEYCONF 1 and setting ERRINS 5095",
        instance());
    SET_ERROR_INSERT_VALUE(5095);
  }

  if (ERROR_INSERTED(5096)) {
    jam();
    g_eventLogger->info(
        "LQH %u : ERRINS 5096 set when processing ACCKEYCONF, clearing",
        instance());
    CLEAR_ERROR_INSERT_VALUE;
    // SET_ERROR_INSERT_VALUE(5097); // Kill on scan
  }

  Uint32 tcIndex = signal->theData[0];

  setup_key_pointers(tcIndex);
  Uint32 localKey1 = signal->theData[3];
  Uint32 localKey2 = signal->theData[4];
  TcConnectionrec *const regTcPtr = m_tc_connect_ptr.p;

  if (regTcPtr->transactionState != TcConnectionrec::WAIT_ACC) {
    jam();
    LQHKEY_abort(signal, 3, m_tc_connect_ptr);
  } else if (unlikely(c_acc->checkOpPendingAbort(regTcPtr->accConnectrec))) {
    jam();
    /* Wait for Abort */
  } else {
    c_tup->prepareTUPKEYREQ(localKey1, localKey2, fragptr.p->tupFragptr);
    continueACCKEYCONF(signal, localKey1, localKey2, m_tc_connect_ptr);
  }
  release_frag_access(fragptr.p);
}

void Dblqh::continueACCKEYCONF(Signal *signal, Uint32 localKey1,
                               Uint32 localKey2,
                               const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  /* ------------------------------------------------------------------------
   * IT IS NOW TIME TO CONTACT THE TUPLE MANAGER. THE TUPLE MANAGER NEEDS THE
   * INFORMATION ON WHICH TABLE AND FRAGMENT, THE LOCAL KEY AND IT NEEDS TO
   * KNOW THE TYPE OF OPERATION TO PERFORM. TUP CAN SEND THE ATTRINFO DATA
   * EITHER TO THE TC BLOCK OR DIRECTLY TO THE APPLICATION. THE SCHEMA VERSION
   * IS NEEDED SINCE TWO SCHEMA VERSIONS CAN BE ACTIVE SIMULTANEOUSLY ON A
   * TABLE.
   * ----------------------------------------------------------------------- */
  if (unlikely(regTcPtr->operation == ZWRITE)) {
    ndbassert(regTcPtr->seqNoReplica == 0 ||
              regTcPtr->activeCreat == Fragrecord::AC_NR_COPY);
    Uint32 op = signal->theData[1];
    Uint32 requestInfo = regTcPtr->reqinfo;
    if (likely(op == ZINSERT || op == ZUPDATE)) {
      jam();
      regTcPtr->operation = op;
    } else {
      jam();
      warningEvent("Converting %d to ZUPDATE", op);
      op = regTcPtr->operation = ZUPDATE;
    }
    if (regTcPtr->seqNoReplica == 0) {
      jam();
      requestInfo &=
          ~(LqhKeyReq::RI_OPERATION_MASK << LqhKeyReq::RI_OPERATION_SHIFT);
      LqhKeyReq::setOperation(requestInfo, op);
      regTcPtr->reqinfo = requestInfo;
    }
  }  // if

  /* ------------------------------------------------------------------------
   * IT IS NOW TIME TO CONTACT THE TUPLE MANAGER. THE TUPLE MANAGER NEEDS THE
   * INFORMATION ON WHICH TABLE AND FRAGMENT, THE LOCAL KEY AND IT NEEDS TO
   * KNOW THE TYPE OF OPERATION TO PERFORM. TUP CAN SEND THE ATTRINFO DATA
   * EITHER TO THE TC BLOCK OR DIRECTLY TO THE APPLICATION. THE SCHEMA VERSION
   * IS NEEDED SINCE TWO SCHEMA VERSIONS CAN BE ACTIVE SIMULTANEOUSLY ON A
   * TABLE.
   * ----------------------------------------------------------------------- */
  Fragrecord *regFragptr = fragptr.p;
  if (!regTcPtr->m_disk_table) {
    jamDebug();
    acckeyconf_tupkeyreq(signal, regTcPtr, regFragptr, localKey1, localKey2,
                         RNIL);
  } else {
    jamDebug();
    ndbassert(!m_is_query_block);
    acckeyconf_load_diskpage(signal, tcConnectptr, regFragptr, localKey1,
                             localKey2);
  }
}

void Dblqh::acckeyconf_tupkeyreq(Signal *signal, TcConnectionrec *regTcPtr,
                                 Fragrecord *regFragptrP, Uint32 page_no,
                                 Uint32 page_idx, Uint32 disk_page) {
  /* ------------------------------------------------------------------------
   * IT IS NOW TIME TO CONTACT THE TUPLE MANAGER. THE TUPLE MANAGER NEEDS THE
   * INFORMATION ON WHICH TABLE AND FRAGMENT, THE LOCAL KEY AND IT NEEDS TO
   * KNOW THE TYPE OF OPERATION TO PERFORM. TUP CAN SEND THE ATTRINFO DATA
   * EITHER TO THE TC BLOCK OR DIRECTLY TO THE APPLICATION. THE SCHEMA VERSION
   * IS NEEDED SINCE TWO SCHEMA VERSIONS CAN BE ACTIVE SIMULTANEOUSLY ON A
   * TABLE.
   * ----------------------------------------------------------------------- */
  TupKeyReq *const tupKeyReq = (TupKeyReq *)signal->getDataPtrSend();
  Uint32 Ttupreq = 0;
  Uint32 row_page_no = regTcPtr->m_row_id.m_page_no;
  Uint32 row_page_idx = regTcPtr->m_row_id.m_page_idx;
  Uint32 use_rowid = regTcPtr->m_use_rowid;
  Uint32 interpreted_exec = regTcPtr->opExec;

  tupKeyReq->keyRef1 = page_no;
  tupKeyReq->keyRef2 = page_idx;
  tupKeyReq->disk_page = disk_page;
  tupKeyReq->m_row_id_page_no = row_page_no;
  tupKeyReq->m_row_id_page_idx = row_page_idx;
  TupKeyReq::setRowidFlag(Ttupreq, use_rowid);
  TupKeyReq::setInterpretedFlag(Ttupreq, interpreted_exec);
  tupKeyReq->request = Ttupreq;

  TRACE_OP(regTcPtr, "TUPKEYREQ");

  const Uint32 totReclenAi = regTcPtr->totReclenAi;
  const Uint32 attrInfoIVal = regTcPtr->attrInfoIVal;
  const Uint32 op = regTcPtr->operation;
  regTcPtr->m_row_id.m_page_no = page_no;
  regTcPtr->m_row_id.m_page_idx = page_idx;
  regTcPtr->transactionState = TcConnectionrec::WAIT_TUP;
  regTcPtr->m_use_rowid |= (op == ZINSERT || op == ZREFRESH);
  /* ---------------------------------------------------------------------
   * Clear interpreted mode bit since we do not want the next replica to
   * use interpreted mode. The next replica will receive a normal write.
   * --------------------------------------------------------------------- */
  regTcPtr->opExec = 0;

#ifdef ERROR_INSERT
  /* Ensure c_executing_redo_log isn't set when a read happens */
  ndbrequire((op != ZREAD && op != ZREAD_EX) || c_executing_redo_log == 0);
#endif
  /* Pass AttrInfo section if available in the TupKeyReq signal
   * We are still responsible for releasing it, TUP is just
   * borrowing it
   */
  if (totReclenAi > 0) {
    ndbassert(attrInfoIVal != RNIL);
    c_tup->copyAttrinfo(totReclenAi, attrInfoIVal);
  }
#ifdef VM_TRACE
  tupKeyReq->fragPtr = regFragptrP->tupFragptr;
#endif
  if (likely(c_tup->execTUPKEYREQ(signal, regTcPtr, nullptr))) {
    execTUPKEYCONF(signal);
    return;
  } else {
    execTUPKEYREF(signal);
    return;
  }
}

void Dblqh::acckeyconf_load_diskpage(Signal *signal,
                                     TcConnectionrecPtr regTcPtr,
                                     Fragrecord *regFragptrP, Uint32 lkey1,
                                     Uint32 lkey2) {
  int res;
  Uint32 disk_flag = regTcPtr.p->operation;
  disk_flag +=
      (LqhKeyReq::getNrCopyFlag(regTcPtr.p->reqinfo) | c_executing_redo_log)
          ? Page_cache_client::COPY_FRAG
          : 0;
  if ((res = c_tup->load_diskpage(signal, regTcPtr.p->tupConnectrec,
                                  regFragptrP->tupFragptr, lkey1, lkey2,
                                  disk_flag)) > 0) {
    jamDebug();
    acckeyconf_tupkeyreq(signal, regTcPtr.p, regFragptrP, lkey1, lkey2, res);
  } else if (res == 0) {
    jamDebug();
    regTcPtr.p->transactionState = TcConnectionrec::WAIT_TUP;
    DEB_COPY(("(%u)get_page returned 0 for %u", instance(), regTcPtr.i));
    regTcPtr.p->m_row_id.m_page_no = lkey1;
    regTcPtr.p->m_row_id.m_page_idx = lkey2;
  } else {
    jamDebug();
    regTcPtr.p->transactionState = TcConnectionrec::WAIT_TUP;
    TupKeyRef *ref = (TupKeyRef *)signal->getDataPtr();
    ref->userRef = regTcPtr.i;
    if (res == -1) {
      jam();
      DEB_COPY(("(%u)get_page returned with -1", instance()));
      ref->errorCode = ~0;
    } else {
      jam();
      ref->errorCode = -res;
      DEB_COPY(("(%u)get_page returned with %d", instance(), -res));
    }
    execTUPKEYREF(signal);
    return;
  }
}

void Dblqh::acckeyconf_load_diskpage_callback(Signal *signal,
                                              Uint32 callbackData,
                                              Uint32 disk_page) {
  jamEntry();
  ndbassert(!m_is_query_block);
  setup_key_pointers(callbackData);
  TcConnectionrecPtr tcConnectptr = m_tc_connect_ptr;
  FragrecordPtr fragPtr = fragptr;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  TcConnectionrec::TransactionState state = regTcPtr->transactionState;
  if (likely(disk_page > 0 && state == TcConnectionrec::WAIT_TUP)) {
    /**
     * We have returned from a real-time break, we need to set
     * up the proper pointers for a key execution.
     */
    jam();
    c_tup->prepareTUPKEYREQ(regTcPtr->m_row_id.m_page_no,
                            regTcPtr->m_row_id.m_page_idx,
                            fragPtr.p->tupFragptr);
    acckeyconf_tupkeyreq(signal, regTcPtr, fragPtr.p,
                         regTcPtr->m_row_id.m_page_no,
                         regTcPtr->m_row_id.m_page_idx, disk_page);
  } else if (state != TcConnectionrec::WAIT_TUP) {
    ndbrequire(state == TcConnectionrec::WAIT_TUP_TO_ABORT);
    TupKeyRef *ref = (TupKeyRef *)signal->getDataPtr();
    ref->userRef = callbackData;
    ref->errorCode = disk_page;
    execTUPKEYREF(signal);
  } else {
    TupKeyRef *ref = (TupKeyRef *)signal->getDataPtr();
    ref->userRef = callbackData;
    ref->errorCode = disk_page;
    execTUPKEYREF(signal);
  }
  release_frag_access(fragptr.p);
}

/* --------------------------------------------------------------------------
 * -------                       ENTER TUP...                         -------
 * ENTER TUPKEYCONF WITH
 *           TC_CONNECTPTR,
 *           TDATA2,     LOCAL KEY REFERENCE 1, ONLY INTERESTING AFTER INSERT
 *           TDATA3,     LOCAL KEY REFERENCE 1, ONLY INTERESTING AFTER INSERT
 *           TDATA4,     TOTAL LENGTH OF READ DATA SENT TO TC/APPLICATION
 *           TDATA5      TOTAL LENGTH OF UPDATE DATA SENT TO/FROM TUP
 *        GOTO TUPKEY_CONF
 *
 *  TAKE CARE OF RESPONSES FROM TUPLE MANAGER.
 * -------------------------------------------------------------------------- */
void Dblqh::tupkeyConfLab(Signal *signal,
                          const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;

  /* ---- GET OPERATION TYPE AND CHECK WHAT KIND OF OPERATION IS REQUESTED ---
   */
  const TupKeyConf *const tupKeyConf = (TupKeyConf *)&signal->theData[0];
  Uint32 activeCreat = regTcPtr->activeCreat;
  Uint32 readLen = tupKeyConf->readLength;
  Uint32 writeLen = tupKeyConf->writeLength;

  TRACE_OP(regTcPtr, "TUPKEYCONF");

  c_acc->execACCKEY_ORD(signal, regTcPtr->accConnectrec,
                        regTcPtr->accConnectPtrP);

  jamEntryDebug();
  if (readLen != 0) {
    jamDebug();

    /* SET BIT 15 IN REQINFO */
    LqhKeyReq::setApplicationAddressFlag(regTcPtr->reqinfo, 1);
    regTcPtr->readlenAi = readLen;
  }  // if

  if (regTcPtr->operation == ZREAD &&
      (regTcPtr->opSimple || regTcPtr->dirtyOp)) {
    jamDebug();
    /* ----------------------------------------------------------------------
     * THE OPERATION IS A SIMPLE READ.
     * WE WILL IMMEDIATELY COMMIT THE OPERATION.
     * SINCE WE HAVE NOT RELEASED THE FRAGMENT LOCK
     * (FOR LOCAL CHECKPOINTS) YET
     * WE CAN GO IMMEDIATELY TO COMMIT_CONTINUE_AFTER_BLOCKED.
     * WE HAVE ALREADY SENT THE RESPONSE SO WE ARE NOT INTERESTED IN
     * READ LENGTH
     * --------------------------------------------------------------------- */
    commitContinueAfterBlockedLab(signal, tcConnectptr);
    return;
  }  // if

  ndbassert(!m_is_in_query_thread);
  regTcPtr->totSendlenAi = writeLen;
  /* We will propagate / log writeLen words
   * Check that that is how many we have available to
   * propagate
   */
  ndbassert(regTcPtr->totSendlenAi == regTcPtr->currTupAiLen);

  if (unlikely(activeCreat == Fragrecord::AC_NR_COPY)) {
    jam();
    ndbrequire(regTcPtr->m_nr_delete.m_cnt);
    regTcPtr->m_nr_delete.m_cnt--;
    if (regTcPtr->m_nr_delete.m_cnt) {
      jam();
      /**
       * Let operation wait for pending NR operations
       *   even for before writing log...(as it's simpler)
       */

#ifdef VM_TRACE
      /**
       * Only disk table can have pending ops...
       */
      TablerecPtr tablePtr;
      tablePtr.i = regTcPtr->tableref;
      ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);
      ndbrequire(tablePtr.p->m_disk_table);
#endif

      return;
    }
  }
  rwConcludedLab(signal, tcConnectptr);
  return;
}  // Dblqh::tupkeyConfLab()

void Dblqh::sendBatchedLqhkeyreq(Signal *signal, Uint32 lqhRef, Uint32 siglen,
                                 SectionHandle *handle) {
  jam();
  const Uint32 version = getNodeInfo(refToNode(lqhRef)).m_version;
  if (ndbd_frag_lqhkeyreq(version)) {
    jam();
    sendBatchedFragmentedSignal(lqhRef, GSN_LQHKEYREQ, signal, siglen, JBB,
                                handle, false);
  } else {
    jam();
    sendSignal(lqhRef, GSN_LQHKEYREQ, signal, siglen, JBB, handle);
  }
}

/* --------------------------------------------------------------------------
 *     THE CODE IS FOUND IN THE SIGNAL RECEPTION PART OF LQH
 * -------------------------------------------------------------------------- */
void Dblqh::rwConcludedLab(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  /* -----------------------------------------------------------------------
   *  WE HAVE NOW CONCLUDED READING/WRITING IN ACC AND TUP FOR THIS OPERATION.
   *  IT IS NOW TIME TO LOG THE OPERATION, SEND REQUEST TO NEXT NODE OR TC AND
   *  FOR SOME TYPES OF OPERATIONS IT IS EVEN TIME TO COMMIT THE OPERATION.
   * ---------------------------------------------------------------------- */
  if (regTcPtr->operation == ZREAD) {
    jamDebug();
    /* ---------------------------------------------------------------------
     * A NORMAL READ OPERATION IS NOT LOGGED BUT IS NOT COMMITTED UNTIL THE
     * COMMIT SIGNAL ARRIVES. THUS WE CONTINUE PACKING THE RESPONSE.
     * -------------------------------------------------------------------- */
    packLqhkeyreqLab(signal, tcConnectptr);
    return;
  } else {
    FragrecordPtr regFragptr = fragptr;
    if (regFragptr.p->logFlag == Fragrecord::STATE_FALSE) {
      if (unlikely(regTcPtr->dirtyOp == ZTRUE)) {
        jam();
        /* -----------------------------------------------------------------
         * THIS OPERATION WAS A WRITE OPERATION THAT DO NOT NEED LOGGING AND
         * THAT CAN CAN  BE COMMITTED IMMEDIATELY.
         * ---------------------------------------------------------------- */
        commitContinueAfterBlockedLab(signal, tcConnectptr);
        return;
      } else {
        jamDebug();
        /* -----------------------------------------------------------------
         * A NORMAL WRITE OPERATION ON A FRAGMENT WHICH DO NOT NEED LOGGING.
         * WE WILL PACK THE REQUEST/RESPONSE TO THE NEXT NODE/TO TC.
         * ---------------------------------------------------------------- */
        regTcPtr->logWriteState = TcConnectionrec::NOT_WRITTEN;
        packLqhkeyreqLab(signal, tcConnectptr);
        return;
      }  // if
    } else {
      jam();
      /* -------------------------------------------------------------------
       * A DIRTY OPERATION WHICH NEEDS LOGGING. WE START BY LOGGING THE
       * REQUEST. IN THIS CASE WE WILL RELEASE THE FRAGMENT LOCK FIRST.
       * -------------------------------------------------------------------
       * A NORMAL WRITE OPERATION THAT NEEDS LOGGING AND WILL NOT BE
       * PREMATURELY COMMITTED.
       * ------------------------------------------------------------------ */
      writePrepareLog(signal, tcConnectptr);
    }  // if
  }    // if
}  // Dblqh::rwConcludedLab()

/* ##########################################################################
 * #######                            LOG MODULE                      #######
 *
 * ##########################################################################
 * --------------------------------------------------------------------------
 *       THE LOG MODULE HANDLES THE READING AND WRITING OF THE LOG
 *       IT IS ALSO RESPONSIBLE FOR HANDLING THE SYSTEM RESTART.
 *       IT CONTROLS THE SYSTEM RESTART IN TUP AND ACC AS WELL.
 * ------------------------------------------------------------------------ */

void Dblqh::set_use_mutex_for_log_parts() {
  /**
   * We can avoid using a mutex for log parts in a number of cases.
   * The requirement to avoid using a mutex is that the log part always
   * resides on the same LDM thread as the log part resides on. We calculate
   * the log part by using a calculation where we take the instance key
   * and take it modulo number of log parts. To find the LDM thread we take
   * the same instance key and this time we get it by using modulo the number
   * of LDM threads.
   *
   * Thus if the number of log parts is equal to the number of LDM workers
   * they will always map to the same LDM instance trivially. It is also
   * trivial to understand that we can always avoid a mutex with only 1
   * LDM instance since there is no concurrency in this case.
   *
   * It is also true that we can always avoid using a mutex when we have 2
   * LDM threads since the number of log parts can only be equal to 4, 6, 8,
   * 10, 12, 14, 16, 20, 24, 28 and 32. Thus all log parts with even instance
   * keys will map to the first LDM thread and all log parts with odd instance
   * keys will map to the second LDM thread.
   *
   * The general rule is that if the number of log parts is a multiple of the
   * number of LDM threads, then the log parts will always organise themselves
   * neatly onto only one LDM thread.
   *
   * In any other case the log parts will be spread onto different LDM threads.
   * Thus with 3 LDM threads and 4 log parts we have to use a mutex to protect
   * the log part since it will be accessed from more than one LDM instance.
   *
   * If the number of LDM threads is higher than the number of log parts then
   * it is also trivial to discover that a mutex is required.
   */
  m_use_mutex_for_log_parts = true;
  if (globalData.ndbMtLqhWorkers == globalData.ndbLogParts ||
      globalData.ndbMtLqhWorkers == 1 || globalData.ndbMtLqhWorkers == 2 ||
      !isNdbMtLqh()) {
    m_use_mutex_for_log_parts = false;
  } else if (globalData.ndbMtLqhWorkers < globalData.ndbLogParts) {
    Uint32 multiple = globalData.ndbLogParts / globalData.ndbMtLqhWorkers;
    if ((multiple * globalData.ndbMtLqhWorkers) == globalData.ndbLogParts) {
      jam();
      m_use_mutex_for_log_parts = false;
    }
  }
  if (instance() > 1) {
    return;
  }
  if (m_use_mutex_for_log_parts) {
    g_eventLogger->info(
        "We are running with %u LDM workers and %u"
        " REDO log parts. "
        "This means that we need to use a mutex to access"
        " REDO log parts",
        globalData.ndbMtLqhWorkers, globalData.ndbLogParts);
  } else {
    g_eventLogger->info(
        "We are running with %u LDM workers and %u"
        " REDO log parts. "
        "This means that we can avoid using a mutex to access"
        " REDO log parts",
        globalData.ndbMtLqhWorkers, globalData.ndbLogParts);
  }
}

/**
 * writePrepareLog
 *
 * Attempt to write a prepare log entry, may result in operation
 * being queued or aborted depending on redo log part state
 */
void Dblqh::writePrepareLog(Signal *signal,
                            const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  LogPartRecord *const regLogPartPtr = regTcPtr->m_log_part_ptr_p;

  lock_log_part(regLogPartPtr);

  Uint32 noOfFreeLogPages = count_free_log_pages(regLogPartPtr);
  const bool out_of_log_buffer = noOfFreeLogPages < ZMIN_LOG_PAGES_OPERATION;
  bool abort_on_redo_problems =
      (LqhKeyReq::getQueueOnRedoProblemFlag(regTcPtr->reqinfo) == 0);

  /* -------------------------------------------------- */
  /*       THIS PART IS USED TO WRITE THE LOG           */
  /* -------------------------------------------------- */
  /* -------------------------------------------------- */
  /*       CHECK IF A LOG OPERATION IS ONGOING ALREADY. */
  /*       IF SO THEN QUEUE THE OPERATION FOR LATER     */
  /*       RESTART WHEN THE LOG PART IS FREE AGAIN.     */
  /* -------------------------------------------------- */
  const bool problem = out_of_log_buffer || regLogPartPtr->m_log_problems != 0;
  if (unlikely(problem || ERROR_INSERTED(5083) || ERROR_INSERTED(5032))) {
    /* -----------------------------------------------------------------*/
    /* P_TAIL_PROBLEM indicates that the redo log is full. If redo      */
    /* log writes are queued in this situation, they will have to wait  */
    /* until redo space is freed. Redo space will not be freed          */
    /* until the next LCP completes, which can take a long time. To     */
    /* avoid long waits and timeouts, redo log writes are aborted       */
    /* in case of a P_TAIL_PROBLEM.                                     */
    /* -----------------------------------------------------------------*/
    if (abort_on_redo_problems ||
        regLogPartPtr->m_log_problems & LogPartRecord::P_TAIL_PROBLEM ||
        ERROR_INSERTED(5032)) {
      jam();
      if (ERROR_INSERTED_CLEAR(5032)) {
        const Uint32 saved_noOfFreeLogPages = regLogPartPtr->noOfFreeLogPages;
        // simulate abort on temporary out-of-redo error
        regLogPartPtr->noOfFreeLogPages = ZMIN_LOG_PAGES_OPERATION - 1;
        writePrepareLog_problems(signal, tcConnectptr, regLogPartPtr);
        regLogPartPtr->noOfFreeLogPages = saved_noOfFreeLogPages;
      } else {
        writePrepareLog_problems(signal, tcConnectptr, regLogPartPtr);
      }
    } else {
      jam();
      linkWaitLog(signal, regLogPartPtr, regLogPartPtr->m_log_prepare_queue,
                  tcConnectptr.p);
      regTcPtr->transactionState = TcConnectionrec::LOG_QUEUED;
    }
    unlock_log_part(regLogPartPtr);
    return;
  }

  if (regLogPartPtr->logPartState != LogPartRecord::IDLE) {
    jam();
    /**
     * We are in a state where the REDO log is only writing Prepare REDO
     * log entries through the REDO log queue. Thus we need to insert the
     * entry into the REDO log queue.
     */
    ndbrequire(regLogPartPtr->logPartState == LogPartRecord::ACTIVE);
    linkWaitLog(signal, regLogPartPtr, regLogPartPtr->m_log_prepare_queue,
                tcConnectptr.p);
    regTcPtr->transactionState = TcConnectionrec::LOG_QUEUED;
    unlock_log_part(regLogPartPtr);
    return;
  }  // if

  /* Proceed with writing the log */
  doWritePrepareLog(signal, tcConnectptr);

}  // Dblqh::writePrepareLog()

/**
 * doWritePrepareLog
 *
 * Do the redo log write - log part lock must be held,
 * and all checks must have been done before calling
 */
void Dblqh::doWritePrepareLog(Signal *signal,
                              const TcConnectionrecPtr tcConnectptr) {
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  UintR tcurrentFilepage;

  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  LogPartRecord *const regLogPartPtr = regTcPtr->m_log_part_ptr_p;

  // Require that log part pointer is locked

  increment_committed_mbytes(regLogPartPtr, regTcPtr);
  logFilePtr.i = regLogPartPtr->currentLogfile;
  ptrCheckGuard(logFilePtr, regLogPartPtr->my_block->clogFileFileSize,
                regLogPartPtr->my_block->logFileRecord);
  /* -------------------------------------------------- */
  /*       CHECK IF A NEW MBYTE IS TO BE STARTED. IF    */
  /*       SO INSERT A NEXT LOG RECORD, WRITE THE LOG   */
  /*       AND PLACE THE LOG POINTER ON THE NEW POSITION*/
  /*       IF A NEW FILE IS TO BE USED, CHANGE FILE AND */
  /*       ALSO START OPENING THE NEXT LOG FILE. IF A   */
  /*       LAP HAS BEEN COMPLETED THEN ADD ONE TO LAP   */
  /*       COUNTER.                                     */
  /* -------------------------------------------------- */
  checkNewMbyte(signal, regTcPtr, logPagePtr, logFilePtr, regLogPartPtr);

  /* -------------------------------------------------- */
  /*       INSERT THE OPERATION RECORD LAST IN THE LIST */
  /*       OF NOT COMPLETED OPERATIONS. ALSO RECORD THE */
  /*       FILE NO, PAGE NO AND PAGE INDEX OF THE START */
  /*       OF THIS LOG RECORD.                          */
  /*       IT IS NOT ALLOWED TO INSERT IT INTO THE LIST */
  /*       BEFORE CHECKING THE NEW MBYTE SINCE THAT WILL*/
  /*       CAUSE THE OLD VALUES OF TC_CONNECTPTR TO BE  */
  /*       USED IN WRITE_FILE_DESCRIPTOR.               */
  /* -------------------------------------------------- */
  {
    /* insertLogTcrec */
    TcConnectionrec *tmpTcConnectptr = regLogPartPtr->lastLogTcrec;
    regLogPartPtr->lastLogTcrec = regTcPtr;
    if (tmpTcConnectptr == nullptr) {
      jam();
      regLogPartPtr->firstLogTcrec = regTcPtr;
    } else {
      ndbrequire(Magic::check_ptr(tmpTcConnectptr));
      tmpTcConnectptr->nextLogTcrec = regTcPtr;
    }  // if
    regTcPtr->nextLogTcrec = nullptr;
    regTcPtr->prevLogTcrec = tmpTcConnectptr;
  }

  Uint32 fileNo = logFilePtr.p->fileNo;
  tcurrentFilepage = logFilePtr.p->currentFilepage;
  logPagePtr.i = logFilePtr.p->currentLogpage;
  ptrCheckGuard(logPagePtr, regLogPartPtr->logPageFileSize,
                regLogPartPtr->logPageRecord);
  Uint32 pageIndex = logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX];
  regTcPtr->logStartFileNo = fileNo;
  regTcPtr->logStartPageNo = tcurrentFilepage;
  regTcPtr->logStartPageIndex = pageIndex;
  /* -------------------------------------------------- */
  /*       WRITE THE LOG HEADER OF THIS OPERATION.      */
  /* -------------------------------------------------- */
  writeLogHeader(signal, tcConnectptr.p, logPagePtr, logFilePtr, regLogPartPtr);
  /* -------------------------------------------------- */
  /*       WRITE THE TUPLE KEY OF THIS OPERATION.       */
  /* -------------------------------------------------- */
  writeKey(signal, tcConnectptr.p, logPagePtr, logFilePtr, regLogPartPtr);
  /* -------------------------------------------------- */
  /*       WRITE THE ATTRIBUTE INFO OF THIS OPERATION.  */
  /* -------------------------------------------------- */
  writeAttrinfoLab(signal, tcConnectptr.p, logPagePtr, logFilePtr,
                   regLogPartPtr);

  /* -------------------------------------------------- */
  /*       RESET THE STATE OF THE LOG PART. IF ANY      */
  /*       OPERATIONS HAVE QUEUED THEN START THE FIRST  */
  /*       OF THESE.                                    */
  /* -------------------------------------------------- */
  /* -------------------------------------------------- */
  /*       CONTINUE WITH PACKING OF LQHKEYREQ           */
  /* -------------------------------------------------- */
  tcurrentFilepage = logFilePtr.p->currentFilepage;
  if (logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] == ZPAGE_HEADER_SIZE) {
    jam();
    tcurrentFilepage--;
  }  // if
  regTcPtr->logStopPageNo = tcurrentFilepage;
  regTcPtr->logWriteState = TcConnectionrec::WRITTEN;
  if (regTcPtr->abortState != TcConnectionrec::ABORT_IDLE) {
    /* -------------------------------------------------- */
    /*       AN ABORT HAVE BEEN ORDERED. THE ABORT WAITED */
    /*       FOR THE LOG WRITE TO BE COMPLETED. NOW WE    */
    /*       CAN PROCEED WITH THE NORMAL ABORT HANDLING.  */
    /* -------------------------------------------------- */
    jam();
    unlock_log_part(regLogPartPtr);
    abortCommonLab(signal, tcConnectptr);
    return;
  }  // if
  if (regTcPtr->dirtyOp != ZTRUE) {
    unlock_log_part(regLogPartPtr);
    packLqhkeyreqLab(signal, tcConnectptr);
  } else {
    jam();
    /* ----------------------------------------------------------------------
     * I NEED TO INSERT A COMMIT LOG RECORD SINCE WE ARE WRITING LOG IN THIS
     * TRANSACTION. SINCE WE RELEASED THE LOG LOCK JUST NOW NO ONE ELSE CAN BE
     * ACTIVE IN WRITING THE LOG. WE THUS WRITE THE LOG WITHOUT GETTING A LOCK
     * SINCE WE ARE ONLY WRITING A COMMIT LOG RECORD.
     * -------------------------------------------------------------------- */
    writeCommitLog(signal, regTcPtr, regLogPartPtr);
    removeLogTcrec(regTcPtr, regLogPartPtr);
    unlock_log_part(regLogPartPtr);
    /* ----------------------------------------------------------------------
     * DIRTY OPERATIONS SHOULD COMMIT BEFORE THEY PACK THE REQUEST/RESPONSE.
     * -------------------------------------------------------------------- */
    localCommitLab(signal, tcConnectptr);
  }  // if
}

void Dblqh::writePrepareLog_problems(Signal *signal,
                                     const TcConnectionrecPtr tcConnectptr,
                                     LogPartRecord *logPartPtrP) {
  jam();
  Uint32 problems = logPartPtrP->m_log_problems;

  if (logPartPtrP->noOfFreeLogPages < ZMIN_LOG_PAGES_OPERATION) {
    jam();
    terrorCode = ZTEMPORARY_REDO_LOG_FAILURE;
  } else if ((problems & LogPartRecord::P_TAIL_PROBLEM) != 0) {
    jam();
    terrorCode = ZTAIL_PROBLEM_IN_LOG_ERROR;
  } else if ((problems & LogPartRecord::P_REDO_IO_PROBLEM) != 0) {
    jam();
    terrorCode = ZREDO_IO_PROBLEM;
  } else if ((problems & LogPartRecord::P_FILE_CHANGE_PROBLEM) != 0) {
    jam();
    terrorCode = ZFILE_CHANGE_PROBLEM_IN_LOG_ERROR;
  } else {
    if (ERROR_INSERTED(5083)) {
      terrorCode = 266;
    }
  }
  abortErrorLab(signal, tcConnectptr);
}

void Dblqh::update_log_problem(Signal *signal, LogPartRecord *partPtrP,
                               Uint32 problem, bool value) {
  Uint32 problems = partPtrP->m_log_problems;
  if (value) {
    /**
     * set
     */
    jam();
    if ((problems & problem) == 0) {
      jam();
      problems |= problem;
    }
  } else {
    /**
     * clear
     */
    jam();
    if ((problems & problem) != 0) {
      jam();
      problems &= ~(Uint32)problem;

      if (partPtrP->LogLqhKeyReqSent == ZFALSE &&
          (!partPtrP->m_log_prepare_queue.isEmpty() ||
           !partPtrP->m_log_complete_queue.isEmpty())) {
        jam();

        partPtrP->LogLqhKeyReqSent = ZTRUE;
        signal->theData[0] = ZLOG_LQHKEYREQ;
        signal->theData[1] = partPtrP->ptrI;
        sendSignal(partPtrP->myRef, GSN_CONTINUEB, signal, 2, JBB);
      }
    }
  }
  partPtrP->m_log_problems = problems;
}

/* ------------------------------------------------------------------------- */
/* -------                        SEND LQHKEYREQ                             */
/*                                                                           */
/* NO STATE CHECKING SINCE THE SIGNAL IS A LOCAL SIGNAL. THE EXECUTION OF    */
/* THE OPERATION IS COMPLETED. IT IS NOW TIME TO SEND THE OPERATION TO THE   */
/* NEXT REPLICA OR TO TC.                                                    */
/* ------------------------------------------------------------------------- */
void Dblqh::packLqhkeyreqLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (regTcPtr->nextReplica == ZNIL) {
    /* -------------------------------------------------------------------------
     */
    /* -------               SEND LQHKEYCONF                             -------
     */
    /*                                                                           */
    /* -------------------------------------------------------------------------
     */
    sendLqhkeyconfTc(signal, regTcPtr->tcBlockref, tcConnectptr);
    if (!(regTcPtr->dirtyOp ||
          (regTcPtr->operation == ZREAD && regTcPtr->opSimple))) {
      jamDebug();
      regTcPtr->transactionState = TcConnectionrec::PREPARED;
      releaseOprec(signal, tcConnectptr);
    } else {
      jamDebug();

      /*************************************************************>*/
      /*       DIRTY WRITES ARE USED IN TWO SITUATIONS. THE FIRST    */
      /*       SITUATION IS WHEN THEY ARE USED TO UPDATE COUNTERS AND*/
      /*       OTHER ATTRIBUTES WHICH ARE NOT SENSITIVE TO CONSISTE- */
      /*       NCY. THE SECOND SITUATION IS BY OPERATIONS THAT ARE   */
      /*       SENT AS PART OF A COPY FRAGMENT PROCESS.              */
      /*                                                             */
      /*       DURING A COPY FRAGMENT PROCESS THERE IS NO LOGGING    */
      /*       ONGOING SINCE THE FRAGMENT IS NOT COMPLETE YET. THE   */
      /*       LOGGING STARTS AFTER COMPLETING THE LAST COPY TUPLE   */
      /*       OPERATION. THE EXECUTION OF THE LAST COPY TUPLE DOES  */
      /*       ALSO START A LOCAL CHECKPOINT SO THAT THE FRAGMENT    */
      /*       REPLICA IS RECOVERABLE. THUS GLOBAL CHECKPOINT ID FOR */
      /*       THOSE OPERATIONS ARE NOT INTERESTING.                 */
      /*                                                             */
      /*       A DIRTY WRITE IS BY DEFINITION NOT CONSISTENT. THUS   */
      /*       IT CAN USE ANY GLOBAL CHECKPOINT. THE IDEA HERE IS TO */
      /*       ALWAYS USE THE LATEST DEFINED GLOBAL CHECKPOINT ID IN */
      /*       THIS NODE.                                            */
      /*************************************************************>*/
      cleanUp(signal, tcConnectptr);
    }  // if
    return;
  }  // if
  ndbassert(!m_is_query_block);
  /* -------------------------------------------------------------------------
   */
  /* -------            SEND LQHKEYREQ                                 -------
   */
  /*                                                                           */
  /* -------------------------------------------------------------------------
   */
  /* -------------------------------------------------------------------------
   */
  /* THERE ARE MORE REPLICAS TO SEND THE OPERATION TO. A NEW LQHKEYREQ WILL BE
   */
  /* PREPARED FOR THE NEXT REPLICA. */
  /* -------------------------------------------------------------------------
   */
  /* CLEAR REPLICA TYPE, ATTRINFO INDICATOR (IN LQHKEYREQ), */
  /* INTERPRETED EXECUTION, SEQUENTIAL NUMBER OF REPLICA. */
  // Set bit indicating Client and TC record not the same.
  // Set readlenAi indicator if readlenAi != 0
  // Stored Procedure Indicator not set.
  /* -------------------------------------------------------------------------
   */
  LqhKeyReq *const lqhKeyReq = (LqhKeyReq *)&signal->theData[0];

  UintR Treqinfo;
  UintR sig0, sig1, sig2, sig3, sig4, sig5, sig6;
  Treqinfo = preComputedRequestInfoMask & regTcPtr->reqinfo;

  jam();
  UintR TAiLen = 0;
  /* Long LQHKeyReq uses section size for key length */
  Uint32 lqhKeyLen = 0;

  UintR TapplAddressIndicator = (regTcPtr->nextSeqNoReplica == 0 ? 0 : 1);
  LqhKeyReq::setApplicationAddressFlag(Treqinfo, TapplAddressIndicator);
  LqhKeyReq::setInterpretedFlag(Treqinfo, regTcPtr->opExec);
  LqhKeyReq::setSeqNoReplica(Treqinfo, regTcPtr->nextSeqNoReplica);
  LqhKeyReq::setAIInLqhKeyReq(Treqinfo, TAiLen);
  LqhKeyReq::setKeyLen(Treqinfo, lqhKeyLen);

  regTcPtr->m_use_rowid |=
      fragptr.p->m_copy_started_state == Fragrecord::AC_NR_COPY;
  LqhKeyReq::setRowidFlag(Treqinfo, regTcPtr->m_use_rowid);

#ifdef VM_TRACE
  if (LqhKeyReq::getRowidFlag(Treqinfo)) {
    // ndbassert(LqhKeyReq::getOperation(Treqinfo) == ZINSERT);
  } else {
    if (fragptr.p->m_copy_started_state != Fragrecord::AC_IGNORED) {
      Uint32 nextNodeId = regTcPtr->nextReplica;
      ndbassert(LqhKeyReq::getOperation(Treqinfo) != ZINSERT ||
                get_node_status(nextNodeId) != ZNODE_UP);
    }
  }
#endif

  UintR TreadLenAiInd = (regTcPtr->readlenAi == 0 ? 0 : 1);
  UintR TsameLqhAndClient = (tcConnectptr.i == regTcPtr->tcOprec ? 0 : 1);
  LqhKeyReq::setSameClientAndTcFlag(Treqinfo, TsameLqhAndClient);
  LqhKeyReq::setReturnedReadLenAIFlag(Treqinfo, TreadLenAiInd);

  /* Long LQHKeyReq uses section size for AttrInfo length */
  UintR TotReclenAi = 0;
  LqhKeyReq::setReorgFlag(TotReclenAi, regTcPtr->m_reorg);

  /* -------------------------------------------------------------------------
   */
  /* WE ARE NOW PREPARED TO SEND THE LQHKEYREQ. WE HAVE TO DECIDE IF ATTRINFO */
  /* IS INCLUDED IN THE LQHKEYREQ SIGNAL AND THEN SEND IT. */
  /* TAKE OVER SCAN OPERATION IS NEVER USED ON BACKUPS, LOG RECORDS AND
   * START-UP*/
  /* OF NEW REPLICA AND THUS ONLY TOT_SENDLEN_AI IS USED THE UPPER 16 BITS ARE
   */
  /* ZERO. */
  /* -------------------------------------------------------------------------
   */
  sig0 = tcConnectptr.i;
  sig1 = regTcPtr->savePointId;
  sig2 = regTcPtr->hashValue;
  sig4 = regTcPtr->tcBlockref;

  lqhKeyReq->clientConnectPtr = sig0;
  lqhKeyReq->attrLen = TotReclenAi;
  lqhKeyReq->savePointId = sig1;
  lqhKeyReq->hashValue = sig2;
  lqhKeyReq->requestInfo = Treqinfo;
  lqhKeyReq->tcBlockref = sig4;

  sig0 = regTcPtr->tableref + ((regTcPtr->schemaVersion << 16) & 0xFFFF0000);
  sig1 = regTcPtr->fragmentid + (regTcPtr->nodeAfterNext[0] << 16);
  sig2 = regTcPtr->transid[0];
  sig3 = regTcPtr->transid[1];
  sig4 = regTcPtr->applRef;
  sig5 = regTcPtr->applOprec;
  sig6 = regTcPtr->tcOprec;
  UintR nextPos = (TapplAddressIndicator << 1);

  lqhKeyReq->tableSchemaVersion = sig0;
  lqhKeyReq->fragmentData = sig1;
  lqhKeyReq->transId1 = sig2;
  lqhKeyReq->transId2 = sig3;
  lqhKeyReq->numFiredTriggers = regTcPtr->numFiredTriggers;
  lqhKeyReq->variableData[0] = sig4;
  lqhKeyReq->variableData[1] = sig5;
  lqhKeyReq->variableData[2] = sig6;

  nextPos += TsameLqhAndClient;

  if ((regTcPtr->lastReplicaNo - regTcPtr->nextSeqNoReplica) > 1) {
    sig0 = (UintR)regTcPtr->nodeAfterNext[1] +
           (UintR)(regTcPtr->nodeAfterNext[2] << 16);
    lqhKeyReq->variableData[nextPos] = sig0;
    nextPos++;
  }  // if
  sig0 = regTcPtr->readlenAi;
  lqhKeyReq->variableData[nextPos] = sig0;
  nextPos += TreadLenAiInd;

  ndbassert(!m_is_in_query_thread);
  sig0 = regTcPtr->gci_hi;
  Local_key tmp = regTcPtr->m_row_id;

  lqhKeyReq->variableData[nextPos + 0] = tmp.m_page_no;
  lqhKeyReq->variableData[nextPos + 1] = tmp.m_page_idx;
  nextPos += 2 * LqhKeyReq::getRowidFlag(Treqinfo);

  lqhKeyReq->variableData[nextPos + 0] = sig0;
  nextPos += LqhKeyReq::getGCIFlag(Treqinfo);

  // pass full instance key for remote to map to real instance
  Uint32 instanceNo =
      getInstanceNo(regTcPtr->nextReplica, fragptr.p->lqhInstanceKey);
  BlockReference lqhRef = numberToRef(DBLQH, instanceNo, regTcPtr->nextReplica);

  {
    /* Long LQHKEYREQ, attach KeyInfo and AttrInfo
     * sections to signal
     */
    SectionHandle handle(this);
    handle.m_cnt = 0;

    if (likely(regTcPtr->primKeyLen > 0)) {
      SegmentedSectionPtr keyInfoSection;

      ndbassert(regTcPtr->keyInfoIVal != RNIL);
      getSection(keyInfoSection, regTcPtr->keyInfoIVal);

      handle.m_ptr[LqhKeyReq::KeyInfoSectionNum] = keyInfoSection;
      handle.m_cnt = 1;

      if (regTcPtr->totSendlenAi > 0) {
        SegmentedSectionPtr attrInfoSection;

        ndbassert(regTcPtr->attrInfoIVal != RNIL);
        getSection(attrInfoSection, regTcPtr->attrInfoIVal);

        handle.m_ptr[LqhKeyReq::AttrInfoSectionNum] = attrInfoSection;
        handle.m_cnt = 2;
      } else {
        /* No AttrInfo to be sent on.  This can occur for delete
         * or with an interpreted update when no actual update
         * is made
         * In this case, we free any attrInfo section now.
         */
        if (regTcPtr->attrInfoIVal != RNIL) {
          ndbassert(!(regTcPtr->m_flags & TcConnectionrec::OP_SAVEATTRINFO));
          releaseSection(regTcPtr->attrInfoIVal);
          regTcPtr->attrInfoIVal = RNIL;
        }
      }
    } else {
      /* Zero-length primary key, better not have any
       * AttrInfo
       */
      ndbrequire(regTcPtr->totSendlenAi == 0);
      ndbrequire(regTcPtr->keyInfoIVal == RNIL);
      ndbrequire(regTcPtr->attrInfoIVal == RNIL);
      ndbrequire(LqhKeyReq::getRowidFlag(Treqinfo));
      ndbrequire(LqhKeyReq::getNrCopyFlag(Treqinfo));
    }

    sendBatchedLqhkeyreq(signal, lqhRef, LqhKeyReq::FixedSignalLength + nextPos,
                         &handle);

    /* Long sections were freed as part of sendSignal */
    ndbassert(handle.m_cnt == 0);
    regTcPtr->keyInfoIVal = RNIL;
    regTcPtr->attrInfoIVal = RNIL;
  }
  /* LQHKEYREQ sent */

  regTcPtr->transactionState = TcConnectionrec::PREPARED;
  if (unlikely(regTcPtr->dirtyOp == ZTRUE)) {
    jam();
    /*************************************************************>*/
    /*       DIRTY WRITES ARE USED IN TWO SITUATIONS. THE FIRST    */
    /*       SITUATION IS WHEN THEY ARE USED TO UPDATE COUNTERS AND*/
    /*       OTHER ATTRIBUTES WHICH ARE NOT SENSITIVE TO CONSISTE- */
    /*       NCY. THE SECOND SITUATION IS BY OPERATIONS THAT ARE   */
    /*       SENT AS PART OF A COPY FRAGMENT PROCESS.              */
    /*                                                             */
    /*       DURING A COPY FRAGMENT PROCESS THERE IS NO LOGGING    */
    /*       ONGOING SINCE THE FRAGMENT IS NOT COMPLETE YET. THE   */
    /*       LOGGING STARTS AFTER COMPLETING THE LAST COPY TUPLE   */
    /*       OPERATION. THE EXECUTION OF THE LAST COPY TUPLE DOES  */
    /*       ALSO START A LOCAL CHECKPOINT SO THAT THE FRAGMENT    */
    /*       REPLICA IS RECOVERABLE. THUS GLOBAL CHECKPOINT ID FOR */
    /*       THOSE OPERATIONS ARE NOT INTERESTING.                 */
    /*                                                             */
    /*       A DIRTY WRITE IS BY DEFINITION NOT CONSISTENT. THUS   */
    /*       IT CAN USE ANY GLOBAL CHECKPOINT. THE IDEA HERE IS TO */
    /*       ALWAYS USE THE LATEST DEFINED GLOBAL CHECKPOINT ID IN */
    /*       THIS NODE.                                            */
    /*************************************************************>*/
    cleanUp(signal, tcConnectptr);
    return;
  }  // if
  /* ------------------------------------------------------------------------
   *   ALL INFORMATION NEEDED BY THE COMMIT PHASE AND COMPLETE PHASE IS
   *   KEPT IN THE TC_CONNECT RECORD. TO ENSURE PROPER USE OF MEMORY
   *   RESOURCES WE DEALLOCATE THE ATTRINFO RECORD AND KEY RECORDS
   *   AS SOON AS POSSIBLE.
   * ------------------------------------------------------------------------ */
  releaseOprec(signal, tcConnectptr);
}  // Dblqh::packLqhkeyreqLab()

/* ========================================================================= */
/* ==== CHECK IF THE LOG RECORD FITS INTO THE CURRENT MBYTE,         ======= */
/*      OTHERWISE SWITCH TO NEXT MBYTE.                                      */
/*                                                                           */
/* ========================================================================= */
void Dblqh::checkNewMbyte(Signal *signal, const TcConnectionrec *regTcPtr,
                          LogPageRecordPtr &logPagePtr,
                          LogFileRecordPtr &logFilePtr,
                          LogPartRecord *logPartPtrP) {
  UintR tcnmTmp;
  UintR ttotalLogSize;

  /* -------------------------------------------------- */
  /*       CHECK IF A NEW MBYTE OF LOG RECORD IS TO BE  */
  /*       OPENED BEFORE WRITING THE LOG RECORD. NO LOG */
  /*       RECORDS ARE ALLOWED TO SPAN A MBYTE BOUNDARY */
  /*                                                    */
  /*       INPUT:  TC_CONNECTPTR   THE OPERATION        */
  /*               LOG_FILE_PTR    THE LOG FILE         */
  /*       OUTPUT: LOG_FILE_PTR    THE NEW LOG FILE     */
  /* -------------------------------------------------- */
  ttotalLogSize = ZLOG_HEAD_SIZE + regTcPtr->currTupAiLen;
  ttotalLogSize = ttotalLogSize + regTcPtr->primKeyLen;
  tcnmTmp = logFilePtr.p->remainingWordsInMbyte;
  if ((ttotalLogSize + ZNEXT_LOG_SIZE) <= tcnmTmp) {
    ndbrequire(tcnmTmp >= ttotalLogSize);
    logFilePtr.p->remainingWordsInMbyte = tcnmTmp - ttotalLogSize;
    return;
  } else {
    jam();
    /* -------------------------------------------------- */
    /*       IT WAS NOT ENOUGH SPACE IN THIS MBYTE FOR    */
    /*       THIS LOG RECORD. MOVE TO NEXT MBYTE          */
    /*       THIS MIGHT INCLUDE CHANGING LOG FILE         */
    /* -------------------------------------------------- */
    /*       WE HAVE TO INSERT A NEXT LOG RECORD FIRST    */
    /* -------------------------------------------------- */
    /*       THEN CONTINUE BY WRITING THE FILE DESCRIPTORS*/
    /* -------------------------------------------------- */
    logPagePtr.i = logFilePtr.p->currentLogpage;
    ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                  logPartPtrP->logPageRecord);
    changeMbyte(signal, logPagePtr, logFilePtr, logPartPtrP);
    tcnmTmp = logFilePtr.p->remainingWordsInMbyte;
  }  // if
  ndbrequire(tcnmTmp >= ttotalLogSize);
  logFilePtr.p->remainingWordsInMbyte = tcnmTmp - ttotalLogSize;
}  // Dblqh::checkNewMbyte()

/* --------------------------------------------------------------------------
 * -------               WRITE OPERATION HEADER TO LOG                -------
 *
 *       SUBROUTINE SHORT NAME: WLH
 * ------------------------------------------------------------------------- */
void Dblqh::writeLogHeader(Signal *signal, const TcConnectionrec *regTcPtr,
                           LogPageRecordPtr &logPagePtr,
                           LogFileRecordPtr &logFilePtr,
                           LogPartRecord *logPartPtrP) {
  Uint32 keyLen = regTcPtr->primKeyLen;
  Uint32 aiLen = regTcPtr->currTupAiLen;
  Local_key rowid = regTcPtr->m_row_id;
  Uint32 totLogLen = ZLOG_HEAD_SIZE + aiLen + keyLen;
  Uint32 logPos = logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX];
  Uint32 hashValue = regTcPtr->hashValue;
  Uint32 operation = regTcPtr->operation;
  logPartPtrP->m_total_written_words += totLogLen;

  if ((logPos + ZLOG_HEAD_SIZE) < ZPAGE_SIZE) {
    Uint32 *dataPtr = &logPagePtr.p->logPageWord[logPos];
    logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] = logPos + ZLOG_HEAD_SIZE;
    dataPtr[0] = ZPREP_OP_TYPE;
    dataPtr[1] = totLogLen;
    dataPtr[2] = hashValue;
    dataPtr[3] = operation;
    dataPtr[4] = aiLen;
    dataPtr[5] = keyLen;
    dataPtr[6] = rowid.m_page_no;
    dataPtr[7] = rowid.m_page_idx;
  } else {
    writeLogWord(signal, ZPREP_OP_TYPE, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, totLogLen, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, hashValue, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, operation, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, aiLen, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, keyLen, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, rowid.m_page_no, logPagePtr, logFilePtr, logPartPtrP);
    writeLogWord(signal, rowid.m_page_idx, logPagePtr, logFilePtr, logPartPtrP);
  }  // if
}  // Dblqh::writeLogHeader()

/* --------------------------------------------------------------------------
 * -------               WRITE TUPLE KEY TO LOG                       -------
 *
 *       SUBROUTINE SHORT NAME: WK
 * ----------------------------------------------------------------------- */
void Dblqh::writeKey(Signal *signal, const TcConnectionrec *regTcPtr,
                     LogPageRecordPtr &logPagePtr, LogFileRecordPtr &logFilePtr,
                     LogPartRecord *logPartPtrP) {
  SectionReader keyInfoReader(regTcPtr->keyInfoIVal, g_sectionSegmentPool);
  const Uint32 *srcPtr;
  Uint32 length;
  Uint32 wordsWritten = 0;

  /* Write contiguous chunks of words from the KeyInfo
   * section to the log
   */
  while (keyInfoReader.getWordsPtr(srcPtr, length)) {
    writeLogWords(signal, srcPtr, length, logPagePtr, logFilePtr, logPartPtrP);
    wordsWritten += length;
  }

  ndbassert(wordsWritten == regTcPtr->primKeyLen);
}  // Dblqh::writeKey()

/* --------------------------------------------------------------------------
 * -------               WRITE ATTRINFO TO LOG                        -------
 *
 *       SUBROUTINE SHORT NAME: WA
 * ----------------------------------------------------------------------- */
void Dblqh::writeAttrinfoLab(Signal *signal, const TcConnectionrec *regTcPtr,
                             LogPageRecordPtr &logPagePtr,
                             LogFileRecordPtr &logFilePtr,
                             LogPartRecord *logPartPtrP) {
  Uint32 totLen = regTcPtr->currTupAiLen;
  if (totLen == 0) return;

  jamDebug();
  ndbassert(regTcPtr->attrInfoIVal != RNIL);
  SectionReader attrInfoReader(regTcPtr->attrInfoIVal, g_sectionSegmentPool);
  const Uint32 *srcPtr;
  Uint32 length;
  Uint32 wordsWritten = 0;

  /* Write contiguous chunks of words from the
   * AttrInfo section to the log
   */
  while (attrInfoReader.getWordsPtr(srcPtr, length)) {
    writeLogWords(signal, srcPtr, length, logPagePtr, logFilePtr, logPartPtrP);
    wordsWritten += length;
  }
  ndbassert(wordsWritten == totLen);
}  // Dblqh::writeAttrinfoLab()

void Dblqh::cleanUp(Signal *signal, TcConnectionrecPtr tcConnectptr) {
  releaseOprec(signal, tcConnectptr);
  deleteTransidHash(signal, tcConnectptr);
  releaseTcrec(signal, tcConnectptr);
}  // Dblqh::cleanUp()

void Dblqh::releaseOprec(Signal *signal, TcConnectionrecPtr regTcPtr) {
  if (regTcPtr.p->m_dealloc_state != TcConnectionrec::DA_IDLE) {
    handleDeallocOp(signal, regTcPtr);
  }

  /* Release long sections if present */
  releaseSection(regTcPtr.p->keyInfoIVal);
  regTcPtr.p->keyInfoIVal = RNIL;
  releaseSection(regTcPtr.p->attrInfoIVal);
  regTcPtr.p->attrInfoIVal = RNIL;
}  // Dblqh::releaseOprec()

/* ------------------------------------------------------------------------- */
/* ------         DELETE TRANSACTION ID FROM HASH TABLE              ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::deleteTransidHash(Signal *signal,
                              TcConnectionrecPtr &tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  TcConnectionrecPtr prevHashptr;
  TcConnectionrecPtr nextHashptr;
  /**
   * This operation has not been inserted in the hash list at all.
   * (It is a non-transactional 'dirtyOp', or the request failed
   *  before it was ever inserted in the hash list.)
   */
  if (regTcPtr->hashIndex == RNIL) {
    jamDebug();
    /* If this operation is 'non-dirty', there should be no duplicates */
    ndbrequire(regTcPtr->dirtyOp == ZTRUE ||
               findTransaction(regTcPtr->transid[0], regTcPtr->transid[1],
                               regTcPtr->tcOprec, regTcPtr->tcHashKeyHi,
                               (regTcPtr->tcScanRec == RNIL), false,
                               tcConnectptr) == ZNOT_FOUND);
    return;
  }

  prevHashptr.i = regTcPtr->prevHashRec;
  nextHashptr.i = regTcPtr->nextHashRec;
  /* prevHashptr and nextHashptr may be RNIL when the bucket has 1 element */

  if (prevHashptr.i != RNIL) {
    jamDebug();
    ndbrequire(tcConnect_pool.getValidPtr(prevHashptr));
    ndbassert(prevHashptr.p->nextHashRec == tcConnectptr.i);
    prevHashptr.p->nextHashRec = nextHashptr.i;
  } else {
    jamDebug();
    /* -------------------------------------------------------------------------
     */
    /* THE OPERATION WAS PLACED FIRST IN THE LIST OF THE HASH TABLE. NEED TO SET
     */
    /* A NEW LEADER OF THE LIST. */
    /* -------------------------------------------------------------------------
     */
    const Uint32 hashIndex = regTcPtr->hashIndex;
    ndbassert(hashIndex == getHashIndex(regTcPtr));
    ndbassert(ctransidHash[hashIndex] == tcConnectptr.i);
    ctransidHash[hashIndex] = nextHashptr.i;
  }  // if
  if (nextHashptr.i != RNIL) {
    jamDebug();
    ndbrequire(tcConnect_pool.getValidPtr(nextHashptr));
    ndbassert(nextHashptr.p->prevHashRec == tcConnectptr.i);
    nextHashptr.p->prevHashRec = prevHashptr.i;
  }  // if
  regTcPtr->hashIndex = regTcPtr->prevHashRec = regTcPtr->nextHashRec = RNIL;
}  // Dblqh::deleteTransidHash()

/* -------------------------------------------------------------------------
 * -------       RELEASE OPERATION FROM ACTIVE LIST ON FRAGMENT      -------
 *
 *       SUBROUTINE SHORT NAME = RAF
 * ------------------------------------------------------------------------- */
/* ######################################################################### */
/* #######                   TRANSACTION MODULE                      ####### */
/*      THIS MODULE HANDLES THE COMMIT AND THE COMPLETE PHASE.               */
/* ######################################################################### */
void Dblqh::warningReport(Signal *signal, int place) {
  switch (place) {
    case 0:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMMIT in wrong state in Dblqh" << endl;
#endif
      break;
    case 1:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMMIT with wrong transid in Dblqh" << endl;
#endif
      break;
    case 2:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMPLETE in wrong state in Dblqh" << endl;
#endif
      break;
    case 3:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMPLETE with wrong transid in Dblqh" << endl;
#endif
      break;
    case 4:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMMITREQ in wrong state in Dblqh" << endl;
#endif
      break;
    case 5:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMMITREQ with wrong transid in Dblqh" << endl;
#endif
      break;
    case 6:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMPLETEREQ in wrong state in Dblqh" << endl;
#endif
      break;
    case 7:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMPLETEREQ with wrong transid in Dblqh" << endl;
#endif
      break;
    case 8:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received ABORT with non-existing transid in Dblqh" << endl;
#endif
      break;
    case 9:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received ABORTREQ with non-existing transid in Dblqh"
             << endl;
#endif
      break;
    case 10:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received ABORTREQ in wrong state in Dblqh" << endl;
#endif
      break;
    case 11:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMMIT when tc-rec released in Dblqh" << endl;
#endif
      break;
    case 12:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received COMPLETE when tc-rec released in Dblqh" << endl;
#endif
      break;
    case 13:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received LQHKEYREF when tc-rec released in Dblqh" << endl;
#endif
      break;
    case 14:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received LQHKEYREF with wrong transid in Dblqh" << endl;
#endif
      break;
    case 15:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received LQHKEYREF when already aborting in Dblqh" << endl;
#endif
      break;
    case 16:
      jam();
      ndbrequire(cstartPhase == ZNIL);
#ifdef ABORT_TRACE
      ndbout << "W: Received LQHKEYREF in wrong state in Dblqh" << endl;
#endif
      break;
    case 17:
      jam();
#ifdef ABORT_TRACE
      ndbout << "W: Received LQHKEYREF with wrong transid in" < < < <
          " COPY_CONNECTED state" << endl;
#endif
      break;
    default:
      jam();
#ifdef ABORT_TRACE
      ndbout << "LQH WarningReport: " << place << endl;
#endif
      break;
  }  // switch
  return;
}  // Dblqh::warningReport()

void Dblqh::errorReport(Signal *signal, int place) {
  switch (place) {
    case 0:
      jam();
      break;
    case 1:
      jam();
      break;
    case 2:
      jam();
      break;
    case 3:
      jam();
      break;
    default:
      jam();
      break;
  }  // switch
  systemErrorLab(signal, __LINE__);
  return;
}  // Dblqh::errorReport()

void Dblqh::execFIRE_TRIG_REQ(Signal *signal) {
  Uint32 tcOprec = signal->theData[0];
  Uint32 transid1 = signal->theData[1];
  Uint32 transid2 = signal->theData[2];
  Uint32 pass = signal->theData[3];
  Uint32 senderRef = signal->getSendersBlockRef();

  jamEntry();

  if (ERROR_INSERTED_CLEAR(5064)) {
    // throw away...should cause timeout in TC
    return;
  }

  CRASH_INSERTION(5072);

  Uint32 err;
  TcConnectionrecPtr tcConnectptr;
  if (findTransaction(transid1, transid2, tcOprec, senderRef, true, false,
                      tcConnectptr) == ZOK &&
      !ERROR_INSERTED_CLEAR(5065) && !ERROR_INSERTED(5070) &&
      !ERROR_INSERTED(5071)) {
    TcConnectionrec *const regTcPtr = tcConnectptr.p;

    if (unlikely(regTcPtr->transactionState != TcConnectionrec::PREPARED ||
                 ERROR_INSERTED_CLEAR(5067))) {
      err = FireTrigRef::FTR_IncorrectState;
      goto do_err;
    }

    /**
     *
     */
    signal->theData[0] = regTcPtr->tupConnectrec;
    signal->theData[1] = regTcPtr->tcBlockref;
    signal->theData[2] = regTcPtr->tcOprec;
    signal->theData[3] = transid1;
    signal->theData[4] = transid2;
    signal->theData[5] = pass;
    Uint32 tup = refToMain(ctupBlockref);
    EXECUTE_DIRECT(tup, GSN_FIRE_TRIG_REQ, signal, 6);

    err = signal->theData[0];
    Uint32 cnt = signal->theData[1];

    if (ERROR_INSERTED_CLEAR(5066)) {
      err = 5066;
    }

    if (ERROR_INSERTED_CLEAR(5068)) tcOprec++;
    if (ERROR_INSERTED_CLEAR(5069)) transid1++;

    if (err == 0) {
      jam();
      Uint32 Tdata[FireTrigConf::SignalLength];
      FireTrigConf *conf = CAST_PTR(FireTrigConf, Tdata);
      conf->tcOpRec = tcOprec;
      conf->transId[0] = transid1;
      conf->transId[1] = transid2;
      conf->numFiredTriggers = cnt;
      sendFireTrigConfTc(signal, regTcPtr->tcBlockref, Tdata);
      return;
    }
  } else {
    jam();
    err = FireTrigRef::FTR_UnknownOperation;
  }

do_err:
  if (ERROR_INSERTED_CLEAR(5070)) tcOprec++;

  if (ERROR_INSERTED_CLEAR(5071)) transid1++;

  FireTrigRef *ref = CAST_PTR(FireTrigRef, signal->getDataPtrSend());
  ref->tcOpRec = tcOprec;
  ref->transId[0] = transid1;
  ref->transId[1] = transid2;
  ref->errCode = err;
  sendSignal(senderRef, GSN_FIRE_TRIG_REF, signal, FireTrigRef::SignalLength,
             JBB);

  return;
}

void Dblqh::sendFireTrigConfTc(Signal *signal, BlockReference atcBlockref,
                               Uint32 Tdata[]) {
  Uint32 instanceKey = refToInstance(atcBlockref);

  ndbassert(refToMain(atcBlockref) == DBTC);
  if (instanceKey > MAX_NDBMT_TC_THREADS) {
    jam();
    memcpy(signal->theData, Tdata, 4 * FireTrigConf::SignalLength);
    sendSignal(atcBlockref, GSN_FIRE_TRIG_CONF, signal,
               FireTrigConf::SignalLength, JBB);
    return;
  }

  HostRecordPtr Thostptr;
  Thostptr.i = refToNode(atcBlockref);
  ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
  Uint32 len = FireTrigConf::SignalLength;
  struct PackedWordsContainer *container = &Thostptr.p->tc_pack[instanceKey];

  if (container->noOfPackedWords > (25 - len)) {
    jam();
    sendPackedSignal(signal, container);
  } else {
    jam();
    updatePackedList(signal, Thostptr.p, Thostptr.i);
  }

  ndbassert(FireTrigConf::SignalLength == 4);
  Uint32 *dst = &container->packedWords[container->noOfPackedWords];
  container->noOfPackedWords += len;
  Thostptr.p->tc_pack_mask.set(instanceKey);
  dst[0] = Tdata[0] | (ZFIRE_TRIG_CONF << 28);
  dst[1] = Tdata[1];
  dst[2] = Tdata[2];
  dst[3] = Tdata[3];
}

bool Dblqh::check_fire_trig_pass(Uint32 opId, Uint32 pass) {
  /**
   * Check that trigger only fires once per pass
   *   (per primary key)
   */
  TcConnectionrecPtr regTcPtr;
  regTcPtr.i = opId;
  ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));
  if (regTcPtr.p->m_fire_trig_pass <= pass) {
    regTcPtr.p->m_fire_trig_pass = pass + 1;
    return true;
  }
  return false;
}

/* ************************************************************************>>
 *  COMMIT: Start commit request from TC. This signal is originally sent as a
 *  packed signal and this function is called from execPACKED_SIGNAL.
 *  This is the normal commit protocol where TC first send this signal to the
 *  backup node which then will send COMMIT to the primary node. If
 *  everything is ok the primary node send COMMITTED back to TC.
 * ************************************************************************>> */
void Dblqh::execCOMMIT(Signal *signal) {
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  Uint32 gci_hi = signal->theData[1];
  Uint32 transid1 = signal->theData[2];
  Uint32 transid2 = signal->theData[3];
  Uint32 gci_lo = signal->theData[4];
  jamEntryDebug();
  if (unlikely(!tcConnect_pool.getValidPtr(tcConnectptr))) {
    jam();
    warningReport(signal, 0);
    return;
  }  // if
  if (ERROR_INSERTED(5011)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_COMMIT, signal, 2000, signal->getLength());
    return;
  }  // if
  if (ERROR_INSERTED(5012)) {
    SET_ERROR_INSERT_VALUE(5017);
    sendSignalWithDelay(cownref, GSN_COMMIT, signal, 2000, signal->getLength());
    return;
  }  // if
  if (ERROR_INSERTED(5062) &&
      ((refToMain(signal->getSendersBlockRef()) == DBTC) ||
       signal->getSendersBlockRef() == reference())) {
    Uint32 save = signal->getSendersBlockRef();
    g_eventLogger->info("Delaying execCOMMIT");
    sendSignalWithDelay(cownref, GSN_COMMIT, signal, 2000, signal->getLength());

    if (refToMain(save) == DBTC) {
      g_eventLogger->info("killing %u", refToNode(save));
      signal->theData[0] = 9999;
      sendSignal(numberToRef(CMVMI, refToNode(save)), GSN_NDB_TAMPER, signal, 1,
                 JBB);
    }
    return;
  }
  if (ERROR_INSERTED(5110)) {
    jam();
    if (tcConnectptr.p->tableref > 2) {
      jam();
      g_eventLogger->info("LQH %u delaying commit", instance());
      sendSignalWithDelay(cownref, GSN_COMMIT, signal, 200, signal->getLength());
      return;
    }
  }

  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  if (likely((tcConnectptr.p->transid[0] == transid1) &&
             (tcConnectptr.p->transid[1] == transid2))) {
    TcConnectionrec *const regTcPtr = tcConnectptr.p;
    TRACE_OP(regTcPtr, "COMMIT");

    if (ERROR_INSERTED(5048) || ERROR_INSERTED(5049)) {
      jam();
      TablerecPtr tablePtr;
      tablePtr.i = tcConnectptr.p->tableref;
      ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);
      if (tcConnectptr.p->operation == ZUPDATE &&
          (ERROR_INSERT_EXTRA == 0 || ERROR_INSERT_EXTRA == tablePtr.i)) {
        g_eventLogger->info("COMMIT an UPDATE on table:%u", tablePtr.i);
        CRASH_INSERTION(5048);
        if (ERROR_INSERTED(5049)) {
          SET_ERROR_INSERT_VALUE(5048);
        }
      }
    }
    if (ERROR_INSERTED(5093)) {
      if (tcConnectptr.p->operation == ZREAD) {
        jam();
        CLEAR_ERROR_INSERT_VALUE;
        g_eventLogger->info("Delaying COMMIT for READ");
        sendSignalWithDelay(cownref, GSN_COMMIT, signal, 3000, 5);
        return;
      }
    }
    commitReqLab(signal, gci_hi, gci_lo, tcConnectptr);
    return;
  }  // if
  warningReport(signal, 1);
  return;
}  // Dblqh::execCOMMIT()

/* ************************************************************************>>
 *  COMMITREQ: Commit request from TC. This is the commit protocol used if
 *  one of the nodes is not behaving correctly. TC explicitly sends COMMITREQ
 *  to both the backup and primary node and gets a COMMITCONF back if the
 *  COMMIT was ok.
 * ************************************************************************>> */
void Dblqh::execCOMMITREQ(Signal *signal) {
  jamEntryDebug();
  Uint32 reqPtr = signal->theData[0];
  BlockReference reqBlockref = signal->theData[1];
  Uint32 gci_hi = signal->theData[2];
  Uint32 transid1 = signal->theData[3];
  Uint32 transid2 = signal->theData[4];
  BlockReference old_blockref = signal->theData[5];
  Uint32 tcOprec = signal->theData[6];
  Uint32 gci_lo = signal->theData[7];

  ndbrequire(signal->getLength() >= 8);

  if (ERROR_INSERTED(5004)) {
    systemErrorLab(signal, __LINE__);
  }
  if (ERROR_INSERTED(5017)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_COMMITREQ, signal, 2000,
                        signal->getLength());
    return;
  }  // if
  bool partial_fit_ok;
  if (reqBlockref == old_blockref) {
    jam();
    partial_fit_ok = false;
  } else {
    jam();
    partial_fit_ok = true;
  }
  TcConnectionrecPtr tcConnectptr;
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  if (findTransaction(transid1, transid2, tcOprec, old_blockref, true,
                      partial_fit_ok, tcConnectptr) != ZOK) {
    warningReport(signal, 5);
    return;
  }  // if
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  switch (regTcPtr->transactionState) {
    case TcConnectionrec::PREPARED:
    case TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL:
    case TcConnectionrec::LOG_COMMIT_WRITTEN_WAIT_SIGNAL:
      jam();
      /*-------------------------------------------------------*/
      /*       THE NORMAL CASE.                                */
      /*-------------------------------------------------------*/
      regTcPtr->reqBlockref = reqBlockref;
      regTcPtr->reqRef = reqPtr;
      regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;
      commitReqLab(signal, gci_hi, gci_lo, tcConnectptr);
      return;
    case TcConnectionrec::COMMITTED:
      jam();
      /*---------------------------------------------------------*/
      /*       FOR SOME REASON THE COMMIT PHASE HAVE BEEN        */
      /*       FINISHED AFTER A TIME OUT. WE NEED ONLY SEND A    */
      /*       COMMITCONF SIGNAL.                                */
      /*---------------------------------------------------------*/
      regTcPtr->reqBlockref = reqBlockref;
      regTcPtr->reqRef = reqPtr;
      regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;
      signal->theData[0] = regTcPtr->reqRef;
      signal->theData[1] = cownNodeid;
      signal->theData[2] = regTcPtr->transid[0];
      signal->theData[3] = regTcPtr->transid[1];
      sendSignal(regTcPtr->reqBlockref, GSN_COMMITCONF, signal, 4, JBB);
      break;
    case TcConnectionrec::WAIT_TUP_COMMIT:
      jam();
      regTcPtr->reqBlockref = reqBlockref;
      regTcPtr->reqRef = reqPtr;
      regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;
      /*empty*/;
      return;
    default:
      jam();
      ndbabort();
      return;
  }  // switch
  return;
}  // Dblqh::execCOMMITREQ()

/* ************************************************************************>>
 *  COMPLETE : Complete the transaction. Sent as a packed signal from TC.
 *  Works the same way as COMMIT protocol. This is the normal case with both
 *  primary and backup working (See COMMIT).
 * ************************************************************************>> */
void Dblqh::execCOMPLETE(Signal *signal) {
  jamEntryDebug();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  Uint32 transid1 = signal->theData[1];
  Uint32 transid2 = signal->theData[2];
  if (unlikely(!tcConnect_pool.getValidPtr(tcConnectptr))) {
    jam();
    warningReport(signal, 2);
    return;
  }  // if
  CRASH_INSERTION(5042);

  if (ERROR_INSERTED(5013)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_COMPLETE, signal, 2000, 3);
    return;
  }  // if
  if (ERROR_INSERTED(5014)) {
    SET_ERROR_INSERT_VALUE(5018);
    sendSignalWithDelay(cownref, GSN_COMPLETE, signal, 2000, 3);
    return;
  }  // if
  if (ERROR_INSERTED(5063) &&
      ((refToMain(signal->getSendersBlockRef()) == DBTC) ||
       signal->getSendersBlockRef() == reference())) {
    Uint32 save = signal->getSendersBlockRef();
    g_eventLogger->info("Delaying execCOMPLETE");
    sendSignalWithDelay(cownref, GSN_COMPLETE, signal, 2000,
                        signal->getLength());

    if (refToMain(save) == DBTC) {
      g_eventLogger->info("killing %u", refToNode(save));
      signal->theData[0] = 9999;
      sendSignal(numberToRef(CMVMI, refToNode(save)), GSN_NDB_TAMPER, signal, 1,
                 JBB);
    }
    return;
  }
  if (ERROR_INSERTED(5111)) {
    jam();
    if (tcConnectptr.p->tableref > 2) {
      jam();
      g_eventLogger->info("LQH %u delaying complete", instance());
      sendSignalWithDelay(cownref, GSN_COMPLETE, signal, 200,
                          signal->getLength());
      return;
    }
  }
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  if (likely(tcConnectptr.p->transactionState == TcConnectionrec::COMMITTED) &&
      (tcConnectptr.p->transid[0] == transid1) &&
      (tcConnectptr.p->transid[1] == transid2)) {
    TcConnectionrec *const regTcPtr = tcConnectptr.p;
    if (ERROR_INSERTED(5093)) {
      if ((tcConnectptr.p->seqNoReplica != 0) &&
          (tcConnectptr.p->operation == ZDELETE)) {
        jam();
        CLEAR_ERROR_INSERT_VALUE;
        g_eventLogger->info("Delaying COMPLETE for DELETE at Backup replica");
        sendSignalWithDelay(cownref, GSN_COMPLETE, signal, 1000, 3);
        return;
      }
    }
    TRACE_OP(regTcPtr, "COMPLETE");

    if (tcConnectptr.p->seqNoReplica != 0 &&
        tcConnectptr.p->activeCreat == Fragrecord::AC_NORMAL) {
      jam();
      localCommitLab(signal, tcConnectptr);
      return;
    } else if (tcConnectptr.p->seqNoReplica == 0) {
      jam();
      completeTransLastLab(signal, tcConnectptr);
      return;
    } else {
      jam();
      completeTransNotLastLab(signal, tcConnectptr);
      return;
    }
  }  // if
  if (tcConnectptr.p->transactionState != TcConnectionrec::COMMITTED) {
    jam();
    warningReport(signal, 2);
  } else {
    jam();
    warningReport(signal, 3);
  }  // if
}  // Dblqh::execCOMPLETE()

/* ************************************************************************>>
 * COMPLETEREQ: Complete request from TC. Same as COMPLETE but used if one
 * node is not working ok (See COMMIT).
 * ************************************************************************>> */
void Dblqh::execCOMPLETEREQ(Signal *signal) {
  jamEntryDebug();
  Uint32 reqPtr = signal->theData[0];
  BlockReference reqBlockref = signal->theData[1];
  Uint32 transid1 = signal->theData[2];
  Uint32 transid2 = signal->theData[3];
  BlockReference old_blockref = signal->theData[4];
  Uint32 tcOprec = signal->theData[5];
  if (ERROR_INSERTED(5005)) {
    systemErrorLab(signal, __LINE__);
  }
  if (ERROR_INSERTED(5018)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_COMPLETEREQ, signal, 2000, 6);
    return;
  }  // if
  bool partial_fit_ok;
  if (reqBlockref == old_blockref) {
    jam();
    partial_fit_ok = false;
  } else {
    jam();
    partial_fit_ok = true;
  }
  TcConnectionrecPtr tcConnectptr;
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  if (findTransaction(transid1, transid2, tcOprec, old_blockref, true,
                      partial_fit_ok, tcConnectptr) != ZOK) {
    jam();
    /*---------------------------------------------------------*/
    /*       FOR SOME REASON THE COMPLETE PHASE STARTED AFTER  */
    /*       A TIME OUT. THE TRANSACTION IS GONE. WE NEED TO   */
    /*       REPORT COMPLETION ANYWAY.                         */
    /*---------------------------------------------------------*/
    signal->theData[0] = reqPtr;
    signal->theData[1] = cownNodeid;
    signal->theData[2] = transid1;
    signal->theData[3] = transid2;
    sendSignal(reqBlockref, GSN_COMPLETECONF, signal, 4, JBB);
    warningReport(signal, 7);
    return;
  }  // if
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  switch (regTcPtr->transactionState) {
    case TcConnectionrec::COMMITTED:
      jam();
      regTcPtr->reqBlockref = reqBlockref;
      regTcPtr->reqRef = reqPtr;
      regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;
      /*empty*/;
      break;
      /*---------------------------------------------------------*/
      /*       THE NORMAL CASE.                                  */
      /*---------------------------------------------------------*/
    case TcConnectionrec::WAIT_TUP_COMMIT:
      jam();
      /*---------------------------------------------------------*/
      /*       FOR SOME REASON THE COMPLETE PHASE STARTED AFTER  */
      /*       A TIME OUT. WE HAVE SET THE PROPER VARIABLES SUCH */
      /*       THAT A COMPLETECONF WILL BE SENT WHEN COMPLETE IS */
      /*       FINISHED.                                         */
      /*---------------------------------------------------------*/
      regTcPtr->reqBlockref = reqBlockref;
      regTcPtr->reqRef = reqPtr;
      regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;
      return;
    default:
      jam();
      ndbabort();
      return;
  }  // switch
  if (regTcPtr->seqNoReplica != 0 &&
      regTcPtr->activeCreat != Fragrecord::AC_NR_COPY) {
    /**
     * TODO RONM: Align this code with execCOMPLETEREQ which
     * handles AC_IGNORED differently. Need to handle
     * cnewestGci and fragPtr.p->newestGci also for those
     * cases properly.
     */
    jamDebug();
    localCommitLab(signal, tcConnectptr);
  } else if (regTcPtr->seqNoReplica == 0) {
    jam();
    completeTransLastLab(signal, tcConnectptr);
  } else {
    jam();
    completeTransNotLastLab(signal, tcConnectptr);
  }
}  // Dblqh::execCOMPLETEREQ()

/* ************> */
/*  COMPLETED  > */
/* ************> */
void Dblqh::execLQHKEYCONF(Signal *signal) {
  LqhKeyConf *const lqhKeyConf = (LqhKeyConf *)signal->getDataPtr();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = lqhKeyConf->opPtr;
  if (unlikely(!tcConnect_pool.getValidPtr(tcConnectptr))) {
    errorReport(signal, 2);
    return;
  }  // if
  jamEntryDebug();
  TcConnectionrec::ConnectState connectState = tcConnectptr.p->connectState;
  switch (connectState) {
    case TcConnectionrec::LOG_CONNECTED:
      jam();
      completedLab(signal, tcConnectptr);
      return;
    case TcConnectionrec::COPY_CONNECTED:
      jam();
      if (ERROR_INSERTED(5106) && signal->getSendersBlockRef() != reference()) {
        g_eventLogger->info("LQH %u delaying copy LQHKEYCONF", instance());
        sendSignalWithDelay(reference(), GSN_LQHKEYCONF, signal, 500, 7);
        return;
      }
      setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
      copyCompletedLab(signal, tcConnectptr);
      release_frag_access(prim_tab_fragptr.p);
      return;
    default:
      jamLine(tcConnectptr.p->connectState);
      ndbabort();
  }  // switch
  return;
}  // Dblqh::execLQHKEYCONF()

/* ------------------------------------------------------------------------- */
/* -------                       COMMIT PHASE                        ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::commitReqLab(Signal *signal, Uint32 gci_hi, Uint32 gci_lo,
                         TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  TcConnectionrec::LogWriteState logWriteState = regTcPtr->logWriteState;
  TcConnectionrec::TransactionState transState = regTcPtr->transactionState;
  ndbassert(!m_is_in_query_thread);
  regTcPtr->gci_hi = gci_hi;
  regTcPtr->gci_lo = gci_lo;
  /**
   * TODO RONM: Ensure that cnewestGci and fragPtr.p->newestGci are kept in
   * synch in all possible node restart variants. Currently it isn't updated
   * when AC_IGNORED set sometimes and sometimes not. How can we ensure that
   * a starting node gets the proper setting of those variables after a copy
   * phase have been completed.
   */
  if (transState == TcConnectionrec::PREPARED) {
    if (logWriteState == TcConnectionrec::WRITTEN) {
      jam();
      regTcPtr->transactionState = TcConnectionrec::PREPARED_RECEIVED_COMMIT;
      TcConnectionrecPtr saveTcPtr = tcConnectptr;
      signal->theData[0] = regTcPtr->tupConnectrec;
      signal->theData[1] = gci_hi;
      signal->theData[2] = gci_lo;
      c_tup->execTUP_WRITELOG_REQ(signal);
      jamEntryDebug();
      if (regTcPtr->transactionState == TcConnectionrec::LOG_COMMIT_QUEUED) {
        jamDebug();
        return;
      }  // if
      ndbrequire(regTcPtr->transactionState ==
                 TcConnectionrec::LOG_COMMIT_WRITTEN);
      tcConnectptr = saveTcPtr;
    } else if (logWriteState == TcConnectionrec::NOT_STARTED) {
      jam();
    } else if (logWriteState == TcConnectionrec::NOT_WRITTEN) {
      jam();
      /*---------------------------------------------------------------------------*/
      /* IT IS A READ OPERATION OR OTHER OPERATION THAT DO NOT USE THE LOG. */
      /*---------------------------------------------------------------------------*/
      /*---------------------------------------------------------------------------*/
      /* THE LOG HAS NOT BEEN WRITTEN SINCE THE LOG FLAG WAS false. THIS CAN
       * OCCUR */
      /* WHEN WE ARE STARTING A NEW FRAGMENT. */
      /*---------------------------------------------------------------------------*/
      regTcPtr->logWriteState = TcConnectionrec::NOT_STARTED;
    } else {
      ndbrequire(logWriteState == TcConnectionrec::NOT_WRITTEN_WAIT);
      jam();
      /*---------------------------------------------------------------------------*/
      /* THE STATE WAS SET TO NOT_WRITTEN BY THE OPERATION BUT LATER A SCAN OF
       * ALL */
      /* OPERATION RECORD CHANGED IT INTO NOT_WRITTEN_WAIT. THIS INDICATES THAT
       * WE */
      /* ARE WAITING FOR THIS OPERATION TO COMMIT OR ABORT SO THAT WE CAN FIND
       * THE */
      /* STARTING GLOBAL CHECKPOINT OF THIS NEW FRAGMENT. */
      /*---------------------------------------------------------------------------*/
      checkScanTcCompleted(signal, tcConnectptr);
    }  // if
  } else if (transState == TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL) {
    jam();
    regTcPtr->transactionState = TcConnectionrec::LOG_COMMIT_QUEUED;
    return;
  } else if (transState == TcConnectionrec::LOG_COMMIT_WRITTEN_WAIT_SIGNAL) {
    jam();
  } else {
    warningReport(signal, 0);
    return;
  }  // if
  if (unlikely(regTcPtr->seqNoReplica == 0 ||
               regTcPtr->activeCreat == Fragrecord::AC_NR_COPY)) {
    jamDebug();
    localCommitLab(signal, tcConnectptr);
    return;
  }  // if
  commitReplyLab(signal, tcConnectptr.p);
}  // Dblqh::commitReqLab()

void Dblqh::execLQH_WRITELOG_REQ(Signal *signal) {
  jamEntryDebug();
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  ndbrequire(tcConnect_pool.getUncheckedPtrRW(tcConnectptr));
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  Uint32 gci_hi = signal->theData[1];
  Uint32 gci_lo = signal->theData[2];
  Uint32 newestGci = cnewestGci;
  TcConnectionrec::LogWriteState logWriteState = regTcPtr->logWriteState;
  TcConnectionrec::TransactionState transState = regTcPtr->transactionState;
  ndbrequire(Magic::check_ptr(regTcPtr));
  ndbassert(!m_is_in_query_thread);
  regTcPtr->gci_hi = gci_hi;
  regTcPtr->gci_lo = gci_lo;
  if (gci_hi > newestGci) {
    jamDebug();
    /* -------------------------------------------------------------------------
     */
    /*       KEEP TRACK OF NEWEST GLOBAL CHECKPOINT THAT LQH HAS HEARD OF. */
    /* -------------------------------------------------------------------------
     */
    cnewestGci = gci_hi;
    DEB_NEWEST_GCI(("(%u)(%u) cnewestGci = %u, line: %u", instance(),
                    m_is_query_block, cnewestGci, __LINE__));
  }  // if
  if (logWriteState == TcConnectionrec::WRITTEN) {
    /*---------------------------------------------------------------------------*/
    /* I NEED TO INSERT A COMMIT LOG RECORD SINCE WE ARE WRITING LOG IN THIS */
    /* TRANSACTION. */
    /*---------------------------------------------------------------------------*/
    jamDebug();
    LogPartRecord *regLogPartPtr;
    regLogPartPtr = regTcPtr->m_log_part_ptr_p;
    lock_log_part(regLogPartPtr);
    Uint32 noOfFreeLogPages = count_free_log_pages(regLogPartPtr);
    if (!regLogPartPtr->m_log_complete_queue.isEmpty() ||
        (noOfFreeLogPages == 0)) {
      /**
       * There is no space in the REDO log buffer OR there is queued entries
       * to write COMMIT/ABORT REDO log messages. In this case we write only
       * through the REDO log queue.
       */
      /*---------------------------------------------------------------------------*/
      /* THIS LOG PART WAS CURRENTLY ACTIVE WRITING ANOTHER LOG RECORD. WE MUST
       */
      /* WAIT UNTIL THIS PART HAS COMPLETED ITS OPERATION. */
      /*---------------------------------------------------------------------------*/
      // We must delay the write of commit info to the log to safe-guard against
      // a crash due to lack of log pages. We temporary stop all log writes to
      // this log part to ensure that we don't get a buffer explosion in the
      // delayed signal buffer instead.
      /*---------------------------------------------------------------------------*/
      linkWaitLog(signal, regLogPartPtr, regLogPartPtr->m_log_complete_queue,
                  tcConnectptr.p);
      if (transState == TcConnectionrec::PREPARED) {
        jam();
        regTcPtr->transactionState =
            TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL;
      } else {
        jam();
        ndbrequire(transState == TcConnectionrec::PREPARED_RECEIVED_COMMIT);
        regTcPtr->transactionState = TcConnectionrec::LOG_COMMIT_QUEUED;
      }  // if
      unlock_log_part(regLogPartPtr);
      return;
    }  // if
    writeCommitLog(signal, tcConnectptr.p, regLogPartPtr);
    removeLogTcrec(regTcPtr, regLogPartPtr);
    unlock_log_part(regLogPartPtr);
    if (transState == TcConnectionrec::PREPARED) {
      jam();
      regTcPtr->transactionState =
          TcConnectionrec::LOG_COMMIT_WRITTEN_WAIT_SIGNAL;
    } else {
      jamDebug();
      ndbrequire(transState == TcConnectionrec::PREPARED_RECEIVED_COMMIT);
      regTcPtr->transactionState = TcConnectionrec::LOG_COMMIT_WRITTEN;
    }  // if
  }    // if
}  // Dblqh::execLQH_WRITELOG_REQ()

void Dblqh::localCommitLab(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr) {
  FragrecordPtr regFragptr;
  regFragptr.i = tcConnectptr.p->fragmentptr;
  c_fragment_pool.getPtr(regFragptr);
  Fragrecord::FragStatus status = regFragptr.p->fragStatus;
  fragptr = regFragptr;
  m_tc_connect_ptr = tcConnectptr;
  switch (status) {
    case Fragrecord::FSACTIVE:
    case Fragrecord::CRASH_RECOVERING:
    case Fragrecord::ACTIVE_CREATION:
      commitContinueAfterBlockedLab(signal, tcConnectptr);
      return;
    case Fragrecord::FREE:
      ndbabort();
    case Fragrecord::DEFINED:
      ndbabort();
    case Fragrecord::REMOVING:
      ndbabort();
    default:
      ndbabort();
  }  // switch
}  // Dblqh::localCommitLab()

void Dblqh::commitContinueAfterBlockedLab(
    Signal *signal, const TcConnectionrecPtr tcConnectptr) {
  /* -------------------------------------------------------------------------
   */
  /*INPUT:          TC_CONNECTPTR           ACTIVE OPERATION RECORD */
  /* -------------------------------------------------------------------------
   */
  /* -------------------------------------------------------------------------
   */
  /*CONTINUE HERE AFTER BEING BLOCKED FOR A WHILE DURING LOCAL CHECKPOINT. */
  /*The operation is already removed from the active list since there is no */
  /*chance for any real-time breaks before we need to release it. */
  /* -------------------------------------------------------------------------
   */
  /*ALSO AFTER NORMAL PROCEDURE WE CONTINUE */
  /*WE MUST COMMIT TUP BEFORE ACC TO ENSURE THAT NO ONE RACES IN AND SEES A */
  /*DIRTY STATE IN TUP. */
  /* -------------------------------------------------------------------------
   */
  Ptr<TcConnectionrec> regTcPtr = tcConnectptr;
  Ptr<Fragrecord> regFragptr = fragptr;
  Uint32 operation = regTcPtr.p->operation;
  Uint32 dirtyOp = regTcPtr.p->dirtyOp;
  Uint32 opSimple = regTcPtr.p->opSimple;
  bool normalProtocol =
      (regTcPtr.p->m_flags & TcConnectionrec::OP_NORMAL_PROTOCOL);

  if (regTcPtr.p->activeCreat != Fragrecord::AC_IGNORED) {
    if (operation != ZREAD) {
      Uint32 sig0 = regTcPtr.p->tupConnectrec;
      ndbassert(!m_is_in_query_thread);
      regTcPtr.p->transactionState = TcConnectionrec::WAIT_TUP_COMMIT;
      prefetch_op_record_3((Uint32 *)regTcPtr.p->accConnectPtrP);

      Uint32 ret_code = c_tup->exec_prepare_tup_commit(sig0);
      if (likely(ret_code == ZTUP_NOT_COMMITTED)) {
        TupCommitReq *const tupCommitReq =
            (TupCommitReq *)signal->getDataPtrSend();
        jam();
        tupCommitReq->opPtr = sig0;
        tupCommitReq->gci_hi = regTcPtr.p->gci_hi;
        tupCommitReq->hashValue = regTcPtr.p->hashValue;
        tupCommitReq->diskpage = RNIL;
        tupCommitReq->gci_lo = regTcPtr.p->gci_lo;
        tupCommitReq->transId1 = regTcPtr.p->transid[0];
        tupCommitReq->transId2 = regTcPtr.p->transid[1];
        if (TRACENR_FLAG) {
          TRACENR("COMMIT: ");
          switch (regTcPtr.p->operation) {
            case ZREAD:
              TRACENR("READ");
              break;
            case ZUPDATE:
              TRACENR("UPDATE");
              break;
            case ZWRITE:
              TRACENR("WRITE");
              break;
            case ZINSERT:
              TRACENR("INSERT");
              break;
            case ZDELETE:
              TRACENR("DELETE");
              break;
            case ZUNLOCK:
              TRACENR("UNLOCK");
              break;
          }

          TRACENR(" tab: " << regTcPtr.p->tableref << " frag: "
                           << regTcPtr.p->fragmentid << " activeCreat: "
                           << (Uint32)regTcPtr.p->activeCreat);
          if (LqhKeyReq::getNrCopyFlag(regTcPtr.p->reqinfo)) TRACENR(" NrCopy");
          if (LqhKeyReq::getRowidFlag(regTcPtr.p->reqinfo))
            TRACENR(" rowid: " << regTcPtr.p->m_row_id);
          TRACENR(" key: " << getKeyInfoWordOrZero(regTcPtr.p, 0));
        }
        /* Further processing of the commit is controlled from DBTUP. */
        Uint32 commit_result = c_tup->exec_tup_commit(signal);
        if (TRACENR_FLAG) {
          if (commit_result == ZTUP_WAIT_COMMIT) TRACENR(" TIMESLICE");
          TRACENR(endl);
        }
        return;
      } else if (ret_code == ZTUP_WAIT_COMMIT) {
        /**
         * Commit is waiting for disk pages to arrive, our commit will be on
         * hold as well until the disk pages arrive. The remainder of the
         * commit processing is controlled from DBTUP.
         *
         * It might even be that the operation is fully released when
         * we arrive here, so not safe to do any checks here.
         */
        return;
      } else {
        /**
         * Commit operation have already been performed, no work for TUP to
         * do and we can simply proceed with the cleanup of this operation.
         * No locks are required for this part since we will merely clean up
         * operation records.
         */
        ndbrequire(ret_code == ZTUP_COMMITTED);
      }

      TRACE_OP(regTcPtr.p, "ACC_COMMITREQ");

      c_acc->execACC_COMMITREQ(signal, regTcPtr.p->accConnectrec,
                               regTcPtr.p->accConnectPtrP);
    } else {
      if (!dirtyOp) {
        TRACE_OP(regTcPtr.p, "ACC_COMMITREQ");
        c_acc->execACC_COMMITREQ(signal, regTcPtr.p->accConnectrec,
                                 regTcPtr.p->accConnectPtrP);
      }

      if (dirtyOp && normalProtocol == 0) {
        jamDebug();
        /**
         * The dirtyRead does not send anything but TRANSID_AI from LDM
         */
        fragptr = regFragptr;
        cleanUp(signal, regTcPtr);
        return;
      }

      /**
       * The simpleRead will send a LQHKEYCONF
       *   but have already released the locks
       */
      if (opSimple) {
        fragptr = regFragptr;
        packLqhkeyreqLab(signal, regTcPtr);
        return;
      }
      ndbassert(!m_is_in_query_thread);
    }
  }  // if
  jamEntryDebug();
  fragptr = regFragptr;
  tupcommit_conf(signal, regTcPtr, regFragptr.p);
}

void Dblqh::tupcommit_conf_callback(Signal *signal, Uint32 tcPtrI) {
  jamEntry();

  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = tcPtrI;
  ndbrequire(tcConnect_pool.getUncheckedPtrRW(tcConnectptr));
  TcConnectionrec *tcPtr = tcConnectptr.p;

  ndbrequire(tcPtr->transactionState == TcConnectionrec::WAIT_TUP_COMMIT);
  prefetch_op_record_3((Uint32 *)tcPtr->accConnectPtrP);

  TRACE_OP(tcPtr, "ACC_COMMITREQ");

  ndbrequire(Magic::check_ptr(tcPtr));
  c_acc->execACC_COMMITREQ(signal, tcPtr->accConnectrec, tcPtr->accConnectPtrP);
  jamEntryDebug();
  /**
   * Release fragment access lock if we hold it, we hold it during the
   * real commit and we want to release it before calling tupcommit_conf
   * since this will only send signals that require no exclusive access.
   */
  release_frag_access(fragptr.p);
  tcConnectptr.i = tcPtrI;
  tcConnectptr.p = tcPtr;
  tupcommit_conf(signal, tcConnectptr, fragptr.p);
}

void Dblqh::tupcommit_conf(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr,
                           Fragrecord *regFragptr) {
  const TcConnectionrec *tcPtrP = tcConnectptr.p;
  Uint32 dirtyOp = tcPtrP->dirtyOp;
  Uint32 seqNoReplica = tcPtrP->seqNoReplica;
  Uint32 activeCreat = tcPtrP->activeCreat;
  ndbassert(!m_is_in_query_thread);
  if (tcPtrP->gci_hi > regFragptr->newestGci && tcPtrP->operation != ZREAD &&
      tcPtrP->operation != ZREAD_EX) {
    jamDebug();
    /* -------------------------------------------------------------------------
     */
    /*IT IS THE FIRST TIME THIS GLOBAL CHECKPOINT IS INVOLVED IN UPDATING THIS
     */
    /*FRAGMENT. UPDATE THE VARIABLE THAT KEEPS TRACK OF NEWEST GCI IN FRAGMENT
     */
    /* -------------------------------------------------------------------------
     */
    ndbassert(tcPtrP->operation != ZUNLOCK);
    regFragptr->newestGci = tcPtrP->gci_hi;
    DEB_EXTRA_LCP(("(%u)op_type: %u, newestGci: %u, tableId: %u, fragId: %u",
                   instance(), tcPtrP->operation, regFragptr->newestGci,
                   regFragptr->tabRef, regFragptr->fragId));
  }  // if
  if (dirtyOp != ZTRUE) {
    if (seqNoReplica == 0 || activeCreat == Fragrecord::AC_NR_COPY) {
      jamDebug();
      commitReplyLab(signal, tcConnectptr.p);
      return;
    }  // if
    if (seqNoReplica == 0) {
      jamDebug();
      completeTransLastLab(signal, tcConnectptr);
    } else {
      jamDebug();
      completeTransNotLastLab(signal, tcConnectptr);
    }
    return;
  } else {
    /* -------------------------------------------------------------------------
     */
    /*WE MUST HANDLE DIRTY WRITES IN A SPECIAL WAY. THESE OPERATIONS WILL NOT */
    /*SEND ANY COMMIT OR COMPLETE MESSAGES TO OTHER NODES. THEY WILL MERELY SEND
     */
    /*THOSE SIGNALS INTERNALLY. */
    /* -------------------------------------------------------------------------
     */
    if (tcPtrP->abortState == TcConnectionrec::ABORT_IDLE) {
      jamDebug();
      if (unlikely(activeCreat == Fragrecord::AC_NR_COPY)) {
        jam();
        ndbrequire(LqhKeyReq::getNrCopyFlag(tcPtrP->reqinfo));
        ndbrequire(tcPtrP->m_nr_delete.m_cnt == 0);
      }
      packLqhkeyreqLab(signal, tcConnectptr);
    } else {
      ndbrequire(tcPtrP->abortState != TcConnectionrec::NEW_FROM_TC);
      jam();
      sendLqhTransconf(signal, LqhTransConf::Committed, tcConnectptr);
      cleanUp(signal, tcConnectptr);
    }  // if
  }    // if
}  // Dblqh::commitContinueAfterBlockedLab()

void Dblqh::commitReplyLab(Signal *signal, TcConnectionrec *const regTcPtr) {
  /* -------------------------------------------------------------- */
  /* BACKUP AND STAND-BY REPLICAS ONLY UPDATE THE TRANSACTION STATE */
  /* -------------------------------------------------------------- */
  TcConnectionrec::AbortState abortState = regTcPtr->abortState;
  regTcPtr->transactionState = TcConnectionrec::COMMITTED;
  if (abortState == TcConnectionrec::ABORT_IDLE) {
    Uint32 clientBlockref = regTcPtr->clientBlockref;
    if (regTcPtr->seqNoReplica == 0) {
      jamDebug();
      sendCommittedTc(signal, clientBlockref, regTcPtr);
      return;
    } else {
      jamDebug();
      sendCommitLqh(signal, clientBlockref, regTcPtr);
      return;
    }  // if
  } else if (regTcPtr->abortState == TcConnectionrec::REQ_FROM_TC) {
    jam();
    signal->theData[0] = regTcPtr->reqRef;
    signal->theData[1] = cownNodeid;
    signal->theData[2] = regTcPtr->transid[0];
    signal->theData[3] = regTcPtr->transid[1];
    sendSignal(regTcPtr->reqBlockref, GSN_COMMITCONF, signal, 4, JBB);
  } else {
    ndbrequire(regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC);
  }  // if
  return;
}  // Dblqh::commitReplyLab()

/* ------------------------------------------------------------------------- */
/* -------                COMPLETE PHASE                             ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::completeTransNotLastLab(Signal *signal,
                                    const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (likely(regTcPtr->abortState == TcConnectionrec::ABORT_IDLE)) {
    Uint32 clientBlockref = regTcPtr->clientBlockref;
    jamDebug();
    sendCompleteLqh(signal, clientBlockref, regTcPtr);
    cleanUp(signal, tcConnectptr);
  } else {
    jamDebug();
    completeUnusualLab(signal, tcConnectptr);
  }  // if
}  // Dblqh::completeTransNotLastLab()

void Dblqh::completeTransLastLab(Signal *signal,
                                 const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (likely(regTcPtr->abortState == TcConnectionrec::ABORT_IDLE)) {
    Uint32 clientBlockref = regTcPtr->clientBlockref;
    jam();
    /* -------------------------------------------------------------------------
     */
    /*DIRTY WRITES WHICH ARE LAST IN THE CHAIN OF REPLICAS WILL SEND COMPLETED
     */
    /*INSTEAD OF SENDING PREPARED TO THE TC (OR OTHER INITIATOR OF OPERATION).
     */
    /* -------------------------------------------------------------------------
     */
    sendCompletedTc(signal, clientBlockref, regTcPtr);
    cleanUp(signal, tcConnectptr);
  } else {
    jamDebug();
    completeUnusualLab(signal, tcConnectptr);
  }  // if
}  // Dblqh::completeTransLastLab()

void Dblqh::completeUnusualLab(Signal *signal,
                               const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (regTcPtr->abortState == TcConnectionrec::ABORT_FROM_TC) {
    jam();
    sendAborted(signal, tcConnectptr);
  } else if (regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC) {
    jam();
  } else {
    ndbrequire(regTcPtr->abortState == TcConnectionrec::REQ_FROM_TC);
    jam();
    signal->theData[0] = regTcPtr->reqRef;
    signal->theData[1] = cownNodeid;
    signal->theData[2] = regTcPtr->transid[0];
    signal->theData[3] = regTcPtr->transid[1];
    sendSignal(regTcPtr->reqBlockref, GSN_COMPLETECONF, signal, 4, JBB);
  }  // if
  cleanUp(signal, tcConnectptr);
}  // Dblqh::completeUnusualLab()

/* ========================================================================= */
/* =======                        RELEASE TC CONNECT RECORD          ======= */
/*                                                                           */
/*       RELEASE A TC CONNECT RECORD TO THE FREELIST.                        */
/* ========================================================================= */
void Dblqh::releaseTcrec(Signal *signal, TcConnectionrecPtr locTcConnectptr) {
  jamDebug();
  if (unlikely(locTcConnectptr.p->m_dealloc_state ==
               TcConnectionrec::DA_DEALLOC_COUNT_ZOMBIE)) {
    jam();
    /**
     * Need to keep this Tcrec around a little extra time to track
     * the ref count on TUP storage deallocation
     */
    return;
  }

  ndbassert(locTcConnectptr.p->hashIndex == RNIL);
  const Uint32 op = locTcConnectptr.p->operation;

  ndbassert(locTcConnectptr.p->tcScanRec == RNIL);
  ndbassert(locTcConnectptr.p->m_committed_log_space == 0);

  TablerecPtr tabPtr;
  tabPtr.i = locTcConnectptr.p->tableref;
  if (likely(tabPtr.i != RNIL)) {
    ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);

    /**
     * Normal case
     */
    if (op == ZREAD || op == ZUNLOCK) {
      Uint32 pre_usageCountR = tabPtr.p->usageCountR--;
      ndbrequire(pre_usageCountR > 0);
    } else {
      Uint32 pre_usageCountW = tabPtr.p->usageCountW--;
      ndbrequire(pre_usageCountW > 0);
    }
  }
  if (likely(locTcConnectptr.i < ctcConnectReserved)) {
    jamDebug();
    const Uint32 firstFree = cfirstfreeTcConrec;
    Uint32 numFree = ctcNumFree;
    locTcConnectptr.p->tcTimer = 0;
    locTcConnectptr.p->transactionState = TcConnectionrec::TC_NOT_CONNECTED;
    locTcConnectptr.p->nextTcConnectrec = firstFree;
    cfirstfreeTcConrec = locTcConnectptr.i;
    ctcNumFree = numFree + 1;
  } else {
    jam();
    release_op_rec(locTcConnectptr);
  }
}  // Dblqh::releaseTcrec()

void Dblqh::releaseTcrecLog(Signal *signal,
                            TcConnectionrecPtr locTcConnectptr) {
  jamDebug();
  Uint32 numFree = ctcNumFree;
  ndbassert(locTcConnectptr.p->hashIndex == RNIL);
  locTcConnectptr.p->tcTimer = 0;
  locTcConnectptr.p->transactionState = TcConnectionrec::TC_NOT_CONNECTED;
  locTcConnectptr.p->nextTcConnectrec = cfirstfreeTcConrec;
  cfirstfreeTcConrec = locTcConnectptr.i;
  ctcNumFree = numFree + 1;
}  // Dblqh::releaseTcrecLog()

/* ------------------------------------------------------------------------- */
/* -------                       ABORT PHASE                         ------- */
/*                                                                           */
/*THIS PART IS USED AT ERRORS THAT CAUSE ABORT OF TRANSACTION.               */
/* ------------------------------------------------------------------------- */
void Dblqh::remove_commit_marker(TcConnectionrec *const regTcPtr) {
  Ptr<CommitAckMarker> tmp;
  Uint32 commitAckMarker = regTcPtr->commitAckMarker;
  regTcPtr->commitAckMarker = RNIL;
  if (commitAckMarker == RNIL) return;
  jam();
  tmp.i = commitAckMarker;
  ndbrequire(m_commitAckMarkerPool.getValidPtr(tmp));
#ifdef MARKER_TRACE
  g_eventLogger->info("%u remove marker[%.8x %.8x] op: %u ref: %u", instance(),
                      tmp.p->transid1, tmp.p->transid2,
                      Uint32(regTcPtr - tcConnectionrec),
                      tmp.p->reference_count);
#endif
  if (tmp.p->in_hash == false) {
    g_eventLogger->info(
        "%u remove_commit_marker failed[%.8x %.8x]"
        " removed_by_fail_api = %u"
        " ack marker transid[%.8x %.8x]"
        " ack marker ref count = %d",
        instance(), regTcPtr->transid[0], regTcPtr->transid[1],
        tmp.p->removed_by_fail_api, tmp.p->transid1, tmp.p->transid2,
        tmp.p->reference_count);
    ndbrequire(tmp.p->reference_count == 0);
    ndbabort();
    return;
  }
  ndbrequire(tmp.p->reference_count > 0);
  if (regTcPtr->transid[0] != tmp.p->transid1 ||
      regTcPtr->transid[1] != tmp.p->transid2) {
    /**
     * We refer to a commit ack marker that have already been removed
     * and even reused by another transaction.
     */
    g_eventLogger->info(
        "%u remove_commit_marker failed, moved[%.8x %.8x]"
        " removed_by_fail_api = %u"
        " ack marker transid[%.8x %.8x]"
        " ack marker ref count = %d",
        instance(), regTcPtr->transid[0], regTcPtr->transid[1],
        tmp.p->removed_by_fail_api, tmp.p->transid1, tmp.p->transid2,
        tmp.p->reference_count);
    ndbabort();
    return;
  }
  tmp.p->reference_count--;
  if (tmp.p->reference_count == 0) {
    jam();
    CommitAckMarker key;
    key.transid1 = regTcPtr->transid[0];
    key.transid2 = regTcPtr->transid[1];
    CommitAckMarkerPtr removedPtr;
    ndbrequire(m_commitAckMarkerHash.remove(removedPtr, key));
    ndbrequire(removedPtr.i == tmp.i);
    removedPtr.p->in_hash = false;
    m_commitAckMarkerPool.release(removedPtr);
    checkPoolShrinkNeed(DBLQH_COMMIT_ACK_MARKER_TRANSIENT_POOL_INDEX,
                        m_commitAckMarkerPool);
  }
}

/* ***************************************************>> */
/*  ABORT: Abort transaction in connection. Sender TC.   */
/*  This is the normal protocol (See COMMIT)             */
/* ***************************************************>> */
void Dblqh::execABORT(Signal *signal) {
  jamEntry();
  if (ERROR_INSERTED(5096)) {
    jam();
    g_eventLogger->info("LQH %u : ERRINS 5096 Stalling ABORT", instance());
    sendSignalWithDelay(reference(), GSN_ABORT, signal, 10, 4);
    return;
  }
  if (ERROR_INSERTED(5095)) {
    jam();
    g_eventLogger->info(
        "LQH %u : ERRINS 5095 Passing abort and setting ERRINS 5096",
        instance());
    SET_ERROR_INSERT_VALUE(5096);
  }

  Uint32 tcOprec = signal->theData[0];
  BlockReference tcBlockref = signal->theData[1];
  Uint32 transid1 = signal->theData[2];
  Uint32 transid2 = signal->theData[3];
  CRASH_INSERTION(5003);
  if (ERROR_INSERTED(5015)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_ABORT, signal, 2000, 4);
    return;
  }  // if
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  TcConnectionrecPtr tcConnectptr;
  if (findTransaction(transid1, transid2, tcOprec, tcBlockref, true, false,
                      tcConnectptr) != ZOK) {
    jam();
    if (ERROR_INSERTED(5039) &&
        refToNode(signal->getSendersBlockRef()) != getOwnNodeId()) {
      jam();
      SET_ERROR_INSERT_VALUE(5040);
      return;
    }

    if (ERROR_INSERTED(5040) &&
        refToNode(signal->getSendersBlockRef()) != getOwnNodeId()) {
      jam();
      SET_ERROR_INSERT_VALUE(5003);
      return;
    }

    /* -------------------------------------------------------------------------
     */
    // SEND ABORTED EVEN IF NOT FOUND.
    // THE TRANSACTION MIGHT NEVER HAVE ARRIVED HERE.
    /* -------------------------------------------------------------------------
     */
    signal->theData[0] = tcOprec;
    signal->theData[1] = transid1;
    signal->theData[2] = transid2;
    signal->theData[3] = cownNodeid;
    signal->theData[4] = ZTRUE;
    sendSignal(tcBlockref, GSN_ABORTED, signal, 5, JBB);
    warningReport(signal, 8);
    return;
  }  // if

  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (ERROR_INSERTED(5100)) {
    SET_ERROR_INSERT_VALUE(5101);
    return;
  }
  CRASH_INSERTION2(5101, regTcPtr->nextReplica != ZNIL);

  /* -------------------------------------------------------------------------
   */
  /*A GUIDING DESIGN PRINCIPLE IN HANDLING THESE ERROR SITUATIONS HAVE BEEN */
  /*KEEP IT SIMPLE. THUS WE RATHER INSERT A WAIT AND SET THE ABORT_STATE TO */
  /*ACTIVE RATHER THAN WRITE NEW CODE TO HANDLE EVERY SPECIAL SITUATION. */
  /* -------------------------------------------------------------------------
   */
  if (regTcPtr->nextReplica != ZNIL) {
    /* -------------------------------------------------------------------------
     */
    // We will immediately send the ABORT message also to the next LQH node in
    // line.
    /* -------------------------------------------------------------------------
     */
    FragrecordPtr Tfragptr;
    Tfragptr.i = regTcPtr->fragmentptr;
    c_fragment_pool.getPtr(Tfragptr);
    Uint32 Tnode = regTcPtr->nextReplica;
    Uint32 instanceKey = Tfragptr.p->lqhInstanceKey;
    Uint32 instanceNo = getInstanceNo(Tnode, instanceKey);
    BlockReference TLqhRef = numberToRef(DBLQH, instanceNo, Tnode);
    signal->theData[0] = regTcPtr->tcOprec;
    signal->theData[1] = regTcPtr->tcBlockref;
    signal->theData[2] = regTcPtr->transid[0];
    signal->theData[3] = regTcPtr->transid[1];
    sendSignal(TLqhRef, GSN_ABORT, signal, 4, JBB);
  }  // if
  regTcPtr->abortState = TcConnectionrec::ABORT_FROM_TC;

  remove_commit_marker(regTcPtr);
  TRACE_OP(regTcPtr, "ABORT");

  abortStateHandlerLab(signal, tcConnectptr);

  return;
}  // Dblqh::execABORT()

/* ************************************************************************>>
 *  ABORTREQ: Same as ABORT but used in case one node isn't working ok.
 *  (See COMMITREQ)
 * ************************************************************************>> */
void Dblqh::execABORTREQ(Signal *signal) {
  jamEntry();
  Uint32 reqPtr = signal->theData[0];
  BlockReference reqBlockref = signal->theData[1];
  Uint32 transid1 = signal->theData[2];
  Uint32 transid2 = signal->theData[3];
  BlockReference old_blockref = signal->theData[4];
  Uint32 tcOprec = signal->theData[5];
  if (ERROR_INSERTED(5006)) {
    systemErrorLab(signal, __LINE__);
  }
  if (ERROR_INSERTED(5016)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_ABORTREQ, signal, 2000, 6);
    return;
  }  // if
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  bool partial_fit_ok;
  if (reqBlockref == old_blockref) {
    jam();
    partial_fit_ok = false;
  } else {
    jam();
    partial_fit_ok = true;
  }
  TcConnectionrecPtr tcConnectptr;
  if (findTransaction(transid1, transid2, tcOprec, old_blockref, true,
                      partial_fit_ok, tcConnectptr) != ZOK) {
    signal->theData[0] = reqPtr;
    signal->theData[2] = cownNodeid;
    signal->theData[3] = transid1;
    signal->theData[4] = transid2;
    sendSignal(reqBlockref, GSN_ABORTCONF, signal, 5, JBB);
    warningReport(signal, 9);
    return;
  }  // if
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (unlikely(regTcPtr->transactionState != TcConnectionrec::PREPARED)) {
    jam();
    warningReport(signal, 10);
    return;
  }  // if
  regTcPtr->reqBlockref = reqBlockref;
  regTcPtr->reqRef = reqPtr;
  regTcPtr->abortState = TcConnectionrec::REQ_FROM_TC;

  abortCommonLab(signal, tcConnectptr);
  return;
}  // Dblqh::execABORTREQ()

/* ************>> */
/*  ACC_TO_REF  > */
/* ************>> */
void Dblqh::execACC_TO_REF(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr) {
  jamEntry();
  terrorCode = signal->theData[1];
  abortErrorLab(signal, tcConnectptr);
}  // Dblqh::execACC_TO_REF()

/* ************> */
/*  ACCKEYREF  > */
/* ************> */
void Dblqh::execACCKEYREF(Signal *signal) {
  jamEntry();
  terrorCode = signal->theData[1];
  setup_key_pointers(signal->theData[0], false);
  continueACCKEYREF(signal, m_tc_connect_ptr);
}

void Dblqh::continueACCKEYREF(Signal *signal, TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const tcPtr = tcConnectptr.p;
  switch (tcPtr->transactionState) {
    case TcConnectionrec::WAIT_ACC:
      jam();
      if (!m_is_in_query_thread) {
        c_fragment_pool.getPtr(tcPtr->fragmentptr)->m_useStat.m_keyRefCount++;
      }
      break;
    case TcConnectionrec::WAIT_ACC_ABORT:
      jam();
      ndbassert(!m_is_in_query_thread);
      /* -------------------------------------------------------------------------
       */
      /*IGNORE SINCE ABORT OF THIS OPERATION IS ONGOING ALREADY. */
      /* -------------------------------------------------------------------------
       */
      return;
    default:
      ndbabort();
  }  // switch
  const Uint32 errCode = terrorCode;
  tcPtr->errorCode = errCode;

  if (TRACENR_FLAG) {
    TRACENR("ACCKEYREF: " << errCode << " ");
    switch (tcPtr->operation) {
      case ZREAD:
        TRACENR("READ");
        break;
      case ZUPDATE:
        TRACENR("UPDATE");
        break;
      case ZWRITE:
        TRACENR("WRITE");
        break;
      case ZINSERT:
        TRACENR("INSERT");
        break;
      case ZDELETE:
        TRACENR("DELETE");
        break;
      case ZUNLOCK:
        TRACENR("UNLOCK");
        break;
      default:
        TRACENR("<Unknown: " << tcPtr->operation << ">");
        break;
    }

    TRACENR(" tab: " << tcPtr->tableref << " frag: " << tcPtr->fragmentid
                     << " activeCreat: " << (Uint32)tcPtr->activeCreat);
    if (LqhKeyReq::getNrCopyFlag(tcPtr->reqinfo)) TRACENR(" NrCopy");
    if (LqhKeyReq::getRowidFlag(tcPtr->reqinfo))
      TRACENR(" rowid: " << tcPtr->m_row_id);
    TRACENR(" key: " << getKeyInfoWordOrZero(tcPtr, 0));
    TRACENR(endl);
  }

  ndbrequire(tcPtr->activeCreat == Fragrecord::AC_NORMAL);
  ndbrequire(!LqhKeyReq::getNrCopyFlag(tcPtr->reqinfo));

  /**
   * Not only primary replica can get ZTUPLE_ALREADY_EXIST || ZNO_TUPLE_FOUND
   *
   * 1) op1 - primary insert ok
   * 2) op1 - backup insert fail (log full or what ever)
   * 3) op1 - delete ok @ primary
   * 4) op1 - delete fail @ backup
   *
   * -> ZNO_TUPLE_FOUND is possible
   *
   * 1) op1 primary delete ok
   * 2) op1 backup delete fail (log full or what ever)
   * 3) op2 insert ok @ primary
   * 4) op2 insert fail @ backup
   *
   * -> ZTUPLE_ALREADY_EXIST
   */
  tcPtr->abortState = TcConnectionrec::ABORT_FROM_LQH;
  abortCommonLab(signal, tcConnectptr);
}  // Dblqh::execACCKEYREF()

void Dblqh::localAbortStateHandlerLab(Signal *signal,
                                      const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (regTcPtr->abortState != TcConnectionrec::ABORT_IDLE) {
    jam();
    return;
  }  // if
  regTcPtr->abortState = TcConnectionrec::ABORT_FROM_LQH;
  regTcPtr->errorCode = terrorCode;
  abortStateHandlerLab(signal, tcConnectptr);
  return;
}  // Dblqh::localAbortStateHandlerLab()

void Dblqh::abortStateHandlerLab(Signal *signal,
                                 const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  switch (regTcPtr->transactionState) {
    case TcConnectionrec::PREPARED:
      jam();
      /* -------------------------------------------------------------------------
       */
      /*THE OPERATION IS ALREADY PREPARED AND SENT TO THE NEXT LQH OR BACK TO
       * TC.  */
      /*WE CAN SIMPLY CONTINUE WITH THE ABORT PROCESS. */
      /*IF IT WAS A CHECK FOR TRANSACTION STATUS THEN WE REPORT THE STATUS TO
       * THE  */
      /*NEW TC AND CONTINUE WITH THE NEXT OPERATION IN LQH. */
      /* -------------------------------------------------------------------------
       */
      if (regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC) {
        jam();
        sendLqhTransconf(signal, LqhTransConf::Prepared, tcConnectptr);
        return;
      }  // if
      break;
    case TcConnectionrec::LOG_COMMIT_WRITTEN_WAIT_SIGNAL:
    case TcConnectionrec::LOG_COMMIT_QUEUED_WAIT_SIGNAL:
      jam();
      /* -------------------------------------------------------------------------
       */
      // We can only reach these states for multi-updates on a record in a
      // transaction.
      // We know that at least one of those has received the COMMIT signal, thus
      // we declare us only prepared since we then receive the expected COMMIT
      // signal.
      /* -------------------------------------------------------------------------
       */
      ndbrequire(regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC);
      sendLqhTransconf(signal, LqhTransConf::Prepared, tcConnectptr);
      return;
    case TcConnectionrec::WAIT_TUP:
      jam();
      /* -------------------------------------------------------------------------
       */
      // TUP is currently active. We have to wait for the TUPKEYREF or
      // TUPKEYCONF to arrive since we might otherwise jeopardise the local
      // checkpoint consistency in overload situations.
      /* -------------------------------------------------------------------------
       */
      regTcPtr->transactionState = TcConnectionrec::WAIT_TUP_TO_ABORT;
      DEB_COPY(
          ("(%u)transactionState(%u) set to WAIT_TUP_TO_ABORT,"
           " abortState: %u",
           instance(), tcConnectptr.i, regTcPtr->abortState));
      return;
    case TcConnectionrec::WAIT_ACC:
      jam();
      {
        ndbassert(!m_is_in_query_thread);
        fragptr.i = regTcPtr->fragmentptr;
        c_fragment_pool.getPtr(fragptr);
        acquire_frag_abort_access(fragptr.p, regTcPtr);
        abortContinueAfterBlockedLab(signal, tcConnectptr);
        release_frag_access(fragptr.p);
      }
      return;
    case TcConnectionrec::LOG_QUEUED:
      jam();
      if (!remove_from_prepare_log_queue(signal, tcConnectptr)) {
        jam();
        /**
         * We have removed it from the queue and sent a signal from the owner
         * of the log part to the LDM instance handling the fragment of the
         * operation to write into the REDO log.
         *
         * This signal will handle the abort of the operation since we have
         * set abortState to != IDLE before entering this method.
         *
         * We should only come here in the case where the LDM instance isn't
         * necessarily colocated with the REDO log parts. This only happens
         * if there are more LDM instances than there are REDO log parts.
         */
        ndbrequire(globalData.ndbMtLqhWorkers > globalData.ndbLogParts);
        return;
      }
      break;
    case TcConnectionrec::WAIT_TUP_TO_ABORT:
    case TcConnectionrec::LOG_ABORT_QUEUED:
    case TcConnectionrec::WAIT_ACC_ABORT:
      jam();
      /* -------------------------------------------------------------------------
       */
      /*ABORT IS ALREADY ONGOING DUE TO SOME ERROR. WE HAVE ALREADY SET THE
       * STATE  */
      /*OF THE ABORT SO THAT WE KNOW THAT TC EXPECTS A REPORT. WE CAN THUS
       * SIMPLY  */
      /*EXIT. */
      /* -------------------------------------------------------------------------
       */
      return;
    case TcConnectionrec::WAIT_TUP_COMMIT:
    case TcConnectionrec::LOG_COMMIT_QUEUED:
      jam();
      /* -------------------------------------------------------------------------
       */
      /*THIS IS ONLY AN ALLOWED STATE IF A DIRTY WRITE OR SIMPLE READ IS
       * PERFORMED.*/
      /*IF WE ARE MERELY CHECKING THE TRANSACTION STATE IT IS ALSO AN ALLOWED
       * STATE*/
      /* -------------------------------------------------------------------------
       */
      if (regTcPtr->dirtyOp == ZTRUE) {
        jam();
        /* -------------------------------------------------------------------------
         */
        /*COMPLETE THE DIRTY WRITE AND THEN REPORT COMPLETED BACK TO TC. SINCE
         * IT IS */
        /*A DIRTY WRITE IT IS ALLOWED TO COMMIT EVEN IF THE TRANSACTION ABORTS.
         */
        /* -------------------------------------------------------------------------
         */
        return;
      }  // if
      if (regTcPtr->opSimple) {
        jam();
        /* -------------------------------------------------------------------------
         */
        /*A SIMPLE READ IS CURRENTLY RELEASING THE LOCKS OR WAITING FOR ACCESS
         * TO    */
        /*ACC TO CLEAR THE LOCKS. COMPLETE THIS PROCESS AND THEN RETURN AS
         * NORMAL.   */
        /*NO DATA HAS CHANGED DUE TO THIS SIMPLE READ ANYWAY. */
        /* -------------------------------------------------------------------------
         */
        return;
      }  // if
      ndbrequire(regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC);
      jam();
      /* -------------------------------------------------------------------------
       */
      /*WE ARE ONLY CHECKING THE STATUS OF THE TRANSACTION. IT IS COMMITTING. */
      /*COMPLETE THE COMMIT LOCALLY AND THEN SEND REPORT OF COMMITTED TO THE NEW
       * TC*/
      /* -------------------------------------------------------------------------
       */
      sendLqhTransconf(signal, LqhTransConf::Committed, tcConnectptr);
      return;
    case TcConnectionrec::COMMITTED:
      jam();
      ndbrequire(regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC);
      /* -------------------------------------------------------------------------
       */
      /*WE ARE CHECKING TRANSACTION STATUS. REPORT COMMITTED AND CONTINUE WITH
       * THE */
      /*NEXT OPERATION. */
      /* -------------------------------------------------------------------------
       */
      sendLqhTransconf(signal, LqhTransConf::Committed, tcConnectptr);
      return;
    default:
      ndbabort();
      /* -------------------------------------------------------------------------
       */
      /*THE STATE WAS NOT AN ALLOWED STATE ON A NORMAL OPERATION. SCANS AND COPY
       */
      /*FRAGMENT OPERATIONS SHOULD HAVE EXECUTED IN ANOTHER PATH. */
      /* -------------------------------------------------------------------------
       */
  }  // switch
  abortCommonLab(signal, tcConnectptr);
}  // Dblqh::abortStateHandlerLab()

void Dblqh::abortErrorLab(Signal *signal, TcConnectionrecPtr tcConnectptr) {
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (regTcPtr->abortState == TcConnectionrec::ABORT_IDLE) {
    jam();
    regTcPtr->abortState = TcConnectionrec::ABORT_FROM_LQH;
    regTcPtr->errorCode = terrorCode;
  }  // if
  abortCommonLab(signal, tcConnectptr);
  return;
}  // Dblqh::abortErrorLab()

void Dblqh::abortCommonLab(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  const Uint32 activeCreat = regTcPtr->activeCreat;

  remove_commit_marker(regTcPtr);

  if (unlikely(activeCreat == Fragrecord::AC_NR_COPY)) {
    jam();
    if (regTcPtr->m_nr_delete.m_cnt) {
      jam();
      /**
       * Let operation wait for pending NR operations
       */
      DEB_COPY(
          ("(%u)Blocked in abortCommonLab for %u", instance(), tcConnectptr.i));
#ifdef VM_TRACE
      /**
       * Only disk table can have pending ops...
       */
      TablerecPtr tablePtr;
      tablePtr.i = regTcPtr->tableref;
      ptrCheckGuard(tablePtr, ctabrecFileSize, tablerec);
      ndbrequire(tablePtr.p->m_disk_table);
#endif
      return;
    }
  }

  fragptr.i = regTcPtr->fragmentptr;
  if (fragptr.i != RNIL) {
    jam();
    c_fragment_pool.getPtr(fragptr);
    switch (fragptr.p->fragStatus) {
      case Fragrecord::FSACTIVE:
      case Fragrecord::CRASH_RECOVERING:
      case Fragrecord::ACTIVE_CREATION:
        acquire_frag_abort_access(fragptr.p, regTcPtr);
        abortContinueAfterBlockedLab(signal, tcConnectptr);
        release_frag_access(fragptr.p);
        return;
      case Fragrecord::FREE:
        ndbabort();
      case Fragrecord::DEFINED:
        ndbabort();
      case Fragrecord::REMOVING:
        ndbabort();
      default:
        ndbabort();
    }  // switch
  } else {
    jam();
    continueAbortLab(signal, tcConnectptr);
  }  // if
}  // Dblqh::abortCommonLab()

void Dblqh::abortContinueAfterBlockedLab(Signal *signal,
                                         TcConnectionrecPtr regTcPtr) {
  /* ------------------------------------------------------------------------
   *       INPUT:          TC_CONNECTPTR           ACTIVE OPERATION RECORD
   * ------------------------------------------------------------------------
   * ------------------------------------------------------------------------
   *       CAN COME HERE AS RESTART AFTER BEING BLOCKED BY A LOCAL CHECKPOINT.
   * ------------------------------------------------------------------------
   *       ALSO AS PART OF A NORMAL ABORT WITHOUT BLOCKING.
   *       WE MUST ABORT TUP BEFORE ACC TO ENSURE THAT NO ONE RACES IN
   *       AND SEES A STATE IN TUP.
   * ----------------------------------------------------------------------- */
  TRACE_OP(regTcPtr.p, "ACC ABORT");
  Uint32 canBlock = 2;  // 2, block if needed
  switch (regTcPtr.p->transactionState) {
    case TcConnectionrec::WAIT_TUP:
      jam();
      /**
       * This is when getting from execTUPKEYREF
       *   in which case we *do* have ACC lock
       *   and should not (need to) block
       */
      canBlock = 0;
      break;
    default:
      break;
  }

  regTcPtr.p->transactionState = TcConnectionrec::WAIT_ACC_ABORT;
  c_acc->execACC_ABORTREQ(signal, regTcPtr.p->accConnectrec,
                          regTcPtr.p->accConnectPtrP, canBlock);

  if (signal->theData[1] == RNIL) {
    jam();
    ndbassert(!m_is_query_block);
    /* ------------------------------------------------------------------------
     * We need to insert a real-time break by sending ACC_ABORTCONF through the
     * job buffer to ensure that we catch any ACCKEYCONF or TUPKEYCONF or
     * TUPKEYREF that are in the job buffer but not yet processed. Doing
     * everything without that would race and create a state error when they
     * are executed.
     * --------------------------------------------------------------------- */
    return;
  }
  signal->theData[0] = regTcPtr.p->tupConnectrec;
  c_tup->do_tup_abortreq(signal, 0);
  jamEntryDebug();
  continueAbortLab(signal, regTcPtr);
}  // Dblqh::abortContinueAfterBlockedLab()

/* ******************>> */
/*  ACC_ABORTCONF     > */
/* ******************>> */
void Dblqh::execACC_ABORTCONF(Signal *signal) {
  jamEntry();
  /**
   * This is sent from DBACC as a way to ensure a real-time break
   * happens before we receive this and thus we are sure that no
   * signals are in job buffer waiting for this operation to be
   * continued.
   */
  setup_key_pointers(signal->theData[0], false);
  TcConnectionrec *const regTcPtr = m_tc_connect_ptr.p;
  ndbrequire(regTcPtr->transactionState == TcConnectionrec::WAIT_ACC_ABORT);
  ndbassert(!m_is_query_block);

  TRACE_OP(regTcPtr, "ACC_ABORTCONF");
  Uint32 connect_ptr = regTcPtr->tupConnectrec;
  acquire_frag_abort_access(fragptr.p, regTcPtr);
  signal->theData[0] = connect_ptr;
  c_tup->do_tup_abortreq(signal, 0);
  jamEntryDebug();
  release_frag_access(fragptr.p);
  continueAbortLab(signal, m_tc_connect_ptr);
}  // Dblqh::execACC_ABORTCONF()

void Dblqh::continueAbortLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  /* ------------------------------------------------------------------------
   *  AN ERROR OCCURRED IN THE ACTIVE CREATION AFTER THE ABORT PHASE.
   *  WE NEED TO CONTINUE WITH A NORMAL ABORT.
   * ------------------------------------------------------------------------
   *       ALSO USED FOR NORMAL CLEAN UP AFTER A NORMAL ABORT.
   * ------------------------------------------------------------------------
   *       ALSO USED WHEN NO FRAGMENT WAS SET UP ON OPERATION.
   * ---------------------------------------------------------------------- */
  if (regTcPtr->logWriteState == TcConnectionrec::WRITTEN) {
    jam();
    /* ----------------------------------------------------------------------
     * I NEED TO INSERT A ABORT LOG RECORD SINCE WE ARE WRITING LOG IN THIS
     * TRANSACTION.
     * -------------------------------------------------------------------- */
    LogPageRecordPtr logPagePtr;
    LogFileRecordPtr logFilePtr;
    LogPartRecordPtr logPartPtr;
    ndbassert(!m_is_query_block);
    lock_log_part(regTcPtr->m_log_part_ptr_p);
    initLogPointers(signal, tcConnectptr, logPagePtr, logFilePtr, logPartPtr);
    Uint32 noOfFreeLogPages = count_free_log_pages(logPartPtr.p);
    if (noOfFreeLogPages == 0 ||
        !logPartPtr.p->m_log_complete_queue.isEmpty()) {
      jam();
      /* --------------------------------------------------------------------
       * A PREPARE OPERATION IS CURRENTLY WRITING IN THE LOG.
       * WE MUST WAIT ON OUR TURN TO WRITE THE LOG.
       * IT IS NECESSARY TO WRITE ONE LOG RECORD COMPLETELY
       * AT A TIME OTHERWISE WE WILL SCRAMBLE THE LOG.
       * ------------------------------------------------------------------ */
      linkWaitLog(signal, logPartPtr.p, logPartPtr.p->m_log_complete_queue,
                  tcConnectptr.p);
      regTcPtr->transactionState = TcConnectionrec::LOG_ABORT_QUEUED;
      unlock_log_part(regTcPtr->m_log_part_ptr_p);
      return;
    }  // if
    writeAbortLog(signal, regTcPtr, logPagePtr, logFilePtr, logPartPtr.p);
    removeLogTcrec(regTcPtr, logPartPtr.p);
    unlock_log_part(logPartPtr.p);
  } else if (regTcPtr->logWriteState == TcConnectionrec::NOT_STARTED) {
    jam();
  } else if (regTcPtr->logWriteState == TcConnectionrec::NOT_WRITTEN) {
    jam();
    /* ------------------------------------------------------------------
     * IT IS A READ OPERATION OR OTHER OPERATION THAT DO NOT USE THE LOG.
     * ------------------------------------------------------------------ */
    /* ------------------------------------------------------------------
     * THE LOG HAS NOT BEEN WRITTEN SINCE THE LOG FLAG WAS false.
     * THIS CAN OCCUR WHEN WE ARE STARTING A NEW FRAGMENT.
     * ------------------------------------------------------------------ */
    regTcPtr->logWriteState = TcConnectionrec::NOT_STARTED;
  } else {
    ndbrequire(regTcPtr->logWriteState == TcConnectionrec::NOT_WRITTEN_WAIT);
    jam();
    /* ----------------------------------------------------------------
     * THE STATE WAS SET TO NOT_WRITTEN BY THE OPERATION BUT LATER
     * A SCAN OF ALL OPERATION RECORD CHANGED IT INTO NOT_WRITTEN_WAIT.
     * THIS INDICATES THAT WE ARE WAITING FOR THIS OPERATION TO COMMIT
     * OR ABORT SO THAT WE CAN FIND THE
     * STARTING GLOBAL CHECKPOINT OF THIS NEW FRAGMENT.
     * ---------------------------------------------------------------- */
    checkScanTcCompleted(signal, tcConnectptr);
  }  // if
  continueAfterLogAbortWriteLab(signal, tcConnectptr);
  return;
}  // Dblqh::continueAbortLab()

void Dblqh::continueAfterLogAbortWriteLab(
    Signal *signal, const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  bool normalProtocol =
      (regTcPtr->m_flags & TcConnectionrec::OP_NORMAL_PROTOCOL);

  remove_commit_marker(regTcPtr);

  if (regTcPtr->operation == ZREAD && regTcPtr->dirtyOp && !normalProtocol) {
    jam();
    TcKeyRef *const tcKeyRef = (TcKeyRef *)signal->getDataPtrSend();

    tcKeyRef->connectPtr = regTcPtr->applOprec;
    tcKeyRef->transId[0] = regTcPtr->transid[0];
    tcKeyRef->transId[1] = regTcPtr->transid[1];
    tcKeyRef->errorCode = regTcPtr->errorCode;
    sendTCKEYREF(signal, regTcPtr->applRef, regTcPtr->tcBlockref, 0);
    cleanUp(signal, tcConnectptr);
    return;
  }  // if
  if ((regTcPtr->abortState == TcConnectionrec::ABORT_FROM_LQH) ||
      (regTcPtr->abortState == TcConnectionrec::ABORT_FROM_LQH_REPLICA)) {
    LqhKeyRef *const lqhKeyRef = (LqhKeyRef *)signal->getDataPtrSend();

    jam();
    lqhKeyRef->userRef = regTcPtr->clientConnectrec;
    lqhKeyRef->connectPtr = regTcPtr->tcOprec;
    lqhKeyRef->errorCode = regTcPtr->errorCode;
    lqhKeyRef->transId1 = regTcPtr->transid[0];
    lqhKeyRef->transId2 = regTcPtr->transid[1];
    lqhKeyRef->flags = 0;
    if (regTcPtr->abortState == TcConnectionrec::ABORT_FROM_LQH_REPLICA) {
      jam();
      LqhKeyRef::setReplicaErrorFlag(lqhKeyRef->flags, 1);
    }
    Uint32 block = refToMain(regTcPtr->clientBlockref);
    if (block != getRESTORE()) {
      sendSignal(regTcPtr->clientBlockref, GSN_LQHKEYREF, signal,
                 LqhKeyRef::SignalLength, JBB);
    } else {
      ndbrequire(refToNode(regTcPtr->clientBlockref) == cownNodeid &&
                 refToInstance(regTcPtr->clientBlockref) == instance());
      EXECUTE_DIRECT(getRESTORE(), GSN_LQHKEYREF, signal,
                     LqhKeyRef::SignalLength);
    }
  } else if (regTcPtr->abortState == TcConnectionrec::ABORT_FROM_TC) {
    jam();
    ndbassert(!m_is_in_query_thread);
    sendAborted(signal, tcConnectptr);
  } else if (regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC) {
    jam();
    ndbassert(!m_is_query_block);
    sendLqhTransconf(signal, LqhTransConf::Aborted, tcConnectptr);
  } else {
    ndbassert(!m_is_query_block);
    ndbrequire(regTcPtr->abortState == TcConnectionrec::REQ_FROM_TC);
    jam();
    signal->theData[0] = regTcPtr->reqRef;
    signal->theData[1] = tcConnectptr.i;
    signal->theData[2] = cownNodeid;
    signal->theData[3] = regTcPtr->transid[0];
    signal->theData[4] = regTcPtr->transid[1];
    sendSignal(regTcPtr->reqBlockref, GSN_ABORTCONF, signal, 5, JBB);
  }  // if
  cleanUp(signal, tcConnectptr);
}  // Dblqh::continueAfterLogAbortWriteLab()

void Dblqh::sendTCKEYREF(Signal *signal, Uint32 ref, Uint32 routeRef,
                         Uint32 cnt) {
  const Uint32 nodeId = refToNode(ref);
  const bool connectedToNode = getNodeInfo(nodeId).m_connected;
  /**
   * ROUTE_ORD signals should not be sent via SPJ as it does not handle these
   * and (unlike TC) may not be connected to the API anyway.
   */
  ndbrequire(routeRef == 0 || nodeId == getOwnNodeId() ||
             refToMain(routeRef) == DBTC);

  if (likely(connectedToNode && !ERROR_INSERTED_CLEAR(5079))) {
    jam();
    sendSignal(ref, GSN_TCKEYREF, signal, TcKeyRef::SignalLength, JBB);
  } else {
    if (routeRef &&
        getNodeInfo(refToNode(routeRef)).m_version >= MAKE_VERSION(5, 1, 14)) {
      jam();
      memmove(signal->theData + 25, signal->theData,
              4 * TcKeyRef::SignalLength);
      RouteOrd *ord = (RouteOrd *)signal->getDataPtrSend();
      ord->dstRef = ref;
      ord->srcRef = reference();
      ord->gsn = GSN_TCKEYREF;
      ord->cnt = 0;
      LinearSectionPtr ptr[3];
      ptr[0].p = signal->theData + 25;
      ptr[0].sz = TcKeyRef::SignalLength;
      sendSignal(routeRef, GSN_ROUTE_ORD, signal, RouteOrd::SignalLength, JBB,
                 ptr, 1);
    } else {
      jam();
      memmove(signal->theData + 3, signal->theData, 4 * TcKeyRef::SignalLength);
      signal->theData[0] = ZRETRY_TCKEYREF;
      signal->theData[1] = cnt + 1;
      signal->theData[2] = ref;
      sendSignalWithDelay(reference(), GSN_CONTINUEB, signal, 100,
                          TcKeyRef::SignalLength + 3);
    }
  }
}

/* ##########################################################################
 * #######                       MODULE TO HANDLE TC FAILURE          #######
 *
 * ########################################################################## */

/* ************************************************************************>>
 *  NODE_FAILREP: Node failure report. Sender Ndbcntr. Set status of failed
 *  node to down and reply with NF_COMPLETEREP to DIH which will report that
 *  LQH has completed failure handling.
 * ************************************************************************>> */
void Dblqh::execNODE_FAILREP(Signal *signal) {
  UintR TfoundNodes = 0;
  UintR TnoOfNodes;
  UintR Tdata[MAX_NDB_NODES];
  Uint32 i;

  NodeFailRep *const nodeFail = (NodeFailRep *)&signal->theData[0];

  if (signal->getLength() == NodeFailRep::SignalLength) {
    jam();
    ndbrequire(signal->getNoOfSections() == 1);
    ndbrequire(getNodeInfo(refToNode(signal->getSendersBlockRef())).m_version);
    SegmentedSectionPtr ptr;
    SectionHandle handle(this, signal);
    ndbrequire(handle.getSection(ptr, 0));
    memset(nodeFail->theNodes, 0, sizeof(nodeFail->theNodes));
    copy(nodeFail->theNodes, ptr);
    releaseSections(handle);
  } else {
    memset(nodeFail->theNodes + NdbNodeBitmask48::Size, 0, _NDB_NBM_DIFF_BYTES);
  }
  TnoOfNodes = nodeFail->noOfNodes;
  UintR index = 0;
  for (i = 1; i < MAX_NDB_NODES; i++) {
    jam();
    if (NdbNodeBitmask::get(nodeFail->theNodes, i)) {
      jam();
      Tdata[index] = i;
      index++;
    }  // if
  }    // for

#ifdef ERROR_INSERT
  c_master_node_id = nodeFail->masterNodeId;
#endif

  ndbrequire(index == TnoOfNodes);
  ndbrequire(cnoOfNodes - 1 < MAX_NDB_NODES);
  for (i = 0; i < TnoOfNodes; i++) {
    const Uint32 nodeId = Tdata[i];

    {
      HostRecordPtr Thostptr;
      Thostptr.i = nodeId;
      ptrCheckGuard(Thostptr, chostFileSize, hostRecord);
      Thostptr.p->nodestatus = ZNODE_DOWN;
      DEB_NODE_STATUS(
          ("(%u,%u) Node %u DOWN", instance(), m_is_query_block, Thostptr.i));
    }

    for (Uint32 j = 0; j < cnoOfNodes; j++) {
      jam();
      if (cnodeData[j] == nodeId) {
        jam();
        cnodeStatus[j] = ZNODE_DOWN;

        TfoundNodes++;
      }  // if
    }    // for

    /* Perform block-level ndbd failure handling */
    Callback cb = {safe_cast(&Dblqh::ndbdFailBlockCleanupCallback), Tdata[i]};
    simBlockNodeFailure(signal, Tdata[i], cb);
  }  // for
  ndbrequire(TnoOfNodes == TfoundNodes);
}  // Dblqh::execNODE_FAILREP()

void Dblqh::ndbdFailBlockCleanupCallback(Signal *signal, Uint32 failedNodeId,
                                         Uint32 ignoredRc) {
  jamEntry();

  NFCompleteRep *const nfCompRep = (NFCompleteRep *)&signal->theData[0];
  nfCompRep->blockNo = m_is_query_block ? DBQLQH : DBLQH;
  nfCompRep->nodeId = cownNodeid;
  nfCompRep->failedNodeId = failedNodeId;
  BlockReference dihRef =
      !isNdbMtLqh() ? DBDIH_REF : m_is_query_block ? DBQLQH_REF : DBLQH_REF;
  sendSignal(dihRef, GSN_NF_COMPLETEREP, signal, NFCompleteRep::SignalLength,
             JBB);
}

/* ************************************************************************>>
 *  LQH_TRANSREQ: Report status of all transactions where TC was coordinated
 *  by a crashed TC
 * ************************************************************************>> */
/* ************************************************************************>>
 *  THIS SIGNAL IS RECEIVED AFTER A NODE CRASH.
 *  THE NODE HAD A TC AND COORDINATED A NUMBER OF TRANSACTIONS.
 *  NOW THE MASTER NODE IS PICKING UP THOSE TRANSACTIONS
 *  TO COMPLETE THEM. EITHER ABORT THEM OR COMMIT THEM.
 * ************************************************************************>> */
void Dblqh::execLQH_TRANSREQ(Signal *signal) {
  jamEntry();

  if (!checkNodeFailSequence(signal)) {
    jam();
    return;
  }
  LqhTransReq *const lqhTransReq = (LqhTransReq *)&signal->theData[0];
  Uint32 newTcPtr = lqhTransReq->senderData;
  BlockReference newTcBlockref = lqhTransReq->senderRef;
  Uint32 oldNodeId = lqhTransReq->failedNodeId;
  Uint32 instanceId = lqhTransReq->instanceId;
  if (signal->getLength() < LqhTransReq::SignalLength) {
    /**
     * TC that performs take over doesn't support taking over one
     * TC instance at a time => we read an unitialised variable,
     * set it to RNIL to indicate we try take over all instances.
     * This code is really only needed in ndbd since ndbmtd handles
     * it also in LQH proxy.
     */
    instanceId = RNIL;
  }
  TcNodeFailRecordPtr tcNodeFailPtr;
  tcNodeFailPtr.i = oldNodeId;
  ptrCheckGuard(tcNodeFailPtr, ctcNodeFailrecFileSize, tcNodeFailRecord);
  if ((tcNodeFailPtr.p->tcFailStatus == TcNodeFailRecord::TC_STATE_TRUE) ||
      (tcNodeFailPtr.p->tcFailStatus == TcNodeFailRecord::TC_STATE_BREAK)) {
    jam();
    tcNodeFailPtr.p->lastNewTcBlockref = newTcBlockref;
    /* ------------------------------------------------------------------------
     * WE HAVE RECEIVED A SIGNAL SPECIFYING THAT WE NEED TO HANDLE THE FAILURE
     * OF A TC.  NOW WE RECEIVE ANOTHER SIGNAL WITH THE SAME ORDER. THIS CAN
     * OCCUR IF THE NEW TC FAILS. WE MUST BE CAREFUL IN THIS CASE SO THAT WE DO
     * NOT START PARALLEL ACTIVITIES TRYING TO DO THE SAME THING. WE SAVE THE
     * NEW BLOCK REFERENCE TO THE LAST NEW TC IN A VARIABLE AND ASSIGN TO IT TO
     * NEW_TC_BLOCKREF WHEN THE OLD PROCESS RETURNS TO LQH_TRANS_NEXT. IT IS
     * CERTAIN TO COME THERE SINCE THIS IS THE ONLY PATH TO TAKE CARE OF THE
     * NEXT TC CONNECT RECORD. WE SET THE STATUS TO BREAK TO INDICATE TO THE OLD
     * PROCESS WHAT IS HAPPENING.
     * ------------------------------------------------------------------------
     */
    tcNodeFailPtr.p->lastNewTcRef = newTcPtr;
    tcNodeFailPtr.p->lastTakeOverInstanceId = instanceId;
    tcNodeFailPtr.p->tcFailStatus = TcNodeFailRecord::TC_STATE_BREAK;
    return;
  }  // if
  tcNodeFailPtr.p->oldNodeId = oldNodeId;
  tcNodeFailPtr.p->newTcBlockref = newTcBlockref;
  tcNodeFailPtr.p->newTcRef = newTcPtr;
  tcNodeFailPtr.p->takeOverInstanceId = instanceId;
  tcNodeFailPtr.p->maxInstanceId = 0;
  tcNodeFailPtr.p->tcRecNow = 0;
  tcNodeFailPtr.p->tcFailStatus = TcNodeFailRecord::TC_STATE_TRUE;
  signal->theData[0] = ZLQH_TRANS_NEXT;
  signal->theData[1] = tcNodeFailPtr.i;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
  return;
}  // Dblqh::execLQH_TRANSREQ()

bool Dblqh::check_tc_and_update_max_instance(BlockReference ref,
                                             TcNodeFailRecord *tcNodeFailPtr) {
  if (refToNode(ref) == tcNodeFailPtr->oldNodeId) {
    jam();
    Uint32 instanceId = refToInstance(ref);
    if (instanceId > tcNodeFailPtr->maxInstanceId) {
      /**
       * Inform take over TC instance about the maximum instance id
       * such that the TC instance knows when to stop the take over
       * process.
       */
      tcNodeFailPtr->maxInstanceId = instanceId;
    }
    if ((tcNodeFailPtr->takeOverInstanceId == RNIL) ||
        (instanceId == tcNodeFailPtr->takeOverInstanceId)) {
      jam();
      return true;
    }
  }
  jam();
  return false;
}

void Dblqh::lqhTransNextLab(Signal *signal, TcNodeFailRecordPtr tcNodeFailPtr) {
  if (tcNodeFailPtr.p->tcFailStatus == TcNodeFailRecord::TC_STATE_BREAK) {
    jam();
    /* ----------------------------------------------------------------------
     *  AN INTERRUPTION TO THIS NODE FAIL HANDLING WAS RECEIVED AND A NEW
     *  TC HAVE BEEN ASSIGNED TO TAKE OVER THE FAILED TC. PROBABLY THE OLD
     *  NEW TC HAVE FAILED.
     * ---------------------------------------------------------------------- */
    tcNodeFailPtr.p->newTcBlockref = tcNodeFailPtr.p->lastNewTcBlockref;
    tcNodeFailPtr.p->newTcRef = tcNodeFailPtr.p->lastNewTcRef;
    tcNodeFailPtr.p->takeOverInstanceId =
        tcNodeFailPtr.p->lastTakeOverInstanceId;
    tcNodeFailPtr.p->maxInstanceId = 0;
    tcNodeFailPtr.p->tcRecNow = 0;
    tcNodeFailPtr.p->tcFailStatus = TcNodeFailRecord::TC_STATE_TRUE;
  }  // if
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = tcNodeFailPtr.p->tcRecNow;
  for (Uint32 i = 0; i < 100; i++) {
    bool found = getNextTcConRec(tcNodeFailPtr.p->tcRecNow, tcConnectptr, 10);
    if (tcNodeFailPtr.p->tcRecNow != RNIL && !found) {
      /**
       * We scanned without finding any records for a long
       * time, thus we will treat this as looping 10 times
       * in this loop.
       */
      jam();
      i += 10;
      continue;
    } else if (tcNodeFailPtr.p->tcRecNow == RNIL) {
      jam();
      /**
       * Finished with scanning operation record
       *
       * now scan markers
       */
#ifdef ERROR_INSERT
      if (ERROR_INSERTED(5061)) {
        CLEAR_ERROR_INSERT_VALUE;
        for (Uint32 i = 0; i < cnoOfNodes; i++) {
          Uint32 node = cnodeData[i];
          if (node != getOwnNodeId() && cnodeStatus[i] == ZNODE_UP) {
            g_eventLogger->info("clearing ERROR_INSERT in LQH:%u", node);
            signal->theData[0] = 0;
            sendSignal(numberToRef(getDBLQH(), node), GSN_NDB_TAMPER, signal, 1,
                       JBB);
          }
        }

        signal->theData[0] = ZSCAN_MARKERS;
        signal->theData[1] = tcNodeFailPtr.i;
        signal->theData[2] = 0;
        sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 5000, 3);
        return;
      }

      if (ERROR_INSERTED(5050)) {
        g_eventLogger->info(
            "send ZSCAN_MARKERS with 5s delay and killing master: %u",
            c_master_node_id);
        CLEAR_ERROR_INSERT_VALUE;
        signal->theData[0] = ZSCAN_MARKERS;
        signal->theData[1] = tcNodeFailPtr.i;
        signal->theData[2] = 0;
        sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 5000, 3);

        signal->theData[0] = 9999;
        sendSignal(numberToRef(CMVMI, c_error_insert_extra), GSN_NDB_TAMPER,
                   signal, 1, JBB);
        return;
      }
#endif
      scanMarkers(signal, tcNodeFailPtr.i, 0);
      return;
    }  // if
    if (tcConnectptr.p->transactionState != TcConnectionrec::IDLE) {
      if (tcConnectptr.p->transactionState !=
          TcConnectionrec::TC_NOT_CONNECTED) {
        if (tcConnectptr.p->tcScanRec == RNIL) {
          if (check_tc_and_update_max_instance(tcConnectptr.p->tcBlockref,
                                               tcNodeFailPtr.p)) {
            /**
             * We send the take over message only for operations that belong
             * to the failed node and also are part of the TC instance that
             * we are currently taking over, instanceId == RNIL means that the
             * signal came from an old version that didn't support multi-TC
             * instance take over. In this case we try to take over all
             * instances in one go.
             */
            switch (tcConnectptr.p->operation) {
              case ZUNLOCK:
                jam(); /* Skip over */
                break;
              case ZREAD:
                jam();
                if (tcConnectptr.p->opSimple == ZTRUE) {
                  jam();
                  break; /* Skip over */
                }
                [[fallthrough]];
              default:
                jam();
                tcConnectptr.p->tcNodeFailrec = tcNodeFailPtr.i;
                tcConnectptr.p->abortState = TcConnectionrec::NEW_FROM_TC;
                abortStateHandlerLab(signal, tcConnectptr);
                return;
            }  // switch
          }
        } else {
          /**
           * Scans don't require any keeping of state in TC that takes
           * over, thus we need not handle scans one instance at a time.
           * We can handle all scans immediately. The same goes for copy
           * operations since we can have a very limited number of copy
           * operations ongoing in parallel.
           */
          scanptr.i = tcConnectptr.p->tcScanRec;
          ndbrequire(c_scanRecordPool.getValidPtr(scanptr));
          switch (scanptr.p->scanType) {
            case ScanRecord::COPY: {
              jam();
              if (scanptr.p->scanNodeId == tcNodeFailPtr.p->oldNodeId) {
                jam();
                /* ------------------------------------------------------------
                 * THE RECEIVER OF THE COPY HAVE FAILED.
                 * WE HAVE TO CLOSE THE COPY PROCESS.
                 * -----------------------------------------------------------
                 */
                if (0) g_eventLogger->info("close copy");
                tcConnectptr.p->tcNodeFailrec = tcNodeFailPtr.i;
                tcConnectptr.p->abortState = TcConnectionrec::NEW_FROM_TC;
                ndbassert(!m_is_query_block);
                setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
                closeCopyRequestLab(signal, tcConnectptr);
                release_frag_access(prim_tab_fragptr.p);
                return;
              }
              break;
            }
            case ScanRecord::SCAN: {
              jam();
              if (refToNode(tcConnectptr.p->tcBlockref) ==
                  tcNodeFailPtr.p->oldNodeId) {
                jam();
                tcConnectptr.p->tcNodeFailrec = tcNodeFailPtr.i;
                tcConnectptr.p->abortState = TcConnectionrec::NEW_FROM_TC;
                ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
                ndbassert(!m_is_query_block);
                setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
                closeScanRequestLab(signal, tcConnectptr);
                release_frag_access(prim_tab_fragptr.p);
                return;
              }  // if
              break;
            }
            default:
              g_eventLogger->info("scanptr.p->scanType: %u",
                                  scanptr.p->scanType);
              g_eventLogger->info("tcConnectptr.p->transactionState: %u",
                                  tcConnectptr.p->transactionState);
              ndbabort();
          }
        }
      } else {
#if defined VM_TRACE || defined ERROR_INSERT
        jam();
        ndbrequire(tcConnectptr.p->tcScanRec == RNIL);
#endif
      }
    } else {
#if defined VM_TRACE || defined ERROR_INSERT
      jam();
      ndbrequire(tcConnectptr.p->tcScanRec == RNIL);
#endif
    }
  }  // for
  signal->theData[0] = ZLQH_TRANS_NEXT;
  signal->theData[1] = tcNodeFailPtr.i;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
}  // Dblqh::lqhTransNextLab()

void Dblqh::scanMarkers(Signal *signal, Uint32 tcNodeFail,
                        Uint32 commitAckMarkerPtrI) {
  jam();

  TcNodeFailRecordPtr tcNodeFailPtr;
  tcNodeFailPtr.i = tcNodeFail;
  ptrCheckGuard(tcNodeFailPtr, ctcNodeFailrecFileSize, tcNodeFailRecord);

  if (tcNodeFailPtr.p->tcFailStatus == TcNodeFailRecord::TC_STATE_BREAK) {
    jam();

    /* ----------------------------------------------------------------------
     *  AN INTERRUPTION TO THIS NODE FAIL HANDLING WAS RECEIVED AND A NEW
     *  TC HAVE BEEN ASSIGNED TO TAKE OVER THE FAILED TC. PROBABLY THE OLD
     *  NEW TC HAVE FAILED.
     * ---------------------------------------------------------------------- */
    lqhTransNextLab(signal, tcNodeFailPtr);
    return;
  }

  const Uint32 RT_BREAK = 128;
  for (Uint32 i = 0; i < RT_BREAK; i++) {
    jam();
    CommitAckMarkerPtr commitAckMarkerPtr;
    bool found =
        getNextCommitAckMarker(commitAckMarkerPtrI, commitAckMarkerPtr, 10);
    if (found) {
      ndbrequire(commitAckMarkerPtr.p->in_hash);
      jam();
      if (check_tc_and_update_max_instance(commitAckMarkerPtr.p->tcRef,
                                           tcNodeFailPtr.p)) {
        jam();
        /**
         * Found marker belonging to crashed node and to instance currently
         * being handled.
         */
        LqhTransConf *const lqhTransConf = (LqhTransConf *)&signal->theData[0];
        lqhTransConf->tcRef = tcNodeFailPtr.p->newTcRef;
        lqhTransConf->lqhNodeId = cownNodeid;
        lqhTransConf->operationStatus = LqhTransConf::Marker;
        lqhTransConf->transId1 = commitAckMarkerPtr.p->transid1;
        lqhTransConf->transId2 = commitAckMarkerPtr.p->transid2;
        lqhTransConf->apiRef = commitAckMarkerPtr.p->apiRef;
        lqhTransConf->apiOpRec = commitAckMarkerPtr.p->apiOprec;
        sendSignal(tcNodeFailPtr.p->newTcBlockref, GSN_LQH_TRANSCONF, signal, 7,
                   JBB);

        signal->theData[0] = ZSCAN_MARKERS;
        signal->theData[1] = tcNodeFailPtr.i;
        signal->theData[2] = commitAckMarkerPtrI;
        sendSignal(cownref, GSN_CONTINUEB, signal, 3, JBB);
        return;
      }
    }
    if (commitAckMarkerPtrI == RNIL) {
      /**
       * Done with iteration
       */
      jam();

      tcNodeFailPtr.p->tcFailStatus = TcNodeFailRecord::TC_STATE_FALSE;
      LqhTransConf *const lqhTransConf = (LqhTransConf *)&signal->theData[0];
      lqhTransConf->tcRef = tcNodeFailPtr.p->newTcRef;
      lqhTransConf->lqhNodeId = cownNodeid;
      lqhTransConf->operationStatus = LqhTransConf::LastTransConf;
      lqhTransConf->maxInstanceId = tcNodeFailPtr.p->maxInstanceId;
      sendSignal(tcNodeFailPtr.p->newTcBlockref, GSN_LQH_TRANSCONF, signal,
                 LqhTransConf::SignalLength, JBB);
      return;
    }
    jam();
  }

  signal->theData[0] = ZSCAN_MARKERS;
  signal->theData[1] = tcNodeFailPtr.i;
  signal->theData[2] = commitAckMarkerPtrI;
  sendSignal(cownref, GSN_CONTINUEB, signal, 3, JBB);
}

/* #########################################################################
 * #######                       SCAN MODULE                         #######
 *
 * #########################################################################
 * -------------------------------------------------------------------------
 * THIS MODULE CONTAINS THE CODE THAT HANDLES A SCAN OF A PARTICULAR FRAGMENT
 * IT OPERATES UNDER THE CONTROL OF TC AND ORDERS ACC TO PERFORM A SCAN OF
 * ALL TUPLES IN THE FRAGMENT. TUP PERFORMS THE NECESSARY SEARCH CONDITIONS
 * TO ENSURE THAT ONLY VALID TUPLES ARE RETURNED TO THE APPLICATION.
 * ------------------------------------------------------------------------- */

/**
 *  ACC_SCANCONF obsolete by usage of Direct Execute
 *
 *  Callee of ACC_SCANREQ will get return value in
 *  signal[8] and call accScanConf{Scan|Copy}Lab()
 *  directly if OK.
 */

Uint32 Dblqh::rt_break_is_scan_prioritised(Uint32 scan_ptr_i) {
  ScanRecordPtr scanPtr;
  scanPtr.i = scan_ptr_i;
  ndbrequire(c_scanRecordPool.getUncheckedPtrRO(scanPtr));
  m_scan_direct_count = 1; /* Initialise before rt break */
  bool ret = is_prioritised_scan(scanPtr.p->scanApiBlockref);
  ndbrequire(Magic::check_ptr(scanPtr.p));
  return ret;
}

/* ************>> */
/*  ACC_SCANREF > */
/* ************>> */
void Dblqh::execACC_SCANREF(Signal *signal,
                            const TcConnectionrecPtr tcConnectptr) {
  const AccScanRef refCopy = *(const AccScanRef *)signal->getDataPtr();
  const AccScanRef *ref = &refCopy;
  ndbrequire(ref->errorCode != 0);

  tcConnectptr.p->errorCode = ref->errorCode;

  /*
   * MRR scan can hit this between 2 DBTUX scans.  Previous range has
   * terminated via last NEXT_SCANCONF, then index is set to Dropping,
   * and then next range is started and returns ACC_SCANREF.
   */
  if (scanptr.p->scanStoredProcId != RNIL) {
    jam();
    scanptr.p->scanCompletedStatus = ZTRUE;
    accScanCloseConfLab(signal, tcConnectptr);
    return;
  }
  tupScanCloseConfLab(signal, tcConnectptr);
}  // Dblqh::execACC_SCANREF()

/* ***************> */
/*  NEXT_SCANREF  > */
/* ***************> */
void Dblqh::execNEXT_SCANREF(Signal *signal) {
  jamEntry();
  ndbrequire(refToMain(signal->getSendersBlockRef()) == getDBTUX());
  exec_next_scan_ref(signal);
}

void Dblqh::exec_next_scan_ref(Signal *signal) {
  jamEntry();
  const NextScanRef refCopy = *(const NextScanRef *)signal->getDataPtr();
  const NextScanRef *ref = &refCopy;
  ndbrequire(ref->errorCode != 0);

  scanptr.i = ref->scanPtr;
  ndbrequire(c_scanRecordPool.getValidPtr(scanptr));
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = scanptr.p->scanTcrec;
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  tcConnectptr.p->errorCode = ref->errorCode;

  /*
   * MRR scan may have other ranges left.  But the scan has already
   * failed.  Terminate the scan now.
   */
  scanptr.p->scanCompletedStatus = ZTRUE;
  accScanCloseConfLab(signal, tcConnectptr);
}  // Dblqh::execNEXT_SCANREF()

/**
 *  ACC_SCANCONF obsolete by usage of Direct Execute
 *
 *  Callee of ACC_SCANREQ will get return value in
 *  signal[8] and call accScanConf{Scan|Copy}Lab()
 *  directly if OK.
 */
Uint32 Dblqh::get_scan_api_op_ptr(Uint32 scan_api_ptr_i) {
  ScanRecordPtr scanPtr;
  scanPtr.i = scan_api_ptr_i;
  ndbrequire(c_scanRecordPool.getUncheckedPtrRW(scanPtr));
  Uint32 apiOpPtr = scanPtr.p->scanApiOpPtr;
  ndbrequire(Magic::check_ptr(scanPtr.p));
  return apiOpPtr;
}

/**
 * Query threads can operate on a fragment in any of the LDM threads.
 * To enable a smooth transition of the code we will still use block
 * variables that point to the table and fragment records in the
 * new query blocks. This means that we need to setup those in each
 * real-time break since each new signal can be executing on behalf
 * of a new LDM thread.
 *
 * Some variables are equal in all LDM threads, this is true for
 * number of fragments and table record instances. Thus this will
 * be setup in the STTOR signals and not here.
 */
void Dblqh::setup_query_thread_for_key_access(Uint32 instanceNo) {
  Dblqh *lqh_block = (Dblqh *)globalData.getBlock(DBLQH, instanceNo);
  Dbtup *tup_block = (Dbtup *)globalData.getBlock(DBTUP, instanceNo);
  Dbacc *acc_block = (Dbacc *)globalData.getBlock(DBACC, instanceNo);
  c_acc->m_ldm_instance_used = acc_block;
  c_tup->m_ldm_instance_used = tup_block;
  this->m_ldm_instance_used = lqh_block;

  this->c_fragment_pool.setArrayPtr(lqh_block->c_fragment_pool.getArrayPtr());
  this->tablerec = lqh_block->tablerec;

  c_acc->fragmentrec = acc_block->fragmentrec;
  c_acc->directoryPoolPtr = acc_block->directoryPoolPtr;
  /**
   * The page8 pool is relying on the page32 pool and this use the
   * memroot of the process as start of ArrayPool and this is common
   * for all threads in the node. Thus no need to copy it from LDM
   * instance since the query threads point to the same base.
   */

  c_tup->fragrecord = tup_block->fragrecord;
  c_tup->tablerec = tup_block->tablerec;
  c_tup->tableDescriptor = tup_block->tableDescriptor;
  c_tup->cnoOfTabDescrRec = tup_block->cnoOfTabDescrRec;
  c_tup->c_page_map_pool_ptr = tup_block->c_page_map_pool_ptr;
}

void Dblqh::setup_query_thread_for_scan_access(Uint32 instanceNo) {
  Dblqh *lqh_block = (Dblqh *)globalData.getBlock(DBLQH, instanceNo);
  Dbtup *tup_block = (Dbtup *)globalData.getBlock(DBTUP, instanceNo);
  Dbacc *acc_block = (Dbacc *)globalData.getBlock(DBACC, instanceNo);
  Dbtux *tux_block = (Dbtux *)globalData.getBlock(DBTUX, instanceNo);
  c_acc->m_ldm_instance_used = acc_block;
  c_tup->m_ldm_instance_used = tup_block;
  this->m_ldm_instance_used = lqh_block;

  this->c_fragment_pool.setArrayPtr(lqh_block->c_fragment_pool.getArrayPtr());
  this->tablerec = lqh_block->tablerec;

  c_acc->fragmentrec = acc_block->fragmentrec;
  c_acc->directoryPoolPtr = acc_block->directoryPoolPtr;

  c_tup->fragrecord = tup_block->fragrecord;
  c_tup->tablerec = tup_block->tablerec;
  c_tup->tableDescriptor = tup_block->tableDescriptor;
  c_tup->cnoOfTabDescrRec = tup_block->cnoOfTabDescrRec;
  c_tup->c_page_map_pool_ptr = tup_block->c_page_map_pool_ptr;

  c_tux->c_fragPool.setArrayPtr(tux_block->c_fragPool.getArrayPtr());
  c_tux->c_indexPool.setArrayPtr(tux_block->c_indexPool.getArrayPtr());
  c_tux->c_descPagePool.setArrayPtr(tux_block->c_descPagePool.getArrayPtr());
}

void Dblqh::reset_query_thread_access() {
  this->c_fragment_pool.setArrayPtr(0);
  this->tablerec = 0;
  c_acc->fragmentrec = 0;
  c_acc->directoryPoolPtr = 0;
  c_tup->fragrecord = 0;
  c_tup->tablerec = 0;
  c_tup->cnoOfTabDescrRec = 0;
  c_tup->c_page_map_pool_ptr = 0;
  c_tux->c_fragPool.setArrayPtr(0);
  c_tux->c_indexPool.setArrayPtr(0);
}

void Dblqh::setup_query_thread_for_restore_access(Uint32 instance_no,
                                                  Uint32 newestGci) {
  setup_query_thread_for_scan_access(instance_no);
  this->m_current_ldm_instance = instance_no;
  this->cnewestGci = newestGci;
  DEB_NEWEST_GCI(("(%u)(%u) cnewestGci = %u, line: %u", instance(),
                  m_is_query_block, this->cnewestGci, __LINE__));
  m_is_in_query_thread = false;
  c_tup->m_is_in_query_thread = false;
  c_acc->m_is_in_query_thread = false;
  ndbrequire(m_is_recover_block == false);
  m_is_recover_block = true;
}

void Dblqh::reset_restore_thread_access() {
  if (m_is_recover_block) {
    reset_query_thread_access();
    m_is_in_query_thread = true;
    c_tup->m_is_in_query_thread = true;
    c_acc->m_is_in_query_thread = true;
    m_is_recover_block = false;
  }
}

/**
 * Here we have some code to handle real-time breaks during
 * PREPARE of key operations and scanning operations. All
 * real-time breaks are handled through signals sent to the
 * DBLQH block, it isn't allowed to send an asynchronous
 * signal to any other block as part of processing an
 * LQHKEYREQ, SCAN_FRAGREQ and SCAN_NEXTREQ.
 *
 * There is one exception where DBTUP sends a CONTINUEB signal
 * to itself when performing a full partition scan,
 * as part of handling this method the setup_scan_pointers
 * method is called to initialise all variables.
 *
 * At receive of SCAN_FRAGREQ and LQHKEYREQ we build up the
 * block pointers for key operations and scan operations as
 * part of setting up the infrastructure to execute the key
 * and scan operation.
 *
 * For key operations there are only two real-time breaks
 * that can occur.
 * 1) After a row lock wait we return from DBACC in
 *    execACCKEYCONF.
 * 2) After reading the disk page we return in
 *    acckeyconf_load_diskpage_callback.
 *
 * In a scan we have many more places from where we can return
 * after a real-time break.
 *
 * 1) Normal real-time break when a scan must take a break to
 *    not overextend its scheduling budget. This returns in
 *    execACC_CHECK_SCAN from where we send the NEXT_SCANREQ
 *    signal to the scanning block.
 * 2) Normal break to communicate with the application.
 *    The application will return by sending SCAN_NEXTREQ and
 *    thus this will happen in execSCAN_NEXTREQ.
 * 3) When we try to access a locked row in DBTUP and DBTUX and
 *    when we are waiting for too many rows in DBACC we will
 *    send the signal CHECK_LCP_STOP to DBLQH. This will cause
 *    1 millisecond sleep and we will return with a CONTINUEB
 *    signal in DBLQH that will the method checkLcpStopBlockedLab.
 * 4) When we hit a row lock in the scanning block we will send
 *    NEXT_SCANCONF with no rows returned. This signal is sent
 *    as a buffered signal. This gives the scan a chance to
 *    discover a quick lock release. This real-time break will
 *    be started in execNEXT_SCANCONF in DBLQH.
 * 5) In various places when we hit a row lock and similar things
 *    in DBACC we will send the signal ACC_CHECK_SCAN to DBLQH.
 *    DBLQH will send this signal back to DBACC after the real-time
 *    break. This is handled in execACC_CHECK_SCAN in DBLQH.
 * 6) When performing a long scan in DBTUP (normally an LCP scan),
 *    we can hit a limit on how many pages we are allowed to scan
 *    before taking a real-time break. This is handled as a CONTINUEB
 *    signal in DBTUP that will call setup_scan_pointers in DBLQH.
 * 7) During a copy fragment scan we will return with a LQHKEYCONF
 *    or LQHKEYREF to continue scanning. We will setup the
 *    infrastructure to continue the scan just before calling
 *    nextRecordCopy from copyCompletedLab. Most of the time we will
 *    not continue the scan from such a signal, so we postpone it
 *    as long as we can.
 * 8) After having our Copy fragment scan halted we return to
 *    executing the copy fragment scan when executing the
 *    execRESUME_COPY_FRAG_REQ method.
 * 9) As part of closing down a scan process we can send a signal
 *    ACC_ABORTCONF to ensure that all ACCKEYCONF and ACCKEYREF
 *    have been received before we close the scan. This is received
 *    in both DBTUP and DBTUX.
 * 10)We also setup scan pointers when executing LQH_TRANSREQ to
 *    ensure that we have always setup those pointers properly in
 *    all paths.
 * 11)We take a real-time break also when restarting a scan operation
 *    that have been released from the queue. This ensures that we
 *    never execute multiple scans in the same real-time break. This
 *    would complicate the handling of pointers considerably.
 *    We handle this special case by sending a CONTINUEB to ourselves
 *    and checking that no one completed the scan while we were
 *    in the job buffer.
 * 12) During scan in disk page order we can end up in
 *     disk_page_tup_scan_callback after retrieving the disk page.
 */
void Dblqh::setup_key_pointers(Uint32 tcIndex, bool acquire_lock) {
  /**
   * We come here after a real-time break for a key operation.
   * At this point we need to setup the pointers in preparation
   * for calling TUP and ACC and the pointers internally in LQH.
   */
  jamDebug();
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  TcConnectionrecPtr tcConnectptr;
  FragrecordPtr fragPtr;
  tcConnectptr.i = tcIndex;
  ndbassert(!m_is_query_block);
  ndbrequire(tcConnect_pool.getUncheckedPtrRW(tcConnectptr));
  c_tup->prepare_op_pointer(tcConnectptr.p->tupConnectrec,
                            tcConnectptr.p->tupConnectPtrP);
  ndbrequire(c_fragment_pool.getPtr(fragPtr, tcConnectptr.p->fragmentptr));
  fragptr = fragPtr;
  c_tup->prepare_tab_pointers(fragPtr.p->tupFragptr);
  m_tc_connect_ptr = tcConnectptr;
  ndbrequire(Magic::check_ptr(tcConnectptr.p));
  ndbrequire(Magic::check_ptr(tcConnectptr.p->tupConnectPtrP));
  ndbrequire(Magic::check_ptr(tcConnectptr.p->accConnectPtrP));
  if (qt_likely((globalData.ndbMtQueryThreads > 0) && acquire_lock)) {
    acquire_frag_prepare_key_access(fragptr.p, m_tc_connect_ptr.p);
  }
}

/**
 * After a real-time break it is necessary to setup the scan context
 * to avoid having to recompute these variables every time we need them.
 *
 * At reception of SCAN_FRAGREQ the scan context is setup bit by bit
 * rather than in one method since we don't have access to all variables
 * at start of the SCAN_FRAGREQ.
 *
 * The following variables are computed:
 *
 * scanptr
 *   This is the pointer and i-value stored in LQH block object for the scan
 *   record.
 * fragptr
 *   This is pointer and i-value stored in LQH block object for the index
 *   fragment being scanned.
 * prim_tab_fragptr
 *   This is pointer and i-value stored in LQH block object for the table
 *   fragment being scanned.
 * m_tc_connect_ptr
 *   This is the pointer and i-value of the TC connect record used by the scan
 *   record. There is always one TC connect record attached to each active scan
 *   record.
 *
 * TUP variables setup by prepare_op_pointer:
 * ..........................................
 *
 * prepare_op_pointer
 *   The pointer and i-value of the operation record that is connected to
 *   TC connect record in LQH.
 *
 * TUP variables setup by prepare_tab_pointers:
 * ............................................
 *
 * prepare_fragptr
 *   The pointer and i-value of the table fragment record used by the scan.
 * prepare_tabptr
 *   The pointer and i-value of the table record used by the scan.
 *
 * TUP variable setup by copyAttrinfo:
 * ...................................
 * The Attrinfo that contains the scan stored procedure is an Attrinfo program.
 * It is stored in signal segments while being executed. When TUP executes the
 * scan the Attrinfo program must be moved to a linear array. This linear array
 * is the cinBuffer variable in TUP. Using a linear array simplifies the code
 * in TUP greatly. Avoiding to copy this for each row we scan saves a lot of
 * computations.
 *
 * The method prepare_scan_ctx calls a method in ACC, TUP or TUX. This method
 * sets up the scan context for the block where the scan is performed. ACC is
 * used for full table scans. TUP is used for LCP scans and Node Restart
 * scans. TUP can also be called for full table scans from the NDB API
 * (but not from SQL). TUX is used for all range scans and represents the
 * majority of the scans performed. These methods describe what context they
 * setup. TUX scan context setup is described in DbtuxScan.cpp.
 *
 * Additional TUP variables to setup before calling execTUPKEYREQ
 * --------------------------------------------------------------
 * execTUPKEYREQ is used to read, update, delete, insert and read for scan rows
 * in TUP.
 * Before we can call this method we must also setup a few additional
 * variables:
 *
 * prepare_pageptr
 *   This is the pointer and i-value of the page where the fixed part of the
 *   row resides.
 * prepare_tuple_ptr
 *   This is a pointer to the start of the tuple in the fixed part of the row.
 * prepare_page_no
 *   This is the physical page number of the fixed size page of the row.
 *
 * These are setup differently dependent on from where we come:
 * prepareTUPKEYREQ
 * ................
 *   This method is called after a real-time break for key operations. It also
 *   sets up prepare_tabptr and prepare_fragptr.
 *
 * prepare_scanTUPKEYREQ
 * .....................
 *   This method is used when preparing to call execTUPKEYREQ from scans in
 *   TUP and ACC. This method gets a page id that is a logical page id since
 *   TUP has the logical page id since it scans in row id order and ACC stores
 *   the logical page id of the row. TUX stores the physical page id of the
 *   row instead to speed up things since it only requires the physical page
 *   id when reading the row.
 *
 * prepare_scan_tux_TUPKEYREQ
 * ..........................
 *   Used when preparing to call execTUPKEYREQ from TUX scans.
 *
 * All these methods also try to prefetch the fixed part of the row to avoid
 * cache miss waits.
 */
void Dblqh::setup_scan_pointers_from_tc_con(TcConnectionrecPtr tcConnectptr,
                                            Uint32 line) {
  /**
   * We come here after a real-time break, we need to setup
   * infrastructure in TUP and here in LQH for execution of
   * direct signals.
   */
  jamDebug();
  FragrecordPtr loc_fragptr;
  FragrecordPtr loc_prim_tab_fragptr;
  ScanRecordPtr loc_scanptr;
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  loc_scanptr.i = tcConnectptr.p->tcScanRec;
  loc_fragptr.i = tcConnectptr.p->fragmentptr;
  if (qt_likely(m_is_query_block)) {
    setup_query_thread_for_scan_access(tcConnectptr.p->ldmInstance);
  }
  ndbrequire(c_scanRecordPool.getUncheckedPtrRW(loc_scanptr));
  loc_prim_tab_fragptr.i = loc_scanptr.p->fragPtrI;
  c_fragment_pool.getPtr(loc_fragptr);
  m_scan_direct_count = 1;
  m_tot_scan_direct_count = 0;
  SimulatedBlock *block = loc_scanptr.p->scanBlock;
  scanptr = loc_scanptr;
  fragptr = loc_fragptr;
  c_fragment_pool.getPtr(loc_prim_tab_fragptr);
  c_tup->prepare_op_pointer(tcConnectptr.p->tupConnectrec,
                            tcConnectptr.p->tupConnectPtrP);
  m_tc_connect_ptr = tcConnectptr;
  prim_tab_fragptr = loc_prim_tab_fragptr;
  c_tup->prepare_tab_pointers(loc_prim_tab_fragptr.p->tupFragptr);
  ndbrequire(Magic::check_ptr(loc_scanptr.p));
  if (likely(loc_scanptr.p->scanStoredProcId != RNIL)) {
    jamDebug();
    Uint32 storedProcLen = c_tup->copyAttrinfo(loc_scanptr.p->scanStoredProcId,
                                               bool(tcConnectptr.p->opExec));
    (void)storedProcLen;
    ndbassert(loc_scanptr.p->scanAiLength == storedProcLen);
  }
  /**
   * Important to acquire shared access before we prepare the
   * scan context in DBTUX. Preparing scan context in DBTUX involves
   * setting up the variable scanLinkedPos and before we have
   * acquired shared access another thread can still move the scan
   * position, so it isn't safe to prepare scan context without having
   * ensured shared access to the fragment.
   */
  acquire_frag_scan_access_new(prim_tab_fragptr.p, m_tc_connect_ptr.p);
  block->prepare_scan_ctx(loc_scanptr.p->scanAccPtr);
  ndbrequire(Magic::check_ptr(tcConnectptr.p->tupConnectPtrP));
  ndbrequire(Magic::check_ptr(tcConnectptr.p->accConnectPtrP));
  loc_scanptr.p->scan_startLine = line;
}

void Dblqh::setup_scan_pointers(Uint32 scanPtrI, Uint32 line) {
  /**
   * We come here after a real-time break, we need to setup
   * infrastructure in TUP and here in LQH for execution of
   * direct signals.
   */
  jamDebug();
  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  FragrecordPtr loc_fragptr;
  FragrecordPtr loc_prim_tab_fragptr;
  ScanRecordPtr loc_scanptr;
  TcConnectionrecPtr loc_tcConnectptr;
  loc_scanptr.i = scanPtrI;
  ndbrequire(c_scanRecordPool.getUncheckedPtrRW(loc_scanptr));
  loc_tcConnectptr.i = loc_scanptr.p->scanTcrec;
  loc_prim_tab_fragptr.i = loc_scanptr.p->fragPtrI;
  ndbrequire(tcConnect_pool.getUncheckedPtrRW(loc_tcConnectptr));
  if (qt_likely(m_is_query_block)) {
    setup_query_thread_for_scan_access(loc_tcConnectptr.p->ldmInstance);
  }
  loc_fragptr.i = loc_tcConnectptr.p->fragmentptr;
  c_fragment_pool.getPtr(loc_prim_tab_fragptr);
  c_fragment_pool.getPtr(loc_fragptr);
  m_scan_direct_count = 1;
  m_tot_scan_direct_count = 0;
  prim_tab_fragptr = loc_prim_tab_fragptr;
  scanptr = loc_scanptr;
  ndbrequire(Magic::check_ptr(loc_scanptr.p));
  SimulatedBlock *block = loc_scanptr.p->scanBlock;
  ndbrequire(Magic::check_ptr(loc_tcConnectptr.p));
  c_tup->prepare_op_pointer(loc_tcConnectptr.p->tupConnectrec,
                            loc_tcConnectptr.p->tupConnectPtrP);
  fragptr = loc_fragptr;
  m_tc_connect_ptr = loc_tcConnectptr;
  c_tup->prepare_tab_pointers(loc_prim_tab_fragptr.p->tupFragptr);
  if (likely(loc_scanptr.p->scanStoredProcId != RNIL)) {
    jamDebug();
    Uint32 storedProcLen = c_tup->copyAttrinfo(
        loc_scanptr.p->scanStoredProcId, bool(loc_tcConnectptr.p->opExec));
    (void)storedProcLen;
    ndbassert(loc_scanptr.p->scanAiLength == storedProcLen);
  }
  acquire_frag_scan_access_new(prim_tab_fragptr.p, m_tc_connect_ptr.p);
  block->prepare_scan_ctx(loc_scanptr.p->scanAccPtr);
  ndbrequire(Magic::check_ptr(loc_tcConnectptr.p->tupConnectPtrP));
  ndbrequire(Magic::check_ptr(loc_tcConnectptr.p->accConnectPtrP));
  loc_scanptr.p->scan_startLine = line;
}

void Dblqh::checkLcpStopBlockedLab(Signal *signal, Uint32 scanPtrI) {
  /**
   * We are back from a real-time break, we need to
   * setup the pointer infrastructure before starting
   * off the execution.
   */
  setup_scan_pointers(scanPtrI, __LINE__);
  ScanRecord *const scanPtr = scanptr.p;
  SimulatedBlock *block = scanPtr->scanBlock;
  Fragrecord::FragStatus fragstatus = fragptr.p->fragStatus;
  ExecFunction f = block->getExecuteFunction(GSN_ACC_CHECK_SCAN);
  scanPtr->scan_lastSeen = __LINE__;
  signal->theData[0] = scanPtr->scanAccPtr;
  signal->theData[1] = AccCheckScan::ZNOT_CHECK_LCP_STOP;
  ndbrequire(is_scan_ok(scanPtr, fragstatus));
  block->EXECUTE_DIRECT_FN(f, signal);
  release_frag_access(prim_tab_fragptr.p);
}  // Dblqh::checkLcpStopBlockedLab()

void Dblqh::execACC_CHECK_SCAN(Signal *signal) {
  /**
   * A real-time break was invoked in the middle of the scan
   * processing.
   *
   * This could either be invoked by DBACC due to some lock
   * wait issue or it could be a normal scheduling break
   * invoked since we have executed too many signals in a
   * row. In this case it is the NEXT_SCANREQ signal that
   * is sent.
   */
  jamDebug();
  Uint32 scanPtrI = signal->theData[0];
  setup_scan_pointers(scanPtrI, __LINE__);
  Uint32 sig_number = signal->theData[1];
  ScanRecord *scanPtr = scanptr.p;
  SimulatedBlock *block = scanPtr->scanBlock;
  if (sig_number == GSN_NEXT_SCANREQ) {
    jamDebug();
    ExecFunction f = block->getExecuteFunction(GSN_NEXT_SCANREQ);
    signal->theData[0] = scanPtr->scanAccPtr;
    signal->theData[1] = signal->theData[2];
    signal->theData[2] = signal->theData[3];
    block->EXECUTE_DIRECT_FN(f, signal);
  } else {
    jamDebug();
    ExecFunction f = block->getExecuteFunction(GSN_ACC_CHECK_SCAN);
    ndbrequire(sig_number == GSN_ACC_CHECK_SCAN);
    signal->theData[0] = scanPtr->scanAccPtr;
    signal->theData[1] = signal->theData[2];
    block->EXECUTE_DIRECT_FN(f, signal);
  }
  release_frag_access(prim_tab_fragptr.p);
}

/* ***************>> */
/*  NEXT_SCANCONF  > */
/* ***************>> */
void Dblqh::execNEXT_SCANCONF(Signal *signal) {
  /**
   * The scan block sent an asynchronous signal, when this arrives
   * we have had a real-time break, so we need to setup the scan
   * execution environment again.
   */
  jamEntryDebug();
  NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
  setup_scan_pointers(nextScanConf->scanPtr, __LINE__);
  const Uint32 pageNo = nextScanConf->localKey[0];
  const Uint32 pageIdx = nextScanConf->localKey[1];
  jamDebug();

  ScanRecord *const scanPtr = scanptr.p;
  scanPtr->m_row_id.m_page_idx = pageIdx;
  scanPtr->m_row_id.m_page_no = pageNo;
  continue_next_scan_conf(signal, scanPtr->scanState, scanPtr);
  release_frag_access(prim_tab_fragptr.p);
}

void Dblqh::exec_next_scan_conf(Signal *signal) {
  /**
   * The scan block sent an immediate signal requiring no
   * real-time break.
   */
  jamDebug();
  NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
  ScanRecord *const scanPtr = scanptr.p;
  const Uint32 pageNo = nextScanConf->localKey[0];
  const Uint32 pageIdx = nextScanConf->localKey[1];
  /**
   * The local key is a row id when scanning ACC and TUP. In TUX we store
   * the physical row id and we don't need the row id for anything in an
   * ordered index scan. So in the case of a TUX scan we will return a
   * physical row id and not the logical row id.
   */
  scanPtr->m_row_id.m_page_idx = pageIdx;
  scanPtr->m_row_id.m_page_no = pageNo;
  continue_next_scan_conf(signal, scanPtr->scanState, scanPtr);
}

void Dblqh::continue_next_scan_conf(Signal *signal,
                                    ScanRecord::ScanState scanState,
                                    ScanRecord *const scanPtr) {
#ifdef VM_TRACE
  NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
  if (signal->getLength() > 2 && nextScanConf->accOperationPtr != RNIL) {
    Ptr<TcConnectionrec> regTcPtr;
    regTcPtr.i = scanPtr->scanTcrec;
    ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));
    ndbassert(regTcPtr.p->fragmentid == nextScanConf->fragId);
  }
#endif
  switch (scanState) {
    case ScanRecord::WAIT_NEXT_SCAN: {
      jamDebug();
      NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
      nextScanConfScanLab(signal, scanPtr, nextScanConf->fragId,
                          nextScanConf->accOperationPtr, m_tc_connect_ptr);
      return;
    }
    case ScanRecord::WAIT_NEXT_SCAN_COPY:
      jamDebug();
      ndbassert(!m_is_query_block);
      nextScanConfCopyLab(signal, m_tc_connect_ptr);
      return;
    case ScanRecord::WAIT_CLOSE_SCAN:
      jamDebug();
      accScanCloseConfLab(signal, m_tc_connect_ptr);
      return;
    case ScanRecord::WAIT_CLOSE_COPY:
      jamDebug();
      ndbassert(!m_is_query_block);
      accCopyCloseConfLab(signal, m_tc_connect_ptr);
      return;
    default:
      jamLine(scanPtr->scanState);
      ndbabort();
  }  // switch
}

/* --------------------------------------------------------------------------
 *       ENTER SCAN_NEXTREQ
 * --------------------------------------------------------------------------
 *       PRECONDITION:
 *       TRANSACTION_STATE = SCAN_STATE
 *       SCAN_STATE = WAIT_SCAN_NEXTREQ
 *
 * Case scanLockHold: ZTRUE  = Unlock previous round of
 *                             scanned row(s) and fetch next set of rows.
 *                    ZFALSE = Fetch new set of rows.
 * Number of rows to read depends on parallelism and how many rows
 * left to scan in the fragment. SCAN_NEXTREQ can also be sent with
 * closeFlag == ZTRUE to close the scan.
 * ------------------------------------------------------------------------- */
void Dblqh::execSCAN_NEXTREQ(Signal *signal) {
  jamEntry();
  const ScanFragNextReq *const nextReq = (ScanFragNextReq *)&signal->theData[0];
  const Uint32 transid1 = nextReq->transId1;
  const Uint32 transid2 = nextReq->transId2;
  const Uint32 senderData = nextReq->senderData;
  Uint32 senderBlockRef = signal->getSendersBlockRef();
  // bug#13834481 hashHi!=0 caused timeout (tx not found)

  TcConnectionrecPtr tcConnectptr;
  if (unlikely(findTransaction(transid1, transid2, senderData, senderBlockRef,
                               false, false, tcConnectptr) != ZOK)) {
    jam();
    LQH_DEBUG(senderData
              << " Received SCAN_NEXTREQ in LQH with close flag when closed");
    ndbrequire(ScanFragNextReq::getCloseFlag(nextReq->requestInfo));
    return;
  }

  // Crash node if signal sender is same node
  CRASH_INSERTION2(5021, refToNode(senderBlockRef) == cownNodeid);
  // Crash node if signal sender is NOT same node
  CRASH_INSERTION2(5022, refToNode(senderBlockRef) != cownNodeid);

  if (ERROR_INSERTED(5023)) {
    // Drop signal if sender is same node
    if (refToNode(senderBlockRef) == cownNodeid &&
        refToMain(senderBlockRef) != getBACKUP()) {
      CLEAR_ERROR_INSERT_VALUE;
      return;
    }
  }  // if
  if (ERROR_INSERTED(5024)) {
    // Drop signal if sender is NOT same node
    if (refToNode(senderBlockRef) != cownNodeid) {
      CLEAR_ERROR_INSERT_VALUE;
      return;
    }
  }  // if
  if (ERROR_INSERTED(5025)) {
    /**
     * This does not work as signal->getSendersBlockRef() is used
     *   as "hashHi"...not having a real data-word for this is not optimal
     *   but it will work...summary: disable this ERROR_INSERT
     */
    CLEAR_ERROR_INSERT_VALUE;
  }

  if (ERROR_INSERTED(5030)) {
    if (refToMain(senderBlockRef) != getBACKUP()) {
      ndbout << "ERROR 5030" << endl;
      CLEAR_ERROR_INSERT_VALUE;
      // Drop signal
      return;
    }
  }  // if

  Uint32 pos = 0;
  if (ScanFragNextReq::getCorrFactorFlag(nextReq->requestInfo)) {
    jamDebug();
    Uint32 corrFactorLo = nextReq->variableData[pos++];
    tcConnectptr.p->m_corrFactorLo &= 0xFFFF0000;
    tcConnectptr.p->m_corrFactorLo |= corrFactorLo;
  }

  setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
  scanptr.p->scanTcWaiting = cLqhTimeOutCount;
  /* ------------------------------------------------------------------
   * If close flag is set this scan should be closed
   * If we are waiting for SCAN_NEXTREQ set flag to stop scanning and
   * continue execution else set flags and wait until the scan
   * completes itself
   * ------------------------------------------------------------------ */
  if (ScanFragNextReq::getCloseFlag(nextReq->requestInfo)) {
    jamDebug();
    if (ERROR_INSERTED(5034)) {
      CLEAR_ERROR_INSERT_VALUE;
    }
    closeScanRequestLab(signal, tcConnectptr);
    release_frag_access(prim_tab_fragptr.p);
    return;
  }  // if
  scanptr.p->prioAFlag = ScanFragNextReq::getPrioAFlag(nextReq->requestInfo);
  scanptr.p->m_exec_direct_batch_size_words = 0;

  ScanRecord *const scanPtr = scanptr.p;
  const Uint32 max_rows = nextReq->batch_size_rows;
  const Uint32 max_bytes = nextReq->batch_size_bytes;
  scanPtr->m_max_batch_size_bytes = max_bytes;
  {
    /**
     * To speed up drop table we check for table being dropped here.
     * This can speed up drop table by minutes, so even though it is
     * a small cost at every scan batch, it will provide us a much
     * more reliable time of execution for drop tables. Drop table
     * can also be delayed by user transactions. So we could potentially
     * make this check for all scan types. This will however require
     * user transactions to add checks for this error, this they should
     * however already have since it is checked at first SCAN_FRAGREQ.
     *
     * We don't need to worry about backups since they will take a lock
     * before they start, so this won't happen to a backup. No drop tables
     * can run concurrently with drop table. Also any ALTER TABLE activity
     * cannot run at the same time as a drop table, so this is also safe
     * for all sorts of table reorg scans. What remains is node recovery
     * scans to synchronize data.
     */
    TablerecPtr tabPtr;
    tabPtr.i = tcConnectptr.p->tableref;
    ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
    if (unlikely(tabPtr.p->tableStatus != Tablerec::TABLE_DEFINED &&
                 tabPtr.p->tableStatus != Tablerec::TABLE_READ_ONLY)) {
      tcConnectptr.p->errorCode = get_table_state_error(tabPtr);
      closeScanRequestLab(signal, tcConnectptr);
      release_frag_access(prim_tab_fragptr.p);
      return;
    }
  }
  if (unlikely(max_rows > scanPtr->m_max_batch_size_rows)) {
    jam();
    /**
     * Extend list...
     * Will never happen for LCP, Backup and NR.
     */
    ndbrequire(scanPtr->m_reserved == 0);
    if (unlikely(!seize_acc_ptr_list(scanPtr, scanPtr->m_max_batch_size_rows,
                                     max_rows))) {
      jam();
      tcConnectptr.p->errorCode = ScanFragRef::ZTOO_MANY_ACTIVE_SCAN_ERROR;
      closeScanRequestLab(signal, tcConnectptr);
      release_frag_access(prim_tab_fragptr.p);
      return;
    }
    scanPtr->m_max_batch_size_rows = max_rows;
  } else if (unlikely(max_rows < scanPtr->m_max_batch_size_rows)) {
    jam();
    scanPtr->m_max_batch_size_rows = max_rows;
  }

  /* --------------------------------------------------------------------
   * If scanLockHold = true we need to unlock previous round of
   * scanned records.
   * scanReleaseLocks will set states for this and send a NEXT_SCANREQ.
   * When confirm signal NEXT_SCANCONF arrives we call
   * continueScanNextReqLab to continue scanning new rows and
   * acquiring new locks.
   * -------------------------------------------------------------------- */
  if (unlikely((scanPtr->scanLockHold == ZTRUE) &&
               (scanPtr->m_curr_batch_size_rows > 0))) {
    jam();
    scanPtr->scanReleaseCounter = 1;
    scanReleaseLocksLab(signal, tcConnectptr.p);
    release_frag_access(prim_tab_fragptr.p);
    return;
  }  // if

  /* -----------------------------------------------------------------------
   * We end up here when scanLockHold = false or no rows was locked from
   * previous round.
   * Simply continue scanning.
   * ----------------------------------------------------------------------- */
  continueScanNextReqLab(signal, tcConnectptr.p);
  release_frag_access(prim_tab_fragptr.p);
}  // Dblqh::execSCAN_NEXTREQ()

void Dblqh::continueScanNextReqLab(Signal *signal,
                                   TcConnectionrec *const regTcPtr) {
  ScanRecord *const scanPtr = scanptr.p;
  if (unlikely(scanPtr->scanCompletedStatus == ZTRUE)) {
    jam();
    closeScanLab(signal, regTcPtr);
    return;
  }  // if

  if (scanPtr->m_last_row) {
    jamDebug();
    scanPtr->scanCompletedStatus = ZTRUE;
    scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
    scanPtr->scan_lastSeen = __LINE__;
    sendScanFragConf(signal, ZFALSE, regTcPtr);
    return;
  }

  // Update timer on tcConnectRecord
  regTcPtr->tcTimer = cLqhTimeOutCount;
  init_acc_ptr_list(scanPtr);
  scanPtr->scanFlag = NextScanReq::ZSCAN_NEXT;
  scanNextLoopLab(signal, regTcPtr->clientConnectrec, RNIL, scanPtr, fragptr.p);
}  // Dblqh::continueScanNextReqLab()

void Dblqh::scanNextLoopLab(Signal *signal, Uint32 clientPtrI, Uint32 accOpPtr,
                            ScanRecord *const scanPtr,
                            Fragrecord *const fragPtr) {
  Fragrecord::FragStatus fragstatus = fragPtr->fragStatus;
  Uint32 scanFlag = scanPtr->scanFlag;
  const Uint32 sig0 = scanPtr->scanAccPtr;
  SimulatedBlock *block = scanPtr->scanBlock;
  ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;
  Uint32 in_send_next_scan = m_in_send_next_scan;

  signal->theData[0] = sig0;
  signal->theData[1] = accOpPtr;
  signal->theData[2] = scanFlag;

  ndbrequire(is_scan_ok(scanPtr, fragstatus));
  scanPtr->scanState = ScanRecord::WAIT_NEXT_SCAN;
  scanPtr->scan_lastSeen = __LINE__;
  if (unlikely(in_send_next_scan == 0)) {
    send_next_NEXT_SCANREQ(signal, block, f, scanPtr, clientPtrI);
    return;
  }
  /**
   * At this point we don't call send_next_NEXT_SCANREQ since we
   * want to unwind the call stack before entering this function.
   * m_in_send_next_scan equal to 1 indicates that we are
   * executing this function already and that we will check the
   * result of this function when we return to send_next_NEXT_SCANREQ
   * after unwinding the stack.
   *
   * It is imperative that we return immediately in all the call
   * stack until we return to send_next_NEXT_SCANREQ again. This
   * secures the signal object that is setup already to send the
   * NEXT_SCANREQ signal.
   *
   * We indicate that we have another signal to process by setting
   * m_in_send_next_scan to 2.
   */
  ndbassert(in_send_next_scan == 1);
  m_in_send_next_scan = 2;
}  // Dblqh::scanNextLoopLab()

void Dblqh::scanLockReleasedLab(Signal *signal,
                                TcConnectionrec *const regTcPtr) {
  ScanRecord *const scanPtr = scanptr.p;
  if (scanPtr->scanReleaseCounter == scanPtr->m_curr_batch_size_rows) {
    if ((scanPtr->scanErrorCounter > 0) ||
        (scanPtr->scanCompletedStatus == ZTRUE)) {
      jam();
      scanPtr->m_curr_batch_size_rows = 0;
      scanPtr->m_curr_batch_size_bytes = 0;
      closeScanLab(signal, regTcPtr);
    } else if (scanPtr->m_last_row && !scanPtr->scanLockHold) {
      jam();
      closeScanLab(signal, regTcPtr);
    } else if (scanPtr->check_scan_batch_completed() &&
               scanPtr->scanLockHold != ZTRUE) {
      jam();
      scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
      scanPtr->scan_lastSeen = __LINE__;
      sendScanFragConf(signal, ZFALSE, regTcPtr);
    } else {
      jam();
      /*
       * We came here after releasing locks after
       * receiving SCAN_NEXTREQ from TC. We only come here
       * when scanLockHold == ZTRUE
       */
      ndbassert(!m_is_query_block);
      scanPtr->m_curr_batch_size_rows = 0;
      scanPtr->m_curr_batch_size_bytes = 0;
      continueScanNextReqLab(signal, regTcPtr);
      return;
    }  // if
  } else if (scanPtr->scanCompletedStatus != ZTRUE) {
    jam();
    /*
    We come here when we have been scanning for a long time and not been able
    to find m_max_batch_size_rows records to return. We needed to release
    the record we didn't want, but now we are returning all found records to
    the API.
    */
    scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
    scanPtr->scan_lastSeen = __LINE__;
    sendScanFragConf(signal, ZFALSE, regTcPtr);
  } else {
    jam();
    closeScanLab(signal, regTcPtr);
  }
  return;
}  // Dblqh::scanLockReleasedLab()

/* -------------------------------------------------------------------------
 *       WE NEED TO RELEASE LOCKS BEFORE CONTINUING
 * ------------------------------------------------------------------------- */
void Dblqh::scanReleaseLocksLab(Signal *signal,
                                TcConnectionrec *const regTcPtr) {
  ScanRecord *const scanPtr = scanptr.p;
  Fragrecord::FragStatus fragstatus = fragptr.p->fragStatus;
  ndbrequire(is_scan_ok(scanPtr, fragstatus));
  check_send_scan_hb_rep(signal, scanPtr, regTcPtr);
  while (true) {
    const Uint32 sig1 = get_acc_ptr_from_scan_record(
        scanPtr, scanPtr->scanReleaseCounter - 1, false);
    const Uint32 sig0 = scanPtr->scanAccPtr;
    SimulatedBlock *block = scanPtr->scanBlock;
    ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;

    signal->theData[1] = sig1;
    signal->theData[2] = NextScanReq::ZSCAN_COMMIT;
    signal->theData[0] = sig0;
    /* EXECUTE_DIRECT optimised to NEXT_SCANREQ in TUP/ACC/TUX */

    /**
     * DESIGN PATTERN DESCRIPTION:
     * ---------------------------
     * When calling another block immediately there is a set of ways to
     * do this.
     *
     * Standard, non-optimised manner:
     * Use EXECUTE_DIRECT with four parameters that specify block reference
     * of receiver, the signal object, the global signal number and the
     * length of the signal (There is also a variant used when sending such
     * a signal to a different instance). This method is fairly optimised
     * but has quite a lot of potential for performance improvement.
     *
     * Standard, optimised manner:
     * In this case we optimise things by translating to block object and
     * retrieving the function pointer in the block call. This gives
     * the compiler assistance to separate the loads and stores more from
     * each other.
     * The call will be seen as:
     * block->EXECUTE_DIRECT(signal, f);
     *
     * This manner optimises the code but retains the flexibility and also
     * the possibility to trace signal execution between blocks.
     *
     * Non-standard, optimised manner:
     * In this case we remove some of the flexibility of the call to enhance
     * the performance yet a bit more. In this case we remove the possibility
     * for a flexible receiver of the signal, it is directed to a certain
     * block and we also call the method directly without any indirection.
     * The call will be seen as e.g.:
     * c_tup->execTUPKEYREQ(signal);
     *
     * The standard manner of calling EXECUTE_DIRECT are both always calling
     * functions with one parameter being the signal object and no return
     * value. There is however two ways of sending signals back. In some
     * cases a signal is always sent back when returning from the
     * EXECUTE_DIRECT call, in this case the signal object returned contains
     * a signal object with data for the return signal. In some cases
     * one gets a return signal in this manner by a combination of the signal
     * number and the parameters. In the example below with NEXT_SCANREQ
     * we always gets a return signal when specifying ZSCAN_COMMIT
     * but not in other cases.
     *
     * The other manner of sending a return signal is to perform a new
     * EXECUTE_DIRECT signal. In those cases one needs to ensure that the
     * call chain is bounded to not run out of stack. It is also a good
     * idea to try and ensure that the EXECUTE_DIRECT can use the
     * tail-call optimisation to avoid using too much stack which is bad
     * for CPU caching. This means returning immediately after
     * EXECUTE_DIRECT but also avoiding having objects that needs
     * destruction after return and also avoiding taking the reference of
     * stack variables.
     *
     * Using the non-standard manner one can obviously also change more
     * ways, one can return a bool for example as in the example with
     * execTUPKEYREQ, one can add parameters and one can even change the
     * name to a different name not according to the standard naming
     * conventions. Obviously doing this removes flexibility of using
     * blocks in a flexible manner.
     */
    block->EXECUTE_DIRECT_FN(f, signal);
    ndbrequire(signal->theData[0] == 0); /* Failure is not an option */
    if (scanPtr->scanReleaseCounter < scanPtr->m_curr_batch_size_rows) {
      jam();
      scanPtr->scanReleaseCounter++;
      /* Continue looping */
    } else {
      scanLockReleasedLab(signal, regTcPtr);
      /* No more records to commit for this scan at this time */
      return;
    }
  }
}  // Dblqh::scanReleaseLocksLab()

/* -------------------------------------------------------------------------
 *       ENTER SCAN_NEXTREQ
 * -------------------------------------------------------------------------
 *       SCAN_NEXT_REQ SIGNAL ARRIVED IN THE MIDDLE OF EXECUTION OF THE SCAN.
 *       IT WAS A REQUEST TO CLOSE THE SCAN. WE WILL CLOSE THE SCAN IN A
 *       CAREFUL MANNER TO ENSURE THAT NO ERROR OCCURS.
 * -------------------------------------------------------------------------
 *       PRECONDITION:
 *       TRANSACTION_STATE = SCAN_STATE_USED
 *       TSCAN_COMPLETED = ZTRUE
 * -------------------------------------------------------------------------
 *       WE CAN ALSO ARRIVE AT THIS LABEL AFTER A NODE CRASH OF THE SCAN
 *       COORDINATOR.
 * ------------------------------------------------------------------------- */
void Dblqh::closeScanRequestLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;
  LQH_DEBUG("transactionState = " << tcConnectptr.p->transactionState);
  switch (tcConnectptr.p->transactionState) {
    case TcConnectionrec::SCAN_STATE_USED:
      LQH_DEBUG("scanState = " << scanPtr->scanState);
      switch (scanPtr->scanState) {
        case ScanRecord::IN_QUEUE:
          jam();
          tupScanCloseConfLab(signal, tcConnectptr);
          return;
        case ScanRecord::WAIT_NEXT_SCAN:
          jam();
          /* -------------------------------------------------------------------
           *  SET COMPLETION STATUS AND WAIT FOR OPPORTUNITY TO STOP THE SCAN.
           * -------------------------------------------------------------------
           */
          scanPtr->scanCompletedStatus = ZTRUE;
          break;
        case ScanRecord::WAIT_START_QUEUED_SCAN:
          jam();
          /**
           * We are currently starting up a queued scan, need to retain
           * scan record until this signal arrives back.
           */
          tupScanCloseConfLab(signal, tcConnectptr);
          return;
        case ScanRecord::QUIT_START_QUEUE_SCAN:
          jam();
          /**
           * Scan is already closed, but waiting for a CONTINUEB signal, let
           * that be handled and do no more. Response back to TC about
           * closed state has already been sent, so no need to send it again.
           */
          break;
        case ScanRecord::WAIT_ACC_SCAN:
          jam();
          /* -------------------------------------------------------------------
           *  WE ARE CURRENTLY STARTING UP THE SCAN. SET COMPLETED STATUS
           *  AND WAIT FOR COMPLETION OF STARTUP.
           * -------------------------------------------------------------------
           */
          scanPtr->scanCompletedStatus = ZTRUE;
          break;
        case ScanRecord::WAIT_CLOSE_SCAN:
          jam();
          scanPtr->scanCompletedStatus = ZTRUE;
          return;
          /* -------------------------------------------------------------------
           *       CLOSE IS ALREADY ONGOING. WE NEED NOT DO ANYTHING.
           * -------------------------------------------------------------------
           */
        case ScanRecord::WAIT_SCAN_NEXTREQ:
          jam();
          /* -------------------------------------------------------------------
           * WE ARE WAITING FOR A SCAN_NEXTREQ FROM SCAN COORDINATOR(TC)
           * WHICH HAVE CRASHED. CLOSE THE SCAN
           * -------------------------------------------------------------------
           */
          scanPtr->scanCompletedStatus = ZTRUE;

          if (scanPtr->scanLockHold == ZTRUE) {
            if (scanPtr->m_curr_batch_size_rows > 0) {
              jam();
              scanPtr->scanReleaseCounter = 1;
              scanReleaseLocksLab(signal, tcConnectptr.p);
              return;
            }  // if
          }    // if
          closeScanLab(signal, tcConnectptr.p);
          return;
        default:
          ndbabort();
      }  // switch
      break;
    case TcConnectionrec::SCAN_TUPKEY:
      jam();
      /* ---------------------------------------------------------------------
       *       SET COMPLETION STATUS AND WAIT FOR OPPORTUNITY TO STOP THE SCAN.
       * ---------------------------------------------------------------------
       */
      scanPtr->scanCompletedStatus = ZTRUE;
      break;
    default:
      ndbabort();
  }  // switch
  if (scanPtr->scanBlock == c_tux) {
    jamDebug();
    c_tux->relinkScan(__LINE__);
  }
}  // Dblqh::closeScanRequestLab()

bool Dblqh::seize_acc_ptr_list(ScanRecord *scanP, Uint32 curr_batch_size,
                               Uint32 new_batch_size) {
  /*  1 maps to 0 segments
   * >1 maps to enough segments to store
   */
  Uint32 segments = (new_batch_size + (SectionSegment::DataLength - 2)) /
                    SectionSegment::DataLength;

  if (segments <= scanP->scan_acc_segments) {
    // No need to allocate more segments.
    return true;
  }

  /* Should never get here for reserved scans */
  ndbrequire(!scanP->m_reserved);

  if (new_batch_size > 1) {
    for (Uint32 i = 1 + scanP->scan_acc_segments; i <= segments; i++) {
      Uint32 seg = seizeSingleSegment();
      if (unlikely(seg == RNIL)) {
        jam();
        /* Cleanup any allocated segments and return */
        scanP->scan_acc_segments = (i - 1);
        release_acc_ptr_list(scanP);
        return false;
      }
      scanP->scan_acc_op_ptr[i] = seg;
    }
  }
  scanP->scan_acc_segments = segments;
  return true;
}

void Dblqh::release_acc_ptr_list(ScanRecord *scanP) {
  Uint32 i, segments;
  segments = scanP->scan_acc_segments;

  for (i = 1; i <= segments; i++) {
    releaseSection(scanP->scan_acc_op_ptr[i]);
  }
  scanP->scan_acc_segments = 0;
  scanP->scan_acc_index = 0;
}

Uint32 Dblqh::seizeSingleSegment() {
  Uint32 junk = 0;
  Uint32 iVal = RNIL;

  /* Rather grungy way to grab a segment */
  if (!appendToSection(iVal, &junk, 1)) return RNIL;

  return iVal;
}

void Dblqh::init_acc_ptr_list(ScanRecord *scanP) { scanP->scan_acc_index = 0; }

Uint32 Dblqh::get_acc_ptr_from_scan_record(ScanRecord *scanP, Uint32 index,
                                           bool crash_flag) {
  Uint32 *acc_ptr;
  if (!((index < MAX_PARALLEL_OP_PER_SCAN) && index < scanP->scan_acc_index)) {
    ndbrequire(crash_flag);
    return RNIL;
  }
  i_get_acc_ptr(scanP, acc_ptr, index);
  return *acc_ptr;
}

void Dblqh::set_acc_ptr_in_scan_record(ScanRecord *scanP, Uint32 index,
                                       Uint32 acc) {
  Uint32 *acc_ptr;
  ndbrequire((index == 0 || scanP->scan_acc_index == index) &&
             (index < MAX_PARALLEL_OP_PER_SCAN));
  scanP->scan_acc_index = index + 1;
  i_get_acc_ptr(scanP, acc_ptr, index);
  *acc_ptr = acc;
}

/**
 * The design of the SCAN algorithm within one LDM instance
 * --------------------------------------------------------
 * DBLQH controls the execution of scans on tables on behalf of DBTC and
 * DBSPJ. Here follows a signal overview of how a scan is performed within
 * one LDM instance. For a description of the global scan protocol
 * see DbtcMain.cpp as a comment before execSCAN_TABREQ.
 *
 * DBLQH only controls execution of a scan towards one partition of a
 * table. DBTC/DBSPJ is responsible for execution of scans toward the
 * entire table and ensuring that the API sees a consistent view of the
 * table.
 *
 * There are currently four types of scans implemented in one LDM
 * instance:
 *
 * Full table scan using hash index. This is implemented in DBACC.
 * Full table scan using row by row. This is implemented in DBTUP.
 * Full table scan using row by row in disk order. This is implemented in
 *   DBTUP.
 * Index scan using one or several ranges. This is implemented in DBTUX.
 *
 * DBLQH controls execution of one partition scan, Dependent on the scan
 * type, DBACC/DBTUP/DBTUX is responsible to get the row references to
 * the tuple scanned. DBTUP is responsible for reading of those rows and
 * finally DBACC is responsible for any locking of rows required as part
 * of the scan.
 *
 * Each scan is controlled by an interpreted program created by the API
 * and transported down to DBTUP. This program is sent as part of the
 * SCAN_FRAGREQ signal and passed to DBTUP in the STORED_PROCREQ signal.
 * This program is applied on each row reference passed to DBTUP by
 * execution of the execTUPKEYREQ signal.
 *
 * In index ranges one or more ranges is sent in the keyinfo part of the
 * SCAN_FRAGREQ. This range information is sent to DBTUX one range at a
 * time. Actually with multiple ranges, DBLQH will treat each range as a
 * separate scan towards the other blocks, so a scan will be started and
 * closed towards DBACC/DBTUP/DBTUX for each range involved.
 *
 * As an optimisation all signals locally in one LDM instance have been
 * converted to direct signals.
 * The following signals are used as part of the scan of one partition.
 * ACC_SCANREQ:
 *   This signal initialises an operation record in DBACC/DBTUP/DBTUX for
 *   scan of one range or a full partition. Always sent as a direct signal
 *   and returned immediately through signal object on return.
 *
 * STORED_PROCREQ:
 *   This signal stores the interpreted program used to read each tuple
 *   as part of the scan. The same signal is also used to deallocate the
 *   the interpreted program when the entire scan of all ranges have been
 *   completed. Always sent as a direct signal and returned immediately
 *   through signal object on return.
 *
 * ACC_LOCKREQ:
 *   Certain scans require a lock on the row before the row is read, this
 *   signal acquires such a lock. Always sent as a direct signal. Return
 *   signal not always sent immediately.
 *
 * ACCKEYCONF:
 *   Signal returned when the lock have been acquired, the signal is
 *   normally sent directly when the row is not locked, but for a locked
 *   row the signal can be sent even a second or more later. When sent the
 *   signal is sent as a direct signal.
 *
 * ACCKEYREF:
 *   Signal returned when acquiring lock failed, e.g. due to record deleted
 *   while waiting for it.
 *
 * ACC_ABORTCONF:
 *   Signal returned after aborting a scan using an asynchronous message to
 *   ensure that all asynchronous messages are delivered since setting the
 *   scan state as aborted.
 *
 * NEXT_SCANREQ:
 *   This signal is used with different meaning:
 *   ZSCAN_NEXT:
 *     Get the next row reference to read, returned in NEXT_SCANCONF signal.
 *   ZSCAN_NEXT_COMMIT:
 *     Get the next row reference to read AND unlock the specified row.
 *     Returned in NEXT_SCANCONF signal.
 *   ZSCAN_COMMIT:
 *     Unlock the specified row. Return signal is simply returned when
 *     returning from call to execNEXT_SCANREQ.
 *   ZSCAN_CLOSE:
 *     Close the scan in DBACC/DBTUP/DBTUX.
 *
 *   When sent as ZSCAN_COMMIT and ZSCAN_CLOSE it is always sent as a direct
 *   signal. Otherwise it is sent as direct or asynchronous signal dependent
 *   on the value of the scan_direct_count variable in the DBLQH scan
 *   record. The scan_direct_count variable ensures that we keep the number
 *   of direct signals sent bounded.
 *
 * NEXT_SCANCONF:
 *   Return signal to NEXT_SCANREQ containing row reference to read or
 *   indication of close completed. Always sent as a direct signal.
 *
 * TUPKEYREQ:
 *   This signal does the actual read of the row and sends the read row data
 *   directly to the API using the TRANSID_AI signal. This signal is always
 *   sent as a direct signal.
 *
 * ACC_CHECK_SCAN:
 *   Continue scanning from specified place. Used by DBACC/DBTUP/DBTUX as an
 *   internal signal as part of the scan. This signal can be sent both as
 *   an asynchronous signal and as a direct signal.
 *
 * SCAN_FRAGCONF:
 *   Return signal sent to DBTC/DBSPJ after completing a part of the scan,
 *   the signal carries a set of references to rows sent to the API. After
 *   sending this signal DBLQH will stop and wait for a SCAN_NEXTREQ to
 *   signal asking DBLQH to continue the scan of the partition. The number
 *   of rows scanned before sending SCAN_FRAGCONF is dependent on both
 *   configuration parameters and information in the SCAN_FRAGREQ signal.
 *
 *   This signal is also sent when the scan is fully completed.
 *   This signal is normally a distributed signal, so it is always sent as
 *   an asynchronous signal.
 *
 * SCAN_NEXTREQ:
 *   Request to continue scanning from DBTC/DBSPJ as requested to them from
 *   API.
 *   This signal is normally a distributed signal, so it is always sent as
 *   an asynchronous signal.
 *
 *  Below follows an example signal diagram of a scan of one partition.
 *
 *  DBLQH          ACC          TUP         ACC/TUP/TUX    API      DBTC
 *    |ACC_SCANREQ
 *    |----------------------------------------->|
 *    |<-----------------------------------------|
 *    | STORED_PROCREQ
 *    |------------------------->|
 *    |<-------------------------|
 *    | NEXT_SCANREQ (ZSCAN_NEXT)
 *    |----------------------------------------->|
 *    |                          prepare_scanTUPKEYREQ
 *    |                          |<--------------|
 *    |                          |-------------->|
 *    | NEXT_SCANCONF
 *    |<-----------------------------------------|
 *    | TUPKEYREQ
 *    |------------------------->|  TRANSID_AI
 *    |                          |-------------------------->|
 *    |<-------------------------|
 *    | NEXT_SCANREQ (ZSCAN_NEXT_COMMIT)
 *    |----------------------------------------->|
 *    |                          prepare_scanTUPKEYREQ
 *    |                          |<--------------|
 *    |                          |-------------->|
 *    | NEXT_SCANCONF
 *    |<-----------------------------------------|
 *    | TUPKEYREQ
 *    |------------------------->|  TRANSID_AI
 *    |                          |-------------------------->|
 *    |<-------------------------|
 *    Repeat above for as many rows as required before returning to the
 *    API. The above TRANSID_AI isn't necessary, the interpreted program
 *    could perform selection and decide to not send a specific row since
 *    it doesn't match the condition checked by the interpreted program.
 *    |
 *    | SCAN_FRAGCONF
 *    |---------------------------------------------------------------->|
 *    .... Some time for API and DBTC to process things.
 *    | SCAN_NEXTREQ
 *    |<----------------------------------------------------------------|
 *    | NEXT_SCANREQ (ZSCAN_NEXT_COMMIT)
 *    |----------------------------------------->|
 *    |                          prepare_scanTUPKEYREQ
 *    |                          |<--------------|
 *    |                          |-------------->|
 *    | NEXT_SCANCONF
 *    |<-----------------------------------------|
 *    | TUPKEYREQ
 *    |------------------------->|  TRANSID_AI
 *    |                          |-------------------------->|
 *    |<-------------------------|
 *    Repeat above again until time for next SCAN_FRAGCONF to be sent.
 *    When scan from NEXT_SCANCONF indicates there are no more tuples to
 *    fetch one starts to close the scan.
 *
 *    |
 *    | NEXT_SCANREQ (ZSCAN_NEXT_COMMIT)
 *    |----------------------------------------->|
 *    | NEXT_SCANCONF(no more tuples)
 *    |<-----------------------------------------|
 *    | NEXT_SCANREQ (ZSCAN_CLOSE)
 *    |----------------------------------------->|
 *    | NEXT_SCANCONF
 *    |<-----------------------------------------|
 *    | STORED_PROCREQ (delete interpreted program)
 *    |------------------------->|
 *    |<-------------------------|
 *    | SCAN_FRAGCONF (close flag set)
 *    |---------------------------------------------------------------->|
 *    Now the scan is completed.
 *
 *    Now a number of variations on the above signal diagrams:
 *    Scan with locking:
 *    In this we use the flag ZSCAN_NEXT all the time and never
 *    ZSCAN_NEXT_COMMIT, we handle things a bit differently instead when
 *    receiving SCAN_NEXTREQ where we perform a signal diagram like this:
 *
 *    | NEXT_SCANREQ (ZSCAN_COMMIT)
 *    |----------------------------------------->|
 *    |<-----------------------------------------|
 *    This is repeated for each row sent to the API in the previous
 *    SCAN_FRAGCONF signal.
 *
 *    If the application wants the row locked for longer time he have had
 *    the chance to perform a key lookup operation that took over the lock
 *    such that even when we unlock the scan lock, the transaction still
 *    retains a lock on the row.
 *
 *    After each row scanned we check if we've reached a scan heartbeat
 *    timeout. In case we have we send a SCAN_HBREP signal to DBTC/DBSPJ
 *    to inform about that we're still actively scanning even though no
 *    result rows have been sent. Remember here that a scan in DBLQH can
 *    potentially scan billions of rows while only returning very few to
 *    the API. Thus we can scan for an extended time without returning to
 *    the API. This is handled by the method check_send_scan_hb_rep.
 *
 *    Already from returning from ACC_SCANREQ we can discover that the
 *    partition (== fragment) is empty and go immediately to the close
 *    down code.
 *    For index scans we will send TUX_BOUND_INFO after ACC_SCANREQ and
 *    before sending STORED_PROCREQ to DBTUX. This will provide one range
 *    to DBTUX for scanning, if multiple ranges are to be scanned we
 *    startup a new scan as if it was a new SCAN_FRAGREQ received, but we
 *    don't need to send STORED_PROCREQ since the same interpreted program
 *    will be used. We will however send ACC_SCANREQ and TUX_BOUND_INFO
 *    also for this new range.
 *
 *  There are various reasons for temporarily stopping a scan, this could
 *  lack of operation records, holding too many row locks, one could also
 *  end up in this situation after waiting for a row lock.
 *
 *  To restart the scan again after any type of temporary stop one sends
 *  the signal ACC_CHECK_SCAN either as direct or as an asynchronous signal
 *  to DBACC/DBTUP/DBTUX. This signal is sent from many different places in
 *  DBLQH, DBACC, DBTUP and DBTUX. It is always sent as part of NEXT_SCANREQ
 *  processing.
 *
 *  When executing ACC_CHECK_SCAN one can flag to DBACC/DBTUP/DBTUX that one
 *  should check for a 1 ms delay with the flag ZCHECK_LCP_STOP. In previous
 *  versions this was also related to local checkpoints, this is no longer
 *  the case. Now it's only related to situations where it is required to
 *  perform an extra wait such that resources becomes available again.
 *
 *  DBTUP and DBTUX sends the signal CHECK_LCP_STOP to DBLQH in a number of
 *  situations, among other things when a locked key has been encountered.
 *  When the ACCKEYCONF signal then is received indicating that one acquired
 *  the lock, DBLQH will still wait for CHECK_LCP_STOP from DBLQH to return
 *  after a 1 ms delay. This is on the TODO-list to fix to ensure that we can
 *  proceed with these locked rows immediately after delivery. As it is now
 *  we can get up to 1 ms delay each time we encounter a locked row.
 */
/* -------------------------------------------------------------------------
 * SCAN_FRAGREQ: Request to start scanning the specified fragment of a table.
 * ------------------------------------------------------------------------- */
void Dblqh::send_scan_fragref(Signal *signal, Uint32 transid1, Uint32 transid2,
                              Uint32 senderData, Uint32 senderBlockRef,
                              Uint32 errorCode) {
  ScanFragRef *ref = (ScanFragRef *)&signal->theData[0];
  ref->senderData = senderData;
  ref->transId1 = transid1;
  ref->transId2 = transid2;
  ref->errorCode = errorCode;
  ref->senderRef = reference();
  sendSignal(senderBlockRef, GSN_SCAN_FRAGREF, signal,
             ScanFragRef::SignalLength, JBB);
}

void Dblqh::execSCAN_FRAGREQ(Signal *signal) {
  /* Reassemble if the request was fragmented */
  if (unlikely(!assembleFragments(signal))) {
    jam();
    return;
  }
  jamEntryDebug();

  if (ERROR_INSERTED(5097)) {
    ndbabort();
  }

  ScanFragReq *const scanFragReq = (ScanFragReq *)&signal->theData[0];
  Uint32 errorCode = 0;
  TcConnectionrec *regTcPtr;
  // bug#13834481 hashHi!=0 caused timeout (tx not found)

  ndbassert(m_fragment_lock_status == FRAGMENT_UNLOCKED);
  tabptr.i = scanFragReq->tableId;

  /* Short SCANFRAGREQ has no sections, Long SCANFRAGREQ has 1 or 2
   * Section 0 : Mandatory ATTRINFO section
   * Section 1 : Optional KEYINFO section
   */
  const Uint32 numSections = signal->getNoOfSections();

  SectionHandle handle(this, signal);

  SegmentedSectionPtr attrInfoPtr, keyInfoPtr;
  Uint32 aiLen = 0;
  Uint32 keyLen = 0;

  ndbassert(numSections != 0);
  {
    /* Long request, get Attr + Key len from section sizes */
    ndbrequire(handle.getSection(attrInfoPtr, ScanFragReq::AttrInfoSectionNum));
    aiLen = attrInfoPtr.sz;

    if (numSections == 2) {
      ndbrequire(handle.getSection(keyInfoPtr, ScanFragReq::KeyInfoSectionNum));
      keyLen = keyInfoPtr.sz;
    } else {
      keyInfoPtr.setNull();
    }
  }
  const Uint32 senderBlockRef = signal->senderBlockRef();
  const Uint32 transid1 = scanFragReq->transId1;
  const Uint32 transid2 = scanFragReq->transId2;
  const Uint32 senderData = scanFragReq->senderData;
  const Uint32 senderBlockNo = refToMain(senderBlockRef);
  TcConnectionrecPtr tcConnectptr;
  if (likely(ctcNumFree > ZNUM_RESERVED_UTIL_CONNECT_RECORDS &&
             !ERROR_INSERTED(5055) && !ERROR_INSERTED(5031)) ||
      (ScanFragReq::getLcpScanFlag(scanFragReq->requestInfo)) ||
      (senderBlockNo == getBACKUP()) ||
      (senderBlockNo == DBUTIL &&
       ctcNumFree > ZNUM_RESERVED_TC_CONNECT_RECORDS)) {
    /**
     * We always keep 3 operation records, one for LCP scans and one for
     * Node recovery support (to handle COPY_FRAGREQ when we're aiding a
     * node to startup by synchronizing our data with the starting nodes
     * recovered data and finally one for backup scans.
     *
     * We also provide 100 records not available to ordinary transactions
     * but available to DBUTIL operations. But LCP and Backup operations
     * still have preference over DBUTIL operations.
     */
    seizeTcrec(tcConnectptr);
    jamEntry();
  } else {
    if (unlikely(ERROR_INSERTED_CLEAR(5055) || (!seize_op_rec(tcConnectptr)))) {
      jam();
      /* --------------------------------------------------------------------
       *      NO FREE TC RECORD AVAILABLE, THUS WE CANNOT HANDLE THE REQUEST.
       * -------------------------------------------------------------------- */
      tcConnectptr.i = RNIL;
      errorCode = ZNO_TC_CONNECT_ERROR;
      releaseSections(handle);
      send_scan_fragref(signal, transid1, transid2, senderData, senderBlockRef,
                        ZNO_TC_CONNECT_ERROR);
      return;
    }
    jamEntry();
  }  // if
  jamLineDebug(Uint16(tcConnectptr.i));
  regTcPtr = tcConnectptr.p;
  const Uint32 savePointId = scanFragReq->savePointId;
  regTcPtr->clientConnectrec = senderData;
  regTcPtr->clientBlockref = senderBlockRef;
  regTcPtr->savePointId = savePointId;

  if (qt_likely(m_is_query_block)) {
    /**
     * We are a query thread. Thus this query is performing a range
     * scan with READ COMMITTED. The actual meta data object we are
     * using is part of a specific LDM instance. We get the instance
     * number and use this to setup the block variables of the query
     * thread. This setup is only valid within this real-time break.
     * The next real-time break must also setup this structures.
     */
    const Uint32 fragId = (scanFragReq->fragmentNoKeyLen & 0xFFFF);
    Uint32 instanceNo = getInstanceNoCanFail(tabptr.i, fragId);
    if (unlikely(instanceNo == RNIL)) {
      errorCode = ZINVALID_SCHEMA_VERSION;
      releaseSections(handle);
      goto error_handler;
    }
    regTcPtr->ldmInstance = instanceNo;
    setup_query_thread_for_scan_access(instanceNo);
  }
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  if (unlikely(tabptr.p->tableStatus != Tablerec::TABLE_DEFINED &&
               tabptr.p->tableStatus != Tablerec::TABLE_READ_ONLY)) {
    jam();
    errorCode = get_table_state_error(tabptr);
    releaseSections(handle);
    goto error_handler;
  }

  if (unlikely(table_version_major(scanFragReq->schemaVersion) !=
               table_version_major(tabptr.p->schemaVersion))) {
    errorCode = ZINVALID_SCHEMA_VERSION;
    releaseSections(handle);
    goto error_handler;
  }

  {
    const Uint32 fragId = (scanFragReq->fragmentNoKeyLen & 0xFFFF);
    const Uint32 reqinfo = scanFragReq->requestInfo;
    const Uint32 max_rows = scanFragReq->batch_size_rows;
    const Uint32 scanLockMode = ScanFragReq::getLockMode(reqinfo);
    const Uint8 keyinfo = ScanFragReq::getKeyinfoFlag(reqinfo);
    const Uint8 rangeScan = ScanFragReq::getRangeScanFlag(reqinfo);
    if (unlikely(!getFragmentrec(fragId))) {
      jam();
      errorCode = 1231;
      releaseSections(handle);
      goto error_handler;
    }  // if

    /**
     * A write always has to get keyinfo
     */
    ndbrequire(scanLockMode == 0 || keyinfo);

    ndbrequire(max_rows > 0 && max_rows <= MAX_PARALLEL_OP_PER_SCAN);

    // Verify scan type vs table type (both sides are boolean)
    if (unlikely(rangeScan !=
                 DictTabInfo::isOrderedIndex(fragptr.p->tableType))) {
      jam();
      errorCode = 1232;
      releaseSections(handle);
      goto error_handler;
    }  // if

    if (ScanFragReq::getLcpScanFlag(reqinfo) || senderBlockNo == getBACKUP()) {
      /* LCP and Backup scans come here */
      jam();
      ndbrequire(m_reserved_scans.first(scanptr));
      m_reserved_scans.remove(scanptr);
    } else if (unlikely(!c_scanRecordPool.seize(scanptr))) {
      jam();
      errorCode = ScanFragRef::ZNO_FREE_SCANREC_ERROR;
      releaseSections(handle);
      goto error_handler;
    }
    initScanTc(scanFragReq, transid1, transid2, fragId, ZNIL, senderBlockRef,
               tcConnectptr);
    regTcPtr->opExec = (1 ^ ScanFragReq::getNotInterpretedFlag(reqinfo));
    {
      const Uint32 applRef = scanFragReq->resultRef;

      regTcPtr->primKeyLen = keyLen;
      regTcPtr->applRef = applRef;
    }

    {
      jamDebug();
      regTcPtr->attrInfoIVal = attrInfoPtr.i;
      if (keyLen > 0) {
        regTcPtr->keyInfoIVal = keyInfoPtr.i;
      } else if (unlikely(!keyInfoPtr.isNull())) {
        jam();
        // Release empty and unneeded key info section.
        releaseSection(keyInfoPtr.i);
      }
      /* Scan state machine is now responsible for freeing
       * these sections, usually via releaseOprec()
       */
      handle.clear();

      /*
         Update per fragment statistics. 'attrInfoPtr' is not defined for
         short-signal scans, so we ignore these, since they only happen
         during online upgrades.
       */
      if (!ScanFragReq::getLcpScanFlag(reqinfo) && !m_is_query_block) {
        jamDebug();
        Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
        useStat.m_scanFragReqCount++;
        useStat.m_scanBoundWords += keyLen;
        if (!ScanFragReq::getNotInterpretedFlag(reqinfo)) {
          /* Backup scans do not use interpreted mode. */
          useStat.m_scanProgramWords += getProgramWordCount(attrInfoPtr);
        }
      }
    }

    if (ScanFragReq::getCorrFactorFlag(reqinfo)) {
      /**
       * Correlation factor for SPJ
       */
      const Uint32 corrFactorHi = scanFragReq->variableData[1];
      regTcPtr->m_corrFactorLo = scanFragReq->variableData[0];
      regTcPtr->m_corrFactorHi = corrFactorHi;
    }
    jamLineDebug((Uint16)aiLen);
    errorCode = initScanrec(scanFragReq, aiLen, tcConnectptr);
    if (unlikely(errorCode != ZOK)) {
      jam();
      goto error_handler2;
    }  // if

    /* Check that no equal element already exists */
    ndbrequire(findTransaction(regTcPtr->transid[0], regTcPtr->transid[1],
                               regTcPtr->tcOprec, senderBlockRef, false, false,
                               tcConnectptr) == ZNOT_FOUND);
    TcConnectionrecPtr nextHashptr;
    const Uint32 hashIndex = getHashIndex(regTcPtr);
    nextHashptr.i = ctransidHash[hashIndex];
    ctransidHash[hashIndex] = tcConnectptr.i;
    regTcPtr->prevHashRec = RNIL;
    regTcPtr->nextHashRec = nextHashptr.i;
    regTcPtr->hashIndex = hashIndex;
    if (nextHashptr.i != RNIL) {
      /* ---------------------------------------------------------------------
       *   ENSURE THAT THE NEXT RECORD HAS SET PREVIOUS TO OUR RECORD
       *   IF IT EXISTS
       * ---------------------------------------------------------------------
       */
      ndbrequire(tcConnect_pool.getValidPtr(nextHashptr));
      ndbassert(nextHashptr.p->prevHashRec == RNIL);
      nextHashptr.p->prevHashRec = tcConnectptr.i;
    }  // if
    continueAfterReceivingAllAiLab(signal, tcConnectptr);
    release_frag_access(prim_tab_fragptr.p);
    return;
  }

error_handler2:
  // no scan number allocated
  scanptr.p->scan_lastSeen = __LINE__;
  if (scanptr.p->m_reserved == 0) {
    jam();
    c_scanRecordPool.release(scanptr);
    checkPoolShrinkNeed(DBLQH_SCAN_RECORD_TRANSIENT_POOL_INDEX,
                        c_scanRecordPool);
  } else {
    jam();
    init_release_scanrec(scanptr.p);
    m_reserved_scans.addFirst(scanptr);
  }
error_handler:
  regTcPtr->abortState = TcConnectionrec::ABORT_ACTIVE;
  regTcPtr->tcScanRec = RNIL;
  releaseOprec(signal, tcConnectptr);
  releaseTcrec(signal, tcConnectptr);
  send_scan_fragref(signal, transid1, transid2, senderData, senderBlockRef,
                    errorCode);
}  // Dblqh::execSCAN_FRAGREQ()

void Dblqh::continueAfterReceivingAllAiLab(
    Signal *signal, const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  ScanRecord *const scanPtr = scanptr.p;
  regTcPtr->transactionState = TcConnectionrec::SCAN_STATE_USED;

  if (unlikely(scanPtr->scanState == ScanRecord::IN_QUEUE)) {
    jam();
    scanPtr->scan_lastSeen = __LINE__;
    return;
  }

  scanPtr->scanState = ScanRecord::WAIT_ACC_SCAN;
  AccScanReq *req = (AccScanReq *)&signal->theData[0];

  Uint32 requestInfo = 0;

  AccScanReq::setLockMode(requestInfo, scanPtr->scanLockMode);
  AccScanReq::setReadCommittedFlag(requestInfo, scanPtr->readCommitted);
  AccScanReq::setDescendingFlag(requestInfo, scanPtr->descending);
  AccScanReq::setStatScanFlag(requestInfo, scanPtr->statScan);

  if (refToMain(regTcPtr->clientBlockref) == getBACKUP()) {
    if (scanPtr->lcpScan) {
      AccScanReq::setNoDiskScanFlag(requestInfo, 1);
      AccScanReq::setLcpScanFlag(requestInfo, 1);
    } else {
      /* If backup scan disktables in disk order */
#ifdef BUG25353234_PROPERLY_FIXED
      AccScanReq::setNoDiskScanFlag(requestInfo, !regTcPtr->m_disk_table);
#else
      AccScanReq::setNoDiskScanFlag(requestInfo, 1);
#endif
      AccScanReq::setLcpScanFlag(requestInfo, 0);
    }
  } else {
#ifdef BUG_27776_FIXED
    AccScanReq::setNoDiskScanFlag(requestInfo, !regTcPtr->m_disk_table);
#else
    AccScanReq::setNoDiskScanFlag(requestInfo, 1);
#endif
    AccScanReq::setLcpScanFlag(requestInfo, 0);
  }

  SimulatedBlock *block = scanPtr->scanBlock;
  req->requestInfo = requestInfo;

  acquire_frag_scan_access(prim_tab_fragptr.p, regTcPtr);

  const Uint32 senderData = scanptr.i;
  const Uint32 senderRef = cownref;
  const Uint32 tableId = regTcPtr->tableref;
  const Uint32 fragmentNo = regTcPtr->fragmentid;
  const Uint32 transId1 = regTcPtr->transid[0];
  const Uint32 transId2 = regTcPtr->transid[1];
  const Uint32 savePointId = regTcPtr->savePointId;
  ExecFunction f = block->getExecuteFunction(GSN_ACC_SCANREQ);

  req->senderData = senderData;
  req->senderRef = senderRef;
  req->tableId = tableId;
  req->fragmentNo = fragmentNo;
  req->transId1 = transId1;
  req->transId2 = transId2;
  req->savePointId = savePointId;

  block->EXECUTE_DIRECT_FN(f, signal);
  if (likely(signal->theData[8] == 0)) {
    /* ACC_SCANCONF */
    jamEntryDebug();
    accScanConfScanLab(signal, tcConnectptr);
  } else {
    /* ACC_SCANREF */
    jamEntry();
    AccScanRef *ref = (AccScanRef *)&signal->theData[0];
    ref->errorCode = signal->theData[8];
    execACC_SCANREF(signal, tcConnectptr);
  }
}  // Dblqh::continueAfterReceivingAllAiLab()

void Dblqh::accScanConfScanLab(Signal *signal,
                               const TcConnectionrecPtr tcConnectptr) {
  AccScanConf *const accScanConf = (AccScanConf *)&signal->theData[0];
  ScanRecord *const scanPtr = scanptr.p;

  /* -----------------------------------------------------------------------
   *       PRECONDITION: SCAN_STATE = WAIT_ACC_SCAN
   * ----------------------------------------------------------------------- */
  if (accScanConf->flag == AccScanConf::ZEMPTY_FRAGMENT) {
    jamDebug();
    /* ---------------------------------------------------------------------
     *       THE FRAGMENT WAS EMPTY.
     *       REPORT SUCCESSFUL COPYING.
     * --------------------------------------------------------------------- */
    /*
     * MRR scan + delete can hit this when the fragment was not
     * initially empty, but has become empty after previous range.
     */
    if (scanPtr->scanStoredProcId != RNIL) {
      jam();
      scanPtr->scanCompletedStatus = ZTRUE;
      accScanCloseConfLab(signal, tcConnectptr);
      return;
    }
    tupScanCloseConfLab(signal, tcConnectptr);
    return;
  }  // if

  check_send_scan_hb_rep(signal, scanPtr, tcConnectptr.p);

  scanPtr->scanAccPtr = accScanConf->accPtr;
  if (scanPtr->rangeScan) {
    TuxBoundInfo *req = (TuxBoundInfo *)signal->getDataPtrSend();
    req->errorCode = RNIL;
    req->tuxScanPtrI = scanPtr->scanAccPtr;
    Uint32 len = req->boundAiLength = copyNextRange(req->data, tcConnectptr.p);
    signal->setLength(TuxBoundInfo::SignalLength + len);
    c_tux->execTUX_BOUND_INFO(signal);
    jamEntryDebug();
    if (unlikely(req->errorCode != 0)) {
      jam();
      /*
       * Cannot use STORED_PROCREF to abort since even the REF
       * returns a stored proc id.  So record error and continue.
       * The scan is already Invalid in TUX and returns empty set.
       */
      tcConnectptr.p->errorCode = req->errorCode;
    }
  }

  if (scanPtr->scanStoredProcId == RNIL) {
    TcConnectionrec *const regTcPtr = tcConnectptr.p;
    jamDebug();
    /* Send AttrInfo to TUP to store as 'stored procedure'
     * and get storedProcId back for future reference
     */
    const Uint32 sig0 = regTcPtr->tupConnectrec;
    const Uint32 sig1 = regTcPtr->tableref;
    const Uint32 sig2 = scanPtr->scanSchemaVersion;
    const Uint32 sig5 = scanPtr->scanApiBlockref;
    const Uint32 sig6 = regTcPtr->attrInfoIVal;

    signal->theData[0] = sig0;
    signal->theData[1] = sig1;
    signal->theData[2] = sig2;
    signal->theData[3] = ZSTORED_PROC_SCAN;
    // theData[4] is not used
    signal->theData[5] = sig5;
    signal->theData[6] = sig6;

    /* Pass ATTRINFO as long section, we don't need
     * it after this
     */
    regTcPtr->attrInfoIVal = RNIL;

    c_tup->execSTORED_PROCREQ(signal);
    if (likely(signal->theData[0] == 0)) {
      /* STORED_PROCCONF */
      jamEntryDebug();
      Uint32 storedProcId = signal->theData[1];
      scanPtr->scanStoredProcId = storedProcId;
      c_tup->copyAttrinfo(storedProcId, bool(regTcPtr->opExec));
      storedProcConfScanLab(signal, tcConnectptr);
      return;
    } else {
      /* STORED_PROCREF */
      jamEntry();
      Uint32 storedProcId = signal->theData[2];
      scanPtr->scanCompletedStatus = ZTRUE;
      scanPtr->scanStoredProcId = storedProcId;
      tcConnectptr.p->errorCode = signal->theData[1];
      closeScanLab(signal, tcConnectptr.p);
      return;
    }
  } else {
    /* TUP already has the Stored procedure, continue */
    jam();
    TcConnectionrec *const regTcPtr = tcConnectptr.p;

    if (regTcPtr->opExec)  // has interpreter code, possibly using param
    {
      // Advance to next parameter in the attrInfo
      c_tup->nextAttrInfoParam(scanPtr->scanStoredProcId);
    }
    c_tup->copyAttrinfo(scanPtr->scanStoredProcId, bool(regTcPtr->opExec));
    storedProcConfScanLab(signal, tcConnectptr);
    return;
  }
}  // Dblqh::accScanConfScanLab()

/* -------------------------------------------------------------------------
 *       ENTER STORED_PROCCONF WITH
 *         0 success == CONF, 1 failure == REF
 *         STORED_PROC_ID
 * ------------------------------------------------------------------------- */
void Dblqh::storedProcConfScanLab(Signal *signal,
                                  const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;
  if (unlikely(scanPtr->scanCompletedStatus == ZTRUE)) {
    jam();
    // STOP THE SCAN PROCESS IF THIS HAS BEEN REQUESTED.
    closeScanLab(signal, tcConnectptr.p);
    return;
  }  // if
  if (unlikely(scanPtr->check_scan_batch_completed())) {
    jam();
    scanPtr->m_last_row = 0;
    scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
    scanPtr->scan_lastSeen = __LINE__;
    sendScanFragConf(signal, ZFALSE, tcConnectptr.p);
    return;
  }
  Fragrecord::FragStatus fragstatus = fragptr.p->fragStatus;
  if (likely(is_scan_ok(scanPtr, fragstatus))) {
    /**
     * At this point we have set up everything in DBLQH, in the
     * scan block (DBACC for full table scans, DBTUP for LCP scans
     * and node recovery scans and full table scans ordered by an
     * NDB API application and DBTUX for ordered index scans
     * (range scans).
     *
     * We will now rely on all operation pointers, scan pointers
     * fragment pointers and table pointers to be setup on the
     * block object for quick access. If we get a real-time break
     * we have to ensure that those are setup properly again.
     *
     * Real-time breaks have occurred when we receive the signal
     * SCAN_NEXTREQ, we could also get a real-time break from
     * the scan block by sending NEXT_SCANCONF as an asynchronous
     * signal. It can be ordered by the scan block by sending a
     * direct signal CHECK_LCP_STOP to DBLQH, this will lead to
     * a CONTINUEB with delay of 1 millisecond.
     *
     * Finally it can happen by sending a CONTINUEB signal from
     * send_next_NEXT_SCANREQ to ourselves. This signal simply
     * means we are in the middle of processing a local scan
     * operation, but we need to take a real-time break to
     * provide an opportunity for other operations to execute
     * as well.
     *
     * This is not always the first call to send_next_NEXT_SCANREQ.
     * The reason is that we can execute multiple ranges as part
     * of one scan. See scanNextLoopLab for more description of
     * stack unwinding when sending NEXT_SCANREQ.
     */
    const Uint32 sig0 = scanPtr->scanAccPtr;
    SimulatedBlock *block = scanPtr->scanBlock;
    ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;
    Uint32 in_send_next_scan = m_in_send_next_scan;
    signal->theData[1] = RNIL;
    signal->theData[2] = NextScanReq::ZSCAN_NEXT;
    signal->theData[0] = sig0;
    scanPtr->scanState = ScanRecord::WAIT_NEXT_SCAN;
    scanPtr->scan_lastSeen = __LINE__;
    if (likely(in_send_next_scan == 0)) {
      send_next_NEXT_SCANREQ(signal, block, f, scanPtr,
                             tcConnectptr.p->clientConnectrec);
      return;
    }
    ndbassert(in_send_next_scan == 1);
    m_in_send_next_scan = 2;
    return;
  } else {
    jamLine(fragptr.p->fragStatus);
    g_eventLogger->info("fragptr.p->fragStatus: %u", fragptr.p->fragStatus);
    // wl4391_todo SR 2-node CRASH_RECOVERING from BACKUP
    ndbabort();
  }
}  // Dblqh::storedProcConfScanLab()

void Dblqh::abort_scan(Signal *signal, Uint32 scan_ptr_i, Uint32 errcode,
                       const TcConnectionrecPtr tcConnectptr) {
  jam();
  scanptr.i = scan_ptr_i;
  ndbrequire(c_scanRecordPool.getValidPtr(scanptr));

  tcConnectptr.p->errorCode = errcode;
  tupScanCloseConfLab(signal, tcConnectptr);
  return;
}

/*---------------------------------------------------------------------*/
/* Send this 'I am alive' signal to TC when it is received from ACC    */
/* We include the scanPtr.i that comes from ACC in signalData[1], this */
/* tells TC which fragment record to check for a timeout.              */
/*---------------------------------------------------------------------*/
void Dblqh::check_send_scan_hb_rep(Signal *signal, ScanRecord *scanPtrP,
                                   TcConnectionrec *tcPtrP) {
  switch (scanPtrP->scanType) {
    case ScanRecord::SCAN:
      break;
    case ScanRecord::COPY:
      return;
#ifdef NDEBUG
    case ScanRecord::ST_IDLE:
    default:
      return;
#else
    case ScanRecord::ST_IDLE:
      ndbabort();
#endif
  }

  const Uint32 now = cLqhTimeOutCount;  // measure in 10ms
  const Uint32 last =
      scanPtrP->scanTcWaiting;  // last time we reported to TC (10ms)
  const Uint32 timeout = cTransactionDeadlockDetectionTimeout;  // (ms)
  const Uint32 limit = timeout / 16;
  const Uint32 time_waiting = (now - last) * 10;  // Convert to ms

  ndbassert(limit > 0);
  /**
   * We need to ensure we send heartbeats before TC decides to timeout based
   * on a deadlock. We use 1/16th of the timeout period for this. The other
   * thing to consider is wraparound, but this will work fine since then we
   * we will get an immediate very high number and immediate timeout.
   *
   * When the sender doesn't know which block instance handled the scan we
   * will send an early heartbeat report already after 10 ms to ensure that
   * we can close the scan quickly if a problem occurs.
   */

  if (time_waiting > limit ||
      (time_waiting > 0 && scanPtrP->m_send_early_hbrep)) {
    jam();

    scanPtrP->scanTcWaiting = Uint32(now);
    scanPtrP->m_send_early_hbrep = 0;
    if (tcPtrP->tcTimer != 0) {
      tcPtrP->tcTimer = Uint32(now);
    }

    Uint32 save[4];
    save[0] = signal->theData[0];
    save[1] = signal->theData[1];
    save[2] = signal->theData[2];
    save[3] = signal->getLength();

    signal->theData[0] = tcPtrP->clientConnectrec;
    signal->theData[1] = tcPtrP->transid[0];
    signal->theData[2] = tcPtrP->transid[1];
    sendSignal(tcPtrP->clientBlockref, GSN_SCAN_HBREP, signal, 3, JBB);

    signal->theData[0] = save[0];
    signal->theData[1] = save[1];
    signal->theData[2] = save[2];
    signal->setLength(save[3]);
  }
}

Uint32 Dblqh::copyNextRange(Uint32 *dst, TcConnectionrec *tcPtrP) {
  /**
   * Copy the bound info for the next range from the KeyInfo
   * to *dst
   * There may be zero or more bounds
   * A SectionReader is used to read bound information, its
   * position is saved between calls
   * This method also extracts range numbers from the
   * KeyInfo
   */
  Uint32 totalLen = tcPtrP->primKeyLen;
  if (totalLen == 0) {
    return 0;
  }

  Uint32 *save = dst;
  do {
    ndbassert(tcPtrP->keyInfoIVal != RNIL);
    SectionReader keyInfoReader(tcPtrP->keyInfoIVal, g_sectionSegmentPool);

    if (tcPtrP->m_flags & TcConnectionrec::OP_SCANKEYINFOPOSSAVED) {
      /* Second or higher range in an MRR scan
       * Restore SectionReader to the last position it was in
       */
      bool ok = keyInfoReader.setPos(tcPtrP->scanKeyInfoPos);
      ndbrequire(ok);
    }

    /* Get first word of next range and extract range
     * length, number from it.
     * For non MRR, these will be zero.
     */
    Uint32 firstWord;
    ndbrequire(keyInfoReader.getWord(&firstWord));
    const Uint32 rangeLen = (firstWord >> 16) ? (firstWord >> 16) : totalLen;
    Uint32 range_no = (firstWord & 0xFFF0) >> 4;
    tcPtrP->m_scan_curr_range_no = range_no;
    tcPtrP->m_corrFactorLo &= 0x0000FFFF;
    tcPtrP->m_corrFactorLo |= (range_no << 16);
    firstWord &= 0xF;  // Remove length+range num from first word

    /* Write range info to dst */
    *(dst++) = firstWord;
    bool ok = keyInfoReader.getWords(dst, rangeLen - 1);
    ndbassert(ok);
    if (unlikely(!ok)) break;

    if (ERROR_INSERTED(5074)) break;

    tcPtrP->primKeyLen -= rangeLen;

    if (rangeLen == totalLen) {
      /* All range information has been copied, free the section */
      releaseSection(tcPtrP->keyInfoIVal);
      tcPtrP->keyInfoIVal = RNIL;
    } else {
      /* Save position of SectionReader for next range (if any) */
      tcPtrP->scanKeyInfoPos = keyInfoReader.getPos();
      tcPtrP->m_flags |= TcConnectionrec::OP_SCANKEYINFOPOSSAVED;
    }

    return rangeLen;
  } while (0);

  /**
   * We enter here if there was some error in the keyinfo
   *   this has (once) been seen in customer lab,
   *   never at in the wild, and never in internal lab.
   *   root-cause unknown, maybe ndbapi application bug
   *
   * Crash in debug, or ERROR_INSERT (unless 5074)
   * else
   *   generate an incorrect bound...that will make TUX abort the scan
   */
#ifdef ERROR_INSERT
  ndbrequire(ERROR_INSERTED_CLEAR(5074));
#else
  ndbassert(false);
#endif

  *save = TuxBoundInfo::InvalidBound;
  return 1;
}

/* -------------------------------------------------------------------------
 * When executing a scan we must come up to the surface at times to wait
 * for a resource to become available
 * ------------------------------------------------------------------------- */
void Dblqh::execCHECK_LCP_STOP(Signal *signal) {
  const CheckLcpStop *cls = (const CheckLcpStop *)signal->theData;
  const Uint32 scanPtrI = cls->scanPtrI;
  jamEntryDebug();
  switch (cls->scanState) {
    case CheckLcpStop::ZSCAN_RUNNABLE: {
      jamDebug();
      return;
    }
    case CheckLcpStop::ZSCAN_RESOURCE_WAIT:
    case CheckLcpStop::ZSCAN_RESOURCE_WAIT_STOPPABLE: {
      jam();
      /**
       * Scan waiting for a resource, we will use a
       * delayed CONTINUEB to continue it later
       *
       * Not appropriate to send SCAN_HBREP here since we are not making
       * progress and thus sending SCAN_HBREP would nullify TC from being
       * able to discover deadlocks.
       */

      /* Tracking */
      ScanRecordPtr loc_scanptr;
      loc_scanptr.i = scanPtrI;
      ndbrequire(c_scanRecordPool.getUncheckedPtrRW(loc_scanptr));
      loc_scanptr.p->scan_lastSeen = __LINE__;
      loc_scanptr.p->scan_check_lcp_stop++;
      ndbrequire(Magic::check_ptr(loc_scanptr.p));
      if (loc_scanptr.p->scan_check_lcp_stop >= 5 &&
          cls->scanState == CheckLcpStop::ZSCAN_RESOURCE_WAIT_STOPPABLE) {
        jam();
        /**
         * STOPPABLE resource wait is e.g. Out of operation records, here
         * we only wait for a short time to slow things down a bit, but
         * we don't wait indefinitely like we do for e.g. locked rows.
         * Lock timeouts are handled by DBTC, but we can conclude on our
         * own to abort if we are out of operation records.
         *
         * Report error as out of operation records, will be reported as
         * out of operation records in local data manager to application
         * with the advice to raise TransactionMemory.
         */
        loc_scanptr.p->scanCompletedStatus = ZTRUE;
        Ptr<TcConnectionrec> regTcPtr;
        regTcPtr.i = loc_scanptr.p->scanTcrec;
        ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));
        regTcPtr.p->errorCode = ZNO_TC_CONNECT_ERROR;
      }
      if (loc_scanptr.p->scanCompletedStatus == ZTRUE) {
        jam();
        /* Tell caller to abort scan */
        signal->theData[0] = CheckLcpStop::ZABORT_SCAN;
        return;
      }

      signal->theData[0] = ZCHECK_LCP_STOP_BLOCKED;
      signal->theData[1] = scanPtrI;
      sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 1, 2);

      /* Tell caller to take a break */
      signal->theData[0] = CheckLcpStop::ZTAKE_A_BREAK;
      return;
    }
    case CheckLcpStop::ZSCAN_RUNNABLE_YIELD: {
      jam();
      /**
       * Scan voluntarily yielding cpu
       * We will use an immediate CONTINUEB to continue
       * it ASAP.
       */
      {
        /* Consider sending SCAN_HBREP if appropriate */
        ScanRecordPtr scan;
        TcConnectionrecPtr tc;
        scan.i = scanPtrI;
        ndbrequire(c_scanRecordPool.getValidPtr(scan));
        tc.i = scan.p->scanTcrec;
        ndbrequire(tcConnect_pool.getValidPtr(tc));
        check_send_scan_hb_rep(signal, scan.p, tc.p);

        /* Tracking */
        scan.p->scan_lastSeen = __LINE__;
        scan.p->scan_check_lcp_stop++;
      }

      signal->theData[0] = ZCHECK_LCP_STOP_BLOCKED;
      signal->theData[1] = scanPtrI;
      sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);

      /* Tell caller to take a break */
      signal->theData[0] = CheckLcpStop::ZTAKE_A_BREAK;
      return;
    }
    default:
      jam();
      jamLine(cls->scanState);
      ndbabort();
  }
}  // Dblqh::execCHECK_LCP_STOP()

/* -------------------------------------------------------------------------
 *       ENTER NEXT_SCANCONF
 * -------------------------------------------------------------------------
 *       PRECONDITION: SCAN_STATE = WAIT_NEXT_SCAN
 * ------------------------------------------------------------------------- */
void Dblqh::nextScanConfScanLab(Signal *signal, ScanRecord *const scanPtr,
                                Uint32 fragId, Uint32 accOpPtr,
                                const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (likely(fragId != RNIL && accOpPtr != RNIL)) {
    jamDebug();
    check_send_scan_hb_rep(signal, scanPtr, tcConnectptr.p);
    scanPtr->scan_check_lcp_stop = 0;
    set_acc_ptr_in_scan_record(scanPtr, scanPtr->m_curr_batch_size_rows,
                               accOpPtr);

    if (unlikely(signal->getLength() == NextScanConf::SignalLengthNoKeyInfo)) {
      /**
       * We have found a deleted row id as part of a LCP scan.
       * We don't use TRANSID_AI in this case to avoid having to go through
       * TUP in this case. We will however call scanTupkeyConfLab to fake
       * that we return successfully from TUPKEYREQ. This is to simplify
       * the code and use the normal patterns. This means that the record
       * will be part of scan batch size which is necessary to ensure that
       * we don't risk running out of buffer space in the BACKUP block while
       * recording deleted row ids.
       *
       * We return with accOpPtr set to RNIL in this case to avoid
       * complications when releasing locks.
       */
      ndbassert(!m_is_query_block);
      NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
      Uint32 gci = nextScanConf->gci;
      Uint32 readLength;
      /**
       * Coming here only happens for LCP scans and these always returns
       * row ids, both in TRANSID_AIs and in return NEXT_SCANCONF.
       */
      if (scanPtr->m_row_id.m_page_idx == ZNIL) {
        jam();
        /* gci transports record_size in this case */
        c_backup->record_deleted_pageid(scanPtr->m_row_id.m_page_no, gci);
        readLength = 2;
      } else {
        jam();
        c_backup->record_deleted_rowid(scanPtr->m_row_id.m_page_no,
                                       scanPtr->m_row_id.m_page_idx, gci);
        readLength = 3;
      }
      ndbrequire(scanPtr->m_curr_batch_size_rows < MAX_PARALLEL_OP_PER_SCAN);
      scanPtr->m_exec_direct_batch_size_words += readLength;
      scanPtr->m_curr_batch_size_bytes += readLength * sizeof(Uint32);
      scanPtr->m_curr_batch_size_rows++;
      scanPtr->m_last_row = false;
      scanPtr->scanFlag = NextScanReq::ZSCAN_NEXT;

      if (!scanPtr->check_scan_batch_completed()) {
        jam();
        scanNextLoopLab(signal, regTcPtr->clientConnectrec, RNIL, scanPtr,
                        fragptr.p);
        return;
      } else {
        jam();
        scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
        scanPtr->scan_lastSeen = __LINE__;
        sendScanFragConf(signal, ZFALSE, regTcPtr);
        return;
      }
    }

    /* ----------------------------------------------------------------------
     *       STOP THE SCAN PROCESS IF THIS HAS BEEN REQUESTED.
     * ---------------------------------------------------------------------- */
    if (unlikely(scanPtr->scanCompletedStatus == ZTRUE)) {
      if ((scanPtr->scanLockHold == ZTRUE) &&
          (scanPtr->m_curr_batch_size_rows > 0)) {
        jam();
        scanPtr->scanReleaseCounter = 1;
        scanReleaseLocksLab(signal, regTcPtr);
        return;
      }  // if
      jam();
      closeScanLab(signal, regTcPtr);
      return;
    }  // if

    Fragrecord *fragPtrP = prim_tab_fragptr.p;
    bool disk_table = regTcPtr->m_disk_table;
    regTcPtr->transactionState = TcConnectionrec::SCAN_TUPKEY;
    if (likely(!disk_table)) {
      jamDebug();
      next_scanconf_tupkeyreq(signal, scanPtr, regTcPtr, fragPtrP, RNIL);
      return;
    } else {
      jamDebug();
      scanPtr->scan_lastSeen = __LINE__;
      next_scanconf_load_diskpage(signal, scanPtr, tcConnectptr, fragPtrP);
      return;
    }
  } else {
    // If accOperationPtr == RNIL no record was returned by ACC
    /* ---------------------------------------------------------------------
     *       THERE ARE NO MORE TUPLES TO FETCH. IF WE HAVE ANY
     *       OPERATIONS STILL NEEDING A LOCK WE REPORT TO THE
     *       APPLICATION AND CLOSE THE SCAN WHEN THE NEXT SCAN
     *       REQUEST IS RECEIVED. IF WE DO NOT HAVE ANY NEED FOR
     *       LOCKS WE CAN CLOSE THE SCAN IMMEDIATELY.
     * --------------------------------------------------------------------- */
    /*************************************************************
     *       STOP THE SCAN PROCESS IF THIS HAS BEEN REQUESTED.
     ************************************************************ */

    if (fragId == RNIL && !scanPtr->scanLockHold) {
      jamDebug();
      closeScanLab(signal, tcConnectptr.p);
      return;
    }

    if (scanPtr->scanCompletedStatus == ZTRUE) {
      if ((scanPtr->scanLockHold == ZTRUE) &&
          (scanPtr->m_curr_batch_size_rows > 0)) {
        jam();
        scanPtr->scanReleaseCounter = 1;
        scanPtr->scan_check_lcp_stop = 0;
        scanReleaseLocksLab(signal, tcConnectptr.p);
        return;
      }  // if
      jam();
      closeScanLab(signal, tcConnectptr.p);
      return;
    }  // if

    if (scanPtr->m_curr_batch_size_rows > 0) {
      if (fragId == RNIL && regTcPtr->primKeyLen == 0) {
        jam();
        scanPtr->scanCompletedStatus = ZTRUE;
      }
      jam();
      scanPtr->scan_check_lcp_stop = 0;
      scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
      scanPtr->scan_lastSeen = __LINE__;
      sendScanFragConf(signal, ZFALSE, tcConnectptr.p);
      return;
    }  // if

    if (fragId == RNIL) {
      jam();
      closeScanLab(signal, tcConnectptr.p);
      return;
    } else {
      SimulatedBlock *block = scanPtr->scanBlock;
      Uint32 sig0 = scanPtr->scanAccPtr;
      jamDebug();
      signal->theData[0] = sig0;
      signal->theData[1] = AccCheckScan::ZCHECK_LCP_STOP;
      scanPtr->scan_lastSeen = __LINE__;
      ExecFunction f = block->getExecuteFunction(GSN_ACC_CHECK_SCAN);
      block->EXECUTE_DIRECT_FN(f, signal);
      return;
    }
  }
}  // Dblqh::nextScanConfScanLab()

void Dblqh::next_scanconf_tupkeyreq(Signal *signal, ScanRecord *scanPtr,
                                    TcConnectionrec *regTcPtr,
                                    Fragrecord *fragPtrP, Uint32 disk_page) {
  jamDebug();
  /* No AttrInfo sent to TUP, it uses a stored procedure */
  TupKeyReq *const tupKeyReq = (TupKeyReq *)signal->getDataPtrSend();
  {
    /**
     * The row id here depends on if we are scanning in TUX
     * or in TUP or ACC. TUX returns physical row ids and
     * TUP and ACC returns logical row ids. This is handled
     * by TUP.
     */
    const Uint32 keyRef1 = scanPtr->m_row_id.m_page_no;
    const Uint32 keyRef2 = scanPtr->m_row_id.m_page_idx;
    tupKeyReq->disk_page = disk_page;
    tupKeyReq->keyRef1 = keyRef1;
    tupKeyReq->keyRef2 = keyRef2;
  }
#ifdef VM_TRACE
  tupKeyReq->fragPtr = fragPtrP->tupFragptr;
#endif
  if (c_tup->execTUPKEYREQ(signal, regTcPtr, scanPtr)) {
    execTUPKEYCONF(signal);
    return;
  } else {
    execTUPKEYREF(signal);
    return;
  }
}

void Dblqh::next_scanconf_load_diskpage(Signal *signal,
                                        ScanRecord *const scanPtr,
                                        Ptr<TcConnectionrec> regTcPtr,
                                        Fragrecord *fragPtrP) {
  jam();

  int res;

  Uint32 disk_flag = (scanPtr->m_reserved) ? Page_cache_client::COPY_FRAG : 0;
  if ((res = c_tup->load_diskpage_scan(
           signal, regTcPtr.p->tupConnectrec, fragPtrP->tupFragptr,
           scanPtr->m_row_id.m_page_no, scanPtr->m_row_id.m_page_idx,
           scanPtr->rangeScan, disk_flag)) > 0) {
    next_scanconf_tupkeyreq(signal, scanPtr, regTcPtr.p, fragPtrP, res);
    return;
  } else if (unlikely(res != 0)) {
    jam();
    TupKeyRef *ref = (TupKeyRef *)signal->getDataPtr();
    ref->userRef = regTcPtr.i;
    if (res == -1) {
      jam();
      ref->errorCode = ~0;
    } else {
      jam();
      ref->errorCode = -res;
    }
    execTUPKEYREF(signal);
    return;
  }
  /**
   * We need to wait for the disk subsystem to fetch the page. This means that
   * we will suffer a real-time break and with TUX scans this means that we need
   * to link our scan record into the ordered index to ensure that we know where
   * we are currently scanning when we return from the real-time break.
   */
  if (scanPtr->scanBlock == c_tux) {
    jamDebug();
    c_tux->relinkScan(__LINE__);
  }
  /* WE ARE ENTERING A REAL-TIME BREAK FOR A SCAN HERE */
}

void Dblqh::next_scanconf_load_diskpage_callback(Signal *signal,
                                                 Uint32 callbackData,
                                                 Uint32 disk_page) {
  jamEntry();
  Ptr<TcConnectionrec> regTcPtr;
  regTcPtr.i = callbackData;
  ndbassert(!m_is_query_block);
  ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));
  /**
   * We have returned from a real-time break, we need to set up
   * the proper block pointers for scan execution.
   */
  setup_scan_pointers_from_tc_con(regTcPtr, __LINE__);

  ScanRecord *scanPtr = scanptr.p;
  Fragrecord *fragPtrP = prim_tab_fragptr.p;
  if (disk_page > 0) {
    if (scanPtr->rangeScan) {
      jam();
      c_tup->prepare_scan_tux_TUPKEYREQ(scanPtr->m_row_id.m_page_no,
                                        scanPtr->m_row_id.m_page_idx);
    } else {
      jam();
      c_tup->prepare_scanTUPKEYREQ(scanPtr->m_row_id.m_page_no,
                                   scanPtr->m_row_id.m_page_idx);
    }
    next_scanconf_tupkeyreq(signal, scanPtr, regTcPtr.p, fragPtrP, disk_page);
  } else {
    jam();
    TupKeyRef *ref = (TupKeyRef *)signal->getDataPtr();
    ref->userRef = callbackData;
    ref->errorCode = disk_page;
    execTUPKEYREF(signal);
  }
  release_frag_access(fragPtrP);
}

/* -------------------------------------------------------------------------
 *       STORE KEYINFO IN A LONG SECTION PRIOR TO SENDING
 * -------------------------------------------------------------------------
 *       PRECONDITION:   SCAN_STATE = WAIT_SCAN_KEYINFO
 * ------------------------------------------------------------------------- */
bool Dblqh::keyinfoLab(const Uint32 *src, Uint32 len,
                       const TcConnectionrecPtr tcConnectptr) {
  ndbassert(tcConnectptr.p->keyInfoIVal == RNIL);
  ndbassert(len > 0);

  if (ERROR_INSERTED(5052) || ERROR_INSERTED_CLEAR(5060)) return false;

  return (appendToSection(tcConnectptr.p->keyInfoIVal, src, len));
}  // Dblqh::keyinfoLab()

Uint32 Dblqh::readPrimaryKeys(Uint32 opPtrI, Uint32 *dst, bool xfrm_hash) {
  TcConnectionrecPtr regTcPtr;
  Uint32 Tmp[MAX_KEY_SIZE_IN_WORDS];

  jamEntry();
  regTcPtr.i = opPtrI;
  ndbrequire(tcConnect_pool.getValidPtr(regTcPtr));

  const Uint32 tableId = regTcPtr.p->tableref;
  const Uint32 keyLen = regTcPtr.p->primKeyLen;
  Uint32 *buf = xfrm_hash ? Tmp : dst;

  copy(buf, regTcPtr.p->keyInfoIVal);

  if (xfrm_hash) {
    jam();
    Uint32 keyPartLen[MAX_ATTRIBUTES_IN_INDEX];
    return xfrm_key_hash(tableId, Tmp, dst, ~0, keyPartLen);
  }
  return keyLen;
}

Uint32 Dblqh::readPrimaryKeys(ScanRecord *scanP, TcConnectionrec *tcConP,
                              Uint32 *dst) {
  Uint32 tableId = prim_tab_fragptr.p->tabRef;
  Uint32 fragId = tcConP->fragmentid;
  Uint32 fragPageId = scanP->m_row_id.m_page_no;
  Uint32 pageIndex = scanP->m_row_id.m_page_idx;

  if (likely(scanP->rangeScan)) {
    jamDebug();
    fragPageId = c_tup->get_current_frag_page_id();
  }
  int ret = c_tup->accReadPk(fragPageId, pageIndex, dst, /*xfrm=*/false);
  jamEntry();
  if (0)
    g_eventLogger->info(
        "readPrimaryKeys(table: %d fragment: %d [ %d %d ] -> %d", tableId,
        fragId, fragPageId, pageIndex, ret);
  ndbassert(ret > 0);

  return ret;
}

/* -------------------------------------------------------------------------
 *         ENTER TUPKEYCONF
 * -------------------------------------------------------------------------
 *       PRECONDITION:   TRANSACTION_STATE = SCAN_TUPKEY
 * ------------------------------------------------------------------------- */
void Dblqh::scanTupkeyConfLab(Signal *signal, TcConnectionrec *regTcPtr) {
  ScanRecord *const scanPtr = scanptr.p;
  Uint32 scan_direct_count = m_scan_direct_count;
  const TupKeyConf *conf = (TupKeyConf *)signal->getDataPtr();
  Uint32 read_len = conf->readLength;
  Uint32 last_row = conf->lastRow | scanPtr->m_first_match_flag;
  m_scan_direct_count = scan_direct_count + 1;

  if (!scanPtr->lcpScan && !m_is_query_block) {
    Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
    ndbassert(useStat.m_scanFragReqCount > 0);

    useStat.m_scanRowsExamined++;
    useStat.m_scanInstructionCount += conf->noExecInstructions;
  }

  regTcPtr->transactionState = TcConnectionrec::SCAN_STATE_USED;

  const Uint32 rows = scanPtr->m_curr_batch_size_rows;
  const Uint32 accOpPtr = get_acc_ptr_from_scan_record(scanPtr, rows, false);
  if (accOpPtr != (Uint32)-1) {
    c_acc->execACCKEY_ORD_no_ptr(signal, accOpPtr);
    jamEntry();
  } else {
    ndbassert(refToBlock(scanPtr->scanBlockref) != getDBACC());
  }

  if (unlikely(scanPtr->scanCompletedStatus == ZTRUE)) {
    /* ---------------------------------------------------------------------
     *       STOP THE SCAN PROCESS IF THIS HAS BEEN REQUESTED.
     * --------------------------------------------------------------------- */
    if ((scanPtr->scanLockHold == ZTRUE) && rows) {
      jam();
      scanPtr->scanReleaseCounter = 1;
      scanReleaseLocksLab(signal, regTcPtr);
      return;
    }  // if
    jam();
    closeScanLab(signal, regTcPtr);
    return;
  }  // if
  if (unlikely(scanPtr->scanKeyinfoFlag)) {
    jam();
    // Inform API about keyinfo len as well
    read_len += sendKeyinfo20(signal, scanPtr, regTcPtr);
  }  // if
  ndbrequire(scanPtr->m_curr_batch_size_rows < MAX_PARALLEL_OP_PER_SCAN);
  scanPtr->m_exec_direct_batch_size_words += read_len;
  scanPtr->m_curr_batch_size_bytes += read_len * sizeof(Uint32);
  scanPtr->m_curr_batch_size_rows = rows + 1;
  scanPtr->m_last_row = last_row;

  const NodeBitmask &all = globalTransporterRegistry.get_status_slowdown();
  if (unlikely(!all.isclear())) {
    if (all.get(refToNode(scanptr.p->scanApiBlockref))) {
      /**
       * End scan batch if transporter-buffer are in slowdown state
       *
       */
      scanPtr->m_stop_batch = 1;

      c_scanSlowDowns++;
    }
  }

  if (scanPtr->check_scan_batch_completed() || last_row) {
    if (scanPtr->scanLockHold == ZTRUE) {
      jam();
      scanPtr->scanState = ScanRecord::WAIT_SCAN_NEXTREQ;
      scanPtr->scan_lastSeen = __LINE__;
      sendScanFragConf(signal, ZFALSE, regTcPtr);
      return;
    } else {
      jam();
      scanPtr->scanReleaseCounter = rows + 1;
      scanReleaseLocksLab(signal, regTcPtr);
      return;
    }
  }
  if (unlikely(scanPtr->scanLockHold == ZTRUE)) {
    jamDebug();
    scanPtr->scanFlag = NextScanReq::ZSCAN_NEXT;
    scanNextLoopLab(signal, regTcPtr->clientConnectrec, RNIL, scanPtr,
                    fragptr.p);
    return;
  } else {
    jamDebug();
    scanPtr->scanFlag = NextScanReq::ZSCAN_NEXT_COMMIT;
    Uint32 accOpPtr = get_acc_ptr_from_scan_record(
        scanPtr, scanPtr->m_curr_batch_size_rows - 1, false);
    scanNextLoopLab(signal, regTcPtr->clientConnectrec, accOpPtr, scanPtr,
                    fragptr.p);
    return;
  }
}  // Dblqh::scanTupkeyConfLab()

/* -------------------------------------------------------------------------
 *         ENTER TUPKEYREF WITH
 *               TC_CONNECTPTR,
 *               TERROR_CODE
 * -------------------------------------------------------------------------
 *       PRECONDITION:   TRANSACTION_STATE = SCAN_TUPKEY
 * ------------------------------------------------------------------------- */
void Dblqh::scanTupkeyRefLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  ScanRecord *const scanPtr = scanptr.p;
  regTcPtr->transactionState = TcConnectionrec::SCAN_STATE_USED;

  if (!scanPtr->lcpScan && !m_is_query_block) {
    Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
    ndbassert(useStat.m_scanFragReqCount > 0);

    useStat.m_scanRowsExamined++;

    const TupKeyRef *const ref =
        reinterpret_cast<const TupKeyRef *>(signal->getDataPtr());
    useStat.m_scanInstructionCount += ref->noExecInstructions;
  }

  Uint32 rows = scanPtr->m_curr_batch_size_rows;
  Uint32 accOpPtr = get_acc_ptr_from_scan_record(scanPtr, rows, false);
  if (accOpPtr != (Uint32)-1) {
    c_acc->execACCKEY_ORD_no_ptr(signal, accOpPtr);
    jamEntryDebug();
  } else {
    ndbassert(refToBlock(scanPtr->scanBlockref) != getDBACC());
    jamDebug();
  }
  if (unlikely(scanPtr->scanCompletedStatus == ZTRUE)) {
    /* ---------------------------------------------------------------------
     *       STOP THE SCAN PROCESS IF THIS HAS BEEN REQUESTED.
     * --------------------------------------------------------------------- */
    if ((scanPtr->scanLockHold == ZTRUE) && rows) {
      jam();
      scanPtr->scanReleaseCounter = 1;
      scanReleaseLocksLab(signal, tcConnectptr.p);
      return;
    }  // if
    jam();
    closeScanLab(signal, tcConnectptr.p);
    return;
  }  // if
  if (unlikely((terrorCode != ZUSER_SEARCH_CONDITION_FALSE_CODE) &&
               (terrorCode != ZNO_TUPLE_FOUND))) {
#ifdef VM_TRACE
    ndbout << "Dblqh::scanTupkeyRefLab() aborting scan terrorCode="
           << terrorCode << endl;
#endif
    jamDebug();
    scanPtr->scanErrorCounter++;
    tcConnectptr.p->errorCode = terrorCode;

    if (scanPtr->scanLockHold == ZTRUE && rows > 0) {
      jam();
      scanPtr->scanReleaseCounter = 1;
    } else {
      jam();
      scanPtr->m_curr_batch_size_rows = rows + 1;
      scanPtr->scanReleaseCounter = rows + 1;
    }  // if
    /* --------------------------------------------------------------------
     *       WE NEED TO RELEASE ALL LOCKS CURRENTLY
     *       HELD BY THIS SCAN.
     * -------------------------------------------------------------------- */
    scanReleaseLocksLab(signal, tcConnectptr.p);
    return;
  }  // if

  // 'time_passed' is in slices of 10ms
  const Uint32 time_passed = cLqhTimeOutCount - tcConnectptr.p->tcTimer;
  if (unlikely(rows && time_passed > 1) &&
      (refToMain(scanPtr->scanApiBlockref) != DBSPJ || time_passed > 10)) {
    /* -----------------------------------------------------------------------
     *  WE NEED TO ENSURE THAT WE DO NOT SEARCH FOR THE NEXT TUPLE FOR A
     *  LONG TIME WHILE WE KEEP A LOCK ON A FOUND TUPLE. WE RATHER REPORT
     *  THE FOUND TUPLE IF FOUND TUPLES ARE RARE. If more than 10 ms passed we
     *  send the found tuples to the API. For requests coming from SPJ we allow
     *  scans to go on for an extended period of 100ms
     * -----------------------------------------------------------------------
     */
    scanPtr->scanReleaseCounter = rows + 1;
    scanReleaseLocksLab(signal, tcConnectptr.p);
    return;
  }
  scanPtr->scanFlag = NextScanReq::ZSCAN_NEXT_COMMIT;
  scanPtr->scan_acc_index--;
  scanNextLoopLab(signal, tcConnectptr.p->clientConnectrec, accOpPtr, scanPtr,
                  fragptr.p);
}  // Dblqh::scanTupkeyRefLab()

/* -------------------------------------------------------------------------
 *   THE SCAN HAS BEEN COMPLETED. EITHER BY REACHING THE END OR BY COMMAND
 *   FROM THE APPLICATION OR BY SOME SORT OF ERROR CONDITION.
 * ------------------------------------------------------------------------- */
void Dblqh::closeScanLab(Signal *signal, TcConnectionrec *regTcPtr) {
  FragrecordPtr regFragPtr = fragptr;
  ScanRecord *const scanPtr = scanptr.p;
  SimulatedBlock *block = scanPtr->scanBlock;
  const Uint32 sig0 = scanPtr->scanAccPtr;
  Fragrecord::FragStatus fragstatus = regFragPtr.p->fragStatus;
  ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;

  scanPtr->scanState = ScanRecord::WAIT_CLOSE_SCAN;
  scanPtr->scan_lastSeen = __LINE__;
  scanPtr->scan_check_lcp_stop = 0;
  regTcPtr->transactionState = TcConnectionrec::SCAN_STATE_USED;
  signal->theData[1] = RNIL;
  signal->theData[2] = NextScanReq::ZSCAN_CLOSE;
  signal->theData[0] = sig0;
  ndbrequire(is_scan_ok(scanPtr, fragstatus));
  scanPtr->scanAccPtr = RNIL;
  block->EXECUTE_DIRECT_FN(f, signal);
}  // Dblqh::closeScanLab()

/* -------------------------------------------------------------------------
 *       ENTER NEXT_SCANCONF
 * -------------------------------------------------------------------------
 *       PRECONDITION: SCAN_STATE = WAIT_CLOSE_SCAN
 * ------------------------------------------------------------------------- */
void Dblqh::accScanCloseConfLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;

  /* Do we have another range to scan? */
  if ((tcConnectptr.p->primKeyLen > 0) &&
      (scanPtr->scanCompletedStatus != ZTRUE)) {
    jam();
    /* Start next range scan...*/
    m_scan_direct_count++;
    continueAfterReceivingAllAiLab(signal, tcConnectptr);
    release_frag_access(prim_tab_fragptr.p);
    return;
  }
  TcConnectionrec *const regTcPtr = tcConnectptr.p;

  const Uint32 sig0 = regTcPtr->tupConnectrec;
  const Uint32 sig1 = regTcPtr->tableref;
  const Uint32 sig2 = scanPtr->scanSchemaVersion;
  const Uint32 sig4 = scanPtr->scanStoredProcId;
  const Uint32 sig5 = scanPtr->scanApiBlockref;

  signal->theData[0] = sig0;
  signal->theData[1] = sig1;
  signal->theData[2] = sig2;
  signal->theData[3] = ZDELETE_STORED_PROC_ID;
  signal->theData[4] = sig4;
  signal->theData[5] = sig5;
  c_tup->execSTORED_PROCREQ(signal);
  scanPtr->scanStoredProcId = RNIL;
  ndbrequire(signal->theData[0] == 0);
  /* -------------------------------------------------------------------------
   *       ENTER STORED_PROCCONF
   * -------------------------------------------------------------------------
   */
  tupScanCloseConfLab(signal, tcConnectptr);
}  // Dblqh::accScanCloseConfLab()

void Dblqh::tupScanCloseConfLab(Signal *signal,
                                TcConnectionrecPtr tcConnectptr) {
  if (scanptr.p->copyPtr != RNIL) {
    jamDebug();
    DEB_COPY(("(%u)tupScanCloseConfLab from COPY_FRAGREQ", instance()));
    tupCopyCloseConfLab(signal, tcConnectptr);
    return;
  }
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  if (regTcPtr->abortState == TcConnectionrec::NEW_FROM_TC) {
    jam();
    TcNodeFailRecordPtr tcNodeFailPtr;
    tcNodeFailPtr.i = regTcPtr->tcNodeFailrec;
    ptrCheckGuard(tcNodeFailPtr, ctcNodeFailrecFileSize, tcNodeFailRecord);
    tcNodeFailPtr.p->tcRecNow = tcConnectptr.i + 1;
    signal->theData[0] = ZLQH_TRANS_NEXT;
    signal->theData[1] = tcNodeFailPtr.i;
    sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
  } else if (regTcPtr->errorCode != 0) {
    jam();
    ScanFragRef *ref = (ScanFragRef *)&signal->theData[0];
    ref->senderData = regTcPtr->clientConnectrec;
    ref->transId1 = regTcPtr->transid[0];
    ref->transId2 = regTcPtr->transid[1];
    ref->errorCode = regTcPtr->errorCode;
    ref->senderRef = reference();
    sendSignal(tcConnectptr.p->clientBlockref, GSN_SCAN_FRAGREF, signal,
               ScanFragRef::SignalLength, JBB);
  } else {
    jam();
    sendScanFragConf(signal, ZSCAN_FRAG_CLOSED, tcConnectptr.p);
  }  // if
  handle_finish_scan(signal, tcConnectptr);
}  // Dblqh::tupScanCloseConfLab()

void Dblqh::handle_finish_scan(Signal *signal,
                               TcConnectionrecPtr tcConnectptr) {
  ScanRecordPtr restart;
  bool restart_flag = finishScanrec(signal, restart, tcConnectptr);
  if (likely(scanptr.p->scanState != ScanRecord::WAIT_START_QUEUED_SCAN)) {
    releaseScanrec(signal);
  } else {
    /**
     * We are waiting for a START QUEUED SCAN signal (CONTINUEB).
     * Until this has arrived we cannot release the scan record.
     */
    jam();
    ndbassert(!m_is_query_block);
    scanptr.p->scanState = ScanRecord::QUIT_START_QUEUE_SCAN;
  }
  deleteTransidHash(signal, tcConnectptr);
  tcConnectptr.p->tcScanRec = RNIL;
  releaseOprec(signal, tcConnectptr);
  releaseTcrec(signal, tcConnectptr);
  if (restart_flag) {
    jam();
    restart.p->scanState = ScanRecord::WAIT_START_QUEUED_SCAN;
    signal->theData[0] = ZSTART_QUEUED_SCAN;
    signal->theData[1] = restart.i;
    sendSignal(reference(), GSN_CONTINUEB, signal, 2, JBB);
  }
}

void Dblqh::restart_queued_scan(Signal *signal, Uint32 scanPtrI) {
  ScanRecordPtr loc_scanptr;
  loc_scanptr.i = scanPtrI;
  ndbassert(!m_is_query_block);
  ndbrequire(c_scanRecordPool.getValidPtr(loc_scanptr));
  if (loc_scanptr.p->scanState == ScanRecord::QUIT_START_QUEUE_SCAN) {
    jam();
    scanptr = loc_scanptr;
    releaseScanrec(signal);
    return;
  }
  ndbrequire(loc_scanptr.p->scanState == ScanRecord::WAIT_START_QUEUED_SCAN);
  ndbrequire(loc_scanptr.p->copyPtr == RNIL);
  setup_scan_pointers(scanPtrI, __LINE__);
  m_scan_direct_count = ZMAX_SCAN_DIRECT_COUNT - 8;
  // Hiding read only version in outer scope
  continueAfterReceivingAllAiLab(signal, m_tc_connect_ptr);
  release_frag_access(prim_tab_fragptr.p);
  return;
}

/* =========================================================================
 * =======              INITIATE SCAN RECORD                         =======
 *
 *       SUBROUTINE SHORT NAME = ISC
 * ========================================================================= */
Uint32 Dblqh::initScanrec(const ScanFragReq *scanFragReq, Uint32 aiLen,
                          const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;

  const Uint32 reqinfo = scanFragReq->requestInfo;
  const Uint32 keyinfo = ScanFragReq::getKeyinfoFlag(reqinfo);
  const Uint32 scanLockHold = ScanFragReq::getHoldLockFlag(reqinfo);
  const Uint32 schemaVersion = scanFragReq->schemaVersion;

  scanPtr->m_send_early_hbrep = tcConnectptr.p->m_query_thread;
  scanPtr->scanAiLength = aiLen;
  scanPtr->copyPtr = RNIL;
  scanPtr->scanStoredProcId = RNIL;
  scanPtr->scanAccPtr = RNIL;
  scanPtr->scanNumber = Uint16(~0);
  m_scan_direct_count = ZMAX_SCAN_DIRECT_COUNT - 6;
  m_tot_scan_direct_count = 0;
  scanPtr->scanType = ScanRecord::SCAN;
  scanPtr->scanState = ScanRecord::SCAN_FREE;
  scanPtr->scanCompletedStatus = ZFALSE;
  scanPtr->scanFlag = ZFALSE;
  scanPtr->scanErrorCounter = 0;
  scanPtr->scan_lastSeen = __LINE__;
  scanPtr->scan_check_lcp_stop = 0;
  scanPtr->m_stop_batch = 0;
  scanPtr->m_curr_batch_size_rows = 0;
  scanPtr->m_curr_batch_size_bytes = 0;
  scanPtr->m_exec_direct_batch_size_words = 0;
  scanPtr->m_last_row = 0;
  /* Reserved scans keep their scan_acc_segments between uses */
  ndbrequire(scanPtr->scan_acc_segments == 0 || scanPtr->m_reserved);
  scanPtr->m_row_id.setNull();
  scanPtr->scanKeyinfoFlag = keyinfo;
  scanPtr->scanLockHold = scanLockHold;
  scanPtr->scanSchemaVersion = schemaVersion;

  const Uint32 scanLockMode = ScanFragReq::getLockMode(reqinfo);
  const Uint32 readCommitted = ScanFragReq::getReadCommittedFlag(reqinfo);
  const Uint32 rangeScan = ScanFragReq::getRangeScanFlag(reqinfo);
  const Uint32 prioAFlag = ScanFragReq::getPrioAFlag(reqinfo);
  const Uint32 firstMatch = ScanFragReq::getFirstMatchFlag(reqinfo);

  scanPtr->scanLockMode = scanLockMode;
  scanPtr->readCommitted = readCommitted;
  scanPtr->rangeScan = rangeScan;
  scanPtr->prioAFlag = prioAFlag;
  scanPtr->m_first_match_flag = firstMatch;

  const Uint32 descending = ScanFragReq::getDescendingFlag(reqinfo);
  Uint32 tupScan = ScanFragReq::getTupScanFlag(reqinfo);
  const Uint32 resultRef = scanFragReq->resultRef;
  const Uint32 tcPtrI = tcConnectptr.i;

  scanPtr->descending = descending;
  scanPtr->tupScan = tupScan;
  scanPtr->scanApiBlockref = resultRef;
  scanPtr->scanTcrec = tcPtrI;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  const bool accScan = (rangeScan == 0) && (tupScan == 0);

  Uint32 blockRef;
  SimulatedBlock *block;
  ExecFunction f;
  if (accScan) {
    blockRef = caccBlockref;
    block = c_acc;
    f = c_acc->getExecuteFunction(GSN_NEXT_SCANREQ);
  } else if (!tupScan) {
    blockRef = ctuxBlockref;
    block = c_tux;
    f = c_tux->getExecuteFunction(GSN_NEXT_SCANREQ);
  } else {
    blockRef = ctupBlockref;
    block = c_tup;
    f = c_tup->getExecuteFunction(GSN_NEXT_SCANREQ);
  }
  scanPtr->scanBlockref = blockRef;
  scanPtr->scanBlock = block;
  scanPtr->scanFunction_NEXT_SCANREQ = f;

  const Uint32 lcpScan = ScanFragReq::getLcpScanFlag(reqinfo);
  const Uint32 statScan = ScanFragReq::getStatScanFlag(reqinfo);
  const Uint32 scanTcWaiting = cLqhTimeOutCount;
  const Uint32 scanApiOpPtr = scanFragReq->clientOpPtr;
  const Uint32 max_rows = scanFragReq->batch_size_rows;
  const Uint32 max_bytes = scanFragReq->batch_size_bytes;

  jamDebug();
  scanPtr->lcpScan = lcpScan;
  scanPtr->statScan = statScan;
  scanPtr->scanTcWaiting = scanTcWaiting;
  scanPtr->scanApiOpPtr = scanApiOpPtr;
  scanPtr->m_max_batch_size_rows = max_rows;
  scanPtr->m_max_batch_size_bytes = max_bytes;

  const Uint32 scanPrio = ScanFragReq::getScanPrio(reqinfo);

  if (unlikely(max_rows == 0 || (max_bytes > 0 && max_rows > max_bytes))) {
    jam();
    return ScanFragRef::ZWRONG_BATCH_SIZE;
  }

  if (ERROR_INSERTED(5057)) {
    CLEAR_ERROR_INSERT_VALUE;
    return ScanFragRef::ZTOO_MANY_ACTIVE_SCAN_ERROR;
  }

  {
    DEBUG_RES_OWNER_GUARD(refToBlock(reference()) << 16 | 999);

    if (unlikely(!seize_acc_ptr_list(scanPtr, 0, max_rows))) {
      jam();
      return ScanFragRef::ZTOO_MANY_ACTIVE_SCAN_ERROR;
    }
    init_acc_ptr_list(scanPtr);
  }

  /**
   * Used for scan take over
   */
  FragrecordPtr tFragPtr;
  tFragPtr.i = fragptr.p->tableFragptr;
  c_fragment_pool.getPtr(tFragPtr);
  scanPtr->fragPtrI = fragptr.p->tableFragptr;
  prim_tab_fragptr = tFragPtr;
  c_tup->prepare_tab_pointers(prim_tab_fragptr.p->tupFragptr);

  /**
   * ACC scan uses 1 - (MAX_PARALLEL_SCANS_PER_FRAG - 1) inclusive  =  0-11
   * Range scans uses from MAX_PARALLEL_SCANS_PER_FRAG - MAX = 12-134
   * TUP scans uses from 135 - 252
   * The boundary between Range and TUP scans are configurable and is
   * set in variable c_max_parallel_scans_per_frag.
   */

  /**
   * ACC only supports 12 parallel scans per fragment (hard limit)
   * TUP/TUX does not have any such limit...but when scanning with keyinfo
   *         (for take-over) no more than 255 such scans can be active
   *         at a fragment (dur to 8 bit number in scan-keyinfo protocol)
   *
   * TODO: Make TUP/TUX limits depend on scanKeyinfoFlag (possibly with
   *       other config limit too)
   */

  Uint32 start, stop;
  Uint32 max_parallel_scans_per_frag = c_max_parallel_scans_per_frag;
  if (accScan) {
    jam();
    start = 0;
    stop = MAX_PARALLEL_SCANS_PER_FRAG;
  } else if (rangeScan) {
    jam();
    start = MAX_PARALLEL_SCANS_PER_FRAG;
    stop = start + max_parallel_scans_per_frag;
  } else {
    jam();
    ndbassert(tupScan);
    start = MAX_PARALLEL_SCANS_PER_FRAG + max_parallel_scans_per_frag;
    stop = start + max_parallel_scans_per_frag;
    if (stop > NR_ScanNo) {
      jam();
      stop = NR_ScanNo;
    }
  }
  ndbrequire((start < 32 * tFragPtr.p->m_scanNumberMask.Size) &&
             (stop < 32 * tFragPtr.p->m_scanNumberMask.Size));

  const BlockReference senderBlock = refToMain(regTcPtr->clientBlockref);
  Uint32 free;

  if (senderBlock == getBACKUP()) {
    /**
     * Both LCP scans and Backup scans have predefined scan numbers.
     * They will never be queued and so completing them will not
     * start any queued scans.
     */
    if (lcpScan) {
      jam();
      free = LCP_ScanNo;
      c_check_scanptr_i[ZLCP_CHECK_INDEX] = scanptr.i;
      c_check_scanptr_save_timer[ZLCP_CHECK_INDEX] = regTcPtr->tcTimer;
    } else {
      /* Backup scan */
      jam();
      free = Backup_ScanNo;
      c_check_scanptr_i[ZBACKUP_CHECK_INDEX] = scanptr.i;
      c_check_scanptr_save_timer[ZBACKUP_CHECK_INDEX] = regTcPtr->tcTimer;
    }
    ndbassert(tFragPtr.p->m_scanNumberMask.get(free));
  } else {
    if (!(scanPtr->scanKeyinfoFlag || accScan)) {
      jam();
      fragptr.p->m_activeScans++;
      /**
       * Range scans and TUP scans need no scanNumber except if it is
       * takeover scan. Query threads will never use scan numbers.
       * We keep track of number of active scans currently also in
       * query threads.
       */
      return ZOK;
    }
    ndbassert(!m_is_query_block);
    ndbassert(!lcpScan);
    /*
      This error insert causes an SPJ index scan to be queued (see
      ndbinfo.test). Checking 5084 twice to ensure that the optimized build will
      see this as 'testQueue = false' and not generate code to evaluate
      subsequent terms.
    */
    const bool testQueue = ERROR_INSERTED(5084) && rangeScan &&
                           refToMain(resultRef) == DBSPJ &&
                           ERROR_INSERTED_CLEAR(5084);

    free = testQueue ? Fragrecord::ScanNumberMask::NotFound
                     : tFragPtr.p->m_scanNumberMask.find(start);

    if (free == Fragrecord::ScanNumberMask::NotFound || free >= stop) {
      /**
       * stop isn't inclusive, so we allow only ids in the range
       * [ start, stop ) .
       */
      jam();

      if (scanPrio == 0) {
        jam();
        return ScanFragRef::ZTOO_MANY_ACTIVE_SCAN_ERROR;
      }

      /**
       * Put on queue
       */
      scanPtr->scanState = ScanRecord::IN_QUEUE;
      Local_ScanRecord_fifo queue(
          c_scanRecordPool, rangeScan != 0
                                ? fragptr.p->m_queuedScans
                                : tupScan != 0 ? fragptr.p->m_queuedTupScans
                                               : fragptr.p->m_queuedAccScans);
      queue.addLast(scanptr);
      fragptr.p->m_useStat.m_queuedScanCount++;
      return ZOK;
    }
  }
  scanPtr->scanNumber = free;
  tFragPtr.p->m_scanNumberMask.clear(free);  // Update mask

  fragptr.p->m_activeScans++;
  if (scanPtr->scanKeyinfoFlag) {
    jam();
#if defined VM_TRACE || defined ERROR_INSERT
    ScanRecordPtr tmp;
    ndbrequire(!c_scanTakeOverHash.find(tmp, *scanptr.p));
#endif
#ifdef TRACE_SCAN_TAKEOVER
    g_eventLogger->info(
        "adding (%d %d) table: %d fragId: %d frag.i: %d tableFragptr: %d",
        scanPtr->scanNumber, scanPtr->fragPtrI, tabptr.i,
        scanFragReq->fragmentNoKeyLen & 0xFFFF, fragptr.i,
        fragptr.p->tableFragptr);
#endif
    c_scanTakeOverHash.add(scanptr);
  }
  return ZOK;
}

/* =========================================================================
 * =======             INITIATE TC RECORD AT SCAN                    =======
 *
 *       SUBROUTINE SHORT NAME = IST
 * ========================================================================= */
void Dblqh::initScanTc(const ScanFragReq *req, Uint32 transid1, Uint32 transid2,
                       Uint32 fragId, Uint32 nodeId, Uint32 hashHi,
                       const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  regTcPtr->transid[0] = transid1;
  regTcPtr->transid[1] = transid2;
  regTcPtr->fragmentid = fragId;
  regTcPtr->nextReplica = nodeId;
  regTcPtr->tcHashKeyHi = hashHi;

  regTcPtr->m_reorg = (req == NULL)
                          ? (Uint8)ScanFragReq::REORG_ALL
                          : (Uint8)ScanFragReq::getReorgFlag(req->requestInfo);
  regTcPtr->m_query_thread =
      (req == NULL) ? 0 : ScanFragReq::getQueryThreadFlag(req->requestInfo);
  TablerecPtr tTablePtr;
  tTablePtr.i = tabptr.p->primaryTableId;
  ptrCheckGuard(tTablePtr, ctabrecFileSize, tablerec);
  regTcPtr->m_disk_table =
      tTablePtr.p->m_disk_table &&
      (!req || !ScanFragReq::getNoDiskFlag(req->requestInfo));
  tabptr.p->usageCountR++;

  regTcPtr->dirtyOp = 0;           // dirtyOp-flag not used in scans
  regTcPtr->indTakeOver = ZFALSE;  // not used in scan
  regTcPtr->lastReplicaNo = 0;     // not used in scan
  regTcPtr->errorCode = 0;
  regTcPtr->currTupAiLen = 0;
  regTcPtr->reclenAiLqhkey = 0;
  regTcPtr->m_scan_curr_range_no = 0;
  regTcPtr->m_dealloc_state = TcConnectionrec::DA_IDLE;
  regTcPtr->m_dealloc_data.m_dealloc_ref_count = RNIL;
  regTcPtr->operation = ZREAD;
  regTcPtr->opExec = 0;  // Default 'not interpret', set later if needed
  regTcPtr->abortState = TcConnectionrec::ABORT_IDLE;
  // set TcConnectionrec::OP_SAVEATTRINFO so that a
  // "old" scan (short signals) update currTupAiLen which is checked
  // in scanAttrinfoLab
  regTcPtr->m_flags = TcConnectionrec::OP_SAVEATTRINFO;
  regTcPtr->commitAckMarker = RNIL;
  regTcPtr->activeCreat = Fragrecord::AC_NORMAL;

  {
    const Uint32 scanPtrI = scanptr.i;
    const Uint32 tabPtrI = tabptr.i;
    const Uint32 fragPtrI = fragptr.i;
    const Uint32 tcOprec = regTcPtr->clientConnectrec;
    const Uint32 tcBlockref = regTcPtr->clientBlockref;

    regTcPtr->tcScanRec = scanPtrI;
    regTcPtr->tableref = tabPtrI;
    regTcPtr->fragmentptr = fragPtrI;
    regTcPtr->tcOprec = tcOprec;
    regTcPtr->tcBlockref = tcBlockref;
  }
}  // Dblqh::initScanTc()

/* =========================================================================
 * =======                       FINISH  SCAN RECORD                 =======
 *
 *       REMOVE SCAN RECORD FROM PER FRAGMENT LIST.
 * ========================================================================= */
bool Dblqh::finishScanrec(Signal *signal, ScanRecordPtr &restart_scan,
                          const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;
  Uint32 reserved = scanPtr->m_reserved;

  if (reserved == 0) {
    release_acc_ptr_list(scanPtr);
  }

  Uint32 tupScan = scanPtr->tupScan;
  Uint32 rangeScan = scanPtr->rangeScan;

  if (scanPtr->scanState == ScanRecord::IN_QUEUE) {
    ndbassert(!m_is_query_block);
    Local_ScanRecord_fifo queue(
        c_scanRecordPool, rangeScan != 0
                              ? fragptr.p->m_queuedScans
                              : tupScan != 0 ? fragptr.p->m_queuedTupScans
                                             : fragptr.p->m_queuedAccScans);
    jam();
    ndbrequire(reserved == 0);
    queue.remove(scanptr);
    return false;
  }

  if (scanPtr->scanKeyinfoFlag) {
    jam();
    ScanRecordPtr tmp;
#ifdef TRACE_SCAN_TAKEOVER
    g_eventLogger->info("removing (%d %d)", scanPtr->scanNumber,
                        scanPtr->fragPtrI);
#endif
    ndbrequire(c_scanTakeOverHash.remove(tmp, *scanPtr));
    ndbrequire(tmp.p == scanPtr);
  }

  {
    /**
     * DESIGN PATTERN DESCRIPTION:
     * ---------------------------
     * The scans object below is created on the stack, it is deleted
     * when we reach the end of the code block where it is created, to
     * avoid keeping the object around for too long we remove it from
     * the context by creating a code block around its use.
     *
     * This enables tail-call optimisations below in the code and also
     * avoids keeping the object around even when no longer needed which
     * can easily lead to false positives in asserts in the template
     * code generated by the object.
     *
     * This comment is kept as a DESIGN PATTERN DESCRIPTION, but obviously
     * no longer applies to the code below since we removed the list of
     * active scans.
     */
    fragptr.p->m_activeScans--;
  }

  FragrecordPtr tFragPtr = prim_tab_fragptr;

  const Uint32 scanNumber = scanPtr->scanNumber;

  if (scanNumber == Uint16(~0)) {
    /**
     * Scans without scanNumber cannot be ACC Scans and cannot be takeover
     * scans. No scan number is used by range scans and TUP scans and is
     * always used by query threads.
     */
    return false;
  }
  ndbassert(!m_is_query_block);
  ndbrequire(!tFragPtr.p->m_scanNumberMask.get(scanNumber));
  ScanRecordPtr restart;

  {
    Local_ScanRecord_fifo queue(
        c_scanRecordPool, rangeScan != 0
                              ? fragptr.p->m_queuedScans
                              : tupScan != 0 ? fragptr.p->m_queuedTupScans
                                             : fragptr.p->m_queuedAccScans);
    /**
     * Start of queued scans
     */
    if (likely(!queue.first(restart)) ||
        (scanNumber >= NR_ScanNo && scanNumber <= Backup_ScanNo)) {
      jamDebug();

      /**
       * LCP scan, NR scan, Backup scans won't start any queued scans since
       * no scanNumber useful for normal scans have been freed. Also come
       * here when no scans are queued.
       */
      tFragPtr.p->m_scanNumberMask.set(scanNumber);
      return false;
    }

    if (ERROR_INSERTED(5034)) {
      jam();
      tFragPtr.p->m_scanNumberMask.set(scanNumber);
      return false;
    }
    ndbrequire(restart.p->scanState == ScanRecord::IN_QUEUE);
    queue.remove(restart);
  }

  restart.p->scanNumber = scanNumber;
  fragptr.p->m_activeScans++;
  if (restart.p->scanKeyinfoFlag) {
    jam();
#if defined VM_TRACE || defined ERROR_INSERT
    ScanRecordPtr tmp;
    ndbrequire(!c_scanTakeOverHash.find(tmp, *restart.p));
#endif
    c_scanTakeOverHash.add(restart);
#ifdef TRACE_SCAN_TAKEOVER
    g_eventLogger->info("adding-r (%d %d)", restart.p->scanNumber,
                        restart.p->fragPtrI);
#endif
  }

  /**
   * This state is a bit weird, but that what set in initScanRec
   */
  restart.p->scanState = ScanRecord::SCAN_FREE;
  ndbrequire(tcConnectptr.p->transactionState ==
             TcConnectionrec::SCAN_STATE_USED);
  jam();
  restart_scan = restart;
  return true;
}  // Dblqh::finishScanrec()

/* =========================================================================
 * =======                       RELEASE SCAN RECORD                 =======
 *
 *       RELEASE A SCAN RECORD TO THE FREELIST.
 * ========================================================================= */
void Dblqh::releaseScanrec(Signal *signal) {
  jamDebug();
  ScanRecord *const scanPtr = scanptr.p;
  if (scanPtr->m_reserved == 0) {
    c_scanRecordPool.release(scanptr);
    checkPoolShrinkNeed(DBLQH_SCAN_RECORD_TRANSIENT_POOL_INDEX,
                        c_scanRecordPool);
    return;
  }
  if (scanptr.p->scanNumber == LCP_ScanNo) {
    jam();
    c_check_scanptr_i[ZLCP_CHECK_INDEX] = RNIL;
  } else if (scanptr.p->scanNumber == Backup_ScanNo) {
    jam();
    c_check_scanptr_i[ZBACKUP_CHECK_INDEX] = RNIL;
  } else {
    jam();
    ndbrequire(scanptr.p->scanNumber == NR_ScanNo);
    c_check_scanptr_i[ZCOPY_FRAGREQ_CHECK_INDEX] = RNIL;
  }
  init_release_scanrec(scanPtr);
  m_reserved_scans.addFirst(scanptr);
}  // Dblqh::releaseScanrec()

void Dblqh::init_release_scanrec(ScanRecord *scanPtr) {
  scanPtr->scanState = ScanRecord::SCAN_FREE;
  scanPtr->scanType = ScanRecord::ST_IDLE;
  scanPtr->scanTcWaiting = 0;
  scanPtr->scan_lastSeen = __LINE__;
}

/* ------------------------------------------------------------------------
 * -------              SEND KEYINFO20 TO API                       -------
 *
 * Return: Length in number of Uint32 words
 * ------------------------------------------------------------------------  */
Uint32 Dblqh::sendKeyinfo20(Signal *signal, ScanRecord *scanP,
                            TcConnectionrec *tcConP) {
  ndbrequire(scanP->m_curr_batch_size_rows < MAX_PARALLEL_OP_PER_SCAN);
  KeyInfo20 *keyInfo = (KeyInfo20 *)&signal->theData[0];

  /**
   * Note that this code requires signal->theData to be big enough for
   * a entire key
   */
  const BlockReference ref = scanP->scanApiBlockref;
  const Uint32 scanOp = scanP->m_curr_batch_size_rows;
  Uint32 nodeId = refToNode(ref);
  const bool connectedToNode = getNodeInfo(nodeId).m_connected;
#ifdef NOT_USED
  const Uint32 type = getNodeInfo(nodeId).m_type;
  const bool is_api = (type >= NodeInfo::API && type <= NodeInfo::REP);
  const bool old_dest = (getNodeInfo(nodeId).m_version < MAKE_VERSION(3, 5, 0));
#endif
  const bool longable = true;  // TODO is_api && !old_dest;

  if (isNdbMtLqh()) {
    jam();
    nodeId = 0;  // prevent execute direct
  }

  Uint32 *dst = keyInfo->keyData;
  dst += nodeId == getOwnNodeId() ? 0 : KeyInfo20::DataLength;

  /**
   * This is ugly :-(
   *  currently only SUMA receives KEYINFO20 inside kernel..
   *  and it's not really interested in the actual keyinfo,
   *  only the scanInfo_Node...so send only that and avoid
   *  messing with if's below...
   */
  Uint32 keyLen;
  /* The blockReference ref could belong to an API node.
   * But the refToMain() is supposed to be used with only data nodes
   * as certain BlockReference numbers of API nodes will also
   * return true for 'refToMain(ref) == SUMA' which is not right.
   * So check the node id first before checking for the block */
  if (refToNode(ref) == getOwnNodeId() && refToMain(ref) == SUMA) {
    keyLen = 0;
  } else {
    keyLen = readPrimaryKeys(scanP, tcConP, dst);
  }

  Uint32 fragId = tcConP->fragmentid;
  keyInfo->clientOpPtr = scanP->scanApiOpPtr;
  keyInfo->keyLen = keyLen;
  keyInfo->scanInfo_Node =
      KeyInfo20::setScanInfo(scanOp, scanP->scanNumber) + (fragId << 20);
  keyInfo->transId1 = tcConP->transid[0];
  keyInfo->transId2 = tcConP->transid[1];

  Uint32 *src = signal->theData + 25;
  if (connectedToNode) {
    jam();

    if (nodeId == getOwnNodeId()) {
      EXECUTE_DIRECT(refToBlock(ref), GSN_KEYINFO20, signal,
                     KeyInfo20::HeaderLength + keyLen);
      jamEntry();
      return keyLen;
    } else {
      if (keyLen <= KeyInfo20::DataLength || !longable) {
        while (keyLen > KeyInfo20::DataLength) {
          jam();
          MEMCOPY_NO_WORDS(keyInfo->keyData, src, KeyInfo20::DataLength);
          sendSignal(ref, GSN_KEYINFO20, signal, 25, JBB);
          src += KeyInfo20::DataLength;
          ;
          keyLen -= KeyInfo20::DataLength;
        }

        MEMCOPY_NO_WORDS(keyInfo->keyData, src, keyLen);
        sendSignal(ref, GSN_KEYINFO20, signal, KeyInfo20::HeaderLength + keyLen,
                   JBB);
        return keyLen;
      }

      LinearSectionPtr ptr[3];
      ptr[0].p = src;
      ptr[0].sz = keyLen;
      sendSignal(ref, GSN_KEYINFO20, signal, KeyInfo20::HeaderLength, JBB, ptr,
                 1);
      return keyLen;
    }
  }

  /**
   * If this node does not have a direct connection
   * to the receiving node we want to send the signals
   * routed via the node that controls this read
   */
  Uint32 routeBlockref = tcConP->clientBlockref;

  if (keyLen < KeyInfo20::DataLength || !longable) {
    jam();

    while (keyLen > (KeyInfo20::DataLength - 1)) {
      jam();
      MEMCOPY_NO_WORDS(keyInfo->keyData, src, KeyInfo20::DataLength - 1);
      keyInfo->keyData[KeyInfo20::DataLength - 1] = ref;
      sendSignal(routeBlockref, GSN_KEYINFO20_R, signal, 25, JBB);
      src += KeyInfo20::DataLength - 1;
      keyLen -= KeyInfo20::DataLength - 1;
    }

    MEMCOPY_NO_WORDS(keyInfo->keyData, src, keyLen);
    keyInfo->keyData[keyLen] = ref;
    sendSignal(routeBlockref, GSN_KEYINFO20_R, signal,
               KeyInfo20::HeaderLength + keyLen + 1, JBB);
    return keyLen;
  }

  keyInfo->keyData[0] = ref;
  LinearSectionPtr ptr[3];
  ptr[0].p = src;
  ptr[0].sz = keyLen;
  sendSignal(routeBlockref, GSN_KEYINFO20_R, signal,
             KeyInfo20::HeaderLength + 1, JBB, ptr, 1);
  return keyLen;
}

/**
 * Function used to send NEXT_SCANREQ, we need to decide whether to
 * continue in the same signal or sending a new signal and if sending
 * a new signal we need to decide whether B-level, Bounded delay or
 * even A-level signal.
 *
 * We need to ensure that we keep track of how many outstanding NEXT_SCANREQ
 * we have, each time we send a NEXT_SCANREQ with ZSCAN_NEXT we need to
 * increment this counter to ensure that we don't end up in calling too
 * deep into the stack which otherwise can happen when we use multiple
 * ranges.
 */
void Dblqh::send_next_NEXT_SCANREQ(Signal *signal, SimulatedBlock *block,
                                   ExecFunction f, ScanRecord *const scanPtr,
                                   Uint32 clientPtrI) {
  (void)clientPtrI;
  /**
   * We have a number of different cases here. There are normal
   * scan operations, these always execute at B-level such that
   * they are scheduled among the other user level transactions.
   *
   * We also have prioritised scans, these could be scans for
   * LCPs, Backups, Node recovery or various ALTER TABLE activities.
   *
   * All internal scan activities are treated as prioritised scans.
   * These need to operate with a bounded delay. Therefore we send
   * these signals with a bounded delay signal (implemented through
   * a delayed signal with delay 0). These signals can also set the
   * priority flag to A-level to ensure that they process more rows
   * per scheduling slot than otherwise. This can be necessary at
   * very high loads when we scan for rather small rows.
   *
   * For efficiency reasons we try to execute a number of rows before
   * we send a new signal. We will never go beyond ZMAX_SCAN_DIRECT_COUNT
   * to avoid using too much of the CPU stack and also to avoid executing
   * for too long without putting ourselves back in the job buffer.
   *
   * We try to maintain the coding rule of NDB to never execute for more
   * than about 5-10 microseconds. Executing a 100 byte row scan on normal
   * CPUs in 2015 will take about 1 microsecond. If we instead scan 1000
   * bytes we estimate the time to be about 3 microseconds. So we use the
   * formula 750 ns of fixed cost per row + 8 ns per word. With this formula
   * we want to avoid that current cost has exceeded 5000 ns. If it has we
   * we will schedule a signal rather than execute directly again. Given that
   * the exactness of the formula isn't perfect and that we want scheduling
   * to happen at least before 10 microseconds we will use a simplified
   * formula. We know that scan_direct_count must be between 0 and 3 when
   * coming here and not being immediately decided to send signal, so the
   * fixed part of the cost here is between 750 ns and 3000 ns. So we will
   * allow for up to 4000 ns of words before we decide to send a signal.
   * This means that when the number of words sent exceeds 500, then we
   * we will send a signal.
   *
   * These calculations are valid for HW of 2015. Future HW is likely to be
   * faster and also we're likely to improve the efficiency of creating
   * LCPs by optimising the code. The coding rules for how long a signal
   * can execute should stay more or less constant over time. We had the
   * same coding rules also in the 1990s as we have now. However if we
   * can execute 300 MByte per second in a CPU rather than 150 MByte per
   * second then we can increase those limits. So effectively we should
   * not change the coding rules, but we should adapt our algorithms to
   * make use of the coding rules in an optimal manner. Not fixing this
   * when HW gets faster means isn't likely to cause much problems given
   * that also signals from user transactions are likely to execute faster.
   * So mainly when we optimise the LCP code we should consider changing
   * those values and when we start allowing more computations due to
   * higher CPU throughput also in signals part of user transactions.
   */
#define ZABS_MAX_SCAN_DIRECT_COUNT 128
#define ZMICROS_TO_WAIT_IN_JBB_WITH_MARGIN 500
#define ZROWS_PER_MICRO 2
#define ZMIN_SCAN_WITH_CONCURRENCY 0U

  Uint32 prioAFlag = scanPtr->prioAFlag;
  Uint32 cnf_max_scan_direct_count = c_max_scan_direct_count;
  Uint32 max_scan_direct_count =
      scanPtr->m_reserved == 1
          ? (prioAFlag ? ((2 * ZRESERVED_SCAN_BATCH_SIZE) + 2)
                       : ZMAX_SCAN_DIRECT_COUNT)
          : cnf_max_scan_direct_count;
  do {
    Uint32 scan_direct_count = m_scan_direct_count;
    bool max_words_reached = c_backup->get_max_words_per_scan_batch(
        prioAFlag, scanPtr->m_exec_direct_batch_size_words, scanPtr->lcpScan,
        clientPtrI);
    rmb();
    if (prim_tab_fragptr.p->m_cond_write_key_waiters > 0 ||
        prim_tab_fragptr.p->m_cond_exclusive_waiters > 0) {
      ndbassert(m_is_query_block);
      max_scan_direct_count =
          std::min(max_scan_direct_count, ZMIN_SCAN_WITH_CONCURRENCY);
    }
    if (scan_direct_count >= max_scan_direct_count || max_words_reached) {
      jamDebug();
      /**
       * We will check whether it is ok to execute for a longer
       * time. The design rule that we are trying to achieve is
       * that at most 1 millisecond is what we are allowed to
       * wait in the job buffer for execution of a primary key
       * operation in normal load. The rules here strive to
       * ensure that this is ensured, with some leeway.
       *
       * If an LDM thread is waiting to execute a signal that requires
       * exclusive access to the fragment we will not continue scanning,
       * we will immediately send an ACC_CHECK_SCAN signal and allow the
       * LDM thread to execute as quickly as possible.
       *
       * Executing one row scan takes from a few hundred nanoseconds
       * up to a few microseconds. If the job buffer level is low we
       * can allow ourselves to run for longer time. We will try to
       * avoid running for more than 100 microseconds even when there
       * is an opportunity to do it. This means we set
       * ZABS_MAX_SCAN_DIRECT_COUNT to 128. Thus we will never execute
       * more than 128 rows in one real-time break. This would only
       * occur if there are only few other activities at the same
       * time as well as that the scans don't send back much data to
       * the application.
       *
       * We will check the number of waiting JBB signals in the job
       * buffer. If this number is at or below 5 we should be safe to use
       * up to the maximum amount of direct scans. If it is higher we
       * will gradually decrease the amount of time we are allowed to
       * execute. If there are 10 signals in the job buffer we will set
       * the maximum limit to 50, with 20 we set it to 25 and so forth.
       * We check this by dividing 500 by the JBB level and multiplying
       * by number of rows executed per microsecond which we estimate to
       * 2.
       */
      Uint32 jbb_level = getEstimatedJobBufferLevel();
      Uint32 tot_scan_direct_count =
          m_tot_scan_direct_count + scan_direct_count;
      Uint32 tot_scan_limit = ZABS_MAX_SCAN_DIRECT_COUNT;
      if (jbb_level >= 8) {
        jamDebug();
        tot_scan_limit =
            (ZMICROS_TO_WAIT_IN_JBB_WITH_MARGIN * ZROWS_PER_MICRO) / jbb_level;
      }
      if (!max_words_reached && tot_scan_direct_count < tot_scan_limit &&
          prim_tab_fragptr.p->m_cond_write_key_waiters == 0 &&
          prim_tab_fragptr.p->m_cond_exclusive_waiters == 0) {
        scan_direct_count = 1;
        m_tot_scan_direct_count = tot_scan_direct_count;
        /**
         * We will fall through here down to the code executing the next
         * NEXT_SCANREQ as a direct signal.
         */
      } else {
        scanPtr->m_exec_direct_batch_size_words = 0;
        BlockReference resultRef = scanPtr->scanApiBlockref;

        if (scanPtr->scanBlock == c_tux) {
          /**
           * Range scan
           * Insert scan into TUX index node to ensure we get back to correct
           * position after real-time break.
           */
          c_tux->relinkScan(__LINE__);
        }
        /* Early release to ensure waiters can quickly get started */
        release_frag_access(prim_tab_fragptr.p);
        /* WE ARE ENTERING A REAL-TIME BREAK FOR A SCAN HERE */
        signal->theData[3] = signal->theData[2];
        signal->theData[2] = signal->theData[1];
        signal->theData[0] = scanptr.i;
        signal->theData[1] = GSN_NEXT_SCANREQ;
        if (!is_prioritised_scan(resultRef)) {
          /* Normal user scans */
          jamDebug();
          scanPtr->scan_lastSeen = __LINE__;
          sendSignal(reference(), GSN_ACC_CHECK_SCAN, signal, 4, JBB);
          return;
        }
        if (prioAFlag) {
          /* Prioritised scan at high load situation */
          if (scanPtr->lcpScan) {
            jamDebug();
            scanPtr->scan_lastSeen = __LINE__;
            c_backup->pausing_lcp(4, scan_direct_count);
          } else {
            jamDebug();
            scanPtr->scan_lastSeen = __LINE__;
          }
        } else {
          jamDebug();
          scanPtr->scan_lastSeen = __LINE__;
        }
        /* Prioritised scan operation */
        jamDebug();
        sendSignalWithDelay(reference(), GSN_ACC_CHECK_SCAN, signal,
                            BOUNDED_DELAY, 4);
        return;
      }
    }
    jamDebug();
    m_scan_direct_count = scan_direct_count + 1;
    m_in_send_next_scan = 1;
    /**
     * To ensure that the scheduler behave differently with more
     * execute direct we report that an extra signal was executed
     * as part of this signal execution.
     */
    scanPtr->scan_lastSeen = __LINE__;
    signal->m_extra_signals++;
    jamDebug();
    block->EXECUTE_DIRECT_FN(f, signal);
    if (m_in_send_next_scan == 1) {
      /**
       * No more calls to perform
       */
      jamDebug();
      m_in_send_next_scan = 0;
      return;
    }
    jamDebug();
    ndbassert(m_in_send_next_scan == 2);
    m_in_send_next_scan = 0;
  } while (1);
}

/* ------------------------------------------------------------------------
 * -------        SEND SCAN_FRAGCONF TO TC THAT CONTROLS THE SCAN   -------
 *
 * ------------------------------------------------------------------------ */
void Dblqh::sendScanFragConf(Signal *signal, Uint32 scanCompleted,
                             const TcConnectionrec *const regTcPtr) {
  jamDebug();
  ScanRecord *const scanPtr = scanptr.p;
  const Uint32 completed_ops = scanPtr->m_curr_batch_size_rows;
  const Uint32 total_len = scanPtr->m_curr_batch_size_bytes / sizeof(Uint32);

  ndbassert((scanPtr->m_curr_batch_size_bytes % sizeof(Uint32)) == 0);

  ndbassert(scanPtr->scanTcWaiting != 0);
  scanPtr->scanTcWaiting = 0;

  if (ERROR_INSERTED(5037)) {
    CLEAR_ERROR_INSERT_VALUE;
    return;
  }

  if (!scanPtr->lcpScan && !m_is_query_block) {
    jamDebug();
    Fragrecord::UsageStat &useStat = fragptr.p->m_useStat;
    ndbassert(useStat.m_scanFragReqCount > 0);

    useStat.m_scanRowsReturned += scanPtr->m_curr_batch_size_rows;
    useStat.m_scanWordsReturned +=
        scanPtr->m_curr_batch_size_bytes / sizeof(Uint32);
  }

  if (!scanPtr->scanLockHold) {
    jamDebug();
    scanPtr->m_curr_batch_size_rows = 0;
    scanPtr->m_curr_batch_size_bytes = 0;
  }

  if (!scanCompleted && scanPtr->scanBlock == c_tux) {
    jam();
    /**
     * Insert scan into TUX index node to ensure we get back to correct
     * position after real-time break.
     */
    c_tux->relinkScan(__LINE__);
  }
  /* WE ARE ENTERING A REAL-TIME BREAK FOR A SCAN HERE */
  scanPtr->m_stop_batch = 0;
  ScanFragConf *conf = (ScanFragConf *)&signal->theData[0];
#ifdef NOT_USED
  NodeId tc_node_id = refToNode(regTcPtr->clientBlockref);
#endif
  const Uint32 senderData = regTcPtr->clientConnectrec;
  const Uint32 trans_id1 = regTcPtr->transid[0];
  const Uint32 trans_id2 = regTcPtr->transid[1];
  const BlockReference blockRef = regTcPtr->clientBlockref;

  conf->senderData = senderData;
  conf->completedOps = completed_ops;
  conf->fragmentCompleted = scanCompleted;
  conf->transId1 = trans_id1;
  conf->transId2 = trans_id2;
  conf->total_len = total_len;

  JobBufferLevel prio_level = JBB;
  if (scanPtr->prioAFlag) {
    jamDebug();
    prio_level = JBA;
  }
  Uint32 sig_len = ScanFragConf::SignalLength;
  if (regTcPtr->m_query_thread) {
    sig_len = ScanFragConf::SignalLength_query;
    conf->activeMask = 0;
    conf->senderRef = reference();
  }
  sendSignal(blockRef, GSN_SCAN_FRAGCONF, signal, sig_len, prio_level);
}  // Dblqh::sendScanFragConf()

/* ######################################################################### */
/* #######                NODE RECOVERY MODULE                       ####### */
/*                                                                           */
/* ######################################################################### */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*   THIS MODULE IS USED WHEN A NODE HAS FAILED. IT PERFORMS A COPY OF A     */
/*   FRAGMENT TO A NEW REPLICA OF THE FRAGMENT. IT DOES ALSO SHUT DOWN ALL   */
/*   CONNECTIONS TO THE FAILED NODE.                                         */
/*---------------------------------------------------------------------------*/
Uint32 Dblqh::calculateHash(Uint32 tableId, const Uint32 *src) {
  jam();
  Uint32 tmp[MAX_KEY_SIZE_IN_WORDS * MAX_XFRM_MULTIPLY];
  Uint32 keyPartLen[MAX_ATTRIBUTES_IN_INDEX];
  Uint32 keyLen =
      xfrm_key_hash(tableId, src, tmp, sizeof(tmp) >> 2, keyPartLen);
  ndbrequire(keyLen);

  return md5_hash(tmp, keyLen);
}  // Dblqh::calculateHash()

/**
 * PREPARE COPY FRAG REQ
 */
void Dblqh::execPREPARE_COPY_FRAG_REQ(Signal *signal) {
  jamEntry();
  PrepareCopyFragReq req = *(PrepareCopyFragReq *)signal->getDataPtr();

  CRASH_INSERTION(5045);

  tabptr.i = req.tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  Uint32 max_page = RNIL;

  if (getOwnNodeId() != req.startingNodeId) {
    jam();
    /**
     * This is currently dead code...
     *   but is provided so we can impl. a better scan+delete on
     *   starting node wo/ having to change running node
     */
    ndbrequire(getOwnNodeId() == req.copyNodeId);
    c_tup->get_frag_info(req.tableId, req.fragId, &max_page);

    PrepareCopyFragConf *conf = (PrepareCopyFragConf *)signal->getDataPtrSend();
    conf->senderData = req.senderData;
    conf->senderRef = reference();
    conf->tableId = req.tableId;
    conf->fragId = req.fragId;
    conf->copyNodeId = req.copyNodeId;
    conf->startingNodeId = req.startingNodeId;
    conf->maxPageNo = max_page;
    conf->completedGci = 0;
    sendSignal(req.senderRef, GSN_PREPARE_COPY_FRAG_CONF, signal,
               PrepareCopyFragConf::SignalLength, JBB);

    return;
  }

  Uint32 completedGci = 0;
  /* Assuming 1 at a time... */
  c_fragCopyTable = req.tableId;
  c_fragCopyFrag = req.fragId;
  if (!c_copy_fragment_in_progress) {
    jam();
    sendSignal(NDBCNTR_REF, GSN_COPY_FRAG_IN_PROGRESS_REP, signal, 1, JBB);
  }
  c_copy_fragment_in_progress = true;

  if (c_fragmentCopyStart == 0) {
    c_fragmentCopyStart = NdbTick_CurrentMillisecond();
    g_eventLogger->info("LDM(%u): Starting to copy fragments.", instance());
  }
  c_fragmentsCopied++;
  c_prepare_copy_fragreq_save = req;

  if (!DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    DEB_COPY(("(%u)Copy tab(%u,%u) starts", instance(), c_fragCopyTable,
              c_fragCopyFrag));
    ndbrequire(getFragmentrec(req.fragId));

    /**
     * We set AC_IGNORED to ensure we ignore transactions (but still
     * pass them on to the next replica) before we have seen the first
     * copy row arrive.
     *
     * Here we also get the number of pages that we have in the starting
     * node. This information is used by the live node to send
     * DELETE by ROWID for all rows that potentially could exist in pages
     * no longer existing on the live node.
     */
    fragptr.p->m_copy_started_state = Fragrecord::AC_IGNORED;
    fragptr.p->fragStatus = Fragrecord::ACTIVE_CREATION;
    fragptr.p->logFlag = Fragrecord::STATE_FALSE;
    completedGci = fragptr.p->m_completed_gci;

    c_tup->get_frag_info(req.tableId, req.fragId, &max_page);
    if ((c_copy_frag_halted &&
         c_copy_frag_halt_state == COPY_FRAG_HALT_STATE_IDLE) ||
        (!c_copy_frag_halted &&
         c_copy_frag_halt_state == COPY_FRAG_HALT_WAIT_FIRST_LQHKEYREQ)) {
      jam();
      /**
       * Copy fragment process have been halted due to overload
       * of UNDO log. We will respond to this signal when
       * overload is gone.
       */
      DEB_COPY(("(%u)Halt after PREPARE_COPY_FRAG_REQ, tab(%u,%u)", instance(),
                req.tableId, req.fragId));
      c_copy_frag_halted = true;
      c_copy_frag_halt_state = PREPARE_COPY_FRAG_IS_HALTED;
      return;
    }
  }
  send_prepare_copy_frag_conf(signal, req, completedGci, max_page);
}

void Dblqh::send_prepare_copy_frag_conf(Signal *signal, PrepareCopyFragReq &req,
                                        Uint32 completedGci, Uint32 max_page) {
  PrepareCopyFragConf *conf = (PrepareCopyFragConf *)signal->getDataPtrSend();
  conf->senderData = req.senderData;
  conf->senderRef = reference();
  conf->tableId = req.tableId;
  conf->fragId = req.fragId;
  conf->copyNodeId = req.copyNodeId;
  conf->startingNodeId = req.startingNodeId;
  conf->maxPageNo = max_page;
  conf->completedGci = completedGci;
  sendSignal(req.senderRef, GSN_PREPARE_COPY_FRAG_CONF, signal,
             PrepareCopyFragConf::SignalLength, JBB);
}

/* *************************************** */
/*  COPY_FRAGREQ: Start copying a fragment */
/* *************************************** */
void Dblqh::execCOPY_FRAGREQ(Signal *signal) {
  jamEntry();
  ndbrequire(signal->getLength() >= CopyFragReq::SignalLength);
  const CopyFragReq *const copyFragReq = (CopyFragReq *)&signal->theData[0];

  tabptr.i = copyFragReq->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  const Uint32 fragId = copyFragReq->fragId;
  const Uint32 copyPtr = copyFragReq->userPtr;
  const Uint32 userRef = copyFragReq->userRef;
  const Uint32 nodeId = copyFragReq->nodeId;
  const Uint32 gci = copyFragReq->gci;
  const Uint32 schemaVersion = copyFragReq->schemaVersion;
  const Uint32 distributionKey = copyFragReq->distributionKey;
  const Uint32 tableId = copyFragReq->tableId;

  ndbrequire(cnoActiveCopy < 3);
  ndbrequire(getFragmentrec(fragId));
  ndbrequire(cfirstfreeTcConrec != RNIL);

  Uint32 nodeCount = copyFragReq->nodeCount;
  ndbrequire(signal->getLength() >= CopyFragReq::SignalLength + nodeCount)

      NdbNodeBitmask nodemask;
  {
    ndbrequire(nodeCount <= MAX_REPLICAS);
    for (Uint32 i = 0; i < nodeCount; i++)
      nodemask.set(copyFragReq->nodeList[i]);
  }
  const Uint32 maxPage = copyFragReq->nodeList[nodeCount];
  const Uint32 requestInfo = copyFragReq->nodeList[nodeCount + 1];

  if (requestInfo == CopyFragReq::CFR_NON_TRANSACTIONAL) {
    jam();
  } else {
    fragptr.p->fragDistributionKey = distributionKey;
  }
  Uint32 key = fragptr.p->fragDistributionKey;

  if (DictTabInfo::isOrderedIndex(tabptr.p->tableType)) {
    jam();
    /**
     * Ordered index doesn't need to be copied
     */
    CopyFragConf *const conf = (CopyFragConf *)&signal->theData[0];
    conf->userPtr = copyPtr;
    conf->sendingNodeId = cownNodeid;
    conf->startingNodeId = nodeId;
    conf->tableId = tabptr.i;
    conf->fragId = fragId;
    sendSignal(userRef, GSN_COPY_FRAGCONF, signal, CopyFragConf::SignalLength,
               JBB);
    return;
  }  // if
  if (!nodemask.isclear()) {
    ndbrequire(nodemask.get(getOwnNodeId()));
    ndbrequire(nodemask.get(nodeId));  // cpy dest
    nodemask.clear(getOwnNodeId());
    nodemask.clear(nodeId);

    UpdateFragDistKeyOrd *ord =
        (UpdateFragDistKeyOrd *)signal->getDataPtrSend();
    ord->tableId = tabptr.i;
    ord->fragId = fragId;
    ord->fragDistributionKey = key;
    Uint32 i = 0;
    while ((i = nodemask.find(i + 1)) != NdbNodeBitmask::NotFound) {
      Uint32 instanceNo = getInstanceNo(i, fragptr.p->lqhInstanceKey);
      BlockReference lqhRef = numberToRef(DBLQH, instanceNo, i);
      sendSignal(lqhRef, GSN_UPDATE_FRAG_DIST_KEY_ORD, signal,
                 UpdateFragDistKeyOrd::SignalLength, JBB);
    }
  }
  if (c_copy_fragment_ongoing && nodeCount > 0) {
    jam();
    /**
     * We are already busy copying a fragment. We only handle one
     * fragment at a time to avoid overloading the live node.
     * Queue the request in FIFO order.
     *
     * We handle distribution key update immediately and thus we have
     * to remove the node list from the signal to avoid setting the
     * distribution key again when restarted.
     */
    CopyFragRecordPtr copy_fragptr;
    ndbrequire(c_copy_fragment_pool.seize(copy_fragptr));
    copy_fragptr.p->m_copy_fragreq.userPtr = copyPtr;
    copy_fragptr.p->m_copy_fragreq.userRef = userRef;
    copy_fragptr.p->m_copy_fragreq.tableId = tableId;
    copy_fragptr.p->m_copy_fragreq.fragId = fragId;
    copy_fragptr.p->m_copy_fragreq.nodeId = nodeId;
    copy_fragptr.p->m_copy_fragreq.schemaVersion = schemaVersion;
    copy_fragptr.p->m_copy_fragreq.distributionKey = distributionKey;
    copy_fragptr.p->m_copy_fragreq.gci = gci;
    copy_fragptr.p->m_copy_fragreq.nodeCount = 0;
    copy_fragptr.p->m_copy_fragreq.nodeList[0] = maxPage;
    copy_fragptr.p->m_copy_fragreq.nodeList[1] = requestInfo;
    c_copy_fragment_queue.addLast(copy_fragptr);
    return;
  }
  jam();
  c_copy_fragment_ongoing = true;

  ndbrequire(fragptr.p->m_scanNumberMask.get(NR_ScanNo));
  {
    /* NR Scans allocate reserved scan records */
    ndbrequire(m_reserved_scans.first(scanptr));
    m_reserved_scans.remove(scanptr);
    fragptr.p->m_activeScans++;
  }

  DEB_COPY(("(%u)COPY_FRAGREQ tab(%u,%u)", instance(), tabptr.i, fragId));
  /* -------------------------------------------------------------------------
   */
  // We keep track of how many operation records in ACC that has been booked.
  // Copy fragment has records always booked and thus need not book any. The
  // most operations in parallel use is the m_max_batch_size_rows.
  // This variable has to be set-up here since it is used by releaseScanrec
  // to unbook operation records in ACC.
  /* -------------------------------------------------------------------------
   */
  ScanRecord *const scanPtr = scanptr.p;
  scanPtr->m_max_batch_size_rows = 0;
  scanPtr->rangeScan = 0;
  scanPtr->tupScan = 0;
  /**
   * Will always succeed since we can only call this once at a time for
   * NR operations, LCP scan operation and backup scan operation. All these
   * 3 operations have a reserved record always available for them.
   * The seizeTcrec would crash if this wasn't true and we've run out this
   * resource.
   */
  TcConnectionrecPtr tcConnectptr;
  seizeTcrec(tcConnectptr);
  tcConnectptr.p->clientBlockref = userRef;
  ndbrequire(Magic::check_ptr(tcConnectptr.p));
  /**
   * Remove implicit cast/usage of CopyFragReq
   */
  // initCopyrec(signal);
  {
    const Uint32 tcPtrI = tcConnectptr.i;
    const Uint32 fragPtrI = fragptr.i;
    const BlockReference myRef = reference();
    const BlockReference tupRef = ctupBlockref;

    scanPtr->copyPtr = copyPtr;
    scanPtr->scanNodeId = nodeId;
    scanPtr->scanTcrec = tcPtrI;
    scanPtr->scanApiOpPtr = tcPtrI;
    scanPtr->fragPtrI = fragPtrI;
    scanPtr->scanSchemaVersion = schemaVersion;
    scanPtr->scanApiBlockref = myRef;
    scanPtr->scanBlockref = tupRef;
    scanPtr->scanBlock = c_tup;
    scanPtr->scanFunction_NEXT_SCANREQ =
        c_tup->getExecuteFunction(GSN_NEXT_SCANREQ);
    scanPtr->scanType = ScanRecord::COPY;
    scanPtr->scanCompletedStatus = ZFALSE;
    scanPtr->scanErrorCounter = 0;
    scanPtr->scanNumber = NR_ScanNo;
    scanPtr->scanKeyinfoFlag = 0;  // Don't put into hash
    scanPtr->scanLockHold = ZFALSE;
    scanPtr->m_curr_batch_size_rows = 0;
    scanPtr->m_curr_batch_size_bytes = 0;
    scanPtr->m_exec_direct_batch_size_words = 0;
    scanPtr->readCommitted = 0;
    scanPtr->prioAFlag = ZFALSE;
    scanPtr->m_first_match_flag = 0;
    scanPtr->scanStoredProcId = RNIL;
    scanPtr->scanAccPtr = RNIL;
    scanPtr->scan_lastSeen = __LINE__;
    scanPtr->scan_check_lcp_stop = 0;
    m_scan_direct_count = ZMAX_SCAN_DIRECT_COUNT - 6;
    fragptr.p->m_scanNumberMask.clear(NR_ScanNo);
    c_check_scanptr_i[ZCOPY_FRAGREQ_CHECK_INDEX] = scanptr.i;
    c_check_scanptr_save_timer[ZCOPY_FRAGREQ_CHECK_INDEX] =
        tcConnectptr.p->tcTimer;
  }

  initScanTc(0, 0, (DBLQH << 20) + (cownNodeid << 8), fragId,
             copyFragReq->nodeId, 0, tcConnectptr);
  /**
   * Copy fragment always performed on primary table fragment,
   * never applied on an ordered index.
   */
  prim_tab_fragptr = fragptr;
  c_tup->prepare_tab_pointers(prim_tab_fragptr.p->tupFragptr);
  /* Save TC connect record used */
  c_tc_connect_rec_copy_frag = tcConnectptr.i;

  cactiveCopy[cnoActiveCopy] = fragptr.i;
  cnoActiveCopy++;

  {
    TcConnectionrec *const regTcPtr = tcConnectptr.p;
    const Uint32 tcPtrI = tcConnectptr.i;

    regTcPtr->schemaVersion = scanPtr->scanSchemaVersion;
    regTcPtr->copyCountWords = 0;
    regTcPtr->tcOprec = tcPtrI;
    regTcPtr->savePointId = gci;
    regTcPtr->applRef = 0;
    regTcPtr->transactionState = TcConnectionrec::SCAN_STATE_USED;
  }

  {
    AccScanReq *req = (AccScanReq *)&signal->theData[0];
    Uint32 sig_request_info = 0;
    AccScanReq::setCopyFragScanFlag(sig_request_info, 1);
    if (requestInfo == CopyFragReq::CFR_TRANSACTIONAL) {
      jam();
      /**
       * An node-recovery scan, is shared lock
       *   and may not perform disk-scan (as it then can miss uncommitted
       *   inserts)
       */
      // AccScanReq::setLockMode(sig_request_info, 0);
      // AccScanReq::setReadCommittedFlag(sig_request_info, 0);
      AccScanReq::setNRScanFlag(sig_request_info, 1);
      AccScanReq::setNoDiskScanFlag(sig_request_info, 1);
    } else {
      jam();
      /**
       * The non-transaction scan is really only a "normal" tup scan
       *   committed read.
       */
      // AccScanReq::setLockMode(sig_request_info, 0);
      AccScanReq::setReadCommittedFlag(sig_request_info, 1);
      AccScanReq::setNoDiskScanFlag(sig_request_info, 1);
      scanPtr->readCommitted = 1;
    }
    req->requestInfo = sig_request_info;
    scanPtr->scanState = ScanRecord::WAIT_ACC_COPY;
    const Uint32 scanPtrI = scanptr.i;
    const Uint32 my_ref = cownref;
    const Uint32 tabPtrI = tabptr.i;
    TcConnectionrec *const regTcPtr = tcConnectptr.p;

    req->senderData = scanPtrI;
    req->senderRef = my_ref;
    req->tableId = tabPtrI;
    req->fragmentNo = fragId;

    SimulatedBlock *block = scanPtr->scanBlock;
    const Uint32 transId1 = regTcPtr->transid[0];
    const Uint32 transId2 = regTcPtr->transid[1];
    const Uint32 savePointId = regTcPtr->savePointId;
    ExecFunction f = block->getExecuteFunction(GSN_ACC_SCANREQ);

    req->transId1 = transId1;
    req->transId2 = transId2;
    req->savePointId = savePointId;
    req->maxPage = maxPage;

    block->EXECUTE_DIRECT_FN(f, signal);
  }
  ndbrequire(signal->theData[8] == 0)
      /* ACC_SCANCONF */
      jamEntry();
  accScanConfCopyLab(signal);
  return;
}  // Dblqh::execCOPY_FRAGREQ()

void Dblqh::execUPDATE_FRAG_DIST_KEY_ORD(Signal *signal) {
  jamEntry();
  UpdateFragDistKeyOrd *ord = (UpdateFragDistKeyOrd *)signal->getDataPtr();

  tabptr.i = ord->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  ndbrequire(getFragmentrec(ord->fragId));
  fragptr.p->fragDistributionKey = ord->fragDistributionKey;
}

void Dblqh::accScanConfCopyLab(Signal *signal) {
  ScanRecord *const scanPtr = scanptr.p;
  AccScanConf *const accScanConf = (AccScanConf *)&signal->theData[0];
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = scanPtr->scanTcrec;
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  /*--------------------------------------------------------------------------*/
  /*  PRECONDITION: SCAN_STATE = WAIT_ACC_COPY                                */
  /*--------------------------------------------------------------------------*/
  if (accScanConf->flag == AccScanConf::ZEMPTY_FRAGMENT) {
    jam();
    /*---------------------------------------------------------------------------*/
    /*   THE FRAGMENT WAS EMPTY. */
    /*   REPORT SUCCESSFUL COPYING. */
    /*---------------------------------------------------------------------------*/
    tupCopyCloseConfLab(signal, tcConnectptr);
    return;
  }  // if
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  {
    const Uint32 accPtr = accScanConf->accPtr;
    const Uint32 sig0 = regTcPtr->tupConnectrec;
    const Uint32 sig1 = regTcPtr->tableref;
    const Uint32 sig2 = scanPtr->scanSchemaVersion;
    const Uint32 sig5 = scanPtr->scanApiBlockref;
    scanPtr->scanAccPtr = accPtr;
    signal->theData[0] = sig0;
    signal->theData[1] = sig1;
    signal->theData[2] = sig2;
    signal->theData[3] = ZSTORED_PROC_COPY;
    // theData[4] is not used in TUP with ZSTORED_PROC_COPY
    signal->theData[5] = sig5;
    c_tup->execSTORED_PROCREQ(signal);
    jamEntry();
  }
  /*---------------------------------------------------------------------------*/
  /*   ENTER STORED_PROCCONF WITH */
  /*     0 success == CONF, 1 failure == REF */
  /*     STORED_PROC_ID */
  /*---------------------------------------------------------------------------*/
  ndbrequire(signal->theData[0] == 0);
  scanPtr->scanStoredProcId = signal->theData[1];
  scanPtr->scanAiLength = signal->theData[2];
  c_tup->copyAttrinfo(scanPtr->scanStoredProcId, bool(regTcPtr->opExec));

  if (scanPtr->scanCompletedStatus == ZTRUE) {
    jam();
    /*---------------------------------------------------------------------------*/
    /*   THE COPY PROCESS HAVE BEEN COMPLETED, MOST LIKELY DUE TO A NODE
     * FAILURE.*/
    /*---------------------------------------------------------------------------*/
    closeCopyLab(signal, regTcPtr);
    return;
  }  // if
  fragptr.i = regTcPtr->fragmentptr;
  c_fragment_pool.getPtr(fragptr);
  prim_tab_fragptr = fragptr;
  scanPtr->scanState = ScanRecord::WAIT_NEXT_SCAN_COPY;
  ndbrequire(fragptr.p->fragStatus == Fragrecord::FSACTIVE);

  /**
   * Start sending ROWID for all operations from now on
   */
  fragptr.p->m_copy_started_state = Fragrecord::AC_NR_COPY;
  if (ERROR_INSERTED(5714)) {
    g_eventLogger->info("Starting copy of tab(%u,%u)", fragptr.p->tabRef,
                        fragptr.p->fragId);
  }

  if (false && fragptr.p->tabRef > 4) {
    g_eventLogger->info("STOPPING COPY X = [ %d %d %d %d ]",
                        refToBlock(scanPtr->scanBlockref), scanPtr->scanAccPtr,
                        RNIL, NextScanReq::ZSCAN_NEXT);

    /**
     * RESTART: > DUMP 7020 332 X
     */
    return;
  }
  {
    /**
     * This is always the first call to send_next_NEXT_SCANREQ.
     * This is always a full partition scan and can thus not
     * have multiple ranges and thus cannot be restarted.
     */
    ndbassert(m_in_send_next_scan == 0);
    const Uint32 sig0 = scanPtr->scanAccPtr;
    SimulatedBlock *block = scanPtr->scanBlock;
    ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;

    signal->theData[1] = RNIL;
    signal->theData[2] = NextScanReq::ZSCAN_NEXT;
    signal->theData[0] = sig0;
    scanPtr->scanState = ScanRecord::WAIT_NEXT_SCAN_COPY;
    scanPtr->scan_lastSeen = __LINE__;
    send_next_NEXT_SCANREQ(signal, block, f, scanPtr,
                           regTcPtr->clientConnectrec);
  }
}  // Dblqh::accScanConfCopyLab()

/*---------------------------------------------------------------------------*/
/*       ENTER NEXT_SCANCONF WITH                                            */
/*         SCANPTR,                                                          */
/*         TFRAGID,                                                          */
/*         TACC_OPPTR,                                                       */
/*         TLOCAL_KEY1,                                                      */
/*         TLOCAL_KEY2,                                                      */
/*         TKEY_LENGTH,                                                      */
/*         TKEY1,                                                            */
/*         TKEY2,                                                            */
/*         TKEY3,                                                            */
/*         TKEY4                                                             */
/*---------------------------------------------------------------------------*/
/*       PRECONDITION: SCAN_STATE = WAIT_NEXT_SCAN_COPY                      */
/*---------------------------------------------------------------------------*/
void Dblqh::nextScanConfCopyLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  NextScanConf *const nextScanConf = (NextScanConf *)&signal->theData[0];
  if (nextScanConf->fragId == RNIL) {
    jam();
    /*---------------------------------------------------------------------------*/
    /*   THERE ARE NO MORE TUPLES TO FETCH. WE NEED TO CLOSE */
    /*   THE COPY IN ACC AND DELETE THE STORED PROCEDURE IN TUP */
    /*---------------------------------------------------------------------------*/
    if (tcConnectptr.p->copyCountWords == 0) {
      closeCopyLab(signal, tcConnectptr.p);
      return;
    }  // if
    /*---------------------------------------------------------------------------*/
    // Wait until copying is completed also at the starting node before
    // reporting completion. Signal completion through scanCompletedStatus-flag.
    /*---------------------------------------------------------------------------*/
    scanptr.p->scan_check_lcp_stop = 0;
    scanptr.p->scanCompletedStatus = ZTRUE;
    scanptr.p->scanState = ScanRecord::WAIT_LQHKEY_COPY;
    scanptr.p->scan_lastSeen = __LINE__;
    if (ERROR_INSERTED(5043)) {
      CLEAR_ERROR_INSERT_VALUE;
      tcConnectptr.p->copyCountWords = ~0;
      signal->theData[0] = 9999;
      sendSignal(numberToRef(CMVMI, scanptr.p->scanNodeId), GSN_NDB_TAMPER,
                 signal, 1, JBA);
    }
    return;
  }  // if

  TcConnectionrec *tcConP = tcConnectptr.p;

  tcConP->m_use_rowid = true;
  /**
   * For copy fragment scans we are scanning from TUP, TUP returns
   * row ids in its scan, so we can safely pass it on here to an
   * operation record knowing that it is a row id we are passing.
   */
  tcConP->m_row_id = scanptr.p->m_row_id;

  scanptr.p->m_curr_batch_size_rows++;

  if (signal->getLength() == NextScanConf::SignalLengthNoKeyInfo) {
    jam();
    /**
     * This code handles the case in Node recovery where we have found a record
     * which didn't exist in this live node, it might however require that the
     * starting node deletes it. There is no primary key information since the
     * tuple was deleted and we only keep the fixed size part of the row after
     * deletion.
     *
     * This performs DELETE by ROWID, if there is a row at this ROWID in the
     * starting node it will also know the primary key to delete.
     */
    scanptr.p->scan_check_lcp_stop = 0;
    ndbrequire(nextScanConf->accOperationPtr == RNIL);
    initCopyTc(signal, ZDELETE, tcConP);
    set_acc_ptr_in_scan_record(scanptr.p, 0, RNIL);
    ndbassert(!m_is_in_query_thread);
    tcConP->gci_hi = nextScanConf->gci;
    tcConP->gci_lo = 0;

    tcConP->primKeyLen = 0;
    tcConP->totSendlenAi = 0;
    tcConP->connectState = TcConnectionrec::COPY_CONNECTED;

    /*---------------------------------------------------------------------------*/
    // To avoid using up to many operation records in ACC we will increase the
    // constant to ensure that we never send more than 40 records at a time.
    // This is where the constant 56 comes from. For long records this constant
    // will not matter that much. The current maximum is 6000 words outstanding
    // (including a number of those 56 words not really sent). We also have to
    // ensure that there are never more simultaneous usage of these operation
    // records to ensure that node recovery does not fail because of
    // simultaneous scanning.
    /*---------------------------------------------------------------------------*/
    UintR TnoOfWords = 8;
    TnoOfWords = TnoOfWords + MAGIC_CONSTANT;
    TnoOfWords = TnoOfWords + (TnoOfWords >> 2);

    /*-----------------------------------------------------------------
     * NOTE for transid1!
     * Transid1 in the tcConnection record is used load regulate the
     * copy(node recovery) process.
     * The number of outstanding words are written in the transid1
     * variable. This will be sent to the starting node in the
     * LQHKEYREQ signal and when the answer is returned in the LQHKEYCONF
     * we can reduce the number of outstanding words and check to see
     * if more LQHKEYREQ signals should be sent.
     *
     * However efficient this method is rather unsafe in such way that
     * it overwrites the transid1 original data.
     *
     * Also see TR 587.
     *----------------------------------------------------------------*/
    tcConP->transid[0] = TnoOfWords;  // Data overload, see note!
    ndbrequire(!c_copy_frag_live_node_halted);
    packLqhkeyreqLab(signal, tcConnectptr);
    tcConP->copyCountWords += TnoOfWords;
    scanptr.p->scanState = ScanRecord::WAIT_LQHKEY_COPY;
    if (tcConP->copyCountWords < cmaxWordsAtNodeRec) {
      nextRecordCopy(signal, tcConnectptr);
      return;
    }
    return;
  } else {
    // If accOperationPtr == RNIL no record was returned by ACC
    if (nextScanConf->accOperationPtr == RNIL) {
      jam();
      if (unlikely(scanptr.p->scanCompletedStatus == ZTRUE)) {
        jam();
        /* Copy is being abandoned, shut it down */
        closeCopyLab(signal, tcConnectptr.p);
        return;
      }

      scanptr.p->scan_lastSeen = __LINE__;
      signal->theData[0] = scanptr.i;
      signal->theData[1] = GSN_ACC_CHECK_SCAN;
      signal->theData[2] = AccCheckScan::ZCHECK_LCP_STOP;
      sendSignal(reference(), GSN_ACC_CHECK_SCAN, signal, 3, JBB);
      return;
    }

    initCopyTc(signal, ZINSERT, tcConP);
    set_acc_ptr_in_scan_record(scanptr.p, 0, nextScanConf->accOperationPtr);

    Fragrecord *fragPtrP = fragptr.p;
    scanptr.p->scanState = ScanRecord::WAIT_TUPKEY_COPY;
    scanptr.p->scan_check_lcp_stop = 0;
    tcConP->transactionState = TcConnectionrec::COPY_TUPKEY;
    if (tcConP->m_disk_table) {
      scanptr.p->scan_lastSeen = __LINE__;
      next_scanconf_load_diskpage(signal, scanptr.p, tcConnectptr, fragPtrP);
      return;
    } else {
      next_scanconf_tupkeyreq(signal, scanptr.p, tcConP, fragPtrP, RNIL);
      return;
    }
  }
}  // Dblqh::nextScanConfCopyLab()

/*---------------------------------------------------------------------------*/
/*   USED IN COPYING OPERATION TO RECEIVE ATTRINFO FROM TUP.                 */
/*---------------------------------------------------------------------------*/
/* ************>> */
/*  TRANSID_AI  > */
/* ************>> */
void Dblqh::execTRANSID_AI(Signal *signal) {
  jamEntry();
  /* TransID_AI received from local TUP, data is linear inline in
   * signal buff
   */
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = signal->theData[0];
  ndbrequire(tcConnect_pool.getUncheckedPtrRW(tcConnectptr));
  Uint32 length = signal->length() - TransIdAI::HeaderLength;
  TcConnectionrec::TransactionState transState =
      tcConnectptr.p->transactionState;
  Uint32 *src = &signal->theData[TransIdAI::HeaderLength];
  ndbrequire(Magic::check_ptr(tcConnectptr.p));
  ndbrequire(transState == TcConnectionrec::COPY_TUPKEY);
  bool ok = appendToSection(tcConnectptr.p->attrInfoIVal, src, length);
  if (unlikely(!ok)) {
    jam();
    tcConnectptr.p->errorCode = ZGET_ATTRINBUF_ERROR;
  }
}  // Dblqh::execTRANSID_AI()

/*--------------------------------------------------------------------------*/
/*     ENTER TUPKEYCONF WITH                                                */
/*          TC_CONNECTPTR,                                                  */
/*          TDATA2,                                                         */
/*          TDATA3,                                                         */
/*          TDATA4,                                                         */
/*          TDATA5                                                          */
/*--------------------------------------------------------------------------*/
/*  PRECONDITION:   TRANSACTION_STATE = COPY_TUPKEY                         */
/*--------------------------------------------------------------------------*/
void Dblqh::copyTupkeyRefLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  // const TupKeyRef * tupKeyRef = (TupKeyRef *)signal->getDataPtr();
  ScanRecord *scanP = scanptr.p;
  if (scanP->readCommitted == 0) {
    jam();
    ndbabort();  // Should not be possible...we read with lock
  } else {
    jam();
    /**
     * Any readCommitted scan, can get 626 if it finds a candidate record
     *   that is not visible to the scan (i.e uncommitted inserts)
     *   if scanning with locks (shared/exclusive) this is not visible
     *   to LQH as lock is taken earlier
     */
    ndbrequire(terrorCode == 626);
  }

  ndbrequire(scanptr.p->scanState == ScanRecord::WAIT_TUPKEY_COPY);
  if (tcConnectptr.p->errorCode != 0) {
    jam();
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }

  if (scanptr.p->scanCompletedStatus == ZTRUE) {
    jam();
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }

  ndbrequire(tcConnectptr.p->copyCountWords < cmaxWordsAtNodeRec);
  scanptr.p->scanState = ScanRecord::WAIT_LQHKEY_COPY;
  nextRecordCopy(signal, tcConnectptr);
}

void Dblqh::copyTupkeyConfLab(Signal *signal,
                              const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *scanP = scanptr.p;
  Uint32 scan_direct_count = m_scan_direct_count;
  const TupKeyConf *const tupKeyConf = (TupKeyConf *)signal->getDataPtr();

  UintR readLength = tupKeyConf->readLength;
  Uint32 tableId = tcConnectptr.p->tableref;
  m_scan_direct_count = scan_direct_count + 1;

  if (scanP->readCommitted == 0) {
    jam();
    Uint32 accOpPtr = get_acc_ptr_from_scan_record(scanP, 0, false);
    ndbassert(accOpPtr != (Uint32)-1);
    c_acc->execACCKEY_ORD_no_ptr(signal, accOpPtr);
  }

  if (tcConnectptr.p->errorCode != 0) {
    jam();
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }  // if
  if (scanptr.p->scanCompletedStatus == ZTRUE) {
    jam();
    /*---------------------------------------------------------------------------*/
    /*   THE COPY PROCESS HAVE BEEN CLOSED. MOST LIKELY A NODE FAILURE. */
    /*---------------------------------------------------------------------------*/
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }  // if
  TcConnectionrec *tcConP = tcConnectptr.p;
  tcConnectptr.p->totSendlenAi = readLength;
  tcConnectptr.p->connectState = TcConnectionrec::COPY_CONNECTED;

  /* Read primary keys from TUP into signal buffer space
   * (used to get here via scan keyinfo)
   */
  Uint32 *tmp = signal->getDataPtrSend() + 24;
  Uint32 len = tcConnectptr.p->primKeyLen = readPrimaryKeys(scanP, tcConP, tmp);

  ndbassert(!m_is_in_query_thread);
  tcConP->gci_hi = tmp[len];
  tcConP->gci_lo = 0;
  // Calculate hash (no need to linearise key)
  if (g_key_descriptor_pool.getPtr(tableId)->hasCharAttr) {
    tcConnectptr.p->hashValue = calculateHash(tableId, tmp);
  } else {
    tcConnectptr.p->hashValue = md5_hash(tmp, len);
  }

  // Copy keyinfo into long section for LQHKEYREQ below
  if (unlikely(!keyinfoLab(tmp, len, tcConnectptr))) {
    /* Failed to store keyInfo, fail copy
     * This will result in a COPY_FRAGREF being sent to
     * the starting node, which will cause it to fail
     */
    jamDebug();
    scanptr.p->scanErrorCounter++;
    tcConP->errorCode = ZGET_DATAREC_ERROR;
    scanptr.p->scanCompletedStatus = ZTRUE;

    closeCopyLab(signal, tcConnectptr.p);
    return;
  }

  /*---------------------------------------------------------------------------*/
  // To avoid using up to many operation records in ACC we will increase the
  // constant to ensure that we never send more than 40 records at a time.
  // This is where the constant 56 comes from. For long records this constant
  // will not matter that much. The current maximum is 6000 words outstanding
  // (including a number of those 56 words not really sent). We also have to
  // ensure that there are never more simultaneous usage of these operation
  // records to ensure that node recovery does not fail because of simultaneous
  // scanning.
  /*---------------------------------------------------------------------------*/
  UintR TnoOfWords = readLength + len;
  scanP->m_curr_batch_size_bytes += 4 * TnoOfWords;
  scanP->m_exec_direct_batch_size_words += readLength;
  TnoOfWords = TnoOfWords + MAGIC_CONSTANT;
  TnoOfWords = TnoOfWords + (TnoOfWords >> 2);

  /*-----------------------------------------------------------------
   * NOTE for transid1!
   * Transid1 in the tcConnection record is used load regulate the
   * copy(node recovery) process.
   * The number of outstanding words are written in the transid1
   * variable. This will be sent to the starting node in the
   * LQHKEYREQ signal and when the answer is returned in the LQHKEYCONF
   * we can reduce the number of outstanding words and check to see
   * if more LQHKEYREQ signals should be sent.
   *
   * However efficient this method is rather unsafe in such way that
   * it overwrites the transid1 original data.
   *
   * Also see TR 587.
   *----------------------------------------------------------------*/
  tcConnectptr.p->transid[0] = TnoOfWords;  // Data overload, see note!
  ndbrequire(!c_copy_frag_live_node_halted);
  packLqhkeyreqLab(signal, tcConnectptr);
  tcConnectptr.p->copyCountWords += TnoOfWords;
  scanptr.p->scanState = ScanRecord::WAIT_LQHKEY_COPY;
  if (tcConnectptr.p->copyCountWords < cmaxWordsAtNodeRec) {
    nextRecordCopy(signal, tcConnectptr);
    return;
  }  // if
  return;
}  // Dblqh::copyTupkeyConfLab()

/*---------------------------------------------------------------------------*/
/*     ENTER LQHKEYCONF                                                      */
/*---------------------------------------------------------------------------*/
/*   PRECONDITION: CONNECT_STATE = COPY_CONNECTED                            */
/*---------------------------------------------------------------------------*/
void Dblqh::copyCompletedLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  const LqhKeyConf *const lqhKeyConf = (LqhKeyConf *)signal->getDataPtr();

  ndbrequire(tcConnectptr.p->transid[1] == lqhKeyConf->transId2);
  if (tcConnectptr.p->copyCountWords >= cmaxWordsAtNodeRec) {
    tcConnectptr.p->copyCountWords -=
        lqhKeyConf->transId1;  // Data overload, see note!
    if (scanptr.p->scanCompletedStatus == ZTRUE) {
      jam();
      /*---------------------------------------------------------------------------*/
      // Copy to complete, we will not start any new copying.
      /*---------------------------------------------------------------------------*/
      closeCopyLab(signal, tcConnectptr.p);
      return;
    }  // if
    if (tcConnectptr.p->copyCountWords < cmaxWordsAtNodeRec) {
      jam();
      nextRecordCopy(signal, tcConnectptr);
      return;
    }  // if
    return;
  }  // if
  tcConnectptr.p->copyCountWords -=
      lqhKeyConf->transId1;  // Data overload, see note!
  ndbrequire(tcConnectptr.p->copyCountWords <= cmaxWordsAtNodeRec);
  if (tcConnectptr.p->copyCountWords > 0) {
    jam();
    return;
  }  // if
  /*---------------------------------------------------------------------------*/
  // No more outstanding copies. We will only start new ones from here if it was
  // stopped before and this only happens when copyCountWords is bigger than the
  // threshold value. Since this did not occur we must be waiting for
  // completion. Check that this is so. If not we crash to find out what is
  // going on.
  /*---------------------------------------------------------------------------*/

  if (scanptr.p->scanCompletedStatus == ZTRUE) {
    jam();
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }  // if

  if (scanptr.p->scanState == ScanRecord::WAIT_LQHKEY_COPY &&
      scanptr.p->scanErrorCounter) {
    jam();
    closeCopyLab(signal, tcConnectptr.p);
    return;
  }

  if (c_copy_frag_live_node_performing_halt &&
      scanptr.p->scanState == ScanRecord::WAIT_LQHKEY_COPY) {
    jam();
    /* No more outstanding copy rows. We are only waiting now. */
    DEB_COPY(("(%u):2: Copy fragment process halted", instance()));
    scanptr.p->scanState = ScanRecord::COPY_FRAG_HALTED;
    scanptr.p->scan_lastSeen = __LINE__;
    c_copy_frag_live_node_halted = true;
    c_copy_frag_live_node_performing_halt = false;
    send_halt_copy_frag_conf(signal, false);
    return;
  }

  /**
   * We could come here even when c_copy_frag_live_node_performing_halt
   * is set. In this case scanState is WAIT_NEXT_SCAN_COPY which means
   * we are waiting for an outstanding NEXT_SCANREQ signal.
   */
  ndbassert(!c_copy_frag_live_node_performing_halt ||
            scanptr.p->scanState == ScanRecord::WAIT_NEXT_SCAN_COPY);

  if (scanptr.p->scanState == ScanRecord::WAIT_LQHKEY_COPY) {
    jam();
    /*---------------------------------------------------------------------------*/
    // Make sure that something is in progress. Otherwise we will simply stop
    // and nothing more will happen.
    /*---------------------------------------------------------------------------*/
    systemErrorLab(signal, __LINE__);
    return;
  }  // if
  return;
}  // Dblqh::copyCompletedLab()

void Dblqh::nextRecordCopy(Signal *signal,
                           const TcConnectionrecPtr tcConnectptr) {
  TcConnectionrec *const regTcPtr = tcConnectptr.p;

  fragptr.i = regTcPtr->fragmentptr;
  c_fragment_pool.getPtr(fragptr);
  prim_tab_fragptr = fragptr;
  scanptr.i = regTcPtr->tcScanRec;
  ndbrequire(c_scanRecordPool.getValidPtr(scanptr));
  ndbrequire(scanptr.p->scanState == ScanRecord::WAIT_LQHKEY_COPY);
  /*---------------------------------------------------------------------------*/
  // Make sure that nothing is in progress. Otherwise we will have to
  // simultaneous scans on the same record and this will certainly lead to
  // unexpected behaviour.
  /*---------------------------------------------------------------------------*/
  ScanRecord *const scanPtr = scanptr.p;
  ndbrequire(fragptr.p->fragStatus == Fragrecord::FSACTIVE);

  regTcPtr->errorCode = 0;
  if (c_copy_frag_live_node_performing_halt) {
    jam();
    ndbrequire(c_tc_connect_rec_copy_frag == tcConnectptr.i);

    if (regTcPtr->copyCountWords == 0) {
      jam();
      /* No more outstanding copy rows. We are only waiting now. */
      DEB_COPY(("(%u):Copy fragment process halted", instance()));
      scanPtr->scanState = ScanRecord::COPY_FRAG_HALTED;
      scanPtr->scan_lastSeen = __LINE__;
      c_copy_frag_live_node_halted = true;
      c_copy_frag_live_node_performing_halt = false;
      send_halt_copy_frag_conf(signal, false);
    }
    return;
  }
  Uint32 acc_op_ptr = get_acc_ptr_from_scan_record(scanptr.p, 0, false);
  SimulatedBlock *block = scanPtr->scanBlock;
  ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;
  const Uint32 sig0 = scanPtr->scanAccPtr;
  Uint32 in_send_next_scan = m_in_send_next_scan;

  /**
   * Here I can assign theData[1] through acc_op_ptr in the case of
   * acc_op_ptr != RNIL and also in the case of acc_op_ptr == RNIL
   * since theData[1] is assigned RNIL in the case when acc_op_ptr == RNIL
   */
  signal->theData[0] = sig0;
  signal->theData[1] = acc_op_ptr;
  signal->theData[2] = acc_op_ptr != RNIL ? NextScanReq::ZSCAN_NEXT_COMMIT
                                          : NextScanReq::ZSCAN_NEXT;
  /**
   * No need to commit (unlock) if no previous operation in ACC
   */
  scanPtr->scanState = ScanRecord::WAIT_NEXT_SCAN_COPY;
  if (unlikely(in_send_next_scan == 0)) {
    send_next_NEXT_SCANREQ(signal, block, f, scanPtr,
                           regTcPtr->clientConnectrec);
    return;
  }
  scanPtr->scan_lastSeen = __LINE__;
  ndbassert(in_send_next_scan == 1);
  m_in_send_next_scan = 2;
  /**
   * See explanation in scanNextLoopLab(...)
   */
}  // Dblqh::nextRecordCopy()

void Dblqh::copyLqhKeyRefLab(Signal *signal,
                             const TcConnectionrecPtr tcConnectptr) {
  jamDebug();
  Uint32 copyWords = signal->theData[3];
  scanptr.p->scanErrorCounter++;
  tcConnectptr.p->errorCode = terrorCode;

  LqhKeyConf *conf = (LqhKeyConf *)signal->getDataPtrSend();
  conf->transId1 = copyWords;
  conf->transId2 = tcConnectptr.p->transid[1];
  copyCompletedLab(signal, tcConnectptr);
}  // Dblqh::copyLqhKeyRefLab()

void Dblqh::closeCopyLab(Signal *signal, TcConnectionrec *regTcPtr) {
  ScanRecord *const scanPtr = scanptr.p;

  if (regTcPtr->copyCountWords > 0) {
    /*---------------------------------------------------------------------------*/
    // We are still waiting for responses from the starting node.
    // Wait until all of those have arrived until we start the
    // close process.
    /*---------------------------------------------------------------------------*/
    jam();
    scanPtr->scanState = ScanRecord::WAIT_LQHKEY_COPY;
    scanPtr->scan_lastSeen = __LINE__;
    return;
  }  // if
  fragptr.i = regTcPtr->fragmentptr;
  regTcPtr->transid[0] = 0;
  regTcPtr->transid[1] = 0;
  c_fragment_pool.getPtr(fragptr);
  prim_tab_fragptr = fragptr;

  /**
   * Stop sending ROWID for all operations from now on
   */
  fragptr.p->m_copy_started_state = Fragrecord::AC_NORMAL;
  if (ERROR_INSERTED(5714)) {
    g_eventLogger->info("Copy of tab(%u,%u) complete", fragptr.p->tabRef,
                        fragptr.p->fragId);
  }

  Fragrecord::FragStatus fragstatus = fragptr.p->fragStatus;

  const Uint32 sig0 = scanPtr->scanAccPtr;
  SimulatedBlock *block = scanPtr->scanBlock;
  ExecFunction f = scanPtr->scanFunction_NEXT_SCANREQ;

  scanPtr->scanState = ScanRecord::WAIT_CLOSE_COPY;
  scanPtr->scan_lastSeen = __LINE__;
  scanPtr->scan_check_lcp_stop = 0;
  signal->theData[0] = sig0;
  signal->theData[1] = RNIL;
  signal->theData[2] = NextScanReq::ZSCAN_CLOSE;
  ndbrequire(fragstatus == Fragrecord::FSACTIVE);
  scanPtr->scanAccPtr = RNIL;
  block->EXECUTE_DIRECT_FN(f, signal);
}  // Dblqh::closeCopyLab()

/*---------------------------------------------------------------------------*/
/*   ENTER NEXT_SCANCONF WITH                                                */
/*     SCANPTR,                                                              */
/*     TFRAGID,                                                              */
/*     TACC_OPPTR,                                                           */
/*     TLOCAL_KEY1,                                                          */
/*     TLOCAL_KEY2,                                                          */
/*     TKEY_LENGTH,                                                          */
/*     TKEY1,                                                                */
/*     TKEY2,                                                                */
/*     TKEY3,                                                                */
/*     TKEY4                                                                 */
/*---------------------------------------------------------------------------*/
/*   PRECONDITION: SCAN_STATE = WAIT_CLOSE_COPY                              */
/*---------------------------------------------------------------------------*/
void Dblqh::accCopyCloseConfLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  ScanRecord *const scanPtr = scanptr.p;
  TcConnectionrec *const regTcPtr = tcConnectptr.p;
  const Uint32 sig0 = regTcPtr->tupConnectrec;
  const Uint32 sig1 = regTcPtr->tableref;
  const Uint32 sig2 = scanPtr->scanSchemaVersion;
  const Uint32 sig4 = scanPtr->scanStoredProcId;
  const Uint32 sig5 = scanPtr->scanApiBlockref;

  signal->theData[0] = sig0;
  signal->theData[1] = sig1;
  signal->theData[2] = sig2;
  signal->theData[3] = ZDELETE_STORED_PROC_ID;
  signal->theData[4] = sig4;
  signal->theData[5] = sig5;
  c_tup->execSTORED_PROCREQ(signal);
  jamEntryDebug();
  scanPtr->scanStoredProcId = RNIL;
  tupCopyCloseConfLab(signal, tcConnectptr);
  return;
}  // Dblqh::accCopyCloseConfLab()

/*---------------------------------------------------------------------------*/
/*   ENTER STORED_PROCCONF WITH                                              */
/*     0 success = CONF, 1 failure == REF                                    */
/*     STORED_PROC_ID                                                        */
/*---------------------------------------------------------------------------*/
void Dblqh::tupCopyCloseConfLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  c_tc_connect_rec_copy_frag = RNIL;

  if (tcConnectptr.p->abortState == TcConnectionrec::NEW_FROM_TC) {
    jam();
    TcNodeFailRecordPtr tcNodeFailPtr;
    tcNodeFailPtr.i = tcConnectptr.p->tcNodeFailrec;
    ptrCheckGuard(tcNodeFailPtr, ctcNodeFailrecFileSize, tcNodeFailRecord);
    tcNodeFailPtr.p->tcRecNow = tcConnectptr.i + 1;
    signal->theData[0] = ZLQH_TRANS_NEXT;
    signal->theData[1] = tcNodeFailPtr.i;
    sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);

    CopyFragRef *const ref = (CopyFragRef *)&signal->theData[0];
    ref->userPtr = scanptr.p->copyPtr;
    ref->sendingNodeId = cownNodeid;
    ref->startingNodeId = scanptr.p->scanNodeId;
    ref->tableId = fragptr.p->tabRef;
    ref->fragId = fragptr.p->fragId;
    ref->errorCode = ZNODE_FAILURE_ERROR;
    sendSignal(tcConnectptr.p->clientBlockref, GSN_COPY_FRAGREF, signal,
               CopyFragRef::SignalLength, JBB);
  } else {
    if (c_copy_frag_live_node_performing_halt) {
      jam();
      send_halt_copy_frag_conf(signal, true);
      c_copy_frag_live_node_performing_halt = false;
    }
    ndbrequire(!c_copy_frag_live_node_halted);

    if (scanptr.p->scanErrorCounter > 0) {
      jam();
      CopyFragRef *const ref = (CopyFragRef *)&signal->theData[0];
      ref->userPtr = scanptr.p->copyPtr;
      ref->sendingNodeId = cownNodeid;
      ref->startingNodeId = scanptr.p->scanNodeId;
      ref->tableId = fragptr.p->tabRef;
      ref->fragId = fragptr.p->fragId;
      ref->errorCode = tcConnectptr.p->errorCode;
      sendSignal(tcConnectptr.p->clientBlockref, GSN_COPY_FRAGREF, signal,
                 CopyFragRef::SignalLength, JBB);
    } else {
      jam();
      DEB_COPY(("(%u)COPY_FRAGCONF tab(%u,%u)", instance(),
                tcConnectptr.p->tableref, tcConnectptr.p->fragmentid));
      CopyFragConf *const conf = (CopyFragConf *)&signal->theData[0];
      conf->userPtr = scanptr.p->copyPtr;
      conf->sendingNodeId = cownNodeid;
      conf->startingNodeId = scanptr.p->scanNodeId;
      conf->tableId = tcConnectptr.p->tableref;
      conf->fragId = tcConnectptr.p->fragmentid;
      conf->rows_lo = scanptr.p->m_curr_batch_size_rows;
      conf->bytes_lo = scanptr.p->m_curr_batch_size_bytes;
      sendSignal(tcConnectptr.p->clientBlockref, GSN_COPY_FRAGCONF, signal,
                 CopyFragConf::SignalLength, JBB);
    }  // if
  }    // if
  releaseActiveCopy(signal);
  handle_finish_scan(signal, tcConnectptr);
  if (c_copy_fragment_queue.isEmpty()) {
    jam();
    /**
     * No more COPY_FRAGREQ queued, allow anyone to start that arrives.
     */
    c_copy_fragment_ongoing = false;
  } else {
    jam();
    /**
     * Queued COPY_FRAGREQ exists, this will be sent as normal JBB signal,
     * We will retaint the c_copy_fragment_ongoing to be false to ensure
     * that signals coming directly from DBDIH will be queued. Our signal
     * will be let through as it has nodeCount set to 0 and this can only
     * be sent from here.
     */
    CopyFragRecordPtr copy_fragptr;
    c_copy_fragment_queue.first(copy_fragptr);
    memcpy(&signal->theData[0], &copy_fragptr.p->m_copy_fragreq,
           sizeof(copy_fragptr.p->m_copy_fragreq));
    sendSignal(reference(), GSN_COPY_FRAGREQ, signal, CopyFragReq::SignalLength,
               JBB);
    c_copy_fragment_queue.removeFirst(copy_fragptr);
  }
}  // Dblqh::tupCopyCloseConfLab()

/*---------------------------------------------------------------------------*/
/*   A NODE FAILURE OCCURRED DURING THE COPY PROCESS. WE NEED TO CLOSE THE   */
/*   COPY PROCESS SINCE A NODE FAILURE DURING THE COPY PROCESS WILL ALSO     */
/*   FAIL THE NODE THAT IS TRYING TO START-UP.                               */
/*---------------------------------------------------------------------------*/
void Dblqh::closeCopyRequestLab(Signal *signal,
                                const TcConnectionrecPtr tcConnectptr) {
  scanptr.p->scanErrorCounter++;
  if (0)
    g_eventLogger->info("closeCopyRequestLab: scanState: %d",
                        scanptr.p->scanState);
  switch (scanptr.p->scanState) {
    case ScanRecord::WAIT_TUPKEY_COPY:
    case ScanRecord::WAIT_NEXT_SCAN_COPY:
      jam();
      /*---------------------------------------------------------------------------*/
      /*   SET COMPLETION STATUS AND WAIT FOR OPPORTUNITY TO STOP THE SCAN. */
      //   ALSO SET NO OF WORDS OUTSTANDING TO ZERO TO AVOID ETERNAL WAIT.
      /*---------------------------------------------------------------------------*/
      scanptr.p->scanCompletedStatus = ZTRUE;
      tcConnectptr.p->copyCountWords = 0;
      break;
    case ScanRecord::WAIT_ACC_COPY:
      jam();
      /*---------------------------------------------------------------------------*/
      /*   WE ARE CURRENTLY STARTING UP THE SCAN. SET COMPLETED STATUS AND WAIT
       * FOR*/
      /*   COMPLETION OF STARTUP. */
      /*---------------------------------------------------------------------------*/
      scanptr.p->scanCompletedStatus = ZTRUE;
      break;
    case ScanRecord::WAIT_CLOSE_COPY:
      jam();
      /*---------------------------------------------------------------------------*/
      /*   CLOSE IS ALREADY ONGOING. WE NEED NOT DO ANYTHING. */
      /*---------------------------------------------------------------------------*/
      break;
    case ScanRecord::COPY_FRAG_HALTED:
      jam();
      c_copy_frag_live_node_halted = false;
      [[fallthrough]];
    case ScanRecord::WAIT_LQHKEY_COPY:
      jam();
      /*---------------------------------------------------------------------------*/
      /*   WE ARE WAITING FOR THE FAILED NODE. THE NODE WILL NEVER COME BACK. */
      //   WE NEED TO START THE FAILURE HANDLING IMMEDIATELY.
      //   ALSO SET NO OF WORDS OUTSTANDING TO ZERO TO AVOID ETERNAL WAIT.
      /*---------------------------------------------------------------------------*/
      tcConnectptr.p->copyCountWords = 0;
      closeCopyLab(signal, tcConnectptr.p);
      break;
    default:
      ndbabort();
  }  // switch
  /* Always handled by DBTUP scans, so no need to relinkScan */
}  // Dblqh::closeCopyRequestLab()

/* ****************************************************** */
/*  COPY_ACTIVEREQ: Change state of a fragment to ACTIVE. */
/* ****************************************************** */
void Dblqh::execCOPY_ACTIVEREQ(Signal *signal) {
  /**
   * We come here two times for normal stored tables.
   * We also come here two times for ordered index tables which
   * obviously never need to much in this context.
   *
   * For NOLOGGING tables we come here one time with flags set,
   * the second time to activate REDO logging we obviously need
   * to skip since it isn't needed to activate the REDO logging.
   *
   * We can discover that the table is an ordered index by checking
   * isOrderedIndex on tableType on table object.
   * We can discover that a table is a temporary or NOLOGGING table
   * by looking at the lcpFlag on the fragment.
   *
   * Thus we need to skip the LCP handling for all ordered indexes
   * and for temporary and NOLOGGING tables in all signals.
   */
  CRASH_INSERTION(5026);

  const CopyActiveReq *const req = (CopyActiveReq *)&signal->theData[0];
  jamEntry();
  Uint32 masterPtr = req->userPtr;
  BlockReference masterRef = req->userRef;
  tabptr.i = req->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  Uint32 fragId = req->fragId;
  Uint32 flags = req->flags;
  if (unlikely(signal->getLength() < CopyActiveReq::SignalLength)) {
    jam();
    ndbabort(); /* Don't support upgrade from 7.0 */
    flags = 0;
  }
  if (c_copy_active_ongoing && ((flags & CopyActiveReq::CAR_LOCAL_SEND) == 0)) {
    jam();
    CopyActiveRecordPtr copy_activeptr;
    ndbrequire(c_copy_active_pool.seize(copy_activeptr));
    copy_activeptr.p->m_copy_activereq = *req;
    copy_activeptr.p->m_copy_activereq.flags =
        flags | CopyActiveReq::CAR_LOCAL_SEND;
    c_copy_active_queue.addLast(copy_activeptr);
    return;
  } else {
    flags &= ~CopyActiveReq::CAR_LOCAL_SEND;
  }
  jam();
  c_copy_active_ongoing = true;

  ndbrequire(getFragmentrec(fragId));

  fragptr.p->fragStatus = Fragrecord::FSACTIVE;
  /* Ensure we don't send Rowid's to any nodes we're not copying to */
  fragptr.p->m_copy_started_state = Fragrecord::AC_NORMAL;
  fragptr.p->fragDistributionKey = req->distributionKey;
  fragptr.p->m_copy_complete_flag = 1;

  if (TRACENR_FLAG)
    TRACENR("tab: " << tabptr.i << " frag: " << fragId << " COPY ACTIVE"
                    << " flags: " << hex << flags << endl);

  ndbrequire(cnoActiveCopy < 3);
  cactiveCopy[cnoActiveCopy] = fragptr.i;
  cnoActiveCopy++;
  fragptr.p->masterBlockref = masterRef;
  fragptr.p->masterPtr = masterPtr;

  if (flags) {
    /**
     * We send with flags first that indicates no logging
     * and no wait, we then send without flags to activate
     * REDO logging. We thus use the flags to indicate when
     * a new fragment is to be copied.
     */
    jam();
    log_fragment_copied(signal);
  } else {
    jam();
    DEB_COPY_ACTIVE(
        ("(%u)Activate REDO log of tab(%u,%u)", instance(), tabptr.i, fragId));
    CRASH_INSERTION(5091);
    /**
     * At first COPY_ACTIVEREQ to activate REDO log on any
     * fragment means that the copy fragment process is
     * completed and we can cease to worry about halt and
     * resume of copy fragment process.
     *
     * We can reach this state if we attempted to halt the
     * last fragment to copy and we failed to halt it before
     * it was completed. This can happen e.g. if we waited
     * for the first LQHKEYREQ.
     *
     * It can also happen if we sent HALT_COPY_FRAG_REQ,
     * in this case we might fail to halt the process and
     * the response signal HALT_COPY_FRAG_CONF is raced
     * by the COPY_FRAGCONF and COPY_ACTIVEREQ signals that
     * are sent through a different path. So this path is
     * more uncommon.
     */
    if (!c_copy_frag_halted &&
        c_copy_frag_halt_state == COPY_FRAG_HALT_WAIT_FIRST_LQHKEYREQ) {
      jam();
      DEB_LCP(
          ("(%u)Phase 2 of copy fragment started while waiting for "
           "LQHKEYREQ",
           instance()));
      c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
    }
    if (!c_copy_frag_halted && c_copy_frag_halt_process_locked &&
        c_copy_frag_halt_state == WAIT_HALT_COPY_FRAG_CONF) {
      jam();
      DEB_LCP(("(%u)Phase 2 of copy fragment started while waiting for halt",
               instance()));
      c_copy_frag_halt_process_locked = false;
      c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
    }
    if (c_copy_frag_halted && c_copy_frag_halt_process_locked &&
        c_copy_frag_halt_state == WAIT_RESUME_COPY_FRAG_CONF) {
      jam();
      DEB_LCP(
          ("(%u)Phase 2 of copy fragment started while resuming", instance()));
      c_copy_frag_halted = false;
      c_copy_frag_halt_process_locked = false;
      c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
    }
    ndbrequire(!c_copy_frag_halted &&
               c_copy_frag_halt_state == COPY_FRAG_HALT_STATE_IDLE);
    if (c_copy_fragment_in_progress) {
      jam();
      m_second_activate_fragment_ptr_i = fragptr.i;
      sendSignal(NDBCNTR_REF, GSN_COPY_FRAG_NOT_IN_PROGRESS_REP, signal, 1,
                 JBB);
      return;
    }
  }

  /**
   * 1st phase (CAR_NO_LOGGING & CAR_NO_WAIT)
   * ---------
   * Put fragment into Local LCP queue and start executing them
   * immediately. We respond without waiting for this activity
   * to complete.
   *
   * 2nd phase (No flags)
   * --------------------
   * At first COPY_ACTIVEREQ in this 2nd phase
   * -->
   *
   *    We will only receive one COPY_ACTIVEREQ at a time per LDM from DIH.
   *    However many LDMs can receive them in parallel although not
   *    necessarily all of the LDMs will receive them in parallel.
   *
   *    So when we receive the first COPY_ACTIVEREQ in this instance
   *    then we will send WAIT_ALL_COMPLETE_LCP_REQ to the LQH
   *    proxy. After this the LQH proxy will send WAIT_COMPLETE_LCP_REQ
   *    to all LDMs to ask them to wait for completion of the local
   *    LCP.
   *
   *    When the local LCP is completed the LDM will send
   *    WAIT_COMPLETE_LCP_CONF to the LQH proxy. Then the LQH proxy will
   *    send WAIT_ALL_COMPLETE_LCP_CONF to all LDMs. After this it is
   *    ok to respond to the COPY_ACTIVEREQ possibly waiting and then
   *    the COPY_ACTIVEREQ will continue as usual.
   *
   *    Record reception of the activate redo log. For tables that
   *    are ordered indexes or NOLOGGING tables or temporary tables
   *    we will respond immediately.
   *
   *    When we are done with the first LCP then we will send
   *    WAIT_COMPLETE_LCP_CONF to the LQH proxy. When all LDMs have
   *    completed this first local LCP then the LQH proxy will
   *    send WAIT_ALL_COMPLETE_LCP_CONF to all LDMs.
   *
   *    Then one could check if it makes sense to run even a 2nd
   *    Local LCP before proceeding. This as left as future work.
   *
   * After this we proceed with restart exactly as before.
   *
   * An easy way to interact with the LCP processing is to
   * simply send a LCP_FRAG_ORD to ourselves. The first
   * one will have a firstFragmentFlag. When we receive
   * the first COPY_ACTIVEREQ in the second phase we will
   * send the LCP_FRAG_ORD with the lastFragmentFlag set.
   *
   * We only need some local variable indicating that
   * we are running Local LCP to ensure that we don't
   * send any LCP_FRAG_REP and that we avoid any
   * other sends out of the node.
   *
   * If we want a fragment to be re-executed in the same
   * LCP we simply send a new LCP_FRAG_ORD after it has
   * completed. A natural place to check this is in the
   * completion after one fragment LCP where we can
   * issue a new LCP if there are no queued fragments
   * for LCP.
   *
   * If we decide on a second LCP then we simply enter
   * all fragments into the queue and wait for it to
   * complete.
   */
  if (flags & CopyActiveReq::CAR_NO_WAIT) {
    jam();
    ndbrequire(flags & CopyActiveReq::CAR_NO_LOGGING);
    ndbrequire(fragptr.p->activeTcCounter == 0);
    if (!DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
        fragptr.p->lcpFlag == Fragrecord::LCP_STATE_TRUE &&
        (c_backup->is_partial_lcp_enabled()) && !c_full_local_lcp_started) {
      jam();
      if (handle_lcp_fragment_first_phase(signal)) {
        jam();
        return;
      }
    }
    Uint32 save = fragptr.p->startGci;
    fragptr.p->startGci = 0;
    sendCopyActiveConf(signal, tabptr.i);
    fragptr.p->startGci = save;
    return;
  }
  ndbrequire((flags & CopyActiveReq::CAR_NO_WAIT) == 0 &&
             (flags & CopyActiveReq::CAR_NO_LOGGING) == 0);
  ndbrequire(!c_copy_fragment_in_progress);
  if (c_local_lcp_started) {
    jam();
    /**
     * Copy fragment no longer in progress, we are still
     * waiting for local LCP started to complete before
     * we can proceed to next step.
     */
    m_second_activate_fragment_ptr_i = fragptr.i;
    return;
  }
  ndbrequire(m_node_restart_first_local_lcp_started ||
             !c_backup->is_partial_lcp_enabled());

  activate_redo_log(signal, tabptr.i, fragId);
}  // Dblqh::execCOPY_ACTIVEREQ()

void Dblqh::execCOPY_FRAG_NOT_IN_PROGRESS_REP(Signal *signal) {
  jamEntry();
  ndbrequire(c_copy_fragment_in_progress ||
             c_num_fragments_created_since_restart == 0);
  c_copy_fragment_in_progress = false;
  if (c_num_fragments_created_since_restart == 0) {
    jam();
    DEB_COPY_ACTIVE(
        ("(%u)COPY_FRAG_NOT_IN_PROGRESS_REP no fragments", instance()));
    /* No need to do anything here. */
    return;
  }
  /**
   * We are now sure that no more local LCPs can be started,
   * we still need to wait until the current one (if a current
   * one is running) is completed before we proceed with
   * activation of the REDO logs.
   */
  if (!c_local_lcp_started) {
    jam();
    /* No local LCP ongoing, ready to proceed */
    if (m_second_activate_fragment_ptr_i == RNIL) {
      jam();
      /**
       * We haven't received the first COPY_ACTIVEREQ in
       * activate REDO log phase yet. We act when this signal
       * is received, no need to do anything now.
       */
      DEB_COPY_ACTIVE(
          ("(%u)COPY_FRAG_NOT_IN_PROGRESS_REP no second", instance()));
      return;
    }
    fragptr.i = m_second_activate_fragment_ptr_i;
    m_second_activate_fragment_ptr_i = RNIL;
    c_fragment_pool.getPtr(fragptr);
    DEB_COPY_ACTIVE(("(%u)COPY_FRAG_NOT_IN_PROGRESS_REP activate tab(%u,%u)",
                     instance(), fragptr.p->tabRef, fragptr.p->fragId));
    activate_redo_log(signal, fragptr.p->tabRef, fragptr.p->fragId);
    return;
  }
  DEB_LOCAL_LCP(
      ("(%u)Started second phase of Copy fragment, wait all LCP", instance()));
  /**
   * Local LCP is ongoing, if a full local LCP is ongoing
   * we need not do anything since NDBCNTR will either have
   * already received WAIT_ALL_COMPLETE_LCP_REQ or will
   * receive it when we are ready to send it.
   *
   * If a local LCP is ongoing, but not a full local LCP
   * then we need to send WAIT_ALL_COMPLETE_LCP_REQ from
   * here to indicate that we are ready for the complete
   * local LCP phase.
   */
  if (!c_full_local_lcp_started) {
    jam();
    c_local_lcp_sent_wait_all_complete_lcp_req = true;
    signal->theData[0] = reference();
    signal->theData[1] = c_num_fragments_created_since_restart;
    sendSignal(NDBCNTR_REF, GSN_WAIT_ALL_COMPLETE_LCP_REQ, signal, 2, JBB);
  }
  ndbrequire(c_localLcpId != 0 || c_local_lcp_sent_wait_complete_conf);
}

void Dblqh::activate_redo_log(Signal *signal, Uint32 tabPtrI, Uint32 fragId) {
  tabptr.i = tabPtrI;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  if (DictTabInfo::isOrderedIndex(tabptr.p->tableType) ||
      fragptr.p->lcpFlag != Fragrecord::LCP_STATE_TRUE) {
    jam();
    sendCopyActiveConf(signal, tabptr.i);
    return;
  }
  if (fragptr.p->lcpFlag == Fragrecord::LCP_STATE_TRUE) {
    jam();
    fragptr.p->logFlag = Fragrecord::STATE_TRUE;
  }

  fragptr.p->activeTcCounter = 1;
  /*------------------------------------------------------*/
  /*       SET IT TO ONE TO ENSURE THAT IT IS NOT POSSIBLE*/
  /*       TO DECREASE IT TO ZERO UNTIL WE HAVE COMPLETED */
  /*       THE SCAN.                                      */
  /*------------------------------------------------------*/
  signal->theData[0] = ZSCAN_TC_CONNECT;
  signal->theData[1] = 0;
  signal->theData[2] = tabptr.i;
  signal->theData[3] = fragId;
  sendSignal(cownref, GSN_CONTINUEB, signal, 4, JBB);
}

bool Dblqh::handle_lcp_fragment_first_phase(Signal *signal) {
  if (!m_node_restart_first_local_lcp_started) {
    jam();
    c_saveLcpId = c_lcpId;
    DEB_LCP(("(%u)c_lcpId = %u", instance(), c_lcpId));
    /**
     * Set first fragment flag in LCP_FRAG_ORD by
     * that c_lcpId != RNIL from start, so this will
     * set firstFragmentFlag to true.
     */
    m_first_activate_fragment_ptr_i = fragptr.i;
    signal->theData[0] = 0; /* Indicate normal local LCP started */
    sendSignal(NDBCNTR_REF, GSN_START_LOCAL_LCP_ORD, signal, 1, JBB);
    return true;
  }
  if (fragptr.p->m_local_lcp_instance_started != c_current_local_lcp_instance) {
    jam();
    sendLCP_FRAG_ORD(signal, fragptr.i);
  }
  return false;
}

void Dblqh::start_local_lcp(Signal *signal, Uint32 lcpId, Uint32 localLcpId) {
  if (!m_node_restart_first_local_lcp_started) {
    jam();
    /**
     * No more necessary to stop and wait for NDBCNTR to start the very
     * first local LCP before we proceed.
     */
    m_node_restart_first_local_lcp_started = true;
    c_saveLcpId = c_lcpId;
  }

  /**
   * The (lcpId, localLcpId) tuple must be higher than last start.
   * If equal than a local LCP must be ongoing already and we are
   * now starting a full local LCP.
   */
  ndbrequire(lcpId > m_curr_lcp_id ||
             (lcpId == m_curr_lcp_id && localLcpId > m_curr_local_lcp_id) ||
             (lcpId == m_curr_lcp_id && localLcpId == m_curr_local_lcp_id &&
              c_full_local_lcp_started && c_local_lcp_started));

  m_curr_lcp_id = lcpId;
  m_curr_local_lcp_id = localLcpId;
  c_localLcpId = 1;
  if (!c_local_lcp_started) {
    jam();
    /**
     * First fragment of new local LCP started.
     * Toggle c_current_local_lcp_instance.
     */
    c_local_lcp_started = true;
    c_max_keep_gci_in_lcp =
        (crestartNewestGci == 0 || crestartNewestGci == ZUNDEFINED_GCI_LIMIT)
            ? 2
            : crestartNewestGci;
    c_keep_gci_for_lcp = c_max_keep_gci_in_lcp;
    c_first_set_min_keep_gci = true;
    c_current_local_lcp_instance++;
    c_current_local_lcp_instance &= 1;
    DEB_LOCAL_LCP(("(%u)c_current_local_lcp_instance: %u", instance(),
                   c_current_local_lcp_instance));
  }
  if (m_first_activate_fragment_ptr_i != RNIL) {
    jam();
    fragptr.i = m_first_activate_fragment_ptr_i;
    m_first_activate_fragment_ptr_i = RNIL;
    c_fragment_pool.getPtr(fragptr);
    Uint32 save = fragptr.p->startGci;
    fragptr.p->startGci = 0;
    sendCopyActiveConf(signal, fragptr.p->tabRef);
    fragptr.p->startGci = save;
    DEB_LOCAL_LCP(
        ("(%u)Written START LCP to sysfile for normal path", instance()));
    if (!c_full_local_lcp_started && (fragptr.p->m_local_lcp_instance_started !=
                                      c_current_local_lcp_instance)) {
      jam();
      sendLCP_FRAG_ORD(signal, fragptr.i);
    }
  }
}

/**
 * A special case that was introduced as a fairly common case by WL#13930
 * where we can create any number of LDMs independent of the number of
 * REDO log parts. The special case is that some LDM can be empty during
 * a local LCP.
 *
 * This requires some special care since in PGMAN we track the number of
 * LDMs that has sent SYNC_EXTENT_PAGES_REQ to the "extra" PGMAN worker
 * that handles extent pages. Thus we cannot simply ignore the local LCPs
 * in the empty LDMs.
 *
 * The empty LCP discovers that a local LCP is started by receiving the
 * below signal START_LOCAL_LCP_ORD from NDBCNTR.
 *
 * We discover that we are an empty LDM through the variable
 * c_num_fragments_created_since_restart, if this is 0 we are an empty
 * LDM thread, meaning we have no table fragments to manage, thus for
 * the moment we don't make any use at all. Normally this should be a
 * temporary state since we will spread tables on all LDMs. But it could
 * happen temporarily.
 *
 * Since we have no fragments to checkpoint we will immediately report back
 * WAIT_ALL_COMPLETE_LCP_REQ giving NDBCNTR knowledge that we are an empty
 * LDM thread. NDBCNTR uses this knowledge to ignore this signal and will
 * not start the process of completing the local LCP. The code in DBLQH is
 * not designed to receive a signal to complete the LCP before we have
 * executed phase 1 of the COPY_ACTIVEREQ signals to completion.
 *
 * In the empty LDMs nothing more will happen until some other LDM reports
 * that the local LCP is ready to complete. We prepare for this already
 * here by setting c_localLcpId = 1 to indicate that a local LCP is ongoing
 * also in the empty LDMs.
 *
 * We can be certain that at least 1 LDM will report the completion of the
 * LCP since we must have at least SYSTAB_0 defined with at least 1 fragment
 * in one of the LDM threads.
 *
 * This is indicated by sending WAIT_COMPLETE_LCP_REQ to all LDMs.
 * When we receive this signal we will save the old c_lcpId in preparation
 * for calling complete_local_lcp.
 *
 * Since it isn't a full local LCP we will now start the completion of the
 * by sending LCP_FRAG_ORD with the last fragment flag set.
 *
 * When the empty LDM sees this it will as all other LDM threads start to
 * interact with the Backup block by sending END_LCPREQ. This in turn will
 * send the SYNC_EXTENT_PAGES_REQ to PGMAN and receive SYNC_EXTENT_PAGES_CONF
 * when the synchronisation of extent pages are done. After receiving this
 * signal the empty LDM will receive END_LCPCONF from the Backup block
 * and this will trigger a call to complete_local_lcp since we set
 * c_localLcpId = 1 in the below method. This method will also restore the
 * old c_lcpId saved when receiving the WAIT_COMPLETE_LCP_REQ signal.
 *
 * At this point the EMPTY LDM is done with its participation in the LCP.
 *
 * When a distributed LCP occurs the empty LDM will not receive any
 * LCP_FRAG_ORD other than the one with the last fragment flag. This will
 * trigger the same interaction as above except that the completion of
 * a distributed LCP uses the LCP_ALL_COMPLETE protocol rather than just
 * calling complete_local_lcp. It will trigger the same execution of
 * signals as described above for local LCPs with END_LCPREQ,
 * SYNC_EXTENT_PAGES.., and END_LCPCONF.
 *
 * When starting a full local LCP we will treat it the same way in an
 * empty LDM as other local LCPs.
 */
void Dblqh::execSTART_LOCAL_LCP_ORD(Signal *signal) {
  Uint32 lcpId = signal->theData[0];
  Uint32 localLcpId = signal->theData[1];
  if (c_num_fragments_created_since_restart == 0) {
    jam();
    c_local_lcp_sent_wait_all_complete_lcp_req = true;
    c_localLcpId = 1;
    c_local_lcp_started = true;
    m_curr_lcp_id = lcpId;
    m_curr_local_lcp_id = localLcpId;
    signal->theData[0] = reference();
    signal->theData[1] = 0;
    sendSignal(NDBCNTR_REF, GSN_WAIT_ALL_COMPLETE_LCP_REQ, signal, 2, JBB);
    return;
  }
  start_local_lcp(signal, lcpId, localLcpId);
}

void Dblqh::execSTART_FULL_LOCAL_LCP_ORD(Signal *signal) {
  Uint32 lcpId = signal->theData[0];
  Uint32 localLcpId = signal->theData[1];

  c_full_local_lcp_started = true;

  if (c_local_lcp_started && c_localLcpId == 0) {
    /**
     * We have started a local LCP already and also completed it.
     * But the local LCP isn't completed in all other threads yet.
     *
     * If we haven't already sent WAIT_ALL_COMPLETE_LCP_REQ we will
     * send it now, if this is sent but not WAIT_COMPLETE_LCP_CONF
     * then we will send that signal.
     */
    if (c_local_lcp_sent_wait_all_complete_lcp_req) {
      jam();
      if (!c_local_lcp_sent_wait_complete_conf) {
        c_local_lcp_sent_wait_complete_conf = true;
        WaitCompleteLcpConf *conf =
            (WaitCompleteLcpConf *)signal->getDataPtrSend();
        conf->senderRef = reference();
        conf->lcpId = lcpId;
        conf->localLcpId = localLcpId;
        conf->maxGciInLcp = c_max_gci_in_lcp;
        conf->maxKeepGci = c_max_keep_gci_in_lcp;
        sendSignal(NDBCNTR_REF, GSN_WAIT_COMPLETE_LCP_CONF, signal,
                   WaitCompleteLcpConf::SignalLength, JBB);
      }
      return;
    } else {
      jam();
      c_local_lcp_sent_wait_all_complete_lcp_req = true;
      signal->theData[0] = reference();
      signal->theData[1] = c_num_fragments_created_since_restart;
      sendSignal(NDBCNTR_REF, GSN_WAIT_ALL_COMPLETE_LCP_REQ, signal, 2, JBB);
    }
    return;
  }
  if (c_num_fragments_created_since_restart == 0) {
    jam();
    c_local_lcp_sent_wait_all_complete_lcp_req = true;
    c_localLcpId = 1;
    c_local_lcp_started = true;
    m_curr_lcp_id = lcpId;
    m_curr_local_lcp_id = localLcpId;
    signal->theData[0] = reference();
    signal->theData[1] = 0;
    sendSignal(NDBCNTR_REF, GSN_WAIT_ALL_COMPLETE_LCP_REQ, signal, 2, JBB);
    return;
  }
  start_local_lcp(signal, lcpId, localLcpId);
  DEB_LOCAL_LCP(("(%u): start_full_local_lcp", instance()));
  c_current_local_lcp_table_id = 0;
  start_lcp_on_table(signal);
}

void Dblqh::sendLCP_FRAG_ORD(Signal *signal, Uint32 fragPtrI) {
  /* Send LCP_FRAG_ORD for the fragment. */
  LcpFragOrd *lcpFragOrd = (LcpFragOrd *)signal->getDataPtrSend();
  fragptr.i = fragPtrI;
  c_fragment_pool.getPtr(fragptr);
  fragptr.p->m_local_lcp_instance_started++;
  fragptr.p->m_local_lcp_instance_started &= 1;
  ndbrequire(c_current_local_lcp_instance ==
             fragptr.p->m_local_lcp_instance_started);

  lcpFragOrd->tableId = fragptr.p->tabRef;
  lcpFragOrd->fragmentId = fragptr.p->fragId;
  lcpFragOrd->lcpNo = 0;
  lcpFragOrd->lcpId = m_curr_lcp_id;
  lcpFragOrd->lastFragmentFlag = false;
  lcpFragOrd->keepGci = 0;
  sendSignal(reference(), GSN_LCP_FRAG_ORD, signal, LcpFragOrd::SignalLength,
             JBB);
}

void Dblqh::complete_local_lcp(Signal *signal) {
  /**
   * We have completed our local LCP, we still need to wait for the
   * rest of the LDMs to finish their local LCP.
   */
  DEB_LCP(("(%u)Completed local LCP", instance()));
  c_localLcpId = 0;
  c_lcpId = c_saveLcpId;
  DEB_LCP(("(%u)Restored c_lcpId = %u", instance(), c_lcpId));

  if (c_full_local_lcp_started) {
    jam();
    if (!c_local_lcp_sent_wait_all_complete_lcp_req) {
      jam();
      c_local_lcp_sent_wait_all_complete_lcp_req = true;
      signal->theData[0] = reference();
      signal->theData[0] = c_num_fragments_created_since_restart;
      sendSignal(NDBCNTR_REF, GSN_WAIT_ALL_COMPLETE_LCP_REQ, signal, 2, JBB);
      return;
    }
  }
  jam();
  ndbrequire(c_local_lcp_sent_wait_all_complete_lcp_req);
  ndbrequire(!c_local_lcp_sent_wait_complete_conf);
  c_local_lcp_sent_wait_complete_conf = true;
  WaitCompleteLcpConf *conf = (WaitCompleteLcpConf *)signal->getDataPtrSend();
  conf->senderRef = reference();
  conf->lcpId = m_curr_lcp_id;
  conf->localLcpId = m_curr_local_lcp_id;
  conf->maxGciInLcp = c_max_gci_in_lcp;
  conf->maxKeepGci = c_max_keep_gci_in_lcp;
  sendSignal(NDBCNTR_REF, GSN_WAIT_COMPLETE_LCP_CONF, signal,
             WaitCompleteLcpConf::SignalLength, JBB);
}

void Dblqh::execWAIT_COMPLETE_LCP_REQ(Signal *signal) {
  /**
   * Check if we need to handle the case where an LDM have no
   * fragments defined. This could e.g. happen after a config change
   * where we have added more LDMs to a node.
   *
   * If this happens then we will still have
   * m_node_restart_first_local_lcp_started equal to false. In this
   * case we skip immediately to complete_local_lcp. Could
   * also happen since we configured to not use partial LCP.
   *
   * With new multithread code we can also start quite a few LDMs, it isn't
   * necessary that all LDMs have received any fragments yet if only a few
   * tables have been created so far. If enough tables are created all LDMs
   * will eventually be used.
   */
  c_local_lcp_sent_wait_complete_conf = false;
  c_local_lcp_sent_wait_all_complete_lcp_req = true;
  if (!m_node_restart_first_local_lcp_started &&
      c_num_fragments_created_since_restart == 0) {
    jam();
    c_saveLcpId = c_lcpId;
  }
  if (!c_full_local_lcp_started) {
    jam();
    /**
     * Normal path, no LCP was started due to UNDO log overload.
     * We still started a LCP and now that all fragments have
     * completed synchronisation we can complete the
     * local LCP and as soon as this is done we can continue
     * the restart processing.
     */
    send_lastLCP_FRAG_ORD(signal);
    return;
  }
  if ((c_localLcpId == 0) || (c_num_fragments_created_since_restart == 0)) {
    jam();
    /**
     * We had an local LCP ordered due to UNDO log overload, this
     * have already completed LCP of all fragments, so we're ready to
     * continue.
     */
    c_localLcpId = 0;
    c_local_lcp_sent_wait_complete_conf = true;
    signal->theData[0] = reference();
    WaitCompleteLcpConf *conf = (WaitCompleteLcpConf *)signal->getDataPtrSend();
    conf->senderRef = reference();
    conf->lcpId = m_curr_lcp_id;
    conf->localLcpId = m_curr_local_lcp_id;
    if (c_num_fragments_created_since_restart > 0) {
      jam();
      conf->maxGciInLcp = c_max_gci_in_lcp;
      conf->maxKeepGci = c_max_keep_gci_in_lcp;
    } else {
      jam();
      conf->maxGciInLcp = 0;
      conf->maxKeepGci = 0;
    }
    sendSignal(NDBCNTR_REF, GSN_WAIT_COMPLETE_LCP_CONF, signal,
               WaitCompleteLcpConf::SignalLength, JBB);
    return;
  }
  ndbrequire(c_num_fragments_created_since_restart > 0);
  /**
   * A local LCP was ordered due to UNDO log overload, this haven't
   * completed yet. So we need to wait until it is completed until
   * we proceed to next step.
   */
}

void Dblqh::send_lastLCP_FRAG_ORD(Signal *signal) {
  /**
   * To ensure that we reach the correct path we set lcpId equal to
   * c_lcpId here. It will later be restored to its original value
   * using c_saveLcpId.
   */
  DEB_LOCAL_LCP(("(%u)Send last LCP_FRAG_ORD, c_full_local_lcp_started: %u",
                 instance(), c_full_local_lcp_started));
  LcpFragOrd *lcpFragOrd = (LcpFragOrd *)signal->getDataPtrSend();
  lcpFragOrd->tableId = RNIL;
  lcpFragOrd->fragmentId = RNIL;
  lcpFragOrd->lcpNo = 0;
  lcpFragOrd->lcpId = c_lcpId;
  lcpFragOrd->lastFragmentFlag = true;
  lcpFragOrd->keepGci = 0;
  sendSignal(reference(), GSN_LCP_FRAG_ORD, signal, LcpFragOrd::SignalLength,
             JBB);
}

void Dblqh::execWAIT_ALL_COMPLETE_LCP_CONF(Signal *signal) {
  /**
   * We have completed waiting for Local LCPs to complete in LDMs.
   * All LDMs will receive this, but it is not necessary that we
   * have any waiting fragment to activate, this could happen
   * either if the LDM is a new one or if the parallelism is lower
   * in the DIH than the number of LDMs.
   *
   * When this signal arrives we are certain that the LCP is
   * fully restorable since we also waited for the GCI to restore
   * before we sent this message. So we are also ready here to
   * cut the log tail of the UNDO log and the log tail of the
   * REDO log.
   */
  ndbrequire(c_local_lcp_sent_wait_complete_conf);
  c_local_lcp_sent_wait_complete_conf = false;
  ndbrequire(c_local_lcp_sent_wait_all_complete_lcp_req);
  c_local_lcp_sent_wait_all_complete_lcp_req = false;
  c_local_lcp_started = false;
  c_full_local_lcp_started = false;
  DEB_LOCAL_LCP(("(%u)All LDMs have completed local LCP", instance()));
  if (m_second_activate_fragment_ptr_i == RNIL) {
    jam();
    return;
  }
  fragptr.i = m_second_activate_fragment_ptr_i;
  m_second_activate_fragment_ptr_i = RNIL;
  c_fragment_pool.getPtr(fragptr);
  activate_redo_log(signal, fragptr.p->tabRef, fragptr.p->fragId);
}

void Dblqh::start_lcp_on_table(Signal *signal) {
  while (c_current_local_lcp_table_id < ctabrecFileSize) {
    jam();
    tabptr.i = c_current_local_lcp_table_id;
    ptrAss(tabptr, tablerec);
    if ((tabptr.p->tableStatus == Tablerec::TABLE_DEFINED ||
         tabptr.p->tableStatus == Tablerec::TABLE_READ_ONLY) &&
        (!DictTabInfo::isOrderedIndex(tabptr.p->tableType))) {
      jam();
      for (Uint32 i = 0; i < NDB_ARRAY_SIZE(tabptr.p->fragid); i++) {
        jam();
        if (tabptr.p->fragid[i] != ZNIL) {
          jam();
          fragptr.i = tabptr.p->fragrec[i];
          c_fragment_pool.getPtr(fragptr);
          bool ret = handle_lcp_fragment_first_phase(signal);
          ndbrequire(!ret);
        }
      }
      c_current_local_lcp_table_id++;
      signal->theData[0] = ZSTART_LOCAL_LCP;
      sendSignal(reference(), GSN_CONTINUEB, signal, 1, JBB);
      return;
    }
    c_current_local_lcp_table_id++;
  }
  /**
   * We're done starting all fragments, so now time to send
   * the LCP_FRAG_ORD with lastFragmentFlag set to true.
   */
  send_lastLCP_FRAG_ORD(signal);
}

void Dblqh::execCUT_REDO_LOG_TAIL_REQ(Signal *signal) {
  Uint32 keepGci = signal->theData[0];
  setLogTail(signal, keepGci);
  sendSignal(NDBCNTR_REF, GSN_CUT_REDO_LOG_TAIL_CONF, signal, 1, JBB);
}

void Dblqh::scanTcConnectLab(Signal *signal, Uint32 tstartTcConnect,
                             Uint32 fragId) {
  ndbrequire(getFragmentrec(fragId));
  TcConnectionrecPtr tcConnectptr;
  Uint32 next = tstartTcConnect;
  for (Uint32 i = 0; i < 200; i++) {
    bool found = getNextTcConRec(next, tcConnectptr, 10);
    if (next != RNIL && !found) {
      jam();
      i += 10;
      continue;
    } else if (next == RNIL) {
      jam();
      break;
    }
    if (tcConnectptr.p->transactionState != TcConnectionrec::IDLE) {
      switch (tcConnectptr.p->logWriteState) {
        case TcConnectionrec::NOT_WRITTEN:
          jam();
          if (fragptr.i == tcConnectptr.p->fragmentptr) {
            jam();
            fragptr.p->activeTcCounter = fragptr.p->activeTcCounter + 1;
            tcConnectptr.p->logWriteState = TcConnectionrec::NOT_WRITTEN_WAIT;
          }  // if
          break;
        default:
          jam();
          /*empty*/;
          break;
      }  // switch
    }    // if
  }      // for
  if (next < RNIL) {
    jam();
    signal->theData[0] = ZSCAN_TC_CONNECT;
    signal->theData[1] = next;
    signal->theData[2] = tabptr.i;
    signal->theData[3] = fragId;
    sendSignal(cownref, GSN_CONTINUEB, signal, 4, JBB);
  } else {
    jam();
    /*------------------------------------------------------*/
    /*       THE SCAN HAVE BEEN COMPLETED. WE CHECK IF ALL  */
    /*       OPERATIONS HAVE ALREADY BEEN COMPLETED.        */
    /*------------------------------------------------------*/
    ndbrequire(fragptr.p->activeTcCounter > 0);
    fragptr.p->activeTcCounter--;
    if (fragptr.p->activeTcCounter == 0) {
      jam();
      /*------------------------------------------------------*/
      /*       SET START GLOBAL CHECKPOINT TO THE NEXT        */
      /*       CHECKPOINT WE HAVE NOT YET HEARD ANYTHING ABOUT*/
      /*       THIS GCP WILL BE COMPLETELY COVERED BY THE LOG.*/
      /*------------------------------------------------------*/
      fragptr.p->startGci = cnewestGci + 1;
      sendCopyActiveConf(signal, tabptr.i);
    }  // if
  }    // if
  return;
}  // Dblqh::scanTcConnectLab()

/* ========================================================================= */
/* =======              INITIATE TC RECORD AT COPY FRAGMENT          ======= */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = ICT                                         */
/* ========================================================================= */
void Dblqh::initCopyTc(Signal *signal, Operation_t op,
                       TcConnectionrec *regTcPtr) {
  regTcPtr->operation = ZREAD;
  regTcPtr->opExec = 0; /* NOT INTERPRETED MODE */
  regTcPtr->schemaVersion = scanptr.p->scanSchemaVersion;
  Uint32 reqinfo = 0;
  LqhKeyReq::setDirtyFlag(reqinfo, 1);
  LqhKeyReq::setSimpleFlag(reqinfo, 1);
  LqhKeyReq::setOperation(reqinfo, op);
  LqhKeyReq::setGCIFlag(reqinfo, 1);
  LqhKeyReq::setNrCopyFlag(reqinfo, 1);
  /* AILen in LQHKEYREQ  IS ZERO */
  regTcPtr->reqinfo = reqinfo;
  /* ------------------------------------------------------------------------ */
  /* THE RECEIVING NODE WILL EXPECT THAT IT IS THE LAST NODE AND WILL         */
  /* SEND COMPLETED AS THE RESPONSE SIGNAL SINCE DIRTY_OP BIT IS SET.         */
  /* ------------------------------------------------------------------------ */
  regTcPtr->nodeAfterNext[0] = ZNIL;
  regTcPtr->nodeAfterNext[1] = ZNIL;
  regTcPtr->tcBlockref = cownref;
  regTcPtr->readlenAi = 0;
  regTcPtr->nextSeqNoReplica = 0;
  regTcPtr->dirtyOp = ZFALSE;
  regTcPtr->lastReplicaNo = 0;
  regTcPtr->currTupAiLen = 0;
  regTcPtr->tcTimer = cLqhTimeOutCount;
  regTcPtr->indTakeOver = ZFALSE;
}  // Dblqh::initCopyTc()

/* ------------------------------------------------------------------------- */
/* -------               SEND COPY_ACTIVECONF TO MASTER DIH          ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::sendCopyActiveConf(Signal *signal, Uint32 tableId) {
  releaseActiveCopy(signal);
  CopyActiveConf *const conf = (CopyActiveConf *)&signal->theData[0];
  conf->userPtr = fragptr.p->masterPtr;
  conf->tableId = tableId;
  conf->fragId = fragptr.p->fragId;
  conf->startingNodeId = cownNodeid;
  conf->startGci = fragptr.p->startGci;
  sendSignal(fragptr.p->masterBlockref, GSN_COPY_ACTIVECONF, signal,
             CopyActiveConf::SignalLength, JBB);
  if (c_copy_active_queue.isEmpty()) {
    /**
     * No queued COPY_ACTIVEREQ. Reset c_copy_active_ongoing to ensure we
     * can receive new such signals immediately.
     */
    jam();
    c_copy_active_ongoing = false;
  } else {
    jam();
    CopyActiveRecordPtr copy_activeptr;
    c_copy_active_queue.first(copy_activeptr);
    memcpy(&signal->theData[0], &copy_activeptr.p->m_copy_activereq,
           sizeof(CopyActiveReq));
    sendSignal(reference(), GSN_COPY_ACTIVEREQ, signal,
               CopyActiveReq::SignalLength, JBB);
    c_copy_active_queue.removeFirst(copy_activeptr);
  }
}  // Dblqh::sendCopyActiveConf()

/**
 * Client side of HALT/RESUME Copy fragment
 * ----------------------------------------
 * This is executed by starting node when we are getting low
 * on space in UNDO log. The aim is to avoid getting error
 * 1501 when inserting a new row in the starting node. While
 * we are halted we will execute a local LCP to ensure that
 * more space is created in the UNDO log for disk data
 * pages.
 *
 * We discover that we need to halt the execution of copy
 * fragment in the starting node. When we receive this condition
 * we might have copy fragments ongoing. The variables
 * c_fragCopyTable and c_fragCopyFrag is set to RNIL when no
 * copy fragment is ongoing. Otherwise they point to the currently
 * active fragment being copied.
 *
 * The starting node see the following flow of signals.
 *
 * 0) Before any copy started
 * Indicated by m_copy_started_state is AC_NORMAL, fragStatus is
 * FSACTIVE and m_copy_complete_flag is 0.
 *
 * 1) PREPARE_COPY_FRAGREQ
 * Sent before the live node gets the COPY_FRAGREQ signal.
 * Indicated by setting m_copy_started_state to AC_IGNORED and setting
 * fragStatus to ACTIVE_CREATION.
 *
 * 2) First LQHKEYREQ signal received (=> COPY_FRAGREQ received
 *    at live node).
 * Indicated by setting m_copy_started_state to AC_NR_COPY.
 *
 * 3) COPY_ACTIVEREQ received with CAR_NO_LOGGING and CAR_NO_WAIT set
 * Indicated by setting m_copy_started to AC_NORMAL and fragStatus to
 * FSACTIVE and m_copy_complete_flag to 1.
 *
 * If we find a copy fragment active it will be in either 1) or 2) above.
 * If all fragments are in either 0) or 3) then no active copy fragment
 * is ongoing.
 */
void Dblqh::send_halt_copy_frag(Signal *signal) {
  ndbrequire(c_undo_log_overloaded);
  ndbrequire(!c_copy_frag_halt_process_locked);
  ndbrequire(!(c_copy_frag_halted &&
               c_copy_frag_halt_state == WAIT_RESUME_COPY_FRAG_CONF));
  ndbrequire(!(!c_copy_frag_halted &&
               c_copy_frag_halt_state == WAIT_HALT_COPY_FRAG_CONF));
  ndbassert(is_copy_frag_in_progress());
  if (c_fragCopyTable == RNIL) {
    jam();
    /**
     * No active checkpoint ongoing.
     * Set c_copy_frag_halted to true and c_copy_frag_halt_state to
     * COPY_FRAG_HALT_STATE_IDLE. Will halt copy fragment when receiving
     * PREPARE_COPY_FRAGREQ.
     */
    DEB_COPY(("(%u): Halted, no active copy", instance()));
    c_copy_frag_halted = true;
    c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
    return;
  }

  tabptr.i = c_fragCopyTable;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  ndbrequire(getFragmentrec(c_fragCopyFrag));

  if (fragptr.p->m_copy_started_state == Fragrecord::AC_IGNORED) {
    jam();
    /**
     * State 1) above
     * We have received the PREPARE_COPY_FRAGREQ already, but we have
     * not yet received the first LQHKEYREQ yet. So we cannot be sure
     * that the starting node have received COPY_FRAGREQ yet. We
     * indicate that we wait for first LQHKEYREQ by setting
     * c_copy_frag_halted to true AND c_copy_frag_halt_state to
     * WAIT_FIRST_LQHKEYREQ.
     *
     * It is possible that not no first LQHKEYREQ arrives if no rows
     * are sent before the copy fragment of this fragment is
     * completed. In this case we will either see a new
     * PREPARE_COPY_FRAG_REQ arrive or we will see that the
     * first phase of copy fragment is completed.
     */
    DEB_COPY(("(%u): Halt when first LQHKEYREQ arrives, tab(%u,%u)", instance(),
              c_fragCopyTable, c_fragCopyFrag));
    c_copy_frag_halted = false;
    c_copy_frag_halt_state = COPY_FRAG_HALT_WAIT_FIRST_LQHKEYREQ;
    return;
  }
  ndbrequire(fragptr.p->m_copy_started_state == Fragrecord::AC_NR_COPY);
  jam();
  /**
   * We can be sure that the live node have received the COPY_FRAGREQ.
   * Send HALT_COPY_FRAG_REQ to live node to stop copy fragment process
   * temporarily.
   */
  DEB_COPY(("(%u): Halt copy fragment process in live node, tab(%u,%u)",
            instance(), c_fragCopyTable, c_fragCopyFrag));
  c_copy_frag_halted = false;
  c_copy_frag_halt_process_locked = true;
  c_copy_frag_halt_state = WAIT_HALT_COPY_FRAG_CONF;
  Uint32 nodeId = c_prepare_copy_fragreq_save.copyNodeId;
  Uint32 instanceKey = fragptr.p->lqhInstanceKey;
  Uint32 instanceNo = getInstanceNo(nodeId, instanceKey);
  BlockReference ref = numberToRef(DBLQH, instanceNo, nodeId);

  HaltCopyFragReq *req = (HaltCopyFragReq *)signal->getDataPtrSend();
  req->senderRef = reference();
  req->senderData = 0;
  req->tableId = fragptr.p->tabRef;
  req->fragmentId = fragptr.p->fragId;
  sendSignal(ref, GSN_HALT_COPY_FRAG_REQ, signal, HaltCopyFragReq::SignalLength,
             JBB);
}

bool Dblqh::is_copy_frag_in_progress(void) {
  if (m_second_activate_fragment_ptr_i == RNIL && c_copy_fragment_in_progress) {
    jam();
    return true;
  }
  return false;
}

void Dblqh::execHALT_COPY_FRAG_CONF(Signal *signal) {
  HaltCopyFragConf *conf = (HaltCopyFragConf *)signal->getDataPtr();
  Uint32 cause = conf->cause;
  tabptr.i = conf->tableId;
  Uint32 fragId = conf->fragmentId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  ndbrequire(getFragmentrec(fragId));
  c_copy_frag_halt_process_locked = false;
  if (cause == HaltCopyFragConf::COPY_FRAG_HALTED) {
    jam();
    ndbrequire(is_copy_frag_in_progress());
    DEB_COPY(
        ("(%u)Halted copy fragment process in live node,"
         " tab(%u,%u)",
         instance(), tabptr.i, fragId));
    c_copy_frag_halted = true;
    c_copy_frag_halt_state = COPY_FRAG_IS_HALTED;
    if (!c_undo_log_overloaded) {
      jam();
      send_resume_copy_frag(signal);
    }
    return;
  }
  ndbrequire(cause == HaltCopyFragConf::COPY_FRAG_COMPLETED);
  /**
   * The copy fragment completed before we got to it. Let's restart
   * the halt process.
   */
  DEB_COPY(
      ("(%u)Completed copy fragment process in live node"
       ", tab(%u,%u)",
       instance(), tabptr.i, fragId));
  ndbrequire(!c_copy_frag_halted);
  c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
  if (c_undo_log_overloaded && is_copy_frag_in_progress()) {
    jam();
    DEB_COPY(("(%u): Restart halt copy fragment process", instance()));
    send_halt_copy_frag(signal);
  }
  return;
}

void Dblqh::send_resume_copy_frag(Signal *signal) {
  ndbrequire(!c_undo_log_overloaded);
  ndbrequire(!c_copy_frag_halt_process_locked);
  ndbassert(is_copy_frag_in_progress());
  if (c_copy_frag_halted) {
    if (c_copy_frag_halt_state == COPY_FRAG_HALT_STATE_IDLE) {
      jam();
      /**
       * No need to do anything. We had not yet been able to halt any
       * copy fragment process. So simply continue after resetting
       * c_copy_frag_halted flag.
       */
      DEB_COPY(("(%u): Copy fragment process resumed, was idle", instance()));
      c_copy_frag_halted = false;
      return;
    } else if (c_copy_frag_halt_state == COPY_FRAG_IS_HALTED) {
      jam();
      /**
       * The live node has halted its copy fragment scan. We need to
       * resume the copy fragment scan again.
       * Only after receiving RESUME_COPY_FRAG_CONF are we able to
       * reset the halt state flag.
       */
      tabptr.i = c_fragCopyTable;
      Uint32 fragId = c_fragCopyFrag;
      ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
      ndbrequire(getFragmentrec(fragId));

      DEB_COPY(("(%u): Send RESUME_COPY_FRAG_REQ, tab(%u,%u)", instance(),
                fragptr.p->tabRef, fragptr.p->fragId));
      c_copy_frag_halt_process_locked = true;
      Uint32 nodeId = c_prepare_copy_fragreq_save.copyNodeId;
      Uint32 instanceKey = fragptr.p->lqhInstanceKey;
      Uint32 instanceNo = getInstanceNo(nodeId, instanceKey);
      BlockReference ref = numberToRef(DBLQH, instanceNo, nodeId);
      c_copy_frag_halt_state = WAIT_RESUME_COPY_FRAG_CONF;
      ResumeCopyFragReq *req = (ResumeCopyFragReq *)signal->getDataPtrSend();
      req->senderRef = reference();
      req->senderData = 0;
      req->tableId = fragptr.p->tabRef;
      req->fragmentId = fragptr.p->fragId;
      sendSignal(ref, GSN_RESUME_COPY_FRAG_REQ, signal,
                 HaltCopyFragReq::SignalLength, JBB);
      return;
    } else if (c_copy_frag_halt_state == PREPARE_COPY_FRAG_IS_HALTED) {
      jam();
      Uint32 tableId = c_prepare_copy_fragreq_save.tableId;
      Uint32 fragId = c_prepare_copy_fragreq_save.fragId;
      DEB_COPY(("(%u): Resume PREPARE_COPY_FRAGREQ, tab(%u,%u)", instance(),
                tableId, fragId));
      Uint32 max_page;
      tabptr.i = tableId;
      ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
      ndbrequire(getFragmentrec(fragId));
      Uint32 completedGci = fragptr.p->m_completed_gci;
      c_tup->get_frag_info(tableId, fragId, &max_page);
      send_prepare_copy_frag_conf(signal, c_prepare_copy_fragreq_save,
                                  completedGci, max_page);
      c_copy_frag_halted = false;
      c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
      return;
    } else {
      jamLine(Uint16(c_copy_frag_halt_state));
      ndbabort();
      return;  // Compiler silencer
    }
  } else if (c_copy_frag_halt_state == COPY_FRAG_HALT_WAIT_FIRST_LQHKEYREQ) {
    jam();
    /**
     * No need to do anything. We had not yet been able to halt any
     * copy fragment process. So simply continue after resetting to
     * an idle state again.
     */
    c_copy_frag_halted = false;
    DEB_COPY(("(%u): Resumed, was still waiting for LQHKEYREQ", instance()));
    c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
    return;
  } else {
    jamLine(c_copy_frag_halt_state);
    ndbabort();
  }
}

void Dblqh::execRESUME_COPY_FRAG_CONF(Signal *signal) {
  jamEntry();
  jamLine(Uint16(c_copy_frag_halt_state));
  c_copy_frag_halted = false;
  c_copy_frag_halt_process_locked = false;
  c_copy_frag_halt_state = COPY_FRAG_HALT_STATE_IDLE;
  DEB_COPY(("(%u) execRESUME_COPY_FRAG_CONF, tab(%u,%u)", instance(),
            c_fragCopyTable, c_fragCopyFrag));
  if (c_undo_log_overloaded && is_copy_frag_in_progress()) {
    jam();
    /**
     * UNDO log is overloaded again. We need to halt it again.
     */
    DEB_COPY(("(%u): Need to halt again", instance()));
    send_halt_copy_frag(signal);
    return;
  } else {
    jam();
    /**
     * Normal path, the UNDO isn't overloaded anymore and we have
     * resumed normal operation. We have already set the correct
     * state, so we can simply return and the copy processes will
     * continue as normal. The resume was done by the live node.
     */
    DEB_COPY(("(%u): Resumed copy fragment, tab(%u,%u)", instance(),
              c_fragCopyTable, c_fragCopyFrag));
    return;
  }
}

/**
 * LGMAN UNDO log level reporting
 * ------------------------------
 * Every time LGMAN reaches some predefined levels it will
 * send information of the level change to all LQHs. This will
 * enable LQH to do various things to minimize the risk of
 * running out of UNDO log. During a restart we can stop
 * temporarily the synchronisation process. During normal
 * operation we can increase the speed of writing local
 * checkpoints.
 */
void Dblqh::execUNDO_LOG_LEVEL_REP(Signal *signal) {
#define OVERLOAD_LEVEL 90
  UndoLogLevelRep *rep = (UndoLogLevelRep *)signal->getDataPtr();
  Uint32 levelUsed = rep->levelUsed;

  c_backup->set_undo_log_level(levelUsed);

  DEB_LOCAL_LCP_EXTRA(("(%u)UNDO_LOG_LEVEL: %u percent, copy in progress: %u",
                       instance(), levelUsed, c_copy_fragment_in_progress));
  if (c_copy_fragment_in_progress) {
    if (levelUsed >= OVERLOAD_LEVEL) {
      if (c_undo_log_overloaded) {
        jam();
        /**
         * Nothing new, we are already set to overload state, no
         * need to report it again until we have dipped into
         * normal state again for a while.
         */
      } else {
        jam();
        /**
         * We have reached an overload state. We need to
         * halt copy fragment process. In addition we also
         * need to ensure that we run a complete local LCP
         * before we can continue again since this is the
         * only method to bring down the UNDO log level.
         */
        c_undo_log_overloaded = true;
        if (!c_copy_frag_halt_process_locked) {
          jam();
          send_halt_copy_frag(signal);
        }
      }
    } else {
      if (!c_undo_log_overloaded) {
        jam();
        /**
         * Nothing new, we are already set to normal state, no
         * need to report it again until we have dipped into
         * overload state for a while.
         */
      } else {
        jam();
        c_undo_log_overloaded = false;
        if (!c_copy_frag_halt_process_locked) {
          jam();
          send_resume_copy_frag(signal);
        }
      }
    }
  }
}

/**
 * Server side of HALT/RESUME Copy fragment
 * ----------------------------------------
 * This is executed on the live node that is copying the rows to the
 * starting node.
 */
void Dblqh::execHALT_COPY_FRAG_REQ(Signal *signal) {
  HaltCopyFragReq *req = (HaltCopyFragReq *)signal->getDataPtr();
  c_halt_copy_fragreq_save = *req;

  if (c_tc_connect_rec_copy_frag == RNIL) {
    jamEntry();
    /**
     * No active copy fragment, obviously copy has been completed
     * already, we will not arrive here unless there was an active
     * copy fragment going on. So if none is active anymore it
     * means we've already completed the copy. We return immediately.
     */
    DEB_COPY(("(%u):HALT_COPY_FRAG_REQ: no active copy", instance()));
    send_halt_copy_frag_conf(signal, true);
    return;
  }
  jamEntry();
  /**
   * Active copy fragment found, we will wait for it to be fully
   * halted before responding. When successfully halted it we
   * will respond, we will also respond if not able to halt it
   * before it was completed.
   */
  DEB_COPY(("(%u):HALT_COPY_FRAG_REQ: start halting", instance()));
  c_copy_frag_live_node_performing_halt = true;
  c_copy_frag_live_node_halted = false;
}

void Dblqh::send_halt_copy_frag_conf(Signal *signal, bool completed) {
  HaltCopyFragConf *conf = (HaltCopyFragConf *)signal->getDataPtrSend();
  conf->cause = completed ? HaltCopyFragConf::COPY_FRAG_COMPLETED
                          : HaltCopyFragConf::COPY_FRAG_HALTED;

  conf->senderData = c_halt_copy_fragreq_save.senderData;
  conf->tableId = c_halt_copy_fragreq_save.tableId;
  conf->fragmentId = c_halt_copy_fragreq_save.fragmentId;
  sendSignal(c_halt_copy_fragreq_save.senderRef, GSN_HALT_COPY_FRAG_CONF,
             signal, HaltCopyFragConf::SignalLength, JBB);
}

void Dblqh::execRESUME_COPY_FRAG_REQ(Signal *signal) {
  jamEntry();
  ndbrequire(c_copy_frag_live_node_halted);
  ndbrequire(!c_copy_frag_live_node_performing_halt);
  ndbrequire(c_tc_connect_rec_copy_frag != RNIL);
  c_copy_frag_live_node_halted = false;
  DEB_COPY(("(%u):RESUME_COPY_FRAG_REQ received", instance()));

  send_resume_copy_frag_conf(signal);

  /**
   * Resume copy fragment process by reissuing nextRecordCopy
   */
  TcConnectionrecPtr tcConnectptr;
  tcConnectptr.i = c_tc_connect_rec_copy_frag;
  ndbrequire(tcConnect_pool.getValidPtr(tcConnectptr));
  setup_scan_pointers_from_tc_con(tcConnectptr, __LINE__);
  ndbrequire(tcConnectptr.p->copyCountWords == 0);
  ndbrequire(scanptr.p->scanState == ScanRecord::COPY_FRAG_HALTED);
  scanptr.p->scanState = ScanRecord::WAIT_LQHKEY_COPY;
  nextRecordCopy(signal, tcConnectptr);
  release_frag_access(prim_tab_fragptr.p);
}

void Dblqh::send_resume_copy_frag_conf(Signal *signal) {
  ResumeCopyFragReq req = *(ResumeCopyFragReq *)signal->getDataPtr();
  ResumeCopyFragConf *conf = (ResumeCopyFragConf *)signal->getDataPtrSend();
  conf->senderData = req.senderData;
  conf->tableId = req.tableId;
  conf->fragmentId = req.fragmentId;
  sendSignal(req.senderRef, GSN_RESUME_COPY_FRAG_CONF, signal,
             ResumeCopyFragConf::SignalLength, JBB);
}

/* ##########################################################################
 * #######                       LOCAL CHECKPOINT MODULE              #######
 *
 * ##########################################################################
 * --------------------------------------------------------------------------
 *  THIS MODULE HANDLES THE EXECUTION AND CONTROL OF LOCAL CHECKPOINTS
 *  IT CONTROLS THE LOCAL CHECKPOINTS IN TUP AND ACC. IT DOES ALSO INTERACT
 *  WITH DIH TO CONTROL WHICH GLOBAL CHECKPOINTS THAT ARE RECOVERABLE
 *
 * We can prepare a fragment checkpoint while we are executing another
 * fragment checkpoint. The reason for this is to make sure that we have
 * quick progress even with many small fragments.
 *
 * Preparing a fragment for checkpoint execution means opening a header file
 * for the fragment and then opening a new file to contain the data from this
 * checkpoint. To perform a restore one might have to execute several
 * checkpoints from the oldest to the newest. How to perform recovery is
 * found in the fragment checkpoint header file.
 *
 * There is also a background process after completing the fragment checkpoint
 * performed by the BACKUP block. This background process will delete old
 * checkpoint files to ensure that we don't run out of file space. This
 * process might be interrupted by a crash, it will however be completed
 * next time the fragment is checkpointed.
 * ------------------------------------------------------------------------- */

void Dblqh::force_lcp(Signal *signal) {
  /* If there is a system or node restart in progress,
   * request an lcp to be triggered when the restart completes
   * without waiting for transaction load or
   * expiry of TimeBetweenLocalCheckpoints,
   * in order to reduce the redo log handling during any
   * potential multi-node crashes and ensure the recoverability.
   */
  if (!getNodeState().getSystemRestartInProgress() &&
      !getNodeState().getNodeRestartInProgress() &&
      cLqhTimeOutCount == c_last_force_lcp_time) {
    jam();
    return;
  }

  c_last_force_lcp_time = cLqhTimeOutCount;
  signal->theData[0] = DumpStateOrd::DihStartLcpImmediately;
  sendSignal(DBDIH_REF, GSN_DUMP_STATE_ORD, signal, 1, JBB);
}

/**
 * At this point DBDIH master is about to start the next
 * distributed LCP. At this point DIH has locked the
 * meta data, so no table creations are in process at
 * this time. So all tables that exists now will be part of
 * this LCP. New tables created after this point will show
 * up here with LQHFRAGREQ signals and for each fragment
 * the LCP will execute it will call lcp_max_completed_gci.
 * So we set it to 0 here to indicate the minimum is not yet
 * set. After that we will call the set_min_keep_gci
 * function each time we come to LQHFRAGREQ and coming to
 * lcp_max_completed_gci.
 *
 * We only receive this signal when we are participating in
 * the distributed LCP to avoid messing things up for local LCP
 * execution.
 *
 * We keep the gci - 1 from here as well just to verify that
 * the keepGci isn't set before this GCI, this would indicate
 * some severe problem of our understanding of the code.
 */
void Dblqh::execSTART_NODE_LCP_REQ(Signal *signal) {
  jamEntry();
#ifdef DEBUG_LCP
  Uint32 current_gci = signal->theData[0];
  Uint32 backup_restorable_gci = c_backup->getRestorableGci();
#endif
  Uint32 restorable_gci = signal->theData[1];
  c_keep_gci_for_lcp = restorable_gci;
  DEB_LCP(
      ("(%u)c_keep_gci_for_lcp = %u,"
       " current_gci = %u, restorable_gci = %u"
       ", cnewestCompletedGci = %u, "
       "backup_restorable_gci = %u",
       instance(), c_keep_gci_for_lcp, current_gci, restorable_gci,
       cnewestCompletedGci, backup_restorable_gci));
  c_max_keep_gci_in_lcp = c_keep_gci_for_lcp;
  c_first_set_min_keep_gci = true;
  BlockReference ref;
  if (isNdbMtLqh()) {
    jam();
    ref = DBLQH_REF;
  } else {
    jam();
    ref = DBDIH_REF;
  }
  signal->theData[0] = 1;
  sendSignal(ref, GSN_START_NODE_LCP_CONF, signal, 1, JBB);

  if (getNodeState().startLevel >= NodeState::SL_STOPPING_4) {
    /**
     * The restorable_gci is not restorable in our node,
     * so don't update Backup's view of restorable GCI
     * at this time since that would create an LCP that
     * isn't restorable.
     *
     * By not updating the restorable GCI in Backup we
     * ensure that the LCP won't complete if any updates
     * have occurred in the LCP. Thus we don't risk that
     * we overwrite all restorable LCP files.
     */
    jam();
    return;
  }
  if (cstartPhase != ZNIL) {
    jam();
    /**
     * The node is not yet complete with its restart.
     * So we cannot yet guarantee that the restorable
     * GCI is restorable in this node even if it is
     * restorable in the cluster.
     */
    return;
  }
  jam();
  signal->theData[0] = restorable_gci;
  EXECUTE_DIRECT(getBACKUP(), GSN_RESTORABLE_GCI_REP, signal, 1);
}

void Dblqh::set_min_keep_gci(Uint32 max_completed_gci) {
  if (c_first_set_min_keep_gci) {
    jam();
    c_first_set_min_keep_gci = false;
    c_max_keep_gci_in_lcp = max_completed_gci;
    DEB_LCP(("(%u)First: c_max_keep_gci_in_lcp = %u", instance(),
             c_max_keep_gci_in_lcp));
  } else if (c_max_keep_gci_in_lcp > max_completed_gci) {
    jam();
    c_max_keep_gci_in_lcp = max_completed_gci;
    DEB_LCP(
        ("(%u)c_max_keep_gci_in_lcp = %u", instance(), c_max_keep_gci_in_lcp));
  }
}

void Dblqh::execLCP_FRAG_ORD(Signal *signal) {
  jamEntry();
  CRASH_INSERTION(5010);

  LcpFragOrd lcpFragOrdCopy = *(LcpFragOrd *)&signal->theData[0];
  LcpFragOrd *lcpFragOrd = &lcpFragOrdCopy;

  Uint32 lcpId = lcpFragOrd->lcpId;

  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);

  if (c_lcpId != lcpFragOrd->lcpId) {
    jam();

    lcpPtr.p->firstFragmentFlag = true;
    c_max_gci_in_lcp = 0;
    c_fragments_in_lcp = 0;

#ifdef ERROR_INSERT
    if (check_ndb_versions()) {
      /**
       * Only (so-far) in error insert
       *   check that keepGci (tail of REDO) is smaller than of head of REDO
       *
       */
      if (!((cnewestCompletedGci >= lcpFragOrd->keepGci) &&
            (cnewestGci >= lcpFragOrd->keepGci))) {
        g_eventLogger->info(
            "lcpFragOrd->keepGci: %u cnewestCompletedGci: %u cnewestGci: %u",
            lcpFragOrd->keepGci, cnewestCompletedGci, cnewestGci);
      }
      ndbrequire(cnewestCompletedGci >= lcpFragOrd->keepGci);
      ndbrequire(cnewestGci >= lcpFragOrd->keepGci);
    }
#endif

    c_lcpId = lcpFragOrd->lcpId;
    ndbrequire(is_lcp_idle(lcpPtr.p));
    if (signal->getSendersBlockRef() != reference()) {
      jam();
      if (c_lcpId > m_curr_lcp_id) {
        jam();
        m_curr_lcp_id = c_lcpId;
        m_curr_local_lcp_id = 0;
      } else {
        m_curr_local_lcp_id++;
        DEB_LCP(
            ("(%u)Starting another distributed LCP with same id,"
             " stepping up local LCP id, LCP(%u,%u)",
             instance(), m_curr_lcp_id, m_curr_local_lcp_id));
        ndbrequire(c_lcpId == m_curr_lcp_id);
      }
      g_eventLogger->debug("(%u)Starting distributed LCP(%u,%u)", instance(),
                           m_curr_lcp_id, m_curr_local_lcp_id);
      m_first_distributed_lcp_started = true;
      signal->theData[0] = c_lcpId;
      sendSignal(NDBCNTR_REF, GSN_START_DISTRIBUTED_LCP_ORD, signal, 1, JBB);
    } else {
      g_eventLogger->info("(%u)Starting local LCP(%u,%u)", instance(),
                          m_curr_lcp_id, m_curr_local_lcp_id);
      ndbrequire(lcpFragOrd->keepGci == 0);
    }
    ndbrequire(clcpCompletedState == LCP_IDLE);
    clcpCompletedState = LCP_RUNNING;

    /**
     * We preset some variables that will stay the same for the entire
     * LCP execution.
     */
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId = c_lcpId;
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.keepGci = lcpFragOrd->keepGci;
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.lastFragmentFlag = false;
    /* These should be set before each LCP fragment execution */
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.tableId = RNIL;
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.fragmentId = RNIL;
    lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpNo = RNIL;

    if (!c_queued_lcp_frag_ord.isEmpty()) {
      jam();
      lcpPtr.p->m_early_lcps_need_synch = true;
    } else {
      jam();
      lcpPtr.p->m_early_lcps_need_synch = false;
    }
    ndbrequire(lcpPtr.p->m_wait_early_lcp_synch == false);
    {
      Logfile_client lgman(this, c_lgman, 0);
      if (lgman.exists_logfile_group()) {
        jam();
        LcpFragOrd *ord = (LcpFragOrd *)signal->getDataPtr();
        ord->tableId = 0;
        ord->fragmentId = 0;
        ord->lcpId = c_lcpId;
        lgman.exec_lcp_frag_ord(signal, get_current_local_lcp_id());
      }
    }
  } else {
    jam();
    ndbrequire(c_lcpId == lcpFragOrd->lcpId);
    if (lcpPtr.p->lastFragmentFlag ||
        (clcpCompletedState == LCP_IDLE &&
         ((lcpFragOrd->lcpId < c_lcpId_sent_last_LCP_FRAG_ORD) ||
          (lcpFragOrd->lcpId == c_lcpId_sent_last_LCP_FRAG_ORD &&
           c_localLcpId == 0) ||
          (lcpFragOrd->lcpId == c_lcpId_sent_last_LCP_FRAG_ORD &&
           c_localLcpId != 0 &&
           c_localLcpId <= c_localLcpId_sent_last_LCP_FRAG_ORD)))) {
      jam();
      /**
       * Drop any message received after LCP_FRAG_ORD with last fragment
       * marker, must be ndbd we're running since Proxy should handle this.
       * Can happen after a master takeover. Can also happen with empty
       * LDMs.
       *
       * If this is the first LCP_FRAG_ORD with last flag AND the LCP
       * state is IDLE we still need to verify that this isn't an empty
       * LCP. We do this by checking that we haven't processed this
       * LCP id yet.
       *
       * DIH doesn't keep track of number of outstanding messages, so
       * no need to do anything when receiving multiple LCP_FRAG_ORDs
       * that are discarded.
       *
       * Can happen with LDM threads with no fragments to restore even for
       * ndbmtd.
       *
       * If last flag is already set on LCP we will immediately drop it.
       * If LCP state is IDLE we will drop it if either of the following
       * statements are true.
       *
       * 1) LCP id is lower than sent last on before
       * 2) LCP id is equal to what sent before AND it is distributed LCP
       * 3) LCP is equal to what sent before AND it is local LCP and
       *    local LCP is lower than or equal to what sent before
       */
      DEB_EMPTY_LCP(
          ("(%u) Drop LastFragment LCP_FRAG_ORD"
           ", flag: %u, clcpCompletedState = %u"
           ", sent_last_LCP_FRAG_ORD(%u,%u), LCP(%u,%u)",
           instance(), lcpPtr.p->lastFragmentFlag, clcpCompletedState,
           c_lcpId_sent_last_LCP_FRAG_ORD, c_localLcpId_sent_last_LCP_FRAG_ORD,
           lcpFragOrd->lcpId, c_localLcpId));
      return;
    }
  }

  if (lcpFragOrd->lastFragmentFlag) {
    jam();
    lcpPtr.p->lastFragmentFlag = true;
    c_lcpId_sent_last_LCP_FRAG_ORD = lcpFragOrd->lcpId;
    c_localLcpId_sent_last_LCP_FRAG_ORD = c_localLcpId;
    DEB_LAST_LCP(
        ("(%u)Received last fragment flag"
         " set sent_last_LCP_FRAG_ORD to (%u,%u)",
         instance(), c_lcpId_sent_last_LCP_FRAG_ORD,
         c_localLcpId_sent_last_LCP_FRAG_ORD));
    CRASH_INSERTION(5054);
    if (is_lcp_idle(lcpPtr.p)) {
      jam();
      /* ----------------------------------------------------------
       *       NOW THE COMPLETE LOCAL CHECKPOINT ROUND IS COMPLETED.
       * -------------------------------------------------------- */
      completeLcpRoundLab(signal, lcpId);
    }
    return;
  }  // if

  c_fragments_in_lcp++;
  tabptr.i = lcpFragOrd->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  if (unlikely(tabptr.p->tableStatus != Tablerec::TABLE_DEFINED &&
               tabptr.p->tableStatus != Tablerec::TABLE_READ_ONLY)) {
    /**
     * There is no way to discover if we had multiple messages for this
     * since the table is already deleted and we don't keep information
     * about it anymore. Should not be a problem since the signal is
     * likely to be dropped somewhere and an extra LCP_FRAG_REP to a
     * dropped table will simply be dropped again in DBDIH.
     */
    jam();
    ndbrequire(c_localLcpId == 0);
    LcpRecord::FragOrd fragOrd;
    fragOrd.lcpFragOrd = *lcpFragOrd;

    Fragrecord tmp;
    tmp.maxGciInLcp = cnewestGci;
    tmp.maxGciCompletedInLcp = cnewestCompletedGci;
    sendLCP_FRAG_REP(signal, fragOrd, &tmp);
    return;
  }

  ndbrequire(getFragmentrec(lcpFragOrd->fragmentId));
  bool add_to_queue = true;
  if (fragptr.p->m_create_table_flag_lcp_frag_ord) {
    jam();
    fragptr.p->m_create_table_flag_lcp_frag_ord = false;
    ndbrequire(fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_QUEUED ||
               fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_EXECUTING);
    if (!is_lcp_idle(lcpPtr.p)) {
      jam();
      /**
       * Checkpoint has already started, we only need to record the fact that
       * this table has received an order to execute fragment as part of LCP.
       *
       * LCP_QUEUED:
       * -----------
       * Fragment have been inserted into the LCP list already. But it is still
       * in the queue. We will remove it from queue and insert it as for a
       * normal fragment and also reset the variables from the CREATE TABLE
       * insert. LCP_EXECUTING:
       * --------------
       * We are already in the process of executing this checkpoint.
       * We cannot have received any LCP_FRAG_ORD previous to this
       * since any such signal would clear the m_create_table_flag_lcp_frag_ord
       * flag. Thus we only need to update the fragment record to ensure
       * it contains the necessary information received in the LCP_FRAG_ORD
       * signal.
       */
      jam();
      fragptr.p->lcp_frag_ord_lcp_no = lcpFragOrd->lcpNo;
      fragptr.p->lcp_frag_ord_lcp_id = lcpFragOrd->lcpId;
      cnoOfFragsCheckpointed++;
      c_fragments_in_lcp++;
      DEB_EARLY_LCP(("(%u)LCP_FRAG_ORD for tab(%u,%u), clear flag", instance(),
                     fragptr.p->tabRef, fragptr.p->fragId));
      return;
    }
    DEB_EARLY_LCP(("(%u)LCP_FRAG_ORD for tab(%u,%u), clear flag and start",
                   instance(), fragptr.p->tabRef, fragptr.p->fragId));
    ndbrequire(fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_QUEUED);
    /**
     * We will insert it into LCP queue below, so we need to remove it
     * here to avoid inserting it twice.
     */
    add_to_queue = false;
  }
  if (fragptr.p->lcp_frag_ord_state ==
      Fragrecord::LCP_EXECUTED_BY_CREATE_TABLE) {
    jam();
    if (fragptr.p->lcp_frag_ord_lcp_no == lcpFragOrd->lcpNo &&
        fragptr.p->lcp_frag_ord_lcp_id == lcpFragOrd->lcpId) {
      /**
       * We already executed a checkpoint as part of this LCP. Send
       * an LCP_FRAG_REP based on this information.
       */
      jam();
      DEB_EARLY_LCP(("(%u)LCP_FRAG_ORD for tab(%u,%u), already executed",
                     instance(), fragptr.p->tabRef, fragptr.p->fragId));
      LcpRecord::FragOrd fragOrd;
      fragOrd.lcpFragOrd.lcpNo = fragptr.p->lcp_frag_ord_lcp_no;
      fragOrd.lcpFragOrd.lcpId = fragptr.p->lcp_frag_ord_lcp_id;
      fragOrd.lcpFragOrd.fragmentId = fragptr.p->fragId;
      fragOrd.lcpFragOrd.tableId = fragptr.p->tabRef;
      fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTED;
      sendLCP_FRAG_REP(signal, fragOrd, fragptr.p);
      ndbrequire(lcpPtr.p->lastFragmentFlag == false);
      c_fragments_in_lcp++;
      return;
    }
    DEB_EARLY_LCP(("(%u)LCP_FRAG_ORD for tab(%u,%u), executed, new LCP",
                   instance(), fragptr.p->tabRef, fragptr.p->fragId));
    /**
     * This is the next LCP, continue as normal and set the state to the
     * expected state at this point.
     */
    fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTED;
    ndbrequire(fragptr.p->m_create_table_flag_lcp_frag_ord == false);
    ndbrequire(fragptr.p->m_create_table_insert_lcp == false);
  }

  if (fragptr.p->lcp_frag_ord_lcp_id == lcpFragOrd->lcpId &&
      c_localLcpId == 0) {
    /**
     * The LCP_FRAG_ORD have already been received, we need to send a report
     * back to the Proxy for ndbmtd to keep the outstanding counter up-to-date.
     * For ndbd we can simply drop the signal.
     */
    jam();
    if (!isNdbMtLqh()) {
      jam();
      return;
    }
    /**
     * This signal is identified by its length, it will be used to decrease
     * the number of outstanding LCP_FRAG_ORD operations to the LQH instances.
     * From a modular point of view DBLQH will always drop this LCP_FRAG_ORD
     * since it already received it. In the case of ndbmtd we do however need
     * to ensure that DBLQH proxy also knows about the drop since it keeps
     * track of the number of outstanding LCP_FRAG_ORDs. When DBLQH proxy
     * receives this signal it will update the counter and then drop the
     * signal. So no signal will be sent to DBDIH in any case.
     */
    sendSignal(DBLQH_REF, GSN_LCP_FRAG_REP, signal, 1, JBA);
    return;
  }

  /**
   * Add the fragment to the queue of LCP_FRAG_ORDs.
   * We need to store lcpId as a flag that we received an
   * LCP_FRAG_ORD for this LCP, we need the lcpNo for
   * later when executing the LCP and we need the state
   * to indicate if we have completed the LCP yet which
   * is needed for drop table.
   */
  fragptr.p->lcp_frag_ord_lcp_no = lcpFragOrd->lcpNo;
  fragptr.p->lcp_frag_ord_lcp_id = lcpFragOrd->lcpId;
  fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_QUEUED;
  if (add_to_queue) {
    jam();
    c_queued_lcp_frag_ord.addLast(fragptr);
  }
  cnoOfFragsCheckpointed++;

  if (lcpPtr.p->lcpPrepareState != LcpRecord::LCP_IDLE) {
    jam();
    return;
  }  // if
  prepare_next_fragment_checkpoint(signal, false);
}  // Dblqh::execLCP_FRAGORD()

void Dblqh::handleFirstFragment(Signal *signal) {
  if (lcpPtr.p->firstFragmentFlag) {
    jam();
    LcpFragOrd *ord = (LcpFragOrd *)signal->getDataPtrSend();
    lcpPtr.p->firstFragmentFlag = false;

    if (!isNdbMtLqh()) {
      /**
       * First fragment mean that last LCP is complete :-)
       */
      jam();
      *ord = lcpPtr.p->currentPrepareFragment.lcpFragOrd;
      EXECUTE_DIRECT_MT(TSMAN, GSN_LCP_FRAG_ORD, signal, signal->length(), 0);
      jamEntry();
    } else {
      /**
       * Handle by LqhProxy
       */
    }
  }
}

void Dblqh::execLCP_PREPARE_REF(Signal *signal) {
  jamEntry();

  LcpPrepareRef *ref = (LcpPrepareRef *)signal->getDataPtr();

  lcpPtr.i = ref->senderData;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpPrepareState == LcpRecord::LCP_PREPARING);

  fragptr.i = lcpPtr.p->currentPrepareFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);

  ndbrequire(ref->tableId == fragptr.p->tabRef);
  ndbrequire(ref->fragmentId == fragptr.p->fragId);

  tabptr.i = ref->tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

  ndbrequire(refToMain(signal->getSendersBlockRef()) == getBACKUP());

  /**
   * Only table no longer present is acceptable - anything
   * else is a hard error.
   * This sometimes manifests as error 785 - 'Schema object is busy with
   * another...' which we treat in the same way. This happens when the table
   * is dropping when we ask for the table information. So both are symptoms
   * of a table which is being dropped or already been dropped.
   */
  if (ref->errorCode != GetTabInfoRef::TableNotDefined &&
      ref->errorCode != DropTableRef::ActiveSchemaTrans) {
    g_eventLogger->critical("Fatal : LCP_PREPARE_REF t%uf%u errorCode %u",
                            ref->tableId, ref->fragmentId, ref->errorCode);
    ndbabort();
    return;
  };
  ndbrequire(c_localLcpId == 0);

  handleFirstFragment(signal);
  /* Carry on with the next table... */
  lcpPtr.p->lcpPrepareState = LcpRecord::LCP_COMPLETED;
  if (lcpPtr.p->lcpRunState == LcpRecord::LCP_IDLE ||
      lcpPtr.p->lcpRunState == LcpRecord::LCP_COMPLETED) {
    jam();
    /**
     * Our LCP prepare was the only outstanding LCP action. So currently
     * no LCP to watch. We will stop it, if there are waiting fragments to
     * prepare for LCP then we will start watchdog again.
     */
    stopLcpFragWatchdog();
    lcpPtr.p->lcpRunState = LcpRecord::LCP_IDLE;
  }
  completed_fragment_checkpoint(signal, lcpPtr.p->currentPrepareFragment);
  prepare_next_fragment_checkpoint(signal, true);
}

void Dblqh::execLCP_PREPARE_CONF(Signal *signal) {
  jamEntry();

  LcpPrepareConf *conf = (LcpPrepareConf *)signal->getDataPtr();

  ndbrequire(refToMain(signal->getSendersBlockRef()) == getBACKUP());

  lcpPtr.i = conf->senderData;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpPrepareState == LcpRecord::LCP_PREPARING);

  fragptr.i = lcpPtr.p->currentPrepareFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);

  ndbrequire(conf->tableId == fragptr.p->tabRef);
  ndbrequire(conf->fragmentId == fragptr.p->fragId);

  handleFirstFragment(signal);
  lcpPtr.p->lcpPrepareState = LcpRecord::LCP_PREPARED;
  if (lcpPtr.p->lcpRunState == LcpRecord::LCP_COMPLETED ||
      lcpPtr.p->lcpRunState == LcpRecord::LCP_IDLE) {
    /**
     * No fragment was currently performing checkpoint, we can start
     * immediately, in most cases we will start when the current
     * fragment checkpoint is completed.
     * We can also start preparing the next fragment immediately.
     */
    jam();
    lcpPtr.p->currentRunFragment = lcpPtr.p->currentPrepareFragment;
    perform_fragment_checkpoint(signal);
    prepare_next_fragment_checkpoint(signal, false);
  }
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
}

#ifdef NDB_DEBUG_FULL
static struct TraceLCP {
  void sendSignal(Uint32 ref, Uint32 gsn, Signal *signal, Uint32 len,
                  Uint32 prio);
  void save(Signal *);
  void restore(SimulatedBlock &, Signal *sig);
  struct Sig {
    enum { Sig_save = 0, Sig_send = 1 } type;
    SignalHeader header;
    Uint32 theData[25];
  };
  Vector<Sig> m_signals;
} g_trace_lcp;
template class Vector<TraceLCP::Sig>;
#endif

void Dblqh::perform_fragment_checkpoint(Signal *signal) {
  ndbassert(!m_is_query_block);
  lcpPtr.p->lcpRunState = LcpRecord::LCP_CHECKPOINTING;

  fragptr.i = lcpPtr.p->currentRunFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);

  /* ----------------------------------------------------------------------
   *    UPDATE THE MAX_GCI_IN_LCP AND MAX_GCI_COMPLETED_IN_LCP NOW BEFORE
   *    ACTIVATING THE FRAGMENT AGAIN.
   * --------------------------------------------------------------------- */
  ndbrequire(lcpPtr.p->currentRunFragment.lcpFragOrd.lcpNo < MAX_LCP_STORED);

  BackupFragmentReq *req = (BackupFragmentReq *)signal->getDataPtr();
  req->tableId = lcpPtr.p->currentRunFragment.lcpFragOrd.tableId;
  req->fragmentNo = 0;
  req->backupPtr = m_backup_ptr;
  req->backupId = lcpPtr.p->currentRunFragment.lcpFragOrd.lcpId;
  req->count = 0;
  req->senderRef = reference();

#ifdef NDB_DEBUG_FULL
  if (ERROR_INSERTED(5904)) {
    g_trace_lcp.sendSignal(BACKUP_REF, GSN_BACKUP_FRAGMENT_REQ, signal,
                           BackupFragmentReq::SignalLength, JBA);
  } else
#endif
  {
    if (ERROR_INSERTED(5044) &&
        (fragptr.p->tabRef == c_error_insert_table_id) &&
        fragptr.p->fragId)  // Not first frag
    {
      /**
       * Force CRASH_INSERTION in 10s
       */
      g_eventLogger->info("table: %d frag: %d", fragptr.p->tabRef,
                          fragptr.p->fragId);
      SET_ERROR_INSERT_VALUE(5027);
      sendSignalWithDelay(reference(), GSN_START_RECREQ, signal, 10000, 1);
    }
    {
      BlockReference backupRef = calcInstanceBlockRef(getBACKUP());
      sendSignal(backupRef, GSN_BACKUP_FRAGMENT_REQ, signal,
                 BackupFragmentReq::SignalLength, JBA);
    }
  }
}

void Dblqh::execBACKUP_FRAGMENT_REF(Signal *signal) {
  BackupFragmentRef *ref = (BackupFragmentRef *)signal->getDataPtr();

  if (ref->errorCode != GetTabInfoRef::TableNotDefined &&
      ref->errorCode != DropTableRef::ActiveSchemaTrans) {
    jam();
    BackupFragmentRef *ref = (BackupFragmentRef *)signal->getDataPtr();
    char buf[100];
    BaseString::snprintf(buf, sizeof(buf),
                         "Unable to store fragment during LCP. NDBFS Error: %u",
                         ref->errorCode);

    progError(__LINE__,
              (ref->errorCode & FsRef::FS_ERR_BIT) ? NDBD_EXIT_AFS_UNKNOWN
                                                   : ref->errorCode,
              buf);
  }
  /**
   * Handle dropped tables in the middle of a multi-file fragment LCP.
   */
  jam();
  ndbrequire(c_localLcpId == 0);
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);

  Uint32 backupId = ref->backupId;
  Uint32 backupPtr = ref->backupPtr;
  BackupFragmentConf *conf = (BackupFragmentConf *)signal->getDataPtrSend();
  conf->backupId = backupId;
  conf->backupPtr = backupPtr;
  conf->tableId = lcpPtr.p->currentRunFragment.lcpFragOrd.tableId;
  conf->fragmentNo = 0;
  conf->noOfRecordsLow = 0;
  conf->noOfRecordsHigh = 0;
  conf->noOfBytesLow = 0;
  conf->noOfBytesHigh = 0;
  execBACKUP_FRAGMENT_CONF(signal);
}

bool Dblqh::is_disk_columns_in_table(Uint32 tableId) {
  return c_tup->is_disk_columns_in_table(tableId);
}

Uint32 Dblqh::get_current_local_lcp_id(void) { return m_curr_local_lcp_id; }

void Dblqh::get_lcp_frag_stats(Uint64 &row_count, Uint64 &prev_row_count,
                               Uint64 &row_change_count,
                               Uint64 &memory_used_in_bytes,
                               Uint32 &max_page_cnt) {
  /**
   * Now the LCP is actually starting, we set the maxGciInLcp and
   * maxGciCompletedInLcp at this point and we get the row count
   * change row counts to calculate various values for the LCP.
   */
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
  fragptr.i = lcpPtr.p->currentRunFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);
  fragptr.p->maxGciInLcp = fragptr.p->newestGci;
  /**
   * Calculating MaxGciCompleted is straightforward when we are
   * performing a fragment LCP during normal operation and it is
   * also straightforward when performed during system restart.
   * In both those cases it is simply set to cnewestCompletedGci.
   *
   * However in node restarts there are different situations to cater
   * for.
   * 1) If the copy fragment haven't been completed yet then it should
   *    set to the GCI we were able to restore in this node.
   *    However if we were restoring an LCP that was created during a
   *    node restart whereafter the node crashed before completing an
   *    LCP where REDO log was active, in that case the MaxGciCompleted
   *    is the same as the one we restored.
   *
   *    We need the following information.
   *    GCI we restored from the REDO log (crestartNewestGci)
   *    MaxGciCompleted of this fragment when restored
   *      (fragptr.p->m_completed_gci)
   *    We set it to the maximum of those two.
   *
   * 2) If the copy fragment have been completed when we come here to
   *    start the fragment LCP in that case we can also set
   *    MaxGciCompleted to cnewestCompletedGci at the start of the
   *    fragment LCP.
   *
   * 3) If we find that RESTORE restored a newer version than
   *    cnewestCompletedGci (can happen when RESTORE used an LCP
   *    created during copy phase of restart) than we use this
   *    GCI rather than the cnewestCompletedGci that represents
   *    the GCI that DIH supposed we could restore (in reality
   *    we are only able to restore using the other live node
   *    in this state).
   */
  if ((!fragptr.p->m_copy_complete_flag) &&
      (cstartType == NodeState::ST_NODE_RESTART ||
       cstartType == NodeState::ST_INITIAL_NODE_RESTART)) {
    if (crestartNewestGci > fragptr.p->m_completed_gci) {
      jam();
      fragptr.p->maxGciCompletedInLcp = crestartNewestGci;
    } else {
      jam();
      fragptr.p->maxGciCompletedInLcp = fragptr.p->m_completed_gci;
    }
  } else if (cstartType == NodeState::ST_NODE_RESTART &&
             fragptr.p->m_completed_gci > cnewestCompletedGci) {
    jam();
    ndbrequire(c_local_sysfile.m_node_restorable_on_its_own ==
               ReadLocalSysfileReq::NODE_NOT_RESTORABLE_ON_ITS_OWN);
    fragptr.p->maxGciCompletedInLcp = fragptr.p->m_completed_gci;
  } else {
    jam();
    fragptr.p->maxGciCompletedInLcp = cnewestCompletedGci;
  }
  c_tup->get_lcp_frag_stats(fragptr.p->tupFragptr, fragptr.p->newestGci,
                            max_page_cnt, row_count, prev_row_count,
                            row_change_count, memory_used_in_bytes, true);
}

Uint32 Dblqh::get_lcp_newest_gci(void) {
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
  fragptr.i = lcpPtr.p->currentRunFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);
  return fragptr.p->newestGci;
}

void Dblqh::lcp_complete_scan(Uint32 &newestGci) {
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
  fragptr.i = lcpPtr.p->currentRunFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);
  /**
   * Update maxGciInLcp after scan has been performed
   */
#if defined VM_TRACE || defined ERROR_INSERT
  if (fragptr.p->newestGci != fragptr.p->maxGciInLcp) {
    DEB_LCP(("tab(%u,%u) increasing maxGciInLcp from %u to %u",
             fragptr.p->tabRef, fragptr.p->fragId, fragptr.p->maxGciInLcp,
             fragptr.p->newestGci));
  }
#endif
  newestGci = fragptr.p->newestGci;
  fragptr.p->maxGciInLcp = fragptr.p->newestGci;
  if (fragptr.p->newestGci > c_max_gci_in_lcp) {
    jam();
    c_max_gci_in_lcp = fragptr.p->newestGci;
    DEB_LCP(("(%u)New c_max_gci_in_lcp = %u", instance(), c_max_gci_in_lcp));
  }
  DEB_LCP(("(%u)complete_scan: newestGci = %u, tab(%u,%u)", instance(),
           newestGci, fragptr.p->tabRef, fragptr.p->fragId));
}

void Dblqh::lcp_max_completed_gci(Uint32 &completedGci, Uint32 max_gci_written,
                                  Uint32 restorable_gci) {
  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
  fragptr.i = lcpPtr.p->currentRunFragment.fragPtrI;
  c_fragment_pool.getPtr(fragptr);

  if (max_gci_written <= restorable_gci &&
      fragptr.p->maxGciCompletedInLcp > restorable_gci) {
    jam();
    /**
     * In this case we haven't written any transactions in the LCP
     * that isn't restorable at this point in time. So the LCP
     * is already restorable. We will only record a
     * MaxGciCompleted that is at most the restorable GCI.
     *
     * The only repercussion of this decision is that we might need
     * to execute one extra GCI in the REDO log for a fragment that
     * we know won't have any writes there. So should be of no
     * concern at all.
     *
     * It is important to record the changed maxGciCompletedInLcp
     * to ensure that DIH doesn't record a higher MaxGciCompleted
     * than we record in the local files.
     *
     * This also simplifies the recovery.
     */
    fragptr.p->maxGciCompletedInLcp = restorable_gci;
  }
  if (fragptr.p->maxGciCompletedInLcp < c_keep_gci_for_lcp) {
    jam();
    /**
     * maxGciCompletedInLcp can never be smaller than the restorable GCI
     * at the time when we start the LCP.
     */
    fragptr.p->maxGciCompletedInLcp = c_keep_gci_for_lcp;
  }

  completedGci = fragptr.p->maxGciCompletedInLcp;
  DEB_LCP(("(%u)maxGciCompletedInLcp = %u, tab(%u,%u)", instance(),
           completedGci, fragptr.p->tabRef, fragptr.p->fragId));
  set_min_keep_gci(fragptr.p->maxGciCompletedInLcp);
}

void Dblqh::execBACKUP_FRAGMENT_CONF(Signal *signal) {
  jamEntry();

  if (ERROR_INSERTED(5073) || ERROR_INSERTED(5109)) {
    g_eventLogger->info("Delaying BACKUP_FRAGMENT_CONF");
    sendSignalWithDelay(reference(), GSN_BACKUP_FRAGMENT_CONF, signal, 500,
                        signal->getLength());
    /**
     * If an ALTER table is ongoing the error 5109 is removed to makes tup to
     * commit the alter table
     */
    if (ERROR_INSERTED(5109) && handleLCPSurfacing(signal)) {
      CLEAR_ERROR_INSERT_VALUE;
    }
    return;
  }

  BackupFragmentConf *conf = (BackupFragmentConf *)signal->getDataPtr();
  Uint64 noOfRecordsLow = conf->noOfRecordsLow;
  Uint64 noOfRecordsHigh = conf->noOfRecordsHigh;
  Uint64 noOfRecords = (noOfRecordsHigh << 32) + noOfRecordsLow;

  Uint64 noOfBytesLow = conf->noOfBytesLow;
  Uint64 noOfBytesHigh = conf->noOfBytesHigh;
  Uint64 noOfBytes = (noOfBytesHigh << 32) + noOfBytesLow;

  lcpPtr.i = 0;
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->lcpRunState == LcpRecord::LCP_CHECKPOINTING);
  lcpPtr.p->lcpRunState = LcpRecord::LCP_COMPLETED;
  lcpPtr.p->m_no_of_records = noOfRecords;
  lcpPtr.p->m_no_of_bytes = noOfBytes;

  /* ------------------------------------------------------------------------
   *   THE LOCAL CHECKPOINT HAS BEEN COMPLETED. IT IS NOW TIME TO START
   *   A LOCAL CHECKPOINT ON THE NEXT FRAGMENT OR COMPLETE THIS LCP ROUND.
   * ------------------------------------------------------------------------
   *   WE START BY SENDING LCP_REPORT TO DIH TO REPORT THE COMPLETED LCP.
   *   TO CATER FOR NODE CRASHES WE SEND IT IN PARALLEL TO ALL NODES.
   * ----------------------------------------------------------------------- */

  completed_fragment_checkpoint(signal, lcpPtr.p->currentRunFragment);
  handleLCPSurfacing(signal);

  if (lcpPtr.p->lcpPrepareState == LcpRecord::LCP_PREPARED) {
    /**
     * We have completed a fragment checkpoint. We can start the next
     * fragment checkpoint which is already prepared and ready.
     *
     * After that we will start preparing the next fragment for
     * checkpointing.
     */
    jam();
    lcpPtr.p->currentRunFragment = lcpPtr.p->currentPrepareFragment;
    perform_fragment_checkpoint(signal);
    prepare_next_fragment_checkpoint(signal, false);
    return;
  } else if (lcpPtr.p->lcpPrepareState == LcpRecord::LCP_PREPARING) {
    /**
     * We completed the fragment checkpointing before the prepare of the
     * next was done. We will not do anything here since we will wait for
     * the prepare to complete and then new action will be taken.
     */
    jam();
    return;
  }
  jam();
  ndbrequire(lcpPtr.p->lcpPrepareState == LcpRecord::LCP_IDLE);

  stopLcpFragWatchdog();

  /**
   * No new fragment had even started to be prepared. This can only mean
   * that this checkpoint have come to an end. Or at least the queue has
   * come to an end. We check if we have received the last fragment and
   * if so we complete the checkpoint. Otherwise we simply wait for
   * more orders to checkpoint fragments.
   */
  lcpPtr.p->lcpRunState = LcpRecord::LCP_IDLE;
  if (lcpPtr.p->lastFragmentFlag) {
    jam();
    /* ----------------------------------------------------------------------
     *       NOW THE COMPLETE LOCAL CHECKPOINT ROUND IS COMPLETED.
     * --------------------------------------------------------------------- */
    completeLcpRoundLab(signal, lcpPtr.p->currentRunFragment.lcpFragOrd.lcpId);
    return;
  }  // if
  return;
}  // Dblqh::execBACKUP_FRAGMENT_CONF()

void Dblqh::sendLCP_FRAG_REP(Signal *signal, const LcpRecord::FragOrd &fragOrd,
                             const Fragrecord *fragPtrP) const {
  ndbrequire(fragOrd.lcpFragOrd.lcpNo < MAX_LCP_STORED);
  LcpFragRep *const lcpReport = (LcpFragRep *)&signal->theData[0];
  lcpReport->nodeId = cownNodeid;
  lcpReport->lcpId = fragOrd.lcpFragOrd.lcpId;
  lcpReport->lcpNo = fragOrd.lcpFragOrd.lcpNo;
  lcpReport->tableId = fragOrd.lcpFragOrd.tableId;
  lcpReport->fragId = fragOrd.lcpFragOrd.fragmentId;
  lcpReport->maxGciCompleted = fragPtrP->maxGciCompletedInLcp;
  lcpReport->maxGciStarted = fragPtrP->maxGciInLcp;

  Uint32 ref = DBDIH_REF;
  if (isNdbMtLqh()) {
    jam();
    ref = DBLQH_REF;
  }
  lcpReport->nodeId = LcpFragRep::BROADCAST_REQ;
  sendSignal(ref, GSN_LCP_FRAG_REP, signal, LcpFragRep::SignalLength, JBA);
}

void Dblqh::completed_fragment_checkpoint(Signal *signal,
                                          const LcpRecord::FragOrd &fragOrd) {
  /* ------------------------------------------------------------------------
   *       UPDATE THE LATEST LOCAL CHECKPOINT COMPLETED ON FRAGMENT.
   *       UPDATE THE LCP_ID OF THIS CHECKPOINT.
   *       REMOVE THE LINK BETWEEN THE FRAGMENT RECORD AND THE LCP RECORD.
   * ----------------------------------------------------------------------- */
  /**
   * Send rep when fragment is done + unblocked
   */
  FragrecordPtr curr_fragptr;
  curr_fragptr.i = fragOrd.fragPtrI;
  c_fragment_pool.getPtr(curr_fragptr);
  if (curr_fragptr.p->m_create_table_flag_lcp_frag_ord) {
    /**
     * At this point we have accomplished the desired action of running an
     * LCP to ensure that we can deduce how far back we need to UNDO for
     * this table.
     *
     * We set state to LCP_EXECUTED_BY_CREATE_TABLE to ensure that we know
     * that an LCP have been executed this LCP. When an LCP_FRAG_ORD is
     * received we will send LCP_FRAG_REP immediately if it is part of
     * the same LCP.
     *
     * No need to send LCP_FRAG_REP since no LCP_FRAG_ORD was requested in
     * this case.
     */
    jam();
    curr_fragptr.p->lcp_frag_ord_state =
        Fragrecord::LCP_EXECUTED_BY_CREATE_TABLE;
    curr_fragptr.p->m_create_table_flag_lcp_frag_ord = false;
    DEB_EARLY_LCP(("(%u)LCP executed for tab(%u,%u), flag cleared", instance(),
                   curr_fragptr.p->tabRef, curr_fragptr.p->fragId));
  } else {
    jam();
    curr_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTED;
    if (c_localLcpId == 0) {
      /* Only need to send LCP_FRAG_REP during distributed LCP. */
      jam();
      sendLCP_FRAG_REP(signal, fragOrd, curr_fragptr.p);
    }
  }
}

void Dblqh::prepare_next_fragment_checkpoint(Signal *signal,
                                             bool complete_possible) {
  FragrecordPtr curr_fragptr;

  if (lcpPtr.p->m_wait_early_lcp_synch) {
    jam();
    /**
     * We are waiting for a signal back from Backup block containing
     * a signal that all outstanding fragments that was put into LCP
     * queue by CREATE TABLE statements have been executed and have
     * completed.
     * We could receive the lastFragmentFlag while waiting and it is
     * important to not be in IDLE state when this arrives since this
     * will flag that the LCP is completed which is not the case here.
     */
    lcpPtr.p->lcpPrepareState = LcpRecord::LCP_PREPARING;
    return;
  }
  do {
    if (c_queued_lcp_frag_ord.isEmpty()) {
      jam();
      lcpPtr.p->lcpPrepareState = LcpRecord::LCP_IDLE;
      break;
    }
    jam();
    /* ----------------------------------------------------------------------
     *  Remove first queued fragment from queue.
     *  Transfer the state from the queued to the active LCP.
     * --------------------------------------------------------------------- */
    c_queued_lcp_frag_ord.first(curr_fragptr);
    c_queued_lcp_frag_ord.removeFirst(curr_fragptr);
    curr_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTING;
    if (exec_prepare_next_fragment_checkpoint(signal, curr_fragptr)) {
      jam();
      break;
    }
  } while (1);
  if (is_lcp_idle(lcpPtr.p)) {
    /**
     * We have no queued fragments waiting to be prepared. We also
     * have no ongoing fragment executing its LCP. If we also received
     * the last fragment then we have completed this LCP.
     */
    jam();
    if (lcpPtr.p->lastFragmentFlag) {
      jam();
      ndbrequire(complete_possible);
      completeLcpRoundLab(signal, c_lcpId);
    }
  }
}

bool Dblqh::exec_prepare_next_fragment_checkpoint(Signal *signal,
                                                  FragrecordPtr curr_fragptr) {
  TablerecPtr tabPtr;
  tabPtr.i = curr_fragptr.p->tabRef;
  ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
  lcpPtr.p->currentPrepareFragment.fragPtrI = curr_fragptr.i;
  if (curr_fragptr.p->m_create_table_flag_lcp_frag_ord) {
    jam();
    curr_fragptr.p->lcp_frag_ord_lcp_id =
        lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId;
    curr_fragptr.p->lcp_frag_ord_lcp_no = 0;
  }
  lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpNo =
      curr_fragptr.p->lcp_frag_ord_lcp_no;
  lcpPtr.p->currentPrepareFragment.lcpFragOrd.fragmentId =
      curr_fragptr.p->fragId;
  lcpPtr.p->currentPrepareFragment.lcpFragOrd.tableId = tabPtr.i;

  if (lcpPtr.p->m_early_lcps_need_synch) {
    if (curr_fragptr.p->m_create_table_insert_lcp) {
      jam();
      ndbrequire(lcpPtr.p->m_wait_early_lcp_synch == false);
    } else {
      jam();
      /**
       * One or more tables were created between two LCPs.
       * We need to ensure that those fragments complete their
       * LCP before any other fragment is allowed to complete.
       * This will ensure that the LCP that any table requires
       * the UNDO log to roll back to is sufficient for these
       * tables
       *
       * See comment in insert_new_fragments_into_lcp for more
       * details.
       */
      DEB_EARLY_LCP(("(%u)WAIT_LCP_IDLE_REQ: tab(%u,%u)", instance(),
                     curr_fragptr.p->tabRef, curr_fragptr.p->fragId));
      lcpPtr.p->lcpPrepareState = LcpRecord::LCP_PREPARING;
      lcpPtr.p->m_early_lcps_need_synch = false;
      lcpPtr.p->m_wait_early_lcp_synch = true;
      curr_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_EXECUTING;
      BlockReference backupRef = calcInstanceBlockRef(getBACKUP());
      signal->theData[0] = m_backup_ptr;
      sendSignal(backupRef, GSN_WAIT_LCP_IDLE_REQ, signal, 1, JBB);
      return true;
    }
  }

  curr_fragptr.p->m_create_table_insert_lcp = false;

  if (unlikely(tabPtr.p->tableStatus != Tablerec::TABLE_DEFINED &&
               tabPtr.p->tableStatus != Tablerec::TABLE_READ_ONLY)) {
    jam();
    /**
     * Fake that the fragment is done
     */
    ndbrequire(c_localLcpId == 0);
    completed_fragment_checkpoint(signal, lcpPtr.p->currentPrepareFragment);
    return false;
  }

  /**
   * We need to perform LCPs also of read-only tables since there might
   * have been changes to the table between now and when the table was
   * made read only.
   */

  lcpPtr.p->lcpPrepareState = LcpRecord::LCP_PREPARING;
  LcpPrepareReq *req = (LcpPrepareReq *)signal->getDataPtr();
  req->senderData = lcpPtr.i;
  req->senderRef = reference();
  req->lcpNo = lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpNo;
  req->tableId = lcpPtr.p->currentPrepareFragment.lcpFragOrd.tableId;
  req->fragmentId = lcpPtr.p->currentPrepareFragment.lcpFragOrd.fragmentId;
  req->lcpId = lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId;
  req->backupPtr = m_backup_ptr;
  req->backupId = lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId;
  req->createGci = curr_fragptr.p->createGci;
  req->localLcpId = m_curr_local_lcp_id;
  BlockReference backupRef = calcInstanceBlockRef(getBACKUP());
  if (!ERROR_INSERTED(5053)) {
    sendSignal(backupRef, GSN_LCP_PREPARE_REQ, signal,
               LcpPrepareReq::SignalLength, JBA);
  } else {
    sendSignalWithDelay(backupRef, GSN_LCP_PREPARE_REQ, signal, 150,
                        LcpPrepareReq::SignalLength);
  }

  /* Now start the LCP fragment watchdog */
  if (lcpPtr.p->lcpRunState == LcpRecord::LCP_IDLE) {
    jam();
    startLcpFragWatchdog(signal);
  }
  return true;
}  // Dblqh::prepare_next_fragment_checkpoint()

void Dblqh::execWAIT_LCP_IDLE_CONF(Signal *signal) {
  FragrecordPtr curr_fragptr;
  lcpPtr.i = signal->theData[0];
  ptrCheckGuard(lcpPtr, clcpFileSize, lcpRecord);
  ndbrequire(lcpPtr.p->m_wait_early_lcp_synch);
  ndbrequire(!lcpPtr.p->m_early_lcps_need_synch);

  /**
   * No action is supposed to remove the fragment from the LCP list
   * while we are waiting for the previous fragments to complete. Only
   * completion of checkpoint for this fragment and an attempt to
   * prepare the fragment could cause this fragment to be removed and
   * none of those actions are possible while we are waiting for
   * WAIT_LCP_IDLE_CONF to return from the Backup block.
   *
   * It is also possible for Drop Table to remove if the state is
   * LCP_QUEUED, to ensure this won't happen the state of the fragment
   * was set to LCP_EXECUTING before starting the wait.
   *
   * The reason to be so careful with this protection against removal
   * is to avoid having to handle that the LCP completes while we are
   * waiting for the other fragments to complete. If that was possible
   * we would have to check for completion of LCP here as well. Now we
   * can skip that since it is safe that we are not arriving here in
   * that state.
   *
   * It is possible for the fragment LCP to be generated by a table
   * creation. This table should not be removed when the table is
   * in the state LCP_EXECUTING, so we rely on that this fragment is
   * still here when we return here. The fragment LCP can be dropped
   * however when we restart it again.
   */
  curr_fragptr.i = lcpPtr.p->currentPrepareFragment.fragPtrI;
  c_fragment_pool.getPtr(curr_fragptr);
  ndbrequire(curr_fragptr.p->lcp_frag_ord_state == Fragrecord::LCP_EXECUTING);
  c_queued_lcp_frag_ord.addFirst(curr_fragptr);
  curr_fragptr.p->lcp_frag_ord_state = Fragrecord::LCP_QUEUED;
  ndbrequire(curr_fragptr.p->m_create_table_insert_lcp == false);
  lcpPtr.p->m_wait_early_lcp_synch = false;
  DEB_EARLY_LCP(("(%u) WAIT_LCP_IDLE_CONF, tab(%u,%u)", instance(),
                 curr_fragptr.p->tabRef, curr_fragptr.p->fragId));
  prepare_next_fragment_checkpoint(signal, true);
}

/* --------------------------------------------------------------------------
 *       THE LOCAL CHECKPOINT ROUND IS NOW COMPLETED. SEND COMPLETED MESSAGE
 *       TO THE MASTER DIH.
 * ------------------------------------------------------------------------- */
void Dblqh::completeLcpRoundLab(Signal *signal, Uint32 lcpId) {
  if (!isNdbMtLqh() && c_fragments_in_lcp == 0) {
    jam();
    lcpPtr.i = 0;
    ptrAss(lcpPtr, lcpRecord);
    sendLCP_COMPLETE_REP(signal,
                         lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId);
    return;
  }
  startLcpFragWatchdog(signal);
  DEB_EMPTY_LCP(("(%u)Start complete LCP %u", instance(), lcpId));
  clcpCompletedState = LCP_CLOSE_STARTED;
  EndLcpReq *req = (EndLcpReq *)signal->getDataPtr();
  req->senderData = lcpId;
  req->senderRef = reference();
  req->backupPtr = m_backup_ptr;
  req->backupId = lcpId;
  BlockReference backupRef = calcInstanceBlockRef(getBACKUP());
  sendSignal(backupRef, GSN_END_LCPREQ, signal, EndLcpReq::SignalLength, JBA);
}  // Dblqh::completeLcpRoundLab()

void Dblqh::execEND_LCPCONF(Signal *signal) {
  EndLcpConf *conf = (EndLcpConf *)signal->getDataPtr();
  jamEntry();

  ndbrequire(clcpCompletedState == LCP_CLOSE_STARTED);
  BlockReference backupRef = calcInstanceBlockRef(getBACKUP());
  if (!isNdbMtLqh() && conf->senderRef == backupRef) {
    /**
     * ndbd also needs to send to TSMAN (handled by Proxy block in ndbmtd).
     */
    jam();
    Uint32 lcpId = conf->senderData;
    EndLcpReq *req = (EndLcpReq *)signal->getDataPtr();
    req->senderData = lcpId;
    req->senderRef = reference();
    req->backupPtr = m_backup_ptr;
    req->backupId = lcpId;
    sendSignal(TSMAN_REF, GSN_END_LCPREQ, signal, EndLcpReq::SignalLength, JBA);
    return;
  }
  DEB_EMPTY_LCP(
      ("(%u)END_LCPCONF received LCP %u", instance(), conf->senderData));
  stopLcpFragWatchdog();
  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);
  sendLCP_COMPLETE_REP(signal,
                       lcpPtr.p->currentPrepareFragment.lcpFragOrd.lcpId);
  CRASH_INSERTION(5056);
}  // Dblqh::execEND_LCPCONF()

void Dblqh::sendLCP_COMPLETE_REP(Signal *signal, Uint32 lcpId) {
  ndbrequire((cnoOfNodes - 1) < (MAX_NDB_NODES - 1));
  /* ------------------------------------------------------------------------
   *       WE SEND COMP_LCP_ROUND TO ALL NODES TO PREPARE FOR NODE CRASHES.
   * ----------------------------------------------------------------------- */
  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);

  infoEvent(
      "LDM(%u): Completed LCP, #frags = %u"
      " #records = %llu, #bytes = %llu",
      instance(), cnoOfFragsCheckpointed, lcpPtr.p->m_no_of_records,
      lcpPtr.p->m_no_of_bytes);

  lcpPtr.p->lastFragmentFlag = false;
  lcpPtr.p->firstFragmentFlag = false;
  lcpPtr.p->m_no_of_records = 0;
  lcpPtr.p->m_no_of_bytes = 0;
  cnoOfFragsCheckpointed = 0;
  clcpCompletedState = LCP_IDLE;
  c_fragments_in_lcp = 0;

  if (c_localLcpId != 0) {
    jam();
    /**
     * We have completed a local LCP, report it locally and continue
     * restart processing.
     */
    complete_local_lcp(signal);
    return;
  }
  /**
   * Before we report that we're done to cluster level we want to
   * wait until the GCI that makes the LCP completely done is safe.
   * After that we have also ensured that the UNDO and REDO log tails
   * have been cut. Only after this is done are we ready to report
   * back to DIH that the LCP is complete in this LDM.
   *
   * This coordination happens in NDBCNTR.
   */
  ndbrequire(c_keep_gci_for_lcp <= c_max_keep_gci_in_lcp ||
             c_num_fragments_created_since_restart == 0);
  LcpAllCompleteReq *req = (LcpAllCompleteReq *)signal->getDataPtrSend();
  req->senderRef = reference();
  req->lcpId = lcpId;
  req->maxGciInLcp = c_max_gci_in_lcp;
  req->maxKeepGci = c_max_keep_gci_in_lcp;
  sendSignal(NDBCNTR_REF, GSN_LCP_ALL_COMPLETE_REQ, signal,
             LcpAllCompleteReq::SignalLength, JBB);
}

void Dblqh::execLCP_ALL_COMPLETE_CONF(Signal *signal) {
  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);
  Uint32 lcpId = signal->theData[0];
  LcpCompleteRep *rep = (LcpCompleteRep *)signal->getDataPtrSend();
  rep->nodeId = getOwnNodeId();
  rep->lcpId = lcpId;
  rep->blockNo = DBLQH;

  Uint32 ref = DBDIH_REF;
  if (isNdbMtLqh()) {
    jam();
    ref = DBLQH_REF;
  }
  rep->nodeId = LcpFragRep::BROADCAST_REQ;

  sendSignal(ref, GSN_LCP_COMPLETE_REP, signal, LcpCompleteRep::SignalLength,
             JBA);

  if (cstartRecReq < SRR_FIRST_LCP_DONE) {
    jam();
    ndbrequire(cstartRecReq == SRR_REDO_COMPLETE);
    cstartRecReq = SRR_FIRST_LCP_DONE;
  }
  return;

}  // Dblqh::sendLCP_COMPLETE_REP()

/* ------------------------------------------------------------------------- */
/* -------               SET THE LOG TAIL IN THE LOG FILES           ------- */
/*                                                                           */
/*THIS SUBROUTINE HAVE BEEN BUGGY AND IS RATHER COMPLEX. IT IS IMPORTANT TO  */
/*REMEMBER THAT WE SEARCH FROM THE TAIL UNTIL WE REACH THE HEAD (CURRENT).   */
/*THE TAIL AND HEAD CAN BE ON THE SAME MBYTE. WE SEARCH UNTIL WE FIND A MBYTE*/
/*THAT WE NEED TO KEEP. WE THEN SET THE TAIL TO BE THE PREVIOUS. IF WE DO    */
/*NOT FIND A MBYTE THAT WE NEED TO KEEP UNTIL WE REACH THE HEAD THEN WE USE  */
/*THE HEAD AS TAIL. FINALLY WE HAVE TO MOVE BACK THE TAIL TO ALSO INCLUDE    */
/*ALL PREPARE RECORDS. THIS MEANS THAT LONG-LIVED TRANSACTIONS ARE DANGEROUS */
/*FOR SHORT LOGS.                                                            */
/* ------------------------------------------------------------------------- */

void Dblqh::setLogTail(Signal *signal, Uint32 keepGci) {
  LogPartRecordPtr sltLogPartPtr;
  LogFileRecordPtr sltLogFilePtr;
  UintR tsltMbyte;
  UintR tsltStartMbyte;
  UintR tsltIndex;
  UintR tsltFlag;

  for (sltLogPartPtr.i = 0; sltLogPartPtr.i < clogPartFileSize;
       sltLogPartPtr.i++) {
    jam();
    bool TchangeMB = false;
    ptrAss(sltLogPartPtr, logPartRecord);
    lock_log_part(sltLogPartPtr.p);
  retry:
    findLogfile(signal, sltLogPartPtr.p->logTailFileNo, sltLogPartPtr.p,
                &sltLogFilePtr);

    tsltMbyte = sltLogPartPtr.p->logTailMbyte;
    tsltStartMbyte = tsltMbyte;
    tsltFlag = ZFALSE;
    if (sltLogFilePtr.i == sltLogPartPtr.p->currentLogfile) {
      /* -------------------------------------------------------------------------
       */
      /*THE LOG AND THE TAIL IS ALREADY IN THE SAME FILE. */
      /* -------------------------------------------------------------------------
       */
      if (sltLogFilePtr.p->currentMbyte >= sltLogPartPtr.p->logTailMbyte) {
        jam();
        /* -------------------------------------------------------------------------
         */
        /*THE CURRENT MBYTE IS AHEAD OF OR AT THE TAIL. THUS WE WILL ONLY LOOK
         * FOR   */
        /*THE TAIL UNTIL WE REACH THE CURRENT MBYTE WHICH IS IN THIS LOG FILE.
         */
        /*IF THE LOG TAIL IS AHEAD OF THE CURRENT MBYTE BUT IN THE SAME LOG FILE
         */
        /*THEN WE HAVE TO SEARCH THROUGH ALL FILES BEFORE WE COME TO THE CURRENT
         */
        /*MBYTE. WE ALWAYS STOP WHEN WE COME TO THE CURRENT MBYTE SINCE THE TAIL
         */
        /*CAN NEVER BE BEFORE THE HEAD. */
        /* -------------------------------------------------------------------------
         */
        tsltFlag = ZTRUE;
      }  // if
    }    // if

    /* -------------------------------------------------------------------------
     */
    /*NOW START SEARCHING FOR THE NEW TAIL, STARTING AT THE CURRENT TAIL AND */
    /*PROCEEDING UNTIL WE FIND A MBYTE WHICH IS NEEDED TO KEEP OR UNTIL WE REACH
     */
    /*CURRENT MBYTE (THE HEAD). */
    /* -------------------------------------------------------------------------
     */
  SLT_LOOP:
    for (tsltIndex = tsltStartMbyte; tsltIndex <= clogFileSize - 1;
         tsltIndex++) {
      if (sltLogFilePtr.p->logMaxGciStarted[tsltIndex] >= keepGci) {
        /* -------------------------------------------------------------------------
         */
        /*WE ARE NOT ALLOWED TO STEP THE LOG ANY FURTHER AHEAD */
        /*SET THE NEW LOG TAIL AND CONTINUE WITH NEXT LOG PART. */
        /*THIS MBYTE IS NOT TO BE INCLUDED SO WE NEED TO STEP BACK ONE MBYTE. */
        /* -------------------------------------------------------------------------
         */
        /* Check keepGCI MB has a reasonable GCI value */
        ndbrequire(sltLogFilePtr.p->logMaxGciStarted[tsltIndex] !=
                   ((Uint32)-1));
        if (tsltIndex != 0) {
          jam();
          tsltMbyte = tsltIndex - 1;
        } else {
          jam();
          /* -------------------------------------------------------------------------
           */
          /*STEPPING BACK INCLUDES ALSO STEPPING BACK TO THE PREVIOUS LOG FILE.
           */
          /* -------------------------------------------------------------------------
           */
          tsltMbyte = clogFileSize - 1;
          sltLogFilePtr.i = sltLogFilePtr.p->prevLogFile;
          ptrCheckGuard(sltLogFilePtr, clogFileFileSize, logFileRecord);
        }  // if
        goto SLT_BREAK;
      } else {
        jam();
        if (tsltFlag == ZTRUE) {
          /* -------------------------------------------------------------------------
           */
          /*WE ARE IN THE SAME FILE AS THE CURRENT MBYTE AND WE CAN REACH THE
           * CURRENT  */
          /*MBYTE BEFORE WE REACH A NEW TAIL. */
          /* -------------------------------------------------------------------------
           */
          if (tsltIndex == sltLogFilePtr.p->currentMbyte) {
            jam();
            /* -------------------------------------------------------------------------
             */
            /*THE TAIL OF THE LOG IS ACTUALLY WITHIN THE CURRENT MBYTE. THUS WE
             * SET THE  */
            /*LOG TAIL TO BE THE CURRENT MBYTE. */
            /* -------------------------------------------------------------------------
             */
            tsltMbyte = sltLogFilePtr.p->currentMbyte;
            goto SLT_BREAK;
          }  // if
        }    // if
      }      // if
    }        // for
    sltLogFilePtr.i = sltLogFilePtr.p->nextLogFile;
    ptrCheckGuard(sltLogFilePtr, clogFileFileSize, logFileRecord);
    if (sltLogFilePtr.i == sltLogPartPtr.p->currentLogfile) {
      jam();
      tsltFlag = ZTRUE;
    }  // if
    tsltStartMbyte = 0;
    goto SLT_LOOP;
  SLT_BREAK:
    jam();
    {
      UintR ToldTailFileNo = sltLogPartPtr.p->logTailFileNo;
      UintR ToldTailMByte = sltLogPartPtr.p->logTailMbyte;

      /* -------------------------------------------------------------------------
       */
      /*SINCE LOG_MAX_GCI_STARTED ONLY KEEP TRACK OF COMMIT LOG RECORDS WE ALSO
       */
      /*HAVE TO STEP BACK THE TAIL SO THAT WE INCLUDE ALL PREPARE RECORDS */
      /*NEEDED FOR THOSE COMMIT RECORDS IN THIS MBYTE. THIS IS A RATHER */
      /*CONSERVATIVE APPROACH BUT IT WORKS. */
      /* -------------------------------------------------------------------------
       */
      arrGuard(tsltMbyte, clogFileSize);
      sltLogPartPtr.p->logTailFileNo =
          sltLogFilePtr.p->logLastPrepRef[tsltMbyte] >> 16;
      sltLogPartPtr.p->logTailMbyte =
          sltLogFilePtr.p->logLastPrepRef[tsltMbyte] & 65535;

      if (DEBUG_REDO_EXEC) {
        g_eventLogger->info(
            "(%u)part: %u setLogTail(gci: %u):"
            " file: %u mb: %u",
            instance(), sltLogPartPtr.p->logPartNo, keepGci,
            sltLogPartPtr.p->logTailFileNo, sltLogPartPtr.p->logTailMbyte);
      }

      bool tailmoved = !(ToldTailFileNo == sltLogPartPtr.p->logTailFileNo &&
                         ToldTailMByte == sltLogPartPtr.p->logTailMbyte);

      LogFileRecordPtr tmpfile;
      tmpfile.i = sltLogPartPtr.p->currentLogfile;
      ptrCheckGuard(tmpfile, clogFileFileSize, logFileRecord);

      LogPosition head = {tmpfile.p->fileNo, tmpfile.p->currentMbyte};
      LogPosition tail = {sltLogPartPtr.p->logTailFileNo,
                          sltLogPartPtr.p->logTailMbyte};
      Uint64 free_mb =
          free_log(head, tail, sltLogPartPtr.p->noLogFiles, clogFileSize);

#ifdef DEBUG_CUT_REDO
      {
        TcConnectionrec *tmpPtrP;
        tmpPtrP = sltLogPartPtr.p->firstLogTcrec;
        Uint32 fileNo = -1;
        Uint32 mbyte = -1;
        if (tmpPtrP != nullptr) {
          jam();
          ndbrequire(Magic::check_ptr(tmpPtrP));
          fileNo = tmpPtrP->logStartFileNo;
          mbyte = tmpPtrP->logStartPageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE;
        }
        DEB_CUT_REDO(
            ("(%u)Logpart: %u, gci: %u, tail(%u,%u), old_tail(%u,%u)"
             ", tc_tail(%u,%u), mb: %llu",
             instance(), sltLogPartPtr.p->logPartNo, keepGci,
             sltLogPartPtr.p->logTailFileNo, sltLogPartPtr.p->logTailMbyte,
             ToldTailFileNo, ToldTailMByte, fileNo, mbyte, free_mb));
      }
#endif

      if (free_mb <= c_free_mb_force_lcp_limit) {
        /**
         * Force a new LCP
         */
        force_lcp(signal);
      }
      Uint32 committed_mbytes = get_committed_mbytes(sltLogPartPtr.p);
      if (tailmoved &&
          (free_mb > (c_free_mb_tail_problem_limit + committed_mbytes))) {
        jam();
        update_log_problem(signal, sltLogPartPtr.p,
                           LogPartRecord::P_TAIL_PROBLEM, false);
      } else if (!tailmoved && free_mb <= c_free_mb_force_lcp_limit) {
        jam();
        /**
         * Tail didn't move...and we forced a new LCP
         *   This could be as currentMb, contains backreferences making it
         *   Check if changing mb forward will help situation
         */
        if ((free_mb + committed_mbytes) < 4) {
          /**
           * Less than 4 mb free, no point in trying to changeMbyte forward...
           */
          jam();
          goto next;
        }

        if (TchangeMB) {
          jam();
          /**
           * We already did move forward...
           */
          goto next;
        }

        TcConnectionrec *tmpPtrP;
        tmpPtrP = sltLogPartPtr.p->firstLogTcrec;
        if (tmpPtrP != nullptr) {
          jam();
          ndbrequire(Magic::check_ptr(tmpPtrP));
          Uint32 fileNo = tmpPtrP->logStartFileNo;
          Uint32 mbyte = tmpPtrP->logStartPageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE;

          if (fileNo == sltLogPartPtr.p->logTailFileNo &&
              mbyte == sltLogPartPtr.p->logTailMbyte) {
            jam();
            /**
             * An uncommitted operation...still pending...
             *   with back-reference to tail...not much to do
             *   (theoretically we could rewrite log-entry here...
             *    but this is for future)
             * skip to next
             */
            goto next;
          }
        }

        {
          /**
           * Try forcing a changeMbyte
           */
          LogPageRecordPtr logPagePtr;
          LogFileRecordPtr logFilePtr;
          LogPartRecordPtr logPartPtr;
          jam();
          logPartPtr = sltLogPartPtr;
          logFilePtr.i = logPartPtr.p->currentLogfile;
          ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
          logPagePtr.i = logFilePtr.p->currentLogpage;
          ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                        logPartPtr.p->logPageRecord);
          changeMbyte(signal, logPagePtr, logFilePtr, logPartPtr.p);
          TchangeMB = true;  // don't try this twice...
          goto retry;
        }
      }
    }
  next:
    (void)1;
    unlock_log_part(sltLogPartPtr.p);
  }  // for
}  // Dblqh::setLogTail()

/* ######################################################################### */
/* #######                       GLOBAL CHECKPOINT MODULE            ####### */
/*                                                                           */
/* ######################################################################### */
/*---------------------------------------------------------------------------*/
/* THIS MODULE HELPS DIH IN DISCOVERING WHEN GLOBAL CHECKPOINTS ARE          */
/* RECOVERABLE. IT HANDLES THE REQUEST GCP_SAVEREQ THAT REQUESTS LQH TO      */
/* SAVE A PARTICULAR GLOBAL CHECKPOINT TO DISK AND RESPOND WHEN COMPLETED.   */
/*---------------------------------------------------------------------------*/
/* *************** */
/*  GCP_SAVEREQ  > */
/* *************** */

#if defined VM_TRACE || defined ERROR_INSERT
static Uint32 m_gcp_monitor = 0;
#endif

void Dblqh::execGCP_SAVEREQ(Signal *signal) {
  jamEntry();
  const GCPSaveReq *const saveReq = (GCPSaveReq *)&signal->theData[0];

  CRASH_INSERTION(5000);

  if (ERROR_INSERTED(5007)) {
    CLEAR_ERROR_INSERT_VALUE;
    sendSignalWithDelay(cownref, GSN_GCP_SAVEREQ, signal, 10000,
                        signal->length());
    return;
  }

  const Uint32 dihBlockRef = saveReq->dihBlockRef;
  const Uint32 dihPtr = saveReq->dihPtr;
  const Uint32 gci = saveReq->gci;

  DEB_GCP(("(%u) GCP_SAVEREQ(%u)", instance(), gci));

  ndbrequire(refToNode(signal->getSendersBlockRef()) == getOwnNodeId());

#if defined VM_TRACE || defined ERROR_INSERT
  if (!isNdbMtLqh()) {  // wl4391_todo mt-safe
    ndbrequire(m_gcp_monitor == 0 || (m_gcp_monitor == gci) ||
               (m_gcp_monitor + 1) == gci);
  }
  m_gcp_monitor = gci;
#endif

  if (getNodeState().startLevel >= NodeState::SL_STOPPING_4) {
    DEB_GCP(("(%u)SL_STOPPING_4: gci = %u", instance(), gci));
    GCPSaveRef *const saveRef = (GCPSaveRef *)&signal->theData[0];
    saveRef->dihPtr = dihPtr;
    saveRef->nodeId = getOwnNodeId();
    saveRef->gci = gci;
    saveRef->errorCode = GCPSaveRef::NodeShutdownInProgress;
    sendSignal(dihBlockRef, GSN_GCP_SAVEREF, signal, GCPSaveRef::SignalLength,
               JBB);
    return;
  }

  /**
   * Cannot update cnewestCompletedGci during SL_STOPPING_4 since we
   * no longer participate in GCPs.
   */
  Uint32 saveNewestCompletedGci = cnewestCompletedGci;
  cnewestCompletedGci = gci;

  if (cstartRecReq < SRR_REDO_COMPLETE) {
    DEB_GCP(("(%u)!SRR_REDO_COMPLETE: gci = %u", instance(), gci));
    /**
     * REDO running is not complete
     */
    GCPSaveRef *const saveRef = (GCPSaveRef *)&signal->theData[0];
    saveRef->dihPtr = dihPtr;
    saveRef->nodeId = getOwnNodeId();
    saveRef->gci = gci;
    saveRef->errorCode = GCPSaveRef::NodeRestartInProgress;
    sendSignal(dihBlockRef, GSN_GCP_SAVEREF, signal, GCPSaveRef::SignalLength,
               JBB);
    return;
  }

  ndbrequire(gci >= saveNewestCompletedGci);

  if (gci == saveNewestCompletedGci) {
    /*---------------------------------------------------------------------------*/
    /* GLOBAL CHECKPOINT HAVE ALREADY BEEN HANDLED. REQUEST MUST HAVE BEEN SENT
     */
    /* FROM NEW MASTER DIH. */
    /*---------------------------------------------------------------------------*/
    DEB_GCP(("(%u)GCP already sent: gci = %u", instance(), gci));
    if (ccurrentGcprec == RNIL) {
      jam();
      /*---------------------------------------------------------------------------*/
      /* THIS INDICATES THAT WE HAVE ALREADY SENT GCP_SAVECONF TO PREVIOUS
       * MASTER. */
      /* WE SIMPLY SEND IT ALSO TO THE NEW MASTER. */
      /*---------------------------------------------------------------------------*/
      GCPSaveConf *const saveConf = (GCPSaveConf *)&signal->theData[0];
      saveConf->dihPtr = dihPtr;
      saveConf->nodeId = getOwnNodeId();
      saveConf->gci = cnewestCompletedGci;
      sendSignal(dihBlockRef, GSN_GCP_SAVECONF, signal,
                 GCPSaveConf::SignalLength, JBA);
      return;
    }
    jam();
    /*---------------------------------------------------------------------------*/
    /* WE HAVE NOT YET SENT THE RESPONSE TO THE OLD MASTER. WE WILL SET THE NEW
     */
    /* RECEIVER OF THE RESPONSE AND THEN EXIT SINCE THE PROCESS IS ALREADY */
    /* STARTED. */
    /*---------------------------------------------------------------------------*/
    gcpPtr.i = ccurrentGcprec;
    ptrCheckGuard(gcpPtr, cgcprecFileSize, gcpRecord);
    gcpPtr.p->gcpUserptr = dihPtr;
    gcpPtr.p->gcpBlockref = dihBlockRef;
    ndbrequire(refToMain(gcpPtr.p->gcpBlockref) == DBDIH ||
               refToMain(gcpPtr.p->gcpBlockref) == DBLQH);
    return;
  }  // if

  ndbrequire(ccurrentGcprec == RNIL);
  cnewestCompletedGci = gci;
  if (gci > cnewestGci) {
    jam();
    cnewestGci = gci;
    DEB_NEWEST_GCI(("(%u)(%u) cnewestGci = %u, line: %u", instance(),
                    m_is_query_block, cnewestGci, __LINE__));
  }  // if

  if (cstartRecReq < SRR_FIRST_LCP_DONE) {
    /**
     * First LCP has not been done
     */
    jam();
    DEB_GCP(("(%u)!SRR_FIRST_LCP_DONE: gci = %u", instance(), gci));
    c_local_sysfile.m_save_gci = gci;
    c_local_sysfile.m_dihPtr = dihPtr;
    c_local_sysfile.m_dihRef = dihBlockRef;
    c_send_gcp_saveref_needed = true;
    if ((m_node_restart_first_local_lcp_started ||
         m_first_distributed_lcp_started) &&
        is_first_instance()) {
      jam();
      write_local_sysfile_gcp_complete(signal, gci - 1);
    } else {
      jam();
      write_local_sysfile_gcp_complete_done(signal);
    }
    return;
  }

  CRASH_INSERTION(5052);

#ifdef GCP_TIMER_HACK
  globalData.gcp_timer_save[0] = NdbTick_getCurrentTicks();
#endif

  if (cstartPhase == ZNIL) {
    jam();
    /**
     * The node have completed its start at least up to phase 50 which
     * means our node is fully restorable and we can treat this GCI
     * as restorable.
     *
     * After completing the restart LCP but before the node restart
     * is completed we won't send any writes to local sysfile, but
     * also we won't report the GCI as restorable just yet.
     * This will not have any major impact since after the restart LCP
     * is completed a very short time should pass before we get to
     * phase 9 where the LQH restart is fully completed and we know
     * that we are restorable again.
     */
    sendRESTORABLE_GCI_REP(signal, gci);
  }

  if (clogPartFileSize == 0) {
    jam();
    GCPSaveConf *const saveConf = (GCPSaveConf *)&signal->theData[0];
    saveConf->dihPtr = dihPtr;
    saveConf->nodeId = getOwnNodeId();
    saveConf->gci = gci;
    sendSignal(dihBlockRef, GSN_GCP_SAVECONF, signal, GCPSaveConf::SignalLength,
               JBA);
    return;
  }

  jam();
  ccurrentGcprec = 0;
  gcpPtr.i = ccurrentGcprec;
  ptrCheckGuard(gcpPtr, cgcprecFileSize, gcpRecord);

  gcpPtr.p->gcpBlockref = dihBlockRef;
  gcpPtr.p->gcpUserptr = dihPtr;
  gcpPtr.p->gcpId = gci;
  ndbrequire(refToMain(gcpPtr.p->gcpBlockref) == DBDIH ||
             refToMain(gcpPtr.p->gcpBlockref) == DBLQH);

  if (cstartPhase != ZNIL) {
    jam();
    if (is_first_instance()) {
      if (c_start_phase_9_waiting) {
        jam();
        /**
         * We have reached Start phase 9 and no one is writing local sysfile
         * since we arrive here. Thus we will write restart completed into the
         * local sysfile before we flush the GCI into the REDO logs.
         */
        DEB_START_PHASE9(
            ("(%u) write local sysfile restart complete", instance()));
        write_local_sysfile_restart_complete(signal);
      } else {
        jam();
        /**
         * We need to keep the local sysfile up to date with the
         * maximum restartable GCI until the restart is completed.
         * This GCI is not necessarily restartable, it is only a
         * maximum GCI that can be restarted. DIH decides what is
         * restartable. LQH keeps this information only to verify
         * that DIH is performing its action correctly.
         */
        write_local_sysfile_gcp_complete_late(signal, gci);
      }
      return;
    }
  }
  DEB_GCP(("%u) Start synch GCP: %u", instance(), gci));
  start_synch_gcp(signal);
}

void Dblqh::start_synch_gcp(Signal *signal) {
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;

  ccurrentGcprec = 0;
  gcpPtr.i = ccurrentGcprec;
  ptrCheckGuard(gcpPtr, cgcprecFileSize, gcpRecord);
  bool tlogActive = false;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    ptrAss(logPartPtr, logPartRecord);
    lock_log_part(logPartPtr.p);
    if (logPartPtr.p->logPartState == LogPartRecord::ACTIVE) {
      jam();
      logPartPtr.p->waitWriteGciLog = LogPartRecord::WWGL_TRUE;
      tlogActive = true;
      if (logPartPtr.p->LogLqhKeyReqSent == ZFALSE) {
        jam();
        logPartPtr.p->LogLqhKeyReqSent = ZTRUE;
        signal->theData[0] = ZLOG_LQHKEYREQ;
        signal->theData[1] = logPartPtr.i;
        sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
      }
    } else {
      jam();
      logPartPtr.p->waitWriteGciLog = LogPartRecord::WWGL_FALSE;
      logFilePtr.i = logPartPtr.p->currentLogfile;
      ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
      logPagePtr.i = logFilePtr.p->currentLogpage;
      ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                    logPartPtr.p->logPageRecord);
      writeCompletedGciLog(signal, logPagePtr, logFilePtr, logPartPtr.p);
    }  // if
    unlock_log_part(logPartPtr.p);
  }  // for
  if (tlogActive == true) {
    jam();
    return;
  }  // if
  initGcpRecLab(signal);
}

void Dblqh::sendRESTORABLE_GCI_REP(Signal *signal, Uint32 gci) {
  /**
   * Report completed GCI (one less than the one we are now saving), to
   * give the NDBCNTR block a chance to know when it is ready to cut the
   * log tails.
   */
  signal->theData[0] = gci - 1;
  if (is_first_instance()) {
    jam();
    sendSignal(NDBCNTR_REF, GSN_RESTORABLE_GCI_REP, signal, 1, JBB);
  }
  /**
   * Report completed GCI (one less than the one we are now saving), to
   * give the Backup block a chance to remove old LCP files.
   * Without this signal arriving to Backup block the node restart will
   * be blocked waiting for the proper GCI to delete the old files
   * and also waiting for this to ensure that it will validate the
   * LCP control files.
   */
  signal->theData[0] = gci - 1;
  EXECUTE_DIRECT(getBACKUP(), GSN_RESTORABLE_GCI_REP, signal, 1);
}

void Dblqh::write_local_sysfile_gcp_complete_done(Signal *signal) {
  if (c_send_gcp_saveref_needed) {
    jam();
    c_send_gcp_saveref_needed = false;
    GCPSaveRef *const saveRef = (GCPSaveRef *)&signal->theData[0];
    saveRef->dihPtr = c_local_sysfile.m_dihPtr;
    saveRef->nodeId = getOwnNodeId();
    saveRef->gci = c_local_sysfile.m_save_gci;
    saveRef->errorCode = GCPSaveRef::NodeRestartInProgress;
    sendSignal(c_local_sysfile.m_dihRef, GSN_GCP_SAVEREF, signal,
               GCPSaveRef::SignalLength, JBB);
    if (ERROR_INSERTED(5052)) {
      jam();
      signal->theData[0] = 9999;
      sendSignalWithDelay(CMVMI_REF, GSN_NDB_TAMPER, signal, 300, 1);
    }
  }
}

/**
 * This is needed for ndbmtd to serialize
 * SUB_GCP_COMPLETE_REP vs FIRE_TRIG_ORD
 */
void Dblqh::execSUB_GCP_COMPLETE_REP(Signal *signal) {
  jamEntry();
  Uint32 len = signal->getLength();
  EXECUTE_DIRECT(getDBTUP(), GSN_SUB_GCP_COMPLETE_REP, signal, len);
  sendSignal(SUMA_REF, GSN_SUB_GCP_COMPLETE_REP, signal, len, JBB);
}

/* ------------------------------------------------------------------------- */
/*  START TIME SUPERVISION OF THE LOG PARTS.                                 */
/* ------------------------------------------------------------------------- */
void Dblqh::startTimeSupervision(Signal *signal, LogPartRecord *logPartPtrP) {
  /* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   */
  /* WE HAVE TO START CHECKING IF THE LOG IS TO BE WRITTEN EVEN IF PAGES ARE */
  /* FULL. INITIALISE THE VALUES OF WHERE WE ARE IN THE LOG CURRENTLY. */
  /* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   */
  logPartPtrP->logPartTimer = 0;
  logPartPtrP->logTimer = 1;
  signal->theData[0] = ZTIME_SUPERVISION;
  signal->theData[1] = logPartPtrP->ptrI;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
}  // Dblqh::startTimeSupervision()

/*---------------------------------------------------------------------------*/
/* WE SET THE GLOBAL CHECKPOINT VARIABLES AFTER WRITING THE COMPLETED GCI LOG*/
/* RECORD. THIS ENSURES THAT WE WILL ENCOUNTER THE COMPLETED GCI RECORD WHEN */
/* WE EXECUTE THE FRAGMENT LOG.                                              */
/*---------------------------------------------------------------------------*/
void Dblqh::initGcpRecLab(Signal *signal) {
  /* ======================================================================== */
  /* =======               INITIATE GCP RECORD                        ======= */
  /*                                                                          */
  /*       SUBROUTINE SHORT NAME = IGR                                        */
  /* ======================================================================== */
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  ndbrequire(clogPartFileSize <= NDB_MAX_LOG_PARTS);
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    lock_log_part(logPartPtr.p);
    /*--------------------------------------------------*/
    /*       BY SETTING THE GCPREC = 0 WE START THE     */
    /*       CHECKING BY CHECK_GCP_COMPLETED. THIS      */
    /*       CHECKING MUST NOT BE STARTED UNTIL WE HAVE */
    /*       INSERTED ALL COMPLETE GCI LOG RECORDS IN   */
    /*       ALL LOG PARTS.                             */
    /*--------------------------------------------------*/
    logPartPtr.p->gcprec = 0;
    gcpPtr.p->gcpLogPartState[logPartPtr.i] = ZWAIT_DISK;
    gcpPtr.p->gcpSyncReady[logPartPtr.i] = ZFALSE;
    logFilePtr.i = logPartPtr.p->currentLogfile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    gcpPtr.p->gcpFilePtr[logPartPtr.i] = logFilePtr.i;
    logPagePtr.i = logFilePtr.p->currentLogpage;
    ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                  logPartPtr.p->logPageRecord);
    if (logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] == ZPAGE_HEADER_SIZE) {
      jam();
      /*--------------------------------------------------*/
      /*       SINCE THE CURRENT FILEPAGE POINTS AT THE   */
      /*       NEXT WORD TO BE WRITTEN WE HAVE TO ADJUST  */
      /*       FOR THIS BY DECREASING THE FILE PAGE BY ONE*/
      /*       IF NO WORD HAS BEEN WRITTEN ON THE CURRENT */
      /*       FILEPAGE.                                  */
      /*--------------------------------------------------*/
      gcpPtr.p->gcpPageNo[logPartPtr.i] = logFilePtr.p->currentFilepage - 1;
      gcpPtr.p->gcpWordNo[logPartPtr.i] = ZPAGE_SIZE - 1;
    } else {
      jam();
      gcpPtr.p->gcpPageNo[logPartPtr.i] = logFilePtr.p->currentFilepage;
      gcpPtr.p->gcpWordNo[logPartPtr.i] =
          logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] - 1;
    }  // if
    startTimeSupervision(signal, logPartPtr.p);
    unlock_log_part(logPartPtr.p);
  }  // for
  // initialize un-used part
  Uint32 Ti;
  for (Ti = clogPartFileSize; Ti < NDB_MAX_LOG_PARTS; Ti++) {
    gcpPtr.p->gcpFilePtr[Ti] = ZNIL;
    gcpPtr.p->gcpPageNo[Ti] = ZNIL;
    gcpPtr.p->gcpSyncReady[Ti] = false;
    gcpPtr.p->gcpWordNo[Ti] = ZNIL;
  }
}  // Dblqh::initGcpRecLab()

/* ========================================================================= */
/* ==== CHECK IF ANY GLOBAL CHECKPOINTS ARE COMPLETED AFTER A COMPLETED===== */
/*      DISK WRITE.                                                          */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = CGC                                         */
/* return: true if gcp was completed */
/* ========================================================================= */
bool Dblqh::checkGcpCompleted(Signal *signal, Uint32 tcgcPageWritten,
                              Uint32 tcgcWordWritten,
                              LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  UintR tcgcFlag;
  UintR tcgcJ;

  gcpPtr.i = logPartPtrP->gcprec;
  Uint32 logPartPtrI = logPartPtrP->ptrI;
  if (gcpPtr.i != RNIL) {
    jam();
    ndbrequire(logPartPtrI < NDB_MAX_LOG_PARTS);
    /* -------------------------------------------------------------------------
     */
    /* IF THE GLOBAL CHECKPOINT IS NOT WAITING FOR COMPLETION THEN WE CAN QUIT
     */
    /* THE SEARCH IMMEDIATELY. */
    /* -------------------------------------------------------------------------
     */
    ptrCheckGuard(gcpPtr, cgcprecFileSize, gcpRecord);
    if (gcpPtr.p->gcpFilePtr[logPartPtrI] == logFilePtr.i) {
      /* -------------------------------------------------------------------------
       */
      /* IF THE COMPLETED DISK OPERATION WAS ON ANOTHER FILE THAN THE ONE WE ARE
       */
      /* WAITING FOR, THEN WE CAN ALSO QUIT THE SEARCH IMMEDIATELY. */
      /* -------------------------------------------------------------------------
       */
      if (tcgcPageWritten < gcpPtr.p->gcpPageNo[logPartPtrI]) {
        jam();
        /* -------------------------------------------------------------------------
         */
        /* THIS LOG PART HAVE NOT YET WRITTEN THE GLOBAL CHECKPOINT TO DISK. */
        /* -------------------------------------------------------------------------
         */
        return false;
      } else {
        if (tcgcPageWritten == gcpPtr.p->gcpPageNo[logPartPtrI]) {
          if (tcgcWordWritten < gcpPtr.p->gcpWordNo[logPartPtrI]) {
            jam();
            /* -------------------------------------------------------------------------
             */
            /* THIS LOG PART HAVE NOT YET WRITTEN THE GLOBAL CHECKPOINT TO DISK.
             */
            /* -------------------------------------------------------------------------
             */
            return false;
          }  // if
        }    // if
      }      // if
      /* -------------------------------------------------------------------------
       */
      /* THIS LOG PART HAVE WRITTEN THE GLOBAL CHECKPOINT TO DISK. */
      /* -------------------------------------------------------------------------
       */
      logPartPtrP->gcprec = RNIL;
      gcpPtr.p->gcpLogPartState[logPartPtrI] = ZON_DISK;
      tcgcFlag = ZTRUE;
      for (tcgcJ = 0; tcgcJ < clogPartFileSize; tcgcJ++) {
        jam();
        if (gcpPtr.p->gcpLogPartState[tcgcJ] != ZON_DISK) {
          jam();
          /* -------------------------------------------------------------------------
           */
          /*ALL LOG PARTS HAVE NOT SAVED THIS GLOBAL CHECKPOINT TO DISK YET.
           * WAIT FOR  */
          /*THEM TO COMPLETE. */
          /* -------------------------------------------------------------------------
           */
          tcgcFlag = ZFALSE;
        }  // if
      }    // for
      if (tcgcFlag == ZFALSE) {
        return false;
      }

      if (tcgcFlag == ZTRUE) {
        jam();
        /* -------------------------------------------------------------------------
         */
        /*WE HAVE FOUND A COMPLETED GLOBAL CHECKPOINT OPERATION. WE NOW NEED TO
         * SEND */
        /*GCP_SAVECONF, REMOVE THE GCP RECORD FROM THE LIST OF WAITING GCP
         * RECORDS   */
        /*ON THIS LOG PART AND RELEASE THE GCP RECORD. */
        // After changing the log implementation we need to perform a FSSYNCREQ
        // on all log files where the last log word resided first before
        // proceeding.
        /* -------------------------------------------------------------------------
         */
        UintR Ti;
        for (Ti = 0; Ti < clogPartFileSize; Ti++) {
          LogFileRecordPtr loopLogFilePtr;
          loopLogFilePtr.i = gcpPtr.p->gcpFilePtr[Ti];
          ptrCheckGuard(loopLogFilePtr, clogFileFileSize, logFileRecord);
          if (loopLogFilePtr.p->logFileStatus == LogFileRecord::OPEN) {
            jam();
            signal->theData[0] = loopLogFilePtr.p->fileRef;
            signal->theData[1] = cownref;
            signal->theData[2] = gcpPtr.p->gcpFilePtr[Ti];
            sendSignal(NDBFS_REF, GSN_FSSYNCREQ, signal, 3, JBA);
          } else {
            ndbrequire(
                (loopLogFilePtr.p->logFileStatus == LogFileRecord::CLOSED) ||
                (loopLogFilePtr.p->logFileStatus ==
                 LogFileRecord::CLOSING_WRITE_LOG) ||
                (loopLogFilePtr.p->logFileStatus ==
                 LogFileRecord::OPENING_WRITE_LOG));
            signal->theData[0] = loopLogFilePtr.i;
            execFSSYNCCONF(signal);
          }  // if
        }    // for
      }      // if
    }        // if
    return true;
  }  // if
  return false;
}  // Dblqh::checkGcpCompleted()

void Dblqh::execFSSYNCCONF(Signal *signal) {
  GcpRecordPtr localGcpPtr;
  LogFileRecordPtr localLogFilePtr;
  LogPartRecordPtr localLogPartPtr;
  localLogFilePtr.i = signal->theData[0];
  ptrCheckGuard(localLogFilePtr, clogFileFileSize, logFileRecord);
  localLogPartPtr.i = localLogFilePtr.p->logPartRec;
  ptrCheckGuard(localLogPartPtr, clogPartFileSize, logPartRecord);
  localGcpPtr.i = ccurrentGcprec;
  ptrCheckGuard(localGcpPtr, cgcprecFileSize, gcpRecord);
  localGcpPtr.p->gcpSyncReady[localLogPartPtr.i] = ZTRUE;
  UintR Ti;

  if (DEBUG_REDO_EXEC) {
    g_eventLogger->info("(%u)part: %u file: %u gci: %u SYNC CONF", instance(),
                        localLogPartPtr.p->logPartNo, localLogFilePtr.p->fileNo,
                        localGcpPtr.p->gcpId);
  }
  for (Ti = 0; Ti < clogPartFileSize; Ti++) {
    jam();
    if (localGcpPtr.p->gcpSyncReady[Ti] == ZFALSE) {
      jam();
      return;
    }  // if
  }    // for

#ifdef GCP_TIMER_HACK
  globalData.gcp_timer_save[1] = NdbTick_getCurrentTicks();
#endif

  DEB_GCP(("(%u) GCP_SAVECONF for GCI: %u", instance(), localGcpPtr.p->gcpId));

  GCPSaveConf *const saveConf = (GCPSaveConf *)&signal->theData[0];
  saveConf->dihPtr = localGcpPtr.p->gcpUserptr;
  saveConf->nodeId = getOwnNodeId();
  saveConf->gci = localGcpPtr.p->gcpId;
  ndbrequire(refToMain(localGcpPtr.p->gcpBlockref) == DBDIH ||
             refToMain(localGcpPtr.p->gcpBlockref) == DBLQH);
  sendSignal(localGcpPtr.p->gcpBlockref, GSN_GCP_SAVECONF, signal,
             GCPSaveConf::SignalLength, JBA);
  ccurrentGcprec = RNIL;
}  // Dblqh::execFSSYNCCONF()

/* ######################################################################### */
/* #######                            FILE HANDLING MODULE           ####### */
/*                                                                           */
/* ######################################################################### */
/*       THIS MODULE HANDLES RESPONSE MESSAGES FROM THE FILE SYSTEM          */
/* ######################################################################### */
/* ######################################################################### */
/*       SIGNAL RECEPTION MODULE                                             */
/*       THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.            */
/*                                                                           */
/*  THIS MODULE CHECKS THE STATE AND JUMPS TO THE PROPER PART OF THE FILE    */
/*  HANDLING MODULE.                                                         */
/* ######################################################################### */
/* *************** */
/*  FSCLOSECONF  > */
/* *************** */
void Dblqh::execFSCLOSECONF(Signal *signal) {
  jamEntry();
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  logFilePtr.i = signal->theData[0];
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  logPartPtr.i = logFilePtr.p->logPartRec;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  /**
   * Lock only needed for normal operation close log which is the
   * state CLOSING_WRITE_LOG.
   */
  lock_log_part(logPartPtr.p);
  logFilePtr.p->fileRef = RNIL;

  if (DEBUG_REDO_EXEC || DEBUG_REDO_REC) {
    g_eventLogger->info("(%u)part: %u file: %u CLOSE CONF", instance(),
                        logPartPtr.p->logPartNo, logFilePtr.p->fileNo);
  }

  switch (logFilePtr.p->logFileStatus) {
    case LogFileRecord::CLOSE_SR_READ_INVALIDATE_PAGES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);

      readFileInInvalidate(signal, 2, logFilePtr, logPartPtr.p);
      return;

    case LogFileRecord::CLOSE_SR_READ_INVALIDATE_SEARCH_FILES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);

      readFileInInvalidate(signal, 4, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::CLOSE_SR_READ_INVALIDATE_SEARCH_LAST_FILE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);

      readFileInInvalidate(signal, 7, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::CLOSE_SR_WRITE_INVALIDATE_PAGES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);

      writeFileInInvalidate(signal, 1, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::CLOSING_INIT:
      jam();
      c_logFileInitDone++;
      unlock_log_part(logPartPtr.p);
      closingInitLab(signal, logFilePtr);
      return;
    case LogFileRecord::CLOSING_SR:
      jam();
      unlock_log_part(logPartPtr.p);
      closingSrLab(signal, logFilePtr);
      return;
    case LogFileRecord::CLOSING_EXEC_SR:
      jam();
      unlock_log_part(logPartPtr.p);
      closeExecSrLab(signal, logFilePtr);
      return;
    case LogFileRecord::CLOSING_EXEC_SR_COMPLETED:
      jam();
      unlock_log_part(logPartPtr.p);
      closeExecSrCompletedLab(signal, logFilePtr);
      return;
    case LogFileRecord::CLOSING_WRITE_LOG:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);
      return;
    case LogFileRecord::CLOSING_EXEC_LOG:
      jam();
      unlock_log_part(logPartPtr.p);
      closeExecLogLab(signal, logFilePtr);
      return;
    case LogFileRecord::CLOSING_EXEC_LOG_CACHED:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
      unlock_log_part(logPartPtr.p);
      release(signal, m_redo_open_file_cache);
      return;
    case LogFileRecord::CLOSING_SR_FRONTPAGE:
      jam();
      unlock_log_part(logPartPtr.p);
      closingSrFrontPage(signal, logFilePtr);
      return;
    default:
      jam();
      systemErrorLab(signal, __LINE__);
      return;
  }  // switch
}  // Dblqh::execFSCLOSECONF()

/* ************>> */
/*  FSOPENCONF  > */
/* ************>> */
void Dblqh::execFSOPENCONF(Signal *signal) {
  jamEntry();
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;

#ifdef ERROR_INSERT
  if (delayOpenFilePtrI > 0 && signal->theData[0] == delayOpenFilePtrI) {
    /* ERROR_INSERT 5090 : delay executing FSOPENCONF by sending
     * GSN_CONTINUEB in order to simulate a delay in opening a redo log file.
     * theData[0] of FSOPENCONF contains the LogFilePtr.i of the delayed file.
     * Add ZDELAY_FS_OPEN to theData[0] in addition to LogFilePtr.i,
     * in order to hint CONTINUEB to handle this signal.
     */
    Uint32 compact = signal->theData[0];
    signal->theData[0] = compact | (Uint32)ZDELAY_FS_OPEN << 16;
    sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
    return;
  }
#endif

  initFsopenconf(signal, logFilePtr, logPartPtr);
  /**
   * initFsopenconf locks log part when required, but it is only needed for
   * non-restart code. Restart code is handled from the log part owning thread.
   * Thus we can release lock before calling the actual function to start the
   * next part.
   *
   * During normal operation we only get here using OPENING_WRITE_LOG state.
   */
  switch (logFilePtr.p->logFileStatus) {
    case LogFileRecord::OPEN_SR_READ_INVALIDATE_PAGES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      readFileInInvalidate(signal, 0, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_READ_INVALIDATE_SEARCH_FILES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      readFileInInvalidate(signal, 5, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_WRITE_INVALIDATE_PAGES:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      writeFileInInvalidate(signal, 0, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPENING_INIT:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openFileInitLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_FRONTPAGE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openSrFrontpageLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_LAST_FILE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openSrLastFileLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_NEXT_FILE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openSrNextFileLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_EXEC_SR_START:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openExecSrStartLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_EXEC_SR_NEW_MBYTE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openExecSrNewMbyteLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_FOURTH_PHASE:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      openSrFourthPhaseLab(signal, logFilePtr, logPartPtr.p);
      unlock_log_part(logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_FOURTH_NEXT:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openSrFourthNextLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPEN_SR_FOURTH_ZERO:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      openSrFourthZeroLab(signal, logFilePtr, logPartPtr.p);
      return;
    case LogFileRecord::OPENING_WRITE_LOG:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      unlock_log_part(logPartPtr.p);
      return;
    case LogFileRecord::OPEN_EXEC_LOG:
      jam();
      logFilePtr.p->logFileStatus = LogFileRecord::OPEN;
      {
        jam();
        m_redo_open_file_cache.m_lru.addFirst(logFilePtr);
      }
      [[fallthrough]];
    case LogFileRecord::OPEN_EXEC_LOG_CACHED:
      jam();
      unlock_log_part(logPartPtr.p);
      openExecLogLab(signal, logFilePtr, logPartPtr.p);
      return;
    default:
      jam();
      systemErrorLab(signal, __LINE__);
      return;
  }  // switch
}  // Dblqh::execFSOPENCONF()

void Dblqh::execFSOPENREF(Signal *signal) {
  jamEntry();
  FsRef *ref = (FsRef *)signal->getDataPtr();
  Uint32 err = ref->errorCode;
  if (err == FsRef::fsErrInvalidFileSize) {
    char buf[256];
    BaseString::snprintf(buf, sizeof(buf),
                         "Invalid file size for redo logfile, "
                         " size only changable with --initial");
    progError(__LINE__, NDBD_EXIT_INVALID_CONFIG, buf);
    return;
  }

  SimulatedBlock::execFSOPENREF(signal);
}

/* ************>> */
/*  FSREADCONF  > */
/* ************>> */
void Dblqh::execFSREADCONF(Signal *signal) {
  jamEntry();
  LogFileOperationRecordPtr lfoPtr;
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  initFsrwconf(signal, false, lfoPtr, logPagePtr, logFilePtr, logPartPtr);

  switch (lfoPtr.p->lfoState) {
    case LogFileOperationRecord::READ_SR_LAST_MBYTE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrLastMbyteLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_FRONTPAGE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrFrontpageLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_LAST_FILE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrLastFileLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_NEXT_FILE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrNextFileLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_EXEC_SR:
      jam();
      readExecSrLab(signal, lfoPtr, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_EXEC_LOG:
      jam();
      readExecLogLab(signal, lfoPtr, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_INVALIDATE_PAGES:
      jam();
      invalidateLogAfterLastGCI(signal, lfoPtr, logPagePtr, logFilePtr,
                                logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_INVALIDATE_SEARCH_FILES:
      jam();
      invalidateLogAfterLastGCI(signal, lfoPtr, logPagePtr, logFilePtr,
                                logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_FOURTH_PHASE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrFourthPhaseLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    case LogFileOperationRecord::READ_SR_FOURTH_ZERO:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      readSrFourthZeroLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      return;
    default:
      jam();
      systemErrorLab(signal, __LINE__);
      return;
  }  // switch
}  // Dblqh::execFSREADCONF()

/* ************>> */
/*  FSREADCONF  > */
/* ************>> */
void Dblqh::execFSREADREF(Signal *signal) {
  jamEntry();
  LogFileOperationRecordPtr lfoPtr;
  lfoPtr.i = signal->theData[0];
  ptrCheckGuard(lfoPtr, clfoFileSize, logFileOperationRecord);
  switch (lfoPtr.p->lfoState) {
    case LogFileOperationRecord::READ_SR_LAST_MBYTE:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_FRONTPAGE:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_LAST_FILE:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_NEXT_FILE:
      jam();
      break;
    case LogFileOperationRecord::READ_EXEC_SR:
      jam();
      break;
    case LogFileOperationRecord::READ_EXEC_LOG:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_FOURTH_PHASE:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_FOURTH_ZERO:
      jam();
      break;
    case LogFileOperationRecord::READ_SR_INVALIDATE_PAGES:
      jam();
      break;
    default:
      jam();
      break;
  }  // switch
  {
    char msg[100];
    sprintf(msg,
            "File system read failed during LogFileOperationRecord state %d",
            (Uint32)lfoPtr.p->lfoState);
    fsRefError(signal, __LINE__, msg);
  }
}  // Dblqh::execFSREADREF()

/* *************** */
/*  FSWRITECONF  > */
/* *************** */
void Dblqh::execFSWRITECONF(Signal *signal) {
  jamEntry();
  LogFileOperationRecordPtr lfoPtr;
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  initFsrwconf(signal, true, lfoPtr, logPagePtr, logFilePtr, logPartPtr);
  switch (lfoPtr.p->lfoState) {
    case LogFileOperationRecord::WRITE_SR_INVALIDATE_PAGES:
      jam();
      invalidateLogAfterLastGCI(signal, lfoPtr, logPagePtr, logFilePtr,
                                logPartPtr.p);
      CRASH_INSERTION(5047);
      break;
    case LogFileOperationRecord::WRITE_PAGE_ZERO:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      writePageZeroLab(signal, __LINE__, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::LAST_WRITE_IN_FILE:
      jam();
      lastWriteInFileLab(signal, lfoPtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::INIT_WRITE_AT_END:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      initWriteEndLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::INIT_FIRST_PAGE:
      jam();
      c_logMBytesInitDone++;
      releaseLfo(lfoPtr, logPartPtr.p);
      initFirstPageLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::WRITE_GCI_ZERO:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      writeGciZeroLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::WRITE_DIRTY:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      writeDirtyLab(signal, logPartPtr.i);
      break;
    case LogFileOperationRecord::WRITE_INIT_MBYTE:
      jam();
      c_logMBytesInitDone++;
      releaseLfo(lfoPtr, logPartPtr.p);
      writeInitMbyteLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::ACTIVE_WRITE_LOG:
      jam();
      writeLogfileLab(signal, lfoPtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::FIRST_PAGE_WRITE_IN_LOGFILE:
      jam();
      releaseLfo(lfoPtr, logPartPtr.p);
      firstPageWriteLab(signal, logPagePtr, logFilePtr, logPartPtr.p);
      break;
    case LogFileOperationRecord::WRITE_SR_INVALIDATE_PAGES_UPDATE_PAGE0:
      jam();
      // We are done...send completed signal and exit this phase.
      releaseLfo(lfoPtr, logPartPtr.p);
      signal->theData[0] = ZSR_FOURTH_COMP;
      signal->theData[1] = logPartPtr.i;
      sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
      break;
    default:
      jam();
      systemErrorLab(signal, __LINE__);
      return;
  }  // switch
  unlock_log_part(logPartPtr.p);
}  // Dblqh::execFSWRITECONF()

/* ************>> */
/*  FSWRITEREF  > */
/* ************>> */
void Dblqh::execFSWRITEREF(Signal *signal) {
  jamEntry();
  LogFileOperationRecordPtr lfoPtr;
  lfoPtr.i = signal->theData[0];
  ptrCheckGuard(lfoPtr, clfoFileSize, logFileOperationRecord);
  terrorCode = signal->theData[1];
  switch (lfoPtr.p->lfoState) {
    case LogFileOperationRecord::WRITE_PAGE_ZERO:
      jam();
      break;
    case LogFileOperationRecord::LAST_WRITE_IN_FILE:
      jam();
      break;
    case LogFileOperationRecord::INIT_WRITE_AT_END:
      jam();
      break;
    case LogFileOperationRecord::INIT_FIRST_PAGE:
      jam();
      break;
    case LogFileOperationRecord::WRITE_GCI_ZERO:
      jam();
      break;
    case LogFileOperationRecord::WRITE_DIRTY:
      jam();
      break;
    case LogFileOperationRecord::WRITE_INIT_MBYTE:
      jam();
      break;
    case LogFileOperationRecord::ACTIVE_WRITE_LOG:
      jam();
      break;
    case LogFileOperationRecord::FIRST_PAGE_WRITE_IN_LOGFILE:
      jam();
      break;
    case LogFileOperationRecord::WRITE_SR_INVALIDATE_PAGES:
      jam();
      systemErrorLab(signal, __LINE__);
    default:
      jam();
      break;
  }  // switch
  {
    char msg[100];
    sprintf(msg,
            "File system write failed during LogFileOperationRecord state %d",
            (Uint32)lfoPtr.p->lfoState);
    fsRefError(signal, __LINE__, msg);
  }
}  // Dblqh::execFSWRITEREF()

/* ========================================================================= */
/* =======              INITIATE WHEN RECEIVING FSOPENCONF           ======= */
/*                                                                           */
/* ========================================================================= */
void Dblqh::initFsopenconf(Signal *signal, LogFileRecordPtr &logFilePtr,
                           LogPartRecordPtr &logPartPtr) {
  logFilePtr.i = signal->theData[0];
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  logPartPtr.i = logFilePtr.p->logPartRec;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  lock_log_part(logPartPtr.p);
  logFilePtr.p->fileRef = signal->theData[1];
  logFilePtr.p->currentMbyte = 0;
  logFilePtr.p->filePosition = 0;

  if (logFilePtr.p->fileChangeState == LogFileRecord::WAIT_FOR_OPEN_NEXT_FILE ||
      logFilePtr.p->fileChangeState == LogFileRecord::LAST_FILEWRITE_WAITS ||
      logFilePtr.p->fileChangeState == LogFileRecord::FIRST_FILEWRITE_WAITS) {
    jam();
    LogPageRecordPtr logPagePtr;
    logPagePtr.i = logFilePtr.p->currentLogpage;
    ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                  logPartPtr.p->logPageRecord);
    writeFileHeaderOpen(signal, ZNORMAL, logPagePtr, logFilePtr, logPartPtr.p);
    openNextLogfile(signal, logFilePtr.p, logPartPtr.p);

    if (logFilePtr.p->fileChangeState == LogFileRecord::WAIT_FOR_OPEN_NEXT_FILE)
      logFilePtr.p->fileChangeState = LogFileRecord::BOTH_WRITES_ONGOING;
    else if (logFilePtr.p->fileChangeState ==
             LogFileRecord::LAST_FILEWRITE_WAITS)
      logFilePtr.p->fileChangeState = LogFileRecord::FIRST_WRITE_ONGOING;
    else if (logFilePtr.p->fileChangeState ==
             LogFileRecord::FIRST_FILEWRITE_WAITS)
      logFilePtr.p->fileChangeState = LogFileRecord::LAST_WRITE_ONGOING;
  }
}  // Dblqh::initFsopenconf()

/* ========================================================================= */
/* =======       INITIATE WHEN RECEIVING FSREADCONF AND FSWRITECONF  ======= */
/*                                                                           */
/* ========================================================================= */
void Dblqh::initFsrwconf(Signal *signal, bool write,
                         LogFileOperationRecordPtr &lfoPtr,
                         LogPageRecordPtr &logPagePtr,
                         LogFileRecordPtr &logFilePtr,
                         LogPartRecordPtr &logPartPtr) {
  LogPageRecordPtr logP;
  Uint32 noPages, totPages;
  /**
   * The lfoPtr i-value to p-value translation always works since we use
   * the ArrayPool for Lfo records. Also the read of the log part from
   * Lfo record is stable since this is only set during startup. Thus
   * we can acquire the mutex before accessing any other data in the
   * Lfo record.
   *
   * It would otherwise be possible to execute this call before the
   * write of the lfoRecord have been completed in the sender thread.
   * (Important to remember that FSWRITEREQ can be sent by another thread
   *  than the thread receiving FSWRITECONF. So we have to take care that
   *  we have acquired the log part mutex to ensure that the Lfo record
   *  write is completed before we read any data).
   */
  lfoPtr.i = signal->theData[0];
  ptrCheckGuard(lfoPtr, clfoFileSize, logFileOperationRecord);
  logPartPtr.i = lfoPtr.p->logPartPtrI;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  if (write) {
    jam();
    lock_log_part(logPartPtr.p);
  }
  ndbrequire(logPartPtr.p->my_block == this);
  totPages = lfoPtr.p->noPagesRw;
  logFilePtr.i = lfoPtr.p->logFileRec;
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  logPagePtr.i = lfoPtr.p->firstLfoPage;
  ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                logPartPtr.p->logPageRecord);
  logP = logPagePtr;
  noPages = 1;
  ndbassert(totPages > 0);

  if (write) {
    Uint32 bytesWritten = totPages * 32768;
    logPartPtr.p->m_io_tracker.complete_io(bytesWritten);
  }

  for (;;) {
    logP.p->logPageWord[ZPOS_IN_WRITING] = 0;
    logP.p->logPageWord[ZPOS_IN_FREE_LIST] = 0;
    if (noPages == totPages) return;
    if (write)
      logP.i = logP.p->logPageWord[ZNEXT_PAGE];
    else
      logP.i = lfoPtr.p->logPageArray[noPages];
    ptrCheckGuard(logP, logPartPtr.p->logPageFileSize,
                  logPartPtr.p->logPageRecord);
    noPages++;
  }

}  // Dblqh::initFsrwconf()

/* ######################################################################### */
/*       NORMAL OPERATION MODULE                                             */
/*       THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.            */
/*                                                                           */
/*   THIS PART HANDLES THE NORMAL OPENING, CLOSING AND WRITING OF LOG FILES  */
/*   DURING NORMAL OPERATION.                                                */
/* ######################################################################### */
/*---------------------------------------------------------------------------*/
/* THIS SIGNAL IS USED TO SUPERVISE THAT THE LOG RECORDS ARE NOT KEPT IN MAIN*/
/* MEMORY FOR MORE THAN 1 SECOND TO ACHIEVE THE PROPER RELIABILITY.          */
/*---------------------------------------------------------------------------*/
void Dblqh::timeSup(Signal *signal) {
  LogFileOperationRecordPtr lfoPtr;
  LogPageRecordPtr origLogPagePtr;
  LogPageRecordPtr logPagePtr;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;
  Uint32 wordWritten;

  jamEntry();
  logPartPtr.i = signal->theData[0];
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  lock_log_part(logPartPtr.p);
  logFilePtr.i = logPartPtr.p->currentLogfile;
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  logPagePtr.i = logFilePtr.p->currentLogpage;
  ptrCheckGuard(logPagePtr, logPartPtr.p->logPageFileSize,
                logPartPtr.p->logPageRecord);
  if (logPartPtr.p->logPartTimer != logPartPtr.p->logTimer) {
    jam();

    /**
     * No file writes to this log part have been scheduled since
     * the part timer was set.
     * Now we will force a write of pending non-empty page or
     * pages for this part.
     * For the page currently being filled, we make a copy, and
     * write to file from the copy, so that we can continue filling
     * the original page with redo entries.
     * If the current file is not available to write due to a file
     * change problem, then we must wait until it is available,
     * supervised by some higher level timing mechanism.
     */

    if ((logPartPtr.p->m_log_problems & LogPartRecord::P_FILE_CHANGE_PROBLEM) !=
        0) {
      jam();
      unlock_log_part(logPartPtr.p);
      g_eventLogger->info(
          "LDM(%u): Gci record write is waiting for "
          "the redo log file to be changed: "
          "logpart: %u log part state: %u "
          "log part problem: %u "
          "file: %u ref %u logFileStatus %u"
          "fileChangeState %u "
          "current mbyte: %u "
          "logPagePtr.i %u ",
          instance(), logPartPtr.p->logPartNo, logPartPtr.p->logPartState,
          logPartPtr.p->m_log_problems, logFilePtr.p->fileNo,
          logFilePtr.p->fileRef, logFilePtr.p->logFileStatus,
          logFilePtr.p->fileChangeState, logFilePtr.p->currentMbyte,
          logPagePtr.i);
      /* Wait for current file to be ready for writes */
      signal->theData[0] = ZTIME_SUPERVISION;
      signal->theData[1] = logPartPtr.i;
      sendSignalWithDelay(cownref, GSN_CONTINUEB, signal, 50, 2);
      return;
    }

    {  // less merge conflicts
      if (((logFilePtr.p->currentFilepage + 1) & (ZPAGES_IN_MBYTE - 1)) == 0) {
        jam();
        /*---------------------------------------------------------------------------*/
        /* THIS IS THE LAST PAGE IN THIS MBYTE. WRITE NEXT LOG AND SWITCH TO
         * NEXT    */
        /* MBYTE. */
        /*---------------------------------------------------------------------------*/
        changeMbyte(signal, logPagePtr, logFilePtr, logPartPtr.p);
      } else {
        /*---------------------------------------------------------------------------*/
        /* WRITE THE LOG PAGE TO DISK EVEN IF IT IS NOT FULL. KEEP PAGE AND
         * WRITE A  */
        /* COPY. THE ORIGINAL PAGE WILL BE WRITTEN AGAIN LATER ON. */
        /*---------------------------------------------------------------------------*/
        wordWritten = logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] - 1;
        origLogPagePtr.i = logPagePtr.i;
        origLogPagePtr.p = logPagePtr.p;
        seizeLogpage(logPagePtr, logPartPtr.p);
        MEMCOPY_NO_WORDS(&logPagePtr.p->logPageWord[0],
                         &origLogPagePtr.p->logPageWord[0], wordWritten + 1);
        ndbrequire(wordWritten < ZPAGE_SIZE);
        if (logFilePtr.p->noLogpagesInBuffer > 0) {
          jam();
          completedLogPage(signal, ZENFORCE_WRITE, __LINE__, false, lfoPtr,
                           logPagePtr, logFilePtr, logPartPtr.p);
          /*---------------------------------------------------------------------------*/
          /*SINCE WE ARE ONLY WRITING PART OF THE LAST PAGE WE HAVE TO UPDATE
           * THE WORD */
          /*WRITTEN TO REFLECT THE REAL LAST WORD WRITTEN. WE ALSO HAVE TO MOVE
           * THE    */
          /*FILE POSITION ONE STEP BACKWARDS SINCE WE ARE NOT WRITING THE LAST
           * PAGE    */
          /*COMPLETELY. IT WILL BE WRITTEN AGAIN. */
          /*---------------------------------------------------------------------------*/
          lfoPtr.p->lfoWordWritten = wordWritten;
          logFilePtr.p->filePosition = logFilePtr.p->filePosition - 1;
        } else {
          if (wordWritten == (ZPAGE_HEADER_SIZE - 1)) {
            /*---------------------------------------------------------------------------*/
            /*THIS IS POSSIBLE BUT VERY UNLIKELY. IF THE PAGE WAS COMPLETED
             * AFTER THE LAST*/
            /*WRITE TO DISK THEN NO_LOG_PAGES_IN_BUFFER > 0 AND IF NOT WRITTEN
             * SINCE LAST*/
            /*WRITE TO DISK THEN THE PREVIOUS PAGE MUST HAVE BEEN WRITTEN BY
             * SOME        */
            /*OPERATION AND THAT BECAME COMPLETELY FULL. IN ANY CASE WE NEED NOT
             * WRITE AN*/
            /*EMPTY PAGE TO DISK. */
            /*---------------------------------------------------------------------------*/
            jam();
            releaseLogpage(logPagePtr, logPartPtr.p);
          } else {
            jam();
            writeSinglePage(signal, logFilePtr.p->currentFilepage, wordWritten,
                            __LINE__, true, lfoPtr, logPagePtr, logFilePtr,
                            logPartPtr.p);
            lfoPtr.p->lfoState = LogFileOperationRecord::ACTIVE_WRITE_LOG;
          }  // if
        }    // if
      }      // if
    }
  }
  logPartPtr.p->logTimer++;
  unlock_log_part(logPartPtr.p);
}  // Dblqh::timeSup()

void Dblqh::writeLogfileLab(Signal *signal, LogFileOperationRecordPtr lfoPtr,
                            LogFileRecordPtr logFilePtr,
                            LogPartRecord *logPartPtrP) {
  /*---------------------------------------------------------------------------*/
  /* CHECK IF ANY GLOBAL CHECKPOINTS ARE COMPLETED DUE TO THIS COMPLETED DISK */
  /* WRITE. */
  /*---------------------------------------------------------------------------*/
  switch (logFilePtr.p->fileChangeState) {
    case LogFileRecord::NOT_ONGOING:
      jam();
      checkGcpCompleted(signal,
                        ((lfoPtr.p->lfoPageNo + lfoPtr.p->noPagesRw) - 1),
                        lfoPtr.p->lfoWordWritten, logFilePtr, logPartPtrP);
      break;
#if 0
  case LogFileRecord::BOTH_WRITES_ONGOING:
    jam();
    g_eventLogger->info("not crashing!!");
    // Fall-through
#endif
    case LogFileRecord::WRITE_PAGE_ZERO_ONGOING:
    case LogFileRecord::LAST_WRITE_ONGOING:
      jam();
      logFilePtr.p->lastPageWritten =
          (lfoPtr.p->lfoPageNo + lfoPtr.p->noPagesRw) - 1;
      logFilePtr.p->lastWordWritten = lfoPtr.p->lfoWordWritten;
      break;
    default:
      jam();
      systemErrorLab(signal, __LINE__);
      return;
      break;
  }  // switch
  releaseLfoPages(lfoPtr, logPartPtrP);
  releaseLfo(lfoPtr, logPartPtrP);
}  // Dblqh::writeLogfileLab()

/* ######################################################################### */
/*       FILE CHANGE MODULE                                                  */
/*       THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.            */
/*                                                                           */
/*THIS PART OF THE FILE MODULE HANDLES WHEN WE ARE CHANGING LOG FILE DURING  */
/*NORMAL OPERATION. WE HAVE TO BE CAREFUL WHEN WE ARE CHANGING LOG FILE SO   */
/*THAT WE DO NOT COMPLICATE THE SYSTEM RESTART PROCESS TOO MUCH.             */
/*THE IDEA IS THAT WE START BY WRITING THE LAST WRITE IN THE OLD FILE AND WE */
/*ALSO WRITE THE FIRST PAGE OF THE NEW FILE CONCURRENT WITH THAT. THIS FIRST */
/*PAGE IN THE NEW FILE DO NOT CONTAIN ANY LOG RECORDS OTHER THAN A DESCRIPTOR*/
/*CONTAINING INFORMATION ABOUT GCI'S NEEDED AT SYSTEM RESTART AND A NEXT LOG */
/*RECORD.                                                                    */
/*                                                                           */
/*WHEN BOTH OF THOSE WRITES HAVE COMPLETED WE ALSO WRITE PAGE ZERO IN FILE   */
/*ZERO. THE ONLY INFORMATION WHICH IS INTERESTING HERE IS THE NEW FILE NUMBER*/
/*                                                                           */
/*IF OPTIMISATIONS ARE NEEDED OF THE LOG HANDLING THEN IT IS POSSIBLE TO     */
/*AVOID WRITING THE FIRST PAGE OF THE NEW PAGE IMMEDIATELY. THIS COMPLICATES */
/*THE SYSTEM RESTART AND ONE HAS TO TAKE SPECIAL CARE WITH FILE ZERO. IT IS  */
/*HOWEVER NO LARGE PROBLEM TO CHANGE INTO THIS SCENARIO. TO AVOID ALSO THE   */
/*WRITING OF PAGE ZERO IS ALSO POSSIBLE BUT COMPLICATES THE DESIGN EVEN      */
/*FURTHER. IT GETS FAIRLY COMPLEX TO FIND THE END OF THE LOG. SOME SORT OF   */
/*BINARY SEARCH IS HOWEVER MOST LIKELY A GOOD METHODOLOGY FOR THIS.          */
/* ######################################################################### */
void Dblqh::firstPageWriteLab(Signal *signal, LogPageRecordPtr logPagePtr,
                              LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  /*---------------------------------------------------------------------------*/
  /*       RELEASE PAGE ZERO IF THE FILE IS NOT FILE 0. */
  /*---------------------------------------------------------------------------*/
  Uint32 fileNo = logFilePtr.p->fileNo;
  if (fileNo != 0) {
    jam();
    releaseLogpage(logPagePtr, logPartPtrP);
  }  // if
  /*---------------------------------------------------------------------------*/
  /* IF A NEW FILE HAS BEEN OPENED WE SHALL ALWAYS ALSO WRITE TO PAGE O IN */
  /* FILE 0. THE AIM IS TO MAKE RESTARTS EASIER BY SPECIFYING WHICH IS THE */
  /* LAST FILE WHERE LOGGING HAS STARTED. */
  /*---------------------------------------------------------------------------*/
  /* FIRST CHECK WHETHER THE NEXT FILE IS OPENED AND THEN */
  /* THE LAST WRITE IN THE PREVIOUS FILE HAVE COMPLETED */
  /*---------------------------------------------------------------------------*/
  if (logFilePtr.p->fileChangeState == LogFileRecord::WAIT_FOR_OPEN_NEXT_FILE) {
    jam();
    logFilePtr.p->fileChangeState = LogFileRecord::FIRST_FILEWRITE_WAITS;
    return;
  } else if (logFilePtr.p->fileChangeState ==
             LogFileRecord::BOTH_WRITES_ONGOING) {
    jam();
    /*---------------------------------------------------------------------------*/
    /* THE LAST WRITE WAS STILL ONGOING. */
    /*---------------------------------------------------------------------------*/
    logFilePtr.p->fileChangeState = LogFileRecord::LAST_WRITE_ONGOING;
  } else {
    jam();
    ndbrequire(logFilePtr.p->fileChangeState ==
               LogFileRecord::FIRST_WRITE_ONGOING);
    /*---------------------------------------------------------------------------*/
    /* WRITE TO PAGE 0 IN IN FILE 0 NOW. */
    /*---------------------------------------------------------------------------*/
    logFilePtr.p->fileChangeState = LogFileRecord::WRITE_PAGE_ZERO_ONGOING;
    if (fileNo == 0) {
      jam();
      /*---------------------------------------------------------------------------*/
      /* IF THE NEW FILE WAS 0 THEN WE HAVE ALREADY WRITTEN PAGE ZERO IN FILE 0.
       */
      /*---------------------------------------------------------------------------*/
      // use writePageZeroLab to make sure that same code as normal is run
      writePageZeroLab(signal, __LINE__, logFilePtr, logPartPtrP);
    } else {
      jam();
      /*---------------------------------------------------------------------------*/
      /* WRITE PAGE ZERO IN FILE ZERO. LOG_FILE_REC WILL REFER TO THE LOG FILE
       * WE  */
      /* HAVE JUST WRITTEN PAGE ZERO IN TO GET HOLD OF LOG_FILE_PTR FOR THIS */
      /* RECORD QUICKLY. THIS IS NEEDED TO GET HOLD OF THE FILE_CHANGE_STATE. */
      /* THE ONLY INFORMATION WE WANT TO CHANGE IS THE LAST FILE NUMBER IN THE
       */
      /* FILE DESCRIPTOR. THIS IS USED AT SYSTEM RESTART TO FIND THE END OF THE
       */
      /* LOG PART. */
      /*---------------------------------------------------------------------------*/
      LogFileOperationRecordPtr lfoPtr;

      Uint32 currLogFile = logFilePtr.i;
      logFilePtr.i = logPartPtrP->firstLogfile;
      ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
      logPagePtr.i = logFilePtr.p->logPageZero;
      ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                    logPartPtrP->logPageRecord);
      logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_FILE_NO] = fileNo;
      writeSinglePage(signal, 0, ZPAGE_SIZE - 1, __LINE__, true, lfoPtr,
                      logPagePtr, logFilePtr, logPartPtrP);
      lfoPtr.p->logFileRec = currLogFile;
      lfoPtr.p->lfoState = LogFileOperationRecord::WRITE_PAGE_ZERO;
    }  // if
  }    // if
}  // Dblqh::firstPageWriteLab()

void Dblqh::lastWriteInFileLab(Signal *signal, LogFileOperationRecordPtr lfoPtr,
                               LogFileRecordPtr logFilePtr,
                               LogPartRecord *logPartPtrP) {
  LogFileRecordPtr locLogFilePtr;
  /*---------------------------------------------------------------------------*/
  /* CHECK IF ANY GLOBAL CHECKPOINTS ARE COMPLETED DUE TO THIS COMPLETED DISK */
  /* WRITE. */
  /*---------------------------------------------------------------------------*/
  checkGcpCompleted(signal, ((lfoPtr.p->lfoPageNo + lfoPtr.p->noPagesRw) - 1),
                    (ZPAGE_SIZE - 1), logFilePtr, logPartPtrP);
  releaseLfoPages(lfoPtr, logPartPtrP);
  releaseLfo(lfoPtr, logPartPtrP);
  /*---------------------------------------------------------------------------*/
  /* IF THE FILE IS NOT IN USE OR THE NEXT FILE TO BE USED WE WILL CLOSE IT. */
  /*---------------------------------------------------------------------------*/
  locLogFilePtr.i = logPartPtrP->currentLogfile;
  ptrCheckGuard(locLogFilePtr, clogFileFileSize, logFileRecord);
  if (logFilePtr.i != locLogFilePtr.i) {
    if (logFilePtr.i != locLogFilePtr.p->nextLogFile) {
      if (logFilePtr.p->fileNo != 0) {
        jam();
        /*---------------------------------------------------------------------------*/
        /* THE FILE IS NOT FILE ZERO EITHER. WE WILL NOT CLOSE FILE ZERO SINCE
         * WE    */
        /* USE IT TO KEEP TRACK OF THE CURRENT LOG FILE BY WRITING PAGE ZERO IN
         */
        /* FILE ZERO. */
        /*---------------------------------------------------------------------------*/
        /* WE WILL CLOSE THE FILE. */
        /*---------------------------------------------------------------------------*/
        logFilePtr.p->logFileStatus = LogFileRecord::CLOSING_WRITE_LOG;
        closeFile(signal, logFilePtr, __LINE__);
      }  // if
    }    // if
  }      // if
  /*---------------------------------------------------------------------------*/
  /* IF A NEW FILE HAS BEEN OPENED WE SHALL ALWAYS ALSO WRITE TO PAGE O IN */
  /* FILE 0. THE AIM IS TO MAKE RESTARTS EASIER BY SPECIFYING WHICH IS THE */
  /* LAST FILE WHERE LOGGING HAS STARTED. */
  /*---------------------------------------------------------------------------*/
  /* FIRST CHECK WHETHER THE NEXT FILE IS OPENED AND THEN */
  /* THE FIRST WRITE IN THE NEW FILE HAVE COMPLETED */
  /* THIS STATE INFORMATION IS IN THE NEW LOG FILE AND THUS WE HAVE TO MOVE */
  /* THE LOG FILE POINTER TO THIS LOG FILE. */
  /*---------------------------------------------------------------------------*/
  logFilePtr.i = logFilePtr.p->nextLogFile;
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);

  if (logFilePtr.p->fileChangeState == LogFileRecord::WAIT_FOR_OPEN_NEXT_FILE) {
    jam();
    logFilePtr.p->fileChangeState = LogFileRecord::LAST_FILEWRITE_WAITS;
    return;
  } else if (logFilePtr.p->fileChangeState ==
             LogFileRecord::BOTH_WRITES_ONGOING) {
    jam();
    /*---------------------------------------------------------------------------*/
    /* THE FIRST WRITE WAS STILL ONGOING. */
    /*---------------------------------------------------------------------------*/
    logFilePtr.p->fileChangeState = LogFileRecord::FIRST_WRITE_ONGOING;
    return;
  } else {
    ndbrequire(logFilePtr.p->fileChangeState ==
               LogFileRecord::LAST_WRITE_ONGOING);
    /*---------------------------------------------------------------------------*/
    /* WRITE TO PAGE 0 IN IN FILE 0 NOW. */
    /*---------------------------------------------------------------------------*/
    logFilePtr.p->fileChangeState = LogFileRecord::WRITE_PAGE_ZERO_ONGOING;
    Uint32 fileNo = logFilePtr.p->fileNo;
    if (fileNo == 0) {
      jam();
      /*---------------------------------------------------------------------------*/
      /* IF THE NEW FILE WAS 0 THEN WE HAVE ALREADY WRITTEN PAGE ZERO IN FILE 0.
       */
      /*---------------------------------------------------------------------------*/
      // use writePageZeroLab to make sure that same code as normal is run
      writePageZeroLab(signal, __LINE__, logFilePtr, logPartPtrP);
      return;
    } else {
      jam();
      /*---------------------------------------------------------------------------*/
      /* WRITE PAGE ZERO IN FILE ZERO. LOG_FILE_REC WILL REFER TO THE LOG FILE
       * WE  */
      /* HAVE JUST WRITTEN PAGE ZERO IN TO GET HOLD OF LOG_FILE_PTR FOR THIS */
      /* RECORD QUICKLY. THIS IS NEEDED TO GET HOLD OF THE FILE_CHANGE_STATE. */
      /* THE ONLY INFORMATION WE WANT TO CHANGE IS THE LAST FILE NUMBER IN THE
       */
      /* FILE DESCRIPTOR. THIS IS USED AT SYSTEM RESTART TO FIND THE END OF THE
       */
      /* LOG PART. */
      /*---------------------------------------------------------------------------*/
      LogPageRecordPtr logPagePtr;
      Uint32 currLogFile = logFilePtr.i;
      logFilePtr.i = logPartPtrP->firstLogfile;
      ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
      logPagePtr.i = logFilePtr.p->logPageZero;
      ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                    logPartPtrP->logPageRecord);
      logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_FILE_NO] = fileNo;
      writeSinglePage(signal, 0, ZPAGE_SIZE - 1, __LINE__, true, lfoPtr,
                      logPagePtr, logFilePtr, logPartPtrP);
      lfoPtr.p->logFileRec = currLogFile;
      lfoPtr.p->lfoState = LogFileOperationRecord::WRITE_PAGE_ZERO;
      return;
    }  // if
  }    // if
}  // Dblqh::lastWriteInFileLab()

void Dblqh::writePageZeroLab(Signal *signal, Uint32 from,
                             LogFileRecordPtr logFilePtr,
                             LogPartRecord *logPartPtrP) {
  if ((logPartPtrP->m_log_problems & LogPartRecord::P_FILE_CHANGE_PROBLEM) !=
      0) {
    jam();
    update_log_problem(signal, logPartPtrP,
                       LogPartRecord::P_FILE_CHANGE_PROBLEM,
                       /* clear */ false);
  }

  logFilePtr.p->fileChangeState = LogFileRecord::NOT_ONGOING;

  /*---------------------------------------------------------------------------*/
  /* IT COULD HAVE ARRIVED PAGE WRITES TO THE CURRENT FILE WHILE WE WERE */
  /* WAITING FOR THIS DISK WRITE TO COMPLETE. THEY COULD NOT CHECK FOR */
  /* COMPLETED GLOBAL CHECKPOINTS. THUS WE SHOULD DO THAT NOW INSTEAD. */
  /*---------------------------------------------------------------------------*/
  bool res =
      checkGcpCompleted(signal, logFilePtr.p->lastPageWritten,
                        logFilePtr.p->lastWordWritten, logFilePtr, logPartPtrP);
  if (res && false) {
    gcpPtr.i = ccurrentGcprec;
    ptrCheckGuard(gcpPtr, cgcprecFileSize, gcpRecord);

    infoEvent("Completing GCP %u in writePageZeroLab from %u", gcpPtr.p->gcpId,
              from);
  }
}  // Dblqh::writePageZeroLab()

/* ######################################################################### */
/*       INITIAL START MODULE                                                */
/*       THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.            */
/*                                                                           */
/*THIS MODULE INITIALISES ALL THE LOG FILES THAT ARE NEEDED AT A SYSTEM      */
/*RESTART AND WHICH ARE USED DURING NORMAL OPERATIONS. IT CREATES THE FILES  */
/*AND SETS A PROPER SIZE OF THEM AND INITIALISES THE FIRST PAGE IN EACH FILE */
/* ######################################################################### */
void Dblqh::openFileInitLab(Signal *signal, LogFileRecordPtr logFilePtr,
                            LogPartRecord *logPartPtrP) {
  LogPageRecordPtr logPagePtr;
  LogFileOperationRecordPtr lfoPtr;

  logFilePtr.p->logFileStatus = LogFileRecord::OPEN_INIT;
  seizeLogpage(logPagePtr, logPartPtrP);
  if (m_use_om_init == 0) {
    jam();
    initLogpage(logPagePtr, logFilePtr.p, logPartPtrP);
    writeSinglePage(signal, (clogFileSize * ZPAGES_IN_MBYTE) - 1,
                    ZPAGE_SIZE - 1, __LINE__, false, lfoPtr, logPagePtr,
                    logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::INIT_WRITE_AT_END;
  } else {
    jam();
    initWriteEndLab(signal, logPagePtr, logFilePtr, logPartPtrP);
  }
}  // Dblqh::openFileInitLab()

void Dblqh::initWriteEndLab(Signal *signal, LogPageRecordPtr logPagePtr,
                            LogFileRecordPtr logFilePtr,
                            LogPartRecord *logPartPtrP) {
  initLogpage(logPagePtr, logFilePtr.p, logPartPtrP);
  if (logFilePtr.p->fileNo == 0) {
    jam();
    /*---------------------------------------------------------------------------*/
    /* PAGE ZERO IN FILE ZERO MUST SET LOG LAP TO ONE SINCE IT HAS STARTED */
    /* WRITING TO THE LOG, ALSO GLOBAL CHECKPOINTS ARE SET TO ZERO. */
    /* Set number of log parts used to ensure we use correct number of log parts
     */
    /* at system restart. Was previously hardcoded to 4. */
    /*---------------------------------------------------------------------------*/
    logPagePtr.p->logPageWord[ZPOS_NO_LOG_PARTS] = globalData.ndbLogParts;
    logPagePtr.p->logPageWord[ZPOS_LOG_LAP] = 1;
    logPagePtr.p->logPageWord[ZPOS_MAX_GCI_STARTED] = 0;
    logPagePtr.p->logPageWord[ZPOS_MAX_GCI_COMPLETED] = 0;
    logFilePtr.p->logMaxGciStarted[0] = 0;
    logFilePtr.p->logMaxGciCompleted[0] = 0;
  }  // if
  /*---------------------------------------------------------------------------*/
  /* REUSE CODE FOR INITIALISATION OF FIRST PAGE IN ALL LOG FILES. */
  /*---------------------------------------------------------------------------*/
  writeFileHeaderOpen(signal, ZINIT, logPagePtr, logFilePtr, logPartPtrP);
}  // Dblqh::initWriteEndLab()

void Dblqh::initFirstPageLab(Signal *signal, LogPageRecordPtr logPagePtr,
                             LogFileRecordPtr logFilePtr,
                             LogPartRecord *logPartPtrP) {
  if (logFilePtr.p->fileNo == 0) {
    jam();
    /*---------------------------------------------------------------------------*/
    /* IN FILE ZERO WE WILL INSERT A PAGE ONE WHERE WE WILL INSERT A COMPLETED
     */
    /* GCI RECORD FOR GCI = 0. */
    /*---------------------------------------------------------------------------*/
    LogFileOperationRecordPtr lfoPtr;

    initLogpage(logPagePtr, logFilePtr.p, logPartPtrP);
    logPagePtr.p->logPageWord[ZPOS_LOG_LAP] = 1;
    logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE] = ZCOMPLETED_GCI_TYPE;
    logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + 1] = 1;
    writeSinglePage(signal, 1, ZPAGE_SIZE - 1, __LINE__, false, lfoPtr,
                    logPagePtr, logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::WRITE_GCI_ZERO;
    return;
  }  // if
  logFilePtr.p->currentMbyte = 1;
  writeInitMbyte(signal, logPagePtr, logFilePtr, logPartPtrP);
}  // Dblqh::initFirstPageLab()

void Dblqh::writeGciZeroLab(Signal *signal, LogPageRecordPtr logPagePtr,
                            LogFileRecordPtr logFilePtr,
                            LogPartRecord *logPartPtrP) {
  logFilePtr.p->currentMbyte = 1;
  writeInitMbyte(signal, logPagePtr, logFilePtr, logPartPtrP);
}  // Dblqh::writeGciZeroLab()

void Dblqh::writeInitMbyteLab(Signal *signal, LogPageRecordPtr logPagePtr,
                              LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  logFilePtr.p->currentMbyte = logFilePtr.p->currentMbyte + 1;
  if (logFilePtr.p->currentMbyte == clogFileSize) {
    jam();
    releaseLogpage(logPagePtr, logPartPtrP);
    logFilePtr.p->logFileStatus = LogFileRecord::CLOSING_INIT;
    closeFile(signal, logFilePtr, __LINE__);
    return;
  }  // if
  writeInitMbyte(signal, logPagePtr, logFilePtr, logPartPtrP);
}  // Dblqh::writeInitMbyteLab()

void Dblqh::closingInitLab(Signal *signal, LogFileRecordPtr logFilePtr) {
  LogPartRecordPtr logPartPtr;
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
  logPartPtr.i = logFilePtr.p->logPartRec;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  if (logFilePtr.p->nextLogFile == logPartPtr.p->firstLogfile) {
    jam();
    checkInitCompletedLab(signal, logPartPtr.p);
  } else {
    jam();
    logFilePtr.i = logFilePtr.p->nextLogFile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    openLogfileInit(signal, logFilePtr);
  }  // if
}  // Dblqh::closingInitLab()

void Dblqh::checkInitCompletedLab(Signal *signal, LogPartRecord *logPartPtrP) {
  LogFileRecordPtr logFilePtr;
  logPartPtrP->logPartState = LogPartRecord::SR_FIRST_PHASE_COMPLETED;
  g_eventLogger->info(
      "LDM(%u): Completed REDO log initialisation of"
      " logPart %u",
      instance(), logPartPtrP->logPartNo);
  csrExecUndoLogState = EULS_COMPLETED;
  /*---------------------------------------------------------------------------*/
  /* WE HAVE NOW INITIALISED ALL FILES IN THIS LOG PART. WE CAN NOW SET THE */
  /* THE LOG LAP TO ONE SINCE WE WILL START WITH LOG LAP ONE. LOG LAP = ZERO */
  /* MEANS THIS PART OF THE LOG IS NOT WRITTEN YET. */
  /*---------------------------------------------------------------------------*/
  logPartPtrP->logLap = 1;

  if (m_use_om_init && (logPartPtrP->ptrI + 1) != clogPartFileSize) {
    jam();
    LogPartRecordPtr logPartPtr;
    logPartPtr.i = logPartPtrP->ptrI + 1;
    ptrAss(logPartPtr, logPartRecord);
    logFilePtr.i = logPartPtr.p->firstLogfile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    openLogfileInit(signal, logFilePtr);
    return;
  }

  LogPartRecordPtr logPartPtr;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    if (logPartPtr.p->logPartState != LogPartRecord::SR_FIRST_PHASE_COMPLETED) {
      jam();
      /*---------------------------------------------------------------------------*/
      /* THIS PART HAS STILL NOT COMPLETED. WAIT FOR THIS TO OCCUR. */
      /*---------------------------------------------------------------------------*/
      return;
    }  // if
  }

#if defined(USE_INIT_GLOBAL_VARIABLES)
  enable_global_variables();
#endif
  g_eventLogger->info("LDM(%u): Completed REDO initialisation", instance());
  logfileInitCompleteReport(signal);
  sendNdbSttorryLab(signal);
}

/* ========================================================================= */
/* =======       INITIATE LOG FILE OPERATION RECORD WHEN ALLOCATED   ======= */
/*                                                                           */
/* ========================================================================= */
void Dblqh::initLfo(LogFileOperationRecord *lfoPtrP, Uint32 logFilePtrI) {
  lfoPtrP->firstLfoPage = RNIL;
  lfoPtrP->lfoState = LogFileOperationRecord::IDLE;
  lfoPtrP->logFileRec = logFilePtrI;
  lfoPtrP->noPagesRw = 0;
  lfoPtrP->lfoPageNo = ZNIL;
}  // Dblqh::initLfo()

/* ========================================================================= */
/* =======              INITIATE LOG FILE WHEN ALLOCATED             ======= */
/*                                                                           */
/*       INPUT:  TFILE_NO        NUMBER OF THE FILE INITIATED                */
/*               LOG_PART_PTR    NUMBER OF LOG PART                          */
/*       SUBROUTINE SHORT NAME = IL                                          */
/* ========================================================================= */
void Dblqh::initLogfile(LogFileRecordPtr logFilePtr, Uint32 partNo,
                        Uint32 fileNo, Uint32 logPartPtrI) {
  UintR tilTmp;
  UintR tilIndex;

  logFilePtr.p->currentFilepage = 0;
  logFilePtr.p->currentLogpage = RNIL;
  logFilePtr.p->fileName[0] = (UintR)-1;
  logFilePtr.p->fileName[1] = (UintR)-1; /* = H'FFFFFFFF = -1 */
  logFilePtr.p->fileName[2] = fileNo;    /* Sfile_no */
  tilTmp = 1;                            /* VERSION 1 OF FILE NAME */
  tilTmp = (tilTmp << 8) + 1; /* FRAGMENT LOG => .FRAGLOG AS EXTENSION */
  tilTmp = (tilTmp << 8) + (8 + partNo); /* DIRECTORY = D(8+Part)/DBLQH */
  tilTmp = (tilTmp << 8) + 255;          /* IGNORE Pxx PART OF FILE NAME */
  logFilePtr.p->fileName[3] = tilTmp;
  /* =========================================================================
   */
  /*       FILE NAME BECOMES /D2/DBLQH/Tpart_no/Sfile_no.FRAGLOG */
  /* =========================================================================
   */
  logFilePtr.p->fileNo = fileNo;
  logFilePtr.p->filePosition = 0;
  logFilePtr.p->firstLfo = RNIL;
  logFilePtr.p->lastLfo = RNIL;
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
  logFilePtr.p->logPartRec = logPartPtrI;
  logFilePtr.p->noLogpagesInBuffer = 0;
  logFilePtr.p->firstFilledPage = RNIL;
  logFilePtr.p->lastFilledPage = RNIL;
  logFilePtr.p->lastPageWritten = 0;
  logFilePtr.p->logPageZero = RNIL;
  logFilePtr.p->currentMbyte = 0;
  for (tilIndex = 0; tilIndex < clogFileSize; tilIndex++) {
    logFilePtr.p->logMaxGciCompleted[tilIndex] = (UintR)-1;
    logFilePtr.p->logMaxGciStarted[tilIndex] = (UintR)-1;
    logFilePtr.p->logLastPrepRef[tilIndex] = 0;
  }  // for
}  // Dblqh::initLogfile()

/* ========================================================================= */
/* =======              INITIATE LOG PAGE WHEN ALLOCATED             ======= */
/*                                                                           */
/* ========================================================================= */
void Dblqh::initLogpage(LogPageRecordPtr logPagePtr, LogFileRecord *logFilePtrP,
                        LogPartRecord *logPartPtrP) {
  TcConnectionrecPtr ilpTcConnectptr;

  /* Ensure all non-used header words are zero */
  std::memset(logPagePtr.p, 0, sizeof(Uint32) * ZPAGE_HEADER_SIZE);
  logPagePtr.p->logPageWord[ZPOS_LOG_LAP] = logPartPtrP->logLap;
  logPagePtr.p->logPageWord[ZPOS_MAX_GCI_COMPLETED] =
      logPartPtrP->logPartNewestCompletedGCI;
  logPagePtr.p->logPageWord[ZPOS_MAX_GCI_STARTED] =
      logPartPtrP->my_block->cnewestGci;
  logPagePtr.p->logPageWord[ZPOS_VERSION] = NDB_VERSION;
  logPagePtr.p->logPageWord[ZPOS_NO_LOG_FILES] = logPartPtrP->noLogFiles;
  logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] = ZPAGE_HEADER_SIZE;
  logPagePtr.p->logPageWord[ZPOS_NO_LOG_PARTS] = globalData.ndbLogParts;
  ilpTcConnectptr.p = logPartPtrP->firstLogTcrec;
  if (ilpTcConnectptr.p != nullptr) {
    jam();
    ndbrequire(Magic::check_ptr(ilpTcConnectptr.p));
    logPagePtr.p->logPageWord[ZLAST_LOG_PREP_REF] =
        (ilpTcConnectptr.p->logStartFileNo << 16) +
        (ilpTcConnectptr.p->logStartPageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE);
  } else {
    jam();
    logPagePtr.p->logPageWord[ZLAST_LOG_PREP_REF] =
        (logFilePtrP->fileNo << 16) +
        (logFilePtrP->currentFilepage >> ZTWOLOG_NO_PAGES_IN_MBYTE);
  }  // if
}  // Dblqh::initLogpage()

/* ------------------------------------------------------------------------- */
/* -------               OPEN LOG FILE FOR READ AND WRITE            ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = OFR                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::openFileRw(Signal *signal, LogFileRecordPtr olfLogFilePtr,
                       bool writeBuffer) {
  FsOpenReq *req = (FsOpenReq *)signal->getDataPtrSend();
  req->userReference = cownref;
  req->userPointer = olfLogFilePtr.i;
  req->fileNumber[0] = olfLogFilePtr.p->fileName[0];
  req->fileNumber[1] = olfLogFilePtr.p->fileName[1];
  req->fileNumber[2] = olfLogFilePtr.p->fileName[2];
  req->fileNumber[3] = olfLogFilePtr.p->fileName[3];
  req->fileFlags = FsOpenReq::OM_READWRITE | FsOpenReq::OM_AUTOSYNC |
                   FsOpenReq::OM_CHECK_SIZE | FsOpenReq::OM_ZEROS_ARE_SPARSE;
  if (c_o_direct) {
    jam();
    req->fileFlags |= FsOpenReq::OM_DIRECT;
    if (c_o_direct_sync_flag) {
      jam();
      req->fileFlags |= FsOpenReq::OM_DIRECT_SYNC;
    }
  }
  if (writeBuffer) {
    req->fileFlags |= FsOpenReq::OM_WRITE_BUFFER;
  }
  if (c_encrypted_filesystem) {
    jam();
    req->fileFlags |= FsOpenReq::OM_ENCRYPT_XTS;
  }

  req->auto_sync_size = MAX_REDO_PAGES_WITHOUT_SYNCH * sizeof(LogPageRecord);
  Uint64 sz = clogFileSize;
  sz *= 1024;
  sz *= 1024;
  req->file_size_hi = (Uint32)(sz >> 32);
  req->file_size_lo = (Uint32)(sz & 0xFFFFFFFF);
  req->page_size = File_formats::NDB_PAGE_SIZE;
  if (req->fileFlags & FsOpenReq::OM_ENCRYPT_CIPHER_MASK) {
    LinearSectionPtr lsptr[3];

    // Use a dummy file name
    ndbrequire(FsOpenReq::getVersion(req->fileNumber) != 4);
    lsptr[FsOpenReq::FILENAME].p = nullptr;
    lsptr[FsOpenReq::FILENAME].sz = 0;

    req->fileFlags |= FsOpenReq::OM_ENCRYPT_KEY;

    EncryptionKeyMaterial nmk;
    nmk.length = globalData.nodeMasterKeyLength;
    memcpy(&nmk.data, globalData.nodeMasterKey, globalData.nodeMasterKeyLength);
    lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].p = (const Uint32 *)&nmk;
    lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].sz = nmk.get_needed_words();

    sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength, JBA,
               lsptr, 2);
  } else {
    sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength, JBA);
  }
}  // Dblqh::openFileRw()

/* ------------------------------------------------------------------------- */
/* -------               OPEN LOG FILE DURING INITIAL START          ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = OLI                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::openLogfileInit(Signal *signal, LogFileRecordPtr logFilePtr) {
  logFilePtr.p->logFileStatus = LogFileRecord::OPENING_INIT;
  FsOpenReq *req = (FsOpenReq *)signal->getDataPtrSend();
  req->userReference = cownref;
  req->userPointer = logFilePtr.i;
  req->fileNumber[0] = logFilePtr.p->fileName[0];
  req->fileNumber[1] = logFilePtr.p->fileName[1];
  req->fileNumber[2] = logFilePtr.p->fileName[2];
  req->fileNumber[3] = logFilePtr.p->fileName[3];
  req->fileFlags = FsOpenReq::OM_READWRITE | FsOpenReq::OM_TRUNCATE |
                   FsOpenReq::OM_CREATE | FsOpenReq::OM_AUTOSYNC |
                   FsOpenReq::OM_WRITE_BUFFER | FsOpenReq::OM_ZEROS_ARE_SPARSE;
  if (c_o_direct) {
    jam();
    req->fileFlags |= FsOpenReq::OM_DIRECT;
    if (c_o_direct_sync_flag) {
      jam();
      req->fileFlags |= FsOpenReq::OM_DIRECT_SYNC;
    }
  }
  Uint64 sz = Uint64(clogFileSize) * 1024 * 1024;
  req->file_size_hi = Uint32(sz >> 32);
  req->file_size_lo = Uint32(sz);
  req->page_size = File_formats::NDB_PAGE_SIZE;
  if (m_use_om_init) {
    jam();
    req->fileFlags |= FsOpenReq::OM_INIT;
  } else {
    jam();
    req->fileFlags |= FsOpenReq::OM_SPARSE_INIT;
  }
  if (c_encrypted_filesystem) {
    jam();
    req->fileFlags |= FsOpenReq::OM_ENCRYPT_XTS;
  }

  req->auto_sync_size = MAX_REDO_PAGES_WITHOUT_SYNCH * sizeof(LogPageRecord);
  if (req->fileFlags & FsOpenReq::OM_ENCRYPT_CIPHER_MASK) {
    LinearSectionPtr lsptr[3];

    // Use a dummy file name
    ndbrequire(FsOpenReq::getVersion(req->fileNumber) != 4);
    lsptr[FsOpenReq::FILENAME].p = nullptr;
    lsptr[FsOpenReq::FILENAME].sz = 0;

    req->fileFlags |= FsOpenReq::OM_ENCRYPT_KEY;
    EncryptionKeyMaterial nmk;
    nmk.length = globalData.nodeMasterKeyLength;
    memcpy(&nmk.data, globalData.nodeMasterKey, globalData.nodeMasterKeyLength);
    lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].p = (const Uint32 *)&nmk;
    lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].sz = nmk.get_needed_words();

    sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength, JBA,
               lsptr, 2);
  } else {
    sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength, JBA);
  }
}  // Dblqh::openLogfileInit()

void Dblqh::execFSWRITEREQ(const FsReadWriteReq *req)
    const /* called direct cross threads from Ndbfs */
{
  /**
   * This is currently run in other thread -> no jam
   *   and no global variables
   *
   * This method is called from NDB file system while initialising a REDO log
   * file, so we need to ensure that we don't touch any block variables other
   * than to read stable variables. This is only called during initial
   * restart. The pages are allocated by NDBFS from DataMemory, so these can
   * be written to safely since they are owned by the file system thread.
   */
  Ptr<GlobalPage> page_ptr;
  ndbrequire(req->getFormatFlag(req->operationFlag) == req->fsFormatSharedPage);
  ndbrequire(
      m_shared_page_pool.getPtr(page_ptr, req->data.sharedPage.pageNumber));

  LogFileRecordPtr currLogFilePtr;
  currLogFilePtr.i = req->userPointer;
  ptrCheckGuard(currLogFilePtr, clogFileFileSize, logFileRecord);

  LogPartRecordPtr currLogPartPtr;
  currLogPartPtr.i = currLogFilePtr.p->logPartRec;
  ptrCheckGuard(currLogPartPtr, clogPartFileSize, logPartRecord);

  Uint32 page_no = req->varIndex;
  LogPageRecordPtr currLogPagePtr;
  currLogPagePtr.p = (LogPageRecord *)page_ptr.p;

  std::memset(page_ptr.p, 0, sizeof(LogPageRecord));
  if (page_no == 0) {
    // keep writing these afterwards
  } else if (((page_no % ZPAGES_IN_MBYTE) == 0) ||
             (page_no == ((clogFileSize * ZPAGES_IN_MBYTE) - 1))) {
    currLogPagePtr.p->logPageWord[ZPOS_LOG_LAP] = currLogPartPtr.p->logLap;
    currLogPagePtr.p->logPageWord[ZPOS_MAX_GCI_COMPLETED] =
        currLogPartPtr.p->logPartNewestCompletedGCI;
    currLogPagePtr.p->logPageWord[ZPOS_MAX_GCI_STARTED] = cnewestGci;
    currLogPagePtr.p->logPageWord[ZPOS_VERSION] = NDB_VERSION;
    currLogPagePtr.p->logPageWord[ZPOS_NO_LOG_FILES] =
        currLogPartPtr.p->noLogFiles;
    currLogPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] = ZPAGE_HEADER_SIZE;
    currLogPagePtr.p->logPageWord[ZLAST_LOG_PREP_REF] =
        (currLogFilePtr.p->fileNo << 16) +
        (currLogFilePtr.p->currentFilepage >> ZTWOLOG_NO_PAGES_IN_MBYTE);

    currLogPagePtr.p->logPageWord[ZNEXT_PAGE] = RNIL;
    currLogPagePtr.p->logPageWord[ZPOS_CHECKSUM] =
        calcPageCheckSum(currLogPagePtr);
  } else if (0) {
    currLogPagePtr.p->logPageWord[ZNEXT_PAGE] = RNIL;
    currLogPagePtr.p->logPageWord[ZPOS_CHECKSUM] =
        calcPageCheckSum(currLogPagePtr);
  }
}

/* OPEN FOR READ/WRITE, DO CREATE AND DO TRUNCATE FILE */
/* ------------------------------------------------------------------------- */
/* -------               OPEN NEXT LOG FILE                          ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = ONL                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::openNextLogfile(Signal *signal, LogFileRecord *logFilePtrP,
                            LogPartRecord *logPartPtrP) {
  LogFileRecordPtr onlLogFilePtr;

  if (logPartPtrP->noLogFiles > 2) {
    jam();
    /* -------------------------------------------------- */
    /*       IF ONLY 1 OR 2 LOG FILES EXIST THEN THEY ARE */
    /*       ALWAYS OPEN AND THUS IT IS NOT NECESSARY TO  */
    /*       OPEN THEM NOW.                               */
    /* -------------------------------------------------- */
    onlLogFilePtr.i = logFilePtrP->nextLogFile;
    ptrCheckGuard(onlLogFilePtr, logPartPtrP->my_block->clogFileFileSize,
                  logPartPtrP->my_block->logFileRecord);

#ifdef ERROR_INSERT
    if (delayOpenFilePtrI == 0 && onlLogFilePtr.p->fileNo > 3 &&
        ERROR_INSERTED_CLEAR(5090)) {
      /* Instruct execFSOPENCONF to delay the execution of the
       * signal for fileNo>3 to simulate a delay in opening it.
       * (Choice of '>3': File 0 is held open. Let files 1-3
       * being filled and opened normally. The next file belonging
       * to the log part being filled by the test will be delayed).
       */
      delayOpenFilePtrI = logFilePtrP->nextLogFile;
    }
#endif

    if (onlLogFilePtr.p->logFileStatus != LogFileRecord::CLOSED) {
      ndbrequire(onlLogFilePtr.p->fileNo == 0);
      return;
    }  // if
    onlLogFilePtr.p->logFileStatus = LogFileRecord::OPENING_WRITE_LOG;
    FsOpenReq *req = (FsOpenReq *)signal->getDataPtrSend();
    req->userReference = logPartPtrP->myRef;
    req->userPointer = onlLogFilePtr.i;
    req->fileNumber[0] = onlLogFilePtr.p->fileName[0];
    req->fileNumber[1] = onlLogFilePtr.p->fileName[1];
    req->fileNumber[2] = onlLogFilePtr.p->fileName[2];
    req->fileNumber[3] = onlLogFilePtr.p->fileName[3];
    req->fileFlags = FsOpenReq::OM_READWRITE | FsOpenReq::OM_AUTOSYNC |
                     FsOpenReq::OM_CHECK_SIZE | FsOpenReq::OM_WRITE_BUFFER |
                     FsOpenReq::OM_ZEROS_ARE_SPARSE;
    if (c_o_direct) {
      jam();
      req->fileFlags |= FsOpenReq::OM_DIRECT;
      if (c_o_direct_sync_flag) {
        jam();
        req->fileFlags |= FsOpenReq::OM_DIRECT_SYNC;
      }
    }
    if (c_encrypted_filesystem) {
      jam();
      req->fileFlags |= FsOpenReq::OM_ENCRYPT_XTS;
    }
    req->auto_sync_size = MAX_REDO_PAGES_WITHOUT_SYNCH * sizeof(LogPageRecord);
    Uint64 sz = logPartPtrP->my_block->clogFileSize;
    sz *= 1024;
    sz *= 1024;
    req->file_size_hi = (Uint32)(sz >> 32);
    req->file_size_lo = (Uint32)(sz & 0xFFFFFFFF);
    req->page_size = File_formats::NDB_PAGE_SIZE;
    if (req->fileFlags & FsOpenReq::OM_ENCRYPT_CIPHER_MASK) {
      LinearSectionPtr lsptr[3];

      // Use a dummy file name
      ndbrequire(FsOpenReq::getVersion(req->fileNumber) != 4);
      lsptr[FsOpenReq::FILENAME].p = nullptr;
      lsptr[FsOpenReq::FILENAME].sz = 0;

      req->fileFlags |= FsOpenReq::OM_ENCRYPT_KEY;
      EncryptionKeyMaterial nmk;
      nmk.length = globalData.nodeMasterKeyLength;
      memcpy(&nmk.data, globalData.nodeMasterKey,
             globalData.nodeMasterKeyLength);
      lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].p = (const Uint32 *)&nmk;
      lsptr[FsOpenReq::ENCRYPT_KEY_MATERIAL].sz = nmk.get_needed_words();

      sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength, JBA,
                 lsptr, 2);
    } else {
      sendSignal(NDBFS_REF, GSN_FSOPENREQ, signal, FsOpenReq::SignalLength,
                 JBA);
    }
  }  // if
}  // Dblqh::openNextLogfile()

/* OPEN FOR READ/WRITE, DON'T CREATE AND DON'T TRUNCATE FILE */
/* ------------------------------------------------------------------------- */
/* -------                       RELEASE LFO RECORD                  ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::releaseLfo(LogFileOperationRecordPtr lfoPtr,
                       LogPartRecord *logPartPtrP) {
#ifdef VM_TRACE
  // Check that lfo record isn't already in free list
  LogFileOperationRecordPtr TlfoPtr;
  TlfoPtr.i = logPartPtrP->firstFreeLfo;
  while (TlfoPtr.i != RNIL) {
    ptrCheckGuard(TlfoPtr, logPartPtrP->my_block->clfoFileSize,
                  logPartPtrP->my_block->logFileOperationRecord);
    ndbrequire(TlfoPtr.i != lfoPtr.i);
    TlfoPtr.i = TlfoPtr.p->nextLfo;
  }
#endif
  lfoPtr.p->nextLfo = logPartPtrP->firstFreeLfo;
  lfoPtr.p->lfoTimer = 0;
  logPartPtrP->firstFreeLfo = lfoPtr.i;
  lfoPtr.p->lfoState = LogFileOperationRecord::IDLE;
}  // Dblqh::releaseLfo()

/* ------------------------------------------------------------------------- */
/* ------- RELEASE ALL LOG PAGES CONNECTED TO A LFO RECORD           ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME = RLP                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::releaseLfoPages(LogFileOperationRecordPtr lfoPtr,
                            LogPartRecord *logPartPtrP) {
  LogPageRecordPtr logPagePtr;
  logPagePtr.i = lfoPtr.p->firstLfoPage;
  while (logPagePtr.i != RNIL) {
    ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                  logPartPtrP->logPageRecord);
    Uint32 tmp = logPagePtr.p->logPageWord[ZNEXT_PAGE];
    releaseLogpage(logPagePtr, logPartPtrP);
    logPagePtr.i = tmp;
  }
  lfoPtr.p->firstLfoPage = RNIL;
}  // Dblqh::releaseLfoPages()

/* ------------------------------------------------------------------------- */
/* -------                       RELEASE LOG PAGE                    ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::releaseLogpage(LogPageRecordPtr logPagePtr,
                           LogPartRecord *logPartPtrP) {
  check_log_page_ptr_i(logPartPtrP, logPagePtr.i);
#ifdef VM_TRACE
  // Check that log page isn't already in free list
  ndbrequire(logPagePtr.p->logPageWord[ZPOS_IN_FREE_LIST] == 0);
#endif

  logPartPtrP->noOfFreeLogPages++;
  logPagePtr.p->logPageWord[ZNEXT_PAGE] = logPartPtrP->firstFreeLogPage;
  logPagePtr.p->logPageWord[ZPOS_IN_WRITING] = 0;
  logPagePtr.p->logPageWord[ZPOS_IN_FREE_LIST] = 1;
  logPartPtrP->firstFreeLogPage = logPagePtr.i;
}  // Dblqh::releaseLogpage()

/* ------------------------------------------------------------------------- */
/* -------       SEIZE LFO RECORD                                    ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::seizeLfo(LogFileOperationRecordPtr &lfoPtr,
                     LogPartRecord *logPartPtrP) {
  lfoPtr.i = logPartPtrP->firstFreeLfo;
  ptrCheckGuard(lfoPtr, logPartPtrP->my_block->clfoFileSize,
                logPartPtrP->my_block->logFileOperationRecord);
  logPartPtrP->firstFreeLfo = lfoPtr.p->nextLfo;
  lfoPtr.p->nextLfo = RNIL;
  lfoPtr.p->lfoTimer = logPartPtrP->my_block->cLqhTimeOutCount;
}  // Dblqh::seizeLfo()

/* ------------------------------------------------------------------------- */
/* -------       SEIZE LOG FILE RECORD                               ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::seizeLogfile(LogFileRecordPtr &logFilePtr,
                         LogPartRecord *logPartPtrP) {
  logFilePtr.i = logPartPtrP->firstFreeLogFile;
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  /* -------------------------------------------------------------------------
   */
  /*IF LIST IS EMPTY THEN A SYSTEM CRASH IS INVOKED SINCE LOG_FILE_PTR = RNIL */
  /* -------------------------------------------------------------------------
   */
  logPartPtrP->firstFreeLogFile = logFilePtr.p->nextLogFile;
  logFilePtr.p->nextLogFile = RNIL;
}  // Dblqh::seizeLogfile()

/* ------------------------------------------------------------------------- */
/* -------       SEIZE LOG PAGE RECORD                               ------- */
/*                                                                           */
/* ------------------------------------------------------------------------- */
void Dblqh::seizeLogpage(LogPageRecordPtr &logPagePtr,
                         LogPartRecord *logPartPtrP) {
  logPartPtrP->noOfFreeLogPages--;
  logPagePtr.i = logPartPtrP->firstFreeLogPage;
  ptrCheckGuard(logPagePtr, logPartPtrP->logPageFileSize,
                logPartPtrP->logPageRecord);
  /* -------------------------------------------------------------------------
   */
  /*IF LIST IS EMPTY THEN A SYSTEM CRASH IS INVOKED SINCE LOG_PAGE_PTR = RNIL */
  /* -------------------------------------------------------------------------
   */
  logPartPtrP->firstFreeLogPage = logPagePtr.p->logPageWord[ZNEXT_PAGE];
#ifdef VM_TRACE
  std::memset(logPagePtr.p, 0, sizeof(LogPageRecord));
#endif
  logPagePtr.p->logPageWord[ZNEXT_PAGE] = RNIL;
  logPagePtr.p->logPageWord[ZPOS_IN_FREE_LIST] = 0;
  /**
   * During an initial start of a data node with Diskless set to
   * 1 we need to initialise this variable to 0. Normally the
   * log page is initialised when read from file system. The
   * code above that zeroes the entire page in debug builds
   * shows that it is safe to perform this initialisation.
   */
  logPagePtr.p->logPageWord[ZPOS_LOG_LAP] = 0;
  check_log_page_ptr_i(logPartPtrP, logPagePtr.i);
}  // Dblqh::seizeLogpage()

/* ------------------------------------------------------------------------- */
/* -------               WRITE FILE DESCRIPTOR INFORMATION           ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME: WFD                                          */
// Pointer handling:
// logFilePtr in
// logPartPtr in
/* ------------------------------------------------------------------------- */
void Dblqh::writeFileDescriptor(Signal *signal, LogFileRecord *logFilePtrP,
                                LogPartRecord *logPartPtrP) {
  TcConnectionrecPtr wfdTcConnectptr;
  UintR twfdFileNo;
  UintR twfdMbyte;

  /* -------------------------------------------------- */
  /*       START BY WRITING TO LOG FILE RECORD          */
  /* -------------------------------------------------- */
  arrGuard(logFilePtrP->currentMbyte, logPartPtrP->my_block->clogFileSize);
  if (DEBUG_REDO_EXEC) {
    g_eventLogger->info(
        "(%u)part: %u file: %u setting"
        " logMaxGciCompleted[%u] = %u"
        " logMaxGciStarted[%u]: %u lastPrepRef[%u]: ",
        instance(), logPartPtrP->logPartNo, logFilePtrP->fileNo,
        logFilePtrP->currentMbyte, logPartPtrP->logPartNewestCompletedGCI,
        logFilePtrP->currentMbyte, logPartPtrP->my_block->cnewestGci,
        logFilePtrP->currentMbyte);
    if (logPartPtrP->firstLogTcrec == nullptr) {
      g_eventLogger->info("(%u)file: %u mb: %u (RNIL)", instance(),
                          logFilePtrP->fileNo, logFilePtrP->currentMbyte);
    } else {
      wfdTcConnectptr.p = logPartPtrP->firstLogTcrec;
      ndbrequire(Magic::check_ptr(wfdTcConnectptr.p));
      g_eventLogger->info(
          "(%u)file: %u mb: %u", instance(), wfdTcConnectptr.p->logStartFileNo,
          wfdTcConnectptr.p->logStartPageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE);
    }
  }
  logFilePtrP->logMaxGciCompleted[logFilePtrP->currentMbyte] =
      logPartPtrP->logPartNewestCompletedGCI;
  logFilePtrP->logMaxGciStarted[logFilePtrP->currentMbyte] =
      logPartPtrP->my_block->cnewestGci;
  wfdTcConnectptr.p = logPartPtrP->firstLogTcrec;
  if (wfdTcConnectptr.p != nullptr) {
    jam();
    twfdFileNo = wfdTcConnectptr.p->logStartFileNo;
    ndbrequire(Magic::check_ptr(wfdTcConnectptr.p));
    twfdMbyte = wfdTcConnectptr.p->logStartPageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE;
    logFilePtrP->logLastPrepRef[logFilePtrP->currentMbyte] =
        (twfdFileNo << 16) + twfdMbyte;
  } else {
    jam();
    logFilePtrP->logLastPrepRef[logFilePtrP->currentMbyte] =
        (logFilePtrP->fileNo << 16) + logFilePtrP->currentMbyte;
  }  // if
}  // Dblqh::writeFileDescriptor()

/* ------------------------------------------------------------------------- */
/* -------               WRITE THE HEADER PAGE OF A NEW FILE         ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME:  WMO                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::writeFileHeaderOpen(Signal *signal, Uint32 wmoType,
                                LogPageRecordPtr &logPagePtr,
                                LogFileRecordPtr logFilePtr,
                                LogPartRecord *logPartPtrP) {
  UintR twmoNoLogDescriptors;

  /* -------------------------------------------------- */
  /*       WRITE HEADER INFORMATION IN THE NEW FILE.    */
  /* -------------------------------------------------- */
  logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_LOG_TYPE] = ZFD_TYPE;
  logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_FILE_NO] =
      logFilePtr.p->fileNo;
  /*
   * When writing a file header on open, we write cmaxLogFilesInPageZero,
   * though the entries for the first file (this file), will be invalid,
   * as we do not know e.g. which GCIs will be included by log records
   * in the MBs in this file.  On the first lap these will be initial values
   * on subsequent laps, they will be values from the previous lap.
   * We take care when reading these values back, not to use the values for
   * the current file.
   */
  if (logPartPtrP->noLogFiles > logPartPtrP->my_block->cmaxLogFilesInPageZero) {
    jam();
    twmoNoLogDescriptors = logPartPtrP->my_block->cmaxLogFilesInPageZero;
  } else {
    jam();
    twmoNoLogDescriptors = logPartPtrP->noLogFiles;
  }  // if
  logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_NO_FD] =
      twmoNoLogDescriptors;

  {
    Uint32 pos = ZPAGE_HEADER_SIZE + ZFD_HEADER_SIZE;
    LogFileRecordPtr filePtr = logFilePtr;
    for (Uint32 fd = 0; fd < twmoNoLogDescriptors; fd++) {
      jam();
      ptrCheckGuard(filePtr, logPartPtrP->my_block->clogFileFileSize,
                    logPartPtrP->my_block->logFileRecord);
      for (Uint32 mb = 0; mb < clogFileSize; mb++) {
        jam();
        Uint32 pos0 = pos + fd * (ZFD_MBYTE_SIZE * clogFileSize) + mb;
        Uint32 pos1 = pos0 + clogFileSize;
        Uint32 pos2 = pos1 + clogFileSize;
        arrGuard(pos0, ZPAGE_SIZE);
        arrGuard(pos1, ZPAGE_SIZE);
        arrGuard(pos2, ZPAGE_SIZE);
        logPagePtr.p->logPageWord[pos0] = filePtr.p->logMaxGciCompleted[mb];
        logPagePtr.p->logPageWord[pos1] = filePtr.p->logMaxGciStarted[mb];
        logPagePtr.p->logPageWord[pos2] = filePtr.p->logLastPrepRef[mb];
      }
      filePtr.i = filePtr.p->prevLogFile;
    }
    pos += (twmoNoLogDescriptors * ZFD_MBYTE_SIZE * clogFileSize);
    arrGuard(pos, ZPAGE_SIZE);
    logPagePtr.p->logPageWord[ZCURR_PAGE_INDEX] = pos;
    logPagePtr.p->logPageWord[pos] = ZNEXT_LOG_RECORD_TYPE;
  }

  /* ------------------------------------------------------- */
  /*       THIS IS A SPECIAL WRITE OF THE FIRST PAGE IN THE  */
  /*       LOG FILE. THIS HAS SPECIAL SIGNIFANCE TO FIND     */
  /*       THE END OF THE LOG AT SYSTEM RESTART.             */
  /* ------------------------------------------------------- */
  LogFileOperationRecordPtr lfoPtr;

  if (wmoType == ZINIT) {
    jam();
    writeSinglePage(signal, 0, ZPAGE_SIZE - 1, __LINE__, false, lfoPtr,
                    logPagePtr, logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::INIT_FIRST_PAGE;
  } else {
    jam();
    writeSinglePage(signal, 0, ZPAGE_SIZE - 1, __LINE__, true, lfoPtr,
                    logPagePtr, logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::FIRST_PAGE_WRITE_IN_LOGFILE;
  }  // if
  logFilePtr.p->filePosition = 1;
  if (wmoType == ZNORMAL) {
    jam();
    /* -------------------------------------------------- */
    /*       ALLOCATE A NEW PAGE SINCE THE CURRENT IS     */
    /*       WRITTEN.                                     */
    /* -------------------------------------------------- */
    seizeLogpage(logPagePtr, logPartPtrP);
    initLogpage(logPagePtr, logFilePtr.p, logPartPtrP);
    logFilePtr.p->currentLogpage = logPagePtr.i;
    logFilePtr.p->currentFilepage = logFilePtr.p->currentFilepage + 1;
  }  // if
}  // Dblqh::writeFileHeaderOpen()

/* -------------------------------------------------- */
/*       THE NEW FILE POSITION WILL ALWAYS BE 1 SINCE */
/*       WE JUST WROTE THE FIRST PAGE IN THE LOG FILE */
/* -------------------------------------------------- */
/* ------------------------------------------------------------------------- */
/* -------               WRITE A MBYTE HEADER DURING INITIAL START   ------- */
/*                                                                           */
/*       SUBROUTINE SHORT NAME: WIM                                          */
/* ------------------------------------------------------------------------- */
void Dblqh::writeInitMbyte(Signal *signal, LogPageRecordPtr logPagePtr,
                           LogFileRecordPtr logFilePtr,
                           LogPartRecord *logPartPtrP) {
  if (m_use_om_init == 0) {
    jam();
    LogFileOperationRecordPtr lfoPtr;

    initLogpage(logPagePtr, logFilePtr.p, logPartPtrP);
    writeSinglePage(signal, logFilePtr.p->currentMbyte * ZPAGES_IN_MBYTE,
                    ZPAGE_SIZE - 1, __LINE__, false, lfoPtr, logPagePtr,
                    logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::WRITE_INIT_MBYTE;
    checkReportStatus(signal);
  } else {
    jam();
    logFilePtr.p->currentMbyte = clogFileSize - 1;
    writeInitMbyteLab(signal, logPagePtr, logFilePtr, logPartPtrP);
  }
}  // Dblqh::writeInitMbyte()

/* ------------------------------------------------------------------------- */
/* -------               WRITE A SINGLE PAGE INTO A FILE             ------- */
/*                                                                           */
/*       INPUT:          TWSP_PAGE_NO    THE PAGE NUMBER WRITTEN             */
/*       SUBROUTINE SHORT NAME:  WSP                                         */
/* ------------------------------------------------------------------------- */
void Dblqh::writeSinglePage(Signal *signal, Uint32 pageNo, Uint32 wordWritten,
                            Uint32 place, bool sync,
                            LogFileOperationRecordPtr &lfoPtr,
                            LogPageRecordPtr logPagePtr,
                            LogFileRecordPtr logFilePtr,
                            LogPartRecord *logPartPtrP) {
  seizeLfo(lfoPtr, logPartPtrP);
  initLfo(lfoPtr.p, logFilePtr.i);
  lfoPtr.p->firstLfoPage = logPagePtr.i;
  logPagePtr.p->logPageWord[ZNEXT_PAGE] = RNIL;

  writeDbgInfoPageHeader(logPagePtr, logFilePtr.p, logPartPtrP, place, pageNo,
                         wordWritten);
  // Calculate checksum for page
  logPagePtr.p->logPageWord[ZPOS_CHECKSUM] = calcPageCheckSum(logPagePtr);

  lfoPtr.p->lfoPageNo = pageNo;
  lfoPtr.p->lfoWordWritten = wordWritten;
  lfoPtr.p->noPagesRw = 1;
  /* -------------------------------------------------- */
  /*       SET TIMER ON THIS LOG PART TO SIGNIFY THAT A */
  /*       LOG RECORD HAS BEEN SENT AT THIS TIME.       */
  /* -------------------------------------------------- */
  logPartPtrP->logPartTimer = logPartPtrP->logTimer;

  FsReadWriteReq *req = (FsReadWriteReq *)signal->getDataPtrSend();
  req->filePointer = logFilePtr.p->fileRef;
  req->userReference = logPartPtrP->myRef;
  req->userPointer = lfoPtr.i;
  req->operationFlag = 0;
  req->setFormatFlag(req->operationFlag, FsReadWriteReq::fsFormatArrayOfPages);
  req->setSyncFlag(req->operationFlag, sync);
  req->varIndex = logPartPtrP->ptrI;
  req->numberOfPages = 1; /* ONE PAGE WRITTEN */
  req->data.arrayOfPages.varIndex = logPagePtr.i;
  req->data.arrayOfPages.fileOffset = pageNo;
  sendSignal(NDBFS_REF, GSN_FSWRITEREQ, signal, 8, JBA);

  if (logFilePtr.p->fileRef == RNIL) {
    signal->theData[0] = DumpStateOrd::LqhFailedHandlingGCP_SAVEREQ;
    logPartPtrP->my_block->execDUMP_STATE_ORD(signal);
  }
  ndbrequire(logFilePtr.p->fileRef != RNIL);

  logPartPtrP->m_io_tracker.send_io(32768);

  if (DEBUG_REDO_EXEC) {
    g_eventLogger->info(
        "(%u)writeSingle 1 page at part: %u file: %u"
        " page: %u (mb: %u)",
        instance(), logPartPtrP->logPartNo, logFilePtr.p->fileNo, pageNo,
        pageNo >> ZTWOLOG_NO_PAGES_IN_MBYTE);
  }
}  // Dblqh::writeSinglePage()

/* ##########################################################################
 *     SYSTEM RESTART PHASE ONE MODULE
 *     THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.
 *
 *     THIS MODULE CONTAINS THE CODE FOR THE FIRST PHASE OF THE SYSTEM RESTART.
 *     THE AIM OF THIS PHASE IS TO FIND THE END OF THE LOG AND TO FIND
 *     INFORMATION ABOUT WHERE GLOBAL CHECKPOINTS ARE COMPLETED AND STARTED
 *     IN THE LOG. THIS INFORMATION IS NEEDED TO START PHASE THREE OF
 *     THE SYSTEM RESTART.
 * ########################################################################## */
/* --------------------------------------------------------------------------
 *     A SYSTEM RESTART OR NODE RESTART IS ONGOING. WE HAVE NOW OPENED FILE 0
 *     NOW WE NEED TO READ PAGE 0 TO FIND WHICH LOG FILE THAT WAS OPEN AT
 *     CRASH TIME.
 * -------------------------------------------------------------------------- */
void Dblqh::openSrFrontpageLab(Signal *signal, LogFileRecordPtr logFilePtr,
                               LogPartRecord *logPartPtrP) {
  LogFileOperationRecordPtr lfoPtr;

  readSinglePage(signal, 0, lfoPtr, logFilePtr, logPartPtrP);
  lfoPtr.p->lfoState = LogFileOperationRecord::READ_SR_FRONTPAGE;
}  // Dblqh::openSrFrontpageLab()

/* -------------------------------------------------------------------------
 * WE HAVE NOW READ PAGE 0 IN FILE 0. CHECK THE LAST OPEN FILE. ACTUALLY THE
 * LAST OPEN FILE COULD BE THE NEXT AFTER THAT. CHECK THAT FIRST. WHEN THE
 * LAST WAS FOUND WE CAN FIND ALL THE NEEDED INFORMATION WHERE TO START AND
 * STOP READING THE LOG.
 * -------------------------------------------------------------------------- */
void Dblqh::readSrFrontpageLab(Signal *signal, LogPageRecordPtr logPagePtr,
                               LogFileRecordPtr logFilePtr,
                               LogPartRecord *logPartPtrP) {
  Uint32 num_parts_used;
  {
    jam();
    num_parts_used = logPagePtr.p->logPageWord[ZPOS_NO_LOG_PARTS];
  }
  /* Verify that number of log parts >= number of LQH workers */
  if (num_parts_used != globalData.ndbLogParts) {
    char buf[255];
    BaseString::snprintf(
        buf, sizeof(buf),
        "Can only change NoOfLogParts through initial node restart, old"
        " value of NoOfLogParts = %d, tried using %d",
        num_parts_used, globalData.ndbLogParts);
    progError(__LINE__, NDBD_EXIT_INVALID_CONFIG, buf);
  }

  Uint32 fileNo = logPagePtr.p->logPageWord[ZPAGE_HEADER_SIZE + ZPOS_FILE_NO];
  /* ------------------------------------------------------------------------
   *    CLOSE FILE 0 SO THAT WE HAVE CLOSED ALL FILES WHEN STARTING TO READ
   *    THE FRAGMENT LOG. ALSO RELEASE PAGE ZERO.
   * ------------------------------------------------------------------------ */
  releaseLogpage(logPagePtr, logPartPtrP);
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSING_SR_FRONTPAGE;
  closeFile(signal, logFilePtr, __LINE__);
  /* Lookup index of last file */
  LogFileRecordPtr locLogFilePtr;
  findLogfile(signal, fileNo, logPartPtrP, &locLogFilePtr);

  /* Store in logPart record for use once file 0 is closed */
  logPartPtrP->srLastFileIndex = locLogFilePtr.i;
}  // Dblqh::readSrFrontpageLab()

void Dblqh::openSrLastFileLab(Signal *signal, LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  LogFileOperationRecordPtr lfoPtr;

  readSinglePage(signal, 0, lfoPtr, logFilePtr, logPartPtrP);
  lfoPtr.p->lfoState = LogFileOperationRecord::READ_SR_LAST_FILE;
}  // Dblqh::openSrLastFileLab()

void Dblqh::readSrLastFileLab(Signal *signal, LogPageRecordPtr logPagePtr,
                              LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  logPartPtrP->logLap = logPagePtr.p->logPageWord[ZPOS_LOG_LAP];
  if (DEBUG_REDO_REC) {
    g_eventLogger->info(
        "(%u)readSrLastFileLab part: %u logExecState: %u"
        " logPartState: %u logLap: %u",
        instance(), logPartPtrP->logPartNo, logPartPtrP->logExecState,
        logPartPtrP->logPartState, logPartPtrP->logLap);
  }
  if (logPartPtrP->noLogFiles > cmaxValidLogFilesInPageZero) {
    jam();
    initGciInLogFileRec(signal, cmaxValidLogFilesInPageZero, logPagePtr,
                        logFilePtr);
  } else {
    jam();
    initGciInLogFileRec(signal, logPartPtrP->noLogFiles, logPagePtr,
                        logFilePtr);
  }  // if
  /* ------------------------------------------------------------------------
   *    NOW WE HAVE FOUND THE LAST LOG FILE. WE ALSO NEED TO FIND THE LAST
   *    MBYTE THAT WAS LAST WRITTEN BEFORE THE SYSTEM CRASH.
   * ------------------------------------------------------------------------ */
  logPartPtrP->lastLogfile = logFilePtr.i;
  /**
   * It is safe to read page 0 of the first MByte since we always ensure that
   * this page is up to date before we update current file number in page 0
   * of file 0. Given that we already have page 0 read, we can now call
   * readSrLastMbyteLab immediately, no need to reread page 0.
   */
  logFilePtr.p->currentMbyte = 0;
  readSrLastMbyteLab(signal, logPagePtr, logFilePtr, logPartPtrP);
}  // Dblqh::readSrLastFileLab()

void Dblqh::readSrLastMbyteLab(Signal *signal, LogPageRecordPtr logPagePtr,
                               LogFileRecordPtr logFilePtr,
                               LogPartRecord *logPartPtrP) {
  if (logPartPtrP->lastMbyte == ZNIL) {
    if (logPagePtr.p->logPageWord[ZPOS_LOG_LAP] < logPartPtrP->logLap) {
      jam();
      logPartPtrP->lastMbyte = logFilePtr.p->currentMbyte - 1;
      if (DEBUG_REDO_REC) {
        g_eventLogger->info(
            "(%u)readSrLastMbyteLab part: %u file: %u"
            " lastMbyte: %u",
            instance(), logPartPtrP->logPartNo, logFilePtr.p->fileNo,
            logPartPtrP->lastMbyte);
      }
    }  // if
  }    // if

  arrGuard(logFilePtr.p->currentMbyte, clogFileSize);
  logFilePtr.p->logMaxGciCompleted[logFilePtr.p->currentMbyte] =
      logPagePtr.p->logPageWord[ZPOS_MAX_GCI_COMPLETED];
  logFilePtr.p->logMaxGciStarted[logFilePtr.p->currentMbyte] =
      logPagePtr.p->logPageWord[ZPOS_MAX_GCI_STARTED];
  logFilePtr.p->logLastPrepRef[logFilePtr.p->currentMbyte] =
      logPagePtr.p->logPageWord[ZLAST_LOG_PREP_REF];
  releaseLogpage(logPagePtr, logPartPtrP);

  if (logFilePtr.p->currentMbyte < (clogFileSize - 1)) {
    jam();
    LogFileOperationRecordPtr lfoPtr;

    logFilePtr.p->currentMbyte++;
    readSinglePage(signal, ZPAGES_IN_MBYTE * logFilePtr.p->currentMbyte, lfoPtr,
                   logFilePtr, logPartPtrP);
    lfoPtr.p->lfoState = LogFileOperationRecord::READ_SR_LAST_MBYTE;
    return;
  } else {
    jam();
    /* ----------------------------------------------------------------------
     *    THE LOG WAS IN THE LAST MBYTE WHEN THE CRASH OCCURRED SINCE ALL
     *    LOG LAPS ARE EQUAL TO THE CURRENT LOG LAP.
     * -------------------------------------------------------------------- */
    if (logPartPtrP->lastMbyte == ZNIL) {
      jam();
      logPartPtrP->lastMbyte = clogFileSize - 1;
    }  // if
  }    // if
  if (ERROR_INSERTED(5092)) {
    jam();
    suspendFile(signal, logFilePtr, 3000);  // Slow close
  }

  logFilePtr.p->logFileStatus = LogFileRecord::CLOSING_SR;
  closeFile(signal, logFilePtr, __LINE__);

  /* Head file is initialised by reading per-MB headers rather than per-file
   * headers.  Therefore, when stepping back through the redo files to get
   * the previous file's metadata, we must be careful not to read the
   * per-file header info over the just-read per-MB headers, invalidating
   * the head metainfo.
   */
  Uint32 nonHeadFileCount = logPartPtrP->noLogFiles - 1;

  if (logPartPtrP->noLogFiles > cmaxValidLogFilesInPageZero) {
    /* Step back from head to get file:mb metadata from a
     * previous file's page zero
     */
    Uint32 fileNo;
    if (logFilePtr.p->fileNo >= cmaxValidLogFilesInPageZero) {
      jam();
      fileNo = logFilePtr.p->fileNo - cmaxValidLogFilesInPageZero;
    } else {
      /* Wrap at 0:0 */
      jam();
      fileNo = (logPartPtrP->noLogFiles + logFilePtr.p->fileNo) -
               cmaxValidLogFilesInPageZero;
    }  // if

    jam();
    logPartPtrP->srRemainingFiles =
        nonHeadFileCount - cmaxValidLogFilesInPageZero;

    /* Check we're making progress */
    ndbrequire(fileNo != logFilePtr.p->fileNo);
    LogFileRecordPtr locLogFilePtr;
    findLogfile(signal, fileNo, logPartPtrP, &locLogFilePtr);
    ndbrequire(locLogFilePtr.p->logFileStatus == LogFileRecord::CLOSED);
    locLogFilePtr.p->logFileStatus = LogFileRecord::OPEN_SR_NEXT_FILE;
    openFileRw(signal, locLogFilePtr, false); /* No write buffering */
  }                                           // if
  /* ------------------------------------------------------------------------
   *   THERE WERE NO NEED TO READ ANY MORE PAGE ZERO IN OTHER FILES.
   *   WE NOW HAVE ALL THE NEEDED INFORMATION ABOUT THE GCI'S THAT WE NEED.
   *   NOW JUST WAIT FOR CLOSE OPERATIONS TO COMPLETE.
   * ---------------------------------------------------------------------- */
}  // Dblqh::readSrLastMbyteLab()

void Dblqh::openSrNextFileLab(Signal *signal, LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  LogFileOperationRecordPtr lfoPtr;

  readSinglePage(signal, 0, lfoPtr, logFilePtr, logPartPtrP);
  lfoPtr.p->lfoState = LogFileOperationRecord::READ_SR_NEXT_FILE;
}  // Dblqh::openSrNextFileLab()

void Dblqh::readSrNextFileLab(Signal *signal, LogPageRecordPtr logPagePtr,
                              LogFileRecordPtr logFilePtr,
                              LogPartRecord *logPartPtrP) {
  if (logPartPtrP->srRemainingFiles > cmaxValidLogFilesInPageZero) {
    jam();
    initGciInLogFileRec(signal, cmaxValidLogFilesInPageZero, logPagePtr,
                        logFilePtr);
  } else {
    jam();
    initGciInLogFileRec(signal, logPartPtrP->srRemainingFiles, logPagePtr,
                        logFilePtr);
  }  // if
  releaseLogpage(logPagePtr, logPartPtrP);
  if (ERROR_INSERTED(5092)) {
    jam();
    suspendFile(signal, logFilePtr, 3000);  // Slow close
  }
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSING_SR;
  closeFile(signal, logFilePtr, __LINE__);
  if (logPartPtrP->srRemainingFiles > cmaxValidLogFilesInPageZero) {
    /* Step back from head to get file:mb metadata from a
     * previous file's page zero
     */
    Uint32 fileNo;
    if (logFilePtr.p->fileNo >= cmaxValidLogFilesInPageZero) {
      jam();
      fileNo = logFilePtr.p->fileNo - cmaxValidLogFilesInPageZero;
    } else {
      /* Wrap at 0:0 */
      jam();
      fileNo = (logPartPtrP->noLogFiles + logFilePtr.p->fileNo) -
               cmaxValidLogFilesInPageZero;
    }  // if

    jam();
    logPartPtrP->srRemainingFiles =
        logPartPtrP->srRemainingFiles - cmaxValidLogFilesInPageZero;

    /* Check we're making progress */
    ndbrequire(fileNo != logFilePtr.p->fileNo);

    /**
     * Note that we are opening another file without waiting for
     * the previous FSCLOSECONF.
     * This can result in > 4 concurrently open files.
     */
    LogFileRecordPtr locLogFilePtr;
    findLogfile(signal, fileNo, logPartPtrP, &locLogFilePtr);
    ndbrequire(locLogFilePtr.p->logFileStatus == LogFileRecord::CLOSED);
    locLogFilePtr.p->logFileStatus = LogFileRecord::OPEN_SR_NEXT_FILE;
    openFileRw(signal, locLogFilePtr, false); /* No write buffering */
  }                                           // if
  /* ------------------------------------------------------------------------
   *   THERE WERE NO NEED TO READ ANY MORE PAGE ZERO IN OTHER FILES.
   *   WE NOW HAVE ALL THE NEEDED INFORMATION ABOUT THE GCI'S THAT WE NEED.
   *   NOW JUST WAIT FOR CLOSE OPERATIONS TO COMPLETE.
   * --------------------------------------------------------------------- */
}  // Dblqh::readSrNextFileLab()

void Dblqh::closingSrFrontPage(Signal *signal, LogFileRecordPtr logFilePtr) {
  LogPartRecordPtr logPartPtr;
  jam();
  /* Front page (file 0) has closed, now it's safe to continue
   * to read any page (including file 0) as part of restoring
   * redo metadata
   */
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
  logPartPtr.i = logFilePtr.p->logPartRec;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  logFilePtr.i = logPartPtr.p->firstLogfile;

  /* Pre-restart head file index was stored in logPartPtr.p->srLastFileIndex
   * prior to closing this file, now let's use it...
   */
  ndbrequire(logPartPtr.p->srLastFileIndex != RNIL);

  LogFileRecordPtr oldHead;
  oldHead.i = logPartPtr.p->srLastFileIndex;
  ptrCheckGuard(oldHead, clogFileFileSize, logFileRecord);

  /* Reset srLastFileIndex */
  logPartPtr.p->srLastFileIndex = RNIL;

  /* And now open the head file to begin redo meta reload */
  oldHead.p->logFileStatus = LogFileRecord::OPEN_SR_LAST_FILE;
  openFileRw(signal, oldHead, false); /* No write buffering */
}

void Dblqh::closingSrLab(Signal *signal, LogFileRecordPtr logFilePtr) {
  LogPartRecordPtr logPartPtr;
  logFilePtr.p->logFileStatus = LogFileRecord::CLOSED;
  logPartPtr.i = logFilePtr.p->logPartRec;
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  logFilePtr.i = logPartPtr.p->firstLogfile;
  do {
    jam();
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    if (logFilePtr.p->logFileStatus != LogFileRecord::CLOSED) {
      jam();
      /* --------------------------------------------------------------------
       *  EXIT AND WAIT FOR REMAINING LOG FILES TO COMPLETE THEIR WORK.
       * ------------------------------------------------------------------ */
      return;
    }  // if
    logFilePtr.i = logFilePtr.p->nextLogFile;
  } while (logFilePtr.i != logPartPtr.p->firstLogfile);
  /* ------------------------------------------------------------------------
   *  ALL FILES IN THIS PART HAVE BEEN CLOSED. THIS INDICATES THAT THE FIRST
   *  PHASE OF THE SYSTEM RESTART HAVE BEEN CONCLUDED FOR THIS LOG PART.
   *  CHECK IF ALL OTHER LOG PARTS ARE ALSO COMPLETED.
   * ---------------------------------------------------------------------- */
  logPartPtr.p->logPartState = LogPartRecord::SR_FIRST_PHASE_COMPLETED;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    if (logPartPtr.p->logPartState != LogPartRecord::SR_FIRST_PHASE_COMPLETED) {
      jam();
      /* --------------------------------------------------------------------
       * EXIT AND WAIT FOR THE REST OF THE LOG PARTS TO COMPLETE.
       * ------------------------------------------------------------------ */
      return;
    }  // if
  }    // for
  /* ------------------------------------------------------------------------
   *       THE FIRST PHASE HAVE BEEN COMPLETED.
   * ---------------------------------------------------------------------- */
  g_eventLogger->info(
      "LDM(%u):"
      "Ready to start execute REDO log phase,"
      " prepare REDO log phase completed",
      instance());

  signal->theData[0] = ZSR_PHASE3_START;
  signal->theData[1] = ZSR_PHASE1_COMPLETED;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
}  // Dblqh::closingSrLab()

/* ##########################################################################
 * #######                  SYSTEM RESTART PHASE TWO MODULE           #######
 *
 *  THIS MODULE HANDLES THE SYSTEM RESTART WHERE LQH CONTROLS TUP AND ACC TO
 *  ENSURE THAT THEY HAVE KNOWLEDGE OF ALL FRAGMENTS AND HAVE DONE THE NEEDED
 *  READING OF DATA FROM FILE AND EXECUTION OF LOCAL LOGS. THIS PROCESS
 *  EXECUTES CONCURRENTLY WITH PHASE ONE OF THE SYSTEM RESTART. THIS PHASE
 *  FINDS THE INFORMATION ABOUT THE FRAGMENT LOG NEEDED TO EXECUTE THE FRAGMENT
 *  LOG.
 *  WHEN TUP AND ACC HAVE PREPARED ALL FRAGMENTS THEN LQH ORDERS THOSE LQH'S
 *  THAT ARE RESPONSIBLE TO EXECUTE THE FRAGMENT LOGS TO DO SO. IT IS POSSIBLE
 *  THAT ANOTHER NODE EXECUTES THE LOG FOR A FRAGMENT RESIDING AT THIS NODE.
 * ######################################################################## */
/* ***************>> */
/*  START_FRAGREQ  > */
/* ***************>> */
void Dblqh::execSTART_FRAGREQ(Signal *signal) {
  /**
   * We don't need to worry about NOLOGGING tables and temporary tables
   * here. These fragments are added at restart, but not started since they
   * by definition are restored as empty fragments.
   */
  const StartFragReq *const startFragReq = (StartFragReq *)&signal->theData[0];
  jamEntry();

  c_fragmentsStarted++;

  tabptr.i = startFragReq->tableId;
  Uint32 fragId = startFragReq->fragId;

  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  if (!getFragmentrec(fragId)) {
    startFragRefLab(signal);
    return;
  }  // if
  tabptr.p->tableStatus = Tablerec::TABLE_DEFINED;
  DEB_SCHEMA_VERSION(
      ("(%u)tab: %u tableStatus = TABLE_DEFINED(3)", instance(), tabptr.i));
  c_pgman->set_table_ready_for_prep_lcp_writes(tabptr.i, true);

  Uint32 lcpNo = startFragReq->lcpNo;
  Uint32 noOfLogNodes = startFragReq->noOfLogNodes;
  Uint32 lcpId = startFragReq->lcpId;
  Uint32 requestInfo = startFragReq->requestInfo;
  Uint32 nodeRestorableGci = startFragReq->nodeRestorableGci;
  if (signal->getLength() < StartFragReq::SignalOldLength) {
    jam();
    requestInfo = StartFragReq::SFR_RESTORE_LCP;
  }
  if (signal->getLength() < StartFragReq::SignalLength) {
    jam();
    nodeRestorableGci = 0;
  }

  bool doprint = false;
#ifdef ERROR_INSERT
  /**
   * Always printSTART_FRAG_REQ (for debugging) if ERROR_INSERT is set
   */
  doprint = true;
#endif
#ifdef DEBUG_LCP
  doprint = true;
#endif
  if (doprint || noOfLogNodes > 1) {
    printSTART_FRAG_REQ(stdout, signal->getDataPtr(), signal->getLength(),
                        number());
  }

  ndbrequire(noOfLogNodes <= MAX_LOG_EXEC);
  fragptr.p->fragStatus = Fragrecord::CRASH_RECOVERING;
  fragptr.p->srBlockref = startFragReq->userRef;
  fragptr.p->srUserptr = startFragReq->userPtr;
  fragptr.p->srChkpnr = lcpNo;
  if (lcpNo == (MAX_LCP_STORED - 1)) {
    jam();
    fragptr.p->lcpId[lcpNo] = lcpId;
  } else if (lcpNo < (MAX_LCP_STORED - 1)) {
    jam();
    fragptr.p->lcpId[lcpNo] = lcpId;
  } else {
    ndbrequire(lcpNo == ZNIL);
    jam();
  }  // if
  fragptr.p->srNoLognodes = noOfLogNodes;
  fragptr.p->logFlag = Fragrecord::STATE_FALSE;
  fragptr.p->srStatus = Fragrecord::SS_IDLE;

  if (requestInfo == StartFragReq::SFR_COPY_FRAG) {
    jam();
    ndbrequire(lcpNo == ZNIL);
    fragptr.p->srLqhLognode[0] = startFragReq->lqhLogNode[0];  // src

    // Magic no, meaning to COPY_FRAGREQ instead of read from disk
    fragptr.p->srChkpnr = Z8NIL;
    c_fragmentsStartedWithCopy++;
    ndbrequire(noOfLogNodes == 0);
  }

  if (noOfLogNodes > 0) {
    jam();
    for (Uint32 i = 0; i < noOfLogNodes; i++) {
      jam();
      fragptr.p->srStartGci[i] = startFragReq->startGci[i];
      fragptr.p->srLastGci[i] = startFragReq->lastGci[i];
      fragptr.p->srLqhLognode[i] = startFragReq->lqhLogNode[i];
    }  // for
    fragptr.p->newestGci = startFragReq->lastGci[noOfLogNodes - 1];
    DEB_NEWEST_GCI(
        ("%u) execSTART_FRAGREQ(1): tab(%u,%u)"
         " fragptr.p->newestGci = %u",
         instance(), fragptr.p->tabRef, fragptr.p->fragId,
         fragptr.p->newestGci));
    fragptr.p->m_completed_gci = startFragReq->lastGci[noOfLogNodes - 1];
  } else {
    jam();
    /**
     * This is a really weird piece of code
     *   it's probably incorrect, but seems to mask problems...
     *
     * This code can only be executed by node restarts. In this
     * case having no log nodes simply means that we restore
     * entirely from the live node. This is indicated by
     * nodeRestorableGci == 0.
     * In reality there should be some REDO log to execute, but
     * this should only happen immediately after creating table
     * and no LCP executed before crash, so should be ok to skip
     * the REDO log and instead restore from live node for this
     * specific case. To use the REDO log would require ensuring
     * that not multiple failures have occurred, so this makes
     * code a bit simpler although a bit less efficient in this
     * specific case.
     */
    if (cnewestGci > fragptr.p->newestGci) {
      jam();
      fragptr.p->newestGci = cnewestGci;
      DEB_NEWEST_GCI(
          ("%u) execSTART_FRAGREQ(2): tab(%u,%u)"
           " fragptr.p->newestGci = %u",
           instance(), fragptr.p->tabRef, fragptr.p->fragId,
           fragptr.p->newestGci));
    } else {
      DEB_NEWEST_GCI(
          ("%u) execSTART_FRAGREQ(3): tab(%u,%u)"
           " fragptr.p->newestGci = %u, cnewestGci = %u",
           instance(), fragptr.p->tabRef, fragptr.p->fragId,
           fragptr.p->newestGci, cnewestGci));
    }
    fragptr.p->m_completed_gci = 0;
  }  // if

  /**
   * To slightly speed up the restart newer versions send the newest
   * GCI that the node can restore on its own. This is the last GCI
   * where the node completed the GCI protocol. This is an important
   * number as we cannot use any LCPs that have written any GCI which
   * is newer than this number.
   *
   * In upgrade cases we hold off with starting to send RESTORE_LCP_REQ
   * until we have received START_RECREQ where this also arrives. In
   * newer versions we added this already to the START_FRAGREQ signal.
   */
  if (nodeRestorableGci != 0) {
    jam();
    if (crestartNewestGci == 0 || crestartNewestGci == ZUNDEFINED_GCI_LIMIT) {
      jam();
      crestartNewestGci = nodeRestorableGci;
    } else {
      ndbrequire(crestartNewestGci == nodeRestorableGci);
    }
  }

  c_lcp_waiting_fragments.addLast(fragptr);
  if (requestInfo == StartFragReq::SFR_COPY_FRAG) {
    jam();
  } else if (lcpNo == ZNIL) {
    /**
     *  THERE WAS NO LOCAL CHECKPOINT AVAILABLE FOR THIS FRAGMENT. WE DO
     *  NOT NEED TO READ IN THE LOCAL FRAGMENT.
     *
     * Given that we might have completed the local checkpoint before DIH
     * got to know about it in LCP format introduced in 7.5 we will still
     * try to restore the LCP locally. If no LCP control files then we will
     * not attempt to execute the LCP however, rather we will delete the
     * LCP files and ensure that a control file exists there but no data
     * files.
     *
     * fragPtr.p->srChkpnr == ZNIL indicates to RESTORE block that DIH didn't
     * know about any LCP for this fragment.
     */
    jam();
  } else {
    jam();

    if (ERROR_INSERTED(5055)) {
      ndbrequire(c_restart_lcpId == 0 || lcpId == 0 ||
                 c_restart_lcpId == lcpId || c_restart_lcpId == (lcpId + 1));
    }
  }
  if (nodeRestorableGci != 0 && c_lcp_restoring_fragments.isEmpty()) {
    jam();
    send_restore_lcp(signal);
  }
}  // Dblqh::execSTART_FRAGREQ()

Uint32 Dblqh::get_recover_thread_instance() {
  /**
   * We are performing a restore in an LDM instance. We haven't used up our
   * quota of the recover threads working on our behalf. We check if any
   * recover thread is available for our use or not. If no one is available
   * we return 0, otherwise we return the instance number of the recover
   * thread we can use (instance number 0 is the proxy instance).
   */
  Uint32 num_recover_threads =
      globalData.ndbMtQueryThreads + globalData.ndbMtRecoverThreads;
  if (num_recover_threads == 0) {
    return 0;
  }
  Uint32 *num_recover_active = c_restore_mutex_lqh->m_num_recover_active;
  NdbMutex_Lock(c_restore_mutex_lqh->m_restore_mutex);
  for (Uint32 i = 1; i <= num_recover_threads; i++) {
    if (num_recover_active[i] == 0) {
      num_recover_active[i] = 1;
      NdbMutex_Unlock(c_restore_mutex_lqh->m_restore_mutex);
      return i;
    }
  }
  NdbMutex_Unlock(c_restore_mutex_lqh->m_restore_mutex);
  return 0;
}

void Dblqh::completed_restore(Uint32 instance) {
  /**
   * A restore of a fragment has completed, we release the recover thread
   * for someone else to use.
   */
  Uint32 *num_recover_active = c_restore_mutex_lqh->m_num_recover_active;
  NdbMutex_Lock(c_restore_mutex_lqh->m_restore_mutex);
  ndbrequire(num_recover_active[instance] == 1);
  num_recover_active[instance] = 0;
  NdbMutex_Unlock(c_restore_mutex_lqh->m_restore_mutex);
}

bool Dblqh::instance_completed_restore(Uint32 instance) {
  /**
   * One of the LDM instances have completed its set of fragments to
   * restore. We check if there is any other LDM instance still working
   * on the restore. If all LDM instances are done and we have had help
   * of recover threads we will report true, otherwise we will return
   * false. This is used to send a signal to the recover threads to
   * log output about its restore operations.
   */
  Uint32 num_recover_threads =
      globalData.ndbMtQueryThreads + globalData.ndbMtRecoverThreads;
  if (num_recover_threads == 0) {
    jam();
    return false;
  }
  bool done_flag;
  Uint32 *instance_recover_active =
      c_restore_mutex_lqh->m_instance_recover_active;
  NdbMutex_Lock(c_restore_mutex_lqh->m_restore_mutex);
  ndbrequire(instance_recover_active[instance] == 1);
  instance_recover_active[instance] = 0;
  ndbrequire(c_restore_mutex_lqh->m_num_instances_active > 0);
  c_restore_mutex_lqh->m_num_instances_active--;
  done_flag = (c_restore_mutex_lqh->m_num_instances_active == 0);
  NdbMutex_Unlock(c_restore_mutex_lqh->m_restore_mutex);
  return done_flag;
}

void Dblqh::send_restore_lcp(Signal *signal) {
  ndbassert(!m_is_query_block);
  while (true) {
    DEB_LCP_RESTORE(("(%u) send_restore_lcp", instance()));
    if (!c_lcp_waiting_fragments.first(fragptr)) {
      jam();
      DEB_LCP_RESTORE(("(%u) send_restore_lcp, RET1", instance()));
      return;
    }
    if (m_num_restores_active == m_num_restore_threads) {
      jam();
      DEB_LCP_RESTORE(("(%u) send_restore_lcp, RET2", instance()));
      return;
    }

    if (fragptr.p->srChkpnr != Z8NIL) {
      Uint32 instanceNo = 0;
      if (m_num_local_restores_active == 0) {
        jam();
        DEB_LCP_RESTORE(("(%u) send_restore_lcp, LOC1", instance()));
        m_num_local_restores_active++;
      } else {
        jam();
        instanceNo = get_recover_thread_instance();
        if (instanceNo == 0) {
          /**
           * No recover thread available at the moment. Cannot start
           * since all recover threads are active as well as my own
           * thread.
           */
          jam();
          DEB_LCP_RESTORE(("(%u) send_restore_lcp, RET3", instance()));
          return;
        }
      }
      c_lcp_waiting_fragments.remove(fragptr);
      c_lcp_restoring_fragments.addLast(fragptr);
      m_num_restores_active++;
      /**
       * We're sending the DIH view to the RESTORE block, this is necessary
       * in upgrade situations. In the case when the LCP was created by 7.5
       * and later the RESTORE block will itself discover the LCP id used to
       * recover, it will use the GCI to restore to get this information
       * and the information stored in the LCP control files.
       *
       * The RESTORE block will return the LCP id used to restore in the
       * CONF signal. This makes it possible for DBTUP to use the correct
       * LCP id to restore the disk data. (This includes the local LCP id).
       */
      jam();
      TablerecPtr tabPtr;
      tabPtr.i = fragptr.p->tabRef;
      ptrCheckGuard(tabPtr, ctabrecFileSize, tablerec);
      if (!tabPtr.p->m_restore_started) {
        jam();
        /**
         * The table record is owned by this LDM instance. It needs some
         * special setup for restore operations. We will set up the
         * table record for restore operation and indicate that we have
         * done so.
         *
         * After we have completed all restore operations we will restore
         * the table record for all restored tables.
         */
        tabPtr.p->m_restore_started = true;
        c_tup->start_restore_table(tabPtr.i);
      }
      RestoreLcpReq *req = (RestoreLcpReq *)signal->getDataPtrSend();
      req->senderData = fragptr.i;
      req->senderRef = reference();
      req->tableId = fragptr.p->tabRef;
      req->fragmentId = fragptr.p->fragId;
      req->lcpNo = fragptr.p->srChkpnr;
      if (fragptr.p->srChkpnr == ZNIL) {
        jam();
        req->lcpId = 0;
        req->maxGciCompleted = 0;
      } else {
        jam();
        req->lcpId = fragptr.p->lcpId[fragptr.p->srChkpnr];
        req->maxGciCompleted = fragptr.p->srStartGci[0] - 1;
      }
      req->restoreGcpId = crestartNewestGci;
      if (c_local_sysfile.m_max_restorable_gci > ZUNDEFINED_GCI_LIMIT) {
        jam();
        if (c_local_sysfile.m_max_restorable_gci < crestartNewestGci) {
          g_eventLogger->info(
              "(%u), tab(%u,%u), lcpNo: %u, "
              "m_max_restorable_gci: %u, "
              "crestartNewestGci: %u"
              ", srStartGci: %u",
              instance(), fragptr.p->tabRef, fragptr.p->fragId,
              fragptr.p->srChkpnr, c_local_sysfile.m_max_restorable_gci,
              crestartNewestGci, fragptr.p->srStartGci[0]);
          ndbrequire(c_local_sysfile.m_max_restorable_gci >= crestartNewestGci);
        }
        req->restoreGcpId = c_local_sysfile.m_max_restorable_gci;
        ndbrequire(c_local_sysfile.m_max_restorable_gci >= crestartNewestGci);
        if (c_local_sysfile.m_max_restorable_gci > crestartNewestGci) {
          jam();
          req->restoreGcpId = c_local_sysfile.m_max_restorable_gci;
        }
      }
      /**
       * DIH could potentially send a createGci that is newer than
       * what is restorable. This could happen when the table was created
       * very close to the crash.
       * We will still keep the createGci as is to discover if it is the
       * same table since all LCP control files are tagged with the
       * createGci.
       */
      req->createGci = fragptr.p->createGci;
      req->cnewestGci = cnewestGci;
      BlockReference restoreRef = calcInstanceBlockRef(RESTORE);
      if (instanceNo == 0) {
        /* We will take care of the restore ourselves. */
        jam();
        DEB_LCP_RESTORE(("(%u) Restore tab(%u,%u) locally", instance(),
                         fragptr.p->tabRef, fragptr.p->fragId));
      } else {
        jam();
        /* We will send the restore to a recover thread. */
        restoreRef = numberToRef(QRESTORE, instanceNo, getOwnNodeId());
        DEB_LCP_RESTORE(("(%u) Restore tab(%u,%u) in recover thread",
                         instance(), fragptr.p->tabRef, fragptr.p->fragId));
      }
      sendSignal(restoreRef, GSN_RESTORE_LCP_REQ, signal,
                 RestoreLcpReq::SignalLength, JBB);
    } else {
      jam();
      if (m_num_copy_restores_active > 0) {
        jam();
        return;
      }
      if (m_num_local_restores_active > 0) {
        jam();
        return;
      }
      m_num_copy_restores_active++;
      m_num_restores_active++;

      c_lcp_waiting_fragments.remove(fragptr);
      c_lcp_restoring_fragments.addLast(fragptr);

      tabptr.i = fragptr.p->tabRef;
      ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);

      fragptr.p->fragStatus = Fragrecord::ACTIVE_CREATION;
      CopyFragReq *req = CAST_PTR(CopyFragReq, signal->getDataPtrSend());
      req->senderData = fragptr.i;
      req->senderRef = reference();
      req->tableId = fragptr.p->tabRef;
      req->fragId = fragptr.p->fragId;
      req->nodeId = getOwnNodeId();
      req->schemaVersion = tabptr.p->schemaVersion;
      req->distributionKey = 0;
      req->gci = fragptr.p->lcpId[0];
      req->nodeCount = 0;
      req->nodeList[1] = CopyFragReq::CFR_NON_TRANSACTIONAL;
      Uint32 nodeId = fragptr.p->srLqhLognode[0];
      Uint32 instanceKey = fragptr.p->lqhInstanceKey;
      Uint32 instanceNo = getInstanceNo(nodeId, instanceKey);
      BlockReference ref = numberToRef(DBLQH, instanceNo, nodeId);
      sendSignal(ref, GSN_COPY_FRAGREQ, signal, CopyFragReq::SignalLength, JBB);
    }
  }
}

void Dblqh::execCOPY_FRAGREF(Signal *signal) {
  jamEntry();

  const CopyFragRef *ref = CAST_CONSTPTR(CopyFragRef, signal->getDataPtr());
  Uint32 errorCode = ref->errorCode;

  SystemError *sysErr = (SystemError *)&signal->theData[0];
  sysErr->errorCode = SystemError::CopyFragRefError;
  sysErr->errorRef = reference();
  sysErr->data[0] = errorCode;
  sysErr->data[1] = 0;
  sendSignal(NDBCNTR_REF, GSN_SYSTEM_ERROR, signal, SystemError::SignalLength,
             JBA);
}

void Dblqh::execCOPY_FRAGCONF(Signal *signal) {
  jamEntry();
  {
    const CopyFragConf *conf =
        CAST_CONSTPTR(CopyFragConf, signal->getDataPtr());
    ndbrequire(c_fragment_pool.getPtr(fragptr, conf->senderData));
    fragptr.p->fragStatus = Fragrecord::CRASH_RECOVERING;

    Uint32 rows_lo = conf->rows_lo;
    Uint32 bytes_lo = conf->bytes_lo;
    signal->theData[0] = NDB_LE_NR_CopyFragDone;
    signal->theData[1] = getOwnNodeId();
    signal->theData[2] = fragptr.p->tabRef;
    signal->theData[3] = fragptr.p->fragId;
    signal->theData[4] = rows_lo;
    signal->theData[5] = 0;
    signal->theData[6] = bytes_lo;
    signal->theData[7] = 0;
    sendSignal(CMVMI_REF, GSN_EVENT_REP, signal, 8, JBB);
    g_eventLogger->debug("(%u)tab(%u,%u), COPY_FRAGCONF: %u rows inserted",
                         instance(), fragptr.p->tabRef, fragptr.p->fragId,
                         rows_lo);
  }

  {
    RestoreLcpConf *conf = (RestoreLcpConf *)signal->getDataPtr();
    conf->senderRef = reference();
    conf->senderData = fragptr.i;
    conf->restoredLcpId = RNIL;
    conf->restoredLocalLcpId = RNIL;
    conf->afterRestore = 0;
    execRESTORE_LCP_CONF(signal);
  }
}

void Dblqh::startFragRefLab(Signal *signal) {
  const StartFragReq *const startFragReq = (StartFragReq *)&signal->theData[0];
  BlockReference userRef = startFragReq->userRef;
  Uint32 userPtr = startFragReq->userPtr;
  signal->theData[0] = userPtr;
  signal->theData[1] = terrorCode;
  signal->theData[2] = cownNodeid;
  sendSignal(userRef, GSN_START_FRAGREF, signal, 3, JBB);
  return;
}  // Dblqh::startFragRefLab()

void Dblqh::execRESTORE_LCP_REF(Signal *signal) {
  jamEntry();
  ndbabort();
  return;
}

void Dblqh::move_start_gci_forward(Signal *signal, Uint32 new_start_gci) {
  Uint32 remove_range = 0;
  for (Uint32 i = 0; i < fragptr.p->srNoLognodes; i++) {
    jam();
    if (fragptr.p->srStartGci[i] == new_start_gci) {
      jam();
      /**
       * The RESTORE block didn't move forward the starting point of
       * the REDO log execution.
       */
      break;
    }
    ndbrequire(new_start_gci > fragptr.p->srStartGci[i]);
    if (fragptr.p->srLastGci[i] >= new_start_gci) {
      jam();
      /**
       * We move it forward within this range, no need to remove any
       * range.
       */
      fragptr.p->srStartGci[i] = new_start_gci;
      break;
    }
    /**
     * The entire first range need to be removed.
     */
    ndbrequire((i + 1) <= fragptr.p->srNoLognodes);
    remove_range++;
  }
  if (remove_range == 0) {
    jam();
    return;
  }
  /**
   * Remove ranges by moving them one step at a time closer to index 0.
   */
  for (Uint32 i = 0; i < remove_range; i++) {
    Uint32 index = fragptr.p->srNoLognodes - 1;
    for (Uint32 j = 0; j < index; j++) {
      fragptr.p->srStartGci[j] = fragptr.p->srStartGci[j + 1];
      fragptr.p->srLastGci[j] = fragptr.p->srLastGci[j + 1];
      fragptr.p->srLqhLognode[j] = fragptr.p->srLqhLognode[j + 1];
    }
    fragptr.p->srNoLognodes--;
  }
}

void Dblqh::execRESTORE_LCP_CONF(Signal *signal) {
  jamEntry();
  RestoreLcpConf *conf = (RestoreLcpConf *)signal->getDataPtr();
  fragptr.i = conf->senderData;
  Uint32 restoredLcpId = conf->restoredLcpId;
  Uint32 restoredLocalLcpId = conf->restoredLocalLcpId;
  Uint32 maxGciCompleted = conf->maxGciCompleted;
  Uint32 afterRestore = conf->afterRestore;
  BlockReference ref = conf->senderRef;
  Uint32 block = refToMain(ref);
  ndbrequire(m_num_restores_active > 0);
  m_num_restores_active--;
  if (block == DBQTUP) {
    jam();
    /* Returning from Recover thread */
    DEB_LCP_RESTORE(("(%u) Recover thread restored", instance()));
    completed_restore(refToInstance(ref));
  } else if (block == DBLQH) {
    jam();
    /* Returning from COPY_FRAGCONF */
    DEB_LCP_RESTORE(("(%u) Copy thread restored", instance()));
    ndbrequire(m_num_copy_restores_active > 0);
    m_num_copy_restores_active--;
  } else {
    jam();
    DEB_LCP_RESTORE(("(%u) Local thread restored", instance()));
    ndbrequire(m_num_local_restores_active > 0);
    m_num_local_restores_active--;
    /* Returning from local Restore operation */
    ndbrequire(block == DBTUP);
  }
  c_fragment_pool.getPtr(fragptr);

  {
    /**
     * Calculate average row size after restore.
     */
    Uint32 max_page_cnt;
    Uint64 row_count;
    Uint64 prev_row_count;
    Uint64 row_change_count;
    Uint64 memory_used_in_bytes;
    c_tup->get_lcp_frag_stats(fragptr.p->tupFragptr,
                              0, /* Ignored when reset flag is false */
                              max_page_cnt, row_count, prev_row_count,
                              row_change_count, memory_used_in_bytes, false);
  }
  c_lcp_restoring_fragments.remove(fragptr);
  c_lcp_complete_fragments.addLast(fragptr);

  DEB_RESTORE(
      ("(%u) restore tab(%u,%u) LCP(%u,%u), maxGciComp: %u"
       " restart LCP(%u,%u) restart MAX LCP(%u,%u), afterRestore: %u",
       instance(), fragptr.p->tabRef, fragptr.p->fragId, restoredLcpId,
       restoredLocalLcpId, maxGciCompleted, c_restart_lcpId,
       c_restart_localLcpId, c_restart_maxLcpId, c_restart_maxLocalLcpId,
       afterRestore));
  if (afterRestore != 0) {
    jam();
    if (restoredLcpId == 0 && restoredLocalLcpId == 0 && maxGciCompleted == 0) {
      jam();
      /**
       * The RESTORE block could not find any LCP for this fragment
       * to restore. So in order to ensure that we don't attempt
       * to execute any UNDO log record we act as if we had hit
       * a CREATE TABLE in the UNDO log and set the UNDO log
       * execution for this fragment to completed.
       *
       * There is no need to move start GCI forward for this fragment
       * since we have not found any newer LCP for sure.
       */
      c_tup->disk_restart_lcp_id(fragptr.p->tabRef, fragptr.p->fragId, RNIL, 0);
    } else {
      jam();
      /**
       * Keep track of minimal lcp-id (including local lcp id)
       * also keep track of maximum tuple of (lcpId, localLcpId).
       * The first time we come we will set the lcp id and local
       * lcp id, after that only set it if the pair is smaller
       * than the previously smallest.
       */
      if ((c_restart_lcpId == 0) || (c_restart_lcpId > restoredLcpId) ||
          (c_restart_lcpId == restoredLcpId &&
           c_restart_localLcpId > restoredLocalLcpId)) {
        jam();
        c_restart_lcpId = restoredLcpId;
        c_restart_localLcpId = restoredLocalLcpId;
      }
      if ((restoredLcpId > c_restart_maxLcpId) ||
          ((restoredLcpId == c_restart_maxLcpId &&
            restoredLocalLcpId > c_restart_maxLocalLcpId))) {
        jam();
        c_restart_maxLcpId = restoredLcpId;
        c_restart_maxLocalLcpId = restoredLocalLcpId;
      }
      c_tup->disk_restart_lcp_id(fragptr.p->tabRef, fragptr.p->fragId,
                                 restoredLcpId, restoredLocalLcpId);
      Uint32 startGci = maxGciCompleted + 1;
      fragptr.p->m_completed_gci = maxGciCompleted;
      move_start_gci_forward(signal, startGci);
    }
  } else {
    ndbrequire(fragptr.p->srNoLognodes == 0);
  }
  if (fragptr.p->srNoLognodes == 0) {
    jam();
    /**
     * 3 potential reasons for getting here:
     * -------------------------------------
     * 1) We have removed at least one range and have no one left. This means
     *    we are now completed also with REDO logging and we can set the
     *    fragment state to active and also set it to enable logging.
     *
     * 2) We are restoring using SFR_COPY_FRAG and in this case afterRestore
     *    is set to 0 and number of log nodes is 0. So REDO logging is
     *    completed.
     *
     * 3) We are performing a node restart and no LCP was found, we ignore
     *    any REDO logging in this case and thus we have also here completed
     *    REDO logging.
     *
     * We need to be careful in setting up all fragments as if REDO logging
     * was done since we could potentially start up an LCP on the fragment
     * even before the copy fragment process is started.
     */
    sendSTART_FRAGCONF(signal);
  }

  if (!c_lcp_waiting_fragments.isEmpty()) {
    send_restore_lcp(signal);
    return;
  }

  if (c_lcp_restoring_fragments.isEmpty() &&
      cstartRecReq == SRR_START_REC_REQ_ARRIVED) {
    jam();
    /* ----------------------------------------------------------------
     *  WE HAVE ALSO RECEIVED AN INDICATION THAT NO MORE FRAGMENTS
     *  NEEDS RESTART.
     *  NOW IT IS TIME TO START EXECUTING THE UNDO LOG.
     * ----------------------------------------------------------------
     *  WE ARE NOW IN A POSITION TO ORDER TUP TO START
     *  EXECUTING THEIR UNDO LOGS. THIS MUST BE DONE BEFORE THE
     *  FRAGMENT LOGS CAN BE EXECUTED.
     * ---------------------------------------------------------------- */

    mark_end_of_lcp_restore(signal);

    /* Log Event denoting the completion of the LCP restore */
    signal->theData[0] = NDB_LE_LCPRestored;
    signal->theData[1] = c_restart_lcpId;
    sendSignal(CMVMI_REF, GSN_EVENT_REP, signal, 2, JBB);

    csrExecUndoLogState = EULS_STARTED;
    lcpPtr.i = 0;
    ptrAss(lcpPtr, lcpRecord);
    lcpPtr.p->m_outstanding = 1;

    if (cstartType == NodeState::ST_INITIAL_NODE_RESTART) {
      jam();
      /**
       * Skip lgman undo...
       */
      signal->theData[0] = LGMAN_REF;
      sendSignal(reference(), GSN_START_RECCONF, signal, 1, JBB);
      return;
    }

    if (!isNdbMtLqh()) {
      jam();
      signal->theData[0] = c_restart_lcpId;
      signal->theData[1] = c_restart_localLcpId;
      sendSignal(LGMAN_REF, GSN_START_RECREQ, signal, 2, JBB);
    } else {
      jam();
      signal->theData[0] = c_restart_lcpId;
      signal->theData[1] = c_restart_localLcpId;
      signal->theData[2] = LGMAN;
      sendSignal(DBLQH_REF, GSN_START_RECREQ, signal, 3, JBB);
    }
  }
}

#define WLS_GCP_COMPLETE 0
#define WLS_RESTART_COMPLETE 2
#define WLS_GCP_COMPLETE_LATE 3

/**
 * The local sysfile ensures that we keep track of what is recoverable
 * locally. It doesn't have to be updated all the time since this is
 * the job of the distributed sysfile in DBDIH. It is used to keep
 * track of GCI restorable during restarts. We also validate that
 * GCI coming from DBDIH is recoverable. We need to maintain the
 * maximum restorable GCI until we have written that the restart
 * is completed into the local sysfile.
 *
 * After the restart is completed we need not update the local sysfile
 * anymore.
 */
void Dblqh::write_local_sysfile_gcp_complete(Signal *signal, Uint32 gci) {
  write_local_sysfile(signal, WLS_GCP_COMPLETE, gci);
}

void Dblqh::write_local_sysfile_gcp_complete_late(Signal *signal, Uint32 gci) {
  write_local_sysfile(signal, WLS_GCP_COMPLETE_LATE, gci);
}

void Dblqh::write_local_sysfile_restart_complete(Signal *signal) {
  write_local_sysfile(signal, WLS_RESTART_COMPLETE, 0);
}

void Dblqh::write_local_sysfile(Signal *signal, Uint32 type, Uint32 gci) {
  WriteLocalSysfileReq *req = (WriteLocalSysfileReq *)signal->getDataPtrSend();
  req->userPointer = type;
  req->userReference = reference();
  Uint32 nodeRestorableFlag;
  ndbrequire(is_first_instance());
  switch (type) {
    case WLS_GCP_COMPLETE:
    case WLS_GCP_COMPLETE_LATE: {
      jam();
      nodeRestorableFlag = ReadLocalSysfileReq::NODE_NOT_RESTORABLE_ON_ITS_OWN;
      req->lastWrite = 0;
      break;
    }
    case WLS_RESTART_COMPLETE: {
      jam();
      nodeRestorableFlag = ReadLocalSysfileReq::NODE_RESTORABLE_ON_ITS_OWN;
      req->lastWrite = 1;
      break;
    }
    default: {
      ndbabort();
      return;  // Keep compiler quiet
    }
  }
  c_outstanding_write_local_sysfile = true;
  req->nodeRestorableOnItsOwn = nodeRestorableFlag;
  req->maxGCIRestorable = gci;
  sendSignal(NDBCNTR_REF, GSN_WRITE_LOCAL_SYSFILE_REQ, signal,
             WriteLocalSysfileReq::SignalLength, JBB);
}

void Dblqh::execWRITE_LOCAL_SYSFILE_CONF(Signal *signal) {
  jam();
  WriteLocalSysfileConf *conf = (WriteLocalSysfileConf *)signal->getDataPtr();
  ndbrequire(is_first_instance());
  c_outstanding_write_local_sysfile = false;
  switch (conf->userPointer) {
    case WLS_GCP_COMPLETE: {
      jam();
      /**
       * This return signal is only sent to first instance and the only impact
       * of it is to send GCP_SAVEREF. All sending of RESTORABLE_GCI_REP is
       * taken care of by the NDBCNTR block in this case.
       */
      ndbrequire(cstartPhase != ZNIL);
      if (c_start_phase_9_waiting) {
        jam();
        GcpRecordPtr localGcpPtr;
        /**
         * We have reached phase 9 during writing of the local sysfile.
         * We proceed immediately to update the local sysfile with the
         * fact that the restart is complete. After this we can synchronize
         * the GCP and report GCP_SAVECONF.
         */
        c_send_gcp_saveref_needed = false;

        if (ccurrentGcprec == RNIL) {
          jam();
          /**
           * Setup GCP record values from local record as we will
           * send a GCP_SAVECONF later
           */
          localGcpPtr.i = 0;
          ptrCheckGuard(localGcpPtr, cgcprecFileSize, gcpRecord);

          localGcpPtr.p->gcpBlockref = c_local_sysfile.m_dihRef;
          localGcpPtr.p->gcpUserptr = c_local_sysfile.m_dihPtr;
          localGcpPtr.p->gcpId = c_local_sysfile.m_save_gci;
          ndbrequire(refToMain(localGcpPtr.p->gcpBlockref) == DBDIH ||
                     refToMain(localGcpPtr.p->gcpBlockref) == DBLQH);
        } else {
          jam();
          /**
           * Check that the GCP record is sane
           */
          localGcpPtr.i = ccurrentGcprec;
          ptrCheckGuard(localGcpPtr, cgcprecFileSize, gcpRecord);
          ndbrequire(refToMain(localGcpPtr.p->gcpBlockref) == DBDIH ||
                     refToMain(localGcpPtr.p->gcpBlockref) == DBLQH);
        }

        write_local_sysfile_restart_complete(signal);
      } else {
        jam();
        write_local_sysfile_gcp_complete_done(signal);
      }
      return;
    }
    case WLS_GCP_COMPLETE_LATE: {
      jam();
      ndbrequire(cstartPhase != ZNIL);
      if (c_start_phase_9_waiting) {
        jam();
        GcpRecordPtr localGcpPtr;
        /**
         * We have reached phase 9 during writing of local sysfile, proceed
         * to write local sysfile with the information that the restart is
         * completed before syncing the GCP.
         */
        write_local_sysfile_restart_complete(signal);
      } else {
        jam();
        /**
         * We have completed the first LCP, but the restart isn't quite done
         * yet. We wrote the local sysfile to keep the maximum restartable
         * GCI value up to date. Now proceed to synch the GCP in this first
         * instance of LDM threads.
         */
        start_synch_gcp(signal);
      }
      return;
    }
    case WLS_RESTART_COMPLETE: {
      jam();
      /**
       * Restart is complete, we have written this into the local sysfile
       * and we are ready to proceed with the last phases of restart and
       * syncing this GCP as requested.
       */
      ndbrequire(cstartPhase != ZNIL);
      ndbrequire(c_start_phase_9_waiting);
      g_eventLogger->info("Restart complete, updated local sysfile");
      write_local_sysfile_restart_complete_done(signal);
      start_synch_gcp(signal);
      return;
    }
    default: {
      ndbabort();
    }
  }
}

/* ***************> */
/*  START_RECREQ  > */
/* ***************> */
void Dblqh::execSTART_RECREQ(Signal *signal) {
  CRASH_INSERTION(5027);

  jamEntry();
  StartRecReq *const req = (StartRecReq *)&signal->theData[0];

  if (signal->getNoOfSections() >= 1) {
    jam();
    Uint32 senderVersion =
        getNodeInfo(refToNode(signal->getSendersBlockRef())).m_version;
    ndbrequire(ndbd_send_node_bitmask_in_section(senderVersion));
    SegmentedSectionPtr ptr;
    SectionHandle handle(this, signal);
    ndbrequire(handle.getSection(ptr, 0));
    ndbrequire(ptr.sz <= NdbNodeBitmask::Size);
    memset(req->sr_nodes, 0, sizeof(req->sr_nodes));
    copy(req->sr_nodes, ptr);
    releaseSections(handle);
  } else {
    memset(req->sr_nodes + NdbNodeBitmask48::Size, 0, _NDB_NBM_DIFF_BYTES);
  }

  cmasterDihBlockref = req->senderRef;

  ndbrequire(crestartNewestGci == 0 ||
             crestartNewestGci == ZUNDEFINED_GCI_LIMIT ||
             crestartNewestGci == req->lastCompletedGci);

  crestartOldestGci = req->keepGci;
  crestartNewestGci = req->lastCompletedGci;
  cnewestGci = req->newestGci;
  DEB_NEWEST_GCI(("(%u)(%u) cnewestGci = %u, line: %u", instance(),
                  m_is_query_block, cnewestGci, __LINE__));
  cstartRecReqData = req->senderData;

  if (check_ndb_versions()) {
    ndbrequire(crestartOldestGci <= crestartNewestGci);
  }

  ndbrequire(req->receivingNodeId == cownNodeid);

  cnewestCompletedGci = cnewestGci;
  c_backup->setRestorableGci(crestartNewestGci);
  cstartRecReq = SRR_START_REC_REQ_ARRIVED;  // StartRecReq has arrived

  if (signal->getLength() == StartRecReq::SignalLength) {
    jam();
    NdbNodeBitmask tmp;
    tmp.assign(NdbNodeBitmask::Size, req->sr_nodes);
    if (!tmp.equal(m_sr_nodes)) {
      char buf0[NdbNodeBitmask::TextLength + 1];
      char buf1[NdbNodeBitmask::TextLength + 1];
      g_eventLogger->info("execSTART_RECREQ changing srnodes from %s to %s",
                          m_sr_nodes.getText(buf0), tmp.getText(buf1));
    }
    m_sr_nodes.assign(NdbNodeBitmask::Size, req->sr_nodes);
  } else {
    jam();
    cstartRecReqData = RNIL;
  }

  {
    LogPartRecordPtr logPartPtr;
    for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
      ptrAss(logPartPtr, logPartRecord);
      logPartPtr.p->logPartNewestCompletedGCI = cnewestCompletedGci;
    }  // for
  }
  /* ------------------------------------------------------------------------
   *   WE HAVE TO SET THE OLDEST AND THE NEWEST GLOBAL CHECKPOINT IDENTITY
   *   THAT WILL SURVIVE THIS SYSTEM RESTART. THIS IS NEEDED SO THAT WE CAN
   *   SET THE LOG HEAD AND LOG TAIL PROPERLY BEFORE STARTING THE SYSTEM AGAIN.
   *   WE ALSO NEED TO SET CNEWEST_GCI TO ENSURE THAT LOG RECORDS ARE EXECUTED
   *   WITH A PROPER GCI.
   *------------------------------------------------------------------------ */

  DEB_LCP(("(%u)START_RECREQ: nodeRestorableGci: %u", instance(),
           crestartNewestGci));
  if (c_lcp_restoring_fragments.isEmpty() &&
      c_lcp_waiting_fragments.isEmpty()) {
    jam();

    mark_end_of_lcp_restore(signal);

    csrExecUndoLogState = EULS_STARTED;

    lcpPtr.i = 0;
    ptrAss(lcpPtr, lcpRecord);
    lcpPtr.p->m_outstanding = 1;

    if (cstartType == NodeState::ST_INITIAL_NODE_RESTART) {
      jam();
      /**
       * Skip lgman undo...
       */
      signal->theData[0] = LGMAN_REF;
      sendSignal(reference(), GSN_START_RECCONF, signal, 1, JBB);
      return;
    }

    if (c_restart_lcpId == 0 && c_restart_localLcpId == 0 &&
        m_restart_local_latest_lcp_id > 1) {
      /**
       * Not a single fragment was restored using an LCP. This should only
       * in nodes added in new node groups since SYSTAB_0 should be in all
       * original nodes from the initial start.
       */
      jam();
      c_restart_lcpId = m_restart_local_latest_lcp_id - 1;
    }
    if (!isNdbMtLqh()) {
      jam();
      signal->theData[0] = c_restart_lcpId;
      signal->theData[1] = c_restart_localLcpId;
      sendSignal(LGMAN_REF, GSN_START_RECREQ, signal, 2, JBB);
    } else {
      jam();
      signal->theData[0] = c_restart_lcpId;
      signal->theData[1] = c_restart_localLcpId;
      signal->theData[2] = LGMAN;
      sendSignal(DBLQH_REF, GSN_START_RECREQ, signal, 3, JBB);
    }
  }  // if
  if (c_lcp_restoring_fragments.isEmpty() &&
      !c_lcp_waiting_fragments.isEmpty()) {
    jam();
    /**
     * This covers the upgrade case where we now know the nodeRestorableGci
     * for our node and thus are prepared to move on with restoring fragments.
     * When the master is on a newer version we don't need to wait for this to
     * happen since there we send this information along with START_FRAGREQ
     * already.
     *
     * We can come here without any START_FRAGREQ being sent.
     */
    send_restore_lcp(signal);
  }
}  // Dblqh::execSTART_RECREQ()

/* ***************>> */
/*  START_RECCONF  > */
/* ***************>> */
void Dblqh::execSTART_RECCONF(Signal *signal) {
  jamEntry();
  lcpPtr.i = 0;
  ptrAss(lcpPtr, lcpRecord);
  ndbrequire(csrExecUndoLogState == EULS_STARTED);
  ndbrequire(lcpPtr.p->m_outstanding);

  Uint32 sender = signal->theData[0];

  if (ERROR_INSERTED(5055)) {
    CLEAR_ERROR_INSERT_VALUE;
  }

  lcpPtr.p->m_outstanding--;
  if (lcpPtr.p->m_outstanding) {
    jam();
    return;
  }

  switch (refToBlock(sender)) {
    case TSMAN:
      jam();
      break;
    case LGMAN:
      jam();
      c_tup->verify_undo_log_execution();
      lcpPtr.p->m_outstanding++;
      if (!isNdbMtLqh()) {
        jam();
        signal->theData[0] = c_restart_lcpId;
        sendSignal(TSMAN_REF, GSN_START_RECREQ, signal, 1, JBB);
      } else {
        jam();
        signal->theData[0] = c_restart_lcpId;
        signal->theData[1] = 0;
        signal->theData[2] = TSMAN;
        sendSignal(DBLQH_REF, GSN_START_RECREQ, signal, 3, JBB);
      }
      return;
      break;
    default:
      ndbabort();
  }

  jam();
  signal->theData[0] = c_restart_maxLcpId;
  signal->theData[1] = c_restart_maxLocalLcpId;
  sendSignal(NDBCNTR_REF, GSN_SET_LOCAL_LCP_ID_REQ, signal, 2, JBB);
}

void Dblqh::execSET_LOCAL_LCP_ID_CONF(Signal *signal) {
  jam();
  c_restart_maxLcpId = signal->theData[0];
  c_restart_maxLocalLcpId = signal->theData[1];
  m_curr_lcp_id = c_restart_maxLcpId;
  m_curr_local_lcp_id = c_restart_maxLocalLcpId;

  csrExecUndoLogState = EULS_COMPLETED;

  g_eventLogger->info("LDM(%u): Completed DD Undo log application", instance());

  sendLOCAL_RECOVERY_COMPLETE_REP(signal,
                                  LocalRecoveryCompleteRep::UNDO_DD_COMPLETED);
  if (cstartType == NodeState::ST_INITIAL_NODE_RESTART) {
    jam();
    cstartRecReq = SRR_REDO_COMPLETE;  // REDO complete

    rebuildOrderedIndexes(signal, 0);
    return;
  }
  c_executing_redo_log = 1;
  g_eventLogger->info(
      "LDM(%u): Starting REDO log execution"
      " phase %u",
      instance(), csrPhasesCompleted);
  startExecSr(signal);
}

void Dblqh::sendLOCAL_RECOVERY_COMPLETE_REP(
    Signal *signal, LocalRecoveryCompleteRep::PhaseIds phaseId) {
  LocalRecoveryCompleteRep *rep =
      (LocalRecoveryCompleteRep *)signal->getDataPtrSend();

  rep->nodeId = getOwnNodeId();
  rep->phaseId = phaseId;
  if (isNdbMtLqh()) {
    jam();
    rep->senderData = cstartRecReqData;
    rep->instanceId = instance();
    sendSignal(DBLQH_REF, GSN_LOCAL_RECOVERY_COMP_REP, signal,
               LocalRecoveryCompleteRep::SignalLengthLocal, JBB);
  } else {
    jam();
    sendSignal(cmasterDihBlockref, GSN_LOCAL_RECOVERY_COMP_REP, signal,
               LocalRecoveryCompleteRep::SignalLengthMaster, JBB);
  }
}

/* ***************> */
/*  START_RECREF  > */
/* ***************> */
void Dblqh::execSTART_RECREF(Signal *signal) {
  jamEntry();
  ndbabort();
}  // Dblqh::execSTART_RECREF()

void Dblqh::rebuildOrderedIndexes(Signal *signal, Uint32 tableId) {
  jamEntry();

  if (tableId == 0) {
    jam();
    g_eventLogger->info("LDM(%u): Starting to rebuild ordered indexes",
                        instance());

    sendLOCAL_RECOVERY_COMPLETE_REP(
        signal, LocalRecoveryCompleteRep::EXECUTE_REDO_LOG_COMPLETED);
  }
  if (tableId >= ctabrecFileSize) {
    jam();
    LogPartRecordPtr logPartPtr;

    for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
      jam();
      ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
      LogFileRecordPtr logFile;
      logFile.i = logPartPtr.p->currentLogfile;
      ptrCheckGuard(logFile, clogFileFileSize, logFileRecord);

      LogPosition head = {logFile.p->fileNo, logFile.p->currentMbyte};
      LogPosition tail = {logPartPtr.p->logTailFileNo,
                          logPartPtr.p->logTailMbyte};
      Uint64 free_mb =
          free_log(head, tail, logPartPtr.p->noLogFiles, clogFileSize);
      Uint32 committed_mbytes = get_committed_mbytes(logPartPtr.p);
      if (free_mb <= (c_free_mb_tail_problem_limit + committed_mbytes)) {
        jam();
        update_log_problem(signal, logPartPtr.p, LogPartRecord::P_TAIL_PROBLEM,
                           true);
      }
    }

    if (!isNdbMtLqh()) {
      /**
       * There should be no disk-ops in flight here...check it
       */
      signal->theData[0] = 12003;
      sendSignal(LGMAN_REF, GSN_DUMP_STATE_ORD, signal, 1, JBB);
    }

    StartRecConf *conf = (StartRecConf *)signal->getDataPtrSend();
    conf->startingNodeId = getOwnNodeId();
    conf->senderData = cstartRecReqData;
    sendSignal(cmasterDihBlockref, GSN_START_RECCONF, signal,
               StartRecConf::SignalLength, JBB);

    g_eventLogger->info(
        "LDM(%u): We have completed restoring our"
        " fragments and executed REDO log and rebuilt"
        " ordered indexes",
        instance());
    return;
  }

  tabptr.i = tableId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  if (!(DictTabInfo::isOrderedIndex(tabptr.p->tableType) &&
        tabptr.p->tableStatus == Tablerec::TABLE_DEFINED)) {
    jam();
    signal->theData[0] = ZREBUILD_ORDERED_INDEXES;
    signal->theData[1] = tableId + 1;
    sendSignal(reference(), GSN_CONTINUEB, signal, 2, JBB);
    return;
  }

  signal->theData[0] = NDB_LE_RebuildIndex;
  signal->theData[1] = instance();
  signal->theData[2] = tableId;
  sendSignal(CMVMI_REF, GSN_EVENT_REP, signal, 3, JBB);

  DEB_INDEX_BUILD(
      ("(%u) Rebuild indexes for primary table %u"
       ", index table %u",
       instance(), tabptr.p->primaryTableId, tableId));

  ndbassert(!m_is_query_block);
  BuildIndxImplReq *const req = (BuildIndxImplReq *)signal->getDataPtrSend();
  req->senderRef = reference();
  req->senderData = tableId;
  req->requestType = BuildIndxImplReq::RF_BUILD_OFFLINE;
  req->buildId = 0;   // not yet..
  req->buildKey = 0;  // ..in use
  req->transId = 0;
  req->indexType = tabptr.p->tableType;
  req->indexId = tableId;
  req->tableId = tabptr.p->primaryTableId;
  req->parallelism = 0;
  sendSignal(calcInstanceBlockRef(DBTUP), GSN_BUILD_INDX_IMPL_REQ, signal,
             BuildIndxImplReq::SignalLength, JBB);
}

void Dblqh::execBUILD_INDX_IMPL_REF(Signal *signal) {
  jamEntry();
  ndbabort();  // TODO error message
}

void Dblqh::execBUILD_INDX_IMPL_CONF(Signal *signal) {
  jamEntry();
  BuildIndxImplConf *conf = (BuildIndxImplConf *)signal->getDataPtr();
  Uint32 tableId = conf->senderData;
  rebuildOrderedIndexes(signal, tableId + 1);
  g_eventLogger->info("LDM(%u): index id %u rebuild done", instance(), tableId);
}

/* ***************>> */
/*  START_EXEC_SR  > */
/* ***************>> */
void Dblqh::execSTART_EXEC_SR(Signal *signal) {
  jamEntry();
  fragptr.i = signal->theData[0];
  Uint32 next = RNIL;

  if (fragptr.i == RNIL) {
    jam();
    /* ----------------------------------------------------------------------
     *    NO MORE FRAGMENTS TO START EXECUTING THE LOG ON.
     *    SEND EXEC_SRREQ TO ALL LQH TO INDICATE THAT THIS NODE WILL
     *    NOT REQUEST ANY MORE FRAGMENTS TO EXECUTE THE FRAGMENT LOG ON.
     * ----------------------------------------------------------------------
     *    WE NEED TO SEND THOSE SIGNALS EVEN IF WE HAVE NOT REQUESTED
     *    ANY FRAGMENTS PARTICIPATE IN THIS PHASE.
     * --------------------------------------------------------------------- */
    signal->theData[0] = cownNodeid;
    if (!isNdbMtLqh()) {
      jam();
      NodeReceiverGroup rg(DBLQH, m_sr_nodes);
      sendSignal(rg, GSN_EXEC_SRREQ, signal, 1, JBB);
    } else {
      jam();
      const Uint32 sz = NdbNodeBitmask::Size;
      m_sr_nodes.copyto(sz, &signal->theData[1]);
      sendSignal(DBLQH_REF, GSN_EXEC_SRREQ, signal, 1 + sz, JBB);
    }
    return;
  } else {
    jam();
    c_lcp_complete_fragments.getPtr(fragptr);
    next = fragptr.p->nextList;

    if (fragptr.p->srNoLognodes > csrPhasesCompleted) {
      jam();
      cnoOutstandingExecFragReq++;

      Uint32 index = csrPhasesCompleted;
      arrGuard(index, MAX_LOG_EXEC);
      Uint32 Tnode = fragptr.p->srLqhLognode[index];
      Uint32 instanceKey = fragptr.p->lqhInstanceKey;
      Uint32 instanceNo = getInstanceNo(Tnode, instanceKey);
      BlockReference ref = numberToRef(DBLQH, instanceNo, Tnode);
      fragptr.p->srStatus = Fragrecord::SS_STARTED;

      /* --------------------------------------------------------------------
       *  SINCE WE CAN HAVE SEVERAL LQH NODES PER FRAGMENT WE CALCULATE
       *  THE LQH POINTER IN SUCH A WAY THAT WE CAN DEDUCE WHICH OF THE
       *  LQH NODES THAT HAS RESPONDED WHEN EXEC_FRAGCONF IS RECEIVED.
       * ------------------------------------------------------------------- */
      ExecFragReq *const execFragReq = (ExecFragReq *)&signal->theData[0];
      execFragReq->userPtr = fragptr.i;
      execFragReq->userRef = cownref;
      execFragReq->tableId = fragptr.p->tabRef;
      execFragReq->fragId = fragptr.p->fragId;
      execFragReq->startGci = fragptr.p->srStartGci[index];
      execFragReq->lastGci = fragptr.p->srLastGci[index];
      execFragReq->dst = ref;

      if (isNdbMtLqh()) {
        jam();
        // send via local proxy
        sendSignal(DBLQH_REF, GSN_EXEC_FRAGREQ, signal,
                   ExecFragReq::SignalLength, JBB);
      } else {
        jam();
        // send via remote proxy
        sendSignal(numberToRef(DBLQH, refToNode(ref)), GSN_EXEC_FRAGREQ, signal,
                   ExecFragReq::SignalLength, JBB);
      }
    }
    signal->theData[0] = next;
    sendSignal(cownref, GSN_START_EXEC_SR, signal, 1, JBB);
  }  // if
}  // Dblqh::execSTART_EXEC_SR()

/* ***************> */
/*  EXEC_FRAGREQ  > */
/* ***************> */
/* --------------------------------------------------------------------------
 *  THIS SIGNAL IS USED TO REQUEST THAT A FRAGMENT PARTICIPATES IN EXECUTING
 *  THE LOG IN THIS NODE.
 * ------------------------------------------------------------------------- */
void Dblqh::execEXEC_FRAGREQ(Signal *signal) {
  ExecFragReq *const execFragReq = (ExecFragReq *)&signal->theData[0];
  jamEntry();
  tabptr.i = execFragReq->tableId;
  Uint32 fragId = execFragReq->fragId;
  ptrCheckGuard(tabptr, ctabrecFileSize, tablerec);
  ndbrequire(getFragmentrec(fragId));

  ndbrequire(fragptr.p->execSrNoReplicas < MAX_REPLICAS);
  fragptr.p->execSrBlockref[fragptr.p->execSrNoReplicas] = execFragReq->userRef;
  fragptr.p->execSrUserptr[fragptr.p->execSrNoReplicas] = execFragReq->userPtr;
  fragptr.p->execSrStartGci[fragptr.p->execSrNoReplicas] =
      execFragReq->startGci;
  fragptr.p->execSrLastGci[fragptr.p->execSrNoReplicas] = execFragReq->lastGci;
  fragptr.p->execSrStatus = Fragrecord::ACTIVE;
  fragptr.p->execSrNoReplicas++;
  /* Need to increment number of fragments to execute for the log part */
  Uint32 instanceKey = getInstanceKey(fragptr.p->tabRef, fragptr.p->fragId);
  LogPartRecord *logPartPtrP = get_log_part_record(instanceKey);
  lock_log_part(logPartPtrP);
  logPartPtrP->m_noFragmentsExecSr++;
  unlock_log_part(logPartPtrP);
}  // Dblqh::execEXEC_FRAGREQ()

void Dblqh::sendExecFragRefLab(Signal *signal) {
  ExecFragReq *const execFragReq = (ExecFragReq *)&signal->theData[0];
  BlockReference retRef = execFragReq->userRef;
  Uint32 retPtr = execFragReq->userPtr;

  signal->theData[0] = retPtr;
  signal->theData[1] = terrorCode;
  sendSignal(retRef, GSN_EXEC_FRAGREF, signal, 2, JBB);
  return;
}  // Dblqh::sendExecFragRefLab()

void Dblqh::sendSTART_FRAGCONF(Signal *signal) {
  /**
   * This signal is ignored in DIH currently, but we still send it to enable
   * future functionality beased on this if needed.
   *
   * This method is called when we are sure that we have completed any REDO
   * log execution needed.
   */
  fragptr.p->logFlag = Fragrecord::STATE_TRUE;
  fragptr.p->fragStatus = Fragrecord::FSACTIVE;

  signal->theData[0] = fragptr.p->srUserptr;
  signal->theData[1] = cownNodeid;
  sendSignal(fragptr.p->srBlockref, GSN_START_FRAGCONF, signal, 2, JBB);
}

/* ***************>> */
/*  EXEC_FRAGCONF  > */
/* ***************>> */
void Dblqh::execEXEC_FRAGCONF(Signal *signal) {
  jamEntry();
  fragptr.i = signal->theData[0];
  c_fragment_pool.getPtr(fragptr);
  fragptr.p->srStatus = Fragrecord::SS_COMPLETED;

  ndbrequire(cnoOutstandingExecFragReq);
  cnoOutstandingExecFragReq--;
  if (fragptr.p->srNoLognodes == csrPhasesCompleted + 1) {
    jam();
    sendSTART_FRAGCONF(signal);
  }
}  // Dblqh::execEXEC_FRAGCONF()

/* ***************> */
/*  EXEC_FRAGREF  > */
/* ***************> */
void Dblqh::execEXEC_FRAGREF(Signal *signal) {
  jamEntry();
  terrorCode = signal->theData[1];
  systemErrorLab(signal, __LINE__);
  return;
}  // Dblqh::execEXEC_FRAGREF()

/* *************** */
/*  EXEC_SRCONF  > */
/* *************** */
void Dblqh::execEXEC_SRCONF(Signal *signal) {
  jamEntry();
  Uint32 nodeId = signal->theData[0];
  arrGuard(nodeId, MAX_NDB_NODES);
  g_eventLogger->info(
      "LDM(%u): Node %u completed LDM restart"
      " phase 3",
      instance(), nodeId);
  m_sr_exec_sr_conf.set(nodeId);

  if (!m_sr_nodes.equal(m_sr_exec_sr_conf)) {
    jam();
    /* ------------------------------------------------------------------
     *  ALL NODES HAVE NOT REPORTED COMPLETION OF EXECUTING FRAGMENT
     *  LOGS YET.
     * ----------------------------------------------------------------- */
    return;
  }

  if (cnoOutstandingExecFragReq != 0) {
    /**
     * This should now have been fixed!
     *   but could occur during upgrade
     * old: wl4391_todo workaround until timing fixed
     */
    jam();
    m_sr_exec_sr_conf.clear(nodeId);
    g_eventLogger->info("delay: reqs=%u", cnoOutstandingExecFragReq);
    ndbabort();
    sendSignalWithDelay(reference(), GSN_EXEC_SRCONF, signal, 10,
                        signal->getLength());
    return;
  }

  /* ------------------------------------------------------------------------
   *  CLEAR NODE SYSTEM RESTART EXECUTION STATE TO PREPARE FOR NEXT PHASE OF
   *  LOG EXECUTION.
   * ----------------------------------------------------------------------- */
  m_sr_exec_sr_conf.clear();

  init_frags_to_execute_sr();
  /* ------------------------------------------------------------------------
   *  NOW CHECK IF ALL FRAGMENTS IN THIS PHASE HAVE COMPLETED. IF SO START THE
   *  NEXT PHASE.
   * ----------------------------------------------------------------------- */
  ndbrequire(cnoOutstandingExecFragReq == 0);

  execSrCompletedLab(signal);
  return;
}  // Dblqh::execEXEC_SRCONF()

void Dblqh::execSrCompletedLab(Signal *signal) {
  csrPhasesCompleted++;
  restart_synch_state(signal, COMPLETED_REDO_LOG_EXEC_SYNCH,
                      ZCONTINUE_REDO_LOG_EXEC_COMPLETED);
}

void Dblqh::continue_execSrCompletedLab(Signal *signal) {
  /* ------------------------------------------------------------------------
   *  ALL FRAGMENTS WERE COMPLETED. THIS PHASE IS COMPLETED. IT IS NOW TIME TO
   *  START THE NEXT PHASE.
   * ----------------------------------------------------------------------- */
  if (csrPhasesCompleted >= MAX_LOG_EXEC) {
    jam();
    /* ----------------------------------------------------------------------
     *  THIS WAS THE LAST PHASE. WE HAVE NOW COMPLETED THE EXECUTION THE
     *  FRAGMENT LOGS IN ALL NODES. BEFORE WE SEND START_RECCONF TO THE
     *  MASTER DIH TO INDICATE A COMPLETED SYSTEM RESTART IT IS NECESSARY
     *  TO FIND THE HEAD AND THE TAIL OF THE LOG WHEN NEW OPERATIONS START
     *  TO COME AGAIN.
     *
     * THE FIRST STEP IS TO FIND THE HEAD AND TAIL MBYTE OF EACH LOG PART.
     * TO DO THIS WE REUSE THE CONTINUEB SIGNAL SR_LOG_LIMITS. THEN WE
     * HAVE TO FIND THE ACTUAL PAGE NUMBER AND PAGE INDEX WHERE TO
     * CONTINUE WRITING THE LOG AFTER THE SYSTEM RESTART.
     * --------------------------------------------------------------------- */
    g_eventLogger->info(
        "LDM(%u): REDO log execution completed, now"
        " finding the new log head + tail",
        instance());
    LogPartRecordPtr logPartPtr;
    for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
      jam();
      ptrAss(logPartPtr, logPartRecord);
      logPartPtr.p->logPartState = LogPartRecord::SR_FOURTH_PHASE_STARTED;
      logPartPtr.p->logLastGci = crestartNewestGci;
      logPartPtr.p->logStartGci = crestartOldestGci;
      logPartPtr.p->logExecState = LogPartRecord::LES_SEARCH_STOP;
      if (logPartPtr.p->headFileNo == ZNIL) {
        jam();
        /* -----------------------------------------------------------------
         *  IF WE HAVEN'T FOUND ANY HEAD OF THE LOG THEN WE ARE IN SERIOUS
         *  PROBLEM.  THIS SHOULD NOT OCCUR. IF IT OCCURS ANYWAY THEN WE
         *  HAVE TO FIND A CURE FOR THIS PROBLEM.
         * ----------------------------------------------------------------- */
        systemErrorLab(signal, __LINE__);
        return;
      }  // if

      if (DEBUG_REDO_REC) {
        g_eventLogger->info(
            "(%u)part: %u srLogLimits SR_FOURTH_PHASE %u-%u"
            " (file: %u mb: %u)",
            instance(), logPartPtr.p->logPartNo, logPartPtr.p->logStartGci,
            logPartPtr.p->logLastGci, logPartPtr.p->lastLogfile,
            logPartPtr.p->lastMbyte);
      }

      signal->theData[0] = ZSR_LOG_LIMITS;
      signal->theData[1] = logPartPtr.i;
      signal->theData[2] = logPartPtr.p->lastLogfile;
      signal->theData[3] = logPartPtr.p->lastMbyte;
      sendSignal(cownref, GSN_CONTINUEB, signal, 4, JBB);
    }  // for
    if (clogPartFileSize == 0) {
      jam();
      g_eventLogger->info(
          "LDM(%u): All redo actions complete (apply,"
          " invalidate)",
          instance());
      restart_synch_state(signal, COMPLETED_REDO_FOURTH_PHASE_SYNCH,
                          ZCONTINUE_SR_FOURTH_COMP);
    }
  } else {
    jam();
    /* ----------------------------------------------------------------------
     *   THERE ARE YET MORE PHASES TO RESTART.
     *   WE MUST INITIALISE DATA FOR NEXT PHASE AND SEND START SIGNAL.
     * --------------------------------------------------------------------- */
    csrPhaseStarted = ZSR_PHASE1_COMPLETED;  // Set correct state first...
    g_eventLogger->info(
        "LDM(%u): Starting REDO log execution"
        " phase %u",
        instance(), csrPhasesCompleted);
    startExecSr(signal);
  }  // if
}  // Dblqh::execSrCompletedLab()

/* ************>> */
/*  EXEC_SRREQ  > */
/* ************>> */
void Dblqh::execEXEC_SRREQ(Signal *signal) {
  jamEntry();
  Uint32 nodeId = signal->theData[0];
  ndbrequire(nodeId < MAX_NDB_NODES);
  g_eventLogger->info("LDM(%u): Node %u ready to execute REDO log", instance(),
                      nodeId);
  m_sr_exec_sr_req.set(nodeId);
  if (!m_sr_exec_sr_req.equal(m_sr_nodes)) {
    jam();
    return;
  }

  /* ------------------------------------------------------------------------
   *  CLEAR NODE SYSTEM RESTART STATE TO PREPARE FOR NEXT PHASE OF LOG
   *  EXECUTION
   * ----------------------------------------------------------------------- */
  m_sr_exec_sr_req.clear();

  g_eventLogger->info(
      "LDM(%u): All starting nodes ready"
      " to execute REDO log.  Phases completed = %u"
      ", fragments to execute: %u",
      instance(), csrPhasesCompleted, get_frags_to_execute_sr());

  if (clogPartFileSize == 0) {
    jam();
    restart_synch_state(signal, START_PHASE3_SYNCH, ZCONTINUE_PHASE3_START);
    return;
  }  // if
  /* ------------------------------------------------------------------------
   *  NOW ALL NODES HAVE SENT ALL EXEC_FRAGREQ. NOW WE CAN START EXECUTING THE
   *  LOG FROM THE MINIMUM GCI NEEDED UNTIL THE MAXIMUM GCI NEEDED.
   *
   *  WE MUST FIRST CHECK IF THE FIRST PHASE OF THE SYSTEM RESTART HAS BEEN
   *  COMPLETED. THIS HANDLING IS PERFORMED IN THE FILE SYSTEM MODULE
   * ----------------------------------------------------------------------- */

  g_eventLogger->info(
      "LDM(%u):"
      "Ready to start execute REDO log phase,"
      " collect REDO log execution info phase completed",
      instance());

  signal->theData[0] = ZSR_PHASE3_START;
  signal->theData[1] = ZSR_PHASE2_COMPLETED;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
  return;
}  // Dblqh::execEXEC_SRREQ()

/* ######################################################################### */
/*       SYSTEM RESTART PHASE THREE MODULE                                   */
/*       THIS MODULE IS A SUB-MODULE OF THE FILE SYSTEM HANDLING.            */
/*                                                                           */
/* THIS MODULE IS CONCERNED WITH EXECUTING THE FRAGMENT LOG. IT DOES ALSO    */
/* CONTAIN SIGNAL RECEPTIONS LQHKEYCONF AND LQHKEYREF SINCE LQHKEYREQ IS USED*/
/* TO EXECUTE THE LOG RECORDS.                                               */
/*                                                                           */
/* BEFORE IT STARTS IT HAS BEEN DECIDED WHERE TO START AND WHERE TO STOP     */
/* READING THE FRAGMENT LOG BY USING THE INFORMATION ABOUT GCI DISCOVERED IN */
/* PHASE ONE OF THE SYSTEM RESTART.                                          */
/* ######################################################################### */
/*---------------------------------------------------------------------------*/
/* PHASE THREE OF THE SYSTEM RESTART CAN NOW START. ONE OF THE PHASES HAVE   */
/* COMPLETED.                                                                */
/*---------------------------------------------------------------------------*/
void Dblqh::srPhase3Start(Signal *signal) {
  UintR tsrPhaseStarted;

  jamEntry();

  tsrPhaseStarted = signal->theData[1];
  if (csrPhaseStarted == ZSR_NO_PHASE_STARTED) {
    jam();
    DEB_RESTART_SYNCH(
        ("(%u) srPhase3Start: Wait for other phase,"
         " got phase %u",
         instance(), tsrPhaseStarted));
    csrPhaseStarted = tsrPhaseStarted;
    return;
  }  // if
  ndbrequire(csrPhaseStarted != tsrPhaseStarted);
  ndbrequire(csrPhaseStarted != ZSR_BOTH_PHASES_STARTED);

  csrPhaseStarted = ZSR_BOTH_PHASES_STARTED;

  LogPartRecordPtr logPartPtr;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    logPartPtr.p->logPartState = LogPartRecord::SR_THIRD_PHASE_STARTED;
    logPartPtr.p->logStartGci = (UintR)-1;
    if (csrPhasesCompleted == 0) {
      jam();
      /* --------------------------------------------------------------------
       *  THE FIRST PHASE WE MUST ENSURE THAT IT REACHES THE END OF THE LOG.
       * ------------------------------------------------------------------- */
      logPartPtr.p->logLastGci = crestartNewestGci;
    } else {
      jam();
      logPartPtr.p->logLastGci = 2;
    }  // if
  }    // for
  restart_synch_state(signal, START_PHASE3_SYNCH, ZCONTINUE_PHASE3_START);
}

void Dblqh::continue_srPhase3Start(Signal *signal) {
  jam();
  c_lcp_complete_fragments.first(fragptr);
  signal->theData[0] = ZSR_GCI_LIMITS;
  signal->theData[1] = fragptr.i;
  sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
  return;
}  // Dblqh::srPhase3Start()

/* --------------------------------------------------------------------------
 *   WE NOW WE NEED TO FIND THE LIMITS WITHIN WHICH TO EXECUTE
 *   THE FRAGMENT LOG
 * ------------------------------------------------------------------------- */
void Dblqh::srGciLimits(Signal *signal) {
  jamEntry();
  fragptr.i = signal->theData[0];
  Uint32 loopCount = 0;
  while (fragptr.i != RNIL) {
    jam();
    c_lcp_complete_fragments.getPtr(fragptr);
    ndbrequire(fragptr.p->execSrNoReplicas - 1 < MAX_REPLICAS);
    Uint32 instanceKey = getInstanceKey(fragptr.p->tabRef, fragptr.p->fragId);
    LogPartRecord *logPartPtrP = get_log_part_record(instanceKey);
    lock_log_part(logPartPtrP);
    for (Uint32 i = 0; i < fragptr.p->execSrNoReplicas; i++) {
      jam();
      if (fragptr.p->execSrStartGci[i] < logPartPtrP->logStartGci) {
        jam();
        logPartPtrP->logStartGci = fragptr.p->execSrStartGci[i];
      }  // if
      if (fragptr.p->execSrLastGci[i] > logPartPtrP->logLastGci) {
        jam();
        /**
         * We cannot run past the end point in the REDO log in our node.
         */
        ndbrequire(csrPhasesCompleted != 0);
        logPartPtrP->logLastGci = fragptr.p->execSrLastGci[i];
      }
    }
    unlock_log_part(logPartPtrP);

    loopCount++;
    if (loopCount > 20) {
      jam();
      signal->theData[0] = ZSR_GCI_LIMITS;
      signal->theData[1] = fragptr.p->nextList;
      sendSignal(cownref, GSN_CONTINUEB, signal, 2, JBB);
      return;
    } else {
      jam();
      fragptr.i = fragptr.p->nextList;
    }  // if
  }
  restart_synch_state(signal, START_LOG_LIMITS_SYNCH, ZCONTINUE_SR_GCI_LIMITS);
}

void Dblqh::init_frags_to_execute_sr() {
  LogPartRecordPtr logPartPtr;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    logPartPtr.p->m_noFragmentsExecSr = 0;
  }
}

Uint32 Dblqh::get_frags_to_execute_sr() {
  LogPartRecordPtr logPartPtr;
  Uint32 no_frags_to_execute = 0;
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    no_frags_to_execute += logPartPtr.p->m_noFragmentsExecSr;
  }
  return no_frags_to_execute;
}

void Dblqh::continue_srGciLimits(Signal *signal) {
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;

  if (clogPartFileSize == 0) {
    jam();
    execLogComp_extra_files_closed(signal);
    return;
  }
  if (get_frags_to_execute_sr() == 0 && csrPhasesCompleted > 0) {
    jam();
    execLogComp_extra_files_closed(signal);
    return;
  }
  for (logPartPtr.i = 0; logPartPtr.i < clogPartFileSize; logPartPtr.i++) {
    jam();
    ptrAss(logPartPtr, logPartRecord);
    if (logPartPtr.p->m_noFragmentsExecSr == 0) {
      jam();
      logPartPtr.p->logLastGci = crestartNewestGci;
      logPartPtr.p->logStartGci = crestartNewestGci;
    } else if (logPartPtr.p->logStartGci == (UintR)-1) {
      /* -------------------------------------------------------------------
       *  THERE WERE NO FRAGMENTS TO INSTALL WE WILL EXECUTE THE LOG AS
       *  SHORT AS POSSIBLE TO REACH THE END OF THE LOG. THIS WE DO BY
       *  STARTING AT THE STOP GCI.
       * ----------------------------------------------------------------- */
      jam();
      logPartPtr.p->logStartGci = logPartPtr.p->logLastGci;
    }
    logPartPtr.p->logExecState = LogPartRecord::LES_SEARCH_STOP;
    if (DEBUG_REDO_REC) {
      g_eventLogger->info(
          "(%u)part: %u srLogLimits (srGciLimits) %u-%u"
          " (file: %u mb: %u)",
          instance(), logPartPtr.p->logPartNo, logPartPtr.p->logStartGci,
          logPartPtr.p->logLastGci, logPartPtr.p->lastLogfile,
          logPartPtr.p->lastMbyte);
    }
    signal->theData[0] = ZSR_LOG_LIMITS;
    signal->theData[1] = logPartPtr.i;
    signal->theData[2] = logPartPtr.p->lastLogfile;
    signal->theData[3] = logPartPtr.p->lastMbyte;
    sendSignal(cownref, GSN_CONTINUEB, signal, 4, JBB);

    logFilePtr.i = logPartPtr.p->lastLogfile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    g_eventLogger->info(
        "LDM(%u): Log part %u will execute REDO log"
        " records from GCI %u -> %u and last log file is "
        "number %u and last MByte in this file is %u",
        instance(), logPartPtr.p->logPartNo, logPartPtr.p->logStartGci,
        logPartPtr.p->logLastGci,
        logFilePtr.p->fileNo, /* fileNo of last log file */
        logPartPtr.p->lastMbyte);
  }  // for
}  // Dblqh::srGciLimits()

/* --------------------------------------------------------------------------
 *       IT IS NOW TIME TO FIND WHERE TO START EXECUTING THE LOG.
 *       THIS SIGNAL IS SENT FOR EACH LOG PART AND STARTS THE EXECUTION
 *       OF THE LOG FOR THIS PART.
 *-------------------------------------------------------------------------- */
void Dblqh::srLogLimits(Signal *signal) {
  Uint32 tlastPrepRef = 0;
  Uint32 tmbyte;
  LogFileRecordPtr logFilePtr;
  LogPartRecordPtr logPartPtr;

  jamEntry();
  logPartPtr.i = signal->theData[0];
  ptrCheckGuard(logPartPtr, clogPartFileSize, logPartRecord);
  logFilePtr.i = signal->theData[1];
  ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
  tmbyte = signal->theData[2];
  Uint32 loopCount = 0;
  /* ------------------------------------------------------------------------
   *   WE ARE SEARCHING FOR THE START AND STOP MBYTE OF THE LOG THAT IS TO BE
   *   EXECUTED.
   * ----------------------------------------------------------------------- */
  while (true) {
    ndbrequire(tmbyte < clogFileSize);
    if (logPartPtr.p->logExecState == LogPartRecord::LES_SEARCH_STOP) {
      if (logFilePtr.p->logMaxGciCompleted[tmbyte] <=
          logPartPtr.p->logLastGci) {
        jam();
        /* ------------------------------------------------------------------
         *  WE ARE STEPPING BACKWARDS FROM MBYTE TO MBYTE. THIS IS THE FIRST
         *  MBYTE WHICH IS TO BE INCLUDED IN THE LOG EXECUTION. THE STOP GCI
         *  HAS NOT BEEN COMPLETED BEFORE THIS MBYTE. THUS THIS MBYTE HAVE
         *  TO BE EXECUTED.
         * ---------------------------------------------------------------- */
        logPartPtr.p->stopLogfile = logFilePtr.i;
        logPartPtr.p->stopMbyte = tmbyte;
        logPartPtr.p->logExecState = LogPartRecord::LES_SEARCH_START;
        if (DEBUG_REDO_REC) {
          g_eventLogger->info(
              "(%u)part: %u srLogLimits found stop pos"
              " file: %u mb: %u"
              " logMaxGciCompleted[tmbyte]: %u (lastGci: %u)",
              instance(), logPartPtr.p->logPartNo, logFilePtr.p->fileNo, tmbyte,
              logFilePtr.p->logMaxGciCompleted[tmbyte],
              logPartPtr.p->logLastGci);
        }
      }  // if
      else if (DEBUG_REDO_REC) {
        g_eventLogger->info(
            "(%u)SEARCH STOP SKIP part: %u file: %u mb: %u "
            "logMaxGciCompleted: %u > %u",
            instance(), logPartPtr.p->logPartNo, logFilePtr.p->fileNo, tmbyte,
            logFilePtr.p->logMaxGciCompleted[tmbyte], logPartPtr.p->logLastGci);
      }
    }  // if
    /* ----------------------------------------------------------------------
     * WHEN WE HAVEN'T FOUND THE STOP MBYTE IT IS NOT NECESSARY TO LOOK FOR THE
     * START MBYTE. THE REASON IS THE FOLLOWING LOGIC CHAIN:
     *   MAX_GCI_STARTED >= MAX_GCI_COMPLETED >= LAST_GCI >= START_GCI
     * THUS MAX_GCI_STARTED >= START_GCI. THUS MAX_GCI_STARTED < START_GCI CAN
     * NOT BE TRUE AS WE WILL CHECK OTHERWISE.
     * -------------------------------------------------------------------- */
    if (logPartPtr.p->logExecState == LogPartRecord::LES_SEARCH_START) {
      if (logFilePtr.p->logMaxGciStarted[tmbyte] < logPartPtr.p->logStartGci) {
        jam();
        /* ------------------------------------------------------------------
         *  WE HAVE NOW FOUND THE START OF THE EXECUTION OF THE LOG.
         *  WE STILL HAVE TO MOVE IT BACKWARDS TO ALSO INCLUDE THE
         *  PREPARE RECORDS WHICH WERE STARTED IN A PREVIOUS MBYTE.
         * ---------------------------------------------------------------- */
        if (DEBUG_REDO_REC) {
          g_eventLogger->info(
              "(%u)part: %u srLogLimits found start"
              " pos file: %u mb: %u"
              " logMaxGciStarted[tmbyte]: %u (startGci: %u)",
              instance(), logPartPtr.p->logPartNo, logFilePtr.p->fileNo, tmbyte,
              logFilePtr.p->logMaxGciCompleted[tmbyte],
              logPartPtr.p->logStartGci);
          g_eventLogger->info(
              "(%u)part: %u srLogLimits lastPrepRef => file:"
              " %u mb: %u",
              instance(), logPartPtr.p->logPartNo,
              logFilePtr.p->logLastPrepRef[tmbyte] >> 16,
              logFilePtr.p->logLastPrepRef[tmbyte] & 65535);
        }
        tlastPrepRef = logFilePtr.p->logLastPrepRef[tmbyte];
        logPartPtr.p->startMbyte = tlastPrepRef & 65535;
        LogFileRecordPtr locLogFilePtr;
        findLogfile(signal, tlastPrepRef >> 16, logPartPtr.p, &locLogFilePtr);
        logPartPtr.p->startLogfile = locLogFilePtr.i;
        logPartPtr.p->logExecState = LogPartRecord::LES_EXEC_LOG;
      } else if (DEBUG_REDO_REC) {
        g_eventLogger->info(
            "(%u)SEARCH START SKIP part: %u file: %u mb: %u "
            "logMaxGciCompleted: %u >= %u",
            instance(), logPartPtr.p->logPartNo, logFilePtr.p->fileNo, tmbyte,
            logFilePtr.p->logMaxGciStarted[tmbyte], logPartPtr.p->logStartGci);
      }
    }  // if
    if (logPartPtr.p->logExecState != LogPartRecord::LES_EXEC_LOG) {
      if (tmbyte == 0) {
        jam();
        tmbyte = clogFileSize - 1;
        logFilePtr.i = logFilePtr.p->prevLogFile;
        ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
      } else {
        jam();
        tmbyte--;
      }  // if
      if (logPartPtr.p->lastLogfile == logFilePtr.i) {
        ndbrequire(logPartPtr.p->lastMbyte != tmbyte);
      }  // if
      if (loopCount > 20) {
        jam();
        signal->theData[0] = ZSR_LOG_LIMITS;
        signal->theData[1] = logPartPtr.i;
        signal->theData[2] = logFilePtr.i;
        signal->theData[3] = tmbyte;
        sendSignal(cownref, GSN_CONTINUEB, signal, 4, JBB);
        return;
      }  // if
      loopCount++;
    } else {
      jam();
      break;
    }  // if
  }    // while

  if (DEBUG_REDO_REC) {
    LogFileRecordPtr tmp;
    tmp.i = logPartPtr.p->stopLogfile;
    ptrCheckGuard(tmp, clogFileFileSize, logFileRecord);
    g_eventLogger->info(
        "(%u)srLogLimits part: %u gci: %u-%u start"
        " file: %u mb: %u"
        " stop file: %u mb: %u",
        instance(), logPartPtr.p->logPartNo, logPartPtr.p->logStartGci,
        logPartPtr.p->logLastGci, tlastPrepRef >> 16, tlastPrepRef & 65535,
        tmp.p->fileNo, logPartPtr.p->stopMbyte);
  }

  /* ------------------------------------------------------------------------
   *  WE HAVE NOW FOUND BOTH THE START AND THE STOP OF THE LOG. NOW START
   *  EXECUTING THE LOG. THE FIRST ACTION IS TO OPEN THE LOG FILE WHERE TO
   *  START EXECUTING THE LOG.
   * ----------------------------------------------------------------------- */
  if (logPartPtr.p->logPartState == LogPartRecord::SR_THIRD_PHASE_STARTED) {
    jam();
    logFilePtr.i = logPartPtr.p->startLogfile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    logFilePtr.p->logFileStatus = LogFileRecord::OPEN_EXEC_SR_START;
    openFileRw(signal, logFilePtr);
    g_eventLogger->info(
        "LDM(%u): Start executing REDO log for"
        " part %u",
        instance(), logPartPtr.p->logPartNo);

    send_runredo_event(signal, logPartPtr.p, logPartPtr.p->logStartGci);
  } else {
    jam();

    g_eventLogger->info(
        "LDM(%u): Found log limits for REDO"
        " post-restart for log part %u",
        instance(), logPartPtr.p->logPartNo);

    ndbrequire(logPartPtr.p->logPartState ==
               LogPartRecord::SR_FOURTH_PHASE_STARTED);
    /* --------------------------------------------------------------------
     *  WE HAVE NOW FOUND THE TAIL MBYTE IN THE TAIL FILE.
     *  SET THOSE PARAMETERS IN THE LOG PART.
     *  WE HAVE ALSO FOUND THE HEAD MBYTE. WE STILL HAVE TO SEARCH
     *  FOR THE PAGE NUMBER AND PAGE INDEX WHERE TO SET THE HEAD.
     * ------------------------------------------------------------------- */
    logFilePtr.i = logPartPtr.p->startLogfile;
    ptrCheckGuard(logFilePtr, clogFileFileSize, logFileRecord);
    logPartPtr.p->logTailFileNo = logFilePtr.p->fileNo;
    logPartPtr.p->logTailMbyte = logPartPtr.p->startMbyte;
    /* --------------------------------------------------------------------
     *  THE HEAD WE ACTUALLY FOUND DURING EXECUTION OF LOG SO WE USE
     *  THIS INFO HERE RATHER THAN THE MBYTE WE FOUND TO BE THE HEADER.
     * ------------------------------------------------------------------- */
    LogFileRecordPtr locLogFilePtr;
    findLogfile(signal, logPartPtr.p->headFileNo, logPartPtr.p, &locLogFilePtr);
    locLogFilePtr.p->logFileStatus = LogFileRecord::OPEN_SR_FOURTH_PHASE;
    openFileRw(signal, locLogFilePtr);
  }  // if
}  // Dblqh::srLogLimits()

void Dblqh::openExecSrStartLab(Signal *signal, LogFileRecordPtr logFilePtr,
                               LogPartRecord *logPartPtrP) {
  logPartPtrP->currentLogfile = logFilePtr.i;
  logFilePtr.p->currentMbyte = logPartPtrP->startMbyte;
  /* ------------------------------------------------------------------------
   *     WE NEED A TC CONNECT RECORD TO HANDLE EXECUTION OF LOG RECORDS.
   *     This will always succeed since we don't interact with user
   *     operations during recovery when we are applying the REDO log content.
   * ------------------------------------------------------------------------ */
  TcConnectionrecPtr tcConnectptr;
  seizeTcrec(tcConnectptr);
  ndbrequire(Magic::check_ptr(tcConnectptr.p));
  logPartPtrP->logTcConrec = tcConnectptr.i;
  /* ------------------------------------------------------------------------
   *   THE FIRST LOG RECORD TO EXECUTE IS ALWAYS AT A NEW MBYTE.
   *   SET THE NUMBER OF PAGES IN THE MAIN MEMORY BUFFER TO ZERO AS AN INITIAL
   *   VALUE. THIS VALUE WILL BE UPDATED AND ENSURED THAT IT RELEASES PAGES IN
   *   THE SUBROUTINE READ_EXEC_SR.
   * ----------------------------------------------------------------------- */
  logPartPtrP->mmBufferSize = 0;
  readExecSrNewMbyte(signal, logFilePtr, logPartPtrP);
}  // Dblqh::openExecSrStartLab()

/* ---------------------------------------------------------------------------
 *  WE WILL ALWAYS ENSURE THAT WE HAVE AT LEAST 16 KBYTE OF LOG PAGES WHEN WE
 *  START READING A LOG RECORD. THE ONLY EXCEPTION IS WHEN WE COME CLOSE TO A
 *  MBYTE BOUNDARY. SINCE WE KNOW THAT LOG RECORDS ARE NEVER WRITTEN ACROSS A
 *  MBYTE BOUNDARY THIS IS NOT A PROBLEM.
 *
 *  WE START BY READING 64 KBYTE BEFORE STARTING TO EXECUTE THE LOG RECORDS.
 *  WHEN WE COME BELOW 64 KBYTE WE READ ANOTHER SET OF LOG PAGES. WHEN WE
 *  GO BELOW 16 KBYTE WE WAIT UNTIL THE READ PAGES HAVE ENTERED THE BLOCK.
 * ------------------------------------------------------------------------- */
/* --------------------------------------------------------------------------
 *       NEW PAGES FROM LOG FILE DURING EXECUTION OF LOG HAS ARRIVED.
 * ------------------------------------------------------------------------- */
void Dblqh::readExecSrLab(Signal *signal, LogFileOperationRecord