﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/ssm/model/ResourceType.h>
#include <aws/core/utils/HashingUtils.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>

using namespace Aws::Utils;


namespace Aws
{
  namespace SSM
  {
    namespace Model
    {
      namespace ResourceTypeMapper
      {

        static const int ManagedInstance_HASH = HashingUtils::HashString("ManagedInstance");
        static const int Document_HASH = HashingUtils::HashString("Document");
        static const int EC2Instance_HASH = HashingUtils::HashString("EC2Instance");


        ResourceType GetResourceTypeForName(const Aws::String& name)
        {
          int hashCode = HashingUtils::HashString(name.c_str());
          if (hashCode == ManagedInstance_HASH)
          {
            return ResourceType::ManagedInstance;
          }
          else if (hashCode == Document_HASH)
          {
            return ResourceType::Document;
          }
          else if (hashCode == EC2Instance_HASH)
          {
            return ResourceType::EC2Instance;
          }
          EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
          if(overflowContainer)
          {
            overflowContainer->StoreOverflow(hashCode, name);
            return static_cast<ResourceType>(hashCode);
          }

          return ResourceType::NOT_SET;
        }

        Aws::String GetNameForResourceType(ResourceType enumValue)
        {
          switch(enumValue)
          {
          case ResourceType::ManagedInstance:
            return "ManagedInstance";
          case ResourceType::Document:
            return "Document";
          case ResourceType::EC2Instance:
            return "EC2Instance";
          default:
            EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
            if(overflowContainer)
            {
              return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
            }

            return {};
          }
        }

      } // namespace ResourceTypeMapper
    } // namespace Model
  } // namespace SSM
} // namespace Aws
