/*======
This file is part of Percona Server for MongoDB.

Copyright (C) 2025-present Percona and/or its affiliates. All rights reserved.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the Server Side Public License, version 1,
    as published by MongoDB, Inc.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    Server Side Public License for more details.

    You should have received a copy of the Server Side Public License
    along with this program. If not, see
    <http://www.mongodb.com/licensing/server-side-public-license>.

    As a special exception, the copyright holders give permission to link the
    code of portions of this program with the OpenSSL library under certain
    conditions as described in each individual source file and distribute
    linked combinations including the program with the OpenSSL library. You
    must comply with the Server Side Public License in all respects for
    all of the code used other than as permitted herein. If you modify file(s)
    with this exception, you may extend this exception to your version of the
    file(s), but you are not obligated to do so. If you do not wish to do so,
    delete this exception statement from your version. If you delete this
    exception statement from all source files in the program, then also delete
    it in the license file.
======= */

#pragma once

#include <set>
#include <string>
#include <tuple>
#include <utility>

#include <boost/optional/optional.hpp>

#include "mongo/base/status_with.h"
#include "mongo/base/string_data.h"
#include "mongo/db/auth/oidc/oidc_server_parameters_gen.h"
#include "mongo/db/auth/oidc_protocol_gen.h"
#include "mongo/db/auth/role_name.h"
#include "mongo/db/auth/sasl_mechanism_policies.h"
#include "mongo/db/auth/sasl_mechanism_registry.h"
#include "mongo/db/auth/user.h"
#include "mongo/db/operation_context.h"
#include "mongo/db/service_context.h"


namespace mongo {

namespace crypto {
class JWSValidatedToken;
}  // namespace crypto

class SaslOidcServerMechanism final : public MakeServerMechanism<OidcPolicy> {
public:
    explicit SaslOidcServerMechanism(std::string authenticationDatabase)
        : MakeServerMechanism<OidcPolicy>{std::move(authenticationDatabase)},
          _expirationTime{Date_t::min()} {}

    boost::optional<unsigned int> currentStep() const final {
        return _step;
    }

    boost::optional<unsigned int> totalSteps() const final {
        return 2u;
    }

    UserRequest getUserRequest() const final;

    boost::optional<Date_t> getExpirationTime() const final;

    void appendExtraInfo(BSONObjBuilder* builder) const final;

private:
    StatusWith<std::tuple<bool, std::string>> stepImpl(OperationContext* opCtx,
                                                       StringData input) final;
    StatusWith<std::tuple<bool, std::string>> step1(ServiceContext* serviceContext,
                                                    const auth::OIDCMechanismClientStep1& request);
    StatusWith<std::tuple<bool, std::string>> step2(ServiceContext* serviceContext,
                                                    const auth::OIDCMechanismClientStep2& request);

    void processPrincipalName(const OidcIdentityProviderConfig& idp,
                              const crypto::JWSValidatedToken& token);
    void processAuthorizationClaim(const OidcIdentityProviderConfig& idp,
                                   const crypto::JWSValidatedToken& token);

    void processLogClaims(const OidcIdentityProviderConfig& idp,
                          const crypto::JWSValidatedToken& token);
    unsigned int _step{0};
    boost::optional<std::set<RoleName>> _roles;
    Date_t _expirationTime;
    BSONObj _claimsObj;
};

class OidcServerFactory final : public MakeServerFactory<SaslOidcServerMechanism> {
public:
    using MakeServerFactory<SaslOidcServerMechanism>::MakeServerFactory;

    static constexpr bool isInternal{false};

    bool canMakeMechanismForUser(const User* user) const final {
        return user->getCredentials().isExternal;
    }
};

}  // namespace mongo
