// Copyright (C) 2014 Klaralvdalens Datakonsult AB (KDAB).
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#include "quick3dentityloader_p_p.h"

#include <QtQml/QQmlContext>
#include <QtQml/QQmlEngine>
#include <QtQml/QQmlIncubator>

#include <QtQml/private/qqmlengine_p.h>

QT_BEGIN_NAMESPACE

namespace Qt3DCore {
namespace Quick {

namespace {
struct Quick3DQmlOwner
{
    Quick3DQmlOwner(QQmlEngine *e, QObject *o)
        : engine(e)
        , object(o)
    {}

    QQmlEngine *engine;
    QObject *object;

    QQmlContext *context() const
    {
        return engine->contextForObject(object);
    }
};

Quick3DQmlOwner _q_findQmlOwner(QObject *object)
{
    auto o = object;
    while (!qmlEngine(o) && o->parent())
        o = o->parent();
    return Quick3DQmlOwner(qmlEngine(o), o);
}
}

class Quick3DEntityLoaderIncubator : public QQmlIncubator
{
public:
    Quick3DEntityLoaderIncubator(Quick3DEntityLoader *loader)
        : QQmlIncubator(AsynchronousIfNested),
          m_loader(loader)
    {
    }

protected:
    void statusChanged(Status status) final
    {
        Quick3DEntityLoaderPrivate *priv = Quick3DEntityLoaderPrivate::get(m_loader);

        switch (status) {
        case Ready: {
            Q_ASSERT(priv->m_entity == nullptr);
            priv->m_entity = qobject_cast<QEntity *>(object());
            Q_ASSERT(priv->m_entity != nullptr);
            priv->m_entity->setParent(m_loader);
            emit m_loader->entityChanged();
            priv->setStatus(Quick3DEntityLoader::Ready);
            break;
        }

        case Loading: {
            priv->setStatus(Quick3DEntityLoader::Loading);
            break;
        }

        case Error: {
            QQmlEnginePrivate::warning(_q_findQmlOwner(m_loader).engine, errors());
            priv->clear();
            emit m_loader->entityChanged();
            priv->setStatus(Quick3DEntityLoader::Error);
            break;
        }

        default:
            break;
        }
    }

private:
    Quick3DEntityLoader *m_loader;
};

/*!
    \qmltype EntityLoader
    \inqmlmodule Qt3D.Core
    \inherits Entity
    \since 5.5
    \brief Provides a way to dynamically load an Entity subtree.

    An EntityLoader provides the facitily to load predefined set of entities
    from qml source file. EntityLoader itself is an entity and the loaded entity
    tree is set as a child of the loader. The loaded entity tree root can be
    accessed with EntityLoader::entity property.

    \badcode
        EntityLoader {
            id: loader
            source: "./SphereEntity.qml"
        }
    \endcode
*/

Quick3DEntityLoader::Quick3DEntityLoader(QNode *parent)
    : QEntity(*new Quick3DEntityLoaderPrivate, parent)
{
}

Quick3DEntityLoader::~Quick3DEntityLoader()
{
    Q_D(Quick3DEntityLoader);
    d->clear();
}

/*!
    \qmlproperty QtQml::QtObject EntityLoader::entity
    Holds the loaded entity tree root.
    \readonly

    This property allows access to the content of the loader. It references
    either a valid Entity object if the status property equals
    EntityLoader.Ready, it is equal to null otherwise.
*/
QObject *Quick3DEntityLoader::entity() const
{
    Q_D(const Quick3DEntityLoader);
    return d->m_entity;
}

/*!
    \qmlproperty url Qt3D.Core::EntityLoader::source
    Holds the source url.
*/
QUrl Quick3DEntityLoader::source() const
{
    Q_D(const Quick3DEntityLoader);
    return d->m_source;
}

void Quick3DEntityLoader::setSource(const QUrl &url)
{
    Q_D(Quick3DEntityLoader);

    if (url == d->m_source)
        return;

    d->clear();
    d->m_source = url;
    emit sourceChanged();
    d->loadFromSource();
}

QQmlComponent *Quick3DEntityLoader::sourceComponent() const
{
    Q_D(const Quick3DEntityLoader);
    return d->m_sourceComponent;
}

void Quick3DEntityLoader::setSourceComponent(QQmlComponent *component)
{
    Q_D(Quick3DEntityLoader);
    if (d->m_sourceComponent == component)
        return;

    d->clear();
    d->m_sourceComponent = component;
    emit sourceComponentChanged();
    d->loadComponent(d->m_sourceComponent);
}

/*!
    \qmlproperty Status Qt3D.Core::EntityLoader::status

    Holds the status of the entity loader.
    \list
    \li EntityLoader.Null
    \li EntityLoader.Loading
    \li EntityLoader.Ready
    \li EntityLoader.Error
    \endlist
 */
Quick3DEntityLoader::Status Quick3DEntityLoader::status() const
{
    Q_D(const Quick3DEntityLoader);
    return d->m_status;
}

Quick3DEntityLoaderPrivate::Quick3DEntityLoaderPrivate()
    : QEntityPrivate(),
      m_incubator(nullptr),
      m_context(nullptr),
      m_component(nullptr),
      m_sourceComponent(nullptr),
      m_entity(nullptr),
      m_status(Quick3DEntityLoader::Null)
{
}

void Quick3DEntityLoaderPrivate::clear()
{
    if (m_incubator) {
        m_incubator->clear();
        delete m_incubator;
        m_incubator = nullptr;
    }

    if (m_entity) {
        m_entity->setParent(Q_NODE_NULLPTR);
        delete m_entity;
        m_entity = nullptr;
    }

    // Only delete m_component if we were loading from a URL otherwise it means
    // m_component = m_sourceComponent which we don't own.
    if (m_component && m_component != m_sourceComponent)
        delete m_component;
    m_component = nullptr;

    if (m_context) {
        delete m_context;
        m_context = nullptr;
    }
}

void Quick3DEntityLoaderPrivate::loadFromSource()
{
    Q_Q(Quick3DEntityLoader);

    if (m_source.isEmpty()) {
        emit q->entityChanged();
        return;
    }

    loadComponent(m_source);
}

void Quick3DEntityLoaderPrivate::loadComponent(const QUrl &source)
{
    Q_Q(Quick3DEntityLoader);

    Q_ASSERT(m_entity == nullptr);
    Q_ASSERT(m_component == nullptr);
    Q_ASSERT(m_context == nullptr);

    auto owner = _q_findQmlOwner(q);
    m_component = new QQmlComponent(owner.engine, owner.object);
    QObject::connect(m_component, SIGNAL(statusChanged(QQmlComponent::Status)),
                     q, SLOT(_q_componentStatusChanged(QQmlComponent::Status)));
    m_component->loadUrl(source, QQmlComponent::Asynchronous);
}

void Quick3DEntityLoaderPrivate::loadComponent(QQmlComponent *component)
{
    Q_ASSERT(m_entity == nullptr);
    Q_ASSERT(m_component == nullptr);
    Q_ASSERT(m_context == nullptr);

    m_component = component;
    _q_componentStatusChanged(m_component ? m_component->status() : QQmlComponent::Null);
}

void Quick3DEntityLoaderPrivate::_q_componentStatusChanged(QQmlComponent::Status status)
{
    Q_Q(Quick3DEntityLoader);

    Q_ASSERT(m_entity == nullptr);
    Q_ASSERT(m_context == nullptr);
    Q_ASSERT(m_incubator == nullptr);

    if (!m_component) {
        clear();
        emit q->entityChanged();
        return;
    }

    auto owner = _q_findQmlOwner(q);
    if (!m_component->errors().isEmpty()) {
        QQmlEnginePrivate::warning(owner.engine, m_component->errors());
        clear();
        emit q->entityChanged();
        return;
    }

    // Still loading
    if (status != QQmlComponent::Ready)
        return;

    m_context = new QQmlContext(owner.context());
    m_context->setContextObject(owner.object);

    m_incubator = new Quick3DEntityLoaderIncubator(q);
    m_component->create(*m_incubator, m_context);
}

void Quick3DEntityLoaderPrivate::setStatus(Quick3DEntityLoader::Status status)
{
    Q_Q(Quick3DEntityLoader);
    if (status != m_status) {
        m_status = status;
        const bool blocked = q->blockNotifications(true);
        emit q->statusChanged(m_status);
        q->blockNotifications(blocked);
    }
}

} // namespace Quick
} // namespace Qt3DCore

QT_END_NAMESPACE

#include "moc_quick3dentityloader_p.cpp"
