// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_SPDY_CORE_HPACK_HPACK_DECODER_INTERFACE_H_
#define NET_SPDY_CORE_HPACK_HPACK_DECODER_INTERFACE_H_

// HpackDecoderInterface is the base class for HPACK block decoders.
// HPACK is defined in http://tools.ietf.org/html/rfc7541

#include <stddef.h>

#include <memory>

#include "net/spdy/core/hpack/hpack_header_table.h"
#include "net/spdy/core/spdy_header_block.h"
#include "net/spdy/core/spdy_headers_handler_interface.h"
#include "net/spdy/platform/api/spdy_export.h"

namespace net {

class SPDY_EXPORT_PRIVATE HpackDecoderInterface {
 public:
  virtual ~HpackDecoderInterface() {}

  // Called upon acknowledgement of SETTINGS_HEADER_TABLE_SIZE.
  virtual void ApplyHeaderTableSizeSetting(size_t size_setting) = 0;

  // If a SpdyHeadersHandlerInterface is provided, the decoder will emit
  // headers to it rather than accumulating them in a SpdyHeaderBlock.
  // Does not take ownership of the handler, but does use the pointer until
  // the current HPACK block is completely decoded.
  virtual void HandleControlFrameHeadersStart(
      SpdyHeadersHandlerInterface* handler) = 0;

  // Called as HPACK block fragments arrive. Returns false if an error occurred
  // while decoding the block. Does not take ownership of headers_data.
  virtual bool HandleControlFrameHeadersData(const char* headers_data,
                                             size_t headers_data_length) = 0;

  // Called after a HPACK block has been completely delivered via
  // HandleControlFrameHeadersData(). Returns false if an error occurred.
  // |compressed_len| if non-null will be set to the size of the encoded
  // buffered block that was accumulated in HandleControlFrameHeadersData(),
  // to support subsequent calculation of compression percentage.
  // Discards the handler supplied at the start of decoding the block.
  // TODO(jamessynge): Determine if compressed_len is needed; it is used to
  // produce UUMA stat Net.SpdyHpackDecompressionPercentage, but only for
  // deprecated SPDY3.
  virtual bool HandleControlFrameHeadersComplete(size_t* compressed_len) = 0;

  // Accessor for the most recently decoded headers block. Valid until the next
  // call to HandleControlFrameHeadersData().
  // TODO(birenroy): Remove this method when all users of HpackDecoder specify
  // a SpdyHeadersHandlerInterface.
  virtual const SpdyHeaderBlock& decoded_block() const = 0;

  virtual void SetHeaderTableDebugVisitor(
      std::unique_ptr<HpackHeaderTable::DebugVisitorInterface> visitor) = 0;

  // Set how much encoded data this decoder is willing to buffer.
  // TODO(jamessynge): Resolve definition of this value, as it is currently
  // too tied to a single implementation. We probably want to limit one or more
  // of these: individual name or value strings, header entries, the entire
  // header list, or the HPACK block; we probably shouldn't care about the size
  // of individual transport buffers.
  virtual void set_max_decode_buffer_size_bytes(
      size_t max_decode_buffer_size_bytes) = 0;

  // Returns the estimate of dynamically allocated memory in bytes.
  virtual size_t EstimateMemoryUsage() const = 0;
};

}  // namespace net

#endif  // NET_SPDY_CORE_HPACK_HPACK_DECODER_INTERFACE_H_
