/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.telephony.mbms;

import android.annotation.IntDef;
import android.annotation.NonNull;
import android.annotation.Nullable;
import android.content.Context;
import android.telephony.MbmsGroupCallSession;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.List;
import java.util.concurrent.Executor;

/**
 * A callback class that is used to receive information from the middleware on MBMS group-call
 * services. An instance of this object should be passed into
 * {@link MbmsGroupCallSession#create(Context, int, Executor, MbmsGroupCallSessionCallback)}.
 */
public interface MbmsGroupCallSessionCallback {
    /** @hide */
    @Retention(RetentionPolicy.SOURCE)
    @IntDef(value = {
            MbmsErrors.ERROR_NO_UNIQUE_MIDDLEWARE,
            MbmsErrors.ERROR_MIDDLEWARE_LOST,
            MbmsErrors.ERROR_MIDDLEWARE_NOT_BOUND,
            MbmsErrors.InitializationErrors.ERROR_APP_PERMISSIONS_NOT_GRANTED,
            MbmsErrors.InitializationErrors.ERROR_DUPLICATE_INITIALIZE,
            MbmsErrors.InitializationErrors.ERROR_UNABLE_TO_INITIALIZE,
            MbmsErrors.GeneralErrors.ERROR_MIDDLEWARE_NOT_YET_READY,
            MbmsErrors.GeneralErrors.ERROR_OUT_OF_MEMORY,
            MbmsErrors.GeneralErrors.ERROR_MIDDLEWARE_TEMPORARILY_UNAVAILABLE,
            MbmsErrors.GeneralErrors.ERROR_IN_E911,
            MbmsErrors.GeneralErrors.ERROR_NOT_CONNECTED_TO_HOME_CARRIER_LTE,
            MbmsErrors.GeneralErrors.ERROR_UNABLE_TO_READ_SIM,
            MbmsErrors.GeneralErrors.ERROR_CARRIER_CHANGE_NOT_ALLOWED}, prefix = { "ERROR_" })
    @interface GroupCallError{}

    /**
     * Called by the middleware when it has detected an error condition. The possible error codes
     * are listed in {@link MbmsErrors}.
     * @param errorCode The error code.
     * @param message A human-readable message generated by the middleware for debugging purposes.
     */
    default void onError(@GroupCallError int errorCode, @Nullable String message) {}

    /**
     * Indicates that the list of currently available SAIs has been updated. The app may use this
     * information to filter the list of group calls when displaying available group calls to
     * the user by matching the SAIs with a list of group calls separately negotiated with the
     * carrier. The app may also report the aggregate list of SAIs to the group call application
     * server so that a network entity can determine when, and where to activate the broadcast of
     * particular group calls.
     * @param currentSais The available SAIs on the current cell.
     * @param availableSais A list of lists of available SAIS in neighboring cells, where each list
     *                      contains the available SAIs in an individual cell.
     */
    default void onAvailableSaisUpdated(@NonNull List<Integer> currentSais,
            @NonNull List<List<Integer>> availableSais) {}

    /**
     * Called soon after the app calls {@link MbmsGroupCallSession#create}. The information supplied
     * via this callback may be used to establish a data-link interface with the modem.
     *
     * In order to establish the data-link interface, the multicast IP and port must be obtained
     * out-of-band from the carrier. A {@link java.net.MulticastSocket} may then be constructed
     * using a {@link java.net.NetworkInterface} with the name and interface supplied by this
     * callback.
     *
     * @param interfaceName The interface name for the data link.
     * @param index The index for the data link.
     */
    default void onServiceInterfaceAvailable(@NonNull String interfaceName, int index) {}

    /**
     * Called to indicate that the middleware has been initialized and is ready.
     *
     * Before this method is called, calling any method on an instance of
     * {@link MbmsGroupCallSession} will result in an {@link IllegalStateException} or an error
     * delivered via {@link #onError(int, String)} with error code
     * {@link MbmsErrors.GeneralErrors#ERROR_MIDDLEWARE_NOT_YET_READY}.
     */
    default void onMiddlewareReady() {}
}
