/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


// --------------------------------------------------------------------
//  AmrDeriveDist.cpp
// --------------------------------------------------------------------
//    This app will read a plotfile and write a new plot file
//    with the same components but with data distributed over
//    parallel disks.
// --------------------------------------------------------------------

#include <winstd.H>
#include <new>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"

//
// This MUST be defined if don't have pubsetbuf() in I/O Streams Library.
//
#ifdef BL_USE_SETBUF
#define pubsetbuf setbuf
#endif

static
void
PrintUsage (const char* progName)
{
    cout << '\n';
    cout << "Usage:" << '\n';
    cout << progName << '\n';
    cout << "    infile  = inputFileName" << '\n';
    cout << "    outfile = outputFileName" << '\n';
    cout << "   [-help]" << '\n';
    cout << "   [-verbose]" << '\n';
    cout << '\n';

    exit(1);
}

int
main (int   argc,
      char* argv[])
{
    if(argc == 1)
        PrintUsage(argv[0]);
    //
    // Make sure to catch new failures.
    //
    set_new_handler(Utility::OutOfMemory);

    ParallelDescriptor::StartParallel(&argc, &argv);

    ParmParse pp(argc-1,argv+1);

    if(pp.contains("help"))
        PrintUsage(argv[0]);

    FArrayBox::setFormat(FABio::FAB_IEEE_32);
    //
    // Set default info.
    //
    // Scan the arguments.
    //
    std::string iFileDir, iFile, oFile, oFileDir;

    bool verbose = false;
    if(pp.contains("verbose")) {
        verbose = true;
        AmrData::SetVerbose(true);
    }
    pp.query("infile", iFile);
    if(iFile.isNull()) {
      BoxLib::Abort("You must specify `infile'");
    }
    pp.query("outfile", oFile);
    if(oFile.isNull()) {
      BoxLib::Abort("You must specify `outfile'");
    }

    int i;
    double dTime0, dTime1;
    Real rTime;
    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    int nProcs(ParallelDescriptor::NProcs());
    
    dTime0 = Utility::second();
    DataServices dataServices(iFile, fileType);

    if( ! dataServices.AmrDataOk()) {
        //
        // This calls ParallelDescriptor::EndParallel() and exit()
        //
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
    }

    AmrData &amrData = dataServices.AmrDataRef();

    const Array<std::string> &derives = amrData.PlotVarNames();
    int nComps(amrData.NComp());

    int finestLevel = amrData.FinestLevel();
    //
    // Make the result MultiFab's.
    //
    Array<MultiFab *> mfout(finestLevel + 1);


    for(int iLevel = 0; iLevel <= finestLevel; ++iLevel) {
      mfout[iLevel] = new MultiFab(amrData.boxArray(iLevel), nComps, 0);
      for(int iComp = 0; iComp < nComps; ++iComp) {
        mfout[iLevel]->copy(amrData.GetGrids(iLevel, iComp), 0, iComp, 1);
      }
    }
    dTime1 = Utility::second() - dTime0;
    rTime = (Real) dTime1;
    ParallelDescriptor::ReduceRealMax(rTime);
    if(ParallelDescriptor::IOProcessor()) {
      cout << "[" << nProcs << " procs]:  Read pltfile time  = " << rTime << endl;
      cout << endl;
    }



    //
    // write the new plot file
    //
    dTime0 = Utility::second();
    if(ParallelDescriptor::IOProcessor()) {
        if(!Utility::UtilCreateDirectory(oFile,0755)) {
            Utility::CreateDirectoryFailed(oFile);
	}
    }
    //
    // Force other processors to wait till directory is built.
    //
    ParallelDescriptor::Barrier();

    std::string oFileHeader(oFile);
    oFileHeader += "/Header";
  
    VisMF::IO_Buffer io_buffer(VisMF::IO_Buffer_Size);

    ofstream os;
  
    os.rdbuf()->pubsetbuf(io_buffer.dataPtr(), io_buffer.length());
  
    if(verbose && ParallelDescriptor::IOProcessor()) {
        cout << "Opening file = " << oFileHeader << '\n';
    }

    os.open(oFileHeader.c_str(), ios::out|ios::binary);

    if(os.fail()) {
        Utility::FileOpenFailed(oFileHeader);
    }

    //
    // Start writing plotfile.
    //
    os << amrData.PlotFileVersion() << '\n';
    os << nComps << '\n';
    for(int n = 0; n < nComps; n++) os << derives[n] << '\n';
    os << BL_SPACEDIM << '\n';
    os << amrData.Time() << '\n';
    os << finestLevel << '\n';
    for(i = 0; i < BL_SPACEDIM; i++) os << amrData.ProbLo()[i] << ' ';
    os << '\n';
    for(i = 0; i < BL_SPACEDIM; i++) os << amrData.ProbHi()[i] << ' ';
    os << '\n';
    for(i = 0; i < finestLevel; i++) os << amrData.RefRatio()[i] << ' ';
    os << '\n';
    for(i = 0; i <= finestLevel; i++) os << amrData.ProbDomain()[i] << ' ';
    os << '\n';
    for(i = 0; i <= finestLevel; i++) os << 0 << ' ';
    os << '\n';
    for(i = 0; i <= finestLevel; i++) {
        for(int k = 0; k < BL_SPACEDIM; k++)
            os << amrData.DxLevel()[i][k] << ' ';
        os << '\n';
    }
    os << amrData.CoordSys() << '\n';
    os << "0\n"; // The bndry data width.
    //
    // Write out level by level.
    //
    for(int iLevel = 0; iLevel <= finestLevel; ++iLevel) {
        //
        // Write state data.
        //
        int nGrids = amrData.boxArray(iLevel).length();
        char buf[64];
        sprintf(buf, "Level_%d", iLevel);
    
        if(ParallelDescriptor::IOProcessor())
        {
            os << iLevel << ' ' << nGrids << ' ' << amrData.Time() << '\n';
            os << 0 << '\n';
    
            for(i = 0; i < nGrids; ++i) {
                for(int n = 0; n < BL_SPACEDIM; n++) {
                    os << amrData.GridLocLo()[iLevel][i][n]
                       << ' '
                       << amrData.GridLocHi()[iLevel][i][n]
                       << '\n';
                }
            }
            //
            // Build the directory to hold the MultiFabs at this level.
            //
            std::string Level(oFile);
            Level += '/';
            Level += buf;
    
            if( ! Utility::UtilCreateDirectory(Level, 0755)) {
                Utility::CreateDirectoryFailed(Level);
	    }
        }
        //
        // Force other processors to wait till directory is built.
        //
        ParallelDescriptor::Barrier();
        //
        // Now build the full relative pathname of the MultiFab.
        //
        static const std::string MultiFabBaseName("/MultiFab");
    
        std::string PathName(oFile);
        PathName += '/';
        PathName += buf;
        PathName += MultiFabBaseName;
    
        if(ParallelDescriptor::IOProcessor()) {
            //
            // The full name relative to the Header file.
            //
            std::string RelativePathName(buf);
            RelativePathName += '/';
            RelativePathName += MultiFabBaseName;
            os << RelativePathName << '\n';
        }

        VisMF::Write(*mfout[iLevel], PathName, VisMF::OneFilePerCPU);
        delete mfout[iLevel];
    }

    os.close();

    dTime1 = Utility::second() - dTime0;
    rTime = (Real) dTime1;
    ParallelDescriptor::ReduceRealMax(rTime);
    if(ParallelDescriptor::IOProcessor()) {
      cout << "[" << nProcs << " procs]:  Write pltfile time = " << rTime << endl;
      cout << endl;
    }

    //
    // This calls ParallelDescriptor::EndParallel() and exit()
    //
    DataServices::Dispatch(DataServices::ExitRequest, NULL);
}
