#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Implemention conformance tests."""

import os
import shutil
import tests
import unittest

# if this fails to import, then the implementations module is bust
# and this test script will never be reached to be imported
import config_manager.implementations


# The following tests test implementation conformance with the Implementation
# contract 
# what would be ideal is that registration of an implementation triggers a test
# suite for it. 
# something along the lines of:
# * implementations are loaded via plugins (see bzr code for plugin logic
#   inspiration.
# * implementations should always load in enough to say they are not available.
# * the implementation registrar is used by test_suite() to test the
#   implementations.
#   - downside: sufficiently invalid implementations will not be tested.
# * the test cases are parameterised by injecting an implementation into
#   them. No service location is needed for this because implementations
#   are at the bottom of the stack.
# for now, though, one test suite, tests the fake implementation directly,
# and we will refactor that into the above concept as we go.


class TestImplementation(unittest.TestCase):

    def test_null_constructor(self):
        # implementations must be default constructable
        self.implementation_class()

    def test_lists_supported_schemes(self):
        # implementations must provide a list of the url schemes they handle.
        self.assertNotEqual(0, len(self.implementation_class().schemes))
    
    def test_is_registered_once_only(self):
        instances = [imp for imp in 
            config_manager.implementations.all_implementations if
            isinstance(imp, self.implementation_class)]
        self.assertEqual(1, len(instances))

    def test_provides_test_repository(self):
        # a test repository is something implementations create and we can
        # test against - the implementation acts like a factory for this for us.
        implementation = self.implementation_class()
        test_repo = implementation.get_test_repo()
        test_repo.setUp() 
        url = test_repo.get_branch_url()
        not_valid_url = test_repo.get_not_branch_url()
        bad_url = test_repo.get_bad_url()
        test_repo.tearDown()
        self.failUnless(url)
        self.failUnless(not_valid_url)
        self.failUnless(bad_url)

    def test_schemes_are_registered(self):
        # This tests that each scheme offered by this implementation is 
        # registered.
        implementation = self.implementation_class()
        for scheme in implementation.schemes:
            implementations = config_manager.implementations.schemes[scheme]
            this_imps = [imp for imp in implementations 
                         if isinstance(imp, self.implementation_class)]
            self.assertEqual(1, len(this_imps))

    def test_checkout_not_valid_url_returns_false(self):
        # a not valid url is one with the right url scheme but 
        # content that is not for this vcs. I.e. a http url
        # but not a svn server there for svn vcs's.
        implementation = self.implementation_class()
        test_repo = implementation.get_test_repo()
        test_repo.setUp() 
        try:
            not_valid_url = test_repo.get_not_branch_url()
            result = implementation.checkout(not_valid_url, "/tmp/blargh")
            self.assertEqual(False, result)
            self.failIf(os.path.exists("/tmp/blargh"))
        finally:
            test_repo.tearDown()
        
    def test_checkout_bad_url_raises(self):
        # a bad url is one with the right scheme, right content,
        # but unusable for some reason - i.e. server failure, permissions etc.
        # i.e. an http:// url with a svn server, but a 403 on some files or
        # so on.
        implementation = self.implementation_class()
        test_repo = implementation.get_test_repo()
        test_repo.setUp() 
        try:
            bad_url = test_repo.get_bad_url()
            self.assertRaises(config_manager.implementations.UnusableURL,
                              implementation.checkout, bad_url, "/tmp/blargh")
            self.failIf(os.path.exists("/tmp/blargh"))
        finally:
            test_repo.tearDown()

    def test_checkout_bad_url_raises(self):
        # a not valid url is one with the right scheme, right content,
        # but unusable for some reason - i.e. server failure, permissions etc.
        implementation = self.implementation_class()
        test_repo = implementation.get_test_repo()
        test_repo.setUp() 
        try:
            bad_url = test_repo.get_bad_url()
            shutil.rmtree("/tmp/blargh", ignore_errors=True)
            self.assertRaises(config_manager.implementations.UnusableURL,
                              implementation.checkout, bad_url, "/tmp/blargh")
            self.failIf(os.path.exists("/tmp/blargh"))
        finally:
            test_repo.tearDown()

    def test_checkout_returns_true(self):
        # successful checkouts return True
        implementation = self.implementation_class()
        test_repo = implementation.get_test_repo()
        test_repo.setUp() 
        try:
            url = test_repo.get_branch_url()
            result = implementation.checkout(url, "/tmp/blargh")
            try:
                self.assertEqual(True, result)
                self.failUnless(os.path.exists("/tmp/blargh"))
            finally:
                shutil.rmtree("/tmp/blargh")
        finally:
            test_repo.tearDown()


class adaptingVisitor(tests.TestUtil.TestVisitor):
    """I accure the adapted test cases I visit into my suite."""

    def __init__(self, adapter):
        tests.TestUtil.TestVisitor.__init__(self)
        self._suite = None
        self._adapter = adapter

    def suite(self):
        """answer the suite we are building"""
        if self._suite is None:
            self._suite = tests.TestSuite()
        return self._suite

    def visitCase(self, aCase):
        self.suite().addTests(self._adapter.adapt(aCase))


def test_suite():
    implementation_adapter = config_manager.implementations.tests.\
        ImplementationTestAdapter()
    visitor = adaptingVisitor(implementation_adapter)
    loader = tests.TestLoader()
    implementation_tests = loader.loadTestsFromName(__name__)
    implementation_tests.visit(visitor)
    return visitor.suite()
