-- ------------------------------------------------------------------------- --
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/objects/cache.adb,v $
--  Description     : Base class for all persistent objects                  --
--  Author          : Michael Erdmann <Michael.Erdmann@snafu.de>             --
--  Created On      : 30-April-2005                                          --
--  Last Modified By: $Author: merdmann $                                    --
--  Last Modified On: $Date: 2007/01/21 20:27:21 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2006-2007 Michael Erdmann                                  --
--                                                                           --
--  GNADE is copyrighted by the persons and institutions enumerated in the   --
--  AUTHORS file. This file is located in the root directory of the          --
--  GNADE distribution.                                                      --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNADE;  see file COPYING. If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from      --
--  GNADE Ada units, or you link GNADE Ada units or libraries with other     --
--  files  to produce an executable, these  units or libraries do not by     --
--  itself cause the resulting  executable  to  be covered  by the  GNU      --
--  General  Public  License.  This exception does not however invalidate    --
--  any other reasons why  the executable file  might be covered by the      --
--  GNU Public License.                                                      --
--                                                                           --
-- ------------------------------------------------------------------------- --
with Ada.Streams;               	use Ada.Streams;
with Ada.Streams.Stream_IO;     	use Ada.Streams.Stream_IO;
with Ada.Text_IO;			use Ada.Text_IO;
with Ada.Storage_IO;
use  Ada;

with System.Storage_Elements;		use System.Storage_Elements;
use  System;

with GNU.DB.SQLCLI;			use GNU.DB.SQLCLI;
with SQL_Standard;      		use SQL_Standard;

with Util.Trace_Helper;
with Util.Dynamic_Hashtable;
with Util.Stack;

with Util.Types;			use Util.Types;
use  Util;

with Objects;				use Objects;
with BLOB_Stream;			use BLOB_Stream;
with MD5;				use MD5;

package body Cache is

   Version : constant String :=
       "$Id: cache.adb,v 1.2 2007/01/21 20:27:21 merdmann Exp $";

   -- Trace facitlity package
   package Tracer is new Util.Trace_Helper( Module => "Cache");
   use Tracer;

   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   -- ++++++++          P R I V A T E    D A T A                ++++++++++ --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   --                                                                      --
   -- Cache Tables                                                         --
   -- ============                                                         --
   -- The cache mainaint two major hash tables. The so called object table --
   -- which maps the object id into the actual cache entries and the so    --
   -- called finger print table, which maps the MD5 fingerprints onto      --
   -- object id's.                                                         --
   --                                                                      --
   -- Pools for BLOBS and Cache_Entries                                    --
   -- =================================                                    --
   -- Since BLOBS and Cache_Entries are allocated diynamically they are    --
   -- only once allocate and put into a pool when they are freed.          --
   --                                                                      --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   TTL_Default : constant Natural := 10;

   Cache_Hits : Natural := 0;
   Cache_Get  : Natural := 0;
   Cache_Put  : Natural := 0;

   ----------------------
   -- Cache_Entry_Type --
   ----------------------
   type Cache_Entry_Type is record
         TTL        : Natural     := TTL_Default;
         Hi_Version : Natural := 0;
         Lo_Version : Natural := 0;
         Instance   : BLOB_Access := Null;
         Length     : Natural := 0;
         Ref_Count  : Natural := 0;
      end record;

   type Cache_Entry_Access is access Cache_Entry_Type;

   package Cache_Entry_Pool is
     new Util.Stack( Element_Type => Cache_Entry_Access);
   use Cache_Entry_Pool;

   package BLOB_Pool is new Util.Stack( Element_Type => BLOB_Access );
   use BLOB_Pool;

   --------------------
   -- 	Object_Table --
   --------------------
   package Object_Table is
      new Dynamic_Hashtable( Payload_Type => Cache_Entry_Access );
   use Object_Table;

   OT            : Object_Table.Object( 1000 ) ;
   BLOBS         : BLOB_Pool.Object := BLOB_Pool.Create;
   Cache_Entries : Cache_Entry_Pool.Object := Cache_Entry_Pool.Create;

   -----------------------
   -- Fingerprint_Table --
   -----------------------
   package Fingerprint_Table is
     new Dynamic_Hashtable( Payload_Type => Integer );

   FP       : Fingerprint_Table.Object( 1000 );

   --------------------
   -- Object_ID_Type --
   --------------------
   type Object_ID_Type is record
         Id      : Integer;
         Version : Integer;
      end record;

   package ObjectID_IO is new Storage_IO( Element_Type => Object_ID_Type );

   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   -- ++++++++        P R I V A T E     M E T H O D S           ++++++++++ --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   --                                                                      --
   --                                                                      --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --

   --------------
   -- New_Blob --
   --------------
   function New_BLOB return Blob_Access is
      Result : Blob_Access;
   begin
      Enter("New_BLOB");

      Pop( BLOBS, Result );

      Leave("New_Blob");
      return Result;

   exception
      when Blob_Pool.Stack_Empty =>
         return new BLOB_Type;
   end New_Blob;

   ---------------
   -- Free_Blob --
   ---------------
   procedure Free_Blob(
      Blob : BLOB_Access ) is
   begin
      Enter("Free_Blob");
      Push( BLOBS, Blob );

      Leave("Free_Blob");
   end Free_Blob;

   ---------------------
   -- New_Cache_Entry --
   ---------------------
   function New_Cache_Entry return Cache_Entry_Access is
      Result : Cache_Entry_Access := null;
   begin
      Enter("New_Cache_Entry");

      Pop( Cache_Entries, Result );

      Leave("New_Cache_Entry");
      return Result;

   exception
      when Cache_Entry_Pool.Stack_Empty =>
         return new Cache_Entry_Type;
   end New_Cache_Entry;

   ---------------
   -- Free_Blob --
   ---------------
   procedure Free_Cache_Entry(
      Item : Cache_Entry_Access ) is
   begin
      Enter("Free_Cache_Entires");
      Push( Cache_Entries, Item );

      Leave("Free_Cache_Entry");
   end Free_Cache_Entry;


   --------------
   -- Make_Key --
   --------------
   function Make_Key(
      Id   : in Integer;
      Vers : in Integer ) return Key_Type is

      use ObjectID_IO;

      OID    : Object_ID_Type := (Id,Vers);
      Result : Buffer_Type;
   begin
      Enter("Make_Key(" &
            Integer'Image(Id) & ", " &
            Integer'Image(Vers) & ")" );
      Write( Result, OID );
      return Result;
   end Make_Key;

   ------------------
   -- To_Object_ID --
   ------------------
   function To_Object_ID(
      Key  : in Key_Type ) return Object_ID_Type is

      use ObjectID_IO;
      Data : Buffer_Type := Key( 1..Buffer_Type'Length );

      Result : Object_ID_Type ;
   begin
      Read( Data, Result );
      return Result;
   end To_Object_ID;

   --------------
   -- Make_Key --
   --------------
   function Make_Key(
      FP : in GNADE.BINARY ) return Key_Type is

      Result : Key_Type(1..16);
   begin
      for i in FP'Range loop
         Result( Storage_Offset(i)) := Storage_Element(FP(i));
      end loop;

      return Result;
   end Make_Key;

   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   -- ++++++++        P U B L I C     M E T H O D S             ++++++++++ --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --
   --                                                                      --
   --                                                                      --
   -- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ --


   -----------
   -- Flush --
   -----------
   procedure Flush is
      -- flush the cache by writing out all object in the cache.
   begin
      Enter("Flush");

      Info("Caching Statistics:");
      Info("   Put :" & Natural'Image(Cache_Put));
      Info("   Get :" & Natural'Image(Cache_Get));
      Info("   Hits:" & Natural'Image(Cache_Hits));

      Leave("Flush");
   end Flush;

   ----------------
   -- Invalidate --
   ----------------
   procedure Invalidate(
      Id      : in Integer;
      Version : in Integer ) is

      Item : Cache_Entry_Access := null;
      Key  : constant Key_Type := Make_Key( Id, Version );
   begin
      Enter("Invalidate(" &
            Integer'Image(Id) & ", " &
            Integer'Image(Version) & ")" );

      Object_Table.Get( OT, Key, Item );
      Item.Ref_Count := Item.Ref_Count - 1;

      if Item.Ref_Count < 1 then
         Free_Blob( Item.Instance );
         Object_Table.Delete( OT, Key );

         Free_Cache_Entry( Item );
      end if;
      Leave("Invalidate");

   exception
      when Object_Table.Key_Not_Found =>
         null;
   end Invalidate;

   ---------
   -- Put --
   ---------
   procedure Put_Object(
      Id         : in Integer;
      Version    : in out Integer;
      Fprint     : in GNADE.BINARY;
      BLOB       : in GNADE.BINARY;
      Class_Name : in String) is
      -- store an object in the cache
      Item    : Cache_Entry_Access := null;
      Object_Version : Integer := Version;
   begin
      Enter("Put_Object(" &
            Integer'Image(ID) & ", " &
            Integer'Image(Version) & ", " &
            Class_Name &")");

      Cache_Put := Cache_Put + 1;

      begin
         Object_Table.Get( OT, Make_Key(Id, Version), Item);

         Item.Instance(1..BLOB'Length) := BLOB;
         Item.Length := BLOB'Length;

         Item.TTL := Item.TTL + 1;

         Cache_Hits := Cache_Hits + 1;

      exception
         when Object_Table.Key_Not_Found =>
            Info("Key_Not_Found =>");

            Item := new Cache_Entry_Type;

            Item.Instance := New_Blob;
            Item.Instance(1..BLOB'Length) := BLOB;
            Item.Length := BLOB'Length;

            Object_Table.Put( OT, Make_Key(Id,Version), Item  );
      end;

      Insert_Object(
         ID,
         Object_Version,
         BLOB,
         Fprint,
         Class_Name
      );

      case Version is
         when Highest_Version =>
            Item.Hi_Version := Object_Version;
         when Lowest_Version =>
            Item.Lo_Version := Object_Version;
         when others =>
            null;
      end case;

      Version := Object_Version;

      Leave("Put_Object(" &
            Integer'Image(ID) & ", " &
            Integer'Image(Version) & ", " &
            Class_Name &")");
   end Put_Object;

   ----------------
   -- Get_Object --
   ----------------
   procedure Get_Object(
      Id      : in Integer;
      Version : in out Integer;
      BLOB    : out BLOB_Access;
      Length  : out Natural) is
      -- get an object by means of the object id aqnd version
      Object_Version : Integer := Version;
      Item    : Cache_Entry_Access := null;
   begin
      Enter("Get_Object( " & Integer'Image(Id) & ", " & Integer'Image(Version) & ")" );

      Cache_Get := Cache_Get + 1;

      begin
         Object_Table.Get( OT, Make_Key(Id, Version), Item);

         Cache_Hits := Cache_Hits + 1;

         BLOB     := Item.Instance;
         Length   := Item.Length;

         Item.TTL       := Item.TTL + 1;
         Item.Ref_Count := Item.Ref_Count + 1;

         case Version is
            when Highest_Version =>
               Object_Version := Item.Hi_Version;
            when Lowest_Version =>
               Object_Version := Item.Lo_Version;
            when others =>
               null;
         end case;

      exception
         when Object_Table.Key_Not_Found =>
            begin
               BLOB := New_Blob;
               Objects.Get_Object( Id, Object_Version, BLOB.all, Length );

               Item := new Cache_Entry_Type ;
               Item.Instance := BLOB;
               Item.Length := Length;
               Item.Ref_Count := 1;

               case Version is
                  when Highest_Version =>
                     Item.Hi_Version := Object_Version;
                  when Lowest_Version =>
                     Item.Lo_Version := Object_Version;
                  when others =>
                     null;
               end case;

               Object_Table.Put( OT, Make_Key(Id,Version), Item  );
            end ;
      end;

      Version := Object_Version;
      Leave("Get_Object(" &
            "Id=" & Integer'Image( Id ) & ", " &
            "Version=" & Integer'Image( Version ) & ", " &
            "Length=" & Natural'Image( Length ) & ")" );
   end Get_Object;

   ----------------
   -- Get_Object --
   ----------------
   procedure Get_Object(
      Fprint  : in GNADE.BINARY;
      Version : in out Integer;
      BLOB    : out BLOB_Access;
      Length  : out Natural) is
      ID      : Integer;
   begin
      Enter("Get_Object( <key>, Version=" & Integer'Image(Version) & ")" );

      begin
         Fingerprint_Table.Get( FP, Make_Key( Fprint ), ID );
         Cache.Get_Object( ID, Version, BLOB, Length );

      exception
         when Fingerprint_Table.Key_Not_Found =>
            declare
               Object_Version : Integer := Version;
               Item    : Cache_Entry_Access := new Cache_Entry_Type ;
            begin
               Cache_Get := Cache_Get + 1;

               BLOB := New_Blob;
               Objects.Get_Object( Fprint, Object_Version, ID, BLOB.all, Length );
               Fingerprint_Table.Put( FP, Make_Key( Fprint ), ID );

               Item.Instance  := BLOB;
               Item.Length    := Length;
               Item.Ref_Count := 1;

               case Version is
                  when Highest_Version =>
                     Item.Hi_Version := Object_Version;
                  when Lowest_Version =>
                     Item.Lo_Version := Object_Version;
                  when others =>
                     null;
               end case;

               Object_Table.Put( OT, Make_Key(ID,Version), Item  );

               Version := Object_Version;
            end ;

      end ;

      Leave("Get_Object( <key>," &
               "Id=" & Integer'Image(ID) &
               "Version=" & Integer'Image(Version) & ", " &
               "Length=" & Natural'Image(Length) & ")" );
   end ;

end Cache;
